InstallMethod( \=,
   "for two nilpotent orbits",
   true, [ IsNilpotentOrbit, IsNilpotentOrbit ], 0,
   function( a, b )

   return AmbientLieAlgebra(a) = AmbientLieAlgebra(b) and
          WeightedDynkinDiagram(a) = WeightedDynkinDiagram(b);

end );

InstallMethod( NilpotentOrbit,
   "for a Lie algebra and list",
   true, [ IsLieAlgebra, IsList ], 0,
   function( L, list )

      local o, fam;

      if not IsBound( L!.nilorbType ) then
         fam:= NewFamily( "nilorbsfam", IsNilpotentOrbit );
         fam!.invCM:= CartanMatrix( RootSystem(L) )^-1;
         L!.nilorbType:= NewType( fam, 
                           IsNilpotentOrbit and IsAttributeStoringRep );
      fi;
      o:= Objectify( L!.nilorbType, rec() );
      SetAmbientLieAlgebra( o, L );
      SetWeightedDynkinDiagram( o, list );
      if HasSemiSimpleType(L) then
         SetSemiSimpleType( o, SemiSimpleType(L) );
      fi; 
      return o;

end );


InstallMethod( SL2Triple,
   "for a Lie algebra and a nilpotent element",
   true, [ IsLieAlgebra, IsObject ], 0,

function( L, x )

  # Same as FindSl2 in the library; only returns the elements, rather
  # than the subalgebra. (Advantage: GAP doesn't change the basis.)

   local n,         # the dimension of `L'
         F,         # the field of `L'
         B,         # basis of `L'
         T,         # the table of structure constants of `L'
         xc,        # coefficient vector
         eqs,       # a system of equations
         i,j,k,l,   # loop variables
         cij,       # the element `T[i][j]'
         b,         # the right hand side of the equation system
         v,         # solution of the equations
         z,         # element of `L'
         h,         # element of `L'
         R,         # centralizer of `x' in `L'
         BR,        # basis of `R'
         Rvecs,     # basis vectors of `R'
         H,         # the matrix of `ad H' restricted to `R'
         e0,        # coefficient vector
         e1,        # coefficient vector
         y;         # element of `L'

    if not IsNilpotentElement( L, x ) then
      Error( "<x> must be a nilpotent element of the Lie algebra <L>" );
    fi;

    n:= Dimension( L );
    F:= LeftActingDomain( L );
    B:= Basis( L );
    T:= StructureConstantsTable( B );

    xc:= Coefficients( B, x );
    eqs:= NullMat( 2*n, 2*n, F );

    # First we try to find elements `z' and `h' such that `[x,z]=h'
    # and `[h,x]=2x' (i.e., such that two of the three defining equations
    # of sl_2 are satisfied).
    # This results in a system of `2n' equations for `2n' variables.

    for i in [1..n] do
      for j in [1..n] do
        cij:= T[i][j];
        for k in [1..Length(cij[1])] do
          l:= cij[1][k];
          eqs[i][l] := eqs[i][l] + xc[j]*cij[2][k];
          eqs[n+i][n+l]:= eqs[n+i][n+l] + xc[j]*cij[2][k];
        od;
      od;
      eqs[n+i][i]:= One( F );
    od;

    b:= [];
    for i in [1..n] do
      b[i]:= Zero( F );
      b[n+i]:= 2*One( F )*xc[i];
    od;

    v:= SolutionMat( eqs, b );

    if v = fail then
      # There is no sl_2 containing <x>.
      return fail;
    fi;

    z:= LinearCombination( B, v{ [   1 ..   n ] } );
    h:= LinearCombination( B, v{ [ n+1 .. 2*n ] } );

    R:= LieCentralizer( L, SubalgebraNC( L, [ x ] ) );
    BR:= Basis( R );
    Rvecs:= BasisVectors( BR );

    # `ad h' maps `R' into `R'. `H' will be the matrix of that map.

    H:= List( Rvecs, v -> Coefficients( BR, h * v ) );

    # By the proof of the lemma of Jacobson-Morozov (see Jacobson,
    # Lie Algebras, p. 98) there is an element `e1' in `R' such that
    # `(H+2)e1=e0' where `e0=[h,z]+2z'.
    # If we set `y=z-e1' then `x,h,y' will span a subalgebra of `L'
    # isomorphic to sl_2.

    H:= H+2*IdentityMat( Dimension( R ), F );


    e0:= Coefficients( BR, h * z + 2*z );
    e1:= SolutionMat( H, e0 );

    if e1 = fail then
      # There is no sl_2 containing <x>.
      return fail;
    fi;

    y:= z-LinearCombination(Rvecs,e1);

    return [y,h,x];

end );



InstallMethod( PrintObj,
        "for nilpotent orbit",
        true,
       [ IsNilpotentOrbit ], 0,
       function( o )


     Print("<nilpotent orbit in Lie algebra");
     if HasSemiSimpleType(o) then
        Print( " of type ", SemiSimpleType(o) );
     fi;
     Print(">");

end );

InstallMethod( RandomSL2Triple,
   "for a nilpotent orbit",
   true, [ IsNilpotentOrbit], 0,
   function( o )

   local Ci, L, c, H, v, h, adh, sp, i, j, k, e, f, found, found_good, co,
         x, eqns, sol, allcft1;

   Ci:= FamilyObj( o )!.invCM;
   L:= AmbientLieAlgebra( o );
   c:= ChevalleyBasis(L);
   H:= c[3];

   v:= Ci*WeightedDynkinDiagram( o );
   h:= Sum( List( [1..Length(H)], x -> v[x]*H[x] ) );
   adh:= AdjointMatrix( Basis(L), h );
   sp:= NullspaceMat( TransposedMat( adh-2*adh^0 ) );
   e:= List( sp, x -> Sum( List( [1..Dimension(L)], ii -> x[ii]*Basis(L)[ii] ) ) );
   sp:= NullspaceMat( TransposedMat( adh+2*adh^0 ) );
   f:= List( sp, x -> Sum( List( [1..Dimension(L)], ii -> x[ii]*Basis(L)[ii] ) ) );

   allcft1:= false;
   
   while not allcft1 do
      found:= false;
      j:= 1;
      while not found do 
            co:= List( [1..Length(e)], x-> Random( Rationals ) );
            x:= Sum( List( [1..Length(e)], ii -> co[ii]*e[ii] ) );
            sp:= Subspace( L, List( f, y -> x*y ) );
            if Dimension(sp) = Length(e) and h in sp then
               found:= true; 
            else
               j:= j+1;
               if j > 100 then
                  Error("Tried hard but found no representative of the nilpotent orbit. Are you sure the weighted Dynkin diagram really corresponds to a nilpotent orbit?");
               fi;
            fi;
      od;

      for k in [Length(co),Length(co)-1..1] do
          i:= -1; found_good:= false;
          while not found_good do
                i:= i+1;
                co[k]:= i;
                x:= Sum( List( [1..Length(e)], ii -> co[ii]*e[ii] ) );
                sp:= Subspace( L, List( f, y -> x*y ) );
                if Dimension(sp) = Length(e) and h in sp then
                   found_good:= true; 
                fi;   
          od;
      od;
      allcft1:= ForAll( co, x -> (x=0) or (x=1) );

   od;
 
   eqns:= List( f, u -> Coefficients( Basis(sp), x*u ) );
   sol:= SolutionMat( eqns, Coefficients( Basis(sp), h ) ); 
   return [Sum([1..Length(f)], ii -> sol[ii]*f[ii] ),h,x]; 


end );

InstallMethod( SL2Triple,
   "for a nilpotent orbit",
   true, [ IsNilpotentOrbit], 0,
   function( o )


   return RandomSL2Triple( o );

end );



SLAfcts.weighted_dyn_diags_Dn:= function( n )


    local part, good, p, q, k, new, cur, m, diags, is_very_even, d, 
          d0, h, a, b, diag;

    part:= Partitions( 2*n );
    new:= [ ];
    for p in part do
        q:= [ ];
        cur:= p[1];
        k:= 1;
        m:= 0;
        while k <= Length(p) do
            m:= m+1;
            if k < Length(p) and p[k+1] <> cur then
               Add( q, [ cur, m ] );
               cur:= p[k+1];
               m:= 0;
            fi;
            k:= k+1;
        od;
        Add( q, [ cur,m ] );
        good:= true;
        for k in [1..Length(q)] do           
            if IsEvenInt(q[k][1]) and not IsEvenInt(q[k][2]) then
               good:= false;
               break;
            fi;
        od;
        if good then
           Add( new, p );
        fi;
    od;
    part:= new;

    diags:= [ ];
    new:= [ ];
    for p in part do
        is_very_even:= ForAll( p, x -> IsEvenInt(x) );
        d:= [ ];
        for k in p do
            d0:= List( [1..k], i -> k -2*(i-1)-1 );
            Append( d, d0 );
        od;
        Sort(d);
        d:= Reversed( d );
        h:= d{[1..n]};
        if not is_very_even then
           diag:= List( [1..n-1], i -> h[i]-h[i+1] );
           Add( diag, h[n-1]+h[n] );
           Add( diags, diag );
           Add( new, p );
        else
           if IsInt(n/4) then a:= 0; else a:= 2; fi;
           b:= 2-a;
           diag:= List( [1..n-2], i -> h[i]-h[i+1] );
           Add( diag, a ); Add( diag, b ); 
           Add( diags, diag );
           Add( new, p );
           diag:= List( [1..n-2], i -> h[i]-h[i+1] );
           Add( diag, b ); Add( diag, a ); 
           Add( diags, diag );
           Add( new, p );
        fi;
    od;

    return [diags,new];


end;


SLAfcts.weighted_dyn_diags_An:= function( n )


    local part, diags, p, d, k, d0, h;

    part:= Partitions( n+1 );
    diags:= [ ];
    for p in part do
        d:= [ ];
        for k in p do
            d0:= List( [1..k], i -> k -2*(i-1)-1 );
            Append( d, d0 );
        od;
        Sort(d);
        h:= Reversed( d );
        Add( diags, List( [1..n], i -> h[i]-h[i+1] ) );
    od;

    return [diags,part];


end;

SLAfcts.weighted_dyn_diags_Bn:= function( n )


    local part, good, p, q, k, new, cur, m, diags, is_very_even, d, 
          d0, h, a, b, diag;

    part:= Partitions( 2*n+1 );
    new:= [ ];
    for p in part do
        q:= [ ];
        cur:= p[1];
        k:= 1;
        m:= 0;
        while k <= Length(p) do
            m:= m+1;
            if k < Length(p) and p[k+1] <> cur then
               Add( q, [ cur, m ] );
               cur:= p[k+1];
               m:= 0;
            fi;
            k:= k+1;
        od;
        Add( q, [ cur,m ] );
        good:= true;
        for k in [1..Length(q)] do           
            if IsEvenInt(q[k][1]) and not IsEvenInt(q[k][2]) then
               good:= false;
               break;
            fi;
        od;
        if good then
           Add( new, p );
        fi;
    od;
    part:= new;

    diags:= [ ];
    for p in part do
        d:= [ ];
        for k in p do
            d0:= List( [1..k], i -> k -2*(i-1)-1 );
            Append( d, d0 );
        od;
        # Get rid of a zero;
        for k in [1..Length(d)] do
            if d[k] = 0 then
               RemoveElmList( d, k );
               break;
            fi;
        od;
        Sort(d);
        d:= Reversed( d );
        h:= d{[1..n]};
        diag:= List( [1..n-1], i -> h[i]-h[i+1] );
        Add( diag, h[n] );
        Add( diags, diag );
    od;

    return [diags,part];


end;


SLAfcts.weighted_dyn_diags_Cn:= function( n )


    local part, good, p, q, k, new, cur, m, diags, is_very_even, d, 
          d0, h, a, b, diag;

    part:= Partitions( 2*n );
    new:= [ ];
    for p in part do
        q:= [ ];
        cur:= p[1];
        k:= 1;
        m:= 0;
        while k <= Length(p) do
            m:= m+1;
            if k < Length(p) and p[k+1] <> cur then
               Add( q, [ cur, m ] );
               cur:= p[k+1];
               m:= 0;
            fi;
            k:= k+1;
        od;
        Add( q, [ cur,m ] );
        good:= true;
        for k in [1..Length(q)] do           
            if IsOddInt(q[k][1]) and not IsEvenInt(q[k][2]) then
               good:= false;
               break;
            fi;
        od;
        if good then
           Add( new, p );
        fi;
    od;
    part:= new;

    diags:= [ ];
    for p in part do
        d:= [ ];
        for k in p do
            d0:= List( [1..k], i -> k -2*(i-1)-1 );
            Append( d, d0 );
        od;
        Sort(d);
        d:= Reversed( d );
        h:= d{[1..n]};
        diag:= List( [1..n-1], i -> h[i]-h[i+1] );
        Add( diag, 2*h[n] );
        Add( diags, diag );
    od;

    return [diags,part];

end;



SLAfcts.nilporbs:=  function( L )

      
        # This for simple types only (at least for the moment).
        # REMARK: nonzero nilpotent orbits!
        # This is only for the standard simple Lie alg!

      local type, list, elms, orbs, i, j, o, u, a, sl2;

      type:= SemiSimpleType( L );
      if type = "G2" then

         list:= [ [1,0], [0,1], [0,2], [2,2] ]; 
         elms:=
[ [ [ 10, 1 ], [ 13, 2, 14, 3 ], [ 4, 1 ] ],
  [ [ 12, 1 ], [ 13, 1, 14, 2 ], [ 6, 1 ] ], 
  [ [ 8, 1, 10, 1 ], [ 13, 2, 14, 4 ], [ 2, 1, 4, 1 ] ], 
  [ [ 7, 6, 8, 10 ], [ 13, 6, 14, 10 ], [ 1, 1, 2, 1 ] ] ];

      elif type = "F4" then

         list:= [ [1,0,0,0], [0,0,0,1], [0,1,0,0], [2,0,0,0], [0,0,0,2],
                  [0,0,1,0], [2,0,0,1], [0,1,0,1], [1,0,1,0], [0,2,0,0], [2,2,0,0],
                  [1,0,1,2], [0,2,0,2], [2,2,0,2], [2,2,2,2] ];
         # need to permute, to account for slightly strange numbering of the Dynkin
         # diagram in GAP:
         list:= List( list, x -> Permuted( x, (1,2,4) ) );
         elms:=
[ [ [ 48, 1 ], [ 49, 1, 50, 2, 51, 2, 52, 3 ], [ 24, 1 ] ], 
  [ [ 45, 1 ], [ 49, 2, 50, 2, 51, 3, 52, 4 ], [ 21, 1 ] ], 
  [ [ 41, 1, 46, 1 ], [ 49, 2, 50, 3, 51, 4, 52, 6 ], [ 17, 1, 22, 1 ] ], 
  [ [ 40, 2, 42, 2 ], [ 49, 2, 50, 4, 51, 4, 52, 6 ], [ 16, 1, 18, 1 ] ], 
  [ [ 35, 2, 36, 2 ], [ 49, 4, 50, 4, 51, 6, 52, 8 ], [ 11, 1, 12, 1 ] ], 
  [ [ 38, 1, 39, 2, 40, 2 ], [ 49, 3, 50, 4, 51, 6, 52, 8 ], 
      [ 14, 1, 15, 1, 16, 1 ] ], 
  [ [ 33, 3, 39, 4 ], [ 49, 4, 50, 6, 51, 7, 52, 10 ], [ 9, 1, 15, 1 ] ], 
  [ [ 32, 2, 38, 2, 40, 1 ], [ 49, 4, 50, 5, 51, 7, 52, 10 ], 
      [ 8, 1, 14, 1, 16, 1 ] ], 
  [ [ 34, 4, 35, 3, 39, 1 ], [ 49, 4, 50, 6, 51, 8, 52, 11 ], 
      [ 10, 1, 11, 1, 15, 1 ] ], 
  [ [ 32, 2, 33, 2, 34, 2, 42, 2 ], [ 49, 4, 50, 6, 51, 8, 52, 12 ], 
      [ 8, 1, 9, 1, 10, 1, 18, 1 ] ], 
  [ [ 26, 10, 32, 6, 34, 6 ], [ 49, 6, 50, 10, 51, 12, 52, 18 ], 
      [ 2, 1, 8, 1, 10, 1 ] ], 
  [ [ 25, 8, 33, 5, 34, 9 ], [ 49, 8, 50, 10, 51, 14, 52, 19 ], 
      [ 1, 1, 9, 1, 10, 1 ] ], 
  [ [ 29, 8, 30, 9, 31, 5, 37, 1 ], [ 49, 8, 50, 10, 51, 14, 52, 20 ], 
      [ 5, 1, 6, 1, 7, 1, 13, 1 ] ], 
  [ [ 26, 14, 28, 18, 29, 10, 34, 8 ], [ 49, 10, 50, 14, 51, 18, 52, 26 ], 
      [ 2, 1, 4, 1, 5, 1, 10, 1 ] ], 
  [ [ 25, 16, 26, 22, 27, 30, 28, 42 ], [ 49, 16, 50, 22, 51, 30, 52, 42 ], 
      [ 1, 1, 2, 1, 3, 1, 4, 1 ] ] ];
  
      elif type = "E6" then

         list:= [ [0,1,0,0,0,0], [1,0,0,0,0,1], [0,0,0,1,0,0], 
                  [0,2,0,0,0,0], [1,1,0,0,0,1], [2,0,0,0,0,2], [0,0,1,0,1,0],
                  [1,2,0,0,0,1], [1,0,0,1,0,1], [0,1,1,0,1,0], [0,0,0,2,0,0], 
                  [2,2,0,0,0,2], [0,2,0,2,0,0], [1,1,1,0,1,1], [2,1,1,0,1,2], 
                  [1,2,1,0,1,1], [2,0,0,2,0,2], [2,2,0,2,0,2], [2,2,2,0,2,2], 
                  [2,2,2,2,2,2] ];
         elms:=
[ [ [ 72, 1 ], [ 73, 1, 74, 2, 75, 2, 76, 3, 77, 2, 78, 1 ], [ 36, 1 ] ], 
  [ [ 66, 1, 70, 1 ], [ 73, 2, 74, 2, 75, 3, 76, 4, 77, 3, 78, 2 ], 
      [ 30, 1, 34, 1 ] ], 
  [ [ 60, 1, 68, 1, 69, 1 ], [ 73, 2, 74, 3, 75, 4, 76, 6, 77, 4, 78, 2 ], 
      [ 24, 1, 32, 1, 33, 1 ] ], 
  [ [ 61, 2, 62, 2 ], [ 73, 2, 74, 4, 75, 4, 76, 6, 77, 4, 78, 2 ], 
      [ 25, 1, 26, 1 ] ], 
  [ [ 59, 1, 61, 2, 62, 2 ], [ 73, 3, 74, 4, 75, 5, 76, 7, 77, 5, 78, 3 ], 
      [ 23, 1, 25, 1, 26, 1 ] ], 
  [ [ 53, 2, 54, 2, 56, 2, 57, 2 ], [ 73, 4, 74, 4, 75, 6, 76, 8, 77, 6, 78, 
          4 ], [ 17, 1, 18, 1, 20, 1, 21, 1 ] ], 
  [ [ 58, 1, 59, 2, 60, 2, 61, 1 ], 
      [ 73, 3, 74, 4, 75, 6, 76, 8, 77, 6, 78, 3 ], 
      [ 22, 1, 23, 1, 24, 1, 25, 1 ] ], 
  [ [ 49, 3, 50, 3, 59, 4 ], [ 73, 4, 74, 6, 75, 7, 76, 10, 77, 7, 78, 4 ], 
      [ 13, 1, 14, 1, 23, 1 ] ], 
  [ [ 48, 2, 56, 2, 57, 2, 58, 2, 60, 1 ], [ 73, 4, 74, 5, 75, 7, 76, 10, 77, 
          7, 78, 4 ], [ 12, 1, 20, 1, 21, 1, 22, 1, 24, 1 ] ], 
  [ [ 51, 4, 53, 3, 56, 3, 59, 1 ], 
      [ 73, 4, 74, 6, 75, 8, 76, 11, 77, 8, 78, 4 ], 
      [ 15, 1, 17, 1, 20, 1, 23, 1 ] ], 
  [ [ 45, 1, 52, 4, 53, 3, 54, 1, 55, 3 ], 
      [ 73, 4, 74, 6, 75, 8, 76, 12, 77, 8, 78, 4 ], 
      [ 9, 1, 16, 1, 17, 1, 18, 1, 19, 1 ] ], 
  [ [ 47, 6, 48, 6, 49, 4, 50, 4 ], 
      [ 73, 6, 74, 8, 75, 10, 76, 14, 77, 10, 78, 6 ], 
      [ 11, 1, 12, 1, 13, 1, 14, 1 ] ], 
  [ [ 38, 10, 48, 6, 51, 6, 52, 6 ], [ 73, 6, 74, 10, 75, 12, 76, 18, 77, 12, 
          78, 6 ], [ 2, 1, 12, 1, 15, 1, 16, 1 ] ], 
  [ [ 47, 6, 48, 6, 49, 4, 50, 4, 51, 1 ], 
      [ 73, 6, 74, 8, 75, 11, 76, 15, 77, 11, 78, 6 ], 
      [ 11, 1, 12, 1, 13, 1, 14, 1, 15, 1 ] ], 
  [ [ 37, 8, 42, 8, 49, 5, 50, 5, 51, 9 ], 
      [ 73, 8, 74, 10, 75, 14, 76, 19, 77, 14, 78, 8 ], 
      [ 1, 1, 6, 1, 13, 1, 14, 1, 15, 1 ] ], 
  [ [ 38, -5, 43, 5, 44, 10, 47, 7, 48, 2, 51, 6, 52, 2 ], 
      [ 73, 7, 74, 10, 75, 13, 76, 18, 77, 13, 78, 7 ], 
      [ 7, 1, 8, 1, 11, 1, 12, 1, 15, 1 ] ], 
  [ [ 37, 8, 44, 5, 45, 9, 46, 1, 47, 8, 55, 5 ], 
      [ 73, 8, 74, 10, 75, 14, 76, 20, 77, 14, 78, 8 ], 
      [ 1, 1, 8, 1, 9, 1, 10, 1, 11, 1, 19, 1 ] ], 
  [ [ 38, 14, 42, 10, 43, 10, 45, 8, 46, 18 ], 
      [ 73, 10, 74, 14, 75, 18, 76, 26, 77, 18, 78, 10 ], 
      [ 2, 1, 6, 1, 7, 1, 9, 1, 10, 1 ] ], 
  [ [ 37, 12, 38, 8, 39, -8, 41, 22, 42, 12, 44, 8, 45, 22, 46, 8 ], 
      [ 73, 12, 74, 16, 75, 22, 76, 30, 77, 22, 78, 12 ], 
      [ 1, 1, 2, 1, 5, 1, 6, 1, 8, 1, 9, 1 ] ], 
  [ [ 37, 16, 38, 22, 39, 30, 40, 42, 41, 30, 42, 16 ], 
      [ 73, 16, 74, 22, 75, 30, 76, 42, 77, 30, 78, 16 ], 
      [ 1, 1, 2, 1, 3, 1, 4, 1, 5, 1, 6, 1 ] ] ];

      elif type = "E7" then

         list:= [ [1,0,0,0,0,0,0], [0,0,0,0,0,1,0], [0,0,0,0,0,0,2], [0,0,1,0,0,0,0],
                  [2,0,0,0,0,0,0], [0,1,0,0,0,0,1], [1,0,0,0,0,1,0], [0,0,0,1,0,0,0], 
                  [2,0,0,0,0,1,0], [0,0,0,0,0,2,0], [0,2,0,0,0,0,0], [2,0,0,0,0,0,2],
                  [0,0,1,0,0,1,0], [1,0,0,1,0,0,0], [0,0,2,0,0,0,0], [1,0,0,0,1,0,1],
                  [2,0,2,0,0,0,0], [0,1,1,0,0,0,1], [0,0,0,1,0,1,0], [2,0,0,0,0,2,0],
                  [0,0,0,0,2,0,0], [2,0,0,0,0,2,2], [2,1,1,0,0,0,1], [1,0,0,1,0,1,0],
                  [2,0,0,1,0,1,0], [0,0,0,2,0,0,0], [1,0,0,1,0,2,0], [1,0,0,1,0,1,2],
                  [2,0,0,0,2,0,0], [0,1,1,0,1,0,2], [0,0,2,0,0,2,0], [2,0,2,0,0,2,0], 
                  [0,0,0,2,0,0,2], [0,0,0,2,0,2,0], [2,1,1,0,1,1,0], [2,1,1,0,1,0,2],
                  [2,0,0,2,0,0,2], [2,1,1,0,1,2,2], [2,0,0,2,0,2,0], [2,0,2,2,0,2,0],
                  [2,0,0,2,0,2,2], [2,2,2,0,2,0,2], [2,2,2,0,2,2,2], [2,2,2,2,2,2,2] ];

      elms:=
[ [ [ 126, 1 ], [ 127, 2, 128, 2, 129, 3, 130, 4, 131, 3, 132, 2, 133, 1 ], 
      [ 63, 1 ] ], 
  [ [ 120, 1, 123, 1 ], [ 127, 2, 128, 3, 129, 4, 130, 6, 131, 5, 132, 4, 
          133, 2 ], [ 57, 1, 60, 1 ] ], 
  [ [ 110, 1, 111, 1, 112, 1 ], [ 127, 2, 128, 3, 129, 4, 130, 6, 131, 5, 
          132, 4, 133, 3 ], [ 47, 1, 48, 1, 49, 1 ] ], 
  [ [ 105, 1, 119, 1, 122, 1 ], 
      [ 127, 3, 128, 4, 129, 6, 130, 8, 131, 6, 132, 4, 133, 2 ], 
      [ 42, 1, 56, 1, 59, 1 ] ], 
  [ [ 107, 2, 109, 2 ], [ 127, 4, 128, 4, 129, 6, 130, 8, 131, 6, 132, 4, 
          133, 2 ], [ 44, 1, 46, 1 ] ], 
  [ [ 108, 1, 110, 1, 115, 1, 116, 1 ], [ 127, 3, 128, 5, 129, 6, 130, 9, 
          131, 7, 132, 5, 133, 3 ], [ 45, 1, 47, 1, 52, 1, 53, 1 ] ], 
  [ [ 107, 2, 109, 2, 112, 1 ], 
      [ 127, 4, 128, 5, 129, 7, 130, 10, 131, 8, 132, 6, 133, 3 ], 
      [ 44, 1, 46, 1, 49, 1 ] ], 
  [ [ 104, 2, 105, 1, 106, 2, 114, 1 ], [ 127, 4, 128, 6, 129, 8, 130, 12, 
          131, 9, 132, 6, 133, 3 ], [ 41, 1, 42, 1, 43, 1, 51, 1 ] ], 
  [ [ 83, 3, 84, 3, 112, 4 ], 
      [ 127, 6, 128, 7, 129, 10, 130, 14, 131, 11, 132, 8, 133, 4 ], 
      [ 20, 1, 21, 1, 49, 1 ] ], 
  [ [ 97, 2, 98, 2, 99, 2, 106, 2 ], [ 127, 4, 128, 6, 129, 8, 130, 12, 131, 
          10, 132, 8, 133, 4 ], [ 34, 1, 35, 1, 36, 1, 43, 1 ] ], 
  [ [ 100, 1, 101, 1, 102, 2, 103, 2, 104, 1 ], 
      [ 127, 4, 128, 7, 129, 8, 130, 12, 131, 9, 132, 6, 133, 3 ], 
      [ 37, 1, 38, 1, 39, 1, 40, 1, 41, 1 ] ], 
  [ [ 90, 3, 93, 4, 94, 1, 100, 3 ], 
      [ 127, 6, 128, 7, 129, 10, 130, 14, 131, 11, 132, 8, 133, 5 ], 
      [ 27, 1, 30, 1, 31, 1, 37, 1 ] ], 
  [ [ 97, 2, 98, 2, 99, 2, 100, 1, 106, 2 ], 
      [ 127, 5, 128, 7, 129, 10, 130, 14, 131, 11, 132, 8, 133, 4 ], 
      [ 34, 1, 35, 1, 36, 1, 37, 1, 43, 1 ] ], 
  [ [ 90, 3, 91, 4, 102, 3, 112, 1 ], 
      [ 127, 6, 128, 8, 129, 11, 130, 16, 131, 12, 132, 8, 133, 4 ], 
      [ 27, 1, 28, 1, 39, 1, 49, 1 ] ], 
  [ [ 90, 1, 92, 3, 94, 3, 95, 4, 102, 1 ], 
      [ 127, 6, 128, 8, 129, 12, 130, 16, 131, 12, 132, 8, 133, 4 ], 
      [ 27, 1, 29, 1, 31, 1, 32, 1, 39, 1 ] ], 
  [ [ 90, 3, 93, 4, 94, 1, 100, 3, 103, 1 ], 
      [ 127, 6, 128, 8, 129, 11, 130, 16, 131, 13, 132, 9, 133, 5 ], 
      [ 27, 1, 30, 1, 31, 1, 37, 1, 40, 1 ] ], 
  [ [ 64, 10, 91, 6, 92, 6, 94, 6 ], 
      [ 127, 10, 128, 12, 129, 18, 130, 24, 131, 18, 132, 12, 133, 6 ], 
      [ 1, 1, 28, 1, 29, 1, 31, 1 ] ], 
  [ [ 85, 1, 93, 1, 94, 4, 95, 3, 96, 3, 98, 1 ], 
      [ 127, 6, 128, 9, 129, 12, 130, 17, 131, 13, 132, 9, 133, 5 ], 
      [ 22, 1, 30, 1, 31, 1, 32, 1, 33, 1, 35, 1 ] ], 
  [ [ 90, 2, 92, 3, 93, 2, 94, 3, 95, 4 ], 
      [ 127, 6, 128, 9, 129, 12, 130, 18, 131, 14, 132, 10, 133, 5 ], 
      [ 27, 1, 29, 1, 30, 1, 31, 1, 32, 1 ] ], 
  [ [ 83, 4, 84, 4, 88, 6, 92, 6 ], 
      [ 127, 8, 128, 10, 129, 14, 130, 20, 131, 16, 132, 12, 133, 6 ], 
      [ 20, 1, 21, 1, 25, 1, 29, 1 ] ], 
  [ [ 88, 4, 89, 3, 90, 2, 91, 2, 92, 3, 110, 1 ], 
      [ 127, 6, 128, 9, 129, 12, 130, 18, 131, 15, 132, 10, 133, 5 ], 
      [ 25, 1, 26, 1, 27, 1, 28, 1, 29, 1, 47, 1 ] ], 
  [ [ 70, 9, 83, 5, 84, 5, 86, 8, 87, 8 ], 
      [ 127, 10, 128, 13, 129, 18, 130, 26, 131, 21, 132, 16, 133, 9 ], 
      [ 7, 1, 20, 1, 21, 1, 23, 1, 24, 1 ] ], 
  [ [ 64, 10, 91, 6, 92, 6, 93, 1, 94, 6 ], 
      [ 127, 10, 128, 13, 129, 18, 130, 25, 131, 19, 132, 13, 133, 7 ], 
      [ 1, 1, 28, 1, 29, 1, 30, 1, 31, 1 ] ], 
  [ [ 83, 4, 84, 4, 88, 6, 91, 1, 92, 6 ], 
      [ 127, 8, 128, 11, 129, 15, 130, 22, 131, 17, 132, 12, 133, 6 ], 
      [ 20, 1, 21, 1, 25, 1, 28, 1, 29, 1 ] ], 
  [ [ 71, 10, 81, 6, 91, 6, 92, 1, 93, 6, 94, 1 ], 
      [ 127, 10, 128, 13, 129, 18, 130, 26, 131, 20, 132, 14, 133, 7 ], 
      [ 8, 1, 18, 1, 28, 1, 29, 1, 30, 1, 31, 1 ] ], 
  [ [ 83, 6, 84, 2, 85, 4, 86, 2, 87, 4, 88, 6 ], 
      [ 127, 8, 128, 12, 129, 16, 130, 24, 131, 18, 132, 12, 133, 6 ], 
      [ 20, 1, 21, 1, 22, 1, 23, 1, 24, 1, 25, 1 ] ], 
  [ [ 75, 8, 76, 8, 83, 5, 84, 5, 91, 9 ], 
      [ 127, 10, 128, 14, 129, 19, 130, 28, 131, 22, 132, 16, 133, 8 ], 
      [ 12, 1, 13, 1, 20, 1, 21, 1, 28, 1 ] ], 
  [ [ 70, 9, 81, 8, 83, 5, 84, 5, 91, 1, 92, 8 ], 
      [ 127, 10, 128, 14, 129, 19, 130, 28, 131, 22, 132, 16, 133, 9 ], 
      [ 7, 1, 18, 1, 20, 1, 21, 1, 28, 1, 29, 1 ] ], 
  [ [ 71, 5, 80, 7, 81, 6, 82, 5, 83, 10, 91, -2, 92, 1, 93, 2 ], 
      [ 127, 10, 128, 13, 129, 18, 130, 26, 131, 21, 132, 14, 133, 7 ], 
      [ 17, 1, 18, 1, 19, 1, 20, 1, 29, 1, 30, 1 ] ], 
  [ [ 70, 6, 76, 9, 78, 6, 79, 6, 83, 3, 84, 10, 86, 3, 87, 4, 90, -3 ], 
      [ 127, 10, 128, 15, 129, 20, 130, 29, 131, 23, 132, 16, 133, 9 ], 
      [ 13, 1, 15, 1, 16, 1, 21, 1, 23, 1, 24, 1 ] ], 
  [ [ 77, 9, 78, 5, 80, 5, 82, 8, 86, 8, 89, 1 ], 
      [ 127, 10, 128, 14, 129, 20, 130, 28, 131, 22, 132, 16, 133, 8 ], 
      [ 14, 1, 15, 1, 17, 1, 19, 1, 23, 1, 26, 1 ] ], 
  [ [ 64, 14, 78, 18, 80, 8, 81, 10, 82, 10 ], 
      [ 127, 14, 128, 18, 129, 26, 130, 36, 131, 28, 132, 20, 133, 10 ], 
      [ 1, 1, 15, 1, 17, 1, 18, 1, 19, 1 ] ], 
  [ [ 76, 6, 77, 6, 78, 10, 79, 4, 81, 6, 82, 9, 84, 3, 85, -3, 87, 3, 96, 1 ]
        , [ 127, 10, 128, 15, 129, 20, 130, 30, 131, 23, 132, 16, 133, 9 ], 
      [ 14, 1, 15, 1, 16, 1, 18, 1, 19, 1, 21, 1, 33, 1 ] ], 
  [ [ 75, 10, 76, 10, 77, 12, 78, 6, 79, 12, 80, 6 ], 
      [ 127, 12, 128, 18, 129, 24, 130, 36, 131, 28, 132, 20, 133, 10 ], 
      [ 12, 1, 13, 1, 14, 1, 15, 1, 16, 1, 17, 1 ] ], 
  [ [ 64, 14, 78, 18, 79, 1, 80, 8, 81, 10, 82, 10 ], 
      [ 127, 14, 128, 19, 129, 26, 130, 37, 131, 29, 132, 20, 133, 10 ], 
      [ 1, 1, 15, 1, 16, 1, 17, 1, 18, 1, 19, 1 ] ], 
  [ [ 64, 14, 70, 2, 76, 9, 78, 8, 79, 2, 80, 18, 86, 11, 87, 9 ], 
      [ 127, 14, 128, 19, 129, 26, 130, 37, 131, 29, 132, 20, 133, 11 ], 
      [ 1, 1, 7, 1, 13, 1, 15, 1, 17, 1, 23, 1 ] ], 
  [ [ 67, -9, 71, 14, 74, 18, 76, 11, 78, 9, 82, 2, 85, 2, 86, 8, 87, 1 ], 
      [ 127, 14, 128, 19, 129, 26, 130, 38, 131, 29, 132, 20, 133, 11 ], 
      [ 8, 1, 11, 1, 13, 1, 15, 1, 22, 1, 23, 1, 24, 1 ] ], 
  [ [ 64, 18, 69, 28, 70, 15, 78, 10, 79, 15, 80, 24 ], 
      [ 127, 18, 128, 25, 129, 34, 130, 49, 131, 39, 132, 28, 133, 15 ], 
      [ 1, 1, 6, 1, 7, 1, 15, 1, 16, 1, 17, 1 ] ], 
  [ [ 67, 15, 71, 16, 75, 12, 76, 12, 78, 7, 79, 15, 80, 7 ], 
      [ 127, 16, 128, 22, 129, 30, 130, 44, 131, 34, 132, 24, 133, 12 ], 
      [ 4, 1, 8, 1, 12, 1, 13, 1, 15, 1, 16, 1, 17, 1 ] ], 
  [ [ 64, 22, 66, 42, 72, 30, 74, 30, 75, 16, 76, 16 ], 
      [ 127, 22, 128, 30, 129, 42, 130, 60, 131, 46, 132, 32, 133, 16 ], 
      [ 1, 1, 3, 1, 9, 1, 11, 1, 12, 1, 13, 1 ] ], 
  [ [ 70, 15, 71, 18, 72, 24, 73, 15, 74, 10, 75, 28, 85, 1 ], 
      [ 127, 18, 128, 25, 129, 34, 130, 50, 131, 39, 132, 28, 133, 15 ], 
      [ 7, 1, 8, 1, 9, 1, 10, 1, 11, 1, 12, 1, 22, 1 ] ], 
  [ [ 64, 22, 65, 28, 66, -28, 70, 17, 72, 3, 73, 42, 74, 15, 75, 32, 76, 3, 
          81, 3 ], 
      [ 127, 22, 128, 31, 129, 42, 130, 60, 131, 47, 132, 32, 133, 17 ], 
      [ 1, 1, 2, 1, 7, 1, 9, 1, 10, 1, 11, 1, 12, 1 ] ], 
  [ [ 64, 26, 65, 15, 66, -15, 68, 57, 69, 40, 70, 21, 72, 22, 73, 50, 74, 22 
         ], [ 127, 26, 128, 37, 129, 50, 130, 72, 131, 57, 132, 40, 133, 21 ],
      [ 1, 1, 2, 1, 5, 1, 6, 1, 7, 1, 9, 1, 10, 1 ] ], 
  [ [ 64, 34, 65, 49, 66, 66, 67, 96, 68, 75, 69, 52, 70, 27 ], 
      [ 127, 34, 128, 49, 129, 66, 130, 96, 131, 75, 132, 52, 133, 27 ], 
      [ 1, 1, 2, 1, 3, 1, 4, 1, 5, 1, 6, 1, 7, 1 ] ] ];

      elif type = "E8" then

         list:= [ [0,0,0,0,0,0,0,1], [1,0,0,0,0,0,0,0], [0,0,0,0,0,0,1,0], [0,0,0,0,0,0,0,2],
                  [0,1,0,0,0,0,0,0], [1,0,0,0,0,0,0,1], [0,0,0,0,0,1,0,0], [1,0,0,0,0,0,0,2],
                  [0,0,1,0,0,0,0,0], [2,0,0,0,0,0,0,0], [1,0,0,0,0,0,1,0], [0,0,0,0,0,1,0,1],
                  [0,0,0,0,0,0,2,0], [0,0,0,0,0,0,2,2], [0,0,0,0,1,0,0,0], [0,0,1,0,0,0,0,1],
                  [0,1,0,0,0,0,1,0], [1,0,0,0,0,1,0,0], [2,0,0,0,0,0,0,2], [0,0,0,1,0,0,0,0], 
                  [0,1,0,0,0,0,1,2], [0,2,0,0,0,0,0,0], [1,0,0,0,0,1,0,1], [1,0,0,0,1,0,0,0],
                  [1,0,0,0,0,1,0,2], [0,0,0,1,0,0,0,1], [0,0,0,0,0,2,0,0], [2,0,0,0,0,1,0,1],
                  [0,0,0,1,0,0,0,2], [0,0,1,0,0,1,0,0], [0,2,0,0,0,0,0,2], [2,0,0,0,0,0,2,0],
                  [2,0,0,0,0,0,2,2], [0,0,0,1,0,0,1,0], [1,0,0,1,0,0,0,1], [0,0,1,0,0,1,0,1],
                  [0,1,1,0,0,0,1,0], [1,0,0,0,1,0,1,0], [0,0,0,1,0,1,0,0], [1,0,0,0,1,0,1,2],
                  [0,0,0,0,2,0,0,0], [2,0,0,0,0,2,0,0], [0,1,1,0,0,0,1,2], [1,0,0,1,0,1,0,0],
                  [0,0,0,1,0,1,0,2], [2,0,0,0,0,2,0,2], [0,0,0,0,2,0,0,2], [2,1,1,0,0,0,1,2],
                  [2,0,0,0,0,2,2,2], [1,0,0,1,0,1,0,1], [1,0,0,1,0,1,1,0], [1,0,0,1,0,1,0,2],
                  [2,0,0,1,0,1,0,2], [0,0,0,2,0,0,0,2], [2,0,0,0,2,0,0,2], [1,0,0,1,0,1,2,2], 
                  [0,1,1,0,1,0,2,2], [0,0,0,2,0,0,2,0], [2,1,1,0,1,1,0,1], [0,0,0,2,0,0,2,2],
                  [2,1,1,0,1,0,2,2], [2,0,0,2,0,0,2,0], [2,0,0,2,0,0,2,2], [2,1,1,0,1,2,2,2],
                  [2,0,0,2,0,2,0,2], [2,0,0,2,0,2,2,2], [2,2,2,0,2,0,2,2], [2,2,2,0,2,2,2,2],
                  [2,2,2,2,2,2,2,2] ];

         elms:= 
[ [ [ 240, 1 ], 
      [ 241, 2, 242, 3, 243, 4, 244, 6, 245, 5, 246, 4, 247, 3, 248, 2 ], 
      [ 120, 1 ] ], 
  [ [ 233, 1, 234, 1 ], [ 241, 4, 242, 5, 243, 7, 244, 10, 245, 8, 246, 6, 
          247, 4, 248, 2 ], [ 113, 1, 114, 1 ] ], 
  [ [ 224, 1, 225, 1, 226, 1 ], 
      [ 241, 4, 242, 6, 243, 8, 244, 12, 245, 10, 246, 8, 247, 6, 248, 3 ], 
      [ 104, 1, 105, 1, 106, 1 ] ], 
  [ [ 208, 2, 210, 2 ], [ 241, 4, 242, 6, 243, 8, 244, 12, 245, 10, 246, 8, 
          247, 6, 248, 4 ], [ 88, 1, 90, 1 ] ], 
  [ [ 215, 1, 217, 1, 218, 1, 223, 1 ], 
      [ 241, 5, 242, 8, 243, 10, 244, 15, 245, 12, 246, 9, 247, 6, 248, 3 ], 
      [ 95, 1, 97, 1, 98, 1, 103, 1 ] ], 
  [ [ 208, 2, 210, 2, 217, 1 ], [ 241, 6, 242, 8, 243, 11, 244, 16, 245, 13, 
          246, 10, 247, 7, 248, 4 ], [ 88, 1, 90, 1, 97, 1 ] ], 
  [ [ 203, 2, 205, 2, 211, 1, 212, 1 ], 
      [ 241, 6, 242, 9, 243, 12, 244, 18, 245, 15, 246, 12, 247, 8, 248, 4 ], 
      [ 83, 1, 85, 1, 91, 1, 92, 1 ] ], 
  [ [ 162, 3, 163, 3, 217, 4 ], [ 241, 8, 242, 11, 243, 15, 244, 22, 245, 18, 
          246, 14, 247, 10, 248, 6 ], [ 42, 1, 43, 1, 97, 1 ] ], 
  [ [ 197, 2, 200, 2, 202, 1, 204, 1, 210, 1 ], 
      [ 241, 7, 242, 10, 243, 14, 244, 20, 245, 16, 246, 12, 247, 8, 248, 4 ],
      [ 77, 1, 80, 1, 82, 1, 84, 1, 90, 1 ] ], 
  [ [ 190, 2, 191, 2, 193, 2, 201, 2 ], 
      [ 241, 8, 242, 10, 243, 14, 244, 20, 245, 16, 246, 12, 247, 8, 248, 4 ],
      [ 70, 1, 71, 1, 73, 1, 81, 1 ] ], 
  [ [ 190, 2, 192, 2, 193, 2, 194, 1, 200, 2 ], 
      [ 241, 8, 242, 11, 243, 15, 244, 22, 245, 18, 246, 14, 247, 10, 248, 5 ]
        , [ 70, 1, 72, 1, 73, 1, 74, 1, 80, 1 ] ], 
  [ [ 182, 3, 185, 3, 186, 4, 213, 1 ], 
      [ 241, 8, 242, 12, 243, 16, 244, 24, 245, 20, 246, 16, 247, 11, 248, 6 ]
        , [ 62, 1, 65, 1, 66, 1, 93, 1 ] ], 
  [ [ 180, 4, 181, 3, 182, 1, 184, 3, 198, 1 ], 
      [ 241, 8, 242, 12, 243, 16, 244, 24, 245, 20, 246, 16, 247, 12, 248, 6 ]
        , [ 60, 1, 61, 1, 62, 1, 64, 1, 78, 1 ] ], 
  [ [ 128, 10, 173, 6, 175, 6, 191, 6 ], 
      [ 241, 12, 242, 18, 243, 24, 244, 36, 245, 30, 246, 24, 247, 18, 248, 
          10 ], [ 8, 1, 53, 1, 55, 1, 71, 1 ] ], 
  [ [ 189, 2, 190, 2, 191, 1, 192, 1, 193, 2, 194, 2 ], 
      [ 241, 8, 242, 12, 243, 16, 244, 24, 245, 20, 246, 15, 247, 10, 248, 5 ]
        , [ 69, 1, 70, 1, 71, 1, 72, 1, 73, 1, 74, 1 ] ], 
  [ [ 178, 4, 187, 3, 188, 3, 189, 1, 202, 1 ], 
      [ 241, 9, 242, 13, 243, 18, 244, 26, 245, 21, 246, 16, 247, 11, 248, 6 ]
        , [ 58, 1, 67, 1, 68, 1, 69, 1, 82, 1 ] ], 
  [ [ 176, 4, 179, 3, 189, 1, 191, 3, 192, 1, 193, 1 ], 
      [ 241, 9, 242, 14, 243, 18, 244, 27, 245, 22, 246, 17, 247, 12, 248, 6 ]
        , [ 56, 1, 59, 1, 69, 1, 71, 1, 72, 1, 73, 1 ] ], 
  [ [ 165, 3, 177, 3, 178, 2, 187, 2, 194, 4 ], 
      [ 241, 10, 242, 14, 243, 19, 244, 28, 245, 23, 246, 18, 247, 12, 248, 6 
         ], [ 45, 1, 57, 1, 58, 1, 67, 1, 74, 1 ] ], 
  [ [ 162, 4, 163, 4, 173, 6, 177, 6 ], 
      [ 241, 12, 242, 16, 243, 22, 244, 32, 245, 26, 246, 20, 247, 14, 248, 8 
         ], [ 42, 1, 43, 1, 53, 1, 57, 1 ] ], 
  [ [ 172, 3, 178, 4, 180, 2, 181, 2, 182, 3, 198, 1 ], 
      [ 241, 10, 242, 15, 243, 20, 244, 30, 245, 24, 246, 18, 247, 12, 248, 6 
         ], [ 52, 1, 58, 1, 60, 1, 61, 1, 62, 1, 78, 1 ] ], 
  [ [ 128, 10, 175, 6, 178, 6, 186, 6, 189, 1 ], 
      [ 241, 13, 242, 20, 243, 26, 244, 39, 245, 32, 246, 25, 247, 18, 248, 
          10 ], [ 8, 1, 55, 1, 58, 1, 66, 1, 69, 1 ] ], 
  [ [ 165, 1, 174, 2, 176, 4, 177, 3, 178, 1, 179, 3, 181, 2 ], 
      [ 241, 10, 242, 16, 243, 20, 244, 30, 245, 24, 246, 18, 247, 12, 248, 6 
         ], [ 45, 1, 54, 1, 56, 1, 57, 1, 58, 1, 59, 1, 61, 1 ] ], 
  [ [ 162, 4, 163, 4, 173, 6, 177, 6, 181, 1 ], 
      [ 241, 12, 242, 17, 243, 23, 244, 34, 245, 28, 246, 22, 247, 15, 248, 8 
         ], [ 42, 1, 43, 1, 53, 1, 57, 1, 61, 1 ] ], 
  [ [ 164, 3, 165, 3, 166, 3, 167, 3, 175, 4, 188, 4 ], 
      [ 241, 12, 242, 17, 243, 23, 244, 34, 245, 28, 246, 21, 247, 14, 248, 7 
         ], [ 44, 1, 45, 1, 46, 1, 47, 1, 55, 1, 68, 1 ] ], 
  [ [ 135, 10, 165, 6, 177, 6, 178, 1, 179, 1, 181, 6 ], 
      [ 241, 14, 242, 20, 243, 27, 244, 40, 245, 33, 246, 26, 247, 18, 248, 
          10 ], [ 15, 1, 45, 1, 57, 1, 58, 1, 59, 1, 61, 1 ] ], 
  [ [ 167, 4, 169, 6, 170, 4, 171, 1, 172, 6, 184, 1 ], 
      [ 241, 12, 242, 18, 243, 24, 244, 36, 245, 29, 246, 22, 247, 15, 248, 8 
         ], [ 47, 1, 49, 1, 50, 1, 51, 1, 52, 1, 64, 1 ] ], 
  [ [ 165, 2, 166, 4, 167, 6, 168, 6, 169, 4, 170, 2 ], 
      [ 241, 12, 242, 18, 243, 24, 244, 36, 245, 30, 246, 24, 247, 16, 248, 8 
         ], [ 45, 1, 46, 1, 47, 1, 48, 1, 49, 1, 50, 1 ] ], 
  [ [ 143, 8, 144, 8, 162, 5, 163, 5, 181, 9 ], 
      [ 241, 16, 242, 22, 243, 30, 244, 44, 245, 36, 246, 28, 247, 19, 248, 
          10 ], [ 23, 1, 24, 1, 42, 1, 43, 1, 61, 1 ] ], 
  [ [ 142, 10, 149, 5, 160, -2, 164, 6, 168, 7, 173, 2, 179, 5, 191, 1 ], 
      [ 241, 14, 242, 21, 243, 28, 244, 42, 245, 34, 246, 26, 247, 18, 248, 
          10 ], [ 22, 1, 44, 1, 48, 1, 53, 1, 59, 1, 71, 1 ] ], 
  [ [ 164, 1, 165, 2, 166, 4, 167, 6, 168, 6, 169, 4, 170, 2 ], 
      [ 241, 13, 242, 19, 243, 26, 244, 38, 245, 31, 246, 24, 247, 16, 248, 8 
         ], [ 44, 1, 45, 1, 46, 1, 47, 1, 48, 1, 49, 1, 50, 1 ] ], 
  [ [ 154, 6, 157, 6, 158, 6, 160, 2, 163, 10, 184, 2 ], 
      [ 241, 14, 242, 22, 243, 28, 244, 42, 245, 34, 246, 26, 247, 18, 248, 
          10 ], [ 34, 1, 37, 1, 38, 1, 40, 1, 43, 1, 64, 1 ] ], 
  [ [ 154, 5, 155, 5, 156, 9, 158, 8, 164, 8, 170, 1 ], 
      [ 241, 16, 242, 22, 243, 30, 244, 44, 245, 36, 246, 28, 247, 20, 248, 
          10 ], [ 34, 1, 35, 1, 36, 1, 38, 1, 44, 1, 50, 1 ] ], 
  [ [ 128, 14, 141, 18, 157, 10, 169, 8, 171, 10 ], 
      [ 241, 20, 242, 28, 243, 38, 244, 56, 245, 46, 246, 36, 247, 26, 248, 
          14 ], [ 8, 1, 21, 1, 37, 1, 49, 1, 51, 1 ] ], 
  [ [ 159, 6, 161, 3, 162, 6, 163, 3, 164, 4, 165, 4, 168, 4 ], 
      [ 241, 14, 242, 21, 243, 28, 244, 42, 245, 34, 246, 26, 247, 18, 248, 9 
         ], [ 39, 1, 41, 1, 42, 1, 43, 1, 44, 1, 45, 1, 48, 1 ] ], 
  [ [ 151, 8, 152, 9, 156, 5, 166, 8, 170, 5, 181, 1 ], 
      [ 241, 16, 242, 23, 243, 31, 244, 46, 245, 37, 246, 28, 247, 19, 248, 
          10 ], [ 31, 1, 32, 1, 36, 1, 46, 1, 50, 1, 61, 1 ] ], 
  [ [ 142, 10, 147, -2, 158, 2, 159, 2, 162, -5, 164, 3, 168, 7, 169, 2, 179, 
          5 ], 
      [ 241, 15, 242, 22, 243, 30, 244, 44, 245, 36, 246, 28, 247, 19, 248, 
          10 ], [ 22, 1, 38, 1, 39, 1, 44, 2, 48, 1, 49, 1, 59, 1 ] ], 
  [ [ 145, 3, 154, 6, 157, 6, 158, 6, 159, 4, 160, 3, 163, 10 ], 
      [ 241, 16, 242, 24, 243, 32, 244, 47, 245, 38, 246, 29, 247, 20, 248, 
          10 ], [ 25, 1, 34, 1, 37, 1, 38, 1, 39, 1, 40, 1, 43, 1 ] ], 
  [ [ 148, 1, 151, 8, 157, 8, 161, 9, 162, 5, 163, 5, 168, 1 ], 
      [ 241, 16, 242, 23, 243, 31, 244, 46, 245, 38, 246, 29, 247, 20, 248, 
          10 ], [ 28, 1, 31, 1, 37, 1, 41, 1, 42, 1, 43, 1, 48, 1 ] ], 
  [ [ 140, 5, 157, 9, 158, 5, 159, 2, 161, 8, 162, 2, 163, 8 ], 
      [ 241, 16, 242, 24, 243, 32, 244, 48, 245, 39, 246, 30, 247, 20, 248, 
          10 ], [ 20, 1, 37, 1, 38, 1, 39, 1, 41, 1, 42, 1, 43, 1 ] ], 
  [ [ 128, 14, 154, 8, 155, 18, 157, 10, 158, 10, 168, 1 ], 
      [ 241, 20, 242, 29, 243, 39, 244, 58, 245, 48, 246, 37, 247, 26, 248, 
          14 ], [ 8, 1, 34, 1, 35, 1, 37, 1, 38, 1, 48, 1 ] ], 
  [ [ 125, 5, 158, 2, 159, 9, 160, 8, 161, 1, 162, 8, 163, 2, 164, 5 ], 
      [ 241, 16, 242, 24, 243, 32, 244, 48, 245, 40, 246, 30, 247, 20, 248, 
          10 ], [ 5, 1, 38, 1, 39, 1, 40, 1, 41, 1, 42, 1, 43, 1, 44, 1 ] ], 
  [ [ 143, 10, 144, 10, 153, 12, 154, 6, 155, 6, 156, 12 ], 
      [ 241, 20, 242, 28, 243, 38, 244, 56, 245, 46, 246, 36, 247, 24, 248, 
          12 ], [ 23, 1, 24, 1, 33, 1, 34, 1, 35, 1, 36, 1 ] ], 
  [ [ 128, 14, 145, 1, 154, 8, 155, 18, 157, 10, 158, 10, 160, 1 ], 
      [ 241, 20, 242, 30, 243, 40, 244, 59, 245, 48, 246, 37, 247, 26, 248, 
          14 ], [ 8, 1, 25, 1, 34, 1, 35, 1, 37, 1, 38, 1, 40, 1 ] ], 
  [ [ 143, 10, 144, 10, 152, 1, 153, 12, 154, 6, 155, 6, 156, 12 ], 
      [ 241, 20, 242, 29, 243, 39, 244, 58, 245, 47, 246, 36, 247, 24, 248, 
          12 ], [ 23, 1, 24, 1, 32, 1, 33, 1, 34, 1, 35, 1, 36, 1 ] ], 
  [ [ 128, 14, 140, 1, 152, 2, 153, -2, 154, 8, 155, 18, 157, 9, 158, 11, 
          159, -9 ], 
      [ 241, 20, 242, 30, 243, 40, 244, 60, 245, 49, 246, 38, 247, 26, 248, 
          14 ], [ 8, 1, 20, 1, 32, 1, 34, 1, 35, 1, 37, 1, 38, 1 ] ], 
  [ [ 121, 12, 128, 16, 140, 7, 153, 7, 154, 15, 155, 15, 164, 12 ], 
      [ 241, 24, 242, 34, 243, 46, 244, 68, 245, 56, 246, 44, 247, 30, 248, 
          16 ], [ 1, 1, 8, 1, 20, 1, 33, 1, 34, 1, 35, 1, 44, 1 ] ], 
  [ [ 128, 14, 146, 10, 147, 2, 150, 2, 152, 10, 159, 18, 161, 8 ], 
      [ 241, 20, 242, 30, 243, 40, 244, 60, 245, 50, 246, 38, 247, 26, 248, 
          14 ], [ 8, 1, 26, 1, 27, 1, 30, 1, 32, 1, 39, 1, 41, 1 ] ], 
  [ [ 121, 28, 128, 18, 152, 15, 153, 15, 154, 10, 155, 24 ], 
      [ 241, 28, 242, 40, 243, 54, 244, 79, 245, 64, 246, 49, 247, 34, 248, 
          18 ], [ 1, 1, 8, 1, 32, 1, 33, 1, 34, 1, 35, 1 ] ], 
  [ [ 121, 16, 127, 42, 128, 22, 146, 30, 147, 30, 164, 16 ], 
      [ 241, 32, 242, 46, 243, 62, 244, 92, 245, 76, 246, 60, 247, 42, 248, 
          22 ], [ 1, 1, 7, 1, 8, 1, 26, 1, 27, 1, 44, 1 ] ], 
  [ [ 143, 18, 144, 4, 146, 3, 147, 10, 148, 10, 149, 14, 152, 8, 161, 3 ], 
      [ 241, 22, 242, 32, 243, 43, 244, 64, 245, 52, 246, 40, 247, 27, 248, 
          14 ], [ 23, 1, 24, 1, 26, 1, 27, 1, 28, 1, 29, 1, 32, 1, 41, 1 ] ], 
  [ [ 134, 15, 143, 12, 144, 12, 146, 7, 147, 7, 149, 15, 152, 16 ], 
      [ 241, 24, 242, 35, 243, 47, 244, 70, 245, 57, 246, 44, 247, 30, 248, 
          15 ], [ 14, 1, 23, 1, 24, 1, 26, 1, 27, 1, 29, 1, 32, 1 ] ], 
  [ [ 135, 16, 136, 12, 146, 15, 147, 15, 148, 7, 150, 12, 152, 1, 161, 7 ], 
      [ 241, 24, 242, 35, 243, 47, 244, 70, 245, 57, 246, 44, 247, 30, 248, 
          16 ], [ 15, 1, 16, 1, 26, 1, 27, 1, 28, 1, 30, 1, 32, 1, 41, 1 ] ], 
  [ [ 129, 28, 135, 18, 146, 24, 147, 10, 148, 15, 152, 15, 161, 1 ], 
      [ 241, 28, 242, 40, 243, 54, 244, 80, 245, 65, 246, 50, 247, 34, 248, 
          18 ], [ 9, 1, 15, 1, 26, 1, 27, 1, 28, 1, 32, 1, 41, 1 ] ], 
  [ [ 124, 8, 128, 12, 132, -4, 135, 16, 143, 6, 145, 14, 146, 12, 150, 4, 
          151, 18, 154, 4, 155, 10, 159, -4 ], 
      [ 241, 24, 242, 36, 243, 48, 244, 72, 245, 58, 246, 44, 247, 30, 248, 
          16 ], [ 4, 1, 15, 1, 23, 1, 25, 1, 26, 1, 31, 1, 34, 1, 35, 1 ] ], 
  [ [ 138, 10, 139, 24, 140, 15, 141, 15, 142, 18, 143, 28, 153, 1, 169, 1 ], 
      [ 241, 28, 242, 40, 243, 54, 244, 80, 245, 66, 246, 50, 247, 34, 248, 
          18 ], [ 18, 1, 19, 1, 20, 1, 21, 1, 22, 1, 23, 1, 33, 1, 49, 1 ] ], 
  [ [ 127, 42, 128, 22, 143, 16, 144, 16, 146, 30, 147, 30, 152, 1 ], 
      [ 241, 32, 242, 47, 243, 63, 244, 94, 245, 77, 246, 60, 247, 42, 248, 
          22 ], [ 7, 1, 8, 1, 23, 1, 24, 1, 26, 1, 27, 1, 32, 1 ] ], 
  [ [ 127, 28, 128, 22, 134, 42, 137, 17, 138, 28, 143, 3, 144, 32, 146, 3, 
          147, 15, 151, -3 ], 
      [ 241, 32, 242, 48, 243, 64, 244, 95, 245, 78, 246, 60, 247, 42, 248, 
          22 ], [ 8, 1, 14, 1, 17, 1, 18, 1, 24, 1, 26, 1, 27, 1 ] ], 
  [ [ 124, 8, 134, 18, 135, 18, 143, 8, 144, 20, 145, 14, 146, 20, 147, 14 ], 
      [ 241, 28, 242, 42, 243, 56, 244, 84, 245, 68, 246, 52, 247, 36, 248, 
          18 ], [ 4, 1, 14, 1, 15, 1, 23, 1, 24, 1, 25, 1, 26, 1, 27, 1 ] ], 
  [ [ 121, 36, 133, 21, 137, 40, 138, 12, 139, 30, 142, 22, 148, 21 ], 
      [ 241, 36, 242, 52, 243, 70, 244, 103, 245, 84, 246, 64, 247, 43, 248, 
          22 ], [ 1, 1, 13, 1, 17, 1, 18, 1, 19, 1, 22, 1, 28, 1 ] ], 
  [ [ 124, 16, 128, 22, 134, 42, 143, 2, 144, 30, 145, 30, 146, 16, 147, 2 ], 
      [ 241, 32, 242, 48, 243, 64, 244, 96, 245, 78, 246, 60, 247, 42, 248, 
          22 ], [ 4, 1, 8, 1, 14, 1, 23, 1, 24, 1, 25, 1, 26, 1, 27, 1 ] ], 
  [ [ 121, 40, 127, 15, 128, 26, 134, 50, 137, 21, 138, 15, 139, 57, 146, 22, 
          147, -22 ], 
      [ 241, 40, 242, 58, 243, 78, 244, 115, 245, 94, 246, 72, 247, 50, 248, 
          26 ], [ 1, 1, 8, 1, 14, 1, 17, 1, 18, 1, 19, 1, 26, 1 ] ], 
  [ [ 121, 36, 124, 12, 135, 2, 137, 30, 138, 2, 139, 40, 141, 22, 142, 22, 
          146, 20, 147, -20 ], 
      [ 241, 36, 242, 52, 243, 70, 244, 104, 245, 84, 246, 64, 247, 44, 248, 
          22 ], [ 1, 1, 4, 1, 17, 1, 18, 1, 19, 1, 21, 1, 22, 1, 26, 1 ] ], 
  [ [ 124, 21, 127, 15, 128, 26, 129, 40, 134, 50, 137, 1, 138, 57, 139, 15, 
          146, -22, 147, 22 ], 
      [ 241, 40, 242, 58, 243, 78, 244, 116, 245, 94, 246, 72, 247, 50, 248, 
          26 ], [ 4, 1, 8, 1, 9, 1, 14, 1, 17, 1, 18, 1, 19, 1, 27, 1 ] ], 
  [ [ 121, 52, 126, 96, 127, 66, 128, 34, 137, 27, 138, 49, 139, 75 ], 
      [ 241, 52, 242, 76, 243, 102, 244, 151, 245, 124, 246, 96, 247, 66, 
          248, 34 ], [ 1, 1, 6, 1, 7, 1, 8, 1, 17, 1, 18, 1, 19, 1 ] ], 
  [ [ 121, 44, 124, 14, 128, 28, 133, 26, 134, 54, 137, 36, 138, 28, 139, 50 ]
        , [ 241, 44, 242, 64, 243, 86, 244, 128, 245, 104, 246, 80, 247, 54, 
          248, 28 ], [ 1, 1, 4, 1, 8, 1, 13, 1, 14, 1, 17, 1, 18, 1, 19, 1 ] ]
    , 
  [ [ 126, 96, 127, 66, 128, 34, 129, 52, 130, 27, 131, 1, 132, 75, 145, 49 ],
      [ 241, 52, 242, 76, 243, 102, 244, 152, 245, 124, 246, 96, 247, 66, 
          248, 34 ], [ 6, 1, 7, 1, 8, 1, 9, 1, 10, 1, 11, 1, 12, 1, 25, 1 ] ],
  [ [ 121, 60, 122, 66, 123, -66, 125, 108, 127, 22, 128, 38, 130, 22, 131, 
          118, 132, 22, 134, 74, 140, 34 ], 
      [ 241, 60, 242, 88, 243, 118, 244, 174, 245, 142, 246, 108, 247, 74, 
          248, 38 ], [ 1, 1, 2, 1, 5, 1, 8, 1, 10, 1, 11, 1, 14, 1, 20, 1 ] ],
  [ [ 121, 72, 122, 38, 123, -38, 125, 172, 126, 132, 127, 90, 128, 46, 130, 
          68, 131, 142, 132, 68 ], 
      [ 241, 72, 242, 106, 243, 142, 244, 210, 245, 172, 246, 132, 247, 90, 
          248, 46 ], [ 1, 1, 2, 1, 5, 1, 6, 1, 7, 1, 8, 1, 10, 1, 11, 1 ] ], 
  [ [ 121, 92, 122, 136, 123, 182, 124, 270, 125, 220, 126, 168, 127, 114, 
          128, 58 ], 
      [ 241, 92, 242, 136, 243, 182, 244, 270, 245, 220, 246, 168, 247, 114, 
          248, 58 ], [ 1, 1, 2, 1, 3, 1, 4, 1, 5, 1, 6, 1, 7, 1, 8, 1 ] ] ];

      elif type[1] = 'A' then

           list:= SLAfcts.weighted_dyn_diags_An(Length(CartanMatrix(RootSystem(L))));
           orbs:= [ ];
           for i in [1..Length(list[1])] do
               if not IsZero(list[1][i]) then
                  o:= NilpotentOrbit( L, list[1][i] );
                  SetOrbitPartition( o, list[2][i] );
                  Add( orbs, o );
               fi;
           od;
           return orbs;

      elif type[1] = 'B' then

           list:= SLAfcts.weighted_dyn_diags_Bn(Length(CartanMatrix(RootSystem(L))));
           orbs:= [ ];
           for i in [1..Length(list[1])] do
               if not IsZero( list[1][i] ) then
                  o:= NilpotentOrbit( L, list[1][i] );
                  SetOrbitPartition( o, list[2][i] );
                  Add( orbs, o );
               fi;
           od;
           return orbs;

      elif type[1] = 'C' then

           list:= SLAfcts.weighted_dyn_diags_Cn(Length(CartanMatrix(RootSystem(L))));
           orbs:= [ ];
           for i in [1..Length(list[1])] do
               if not IsZero( list[1][i] ) then
                  o:= NilpotentOrbit( L, list[1][i] );
                  SetOrbitPartition( o, list[2][i] );
                  Add( orbs, o );
               fi;
           od;
           return orbs;

      elif type[1] = 'D' then

           list:= SLAfcts.weighted_dyn_diags_Dn(Length(CartanMatrix(RootSystem(L))));
           orbs:= [ ];
           for i in [1..Length(list[1])] do
               if not IsZero( list[1][i] ) then
                  o:= NilpotentOrbit( L, list[1][i] );
                  SetOrbitPartition( o, list[2][i] );
                  Add( orbs, o );
               fi;
           od;
           return orbs;

      fi;

      orbs:= [ ];
      for i in [1..Length( list )] do
          o:= NilpotentOrbit( L, list[i] );
          sl2:= [ ];
          for u in elms[i] do
              a:= Zero(L);
              for j in  [1,3..Length(u)-1] do
                  a := a + u[j+1]*Basis(L)[u[j]];
              od;
              Add( sl2, a );
          od;

          SetSL2Triple( o, sl2 );
          Add( orbs, o );
      od;

      return orbs;

end;

SLAfcts.nilporbs1:= function( L )

    # L simple.

    local tL, F, K, f, o, tK, eK, p, orbs, orb, wd, new, exceptional, eL, list, i;

    tL:= CartanType( CartanMatrix( RootSystem(L) ) );
    if Length(tL.types) > 1 then
       Error("The Lie algebra <L> is not simple.");
    fi;

    F:= LeftActingDomain(L);
    K:= SimpleLieAlgebra( tL.types[1][1], tL.types[1][2], F );
    
    exceptional:= tL.types[1][1] in ["E","F","G"];
    if exceptional then
        f:= IsomorphismOfSemisimpleLieAlgebras( K, L );
    fi;
    o:= SLAfcts.nilporbs( K );

    tK:= CartanType( CartanMatrix( RootSystem(K) ) );

    eK:= tK.enumeration[1];
    eL:= tL.enumeration[1];
    list:= [ ];
    for i in [1..Length(eK)] do list[eK[i]]:= eL[i]; od;
    p:= PermList( list );

    orbs:= [ ];
    for orb in o do
        wd:= Permuted( WeightedDynkinDiagram(orb), p );
        new:= NilpotentOrbit( L, wd );
        if exceptional then
           SetSL2Triple( new, List( SL2Triple( orb ), x -> Image( f, x ) ) );
        else
           SetOrbitPartition( new, OrbitPartition( orb ) );
        fi;
        Add( orbs, new );
    od; 

    return orbs;
end;

InstallMethod( IsSLA,
   "for a Lie algebra",
   true, [ IsLieAlgebra ], 0,
   function( L )
   
      local R, t, c, Ks, i, b, bounds, sm, e, pp;

      R:= RootSystem(L);
      if R = fail then return fail; fi;
      t:= CartanType( CartanMatrix( RootSystem(L) ) );
      if Length( t.types ) = 1 then
         return true;
      else
         c:= CanonicalGenerators(R);
         Ks:= [ ];
         for i in [1..Length(t.enumeration)] do
             Add( Ks, UnderlyingLieAlgebra( SLAfcts.rtsys( L, 
                                        List( c, u -> u{t.enumeration[i]} ) ) ) );
         od;
         b:= Concatenation( List( Ks, U -> BasisVectors( Basis(U) ) ) );
         bounds:= [ [1,Dimension(Ks[1])] ];
         for i in [2..Length(Ks)] do
             sm:= Sum( [1..i-1], j -> Dimension( Ks[j] ) );
             Add( bounds, [sm+1,sm+Dimension(Ks[i])] );
         od;
         e:= Concatenation( t.enumeration );
         pp:= [ ];
         for i in [1..Length(e)] do
             Add( pp, Position(e,i) );
         od;
         SetSLAComponents( L, rec( Ks:= Ks, pp:= pp, bounds:= bounds, 
                      bas:= Basis( L, b ) ) );
         return false;
      fi;

end );


InstallOtherMethod( NilpotentOrbits,
   "for a Lie algebra",
   true, [ IsLieAlgebra ], 10,
   function( L )

    # L semi-simple.

     local Ks, wds, wds0, w1, w2, o, wd, K, p, e, pp, i, orbs;

     if IsSLA(L) then
        return SLAfcts.nilporbs1(L);
     else
        Ks:= SLAComponents( L ).Ks;
        wds:= [ [] ];
        for K in Ks do
            o:= SLAfcts.nilporbs1(K);
            wd:= [ List( [1..Length(CartanMatrix(RootSystem(K)))], i -> 0 ) ];
            Append( wd, List( o, WeightedDynkinDiagram ) );
            wds0:= [ ];
            for w1 in wds do
                for w2 in wd do
                    Add( wds0, Concatenation(w1,w2) );
                od;
            od; 
            wds:= wds0;
        od;
        p:= PositionProperty( wds, x -> ForAll( x, IsZero ) );
        RemoveElmList( wds, p );

        wds0:= [ ];
        pp:= SLAComponents(L).pp;
        for w1 in wds do
            Add( wds0, w1{pp} );
        od;
        wds:= wds0;
             
        orbs:= List( wds, w -> NilpotentOrbit( L, w ) );

     fi;
     return orbs;       

end );


InstallMethod( SignatureTable,
"for Lie algebra", true, [IsLieAlgebra], 0,
function( L )

    local o, R, p, tab, x, w, max, dims, r, u, wt, dc, char, it, i, 
          Ci, h, ev, pos, tp, res, en;
    
    o:= NilpotentOrbits(L);
    R:= RootSystem(L);

    tp:= CartanType( CartanMatrix(R) ).types[1];
    if tp[1] in [ "A", "B", "C", "E", "F", "G" ] then

       p:= PositiveRootsNF(R);
       tab:= [ ];
       for x in o do
           w:= WeightedDynkinDiagram(x);
           max:= p[Length(p)]*w;
           if not IsInt( max ) then # hack to make it work with SqrtField...
              max:= max![1][1][1];
           fi;
           dims:= List([1..max+1], u -> 0 );
           for r in p do
               u:= r*w+1;
               if not IsInt( u ) then
                  u:= u![1][1][1];
               fi;
               dims[u]:= dims[u]+1;
           od;
           dims[1]:= 2*dims[1]+Length(CartanMatrix(R));
           Add( tab, [ dims, w ] );
       od;

       return rec( tipo:= "notD", tab:= tab );

    else

       en:= CartanType( CartanMatrix(R) ).enumeration[1];
       wt:= List( CartanMatrix( R ), x -> 0 );
       wt[en[1]]:= 1;
       dc:= DominantCharacter( L, wt );
       char:= [[],[]];
       for i in [1..Length(dc[1])] do
           it:= WeylOrbitIterator( WeylGroup(R), dc[1][i] );
           while not IsDoneIterator( it ) do
              Add( char[1], NextIterator( it ) );
              Add( char[2], dc[2][i] );
           od;
       od;

       Ci:= FamilyObj(o[1])!.invCM;
       tab:= [ ];
       for x in o do

           h:= Ci*WeightedDynkinDiagram(x);

           dims:= [ ];
           for i in [1..Length(char[1])] do
               ev:= h*char[1][i];

               pos:= PositionProperty( dims, y -> y[1]=ev);
               if pos = fail then
                  Add( dims, [ev, char[2][i]] );
               else
                  dims[pos][2]:= dims[pos][2]+char[2][i];
               fi;
           od;
           Sort( dims, function(a,b) return a[1] < b[1]; end );

           Add( tab, [dims, WeightedDynkinDiagram(x)] );
       od;

       res:= rec( tipo:= "D", char1:= char, tab1:= tab, V1:= 
                HighestWeightModule( L, wt ) );

       wt:= List( CartanMatrix( R ), x -> 0 );
       wt[en[Length(wt)]]:= 1;
       dc:= DominantCharacter( L, wt );
       char:= [[],[]];
       for i in [1..Length(dc[1])] do
           it:= WeylOrbitIterator( WeylGroup(R), dc[1][i] );
           while not IsDoneIterator( it ) do
              Add( char[1], NextIterator( it ) );
              Add( char[2], dc[2][i] );
           od;
       od;

       Ci:= FamilyObj(o[1])!.invCM;
       tab:= [ ];
       for x in o do

           h:= Ci*WeightedDynkinDiagram(x);

           dims:= [ ];
           for i in [1..Length(char[1])] do
               ev:= h*char[1][i];

               pos:= PositionProperty( dims, y -> y[1]=ev);
               if pos = fail then
                  Add( dims, [ev, char[2][i]] );
               else
                  dims[pos][2]:= dims[pos][2]+char[2][i];
               fi;
           od;
           Sort( dims, function(a,b) return a[1] < b[1]; end );

           Add( tab, [dims, WeightedDynkinDiagram(x)] );
       od;

       res.char2:= char; res.tab2:= tab;
       res.V2:= HighestWeightModule( L, wt );
       return res;
    fi;   

end );


SLAfcts.wdd:= function( L, x )

      local sl2, K, H, ch, t, cc, tableM, T, adh, possibles,
            k, dim, rank, poss0, l, c1, c2, fac, f, inds1, inds2, ev, pos, id;

      if not x in L then Error("<x> does not lie in <L>"); fi;

      if IsZero(x) then
         return List( CartanMatrix( RootSystem(L) ), x -> 0 );
      fi;

      T:= SignatureTable(L);
      sl2:= SL2Triple( L, x );
      if T.tipo = "notD" then

         T:= T.tab;
         adh:= TransposedMat( AdjointMatrix( Basis(L), sl2[2] ) );
         possibles:= [ ];

         dim:= Dimension(L);
         rank:= RankMat(adh);
         for k in [1..Length(T)] do
             if T[k][1][1] = dim-rank then
                Add( possibles, T[k] );
             fi;
         od;
   
         k:= 1;
         while Length(possibles) > 1 do 
             rank:= RankMat( adh-k*adh^0 );
             poss0:= [ ];
             for l in [1..Length(possibles)] do
                 if possibles[l][1][k+1] = dim-rank then
                    Add( poss0, possibles[l] );
                 fi;
             od;
             possibles:= poss0;
             k:= k+1;
         od; 

         return possibles[1][2];         

      else

         adh:= MatrixOfAction( Basis( T.V1 ), sl2[2] );
         id:= adh^0;
         c1:= [ ];
         k:= 0; ev:=0;
         while ev < Dimension(T.V1) do
            f:= Dimension(T.V1)-RankMat( adh - k*id );
            if f > 0 then
               Add( c1, [ k, f ] );
               if k=0 then
                  ev:= ev + f;
               else
                  ev:= ev+2*f;
                  Add( c1, [ -k, f ] );
               fi;
            fi;
            k:= k+1;
         od;
         Sort( c1, function(a,b) return a[1]<b[1]; end );
         inds1:= Filtered( [1..Length(T.tab1)], x -> T.tab1[x][1]=c1 );
         if Length(inds1) = 1 then
            return T.tab1[ inds1[1] ][2];
         fi;

         adh:= MatrixOfAction( Basis( T.V2 ), sl2[2] );
         id:= adh^0;
         c2:= [ ];
         k:= 0; ev:=0;
         while ev < Dimension(T.V2) do
            f:= Dimension(T.V2)-RankMat( adh - k*id );
            if f > 0 then
               Add( c2, [ k, f ] );
               if k=0 then
                  ev:= ev + f;
               else
                  ev:= ev+2*f;
                  Add( c2, [ -k, f ] );
               fi;
            fi;
            k:= k+1;
         od;
         Sort( c2, function(a,b) return a[1]<b[1]; end );
         inds2:= Filtered( [1..Length(T.tab2)], x -> T.tab2[x][1]=c2 );

         return T.tab1[ Intersection( inds1, inds2 )[1] ][2];

      fi;
end;

InstallOtherMethod( WeightedDynkinDiagram,
   "for a semisimple Lie algebra and a nilpotent element",
   true, [ IsLieAlgebra, IsObject ], 0,
   function(L,x)

     local r, wd, cf, i, c0, u;

     if IsSLA(L) then
        return SLAfcts.wdd(L,x);
     else
        r:= SLAComponents( L );
        wd:= [ ];
        cf:= Coefficients( r.bas, x );
        for i in [1..Length(r.Ks)] do
            c0:= cf{[r.bounds[i][1]..r.bounds[i][2]]};
            u:= Sum( [1..Length(c0)], k -> c0[k]*Basis(r.Ks[i])[k] );
            Append( wd, SLAfcts.wdd( r.Ks[i], u ) );
        od;
        return wd{r.pp};
     fi;        

end );

InstallMethod( SL2Grading,
"for Lie algebra and element", true, [IsLieAlgebra, IsObject], 0,
function( L, h )

   # Here o  is a nilpotent orbit. We return the grading of the Lie
   # algebra corresponding to "h". Three lists are returned:
   # first the components with grades 1,2,3,... then a list with
   # the components with grades -1,-2,-3... Then the zero component.

   local adh, pos, neg, k, done, sp;

   adh:= AdjointMatrix( Basis(L), h );
   
   pos:= [ ];
   neg:= [ ];
   k:= 1;
   done:= false;
   while not done do
      sp:= NullspaceMat( TransposedMat( adh-k*adh^0 ) );
      Add( pos, List( sp, c -> LinearCombination( Basis(L), c ) ) );
      sp:= NullspaceMat( TransposedMat( adh+k*adh^0 ) );
      Add( neg, List( sp, c -> LinearCombination( Basis(L), c ) ) );

      if Length(pos)>=2 and (pos[ Length(pos) ] = [ ] and pos[ Length(pos)-1 ] = [ ]) then
         done:= true;
      else
         k:= k+1;
      fi;

   od;
   while pos[Length(pos)] = [ ] do Unbind( pos[Length(pos)] ); od;
   while neg[Length(neg)] = [ ] do Unbind( neg[Length(neg)] ); od;

   return [ pos, neg, ShallowCopy( BasisVectors( 
             Basis( LieCentralizer( L, Subalgebra( L,[h] ) ) ) ) ) ];

end );

SLAfcts.rigid_diags_Dn:= function( n )


    local part, good, p, q, cur, k, m, is_even, no_gap, odd_good, u, diags, 
          u0, i, d, L, h, Ci, dims, h0, mat, dim;

    part:= Partitions( 2*n );
    good:= [ ];
    for p in part do
        q:= [ ];
        cur:= p[1];
        k:= 1;
        m:= 0;
        while k <= Length(p) do
            m:= m+1;
            if k < Length(p) and p[k+1] <> cur then
               Add( q, [ cur, m ] );
               cur:= p[k+1];
               m:= 0;
            fi;
            k:= k+1;
        od;
        Add( q, [ cur,m ] );

        is_even:= true;
        for k in [1..Length(q)] do
            if IsEvenInt( q[k][1] ) and not IsEvenInt( q[k][2] ) then
               is_even:= false;
               break;
            fi;
        od;

        no_gap:= true;
        for k in [1..Length(q)-1] do
            if q[k][1] > q[k+1][1]+1 then 
               no_gap:= false;
               break;
            fi;
        od;
        if no_gap then
           no_gap:= q[Length(q)][1] = 1;
        fi;

        odd_good:= true;        
        for k in [1..Length(q)] do
            if IsOddInt(q[k][1]) and q[k][2] = 2 then
               odd_good:= false;
               break;
            fi;
        od;

        if is_even and no_gap and odd_good then Add( good, q ); fi;
    od;
    
    diags:= [ ];
    for q in good do
        u:= [ ];
        for k in [1..Length(q)] do
           u0:= [ ];
           d:= q[k][1]-1;
           while d >= 0 do
              Add( u0, d );
              d:= d-2;
           od;
           for i in [1..q[k][2]] do Append( u, u0 ); od;
        od;
                
        Sort(u);
        u:= Reversed( u );

        d:= [ ];
        for k in [1..n-1] do
            Add( d, u[k]-u[k+1] );
        od;
        Add( d, u[n-1]+u[n] );
        Add( diags, d );
    od;

    L:= SimpleLieAlgebra( "D", n, Rationals );
    h:= ChevalleyBasis(L)[3];
    Ci:= CartanMatrix( RootSystem(L) )^-1;
    dims:= [ ];
    for d in diags do
        h0:= (Ci*d)*h;
        mat:= AdjointMatrix( Basis(L), h0 );
        dim:= Dimension(L);
        dim:= dim - Length( NullspaceMat(mat) );
        dim:= dim - Length( NullspaceMat( mat-IdentityMat( Dimension(L) ) ) );
        Add( dims, dim );
    od;

    return [diags,dims,good];

end;

SLAfcts.rigid_diags_Bn:= function( n )


    local part, good, p, q, cur, k, m, is_even, no_gap, odd_good, u, diags, 
          u0, i, d, L, h, Ci, dims, h0, mat, dim;

    part:= Partitions( 2*n+1 );
    good:= [ ];
    for p in part do
        q:= [ ];
        cur:= p[1];
        k:= 1;
        m:= 0;
        while k <= Length(p) do
            m:= m+1;
            if k < Length(p) and p[k+1] <> cur then
               Add( q, [ cur, m ] );
               cur:= p[k+1];
               m:= 0;
            fi;
            k:= k+1;
        od;
        Add( q, [ cur,m ] );

        is_even:= true;
        for k in [1..Length(q)] do
            if IsEvenInt( q[k][1] ) and not IsEvenInt( q[k][2] ) then
               is_even:= false;
               break;
            fi;
        od;

        no_gap:= true;
        for k in [1..Length(q)-1] do
            if q[k][1] > q[k+1][1]+1 then 
               no_gap:= false;
               break;
            fi;
        od;
        if no_gap then
           no_gap:= q[Length(q)][1] = 1;
        fi;

        odd_good:= true;        
        for k in [1..Length(q)] do
            if IsOddInt(q[k][1]) and q[k][2] = 2 then
               odd_good:= false;
               break;
            fi;
        od;

        if is_even and no_gap and odd_good then Add( good, q ); fi;
    od;
    
    diags:= [ ];
    for q in good do
        u:= [ ];
        for k in [1..Length(q)] do
           u0:= [ ];
           d:= q[k][1]-1;
           while d >= 0 do
              Add( u0, d );
              d:= d-2;
           od;
           for i in [1..q[k][2]] do Append( u, u0 ); od;
        od;

        # get rid of a zero...
        for k in [1..Length(u)] do
            if u[k] = 0 then RemoveElmList( u, k ); break; fi;
        od;
                
        Sort(u);
        u:= Reversed( u );

        d:= [ ];
        for k in [1..n-1] do
            Add( d, u[k]-u[k+1] );
        od;
        Add( d, u[n] );
        Add( diags, d );
    od;

    L:= SimpleLieAlgebra( "B", n, Rationals );
    h:= ChevalleyBasis(L)[3];
    Ci:= CartanMatrix( RootSystem(L) )^-1;
    dims:= [ ];
    for d in diags do
        h0:= (Ci*d)*h;
        mat:= AdjointMatrix( Basis(L), h0 );
        dim:= Dimension(L);
        dim:= dim - Length( NullspaceMat(mat) );
        dim:= dim - Length( NullspaceMat( mat-IdentityMat( Dimension(L) ) ) );
        Add( dims, dim );
    od;

    return [diags,dims,good];

end;

SLAfcts.rigid_diags_Cn:= function( n )


    local part, good, p, q, cur, k, m, is_even, no_gap, odd_good, u, diags, 
          u0, i, d, L, h, Ci, dims, h0, mat, dim;

    part:= Partitions( 2*n );
    good:= [ ];
    for p in part do
        q:= [ ];
        cur:= p[1];
        k:= 1;
        m:= 0;
        while k <= Length(p) do
            m:= m+1;
            if k < Length(p) and p[k+1] <> cur then
               Add( q, [ cur, m ] );
               cur:= p[k+1];
               m:= 0;
            fi;
            k:= k+1;
        od;
        Add( q, [ cur,m ] );

        is_even:= true;
        for k in [1..Length(q)] do
            if IsOddInt( q[k][1] ) and not IsEvenInt( q[k][2] ) then
               is_even:= false;
               break;
            fi;
        od;

        no_gap:= true;
        for k in [1..Length(q)-1] do
            if q[k][1] > q[k+1][1]+1 then 
               no_gap:= false;
               break;
            fi;
        od;
        if no_gap then
           no_gap:= q[Length(q)][1] = 1;
        fi;

        odd_good:= true;        
        for k in [1..Length(q)] do
            if IsEvenInt(q[k][1]) and q[k][2] = 2 then
               odd_good:= false;
               break;
            fi;
        od;

        if is_even and no_gap and odd_good then Add( good, q ); fi;
    od;
    
    diags:= [ ];
    for q in good do
        u:= [ ];
        for k in [1..Length(q)] do
           u0:= [ ];
           d:= q[k][1]-1;
           while d >= 0 do
              Add( u0, d );
              d:= d-2;
           od;
           for i in [1..q[k][2]] do Append( u, u0 ); od;
        od;
                
        Sort(u);
        u:= Reversed( u );

        d:= [ ];
        for k in [1..n-1] do
            Add( d, u[k]-u[k+1] );
        od;
        Add( d, 2*u[n] );
        Add( diags, d );
    od;

    L:= SimpleLieAlgebra( "C", n, Rationals );
    h:= ChevalleyBasis(L)[3];
    Ci:= CartanMatrix( RootSystem(L) )^-1;
    dims:= [ ];
    for d in diags do
        h0:= (Ci*d)*h;
        mat:= AdjointMatrix( Basis(L), h0 );
        dim:= Dimension(L);
        dim:= dim - Length( NullspaceMat(mat) );
        dim:= dim - Length( NullspaceMat( mat-IdentityMat( Dimension(L) ) ) );
        Add( dims, dim );
    od;

    return [diags,dims,good];

end;


SLAfcts.indtableAn:= function( L ) # L of type An....

     local o, rank, p, T, d, sd, i, k, dt, pos;

     o:= NilpotentOrbits(L);
     rank:= Length( CartanMatrix( RootSystem(L) ) );
     p:= [ List( [1..rank+1], x -> 1 ) ];
     Append( p, List( o, OrbitPartition ) );

     T:= [ ];
     for d in p do

         if Length(d) > 1 then
            sd:= List( [1..rank], x -> 0 );
            for i in [1..Length(d)-1] do
                k:= Sum( d{[1..i]} );
                sd[k]:= 2;
            od;

            dt:=[ ];
            for i in [1..rank+1] do
                dt[i]:= Length( Filtered( [1..Length(d)], j -> d[j] >= i ) );
            od;
            for i in [1..Length(dt)] do if dt[i] = 0 then Unbind( dt[i] ); fi; od;
            dt:= Filtered( dt, x -> IsBound(x) );

            pos:= Position( p, dt );
            Add( T, [ sd, WeightedDynkinDiagram(o[pos-1]) ] );
         fi;
     od; 
     
     return T;

end;

InstallMethod( InducedNilpotentOrbits,
"for Lie algebra", true, [IsLieAlgebra], 0,
function( L )

     local tp, T, rank, i, S, pos, o, e, f, wt;

     if not IsSLA(L) then
        Error("currently this only works for simple Lie algebras.");
     fi;

     tp:= CartanType( CartanMatrix( RootSystem(L) ) );
     rank:= tp.types[1][2];

     if tp.types[1][1] = "A" then
        T:= SLAfcts.indtableAn( SimpleLieAlgebra( "A", rank, Rationals ) );
     elif tp.types[1][1] = "B" then
        T:= SLAfcts.SB[rank];
     elif tp.types[1][1] = "C" then
        T:= SLAfcts.SC[rank];
     elif tp.types[1][1] = "D" then
        T:= SLAfcts.SD[rank];        
     elif tp.types[1][1] = "E" then
        T:= SLAfcts.SE[rank];
     elif tp.types[1][1] = "F" then
        T:= SLAfcts.SF[rank];
     elif tp.types[1][1] = "G" then
        T:= SLAfcts.SG[rank];
     fi;

     wt:= List( [1..rank], x -> 2 );
     T:= Concatenation( [[ ShallowCopy(wt), ShallowCopy(wt) ]], T );

     e:= tp.enumeration[1];
     f:= [ ];
     for i in [1..rank] do f[ e[i] ]:= i; od;

     for i in [1..Length(T)] do
         T[i][1]:= T[i][1]{f};
         T[i][2]:= T[i][2]{f};
     od;

     S:= [ ];
     o:= NilpotentOrbits(L);
     for i in [1..Length(T)] do
         pos:= PositionProperty( o, x -> WeightedDynkinDiagram(x) = T[i][2] );
         Add( S, rec( sheetdiag:= T[i][1], norbit:= o[pos] ) );
     od;

     return S;

end );

InstallMethod( RigidNilpotentOrbits,
"for Lie algebra", true, [IsLieAlgebra], 0,
function( L )

     local tp, T, rank, i, S, pos, o, e, f;

     if not IsSLA(L) then
        Error("currently this only works for simple Lie algebras.");
     fi;

     tp:= CartanType( CartanMatrix( RootSystem(L) ) );
     if tp.types[1][1] = "A" then
        T:= [ ];
     else
        rank:= tp.types[1][2];
        if tp.types[1][1] = "B" then
           T:= SLAfcts.rigid_diags_Bn( rank )[1];
        elif tp.types[1][1] = "C" then
           T:= SLAfcts.rigid_diags_Cn( rank )[1];
        elif tp.types[1][1] = "D" then
           T:= SLAfcts.rigid_diags_Dn( rank )[1];        
        elif tp.types[1][1] = "E" then
           if rank = 6 then
              T:= [ [1,0,0,1,0,1], [0,0,0,1,0,0], [0,1,0,0,0,0] ];
           elif rank = 7 then
              T:= [ [1,0,0,1,0,0,0], [0,0,1,0,0,1,0], 
                    [0,0,0,1,0,0,0], [0,1,0,0,0,0,1],
                    [0,0,1,0,0,0,0], [0,0,0,0,0,1,0],
                    [0,1,0,0,0,0,0] ];
           elif rank = 8 then
              T:= [ [ 0, 0, 0, 0, 0, 0, 0, 1 ], [ 1, 0, 0, 0, 0, 0, 0, 0 ], 
                    [ 0, 0, 0, 0, 0, 0, 1, 0 ], [ 0, 1, 0, 0, 0, 0, 0, 0 ], 
                    [ 1, 0, 0, 0, 0, 0, 0, 1 ], [ 0, 0, 0, 0, 0, 1, 0, 0 ], 
                    [ 0, 0, 1, 0, 0, 0, 0, 0 ], [ 1, 0, 0, 0, 0, 0, 1, 0 ], 
                    [ 0, 0, 0, 0, 0, 1, 0, 1 ], [ 0, 0, 0, 0, 1, 0, 0, 0 ], 
                    [ 0, 0, 1, 0, 0, 0, 0, 1 ], [ 0, 1, 0, 0, 0, 0, 1, 0 ], 
                    [ 0, 0, 0, 1, 0, 0, 0, 0 ], [ 1, 0, 0, 0, 1, 0, 0, 0 ], 
                    [ 0, 0, 0, 1, 0, 0, 1, 0 ], [ 1, 0, 0, 1, 0, 0, 0, 1 ], 
                    [ 0, 0, 1, 0, 0, 1, 0, 1 ] ];
           fi;
        elif tp.types[1][1] = "F" then
           T:= [ [ 0, 1, 0, 0 ], [ 1, 0, 0, 0 ], [ 0, 0, 0, 1 ], [ 0, 0, 1, 0 ], [ 1, 0, 0, 1 ] ];
        elif tp.types[1][1] = "G" then
           T:= [ [ 1, 0 ], [ 0, 1 ] ];
        fi;

        e:= tp.enumeration[1];
        f:= [ ];
        for i in [1..rank] do f[ e[i] ]:= i; od;

        for i in [1..Length(T)] do
            T[i]:= T[i]{f};
        od;
     fi;

     o:= NilpotentOrbits(L);
     return Filtered( o, x -> WeightedDynkinDiagram(x) in T );

end );

InstallOtherMethod( Dimension,
        "for a nilpotent orbit", true,
        [ IsNilpotentOrbit ], 0,
        function( o )
    
        local wd, L, R, posR, dim, r0;

        wd:= WeightedDynkinDiagram(o);
	L:= AmbientLieAlgebra(o);
	R:= RootSystem(L);
	posR:= PositiveRootsNF(R);
	dim:= Dimension(L);
	r0:= Length( Filtered( posR, r -> wd*r = 0 ) );
	dim:= dim-Length(CartanMatrix(R))-2*r0;
	r0:= Length( Filtered( posR, r -> wd*r = 1 ) );
	return dim-r0;

end );


InstallMethod( IsRegular,
        "for a nilpotent orbit", true,
        [ IsNilpotentOrbit ], 0,
        function( o )

      return ForAll( WeightedDynkinDiagram(o), x -> x=2 );

end );

InstallMethod( RegularNilpotentOrbit,
        "for a semisimple Lie algebra", true,
        [ IsLieAlgebra ], 0,
        function( L )

      return Filtered( NilpotentOrbits(L), IsRegular )[1];

end );

InstallMethod( IsDistinguished,
        "for a nilpotent orbit", true,
        [ IsNilpotentOrbit ], 0,
        function( o )


        local wd, L, R, posR, d0, r0;

        wd:= WeightedDynkinDiagram(o);
	L:= AmbientLieAlgebra(o);
	R:= RootSystem(L);
	posR:= PositiveRootsNF(R);
	r0:= Length( Filtered( posR, r -> wd*r = 0 ) );
	d0:= Length(CartanMatrix(R))+2*r0;
	r0:= Length( Filtered( posR, r -> wd*r = 2 ) );

        return d0=r0;

end );

InstallMethod( DistinguishedNilpotentOrbits,
        "for a semisimple Lie algebra", true,
        [ IsLieAlgebra ], 0,
        function( L )

      return Filtered( NilpotentOrbits(L), IsDistinguished );

end );

InstallMethod( DisplayWeightedDynkinDiagram,
    "for a nilpotent orbit",
    true, [ IsNilpotentOrbit ], 0,
    function(o )

    local C;
    
    C:= CartanMatrix(RootSystem(AmbientLieAlgebra(o)));
    SLAfcts.printdyndiag(C,WeightedDynkinDiagram(o));
end);


InstallOtherMethod( DisplayWeightedDynkinDiagram,
    "for a Lie algebra and a nipotent element",
    true, [ IsLieAlgebra, IsObject ], 0,
    function( L, x )

    local C;
    
    C:= CartanMatrix(RootSystem(L));
    SLAfcts.printdyndiag(C,WeightedDynkinDiagram(L,x));
end);

InstallMethod( RichardsonOrbits,
        "for a semisimple Lie algebra", true,
        [ IsLieAlgebra ], 0,
        function( L )

     local ind, rich, wdd, i; 

     ind:= InducedNilpotentOrbits(L);
     rich:= [ ];
     wdd:= [ ];
     for i in [1..Length(ind)] do
         if not (1 in ind[i].sheetdiag) then
	    if not WeightedDynkinDiagram(ind[i].norbit) in wdd then
	       Add( rich, ind[i].norbit );
	       Add( wdd, WeightedDynkinDiagram(ind[i].norbit) );
	    fi;
	 fi;
     od;

     return rich;

end );


SLAfcts.diagmat:= function( F, H ) # F : field
                           # H torus, find a matrix C such that 
                           # C^-1*u*C is diagonal for all u in H


    local spaces, i, h, sp0, s, sp, A, fct, f, V, bas, C, one,
          fct0, num, fam, cf, b, c, r, n, U;

    one := One(F);
    n:= Length( H[1] );

    if ForAll( H, IsDiagonalMat ) then # nothing to do..
       return IdentityMat(n)*one;
    fi;

    V:= F^n;
    spaces:= [ ShallowCopy( BasisVectors( Basis(V) ) ) ];
    for i in [1..Length(H)] do
        h:= H[i];
        sp0:= [ ];
        for s in spaces do
            sp:= Basis( Subspace( V, s ), s );
            A:= List( s, u -> Coefficients( sp, h*u ) );
            fct:= Factors( MinimalPolynomial(F,A) );

            num  := IndeterminateNumberOfUnivariateLaurentPolynomial(fct[1]);
            fam  := FamilyObj( fct[1] );

            fct0:= [ ];
            for f in fct do
               if Degree(f) = 1 then
                  Add( fct0, f );
               elif Degree(f) = 2 then # we just take square roots...
                  cf := CoefficientsOfUnivariatePolynomial(f);
                  b  := cf[2]; 
                  c  := cf[1];
                  r  := (-b+Sqrt(b^2-4*c))/2;
                  if not r in F then Print("diagmat: cannot do this over ",F,"\n"); fi;
                  Add( fct0, PolynomialByExtRep( fam, 
                                            [ [], -r, [num,1], one] ) );
                  r  := (-b-Sqrt(b^2-4*c))/2;
                  if not r in F then Print("diagmat: cannot do this over ",F,"\n"); fi;
                  Add( fct0, PolynomialByExtRep( fam, 
                           [ [], -r, [num,1], one] ) );
               else
                  Error("not split!");
                  return fail;
               fi;
            od;

            for f in fct0 do
                U:= NullspaceMat( Value( f, A ) );
                Add( sp0, List( U, x -> LinearCombination( s, x ) ) );
            od;
        od;
        spaces:= sp0;
    od;
   
    bas:= Concatenation( spaces );
    C:= List( bas, x -> Coefficients( Basis(V), x ) );
    return TransposedMat(C); 

end;


SLAfcts.toruspar:= function( F, H ) # H list of mats giving the Lie algebra of torus
                             # output: parametrization of torus 

    local C, ad, add, diag, lat, n, A, d, r, S, P, Q, names, i, j, R, 
          indet, mats, m;

    n:= Length( H[1] );

    C:= SLAfcts.diagmat(F,H);
    add:= List( H, u -> C^-1*u*C );

    diag:= List( add, a -> List( [1..n], i -> a[i][i] ) );
    lat:= NullspaceMat( TransposedMat(diag) );

    if lat = [ ] then lat:= [ List( [1..n], i -> Zero(F) ) ]; fi;

    # saturation...
    A:= lat;
    if A <> 0*A then
       d:= Lcm( List( Filtered( Flat(A), x -> not IsZero(x)), DenominatorRat));
       A:= d*A;
    fi;
    r:=SmithNormalFormIntegerMatTransforms(A);
    Q:= r.coltrans; P:= r.rowtrans;
    S:= List( r.normal, ShallowCopy );
    for i in [1..Length(S)] do 
        if S[i][i] > 0 then S[i][i]:= 1; fi;
    od;
    lat:= P^-1*S*Q^-1;

    lat:= NullspaceMat( TransposedMat(lat) );
    # another saturation...
    A:= lat;
    d:= Lcm( List( Filtered( Flat(A), x -> not IsZero(x) ), DenominatorRat ) );
    A:= d*A;
    r:=SmithNormalFormIntegerMatTransforms(A);
    Q:= r.coltrans; P:= r.rowtrans;
    S:= List( r.normal, ShallowCopy );
    for i in [1..Length(S)] do 
        if S[i][i] > 0 then S[i][i]:= 1; fi;
    od;
    lat:= P^-1*S*Q^-1;

    names:= [ ];
    for i in [1..Length(lat)] do
        Add( names, Concatenation( "a", String(i) ) );
    od;
    R:= PolynomialRing( F, names );
    
    indet:= IndeterminatesOfPolynomialRing( R );
    mats:=  [ ];
    for i in [1..Length(lat)] do
        m:= NullMat( n, n, R );
        for j in [1..n] do
            m[j][j]:= indet[i]^lat[i][j];
        od;
        #Add( mats, C*m*C^-1 );
        Add( mats, m );
    od;

    A:= TransposedMat(lat);
    Q:= HermiteNormalFormIntegerMatTransform( A ).rowtrans;

    return rec( mats:= mats, C:= C, indets:= indet, exps:=A, Q:= Q );
    

end;


SLAfcts.valpars:= function( r, A )

        # here r is the output of torusparam_gen
        # A is a matrix in the torus (so the nonsplit torus)
        # with elements T(t1,...,tm)
        # we return a the vector [t1,..,tm] such that
        # T(t1,...,tm)=A

        local D, Q, t, m;

        D:= r.C^-1*A*r.C;
        if not IsDiagonalMat( D ) then return fail; fi;
        D:= List( [1..Length(D)], i -> D[i][i] );
        Q:= r.Q;
        t:= List( Q, s -> Product( List( [1..Length(s)], i -> D[i]^s[i] ) ) );
        m:= Length(r.exps[1]);
        if not ForAll([m+1..Length(t)], i -> IsOne(t[i]) ) then
           return fail; # the element does not lie in the torus...
        fi;
        return t{[1..m]};
end;



SLAfcts.iseltof:= function( L, C, s )

        # here s is an element of order 2 in the automorphism group of
        # L, C is a reductive subalgebra of L, we determine whether s lies
        # in the connected subgroup of Ad L, with Lie algebra ad C.

        local sol, U, K, H, adH, r;

        sol:= NullspaceMat( TransposedMat(s-s^0) );
        U:= Subalgebra( L, List( sol, x -> x*Basis(L)), "basis" );
        K:= Intersection( U, C );
        H:= CartanSubalgebra( K );
        if Dimension(H) < Dimension( CartanSubalgebra(C) ) then
           # the restriction of s to C is not inner, hence cannot lie in the
           # group
           return false;
        fi;

        adH:= List( Basis(H), x -> AdjointMatrix( Basis(L), x ) );
        r:= SLAfcts.toruspar( CF(840), adH );      
        return SLAfcts.valpars( r, s ) <> fail;

end;



SLAfcts.orthLA:= function( F, n )

        # for n = 2k+1 this gives the Lie algebra of type B_k, k\geq 2
        # for n = 2k this gives the Lie algebra of type D_k, k\geq 4.
        # both in standard nxn matrix rep.

        local bas, i, j, m;

        bas:= [ ];
        for i in [1..n-1] do
            for j in [i+1..n] do
                if not (i=(n+1)/2 and j=(n+1)/2) then 
                   m:= NullMat(n,n);
                   m[i][n+1-j]:= 1; m[j][n+1-i]:= -1;
                   Add( bas, m );
                fi;
            od;
        od;

        return LieAlgebra( F, bas, "basis" );

end;

SLAfcts.simpLA:= function( F, n )

        # sp(n,F) (of rank n/2).

        local bas, i, j, l, m;

        l:= n/2;
        bas:= [ ];
        for i in [1..l] do
            for j in [1..l] do
                m:= NullMat(n,n);
                m[i][j]:=1; m[l+j][l+i]:=-1;
                Add( bas, m );
            od;
        od;

        for i in [1..l] do
            for j in [i..l] do
                m:= NullMat(n,n);
                m[i][l+j]:=1; m[j][l+i]:=1;
                Add( bas, m );
            od;
        od;

        for i in [1..l] do
            for j in [i..l] do
                m:= NullMat(n,n);
                m[l+i][j]:=1; m[l+j][i]:=1;
                Add( bas, m );
            od;
        od;

        return LieAlgebra( F, bas, "basis" );

end;
     



SLAfcts.compgrp:= function( L, sl2, eps ) # L is the output of the previous fct,
                                  # sl2 an sl2-triple...
                                  # eps is -1 if L is simp, 1 of l=o_n...
        
        local n, K, V, M, i, j, phi, dV, dims, d0, vv0, v0, u0, w0, bass0,
              bas, BV0, gens, s, ms, psi, found, wn, eqn, sol, ww, Bww, g,
              imgs, mat, l;        

        n:= Length( GeneratorsOfAlgebra(L)[1]![1] );
        K:= SubalgebraNC( L, sl2 );

        V:= LeftAlgebraModule( K, function(x,v) return x![1]*v; end,
                  LeftActingDomain(L)^n );

        M:= NullMat( n, n );
        if eps=1 then 
           for i in [1..n] do M[i][n+1-i]:=1; od;
        else
           l:= n/2;
           for i in [1..l] do
               M[i][l+i]:=1;
               M[l+i][i]:= -1;
           od;
        fi;

        phi:= function(v,w) #v,w in V
              return v![1]*M*w![1];
        end;

        dV:= DirectSumDecomposition(V);
        dims:= List( dV, Dimension );

        if eps=1 then
           d0:= Filtered( Set( dims ), IsOddInt );
        else
           d0:= Filtered( Set( dims ), IsEvenInt );
        fi;

        vv0:= [ ];
        for i in [1..Length(dV)] do
            v0:= Basis( dV[i] )[dims[i]];
            u0:= sl2[1]^v0;
            while not IsZero(u0) do
                 v0:= u0;
                 u0:= sl2[1]^v0;
            od;
            Add( vv0, v0 );
        od;

        bass0:= [ ];
        for i in  [1..Length(dV)] do
            bas:=[];
            u0:= vv0[i];
            while not IsZero(u0) do
                 Add(bas,u0);
                 u0:= sl2[3]^u0;
            od;
            Add( bass0, bas );
        od;

        BV0:= Basis( V, Concatenation(bass0) );

        gens:= [ ];
        for s in d0 do
            ms:= [ ];
            for i in [1..Length(dV)] do
                if dims[i] = s then
                   Add( ms, vv0[i] );
                fi;
            od;

            psi:= function( v, w )
                  local i0, w0;

                  w0:= w;
                  for i0 in [1..s-1] do 
                      w0:= sl2[3]^w0;
                  od;
                  return phi( v, w0 );
            end;

            found:= false;
            for i in [1..Length(ms)] do
                if not IsZero( psi( ms[i], ms[i] ) ) then
                   wn:= ms[i]; found:= true; break;
                fi;
            od;

            for i in [1..Length(ms)] do
                if found then break; fi;
                for j in [i+1..Length(ms)] do
                    if not IsZero( psi( ms[i], ms[j] ) ) then
                       wn:= ms[i]+ms[j]; found:= true; break;
                    fi;
                od;
            od;

            eqn:= [ List( ms, v -> psi( v, wn ) ) ];
            sol:= NullspaceMat( TransposedMat(eqn) );
            ww:= List( sol, q -> q*ms );
            Add( ww, wn );
            Bww:= Basis( Subspace( V, ww ), ww );

            g:= function( u ) # u in the span of ms..
                local cff;
                cff:= ShallowCopy( Coefficients( Bww, u ) );
                cff[ Length(cff) ]:= -cff[Length(cff)];
                return cff*ww;
            end;

            imgs:= [ ];
            for i in [1..Length(dV)] do
                if dims[i] = s then
                   u0:= g(bass0[i][1]);
                   while not IsZero(u0) do
                        Add( imgs, u0 );
                        u0:= sl2[3]^u0;
                   od;
                else
                   Append( imgs, bass0[i] ); 
                fi;
            od;

            mat:= [ ];
            for i in [1..Length(Basis(V))] do
                u0:= Coefficients( BV0, Basis(V)[i] )*imgs;
                Add( mat, Coefficients( Basis(V), u0 ) );
            od;

            Add( gens, TransposedMat(mat) );
        od;

        return gens;

end;


SLAfcts.compgrpB_D:= function( o )

        local ct, n, M, f, f1, sl2, gg, elms, mats, g, mat, mats0, CC, ad, Ad,
              u, L, p, odd, parts, mults, j, a, b, size, id, CG, even, eps,
              r, bound;

        # L simple of type B-D
        # o is a nilpotent orbit in L

        L:= AmbientLieAlgebra(o);
        id:= IdentityMat( Dimension(L), LeftActingDomain(L) );

        if IsBound( L!.compgrpData ) then
           r:= L!.compgrpData;
           M:= r.liealg;
           f:= r.isomLM;
           f1:= r.isomML;
           ct:= r.types;
           bound:= true;
        else
           ct:= CartanType( CartanMatrix( RootSystem(L) ) ).types;
           r:= rec(types:= ct);
           bound:= false;
        fi;

        if ct[1][1] = "B" then
           n:= 2*ct[1][2]+1;
           p:= OrbitPartition(o);
           odd:= Filtered( p, IsOddInt );
           a:= Length(Set(odd));
           size:= 2^(a-1);
           if not bound then 
              M:= SLAfcts.orthLA( LeftActingDomain(L), n );
              r.liealg:= M;
           fi;
           eps:=1;
        elif ct[1][1] = "C" then
           n:= 2*ct[1][2];
            p:= OrbitPartition(o);
            even:= Filtered( p, IsEvenInt );
            if Length(even)=0 then
               size:=1;
            else
               parts:= [ even[1] ];
               mults:= [ 1 ];
               for j in [2..Length(even)] do
                   if even[j] = parts[ Length(parts) ] then
                      mults[Length(mults)]:= mults[Length(mults)]+1;
                   else
                      Add( parts, even[j] );
                      Add( mults, 1 );
                   fi;
               od;
               a:= Length(parts);
               if ForAll( mults, IsEvenInt ) then
                  b:= a;
               else
                  b:= a-1;
               fi;
               size:= 2^b;
            fi;
            if not bound then 
               M:= SLAfcts.simpLA( LeftActingDomain(L), n );
               r.liealg:= M;
            fi;
            eps:=-1;
        elif ct[1][1] = "D" then
            n:= 2*ct[1][2];
            p:= OrbitPartition(o);
            odd:= Filtered( p, IsOddInt );
            if Length(odd)=0 then
               size:=1;
            else
               parts:= [ odd[1] ];
               mults:= [ 1 ];
               for j in [2..Length(odd)] do
                   if odd[j] = parts[ Length(parts) ] then
                      mults[Length(mults)]:= mults[Length(mults)]+1;
                   else
                      Add( parts, odd[j] );
                      Add( mults, 1 );
                   fi;
               od;
               a:= Length(parts);
               if ForAll( mults, IsEvenInt ) then
                  b:= Maximum(0,a-1);
               else
                  b:= Maximum(0,a-2);
               fi;
               size:= 2^b;
            fi;
            if not bound then 
               M:= SLAfcts.orthLA( LeftActingDomain(L), n );
               r.liealg:= M;
            fi;
            eps:=1;
        fi;

        if size = 1 then return Group([id]); fi;

        if not bound then 
           f:= IsomorphismOfSemisimpleLieAlgebras( L, M );
           f1:= IsomorphismOfSemisimpleLieAlgebras( M, L );
           r.isomLM:= f;
           r.isomML:= f1;
           L!.compgrpData:= r;
        fi;

        sl2:= List( SL2Triple(o), x -> Image( f, x ) );
        gg:= SLAfcts.compgrp( M, sl2, eps );
        elms:= Filtered( Elements( Group(gg) ), x -> DeterminantMat(x)=1 );

        mats:= [ ];
        for g in elms do
            mat:= List( Basis(L), x -> Coefficients( Basis(L),
                Image( f1, g*Image(f,x)*g^-1 ) ) );
            Add( mats, TransposedMat(mat) );
        od;

        mats:= Set(mats);
        CG:= Group(mats);

        if Length(Elements(CG)) = size then
           return CG;
        else
           mats0:= [ ];
           CC:= LieCentralizer( L, Subalgebra(L,SL2Triple(o)) );
           if Dimension(CC) > 0 then
              for u in mats do
                  if SLAfcts.iseltof( L, CC, u ) then
                     Add( mats0, u );
                  fi;
              od;
              # so mats0 is the normal subgroup of elements in the idcomp
              # we need to take coset reps
              mats:= RightTransversal( Group(mats), Group(mats0) );
           fi;

           if Length(mats)=0 then
              return Group( [id]);
           else
             return Group(mats);
           fi;
        fi;

end;


SLAfcts.compgrps:= [

[ "G", 2, [ [0,2],
   [ 14, [ 1, 1, -1 ], [ 2, 2, 1 ], [ 3, 3, -1 ], [ 4, 4, 1 ], [ 5, 5, -1 ], 
  [ 6, 6, -1 ], [ 7, 7, -1 ], [ 8, 8, 1 ], [ 9, 9, -1 ], [ 10, 10, 1 ], 
  [ 11, 11, -1 ], [ 12, 12, -1 ], [ 13, 13, 1 ], [ 14, 14, 1 ] ],
  [ 14, [ 1, 1, 1/4 ], [ 1, 7, 3/4 ], [ 1, 13, -1/2*E(3)+1/2*E(3)^2 ], 
  [ 1, 14, 1/4*E(3)-1/4*E(3)^2 ], [ 2, 2, -1/8 ], 
  [ 2, 3, 3/8*E(3)-3/8*E(3)^2 ], [ 2, 4, 9/8 ], 
  [ 2, 5, -3/8*E(3)+3/8*E(3)^2 ], [ 3, 2, 1/8*E(3)-1/8*E(3)^2 ], 
  [ 3, 3, 5/8 ], [ 3, 4, -1/8*E(3)+1/8*E(3)^2 ], [ 3, 5, 3/8 ], 
  [ 4, 2, 3/8 ], [ 4, 3, -1/8*E(3)+1/8*E(3)^2 ], [ 4, 4, 5/8 ], 
  [ 4, 5, 1/8*E(3)-1/8*E(3)^2 ], [ 5, 2, -3/8*E(3)+3/8*E(3)^2 ], 
  [ 5, 3, 9/8 ], [ 5, 4, 3/8*E(3)-3/8*E(3)^2 ], [ 5, 5, -1/8 ], [ 6, 6, 1 ], 
  [ 7, 1, 3/4 ], [ 7, 7, 1/4 ], [ 7, 13, 1/2*E(3)-1/2*E(3)^2 ], 
  [ 7, 14, -1/4*E(3)+1/4*E(3)^2 ], [ 8, 8, -1/8 ], 
  [ 8, 9, -3/8*E(3)+3/8*E(3)^2 ], [ 8, 10, 9/8 ], 
  [ 8, 11, 3/8*E(3)-3/8*E(3)^2 ], [ 9, 8, -1/8*E(3)+1/8*E(3)^2 ], 
  [ 9, 9, 5/8 ], [ 9, 10, 1/8*E(3)-1/8*E(3)^2 ], [ 9, 11, 3/8 ], 
  [ 10, 8, 3/8 ], [ 10, 9, 1/8*E(3)-1/8*E(3)^2 ], [ 10, 10, 5/8 ], 
  [ 10, 11, -1/8*E(3)+1/8*E(3)^2 ], [ 11, 8, 3/8*E(3)-3/8*E(3)^2 ], 
  [ 11, 9, 9/8 ], [ 11, 10, -3/8*E(3)+3/8*E(3)^2 ], [ 11, 11, -1/8 ], 
  [ 12, 12, 1 ], [ 13, 1, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 13, 7, 1/4*E(3)-1/4*E(3)^2 ], [ 13, 13, -1/2 ], [ 13, 14, 3/4 ], 
  [ 14, 14, 1 ] ] ] ],          
            
            
[ "F", 4, [ [ 1, 0, 0, 0 ],
  [ 52, [ 1, 5, 1 ], [ 2, 2, 1 ], [ 3, 27, 1 ], [ 4, 10, 1 ], [ 5, 1, 1 ], 
  [ 6, 13, 1 ], [ 7, 7, 1 ], [ 8, 12, 1 ], [ 9, 9, 1 ], [ 10, 4, 1 ], 
  [ 11, 14, 1 ], [ 12, 8, 1 ], [ 13, 6, 1 ], [ 14, 11, 1 ], [ 15, 15, 1 ], 
  [ 16, 16, 1 ], [ 17, 19, -1 ], [ 18, 18, 1 ], [ 19, 17, -1 ], 
  [ 20, 22, 1 ], [ 21, 21, 1 ], [ 22, 20, 1 ], [ 23, 23, 1 ], [ 24, 24, 1 ], 
  [ 25, 29, 1 ], [ 26, 26, 1 ], [ 27, 3, 1 ], [ 28, 34, 1 ], [ 29, 25, 1 ], 
  [ 30, 37, 1 ], [ 31, 31, 1 ], [ 32, 36, 1 ], [ 33, 33, 1 ], [ 34, 28, 1 ], 
  [ 35, 38, 1 ], [ 36, 32, 1 ], [ 37, 30, 1 ], [ 38, 35, 1 ], [ 39, 39, 1 ], 
  [ 40, 40, 1 ], [ 41, 43, -1 ], [ 42, 42, 1 ], [ 43, 41, -1 ], 
  [ 44, 46, 1 ], [ 45, 45, 1 ], [ 46, 44, 1 ], [ 47, 47, 1 ], [ 48, 48, 1 ], 
  [ 49, 49, 1 ], [ 50, 50, 1 ], [ 51, 49, 1 ], [ 51, 51, -1 ], [ 51, 52, 1 ], 
  [ 52, 52, 1 ] ] ],

[ [ 0, 2, 0, 0 ],
  [ 52, [ 1, 29, -E(3)^2 ], [ 2, 23, E(3) ], [ 3, 3, -1 ], [ 4, 34, E(3) ], 
  [ 5, 25, -E(3)^2 ], [ 6, 20, E(3)^2 ], [ 7, 31, E(3) ], [ 8, 36, 1 ], 
  [ 9, 21, -E(3)^2 ], [ 10, 28, E(3) ], [ 11, 17, -E(3) ], [ 12, 32, 1 ], 
  [ 13, 22, E(3)^2 ], [ 14, 19, E(3) ], [ 15, 39, E(3)^2 ], [ 16, 18, 1 ], 
  [ 17, 11, -E(3)^2 ], [ 18, 16, 1 ], [ 19, 14, E(3)^2 ], [ 20, 6, E(3) ], 
  [ 21, 9, -E(3) ], [ 22, 13, E(3) ], [ 23, 2, E(3)^2 ], [ 24, 24, -1 ], 
  [ 25, 5, -E(3) ], [ 26, 47, E(3)^2 ], [ 27, 27, -1 ], [ 28, 10, E(3)^2 ], 
  [ 29, 1, -E(3) ], [ 30, 44, E(3) ], [ 31, 7, E(3)^2 ], [ 32, 12, 1 ], 
  [ 33, 45, -E(3) ], [ 34, 4, E(3)^2 ], [ 35, 41, -E(3)^2 ], [ 36, 8, 1 ], 
  [ 37, 46, E(3) ], [ 38, 43, E(3)^2 ], [ 39, 15, E(3) ], [ 40, 42, 1 ], 
  [ 41, 35, -E(3) ], [ 42, 40, 1 ], [ 43, 38, E(3) ], [ 44, 30, E(3)^2 ], 
  [ 45, 33, -E(3)^2 ], [ 46, 37, E(3)^2 ], [ 47, 26, E(3) ], [ 48, 48, -1 ], 
  [ 49, 49, -1 ], [ 49, 50, 1 ], [ 50, 50, 1 ], [ 51, 49, -1 ], 
  [ 51, 50, 2 ], [ 51, 51, 1 ], [ 51, 52, -1 ], [ 52, 50, 3 ], 
  [ 52, 52, -1 ] ] ],

[ [ 1, 2, 0, 0 ],
[ 52, [ 1, 8, -E(4) ], [ 2, 16, -1 ], [ 3, 3, 1 ], [ 4, 34, 1 ], 
  [ 5, 12, E(4) ], [ 6, 6, 1 ], [ 7, 31, -1 ], [ 8, 1, E(4) ], [ 9, 9, 1 ], 
  [ 10, 28, 1 ], [ 11, 17, E(4) ], [ 12, 5, -E(4) ], [ 13, 13, 1 ], 
  [ 14, 19, E(4) ], [ 15, 15, 1 ], [ 16, 2, -1 ], [ 17, 11, -E(4) ], 
  [ 18, 23, -1 ], [ 19, 14, -E(4) ], [ 20, 20, 1 ], [ 21, 21, 1 ], 
  [ 22, 22, 1 ], [ 23, 18, -1 ], [ 24, 24, 1 ], [ 25, 32, E(4) ], 
  [ 26, 40, -1 ], [ 27, 27, 1 ], [ 28, 10, 1 ], [ 29, 36, -E(4) ], 
  [ 30, 30, 1 ], [ 31, 7, -1 ], [ 32, 25, -E(4) ], [ 33, 33, 1 ], 
  [ 34, 4, 1 ], [ 35, 41, -E(4) ], [ 36, 29, E(4) ], [ 37, 37, 1 ], 
  [ 38, 43, -E(4) ], [ 39, 39, 1 ], [ 40, 26, -1 ], [ 41, 35, E(4) ], 
  [ 42, 47, -1 ], [ 43, 38, E(4) ], [ 44, 44, 1 ], [ 45, 45, 1 ], 
  [ 46, 46, 1 ], [ 47, 42, -1 ], [ 48, 48, 1 ], [ 49, 49, 1 ], 
  [ 50, 50, 1 ], [ 51, 49, 1 ], [ 51, 50, 1 ], [ 51, 51, 1 ], 
  [ 51, 52, -1 ], [ 52, 49, 2 ], [ 52, 50, 2 ], [ 52, 52, -1 ] ] ],

[ [ 0, 1, 1, 0 ],
[ 52, [ 1, 1, -1 ], [ 2, 2, 1 ], [ 3, 3, -1 ], [ 4, 4, 1 ], [ 5, 5, 1 ], 
  [ 6, 6, 1 ], [ 7, 7, -1 ], [ 8, 8, 1 ], [ 9, 9, -1 ], [ 10, 10, 1 ], 
  [ 11, 11, 1 ], [ 12, 12, -1 ], [ 13, 13, 1 ], [ 14, 14, -1 ], 
  [ 15, 15, 1 ], [ 16, 16, 1 ], [ 17, 17, -1 ], [ 18, 18, 1 ], 
  [ 19, 19, 1 ], [ 20, 20, 1 ], [ 21, 21, -1 ], [ 22, 22, 1 ], 
  [ 23, 23, 1 ], [ 24, 24, 1 ], [ 25, 25, -1 ], [ 26, 26, 1 ], 
  [ 27, 27, -1 ], [ 28, 28, 1 ], [ 29, 29, 1 ], [ 30, 30, 1 ], 
  [ 31, 31, -1 ], [ 32, 32, 1 ], [ 33, 33, -1 ], [ 34, 34, 1 ], 
  [ 35, 35, 1 ], [ 36, 36, -1 ], [ 37, 37, 1 ], [ 38, 38, -1 ], 
  [ 39, 39, 1 ], [ 40, 40, 1 ], [ 41, 41, -1 ], [ 42, 42, 1 ], 
  [ 43, 43, 1 ], [ 44, 44, 1 ], [ 45, 45, -1 ], [ 46, 46, 1 ], 
  [ 47, 47, 1 ], [ 48, 48, 1 ], [ 49, 49, 1 ], [ 50, 50, 1 ], [ 51, 51, 1 ], 
  [ 52, 52, 1 ] ] ],  

[ [ 2, 0, 0, 2 ],
[ 52, [ 1, 1, -1 ], [ 2, 2, -1 ], [ 3, 3, -1 ], [ 4, 4, -1 ], [ 5, 5, 1 ], 
  [ 6, 6, 1 ], [ 7, 7, 1 ], [ 8, 8, -1 ], [ 9, 9, -1 ], [ 10, 10, -1 ], 
  [ 11, 11, 1 ], [ 12, 12, 1 ], [ 13, 13, 1 ], [ 14, 14, -1 ], 
  [ 15, 15, -1 ], [ 16, 16, -1 ], [ 17, 17, 1 ], [ 18, 18, 1 ], 
  [ 19, 19, -1 ], [ 20, 20, -1 ], [ 21, 21, 1 ], [ 22, 22, -1 ], 
  [ 23, 23, 1 ], [ 24, 24, -1 ], [ 25, 25, -1 ], [ 26, 26, -1 ], 
  [ 27, 27, -1 ], [ 28, 28, -1 ], [ 29, 29, 1 ], [ 30, 30, 1 ], 
  [ 31, 31, 1 ], [ 32, 32, -1 ], [ 33, 33, -1 ], [ 34, 34, -1 ], 
  [ 35, 35, 1 ], [ 36, 36, 1 ], [ 37, 37, 1 ], [ 38, 38, -1 ], 
  [ 39, 39, -1 ], [ 40, 40, -1 ], [ 41, 41, 1 ], [ 42, 42, 1 ], 
  [ 43, 43, -1 ], [ 44, 44, -1 ], [ 45, 45, 1 ], [ 46, 46, -1 ], 
  [ 47, 47, 1 ], [ 48, 48, -1 ], [ 49, 49, 1 ], [ 50, 50, 1 ], 
  [ 51, 51, 1 ], [ 52, 52, 1 ] ] ],

[ [ 2, 2, 0, 2 ],
[ 52, [ 1, 1, -1 ], [ 2, 2, 1 ], [ 3, 3, -1 ], [ 4, 4, 1 ], [ 5, 5, 1 ], 
  [ 6, 6, 1 ], [ 7, 7, -1 ], [ 8, 8, 1 ], [ 9, 9, -1 ], [ 10, 10, 1 ], 
  [ 11, 11, 1 ], [ 12, 12, -1 ], [ 13, 13, 1 ], [ 14, 14, -1 ], 
  [ 15, 15, 1 ], [ 16, 16, 1 ], [ 17, 17, -1 ], [ 18, 18, 1 ], 
  [ 19, 19, 1 ], [ 20, 20, 1 ], [ 21, 21, -1 ], [ 22, 22, 1 ], 
  [ 23, 23, 1 ], [ 24, 24, 1 ], [ 25, 25, -1 ], [ 26, 26, 1 ], 
  [ 27, 27, -1 ], [ 28, 28, 1 ], [ 29, 29, 1 ], [ 30, 30, 1 ], 
  [ 31, 31, -1 ], [ 32, 32, 1 ], [ 33, 33, -1 ], [ 34, 34, 1 ], 
  [ 35, 35, 1 ], [ 36, 36, -1 ], [ 37, 37, 1 ], [ 38, 38, -1 ], 
  [ 39, 39, 1 ], [ 40, 40, 1 ], [ 41, 41, -1 ], [ 42, 42, 1 ], 
  [ 43, 43, 1 ], [ 44, 44, 1 ], [ 45, 45, -1 ], [ 46, 46, 1 ], 
  [ 47, 47, 1 ], [ 48, 48, 1 ], [ 49, 49, 1 ], [ 50, 50, 1 ], [ 51, 51, 1 ], 
  [ 52, 52, 1 ] ] ],

[ [ 0, 0, 0, 2 ],
  [ 52, [ 1, 1, E(3) ], [ 2, 2, E(3) ], [ 3, 3, E(3) ], [ 4, 4, E(3) ], 
  [ 5, 5, E(3)^2 ], [ 6, 6, E(3)^2 ], [ 7, 7, E(3)^2 ], [ 8, 8, 1 ], 
  [ 9, 9, 1 ], [ 10, 10, 1 ], [ 11, 11, E(3) ], [ 12, 12, E(3) ], 
  [ 13, 13, E(3) ], [ 14, 14, E(3)^2 ], [ 15, 15, E(3)^2 ], 
  [ 16, 16, E(3)^2 ], [ 17, 17, 1 ], [ 18, 18, 1 ], [ 19, 19, E(3) ], 
  [ 20, 20, E(3) ], [ 21, 21, E(3)^2 ], [ 22, 22, 1 ], [ 23, 23, E(3) ], 
  [ 24, 24, E(3)^2 ], [ 25, 25, E(3)^2 ], [ 26, 26, E(3)^2 ], 
  [ 27, 27, E(3)^2 ], [ 28, 28, E(3)^2 ], [ 29, 29, E(3) ], [ 30, 30, E(3) ],
  [ 31, 31, E(3) ], [ 32, 32, 1 ], [ 33, 33, 1 ], [ 34, 34, 1 ], 
  [ 35, 35, E(3)^2 ], [ 36, 36, E(3)^2 ], [ 37, 37, E(3)^2 ], 
  [ 38, 38, E(3) ], [ 39, 39, E(3) ], [ 40, 40, E(3) ], [ 41, 41, 1 ], 
  [ 42, 42, 1 ], [ 43, 43, E(3)^2 ], [ 44, 44, E(3)^2 ], [ 45, 45, E(3) ], 
  [ 46, 46, 1 ], [ 47, 47, E(3)^2 ], [ 48, 48, E(3) ], [ 49, 49, 1 ], 
  [ 50, 50, 1 ], [ 51, 51, 1 ], [ 52, 52, 1 ] ],

[ 52, [ 1, 1, 1/9 ], [ 1, 3, 4/9 ], [ 1, 5, -2/9 ], [ 1, 25, 4/9 ], 
  [ 1, 27, -2/9 ], [ 1, 29, 4/9 ], [ 1, 51, 2/3 ], [ 1, 52, -4/9 ], 
  [ 2, 2, 1 ], [ 3, 1, 4/9 ], [ 3, 3, 1/9 ], [ 3, 5, -2/9 ], [ 3, 25, -2/9 ],
  [ 3, 27, 4/9 ], [ 3, 29, 4/9 ], [ 3, 49, -2/3 ], [ 3, 52, 2/9 ], 
  [ 4, 4, 1/9 ], [ 4, 7, -4/9 ], [ 4, 8, -4/9 ], [ 4, 10, 4/9 ], 
  [ 4, 12, 8/9 ], [ 4, 15, 4/9 ], [ 5, 1, -2/9 ], [ 5, 3, -2/9 ], 
  [ 5, 5, 1/9 ], [ 5, 25, 4/9 ], [ 5, 27, 4/9 ], [ 5, 29, 4/9 ], 
  [ 5, 49, 2/3 ], [ 5, 51, -2/3 ], [ 5, 52, 2/9 ], [ 6, 6, 1/9 ], 
  [ 6, 9, -4/9 ], [ 6, 11, -4/9 ], [ 6, 13, 4/9 ], [ 6, 14, 8/9 ], 
  [ 6, 18, 4/9 ], [ 7, 4, -2/9 ], [ 7, 7, 5/9 ], [ 7, 8, 2/9 ], 
  [ 7, 10, -2/9 ], [ 7, 12, 2/9 ], [ 7, 15, 4/9 ], [ 8, 4, -2/9 ], 
  [ 8, 7, 2/9 ], [ 8, 8, 5/9 ], [ 8, 10, 4/9 ], [ 8, 12, 2/9 ], 
  [ 8, 15, -2/9 ], [ 9, 6, -2/9 ], [ 9, 9, 5/9 ], [ 9, 11, 2/9 ], 
  [ 9, 13, -2/9 ], [ 9, 14, 2/9 ], [ 9, 18, 4/9 ], [ 10, 4, 4/9 ], 
  [ 10, 7, -4/9 ], [ 10, 8, 8/9 ], [ 10, 10, 1/9 ], [ 10, 12, -4/9 ], 
  [ 10, 15, 4/9 ], [ 11, 6, -2/9 ], [ 11, 9, 2/9 ], [ 11, 11, 5/9 ], 
  [ 11, 13, 4/9 ], [ 11, 14, 2/9 ], [ 11, 18, -2/9 ], [ 12, 4, 4/9 ], 
  [ 12, 7, 2/9 ], [ 12, 8, 2/9 ], [ 12, 10, -2/9 ], [ 12, 12, 5/9 ], 
  [ 12, 15, -2/9 ], [ 13, 6, 4/9 ], [ 13, 9, -4/9 ], [ 13, 11, 8/9 ], 
  [ 13, 13, 1/9 ], [ 13, 14, -4/9 ], [ 13, 18, 4/9 ], [ 14, 6, 4/9 ], 
  [ 14, 9, 2/9 ], [ 14, 11, 2/9 ], [ 14, 13, -2/9 ], [ 14, 14, 5/9 ], 
  [ 14, 18, -2/9 ], [ 15, 4, 4/9 ], [ 15, 7, 8/9 ], [ 15, 8, -4/9 ], 
  [ 15, 10, 4/9 ], [ 15, 12, -4/9 ], [ 15, 15, 1/9 ], [ 16, 16, 1/9 ], 
  [ 16, 17, 4/9 ], [ 16, 19, -4/9 ], [ 16, 20, 4/9 ], [ 16, 21, -8/9 ], 
  [ 16, 22, 4/9 ], [ 17, 16, 2/9 ], [ 17, 17, 5/9 ], [ 17, 19, -2/9 ], 
  [ 17, 20, 2/9 ], [ 17, 21, 2/9 ], [ 17, 22, -4/9 ], [ 18, 6, 4/9 ], 
  [ 18, 9, 8/9 ], [ 18, 11, -4/9 ], [ 18, 13, 4/9 ], [ 18, 14, -4/9 ], 
  [ 18, 18, 1/9 ], [ 19, 16, -2/9 ], [ 19, 17, -2/9 ], [ 19, 19, 5/9 ], 
  [ 19, 20, 4/9 ], [ 19, 21, -2/9 ], [ 19, 22, -2/9 ], [ 20, 16, 4/9 ], 
  [ 20, 17, 4/9 ], [ 20, 19, 8/9 ], [ 20, 20, 1/9 ], [ 20, 21, 4/9 ], 
  [ 20, 22, 4/9 ], [ 21, 16, -4/9 ], [ 21, 17, 2/9 ], [ 21, 19, -2/9 ], 
  [ 21, 20, 2/9 ], [ 21, 21, 5/9 ], [ 21, 22, 2/9 ], [ 22, 16, 4/9 ], 
  [ 22, 17, -8/9 ], [ 22, 19, -4/9 ], [ 22, 20, 4/9 ], [ 22, 21, 4/9 ], 
  [ 22, 22, 1/9 ], [ 23, 23, 1 ], [ 24, 24, 1 ], [ 25, 1, 4/9 ], 
  [ 25, 3, -2/9 ], [ 25, 5, 4/9 ], [ 25, 25, 1/9 ], [ 25, 27, 4/9 ], 
  [ 25, 29, -2/9 ], [ 25, 51, 2/3 ], [ 25, 52, -4/9 ], [ 26, 26, 1 ], 
  [ 27, 1, -2/9 ], [ 27, 3, 4/9 ], [ 27, 5, 4/9 ], [ 27, 25, 4/9 ], 
  [ 27, 27, 1/9 ], [ 27, 29, -2/9 ], [ 27, 49, -2/3 ], [ 27, 52, 2/9 ], 
  [ 28, 28, 1/9 ], [ 28, 31, -4/9 ], [ 28, 32, -4/9 ], [ 28, 34, 4/9 ], 
  [ 28, 36, 8/9 ], [ 28, 39, 4/9 ], [ 29, 1, 4/9 ], [ 29, 3, 4/9 ], 
  [ 29, 5, 4/9 ], [ 29, 25, -2/9 ], [ 29, 27, -2/9 ], [ 29, 29, 1/9 ], 
  [ 29, 49, 2/3 ], [ 29, 51, -2/3 ], [ 29, 52, 2/9 ], [ 30, 30, 1/9 ], 
  [ 30, 33, -4/9 ], [ 30, 35, -4/9 ], [ 30, 37, 4/9 ], [ 30, 38, 8/9 ], 
  [ 30, 42, 4/9 ], [ 31, 28, -2/9 ], [ 31, 31, 5/9 ], [ 31, 32, 2/9 ], 
  [ 31, 34, -2/9 ], [ 31, 36, 2/9 ], [ 31, 39, 4/9 ], [ 32, 28, -2/9 ], 
  [ 32, 31, 2/9 ], [ 32, 32, 5/9 ], [ 32, 34, 4/9 ], [ 32, 36, 2/9 ], 
  [ 32, 39, -2/9 ], [ 33, 30, -2/9 ], [ 33, 33, 5/9 ], [ 33, 35, 2/9 ], 
  [ 33, 37, -2/9 ], [ 33, 38, 2/9 ], [ 33, 42, 4/9 ], [ 34, 28, 4/9 ], 
  [ 34, 31, -4/9 ], [ 34, 32, 8/9 ], [ 34, 34, 1/9 ], [ 34, 36, -4/9 ], 
  [ 34, 39, 4/9 ], [ 35, 30, -2/9 ], [ 35, 33, 2/9 ], [ 35, 35, 5/9 ], 
  [ 35, 37, 4/9 ], [ 35, 38, 2/9 ], [ 35, 42, -2/9 ], [ 36, 28, 4/9 ], 
  [ 36, 31, 2/9 ], [ 36, 32, 2/9 ], [ 36, 34, -2/9 ], [ 36, 36, 5/9 ], 
  [ 36, 39, -2/9 ], [ 37, 30, 4/9 ], [ 37, 33, -4/9 ], [ 37, 35, 8/9 ], 
  [ 37, 37, 1/9 ], [ 37, 38, -4/9 ], [ 37, 42, 4/9 ], [ 38, 30, 4/9 ], 
  [ 38, 33, 2/9 ], [ 38, 35, 2/9 ], [ 38, 37, -2/9 ], [ 38, 38, 5/9 ], 
  [ 38, 42, -2/9 ], [ 39, 28, 4/9 ], [ 39, 31, 8/9 ], [ 39, 32, -4/9 ], 
  [ 39, 34, 4/9 ], [ 39, 36, -4/9 ], [ 39, 39, 1/9 ], [ 40, 40, 1/9 ], 
  [ 40, 41, 4/9 ], [ 40, 43, -4/9 ], [ 40, 44, 4/9 ], [ 40, 45, -8/9 ], 
  [ 40, 46, 4/9 ], [ 41, 40, 2/9 ], [ 41, 41, 5/9 ], [ 41, 43, -2/9 ], 
  [ 41, 44, 2/9 ], [ 41, 45, 2/9 ], [ 41, 46, -4/9 ], [ 42, 30, 4/9 ], 
  [ 42, 33, 8/9 ], [ 42, 35, -4/9 ], [ 42, 37, 4/9 ], [ 42, 38, -4/9 ], 
  [ 42, 42, 1/9 ], [ 43, 40, -2/9 ], [ 43, 41, -2/9 ], [ 43, 43, 5/9 ], 
  [ 43, 44, 4/9 ], [ 43, 45, -2/9 ], [ 43, 46, -2/9 ], [ 44, 40, 4/9 ], 
  [ 44, 41, 4/9 ], [ 44, 43, 8/9 ], [ 44, 44, 1/9 ], [ 44, 45, 4/9 ], 
  [ 44, 46, 4/9 ], [ 45, 40, -4/9 ], [ 45, 41, 2/9 ], [ 45, 43, -2/9 ], 
  [ 45, 44, 2/9 ], [ 45, 45, 5/9 ], [ 45, 46, 2/9 ], [ 46, 40, 4/9 ], 
  [ 46, 41, -8/9 ], [ 46, 43, -4/9 ], [ 46, 44, 4/9 ], [ 46, 45, 4/9 ], 
  [ 46, 46, 1/9 ], [ 47, 47, 1 ], [ 48, 48, 1 ], [ 49, 1, 2/9 ], 
  [ 49, 3, -4/9 ], [ 49, 5, 2/9 ], [ 49, 25, 2/9 ], [ 49, 27, -4/9 ], 
  [ 49, 29, 2/9 ], [ 49, 49, -1/3 ], [ 49, 52, 4/9 ], [ 50, 50, 1 ], 
  [ 51, 1, 4/9 ], [ 51, 3, -2/9 ], [ 51, 5, -2/9 ], [ 51, 25, 4/9 ], 
  [ 51, 27, -2/9 ], [ 51, 29, -2/9 ], [ 51, 51, -1/3 ], [ 51, 52, 8/9 ], 
  [ 52, 52, 1 ] ],

[ 52, [ 1, 25, 1 ], [ 2, 26, 1 ], [ 3, 5, 1 ], [ 4, 6, 1 ], [ 5, 3, 1 ], 
  [ 6, 4, 1 ], [ 7, 11, 1 ], [ 8, 9, 1 ], [ 9, 8, 1 ], [ 10, 18, 1 ], 
  [ 11, 7, 1 ], [ 12, 14, 1 ], [ 13, 15, 1 ], [ 14, 12, 1 ], [ 15, 13, 1 ], 
  [ 16, 20, -1 ], [ 17, 17, -1 ], [ 18, 10, 1 ], [ 19, 21, 1 ], 
  [ 20, 16, -1 ], [ 21, 19, 1 ], [ 22, 22, -1 ], [ 23, 24, 1 ], 
  [ 24, 23, 1 ], [ 25, 1, 1 ], [ 26, 2, 1 ], [ 27, 29, 1 ], [ 28, 30, 1 ], 
  [ 29, 27, 1 ], [ 30, 28, 1 ], [ 31, 35, 1 ], [ 32, 33, 1 ], [ 33, 32, 1 ], 
  [ 34, 42, 1 ], [ 35, 31, 1 ], [ 36, 38, 1 ], [ 37, 39, 1 ], [ 38, 36, 1 ], 
  [ 39, 37, 1 ], [ 40, 44, -1 ], [ 41, 41, -1 ], [ 42, 34, 1 ], 
  [ 43, 45, 1 ], [ 44, 40, -1 ], [ 45, 43, 1 ], [ 46, 46, -1 ], 
  [ 47, 48, 1 ], [ 48, 47, 1 ], [ 49, 49, -1 ], [ 49, 51, 1 ], 
  [ 50, 50, -1 ], [ 50, 52, 1 ], [ 51, 51, 1 ], [ 52, 52, 1 ] ] ]
      
],

[ "E", 6, [  [ 0, 2, 0, 0, 0, 0 ],  
[ 78, [ 1, 37, 1 ], [ 2, 8, 1 ], [ 3, 12, 1 ], [ 4, 40, 1 ], [ 5, 16, 1 ], [ 6, 42, 1 ], [ 7, 9, 1 ], [ 8, 2, 1 ], [ 9, 7, 1 ], 
  [ 10, 11, 1 ], [ 11, 10, 1 ], [ 12, 3, 1 ], [ 13, 17, -1 ], [ 14, 20, 1 ], [ 15, 23, 1 ], [ 16, 5, 1 ], [ 17, 13, -1 ], [ 18, 21, 1 ], 
  [ 19, 30, 1 ], [ 20, 14, 1 ], [ 21, 18, 1 ], [ 22, 28, 1 ], [ 23, 15, 1 ], [ 24, 27, 1 ], [ 25, 26, 1 ], [ 26, 25, 1 ], [ 27, 24, 1 ], 
  [ 28, 22, 1 ], [ 29, 32, 1 ], [ 30, 19, 1 ], [ 31, 33, -1 ], [ 32, 29, 1 ], [ 33, 31, -1 ], [ 34, 35, 1 ], [ 35, 34, 1 ], [ 36, 36, -1 ], 
  [ 37, 1, 1 ], [ 38, 44, 1 ], [ 39, 48, 1 ], [ 40, 4, 1 ], [ 41, 52, 1 ], [ 42, 6, 1 ], [ 43, 45, 1 ], [ 44, 38, 1 ], [ 45, 43, 1 ], 
  [ 46, 47, 1 ], [ 47, 46, 1 ], [ 48, 39, 1 ], [ 49, 53, -1 ], [ 50, 56, 1 ], [ 51, 59, 1 ], [ 52, 41, 1 ], [ 53, 49, -1 ], [ 54, 57, 1 ], 
  [ 55, 66, 1 ], [ 56, 50, 1 ], [ 57, 54, 1 ], [ 58, 64, 1 ], [ 59, 51, 1 ], [ 60, 63, 1 ], [ 61, 62, 1 ], [ 62, 61, 1 ], [ 63, 60, 1 ], 
  [ 64, 58, 1 ], [ 65, 68, 1 ], [ 66, 55, 1 ], [ 67, 69, -1 ], [ 68, 65, 1 ], [ 69, 67, -1 ], [ 70, 71, 1 ], [ 71, 70, 1 ], [ 72, 72, -1 ], 
  [ 73, 73, -1 ], [ 73, 75, 1 ], [ 74, 74, 1 ], [ 75, 75, 1 ], [ 76, 74, 1 ], [ 76, 75, 1 ], [ 76, 76, -1 ], [ 76, 77, 1 ], [ 77, 77, 1 ], 
  [ 78, 77, 1 ], [ 78, 78, -1 ] ] ],

[ [ 0, 0, 0, 2, 0, 0 ],
  [ 78, [ 1, 37, -1 ], [ 2, 2, -1 ], [ 3, 7, -E(4) ], [ 4, 10, E(4) ], [ 5, 41, -1 ], [ 6, 11, E(4) ], [ 7, 3, E(4) ], 
  [ 8, 14, -E(4) ], [ 9, 18, 1 ], [ 10, 4, -E(4) ], [ 11, 6, -E(4) ], [ 12, 15, -1 ], [ 13, 22, -1 ], [ 14, 8, E(4) ], 
  [ 15, 12, -1 ], [ 16, 16, 1 ], [ 17, 19, 1 ], [ 18, 9, 1 ], [ 19, 17, 1 ], [ 20, 20, -1 ], [ 21, 23, -E(4) ], 
  [ 22, 13, -1 ], [ 23, 21, E(4) ], [ 24, 26, -E(4) ], [ 25, 27, E(4) ], [ 26, 24, E(4) ], [ 27, 25, -E(4) ], 
  [ 28, 33, -1 ], [ 29, 29, -1 ], [ 30, 31, 1 ], [ 31, 30, 1 ], [ 32, 34, E(4) ], [ 33, 28, -1 ], [ 34, 32, -E(4) ], 
  [ 35, 35, -1 ], [ 36, 36, 1 ], [ 37, 1, -1 ], [ 38, 38, -1 ], [ 39, 43, E(4) ], [ 40, 46, -E(4) ], [ 41, 5, -1 ], 
  [ 42, 47, -E(4) ], [ 43, 39, -E(4) ], [ 44, 50, E(4) ], [ 45, 54, 1 ], [ 46, 40, E(4) ], [ 47, 42, E(4) ], 
  [ 48, 51, -1 ], [ 49, 58, -1 ], [ 50, 44, -E(4) ], [ 51, 48, -1 ], [ 52, 52, 1 ], [ 53, 55, 1 ], [ 54, 45, 1 ], 
  [ 55, 53, 1 ], [ 56, 56, -1 ], [ 57, 59, E(4) ], [ 58, 49, -1 ], [ 59, 57, -E(4) ], [ 60, 62, E(4) ], 
  [ 61, 63, -E(4) ], [ 62, 60, -E(4) ], [ 63, 61, E(4) ], [ 64, 69, -1 ], [ 65, 65, -1 ], [ 66, 67, 1 ], [ 67, 66, 1 ], 
  [ 68, 70, -E(4) ], [ 69, 64, -1 ], [ 70, 68, E(4) ], [ 71, 71, -1 ], [ 72, 72, 1 ], [ 73, 73, -1 ], [ 73, 75, 1 ], 
  [ 74, 74, 1 ], [ 75, 75, 1 ], [ 76, 76, 1 ], [ 77, 76, 1 ], [ 77, 77, -1 ], [ 77, 78, 1 ], [ 78, 78, 1 ] ],
  [ 78, [ 1, 1, -1/4 ], [ 1, 3, -1/4 ], [ 1, 7, 1/4 ], [ 1, 37, -1/4 ], [ 1, 39, 1/2 ], [ 1, 43, -1/2 ], [ 1, 75, -3/4 ], 
  [ 1, 76, 1/2 ], [ 2, 2, 1/4 ], [ 2, 38, -1/4 ], [ 2, 74, -1/2 ], [ 2, 76, 1/4 ], [ 3, 1, -1/2*E(4) ], 
  [ 3, 37, 1/2*E(4) ], [ 3, 39, -E(4) ], [ 3, 43, -E(4) ], [ 3, 73, E(4) ], [ 3, 75, -1/2*E(4) ], [ 4, 9, 1/8*E(4) ], 
  [ 4, 12, 1/8*E(4) ], [ 4, 13, 3/8*E(4) ], [ 4, 15, -1/8*E(4) ], [ 4, 17, 3/8*E(4) ], [ 4, 18, -1/8*E(4) ], 
  [ 4, 19, -3/8*E(4) ], [ 4, 21, 1/4*E(4) ], [ 4, 22, -3/8*E(4) ], [ 4, 23, 1/4*E(4) ], [ 4, 25, 3/4*E(4) ], 
  [ 4, 27, 3/4*E(4) ], [ 5, 5, -1/4 ], [ 5, 6, 1/2 ], [ 5, 11, -1/2 ], [ 5, 41, -1/4 ], [ 5, 42, -1/4 ], 
  [ 5, 47, 1/4 ], [ 5, 76, 1/4 ], [ 5, 78, -3/4 ], [ 6, 5, -1/4*E(4) ], [ 6, 41, 1/4*E(4) ], [ 6, 42, 1/4*E(4) ], 
  [ 6, 47, 1/4*E(4) ], [ 6, 76, -1/4*E(4) ], [ 6, 77, 1/2*E(4) ], [ 6, 78, -1/4*E(4) ], [ 7, 1, -1/2*E(4) ], 
  [ 7, 37, 1/2*E(4) ], [ 7, 39, E(4) ], [ 7, 43, E(4) ], [ 7, 73, E(4) ], [ 7, 75, -1/2*E(4) ], [ 8, 9, -1/8*E(4) ], 
  [ 8, 12, -1/8*E(4) ], [ 8, 13, 1/8*E(4) ], [ 8, 15, 1/8*E(4) ], [ 8, 17, 1/8*E(4) ], [ 8, 18, 1/8*E(4) ], 
  [ 8, 19, -1/8*E(4) ], [ 8, 21, -1/4*E(4) ], [ 8, 22, -1/8*E(4) ], [ 8, 23, -1/4*E(4) ], [ 8, 25, 1/4*E(4) ], 
  [ 8, 27, 1/4*E(4) ], [ 9, 4, 1/4 ], [ 9, 8, 3/4 ], [ 9, 9, -1/8 ], [ 9, 10, -1/4 ], [ 9, 12, 1/8 ], [ 9, 13, -3/8 ], 
  [ 9, 14, -3/4 ], [ 9, 15, 1/8 ], [ 9, 16, 1/2 ], [ 9, 17, 3/8 ], [ 9, 18, -1/8 ], [ 9, 19, 3/8 ], [ 9, 20, 3/2 ], 
  [ 9, 21, -1/4 ], [ 9, 22, -3/8 ], [ 9, 23, 1/4 ], [ 9, 25, -3/4 ], [ 9, 27, 3/4 ], [ 10, 9, 1/8*E(4) ], 
  [ 10, 12, 1/8*E(4) ], [ 10, 13, 3/8*E(4) ], [ 10, 15, -1/8*E(4) ], [ 10, 17, 3/8*E(4) ], [ 10, 18, -1/8*E(4) ], 
  [ 10, 19, -3/8*E(4) ], [ 10, 21, -1/4*E(4) ], [ 10, 22, -3/8*E(4) ], [ 10, 23, -1/4*E(4) ], [ 10, 25, -3/4*E(4) ], 
  [ 10, 27, -3/4*E(4) ], [ 11, 5, -1/4*E(4) ], [ 11, 41, 1/4*E(4) ], [ 11, 42, -1/4*E(4) ], [ 11, 47, -1/4*E(4) ], 
  [ 11, 76, -1/4*E(4) ], [ 11, 77, 1/2*E(4) ], [ 11, 78, -1/4*E(4) ], [ 12, 4, -1/4 ], [ 12, 8, -3/4 ], [ 12, 9, -1/8 ],
  [ 12, 10, 1/4 ], [ 12, 12, 1/8 ], [ 12, 13, -3/8 ], [ 12, 14, 3/4 ], [ 12, 15, 1/8 ], [ 12, 16, -1/2 ], 
  [ 12, 17, 3/8 ], [ 12, 18, -1/8 ], [ 12, 19, 3/8 ], [ 12, 20, -3/2 ], [ 12, 21, -1/4 ], [ 12, 22, -3/8 ], 
  [ 12, 23, 1/4 ], [ 12, 25, -3/4 ], [ 12, 27, 3/4 ], [ 13, 4, -1/4 ], [ 13, 8, 1/4 ], [ 13, 9, 1/8 ], [ 13, 10, 1/4 ], 
  [ 13, 12, -1/8 ], [ 13, 13, -1/8 ], [ 13, 14, -1/4 ], [ 13, 15, -1/8 ], [ 13, 16, -1/2 ], [ 13, 17, 1/8 ], 
  [ 13, 18, 1/8 ], [ 13, 19, 1/8 ], [ 13, 20, 1/2 ], [ 13, 21, 1/4 ], [ 13, 22, -1/8 ], [ 13, 23, -1/4 ], 
  [ 13, 25, -1/4 ], [ 13, 27, 1/4 ], [ 14, 9, -1/8*E(4) ], [ 14, 12, -1/8*E(4) ], [ 14, 13, 1/8*E(4) ], 
  [ 14, 15, 1/8*E(4) ], [ 14, 17, 1/8*E(4) ], [ 14, 18, 1/8*E(4) ], [ 14, 19, -1/8*E(4) ], [ 14, 21, 1/4*E(4) ], 
  [ 14, 22, -1/8*E(4) ], [ 14, 23, 1/4*E(4) ], [ 14, 25, -1/4*E(4) ], [ 14, 27, -1/4*E(4) ], [ 15, 4, 1/4 ], 
  [ 15, 8, 3/4 ], [ 15, 9, -1/8 ], [ 15, 10, -1/4 ], [ 15, 12, 1/8 ], [ 15, 13, -3/8 ], [ 15, 14, -3/4 ], 
  [ 15, 15, 1/8 ], [ 15, 16, -1/2 ], [ 15, 17, 3/8 ], [ 15, 18, -1/8 ], [ 15, 19, 3/8 ], [ 15, 20, -3/2 ], 
  [ 15, 21, 1/4 ], [ 15, 22, -3/8 ], [ 15, 23, -1/4 ], [ 15, 25, 3/4 ], [ 15, 27, -3/4 ], [ 16, 9, 1/8 ], 
  [ 16, 12, 1/8 ], [ 16, 13, 3/8 ], [ 16, 15, 1/8 ], [ 16, 17, 3/8 ], [ 16, 18, 1/8 ], [ 16, 19, 3/8 ], [ 16, 22, 3/8 ],
  [ 17, 4, 1/4 ], [ 17, 8, -1/4 ], [ 17, 9, 1/8 ], [ 17, 10, -1/4 ], [ 17, 12, -1/8 ], [ 17, 13, -1/8 ], 
  [ 17, 14, 1/4 ], [ 17, 15, -1/8 ], [ 17, 16, 1/2 ], [ 17, 17, 1/8 ], [ 17, 18, 1/8 ], [ 17, 19, 1/8 ], 
  [ 17, 20, -1/2 ], [ 17, 21, 1/4 ], [ 17, 22, -1/8 ], [ 17, 23, -1/4 ], [ 17, 25, -1/4 ], [ 17, 27, 1/4 ], 
  [ 18, 4, -1/4 ], [ 18, 8, -3/4 ], [ 18, 9, -1/8 ], [ 18, 10, 1/4 ], [ 18, 12, 1/8 ], [ 18, 13, -3/8 ], 
  [ 18, 14, 3/4 ], [ 18, 15, 1/8 ], [ 18, 16, 1/2 ], [ 18, 17, 3/8 ], [ 18, 18, -1/8 ], [ 18, 19, 3/8 ], 
  [ 18, 20, 3/2 ], [ 18, 21, 1/4 ], [ 18, 22, -3/8 ], [ 18, 23, -1/4 ], [ 18, 25, 3/4 ], [ 18, 27, -3/4 ], 
  [ 19, 4, -1/4 ], [ 19, 8, 1/4 ], [ 19, 9, 1/8 ], [ 19, 10, 1/4 ], [ 19, 12, -1/8 ], [ 19, 13, -1/8 ], 
  [ 19, 14, -1/4 ], [ 19, 15, -1/8 ], [ 19, 16, 1/2 ], [ 19, 17, 1/8 ], [ 19, 18, 1/8 ], [ 19, 19, 1/8 ], 
  [ 19, 20, -1/2 ], [ 19, 21, -1/4 ], [ 19, 22, -1/8 ], [ 19, 23, 1/4 ], [ 19, 25, 1/4 ], [ 19, 27, -1/4 ], 
  [ 20, 9, -1/8 ], [ 20, 12, -1/8 ], [ 20, 13, 1/8 ], [ 20, 15, -1/8 ], [ 20, 17, 1/8 ], [ 20, 18, -1/8 ], 
  [ 20, 19, 1/8 ], [ 20, 22, 1/8 ], [ 21, 4, -1/4*E(4) ], [ 21, 8, -3/4*E(4) ], [ 21, 9, 1/8*E(4) ], 
  [ 21, 10, -1/4*E(4) ], [ 21, 12, -1/8*E(4) ], [ 21, 13, 3/8*E(4) ], [ 21, 14, -3/4*E(4) ], [ 21, 15, 1/8*E(4) ], 
  [ 21, 17, -3/8*E(4) ], [ 21, 18, -1/8*E(4) ], [ 21, 19, 3/8*E(4) ], [ 21, 22, -3/8*E(4) ], [ 22, 4, 1/4 ], 
  [ 22, 8, -1/4 ], [ 22, 9, 1/8 ], [ 22, 10, -1/4 ], [ 22, 12, -1/8 ], [ 22, 13, -1/8 ], [ 22, 14, 1/4 ], 
  [ 22, 15, -1/8 ], [ 22, 16, -1/2 ], [ 22, 17, 1/8 ], [ 22, 18, 1/8 ], [ 22, 19, 1/8 ], [ 22, 20, 1/2 ], 
  [ 22, 21, -1/4 ], [ 22, 22, -1/8 ], [ 22, 23, 1/4 ], [ 22, 25, 1/4 ], [ 22, 27, -1/4 ], [ 23, 4, 1/4*E(4) ], 
  [ 23, 8, 3/4*E(4) ], [ 23, 9, 1/8*E(4) ], [ 23, 10, 1/4*E(4) ], [ 23, 12, -1/8*E(4) ], [ 23, 13, 3/8*E(4) ], 
  [ 23, 14, 3/4*E(4) ], [ 23, 15, 1/8*E(4) ], [ 23, 17, -3/8*E(4) ], [ 23, 18, -1/8*E(4) ], [ 23, 19, 3/8*E(4) ], 
  [ 23, 22, -3/8*E(4) ], [ 24, 28, 1/2*E(4) ], [ 24, 30, 1/2*E(4) ], [ 24, 31, -1/2*E(4) ], [ 24, 32, -1/2*E(4) ], 
  [ 24, 33, -1/2*E(4) ], [ 24, 34, 1/2*E(4) ], [ 25, 4, 1/4*E(4) ], [ 25, 8, -1/4*E(4) ], [ 25, 9, -1/8*E(4) ], 
  [ 25, 10, 1/4*E(4) ], [ 25, 12, 1/8*E(4) ], [ 25, 13, 1/8*E(4) ], [ 25, 14, -1/4*E(4) ], [ 25, 15, -1/8*E(4) ], 
  [ 25, 17, -1/8*E(4) ], [ 25, 18, 1/8*E(4) ], [ 25, 19, 1/8*E(4) ], [ 25, 22, -1/8*E(4) ], [ 26, 28, -1/2*E(4) ], 
  [ 26, 30, -1/2*E(4) ], [ 26, 31, 1/2*E(4) ], [ 26, 32, -1/2*E(4) ], [ 26, 33, 1/2*E(4) ], [ 26, 34, 1/2*E(4) ], 
  [ 27, 4, -1/4*E(4) ], [ 27, 8, 1/4*E(4) ], [ 27, 9, -1/8*E(4) ], [ 27, 10, -1/4*E(4) ], [ 27, 12, 1/8*E(4) ], 
  [ 27, 13, 1/8*E(4) ], [ 27, 14, 1/4*E(4) ], [ 27, 15, -1/8*E(4) ], [ 27, 17, -1/8*E(4) ], [ 27, 18, 1/8*E(4) ], 
  [ 27, 19, 1/8*E(4) ], [ 27, 22, -1/8*E(4) ], [ 28, 24, -1/4 ], [ 28, 26, -1/4 ], [ 28, 28, 1/4 ], [ 28, 29, 1/4 ], 
  [ 28, 30, 1/4 ], [ 28, 31, 1/4 ], [ 28, 32, -1/4 ], [ 28, 33, 1/4 ], [ 28, 34, -1/4 ], [ 29, 28, 1 ], [ 29, 30, -1 ], 
  [ 29, 31, -1 ], [ 29, 33, 1 ], [ 30, 24, 1/4 ], [ 30, 26, 1/4 ], [ 30, 28, -1/4 ], [ 30, 29, 1/4 ], [ 30, 30, -1/4 ], 
  [ 30, 31, -1/4 ], [ 30, 32, -1/4 ], [ 30, 33, -1/4 ], [ 30, 34, -1/4 ], [ 31, 24, -1/4 ], [ 31, 26, -1/4 ], 
  [ 31, 28, -1/4 ], [ 31, 29, 1/4 ], [ 31, 30, -1/4 ], [ 31, 31, -1/4 ], [ 31, 32, 1/4 ], [ 31, 33, -1/4 ], 
  [ 31, 34, 1/4 ], [ 32, 24, 1/2*E(4) ], [ 32, 26, -1/2*E(4) ], [ 32, 28, -1/2*E(4) ], [ 32, 30, 1/2*E(4) ], 
  [ 32, 31, -1/2*E(4) ], [ 32, 33, 1/2*E(4) ], [ 33, 24, 1/4 ], [ 33, 26, 1/4 ], [ 33, 28, 1/4 ], [ 33, 29, 1/4 ], 
  [ 33, 30, 1/4 ], [ 33, 31, 1/4 ], [ 33, 32, 1/4 ], [ 33, 33, 1/4 ], [ 33, 34, 1/4 ], [ 34, 24, 1/2*E(4) ], 
  [ 34, 26, -1/2*E(4) ], [ 34, 28, 1/2*E(4) ], [ 34, 30, -1/2*E(4) ], [ 34, 31, 1/2*E(4) ], [ 34, 33, -1/2*E(4) ], 
  [ 35, 35, -1/2 ], [ 35, 36, 3/2 ], [ 36, 35, -1/2 ], [ 36, 36, -1/2 ], [ 37, 1, -1/4 ], [ 37, 3, 1/4 ], 
  [ 37, 7, -1/4 ], [ 37, 37, -1/4 ], [ 37, 39, -1/2 ], [ 37, 43, 1/2 ], [ 37, 75, -3/4 ], [ 37, 76, 1/2 ], 
  [ 38, 2, -9/4 ], [ 38, 38, 1/4 ], [ 38, 74, -3/2 ], [ 38, 76, 3/4 ], [ 39, 1, -1/4*E(4) ], [ 39, 3, 1/4*E(4) ], 
  [ 39, 7, 1/4*E(4) ], [ 39, 37, 1/4*E(4) ], [ 39, 73, -1/2*E(4) ], [ 39, 75, 1/4*E(4) ], [ 40, 45, -1/4*E(4) ], 
  [ 40, 48, -1/4*E(4) ], [ 40, 49, -1/4*E(4) ], [ 40, 51, 1/4*E(4) ], [ 40, 53, -1/4*E(4) ], [ 40, 54, 1/4*E(4) ], 
  [ 40, 55, 1/4*E(4) ], [ 40, 57, -1/4*E(4) ], [ 40, 58, 1/4*E(4) ], [ 40, 59, -1/4*E(4) ], [ 40, 61, -1/4*E(4) ], 
  [ 40, 63, -1/4*E(4) ], [ 41, 5, -1/4 ], [ 41, 6, -1/2 ], [ 41, 11, 1/2 ], [ 41, 41, -1/4 ], [ 41, 42, 1/4 ], 
  [ 41, 47, -1/4 ], [ 41, 76, 1/4 ], [ 41, 78, -3/4 ], [ 42, 5, -1/2*E(4) ], [ 42, 6, -E(4) ], [ 42, 11, -E(4) ], 
  [ 42, 41, 1/2*E(4) ], [ 42, 76, 1/2*E(4) ], [ 42, 77, -E(4) ], [ 42, 78, 1/2*E(4) ], [ 43, 1, -1/4*E(4) ], 
  [ 43, 3, -1/4*E(4) ], [ 43, 7, -1/4*E(4) ], [ 43, 37, 1/4*E(4) ], [ 43, 73, -1/2*E(4) ], [ 43, 75, 1/4*E(4) ], 
  [ 44, 45, 3/4*E(4) ], [ 44, 48, 3/4*E(4) ], [ 44, 49, -1/4*E(4) ], [ 44, 51, -3/4*E(4) ], [ 44, 53, -1/4*E(4) ], 
  [ 44, 54, -3/4*E(4) ], [ 44, 55, 1/4*E(4) ], [ 44, 57, 3/4*E(4) ], [ 44, 58, 1/4*E(4) ], [ 44, 59, 3/4*E(4) ], 
  [ 44, 61, -1/4*E(4) ], [ 44, 63, -1/4*E(4) ], [ 45, 40, 1/8 ], [ 45, 44, 1/8 ], [ 45, 45, -1/8 ], [ 45, 46, -1/8 ], 
  [ 45, 48, 1/8 ], [ 45, 49, -1/8 ], [ 45, 50, -1/8 ], [ 45, 51, 1/8 ], [ 45, 52, 1/8 ], [ 45, 53, 1/8 ], 
  [ 45, 54, -1/8 ], [ 45, 55, 1/8 ], [ 45, 56, 1/8 ], [ 45, 57, -1/8 ], [ 45, 58, -1/8 ], [ 45, 59, 1/8 ], 
  [ 45, 61, -1/8 ], [ 45, 63, 1/8 ], [ 46, 45, -1/4*E(4) ], [ 46, 48, -1/4*E(4) ], [ 46, 49, -1/4*E(4) ], 
  [ 46, 51, 1/4*E(4) ], [ 46, 53, -1/4*E(4) ], [ 46, 54, 1/4*E(4) ], [ 46, 55, 1/4*E(4) ], [ 46, 57, 1/4*E(4) ], 
  [ 46, 58, 1/4*E(4) ], [ 46, 59, 1/4*E(4) ], [ 46, 61, 1/4*E(4) ], [ 46, 63, 1/4*E(4) ], [ 47, 5, -1/2*E(4) ], 
  [ 47, 6, E(4) ], [ 47, 11, E(4) ], [ 47, 41, 1/2*E(4) ], [ 47, 76, 1/2*E(4) ], [ 47, 77, -E(4) ], 
  [ 47, 78, 1/2*E(4) ], [ 48, 40, -1/8 ], [ 48, 44, -1/8 ], [ 48, 45, -1/8 ], [ 48, 46, 1/8 ], [ 48, 48, 1/8 ], 
  [ 48, 49, -1/8 ], [ 48, 50, 1/8 ], [ 48, 51, 1/8 ], [ 48, 52, -1/8 ], [ 48, 53, 1/8 ], [ 48, 54, -1/8 ], 
  [ 48, 55, 1/8 ], [ 48, 56, -1/8 ], [ 48, 57, -1/8 ], [ 48, 58, -1/8 ], [ 48, 59, 1/8 ], [ 48, 61, -1/8 ], 
  [ 48, 63, 1/8 ], [ 49, 40, -3/8 ], [ 49, 44, 1/8 ], [ 49, 45, 3/8 ], [ 49, 46, 3/8 ], [ 49, 48, -3/8 ], 
  [ 49, 49, -1/8 ], [ 49, 50, -1/8 ], [ 49, 51, -3/8 ], [ 49, 52, -3/8 ], [ 49, 53, 1/8 ], [ 49, 54, 3/8 ], 
  [ 49, 55, 1/8 ], [ 49, 56, 1/8 ], [ 49, 57, 3/8 ], [ 49, 58, -1/8 ], [ 49, 59, -3/8 ], [ 49, 61, -1/8 ], 
  [ 49, 63, 1/8 ], [ 50, 45, 3/4*E(4) ], [ 50, 48, 3/4*E(4) ], [ 50, 49, -1/4*E(4) ], [ 50, 51, -3/4*E(4) ], 
  [ 50, 53, -1/4*E(4) ], [ 50, 54, -3/4*E(4) ], [ 50, 55, 1/4*E(4) ], [ 50, 57, -3/4*E(4) ], [ 50, 58, 1/4*E(4) ], 
  [ 50, 59, -3/4*E(4) ], [ 50, 61, 1/4*E(4) ], [ 50, 63, 1/4*E(4) ], [ 51, 40, 1/8 ], [ 51, 44, 1/8 ], [ 51, 45, -1/8 ],
  [ 51, 46, -1/8 ], [ 51, 48, 1/8 ], [ 51, 49, -1/8 ], [ 51, 50, -1/8 ], [ 51, 51, 1/8 ], [ 51, 52, -1/8 ], 
  [ 51, 53, 1/8 ], [ 51, 54, -1/8 ], [ 51, 55, 1/8 ], [ 51, 56, -1/8 ], [ 51, 57, 1/8 ], [ 51, 58, -1/8 ], 
  [ 51, 59, -1/8 ], [ 51, 61, 1/8 ], [ 51, 63, -1/8 ], [ 52, 45, 1/2 ], [ 52, 48, 1/2 ], [ 52, 49, 1/2 ], 
  [ 52, 51, 1/2 ], [ 52, 53, 1/2 ], [ 52, 54, 1/2 ], [ 52, 55, 1/2 ], [ 52, 58, 1/2 ], [ 53, 40, 3/8 ], 
  [ 53, 44, -1/8 ], [ 53, 45, 3/8 ], [ 53, 46, -3/8 ], [ 53, 48, -3/8 ], [ 53, 49, -1/8 ], [ 53, 50, 1/8 ], 
  [ 53, 51, -3/8 ], [ 53, 52, 3/8 ], [ 53, 53, 1/8 ], [ 53, 54, 3/8 ], [ 53, 55, 1/8 ], [ 53, 56, -1/8 ], 
  [ 53, 57, 3/8 ], [ 53, 58, -1/8 ], [ 53, 59, -3/8 ], [ 53, 61, -1/8 ], [ 53, 63, 1/8 ], [ 54, 40, -1/8 ], 
  [ 54, 44, -1/8 ], [ 54, 45, -1/8 ], [ 54, 46, 1/8 ], [ 54, 48, 1/8 ], [ 54, 49, -1/8 ], [ 54, 50, 1/8 ], 
  [ 54, 51, 1/8 ], [ 54, 52, 1/8 ], [ 54, 53, 1/8 ], [ 54, 54, -1/8 ], [ 54, 55, 1/8 ], [ 54, 56, 1/8 ], 
  [ 54, 57, 1/8 ], [ 54, 58, -1/8 ], [ 54, 59, -1/8 ], [ 54, 61, 1/8 ], [ 54, 63, -1/8 ], [ 55, 40, -3/8 ], 
  [ 55, 44, 1/8 ], [ 55, 45, 3/8 ], [ 55, 46, 3/8 ], [ 55, 48, -3/8 ], [ 55, 49, -1/8 ], [ 55, 50, -1/8 ], 
  [ 55, 51, -3/8 ], [ 55, 52, 3/8 ], [ 55, 53, 1/8 ], [ 55, 54, 3/8 ], [ 55, 55, 1/8 ], [ 55, 56, -1/8 ], 
  [ 55, 57, -3/8 ], [ 55, 58, -1/8 ], [ 55, 59, 3/8 ], [ 55, 61, 1/8 ], [ 55, 63, -1/8 ], [ 56, 45, -3/2 ], 
  [ 56, 48, -3/2 ], [ 56, 49, 1/2 ], [ 56, 51, -3/2 ], [ 56, 53, 1/2 ], [ 56, 54, -3/2 ], [ 56, 55, 1/2 ], 
  [ 56, 58, 1/2 ], [ 57, 40, 1/4*E(4) ], [ 57, 44, 1/4*E(4) ], [ 57, 45, -1/4*E(4) ], [ 57, 46, 1/4*E(4) ], 
  [ 57, 48, 1/4*E(4) ], [ 57, 49, -1/4*E(4) ], [ 57, 50, 1/4*E(4) ], [ 57, 51, -1/4*E(4) ], [ 57, 53, 1/4*E(4) ], 
  [ 57, 54, 1/4*E(4) ], [ 57, 55, -1/4*E(4) ], [ 57, 58, 1/4*E(4) ], [ 58, 40, 3/8 ], [ 58, 44, -1/8 ], [ 58, 45, 3/8 ],
  [ 58, 46, -3/8 ], [ 58, 48, -3/8 ], [ 58, 49, -1/8 ], [ 58, 50, 1/8 ], [ 58, 51, -3/8 ], [ 58, 52, -3/8 ], 
  [ 58, 53, 1/8 ], [ 58, 54, 3/8 ], [ 58, 55, 1/8 ], [ 58, 56, 1/8 ], [ 58, 57, -3/8 ], [ 58, 58, -1/8 ], 
  [ 58, 59, 3/8 ], [ 58, 61, 1/8 ], [ 58, 63, -1/8 ], [ 59, 40, -1/4*E(4) ], [ 59, 44, -1/4*E(4) ], 
  [ 59, 45, -1/4*E(4) ], [ 59, 46, -1/4*E(4) ], [ 59, 48, 1/4*E(4) ], [ 59, 49, -1/4*E(4) ], [ 59, 50, -1/4*E(4) ], 
  [ 59, 51, -1/4*E(4) ], [ 59, 53, 1/4*E(4) ], [ 59, 54, 1/4*E(4) ], [ 59, 55, -1/4*E(4) ], [ 59, 58, 1/4*E(4) ], 
  [ 60, 64, -1/4*E(4) ], [ 60, 66, -1/4*E(4) ], [ 60, 67, 1/4*E(4) ], [ 60, 68, 1/2*E(4) ], [ 60, 69, 1/4*E(4) ], 
  [ 60, 70, -1/2*E(4) ], [ 61, 40, -3/4*E(4) ], [ 61, 44, 1/4*E(4) ], [ 61, 45, 3/4*E(4) ], [ 61, 46, -3/4*E(4) ], 
  [ 61, 48, -3/4*E(4) ], [ 61, 49, -1/4*E(4) ], [ 61, 50, 1/4*E(4) ], [ 61, 51, 3/4*E(4) ], [ 61, 53, 1/4*E(4) ], 
  [ 61, 54, -3/4*E(4) ], [ 61, 55, -1/4*E(4) ], [ 61, 58, 1/4*E(4) ], [ 62, 64, 1/4*E(4) ], [ 62, 66, 1/4*E(4) ], 
  [ 62, 67, -1/4*E(4) ], [ 62, 68, 1/2*E(4) ], [ 62, 69, -1/4*E(4) ], [ 62, 70, -1/2*E(4) ], [ 63, 40, 3/4*E(4) ], 
  [ 63, 44, -1/4*E(4) ], [ 63, 45, 3/4*E(4) ], [ 63, 46, 3/4*E(4) ], [ 63, 48, -3/4*E(4) ], [ 63, 49, -1/4*E(4) ], 
  [ 63, 50, -1/4*E(4) ], [ 63, 51, 3/4*E(4) ], [ 63, 53, 1/4*E(4) ], [ 63, 54, -3/4*E(4) ], [ 63, 55, -1/4*E(4) ], 
  [ 63, 58, 1/4*E(4) ], [ 64, 60, -1/2 ], [ 64, 62, -1/2 ], [ 64, 64, 1/4 ], [ 64, 65, 1 ], [ 64, 66, 1/4 ], 
  [ 64, 67, 1/4 ], [ 64, 68, -1/2 ], [ 64, 69, 1/4 ], [ 64, 70, -1/2 ], [ 65, 64, 1/4 ], [ 65, 66, -1/4 ], 
  [ 65, 67, -1/4 ], [ 65, 69, 1/4 ], [ 66, 60, 1/2 ], [ 66, 62, 1/2 ], [ 66, 64, -1/4 ], [ 66, 65, 1 ], 
  [ 66, 66, -1/4 ], [ 66, 67, -1/4 ], [ 66, 68, -1/2 ], [ 66, 69, -1/4 ], [ 66, 70, -1/2 ], [ 67, 60, -1/2 ], 
  [ 67, 62, -1/2 ], [ 67, 64, -1/4 ], [ 67, 65, 1 ], [ 67, 66, -1/4 ], [ 67, 67, -1/4 ], [ 67, 68, 1/2 ], 
  [ 67, 69, -1/4 ], [ 67, 70, 1/2 ], [ 68, 60, -1/2*E(4) ], [ 68, 62, 1/2*E(4) ], [ 68, 64, 1/4*E(4) ], 
  [ 68, 66, -1/4*E(4) ], [ 68, 67, 1/4*E(4) ], [ 68, 69, -1/4*E(4) ], [ 69, 60, 1/2 ], [ 69, 62, 1/2 ], [ 69, 64, 1/4 ],
  [ 69, 65, 1 ], [ 69, 66, 1/4 ], [ 69, 67, 1/4 ], [ 69, 68, 1/2 ], [ 69, 69, 1/4 ], [ 69, 70, 1/2 ], 
  [ 70, 60, -1/2*E(4) ], [ 70, 62, 1/2*E(4) ], [ 70, 64, -1/4*E(4) ], [ 70, 66, 1/4*E(4) ], [ 70, 67, -1/4*E(4) ], 
  [ 70, 69, 1/4*E(4) ], [ 71, 71, -1/2 ], [ 71, 72, 1/2 ], [ 72, 71, -3/2 ], [ 72, 72, -1/2 ], [ 73, 1, 1/4 ], 
  [ 73, 3, -1/4 ], [ 73, 7, 1/4 ], [ 73, 37, 1/4 ], [ 73, 39, -1/2 ], [ 73, 43, 1/2 ], [ 73, 75, -1/4 ], 
  [ 73, 76, 1/2 ], [ 74, 2, 3/4 ], [ 74, 38, 1/4 ], [ 74, 74, -1/2 ], [ 74, 76, 3/4 ], [ 75, 1, 1/2 ], [ 75, 37, 1/2 ], 
  [ 75, 75, -1/2 ], [ 75, 76, 1 ], [ 76, 76, 1 ], [ 77, 5, 1/4 ], [ 77, 6, 1/2 ], [ 77, 11, -1/2 ], [ 77, 41, 1/4 ], 
  [ 77, 42, 1/4 ], [ 77, 47, -1/4 ], [ 77, 76, 3/4 ], [ 77, 78, -1/4 ], [ 78, 5, 1/2 ], [ 78, 41, 1/2 ], 
  [ 78, 76, 1/2 ], [ 78, 78, -1/2 ] ] ],

      
[ [ 2, 0, 0, 2, 0, 2 ],
   [ 78, [ 1, 1, 1 ], [ 2, 2, -1 ], [ 3, 3, -1 ], [ 4, 4, -1 ], [ 5, 5, -1 ], [ 6, 6, -1 ], [ 7, 7, -1 ], [ 8, 8, 1 ], 
  [ 9, 9, 1 ], [ 10, 10, 1 ], [ 11, 11, 1 ], [ 12, 12, 1 ], [ 13, 13, -1 ], [ 14, 14, -1 ], [ 15, 15, -1 ], 
  [ 16, 16, -1 ], [ 17, 17, -1 ], [ 18, 18, -1 ], [ 19, 19, 1 ], [ 20, 20, 1 ], [ 21, 21, 1 ], [ 22, 22, 1 ], 
  [ 23, 23, 1 ], [ 24, 24, -1 ], [ 25, 25, -1 ], [ 26, 26, -1 ], [ 27, 27, -1 ], [ 28, 28, 1 ], [ 29, 29, 1 ], 
  [ 30, 30, 1 ], [ 31, 31, -1 ], [ 32, 32, -1 ], [ 33, 33, -1 ], [ 34, 34, 1 ], [ 35, 35, -1 ], [ 36, 36, 1 ], 
  [ 37, 37, 1 ], [ 38, 38, -1 ], [ 39, 39, -1 ], [ 40, 40, -1 ], [ 41, 41, -1 ], [ 42, 42, -1 ], [ 43, 43, -1 ], 
  [ 44, 44, 1 ], [ 45, 45, 1 ], [ 46, 46, 1 ], [ 47, 47, 1 ], [ 48, 48, 1 ], [ 49, 49, -1 ], [ 50, 50, -1 ], 
  [ 51, 51, -1 ], [ 52, 52, -1 ], [ 53, 53, -1 ], [ 54, 54, -1 ], [ 55, 55, 1 ], [ 56, 56, 1 ], [ 57, 57, 1 ], 
  [ 58, 58, 1 ], [ 59, 59, 1 ], [ 60, 60, -1 ], [ 61, 61, -1 ], [ 62, 62, -1 ], [ 63, 63, -1 ], [ 64, 64, 1 ], 
  [ 65, 65, 1 ], [ 66, 66, 1 ], [ 67, 67, -1 ], [ 68, 68, -1 ], [ 69, 69, -1 ], [ 70, 70, 1 ], [ 71, 71, -1 ], 
  [ 72, 72, 1 ], [ 73, 73, 1 ], [ 74, 74, 1 ], [ 75, 75, 1 ], [ 76, 76, 1 ], [ 77, 77, 1 ], [ 78, 78, 1 ] ] ]   

],


["E", 7, [ [ 1, 0, 0, 0, 0, 1, 0 ],
  [ 133, [ 1, 8, -E(4) ], [ 2, 2, 1 ], [ 3, 66, -1 ], [ 4, 17, -1 ], 
  [ 5, 68, -1 ], [ 6, 19, E(4) ], [ 7, 70, 1 ], [ 8, 1, E(4) ], 
  [ 9, 22, -1 ], [ 10, 11, 1 ], [ 11, 10, 1 ], [ 12, 13, -E(4) ], 
  [ 13, 12, E(4) ], [ 14, 21, -E(4) ], [ 15, 16, 1 ], [ 16, 15, 1 ], 
  [ 17, 4, -1 ], [ 18, 31, E(4) ], [ 19, 6, -E(4) ], [ 20, 26, -E(4) ], 
  [ 21, 14, E(4) ], [ 22, 9, -1 ], [ 23, 36, E(4) ], [ 24, 25, -E(4) ], 
  [ 25, 24, E(4) ], [ 26, 20, E(4) ], [ 27, 34, -1 ], [ 28, 28, 1 ], 
  [ 29, 30, -E(4) ], [ 30, 29, E(4) ], [ 31, 18, -E(4) ], [ 32, 37, E(4) ], 
  [ 33, 39, -1 ], [ 34, 27, -1 ], [ 35, 45, -E(4) ], [ 36, 23, -E(4) ], 
  [ 37, 32, -E(4) ], [ 38, 51, 1 ], [ 39, 33, -1 ], [ 40, 41, E(4) ], 
  [ 41, 40, -E(4) ], [ 42, 48, -1 ], [ 43, 47, -1 ], [ 44, 46, 1 ], 
  [ 45, 35, E(4) ], [ 46, 44, 1 ], [ 47, 43, -1 ], [ 48, 42, -1 ], 
  [ 49, 49, 1 ], [ 50, 54, -1 ], [ 51, 38, 1 ], [ 52, 55, E(4) ], 
  [ 53, 56, -1 ], [ 54, 50, -1 ], [ 55, 52, -E(4) ], [ 56, 53, -1 ], 
  [ 57, 59, -E(4) ], [ 58, 60, -E(4) ], [ 59, 57, E(4) ], [ 60, 58, E(4) ], 
  [ 61, 62, -E(4) ], [ 62, 61, E(4) ], [ 63, 63, -1 ], [ 64, 71, E(4) ], 
  [ 65, 65, 1 ], [ 66, 3, -1 ], [ 67, 80, -1 ], [ 68, 5, -1 ], 
  [ 69, 82, -E(4) ], [ 70, 7, 1 ], [ 71, 64, -E(4) ], [ 72, 85, -1 ], 
  [ 73, 74, 1 ], [ 74, 73, 1 ], [ 75, 76, E(4) ], [ 76, 75, -E(4) ], 
  [ 77, 84, E(4) ], [ 78, 79, 1 ], [ 79, 78, 1 ], [ 80, 67, -1 ], 
  [ 81, 94, -E(4) ], [ 82, 69, E(4) ], [ 83, 89, E(4) ], [ 84, 77, -E(4) ], 
  [ 85, 72, -1 ], [ 86, 99, -E(4) ], [ 87, 88, E(4) ], [ 88, 87, -E(4) ], 
  [ 89, 83, -E(4) ], [ 90, 97, -1 ], [ 91, 91, 1 ], [ 92, 93, E(4) ], 
  [ 93, 92, -E(4) ], [ 94, 81, E(4) ], [ 95, 100, -E(4) ], [ 96, 102, -1 ], 
  [ 97, 90, -1 ], [ 98, 108, E(4) ], [ 99, 86, E(4) ], [ 100, 95, E(4) ], 
  [ 101, 114, 1 ], [ 102, 96, -1 ], [ 103, 104, -E(4) ], [ 104, 103, E(4) ], 
  [ 105, 111, -1 ], [ 106, 110, -1 ], [ 107, 109, 1 ], [ 108, 98, -E(4) ], 
  [ 109, 107, 1 ], [ 110, 106, -1 ], [ 111, 105, -1 ], [ 112, 112, 1 ], 
  [ 113, 117, -1 ], [ 114, 101, 1 ], [ 115, 118, -E(4) ], [ 116, 119, -1 ], 
  [ 117, 113, -1 ], [ 118, 115, E(4) ], [ 119, 116, -1 ], [ 120, 122, E(4) ],
  [ 121, 123, E(4) ], [ 122, 120, -E(4) ], [ 123, 121, -E(4) ], 
  [ 124, 125, E(4) ], [ 125, 124, -E(4) ], [ 126, 126, -1 ], [ 127, 127, 1 ],
  [ 128, 128, 1 ], [ 129, 127, 1 ], [ 129, 129, -1 ], [ 129, 130, 1 ], 
  [ 130, 130, 1 ], [ 131, 130, 1 ], [ 131, 131, -1 ], [ 131, 132, 1 ], 
  [ 132, 132, 1 ], [ 133, 132, 1 ], [ 133, 133, -1 ] ] ],

[ [ 0, 0, 0, 1, 0, 1, 0 ], [ 133, [ 1, 66, -1 ], [ 2, 65, -1 ], [ 3, 64, -1 ], [ 4, 20, -E(4) ], 
  [ 5, 5, 1 ], [ 6, 13, E(4) ], [ 7, 70, -1 ], [ 8, 71, 1 ], [ 9, 14, E(4) ], 
  [ 10, 15, E(4) ], [ 11, 26, -E(4) ], [ 12, 19, E(4) ], [ 13, 6, -E(4) ], 
  [ 14, 9, -E(4) ], [ 15, 10, -E(4) ], [ 16, 21, E(4) ], [ 17, 22, E(4) ], 
  [ 18, 39, 1 ], [ 19, 12, -E(4) ], [ 20, 4, E(4) ], [ 21, 16, -E(4) ], 
  [ 22, 17, -E(4) ], [ 23, 34, -1 ], [ 24, 36, -1 ], [ 25, 33, -1 ], 
  [ 26, 11, E(4) ], [ 27, 30, 1 ], [ 28, 37, -1 ], [ 29, 31, 1 ], 
  [ 30, 27, 1 ], [ 31, 29, 1 ], [ 32, 32, 1 ], [ 33, 25, -1 ], 
  [ 34, 23, -1 ], [ 35, 47, -E(4) ], [ 36, 24, -1 ], [ 37, 28, -1 ], 
  [ 38, 44, E(4) ], [ 39, 18, 1 ], [ 40, 51, -E(4) ], [ 41, 42, E(4) ], 
  [ 42, 41, -E(4) ], [ 43, 48, E(4) ], [ 44, 38, -E(4) ], [ 45, 46, E(4) ], 
  [ 46, 45, -E(4) ], [ 47, 35, E(4) ], [ 48, 43, -E(4) ], [ 49, 55, 1 ], 
  [ 50, 56, 1 ], [ 51, 40, E(4) ], [ 52, 52, -1 ], [ 53, 54, -1 ], 
  [ 54, 53, -1 ], [ 55, 49, 1 ], [ 56, 50, 1 ], [ 57, 58, E(4) ], 
  [ 58, 57, -E(4) ], [ 59, 60, E(4) ], [ 60, 59, -E(4) ], [ 61, 63, -1 ], 
  [ 62, 62, 1 ], [ 63, 61, -1 ], [ 64, 3, -1 ], [ 65, 2, -1 ], [ 66, 1, -1 ], 
  [ 67, 83, E(4) ], [ 68, 68, 1 ], [ 69, 76, -E(4) ], [ 70, 7, -1 ], 
  [ 71, 8, 1 ], [ 72, 77, -E(4) ], [ 73, 78, -E(4) ], [ 74, 89, E(4) ], 
  [ 75, 82, -E(4) ], [ 76, 69, E(4) ], [ 77, 72, E(4) ], [ 78, 73, E(4) ], 
  [ 79, 84, -E(4) ], [ 80, 85, -E(4) ], [ 81, 102, 1 ], [ 82, 75, E(4) ], 
  [ 83, 67, -E(4) ], [ 84, 79, E(4) ], [ 85, 80, E(4) ], [ 86, 97, -1 ], 
  [ 87, 99, -1 ], [ 88, 96, -1 ], [ 89, 74, -E(4) ], [ 90, 93, 1 ], 
  [ 91, 100, -1 ], [ 92, 94, 1 ], [ 93, 90, 1 ], [ 94, 92, 1 ], 
  [ 95, 95, 1 ], [ 96, 88, -1 ], [ 97, 86, -1 ], [ 98, 110, E(4) ], 
  [ 99, 87, -1 ], [ 100, 91, -1 ], [ 101, 107, -E(4) ], [ 102, 81, 1 ], 
  [ 103, 114, E(4) ], [ 104, 105, -E(4) ], [ 105, 104, E(4) ], 
  [ 106, 111, -E(4) ], [ 107, 101, E(4) ], [ 108, 109, -E(4) ], 
  [ 109, 108, E(4) ], [ 110, 98, -E(4) ], [ 111, 106, E(4) ], [ 112, 118, 1 ],
  [ 113, 119, 1 ], [ 114, 103, -E(4) ], [ 115, 115, -1 ], [ 116, 117, -1 ], 
  [ 117, 116, -1 ], [ 118, 112, 1 ], [ 119, 113, 1 ], [ 120, 121, -E(4) ], 
  [ 121, 120, E(4) ], [ 122, 123, -E(4) ], [ 123, 122, E(4) ], 
  [ 124, 126, -1 ], [ 125, 125, 1 ], [ 126, 124, -1 ], [ 127, 129, -1 ], 
  [ 127, 130, 1 ], [ 128, 128, -1 ], [ 128, 130, 1 ], [ 129, 127, -1 ], 
  [ 129, 130, 1 ], [ 130, 130, 1 ], [ 131, 131, 1 ], [ 132, 132, 1 ], 
  [ 133, 132, 1 ], [ 133, 133, -1 ] ] ],

 
[ [ 2, 0, 0, 0, 0, 2, 0 ],
  [ 133, [ 1, 26, E(5) ], [ 2, 78, -E(5)^4 ], [ 3, 4, E(5)^3 ], 
  [ 4, 3, E(5)^2 ], [ 5, 80, E(5)^4 ], [ 6, 45, -E(5)^2 ], 
  [ 7, 70, -E(5)^2 ], [ 8, 32, E(5)^4 ], [ 9, 72, -E(5) ], [ 10, 10, -1 ], 
  [ 11, 74, -E(5) ], [ 12, 30, -E(5) ], [ 13, 40, E(5)^4 ], 
  [ 14, 37, -E(5) ], [ 15, 65, -E(5)^4 ], [ 16, 91, 1 ], [ 17, 68, E(5)^4 ], 
  [ 18, 36, -E(5)^3 ], [ 19, 23, E(5)^3 ], [ 20, 21, 1 ], [ 21, 20, 1 ], 
  [ 22, 85, E(5)^3 ], [ 23, 19, E(5)^2 ], [ 24, 41, E(5) ], [ 25, 29, 1 ], 
  [ 26, 1, E(5)^4 ], [ 27, 56, -E(5)^2 ], [ 28, 79, 1 ], [ 29, 25, 1 ], 
  [ 30, 12, -E(5)^4 ], [ 31, 35, -E(5)^3 ], [ 32, 8, E(5) ], 
  [ 33, 48, -E(5) ], [ 34, 53, E(5)^4 ], [ 35, 31, -E(5)^2 ], 
  [ 36, 18, -E(5)^2 ], [ 37, 14, -E(5)^4 ], [ 38, 51, E(5)^3 ], 
  [ 39, 43, E(5)^3 ], [ 40, 13, E(5) ], [ 41, 24, E(5)^4 ], 
  [ 42, 54, -E(5) ], [ 43, 39, E(5)^2 ], [ 44, 46, -1 ], [ 45, 6, -E(5)^3 ], 
  [ 46, 44, -1 ], [ 47, 50, E(5)^3 ], [ 48, 33, -E(5)^4 ], [ 49, 49, -1 ], 
  [ 50, 47, E(5)^2 ], [ 51, 38, E(5)^2 ], [ 52, 60, -E(5) ], 
  [ 53, 34, E(5) ], [ 54, 42, -E(5)^4 ], [ 55, 61, E(5)^4 ], 
  [ 56, 27, -E(5)^3 ], [ 57, 62, E(5) ], [ 58, 59, 1 ], [ 59, 58, 1 ], 
  [ 60, 52, -E(5)^4 ], [ 61, 55, E(5) ], [ 62, 57, E(5)^4 ], [ 63, 63, -1 ], 
  [ 64, 89, E(5)^4 ], [ 65, 15, -E(5) ], [ 66, 67, E(5)^2 ], 
  [ 67, 66, E(5)^3 ], [ 68, 17, E(5) ], [ 69, 108, -E(5)^3 ], 
  [ 70, 7, -E(5)^3 ], [ 71, 95, E(5) ], [ 72, 9, -E(5)^4 ], [ 73, 73, -1 ], 
  [ 74, 11, -E(5)^4 ], [ 75, 93, -E(5)^4 ], [ 76, 103, E(5) ], 
  [ 77, 100, -E(5)^4 ], [ 78, 2, -E(5) ], [ 79, 28, 1 ], [ 80, 5, E(5) ], 
  [ 81, 99, -E(5)^2 ], [ 82, 86, E(5)^2 ], [ 83, 84, 1 ], [ 84, 83, 1 ], 
  [ 85, 22, E(5)^2 ], [ 86, 82, E(5)^3 ], [ 87, 104, E(5)^4 ], 
  [ 88, 92, 1 ], [ 89, 64, E(5) ], [ 90, 119, -E(5)^3 ], [ 91, 16, 1 ], 
  [ 92, 88, 1 ], [ 93, 75, -E(5) ], [ 94, 98, -E(5)^2 ], [ 95, 71, E(5)^4 ], 
  [ 96, 111, -E(5)^4 ], [ 97, 116, E(5) ], [ 98, 94, -E(5)^3 ], 
  [ 99, 81, -E(5)^3 ], [ 100, 77, -E(5) ], [ 101, 114, E(5)^2 ], 
  [ 102, 106, E(5)^2 ], [ 103, 76, E(5)^4 ], [ 104, 87, E(5) ], 
  [ 105, 117, -E(5)^4 ], [ 106, 102, E(5)^3 ], [ 107, 109, -1 ], 
  [ 108, 69, -E(5)^2 ], [ 109, 107, -1 ], [ 110, 113, E(5)^2 ], 
  [ 111, 96, -E(5) ], [ 112, 112, -1 ], [ 113, 110, E(5)^3 ], 
  [ 114, 101, E(5)^3 ], [ 115, 123, -E(5)^4 ], [ 116, 97, E(5)^4 ], 
  [ 117, 105, -E(5) ], [ 118, 124, E(5) ], [ 119, 90, -E(5)^2 ], 
  [ 120, 125, E(5)^4 ], [ 121, 122, 1 ], [ 122, 121, 1 ], 
  [ 123, 115, -E(5) ], [ 124, 118, E(5)^4 ], [ 125, 120, E(5) ], 
  [ 126, 126, -1 ], [ 127, 127, 1 ], [ 128, 127, 1 ], [ 128, 128, -1 ], 
  [ 128, 132, 1 ], [ 129, 127, 1 ], [ 129, 128, -1 ], [ 129, 130, 1 ], 
  [ 129, 131, -1 ], [ 129, 132, 1 ], [ 130, 127, 1 ], [ 130, 128, -1 ], 
  [ 130, 129, 1 ], [ 130, 131, -1 ], [ 130, 132, 2 ], [ 131, 127, 1 ], 
  [ 131, 131, -1 ], [ 131, 132, 2 ], [ 132, 132, 1 ], [ 133, 132, 1 ], 
  [ 133, 133, -1 ] ] ],

[ [ 1, 0, 0, 1, 0, 1, 0 ],
[ 133, [ 1, 8, -1 ], [ 2, 65, 1 ], [ 3, 66, -1 ], [ 4, 22, 1 ], [ 5, 68, 1 ], [ 6, 19, -1 ], [ 7, 70, -1 ], [ 8, 1, 1 ], [ 9, 17, 1 ], 
  [ 10, 16, -1 ], [ 11, 15, 1 ], [ 12, 13, -1 ], [ 13, 12, 1 ], [ 14, 26, 1 ], [ 15, 11, -1 ], [ 16, 10, 1 ], [ 17, 9, -1 ], [ 18, 36, -1 ], 
  [ 19, 6, 1 ], [ 20, 21, 1 ], [ 21, 20, 1 ], [ 22, 4, -1 ], [ 23, 31, -1 ], [ 24, 30, 1 ], [ 25, 29, 1 ], [ 26, 14, 1 ], [ 27, 39, -1 ], 
  [ 28, 28, 1 ], [ 29, 25, 1 ], [ 30, 24, 1 ], [ 31, 23, -1 ], [ 32, 37, 1 ], [ 33, 34, -1 ], [ 34, 33, 1 ], [ 35, 45, 1 ], [ 36, 18, -1 ], 
  [ 37, 32, -1 ], [ 38, 51, 1 ], [ 39, 27, 1 ], [ 40, 41, 1 ], [ 41, 40, -1 ], [ 42, 48, -1 ], [ 43, 47, 1 ], [ 44, 46, -1 ], [ 45, 35, -1 ], 
  [ 46, 44, -1 ], [ 47, 43, 1 ], [ 48, 42, -1 ], [ 49, 49, -1 ], [ 50, 56, 1 ], [ 51, 38, 1 ], [ 52, 55, -1 ], [ 53, 54, 1 ], [ 54, 53, -1 ], 
  [ 55, 52, 1 ], [ 56, 50, -1 ], [ 57, 60, 1 ], [ 58, 59, 1 ], [ 59, 58, 1 ], [ 60, 57, 1 ], [ 61, 62, 1 ], [ 62, 61, -1 ], [ 63, 63, -1 ], 
  [ 64, 71, -1 ], [ 65, 2, 1 ], [ 66, 3, -1 ], [ 67, 85, 1 ], [ 68, 5, 1 ], [ 69, 82, -1 ], [ 70, 7, -1 ], [ 71, 64, 1 ], [ 72, 80, 1 ], 
  [ 73, 79, -1 ], [ 74, 78, 1 ], [ 75, 76, -1 ], [ 76, 75, 1 ], [ 77, 89, 1 ], [ 78, 74, -1 ], [ 79, 73, 1 ], [ 80, 72, -1 ], [ 81, 99, -1 ], 
  [ 82, 69, 1 ], [ 83, 84, 1 ], [ 84, 83, 1 ], [ 85, 67, -1 ], [ 86, 94, -1 ], [ 87, 93, 1 ], [ 88, 92, 1 ], [ 89, 77, 1 ], [ 90, 102, -1 ], 
  [ 91, 91, 1 ], [ 92, 88, 1 ], [ 93, 87, 1 ], [ 94, 86, -1 ], [ 95, 100, 1 ], [ 96, 97, -1 ], [ 97, 96, 1 ], [ 98, 108, 1 ], [ 99, 81, -1 ], 
  [ 100, 95, -1 ], [ 101, 114, 1 ], [ 102, 90, 1 ], [ 103, 104, 1 ], [ 104, 103, -1 ], [ 105, 111, -1 ], [ 106, 110, 1 ], [ 107, 109, -1 ], 
  [ 108, 98, -1 ], [ 109, 107, -1 ], [ 110, 106, 1 ], [ 111, 105, -1 ], [ 112, 112, -1 ], [ 113, 119, 1 ], [ 114, 101, 1 ], [ 115, 118, -1 ], 
  [ 116, 117, 1 ], [ 117, 116, -1 ], [ 118, 115, 1 ], [ 119, 113, -1 ], [ 120, 123, 1 ], [ 121, 122, 1 ], [ 122, 121, 1 ], [ 123, 120, 1 ], 
  [ 124, 125, 1 ], [ 125, 124, -1 ], [ 126, 126, -1 ], [ 127, 127, 1 ], [ 128, 128, -1 ], [ 128, 130, 1 ], [ 129, 127, 1 ], [ 129, 129, -1 ], 
  [ 129, 130, 1 ], [ 130, 130, 1 ], [ 131, 130, 1 ], [ 131, 131, -1 ], [ 131, 132, 1 ], [ 132, 132, 1 ], [ 133, 132, 1 ], [ 133, 133, -1 ] ] ],

 
[ [ 2, 0, 0, 1, 0, 1, 0 ],
  [ 133, [ 1, 1, -1 ], [ 2, 2, -1 ], [ 3, 3, -1 ], [ 4, 4, -1 ], [ 5, 5, 1 ], [ 6, 6, -1 ], [ 7, 7, -1 ], [ 8, 8, 1 ], [ 9, 9, 1 ], 
  [ 10, 10, 1 ], [ 11, 11, -1 ], [ 12, 12, -1 ], [ 13, 13, 1 ], [ 14, 14, -1 ], [ 15, 15, -1 ], [ 16, 16, 1 ], [ 17, 17, 1 ], [ 18, 18, 1 ], 
  [ 19, 19, 1 ], [ 20, 20, 1 ], [ 21, 21, -1 ], [ 22, 22, -1 ], [ 23, 23, -1 ], [ 24, 24, -1 ], [ 25, 25, -1 ], [ 26, 26, 1 ], [ 27, 27, 1 ], 
  [ 28, 28, 1 ], [ 29, 29, 1 ], [ 30, 30, 1 ], [ 31, 31, 1 ], [ 32, 32, -1 ], [ 33, 33, -1 ], [ 34, 34, -1 ], [ 35, 35, -1 ], [ 36, 36, -1 ], 
  [ 37, 37, 1 ], [ 38, 38, 1 ], [ 39, 39, 1 ], [ 40, 40, -1 ], [ 41, 41, 1 ], [ 42, 42, -1 ], [ 43, 43, 1 ], [ 44, 44, -1 ], [ 45, 45, 1 ], 
  [ 46, 46, -1 ], [ 47, 47, 1 ], [ 48, 48, -1 ], [ 49, 49, -1 ], [ 50, 50, 1 ], [ 51, 51, 1 ], [ 52, 52, 1 ], [ 53, 53, -1 ], [ 54, 54, -1 ], 
  [ 55, 55, -1 ], [ 56, 56, 1 ], [ 57, 57, 1 ], [ 58, 58, -1 ], [ 59, 59, 1 ], [ 60, 60, -1 ], [ 61, 61, 1 ], [ 62, 62, -1 ], [ 63, 63, 1 ], 
  [ 64, 64, -1 ], [ 65, 65, -1 ], [ 66, 66, -1 ], [ 67, 67, -1 ], [ 68, 68, 1 ], [ 69, 69, -1 ], [ 70, 70, -1 ], [ 71, 71, 1 ], [ 72, 72, 1 ], 
  [ 73, 73, 1 ], [ 74, 74, -1 ], [ 75, 75, -1 ], [ 76, 76, 1 ], [ 77, 77, -1 ], [ 78, 78, -1 ], [ 79, 79, 1 ], [ 80, 80, 1 ], [ 81, 81, 1 ], 
  [ 82, 82, 1 ], [ 83, 83, 1 ], [ 84, 84, -1 ], [ 85, 85, -1 ], [ 86, 86, -1 ], [ 87, 87, -1 ], [ 88, 88, -1 ], [ 89, 89, 1 ], [ 90, 90, 1 ], 
  [ 91, 91, 1 ], [ 92, 92, 1 ], [ 93, 93, 1 ], [ 94, 94, 1 ], [ 95, 95, -1 ], [ 96, 96, -1 ], [ 97, 97, -1 ], [ 98, 98, -1 ], [ 99, 99, -1 ], 
  [ 100, 100, 1 ], [ 101, 101, 1 ], [ 102, 102, 1 ], [ 103, 103, -1 ], [ 104, 104, 1 ], [ 105, 105, -1 ], [ 106, 106, 1 ], [ 107, 107, -1 ], 
  [ 108, 108, 1 ], [ 109, 109, -1 ], [ 110, 110, 1 ], [ 111, 111, -1 ], [ 112, 112, -1 ], [ 113, 113, 1 ], [ 114, 114, 1 ], [ 115, 115, 1 ], 
  [ 116, 116, -1 ], [ 117, 117, -1 ], [ 118, 118, -1 ], [ 119, 119, 1 ], [ 120, 120, 1 ], [ 121, 121, -1 ], [ 122, 122, 1 ], [ 123, 123, -1 ], 
  [ 124, 124, 1 ], [ 125, 125, -1 ], [ 126, 126, 1 ], [ 127, 127, 1 ], [ 128, 128, 1 ], [ 129, 129, 1 ], [ 130, 130, 1 ], [ 131, 131, 1 ], 
  [ 132, 132, 1 ], [ 133, 133, 1 ] ] ],

[ [ 2, 0, 0, 2, 0, 2, 0 ],
  [ 133, [ 1, 1, -1 ], [ 2, 65, -1 ], [ 3, 3, -1 ], [ 4, 16, 1 ], [ 5, 68, -1 ], [ 6, 19, -1 ], [ 7, 70, -1 ], [ 8, 8, 1 ], [ 9, 11, -1 ], 
  [ 10, 22, -1 ], [ 11, 9, -1 ], [ 12, 13, 1 ], [ 13, 12, 1 ], [ 14, 26, 1 ], [ 15, 17, 1 ], [ 16, 4, 1 ], [ 17, 15, 1 ], [ 18, 30, 1 ], 
  [ 19, 6, -1 ], [ 20, 21, -1 ], [ 21, 20, -1 ], [ 22, 10, -1 ], [ 23, 25, -1 ], [ 24, 36, -1 ], [ 25, 23, -1 ], [ 26, 14, 1 ], [ 27, 39, 1 ], 
  [ 28, 28, -1 ], [ 29, 31, 1 ], [ 30, 18, 1 ], [ 31, 29, 1 ], [ 32, 32, 1 ], [ 33, 34, -1 ], [ 34, 33, -1 ], [ 35, 45, -1 ], [ 36, 24, -1 ], 
  [ 37, 37, -1 ], [ 38, 48, 1 ], [ 39, 27, 1 ], [ 40, 41, 1 ], [ 41, 40, 1 ], [ 42, 51, -1 ], [ 43, 44, -1 ], [ 44, 43, -1 ], [ 45, 35, -1 ], 
  [ 46, 47, 1 ], [ 47, 46, 1 ], [ 48, 38, 1 ], [ 49, 49, -1 ], [ 50, 56, 1 ], [ 51, 42, -1 ], [ 52, 52, 1 ], [ 53, 54, -1 ], [ 54, 53, -1 ], 
  [ 55, 55, -1 ], [ 56, 50, 1 ], [ 57, 60, 1 ], [ 58, 59, -1 ], [ 59, 58, -1 ], [ 60, 57, 1 ], [ 61, 61, -1 ], [ 62, 62, 1 ], [ 63, 63, -1 ], 
  [ 64, 64, -1 ], [ 65, 2, -1 ], [ 66, 66, -1 ], [ 67, 79, 1 ], [ 68, 5, -1 ], [ 69, 82, -1 ], [ 70, 7, -1 ], [ 71, 71, 1 ], [ 72, 74, -1 ], 
  [ 73, 85, -1 ], [ 74, 72, -1 ], [ 75, 76, 1 ], [ 76, 75, 1 ], [ 77, 89, 1 ], [ 78, 80, 1 ], [ 79, 67, 1 ], [ 80, 78, 1 ], [ 81, 93, 1 ], 
  [ 82, 69, -1 ], [ 83, 84, -1 ], [ 84, 83, -1 ], [ 85, 73, -1 ], [ 86, 88, -1 ], [ 87, 99, -1 ], [ 88, 86, -1 ], [ 89, 77, 1 ], [ 90, 102, 1 ], 
  [ 91, 91, -1 ], [ 92, 94, 1 ], [ 93, 81, 1 ], [ 94, 92, 1 ], [ 95, 95, 1 ], [ 96, 97, -1 ], [ 97, 96, -1 ], [ 98, 108, -1 ], [ 99, 87, -1 ], 
  [ 100, 100, -1 ], [ 101, 111, 1 ], [ 102, 90, 1 ], [ 103, 104, 1 ], [ 104, 103, 1 ], [ 105, 114, -1 ], [ 106, 107, -1 ], [ 107, 106, -1 ], 
  [ 108, 98, -1 ], [ 109, 110, 1 ], [ 110, 109, 1 ], [ 111, 101, 1 ], [ 112, 112, -1 ], [ 113, 119, 1 ], [ 114, 105, -1 ], [ 115, 115, 1 ], 
  [ 116, 117, -1 ], [ 117, 116, -1 ], [ 118, 118, -1 ], [ 119, 113, 1 ], [ 120, 123, 1 ], [ 121, 122, -1 ], [ 122, 121, -1 ], [ 123, 120, 1 ], 
  [ 124, 124, -1 ], [ 125, 125, 1 ], [ 126, 126, -1 ], [ 127, 127, 1 ], [ 128, 128, -1 ], [ 128, 130, 1 ], [ 129, 129, 1 ], [ 130, 130, 1 ], 
  [ 131, 130, 1 ], [ 131, 131, -1 ], [ 131, 132, 1 ], [ 132, 132, 1 ], [ 133, 132, 1 ], [ 133, 133, -1 ] ] ],

[ [ 0, 0, 2, 0, 0, 2, 0 ],
  [ 133, [ 1, 1, -1 ], [ 2, 2, -1 ], [ 3, 3, 1 ], [ 4, 4, -1 ], [ 5, 5, -1 ], [ 6, 6, -1 ], [ 7, 7, 1 ], [ 8, 8, -1 ], [ 9, 9, 1 ], 
  [ 10, 10, -1 ], [ 11, 11, 1 ], [ 12, 12, 1 ], [ 13, 13, -1 ], [ 14, 14, 1 ], [ 15, 15, 1 ], [ 16, 16, -1 ], [ 17, 17, 1 ], [ 18, 18, -1 ], 
  [ 19, 19, 1 ], [ 20, 20, -1 ], [ 21, 21, -1 ], [ 22, 22, -1 ], [ 23, 23, 1 ], [ 24, 24, -1 ], [ 25, 25, -1 ], [ 26, 26, 1 ], [ 27, 27, 1 ], 
  [ 28, 28, 1 ], [ 29, 29, 1 ], [ 30, 30, 1 ], [ 31, 31, -1 ], [ 32, 32, -1 ], [ 33, 33, -1 ], [ 34, 34, 1 ], [ 35, 35, -1 ], [ 36, 36, 1 ], 
  [ 37, 37, -1 ], [ 38, 38, 1 ], [ 39, 39, -1 ], [ 40, 40, 1 ], [ 41, 41, -1 ], [ 42, 42, 1 ], [ 43, 43, -1 ], [ 44, 44, 1 ], [ 45, 45, 1 ], 
  [ 46, 46, -1 ], [ 47, 47, 1 ], [ 48, 48, -1 ], [ 49, 49, -1 ], [ 50, 50, 1 ], [ 51, 51, -1 ], [ 52, 52, 1 ], [ 53, 53, -1 ], [ 54, 54, 1 ], 
  [ 55, 55, 1 ], [ 56, 56, -1 ], [ 57, 57, -1 ], [ 58, 58, 1 ], [ 59, 59, 1 ], [ 60, 60, -1 ], [ 61, 61, 1 ], [ 62, 62, 1 ], [ 63, 63, -1 ], 
  [ 64, 64, -1 ], [ 65, 65, -1 ], [ 66, 66, 1 ], [ 67, 67, -1 ], [ 68, 68, -1 ], [ 69, 69, -1 ], [ 70, 70, 1 ], [ 71, 71, -1 ], [ 72, 72, 1 ], 
  [ 73, 73, -1 ], [ 74, 74, 1 ], [ 75, 75, 1 ], [ 76, 76, -1 ], [ 77, 77, 1 ], [ 78, 78, 1 ], [ 79, 79, -1 ], [ 80, 80, 1 ], [ 81, 81, -1 ], 
  [ 82, 82, 1 ], [ 83, 83, -1 ], [ 84, 84, -1 ], [ 85, 85, -1 ], [ 86, 86, 1 ], [ 87, 87, -1 ], [ 88, 88, -1 ], [ 89, 89, 1 ], [ 90, 90, 1 ], 
  [ 91, 91, 1 ], [ 92, 92, 1 ], [ 93, 93, 1 ], [ 94, 94, -1 ], [ 95, 95, -1 ], [ 96, 96, -1 ], [ 97, 97, 1 ], [ 98, 98, -1 ], [ 99, 99, 1 ], 
  [ 100, 100, -1 ], [ 101, 101, 1 ], [ 102, 102, -1 ], [ 103, 103, 1 ], [ 104, 104, -1 ], [ 105, 105, 1 ], [ 106, 106, -1 ], [ 107, 107, 1 ], 
  [ 108, 108, 1 ], [ 109, 109, -1 ], [ 110, 110, 1 ], [ 111, 111, -1 ], [ 112, 112, -1 ], [ 113, 113, 1 ], [ 114, 114, -1 ], [ 115, 115, 1 ], 
  [ 116, 116, -1 ], [ 117, 117, 1 ], [ 118, 118, 1 ], [ 119, 119, -1 ], [ 120, 120, -1 ], [ 121, 121, 1 ], [ 122, 122, 1 ], [ 123, 123, -1 ], 
  [ 124, 124, 1 ], [ 125, 125, 1 ], [ 126, 126, -1 ], [ 127, 127, 1 ], [ 128, 128, 1 ], [ 129, 129, 1 ], [ 130, 130, 1 ], [ 131, 131, 1 ], 
  [ 132, 132, 1 ], [ 133, 133, 1 ] ] ],

[ [ 2, 0, 0, 0, 0, 0, 0 ],
[ 133, [ 1, 8, 1 ], [ 2, 2, 1 ], [ 3, 66, 1 ], [ 4, 17, 1 ], [ 5, 68, 1 ], 
  [ 6, 19, 1 ], [ 7, 70, 1 ], [ 8, 1, 1 ], [ 9, 22, 1 ], [ 10, 11, 1 ], 
  [ 11, 10, 1 ], [ 12, 13, 1 ], [ 13, 12, 1 ], [ 14, 21, 1 ], [ 15, 16, 1 ], 
  [ 16, 15, 1 ], [ 17, 4, 1 ], [ 18, 31, 1 ], [ 19, 6, 1 ], [ 20, 26, 1 ], 
  [ 21, 14, 1 ], [ 22, 9, 1 ], [ 23, 36, 1 ], [ 24, 25, 1 ], [ 25, 24, 1 ], 
  [ 26, 20, 1 ], [ 27, 34, 1 ], [ 28, 28, 1 ], [ 29, 30, 1 ], [ 30, 29, 1 ], 
  [ 31, 18, 1 ], [ 32, 37, -1 ], [ 33, 39, 1 ], [ 34, 27, 1 ], 
  [ 35, 45, -1 ], [ 36, 23, 1 ], [ 37, 32, -1 ], [ 38, 51, 1 ], 
  [ 39, 33, 1 ], [ 40, 41, -1 ], [ 41, 40, -1 ], [ 42, 48, 1 ], 
  [ 43, 47, 1 ], [ 44, 46, 1 ], [ 45, 35, -1 ], [ 46, 44, 1 ], [ 47, 43, 1 ], 
  [ 48, 42, 1 ], [ 49, 49, 1 ], [ 50, 54, 1 ], [ 51, 38, 1 ], [ 52, 55, -1 ], 
  [ 53, 56, 1 ], [ 54, 50, 1 ], [ 55, 52, -1 ], [ 56, 53, 1 ], [ 57, 59, 1 ], 
  [ 58, 60, 1 ], [ 59, 57, 1 ], [ 60, 58, 1 ], [ 61, 62, 1 ], [ 62, 61, 1 ], 
  [ 63, 63, -1 ], [ 64, 71, 1 ], [ 65, 65, 1 ], [ 66, 3, 1 ], [ 67, 80, 1 ], 
  [ 68, 5, 1 ], [ 69, 82, 1 ], [ 70, 7, 1 ], [ 71, 64, 1 ], [ 72, 85, 1 ], 
  [ 73, 74, 1 ], [ 74, 73, 1 ], [ 75, 76, 1 ], [ 76, 75, 1 ], [ 77, 84, 1 ], 
  [ 78, 79, 1 ], [ 79, 78, 1 ], [ 80, 67, 1 ], [ 81, 94, 1 ], [ 82, 69, 1 ], 
  [ 83, 89, 1 ], [ 84, 77, 1 ], [ 85, 72, 1 ], [ 86, 99, 1 ], [ 87, 88, 1 ], 
  [ 88, 87, 1 ], [ 89, 83, 1 ], [ 90, 97, 1 ], [ 91, 91, 1 ], [ 92, 93, 1 ], 
  [ 93, 92, 1 ], [ 94, 81, 1 ], [ 95, 100, -1 ], [ 96, 102, 1 ], 
  [ 97, 90, 1 ], [ 98, 108, -1 ], [ 99, 86, 1 ], [ 100, 95, -1 ], 
  [ 101, 114, 1 ], [ 102, 96, 1 ], [ 103, 104, -1 ], [ 104, 103, -1 ], 
  [ 105, 111, 1 ], [ 106, 110, 1 ], [ 107, 109, 1 ], [ 108, 98, -1 ], 
  [ 109, 107, 1 ], [ 110, 106, 1 ], [ 111, 105, 1 ], [ 112, 112, 1 ], 
  [ 113, 117, 1 ], [ 114, 101, 1 ], [ 115, 118, -1 ], [ 116, 119, 1 ], 
  [ 117, 113, 1 ], [ 118, 115, -1 ], [ 119, 116, 1 ], [ 120, 122, 1 ], 
  [ 121, 123, 1 ], [ 122, 120, 1 ], [ 123, 121, 1 ], [ 124, 125, 1 ], 
  [ 125, 124, 1 ], [ 126, 126, -1 ], [ 127, 127, 1 ], [ 128, 128, 1 ], 
  [ 129, 127, 1 ], [ 129, 129, -1 ], [ 129, 130, 1 ], [ 130, 130, 1 ], 
  [ 131, 130, 1 ], [ 131, 131, -1 ], [ 131, 132, 1 ], [ 132, 132, 1 ], 
  [ 133, 132, 1 ], [ 133, 133, -1 ] ] ],

[ [ 0, 0, 2, 0, 0, 0, 0 ],
[ 133, [ 1, 1, -1/4 ], [ 1, 64, 9/4 ], [ 1, 127, 3/2 ], [ 1, 129, -3/4 ], 
  [ 2, 2, 1/4 ], [ 2, 11, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 2, 16, -1/2*E(8)-1/2*E(8)^3 ], [ 2, 65, 1/4 ], 
  [ 2, 74, 1/8*E(8)+1/8*E(8)^3 ], [ 2, 79, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 2, 129, -1/2 ], [ 2, 130, 1/4 ], [ 2, 131, 1/2 ], [ 2, 132, -1/2 ], 
  [ 3, 17, -1/2*E(8)-1/2*E(8)^3 ], [ 3, 21, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 3, 22, -1/2*E(8)-1/2*E(8)^3 ], [ 3, 26, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 4, 6, -1/4 ], [ 4, 13, 1/4 ], [ 4, 18, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 4, 23, -1/4*E(8)-1/4*E(8)^3 ], [ 4, 25, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 4, 30, 1/4*E(8)+1/4*E(8)^3 ], [ 5, 81, 1/4 ], [ 5, 86, 1/4 ], 
  [ 5, 88, -1/4 ], [ 5, 93, -1/4 ], [ 6, 4, -1 ], [ 6, 9, -1 ], 
  [ 6, 18, -1/2*E(8)-1/2*E(8)^3 ], [ 6, 23, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 6, 25, -1/2*E(8)-1/2*E(8)^3 ], [ 6, 30, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 7, 7, 1/4 ], [ 7, 12, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 7, 19, 1/4*E(8)+1/4*E(8)^3 ], [ 7, 70, 1/4 ], 
  [ 7, 75, 1/4*E(8)+1/4*E(8)^3 ], [ 7, 82, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 7, 130, -1/2 ], [ 7, 131, 1/2 ], [ 7, 132, 1/4 ], 
  [ 8, 17, -3/2*E(8)-3/2*E(8)^3 ], [ 8, 21, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 8, 22, -3/2*E(8)-3/2*E(8)^3 ], [ 8, 26, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 9, 6, -1/4 ], [ 9, 13, 1/4 ], [ 9, 18, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 9, 23, 1/4*E(8)+1/4*E(8)^3 ], [ 9, 25, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 9, 30, -1/4*E(8)-1/4*E(8)^3 ], [ 10, 24, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 10, 27, 1/8*E(8)+1/8*E(8)^3 ], [ 10, 29, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 10, 31, -1/8*E(8)-1/8*E(8)^3 ], [ 10, 33, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 10, 34, -1/8*E(8)-1/8*E(8)^3 ], [ 10, 36, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 10, 39, -1/8*E(8)-1/8*E(8)^3 ], [ 10, 40, -1/2 ], [ 10, 43, -1/2 ], 
  [ 10, 45, 1/2 ], [ 10, 48, 1/2 ], [ 11, 2, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 11, 65, 1/8*E(8)+1/8*E(8)^3 ], [ 11, 74, -1/8 ], [ 11, 79, -1/8 ], 
  [ 11, 128, 1/4*E(8)+1/4*E(8)^3 ], [ 11, 130, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 12, 7, -1/4*E(8)-1/4*E(8)^3 ], [ 12, 70, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 12, 75, -1/2 ], [ 12, 82, 1/2 ], [ 12, 132, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 12, 133, 1/2*E(8)+1/2*E(8)^3 ], [ 13, 4, 1 ], [ 13, 9, 1 ], 
  [ 13, 18, -1/2*E(8)-1/2*E(8)^3 ], [ 13, 23, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 13, 25, -1/2*E(8)-1/2*E(8)^3 ], [ 13, 30, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 14, 24, 3/8*E(8)+3/8*E(8)^3 ], [ 14, 27, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 14, 29, 3/8*E(8)+3/8*E(8)^3 ], [ 14, 31, -3/8*E(8)-3/8*E(8)^3 ], 
  [ 14, 33, -1/8*E(8)-1/8*E(8)^3 ], [ 14, 34, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 14, 36, -3/8*E(8)-3/8*E(8)^3 ], [ 14, 39, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 14, 40, -3/2 ], [ 14, 43, 1/2 ], [ 14, 45, 3/2 ], [ 14, 48, -1/2 ], 
  [ 15, 24, 1/8*E(8)+1/8*E(8)^3 ], [ 15, 27, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 15, 29, 1/8*E(8)+1/8*E(8)^3 ], [ 15, 31, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 15, 33, 1/8*E(8)+1/8*E(8)^3 ], [ 15, 34, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 15, 36, -1/8*E(8)-1/8*E(8)^3 ], [ 15, 39, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 15, 40, 1/2 ], [ 15, 43, 1/2 ], [ 15, 45, -1/2 ], [ 15, 48, -1/2 ], 
  [ 16, 2, 1/8*E(8)+1/8*E(8)^3 ], [ 16, 65, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 16, 74, -1/8 ], [ 16, 79, -1/8 ], [ 16, 128, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 16, 130, 1/8*E(8)+1/8*E(8)^3 ], [ 17, 3, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 17, 8, 1/8*E(8)+1/8*E(8)^3 ], [ 17, 17, -1/4 ], [ 17, 21, -1/4 ], 
  [ 17, 22, 1/4 ], [ 17, 26, 1/4 ], [ 18, 4, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 18, 6, 1/8*E(8)+1/8*E(8)^3 ], [ 18, 9, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 18, 13, 1/8*E(8)+1/8*E(8)^3 ], [ 18, 18, 1/4 ], [ 18, 23, -1/4 ], 
  [ 18, 25, 1/4 ], [ 18, 30, -1/4 ], [ 18, 68, 1/4 ], 
  [ 19, 7, -1/4*E(8)-1/4*E(8)^3 ], [ 19, 70, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 19, 75, 1/2 ], [ 19, 82, -1/2 ], [ 19, 132, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 19, 133, 1/2*E(8)+1/2*E(8)^3 ], [ 20, 24, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 20, 27, -1/8*E(8)-1/8*E(8)^3 ], [ 20, 29, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 20, 31, -3/8*E(8)-3/8*E(8)^3 ], [ 20, 33, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 20, 34, 1/8*E(8)+1/8*E(8)^3 ], [ 20, 36, -3/8*E(8)-3/8*E(8)^3 ], 
  [ 20, 39, 1/8*E(8)+1/8*E(8)^3 ], [ 20, 40, 3/2 ], [ 20, 43, -1/2 ], 
  [ 20, 45, -3/2 ], [ 20, 48, 1/2 ], [ 21, 3, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 21, 8, -1/8*E(8)-1/8*E(8)^3 ], [ 21, 17, -3/4 ], [ 21, 21, 1/4 ], 
  [ 21, 22, 3/4 ], [ 21, 26, -1/4 ], [ 22, 3, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 22, 8, 1/8*E(8)+1/8*E(8)^3 ], [ 22, 17, 1/4 ], [ 22, 21, 1/4 ], 
  [ 22, 22, -1/4 ], [ 22, 26, -1/4 ], [ 23, 4, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 23, 6, 1/8*E(8)+1/8*E(8)^3 ], [ 23, 9, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 23, 13, 1/8*E(8)+1/8*E(8)^3 ], [ 23, 18, -1/4 ], [ 23, 23, 1/4 ], 
  [ 23, 25, -1/4 ], [ 23, 30, 1/4 ], [ 23, 68, 1/4 ], 
  [ 24, 10, -1/8*E(8)-1/8*E(8)^3 ], [ 24, 14, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 24, 15, -1/8*E(8)-1/8*E(8)^3 ], [ 24, 20, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 24, 24, 1/8 ], [ 24, 27, 1/8 ], [ 24, 28, -1/2 ], [ 24, 29, 1/8 ], 
  [ 24, 31, 1/8 ], [ 24, 32, -1/2 ], [ 24, 33, 1/8 ], [ 24, 34, 1/8 ], 
  [ 24, 36, 1/8 ], [ 24, 39, 1/8 ], [ 24, 40, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 24, 43, 1/4*E(8)+1/4*E(8)^3 ], [ 24, 45, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 24, 48, 1/4*E(8)+1/4*E(8)^3 ], [ 25, 4, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 25, 6, 1/8*E(8)+1/8*E(8)^3 ], [ 25, 9, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 25, 13, 1/8*E(8)+1/8*E(8)^3 ], [ 25, 18, 1/4 ], [ 25, 23, -1/4 ], 
  [ 25, 25, 1/4 ], [ 25, 30, -1/4 ], [ 25, 68, -1/4 ], 
  [ 26, 3, 3/8*E(8)+3/8*E(8)^3 ], [ 26, 8, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 26, 17, 3/4 ], [ 26, 21, -1/4 ], [ 26, 22, -3/4 ], [ 26, 26, 1/4 ], 
  [ 27, 10, -3/8*E(8)-3/8*E(8)^3 ], [ 27, 14, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 27, 15, -3/8*E(8)-3/8*E(8)^3 ], [ 27, 20, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 27, 24, 3/8 ], [ 27, 27, -1/8 ], [ 27, 28, -3/2 ], [ 27, 29, 3/8 ], 
  [ 27, 31, 3/8 ], [ 27, 32, 1/2 ], [ 27, 33, -1/8 ], [ 27, 34, -1/8 ], 
  [ 27, 36, 3/8 ], [ 27, 39, -1/8 ], [ 27, 40, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 27, 43, -1/4*E(8)-1/4*E(8)^3 ], [ 27, 45, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 27, 48, -1/4*E(8)-1/4*E(8)^3 ], [ 28, 24, -1/8 ], [ 28, 27, -1/8 ], 
  [ 28, 29, 1/8 ], [ 28, 31, 1/8 ], [ 28, 33, 1/8 ], [ 28, 34, 1/8 ], 
  [ 28, 36, -1/8 ], [ 28, 39, -1/8 ], [ 29, 10, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 29, 14, -1/8*E(8)-1/8*E(8)^3 ], [ 29, 15, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 29, 20, -1/8*E(8)-1/8*E(8)^3 ], [ 29, 24, 1/8 ], [ 29, 27, 1/8 ], 
  [ 29, 28, 1/2 ], [ 29, 29, 1/8 ], [ 29, 31, 1/8 ], [ 29, 32, 1/2 ], 
  [ 29, 33, 1/8 ], [ 29, 34, 1/8 ], [ 29, 36, 1/8 ], [ 29, 39, 1/8 ], 
  [ 29, 40, -1/4*E(8)-1/4*E(8)^3 ], [ 29, 43, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 29, 45, -1/4*E(8)-1/4*E(8)^3 ], [ 29, 48, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 30, 4, -1/4*E(8)-1/4*E(8)^3 ], [ 30, 6, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 30, 9, 1/4*E(8)+1/4*E(8)^3 ], [ 30, 13, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 30, 18, -1/4 ], [ 30, 23, 1/4 ], [ 30, 25, -1/4 ], [ 30, 30, 1/4 ], 
  [ 30, 68, -1/4 ], [ 31, 10, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 31, 14, 1/8*E(8)+1/8*E(8)^3 ], [ 31, 15, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 31, 20, 1/8*E(8)+1/8*E(8)^3 ], [ 31, 24, 1/8 ], [ 31, 27, 1/8 ], 
  [ 31, 28, 1/2 ], [ 31, 29, 1/8 ], [ 31, 31, 1/8 ], [ 31, 32, 1/2 ], 
  [ 31, 33, 1/8 ], [ 31, 34, 1/8 ], [ 31, 36, 1/8 ], [ 31, 39, 1/8 ], 
  [ 31, 40, 1/4*E(8)+1/4*E(8)^3 ], [ 31, 43, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 31, 45, 1/4*E(8)+1/4*E(8)^3 ], [ 31, 48, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 32, 24, -3/8 ], [ 32, 27, 1/8 ], [ 32, 29, 3/8 ], [ 32, 31, 3/8 ], 
  [ 32, 33, -1/8 ], [ 32, 34, -1/8 ], [ 32, 36, -3/8 ], [ 32, 39, 1/8 ], 
  [ 33, 10, -3/8*E(8)-3/8*E(8)^3 ], [ 33, 14, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 33, 15, -3/8*E(8)-3/8*E(8)^3 ], [ 33, 20, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 33, 24, 3/8 ], [ 33, 27, -1/8 ], [ 33, 28, 3/2 ], [ 33, 29, 3/8 ], 
  [ 33, 31, 3/8 ], [ 33, 32, -1/2 ], [ 33, 33, -1/8 ], [ 33, 34, -1/8 ], 
  [ 33, 36, 3/8 ], [ 33, 39, -1/8 ], [ 33, 40, -3/4*E(8)-3/4*E(8)^3 ], 
  [ 33, 43, 1/4*E(8)+1/4*E(8)^3 ], [ 33, 45, -3/4*E(8)-3/4*E(8)^3 ], 
  [ 33, 48, 1/4*E(8)+1/4*E(8)^3 ], [ 34, 10, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 34, 14, -1/8*E(8)-1/8*E(8)^3 ], [ 34, 15, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 34, 20, -1/8*E(8)-1/8*E(8)^3 ], [ 34, 24, 3/8 ], [ 34, 27, -1/8 ], 
  [ 34, 28, 3/2 ], [ 34, 29, 3/8 ], [ 34, 31, 3/8 ], [ 34, 32, -1/2 ], 
  [ 34, 33, -1/8 ], [ 34, 34, -1/8 ], [ 34, 36, 3/8 ], [ 34, 39, -1/8 ], 
  [ 34, 40, 3/4*E(8)+3/4*E(8)^3 ], [ 34, 43, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 34, 45, 3/4*E(8)+3/4*E(8)^3 ], [ 34, 48, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 35, 35, -1/4 ], [ 35, 38, -1/4 ], [ 35, 41, 1/4 ], [ 35, 44, 1/4 ], 
  [ 35, 49, 1/4*E(8)+1/4*E(8)^3 ], [ 35, 52, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 36, 10, 1/8*E(8)+1/8*E(8)^3 ], [ 36, 14, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 36, 15, 1/8*E(8)+1/8*E(8)^3 ], [ 36, 20, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 36, 24, 1/8 ], [ 36, 27, 1/8 ], [ 36, 28, -1/2 ], [ 36, 29, 1/8 ], 
  [ 36, 31, 1/8 ], [ 36, 32, -1/2 ], [ 36, 33, 1/8 ], [ 36, 34, 1/8 ], 
  [ 36, 36, 1/8 ], [ 36, 39, 1/8 ], [ 36, 40, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 36, 43, -1/4*E(8)-1/4*E(8)^3 ], [ 36, 45, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 36, 48, -1/4*E(8)-1/4*E(8)^3 ], [ 37, 46, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 37, 51, 1/2*E(8)+1/2*E(8)^3 ], [ 38, 35, -3/4 ], [ 38, 38, 1/4 ], 
  [ 38, 41, 3/4 ], [ 38, 44, -1/4 ], [ 38, 49, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 38, 52, -1/4*E(8)-1/4*E(8)^3 ], [ 39, 10, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 39, 14, -1/8*E(8)-1/8*E(8)^3 ], [ 39, 15, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 39, 20, -1/8*E(8)-1/8*E(8)^3 ], [ 39, 24, 3/8 ], [ 39, 27, -1/8 ], 
  [ 39, 28, -3/2 ], [ 39, 29, 3/8 ], [ 39, 31, 3/8 ], [ 39, 32, 1/2 ], 
  [ 39, 33, -1/8 ], [ 39, 34, -1/8 ], [ 39, 36, 3/8 ], [ 39, 39, -1/8 ], 
  [ 39, 40, -3/4*E(8)-3/4*E(8)^3 ], [ 39, 43, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 39, 45, -3/4*E(8)-3/4*E(8)^3 ], [ 39, 48, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 40, 10, -1/8 ], [ 40, 14, -1/8 ], [ 40, 15, 1/8 ], [ 40, 20, 1/8 ], 
  [ 40, 24, -1/16*E(8)-1/16*E(8)^3 ], [ 40, 27, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 40, 29, 1/16*E(8)+1/16*E(8)^3 ], [ 40, 31, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 40, 33, 1/16*E(8)+1/16*E(8)^3 ], [ 40, 34, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 40, 36, 1/16*E(8)+1/16*E(8)^3 ], [ 40, 39, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 41, 35, 1/4 ], [ 41, 38, 1/4 ], [ 41, 41, -1/4 ], [ 41, 44, -1/4 ], 
  [ 41, 49, 1/4*E(8)+1/4*E(8)^3 ], [ 41, 52, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 42, 50, 1/2*E(8)+1/2*E(8)^3 ], [ 42, 53, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 42, 54, -1/2*E(8)-1/2*E(8)^3 ], [ 42, 56, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 42, 59, -1 ], [ 42, 60, 1 ], [ 43, 10, -3/8 ], [ 43, 14, 1/8 ], 
  [ 43, 15, 3/8 ], [ 43, 20, -1/8 ], [ 43, 24, -3/16*E(8)-3/16*E(8)^3 ], 
  [ 43, 27, 1/16*E(8)+1/16*E(8)^3 ], [ 43, 29, 3/16*E(8)+3/16*E(8)^3 ], 
  [ 43, 31, -3/16*E(8)-3/16*E(8)^3 ], [ 43, 33, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 43, 34, 1/16*E(8)+1/16*E(8)^3 ], [ 43, 36, 3/16*E(8)+3/16*E(8)^3 ], 
  [ 43, 39, -1/16*E(8)-1/16*E(8)^3 ], [ 44, 35, 3/4 ], [ 44, 38, -1/4 ], 
  [ 44, 41, -3/4 ], [ 44, 44, 1/4 ], [ 44, 49, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 44, 52, -1/4*E(8)-1/4*E(8)^3 ], [ 45, 10, 1/8 ], [ 45, 14, 1/8 ], 
  [ 45, 15, -1/8 ], [ 45, 20, -1/8 ], [ 45, 24, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 45, 27, -1/16*E(8)-1/16*E(8)^3 ], [ 45, 29, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 45, 31, -1/16*E(8)-1/16*E(8)^3 ], [ 45, 33, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 45, 34, -1/16*E(8)-1/16*E(8)^3 ], [ 45, 36, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 45, 39, 1/16*E(8)+1/16*E(8)^3 ], [ 46, 37, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 46, 46, 1/2 ], [ 46, 51, 1/2 ], [ 47, 50, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 47, 53, 1/2*E(8)+1/2*E(8)^3 ], [ 47, 54, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 47, 56, -1/2*E(8)-1/2*E(8)^3 ], [ 47, 59, -1 ], [ 47, 60, 1 ], 
  [ 48, 10, 3/8 ], [ 48, 14, -1/8 ], [ 48, 15, -3/8 ], [ 48, 20, 1/8 ], 
  [ 48, 24, -3/16*E(8)-3/16*E(8)^3 ], [ 48, 27, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 48, 29, 3/16*E(8)+3/16*E(8)^3 ], [ 48, 31, -3/16*E(8)-3/16*E(8)^3 ], 
  [ 48, 33, -1/16*E(8)-1/16*E(8)^3 ], [ 48, 34, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 48, 36, 3/16*E(8)+3/16*E(8)^3 ], [ 48, 39, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 49, 35, -1/4*E(8)-1/4*E(8)^3 ], [ 49, 38, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 49, 41, -1/4*E(8)-1/4*E(8)^3 ], [ 49, 44, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 50, 42, -1/8*E(8)-1/8*E(8)^3 ], [ 50, 47, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 50, 50, -1/4 ], [ 50, 53, -1/4 ], [ 50, 54, 1/4 ], [ 50, 55, -1/4 ], 
  [ 50, 56, 1/4 ], [ 50, 59, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 50, 60, -1/4*E(8)-1/4*E(8)^3 ], [ 51, 37, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 51, 46, 1/2 ], [ 51, 51, 1/2 ], [ 52, 35, -3/4*E(8)-3/4*E(8)^3 ], 
  [ 52, 38, 1/4*E(8)+1/4*E(8)^3 ], [ 52, 41, -3/4*E(8)-3/4*E(8)^3 ], 
  [ 52, 44, 1/4*E(8)+1/4*E(8)^3 ], [ 53, 42, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 53, 47, -1/8*E(8)-1/8*E(8)^3 ], [ 53, 50, -1/4 ], [ 53, 53, -1/4 ], 
  [ 53, 54, 1/4 ], [ 53, 55, 1/4 ], [ 53, 56, 1/4 ], 
  [ 53, 59, -1/4*E(8)-1/4*E(8)^3 ], [ 53, 60, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 54, 42, 1/8*E(8)+1/8*E(8)^3 ], [ 54, 47, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 54, 50, 1/4 ], [ 54, 53, 1/4 ], [ 54, 54, -1/4 ], [ 54, 55, -1/4 ], 
  [ 54, 56, -1/4 ], [ 54, 59, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 54, 60, -1/4*E(8)-1/4*E(8)^3 ], [ 55, 50, -1 ], [ 55, 53, 1 ], 
  [ 55, 54, -1 ], [ 55, 56, 1 ], [ 56, 42, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 56, 47, 1/8*E(8)+1/8*E(8)^3 ], [ 56, 50, 1/4 ], [ 56, 53, 1/4 ], 
  [ 56, 54, -1/4 ], [ 56, 55, 1/4 ], [ 56, 56, -1/4 ], 
  [ 56, 59, -1/4*E(8)-1/4*E(8)^3 ], [ 56, 60, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 57, 57, 1/2 ], [ 57, 58, -1/2 ], [ 57, 61, E(8)+E(8)^3 ], 
  [ 58, 57, -1/2 ], [ 58, 58, 1/2 ], [ 58, 61, E(8)+E(8)^3 ], 
  [ 59, 42, -1/4 ], [ 59, 47, -1/4 ], [ 59, 50, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 59, 53, 1/4*E(8)+1/4*E(8)^3 ], [ 59, 54, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 59, 56, 1/4*E(8)+1/4*E(8)^3 ], [ 60, 42, 1/4 ], [ 60, 47, 1/4 ], 
  [ 60, 50, 1/4*E(8)+1/4*E(8)^3 ], [ 60, 53, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 60, 54, 1/4*E(8)+1/4*E(8)^3 ], [ 60, 56, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 61, 57, -1/4*E(8)-1/4*E(8)^3 ], [ 61, 58, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 62, 62, -1/2 ], [ 62, 63, 1/2 ], [ 63, 62, 3/2 ], [ 63, 63, 1/2 ], 
  [ 64, 1, 1/4 ], [ 64, 64, -1/4 ], [ 64, 127, 1/2 ], [ 64, 129, -1/4 ], 
  [ 65, 2, 1/4 ], [ 65, 11, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 65, 16, 1/2*E(8)+1/2*E(8)^3 ], [ 65, 65, 1/4 ], 
  [ 65, 74, -1/8*E(8)-1/8*E(8)^3 ], [ 65, 79, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 65, 129, -1/2 ], [ 65, 130, 1/4 ], [ 65, 131, 1/2 ], [ 65, 132, -1/2 ], 
  [ 66, 80, 1/8*E(8)+1/8*E(8)^3 ], [ 66, 84, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 66, 85, 1/8*E(8)+1/8*E(8)^3 ], [ 66, 89, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 67, 69, -1 ], [ 67, 76, 1 ], [ 67, 81, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 67, 86, 1/4*E(8)+1/4*E(8)^3 ], [ 67, 88, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 67, 93, -1/4*E(8)-1/4*E(8)^3 ], [ 68, 18, 1 ], [ 68, 23, 1 ], 
  [ 68, 25, -1 ], [ 68, 30, -1 ], [ 69, 67, -1/4 ], [ 69, 72, -1/4 ], 
  [ 69, 81, 1/8*E(8)+1/8*E(8)^3 ], [ 69, 86, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 69, 88, 1/8*E(8)+1/8*E(8)^3 ], [ 69, 93, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 70, 7, 1/4 ], [ 70, 12, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 70, 19, -1/4*E(8)-1/4*E(8)^3 ], [ 70, 70, 1/4 ], 
  [ 70, 75, -1/4*E(8)-1/4*E(8)^3 ], [ 70, 82, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 70, 130, -1/2 ], [ 70, 131, 1/2 ], [ 70, 132, 1/4 ], 
  [ 71, 80, 1/8*E(8)+1/8*E(8)^3 ], [ 71, 84, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 71, 85, 1/8*E(8)+1/8*E(8)^3 ], [ 71, 89, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 72, 69, -1 ], [ 72, 76, 1 ], [ 72, 81, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 72, 86, -1/4*E(8)-1/4*E(8)^3 ], [ 72, 88, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 72, 93, 1/4*E(8)+1/4*E(8)^3 ], [ 73, 87, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 73, 90, -3/8*E(8)-3/8*E(8)^3 ], [ 73, 92, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 73, 94, 1/8*E(8)+1/8*E(8)^3 ], [ 73, 96, -3/8*E(8)-3/8*E(8)^3 ], 
  [ 73, 97, 3/8*E(8)+3/8*E(8)^3 ], [ 73, 99, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 73, 102, 3/8*E(8)+3/8*E(8)^3 ], [ 73, 103, -1/8 ], [ 73, 106, -3/8 ], 
  [ 73, 108, 1/8 ], [ 73, 111, 3/8 ], [ 74, 2, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 74, 11, -2 ], [ 74, 16, -2 ], [ 74, 65, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 74, 128, -E(8)-E(8)^3 ], [ 74, 130, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 75, 7, -1/4*E(8)-1/4*E(8)^3 ], [ 75, 12, -1/2 ], [ 75, 19, 1/2 ], 
  [ 75, 70, 1/4*E(8)+1/4*E(8)^3 ], [ 75, 132, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 75, 133, -1/2*E(8)-1/2*E(8)^3 ], [ 76, 67, 1/4 ], [ 76, 72, 1/4 ], 
  [ 76, 81, 1/8*E(8)+1/8*E(8)^3 ], [ 76, 86, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 76, 88, 1/8*E(8)+1/8*E(8)^3 ], [ 76, 93, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 77, 87, -1/8*E(8)-1/8*E(8)^3 ], [ 77, 90, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 77, 92, -1/8*E(8)-1/8*E(8)^3 ], [ 77, 94, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 77, 96, 1/8*E(8)+1/8*E(8)^3 ], [ 77, 97, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 77, 99, 1/8*E(8)+1/8*E(8)^3 ], [ 77, 102, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 77, 103, -1/8 ], [ 77, 106, 1/8 ], [ 77, 108, 1/8 ], [ 77, 111, -1/8 ], 
  [ 78, 87, -1/8*E(8)-1/8*E(8)^3 ], [ 78, 90, -3/8*E(8)-3/8*E(8)^3 ], 
  [ 78, 92, -1/8*E(8)-1/8*E(8)^3 ], [ 78, 94, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 78, 96, -3/8*E(8)-3/8*E(8)^3 ], [ 78, 97, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 78, 99, 1/8*E(8)+1/8*E(8)^3 ], [ 78, 102, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 78, 103, 1/8 ], [ 78, 106, 3/8 ], [ 78, 108, -1/8 ], [ 78, 111, -3/8 ], 
  [ 79, 2, 1/2*E(8)+1/2*E(8)^3 ], [ 79, 11, -2 ], [ 79, 16, -2 ], 
  [ 79, 65, -1/2*E(8)-1/2*E(8)^3 ], [ 79, 128, E(8)+E(8)^3 ], 
  [ 79, 130, -1/2*E(8)-1/2*E(8)^3 ], [ 80, 66, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 80, 71, -3/2*E(8)-3/2*E(8)^3 ], [ 80, 80, -1/4 ], [ 80, 84, -3/4 ], 
  [ 80, 85, 1/4 ], [ 80, 89, 3/4 ], [ 81, 5, 1 ], 
  [ 81, 67, 1/4*E(8)+1/4*E(8)^3 ], [ 81, 69, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 81, 72, -1/4*E(8)-1/4*E(8)^3 ], [ 81, 76, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 81, 81, 1/4 ], [ 81, 86, -1/4 ], [ 81, 88, 1/4 ], [ 81, 93, -1/4 ], 
  [ 82, 7, -1/4*E(8)-1/4*E(8)^3 ], [ 82, 12, 1/2 ], [ 82, 19, -1/2 ], 
  [ 82, 70, 1/4*E(8)+1/4*E(8)^3 ], [ 82, 132, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 82, 133, -1/2*E(8)-1/2*E(8)^3 ], [ 83, 87, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 83, 90, 1/8*E(8)+1/8*E(8)^3 ], [ 83, 92, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 83, 94, 1/8*E(8)+1/8*E(8)^3 ], [ 83, 96, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 83, 97, -1/8*E(8)-1/8*E(8)^3 ], [ 83, 99, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 83, 102, -1/8*E(8)-1/8*E(8)^3 ], [ 83, 103, 1/8 ], [ 83, 106, -1/8 ], 
  [ 83, 108, -1/8 ], [ 83, 111, 1/8 ], [ 84, 66, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 84, 71, 1/2*E(8)+1/2*E(8)^3 ], [ 84, 80, -1/4 ], [ 84, 84, 1/4 ], 
  [ 84, 85, 1/4 ], [ 84, 89, -1/4 ], [ 85, 66, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 85, 71, -3/2*E(8)-3/2*E(8)^3 ], [ 85, 80, 1/4 ], [ 85, 84, 3/4 ], 
  [ 85, 85, -1/4 ], [ 85, 89, -3/4 ], [ 86, 5, 1 ], 
  [ 86, 67, -1/4*E(8)-1/4*E(8)^3 ], [ 86, 69, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 86, 72, 1/4*E(8)+1/4*E(8)^3 ], [ 86, 76, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 86, 81, -1/4 ], [ 86, 86, 1/4 ], [ 86, 88, -1/4 ], [ 86, 93, 1/4 ], 
  [ 87, 73, 1/8*E(8)+1/8*E(8)^3 ], [ 87, 77, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 87, 78, 1/8*E(8)+1/8*E(8)^3 ], [ 87, 83, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 87, 87, 1/8 ], [ 87, 90, 3/8 ], [ 87, 91, -1/8 ], [ 87, 92, 1/8 ], 
  [ 87, 94, 1/8 ], [ 87, 95, -3/8 ], [ 87, 96, 3/8 ], [ 87, 97, 3/8 ], 
  [ 87, 99, 1/8 ], [ 87, 102, 3/8 ], [ 87, 103, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 87, 106, -3/16*E(8)-3/16*E(8)^3 ], [ 87, 108, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 87, 111, -3/16*E(8)-3/16*E(8)^3 ], [ 88, 5, -1 ], 
  [ 88, 67, -1/4*E(8)-1/4*E(8)^3 ], [ 88, 69, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 88, 72, 1/4*E(8)+1/4*E(8)^3 ], [ 88, 76, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 88, 81, 1/4 ], [ 88, 86, -1/4 ], [ 88, 88, 1/4 ], [ 88, 93, -1/4 ], 
  [ 89, 66, -1/2*E(8)-1/2*E(8)^3 ], [ 89, 71, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 89, 80, 1/4 ], [ 89, 84, -1/4 ], [ 89, 85, -1/4 ], [ 89, 89, 1/4 ], 
  [ 90, 73, 1/8*E(8)+1/8*E(8)^3 ], [ 90, 77, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 90, 78, 1/8*E(8)+1/8*E(8)^3 ], [ 90, 83, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 90, 87, 1/8 ], [ 90, 90, -1/8 ], [ 90, 91, -1/8 ], [ 90, 92, 1/8 ], 
  [ 90, 94, 1/8 ], [ 90, 95, 1/8 ], [ 90, 96, -1/8 ], [ 90, 97, -1/8 ], 
  [ 90, 99, 1/8 ], [ 90, 102, -1/8 ], [ 90, 103, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 90, 106, 1/16*E(8)+1/16*E(8)^3 ], [ 90, 108, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 90, 111, 1/16*E(8)+1/16*E(8)^3 ], [ 91, 87, -1/2 ], [ 91, 90, -3/2 ], 
  [ 91, 92, 1/2 ], [ 91, 94, 1/2 ], [ 91, 96, 3/2 ], [ 91, 97, 3/2 ], 
  [ 91, 99, -1/2 ], [ 91, 102, -3/2 ], [ 92, 73, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 92, 77, 3/8*E(8)+3/8*E(8)^3 ], [ 92, 78, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 92, 83, 3/8*E(8)+3/8*E(8)^3 ], [ 92, 87, 1/8 ], [ 92, 90, 3/8 ], 
  [ 92, 91, 1/8 ], [ 92, 92, 1/8 ], [ 92, 94, 1/8 ], [ 92, 95, 3/8 ], 
  [ 92, 96, 3/8 ], [ 92, 97, 3/8 ], [ 92, 99, 1/8 ], [ 92, 102, 3/8 ], 
  [ 92, 103, 1/16*E(8)+1/16*E(8)^3 ], [ 92, 106, 3/16*E(8)+3/16*E(8)^3 ], 
  [ 92, 108, 1/16*E(8)+1/16*E(8)^3 ], [ 92, 111, 3/16*E(8)+3/16*E(8)^3 ], 
  [ 93, 5, -1 ], [ 93, 67, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 93, 69, -1/2*E(8)-1/2*E(8)^3 ], [ 93, 72, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 93, 76, -1/2*E(8)-1/2*E(8)^3 ], [ 93, 81, -1/4 ], [ 93, 86, 1/4 ], 
  [ 93, 88, -1/4 ], [ 93, 93, 1/4 ], [ 94, 73, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 94, 77, -3/8*E(8)-3/8*E(8)^3 ], [ 94, 78, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 94, 83, -3/8*E(8)-3/8*E(8)^3 ], [ 94, 87, 1/8 ], [ 94, 90, 3/8 ], 
  [ 94, 91, 1/8 ], [ 94, 92, 1/8 ], [ 94, 94, 1/8 ], [ 94, 95, 3/8 ], 
  [ 94, 96, 3/8 ], [ 94, 97, 3/8 ], [ 94, 99, 1/8 ], [ 94, 102, 3/8 ], 
  [ 94, 103, -1/16*E(8)-1/16*E(8)^3 ], [ 94, 106, -3/16*E(8)-3/16*E(8)^3 ], 
  [ 94, 108, -1/16*E(8)-1/16*E(8)^3 ], [ 94, 111, -3/16*E(8)-3/16*E(8)^3 ], 
  [ 95, 87, -1/2 ], [ 95, 90, 1/2 ], [ 95, 92, 1/2 ], [ 95, 94, 1/2 ], 
  [ 95, 96, -1/2 ], [ 95, 97, -1/2 ], [ 95, 99, -1/2 ], [ 95, 102, 1/2 ], 
  [ 96, 73, 1/8*E(8)+1/8*E(8)^3 ], [ 96, 77, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 96, 78, 1/8*E(8)+1/8*E(8)^3 ], [ 96, 83, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 96, 87, 1/8 ], [ 96, 90, -1/8 ], [ 96, 91, 1/8 ], [ 96, 92, 1/8 ], 
  [ 96, 94, 1/8 ], [ 96, 95, -1/8 ], [ 96, 96, -1/8 ], [ 96, 97, -1/8 ], 
  [ 96, 99, 1/8 ], [ 96, 102, -1/8 ], [ 96, 103, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 96, 106, -1/16*E(8)-1/16*E(8)^3 ], [ 96, 108, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 96, 111, -1/16*E(8)-1/16*E(8)^3 ], [ 97, 73, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 97, 77, 1/8*E(8)+1/8*E(8)^3 ], [ 97, 78, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 97, 83, 1/8*E(8)+1/8*E(8)^3 ], [ 97, 87, 1/8 ], [ 97, 90, -1/8 ], 
  [ 97, 91, 1/8 ], [ 97, 92, 1/8 ], [ 97, 94, 1/8 ], [ 97, 95, -1/8 ], 
  [ 97, 96, -1/8 ], [ 97, 97, -1/8 ], [ 97, 99, 1/8 ], [ 97, 102, -1/8 ], 
  [ 97, 103, -1/16*E(8)-1/16*E(8)^3 ], [ 97, 106, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 97, 108, -1/16*E(8)-1/16*E(8)^3 ], [ 97, 111, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 98, 98, -1/4 ], [ 98, 101, -3/4 ], [ 98, 104, 1/4 ], [ 98, 107, 3/4 ], 
  [ 98, 112, -1/4*E(8)-1/4*E(8)^3 ], [ 98, 115, -3/4*E(8)-3/4*E(8)^3 ], 
  [ 99, 73, -1/8*E(8)-1/8*E(8)^3 ], [ 99, 77, -3/8*E(8)-3/8*E(8)^3 ], 
  [ 99, 78, -1/8*E(8)-1/8*E(8)^3 ], [ 99, 83, -3/8*E(8)-3/8*E(8)^3 ], 
  [ 99, 87, 1/8 ], [ 99, 90, 3/8 ], [ 99, 91, -1/8 ], [ 99, 92, 1/8 ], 
  [ 99, 94, 1/8 ], [ 99, 95, -3/8 ], [ 99, 96, 3/8 ], [ 99, 97, 3/8 ], 
  [ 99, 99, 1/8 ], [ 99, 102, 3/8 ], [ 99, 103, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 99, 106, 3/16*E(8)+3/16*E(8)^3 ], [ 99, 108, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 99, 111, 3/16*E(8)+3/16*E(8)^3 ], [ 100, 109, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 100, 114, -1/2*E(8)-1/2*E(8)^3 ], [ 101, 98, -1/4 ], [ 101, 101, 1/4 ], 
  [ 101, 104, 1/4 ], [ 101, 107, -1/4 ], [ 101, 112, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 101, 115, 1/4*E(8)+1/4*E(8)^3 ], [ 102, 73, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 102, 77, 1/8*E(8)+1/8*E(8)^3 ], [ 102, 78, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 102, 83, 1/8*E(8)+1/8*E(8)^3 ], [ 102, 87, 1/8 ], [ 102, 90, -1/8 ], 
  [ 102, 91, -1/8 ], [ 102, 92, 1/8 ], [ 102, 94, 1/8 ], [ 102, 95, 1/8 ], 
  [ 102, 96, -1/8 ], [ 102, 97, -1/8 ], [ 102, 99, 1/8 ], [ 102, 102, -1/8 ],
  [ 102, 103, 1/16*E(8)+1/16*E(8)^3 ], [ 102, 106, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 102, 108, 1/16*E(8)+1/16*E(8)^3 ], [ 102, 111, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 103, 73, -1/2 ], [ 103, 77, -3/2 ], [ 103, 78, 1/2 ], [ 103, 83, 3/2 ], 
  [ 103, 87, 1/4*E(8)+1/4*E(8)^3 ], [ 103, 90, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 103, 92, -1/4*E(8)-1/4*E(8)^3 ], [ 103, 94, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 103, 96, -3/4*E(8)-3/4*E(8)^3 ], [ 103, 97, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 103, 99, -1/4*E(8)-1/4*E(8)^3 ], [ 103, 102, -3/4*E(8)-3/4*E(8)^3 ], 
  [ 104, 98, 1/4 ], [ 104, 101, 3/4 ], [ 104, 104, -1/4 ], 
  [ 104, 107, -3/4 ], [ 104, 112, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 104, 115, -3/4*E(8)-3/4*E(8)^3 ], [ 105, 113, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 105, 116, 1/8*E(8)+1/8*E(8)^3 ], [ 105, 117, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 105, 119, -1/8*E(8)-1/8*E(8)^3 ], [ 105, 122, -1/4 ], [ 105, 123, 1/4 ], 
  [ 106, 73, -1/2 ], [ 106, 77, 1/2 ], [ 106, 78, 1/2 ], [ 106, 83, -1/2 ], 
  [ 106, 87, 1/4*E(8)+1/4*E(8)^3 ], [ 106, 90, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 106, 92, -1/4*E(8)-1/4*E(8)^3 ], [ 106, 94, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 106, 96, 1/4*E(8)+1/4*E(8)^3 ], [ 106, 97, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 106, 99, -1/4*E(8)-1/4*E(8)^3 ], [ 106, 102, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 107, 98, 1/4 ], [ 107, 101, -1/4 ], [ 107, 104, -1/4 ], 
  [ 107, 107, 1/4 ], [ 107, 112, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 107, 115, 1/4*E(8)+1/4*E(8)^3 ], [ 108, 73, 1/2 ], [ 108, 77, 3/2 ], 
  [ 108, 78, -1/2 ], [ 108, 83, -3/2 ], [ 108, 87, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 108, 90, 3/4*E(8)+3/4*E(8)^3 ], [ 108, 92, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 108, 94, 1/4*E(8)+1/4*E(8)^3 ], [ 108, 96, -3/4*E(8)-3/4*E(8)^3 ], 
  [ 108, 97, 3/4*E(8)+3/4*E(8)^3 ], [ 108, 99, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 108, 102, -3/4*E(8)-3/4*E(8)^3 ], [ 109, 100, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 109, 109, 1/2 ], [ 109, 114, 1/2 ], [ 110, 113, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 110, 116, -1/8*E(8)-1/8*E(8)^3 ], [ 110, 117, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 110, 119, 1/8*E(8)+1/8*E(8)^3 ], [ 110, 122, -1/4 ], [ 110, 123, 1/4 ], 
  [ 111, 73, 1/2 ], [ 111, 77, -1/2 ], [ 111, 78, -1/2 ], [ 111, 83, 1/2 ], 
  [ 111, 87, 1/4*E(8)+1/4*E(8)^3 ], [ 111, 90, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 111, 92, -1/4*E(8)-1/4*E(8)^3 ], [ 111, 94, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 111, 96, 1/4*E(8)+1/4*E(8)^3 ], [ 111, 97, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 111, 99, -1/4*E(8)-1/4*E(8)^3 ], [ 111, 102, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 112, 98, 1/4*E(8)+1/4*E(8)^3 ], [ 112, 101, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 112, 104, 1/4*E(8)+1/4*E(8)^3 ], [ 112, 107, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 113, 105, 1/2*E(8)+1/2*E(8)^3 ], [ 113, 110, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 113, 113, -1/4 ], [ 113, 116, -1/4 ], [ 113, 117, 1/4 ], 
  [ 113, 118, -1 ], [ 113, 119, 1/4 ], [ 113, 122, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 113, 123, 1/4*E(8)+1/4*E(8)^3 ], [ 114, 100, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 114, 109, 1/2 ], [ 114, 114, 1/2 ], [ 115, 98, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 115, 101, -1/4*E(8)-1/4*E(8)^3 ], [ 115, 104, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 115, 107, -1/4*E(8)-1/4*E(8)^3 ], [ 116, 105, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 116, 110, 1/2*E(8)+1/2*E(8)^3 ], [ 116, 113, -1/4 ], [ 116, 116, -1/4 ], 
  [ 116, 117, 1/4 ], [ 116, 118, 1 ], [ 116, 119, 1/4 ], 
  [ 116, 122, 1/4*E(8)+1/4*E(8)^3 ], [ 116, 123, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 117, 105, -1/2*E(8)-1/2*E(8)^3 ], [ 117, 110, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 117, 113, 1/4 ], [ 117, 116, 1/4 ], [ 117, 117, -1/4 ], [ 117, 118, -1 ],
  [ 117, 119, -1/4 ], [ 117, 122, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 117, 123, 1/4*E(8)+1/4*E(8)^3 ], [ 118, 113, -1/4 ], [ 118, 116, 1/4 ], 
  [ 118, 117, -1/4 ], [ 118, 119, 1/4 ], [ 119, 105, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 119, 110, -1/2*E(8)-1/2*E(8)^3 ], [ 119, 113, 1/4 ], [ 119, 116, 1/4 ], 
  [ 119, 117, -1/4 ], [ 119, 118, 1 ], [ 119, 119, -1/4 ], 
  [ 119, 122, 1/4*E(8)+1/4*E(8)^3 ], [ 119, 123, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 120, 120, 1/2 ], [ 120, 121, -1/2 ], [ 120, 124, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 121, 120, -1/2 ], [ 121, 121, 1/2 ], [ 121, 124, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 122, 105, -1 ], [ 122, 110, -1 ], [ 122, 113, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 122, 116, -1/4*E(8)-1/4*E(8)^3 ], [ 122, 117, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 122, 119, -1/4*E(8)-1/4*E(8)^3 ], [ 123, 105, 1 ], [ 123, 110, 1 ], 
  [ 123, 113, -1/4*E(8)-1/4*E(8)^3 ], [ 123, 116, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 123, 117, -1/4*E(8)-1/4*E(8)^3 ], [ 123, 119, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 124, 120, E(8)+E(8)^3 ], [ 124, 121, E(8)+E(8)^3 ], [ 125, 125, -1/2 ], 
  [ 125, 126, 3/2 ], [ 126, 125, 1/2 ], [ 126, 126, 1/2 ], [ 127, 1, 1/4 ], 
  [ 127, 64, 3/4 ], [ 127, 127, -1/2 ], [ 127, 129, 3/4 ], [ 128, 2, 1/4 ], 
  [ 128, 11, -1/2*E(8)-1/2*E(8)^3 ], [ 128, 16, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 128, 65, 1/4 ], [ 128, 74, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 128, 79, -1/8*E(8)-1/8*E(8)^3 ], [ 128, 129, 1/2 ], [ 128, 130, 1/4 ], 
  [ 128, 131, -1/2 ], [ 128, 132, 1/2 ], [ 129, 129, 1 ], [ 130, 2, 1/2 ], 
  [ 130, 65, 1/2 ], [ 130, 129, 1 ], [ 130, 130, 1/2 ], [ 130, 131, -1 ], 
  [ 130, 132, 1 ], [ 131, 2, 1/2 ], [ 131, 7, 1/2 ], [ 131, 65, 1/2 ], 
  [ 131, 70, 1/2 ], [ 131, 129, 1 ], [ 131, 130, 1/2 ], [ 131, 131, -1 ], 
  [ 131, 132, 1/2 ], [ 132, 7, 1/2 ], [ 132, 70, 1/2 ], [ 132, 130, 1 ], 
  [ 132, 131, -1 ], [ 132, 132, 1/2 ], [ 133, 7, 1/4 ], 
  [ 133, 12, -1/4*E(8)-1/4*E(8)^3 ], [ 133, 19, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 133, 70, 1/4 ], [ 133, 75, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 133, 82, 1/4*E(8)+1/4*E(8)^3 ], [ 133, 130, 1/2 ], [ 133, 131, -1/2 ], 
  [ 133, 132, 1/4 ] ],
[ 133, [ 1, 1, -1 ], [ 2, 70, 1 ], [ 3, 49, -1 ], [ 4, 75, 1 ], [ 5, 5, 1 ], 
  [ 6, 74, 1 ], [ 7, 65, 1 ], [ 8, 52, 1 ], [ 9, 82, 1 ], [ 10, 41, -1 ], 
  [ 11, 69, 1 ], [ 12, 67, 1 ], [ 13, 79, 1 ], [ 14, 44, 1 ], [ 15, 35, 1 ], 
  [ 16, 76, 1 ], [ 17, 45, -1 ], [ 18, 81, 1 ], [ 19, 72, 1 ], 
  [ 20, 38, -1 ], [ 21, 48, 1 ], [ 22, 40, 1 ], [ 23, 88, 1 ], 
  [ 24, 36, -1 ], [ 25, 86, 1 ], [ 26, 43, -1 ], [ 27, 39, 1 ], 
  [ 28, 28, -1 ], [ 29, 29, 1 ], [ 30, 93, 1 ], [ 31, 31, 1 ], 
  [ 32, 32, 1 ], [ 33, 33, -1 ], [ 34, 34, -1 ], [ 35, 15, 1 ], 
  [ 36, 24, -1 ], [ 37, 61, 1 ], [ 38, 20, -1 ], [ 39, 27, 1 ], 
  [ 40, 22, 1 ], [ 41, 10, -1 ], [ 42, 58, -1 ], [ 43, 26, -1 ], 
  [ 44, 14, 1 ], [ 45, 17, -1 ], [ 46, 60, -1 ], [ 47, 57, -1 ], 
  [ 48, 21, 1 ], [ 49, 3, -1 ], [ 50, 56, 1 ], [ 51, 59, -1 ], [ 52, 8, 1 ], 
  [ 53, 53, 1 ], [ 54, 54, 1 ], [ 55, 55, -1 ], [ 56, 50, 1 ], 
  [ 57, 47, -1 ], [ 58, 42, -1 ], [ 59, 51, -1 ], [ 60, 46, -1 ], 
  [ 61, 37, 1 ], [ 62, 62, 1 ], [ 63, 63, -1 ], [ 64, 64, -1 ], 
  [ 65, 7, 1 ], [ 66, 112, -1 ], [ 67, 12, 1 ], [ 68, 68, 1 ], 
  [ 69, 11, 1 ], [ 70, 2, 1 ], [ 71, 115, 1 ], [ 72, 19, 1 ], 
  [ 73, 104, -1 ], [ 74, 6, 1 ], [ 75, 4, 1 ], [ 76, 16, 1 ], 
  [ 77, 107, 1 ], [ 78, 98, 1 ], [ 79, 13, 1 ], [ 80, 108, -1 ], 
  [ 81, 18, 1 ], [ 82, 9, 1 ], [ 83, 101, -1 ], [ 84, 111, 1 ], 
  [ 85, 103, 1 ], [ 86, 25, 1 ], [ 87, 99, -1 ], [ 88, 23, 1 ], 
  [ 89, 106, -1 ], [ 90, 102, 1 ], [ 91, 91, -1 ], [ 92, 92, 1 ], 
  [ 93, 30, 1 ], [ 94, 94, 1 ], [ 95, 95, 1 ], [ 96, 96, -1 ], 
  [ 97, 97, -1 ], [ 98, 78, 1 ], [ 99, 87, -1 ], [ 100, 124, 1 ], 
  [ 101, 83, -1 ], [ 102, 90, 1 ], [ 103, 85, 1 ], [ 104, 73, -1 ], 
  [ 105, 121, -1 ], [ 106, 89, -1 ], [ 107, 77, 1 ], [ 108, 80, -1 ], 
  [ 109, 123, -1 ], [ 110, 120, -1 ], [ 111, 84, 1 ], [ 112, 66, -1 ], 
  [ 113, 119, 1 ], [ 114, 122, -1 ], [ 115, 71, 1 ], [ 116, 116, 1 ], 
  [ 117, 117, 1 ], [ 118, 118, -1 ], [ 119, 113, 1 ], [ 120, 110, -1 ], 
  [ 121, 105, -1 ], [ 122, 114, -1 ], [ 123, 109, -1 ], [ 124, 100, 1 ], 
  [ 125, 125, 1 ], [ 126, 126, -1 ], [ 127, 127, 1 ], [ 128, 129, 1 ], 
  [ 128, 133, -1 ], [ 129, 129, 1 ], [ 130, 129, 2 ], [ 130, 132, -1 ], 
  [ 131, 129, 2 ], [ 131, 130, -1 ], [ 131, 131, 1 ], [ 131, 132, -1 ], 
  [ 132, 129, 2 ], [ 132, 130, -1 ], [ 133, 128, -1 ], [ 133, 129, 1 ] ] ],

[ [ 0, 1, 1, 0, 0, 0, 1 ],
[ 133, [ 1, 1, -1 ], [ 2, 23, 1 ], [ 3, 24, 1 ], [ 4, 69, -1 ], 
  [ 5, 68, 1 ], [ 6, 67, -1 ], [ 7, 25, 1 ], [ 8, 27, -1 ], [ 9, 16, -1 ], 
  [ 10, 17, -1 ], [ 11, 75, -1 ], [ 12, 74, -1 ], [ 13, 19, -1 ], 
  [ 14, 21, 1 ], [ 15, 40, 1 ], [ 16, 9, -1 ], [ 17, 10, -1 ], 
  [ 18, 81, 1 ], [ 19, 13, -1 ], [ 20, 43, -1 ], [ 21, 14, 1 ], 
  [ 22, 35, 1 ], [ 23, 2, 1 ], [ 24, 3, 1 ], [ 25, 7, 1 ], [ 26, 38, -1 ], 
  [ 27, 8, -1 ], [ 28, 28, -1 ], [ 29, 29, -1 ], [ 30, 30, 1 ], 
  [ 31, 31, 1 ], [ 32, 32, 1 ], [ 33, 33, 1 ], [ 34, 34, -1 ], 
  [ 35, 22, 1 ], [ 36, 49, 1 ], [ 37, 50, 1 ], [ 38, 26, -1 ], 
  [ 39, 52, -1 ], [ 40, 15, 1 ], [ 41, 45, -1 ], [ 42, 46, -1 ], 
  [ 43, 20, -1 ], [ 44, 48, 1 ], [ 45, 41, -1 ], [ 46, 42, -1 ], 
  [ 47, 59, -1 ], [ 48, 44, 1 ], [ 49, 36, 1 ], [ 50, 37, 1 ], 
  [ 51, 57, -1 ], [ 52, 39, -1 ], [ 53, 53, -1 ], [ 54, 54, 1 ], 
  [ 55, 55, 1 ], [ 56, 61, 1 ], [ 57, 51, -1 ], [ 58, 60, -1 ], 
  [ 59, 47, -1 ], [ 60, 58, -1 ], [ 61, 56, 1 ], [ 62, 62, -1 ], 
  [ 63, 63, 1 ], [ 64, 64, -1 ], [ 65, 86, 1 ], [ 66, 87, 1 ], 
  [ 67, 6, -1 ], [ 68, 5, 1 ], [ 69, 4, -1 ], [ 70, 88, 1 ], [ 71, 90, -1 ], 
  [ 72, 79, -1 ], [ 73, 80, -1 ], [ 74, 12, -1 ], [ 75, 11, -1 ], 
  [ 76, 82, -1 ], [ 77, 84, 1 ], [ 78, 103, 1 ], [ 79, 72, -1 ], 
  [ 80, 73, -1 ], [ 81, 18, 1 ], [ 82, 76, -1 ], [ 83, 106, -1 ], 
  [ 84, 77, 1 ], [ 85, 98, 1 ], [ 86, 65, 1 ], [ 87, 66, 1 ], [ 88, 70, 1 ], 
  [ 89, 101, -1 ], [ 90, 71, -1 ], [ 91, 91, -1 ], [ 92, 92, -1 ], 
  [ 93, 93, 1 ], [ 94, 94, 1 ], [ 95, 95, 1 ], [ 96, 96, 1 ], 
  [ 97, 97, -1 ], [ 98, 85, 1 ], [ 99, 112, 1 ], [ 100, 113, 1 ], 
  [ 101, 89, -1 ], [ 102, 115, -1 ], [ 103, 78, 1 ], [ 104, 108, -1 ], 
  [ 105, 109, -1 ], [ 106, 83, -1 ], [ 107, 111, 1 ], [ 108, 104, -1 ], 
  [ 109, 105, -1 ], [ 110, 122, -1 ], [ 111, 107, 1 ], [ 112, 99, 1 ], 
  [ 113, 100, 1 ], [ 114, 120, -1 ], [ 115, 102, -1 ], [ 116, 116, -1 ], 
  [ 117, 117, 1 ], [ 118, 118, 1 ], [ 119, 124, 1 ], [ 120, 114, -1 ], 
  [ 121, 123, -1 ], [ 122, 110, -1 ], [ 123, 121, -1 ], [ 124, 119, 1 ], 
  [ 125, 125, -1 ], [ 126, 126, 1 ], [ 127, 127, 1 ], [ 128, 128, 1 ], 
  [ 129, 129, 1 ], [ 130, 128, 1 ], [ 130, 129, 1 ], [ 130, 132, -1 ], 
  [ 130, 133, 1 ], [ 131, 128, 1 ], [ 131, 129, 1 ], [ 131, 131, -1 ], 
  [ 131, 133, 1 ], [ 132, 128, 1 ], [ 132, 129, 1 ], [ 132, 130, -1 ], 
  [ 132, 133, 1 ], [ 133, 133, 1 ] ] ],

[ [ 0, 0, 0, 2, 0, 0, 2 ],
 [ 133, [ 1, 1, -1 ], [ 2, 2, -1 ], [ 3, 3, 1 ], [ 4, 4, -1 ], [ 5, 5, 1 ], 
  [ 6, 6, -1 ], [ 7, 7, -1 ], [ 8, 8, -1 ], [ 9, 9, 1 ], [ 10, 10, -1 ], 
  [ 11, 11, -1 ], [ 12, 12, -1 ], [ 13, 13, 1 ], [ 14, 14, 1 ], 
  [ 15, 15, 1 ], [ 16, 16, 1 ], [ 17, 17, -1 ], [ 18, 18, 1 ], 
  [ 19, 19, 1 ], [ 20, 20, -1 ], [ 21, 21, 1 ], [ 22, 22, 1 ], 
  [ 23, 23, -1 ], [ 24, 24, 1 ], [ 25, 25, -1 ], [ 26, 26, -1 ], 
  [ 27, 27, -1 ], [ 28, 28, -1 ], [ 29, 29, -1 ], [ 30, 30, 1 ], 
  [ 31, 31, -1 ], [ 32, 32, 1 ], [ 33, 33, 1 ], [ 34, 34, 1 ], 
  [ 35, 35, 1 ], [ 36, 36, 1 ], [ 37, 37, 1 ], [ 38, 38, -1 ], 
  [ 39, 39, -1 ], [ 40, 40, 1 ], [ 41, 41, -1 ], [ 42, 42, -1 ], 
  [ 43, 43, -1 ], [ 44, 44, 1 ], [ 45, 45, -1 ], [ 46, 46, -1 ], 
  [ 47, 47, 1 ], [ 48, 48, 1 ], [ 49, 49, 1 ], [ 50, 50, 1 ], [ 51, 51, 1 ], 
  [ 52, 52, -1 ], [ 53, 53, -1 ], [ 54, 54, -1 ], [ 55, 55, -1 ], 
  [ 56, 56, 1 ], [ 57, 57, 1 ], [ 58, 58, -1 ], [ 59, 59, 1 ], 
  [ 60, 60, -1 ], [ 61, 61, 1 ], [ 62, 62, 1 ], [ 63, 63, -1 ], 
  [ 64, 64, -1 ], [ 65, 65, -1 ], [ 66, 66, 1 ], [ 67, 67, -1 ], 
  [ 68, 68, 1 ], [ 69, 69, -1 ], [ 70, 70, -1 ], [ 71, 71, -1 ], 
  [ 72, 72, 1 ], [ 73, 73, -1 ], [ 74, 74, -1 ], [ 75, 75, -1 ], 
  [ 76, 76, 1 ], [ 77, 77, 1 ], [ 78, 78, 1 ], [ 79, 79, 1 ], 
  [ 80, 80, -1 ], [ 81, 81, 1 ], [ 82, 82, 1 ], [ 83, 83, -1 ], 
  [ 84, 84, 1 ], [ 85, 85, 1 ], [ 86, 86, -1 ], [ 87, 87, 1 ], 
  [ 88, 88, -1 ], [ 89, 89, -1 ], [ 90, 90, -1 ], [ 91, 91, -1 ], 
  [ 92, 92, -1 ], [ 93, 93, 1 ], [ 94, 94, -1 ], [ 95, 95, 1 ], 
  [ 96, 96, 1 ], [ 97, 97, 1 ], [ 98, 98, 1 ], [ 99, 99, 1 ], 
  [ 100, 100, 1 ], [ 101, 101, -1 ], [ 102, 102, -1 ], [ 103, 103, 1 ], 
  [ 104, 104, -1 ], [ 105, 105, -1 ], [ 106, 106, -1 ], [ 107, 107, 1 ], 
  [ 108, 108, -1 ], [ 109, 109, -1 ], [ 110, 110, 1 ], [ 111, 111, 1 ], 
  [ 112, 112, 1 ], [ 113, 113, 1 ], [ 114, 114, 1 ], [ 115, 115, -1 ], 
  [ 116, 116, -1 ], [ 117, 117, -1 ], [ 118, 118, -1 ], [ 119, 119, 1 ], 
  [ 120, 120, 1 ], [ 121, 121, -1 ], [ 122, 122, 1 ], [ 123, 123, -1 ], 
  [ 124, 124, 1 ], [ 125, 125, 1 ], [ 126, 126, -1 ], [ 127, 127, 1 ], 
  [ 128, 128, 1 ], [ 129, 129, 1 ], [ 130, 130, 1 ], [ 131, 131, 1 ], 
  [ 132, 132, 1 ], [ 133, 133, 1 ] ],
  [ 133, [ 1, 1, 1/4 ], [ 1, 64, -9/4 ], [ 1, 66, -1/4 ], [ 1, 71, -3/4 ], 
  [ 1, 127, -3/2 ], [ 1, 129, 3/4 ], [ 2, 2, 1/4 ], [ 2, 65, -9/4 ], 
  [ 2, 128, 3/2 ], [ 2, 130, -3/4 ], [ 3, 1, 1/4 ], [ 3, 3, -1/2 ], 
  [ 3, 8, -1/2 ], [ 3, 64, 1/4 ], [ 3, 66, 1/4 ], [ 3, 71, 1/4 ], 
  [ 3, 129, -3/4 ], [ 3, 130, 1/2 ], [ 4, 4, -1/2 ], [ 4, 9, 1/2 ], 
  [ 5, 5, -1/2 ], [ 5, 6, -1/2 ], [ 5, 12, -1/2 ], [ 5, 68, 1 ], 
  [ 5, 69, -1/2 ], [ 5, 75, 1 ], [ 5, 130, 1 ], [ 5, 131, -3/2 ], 
  [ 5, 133, 1/2 ], [ 6, 6, 1/4 ], [ 6, 68, 1/2 ], [ 6, 69, -9/4 ], 
  [ 6, 75, 3/2 ], [ 6, 131, -3/4 ], [ 6, 132, 3/2 ], [ 6, 133, -3/4 ], 
  [ 7, 7, -1/2 ], [ 7, 13, 1/2 ], [ 8, 1, 1/4 ], [ 8, 3, 3/2 ], 
  [ 8, 8, -1/2 ], [ 8, 64, -3/4 ], [ 8, 66, -1/4 ], [ 8, 71, -1/4 ], 
  [ 8, 127, -1 ], [ 8, 129, 5/4 ], [ 8, 130, -1/2 ], [ 9, 4, -3/2 ], 
  [ 9, 9, -1/2 ], [ 10, 4, 1/4 ], [ 10, 9, -1/4 ], [ 10, 10, 1/4 ], 
  [ 10, 14, 1/4 ], [ 10, 15, -1/4 ], [ 10, 20, -1/4 ], [ 11, 4, -1/2 ], 
  [ 11, 9, 1/2 ], [ 11, 11, 1/4 ], [ 11, 16, -1/4 ], [ 11, 18, 1/4 ], 
  [ 11, 23, -1/4 ], [ 12, 5, 3/2 ], [ 12, 6, -1/2 ], [ 12, 12, -1/2 ], 
  [ 12, 68, -1 ], [ 12, 69, 3/2 ], [ 12, 75, -1 ], [ 12, 130, -1 ], 
  [ 12, 131, 5/2 ], [ 12, 132, -2 ], [ 12, 133, 1/2 ], [ 13, 7, -3/2 ], 
  [ 13, 13, -1/2 ], [ 14, 4, -1/4 ], [ 14, 9, 1/4 ], [ 14, 10, -3/4 ], 
  [ 14, 14, 1/4 ], [ 14, 15, 3/4 ], [ 14, 20, -1/4 ], [ 15, 4, 3/4 ], 
  [ 15, 9, 1/4 ], [ 15, 10, 3/4 ], [ 15, 14, 3/4 ], [ 15, 15, 1/4 ], 
  [ 15, 20, 1/4 ], [ 16, 4, -3/2 ], [ 16, 9, -1/2 ], [ 16, 11, 3/4 ], 
  [ 16, 16, 1/4 ], [ 16, 18, 3/4 ], [ 16, 23, 1/4 ], [ 17, 4, 1/4 ], 
  [ 17, 9, -1/4 ], [ 17, 10, 1/4 ], [ 17, 11, -1/8 ], [ 17, 14, 1/4 ], 
  [ 17, 15, -1/4 ], [ 17, 16, 1/8 ], [ 17, 17, -1/8 ], [ 17, 18, -1/8 ], 
  [ 17, 20, -1/4 ], [ 17, 21, -1/8 ], [ 17, 22, 1/8 ], [ 17, 23, 1/8 ], 
  [ 17, 24, -1/8 ], [ 17, 26, 1/8 ], [ 17, 27, -1/8 ], [ 17, 29, 1/8 ], 
  [ 17, 33, 1/8 ], [ 18, 4, 1/2 ], [ 18, 9, -1/2 ], [ 18, 11, -3/4 ], 
  [ 18, 16, 3/4 ], [ 18, 18, 1/4 ], [ 18, 23, -1/4 ], [ 19, 7, 1 ], 
  [ 19, 13, 1 ], [ 19, 19, 1 ], [ 20, 4, -3/4 ], [ 20, 9, -1/4 ], 
  [ 20, 10, -9/4 ], [ 20, 14, 3/4 ], [ 20, 15, -3/4 ], [ 20, 20, 1/4 ], 
  [ 21, 4, -1/4 ], [ 21, 9, 1/4 ], [ 21, 10, -3/4 ], [ 21, 11, 1/8 ], 
  [ 21, 14, 1/4 ], [ 21, 15, 3/4 ], [ 21, 16, -1/8 ], [ 21, 17, 3/8 ], 
  [ 21, 18, 1/8 ], [ 21, 20, -1/4 ], [ 21, 21, -1/8 ], [ 21, 22, -3/8 ], 
  [ 21, 23, -1/8 ], [ 21, 24, 3/8 ], [ 21, 26, 1/8 ], [ 21, 27, -1/8 ], 
  [ 21, 29, -3/8 ], [ 21, 33, 1/8 ], [ 22, 4, 3/4 ], [ 22, 9, 1/4 ], 
  [ 22, 10, 3/4 ], [ 22, 11, -3/8 ], [ 22, 14, 3/4 ], [ 22, 15, 1/4 ], 
  [ 22, 16, -1/8 ], [ 22, 17, -3/8 ], [ 22, 18, -3/8 ], [ 22, 20, 1/4 ], 
  [ 22, 21, -3/8 ], [ 22, 22, -1/8 ], [ 22, 23, -1/8 ], [ 22, 24, -3/8 ], 
  [ 22, 26, -1/8 ], [ 22, 27, -3/8 ], [ 22, 29, -1/8 ], [ 22, 33, -1/8 ], 
  [ 23, 4, 3/2 ], [ 23, 9, 1/2 ], [ 23, 11, -9/4 ], [ 23, 16, -3/4 ], 
  [ 23, 18, 3/4 ], [ 23, 23, 1/4 ], [ 24, 4, -1/4 ], [ 24, 9, 1/4 ], 
  [ 24, 10, -1/4 ], [ 24, 11, 3/8 ], [ 24, 14, -1/4 ], [ 24, 15, 1/4 ], 
  [ 24, 16, -3/8 ], [ 24, 17, 3/8 ], [ 24, 18, -1/8 ], [ 24, 20, 1/4 ], 
  [ 24, 21, 3/8 ], [ 24, 22, -3/8 ], [ 24, 23, 1/8 ], [ 24, 24, -1/8 ], 
  [ 24, 26, -3/8 ], [ 24, 27, -1/8 ], [ 24, 29, 1/8 ], [ 24, 33, 1/8 ], 
  [ 25, 25, -1/2 ], [ 25, 30, 1/2 ], [ 26, 4, -3/4 ], [ 26, 9, -1/4 ], 
  [ 26, 10, -9/4 ], [ 26, 11, 3/8 ], [ 26, 14, 3/4 ], [ 26, 15, -3/4 ], 
  [ 26, 16, 1/8 ], [ 26, 17, 9/8 ], [ 26, 18, 3/8 ], [ 26, 20, 1/4 ], 
  [ 26, 21, -3/8 ], [ 26, 22, 3/8 ], [ 26, 23, 1/8 ], [ 26, 24, 9/8 ], 
  [ 26, 26, -1/8 ], [ 26, 27, -3/8 ], [ 26, 29, 3/8 ], [ 26, 33, -1/8 ], 
  [ 27, 4, 1/4 ], [ 27, 9, -1/4 ], [ 27, 10, 3/4 ], [ 27, 11, -3/8 ], 
  [ 27, 14, -1/4 ], [ 27, 15, -3/4 ], [ 27, 16, 3/8 ], [ 27, 17, -9/8 ], 
  [ 27, 18, 1/8 ], [ 27, 20, 1/4 ], [ 27, 21, 3/8 ], [ 27, 22, 9/8 ], 
  [ 27, 23, -1/8 ], [ 27, 24, 3/8 ], [ 27, 26, -3/8 ], [ 27, 27, -1/8 ], 
  [ 27, 29, -3/8 ], [ 27, 33, 1/8 ], [ 28, 28, 1/4 ], [ 28, 32, 1/4 ], 
  [ 28, 35, 1/4 ], [ 28, 38, 1/4 ], [ 29, 4, -3/4 ], [ 29, 9, -1/4 ], 
  [ 29, 10, -3/4 ], [ 29, 11, 9/8 ], [ 29, 14, -3/4 ], [ 29, 15, -1/4 ], 
  [ 29, 16, 3/8 ], [ 29, 17, 9/8 ], [ 29, 18, -3/8 ], [ 29, 20, -1/4 ], 
  [ 29, 21, 9/8 ], [ 29, 22, 3/8 ], [ 29, 23, -1/8 ], [ 29, 24, -3/8 ], 
  [ 29, 26, 3/8 ], [ 29, 27, -3/8 ], [ 29, 29, -1/8 ], [ 29, 33, -1/8 ], 
  [ 30, 25, -3/2 ], [ 30, 30, -1/2 ], [ 31, 25, 1/4 ], [ 31, 30, -1/4 ], 
  [ 31, 31, 1/4 ], [ 31, 34, 1/4 ], [ 31, 36, -1/4 ], [ 31, 39, -1/4 ], 
  [ 32, 28, -3/4 ], [ 32, 32, 1/4 ], [ 32, 35, -3/4 ], [ 32, 38, 1/4 ], 
  [ 33, 4, 3/4 ], [ 33, 9, 1/4 ], [ 33, 10, 9/4 ], [ 33, 11, -9/8 ], 
  [ 33, 14, -3/4 ], [ 33, 15, 3/4 ], [ 33, 16, -3/8 ], [ 33, 17, -27/8 ], 
  [ 33, 18, 3/8 ], [ 33, 20, -1/4 ], [ 33, 21, 9/8 ], [ 33, 22, -9/8 ], 
  [ 33, 23, 1/8 ], [ 33, 24, 9/8 ], [ 33, 26, 3/8 ], [ 33, 27, -3/8 ], 
  [ 33, 29, 3/8 ], [ 33, 33, -1/8 ], [ 34, 25, -1/4 ], [ 34, 30, 1/4 ], 
  [ 34, 31, -3/4 ], [ 34, 34, 1/4 ], [ 34, 36, 3/4 ], [ 34, 39, -1/4 ], 
  [ 35, 28, -3/4 ], [ 35, 32, -3/4 ], [ 35, 35, 1/4 ], [ 35, 38, 1/4 ], 
  [ 36, 25, 3/4 ], [ 36, 30, 1/4 ], [ 36, 31, 3/4 ], [ 36, 34, 3/4 ], 
  [ 36, 36, 1/4 ], [ 36, 39, 1/4 ], [ 37, 28, 1/4 ], [ 37, 32, -1/4 ], 
  [ 37, 35, 1/4 ], [ 37, 37, -1/2 ], [ 37, 38, -1/4 ], [ 37, 42, -1/2 ], 
  [ 38, 28, 9/4 ], [ 38, 32, -3/4 ], [ 38, 35, -3/4 ], [ 38, 38, 1/4 ], 
  [ 39, 25, -3/4 ], [ 39, 30, -1/4 ], [ 39, 31, -9/4 ], [ 39, 34, 3/4 ], 
  [ 39, 36, -3/4 ], [ 39, 39, 1/4 ], [ 40, 28, -1/2 ], [ 40, 32, -1/2 ], 
  [ 40, 35, 1/2 ], [ 40, 38, 1/2 ], [ 40, 40, -1/2 ], [ 40, 43, -1/2 ], 
  [ 41, 41, -1/2 ], [ 41, 44, -1/2 ], [ 42, 28, -3/4 ], [ 42, 32, 3/4 ], 
  [ 42, 35, 1/4 ], [ 42, 37, 3/2 ], [ 42, 38, -1/4 ], [ 42, 42, -1/2 ], 
  [ 43, 28, 3/2 ], [ 43, 32, -1/2 ], [ 43, 35, -3/2 ], [ 43, 38, 1/2 ], 
  [ 43, 40, 3/2 ], [ 43, 43, -1/2 ], [ 44, 41, 3/2 ], [ 44, 44, -1/2 ], 
  [ 45, 41, -1/2 ], [ 45, 44, -1/2 ], [ 45, 45, 1/4 ], [ 45, 48, 1/4 ], 
  [ 45, 49, 1/4 ], [ 45, 52, 1/4 ], [ 46, 28, -1/2 ], [ 46, 32, 1/2 ], 
  [ 46, 35, 1/2 ], [ 46, 37, 1 ], [ 46, 38, -1/2 ], [ 46, 40, -1/2 ], 
  [ 46, 42, -1 ], [ 46, 43, 1/2 ], [ 46, 46, 1 ], [ 47, 41, -1/2 ], 
  [ 47, 44, 1/2 ], [ 47, 47, 1 ], [ 48, 41, 3/2 ], [ 48, 44, -1/2 ], 
  [ 48, 45, -3/4 ], [ 48, 48, 1/4 ], [ 48, 49, -3/4 ], [ 48, 52, 1/4 ], 
  [ 49, 41, 1/2 ], [ 49, 44, 1/2 ], [ 49, 45, -3/4 ], [ 49, 48, -3/4 ], 
  [ 49, 49, 1/4 ], [ 49, 52, 1/4 ], [ 50, 50, -1/2 ], [ 50, 53, -1/2 ], 
  [ 51, 41, -1/2 ], [ 51, 44, 1/2 ], [ 51, 45, 1/4 ], [ 51, 47, 1 ], 
  [ 51, 48, -1/4 ], [ 51, 49, 1/4 ], [ 51, 51, -1/2 ], [ 51, 52, -1/4 ], 
  [ 51, 55, -1/2 ], [ 52, 41, -3/2 ], [ 52, 44, 1/2 ], [ 52, 45, 9/4 ], 
  [ 52, 48, -3/4 ], [ 52, 49, -3/4 ], [ 52, 52, 1/4 ], [ 53, 50, 3/2 ], 
  [ 53, 53, -1/2 ], [ 54, 54, 1/4 ], [ 54, 56, 1/4 ], [ 54, 57, 1/4 ], 
  [ 54, 58, 1/4 ], [ 55, 41, 1/2 ], [ 55, 44, -1/2 ], [ 55, 45, -3/4 ], 
  [ 55, 47, -1 ], [ 55, 48, 3/4 ], [ 55, 49, 1/4 ], [ 55, 51, 3/2 ], 
  [ 55, 52, -1/4 ], [ 55, 55, -1/2 ], [ 56, 54, -3/4 ], [ 56, 56, 1/4 ], 
  [ 56, 57, -3/4 ], [ 56, 58, 1/4 ], [ 57, 54, -3/4 ], [ 57, 56, -3/4 ], 
  [ 57, 57, 1/4 ], [ 57, 58, 1/4 ], [ 58, 54, 9/4 ], [ 58, 56, -3/4 ], 
  [ 58, 57, -3/4 ], [ 58, 58, 1/4 ], [ 59, 54, -1/2 ], [ 59, 56, -1/2 ], 
  [ 59, 57, 1/2 ], [ 59, 58, 1/2 ], [ 59, 59, -1/2 ], [ 59, 60, -1/2 ], 
  [ 60, 54, 3/2 ], [ 60, 56, -1/2 ], [ 60, 57, -3/2 ], [ 60, 58, 1/2 ], 
  [ 60, 59, 3/2 ], [ 60, 60, -1/2 ], [ 61, 61, 1 ], [ 62, 61, 1/2 ], 
  [ 62, 62, -1/2 ], [ 62, 63, 1/2 ], [ 63, 61, 1/2 ], [ 63, 62, -3/2 ], 
  [ 63, 63, -1/2 ], [ 64, 1, -1/4 ], [ 64, 64, 1/4 ], [ 64, 66, -1/4 ], 
  [ 64, 71, 1/4 ], [ 64, 127, -1/2 ], [ 64, 129, 1/4 ], [ 65, 2, -1/4 ], 
  [ 65, 65, 1/4 ], [ 65, 128, 1/2 ], [ 65, 130, -1/4 ], [ 66, 66, -1/2 ], 
  [ 66, 71, -3/2 ], [ 67, 67, -1/2 ], [ 67, 72, 3/2 ], [ 67, 73, 1/4 ], 
  [ 67, 74, -1/2 ], [ 67, 77, 1/4 ], [ 67, 78, -3/4 ], [ 67, 79, 3/2 ], 
  [ 67, 80, 1/4 ], [ 67, 81, -1/2 ], [ 67, 83, -3/4 ], [ 67, 84, 1/4 ], 
  [ 67, 85, -3/4 ], [ 67, 86, 3/2 ], [ 67, 87, 1/4 ], [ 67, 89, -3/4 ], 
  [ 67, 90, 1/4 ], [ 67, 92, -3/4 ], [ 67, 96, -3/4 ], [ 68, 68, -1/2 ], 
  [ 68, 75, -3/2 ], [ 69, 6, -1/4 ], [ 69, 68, 1/2 ], [ 69, 69, 1/4 ], 
  [ 69, 75, -1/2 ], [ 69, 131, -1/4 ], [ 69, 132, 1/2 ], [ 69, 133, -1/4 ], 
  [ 70, 70, -1/2 ], [ 70, 76, 3/2 ], [ 70, 82, -1 ], [ 71, 66, 1/2 ], 
  [ 71, 71, -1/2 ], [ 72, 67, -1/2 ], [ 72, 72, -1/2 ], [ 72, 73, 1/4 ], 
  [ 72, 74, -1/2 ], [ 72, 77, 1/4 ], [ 72, 78, 1/4 ], [ 72, 79, -1/2 ], 
  [ 72, 80, 1/4 ], [ 72, 81, -1/2 ], [ 72, 83, 1/4 ], [ 72, 84, 1/4 ], 
  [ 72, 85, 1/4 ], [ 72, 86, -1/2 ], [ 72, 87, 1/4 ], [ 72, 89, 1/4 ], 
  [ 72, 90, 1/4 ], [ 72, 92, 1/4 ], [ 72, 96, 1/4 ], [ 73, 73, 1/4 ], 
  [ 73, 77, 3/4 ], [ 73, 78, -3/4 ], [ 73, 80, 1/4 ], [ 73, 83, -9/4 ], 
  [ 73, 84, 3/4 ], [ 73, 85, -3/4 ], [ 73, 87, 1/4 ], [ 73, 89, -9/4 ], 
  [ 73, 90, 3/4 ], [ 73, 92, -3/4 ], [ 73, 96, -9/4 ], [ 74, 74, 1/4 ], 
  [ 74, 79, -3/4 ], [ 74, 80, -1/8 ], [ 74, 81, 3/4 ], [ 74, 84, -1/8 ], 
  [ 74, 85, 3/8 ], [ 74, 86, -9/4 ], [ 74, 87, -3/8 ], [ 74, 89, 3/8 ], 
  [ 74, 90, -3/8 ], [ 74, 92, 9/8 ], [ 74, 96, 9/8 ], [ 75, 68, 1/2 ], 
  [ 75, 75, -1/2 ], [ 76, 70, -1/2 ], [ 76, 76, -1/2 ], [ 76, 82, 1 ], 
  [ 77, 73, -1/4 ], [ 77, 77, 1/4 ], [ 77, 78, 3/4 ], [ 77, 80, -1/4 ], 
  [ 77, 83, -3/4 ], [ 77, 84, 1/4 ], [ 77, 85, 3/4 ], [ 77, 87, -1/4 ], 
  [ 77, 89, -3/4 ], [ 77, 90, 1/4 ], [ 77, 92, 3/4 ], [ 77, 96, -3/4 ], 
  [ 78, 73, 1/4 ], [ 78, 77, 3/4 ], [ 78, 78, 1/4 ], [ 78, 80, 1/4 ], 
  [ 78, 83, 3/4 ], [ 78, 84, 3/4 ], [ 78, 85, 1/4 ], [ 78, 87, 1/4 ], 
  [ 78, 89, 3/4 ], [ 78, 90, 3/4 ], [ 78, 92, 1/4 ], [ 78, 96, 3/4 ], 
  [ 79, 74, 1/4 ], [ 79, 79, 1/4 ], [ 79, 80, -1/8 ], [ 79, 81, 3/4 ], 
  [ 79, 84, -1/8 ], [ 79, 85, -1/8 ], [ 79, 86, 3/4 ], [ 79, 87, -3/8 ], 
  [ 79, 89, -1/8 ], [ 79, 90, -3/8 ], [ 79, 92, -3/8 ], [ 79, 96, -3/8 ], 
  [ 80, 80, -1/8 ], [ 80, 84, -3/8 ], [ 80, 85, 3/8 ], [ 80, 87, -3/8 ], 
  [ 80, 89, 9/8 ], [ 80, 90, -9/8 ], [ 80, 92, 9/8 ], [ 80, 96, 27/8 ], 
  [ 81, 74, -1/4 ], [ 81, 79, 3/4 ], [ 81, 80, 1/8 ], [ 81, 81, 1/4 ], 
  [ 81, 84, 1/8 ], [ 81, 85, -3/8 ], [ 81, 86, -3/4 ], [ 81, 87, -1/8 ], 
  [ 81, 89, -3/8 ], [ 81, 90, -1/8 ], [ 81, 92, 3/8 ], [ 81, 96, 3/8 ], 
  [ 82, 82, 1 ], [ 83, 73, -1/4 ], [ 83, 77, 1/4 ], [ 83, 78, -1/4 ], 
  [ 83, 80, -1/4 ], [ 83, 83, 1/4 ], [ 83, 84, 1/4 ], [ 83, 85, -1/4 ], 
  [ 83, 87, -1/4 ], [ 83, 89, 1/4 ], [ 83, 90, 1/4 ], [ 83, 92, -1/4 ], 
  [ 83, 96, 1/4 ], [ 84, 80, 1/8 ], [ 84, 84, -1/8 ], [ 84, 85, -3/8 ], 
  [ 84, 87, 3/8 ], [ 84, 89, 3/8 ], [ 84, 90, -3/8 ], [ 84, 92, -9/8 ], 
  [ 84, 96, 9/8 ], [ 85, 80, -1/8 ], [ 85, 84, -3/8 ], [ 85, 85, -1/8 ], 
  [ 85, 87, -3/8 ], [ 85, 89, -3/8 ], [ 85, 90, -9/8 ], [ 85, 92, -3/8 ], 
  [ 85, 96, -9/8 ], [ 86, 74, -1/4 ], [ 86, 79, -1/4 ], [ 86, 80, 1/8 ], 
  [ 86, 81, 1/4 ], [ 86, 84, 1/8 ], [ 86, 85, 1/8 ], [ 86, 86, 1/4 ], 
  [ 86, 87, -1/8 ], [ 86, 89, 1/8 ], [ 86, 90, -1/8 ], [ 86, 92, -1/8 ], 
  [ 86, 96, -1/8 ], [ 87, 80, 1/8 ], [ 87, 84, 3/8 ], [ 87, 85, -3/8 ], 
  [ 87, 87, -1/8 ], [ 87, 89, -9/8 ], [ 87, 90, -3/8 ], [ 87, 92, 3/8 ], 
  [ 87, 96, 9/8 ], [ 88, 88, -1/2 ], [ 88, 93, 3/2 ], [ 88, 94, 1/4 ], 
  [ 88, 97, 1/4 ], [ 88, 99, -3/4 ], [ 88, 102, -3/4 ], [ 89, 80, 1/8 ], 
  [ 89, 84, -1/8 ], [ 89, 85, 1/8 ], [ 89, 87, 3/8 ], [ 89, 89, -1/8 ], 
  [ 89, 90, -3/8 ], [ 89, 92, 3/8 ], [ 89, 96, -3/8 ], [ 90, 80, -1/8 ], 
  [ 90, 84, 1/8 ], [ 90, 85, 3/8 ], [ 90, 87, 1/8 ], [ 90, 89, -3/8 ], 
  [ 90, 90, -1/8 ], [ 90, 92, -3/8 ], [ 90, 96, 3/8 ], [ 91, 91, 1/4 ], 
  [ 91, 95, 3/4 ], [ 91, 98, 3/4 ], [ 91, 100, -1/4 ], [ 91, 101, 9/4 ], 
  [ 91, 103, 1/2 ], [ 91, 105, -3/4 ], [ 91, 106, 3/2 ], [ 91, 109, -1/2 ], 
  [ 92, 80, 1/8 ], [ 92, 84, 3/8 ], [ 92, 85, 1/8 ], [ 92, 87, -1/8 ], 
  [ 92, 89, 3/8 ], [ 92, 90, -3/8 ], [ 92, 92, -1/8 ], [ 92, 96, -3/8 ], 
  [ 93, 88, -1/2 ], [ 93, 93, -1/2 ], [ 93, 94, 1/4 ], [ 93, 97, 1/4 ], 
  [ 93, 99, 1/4 ], [ 93, 102, 1/4 ], [ 94, 94, 1/4 ], [ 94, 97, 3/4 ], 
  [ 94, 99, -3/4 ], [ 94, 102, -9/4 ], [ 95, 91, -1/4 ], [ 95, 95, 1/4 ], 
  [ 95, 98, -3/4 ], [ 95, 100, -1/4 ], [ 95, 101, 3/4 ], [ 95, 103, -1/2 ], 
  [ 95, 105, -3/4 ], [ 95, 106, 1/2 ], [ 95, 109, -1/2 ], [ 96, 80, -1/8 ], 
  [ 96, 84, 1/8 ], [ 96, 85, -1/8 ], [ 96, 87, 1/8 ], [ 96, 89, 1/8 ], 
  [ 96, 90, -1/8 ], [ 96, 92, 1/8 ], [ 96, 96, -1/8 ], [ 97, 94, -1/4 ], 
  [ 97, 97, 1/4 ], [ 97, 99, 3/4 ], [ 97, 102, -3/4 ], [ 98, 91, -1/4 ], 
  [ 98, 95, -3/4 ], [ 98, 98, 1/4 ], [ 98, 100, 1/4 ], [ 98, 101, 3/4 ], 
  [ 98, 103, 1/2 ], [ 98, 105, -1/4 ], [ 98, 106, 3/2 ], [ 98, 109, -1/2 ], 
  [ 99, 94, 1/4 ], [ 99, 97, 3/4 ], [ 99, 99, 1/4 ], [ 99, 102, 3/4 ], 
  [ 100, 100, -1/2 ], [ 100, 105, -3/2 ], [ 100, 109, -1 ], [ 101, 91, 1/4 ],
  [ 101, 95, -1/4 ], [ 101, 98, -1/4 ], [ 101, 100, 1/4 ], [ 101, 101, 1/4 ],
  [ 101, 103, -1/2 ], [ 101, 105, -1/4 ], [ 101, 106, 1/2 ], 
  [ 101, 109, -1/2 ], [ 102, 94, -1/4 ], [ 102, 97, 1/4 ], [ 102, 99, -1/4 ],
  [ 102, 102, 1/4 ], [ 103, 103, -1/2 ], [ 103, 106, -3/2 ], 
  [ 103, 109, 1/2 ], [ 104, 104, -1/2 ], [ 104, 107, -3/2 ], 
  [ 104, 108, -1/2 ], [ 104, 110, 1/2 ], [ 104, 111, -3/2 ], 
  [ 104, 112, -1/2 ], [ 104, 114, 1/2 ], [ 104, 115, -3/2 ], 
  [ 104, 118, 1/2 ], [ 105, 100, 1/2 ], [ 105, 105, -1/2 ], [ 105, 109, -1 ],
  [ 106, 103, 1/2 ], [ 106, 106, -1/2 ], [ 106, 109, 1/2 ], 
  [ 107, 104, 1/2 ], [ 107, 107, -1/2 ], [ 107, 108, 1/2 ], 
  [ 107, 110, 1/2 ], [ 107, 111, -1/2 ], [ 107, 112, 1/2 ], 
  [ 107, 114, 1/2 ], [ 107, 115, -1/2 ], [ 107, 118, 1/2 ], 
  [ 108, 108, 1/4 ], [ 108, 111, 3/4 ], [ 108, 112, 3/4 ], 
  [ 108, 114, -1/4 ], [ 108, 115, 9/4 ], [ 108, 118, -3/4 ], [ 109, 109, 1 ],
  [ 110, 110, 1 ], [ 110, 114, 1 ], [ 110, 118, 1 ], [ 111, 108, -1/4 ], 
  [ 111, 111, 1/4 ], [ 111, 112, -3/4 ], [ 111, 114, -1/4 ], 
  [ 111, 115, 3/4 ], [ 111, 118, -3/4 ], [ 112, 108, -1/4 ], 
  [ 112, 111, -3/4 ], [ 112, 112, 1/4 ], [ 112, 114, 1/4 ], 
  [ 112, 115, 3/4 ], [ 112, 118, -1/4 ], [ 113, 113, -1/2 ], 
  [ 113, 116, -3/2 ], [ 114, 114, -1/2 ], [ 114, 118, -3/2 ], 
  [ 115, 108, 1/4 ], [ 115, 111, -1/4 ], [ 115, 112, -1/4 ], 
  [ 115, 114, 1/4 ], [ 115, 115, 1/4 ], [ 115, 118, -1/4 ], 
  [ 116, 113, 1/2 ], [ 116, 116, -1/2 ], [ 117, 117, 1/4 ], 
  [ 117, 119, 3/4 ], [ 117, 120, 3/4 ], [ 117, 121, 9/4 ], [ 117, 122, 1/2 ],
  [ 117, 123, 3/2 ], [ 118, 114, 1/2 ], [ 118, 118, -1/2 ], 
  [ 119, 117, -1/4 ], [ 119, 119, 1/4 ], [ 119, 120, -3/4 ], 
  [ 119, 121, 3/4 ], [ 119, 122, -1/2 ], [ 119, 123, 1/2 ], 
  [ 120, 117, -1/4 ], [ 120, 119, -3/4 ], [ 120, 120, 1/4 ], 
  [ 120, 121, 3/4 ], [ 120, 122, 1/2 ], [ 120, 123, 3/2 ], [ 121, 117, 1/4 ],
  [ 121, 119, -1/4 ], [ 121, 120, -1/4 ], [ 121, 121, 1/4 ], 
  [ 121, 122, -1/2 ], [ 121, 123, 1/2 ], [ 122, 122, -1/2 ], 
  [ 122, 123, -3/2 ], [ 123, 122, 1/2 ], [ 123, 123, -1/2 ], [ 124, 124, 1 ],
  [ 124, 125, 1/2 ], [ 124, 126, -1/2 ], [ 125, 125, -1/2 ], 
  [ 125, 126, 3/2 ], [ 126, 125, -1/2 ], [ 126, 126, -1/2 ], [ 127, 1, 1/4 ],
  [ 127, 64, 3/4 ], [ 127, 66, -1/4 ], [ 127, 71, 1/4 ], [ 127, 127, -1/2 ], 
  [ 127, 129, 3/4 ], [ 128, 2, -1/4 ], [ 128, 65, -3/4 ], [ 128, 128, -1/2 ],
  [ 128, 130, 3/4 ], [ 129, 66, -1/2 ], [ 129, 71, -1/2 ], [ 129, 129, 1 ], 
  [ 130, 130, 1 ], [ 131, 68, -1 ], [ 131, 75, -1 ], [ 131, 131, 1 ], 
  [ 132, 6, -1/4 ], [ 132, 68, -1/2 ], [ 132, 69, -3/4 ], [ 132, 75, 1/2 ], 
  [ 132, 131, 3/4 ], [ 132, 132, -1/2 ], [ 132, 133, 3/4 ], [ 133, 133, 1 ] ] ],

 [ [ 2, 0, 0, 2, 0, 2, 2 ],
   [ 133, [ 1, 1, -1 ], [ 2, 2, -1 ], [ 3, 3, -1 ], [ 4, 4, -1 ], [ 5, 5, -1 ], 
  [ 6, 6, -1 ], [ 7, 7, 1 ], [ 8, 8, 1 ], [ 9, 9, 1 ], [ 10, 10, 1 ], 
  [ 11, 11, 1 ], [ 12, 12, 1 ], [ 13, 13, -1 ], [ 14, 14, -1 ], 
  [ 15, 15, -1 ], [ 16, 16, -1 ], [ 17, 17, -1 ], [ 18, 18, -1 ], 
  [ 19, 19, 1 ], [ 20, 20, 1 ], [ 21, 21, 1 ], [ 22, 22, 1 ], [ 23, 23, 1 ], 
  [ 24, 24, 1 ], [ 25, 25, -1 ], [ 26, 26, -1 ], [ 27, 27, -1 ], 
  [ 28, 28, -1 ], [ 29, 29, -1 ], [ 30, 30, 1 ], [ 31, 31, 1 ], 
  [ 32, 32, 1 ], [ 33, 33, 1 ], [ 34, 34, -1 ], [ 35, 35, 1 ], 
  [ 36, 36, -1 ], [ 37, 37, -1 ], [ 38, 38, -1 ], [ 39, 39, 1 ], 
  [ 40, 40, -1 ], [ 41, 41, 1 ], [ 42, 42, 1 ], [ 43, 43, 1 ], 
  [ 44, 44, -1 ], [ 45, 45, -1 ], [ 46, 46, -1 ], [ 47, 47, 1 ], 
  [ 48, 48, 1 ], [ 49, 49, 1 ], [ 50, 50, 1 ], [ 51, 51, -1 ], 
  [ 52, 52, -1 ], [ 53, 53, -1 ], [ 54, 54, 1 ], [ 55, 55, 1 ], 
  [ 56, 56, -1 ], [ 57, 57, -1 ], [ 58, 58, 1 ], [ 59, 59, 1 ], 
  [ 60, 60, -1 ], [ 61, 61, 1 ], [ 62, 62, -1 ], [ 63, 63, 1 ], 
  [ 64, 64, -1 ], [ 65, 65, -1 ], [ 66, 66, -1 ], [ 67, 67, -1 ], 
  [ 68, 68, -1 ], [ 69, 69, -1 ], [ 70, 70, 1 ], [ 71, 71, 1 ], 
  [ 72, 72, 1 ], [ 73, 73, 1 ], [ 74, 74, 1 ], [ 75, 75, 1 ], [ 76, 76, -1 ], 
  [ 77, 77, -1 ], [ 78, 78, -1 ], [ 79, 79, -1 ], [ 80, 80, -1 ], 
  [ 81, 81, -1 ], [ 82, 82, 1 ], [ 83, 83, 1 ], [ 84, 84, 1 ], [ 85, 85, 1 ], 
  [ 86, 86, 1 ], [ 87, 87, 1 ], [ 88, 88, -1 ], [ 89, 89, -1 ], 
  [ 90, 90, -1 ], [ 91, 91, -1 ], [ 92, 92, -1 ], [ 93, 93, 1 ], 
  [ 94, 94, 1 ], [ 95, 95, 1 ], [ 96, 96, 1 ], [ 97, 97, -1 ], [ 98, 98, 1 ], 
  [ 99, 99, -1 ], [ 100, 100, -1 ], [ 101, 101, -1 ], [ 102, 102, 1 ], 
  [ 103, 103, -1 ], [ 104, 104, 1 ], [ 105, 105, 1 ], [ 106, 106, 1 ], 
  [ 107, 107, -1 ], [ 108, 108, -1 ], [ 109, 109, -1 ], [ 110, 110, 1 ], 
  [ 111, 111, 1 ], [ 112, 112, 1 ], [ 113, 113, 1 ], [ 114, 114, -1 ], 
  [ 115, 115, -1 ], [ 116, 116, -1 ], [ 117, 117, 1 ], [ 118, 118, 1 ], 
  [ 119, 119, -1 ], [ 120, 120, -1 ], [ 121, 121, 1 ], [ 122, 122, 1 ], 
  [ 123, 123, -1 ], [ 124, 124, 1 ], [ 125, 125, -1 ], [ 126, 126, 1 ], 
  [ 127, 127, 1 ], [ 128, 128, 1 ], [ 129, 129, 1 ], [ 130, 130, 1 ], 
  [ 131, 131, 1 ], [ 132, 132, 1 ], [ 133, 133, 1 ] ] ]
],

[ "E", 8,

[ [ 0, 0, 0, 0, 0, 0, 0, 2 ],
[ 248, [ 1, 1, 1 ], [ 2, 125, -1 ], [ 3, 3, 1 ], [ 4, 18, 1 ], [ 5, 122, -1 ], [ 6, 183, 1 ], 
  [ 7, 127, 1 ], [ 8, 119, 1 ], [ 9, 9, 1 ], [ 10, 10, 1 ], [ 11, 25, 1 ], [ 12, 12, 1 ], [ 13, 189, -1 ], 
  [ 14, 190, -1 ], [ 15, 118, 1 ], [ 16, 30, 1 ], [ 17, 17, 1 ], [ 18, 4, 1 ], [ 19, 19, 1 ], 
  [ 20, 171, -1 ], [ 21, 195, 1 ], [ 22, 100, 1 ], [ 23, 23, 1 ], [ 24, 24, 1 ], [ 25, 11, 1 ], 
  [ 26, 177, 1 ], [ 27, 165, 1 ], [ 28, 178, 1 ], [ 29, 96, 1 ], [ 30, 16, 1 ], [ 31, 160, -1 ], 
  [ 32, 32, 1 ], [ 33, 172, -1 ], [ 34, 184, -1 ], [ 35, 173, -1 ], [ 36, 106, -1 ], [ 37, 37, 1 ], 
  [ 38, 168, 1 ], [ 39, 169, 1 ], [ 40, 151, -1 ], [ 41, 179, 1 ], [ 42, 103, -1 ], [ 43, 108, 1 ], 
  [ 44, 44, 1 ], [ 45, 147, 1 ], [ 46, 175, -1 ], [ 47, 109, -1 ], [ 48, 158, 1 ], [ 49, 159, 1 ], 
  [ 50, 105, 1 ], [ 51, 140, -1 ], [ 52, 153, -1 ], [ 53, 155, -1 ], [ 54, 107, -1 ], [ 55, 166, -1 ], 
  [ 56, 112, -1 ], [ 57, 146, 1 ], [ 58, 148, 1 ], [ 59, 161, 1 ], [ 60, 113, 1 ], [ 61, 217, 1 ], 
  [ 62, 110, -1 ], [ 63, 126, 1 ], [ 64, 154, -1 ], [ 65, 115, -1 ], [ 66, 213, 1 ], [ 67, 111, 1 ], 
  [ 68, 74, -1 ], [ 69, 133, -1 ], [ 70, 134, -1 ], [ 71, 209, 1 ], [ 72, 114, -1 ], [ 73, 79, -1 ], 
  [ 74, 68, -1 ], [ 75, 141, 1 ], [ 76, 196, 1 ], [ 77, 117, 1 ], [ 78, 84, -1 ], [ 79, 73, -1 ], 
  [ 80, 202, 1 ], [ 81, 116, 1 ], [ 82, 200, 1 ], [ 83, 95, 1 ], [ 84, 78, -1 ], [ 85, 205, 1 ], 
  [ 86, 91, -1 ], [ 87, 92, -1 ], [ 88, 90, 1 ], [ 89, 191, 1 ], [ 90, 88, 1 ], [ 91, 86, -1 ], 
  [ 92, 87, -1 ], [ 93, 186, 1 ], [ 94, 99, -1 ], [ 95, 83, 1 ], [ 96, 29, 1 ], [ 97, 181, 1 ], 
  [ 98, 102, -1 ], [ 99, 94, -1 ], [ 100, 22, 1 ], [ 101, 104, -1 ], [ 102, 98, -1 ], [ 103, 42, -1 ], 
  [ 104, 101, -1 ], [ 105, 50, 1 ], [ 106, 36, -1 ], [ 107, 54, -1 ], [ 108, 43, 1 ], [ 109, 47, -1 ], 
  [ 110, 62, -1 ], [ 111, 67, 1 ], [ 112, 56, -1 ], [ 113, 60, 1 ], [ 114, 72, -1 ], [ 115, 65, -1 ], 
  [ 116, 81, 1 ], [ 117, 77, 1 ], [ 118, 15, 1 ], [ 119, 8, 1 ], [ 120, 120, -1 ], [ 121, 121, 1 ], 
  [ 122, 5, -1 ], [ 123, 123, 1 ], [ 124, 138, 1 ], [ 125, 2, -1 ], [ 126, 63, 1 ], [ 127, 7, 1 ], 
  [ 128, 239, 1 ], [ 129, 129, 1 ], [ 130, 130, 1 ], [ 131, 145, 1 ], [ 132, 132, 1 ], [ 133, 69, -1 ], 
  [ 134, 70, -1 ], [ 135, 238, 1 ], [ 136, 150, 1 ], [ 137, 137, 1 ], [ 138, 124, 1 ], [ 139, 139, 1 ], 
  [ 140, 51, -1 ], [ 141, 75, 1 ], [ 142, 220, 1 ], [ 143, 143, 1 ], [ 144, 144, 1 ], [ 145, 131, 1 ], 
  [ 146, 57, 1 ], [ 147, 45, 1 ], [ 148, 58, 1 ], [ 149, 216, 1 ], [ 150, 136, 1 ], [ 151, 40, -1 ], 
  [ 152, 152, 1 ], [ 153, 52, -1 ], [ 154, 64, -1 ], [ 155, 53, -1 ], [ 156, 226, -1 ], [ 157, 157, 1 ], 
  [ 158, 48, 1 ], [ 159, 49, 1 ], [ 160, 31, -1 ], [ 161, 59, 1 ], [ 162, 223, -1 ], [ 163, 228, 1 ], 
  [ 164, 164, 1 ], [ 165, 27, 1 ], [ 166, 55, -1 ], [ 167, 229, -1 ], [ 168, 38, 1 ], [ 169, 39, 1 ], 
  [ 170, 225, 1 ], [ 171, 20, -1 ], [ 172, 33, -1 ], [ 173, 35, -1 ], [ 174, 227, -1 ], [ 175, 46, -1 ], 
  [ 176, 232, -1 ], [ 177, 26, 1 ], [ 178, 28, 1 ], [ 179, 41, 1 ], [ 180, 233, 1 ], [ 181, 97, 1 ], 
  [ 182, 230, -1 ], [ 183, 6, 1 ], [ 184, 34, -1 ], [ 185, 235, -1 ], [ 186, 93, 1 ], [ 187, 231, 1 ], 
  [ 188, 194, -1 ], [ 189, 13, -1 ], [ 190, 14, -1 ], [ 191, 89, 1 ], [ 192, 234, -1 ], [ 193, 199, -1 ], 
  [ 194, 188, -1 ], [ 195, 21, 1 ], [ 196, 76, 1 ], [ 197, 237, 1 ], [ 198, 204, -1 ], [ 199, 193, -1 ], 
  [ 200, 82, 1 ], [ 201, 236, 1 ], [ 202, 80, 1 ], [ 203, 215, 1 ], [ 204, 198, -1 ], [ 205, 85, 1 ], 
  [ 206, 211, -1 ], [ 207, 212, -1 ], [ 208, 210, 1 ], [ 209, 71, 1 ], [ 210, 208, 1 ], [ 211, 206, -1 ], 
  [ 212, 207, -1 ], [ 213, 66, 1 ], [ 214, 219, -1 ], [ 215, 203, 1 ], [ 216, 149, 1 ], [ 217, 61, 1 ], 
  [ 218, 222, -1 ], [ 219, 214, -1 ], [ 220, 142, 1 ], [ 221, 224, -1 ], [ 222, 218, -1 ], 
  [ 223, 162, -1 ], [ 224, 221, -1 ], [ 225, 170, 1 ], [ 226, 156, -1 ], [ 227, 174, -1 ], [ 228, 163, 1 ],
  [ 229, 167, -1 ], [ 230, 182, -1 ], [ 231, 187, 1 ], [ 232, 176, -1 ], [ 233, 180, 1 ], [ 234, 192, -1 ],
  [ 235, 185, -1 ], [ 236, 201, 1 ], [ 237, 197, 1 ], [ 238, 135, 1 ], [ 239, 128, 1 ], [ 240, 240, -1 ], 
  [ 241, 241, 1 ], [ 241, 246, -1 ], [ 241, 248, 2 ], [ 242, 244, 1 ], [ 242, 245, -1 ], [ 242, 246, -1 ], 
  [ 242, 248, 3 ], [ 243, 243, 1 ], [ 243, 246, -2 ], [ 243, 248, 4 ], [ 244, 244, 1 ], [ 244, 246, -3 ], 
  [ 244, 248, 6 ], [ 245, 242, -1 ], [ 245, 244, 1 ], [ 245, 246, -2 ], [ 245, 248, 5 ], [ 246, 246, -1 ], 
  [ 246, 248, 4 ], [ 247, 247, -1 ], [ 247, 248, 3 ], [ 248, 248, 1 ] ] ],

[ [ 1, 0, 0, 0, 0, 0, 0, 1 ],
  [ 248, [ 1, 1, -1 ], [ 2, 122, 1 ], [ 3, 3, 1 ], [ 4, 18, 1 ], [ 5, 125, 1 ], [ 6, 21, 1 ], [ 7, 127, 1 ], 
  [ 8, 15, -1 ], [ 9, 9, -1 ], [ 10, 12, 1 ], [ 11, 25, 1 ], [ 12, 10, 1 ], [ 13, 14, 1 ], [ 14, 13, 1 ], 
  [ 15, 8, -1 ], [ 16, 30, -1 ], [ 17, 19, 1 ], [ 18, 4, 1 ], [ 19, 17, 1 ], [ 20, 34, 1 ], [ 21, 6, 1 ], 
  [ 22, 29, -1 ], [ 23, 24, -1 ], [ 24, 23, -1 ], [ 25, 11, 1 ], [ 26, 28, 1 ], [ 27, 41, 1 ], 
  [ 28, 26, 1 ], [ 29, 22, -1 ], [ 30, 16, -1 ], [ 31, 46, -1 ], [ 32, 32, 1 ], [ 33, 35, 1 ], 
  [ 34, 20, 1 ], [ 35, 33, 1 ], [ 36, 42, -1 ], [ 37, 37, -1 ], [ 38, 39, -1 ], [ 39, 38, -1 ], 
  [ 40, 55, -1 ], [ 41, 27, 1 ], [ 42, 36, -1 ], [ 43, 50, -1 ], [ 44, 44, -1 ], [ 45, 59, 1 ], 
  [ 46, 31, -1 ], [ 47, 54, 1 ], [ 48, 49, -1 ], [ 49, 48, -1 ], [ 50, 43, -1 ], [ 51, 64, 1 ], 
  [ 52, 53, 1 ], [ 53, 52, 1 ], [ 54, 47, 1 ], [ 55, 40, -1 ], [ 56, 62, 1 ], [ 57, 58, 1 ], 
  [ 58, 57, 1 ], [ 59, 45, 1 ], [ 60, 67, -1 ], [ 61, 61, 1 ], [ 62, 56, 1 ], [ 63, 75, 1 ], 
  [ 64, 51, 1 ], [ 65, 72, -1 ], [ 66, 66, -1 ], [ 67, 60, -1 ], [ 68, 74, 1 ], [ 69, 70, 1 ], 
  [ 70, 69, 1 ], [ 71, 71, -1 ], [ 72, 65, -1 ], [ 73, 79, -1 ], [ 74, 68, 1 ], [ 75, 63, 1 ], 
  [ 76, 85, 1 ], [ 77, 81, -1 ], [ 78, 84, -1 ], [ 79, 73, -1 ], [ 80, 82, 1 ], [ 81, 77, -1 ], 
  [ 82, 80, 1 ], [ 83, 95, 1 ], [ 84, 78, -1 ], [ 85, 76, 1 ], [ 86, 92, 1 ], [ 87, 91, 1 ], 
  [ 88, 90, 1 ], [ 89, 89, -1 ], [ 90, 88, 1 ], [ 91, 87, 1 ], [ 92, 86, 1 ], [ 93, 93, -1 ], 
  [ 94, 99, -1 ], [ 95, 83, 1 ], [ 96, 100, -1 ], [ 97, 97, 1 ], [ 98, 102, -1 ], [ 99, 94, -1 ], 
  [ 100, 96, -1 ], [ 101, 104, 1 ], [ 102, 98, -1 ], [ 103, 106, -1 ], [ 104, 101, 1 ], [ 105, 108, -1 ], 
  [ 106, 103, -1 ], [ 107, 109, 1 ], [ 108, 105, -1 ], [ 109, 107, 1 ], [ 110, 112, 1 ], [ 111, 113, -1 ], 
  [ 112, 110, 1 ], [ 113, 111, -1 ], [ 114, 115, -1 ], [ 115, 114, -1 ], [ 116, 117, -1 ], 
  [ 117, 116, -1 ], [ 118, 119, -1 ], [ 119, 118, -1 ], [ 120, 120, -1 ], [ 121, 121, -1 ], [ 122, 2, 1 ], 
  [ 123, 123, 1 ], [ 124, 138, 1 ], [ 125, 5, 1 ], [ 126, 141, 1 ], [ 127, 7, 1 ], [ 128, 135, -1 ], 
  [ 129, 129, -1 ], [ 130, 132, 1 ], [ 131, 145, 1 ], [ 132, 130, 1 ], [ 133, 134, 1 ], [ 134, 133, 1 ], 
  [ 135, 128, -1 ], [ 136, 150, -1 ], [ 137, 139, 1 ], [ 138, 124, 1 ], [ 139, 137, 1 ], [ 140, 154, 1 ], 
  [ 141, 126, 1 ], [ 142, 149, -1 ], [ 143, 144, -1 ], [ 144, 143, -1 ], [ 145, 131, 1 ], [ 146, 148, 1 ], 
  [ 147, 161, 1 ], [ 148, 146, 1 ], [ 149, 142, -1 ], [ 150, 136, -1 ], [ 151, 166, -1 ], [ 152, 152, 1 ], 
  [ 153, 155, 1 ], [ 154, 140, 1 ], [ 155, 153, 1 ], [ 156, 162, -1 ], [ 157, 157, -1 ], [ 158, 159, -1 ], 
  [ 159, 158, -1 ], [ 160, 175, -1 ], [ 161, 147, 1 ], [ 162, 156, -1 ], [ 163, 170, -1 ], 
  [ 164, 164, -1 ], [ 165, 179, 1 ], [ 166, 151, -1 ], [ 167, 174, 1 ], [ 168, 169, -1 ], [ 169, 168, -1 ],
  [ 170, 163, -1 ], [ 171, 184, 1 ], [ 172, 173, 1 ], [ 173, 172, 1 ], [ 174, 167, 1 ], [ 175, 160, -1 ], 
  [ 176, 182, 1 ], [ 177, 178, 1 ], [ 178, 177, 1 ], [ 179, 165, 1 ], [ 180, 187, -1 ], [ 181, 181, 1 ], 
  [ 182, 176, 1 ], [ 183, 195, 1 ], [ 184, 171, 1 ], [ 185, 192, -1 ], [ 186, 186, -1 ], [ 187, 180, -1 ], 
  [ 188, 194, 1 ], [ 189, 190, 1 ], [ 190, 189, 1 ], [ 191, 191, -1 ], [ 192, 185, -1 ], [ 193, 199, -1 ], 
  [ 194, 188, 1 ], [ 195, 183, 1 ], [ 196, 205, 1 ], [ 197, 201, -1 ], [ 198, 204, -1 ], [ 199, 193, -1 ], 
  [ 200, 202, 1 ], [ 201, 197, -1 ], [ 202, 200, 1 ], [ 203, 215, 1 ], [ 204, 198, -1 ], [ 205, 196, 1 ], 
  [ 206, 212, 1 ], [ 207, 211, 1 ], [ 208, 210, 1 ], [ 209, 209, -1 ], [ 210, 208, 1 ], [ 211, 207, 1 ], 
  [ 212, 206, 1 ], [ 213, 213, -1 ], [ 214, 219, -1 ], [ 215, 203, 1 ], [ 216, 220, -1 ], [ 217, 217, 1 ], 
  [ 218, 222, -1 ], [ 219, 214, -1 ], [ 220, 216, -1 ], [ 221, 224, 1 ], [ 222, 218, -1 ], 
  [ 223, 226, -1 ], [ 224, 221, 1 ], [ 225, 228, -1 ], [ 226, 223, -1 ], [ 227, 229, 1 ], [ 228, 225, -1 ],
  [ 229, 227, 1 ], [ 230, 232, 1 ], [ 231, 233, -1 ], [ 232, 230, 1 ], [ 233, 231, -1 ], [ 234, 235, -1 ], 
  [ 235, 234, -1 ], [ 236, 237, -1 ], [ 237, 236, -1 ], [ 238, 239, -1 ], [ 239, 238, -1 ], 
  [ 240, 240, -1 ], [ 241, 241, 1 ], [ 242, 242, -1 ], [ 242, 244, 1 ], [ 243, 243, 1 ], [ 244, 244, 1 ], 
  [ 245, 244, 1 ], [ 245, 245, -1 ], [ 245, 246, 1 ], [ 246, 246, 1 ], [ 247, 246, 1 ], [ 247, 247, -1 ], 
  [ 247, 248, 1 ], [ 248, 248, 1 ] ] ],

[ [ 2, 0, 0, 0, 0, 0, 0, 0 ],
  [ 248, [ 1, 1, 1 ], [ 2, 132, 1 ], [ 3, 25, -1 ], [ 4, 28, 1 ], [ 5, 154, 1 ], [ 6, 42, -1 ], 
  [ 7, 149, -1 ], [ 8, 13, -1 ], [ 9, 30, -1 ], [ 10, 14, 1 ], [ 11, 55, 1 ], [ 12, 122, 1 ], 
  [ 13, 8, -1 ], [ 14, 10, 1 ], [ 15, 135, 1 ], [ 16, 59, 1 ], [ 17, 41, -1 ], [ 18, 138, 1 ], 
  [ 19, 19, 1 ], [ 20, 36, -1 ], [ 21, 141, 1 ], [ 22, 26, -1 ], [ 23, 46, -1 ], [ 24, 24, 1 ], 
  [ 25, 3, -1 ], [ 26, 22, -1 ], [ 27, 62, -1 ], [ 28, 4, 1 ], [ 29, 127, -1 ], [ 30, 9, -1 ], 
  [ 31, 67, -1 ], [ 32, 35, -1 ], [ 33, 50, 1 ], [ 34, 125, 1 ], [ 35, 32, -1 ], [ 36, 20, -1 ], 
  [ 37, 39, -1 ], [ 38, 54, 1 ], [ 39, 37, -1 ], [ 40, 74, 1 ], [ 41, 17, -1 ], [ 42, 6, -1 ], 
  [ 43, 48, -1 ], [ 44, 64, 1 ], [ 45, 79, 1 ], [ 46, 23, -1 ], [ 47, 52, -1 ], [ 48, 43, -1 ], 
  [ 49, 49, 1 ], [ 50, 33, 1 ], [ 51, 95, 1 ], [ 52, 47, -1 ], [ 53, 53, 1 ], [ 54, 38, 1 ], 
  [ 55, 11, 1 ], [ 56, 61, -1 ], [ 57, 72, 1 ], [ 58, 75, -1 ], [ 59, 16, 1 ], [ 60, 66, -1 ], 
  [ 61, 56, -1 ], [ 62, 27, -1 ], [ 63, 92, -1 ], [ 64, 44, 1 ], [ 65, 85, -1 ], [ 66, 60, -1 ], 
  [ 67, 31, -1 ], [ 68, 68, 1 ], [ 69, 84, -1 ], [ 70, 70, 1 ], [ 71, 81, 1 ], [ 72, 57, 1 ], 
  [ 73, 73, 1 ], [ 74, 40, 1 ], [ 75, 58, -1 ], [ 76, 99, 1 ], [ 77, 82, 1 ], [ 78, 90, 1 ], 
  [ 79, 45, 1 ], [ 80, 91, -1 ], [ 81, 71, 1 ], [ 82, 77, 1 ], [ 83, 106, 1 ], [ 84, 69, -1 ], 
  [ 85, 65, -1 ], [ 86, 100, 1 ], [ 87, 87, 1 ], [ 88, 89, 1 ], [ 89, 88, 1 ], [ 90, 78, 1 ], 
  [ 91, 80, -1 ], [ 92, 63, -1 ], [ 93, 102, -1 ], [ 94, 96, -1 ], [ 95, 51, 1 ], [ 96, 94, -1 ], 
  [ 97, 104, -1 ], [ 98, 108, -1 ], [ 99, 76, 1 ], [ 100, 86, 1 ], [ 101, 109, -1 ], [ 102, 93, -1 ], 
  [ 103, 103, 1 ], [ 104, 97, -1 ], [ 105, 112, 1 ], [ 106, 83, 1 ], [ 107, 113, 1 ], [ 108, 98, -1 ], 
  [ 109, 101, -1 ], [ 110, 110, 1 ], [ 111, 111, 1 ], [ 112, 105, 1 ], [ 113, 107, 1 ], [ 114, 117, 1 ], 
  [ 115, 115, -1 ], [ 116, 119, 1 ], [ 117, 114, 1 ], [ 118, 120, 1 ], [ 119, 116, 1 ], [ 120, 118, 1 ], 
  [ 121, 121, 1 ], [ 122, 12, 1 ], [ 123, 145, -1 ], [ 124, 148, 1 ], [ 125, 34, 1 ], [ 126, 162, -1 ], 
  [ 127, 29, -1 ], [ 128, 133, -1 ], [ 129, 150, -1 ], [ 130, 134, 1 ], [ 131, 175, 1 ], [ 132, 2, 1 ], 
  [ 133, 128, -1 ], [ 134, 130, 1 ], [ 135, 15, 1 ], [ 136, 179, 1 ], [ 137, 161, -1 ], [ 138, 18, 1 ], 
  [ 139, 139, 1 ], [ 140, 156, -1 ], [ 141, 21, 1 ], [ 142, 146, -1 ], [ 143, 166, -1 ], [ 144, 144, 1 ], 
  [ 145, 123, -1 ], [ 146, 142, -1 ], [ 147, 182, -1 ], [ 148, 124, 1 ], [ 149, 7, -1 ], [ 150, 129, -1 ], 
  [ 151, 187, -1 ], [ 152, 155, -1 ], [ 153, 170, 1 ], [ 154, 5, 1 ], [ 155, 152, -1 ], [ 156, 140, -1 ], 
  [ 157, 159, -1 ], [ 158, 174, 1 ], [ 159, 157, -1 ], [ 160, 194, 1 ], [ 161, 137, -1 ], [ 162, 126, -1 ],
  [ 163, 168, -1 ], [ 164, 184, 1 ], [ 165, 199, 1 ], [ 166, 143, -1 ], [ 167, 172, -1 ], [ 168, 163, -1 ],
  [ 169, 169, 1 ], [ 170, 153, 1 ], [ 171, 215, 1 ], [ 172, 167, -1 ], [ 173, 173, 1 ], [ 174, 158, 1 ], 
  [ 175, 131, 1 ], [ 176, 181, -1 ], [ 177, 192, 1 ], [ 178, 195, -1 ], [ 179, 136, 1 ], [ 180, 186, -1 ], 
  [ 181, 176, -1 ], [ 182, 147, -1 ], [ 183, 212, -1 ], [ 184, 164, 1 ], [ 185, 205, -1 ], 
  [ 186, 180, -1 ], [ 187, 151, -1 ], [ 188, 188, 1 ], [ 189, 204, -1 ], [ 190, 190, 1 ], [ 191, 201, 1 ], 
  [ 192, 177, 1 ], [ 193, 193, 1 ], [ 194, 160, 1 ], [ 195, 178, -1 ], [ 196, 219, 1 ], [ 197, 202, 1 ], 
  [ 198, 210, 1 ], [ 199, 165, 1 ], [ 200, 211, -1 ], [ 201, 191, 1 ], [ 202, 197, 1 ], [ 203, 226, 1 ], 
  [ 204, 189, -1 ], [ 205, 185, -1 ], [ 206, 220, 1 ], [ 207, 207, 1 ], [ 208, 209, 1 ], [ 209, 208, 1 ], 
  [ 210, 198, 1 ], [ 211, 200, -1 ], [ 212, 183, -1 ], [ 213, 222, -1 ], [ 214, 216, -1 ], [ 215, 171, 1 ],
  [ 216, 214, -1 ], [ 217, 224, -1 ], [ 218, 228, -1 ], [ 219, 196, 1 ], [ 220, 206, 1 ], [ 221, 229, -1 ],
  [ 222, 213, -1 ], [ 223, 223, 1 ], [ 224, 217, -1 ], [ 225, 232, 1 ], [ 226, 203, 1 ], [ 227, 233, 1 ], 
  [ 228, 218, -1 ], [ 229, 221, -1 ], [ 230, 230, 1 ], [ 231, 231, 1 ], [ 232, 225, 1 ], [ 233, 227, 1 ], 
  [ 234, 237, 1 ], [ 235, 235, -1 ], [ 236, 239, 1 ], [ 237, 234, 1 ], [ 238, 240, 1 ], [ 239, 236, 1 ], 
  [ 240, 238, 1 ], [ 241, 241, 1 ], [ 242, 243, 1 ], [ 242, 245, -1 ], [ 242, 246, 1 ], [ 243, 243, 1 ], 
  [ 244, 242, -1 ], [ 244, 243, 1 ], [ 244, 244, 1 ], [ 244, 245, -1 ], [ 244, 246, 1 ], [ 245, 242, -1 ], 
  [ 245, 243, 1 ], [ 245, 244, 1 ], [ 245, 245, -1 ], [ 245, 246, 1 ], [ 245, 247, -1 ], [ 245, 248, 1 ], 
  [ 246, 244, 1 ], [ 246, 245, -1 ], [ 246, 246, 1 ], [ 246, 247, -1 ], [ 246, 248, 1 ], [ 247, 244, 1 ], 
  [ 247, 245, -1 ], [ 247, 246, 1 ], [ 247, 247, -1 ], [ 248, 246, 1 ], [ 248, 247, -1 ] ]],

[ [ 0, 0, 0, 0, 0, 0, 2, 0 ],
  [ 248, [ 1, 5, 1/4 ], [ 1, 13, -1/4 ], [ 1, 18, -1/4*E(8)-1/4*E(8)^3 ], [ 1, 26, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 1, 30, -1/4*E(8)-1/4*E(8)^3 ], [ 1, 38, 1/4*E(8)+1/4*E(8)^3 ], [ 2, 152, -1/4 ], [ 2, 160, 1/4 ], 
  [ 2, 164, -1/4 ], [ 2, 171, 1/4 ], [ 3, 125, -1 ], [ 3, 133, 1 ], [ 3, 138, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 3, 146, -1/4*E(8)-1/4*E(8)^3 ], [ 3, 150, 1/4*E(8)+1/4*E(8)^3 ], [ 3, 158, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 4, 19, 1/4 ], [ 4, 27, -1/4 ], [ 4, 32, 1/4*E(8)+1/4*E(8)^3 ], [ 4, 40, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 4, 44, -1/4*E(8)-1/4*E(8)^3 ], [ 4, 51, 1/4*E(8)+1/4*E(8)^3 ], [ 5, 1, 1 ], 
  [ 5, 18, -1/2*E(8)-1/2*E(8)^3 ], [ 5, 26, -1/2*E(8)-1/2*E(8)^3 ], [ 5, 30, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 5, 38, 1/2*E(8)+1/2*E(8)^3 ], [ 5, 123, -1 ], [ 6, 6, 1/4 ], [ 6, 25, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 6, 33, 1/4*E(8)+1/4*E(8)^3 ], [ 6, 126, 1/4 ], [ 6, 145, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 6, 153, 1/4*E(8)+1/4*E(8)^3 ], [ 6, 241, -1/2 ], [ 6, 242, 1/2 ], [ 6, 243, 1/2 ], [ 6, 244, -1/2 ], 
  [ 6, 245, 1/4 ], [ 6, 247, -1/4 ], [ 7, 7, 1/4 ], [ 7, 14, 1/4 ], [ 7, 15, 1/4 ], [ 7, 22, 1/4 ], 
  [ 7, 41, -1/4*E(8)-1/4*E(8)^3 ], [ 7, 50, -1/4*E(8)-1/4*E(8)^3 ], [ 8, 8, -1/4 ], [ 8, 128, 9/4 ], 
  [ 8, 247, 3/4 ], [ 8, 248, -3/2 ], [ 9, 9, 1/4 ], [ 9, 10, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 9, 23, 1/2*E(8)+1/2*E(8)^3 ], [ 9, 129, 1/4 ], [ 9, 130, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 9, 143, 1/8*E(8)+1/8*E(8)^3 ], [ 9, 241, -1/4 ], [ 9, 242, -1/2 ], [ 9, 243, 1/4 ], [ 9, 244, -1/4 ], 
  [ 9, 245, 1/2 ], [ 10, 9, 1/8*E(8)+1/8*E(8)^3 ], [ 10, 129, -1/8*E(8)-1/8*E(8)^3 ], [ 10, 130, 1/8 ], 
  [ 10, 143, -1/8 ], [ 10, 241, 1/8*E(8)+1/8*E(8)^3 ], [ 10, 243, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 10, 244, -1/8*E(8)-1/8*E(8)^3 ], [ 11, 11, 1 ], [ 12, 12, 1/4 ], [ 12, 20, -1/4 ], [ 12, 24, -1/4 ], 
  [ 12, 31, 1/4 ], [ 12, 37, -1/2*E(8)-1/2*E(8)^3 ], [ 12, 45, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 12, 48, -1/4*E(8)-1/4*E(8)^3 ], [ 12, 57, -1/4*E(8)-1/4*E(8)^3 ], [ 12, 69, 1 ], [ 13, 1, -1 ], 
  [ 13, 18, -1/2*E(8)-1/2*E(8)^3 ], [ 13, 26, -1/2*E(8)-1/2*E(8)^3 ], [ 13, 30, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 13, 38, 1/2*E(8)+1/2*E(8)^3 ], [ 13, 123, 1 ], [ 14, 7, 1/4 ], [ 14, 14, 1/4 ], [ 14, 15, 1/4 ], 
  [ 14, 22, 1/4 ], [ 14, 41, 1/4*E(8)+1/4*E(8)^3 ], [ 14, 50, 1/4*E(8)+1/4*E(8)^3 ], [ 15, 7, 3/4 ], 
  [ 15, 14, 3/4 ], [ 15, 15, -1/4 ], [ 15, 22, -1/4 ], [ 15, 41, -3/4*E(8)-3/4*E(8)^3 ], 
  [ 15, 50, 1/4*E(8)+1/4*E(8)^3 ], [ 16, 19, -1/4 ], [ 16, 27, 1/4 ], [ 16, 32, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 16, 40, -1/4*E(8)-1/4*E(8)^3 ], [ 16, 44, -1/4*E(8)-1/4*E(8)^3 ], [ 16, 51, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 17, 138, -1/4 ], [ 17, 146, 1/4 ], [ 17, 150, 1/4 ], [ 17, 158, -1/4 ], 
  [ 18, 1, -1/4*E(8)-1/4*E(8)^3 ], [ 18, 5, -1/8*E(8)-1/8*E(8)^3 ], [ 18, 13, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 18, 18, 1/4 ], [ 18, 26, 1/4 ], [ 18, 30, 1/4 ], [ 18, 38, 1/4 ], [ 18, 123, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 18, 137, -1/4 ], [ 19, 4, 1 ], [ 19, 16, -1 ], [ 19, 32, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 19, 40, -1/2*E(8)-1/2*E(8)^3 ], [ 19, 44, -1/2*E(8)-1/2*E(8)^3 ], [ 19, 51, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 20, 12, -1/4 ], [ 20, 20, 1/4 ], [ 20, 24, 1/4 ], [ 20, 31, -1/4 ], [ 20, 37, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 20, 45, -1/2*E(8)-1/2*E(8)^3 ], [ 20, 48, -1/4*E(8)-1/4*E(8)^3 ], [ 20, 57, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 20, 69, 1 ], [ 21, 34, -1/2*E(8)-1/2*E(8)^3 ], [ 21, 42, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 21, 46, 1/2*E(8)+1/2*E(8)^3 ], [ 21, 54, 1/2*E(8)+1/2*E(8)^3 ], [ 22, 7, 3/4 ], [ 22, 14, 3/4 ], 
  [ 22, 15, -1/4 ], [ 22, 22, -1/4 ], [ 22, 41, 3/4*E(8)+3/4*E(8)^3 ], [ 22, 50, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 23, 9, 1/8*E(8)+1/8*E(8)^3 ], [ 23, 129, -1/8*E(8)-1/8*E(8)^3 ], [ 23, 130, -1/8 ], [ 23, 143, 1/8 ], 
  [ 23, 241, 1/8*E(8)+1/8*E(8)^3 ], [ 23, 243, 1/8*E(8)+1/8*E(8)^3 ], [ 23, 244, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 24, 12, -1/4 ], [ 24, 20, 1/4 ], [ 24, 24, 1/4 ], [ 24, 31, -1/4 ], [ 24, 37, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 24, 45, 1/2*E(8)+1/2*E(8)^3 ], [ 24, 48, 1/4*E(8)+1/4*E(8)^3 ], [ 24, 57, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 24, 69, 1 ], [ 25, 6, 1/4*E(8)+1/4*E(8)^3 ], [ 25, 126, -1/4*E(8)-1/4*E(8)^3 ], [ 25, 145, 1/2 ], 
  [ 25, 153, -1/2 ], [ 25, 245, 1/4*E(8)+1/4*E(8)^3 ], [ 25, 246, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 25, 247, 1/4*E(8)+1/4*E(8)^3 ], [ 26, 1, 1/4*E(8)+1/4*E(8)^3 ], [ 26, 5, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 26, 13, -1/8*E(8)-1/8*E(8)^3 ], [ 26, 18, 1/4 ], [ 26, 26, 1/4 ], [ 26, 30, 1/4 ], [ 26, 38, 1/4 ], 
  [ 26, 123, 1/4*E(8)+1/4*E(8)^3 ], [ 26, 137, 1/4 ], [ 27, 4, -1 ], [ 27, 16, 1 ], 
  [ 27, 32, -1/2*E(8)-1/2*E(8)^3 ], [ 27, 40, -1/2*E(8)-1/2*E(8)^3 ], [ 27, 44, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 27, 51, -1/2*E(8)-1/2*E(8)^3 ], [ 28, 55, -1/8*E(8)-1/8*E(8)^3 ], [ 28, 61, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 28, 62, -1/8*E(8)-1/8*E(8)^3 ], [ 28, 64, -1/8*E(8)-1/8*E(8)^3 ], [ 28, 68, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 28, 71, 1/8*E(8)+1/8*E(8)^3 ], [ 28, 72, -1/8*E(8)-1/8*E(8)^3 ], [ 28, 75, 1/2 ], 
  [ 28, 78, 1/8*E(8)+1/8*E(8)^3 ], [ 28, 80, -1/2 ], [ 28, 81, 1/2 ], [ 28, 86, -1/2 ], 
  [ 29, 34, -3/2*E(8)-3/2*E(8)^3 ], [ 29, 42, 1/2*E(8)+1/2*E(8)^3 ], [ 29, 46, 3/2*E(8)+3/2*E(8)^3 ], 
  [ 29, 54, -1/2*E(8)-1/2*E(8)^3 ], [ 30, 1, -1/4*E(8)-1/4*E(8)^3 ], [ 30, 5, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 30, 13, 1/8*E(8)+1/8*E(8)^3 ], [ 30, 18, 1/4 ], [ 30, 26, 1/4 ], [ 30, 30, 1/4 ], [ 30, 38, 1/4 ], 
  [ 30, 123, -1/4*E(8)-1/4*E(8)^3 ], [ 30, 137, 1/4 ], [ 31, 12, 1/4 ], [ 31, 20, -1/4 ], [ 31, 24, -1/4 ],
  [ 31, 31, 1/4 ], [ 31, 37, 1/2*E(8)+1/2*E(8)^3 ], [ 31, 45, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 31, 48, 1/4*E(8)+1/4*E(8)^3 ], [ 31, 57, 1/4*E(8)+1/4*E(8)^3 ], [ 31, 69, 1 ], 
  [ 32, 4, 1/4*E(8)+1/4*E(8)^3 ], [ 32, 16, 1/4*E(8)+1/4*E(8)^3 ], [ 32, 19, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 32, 27, -1/8*E(8)-1/8*E(8)^3 ], [ 32, 32, 1/4 ], [ 32, 40, 1/4 ], [ 32, 44, -1/4 ], [ 32, 51, -1/4 ], 
  [ 32, 122, -1/4 ], [ 33, 6, 1/4*E(8)+1/4*E(8)^3 ], [ 33, 126, -1/4*E(8)-1/4*E(8)^3 ], [ 33, 145, -1/2 ], 
  [ 33, 153, 1/2 ], [ 33, 245, 1/4*E(8)+1/4*E(8)^3 ], [ 33, 246, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 33, 247, 1/4*E(8)+1/4*E(8)^3 ], [ 34, 21, -1/8*E(8)-1/8*E(8)^3 ], [ 34, 29, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 34, 34, 1/4 ], [ 34, 42, 1/4 ], [ 34, 46, 1/4 ], [ 34, 54, 1/4 ], [ 35, 49, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 35, 56, -1/2*E(8)-1/2*E(8)^3 ], [ 35, 58, -1/2*E(8)-1/2*E(8)^3 ], [ 35, 65, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 36, 55, -3/8*E(8)-3/8*E(8)^3 ], [ 36, 61, 3/8*E(8)+3/8*E(8)^3 ], [ 36, 62, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 36, 64, -3/8*E(8)-3/8*E(8)^3 ], [ 36, 68, -1/8*E(8)-1/8*E(8)^3 ], [ 36, 71, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 36, 72, 1/8*E(8)+1/8*E(8)^3 ], [ 36, 75, 3/2 ], [ 36, 78, -1/8*E(8)-1/8*E(8)^3 ], [ 36, 80, -3/2 ], 
  [ 36, 81, -1/2 ], [ 36, 86, 1/2 ], [ 37, 12, -1/8*E(8)-1/8*E(8)^3 ], [ 37, 20, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 37, 24, -1/8*E(8)-1/8*E(8)^3 ], [ 37, 31, 1/8*E(8)+1/8*E(8)^3 ], [ 37, 48, -1/4 ], [ 37, 57, 1/4 ], 
  [ 38, 1, 1/4*E(8)+1/4*E(8)^3 ], [ 38, 5, 1/8*E(8)+1/8*E(8)^3 ], [ 38, 13, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 38, 18, 1/4 ], [ 38, 26, 1/4 ], [ 38, 30, 1/4 ], [ 38, 38, 1/4 ], [ 38, 123, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 38, 137, -1/4 ], [ 39, 55, 1/8*E(8)+1/8*E(8)^3 ], [ 39, 61, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 39, 62, 1/8*E(8)+1/8*E(8)^3 ], [ 39, 64, 1/8*E(8)+1/8*E(8)^3 ], [ 39, 68, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 39, 71, -1/8*E(8)-1/8*E(8)^3 ], [ 39, 72, 1/8*E(8)+1/8*E(8)^3 ], [ 39, 75, 1/2 ], 
  [ 39, 78, -1/8*E(8)-1/8*E(8)^3 ], [ 39, 80, -1/2 ], [ 39, 81, 1/2 ], [ 39, 86, -1/2 ], 
  [ 40, 4, -1/4*E(8)-1/4*E(8)^3 ], [ 40, 16, -1/4*E(8)-1/4*E(8)^3 ], [ 40, 19, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 40, 27, -1/8*E(8)-1/8*E(8)^3 ], [ 40, 32, 1/4 ], [ 40, 40, 1/4 ], [ 40, 44, -1/4 ], [ 40, 51, -1/4 ], 
  [ 40, 122, 1/4 ], [ 41, 7, -1/4*E(8)-1/4*E(8)^3 ], [ 41, 14, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 41, 15, -1/4*E(8)-1/4*E(8)^3 ], [ 41, 22, 1/4*E(8)+1/4*E(8)^3 ], [ 42, 21, -3/8*E(8)-3/8*E(8)^3 ], 
  [ 42, 29, 1/8*E(8)+1/8*E(8)^3 ], [ 42, 34, 3/4 ], [ 42, 42, -1/4 ], [ 42, 46, 3/4 ], [ 42, 54, -1/4 ], 
  [ 43, 49, -3/2*E(8)-3/2*E(8)^3 ], [ 43, 56, 1/2*E(8)+1/2*E(8)^3 ], [ 43, 58, -3/2*E(8)-3/2*E(8)^3 ], 
  [ 43, 65, 1/2*E(8)+1/2*E(8)^3 ], [ 44, 4, -1/4*E(8)-1/4*E(8)^3 ], [ 44, 16, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 44, 19, -1/8*E(8)-1/8*E(8)^3 ], [ 44, 27, -1/8*E(8)-1/8*E(8)^3 ], [ 44, 32, -1/4 ], [ 44, 40, -1/4 ], 
  [ 44, 44, 1/4 ], [ 44, 51, 1/4 ], [ 44, 122, -1/4 ], [ 45, 12, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 45, 20, -1/8*E(8)-1/8*E(8)^3 ], [ 45, 24, 1/8*E(8)+1/8*E(8)^3 ], [ 45, 31, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 45, 48, -1/4 ], [ 45, 57, 1/4 ], [ 46, 21, 1/8*E(8)+1/8*E(8)^3 ], [ 46, 29, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 46, 34, 1/4 ], [ 46, 42, 1/4 ], [ 46, 46, 1/4 ], [ 46, 54, 1/4 ], [ 47, 55, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 47, 61, -3/8*E(8)-3/8*E(8)^3 ], [ 47, 62, -1/8*E(8)-1/8*E(8)^3 ], [ 47, 64, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 47, 68, 1/8*E(8)+1/8*E(8)^3 ], [ 47, 71, -3/8*E(8)-3/8*E(8)^3 ], [ 47, 72, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 47, 75, 3/2 ], [ 47, 78, 1/8*E(8)+1/8*E(8)^3 ], [ 47, 80, -3/2 ], [ 47, 81, -1/2 ], [ 47, 86, 1/2 ], 
  [ 48, 12, -1/4*E(8)-1/4*E(8)^3 ], [ 48, 20, -1/4*E(8)-1/4*E(8)^3 ], [ 48, 24, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 48, 31, 1/4*E(8)+1/4*E(8)^3 ], [ 48, 37, -1 ], [ 48, 45, -1 ], [ 49, 35, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 49, 43, -1/8*E(8)-1/8*E(8)^3 ], [ 49, 49, 1/4 ], [ 49, 56, 1/4 ], [ 49, 58, -1/4 ], [ 49, 65, -1/4 ], 
  [ 50, 7, -3/4*E(8)-3/4*E(8)^3 ], [ 50, 14, 3/4*E(8)+3/4*E(8)^3 ], [ 50, 15, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 50, 22, -1/4*E(8)-1/4*E(8)^3 ], [ 51, 4, 1/4*E(8)+1/4*E(8)^3 ], [ 51, 16, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 51, 19, -1/8*E(8)-1/8*E(8)^3 ], [ 51, 27, -1/8*E(8)-1/8*E(8)^3 ], [ 51, 32, -1/4 ], [ 51, 40, -1/4 ], 
  [ 51, 44, 1/4 ], [ 51, 51, 1/4 ], [ 51, 122, 1/4 ], [ 52, 52, 1 ], [ 53, 55, -1/8 ], [ 53, 61, 1/8 ], 
  [ 53, 62, -1/8 ], [ 53, 64, 1/8 ], [ 53, 68, 1/8 ], [ 53, 71, -1/8 ], [ 53, 72, 1/8 ], [ 53, 78, -1/8 ], 
  [ 54, 21, 3/8*E(8)+3/8*E(8)^3 ], [ 54, 29, -1/8*E(8)-1/8*E(8)^3 ], [ 54, 34, 3/4 ], [ 54, 42, -1/4 ], 
  [ 54, 46, 3/4 ], [ 54, 54, -1/4 ], [ 55, 28, -1/8*E(8)-1/8*E(8)^3 ], [ 55, 36, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 55, 39, 1/8*E(8)+1/8*E(8)^3 ], [ 55, 47, 1/8*E(8)+1/8*E(8)^3 ], [ 55, 53, -1/2 ], [ 55, 55, 1/8 ], 
  [ 55, 60, -1/2 ], [ 55, 61, 1/8 ], [ 55, 62, 1/8 ], [ 55, 64, 1/8 ], [ 55, 68, 1/8 ], [ 55, 71, 1/8 ], 
  [ 55, 72, 1/8 ], [ 55, 75, 1/4*E(8)+1/4*E(8)^3 ], [ 55, 78, 1/8 ], [ 55, 80, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 55, 81, 1/4*E(8)+1/4*E(8)^3 ], [ 55, 86, 1/4*E(8)+1/4*E(8)^3 ], [ 56, 35, -3/8*E(8)-3/8*E(8)^3 ], 
  [ 56, 43, 1/8*E(8)+1/8*E(8)^3 ], [ 56, 49, 3/4 ], [ 56, 56, -1/4 ], [ 56, 58, -3/4 ], [ 56, 65, 1/4 ], 
  [ 57, 12, -1/4*E(8)-1/4*E(8)^3 ], [ 57, 20, -1/4*E(8)-1/4*E(8)^3 ], [ 57, 24, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 57, 31, 1/4*E(8)+1/4*E(8)^3 ], [ 57, 37, 1 ], [ 57, 45, 1 ], [ 58, 35, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 58, 43, -1/8*E(8)-1/8*E(8)^3 ], [ 58, 49, -1/4 ], [ 58, 56, -1/4 ], [ 58, 58, 1/4 ], [ 58, 65, 1/4 ], 
  [ 59, 59, 1/4 ], [ 59, 66, -1/4 ], [ 59, 67, 1/4 ], [ 59, 73, -1/4 ], [ 59, 85, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 59, 90, -1/4*E(8)-1/4*E(8)^3 ], [ 60, 55, -3/8 ], [ 60, 61, 3/8 ], [ 60, 62, 1/8 ], [ 60, 64, 3/8 ], 
  [ 60, 68, -1/8 ], [ 60, 71, -3/8 ], [ 60, 72, -1/8 ], [ 60, 78, 1/8 ], [ 61, 28, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 61, 36, 1/8*E(8)+1/8*E(8)^3 ], [ 61, 39, -1/8*E(8)-1/8*E(8)^3 ], [ 61, 47, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 61, 53, 1/2 ], [ 61, 55, 1/8 ], [ 61, 60, 1/2 ], [ 61, 61, 1/8 ], [ 61, 62, 1/8 ], [ 61, 64, 1/8 ], 
  [ 61, 68, 1/8 ], [ 61, 71, 1/8 ], [ 61, 72, 1/8 ], [ 61, 75, 1/4*E(8)+1/4*E(8)^3 ], [ 61, 78, 1/8 ], 
  [ 61, 80, 1/4*E(8)+1/4*E(8)^3 ], [ 61, 81, 1/4*E(8)+1/4*E(8)^3 ], [ 61, 86, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 62, 28, -3/8*E(8)-3/8*E(8)^3 ], [ 62, 36, 1/8*E(8)+1/8*E(8)^3 ], [ 62, 39, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 62, 47, -1/8*E(8)-1/8*E(8)^3 ], [ 62, 53, -3/2 ], [ 62, 55, 3/8 ], [ 62, 60, 1/2 ], [ 62, 61, 3/8 ], 
  [ 62, 62, -1/8 ], [ 62, 64, 3/8 ], [ 62, 68, -1/8 ], [ 62, 71, 3/8 ], [ 62, 72, -1/8 ], 
  [ 62, 75, 3/4*E(8)+3/4*E(8)^3 ], [ 62, 78, -1/8 ], [ 62, 80, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 62, 81, -1/4*E(8)-1/4*E(8)^3 ], [ 62, 86, -1/4*E(8)-1/4*E(8)^3 ], [ 63, 63, 1 ], 
  [ 64, 28, -1/8*E(8)-1/8*E(8)^3 ], [ 64, 36, -1/8*E(8)-1/8*E(8)^3 ], [ 64, 39, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 64, 47, 1/8*E(8)+1/8*E(8)^3 ], [ 64, 53, 1/2 ], [ 64, 55, 1/8 ], [ 64, 60, 1/2 ], [ 64, 61, 1/8 ], 
  [ 64, 62, 1/8 ], [ 64, 64, 1/8 ], [ 64, 68, 1/8 ], [ 64, 71, 1/8 ], [ 64, 72, 1/8 ], 
  [ 64, 75, -1/4*E(8)-1/4*E(8)^3 ], [ 64, 78, 1/8 ], [ 64, 80, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 64, 81, -1/4*E(8)-1/4*E(8)^3 ], [ 64, 86, -1/4*E(8)-1/4*E(8)^3 ], [ 65, 35, -3/8*E(8)-3/8*E(8)^3 ], 
  [ 65, 43, 1/8*E(8)+1/8*E(8)^3 ], [ 65, 49, -3/4 ], [ 65, 56, 1/4 ], [ 65, 58, 3/4 ], [ 65, 65, -1/4 ], 
  [ 66, 59, -1/4 ], [ 66, 66, 1/4 ], [ 66, 67, -1/4 ], [ 66, 73, 1/4 ], [ 66, 85, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 66, 90, -1/4*E(8)-1/4*E(8)^3 ], [ 67, 59, 3/4 ], [ 67, 66, -3/4 ], [ 67, 67, -1/4 ], [ 67, 73, 1/4 ], 
  [ 67, 85, -3/4*E(8)-3/4*E(8)^3 ], [ 67, 90, 1/4*E(8)+1/4*E(8)^3 ], [ 68, 28, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 68, 36, -1/8*E(8)-1/8*E(8)^3 ], [ 68, 39, -3/8*E(8)-3/8*E(8)^3 ], [ 68, 47, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 68, 53, 3/2 ], [ 68, 55, 3/8 ], [ 68, 60, -1/2 ], [ 68, 61, 3/8 ], [ 68, 62, -1/8 ], [ 68, 64, 3/8 ], 
  [ 68, 68, -1/8 ], [ 68, 71, 3/8 ], [ 68, 72, -1/8 ], [ 68, 75, 3/4*E(8)+3/4*E(8)^3 ], [ 68, 78, -1/8 ], 
  [ 68, 80, 3/4*E(8)+3/4*E(8)^3 ], [ 68, 81, -1/4*E(8)-1/4*E(8)^3 ], [ 68, 86, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 69, 12, 1/4 ], [ 69, 20, 1/4 ], [ 69, 24, 1/4 ], [ 69, 31, 1/4 ], [ 70, 70, 1/4 ], [ 70, 76, -1/4 ], 
  [ 70, 77, 1/4 ], [ 70, 83, -1/4 ], [ 70, 93, -1/4*E(8)-1/4*E(8)^3 ], [ 70, 98, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 71, 28, 1/8*E(8)+1/8*E(8)^3 ], [ 71, 36, 1/8*E(8)+1/8*E(8)^3 ], [ 71, 39, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 71, 47, -1/8*E(8)-1/8*E(8)^3 ], [ 71, 53, -1/2 ], [ 71, 55, 1/8 ], [ 71, 60, -1/2 ], [ 71, 61, 1/8 ], 
  [ 71, 62, 1/8 ], [ 71, 64, 1/8 ], [ 71, 68, 1/8 ], [ 71, 71, 1/8 ], [ 71, 72, 1/8 ], 
  [ 71, 75, -1/4*E(8)-1/4*E(8)^3 ], [ 71, 78, 1/8 ], [ 71, 80, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 71, 81, -1/4*E(8)-1/4*E(8)^3 ], [ 71, 86, -1/4*E(8)-1/4*E(8)^3 ], [ 72, 28, -3/8*E(8)-3/8*E(8)^3 ], 
  [ 72, 36, 1/8*E(8)+1/8*E(8)^3 ], [ 72, 39, 3/8*E(8)+3/8*E(8)^3 ], [ 72, 47, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 72, 53, 3/2 ], [ 72, 55, 3/8 ], [ 72, 60, -1/2 ], [ 72, 61, 3/8 ], [ 72, 62, -1/8 ], [ 72, 64, 3/8 ], 
  [ 72, 68, -1/8 ], [ 72, 71, 3/8 ], [ 72, 72, -1/8 ], [ 72, 75, -3/4*E(8)-3/4*E(8)^3 ], [ 72, 78, -1/8 ], 
  [ 72, 80, -3/4*E(8)-3/4*E(8)^3 ], [ 72, 81, 1/4*E(8)+1/4*E(8)^3 ], [ 72, 86, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 73, 59, -3/4 ], [ 73, 66, 3/4 ], [ 73, 67, 1/4 ], [ 73, 73, -1/4 ], [ 73, 85, -3/4*E(8)-3/4*E(8)^3 ], 
  [ 73, 90, 1/4*E(8)+1/4*E(8)^3 ], [ 74, 74, -1/2 ], [ 74, 84, -1/2 ], [ 74, 91, -E(8)-E(8)^3 ], 
  [ 75, 28, 1/8 ], [ 75, 36, 1/8 ], [ 75, 39, 1/8 ], [ 75, 47, 1/8 ], [ 75, 55, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 75, 61, 1/16*E(8)+1/16*E(8)^3 ], [ 75, 62, 1/16*E(8)+1/16*E(8)^3 ], [ 75, 64, -1/16*E(8)-1/16*E(8)^3 ],
  [ 75, 68, 1/16*E(8)+1/16*E(8)^3 ], [ 75, 71, -1/16*E(8)-1/16*E(8)^3 ], [ 75, 72, -1/16*E(8)-1/16*E(8)^3 ]
    , [ 75, 78, -1/16*E(8)-1/16*E(8)^3 ], [ 76, 70, -1/4 ], [ 76, 76, 1/4 ], [ 76, 77, -1/4 ], 
  [ 76, 83, 1/4 ], [ 76, 93, -1/4*E(8)-1/4*E(8)^3 ], [ 76, 98, -1/4*E(8)-1/4*E(8)^3 ], [ 77, 70, 3/4 ], 
  [ 77, 76, -3/4 ], [ 77, 77, -1/4 ], [ 77, 83, 1/4 ], [ 77, 93, -3/4*E(8)-3/4*E(8)^3 ], 
  [ 77, 98, 1/4*E(8)+1/4*E(8)^3 ], [ 78, 28, 3/8*E(8)+3/8*E(8)^3 ], [ 78, 36, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 78, 39, -3/8*E(8)-3/8*E(8)^3 ], [ 78, 47, 1/8*E(8)+1/8*E(8)^3 ], [ 78, 53, -3/2 ], [ 78, 55, 3/8 ], 
  [ 78, 60, 1/2 ], [ 78, 61, 3/8 ], [ 78, 62, -1/8 ], [ 78, 64, 3/8 ], [ 78, 68, -1/8 ], [ 78, 71, 3/8 ], 
  [ 78, 72, -1/8 ], [ 78, 75, -3/4*E(8)-3/4*E(8)^3 ], [ 78, 78, -1/8 ], [ 78, 80, -3/4*E(8)-3/4*E(8)^3 ], 
  [ 78, 81, 1/4*E(8)+1/4*E(8)^3 ], [ 78, 86, 1/4*E(8)+1/4*E(8)^3 ], [ 79, 95, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 79, 100, -1/2*E(8)-1/2*E(8)^3 ], [ 80, 28, -1/8 ], [ 80, 36, -1/8 ], [ 80, 39, -1/8 ], 
  [ 80, 47, -1/8 ], [ 80, 55, 1/16*E(8)+1/16*E(8)^3 ], [ 80, 61, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 80, 62, 1/16*E(8)+1/16*E(8)^3 ], [ 80, 64, -1/16*E(8)-1/16*E(8)^3 ], [ 80, 68, 1/16*E(8)+1/16*E(8)^3 ],
  [ 80, 71, -1/16*E(8)-1/16*E(8)^3 ], [ 80, 72, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 80, 78, -1/16*E(8)-1/16*E(8)^3 ], [ 81, 28, 3/8 ], [ 81, 36, -1/8 ], [ 81, 39, 3/8 ], [ 81, 47, -1/8 ],
  [ 81, 55, 3/16*E(8)+3/16*E(8)^3 ], [ 81, 61, 3/16*E(8)+3/16*E(8)^3 ], [ 81, 62, -1/16*E(8)-1/16*E(8)^3 ],
  [ 81, 64, -3/16*E(8)-3/16*E(8)^3 ], [ 81, 68, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 81, 71, -3/16*E(8)-3/16*E(8)^3 ], [ 81, 72, 1/16*E(8)+1/16*E(8)^3 ], [ 81, 78, 1/16*E(8)+1/16*E(8)^3 ],
  [ 82, 89, 1/2*E(8)+1/2*E(8)^3 ], [ 82, 94, 1/2*E(8)+1/2*E(8)^3 ], [ 82, 97, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 82, 101, -1/2*E(8)-1/2*E(8)^3 ], [ 83, 70, -3/4 ], [ 83, 76, 3/4 ], [ 83, 77, 1/4 ], [ 83, 83, -1/4 ], 
  [ 83, 93, -3/4*E(8)-3/4*E(8)^3 ], [ 83, 98, 1/4*E(8)+1/4*E(8)^3 ], [ 84, 74, -1/2 ], [ 84, 84, -1/2 ], 
  [ 84, 91, E(8)+E(8)^3 ], [ 85, 59, -1/4*E(8)-1/4*E(8)^3 ], [ 85, 66, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 85, 67, -1/4*E(8)-1/4*E(8)^3 ], [ 85, 73, -1/4*E(8)-1/4*E(8)^3 ], [ 86, 28, -3/8 ], [ 86, 36, 1/8 ], 
  [ 86, 39, -3/8 ], [ 86, 47, 1/8 ], [ 86, 55, 3/16*E(8)+3/16*E(8)^3 ], [ 86, 61, 3/16*E(8)+3/16*E(8)^3 ], 
  [ 86, 62, -1/16*E(8)-1/16*E(8)^3 ], [ 86, 64, -3/16*E(8)-3/16*E(8)^3 ], 
  [ 86, 68, -1/16*E(8)-1/16*E(8)^3 ], [ 86, 71, -3/16*E(8)-3/16*E(8)^3 ], [ 86, 72, 1/16*E(8)+1/16*E(8)^3 ]
    , [ 86, 78, 1/16*E(8)+1/16*E(8)^3 ], [ 87, 89, 3/2*E(8)+3/2*E(8)^3 ], [ 87, 94, -1/2*E(8)-1/2*E(8)^3 ],
  [ 87, 97, -3/2*E(8)-3/2*E(8)^3 ], [ 87, 101, 1/2*E(8)+1/2*E(8)^3 ], [ 88, 102, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 88, 105, -1/2*E(8)-1/2*E(8)^3 ], [ 89, 82, 1/8*E(8)+1/8*E(8)^3 ], [ 89, 87, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 89, 89, 1/4 ], [ 89, 94, 1/4 ], [ 89, 97, 1/4 ], [ 89, 101, 1/4 ], [ 90, 59, -3/4*E(8)-3/4*E(8)^3 ], 
  [ 90, 66, -3/4*E(8)-3/4*E(8)^3 ], [ 90, 67, 1/4*E(8)+1/4*E(8)^3 ], [ 90, 73, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 91, 74, -1/4*E(8)-1/4*E(8)^3 ], [ 91, 84, 1/4*E(8)+1/4*E(8)^3 ], [ 92, 99, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 92, 103, 1/2*E(8)+1/2*E(8)^3 ], [ 92, 104, 1/2*E(8)+1/2*E(8)^3 ], [ 92, 107, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 92, 112, 1 ], [ 92, 115, 1 ], [ 93, 70, -1/4*E(8)-1/4*E(8)^3 ], [ 93, 76, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 93, 77, -1/4*E(8)-1/4*E(8)^3 ], [ 93, 83, -1/4*E(8)-1/4*E(8)^3 ], [ 94, 82, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 94, 87, -1/8*E(8)-1/8*E(8)^3 ], [ 94, 89, 3/4 ], [ 94, 94, -1/4 ], [ 94, 97, 3/4 ], [ 94, 101, -1/4 ], 
  [ 95, 79, 1/2*E(8)+1/2*E(8)^3 ], [ 95, 95, -1/2 ], [ 95, 100, -1/2 ], [ 96, 99, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 96, 103, -1/2*E(8)-1/2*E(8)^3 ], [ 96, 104, -1/2*E(8)-1/2*E(8)^3 ], [ 96, 107, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 96, 112, 1 ], [ 96, 115, 1 ], [ 97, 82, -1/8*E(8)-1/8*E(8)^3 ], [ 97, 87, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 97, 89, 1/4 ], [ 97, 94, 1/4 ], [ 97, 97, 1/4 ], [ 97, 101, 1/4 ], [ 98, 70, -3/4*E(8)-3/4*E(8)^3 ], 
  [ 98, 76, -3/4*E(8)-3/4*E(8)^3 ], [ 98, 77, 1/4*E(8)+1/4*E(8)^3 ], [ 98, 83, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 99, 92, -1/8*E(8)-1/8*E(8)^3 ], [ 99, 96, 1/8*E(8)+1/8*E(8)^3 ], [ 99, 99, -1/4 ], [ 99, 103, 1/4 ], 
  [ 99, 104, -1/4 ], [ 99, 107, 1/4 ], [ 99, 108, 1/4 ], [ 99, 112, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 99, 115, 1/4*E(8)+1/4*E(8)^3 ], [ 100, 79, -1/2*E(8)-1/2*E(8)^3 ], [ 100, 95, -1/2 ], 
  [ 100, 100, -1/2 ], [ 101, 82, -3/8*E(8)-3/8*E(8)^3 ], [ 101, 87, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 101, 89, 3/4 ], [ 101, 94, -1/4 ], [ 101, 97, 3/4 ], [ 101, 101, -1/4 ], 
  [ 102, 88, 1/2*E(8)+1/2*E(8)^3 ], [ 102, 102, -1/2 ], [ 102, 105, -1/2 ], 
  [ 103, 92, 1/8*E(8)+1/8*E(8)^3 ], [ 103, 96, -1/8*E(8)-1/8*E(8)^3 ], [ 103, 99, 1/4 ], 
  [ 103, 103, -1/4 ], [ 103, 104, 1/4 ], [ 103, 107, -1/4 ], [ 103, 108, 1/4 ], 
  [ 103, 112, -1/4*E(8)-1/4*E(8)^3 ], [ 103, 115, 1/4*E(8)+1/4*E(8)^3 ], [ 104, 92, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 104, 96, -1/8*E(8)-1/8*E(8)^3 ], [ 104, 99, -1/4 ], [ 104, 103, 1/4 ], [ 104, 104, -1/4 ], 
  [ 104, 107, 1/4 ], [ 104, 108, -1/4 ], [ 104, 112, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 104, 115, 1/4*E(8)+1/4*E(8)^3 ], [ 105, 88, -1/2*E(8)-1/2*E(8)^3 ], [ 105, 102, -1/2 ], 
  [ 105, 105, -1/2 ], [ 106, 106, -1/2 ], [ 106, 109, 1/2 ], [ 106, 113, -E(8)-E(8)^3 ], 
  [ 107, 92, -1/8*E(8)-1/8*E(8)^3 ], [ 107, 96, 1/8*E(8)+1/8*E(8)^3 ], [ 107, 99, 1/4 ], 
  [ 107, 103, -1/4 ], [ 107, 104, 1/4 ], [ 107, 107, -1/4 ], [ 107, 108, -1/4 ], 
  [ 107, 112, -1/4*E(8)-1/4*E(8)^3 ], [ 107, 115, 1/4*E(8)+1/4*E(8)^3 ], [ 108, 99, 1 ], [ 108, 103, 1 ], 
  [ 108, 104, -1 ], [ 108, 107, -1 ], [ 109, 106, 1/2 ], [ 109, 109, -1/2 ], [ 109, 113, -E(8)-E(8)^3 ], 
  [ 110, 110, -1/2 ], [ 110, 114, -1/2 ], [ 110, 116, E(8)+E(8)^3 ], [ 111, 117, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 111, 118, -1/2*E(8)-1/2*E(8)^3 ], [ 112, 92, 1/4 ], [ 112, 96, 1/4 ], [ 112, 99, -1/4*E(8)-1/4*E(8)^3 ]
    , [ 112, 103, -1/4*E(8)-1/4*E(8)^3 ], [ 112, 104, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 112, 107, -1/4*E(8)-1/4*E(8)^3 ], [ 113, 106, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 113, 109, -1/4*E(8)-1/4*E(8)^3 ], [ 114, 110, -1/2 ], [ 114, 114, -1/2 ], [ 114, 116, -E(8)-E(8)^3 ], 
  [ 115, 92, 1/4 ], [ 115, 96, 1/4 ], [ 115, 99, 1/4*E(8)+1/4*E(8)^3 ], [ 115, 103, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 115, 104, 1/4*E(8)+1/4*E(8)^3 ], [ 115, 107, 1/4*E(8)+1/4*E(8)^3 ], [ 116, 110, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 116, 114, -1/4*E(8)-1/4*E(8)^3 ], [ 117, 111, -1/2*E(8)-1/2*E(8)^3 ], [ 117, 117, -1/2 ], 
  [ 117, 118, 1/2 ], [ 118, 111, -1/2*E(8)-1/2*E(8)^3 ], [ 118, 117, 1/2 ], [ 118, 118, -1/2 ], 
  [ 119, 119, -1/2 ], [ 119, 120, 1/2 ], [ 120, 119, 3/2 ], [ 120, 120, 1/2 ], [ 121, 125, 1 ], 
  [ 121, 133, -1 ], [ 121, 138, 1/4*E(8)+1/4*E(8)^3 ], [ 121, 146, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 121, 150, 1/4*E(8)+1/4*E(8)^3 ], [ 121, 158, -1/4*E(8)-1/4*E(8)^3 ], [ 122, 32, -1 ], [ 122, 40, 1 ], 
  [ 122, 44, -1 ], [ 122, 51, 1 ], [ 123, 5, -1/4 ], [ 123, 13, 1/4 ], [ 123, 18, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 123, 26, 1/4*E(8)+1/4*E(8)^3 ], [ 123, 30, -1/4*E(8)-1/4*E(8)^3 ], [ 123, 38, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 124, 139, 1 ], [ 124, 147, -1 ], [ 124, 152, -1/4*E(8)-1/4*E(8)^3 ], [ 124, 160, 1/4*E(8)+1/4*E(8)^3 ],
  [ 124, 164, 1/4*E(8)+1/4*E(8)^3 ], [ 124, 171, -1/4*E(8)-1/4*E(8)^3 ], [ 125, 3, -1/4 ], 
  [ 125, 121, 1/4 ], [ 125, 138, 1/8*E(8)+1/8*E(8)^3 ], [ 125, 146, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 125, 150, -1/8*E(8)-1/8*E(8)^3 ], [ 125, 158, -1/8*E(8)-1/8*E(8)^3 ], [ 126, 6, 1/4 ], 
  [ 126, 25, -1/4*E(8)-1/4*E(8)^3 ], [ 126, 33, -1/4*E(8)-1/4*E(8)^3 ], [ 126, 126, 1/4 ], 
  [ 126, 145, -1/4*E(8)-1/4*E(8)^3 ], [ 126, 153, -1/4*E(8)-1/4*E(8)^3 ], [ 126, 241, -1/2 ], 
  [ 126, 242, 1/2 ], [ 126, 243, 1/2 ], [ 126, 244, -1/2 ], [ 126, 245, 1/4 ], [ 126, 247, -1/4 ], 
  [ 127, 127, 1/4 ], [ 127, 134, 1/4 ], [ 127, 135, 3/4 ], [ 127, 142, 3/4 ], 
  [ 127, 161, 1/4*E(8)+1/4*E(8)^3 ], [ 127, 170, 3/4*E(8)+3/4*E(8)^3 ], [ 128, 8, 1/4 ], 
  [ 128, 128, -1/4 ], [ 128, 247, 1/4 ], [ 128, 248, -1/2 ], [ 129, 9, 1/4 ], 
  [ 129, 10, -1/2*E(8)-1/2*E(8)^3 ], [ 129, 23, -1/2*E(8)-1/2*E(8)^3 ], [ 129, 129, 1/4 ], 
  [ 129, 130, -1/8*E(8)-1/8*E(8)^3 ], [ 129, 143, -1/8*E(8)-1/8*E(8)^3 ], [ 129, 241, -1/4 ], 
  [ 129, 242, -1/2 ], [ 129, 243, 1/4 ], [ 129, 244, -1/4 ], [ 129, 245, 1/2 ], 
  [ 130, 9, 1/2*E(8)+1/2*E(8)^3 ], [ 130, 10, 2 ], [ 130, 23, -2 ], [ 130, 129, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 130, 241, -1/2*E(8)-1/2*E(8)^3 ], [ 130, 243, -1/2*E(8)-1/2*E(8)^3 ], [ 130, 244, 1/2*E(8)+1/2*E(8)^3 ]
    , [ 131, 131, 1 ], [ 132, 132, 1/4 ], [ 132, 140, -1/4 ], [ 132, 144, -1/4 ], [ 132, 151, 1/4 ], 
  [ 132, 157, 1/8*E(8)+1/8*E(8)^3 ], [ 132, 165, -1/8*E(8)-1/8*E(8)^3 ], [ 132, 168, 1/4*E(8)+1/4*E(8)^3 ],
  [ 132, 177, 1/4*E(8)+1/4*E(8)^3 ], [ 132, 189, 1/4 ], [ 133, 3, 1/4 ], [ 133, 121, -1/4 ], 
  [ 133, 138, 1/8*E(8)+1/8*E(8)^3 ], [ 133, 146, 1/8*E(8)+1/8*E(8)^3 ], [ 133, 150, -1/8*E(8)-1/8*E(8)^3 ],
  [ 133, 158, -1/8*E(8)-1/8*E(8)^3 ], [ 134, 127, 1/4 ], [ 134, 134, 1/4 ], [ 134, 135, 3/4 ], 
  [ 134, 142, 3/4 ], [ 134, 161, -1/4*E(8)-1/4*E(8)^3 ], [ 134, 170, -3/4*E(8)-3/4*E(8)^3 ], 
  [ 135, 127, 1/4 ], [ 135, 134, 1/4 ], [ 135, 135, -1/4 ], [ 135, 142, -1/4 ], 
  [ 135, 161, 1/4*E(8)+1/4*E(8)^3 ], [ 135, 170, -1/4*E(8)-1/4*E(8)^3 ], [ 136, 139, -1 ], [ 136, 147, 1 ],
  [ 136, 152, -1/4*E(8)-1/4*E(8)^3 ], [ 136, 160, 1/4*E(8)+1/4*E(8)^3 ], [ 136, 164, 1/4*E(8)+1/4*E(8)^3 ],
  [ 136, 171, -1/4*E(8)-1/4*E(8)^3 ], [ 137, 18, -1 ], [ 137, 26, 1 ], [ 137, 30, 1 ], [ 137, 38, -1 ], 
  [ 138, 3, 1/4*E(8)+1/4*E(8)^3 ], [ 138, 17, -1 ], [ 138, 121, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 138, 125, 1/2*E(8)+1/2*E(8)^3 ], [ 138, 133, 1/2*E(8)+1/2*E(8)^3 ], [ 138, 138, 1/4 ], 
  [ 138, 146, 1/4 ], [ 138, 150, 1/4 ], [ 138, 158, 1/4 ], [ 139, 124, 1/4 ], [ 139, 136, -1/4 ], 
  [ 139, 152, 1/8*E(8)+1/8*E(8)^3 ], [ 139, 160, 1/8*E(8)+1/8*E(8)^3 ], [ 139, 164, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 139, 171, 1/8*E(8)+1/8*E(8)^3 ], [ 140, 132, -1/4 ], [ 140, 140, 1/4 ], [ 140, 144, 1/4 ], 
  [ 140, 151, -1/4 ], [ 140, 157, -1/8*E(8)-1/8*E(8)^3 ], [ 140, 165, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 140, 168, 1/4*E(8)+1/4*E(8)^3 ], [ 140, 177, 1/4*E(8)+1/4*E(8)^3 ], [ 140, 189, 1/4 ], 
  [ 141, 154, 1/8*E(8)+1/8*E(8)^3 ], [ 141, 162, 3/8*E(8)+3/8*E(8)^3 ], [ 141, 166, -1/8*E(8)-1/8*E(8)^3 ],
  [ 141, 174, -3/8*E(8)-3/8*E(8)^3 ], [ 142, 127, 1/4 ], [ 142, 134, 1/4 ], [ 142, 135, -1/4 ], 
  [ 142, 142, -1/4 ], [ 142, 161, -1/4*E(8)-1/4*E(8)^3 ], [ 142, 170, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 143, 9, 1/2*E(8)+1/2*E(8)^3 ], [ 143, 10, -2 ], [ 143, 23, 2 ], [ 143, 129, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 143, 241, -1/2*E(8)-1/2*E(8)^3 ], [ 143, 243, -1/2*E(8)-1/2*E(8)^3 ], [ 143, 244, 1/2*E(8)+1/2*E(8)^3 ]
    , [ 144, 132, -1/4 ], [ 144, 140, 1/4 ], [ 144, 144, 1/4 ], [ 144, 151, -1/4 ], 
  [ 144, 157, 1/8*E(8)+1/8*E(8)^3 ], [ 144, 165, -1/8*E(8)-1/8*E(8)^3 ], [ 144, 168, -1/4*E(8)-1/4*E(8)^3 ]
    , [ 144, 177, -1/4*E(8)-1/4*E(8)^3 ], [ 144, 189, 1/4 ], [ 145, 6, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 145, 25, 1/2 ], [ 145, 33, -1/2 ], [ 145, 126, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 145, 245, -1/4*E(8)-1/4*E(8)^3 ], [ 145, 246, 1/2*E(8)+1/2*E(8)^3 ], [ 145, 247, -1/4*E(8)-1/4*E(8)^3 ]
    , [ 146, 3, -1/4*E(8)-1/4*E(8)^3 ], [ 146, 17, 1 ], [ 146, 121, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 146, 125, 1/2*E(8)+1/2*E(8)^3 ], [ 146, 133, 1/2*E(8)+1/2*E(8)^3 ], [ 146, 138, 1/4 ], 
  [ 146, 146, 1/4 ], [ 146, 150, 1/4 ], [ 146, 158, 1/4 ], [ 147, 124, -1/4 ], [ 147, 136, 1/4 ], 
  [ 147, 152, 1/8*E(8)+1/8*E(8)^3 ], [ 147, 160, 1/8*E(8)+1/8*E(8)^3 ], [ 147, 164, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 147, 171, 1/8*E(8)+1/8*E(8)^3 ], [ 148, 175, 1/8*E(8)+1/8*E(8)^3 ], [ 148, 181, -1/8*E(8)-1/8*E(8)^3 ],
  [ 148, 182, 3/8*E(8)+3/8*E(8)^3 ], [ 148, 184, 1/8*E(8)+1/8*E(8)^3 ], [ 148, 188, -3/8*E(8)-3/8*E(8)^3 ],
  [ 148, 191, -1/8*E(8)-1/8*E(8)^3 ], [ 148, 192, 3/8*E(8)+3/8*E(8)^3 ], [ 148, 195, 1/8 ], 
  [ 148, 198, -3/8*E(8)-3/8*E(8)^3 ], [ 148, 200, -1/8 ], [ 148, 201, 3/8 ], [ 148, 206, -3/8 ], 
  [ 149, 154, 1/8*E(8)+1/8*E(8)^3 ], [ 149, 162, -1/8*E(8)-1/8*E(8)^3 ], [ 149, 166, -1/8*E(8)-1/8*E(8)^3 ]
    , [ 149, 174, 1/8*E(8)+1/8*E(8)^3 ], [ 150, 3, 1/4*E(8)+1/4*E(8)^3 ], [ 150, 17, 1 ], 
  [ 150, 121, 1/4*E(8)+1/4*E(8)^3 ], [ 150, 125, -1/2*E(8)-1/2*E(8)^3 ], [ 150, 133, -1/2*E(8)-1/2*E(8)^3 ]
    , [ 150, 138, 1/4 ], [ 150, 146, 1/4 ], [ 150, 150, 1/4 ], [ 150, 158, 1/4 ], [ 151, 132, 1/4 ], 
  [ 151, 140, -1/4 ], [ 151, 144, -1/4 ], [ 151, 151, 1/4 ], [ 151, 157, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 151, 165, 1/8*E(8)+1/8*E(8)^3 ], [ 151, 168, -1/4*E(8)-1/4*E(8)^3 ], [ 151, 177, -1/4*E(8)-1/4*E(8)^3 ]
    , [ 151, 189, 1/4 ], [ 152, 2, -1 ], [ 152, 124, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 152, 136, -1/4*E(8)-1/4*E(8)^3 ], [ 152, 139, 1/2*E(8)+1/2*E(8)^3 ], [ 152, 147, 1/2*E(8)+1/2*E(8)^3 ],
  [ 152, 152, 1/4 ], [ 152, 160, 1/4 ], [ 152, 164, -1/4 ], [ 152, 171, -1/4 ], 
  [ 153, 6, 1/4*E(8)+1/4*E(8)^3 ], [ 153, 25, -1/2 ], [ 153, 33, 1/2 ], [ 153, 126, -1/4*E(8)-1/4*E(8)^3 ],
  [ 153, 245, -1/4*E(8)-1/4*E(8)^3 ], [ 153, 246, 1/2*E(8)+1/2*E(8)^3 ], [ 153, 247, -1/4*E(8)-1/4*E(8)^3 ]
    , [ 154, 141, 1/2*E(8)+1/2*E(8)^3 ], [ 154, 149, 3/2*E(8)+3/2*E(8)^3 ], [ 154, 154, 1/4 ], 
  [ 154, 162, 3/4 ], [ 154, 166, 1/4 ], [ 154, 174, 3/4 ], [ 155, 169, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 155, 176, 3/8*E(8)+3/8*E(8)^3 ], [ 155, 178, 1/8*E(8)+1/8*E(8)^3 ], [ 155, 185, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 156, 175, 1/8*E(8)+1/8*E(8)^3 ], [ 156, 181, -1/8*E(8)-1/8*E(8)^3 ], [ 156, 182, -1/8*E(8)-1/8*E(8)^3 ]
    , [ 156, 184, 1/8*E(8)+1/8*E(8)^3 ], [ 156, 188, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 156, 191, -1/8*E(8)-1/8*E(8)^3 ], [ 156, 192, -1/8*E(8)-1/8*E(8)^3 ], [ 156, 195, 1/8 ], 
  [ 156, 198, 1/8*E(8)+1/8*E(8)^3 ], [ 156, 200, -1/8 ], [ 156, 201, -1/8 ], [ 156, 206, 1/8 ], 
  [ 157, 132, 1/2*E(8)+1/2*E(8)^3 ], [ 157, 140, -1/2*E(8)-1/2*E(8)^3 ], [ 157, 144, 1/2*E(8)+1/2*E(8)^3 ],
  [ 157, 151, -1/2*E(8)-1/2*E(8)^3 ], [ 157, 168, -1 ], [ 157, 177, 1 ], [ 158, 3, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 158, 17, -1 ], [ 158, 121, -1/4*E(8)-1/4*E(8)^3 ], [ 158, 125, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 158, 133, -1/2*E(8)-1/2*E(8)^3 ], [ 158, 138, 1/4 ], [ 158, 146, 1/4 ], [ 158, 150, 1/4 ], 
  [ 158, 158, 1/4 ], [ 159, 175, -1/8*E(8)-1/8*E(8)^3 ], [ 159, 181, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 159, 182, -3/8*E(8)-3/8*E(8)^3 ], [ 159, 184, -1/8*E(8)-1/8*E(8)^3 ], [ 159, 188, 3/8*E(8)+3/8*E(8)^3 ]
    , [ 159, 191, 1/8*E(8)+1/8*E(8)^3 ], [ 159, 192, -3/8*E(8)-3/8*E(8)^3 ], [ 159, 195, 1/8 ], 
  [ 159, 198, 3/8*E(8)+3/8*E(8)^3 ], [ 159, 200, -1/8 ], [ 159, 201, 3/8 ], [ 159, 206, -3/8 ], 
  [ 160, 2, 1 ], [ 160, 124, 1/4*E(8)+1/4*E(8)^3 ], [ 160, 136, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 160, 139, 1/2*E(8)+1/2*E(8)^3 ], [ 160, 147, 1/2*E(8)+1/2*E(8)^3 ], [ 160, 152, 1/4 ], 
  [ 160, 160, 1/4 ], [ 160, 164, -1/4 ], [ 160, 171, -1/4 ], [ 161, 127, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 161, 134, -1/4*E(8)-1/4*E(8)^3 ], [ 161, 135, 3/4*E(8)+3/4*E(8)^3 ], [ 161, 142, -3/4*E(8)-3/4*E(8)^3 ]
    , [ 162, 141, 1/2*E(8)+1/2*E(8)^3 ], [ 162, 149, -1/2*E(8)-1/2*E(8)^3 ], [ 162, 154, 1/4 ], 
  [ 162, 162, -1/4 ], [ 162, 166, 1/4 ], [ 162, 174, -1/4 ], [ 163, 169, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 163, 176, -1/8*E(8)-1/8*E(8)^3 ], [ 163, 178, 1/8*E(8)+1/8*E(8)^3 ], [ 163, 185, -1/8*E(8)-1/8*E(8)^3 ]
    , [ 164, 2, -1 ], [ 164, 124, 1/4*E(8)+1/4*E(8)^3 ], [ 164, 136, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 164, 139, 1/2*E(8)+1/2*E(8)^3 ], [ 164, 147, 1/2*E(8)+1/2*E(8)^3 ], [ 164, 152, -1/4 ], 
  [ 164, 160, -1/4 ], [ 164, 164, 1/4 ], [ 164, 171, 1/4 ], [ 165, 132, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 165, 140, 1/2*E(8)+1/2*E(8)^3 ], [ 165, 144, -1/2*E(8)-1/2*E(8)^3 ], [ 165, 151, 1/2*E(8)+1/2*E(8)^3 ],
  [ 165, 168, -1 ], [ 165, 177, 1 ], [ 166, 141, -1/2*E(8)-1/2*E(8)^3 ], [ 166, 149, -3/2*E(8)-3/2*E(8)^3 ]
    , [ 166, 154, 1/4 ], [ 166, 162, 3/4 ], [ 166, 166, 1/4 ], [ 166, 174, 3/4 ], 
  [ 167, 175, -1/8*E(8)-1/8*E(8)^3 ], [ 167, 181, 1/8*E(8)+1/8*E(8)^3 ], [ 167, 182, 1/8*E(8)+1/8*E(8)^3 ],
  [ 167, 184, -1/8*E(8)-1/8*E(8)^3 ], [ 167, 188, -1/8*E(8)-1/8*E(8)^3 ], [ 167, 191, 1/8*E(8)+1/8*E(8)^3 ]
    , [ 167, 192, 1/8*E(8)+1/8*E(8)^3 ], [ 167, 195, 1/8 ], [ 167, 198, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 167, 200, -1/8 ], [ 167, 201, -1/8 ], [ 167, 206, 1/8 ], [ 168, 132, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 168, 140, 1/4*E(8)+1/4*E(8)^3 ], [ 168, 144, -1/4*E(8)-1/4*E(8)^3 ], [ 168, 151, -1/4*E(8)-1/4*E(8)^3 ]
    , [ 168, 157, -1/4 ], [ 168, 165, -1/4 ], [ 169, 155, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 169, 163, 3/2*E(8)+3/2*E(8)^3 ], [ 169, 169, 1/4 ], [ 169, 176, 3/4 ], [ 169, 178, -1/4 ], 
  [ 169, 185, -3/4 ], [ 170, 127, 1/4*E(8)+1/4*E(8)^3 ], [ 170, 134, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 170, 135, -1/4*E(8)-1/4*E(8)^3 ], [ 170, 142, 1/4*E(8)+1/4*E(8)^3 ], [ 171, 2, 1 ], 
  [ 171, 124, -1/4*E(8)-1/4*E(8)^3 ], [ 171, 136, -1/4*E(8)-1/4*E(8)^3 ], [ 171, 139, 1/2*E(8)+1/2*E(8)^3 ]
    , [ 171, 147, 1/2*E(8)+1/2*E(8)^3 ], [ 171, 152, -1/4 ], [ 171, 160, -1/4 ], [ 171, 164, 1/4 ], 
  [ 171, 171, 1/4 ], [ 172, 172, 1 ], [ 173, 175, -1/2 ], [ 173, 181, 1/2 ], [ 173, 182, -3/2 ], 
  [ 173, 184, 1/2 ], [ 173, 188, 3/2 ], [ 173, 191, -1/2 ], [ 173, 192, 3/2 ], [ 173, 198, -3/2 ], 
  [ 174, 141, -1/2*E(8)-1/2*E(8)^3 ], [ 174, 149, 1/2*E(8)+1/2*E(8)^3 ], [ 174, 154, 1/4 ], 
  [ 174, 162, -1/4 ], [ 174, 166, 1/4 ], [ 174, 174, -1/4 ], [ 175, 148, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 175, 156, 3/8*E(8)+3/8*E(8)^3 ], [ 175, 159, -1/8*E(8)-1/8*E(8)^3 ], [ 175, 167, -3/8*E(8)-3/8*E(8)^3 ]
    , [ 175, 173, -1/8 ], [ 175, 175, 1/8 ], [ 175, 180, -3/8 ], [ 175, 181, 1/8 ], [ 175, 182, 3/8 ], 
  [ 175, 184, 1/8 ], [ 175, 188, 3/8 ], [ 175, 191, 1/8 ], [ 175, 192, 3/8 ], 
  [ 175, 195, -1/16*E(8)-1/16*E(8)^3 ], [ 175, 198, 3/8 ], [ 175, 200, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 175, 201, -3/16*E(8)-3/16*E(8)^3 ], [ 175, 206, -3/16*E(8)-3/16*E(8)^3 ], 
  [ 176, 155, 1/2*E(8)+1/2*E(8)^3 ], [ 176, 163, -1/2*E(8)-1/2*E(8)^3 ], [ 176, 169, 1/4 ], 
  [ 176, 176, -1/4 ], [ 176, 178, -1/4 ], [ 176, 185, 1/4 ], [ 177, 132, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 177, 140, 1/4*E(8)+1/4*E(8)^3 ], [ 177, 144, -1/4*E(8)-1/4*E(8)^3 ], [ 177, 151, -1/4*E(8)-1/4*E(8)^3 ]
    , [ 177, 157, 1/4 ], [ 177, 165, 1/4 ], [ 178, 155, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 178, 163, 3/2*E(8)+3/2*E(8)^3 ], [ 178, 169, -1/4 ], [ 178, 176, -3/4 ], [ 178, 178, 1/4 ], 
  [ 178, 185, 3/4 ], [ 179, 179, 1/4 ], [ 179, 186, -1/4 ], [ 179, 187, 3/4 ], [ 179, 193, -3/4 ], 
  [ 179, 205, 1/4*E(8)+1/4*E(8)^3 ], [ 179, 210, 3/4*E(8)+3/4*E(8)^3 ], [ 180, 175, -1/2 ], 
  [ 180, 181, 1/2 ], [ 180, 182, 1/2 ], [ 180, 184, 1/2 ], [ 180, 188, -1/2 ], [ 180, 191, -1/2 ], 
  [ 180, 192, -1/2 ], [ 180, 198, 1/2 ], [ 181, 148, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 181, 156, -3/8*E(8)-3/8*E(8)^3 ], [ 181, 159, 1/8*E(8)+1/8*E(8)^3 ], [ 181, 167, 3/8*E(8)+3/8*E(8)^3 ],
  [ 181, 173, 1/8 ], [ 181, 175, 1/8 ], [ 181, 180, 3/8 ], [ 181, 181, 1/8 ], [ 181, 182, 3/8 ], 
  [ 181, 184, 1/8 ], [ 181, 188, 3/8 ], [ 181, 191, 1/8 ], [ 181, 192, 3/8 ], 
  [ 181, 195, -1/16*E(8)-1/16*E(8)^3 ], [ 181, 198, 3/8 ], [ 181, 200, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 181, 201, -3/16*E(8)-3/16*E(8)^3 ], [ 181, 206, -3/16*E(8)-3/16*E(8)^3 ], 
  [ 182, 148, 1/8*E(8)+1/8*E(8)^3 ], [ 182, 156, -1/8*E(8)-1/8*E(8)^3 ], [ 182, 159, -1/8*E(8)-1/8*E(8)^3 ]
    , [ 182, 167, 1/8*E(8)+1/8*E(8)^3 ], [ 182, 173, -1/8 ], [ 182, 175, 1/8 ], [ 182, 180, 1/8 ], 
  [ 182, 181, 1/8 ], [ 182, 182, -1/8 ], [ 182, 184, 1/8 ], [ 182, 188, -1/8 ], [ 182, 191, 1/8 ], 
  [ 182, 192, -1/8 ], [ 182, 195, -1/16*E(8)-1/16*E(8)^3 ], [ 182, 198, -1/8 ], 
  [ 182, 200, -1/16*E(8)-1/16*E(8)^3 ], [ 182, 201, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 182, 206, 1/16*E(8)+1/16*E(8)^3 ], [ 183, 183, 1 ], [ 184, 148, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 184, 156, 3/8*E(8)+3/8*E(8)^3 ], [ 184, 159, -1/8*E(8)-1/8*E(8)^3 ], [ 184, 167, -3/8*E(8)-3/8*E(8)^3 ]
    , [ 184, 173, 1/8 ], [ 184, 175, 1/8 ], [ 184, 180, 3/8 ], [ 184, 181, 1/8 ], [ 184, 182, 3/8 ], 
  [ 184, 184, 1/8 ], [ 184, 188, 3/8 ], [ 184, 191, 1/8 ], [ 184, 192, 3/8 ], 
  [ 184, 195, 1/16*E(8)+1/16*E(8)^3 ], [ 184, 198, 3/8 ], [ 184, 200, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 184, 201, 3/16*E(8)+3/16*E(8)^3 ], [ 184, 206, 3/16*E(8)+3/16*E(8)^3 ], 
  [ 185, 155, 1/2*E(8)+1/2*E(8)^3 ], [ 185, 163, -1/2*E(8)-1/2*E(8)^3 ], [ 185, 169, -1/4 ], 
  [ 185, 176, 1/4 ], [ 185, 178, 1/4 ], [ 185, 185, -1/4 ], [ 186, 179, -1/4 ], [ 186, 186, 1/4 ], 
  [ 186, 187, -3/4 ], [ 186, 193, 3/4 ], [ 186, 205, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 186, 210, 3/4*E(8)+3/4*E(8)^3 ], [ 187, 179, 1/4 ], [ 187, 186, -1/4 ], [ 187, 187, -1/4 ], 
  [ 187, 193, 1/4 ], [ 187, 205, 1/4*E(8)+1/4*E(8)^3 ], [ 187, 210, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 188, 148, -1/8*E(8)-1/8*E(8)^3 ], [ 188, 156, 1/8*E(8)+1/8*E(8)^3 ], [ 188, 159, 1/8*E(8)+1/8*E(8)^3 ],
  [ 188, 167, -1/8*E(8)-1/8*E(8)^3 ], [ 188, 173, 1/8 ], [ 188, 175, 1/8 ], [ 188, 180, -1/8 ], 
  [ 188, 181, 1/8 ], [ 188, 182, -1/8 ], [ 188, 184, 1/8 ], [ 188, 188, -1/8 ], [ 188, 191, 1/8 ], 
  [ 188, 192, -1/8 ], [ 188, 195, -1/16*E(8)-1/16*E(8)^3 ], [ 188, 198, -1/8 ], 
  [ 188, 200, -1/16*E(8)-1/16*E(8)^3 ], [ 188, 201, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 188, 206, 1/16*E(8)+1/16*E(8)^3 ], [ 189, 132, 1 ], [ 189, 140, 1 ], [ 189, 144, 1 ], [ 189, 151, 1 ], 
  [ 190, 190, 1/4 ], [ 190, 196, -1/4 ], [ 190, 197, 3/4 ], [ 190, 203, -3/4 ], 
  [ 190, 213, 1/4*E(8)+1/4*E(8)^3 ], [ 190, 218, 3/4*E(8)+3/4*E(8)^3 ], [ 191, 148, -1/8*E(8)-1/8*E(8)^3 ],
  [ 191, 156, -3/8*E(8)-3/8*E(8)^3 ], [ 191, 159, 1/8*E(8)+1/8*E(8)^3 ], [ 191, 167, 3/8*E(8)+3/8*E(8)^3 ],
  [ 191, 173, -1/8 ], [ 191, 175, 1/8 ], [ 191, 180, -3/8 ], [ 191, 181, 1/8 ], [ 191, 182, 3/8 ], 
  [ 191, 184, 1/8 ], [ 191, 188, 3/8 ], [ 191, 191, 1/8 ], [ 191, 192, 3/8 ], 
  [ 191, 195, 1/16*E(8)+1/16*E(8)^3 ], [ 191, 198, 3/8 ], [ 191, 200, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 191, 201, 3/16*E(8)+3/16*E(8)^3 ], [ 191, 206, 3/16*E(8)+3/16*E(8)^3 ], 
  [ 192, 148, 1/8*E(8)+1/8*E(8)^3 ], [ 192, 156, -1/8*E(8)-1/8*E(8)^3 ], [ 192, 159, -1/8*E(8)-1/8*E(8)^3 ]
    , [ 192, 167, 1/8*E(8)+1/8*E(8)^3 ], [ 192, 173, 1/8 ], [ 192, 175, 1/8 ], [ 192, 180, -1/8 ], 
  [ 192, 181, 1/8 ], [ 192, 182, -1/8 ], [ 192, 184, 1/8 ], [ 192, 188, -1/8 ], [ 192, 191, 1/8 ], 
  [ 192, 192, -1/8 ], [ 192, 195, 1/16*E(8)+1/16*E(8)^3 ], [ 192, 198, -1/8 ], 
  [ 192, 200, 1/16*E(8)+1/16*E(8)^3 ], [ 192, 201, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 192, 206, -1/16*E(8)-1/16*E(8)^3 ], [ 193, 179, -1/4 ], [ 193, 186, 1/4 ], [ 193, 187, 1/4 ], 
  [ 193, 193, -1/4 ], [ 193, 205, 1/4*E(8)+1/4*E(8)^3 ], [ 193, 210, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 194, 194, -1/2 ], [ 194, 204, -1/2 ], [ 194, 211, 1/4*E(8)+1/4*E(8)^3 ], [ 195, 148, 1/2 ], 
  [ 195, 156, 3/2 ], [ 195, 159, 1/2 ], [ 195, 167, 3/2 ], [ 195, 175, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 195, 181, -1/4*E(8)-1/4*E(8)^3 ], [ 195, 182, -3/4*E(8)-3/4*E(8)^3 ], [ 195, 184, 1/4*E(8)+1/4*E(8)^3 ]
    , [ 195, 188, -3/4*E(8)-3/4*E(8)^3 ], [ 195, 191, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 195, 192, 3/4*E(8)+3/4*E(8)^3 ], [ 195, 198, 3/4*E(8)+3/4*E(8)^3 ], [ 196, 190, -1/4 ], 
  [ 196, 196, 1/4 ], [ 196, 197, -3/4 ], [ 196, 203, 3/4 ], [ 196, 213, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 196, 218, 3/4*E(8)+3/4*E(8)^3 ], [ 197, 190, 1/4 ], [ 197, 196, -1/4 ], [ 197, 197, -1/4 ], 
  [ 197, 203, 1/4 ], [ 197, 213, 1/4*E(8)+1/4*E(8)^3 ], [ 197, 218, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 198, 148, -1/8*E(8)-1/8*E(8)^3 ], [ 198, 156, 1/8*E(8)+1/8*E(8)^3 ], [ 198, 159, 1/8*E(8)+1/8*E(8)^3 ],
  [ 198, 167, -1/8*E(8)-1/8*E(8)^3 ], [ 198, 173, -1/8 ], [ 198, 175, 1/8 ], [ 198, 180, 1/8 ], 
  [ 198, 181, 1/8 ], [ 198, 182, -1/8 ], [ 198, 184, 1/8 ], [ 198, 188, -1/8 ], [ 198, 191, 1/8 ], 
  [ 198, 192, -1/8 ], [ 198, 195, 1/16*E(8)+1/16*E(8)^3 ], [ 198, 198, -1/8 ], 
  [ 198, 200, 1/16*E(8)+1/16*E(8)^3 ], [ 198, 201, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 198, 206, -1/16*E(8)-1/16*E(8)^3 ], [ 199, 215, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 199, 220, 1/2*E(8)+1/2*E(8)^3 ], [ 200, 148, -1/2 ], [ 200, 156, -3/2 ], [ 200, 159, -1/2 ], 
  [ 200, 167, -3/2 ], [ 200, 175, -1/4*E(8)-1/4*E(8)^3 ], [ 200, 181, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 200, 182, -3/4*E(8)-3/4*E(8)^3 ], [ 200, 184, 1/4*E(8)+1/4*E(8)^3 ], [ 200, 188, -3/4*E(8)-3/4*E(8)^3 ]
    , [ 200, 191, 1/4*E(8)+1/4*E(8)^3 ], [ 200, 192, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 200, 198, 3/4*E(8)+3/4*E(8)^3 ], [ 201, 148, 1/2 ], [ 201, 156, -1/2 ], [ 201, 159, 1/2 ], 
  [ 201, 167, -1/2 ], [ 201, 175, -1/4*E(8)-1/4*E(8)^3 ], [ 201, 181, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 201, 182, 1/4*E(8)+1/4*E(8)^3 ], [ 201, 184, 1/4*E(8)+1/4*E(8)^3 ], [ 201, 188, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 201, 191, 1/4*E(8)+1/4*E(8)^3 ], [ 201, 192, -1/4*E(8)-1/4*E(8)^3 ], [ 201, 198, -1/4*E(8)-1/4*E(8)^3 ]
    , [ 202, 209, -1/8*E(8)-1/8*E(8)^3 ], [ 202, 214, -3/8*E(8)-3/8*E(8)^3 ], 
  [ 202, 217, 1/8*E(8)+1/8*E(8)^3 ], [ 202, 221, 3/8*E(8)+3/8*E(8)^3 ], [ 203, 190, -1/4 ], 
  [ 203, 196, 1/4 ], [ 203, 197, 1/4 ], [ 203, 203, -1/4 ], [ 203, 213, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 203, 218, -1/4*E(8)-1/4*E(8)^3 ], [ 204, 194, -1/2 ], [ 204, 204, -1/2 ], 
  [ 204, 211, -1/4*E(8)-1/4*E(8)^3 ], [ 205, 179, 1/4*E(8)+1/4*E(8)^3 ], [ 205, 186, 1/4*E(8)+1/4*E(8)^3 ],
  [ 205, 187, 3/4*E(8)+3/4*E(8)^3 ], [ 205, 193, 3/4*E(8)+3/4*E(8)^3 ], [ 206, 148, -1/2 ], 
  [ 206, 156, 1/2 ], [ 206, 159, -1/2 ], [ 206, 167, 1/2 ], [ 206, 175, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 206, 181, -1/4*E(8)-1/4*E(8)^3 ], [ 206, 182, 1/4*E(8)+1/4*E(8)^3 ], [ 206, 184, 1/4*E(8)+1/4*E(8)^3 ],
  [ 206, 188, 1/4*E(8)+1/4*E(8)^3 ], [ 206, 191, 1/4*E(8)+1/4*E(8)^3 ], [ 206, 192, -1/4*E(8)-1/4*E(8)^3 ],
  [ 206, 198, -1/4*E(8)-1/4*E(8)^3 ], [ 207, 209, -1/8*E(8)-1/8*E(8)^3 ], [ 207, 214, 1/8*E(8)+1/8*E(8)^3 ]
    , [ 207, 217, 1/8*E(8)+1/8*E(8)^3 ], [ 207, 221, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 208, 222, -1/2*E(8)-1/2*E(8)^3 ], [ 208, 225, 1/2*E(8)+1/2*E(8)^3 ], [ 209, 202, -1/2*E(8)-1/2*E(8)^3 ]
    , [ 209, 207, -3/2*E(8)-3/2*E(8)^3 ], [ 209, 209, 1/4 ], [ 209, 214, 3/4 ], [ 209, 217, 1/4 ], 
  [ 209, 221, 3/4 ], [ 210, 179, 1/4*E(8)+1/4*E(8)^3 ], [ 210, 186, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 210, 187, -1/4*E(8)-1/4*E(8)^3 ], [ 210, 193, -1/4*E(8)-1/4*E(8)^3 ], [ 211, 194, E(8)+E(8)^3 ], 
  [ 211, 204, -E(8)-E(8)^3 ], [ 212, 219, 1/8*E(8)+1/8*E(8)^3 ], [ 212, 223, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 212, 224, -1/8*E(8)-1/8*E(8)^3 ], [ 212, 227, 1/8*E(8)+1/8*E(8)^3 ], [ 212, 232, 1/4 ], 
  [ 212, 235, 1/4 ], [ 213, 190, 1/4*E(8)+1/4*E(8)^3 ], [ 213, 196, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 213, 197, 3/4*E(8)+3/4*E(8)^3 ], [ 213, 203, 3/4*E(8)+3/4*E(8)^3 ], [ 214, 202, -1/2*E(8)-1/2*E(8)^3 ],
  [ 214, 207, 1/2*E(8)+1/2*E(8)^3 ], [ 214, 209, 1/4 ], [ 214, 214, -1/4 ], [ 214, 217, 1/4 ], 
  [ 214, 221, -1/4 ], [ 215, 199, -1/2*E(8)-1/2*E(8)^3 ], [ 215, 215, -1/2 ], [ 215, 220, -1/2 ], 
  [ 216, 219, -1/8*E(8)-1/8*E(8)^3 ], [ 216, 223, 1/8*E(8)+1/8*E(8)^3 ], [ 216, 224, 1/8*E(8)+1/8*E(8)^3 ],
  [ 216, 227, -1/8*E(8)-1/8*E(8)^3 ], [ 216, 232, 1/4 ], [ 216, 235, 1/4 ], 
  [ 217, 202, 1/2*E(8)+1/2*E(8)^3 ], [ 217, 207, 3/2*E(8)+3/2*E(8)^3 ], [ 217, 209, 1/4 ], 
  [ 217, 214, 3/4 ], [ 217, 217, 1/4 ], [ 217, 221, 3/4 ], [ 218, 190, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 218, 196, 1/4*E(8)+1/4*E(8)^3 ], [ 218, 197, -1/4*E(8)-1/4*E(8)^3 ], [ 218, 203, -1/4*E(8)-1/4*E(8)^3 ]
    , [ 219, 212, 1/2*E(8)+1/2*E(8)^3 ], [ 219, 216, -1/2*E(8)-1/2*E(8)^3 ], [ 219, 219, -1/4 ], 
  [ 219, 223, 1/4 ], [ 219, 224, -1/4 ], [ 219, 227, 1/4 ], [ 219, 228, 1 ], 
  [ 219, 232, 1/4*E(8)+1/4*E(8)^3 ], [ 219, 235, -1/4*E(8)-1/4*E(8)^3 ], [ 220, 199, 1/2*E(8)+1/2*E(8)^3 ],
  [ 220, 215, -1/2 ], [ 220, 220, -1/2 ], [ 221, 202, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 221, 207, -1/2*E(8)-1/2*E(8)^3 ], [ 221, 209, 1/4 ], [ 221, 214, -1/4 ], [ 221, 217, 1/4 ], 
  [ 221, 221, -1/4 ], [ 222, 208, -1/2*E(8)-1/2*E(8)^3 ], [ 222, 222, -1/2 ], [ 222, 225, -1/2 ], 
  [ 223, 212, -1/2*E(8)-1/2*E(8)^3 ], [ 223, 216, 1/2*E(8)+1/2*E(8)^3 ], [ 223, 219, 1/4 ], 
  [ 223, 223, -1/4 ], [ 223, 224, 1/4 ], [ 223, 227, -1/4 ], [ 223, 228, 1 ], 
  [ 223, 232, 1/4*E(8)+1/4*E(8)^3 ], [ 223, 235, -1/4*E(8)-1/4*E(8)^3 ], [ 224, 212, -1/2*E(8)-1/2*E(8)^3 ]
    , [ 224, 216, 1/2*E(8)+1/2*E(8)^3 ], [ 224, 219, -1/4 ], [ 224, 223, 1/4 ], [ 224, 224, -1/4 ], 
  [ 224, 227, 1/4 ], [ 224, 228, -1 ], [ 224, 232, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 224, 235, -1/4*E(8)-1/4*E(8)^3 ], [ 225, 208, 1/2*E(8)+1/2*E(8)^3 ], [ 225, 222, -1/2 ], 
  [ 225, 225, -1/2 ], [ 226, 226, -1/2 ], [ 226, 229, 1/2 ], [ 226, 233, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 227, 212, 1/2*E(8)+1/2*E(8)^3 ], [ 227, 216, -1/2*E(8)-1/2*E(8)^3 ], [ 227, 219, 1/4 ], 
  [ 227, 223, -1/4 ], [ 227, 224, 1/4 ], [ 227, 227, -1/4 ], [ 227, 228, -1 ], 
  [ 227, 232, 1/4*E(8)+1/4*E(8)^3 ], [ 227, 235, -1/4*E(8)-1/4*E(8)^3 ], [ 228, 219, 1/4 ], 
  [ 228, 223, 1/4 ], [ 228, 224, -1/4 ], [ 228, 227, -1/4 ], [ 229, 226, 1/2 ], [ 229, 229, -1/2 ], 
  [ 229, 233, 1/4*E(8)+1/4*E(8)^3 ], [ 230, 230, -1/2 ], [ 230, 234, -1/2 ], 
  [ 230, 236, -1/4*E(8)-1/4*E(8)^3 ], [ 231, 237, 1/2*E(8)+1/2*E(8)^3 ], [ 231, 238, 1/2*E(8)+1/2*E(8)^3 ],
  [ 232, 212, 1 ], [ 232, 216, 1 ], [ 232, 219, 1/4*E(8)+1/4*E(8)^3 ], [ 232, 223, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 232, 224, 1/4*E(8)+1/4*E(8)^3 ], [ 232, 227, 1/4*E(8)+1/4*E(8)^3 ], [ 233, 226, E(8)+E(8)^3 ], 
  [ 233, 229, E(8)+E(8)^3 ], [ 234, 230, -1/2 ], [ 234, 234, -1/2 ], [ 234, 236, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 235, 212, 1 ], [ 235, 216, 1 ], [ 235, 219, -1/4*E(8)-1/4*E(8)^3 ], [ 235, 223, -1/4*E(8)-1/4*E(8)^3 ],
  [ 235, 224, -1/4*E(8)-1/4*E(8)^3 ], [ 235, 227, -1/4*E(8)-1/4*E(8)^3 ], [ 236, 230, -E(8)-E(8)^3 ], 
  [ 236, 234, E(8)+E(8)^3 ], [ 237, 231, 1/2*E(8)+1/2*E(8)^3 ], [ 237, 237, -1/2 ], [ 237, 238, 1/2 ], 
  [ 238, 231, 1/2*E(8)+1/2*E(8)^3 ], [ 238, 237, 1/2 ], [ 238, 238, -1/2 ], [ 239, 239, -1/2 ], 
  [ 239, 240, 3/2 ], [ 240, 239, 1/2 ], [ 240, 240, 1/2 ], [ 241, 9, -1/4 ], 
  [ 241, 10, 1/2*E(8)+1/2*E(8)^3 ], [ 241, 23, 1/2*E(8)+1/2*E(8)^3 ], [ 241, 129, -1/4 ], 
  [ 241, 130, -1/8*E(8)-1/8*E(8)^3 ], [ 241, 143, -1/8*E(8)-1/8*E(8)^3 ], [ 241, 241, 1/4 ], 
  [ 241, 242, -1/2 ], [ 241, 243, -1/4 ], [ 241, 244, 1/4 ], [ 241, 245, 1/2 ], [ 242, 6, 1/2 ], 
  [ 242, 9, -1/2 ], [ 242, 126, 1/2 ], [ 242, 129, -1/2 ], [ 242, 241, 1/2 ], [ 242, 242, -1 ], 
  [ 242, 243, -1/2 ], [ 242, 244, 1/2 ], [ 242, 245, 1/2 ], [ 242, 247, 1/2 ], [ 243, 6, 1/2 ], 
  [ 243, 9, -1/4 ], [ 243, 10, 1/2*E(8)+1/2*E(8)^3 ], [ 243, 23, 1/2*E(8)+1/2*E(8)^3 ], [ 243, 126, 1/2 ], 
  [ 243, 129, -1/4 ], [ 243, 130, -1/8*E(8)-1/8*E(8)^3 ], [ 243, 143, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 243, 241, 1/4 ], [ 243, 242, -3/2 ], [ 243, 243, -1/4 ], [ 243, 244, 5/4 ], [ 243, 247, 1/2 ], 
  [ 244, 6, 1/2 ], [ 244, 9, -1/2 ], [ 244, 126, 1/2 ], [ 244, 129, -1/2 ], [ 244, 241, 1/2 ], 
  [ 244, 242, -2 ], [ 244, 243, -1/2 ], [ 244, 244, 3/2 ], [ 244, 245, 1/2 ], [ 244, 247, 1/2 ], 
  [ 245, 6, 1/2 ], [ 245, 126, 1/2 ], [ 245, 241, 1 ], [ 245, 242, -1 ], [ 245, 243, -1 ], [ 245, 244, 1 ],
  [ 245, 245, 1/2 ], [ 245, 247, 1/2 ], [ 246, 6, 1/4 ], [ 246, 25, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 246, 33, -1/4*E(8)-1/4*E(8)^3 ], [ 246, 126, 1/4 ], [ 246, 145, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 246, 153, 1/4*E(8)+1/4*E(8)^3 ], [ 246, 241, 1/2 ], [ 246, 242, -1/2 ], [ 246, 243, -1/2 ], 
  [ 246, 244, 1/2 ], [ 246, 245, 1/4 ], [ 246, 247, 3/4 ], [ 247, 247, 1 ], [ 248, 8, -1/4 ], 
  [ 248, 128, -3/4 ], [ 248, 247, 3/4 ], [ 248, 248, -1/2 ] ],
  
[ 248, [ 1, 12, 1/8*E(8)+1/8*E(8)^3 ], [ 1, 20, -1/8*E(8)-1/8*E(8)^3 ], [ 1, 24, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 1, 31, -1/8*E(8)-1/8*E(8)^3 ], [ 1, 48, 1/4 ], [ 1, 57, -1/4 ], [ 2, 152, -1/4 ], [ 2, 160, 1/4 ], 
  [ 2, 164, -1/4 ], [ 2, 171, 1/4 ], [ 3, 132, 1/2*E(8)+1/2*E(8)^3 ], [ 3, 140, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 3, 144, 1/2*E(8)+1/2*E(8)^3 ], [ 3, 151, -1/2*E(8)-1/2*E(8)^3 ], [ 3, 168, 1 ], [ 3, 177, -1 ], 
  [ 4, 6, -1/4*E(8)-1/4*E(8)^3 ], [ 4, 25, -1/2 ], [ 4, 33, 1/2 ], [ 4, 126, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 4, 245, 1/4*E(8)+1/4*E(8)^3 ], [ 4, 246, -1/2*E(8)-1/2*E(8)^3 ], [ 4, 247, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 5, 12, 1/4*E(8)+1/4*E(8)^3 ], [ 5, 20, 1/4*E(8)+1/4*E(8)^3 ], [ 5, 24, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 5, 31, -1/4*E(8)-1/4*E(8)^3 ], [ 5, 37, -1 ], [ 5, 45, -1 ], [ 6, 9, -1/4 ], 
  [ 6, 10, 1/2*E(8)+1/2*E(8)^3 ], [ 6, 23, 1/2*E(8)+1/2*E(8)^3 ], [ 6, 129, -1/4 ], 
  [ 6, 130, 1/8*E(8)+1/8*E(8)^3 ], [ 6, 143, 1/8*E(8)+1/8*E(8)^3 ], [ 6, 241, 1/4 ], [ 6, 242, 1/2 ], 
  [ 6, 243, -1/4 ], [ 6, 244, 1/4 ], [ 6, 245, -1/2 ], [ 7, 35, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 7, 43, -1/8*E(8)-1/8*E(8)^3 ], [ 7, 49, -1/4 ], [ 7, 56, -1/4 ], [ 7, 58, 1/4 ], [ 7, 65, 1/4 ], 
  [ 8, 8, 1/4 ], [ 8, 128, -9/4 ], [ 8, 247, -3/4 ], [ 8, 248, 3/2 ], [ 9, 6, -1/4 ], 
  [ 9, 25, 1/4*E(8)+1/4*E(8)^3 ], [ 9, 33, 1/4*E(8)+1/4*E(8)^3 ], [ 9, 126, -1/4 ], 
  [ 9, 145, 1/4*E(8)+1/4*E(8)^3 ], [ 9, 153, 1/4*E(8)+1/4*E(8)^3 ], [ 9, 241, 1/2 ], [ 9, 242, -1/2 ], 
  [ 9, 243, -1/2 ], [ 9, 244, 1/2 ], [ 9, 245, -1/4 ], [ 9, 247, 1/4 ], [ 10, 124, 1/4 ], 
  [ 10, 136, -1/4 ], [ 10, 152, 1/8*E(8)+1/8*E(8)^3 ], [ 10, 160, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 10, 164, 1/8*E(8)+1/8*E(8)^3 ], [ 10, 171, 1/8*E(8)+1/8*E(8)^3 ], [ 11, 132, 1 ], [ 11, 140, 1 ], 
  [ 11, 144, 1 ], [ 11, 151, 1 ], [ 12, 12, 1/4 ], [ 12, 20, -1/4 ], [ 12, 24, -1/4 ], [ 12, 31, 1/4 ], 
  [ 12, 37, 1/2*E(8)+1/2*E(8)^3 ], [ 12, 45, -1/2*E(8)-1/2*E(8)^3 ], [ 12, 48, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 12, 57, 1/4*E(8)+1/4*E(8)^3 ], [ 12, 69, 1 ], [ 13, 12, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 13, 20, -1/4*E(8)-1/4*E(8)^3 ], [ 13, 24, 1/4*E(8)+1/4*E(8)^3 ], [ 13, 31, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 13, 37, -1 ], [ 13, 45, -1 ], [ 14, 35, -1/8*E(8)-1/8*E(8)^3 ], [ 14, 43, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 14, 49, 1/4 ], [ 14, 56, 1/4 ], [ 14, 58, -1/4 ], [ 14, 65, -1/4 ], [ 15, 35, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 15, 43, -1/8*E(8)-1/8*E(8)^3 ], [ 15, 49, 3/4 ], [ 15, 56, -1/4 ], [ 15, 58, -3/4 ], [ 15, 65, 1/4 ], 
  [ 16, 6, 1/4*E(8)+1/4*E(8)^3 ], [ 16, 25, -1/2 ], [ 16, 33, 1/2 ], [ 16, 126, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 16, 245, -1/4*E(8)-1/4*E(8)^3 ], [ 16, 246, 1/2*E(8)+1/2*E(8)^3 ], [ 16, 247, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 17, 183, 1 ], [ 18, 1, 1/4*E(8)+1/4*E(8)^3 ], [ 18, 5, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 18, 13, 1/8*E(8)+1/8*E(8)^3 ], [ 18, 18, 1/4 ], [ 18, 26, 1/4 ], [ 18, 30, 1/4 ], [ 18, 38, 1/4 ], 
  [ 18, 123, 1/4*E(8)+1/4*E(8)^3 ], [ 18, 137, -1/4 ], [ 19, 9, 1/2*E(8)+1/2*E(8)^3 ], [ 19, 10, 2 ], 
  [ 19, 23, -2 ], [ 19, 129, -1/2*E(8)-1/2*E(8)^3 ], [ 19, 241, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 19, 243, -1/2*E(8)-1/2*E(8)^3 ], [ 19, 244, 1/2*E(8)+1/2*E(8)^3 ], [ 20, 12, -1/4 ], [ 20, 20, 1/4 ], 
  [ 20, 24, 1/4 ], [ 20, 31, -1/4 ], [ 20, 37, 1/2*E(8)+1/2*E(8)^3 ], [ 20, 45, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 20, 48, -1/4*E(8)-1/4*E(8)^3 ], [ 20, 57, -1/4*E(8)-1/4*E(8)^3 ], [ 20, 69, 1 ], 
  [ 21, 70, -1/4*E(8)-1/4*E(8)^3 ], [ 21, 76, -1/4*E(8)-1/4*E(8)^3 ], [ 21, 77, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 21, 83, -1/4*E(8)-1/4*E(8)^3 ], [ 22, 35, 3/8*E(8)+3/8*E(8)^3 ], [ 22, 43, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 22, 49, -3/4 ], [ 22, 56, 1/4 ], [ 22, 58, 3/4 ], [ 22, 65, -1/4 ], [ 23, 124, 1/4 ], 
  [ 23, 136, -1/4 ], [ 23, 152, -1/8*E(8)-1/8*E(8)^3 ], [ 23, 160, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 23, 164, -1/8*E(8)-1/8*E(8)^3 ], [ 23, 171, -1/8*E(8)-1/8*E(8)^3 ], [ 24, 12, -1/4 ], [ 24, 20, 1/4 ], 
  [ 24, 24, 1/4 ], [ 24, 31, -1/4 ], [ 24, 37, -1/2*E(8)-1/2*E(8)^3 ], [ 24, 45, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 24, 48, 1/4*E(8)+1/4*E(8)^3 ], [ 24, 57, 1/4*E(8)+1/4*E(8)^3 ], [ 24, 69, 1 ], [ 25, 139, -1 ], 
  [ 25, 147, 1 ], [ 25, 152, 1/4*E(8)+1/4*E(8)^3 ], [ 25, 160, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 25, 164, -1/4*E(8)-1/4*E(8)^3 ], [ 25, 171, 1/4*E(8)+1/4*E(8)^3 ], [ 26, 1, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 26, 5, -1/8*E(8)-1/8*E(8)^3 ], [ 26, 13, -1/8*E(8)-1/8*E(8)^3 ], [ 26, 18, 1/4 ], [ 26, 26, 1/4 ], 
  [ 26, 30, 1/4 ], [ 26, 38, 1/4 ], [ 26, 123, 1/4*E(8)+1/4*E(8)^3 ], [ 26, 137, 1/4 ], 
  [ 27, 9, -1/2*E(8)-1/2*E(8)^3 ], [ 27, 10, 2 ], [ 27, 23, -2 ], [ 27, 129, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 27, 241, 1/2*E(8)+1/2*E(8)^3 ], [ 27, 243, 1/2*E(8)+1/2*E(8)^3 ], [ 27, 244, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 28, 70, -1/4 ], [ 28, 76, 1/4 ], [ 28, 77, -1/4 ], [ 28, 83, 1/4 ], [ 28, 93, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 28, 98, -1/4*E(8)-1/4*E(8)^3 ], [ 29, 70, 3/4*E(8)+3/4*E(8)^3 ], [ 29, 76, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 29, 77, -1/4*E(8)-1/4*E(8)^3 ], [ 29, 83, -1/4*E(8)-1/4*E(8)^3 ], [ 30, 1, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 30, 5, 1/8*E(8)+1/8*E(8)^3 ], [ 30, 13, 1/8*E(8)+1/8*E(8)^3 ], [ 30, 18, 1/4 ], [ 30, 26, 1/4 ], 
  [ 30, 30, 1/4 ], [ 30, 38, 1/4 ], [ 30, 123, -1/4*E(8)-1/4*E(8)^3 ], [ 30, 137, 1/4 ], [ 31, 12, 1/4 ], 
  [ 31, 20, -1/4 ], [ 31, 24, -1/4 ], [ 31, 31, 1/4 ], [ 31, 37, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 31, 45, 1/2*E(8)+1/2*E(8)^3 ], [ 31, 48, -1/4*E(8)-1/4*E(8)^3 ], [ 31, 57, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 31, 69, 1 ], [ 32, 2, -1 ], [ 32, 124, -1/4*E(8)-1/4*E(8)^3 ], [ 32, 136, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 32, 139, 1/2*E(8)+1/2*E(8)^3 ], [ 32, 147, 1/2*E(8)+1/2*E(8)^3 ], [ 32, 152, 1/4 ], [ 32, 160, 1/4 ], 
  [ 32, 164, -1/4 ], [ 32, 171, -1/4 ], [ 33, 139, -1 ], [ 33, 147, 1 ], [ 33, 152, -1/4*E(8)-1/4*E(8)^3 ],
  [ 33, 160, 1/4*E(8)+1/4*E(8)^3 ], [ 33, 164, 1/4*E(8)+1/4*E(8)^3 ], [ 33, 171, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 34, 28, 1/8 ], [ 34, 36, 1/8 ], [ 34, 39, 1/8 ], [ 34, 47, 1/8 ], [ 34, 55, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 34, 61, -1/16*E(8)-1/16*E(8)^3 ], [ 34, 62, -1/16*E(8)-1/16*E(8)^3 ], [ 34, 64, 1/16*E(8)+1/16*E(8)^3 ]
    , [ 34, 68, -1/16*E(8)-1/16*E(8)^3 ], [ 34, 71, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 34, 72, 1/16*E(8)+1/16*E(8)^3 ], [ 34, 78, 1/16*E(8)+1/16*E(8)^3 ], [ 35, 49, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 35, 56, -1/2*E(8)-1/2*E(8)^3 ], [ 35, 58, -1/2*E(8)-1/2*E(8)^3 ], [ 35, 65, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 36, 70, 3/4 ], [ 36, 76, -3/4 ], [ 36, 77, -1/4 ], [ 36, 83, 1/4 ], [ 36, 93, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 36, 98, -1/4*E(8)-1/4*E(8)^3 ], [ 37, 5, -1/4 ], [ 37, 13, 1/4 ], [ 37, 18, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 37, 26, -1/4*E(8)-1/4*E(8)^3 ], [ 37, 30, 1/4*E(8)+1/4*E(8)^3 ], [ 37, 38, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 38, 1, -1/4*E(8)-1/4*E(8)^3 ], [ 38, 5, -1/8*E(8)-1/8*E(8)^3 ], [ 38, 13, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 38, 18, 1/4 ], [ 38, 26, 1/4 ], [ 38, 30, 1/4 ], [ 38, 38, 1/4 ], [ 38, 123, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 38, 137, -1/4 ], [ 39, 70, 1/4 ], [ 39, 76, -1/4 ], [ 39, 77, 1/4 ], [ 39, 83, -1/4 ], 
  [ 39, 93, -1/4*E(8)-1/4*E(8)^3 ], [ 39, 98, -1/4*E(8)-1/4*E(8)^3 ], [ 40, 2, -1 ], 
  [ 40, 124, 1/4*E(8)+1/4*E(8)^3 ], [ 40, 136, 1/4*E(8)+1/4*E(8)^3 ], [ 40, 139, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 40, 147, 1/2*E(8)+1/2*E(8)^3 ], [ 40, 152, -1/4 ], [ 40, 160, -1/4 ], [ 40, 164, 1/4 ], 
  [ 40, 171, 1/4 ], [ 41, 7, -1/4*E(8)-1/4*E(8)^3 ], [ 41, 14, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 41, 15, -1/4*E(8)-1/4*E(8)^3 ], [ 41, 22, 1/4*E(8)+1/4*E(8)^3 ], [ 42, 28, -3/8 ], [ 42, 36, 1/8 ], 
  [ 42, 39, -3/8 ], [ 42, 47, 1/8 ], [ 42, 55, 3/16*E(8)+3/16*E(8)^3 ], [ 42, 61, 3/16*E(8)+3/16*E(8)^3 ], 
  [ 42, 62, -1/16*E(8)-1/16*E(8)^3 ], [ 42, 64, -3/16*E(8)-3/16*E(8)^3 ], 
  [ 42, 68, -1/16*E(8)-1/16*E(8)^3 ], [ 42, 71, -3/16*E(8)-3/16*E(8)^3 ], [ 42, 72, 1/16*E(8)+1/16*E(8)^3 ]
    , [ 42, 78, 1/16*E(8)+1/16*E(8)^3 ], [ 43, 49, 3/2*E(8)+3/2*E(8)^3 ], [ 43, 56, -1/2*E(8)-1/2*E(8)^3 ],
  [ 43, 58, 3/2*E(8)+3/2*E(8)^3 ], [ 43, 65, -1/2*E(8)-1/2*E(8)^3 ], [ 44, 2, 1 ], 
  [ 44, 124, 1/4*E(8)+1/4*E(8)^3 ], [ 44, 136, 1/4*E(8)+1/4*E(8)^3 ], [ 44, 139, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 44, 147, 1/2*E(8)+1/2*E(8)^3 ], [ 44, 152, 1/4 ], [ 44, 160, 1/4 ], [ 44, 164, -1/4 ], 
  [ 44, 171, -1/4 ], [ 45, 5, 1/4 ], [ 45, 13, -1/4 ], [ 45, 18, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 45, 26, -1/4*E(8)-1/4*E(8)^3 ], [ 45, 30, 1/4*E(8)+1/4*E(8)^3 ], [ 45, 38, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 46, 28, -1/8 ], [ 46, 36, -1/8 ], [ 46, 39, -1/8 ], [ 46, 47, -1/8 ], 
  [ 46, 55, -1/16*E(8)-1/16*E(8)^3 ], [ 46, 61, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 46, 62, -1/16*E(8)-1/16*E(8)^3 ], [ 46, 64, 1/16*E(8)+1/16*E(8)^3 ], [ 46, 68, -1/16*E(8)-1/16*E(8)^3 ]
    , [ 46, 71, 1/16*E(8)+1/16*E(8)^3 ], [ 46, 72, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 46, 78, 1/16*E(8)+1/16*E(8)^3 ], [ 47, 70, -3/4 ], [ 47, 76, 3/4 ], [ 47, 77, 1/4 ], [ 47, 83, -1/4 ], 
  [ 47, 93, 3/4*E(8)+3/4*E(8)^3 ], [ 47, 98, -1/4*E(8)-1/4*E(8)^3 ], [ 48, 1, -1 ], 
  [ 48, 18, -1/2*E(8)-1/2*E(8)^3 ], [ 48, 26, -1/2*E(8)-1/2*E(8)^3 ], [ 48, 30, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 48, 38, 1/2*E(8)+1/2*E(8)^3 ], [ 48, 123, 1 ], [ 49, 7, 1/4 ], [ 49, 14, 1/4 ], [ 49, 15, 1/4 ], 
  [ 49, 22, 1/4 ], [ 49, 41, 1/4*E(8)+1/4*E(8)^3 ], [ 49, 50, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 50, 7, 3/4*E(8)+3/4*E(8)^3 ], [ 50, 14, -3/4*E(8)-3/4*E(8)^3 ], [ 50, 15, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 50, 22, 1/4*E(8)+1/4*E(8)^3 ], [ 51, 2, 1 ], [ 51, 124, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 51, 136, -1/4*E(8)-1/4*E(8)^3 ], [ 51, 139, 1/2*E(8)+1/2*E(8)^3 ], [ 51, 147, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 51, 152, -1/4 ], [ 51, 160, -1/4 ], [ 51, 164, 1/4 ], [ 51, 171, 1/4 ], [ 52, 52, 1 ], [ 53, 55, 1/8 ],
  [ 53, 61, -1/8 ], [ 53, 62, 1/8 ], [ 53, 64, -1/8 ], [ 53, 68, -1/8 ], [ 53, 71, 1/8 ], [ 53, 72, -1/8 ],
  [ 53, 78, 1/8 ], [ 54, 28, 3/8 ], [ 54, 36, -1/8 ], [ 54, 39, 3/8 ], [ 54, 47, -1/8 ], 
  [ 54, 55, 3/16*E(8)+3/16*E(8)^3 ], [ 54, 61, 3/16*E(8)+3/16*E(8)^3 ], [ 54, 62, -1/16*E(8)-1/16*E(8)^3 ],
  [ 54, 64, -3/16*E(8)-3/16*E(8)^3 ], [ 54, 68, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 54, 71, -3/16*E(8)-3/16*E(8)^3 ], [ 54, 72, 1/16*E(8)+1/16*E(8)^3 ], [ 54, 78, 1/16*E(8)+1/16*E(8)^3 ],
  [ 55, 28, -1/8*E(8)-1/8*E(8)^3 ], [ 55, 36, -1/8*E(8)-1/8*E(8)^3 ], [ 55, 39, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 55, 47, 1/8*E(8)+1/8*E(8)^3 ], [ 55, 53, 1/2 ], [ 55, 55, -1/8 ], [ 55, 60, 1/2 ], [ 55, 61, -1/8 ], 
  [ 55, 62, -1/8 ], [ 55, 64, -1/8 ], [ 55, 68, -1/8 ], [ 55, 71, -1/8 ], [ 55, 72, -1/8 ], 
  [ 55, 75, 1/4*E(8)+1/4*E(8)^3 ], [ 55, 78, -1/8 ], [ 55, 80, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 55, 81, 1/4*E(8)+1/4*E(8)^3 ], [ 55, 86, 1/4*E(8)+1/4*E(8)^3 ], [ 56, 7, -3/4 ], [ 56, 14, -3/4 ], 
  [ 56, 15, 1/4 ], [ 56, 22, 1/4 ], [ 56, 41, -3/4*E(8)-3/4*E(8)^3 ], [ 56, 50, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 57, 1, -1 ], [ 57, 18, 1/2*E(8)+1/2*E(8)^3 ], [ 57, 26, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 57, 30, -1/2*E(8)-1/2*E(8)^3 ], [ 57, 38, -1/2*E(8)-1/2*E(8)^3 ], [ 57, 123, 1 ], [ 58, 7, 1/4 ], 
  [ 58, 14, 1/4 ], [ 58, 15, 1/4 ], [ 58, 22, 1/4 ], [ 58, 41, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 58, 50, -1/4*E(8)-1/4*E(8)^3 ], [ 59, 82, 1/8*E(8)+1/8*E(8)^3 ], [ 59, 87, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 59, 89, -1/4 ], [ 59, 94, -1/4 ], [ 59, 97, -1/4 ], [ 59, 101, -1/4 ], [ 60, 55, -3/8 ], 
  [ 60, 61, 3/8 ], [ 60, 62, 1/8 ], [ 60, 64, 3/8 ], [ 60, 68, -1/8 ], [ 60, 71, -3/8 ], [ 60, 72, -1/8 ], 
  [ 60, 78, 1/8 ], [ 61, 28, 1/8*E(8)+1/8*E(8)^3 ], [ 61, 36, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 61, 39, -1/8*E(8)-1/8*E(8)^3 ], [ 61, 47, -1/8*E(8)-1/8*E(8)^3 ], [ 61, 53, 1/2 ], [ 61, 55, 1/8 ], 
  [ 61, 60, 1/2 ], [ 61, 61, 1/8 ], [ 61, 62, 1/8 ], [ 61, 64, 1/8 ], [ 61, 68, 1/8 ], [ 61, 71, 1/8 ], 
  [ 61, 72, 1/8 ], [ 61, 75, 1/4*E(8)+1/4*E(8)^3 ], [ 61, 78, 1/8 ], [ 61, 80, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 61, 81, 1/4*E(8)+1/4*E(8)^3 ], [ 61, 86, 1/4*E(8)+1/4*E(8)^3 ], [ 62, 28, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 62, 36, -1/8*E(8)-1/8*E(8)^3 ], [ 62, 39, -3/8*E(8)-3/8*E(8)^3 ], [ 62, 47, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 62, 53, -3/2 ], [ 62, 55, 3/8 ], [ 62, 60, 1/2 ], [ 62, 61, 3/8 ], [ 62, 62, -1/8 ], [ 62, 64, 3/8 ], 
  [ 62, 68, -1/8 ], [ 62, 71, 3/8 ], [ 62, 72, -1/8 ], [ 62, 75, -3/4*E(8)-3/4*E(8)^3 ], [ 62, 78, -1/8 ], 
  [ 62, 80, -3/4*E(8)-3/4*E(8)^3 ], [ 62, 81, 1/4*E(8)+1/4*E(8)^3 ], [ 62, 86, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 63, 18, -1 ], [ 63, 26, 1 ], [ 63, 30, 1 ], [ 63, 38, -1 ], [ 64, 28, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 64, 36, -1/8*E(8)-1/8*E(8)^3 ], [ 64, 39, 1/8*E(8)+1/8*E(8)^3 ], [ 64, 47, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 64, 53, 1/2 ], [ 64, 55, 1/8 ], [ 64, 60, 1/2 ], [ 64, 61, 1/8 ], [ 64, 62, 1/8 ], [ 64, 64, 1/8 ], 
  [ 64, 68, 1/8 ], [ 64, 71, 1/8 ], [ 64, 72, 1/8 ], [ 64, 75, -1/4*E(8)-1/4*E(8)^3 ], [ 64, 78, 1/8 ], 
  [ 64, 80, -1/4*E(8)-1/4*E(8)^3 ], [ 64, 81, -1/4*E(8)-1/4*E(8)^3 ], [ 64, 86, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 65, 7, -3/4 ], [ 65, 14, -3/4 ], [ 65, 15, 1/4 ], [ 65, 22, 1/4 ], [ 65, 41, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 65, 50, -1/4*E(8)-1/4*E(8)^3 ], [ 66, 82, -1/8*E(8)-1/8*E(8)^3 ], [ 66, 87, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 66, 89, -1/4 ], [ 66, 94, -1/4 ], [ 66, 97, -1/4 ], [ 66, 101, -1/4 ], [ 67, 82, -3/8*E(8)-3/8*E(8)^3 ]
    , [ 67, 87, 1/8*E(8)+1/8*E(8)^3 ], [ 67, 89, 3/4 ], [ 67, 94, -1/4 ], [ 67, 97, 3/4 ], 
  [ 67, 101, -1/4 ], [ 68, 28, -3/8*E(8)-3/8*E(8)^3 ], [ 68, 36, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 68, 39, 3/8*E(8)+3/8*E(8)^3 ], [ 68, 47, -1/8*E(8)-1/8*E(8)^3 ], [ 68, 53, -3/2 ], [ 68, 55, -3/8 ], 
  [ 68, 60, 1/2 ], [ 68, 61, -3/8 ], [ 68, 62, 1/8 ], [ 68, 64, -3/8 ], [ 68, 68, 1/8 ], [ 68, 71, -3/8 ], 
  [ 68, 72, 1/8 ], [ 68, 75, -3/4*E(8)-3/4*E(8)^3 ], [ 68, 78, 1/8 ], [ 68, 80, -3/4*E(8)-3/4*E(8)^3 ], 
  [ 68, 81, 1/4*E(8)+1/4*E(8)^3 ], [ 68, 86, 1/4*E(8)+1/4*E(8)^3 ], [ 69, 131, 1 ], 
  [ 70, 55, 1/8*E(8)+1/8*E(8)^3 ], [ 70, 61, -1/8*E(8)-1/8*E(8)^3 ], [ 70, 62, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 70, 64, 1/8*E(8)+1/8*E(8)^3 ], [ 70, 68, -1/8*E(8)-1/8*E(8)^3 ], [ 70, 71, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 70, 72, 1/8*E(8)+1/8*E(8)^3 ], [ 70, 75, 1/2 ], [ 70, 78, -1/8*E(8)-1/8*E(8)^3 ], [ 70, 80, -1/2 ], 
  [ 70, 81, 1/2 ], [ 70, 86, -1/2 ], [ 71, 28, 1/8*E(8)+1/8*E(8)^3 ], [ 71, 36, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 71, 39, -1/8*E(8)-1/8*E(8)^3 ], [ 71, 47, -1/8*E(8)-1/8*E(8)^3 ], [ 71, 53, 1/2 ], [ 71, 55, -1/8 ], 
  [ 71, 60, 1/2 ], [ 71, 61, -1/8 ], [ 71, 62, -1/8 ], [ 71, 64, -1/8 ], [ 71, 68, -1/8 ], 
  [ 71, 71, -1/8 ], [ 71, 72, -1/8 ], [ 71, 75, -1/4*E(8)-1/4*E(8)^3 ], [ 71, 78, -1/8 ], 
  [ 71, 80, -1/4*E(8)-1/4*E(8)^3 ], [ 71, 81, -1/4*E(8)-1/4*E(8)^3 ], [ 71, 86, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 72, 28, 3/8*E(8)+3/8*E(8)^3 ], [ 72, 36, -1/8*E(8)-1/8*E(8)^3 ], [ 72, 39, -3/8*E(8)-3/8*E(8)^3 ], 
  [ 72, 47, 1/8*E(8)+1/8*E(8)^3 ], [ 72, 53, -3/2 ], [ 72, 55, -3/8 ], [ 72, 60, 1/2 ], [ 72, 61, -3/8 ], 
  [ 72, 62, 1/8 ], [ 72, 64, -3/8 ], [ 72, 68, 1/8 ], [ 72, 71, -3/8 ], [ 72, 72, 1/8 ], 
  [ 72, 75, 3/4*E(8)+3/4*E(8)^3 ], [ 72, 78, 1/8 ], [ 72, 80, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 72, 81, -1/4*E(8)-1/4*E(8)^3 ], [ 72, 86, -1/4*E(8)-1/4*E(8)^3 ], [ 73, 82, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 73, 87, -1/8*E(8)-1/8*E(8)^3 ], [ 73, 89, 3/4 ], [ 73, 94, -1/4 ], [ 73, 97, 3/4 ], [ 73, 101, -1/4 ], 
  [ 74, 88, -1/2*E(8)-1/2*E(8)^3 ], [ 74, 102, -1/2 ], [ 74, 105, -1/2 ], [ 75, 21, -1/8*E(8)-1/8*E(8)^3 ],
  [ 75, 29, -1/8*E(8)-1/8*E(8)^3 ], [ 75, 34, -1/4 ], [ 75, 42, -1/4 ], [ 75, 46, -1/4 ], [ 75, 54, -1/4 ],
  [ 76, 55, -1/8*E(8)-1/8*E(8)^3 ], [ 76, 61, 1/8*E(8)+1/8*E(8)^3 ], [ 76, 62, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 76, 64, -1/8*E(8)-1/8*E(8)^3 ], [ 76, 68, 1/8*E(8)+1/8*E(8)^3 ], [ 76, 71, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 76, 72, -1/8*E(8)-1/8*E(8)^3 ], [ 76, 75, 1/2 ], [ 76, 78, 1/8*E(8)+1/8*E(8)^3 ], [ 76, 80, -1/2 ], 
  [ 76, 81, 1/2 ], [ 76, 86, -1/2 ], [ 77, 55, -3/8*E(8)-3/8*E(8)^3 ], [ 77, 61, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 77, 62, 1/8*E(8)+1/8*E(8)^3 ], [ 77, 64, -3/8*E(8)-3/8*E(8)^3 ], [ 77, 68, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 77, 71, 3/8*E(8)+3/8*E(8)^3 ], [ 77, 72, 1/8*E(8)+1/8*E(8)^3 ], [ 77, 75, -3/2 ], 
  [ 77, 78, -1/8*E(8)-1/8*E(8)^3 ], [ 77, 80, 3/2 ], [ 77, 81, 1/2 ], [ 77, 86, -1/2 ], 
  [ 78, 28, -3/8*E(8)-3/8*E(8)^3 ], [ 78, 36, 1/8*E(8)+1/8*E(8)^3 ], [ 78, 39, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 78, 47, -1/8*E(8)-1/8*E(8)^3 ], [ 78, 53, -3/2 ], [ 78, 55, 3/8 ], [ 78, 60, 1/2 ], [ 78, 61, 3/8 ], 
  [ 78, 62, -1/8 ], [ 78, 64, 3/8 ], [ 78, 68, -1/8 ], [ 78, 71, 3/8 ], [ 78, 72, -1/8 ], 
  [ 78, 75, 3/4*E(8)+3/4*E(8)^3 ], [ 78, 78, -1/8 ], [ 78, 80, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 78, 81, -1/4*E(8)-1/4*E(8)^3 ], [ 78, 86, -1/4*E(8)-1/4*E(8)^3 ], [ 79, 110, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 79, 114, 1/4*E(8)+1/4*E(8)^3 ], [ 80, 21, 1/8*E(8)+1/8*E(8)^3 ], [ 80, 29, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 80, 34, -1/4 ], [ 80, 42, -1/4 ], [ 80, 46, -1/4 ], [ 80, 54, -1/4 ], [ 81, 21, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 81, 29, -1/8*E(8)-1/8*E(8)^3 ], [ 81, 34, 3/4 ], [ 81, 42, -1/4 ], [ 81, 46, 3/4 ], [ 81, 54, -1/4 ], 
  [ 82, 89, 1/2*E(8)+1/2*E(8)^3 ], [ 82, 94, 1/2*E(8)+1/2*E(8)^3 ], [ 82, 97, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 82, 101, -1/2*E(8)-1/2*E(8)^3 ], [ 83, 55, 3/8*E(8)+3/8*E(8)^3 ], [ 83, 61, -3/8*E(8)-3/8*E(8)^3 ], 
  [ 83, 62, -1/8*E(8)-1/8*E(8)^3 ], [ 83, 64, 3/8*E(8)+3/8*E(8)^3 ], [ 83, 68, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 83, 71, -3/8*E(8)-3/8*E(8)^3 ], [ 83, 72, -1/8*E(8)-1/8*E(8)^3 ], [ 83, 75, -3/2 ], 
  [ 83, 78, 1/8*E(8)+1/8*E(8)^3 ], [ 83, 80, 3/2 ], [ 83, 81, 1/2 ], [ 83, 86, -1/2 ], 
  [ 84, 88, -1/2*E(8)-1/2*E(8)^3 ], [ 84, 102, 1/2 ], [ 84, 105, 1/2 ], [ 85, 59, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 85, 66, -1/4*E(8)-1/4*E(8)^3 ], [ 85, 67, -1/4*E(8)-1/4*E(8)^3 ], [ 85, 73, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 86, 21, -3/8*E(8)-3/8*E(8)^3 ], [ 86, 29, 1/8*E(8)+1/8*E(8)^3 ], [ 86, 34, 3/4 ], [ 86, 42, -1/4 ], 
  [ 86, 46, 3/4 ], [ 86, 54, -1/4 ], [ 87, 89, -3/2*E(8)-3/2*E(8)^3 ], [ 87, 94, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 87, 97, 3/2*E(8)+3/2*E(8)^3 ], [ 87, 101, -1/2*E(8)-1/2*E(8)^3 ], [ 88, 102, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 88, 105, -1/2*E(8)-1/2*E(8)^3 ], [ 89, 59, 1/4 ], [ 89, 66, -1/4 ], [ 89, 67, 1/4 ], [ 89, 73, -1/4 ], 
  [ 89, 85, 1/4*E(8)+1/4*E(8)^3 ], [ 89, 90, 1/4*E(8)+1/4*E(8)^3 ], [ 90, 59, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 90, 66, 3/4*E(8)+3/4*E(8)^3 ], [ 90, 67, -1/4*E(8)-1/4*E(8)^3 ], [ 90, 73, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 91, 74, -1/4*E(8)-1/4*E(8)^3 ], [ 91, 84, 1/4*E(8)+1/4*E(8)^3 ], [ 92, 110, -1/2 ], [ 92, 114, -1/2 ], 
  [ 92, 116, -E(8)-E(8)^3 ], [ 93, 34, -1/2*E(8)-1/2*E(8)^3 ], [ 93, 42, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 93, 46, 1/2*E(8)+1/2*E(8)^3 ], [ 93, 54, 1/2*E(8)+1/2*E(8)^3 ], [ 94, 59, -3/4 ], [ 94, 66, 3/4 ], 
  [ 94, 67, 1/4 ], [ 94, 73, -1/4 ], [ 94, 85, -3/4*E(8)-3/4*E(8)^3 ], [ 94, 90, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 95, 92, -1/4 ], [ 95, 96, -1/4 ], [ 95, 99, -1/4*E(8)-1/4*E(8)^3 ], [ 95, 103, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 95, 104, -1/4*E(8)-1/4*E(8)^3 ], [ 95, 107, -1/4*E(8)-1/4*E(8)^3 ], [ 96, 110, 1/2 ], [ 96, 114, 1/2 ],
  [ 96, 116, -E(8)-E(8)^3 ], [ 97, 59, -1/4 ], [ 97, 66, 1/4 ], [ 97, 67, -1/4 ], [ 97, 73, 1/4 ], 
  [ 97, 85, 1/4*E(8)+1/4*E(8)^3 ], [ 97, 90, 1/4*E(8)+1/4*E(8)^3 ], [ 98, 34, 3/2*E(8)+3/2*E(8)^3 ], 
  [ 98, 42, -1/2*E(8)-1/2*E(8)^3 ], [ 98, 46, -3/2*E(8)-3/2*E(8)^3 ], [ 98, 54, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 99, 92, -1/8*E(8)-1/8*E(8)^3 ], [ 99, 96, 1/8*E(8)+1/8*E(8)^3 ], [ 99, 99, 1/4 ], [ 99, 103, -1/4 ], 
  [ 99, 104, 1/4 ], [ 99, 107, -1/4 ], [ 99, 108, -1/4 ], [ 99, 112, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 99, 115, 1/4*E(8)+1/4*E(8)^3 ], [ 100, 92, 1/4 ], [ 100, 96, 1/4 ], [ 100, 99, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 100, 103, -1/4*E(8)-1/4*E(8)^3 ], [ 100, 104, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 100, 107, -1/4*E(8)-1/4*E(8)^3 ], [ 101, 59, 3/4 ], [ 101, 66, -3/4 ], [ 101, 67, -1/4 ], 
  [ 101, 73, 1/4 ], [ 101, 85, -3/4*E(8)-3/4*E(8)^3 ], [ 101, 90, 1/4*E(8)+1/4*E(8)^3 ], [ 102, 74, 1/2 ], 
  [ 102, 84, 1/2 ], [ 102, 91, -E(8)-E(8)^3 ], [ 103, 92, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 103, 96, -1/8*E(8)-1/8*E(8)^3 ], [ 103, 99, 1/4 ], [ 103, 103, -1/4 ], [ 103, 104, 1/4 ], 
  [ 103, 107, -1/4 ], [ 103, 108, 1/4 ], [ 103, 112, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 103, 115, 1/4*E(8)+1/4*E(8)^3 ], [ 104, 92, 1/8*E(8)+1/8*E(8)^3 ], [ 104, 96, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 104, 99, -1/4 ], [ 104, 103, 1/4 ], [ 104, 104, -1/4 ], [ 104, 107, 1/4 ], [ 104, 108, -1/4 ], 
  [ 104, 112, -1/4*E(8)-1/4*E(8)^3 ], [ 104, 115, 1/4*E(8)+1/4*E(8)^3 ], [ 105, 74, -1/2 ], 
  [ 105, 84, -1/2 ], [ 105, 91, -E(8)-E(8)^3 ], [ 106, 111, 1/2*E(8)+1/2*E(8)^3 ], [ 106, 117, -1/2 ], 
  [ 106, 118, 1/2 ], [ 107, 92, -1/8*E(8)-1/8*E(8)^3 ], [ 107, 96, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 107, 99, -1/4 ], [ 107, 103, 1/4 ], [ 107, 104, -1/4 ], [ 107, 107, 1/4 ], [ 107, 108, 1/4 ], 
  [ 107, 112, -1/4*E(8)-1/4*E(8)^3 ], [ 107, 115, 1/4*E(8)+1/4*E(8)^3 ], [ 108, 99, -1 ], [ 108, 103, -1 ],
  [ 108, 104, 1 ], [ 108, 107, 1 ], [ 109, 111, -1/2*E(8)-1/2*E(8)^3 ], [ 109, 117, -1/2 ], 
  [ 109, 118, 1/2 ], [ 110, 99, -1/2*E(8)-1/2*E(8)^3 ], [ 110, 103, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 110, 104, 1/2*E(8)+1/2*E(8)^3 ], [ 110, 107, -1/2*E(8)-1/2*E(8)^3 ], [ 110, 112, -1 ], 
  [ 110, 115, -1 ], [ 111, 117, -1/2*E(8)-1/2*E(8)^3 ], [ 111, 118, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 112, 79, -1/2*E(8)-1/2*E(8)^3 ], [ 112, 95, -1/2 ], [ 112, 100, -1/2 ], 
  [ 113, 106, -1/4*E(8)-1/4*E(8)^3 ], [ 113, 109, -1/4*E(8)-1/4*E(8)^3 ], [ 114, 99, -1/2*E(8)-1/2*E(8)^3 ]
    , [ 114, 103, 1/2*E(8)+1/2*E(8)^3 ], [ 114, 104, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 114, 107, -1/2*E(8)-1/2*E(8)^3 ], [ 114, 112, 1 ], [ 114, 115, 1 ], [ 115, 79, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 115, 95, 1/2 ], [ 115, 100, 1/2 ], [ 116, 95, -1/2*E(8)-1/2*E(8)^3 ], [ 116, 100, 1/2*E(8)+1/2*E(8)^3 ]
    , [ 117, 106, 1/2 ], [ 117, 109, -1/2 ], [ 117, 113, -E(8)-E(8)^3 ], [ 118, 106, 1/2 ], 
  [ 118, 109, -1/2 ], [ 118, 113, E(8)+E(8)^3 ], [ 119, 119, -1/2 ], [ 119, 120, 1/2 ], [ 120, 119, -3/2 ],
  [ 120, 120, -1/2 ], [ 121, 132, -1/2*E(8)-1/2*E(8)^3 ], [ 121, 140, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 121, 144, -1/2*E(8)-1/2*E(8)^3 ], [ 121, 151, 1/2*E(8)+1/2*E(8)^3 ], [ 121, 168, 1 ], [ 121, 177, -1 ],
  [ 122, 32, -1 ], [ 122, 40, 1 ], [ 122, 44, -1 ], [ 122, 51, 1 ], [ 123, 12, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 123, 20, 1/8*E(8)+1/8*E(8)^3 ], [ 123, 24, -1/8*E(8)-1/8*E(8)^3 ], [ 123, 31, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 123, 48, 1/4 ], [ 123, 57, -1/4 ], [ 124, 6, -1/4*E(8)-1/4*E(8)^3 ], [ 124, 126, 1/4*E(8)+1/4*E(8)^3 ],
  [ 124, 145, -1/2 ], [ 124, 153, 1/2 ], [ 124, 245, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 124, 246, 1/2*E(8)+1/2*E(8)^3 ], [ 124, 247, -1/4*E(8)-1/4*E(8)^3 ], [ 125, 132, -1/4*E(8)-1/4*E(8)^3 ]
    , [ 125, 140, -1/4*E(8)-1/4*E(8)^3 ], [ 125, 144, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 125, 151, 1/4*E(8)+1/4*E(8)^3 ], [ 125, 157, -1/4 ], [ 125, 165, -1/4 ], [ 126, 9, -1/4 ], 
  [ 126, 10, -1/2*E(8)-1/2*E(8)^3 ], [ 126, 23, -1/2*E(8)-1/2*E(8)^3 ], [ 126, 129, -1/4 ], 
  [ 126, 130, -1/8*E(8)-1/8*E(8)^3 ], [ 126, 143, -1/8*E(8)-1/8*E(8)^3 ], [ 126, 241, 1/4 ], 
  [ 126, 242, 1/2 ], [ 126, 243, -1/4 ], [ 126, 244, 1/4 ], [ 126, 245, -1/2 ], 
  [ 127, 155, 1/2*E(8)+1/2*E(8)^3 ], [ 127, 163, 3/2*E(8)+3/2*E(8)^3 ], [ 127, 169, -1/4 ], 
  [ 127, 176, -3/4 ], [ 127, 178, 1/4 ], [ 127, 185, 3/4 ], [ 128, 8, -1/4 ], [ 128, 128, 1/4 ], 
  [ 128, 247, -1/4 ], [ 128, 248, 1/2 ], [ 129, 6, -1/4 ], [ 129, 25, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 129, 33, -1/4*E(8)-1/4*E(8)^3 ], [ 129, 126, -1/4 ], [ 129, 145, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 129, 153, -1/4*E(8)-1/4*E(8)^3 ], [ 129, 241, 1/2 ], [ 129, 242, -1/2 ], [ 129, 243, -1/2 ], 
  [ 129, 244, 1/2 ], [ 129, 245, -1/4 ], [ 129, 247, 1/4 ], [ 130, 4, 1 ], [ 130, 16, -1 ], 
  [ 130, 32, -1/2*E(8)-1/2*E(8)^3 ], [ 130, 40, -1/2*E(8)-1/2*E(8)^3 ], [ 130, 44, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 130, 51, -1/2*E(8)-1/2*E(8)^3 ], [ 131, 12, 1/4 ], [ 131, 20, 1/4 ], [ 131, 24, 1/4 ], 
  [ 131, 31, 1/4 ], [ 132, 132, 1/4 ], [ 132, 140, -1/4 ], [ 132, 144, -1/4 ], [ 132, 151, 1/4 ], 
  [ 132, 157, -1/8*E(8)-1/8*E(8)^3 ], [ 132, 165, 1/8*E(8)+1/8*E(8)^3 ], [ 132, 168, -1/4*E(8)-1/4*E(8)^3 ]
    , [ 132, 177, -1/4*E(8)-1/4*E(8)^3 ], [ 132, 189, 1/4 ], [ 133, 132, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 133, 140, 1/4*E(8)+1/4*E(8)^3 ], [ 133, 144, -1/4*E(8)-1/4*E(8)^3 ], [ 133, 151, -1/4*E(8)-1/4*E(8)^3 ]
    , [ 133, 157, -1/4 ], [ 133, 165, -1/4 ], [ 134, 155, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 134, 163, 3/2*E(8)+3/2*E(8)^3 ], [ 134, 169, 1/4 ], [ 134, 176, 3/4 ], [ 134, 178, -1/4 ], 
  [ 134, 185, -3/4 ], [ 135, 155, -1/2*E(8)-1/2*E(8)^3 ], [ 135, 163, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 135, 169, 1/4 ], [ 135, 176, -1/4 ], [ 135, 178, -1/4 ], [ 135, 185, 1/4 ], 
  [ 136, 6, 1/4*E(8)+1/4*E(8)^3 ], [ 136, 126, -1/4*E(8)-1/4*E(8)^3 ], [ 136, 145, -1/2 ], 
  [ 136, 153, 1/2 ], [ 136, 245, 1/4*E(8)+1/4*E(8)^3 ], [ 136, 246, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 136, 247, 1/4*E(8)+1/4*E(8)^3 ], [ 137, 63, 1 ], [ 138, 3, -1/4*E(8)-1/4*E(8)^3 ], [ 138, 17, -1 ], 
  [ 138, 121, -1/4*E(8)-1/4*E(8)^3 ], [ 138, 125, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 138, 133, -1/2*E(8)-1/2*E(8)^3 ], [ 138, 138, 1/4 ], [ 138, 146, 1/4 ], [ 138, 150, 1/4 ], 
  [ 138, 158, 1/4 ], [ 139, 9, 1/8*E(8)+1/8*E(8)^3 ], [ 139, 129, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 139, 130, 1/8 ], [ 139, 143, -1/8 ], [ 139, 241, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 139, 243, 1/8*E(8)+1/8*E(8)^3 ], [ 139, 244, -1/8*E(8)-1/8*E(8)^3 ], [ 140, 132, -1/4 ], 
  [ 140, 140, 1/4 ], [ 140, 144, 1/4 ], [ 140, 151, -1/4 ], [ 140, 157, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 140, 165, 1/8*E(8)+1/8*E(8)^3 ], [ 140, 168, 1/4*E(8)+1/4*E(8)^3 ], [ 140, 177, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 140, 189, 1/4 ], [ 141, 190, 1/4*E(8)+1/4*E(8)^3 ], [ 141, 196, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 141, 197, 3/4*E(8)+3/4*E(8)^3 ], [ 141, 203, 3/4*E(8)+3/4*E(8)^3 ], [ 142, 155, -1/2*E(8)-1/2*E(8)^3 ],
  [ 142, 163, 1/2*E(8)+1/2*E(8)^3 ], [ 142, 169, -1/4 ], [ 142, 176, 1/4 ], [ 142, 178, 1/4 ], 
  [ 142, 185, -1/4 ], [ 143, 4, 1 ], [ 143, 16, -1 ], [ 143, 32, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 143, 40, 1/2*E(8)+1/2*E(8)^3 ], [ 143, 44, 1/2*E(8)+1/2*E(8)^3 ], [ 143, 51, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 144, 132, -1/4 ], [ 144, 140, 1/4 ], [ 144, 144, 1/4 ], [ 144, 151, -1/4 ], 
  [ 144, 157, 1/8*E(8)+1/8*E(8)^3 ], [ 144, 165, -1/8*E(8)-1/8*E(8)^3 ], [ 144, 168, -1/4*E(8)-1/4*E(8)^3 ]
    , [ 144, 177, -1/4*E(8)-1/4*E(8)^3 ], [ 144, 189, 1/4 ], [ 145, 19, -1/4 ], [ 145, 27, 1/4 ], 
  [ 145, 32, -1/4*E(8)-1/4*E(8)^3 ], [ 145, 40, 1/4*E(8)+1/4*E(8)^3 ], [ 145, 44, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 145, 51, -1/4*E(8)-1/4*E(8)^3 ], [ 146, 3, -1/4*E(8)-1/4*E(8)^3 ], [ 146, 17, 1 ], 
  [ 146, 121, -1/4*E(8)-1/4*E(8)^3 ], [ 146, 125, 1/2*E(8)+1/2*E(8)^3 ], [ 146, 133, 1/2*E(8)+1/2*E(8)^3 ],
  [ 146, 138, 1/4 ], [ 146, 146, 1/4 ], [ 146, 150, 1/4 ], [ 146, 158, 1/4 ], 
  [ 147, 9, -1/8*E(8)-1/8*E(8)^3 ], [ 147, 129, 1/8*E(8)+1/8*E(8)^3 ], [ 147, 130, 1/8 ], 
  [ 147, 143, -1/8 ], [ 147, 241, -1/8*E(8)-1/8*E(8)^3 ], [ 147, 243, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 147, 244, 1/8*E(8)+1/8*E(8)^3 ], [ 148, 190, -1/4 ], [ 148, 196, 1/4 ], [ 148, 197, -3/4 ], 
  [ 148, 203, 3/4 ], [ 148, 213, 1/4*E(8)+1/4*E(8)^3 ], [ 148, 218, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 149, 190, -1/4*E(8)-1/4*E(8)^3 ], [ 149, 196, -1/4*E(8)-1/4*E(8)^3 ], [ 149, 197, 1/4*E(8)+1/4*E(8)^3 ]
    , [ 149, 203, 1/4*E(8)+1/4*E(8)^3 ], [ 150, 3, 1/4*E(8)+1/4*E(8)^3 ], [ 150, 17, 1 ], 
  [ 150, 121, 1/4*E(8)+1/4*E(8)^3 ], [ 150, 125, -1/2*E(8)-1/2*E(8)^3 ], [ 150, 133, -1/2*E(8)-1/2*E(8)^3 ]
    , [ 150, 138, 1/4 ], [ 150, 146, 1/4 ], [ 150, 150, 1/4 ], [ 150, 158, 1/4 ], [ 151, 132, 1/4 ], 
  [ 151, 140, -1/4 ], [ 151, 144, -1/4 ], [ 151, 151, 1/4 ], [ 151, 157, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 151, 165, -1/8*E(8)-1/8*E(8)^3 ], [ 151, 168, 1/4*E(8)+1/4*E(8)^3 ], [ 151, 177, 1/4*E(8)+1/4*E(8)^3 ],
  [ 151, 189, 1/4 ], [ 152, 4, 1/4*E(8)+1/4*E(8)^3 ], [ 152, 16, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 152, 19, -1/8*E(8)-1/8*E(8)^3 ], [ 152, 27, -1/8*E(8)-1/8*E(8)^3 ], [ 152, 32, 1/4 ], [ 152, 40, 1/4 ],
  [ 152, 44, -1/4 ], [ 152, 51, -1/4 ], [ 152, 122, -1/4 ], [ 153, 19, -1/4 ], [ 153, 27, 1/4 ], 
  [ 153, 32, 1/4*E(8)+1/4*E(8)^3 ], [ 153, 40, -1/4*E(8)-1/4*E(8)^3 ], [ 153, 44, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 153, 51, 1/4*E(8)+1/4*E(8)^3 ], [ 154, 148, 1/2 ], [ 154, 156, 3/2 ], [ 154, 159, 1/2 ], 
  [ 154, 167, 3/2 ], [ 154, 175, 1/4*E(8)+1/4*E(8)^3 ], [ 154, 181, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 154, 182, 3/4*E(8)+3/4*E(8)^3 ], [ 154, 184, -1/4*E(8)-1/4*E(8)^3 ], [ 154, 188, 3/4*E(8)+3/4*E(8)^3 ],
  [ 154, 191, -1/4*E(8)-1/4*E(8)^3 ], [ 154, 192, -3/4*E(8)-3/4*E(8)^3 ], 
  [ 154, 198, -3/4*E(8)-3/4*E(8)^3 ], [ 155, 169, 1/8*E(8)+1/8*E(8)^3 ], [ 155, 176, 3/8*E(8)+3/8*E(8)^3 ],
  [ 155, 178, 1/8*E(8)+1/8*E(8)^3 ], [ 155, 185, 3/8*E(8)+3/8*E(8)^3 ], [ 156, 190, 1/4 ], 
  [ 156, 196, -1/4 ], [ 156, 197, -1/4 ], [ 156, 203, 1/4 ], [ 156, 213, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 156, 218, 1/4*E(8)+1/4*E(8)^3 ], [ 157, 125, -1 ], [ 157, 133, 1 ], [ 157, 138, -1/4*E(8)-1/4*E(8)^3 ],
  [ 157, 146, 1/4*E(8)+1/4*E(8)^3 ], [ 157, 150, -1/4*E(8)-1/4*E(8)^3 ], [ 157, 158, 1/4*E(8)+1/4*E(8)^3 ],
  [ 158, 3, 1/4*E(8)+1/4*E(8)^3 ], [ 158, 17, -1 ], [ 158, 121, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 158, 125, 1/2*E(8)+1/2*E(8)^3 ], [ 158, 133, 1/2*E(8)+1/2*E(8)^3 ], [ 158, 138, 1/4 ], 
  [ 158, 146, 1/4 ], [ 158, 150, 1/4 ], [ 158, 158, 1/4 ], [ 159, 190, 1/4 ], [ 159, 196, -1/4 ], 
  [ 159, 197, 3/4 ], [ 159, 203, -3/4 ], [ 159, 213, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 159, 218, 3/4*E(8)+3/4*E(8)^3 ], [ 160, 4, -1/4*E(8)-1/4*E(8)^3 ], [ 160, 16, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 160, 19, -1/8*E(8)-1/8*E(8)^3 ], [ 160, 27, -1/8*E(8)-1/8*E(8)^3 ], [ 160, 32, -1/4 ], 
  [ 160, 40, -1/4 ], [ 160, 44, 1/4 ], [ 160, 51, 1/4 ], [ 160, 122, -1/4 ], 
  [ 161, 127, 1/4*E(8)+1/4*E(8)^3 ], [ 161, 134, -1/4*E(8)-1/4*E(8)^3 ], [ 161, 135, 3/4*E(8)+3/4*E(8)^3 ],
  [ 161, 142, -3/4*E(8)-3/4*E(8)^3 ], [ 162, 148, -1/2 ], [ 162, 156, 1/2 ], [ 162, 159, -1/2 ], 
  [ 162, 167, 1/2 ], [ 162, 175, -1/4*E(8)-1/4*E(8)^3 ], [ 162, 181, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 162, 182, 1/4*E(8)+1/4*E(8)^3 ], [ 162, 184, 1/4*E(8)+1/4*E(8)^3 ], [ 162, 188, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 162, 191, 1/4*E(8)+1/4*E(8)^3 ], [ 162, 192, -1/4*E(8)-1/4*E(8)^3 ], [ 162, 198, -1/4*E(8)-1/4*E(8)^3 ]
    , [ 163, 169, -1/8*E(8)-1/8*E(8)^3 ], [ 163, 176, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 163, 178, -1/8*E(8)-1/8*E(8)^3 ], [ 163, 185, 1/8*E(8)+1/8*E(8)^3 ], [ 164, 4, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 164, 16, -1/4*E(8)-1/4*E(8)^3 ], [ 164, 19, -1/8*E(8)-1/8*E(8)^3 ], [ 164, 27, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 164, 32, 1/4 ], [ 164, 40, 1/4 ], [ 164, 44, -1/4 ], [ 164, 51, -1/4 ], [ 164, 122, 1/4 ], 
  [ 165, 125, 1 ], [ 165, 133, -1 ], [ 165, 138, -1/4*E(8)-1/4*E(8)^3 ], [ 165, 146, 1/4*E(8)+1/4*E(8)^3 ],
  [ 165, 150, -1/4*E(8)-1/4*E(8)^3 ], [ 165, 158, 1/4*E(8)+1/4*E(8)^3 ], [ 166, 148, -1/2 ], 
  [ 166, 156, -3/2 ], [ 166, 159, -1/2 ], [ 166, 167, -3/2 ], [ 166, 175, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 166, 181, 1/4*E(8)+1/4*E(8)^3 ], [ 166, 182, 3/4*E(8)+3/4*E(8)^3 ], [ 166, 184, -1/4*E(8)-1/4*E(8)^3 ],
  [ 166, 188, 3/4*E(8)+3/4*E(8)^3 ], [ 166, 191, -1/4*E(8)-1/4*E(8)^3 ], [ 166, 192, -3/4*E(8)-3/4*E(8)^3 ]
    , [ 166, 198, -3/4*E(8)-3/4*E(8)^3 ], [ 167, 190, -1/4 ], [ 167, 196, 1/4 ], [ 167, 197, 1/4 ], 
  [ 167, 203, -1/4 ], [ 167, 213, -1/4*E(8)-1/4*E(8)^3 ], [ 167, 218, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 168, 3, 1/4 ], [ 168, 121, -1/4 ], [ 168, 138, 1/8*E(8)+1/8*E(8)^3 ], [ 168, 146, 1/8*E(8)+1/8*E(8)^3 ]
    , [ 168, 150, -1/8*E(8)-1/8*E(8)^3 ], [ 168, 158, -1/8*E(8)-1/8*E(8)^3 ], [ 169, 127, 1/4 ], 
  [ 169, 134, 1/4 ], [ 169, 135, 3/4 ], [ 169, 142, 3/4 ], [ 169, 161, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 169, 170, -3/4*E(8)-3/4*E(8)^3 ], [ 170, 127, -1/4*E(8)-1/4*E(8)^3 ], [ 170, 134, 1/4*E(8)+1/4*E(8)^3 ]
    , [ 170, 135, 1/4*E(8)+1/4*E(8)^3 ], [ 170, 142, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 171, 4, 1/4*E(8)+1/4*E(8)^3 ], [ 171, 16, 1/4*E(8)+1/4*E(8)^3 ], [ 171, 19, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 171, 27, -1/8*E(8)-1/8*E(8)^3 ], [ 171, 32, -1/4 ], [ 171, 40, -1/4 ], [ 171, 44, 1/4 ], 
  [ 171, 51, 1/4 ], [ 171, 122, 1/4 ], [ 172, 172, 1 ], [ 173, 175, 1/2 ], [ 173, 181, -1/2 ], 
  [ 173, 182, 3/2 ], [ 173, 184, -1/2 ], [ 173, 188, -3/2 ], [ 173, 191, 1/2 ], [ 173, 192, -3/2 ], 
  [ 173, 198, 3/2 ], [ 174, 148, 1/2 ], [ 174, 156, -1/2 ], [ 174, 159, 1/2 ], [ 174, 167, -1/2 ], 
  [ 174, 175, -1/4*E(8)-1/4*E(8)^3 ], [ 174, 181, -1/4*E(8)-1/4*E(8)^3 ], [ 174, 182, 1/4*E(8)+1/4*E(8)^3 ]
    , [ 174, 184, 1/4*E(8)+1/4*E(8)^3 ], [ 174, 188, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 174, 191, 1/4*E(8)+1/4*E(8)^3 ], [ 174, 192, -1/4*E(8)-1/4*E(8)^3 ], [ 174, 198, -1/4*E(8)-1/4*E(8)^3 ]
    , [ 175, 148, 1/8*E(8)+1/8*E(8)^3 ], [ 175, 156, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 175, 159, -1/8*E(8)-1/8*E(8)^3 ], [ 175, 167, -3/8*E(8)-3/8*E(8)^3 ], [ 175, 173, 1/8 ], 
  [ 175, 175, -1/8 ], [ 175, 180, 3/8 ], [ 175, 181, -1/8 ], [ 175, 182, -3/8 ], [ 175, 184, -1/8 ], 
  [ 175, 188, -3/8 ], [ 175, 191, -1/8 ], [ 175, 192, -3/8 ], [ 175, 195, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 175, 198, -3/8 ], [ 175, 200, -1/16*E(8)-1/16*E(8)^3 ], [ 175, 201, -3/16*E(8)-3/16*E(8)^3 ], 
  [ 175, 206, -3/16*E(8)-3/16*E(8)^3 ], [ 176, 127, -1/4 ], [ 176, 134, -1/4 ], [ 176, 135, 1/4 ], 
  [ 176, 142, 1/4 ], [ 176, 161, 1/4*E(8)+1/4*E(8)^3 ], [ 176, 170, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 177, 3, 1/4 ], [ 177, 121, -1/4 ], [ 177, 138, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 177, 146, -1/8*E(8)-1/8*E(8)^3 ], [ 177, 150, 1/8*E(8)+1/8*E(8)^3 ], [ 177, 158, 1/8*E(8)+1/8*E(8)^3 ],
  [ 178, 127, 1/4 ], [ 178, 134, 1/4 ], [ 178, 135, 3/4 ], [ 178, 142, 3/4 ], 
  [ 178, 161, 1/4*E(8)+1/4*E(8)^3 ], [ 178, 170, 3/4*E(8)+3/4*E(8)^3 ], [ 179, 202, -1/2*E(8)-1/2*E(8)^3 ],
  [ 179, 207, -3/2*E(8)-3/2*E(8)^3 ], [ 179, 209, -1/4 ], [ 179, 214, -3/4 ], [ 179, 217, -1/4 ], 
  [ 179, 221, -3/4 ], [ 180, 175, -1/2 ], [ 180, 181, 1/2 ], [ 180, 182, 1/2 ], [ 180, 184, 1/2 ], 
  [ 180, 188, -1/2 ], [ 180, 191, -1/2 ], [ 180, 192, -1/2 ], [ 180, 198, 1/2 ], 
  [ 181, 148, -1/8*E(8)-1/8*E(8)^3 ], [ 181, 156, -3/8*E(8)-3/8*E(8)^3 ], [ 181, 159, 1/8*E(8)+1/8*E(8)^3 ]
    , [ 181, 167, 3/8*E(8)+3/8*E(8)^3 ], [ 181, 173, 1/8 ], [ 181, 175, 1/8 ], [ 181, 180, 3/8 ], 
  [ 181, 181, 1/8 ], [ 181, 182, 3/8 ], [ 181, 184, 1/8 ], [ 181, 188, 3/8 ], [ 181, 191, 1/8 ], 
  [ 181, 192, 3/8 ], [ 181, 195, -1/16*E(8)-1/16*E(8)^3 ], [ 181, 198, 3/8 ], 
  [ 181, 200, -1/16*E(8)-1/16*E(8)^3 ], [ 181, 201, -3/16*E(8)-3/16*E(8)^3 ], 
  [ 181, 206, -3/16*E(8)-3/16*E(8)^3 ], [ 182, 148, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 182, 156, 1/8*E(8)+1/8*E(8)^3 ], [ 182, 159, 1/8*E(8)+1/8*E(8)^3 ], [ 182, 167, -1/8*E(8)-1/8*E(8)^3 ],
  [ 182, 173, -1/8 ], [ 182, 175, 1/8 ], [ 182, 180, 1/8 ], [ 182, 181, 1/8 ], [ 182, 182, -1/8 ], 
  [ 182, 184, 1/8 ], [ 182, 188, -1/8 ], [ 182, 191, 1/8 ], [ 182, 192, -1/8 ], 
  [ 182, 195, 1/16*E(8)+1/16*E(8)^3 ], [ 182, 198, -1/8 ], [ 182, 200, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 182, 201, -1/16*E(8)-1/16*E(8)^3 ], [ 182, 206, -1/16*E(8)-1/16*E(8)^3 ], [ 183, 138, -1/4 ], 
  [ 183, 146, 1/4 ], [ 183, 150, 1/4 ], [ 183, 158, -1/4 ], [ 184, 148, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 184, 156, 3/8*E(8)+3/8*E(8)^3 ], [ 184, 159, -1/8*E(8)-1/8*E(8)^3 ], [ 184, 167, -3/8*E(8)-3/8*E(8)^3 ]
    , [ 184, 173, 1/8 ], [ 184, 175, 1/8 ], [ 184, 180, 3/8 ], [ 184, 181, 1/8 ], [ 184, 182, 3/8 ], 
  [ 184, 184, 1/8 ], [ 184, 188, 3/8 ], [ 184, 191, 1/8 ], [ 184, 192, 3/8 ], 
  [ 184, 195, 1/16*E(8)+1/16*E(8)^3 ], [ 184, 198, 3/8 ], [ 184, 200, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 184, 201, 3/16*E(8)+3/16*E(8)^3 ], [ 184, 206, 3/16*E(8)+3/16*E(8)^3 ], [ 185, 127, -1/4 ], 
  [ 185, 134, -1/4 ], [ 185, 135, 1/4 ], [ 185, 142, 1/4 ], [ 185, 161, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 185, 170, 1/4*E(8)+1/4*E(8)^3 ], [ 186, 202, 1/2*E(8)+1/2*E(8)^3 ], [ 186, 207, 3/2*E(8)+3/2*E(8)^3 ], 
  [ 186, 209, -1/4 ], [ 186, 214, -3/4 ], [ 186, 217, -1/4 ], [ 186, 221, -3/4 ], 
  [ 187, 202, 1/2*E(8)+1/2*E(8)^3 ], [ 187, 207, -1/2*E(8)-1/2*E(8)^3 ], [ 187, 209, 1/4 ], 
  [ 187, 214, -1/4 ], [ 187, 217, 1/4 ], [ 187, 221, -1/4 ], [ 188, 148, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 188, 156, -1/8*E(8)-1/8*E(8)^3 ], [ 188, 159, -1/8*E(8)-1/8*E(8)^3 ], [ 188, 167, 1/8*E(8)+1/8*E(8)^3 ]
    , [ 188, 173, -1/8 ], [ 188, 175, -1/8 ], [ 188, 180, 1/8 ], [ 188, 181, -1/8 ], [ 188, 182, 1/8 ], 
  [ 188, 184, -1/8 ], [ 188, 188, 1/8 ], [ 188, 191, -1/8 ], [ 188, 192, 1/8 ], 
  [ 188, 195, 1/16*E(8)+1/16*E(8)^3 ], [ 188, 198, 1/8 ], [ 188, 200, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 188, 201, -1/16*E(8)-1/16*E(8)^3 ], [ 188, 206, -1/16*E(8)-1/16*E(8)^3 ], [ 189, 11, 1 ], 
  [ 190, 175, -1/8*E(8)-1/8*E(8)^3 ], [ 190, 181, 1/8*E(8)+1/8*E(8)^3 ], [ 190, 182, -3/8*E(8)-3/8*E(8)^3 ]
    , [ 190, 184, -1/8*E(8)-1/8*E(8)^3 ], [ 190, 188, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 190, 191, 1/8*E(8)+1/8*E(8)^3 ], [ 190, 192, -3/8*E(8)-3/8*E(8)^3 ], [ 190, 195, 1/8 ], 
  [ 190, 198, 3/8*E(8)+3/8*E(8)^3 ], [ 190, 200, -1/8 ], [ 190, 201, 3/8 ], [ 190, 206, -3/8 ], 
  [ 191, 148, -1/8*E(8)-1/8*E(8)^3 ], [ 191, 156, -3/8*E(8)-3/8*E(8)^3 ], [ 191, 159, 1/8*E(8)+1/8*E(8)^3 ]
    , [ 191, 167, 3/8*E(8)+3/8*E(8)^3 ], [ 191, 173, 1/8 ], [ 191, 175, -1/8 ], [ 191, 180, 3/8 ], 
  [ 191, 181, -1/8 ], [ 191, 182, -3/8 ], [ 191, 184, -1/8 ], [ 191, 188, -3/8 ], [ 191, 191, -1/8 ], 
  [ 191, 192, -3/8 ], [ 191, 195, 1/16*E(8)+1/16*E(8)^3 ], [ 191, 198, -3/8 ], 
  [ 191, 200, 1/16*E(8)+1/16*E(8)^3 ], [ 191, 201, 3/16*E(8)+3/16*E(8)^3 ], 
  [ 191, 206, 3/16*E(8)+3/16*E(8)^3 ], [ 192, 148, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 192, 156, 1/8*E(8)+1/8*E(8)^3 ], [ 192, 159, 1/8*E(8)+1/8*E(8)^3 ], [ 192, 167, -1/8*E(8)-1/8*E(8)^3 ],
  [ 192, 173, -1/8 ], [ 192, 175, -1/8 ], [ 192, 180, 1/8 ], [ 192, 181, -1/8 ], [ 192, 182, 1/8 ], 
  [ 192, 184, -1/8 ], [ 192, 188, 1/8 ], [ 192, 191, -1/8 ], [ 192, 192, 1/8 ], 
  [ 192, 195, -1/16*E(8)-1/16*E(8)^3 ], [ 192, 198, 1/8 ], [ 192, 200, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 192, 201, 1/16*E(8)+1/16*E(8)^3 ], [ 192, 206, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 193, 202, -1/2*E(8)-1/2*E(8)^3 ], [ 193, 207, 1/2*E(8)+1/2*E(8)^3 ], [ 193, 209, 1/4 ], 
  [ 193, 214, -1/4 ], [ 193, 217, 1/4 ], [ 193, 221, -1/4 ], [ 194, 208, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 194, 222, -1/2 ], [ 194, 225, -1/2 ], [ 195, 141, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 195, 149, 3/2*E(8)+3/2*E(8)^3 ], [ 195, 154, -1/4 ], [ 195, 162, -3/4 ], [ 195, 166, -1/4 ], 
  [ 195, 174, -3/4 ], [ 196, 175, 1/8*E(8)+1/8*E(8)^3 ], [ 196, 181, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 196, 182, 3/8*E(8)+3/8*E(8)^3 ], [ 196, 184, 1/8*E(8)+1/8*E(8)^3 ], [ 196, 188, -3/8*E(8)-3/8*E(8)^3 ],
  [ 196, 191, -1/8*E(8)-1/8*E(8)^3 ], [ 196, 192, 3/8*E(8)+3/8*E(8)^3 ], [ 196, 195, 1/8 ], 
  [ 196, 198, -3/8*E(8)-3/8*E(8)^3 ], [ 196, 200, -1/8 ], [ 196, 201, 3/8 ], [ 196, 206, -3/8 ], 
  [ 197, 175, 1/8*E(8)+1/8*E(8)^3 ], [ 197, 181, -1/8*E(8)-1/8*E(8)^3 ], [ 197, 182, -1/8*E(8)-1/8*E(8)^3 ]
    , [ 197, 184, 1/8*E(8)+1/8*E(8)^3 ], [ 197, 188, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 197, 191, -1/8*E(8)-1/8*E(8)^3 ], [ 197, 192, -1/8*E(8)-1/8*E(8)^3 ], [ 197, 195, -1/8 ], 
  [ 197, 198, 1/8*E(8)+1/8*E(8)^3 ], [ 197, 200, 1/8 ], [ 197, 201, 1/8 ], [ 197, 206, -1/8 ], 
  [ 198, 148, 1/8*E(8)+1/8*E(8)^3 ], [ 198, 156, -1/8*E(8)-1/8*E(8)^3 ], [ 198, 159, -1/8*E(8)-1/8*E(8)^3 ]
    , [ 198, 167, 1/8*E(8)+1/8*E(8)^3 ], [ 198, 173, -1/8 ], [ 198, 175, 1/8 ], [ 198, 180, 1/8 ], 
  [ 198, 181, 1/8 ], [ 198, 182, -1/8 ], [ 198, 184, 1/8 ], [ 198, 188, -1/8 ], [ 198, 191, 1/8 ], 
  [ 198, 192, -1/8 ], [ 198, 195, -1/16*E(8)-1/16*E(8)^3 ], [ 198, 198, -1/8 ], 
  [ 198, 200, -1/16*E(8)-1/16*E(8)^3 ], [ 198, 201, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 198, 206, 1/16*E(8)+1/16*E(8)^3 ], [ 199, 230, E(8)+E(8)^3 ], [ 199, 234, -E(8)-E(8)^3 ], 
  [ 200, 141, -1/2*E(8)-1/2*E(8)^3 ], [ 200, 149, -3/2*E(8)-3/2*E(8)^3 ], [ 200, 154, -1/4 ], 
  [ 200, 162, -3/4 ], [ 200, 166, -1/4 ], [ 200, 174, -3/4 ], [ 201, 141, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 201, 149, 1/2*E(8)+1/2*E(8)^3 ], [ 201, 154, 1/4 ], [ 201, 162, -1/4 ], [ 201, 166, 1/4 ], 
  [ 201, 174, -1/4 ], [ 202, 209, -1/8*E(8)-1/8*E(8)^3 ], [ 202, 214, -3/8*E(8)-3/8*E(8)^3 ], 
  [ 202, 217, 1/8*E(8)+1/8*E(8)^3 ], [ 202, 221, 3/8*E(8)+3/8*E(8)^3 ], [ 203, 175, -1/8*E(8)-1/8*E(8)^3 ],
  [ 203, 181, 1/8*E(8)+1/8*E(8)^3 ], [ 203, 182, 1/8*E(8)+1/8*E(8)^3 ], [ 203, 184, -1/8*E(8)-1/8*E(8)^3 ],
  [ 203, 188, -1/8*E(8)-1/8*E(8)^3 ], [ 203, 191, 1/8*E(8)+1/8*E(8)^3 ], [ 203, 192, 1/8*E(8)+1/8*E(8)^3 ],
  [ 203, 195, -1/8 ], [ 203, 198, -1/8*E(8)-1/8*E(8)^3 ], [ 203, 200, 1/8 ], [ 203, 201, 1/8 ], 
  [ 203, 206, -1/8 ], [ 204, 208, 1/2*E(8)+1/2*E(8)^3 ], [ 204, 222, 1/2 ], [ 204, 225, 1/2 ], 
  [ 205, 179, 1/4*E(8)+1/4*E(8)^3 ], [ 205, 186, 1/4*E(8)+1/4*E(8)^3 ], [ 205, 187, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 205, 193, 3/4*E(8)+3/4*E(8)^3 ], [ 206, 141, 1/2*E(8)+1/2*E(8)^3 ], [ 206, 149, -1/2*E(8)-1/2*E(8)^3 ],
  [ 206, 154, 1/4 ], [ 206, 162, -1/4 ], [ 206, 166, 1/4 ], [ 206, 174, -1/4 ], 
  [ 207, 209, 1/8*E(8)+1/8*E(8)^3 ], [ 207, 214, -1/8*E(8)-1/8*E(8)^3 ], [ 207, 217, -1/8*E(8)-1/8*E(8)^3 ]
    , [ 207, 221, 1/8*E(8)+1/8*E(8)^3 ], [ 208, 222, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 208, 225, 1/2*E(8)+1/2*E(8)^3 ], [ 209, 179, 1/4 ], [ 209, 186, -1/4 ], [ 209, 187, 3/4 ], 
  [ 209, 193, -3/4 ], [ 209, 205, -1/4*E(8)-1/4*E(8)^3 ], [ 209, 210, -3/4*E(8)-3/4*E(8)^3 ], 
  [ 210, 179, -1/4*E(8)-1/4*E(8)^3 ], [ 210, 186, -1/4*E(8)-1/4*E(8)^3 ], [ 210, 187, 1/4*E(8)+1/4*E(8)^3 ]
    , [ 210, 193, 1/4*E(8)+1/4*E(8)^3 ], [ 211, 194, E(8)+E(8)^3 ], [ 211, 204, -E(8)-E(8)^3 ], 
  [ 212, 230, -1/2 ], [ 212, 234, -1/2 ], [ 212, 236, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 213, 154, 1/8*E(8)+1/8*E(8)^3 ], [ 213, 162, 3/8*E(8)+3/8*E(8)^3 ], [ 213, 166, -1/8*E(8)-1/8*E(8)^3 ],
  [ 213, 174, -3/8*E(8)-3/8*E(8)^3 ], [ 214, 179, -1/4 ], [ 214, 186, 1/4 ], [ 214, 187, 1/4 ], 
  [ 214, 193, -1/4 ], [ 214, 205, 1/4*E(8)+1/4*E(8)^3 ], [ 214, 210, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 215, 212, -1 ], [ 215, 216, -1 ], [ 215, 219, 1/4*E(8)+1/4*E(8)^3 ], [ 215, 223, 1/4*E(8)+1/4*E(8)^3 ],
  [ 215, 224, 1/4*E(8)+1/4*E(8)^3 ], [ 215, 227, 1/4*E(8)+1/4*E(8)^3 ], [ 216, 230, 1/2 ], 
  [ 216, 234, 1/2 ], [ 216, 236, 1/4*E(8)+1/4*E(8)^3 ], [ 217, 179, -1/4 ], [ 217, 186, 1/4 ], 
  [ 217, 187, -3/4 ], [ 217, 193, 3/4 ], [ 217, 205, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 217, 210, -3/4*E(8)-3/4*E(8)^3 ], [ 218, 154, -1/8*E(8)-1/8*E(8)^3 ], [ 218, 162, 1/8*E(8)+1/8*E(8)^3 ]
    , [ 218, 166, 1/8*E(8)+1/8*E(8)^3 ], [ 218, 174, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 219, 212, 1/2*E(8)+1/2*E(8)^3 ], [ 219, 216, -1/2*E(8)-1/2*E(8)^3 ], [ 219, 219, 1/4 ], 
  [ 219, 223, -1/4 ], [ 219, 224, 1/4 ], [ 219, 227, -1/4 ], [ 219, 228, -1 ], 
  [ 219, 232, 1/4*E(8)+1/4*E(8)^3 ], [ 219, 235, -1/4*E(8)-1/4*E(8)^3 ], [ 220, 212, 1 ], [ 220, 216, 1 ], 
  [ 220, 219, 1/4*E(8)+1/4*E(8)^3 ], [ 220, 223, 1/4*E(8)+1/4*E(8)^3 ], [ 220, 224, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 220, 227, 1/4*E(8)+1/4*E(8)^3 ], [ 221, 179, 1/4 ], [ 221, 186, -1/4 ], [ 221, 187, -1/4 ], 
  [ 221, 193, 1/4 ], [ 221, 205, 1/4*E(8)+1/4*E(8)^3 ], [ 221, 210, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 222, 194, 1/2 ], [ 222, 204, 1/2 ], [ 222, 211, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 223, 212, -1/2*E(8)-1/2*E(8)^3 ], [ 223, 216, 1/2*E(8)+1/2*E(8)^3 ], [ 223, 219, 1/4 ], 
  [ 223, 223, -1/4 ], [ 223, 224, 1/4 ], [ 223, 227, -1/4 ], [ 223, 228, 1 ], 
  [ 223, 232, 1/4*E(8)+1/4*E(8)^3 ], [ 223, 235, -1/4*E(8)-1/4*E(8)^3 ], [ 224, 212, -1/2*E(8)-1/2*E(8)^3 ]
    , [ 224, 216, 1/2*E(8)+1/2*E(8)^3 ], [ 224, 219, -1/4 ], [ 224, 223, 1/4 ], [ 224, 224, -1/4 ], 
  [ 224, 227, 1/4 ], [ 224, 228, -1 ], [ 224, 232, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 224, 235, -1/4*E(8)-1/4*E(8)^3 ], [ 225, 194, -1/2 ], [ 225, 204, -1/2 ], 
  [ 225, 211, 1/4*E(8)+1/4*E(8)^3 ], [ 226, 231, -1/2*E(8)-1/2*E(8)^3 ], [ 226, 237, -1/2 ], 
  [ 226, 238, 1/2 ], [ 227, 212, 1/2*E(8)+1/2*E(8)^3 ], [ 227, 216, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 227, 219, -1/4 ], [ 227, 223, 1/4 ], [ 227, 224, -1/4 ], [ 227, 227, 1/4 ], [ 227, 228, 1 ], 
  [ 227, 232, 1/4*E(8)+1/4*E(8)^3 ], [ 227, 235, -1/4*E(8)-1/4*E(8)^3 ], [ 228, 219, -1/4 ], 
  [ 228, 223, -1/4 ], [ 228, 224, 1/4 ], [ 228, 227, 1/4 ], [ 229, 231, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 229, 237, -1/2 ], [ 229, 238, 1/2 ], [ 230, 219, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 230, 223, -1/8*E(8)-1/8*E(8)^3 ], [ 230, 224, -1/8*E(8)-1/8*E(8)^3 ], [ 230, 227, 1/8*E(8)+1/8*E(8)^3 ]
    , [ 230, 232, -1/4 ], [ 230, 235, -1/4 ], [ 231, 237, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 231, 238, 1/2*E(8)+1/2*E(8)^3 ], [ 232, 199, 1/2*E(8)+1/2*E(8)^3 ], [ 232, 215, -1/2 ], 
  [ 232, 220, -1/2 ], [ 233, 226, E(8)+E(8)^3 ], [ 233, 229, E(8)+E(8)^3 ], 
  [ 234, 219, 1/8*E(8)+1/8*E(8)^3 ], [ 234, 223, -1/8*E(8)-1/8*E(8)^3 ], [ 234, 224, -1/8*E(8)-1/8*E(8)^3 ]
    , [ 234, 227, 1/8*E(8)+1/8*E(8)^3 ], [ 234, 232, 1/4 ], [ 234, 235, 1/4 ], 
  [ 235, 199, 1/2*E(8)+1/2*E(8)^3 ], [ 235, 215, 1/2 ], [ 235, 220, 1/2 ], 
  [ 236, 215, 1/2*E(8)+1/2*E(8)^3 ], [ 236, 220, -1/2*E(8)-1/2*E(8)^3 ], [ 237, 226, 1/2 ], 
  [ 237, 229, -1/2 ], [ 237, 233, 1/4*E(8)+1/4*E(8)^3 ], [ 238, 226, 1/2 ], [ 238, 229, -1/2 ], 
  [ 238, 233, -1/4*E(8)-1/4*E(8)^3 ], [ 239, 239, -1/2 ], [ 239, 240, 3/2 ], [ 240, 239, -1/2 ], 
  [ 240, 240, -1/2 ], [ 241, 6, -1/4 ], [ 241, 25, 1/4*E(8)+1/4*E(8)^3 ], [ 241, 33, 1/4*E(8)+1/4*E(8)^3 ],
  [ 241, 126, -1/4 ], [ 241, 145, -1/4*E(8)-1/4*E(8)^3 ], [ 241, 153, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 241, 241, 1/2 ], [ 241, 242, 1/2 ], [ 241, 243, -1/2 ], [ 241, 244, -1/2 ], [ 241, 245, 3/4 ], 
  [ 241, 247, 1/4 ], [ 242, 241, 1 ], [ 242, 242, 1 ], [ 242, 243, -1 ], [ 242, 244, -1 ], [ 242, 245, 1 ],
  [ 242, 247, 1 ], [ 243, 6, -1/4 ], [ 243, 9, 1/2 ], [ 243, 25, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 243, 33, 1/4*E(8)+1/4*E(8)^3 ], [ 243, 126, -1/4 ], [ 243, 129, 1/2 ], 
  [ 243, 145, -1/4*E(8)-1/4*E(8)^3 ], [ 243, 153, -1/4*E(8)-1/4*E(8)^3 ], [ 243, 241, 1 ], 
  [ 243, 242, 3/2 ], [ 243, 243, -1 ], [ 243, 244, -1 ], [ 243, 245, 3/4 ], [ 243, 247, 5/4 ], 
  [ 244, 6, -1/2 ], [ 244, 9, 1/2 ], [ 244, 126, -1/2 ], [ 244, 129, 1/2 ], [ 244, 241, 3/2 ], 
  [ 244, 242, 2 ], [ 244, 243, -3/2 ], [ 244, 244, -3/2 ], [ 244, 245, 3/2 ], [ 244, 247, 3/2 ], 
  [ 245, 9, 1/2 ], [ 245, 129, 1/2 ], [ 245, 241, 3/2 ], [ 245, 242, 1 ], [ 245, 243, -3/2 ], 
  [ 245, 244, -1/2 ], [ 245, 245, 1 ], [ 245, 247, 1 ], [ 246, 9, 1/4 ], [ 246, 10, -1/2*E(8)-1/2*E(8)^3 ],
  [ 246, 23, -1/2*E(8)-1/2*E(8)^3 ], [ 246, 129, 1/4 ], [ 246, 130, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 246, 143, 1/8*E(8)+1/8*E(8)^3 ], [ 246, 241, 3/4 ], [ 246, 242, 1/2 ], [ 246, 243, -3/4 ], 
  [ 246, 244, -1/4 ], [ 246, 245, 1/2 ], [ 246, 247, 1 ], [ 247, 247, 1 ], [ 248, 8, -1/4 ], 
  [ 248, 128, -3/4 ], [ 248, 247, 3/4 ], [ 248, 248, -1/2 ] ] ],


[ [ 0, 1, 0, 0, 0, 0, 1, 0 ],
  [ 248, [ 1, 125, 1 ], [ 1, 132, -1/4*E(8)+1/4*E(8)^3 ], [ 1, 133, -1 ], [ 1, 139, 1/4*E(8)-1/4*E(8)^3 ], [ 1, 140, 1/4*E(8)-1/4*E(8)^3 ], [ 1, 147, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 2, 37, 1 ], [ 2, 44, 1 ], [ 2, 45, -1 ], [ 2, 51, -1 ], [ 3, 3, 1/4 ], [ 3, 4, -1/2*E(8)+1/2*E(8)^3 ], [ 3, 11, 1/2*E(8)-1/2*E(8)^3 ], [ 3, 123, 1/4 ], 
  [ 3, 124, 1/8*E(8)-1/8*E(8)^3 ], [ 3, 131, -1/8*E(8)+1/8*E(8)^3 ], [ 3, 241, -1/4 ], [ 3, 242, -1/2 ], [ 3, 244, 3/4 ], [ 3, 245, -1/2 ], [ 4, 3, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 4, 123, 1/8*E(8)-1/8*E(8)^3 ], [ 4, 124, 1/8 ], [ 4, 131, 1/8 ], [ 4, 241, -1/8*E(8)+1/8*E(8)^3 ], [ 4, 243, 1/4*E(8)-1/4*E(8)^3 ], [ 4, 244, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 5, 12, 1/2*E(8)-1/2*E(8)^3 ], [ 5, 19, 1/2*E(8)-1/2*E(8)^3 ], [ 5, 20, 1/2*E(8)-1/2*E(8)^3 ], [ 5, 27, 1/2*E(8)-1/2*E(8)^3 ], [ 5, 121, 1 ], [ 5, 129, 1 ], 
  [ 6, 6, 1/4 ], [ 6, 24, 1/4*E(8)-1/4*E(8)^3 ], [ 6, 31, 1/4*E(8)-1/4*E(8)^3 ], [ 6, 126, 1/4 ], [ 6, 144, -1/4*E(8)+1/4*E(8)^3 ], [ 6, 151, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 6, 241, 1/2 ], [ 6, 242, -1/2 ], [ 6, 245, 1/4 ], [ 6, 247, -1/4 ], [ 7, 7, 1/4 ], [ 7, 14, 1/4 ], [ 7, 15, 1/4 ], [ 7, 22, 1/4 ], [ 7, 39, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 7, 47, -1/4*E(8)+1/4*E(8)^3 ], [ 8, 8, -1/4 ], [ 8, 128, 9/4 ], [ 8, 247, 3/4 ], [ 8, 248, -3/2 ], [ 9, 125, 1 ], [ 9, 132, 1/4*E(8)-1/4*E(8)^3 ], [ 9, 133, -1 ], 
  [ 9, 139, -1/4*E(8)+1/4*E(8)^3 ], [ 9, 140, -1/4*E(8)+1/4*E(8)^3 ], [ 9, 147, 1/4*E(8)-1/4*E(8)^3 ], [ 10, 30, 1/4 ], [ 10, 37, -1/4*E(8)+1/4*E(8)^3 ], [ 10, 38, -1/4 ], 
  [ 10, 44, 1/4*E(8)-1/4*E(8)^3 ], [ 10, 45, 1/4*E(8)-1/4*E(8)^3 ], [ 10, 51, -1/4*E(8)+1/4*E(8)^3 ], [ 11, 3, 1/8*E(8)-1/8*E(8)^3 ], [ 11, 123, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 11, 124, 1/8 ], [ 11, 131, 1/8 ], [ 11, 241, 1/8*E(8)-1/8*E(8)^3 ], [ 11, 243, -1/4*E(8)+1/4*E(8)^3 ], [ 11, 244, 1/8*E(8)-1/8*E(8)^3 ], [ 12, 5, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 12, 12, 1/4 ], [ 12, 13, 1/8*E(8)-1/8*E(8)^3 ], [ 12, 19, -1/4 ], [ 12, 20, 1/4 ], [ 12, 27, -1/4 ], [ 12, 121, -1/4*E(8)+1/4*E(8)^3 ], [ 12, 129, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 12, 136, 1/4 ], [ 13, 12, 1/2*E(8)-1/2*E(8)^3 ], [ 13, 19, 1/2*E(8)-1/2*E(8)^3 ], [ 13, 20, 1/2*E(8)-1/2*E(8)^3 ], [ 13, 27, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 13, 121, -1 ], [ 13, 129, -1 ], [ 14, 7, 1/4 ], [ 14, 14, 1/4 ], [ 14, 15, 1/4 ], [ 14, 22, 1/4 ], [ 14, 39, 1/4*E(8)-1/4*E(8)^3 ], [ 14, 47, 1/4*E(8)-1/4*E(8)^3 ], [ 15, 7, 3/4 ], 
  [ 15, 14, 3/4 ], [ 15, 15, -1/4 ], [ 15, 22, -1/4 ], [ 15, 39, -3/4*E(8)+3/4*E(8)^3 ], [ 15, 47, 1/4*E(8)-1/4*E(8)^3 ], [ 16, 132, 1/4 ], [ 16, 139, 1/4 ], 
  [ 16, 140, -1/4 ], [ 16, 147, -1/4 ], [ 17, 30, 1/4 ], [ 17, 37, 1/4*E(8)-1/4*E(8)^3 ], [ 17, 38, -1/4 ], [ 17, 44, -1/4*E(8)+1/4*E(8)^3 ], [ 17, 45, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 17, 51, 1/4*E(8)-1/4*E(8)^3 ], [ 18, 18, -1/4 ], [ 18, 25, -1/4 ], [ 18, 26, 1/4 ], [ 18, 32, 1/2*E(8)-1/2*E(8)^3 ], [ 18, 33, 1/4 ], [ 18, 40, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 18, 52, -1/4*E(8)+1/4*E(8)^3 ], [ 18, 57, -1/4*E(8)+1/4*E(8)^3 ], [ 18, 63, 1 ], [ 19, 5, 1/8*E(8)-1/8*E(8)^3 ], [ 19, 12, -1/4 ], [ 19, 13, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 19, 19, 1/4 ], [ 19, 20, -1/4 ], [ 19, 27, 1/4 ], [ 19, 121, 1/4*E(8)-1/4*E(8)^3 ], [ 19, 129, -1/4*E(8)+1/4*E(8)^3 ], [ 19, 136, 1/4 ], [ 20, 5, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 20, 12, 1/4 ], [ 20, 13, 1/8*E(8)-1/8*E(8)^3 ], [ 20, 19, -1/4 ], [ 20, 20, 1/4 ], [ 20, 27, -1/4 ], [ 20, 121, 1/4*E(8)-1/4*E(8)^3 ], [ 20, 129, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 20, 136, -1/4 ], [ 21, 28, 1/2*E(8)-1/2*E(8)^3 ], [ 21, 35, 1/2*E(8)-1/2*E(8)^3 ], [ 21, 36, 1/2*E(8)-1/2*E(8)^3 ], [ 21, 43, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 22, 7, 3/4 ], [ 22, 14, 3/4 ], [ 22, 15, -1/4 ], [ 22, 22, -1/4 ], [ 22, 39, 3/4*E(8)-3/4*E(8)^3 ], [ 22, 47, -1/4*E(8)+1/4*E(8)^3 ], [ 23, 23, -1 ], [ 24, 6, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 24, 126, -1/4*E(8)+1/4*E(8)^3 ], [ 24, 144, 1/2 ], [ 24, 151, -1/2 ], [ 24, 245, 1/4*E(8)-1/4*E(8)^3 ], [ 24, 246, -1/2*E(8)+1/2*E(8)^3 ], [ 24, 247, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 25, 18, -1/4 ], [ 25, 25, -1/4 ], [ 25, 26, 1/4 ], [ 25, 32, -1/2*E(8)+1/2*E(8)^3 ], [ 25, 33, 1/4 ], [ 25, 40, 1/2*E(8)-1/2*E(8)^3 ], [ 25, 52, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 25, 57, -1/4*E(8)+1/4*E(8)^3 ], [ 25, 63, -1 ], [ 26, 18, 1/4 ], [ 26, 25, 1/4 ], [ 26, 26, -1/4 ], [ 26, 32, -1/2*E(8)+1/2*E(8)^3 ], [ 26, 33, -1/4 ], 
  [ 26, 40, 1/2*E(8)-1/2*E(8)^3 ], [ 26, 52, -1/4*E(8)+1/4*E(8)^3 ], [ 26, 57, -1/4*E(8)+1/4*E(8)^3 ], [ 26, 63, 1 ], [ 27, 5, 1/8*E(8)-1/8*E(8)^3 ], [ 27, 12, -1/4 ], 
  [ 27, 13, 1/8*E(8)-1/8*E(8)^3 ], [ 27, 19, 1/4 ], [ 27, 20, -1/4 ], [ 27, 27, 1/4 ], [ 27, 121, -1/4*E(8)+1/4*E(8)^3 ], [ 27, 129, 1/4*E(8)-1/4*E(8)^3 ], [ 27, 136, -1/4 ], 
  [ 28, 21, 1/8*E(8)-1/8*E(8)^3 ], [ 28, 28, 1/4 ], [ 28, 29, 1/8*E(8)-1/8*E(8)^3 ], [ 28, 35, -1/4 ], [ 28, 36, 1/4 ], [ 28, 43, -1/4 ], [ 29, 28, 3/2*E(8)-3/2*E(8)^3 ], 
  [ 29, 35, 3/2*E(8)-3/2*E(8)^3 ], [ 29, 36, -1/2*E(8)+1/2*E(8)^3 ], [ 29, 43, -1/2*E(8)+1/2*E(8)^3 ], [ 30, 10, 1 ], [ 30, 17, 1 ], [ 30, 37, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 30, 44, 1/2*E(8)-1/2*E(8)^3 ], [ 30, 45, 1/2*E(8)-1/2*E(8)^3 ], [ 30, 51, 1/2*E(8)-1/2*E(8)^3 ], [ 31, 6, 1/4*E(8)-1/4*E(8)^3 ], [ 31, 126, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 31, 144, -1/2 ], [ 31, 151, 1/2 ], [ 31, 245, 1/4*E(8)-1/4*E(8)^3 ], [ 31, 246, -1/2*E(8)+1/2*E(8)^3 ], [ 31, 247, 1/4*E(8)-1/4*E(8)^3 ], [ 32, 18, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 32, 25, -1/8*E(8)+1/8*E(8)^3 ], [ 32, 26, -1/8*E(8)+1/8*E(8)^3 ], [ 32, 33, 1/8*E(8)-1/8*E(8)^3 ], [ 32, 52, 1/4 ], [ 32, 57, -1/4 ], [ 33, 18, 1/4 ], 
  [ 33, 25, 1/4 ], [ 33, 26, -1/4 ], [ 33, 32, 1/2*E(8)-1/2*E(8)^3 ], [ 33, 33, -1/4 ], [ 33, 40, -1/2*E(8)+1/2*E(8)^3 ], [ 33, 52, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 33, 57, -1/4*E(8)+1/4*E(8)^3 ], [ 33, 63, -1 ], [ 34, 59, -1/8*E(8)+1/8*E(8)^3 ], [ 34, 64, -1/8*E(8)+1/8*E(8)^3 ], [ 34, 66, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 34, 67, -1/8*E(8)+1/8*E(8)^3 ], [ 34, 70, 1/2 ], [ 34, 71, 1/8*E(8)-1/8*E(8)^3 ], [ 34, 72, -1/8*E(8)+1/8*E(8)^3 ], [ 34, 73, 1/8*E(8)-1/8*E(8)^3 ], [ 34, 76, -1/2 ], 
  [ 34, 77, 1/2 ], [ 34, 78, 1/8*E(8)-1/8*E(8)^3 ], [ 34, 83, -1/2 ], [ 35, 21, 1/8*E(8)-1/8*E(8)^3 ], [ 35, 28, -1/4 ], [ 35, 29, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 35, 35, 1/4 ], [ 35, 36, -1/4 ], [ 35, 43, 1/4 ], [ 36, 21, 3/8*E(8)-3/8*E(8)^3 ], [ 36, 28, 3/4 ], [ 36, 29, -1/8*E(8)+1/8*E(8)^3 ], [ 36, 35, -3/4 ], [ 36, 36, -1/4 ], 
  [ 36, 43, 1/4 ], [ 37, 2, 1/4 ], [ 37, 10, -1/4*E(8)+1/4*E(8)^3 ], [ 37, 17, 1/4*E(8)-1/4*E(8)^3 ], [ 37, 30, 1/8*E(8)-1/8*E(8)^3 ], [ 37, 37, -1/4 ], 
  [ 37, 38, 1/8*E(8)-1/8*E(8)^3 ], [ 37, 44, 1/4 ], [ 37, 45, -1/4 ], [ 37, 51, 1/4 ], [ 38, 10, -1 ], [ 38, 17, -1 ], [ 38, 37, 1/2*E(8)-1/2*E(8)^3 ], [ 38, 44, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 38, 45, 1/2*E(8)-1/2*E(8)^3 ], [ 38, 51, 1/2*E(8)-1/2*E(8)^3 ], [ 39, 7, -1/4*E(8)+1/4*E(8)^3 ], [ 39, 14, 1/4*E(8)-1/4*E(8)^3 ], [ 39, 15, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 39, 22, 1/4*E(8)-1/4*E(8)^3 ], [ 40, 18, -1/8*E(8)+1/8*E(8)^3 ], [ 40, 25, 1/8*E(8)-1/8*E(8)^3 ], [ 40, 26, 1/8*E(8)-1/8*E(8)^3 ], [ 40, 33, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 40, 52, 1/4 ], [ 40, 57, -1/4 ], [ 41, 59, -1/8*E(8)+1/8*E(8)^3 ], [ 41, 64, -1/8*E(8)+1/8*E(8)^3 ], [ 41, 66, 1/8*E(8)-1/8*E(8)^3 ], [ 41, 67, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 41, 70, -1/2 ], [ 41, 71, 1/8*E(8)-1/8*E(8)^3 ], [ 41, 72, -1/8*E(8)+1/8*E(8)^3 ], [ 41, 73, 1/8*E(8)-1/8*E(8)^3 ], [ 41, 76, 1/2 ], [ 41, 77, -1/2 ], 
  [ 41, 78, 1/8*E(8)-1/8*E(8)^3 ], [ 41, 83, 1/2 ], [ 42, 59, -3/8*E(8)+3/8*E(8)^3 ], [ 42, 64, -3/8*E(8)+3/8*E(8)^3 ], [ 42, 66, 3/8*E(8)-3/8*E(8)^3 ], [ 42, 67, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 42, 70, 3/2 ], [ 42, 71, 3/8*E(8)-3/8*E(8)^3 ], [ 42, 72, 1/8*E(8)-1/8*E(8)^3 ], [ 42, 73, -1/8*E(8)+1/8*E(8)^3 ], [ 42, 76, -3/2 ], [ 42, 77, -1/2 ], 
  [ 42, 78, -1/8*E(8)+1/8*E(8)^3 ], [ 42, 83, 1/2 ], [ 43, 21, 3/8*E(8)-3/8*E(8)^3 ], [ 43, 28, -3/4 ], [ 43, 29, -1/8*E(8)+1/8*E(8)^3 ], [ 43, 35, 3/4 ], [ 43, 36, 1/4 ], 
  [ 43, 43, -1/4 ], [ 44, 2, 1/4 ], [ 44, 10, 1/4*E(8)-1/4*E(8)^3 ], [ 44, 17, -1/4*E(8)+1/4*E(8)^3 ], [ 44, 30, 1/8*E(8)-1/8*E(8)^3 ], [ 44, 37, 1/4 ], 
  [ 44, 38, 1/8*E(8)-1/8*E(8)^3 ], [ 44, 44, -1/4 ], [ 44, 45, 1/4 ], [ 44, 51, -1/4 ], [ 45, 2, -1/4 ], [ 45, 10, 1/4*E(8)-1/4*E(8)^3 ], [ 45, 17, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 45, 30, 1/8*E(8)-1/8*E(8)^3 ], [ 45, 37, -1/4 ], [ 45, 38, 1/8*E(8)-1/8*E(8)^3 ], [ 45, 44, 1/4 ], [ 45, 45, -1/4 ], [ 45, 51, 1/4 ], 
  [ 46, 53, 1/2*E(8)-1/2*E(8)^3 ], [ 46, 58, 1/2*E(8)-1/2*E(8)^3 ], [ 46, 60, 1/2*E(8)-1/2*E(8)^3 ], [ 46, 65, 1/2*E(8)-1/2*E(8)^3 ], [ 47, 7, -3/4*E(8)+3/4*E(8)^3 ], 
  [ 47, 14, 3/4*E(8)-3/4*E(8)^3 ], [ 47, 15, 1/4*E(8)-1/4*E(8)^3 ], [ 47, 22, -1/4*E(8)+1/4*E(8)^3 ], [ 48, 48, -1 ], [ 49, 59, 1/8 ], [ 49, 64, -1/8 ], 
  [ 49, 66, -1/8 ], [ 49, 67, 1/8 ], [ 49, 71, 1/8 ], [ 49, 72, -1/8 ], [ 49, 73, -1/8 ], [ 49, 78, 1/8 ], [ 50, 59, -3/8*E(8)+3/8*E(8)^3 ], [ 50, 64, -3/8*E(8)+3/8*E(8)^3 ], 
  [ 50, 66, 3/8*E(8)-3/8*E(8)^3 ], [ 50, 67, 1/8*E(8)-1/8*E(8)^3 ], [ 50, 70, -3/2 ], [ 50, 71, 3/8*E(8)-3/8*E(8)^3 ], [ 50, 72, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 50, 73, -1/8*E(8)+1/8*E(8)^3 ], [ 50, 76, 3/2 ], [ 50, 77, 1/2 ], [ 50, 78, -1/8*E(8)+1/8*E(8)^3 ], [ 50, 83, -1/2 ], [ 51, 2, -1/4 ], [ 51, 10, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 51, 17, 1/4*E(8)-1/4*E(8)^3 ], [ 51, 30, 1/8*E(8)-1/8*E(8)^3 ], [ 51, 37, 1/4 ], [ 51, 38, 1/8*E(8)-1/8*E(8)^3 ], [ 51, 44, -1/4 ], 
  [ 51, 45, 1/4 ], [ 51, 51, -1/4 ], [ 52, 18, -1/4*E(8)+1/4*E(8)^3 ], [ 52, 25, -1/4*E(8)+1/4*E(8)^3 ], [ 52, 26, -1/4*E(8)+1/4*E(8)^3 ], [ 52, 32, 1 ], [ 52, 33, -1/4*E(8)+1/4*E(8)^3 ]
    , [ 52, 40, 1 ], [ 53, 46, 1/8*E(8)-1/8*E(8)^3 ], [ 53, 53, -1/4 ], [ 53, 54, 1/8*E(8)-1/8*E(8)^3 ], [ 53, 58, 1/4 ], [ 53, 60, -1/4 ], [ 53, 65, 1/4 ], 
  [ 54, 53, 3/2*E(8)-3/2*E(8)^3 ], [ 54, 58, 3/2*E(8)-3/2*E(8)^3 ], [ 54, 60, -1/2*E(8)+1/2*E(8)^3 ], [ 54, 65, -1/2*E(8)+1/2*E(8)^3 ], [ 55, 55, -1/4 ], [ 55, 61, 1/4 ], 
  [ 55, 62, -1/4 ], [ 55, 68, 1/4 ], [ 55, 82, 1/4*E(8)-1/4*E(8)^3 ], [ 55, 87, 1/4*E(8)-1/4*E(8)^3 ], [ 56, 59, 3/8 ], [ 56, 64, -3/8 ], [ 56, 66, -3/8 ], [ 56, 67, -1/8 ], 
  [ 56, 71, 3/8 ], [ 56, 72, 1/8 ], [ 56, 73, 1/8 ], [ 56, 78, -1/8 ], [ 57, 18, -1/4*E(8)+1/4*E(8)^3 ], [ 57, 25, -1/4*E(8)+1/4*E(8)^3 ], [ 57, 26, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 57, 32, -1 ], [ 57, 33, -1/4*E(8)+1/4*E(8)^3 ], [ 57, 40, -1 ], [ 58, 46, 1/8*E(8)-1/8*E(8)^3 ], [ 58, 53, 1/4 ], [ 58, 54, 1/8*E(8)-1/8*E(8)^3 ], [ 58, 58, -1/4 ], 
  [ 58, 60, 1/4 ], [ 58, 65, -1/4 ], [ 59, 34, -1/8*E(8)+1/8*E(8)^3 ], [ 59, 41, -1/8*E(8)+1/8*E(8)^3 ], [ 59, 42, -1/8*E(8)+1/8*E(8)^3 ], [ 59, 49, 1/2 ], 
  [ 59, 50, -1/8*E(8)+1/8*E(8)^3 ], [ 59, 56, 1/2 ], [ 59, 59, 1/8 ], [ 59, 64, 1/8 ], [ 59, 66, 1/8 ], [ 59, 67, 1/8 ], 
  [ 59, 70, -1/4*E(8)+1/4*E(8)^3 ], [ 59, 71, 1/8 ], [ 59, 72, 1/8 ], [ 59, 73, 1/8 ], [ 59, 76, -1/4*E(8)+1/4*E(8)^3 ], [ 59, 77, -1/4*E(8)+1/4*E(8)^3 ], [ 59, 78, 1/8 ], 
  [ 59, 83, -1/4*E(8)+1/4*E(8)^3 ], [ 60, 46, 3/8*E(8)-3/8*E(8)^3 ], [ 60, 53, -3/4 ], [ 60, 54, -1/8*E(8)+1/8*E(8)^3 ], [ 60, 58, 3/4 ], [ 60, 60, 1/4 ], 
  [ 60, 65, -1/4 ], [ 61, 55, 1/4 ], [ 61, 61, -1/4 ], [ 61, 62, 1/4 ], [ 61, 68, -1/4 ], [ 61, 82, 1/4*E(8)-1/4*E(8)^3 ], [ 61, 87, 1/4*E(8)-1/4*E(8)^3 ], [ 62, 55, -3/4 ], 
  [ 62, 61, 3/4 ], [ 62, 62, 1/4 ], [ 62, 68, -1/4 ], [ 62, 82, 3/4*E(8)-3/4*E(8)^3 ], [ 62, 87, -1/4*E(8)+1/4*E(8)^3 ], [ 63, 18, 1/4 ], [ 63, 25, -1/4 ], 
  [ 63, 26, 1/4 ], [ 63, 33, -1/4 ], [ 64, 34, -1/8*E(8)+1/8*E(8)^3 ], [ 64, 41, -1/8*E(8)+1/8*E(8)^3 ], [ 64, 42, -1/8*E(8)+1/8*E(8)^3 ], [ 64, 49, -1/2 ], 
  [ 64, 50, -1/8*E(8)+1/8*E(8)^3 ], [ 64, 56, -1/2 ], [ 64, 59, 1/8 ], [ 64, 64, 1/8 ], [ 64, 66, 1/8 ], [ 64, 67, 1/8 ], [ 64, 70, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 64, 71, 1/8 ], [ 64, 72, 1/8 ], [ 64, 73, 1/8 ], [ 64, 76, 1/4*E(8)-1/4*E(8)^3 ], [ 64, 77, 1/4*E(8)-1/4*E(8)^3 ], [ 64, 78, 1/8 ], [ 64, 83, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 65, 46, 3/8*E(8)-3/8*E(8)^3 ], [ 65, 53, 3/4 ], [ 65, 54, -1/8*E(8)+1/8*E(8)^3 ], [ 65, 58, -3/4 ], [ 65, 60, -1/4 ], [ 65, 65, 1/4 ], 
  [ 66, 34, 1/8*E(8)-1/8*E(8)^3 ], [ 66, 41, 1/8*E(8)-1/8*E(8)^3 ], [ 66, 42, 1/8*E(8)-1/8*E(8)^3 ], [ 66, 49, -1/2 ], [ 66, 50, 1/8*E(8)-1/8*E(8)^3 ], [ 66, 56, -1/2 ], [ 66, 59, 1/8 ],
  [ 66, 64, 1/8 ], [ 66, 66, 1/8 ], [ 66, 67, 1/8 ], [ 66, 70, -1/4*E(8)+1/4*E(8)^3 ], [ 66, 71, 1/8 ], [ 66, 72, 1/8 ], [ 66, 73, 1/8 ], [ 66, 76, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 66, 77, -1/4*E(8)+1/4*E(8)^3 ], [ 66, 78, 1/8 ], [ 66, 83, -1/4*E(8)+1/4*E(8)^3 ], [ 67, 34, -3/8*E(8)+3/8*E(8)^3 ], [ 67, 41, -3/8*E(8)+3/8*E(8)^3 ], 
  [ 67, 42, 1/8*E(8)-1/8*E(8)^3 ], [ 67, 49, 3/2 ], [ 67, 50, 1/8*E(8)-1/8*E(8)^3 ], [ 67, 56, -1/2 ], [ 67, 59, 3/8 ], [ 67, 64, 3/8 ], [ 67, 66, 3/8 ], [ 67, 67, -1/8 ], 
  [ 67, 70, -3/4*E(8)+3/4*E(8)^3 ], [ 67, 71, 3/8 ], [ 67, 72, -1/8 ], [ 67, 73, -1/8 ], [ 67, 76, -3/4*E(8)+3/4*E(8)^3 ], [ 67, 77, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 67, 78, -1/8 ], [ 67, 83, 1/4*E(8)-1/4*E(8)^3 ], [ 68, 55, 3/4 ], [ 68, 61, -3/4 ], [ 68, 62, -1/4 ], [ 68, 68, 1/4 ], [ 68, 82, 3/4*E(8)-3/4*E(8)^3 ], 
  [ 68, 87, -1/4*E(8)+1/4*E(8)^3 ], [ 69, 69, 1 ], [ 70, 34, 1/8 ], [ 70, 41, -1/8 ], [ 70, 42, 1/8 ], [ 70, 50, -1/8 ], [ 70, 59, -1/16*E(8)+1/16*E(8)^3 ], 
  [ 70, 64, 1/16*E(8)-1/16*E(8)^3 ], [ 70, 66, -1/16*E(8)+1/16*E(8)^3 ], [ 70, 67, -1/16*E(8)+1/16*E(8)^3 ], [ 70, 71, 1/16*E(8)-1/16*E(8)^3 ], [ 70, 72, 1/16*E(8)-1/16*E(8)^3 ], 
  [ 70, 73, -1/16*E(8)+1/16*E(8)^3 ], [ 70, 78, 1/16*E(8)-1/16*E(8)^3 ], [ 71, 34, 1/8*E(8)-1/8*E(8)^3 ], [ 71, 41, 1/8*E(8)-1/8*E(8)^3 ], [ 71, 42, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 71, 49, 1/2 ], [ 71, 50, 1/8*E(8)-1/8*E(8)^3 ], [ 71, 56, 1/2 ], [ 71, 59, 1/8 ], [ 71, 64, 1/8 ], [ 71, 66, 1/8 ], [ 71, 67, 1/8 ], [ 71, 70, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 71, 71, 1/8 ], [ 71, 72, 1/8 ], [ 71, 73, 1/8 ], [ 71, 76, 1/4*E(8)-1/4*E(8)^3 ], [ 71, 77, 1/4*E(8)-1/4*E(8)^3 ], [ 71, 78, 1/8 ], [ 71, 83, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 72, 34, -3/8*E(8)+3/8*E(8)^3 ], [ 72, 41, -3/8*E(8)+3/8*E(8)^3 ], [ 72, 42, 1/8*E(8)-1/8*E(8)^3 ], [ 72, 49, -3/2 ], [ 72, 50, 1/8*E(8)-1/8*E(8)^3 ], [ 72, 56, 1/2 ], 
  [ 72, 59, 3/8 ], [ 72, 64, 3/8 ], [ 72, 66, 3/8 ], [ 72, 67, -1/8 ], [ 72, 70, 3/4*E(8)-3/4*E(8)^3 ], [ 72, 71, 3/8 ], [ 72, 72, -1/8 ], [ 72, 73, -1/8 ], 
  [ 72, 76, 3/4*E(8)-3/4*E(8)^3 ], [ 72, 77, -1/4*E(8)+1/4*E(8)^3 ], [ 72, 78, -1/8 ], [ 72, 83, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 73, 34, 3/8*E(8)-3/8*E(8)^3 ], [ 73, 41, 3/8*E(8)-3/8*E(8)^3 ], [ 73, 42, -1/8*E(8)+1/8*E(8)^3 ], [ 73, 49, -3/2 ], [ 73, 50, -1/8*E(8)+1/8*E(8)^3 ], [ 73, 56, 1/2 ], 
  [ 73, 59, 3/8 ], [ 73, 64, 3/8 ], [ 73, 66, 3/8 ], [ 73, 67, -1/8 ], [ 73, 70, -3/4*E(8)+3/4*E(8)^3 ], [ 73, 71, 3/8 ], [ 73, 72, -1/8 ], [ 73, 73, -1/8 ], 
  [ 73, 76, -3/4*E(8)+3/4*E(8)^3 ], [ 73, 77, 1/4*E(8)-1/4*E(8)^3 ], [ 73, 78, -1/8 ], [ 73, 83, 1/4*E(8)-1/4*E(8)^3 ], [ 74, 92, -1/2*E(8)+1/2*E(8)^3 ], [ 74, 96, 1/2*E(8)-1/2*E(8)^3 ],
  [ 75, 75, 1/4 ], [ 75, 80, -1/4 ], [ 75, 81, 1/4 ], [ 75, 86, -1/4 ], [ 75, 97, 1/4*E(8)-1/4*E(8)^3 ], [ 75, 101, 1/4*E(8)-1/4*E(8)^3 ], [ 76, 34, -1/8 ], 
  [ 76, 41, 1/8 ], [ 76, 42, -1/8 ], [ 76, 50, 1/8 ], [ 76, 59, -1/16*E(8)+1/16*E(8)^3 ], [ 76, 64, 1/16*E(8)-1/16*E(8)^3 ], [ 76, 66, -1/16*E(8)+1/16*E(8)^3 ], 
  [ 76, 67, -1/16*E(8)+1/16*E(8)^3 ], [ 76, 71, 1/16*E(8)-1/16*E(8)^3 ], [ 76, 72, 1/16*E(8)-1/16*E(8)^3 ], [ 76, 73, -1/16*E(8)+1/16*E(8)^3 ], 
  [ 76, 78, 1/16*E(8)-1/16*E(8)^3 ], [ 77, 34, 3/8 ], [ 77, 41, -3/8 ], [ 77, 42, -1/8 ], [ 77, 50, 1/8 ], [ 77, 59, -3/16*E(8)+3/16*E(8)^3 ], [ 77, 64, 3/16*E(8)-3/16*E(8)^3 ], 
  [ 77, 66, -3/16*E(8)+3/16*E(8)^3 ], [ 77, 67, 1/16*E(8)-1/16*E(8)^3 ], [ 77, 71, 3/16*E(8)-3/16*E(8)^3 ], [ 77, 72, -1/16*E(8)+1/16*E(8)^3 ], [ 77, 73, 1/16*E(8)-1/16*E(8)^3 ], 
  [ 77, 78, -1/16*E(8)+1/16*E(8)^3 ], [ 78, 34, 3/8*E(8)-3/8*E(8)^3 ], [ 78, 41, 3/8*E(8)-3/8*E(8)^3 ], [ 78, 42, -1/8*E(8)+1/8*E(8)^3 ], [ 78, 49, 3/2 ], 
  [ 78, 50, -1/8*E(8)+1/8*E(8)^3 ], [ 78, 56, -1/2 ], [ 78, 59, 3/8 ], [ 78, 64, 3/8 ], [ 78, 66, 3/8 ], [ 78, 67, -1/8 ], [ 78, 70, 3/4*E(8)-3/4*E(8)^3 ], [ 78, 71, 3/8 ], 
  [ 78, 72, -1/8 ], [ 78, 73, -1/8 ], [ 78, 76, 3/4*E(8)-3/4*E(8)^3 ], [ 78, 77, -1/4*E(8)+1/4*E(8)^3 ], [ 78, 78, -1/8 ], [ 78, 83, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 79, 79, -1/2 ], [ 79, 84, -1/2 ], [ 79, 88, E(8)-E(8)^3 ], [ 80, 75, -1/4 ], [ 80, 80, 1/4 ], [ 80, 81, -1/4 ], [ 80, 86, 1/4 ], [ 80, 97, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 80, 101, 1/4*E(8)-1/4*E(8)^3 ], [ 81, 75, 3/4 ], [ 81, 80, -3/4 ], [ 81, 81, -1/4 ], [ 81, 86, 1/4 ], [ 81, 97, 3/4*E(8)-3/4*E(8)^3 ], 
  [ 81, 101, -1/4*E(8)+1/4*E(8)^3 ], [ 82, 55, 1/4*E(8)-1/4*E(8)^3 ], [ 82, 61, 1/4*E(8)-1/4*E(8)^3 ], [ 82, 62, 1/4*E(8)-1/4*E(8)^3 ], [ 82, 68, 1/4*E(8)-1/4*E(8)^3 ], [ 83, 34, -3/8 ],
  [ 83, 41, 3/8 ], [ 83, 42, 1/8 ], [ 83, 50, -1/8 ], [ 83, 59, -3/16*E(8)+3/16*E(8)^3 ], [ 83, 64, 3/16*E(8)-3/16*E(8)^3 ], [ 83, 66, -3/16*E(8)+3/16*E(8)^3 ], 
  [ 83, 67, 1/16*E(8)-1/16*E(8)^3 ], [ 83, 71, 3/16*E(8)-3/16*E(8)^3 ], [ 83, 72, -1/16*E(8)+1/16*E(8)^3 ], [ 83, 73, 1/16*E(8)-1/16*E(8)^3 ], [ 83, 78, -1/16*E(8)+1/16*E(8)^3 ], 
  [ 84, 79, -1/2 ], [ 84, 84, -1/2 ], [ 84, 88, -E(8)+E(8)^3 ], [ 85, 89, -1/2*E(8)+1/2*E(8)^3 ], [ 85, 93, 1/2*E(8)-1/2*E(8)^3 ], [ 85, 94, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 85, 98, 1/2*E(8)-1/2*E(8)^3 ], [ 86, 75, -3/4 ], [ 86, 80, 3/4 ], [ 86, 81, 1/4 ], [ 86, 86, -1/4 ], [ 86, 97, 3/4*E(8)-3/4*E(8)^3 ], [ 86, 101, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 87, 55, 3/4*E(8)-3/4*E(8)^3 ], [ 87, 61, 3/4*E(8)-3/4*E(8)^3 ], [ 87, 62, -1/4*E(8)+1/4*E(8)^3 ], [ 87, 68, -1/4*E(8)+1/4*E(8)^3 ], [ 88, 79, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 88, 84, -1/4*E(8)+1/4*E(8)^3 ], [ 89, 85, -1/8*E(8)+1/8*E(8)^3 ], [ 89, 89, 1/4 ], [ 89, 90, -1/8*E(8)+1/8*E(8)^3 ], [ 89, 93, 1/4 ], [ 89, 94, 1/4 ], 
  [ 89, 98, 1/4 ], [ 90, 89, -3/2*E(8)+3/2*E(8)^3 ], [ 90, 93, 3/2*E(8)-3/2*E(8)^3 ], [ 90, 94, 1/2*E(8)-1/2*E(8)^3 ], [ 90, 98, -1/2*E(8)+1/2*E(8)^3 ], [ 91, 104, -1/2*E(8)+1/2*E(8)^3 ]
    , [ 91, 107, 1/2*E(8)-1/2*E(8)^3 ], [ 92, 74, -1/2*E(8)+1/2*E(8)^3 ], [ 92, 92, -1/2 ], [ 92, 96, -1/2 ], [ 93, 85, 1/8*E(8)-1/8*E(8)^3 ], [ 93, 89, 1/4 ], 
  [ 93, 90, 1/8*E(8)-1/8*E(8)^3 ], [ 93, 93, 1/4 ], [ 93, 94, 1/4 ], [ 93, 98, 1/4 ], [ 94, 85, -3/8*E(8)+3/8*E(8)^3 ], [ 94, 89, 3/4 ], [ 94, 90, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 94, 93, 3/4 ], [ 94, 94, -1/4 ], [ 94, 98, -1/4 ], [ 95, 99, 1/2*E(8)-1/2*E(8)^3 ], [ 95, 102, -1/2*E(8)+1/2*E(8)^3 ], [ 95, 103, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 95, 105, 1/2*E(8)-1/2*E(8)^3 ], [ 95, 111, -1 ], [ 95, 114, 1 ], [ 96, 74, 1/2*E(8)-1/2*E(8)^3 ], [ 96, 92, -1/2 ], [ 96, 96, -1/2 ], 
  [ 97, 75, 1/4*E(8)-1/4*E(8)^3 ], [ 97, 80, 1/4*E(8)-1/4*E(8)^3 ], [ 97, 81, 1/4*E(8)-1/4*E(8)^3 ], [ 97, 86, 1/4*E(8)-1/4*E(8)^3 ], [ 98, 85, 3/8*E(8)-3/8*E(8)^3 ], [ 98, 89, 3/4 ], 
  [ 98, 90, -1/8*E(8)+1/8*E(8)^3 ], [ 98, 93, 3/4 ], [ 98, 94, -1/4 ], [ 98, 98, -1/4 ], [ 99, 95, 1/8*E(8)-1/8*E(8)^3 ], [ 99, 99, -1/4 ], [ 99, 100, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 99, 102, -1/4 ], [ 99, 103, 1/4 ], [ 99, 105, 1/4 ], [ 99, 109, -1/4 ], [ 99, 111, 1/4*E(8)-1/4*E(8)^3 ], [ 99, 114, 1/4*E(8)-1/4*E(8)^3 ], [ 100, 99, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 100, 102, 1/2*E(8)-1/2*E(8)^3 ], [ 100, 103, 1/2*E(8)-1/2*E(8)^3 ], [ 100, 105, -1/2*E(8)+1/2*E(8)^3 ], [ 100, 111, -1 ], [ 100, 114, 1 ], [ 101, 75, 3/4*E(8)-3/4*E(8)^3 ], 
  [ 101, 80, 3/4*E(8)-3/4*E(8)^3 ], [ 101, 81, -1/4*E(8)+1/4*E(8)^3 ], [ 101, 86, -1/4*E(8)+1/4*E(8)^3 ], [ 102, 95, -1/8*E(8)+1/8*E(8)^3 ], [ 102, 99, -1/4 ], 
  [ 102, 100, 1/8*E(8)-1/8*E(8)^3 ], [ 102, 102, -1/4 ], [ 102, 103, 1/4 ], [ 102, 105, 1/4 ], [ 102, 109, 1/4 ], [ 102, 111, 1/4*E(8)-1/4*E(8)^3 ], [ 102, 114, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 103, 95, -1/8*E(8)+1/8*E(8)^3 ], [ 103, 99, 1/4 ], [ 103, 100, 1/8*E(8)-1/8*E(8)^3 ], [ 103, 102, 1/4 ], [ 103, 103, -1/4 ], [ 103, 105, -1/4 ], [ 103, 109, -1/4 ], 
  [ 103, 111, 1/4*E(8)-1/4*E(8)^3 ], [ 103, 114, 1/4*E(8)-1/4*E(8)^3 ], [ 104, 91, -1/2*E(8)+1/2*E(8)^3 ], [ 104, 104, 1/2 ], [ 104, 107, 1/2 ], [ 105, 95, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 105, 99, 1/4 ], [ 105, 100, -1/8*E(8)+1/8*E(8)^3 ], [ 105, 102, 1/4 ], [ 105, 103, -1/4 ], [ 105, 105, -1/4 ], [ 105, 109, 1/4 ], [ 105, 111, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 105, 114, 1/4*E(8)-1/4*E(8)^3 ], [ 106, 106, 1/2 ], [ 106, 108, -1/2 ], [ 106, 110, E(8)-E(8)^3 ], [ 107, 91, 1/2*E(8)-1/2*E(8)^3 ], [ 107, 104, 1/2 ], [ 107, 107, 1/2 ], 
  [ 108, 106, -1/2 ], [ 108, 108, 1/2 ], [ 108, 110, E(8)-E(8)^3 ], [ 109, 99, -1 ], [ 109, 102, 1 ], [ 109, 103, -1 ], [ 109, 105, 1 ], [ 110, 106, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 110, 108, 1/4*E(8)-1/4*E(8)^3 ], [ 111, 95, -1/4 ], [ 111, 99, 1/4*E(8)-1/4*E(8)^3 ], [ 111, 100, -1/4 ], [ 111, 102, 1/4*E(8)-1/4*E(8)^3 ], [ 111, 103, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 111, 105, 1/4*E(8)-1/4*E(8)^3 ], [ 112, 117, -1/2*E(8)+1/2*E(8)^3 ], [ 112, 118, -1/2*E(8)+1/2*E(8)^3 ], [ 113, 113, -1/2 ], [ 113, 115, 1/2 ], [ 113, 116, E(8)-E(8)^3 ], 
  [ 114, 95, 1/4 ], [ 114, 99, 1/4*E(8)-1/4*E(8)^3 ], [ 114, 100, 1/4 ], [ 114, 102, 1/4*E(8)-1/4*E(8)^3 ], [ 114, 103, 1/4*E(8)-1/4*E(8)^3 ], [ 114, 105, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 115, 113, 1/2 ], [ 115, 115, -1/2 ], [ 115, 116, E(8)-E(8)^3 ], [ 116, 113, 1/4*E(8)-1/4*E(8)^3 ], [ 116, 115, 1/4*E(8)-1/4*E(8)^3 ], [ 117, 112, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 117, 117, -1/2 ], [ 117, 118, 1/2 ], [ 118, 112, -1/2*E(8)+1/2*E(8)^3 ], [ 118, 117, 1/2 ], [ 118, 118, -1/2 ], [ 119, 119, -1/2 ], [ 119, 120, 1/2 ], [ 120, 119, 3/2 ], 
  [ 120, 120, 1/2 ], [ 121, 5, 1/4 ], [ 121, 12, -1/4*E(8)+1/4*E(8)^3 ], [ 121, 13, -1/4 ], [ 121, 19, 1/4*E(8)-1/4*E(8)^3 ], [ 121, 20, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 121, 27, -1/4*E(8)+1/4*E(8)^3 ], [ 122, 157, 1/4 ], [ 122, 164, 1/4 ], [ 122, 165, -1/4 ], [ 122, 171, -1/4 ], [ 123, 3, 1/4 ], [ 123, 4, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 123, 11, -1/2*E(8)+1/2*E(8)^3 ], [ 123, 123, 1/4 ], [ 123, 124, -1/8*E(8)+1/8*E(8)^3 ], [ 123, 131, 1/8*E(8)-1/8*E(8)^3 ], [ 123, 241, -1/4 ], 
  [ 123, 242, -1/2 ], [ 123, 244, 3/4 ], [ 123, 245, -1/2 ], [ 124, 3, 1/2*E(8)-1/2*E(8)^3 ], [ 124, 4, 2 ], [ 124, 11, 2 ], [ 124, 123, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 124, 241, -1/2*E(8)+1/2*E(8)^3 ], [ 124, 243, E(8)-E(8)^3 ], [ 124, 244, -1/2*E(8)+1/2*E(8)^3 ], [ 125, 1, 1/4 ], [ 125, 9, 1/4 ], [ 125, 132, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 125, 139, 1/8*E(8)-1/8*E(8)^3 ], [ 125, 140, 1/8*E(8)-1/8*E(8)^3 ], [ 125, 147, 1/8*E(8)-1/8*E(8)^3 ], [ 126, 6, 1/4 ], [ 126, 24, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 126, 31, -1/4*E(8)+1/4*E(8)^3 ], [ 126, 126, 1/4 ], [ 126, 144, 1/4*E(8)-1/4*E(8)^3 ], [ 126, 151, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 126, 241, 1/2 ], [ 126, 242, -1/2 ], [ 126, 245, 1/4 ], [ 126, 247, -1/4 ], [ 127, 127, 1/4 ], [ 127, 134, 1/4 ], [ 127, 135, 3/4 ], [ 127, 142, 3/4 ], 
  [ 127, 159, -1/4*E(8)+1/4*E(8)^3 ], [ 127, 167, -3/4*E(8)+3/4*E(8)^3 ], [ 128, 8, 1/4 ], [ 128, 128, -1/4 ], [ 128, 247, 1/4 ], [ 128, 248, -1/2 ], [ 129, 5, 1/4 ], 
  [ 129, 12, 1/4*E(8)-1/4*E(8)^3 ], [ 129, 13, -1/4 ], [ 129, 19, -1/4*E(8)+1/4*E(8)^3 ], [ 129, 20, -1/4*E(8)+1/4*E(8)^3 ], [ 129, 27, 1/4*E(8)-1/4*E(8)^3 ], [ 130, 150, 1 ], 
  [ 130, 157, -1/4*E(8)+1/4*E(8)^3 ], [ 130, 158, -1 ], [ 130, 164, 1/4*E(8)-1/4*E(8)^3 ], [ 130, 165, 1/4*E(8)-1/4*E(8)^3 ], [ 130, 171, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 131, 3, -1/2*E(8)+1/2*E(8)^3 ], [ 131, 4, 2 ], [ 131, 11, 2 ], [ 131, 123, 1/2*E(8)-1/2*E(8)^3 ], [ 131, 241, 1/2*E(8)-1/2*E(8)^3 ], [ 131, 243, -E(8)+E(8)^3 ], 
  [ 131, 244, 1/2*E(8)-1/2*E(8)^3 ], [ 132, 1, -1/4*E(8)+1/4*E(8)^3 ], [ 132, 9, 1/4*E(8)-1/4*E(8)^3 ], [ 132, 16, 1 ], [ 132, 125, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 132, 132, 1/4 ], [ 132, 133, 1/2*E(8)-1/2*E(8)^3 ], [ 132, 139, -1/4 ], [ 132, 140, 1/4 ], [ 132, 147, -1/4 ], [ 133, 1, -1/4 ], [ 133, 9, -1/4 ], [ 133, 132, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 133, 139, 1/8*E(8)-1/8*E(8)^3 ], [ 133, 140, 1/8*E(8)-1/8*E(8)^3 ], [ 133, 147, 1/8*E(8)-1/8*E(8)^3 ], [ 134, 127, 1/4 ], [ 134, 134, 1/4 ], [ 134, 135, 3/4 ], 
  [ 134, 142, 3/4 ], [ 134, 159, 1/4*E(8)-1/4*E(8)^3 ], [ 134, 167, 3/4*E(8)-3/4*E(8)^3 ], [ 135, 127, 1/4 ], [ 135, 134, 1/4 ], [ 135, 135, -1/4 ], [ 135, 142, -1/4 ], 
  [ 135, 159, -1/4*E(8)+1/4*E(8)^3 ], [ 135, 167, 1/4*E(8)-1/4*E(8)^3 ], [ 136, 12, 1 ], [ 136, 19, 1 ], [ 136, 20, -1 ], [ 136, 27, -1 ], 
  [ 137, 150, 1 ], [ 137, 157, 1/4*E(8)-1/4*E(8)^3 ], [ 137, 158, -1 ], [ 137, 164, -1/4*E(8)+1/4*E(8)^3 ], [ 137, 165, -1/4*E(8)+1/4*E(8)^3 ], [ 137, 171, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 138, 138, -1/4 ], [ 138, 145, -1/4 ], [ 138, 146, 1/4 ], [ 138, 152, 1/8*E(8)-1/8*E(8)^3 ], [ 138, 153, 1/4 ], [ 138, 160, -1/8*E(8)+1/8*E(8)^3 ], [ 138, 172, -1/4*E(8)+1/4*E(8)^3 ],
  [ 138, 177, -1/4*E(8)+1/4*E(8)^3 ], [ 138, 183, 1/4 ], [ 139, 1, 1/4*E(8)-1/4*E(8)^3 ], [ 139, 9, -1/4*E(8)+1/4*E(8)^3 ], [ 139, 16, 1 ], [ 139, 125, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 139, 132, -1/4 ], [ 139, 133, 1/2*E(8)-1/2*E(8)^3 ], [ 139, 139, 1/4 ], [ 139, 140, -1/4 ], [ 139, 147, 1/4 ], [ 140, 1, 1/4*E(8)-1/4*E(8)^3 ], [ 140, 9, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 140, 16, -1 ], [ 140, 125, 1/2*E(8)-1/2*E(8)^3 ], [ 140, 132, 1/4 ], [ 140, 133, 1/2*E(8)-1/2*E(8)^3 ], [ 140, 139, -1/4 ], [ 140, 140, 1/4 ], [ 140, 147, -1/4 ], 
  [ 141, 148, 1/8*E(8)-1/8*E(8)^3 ], [ 141, 155, 1/8*E(8)-1/8*E(8)^3 ], [ 141, 156, 3/8*E(8)-3/8*E(8)^3 ], [ 141, 163, 3/8*E(8)-3/8*E(8)^3 ], [ 142, 127, 1/4 ], [ 142, 134, 1/4 ], 
  [ 142, 135, -1/4 ], [ 142, 142, -1/4 ], [ 142, 159, 1/4*E(8)-1/4*E(8)^3 ], [ 142, 167, -1/4*E(8)+1/4*E(8)^3 ], [ 143, 143, -1 ], [ 144, 6, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 144, 24, 1/2 ], [ 144, 31, -1/2 ], [ 144, 126, 1/4*E(8)-1/4*E(8)^3 ], [ 144, 245, 1/4*E(8)-1/4*E(8)^3 ], [ 144, 246, -1/2*E(8)+1/2*E(8)^3 ], [ 144, 247, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 145, 138, -1/4 ], [ 145, 145, -1/4 ], [ 145, 146, 1/4 ], [ 145, 152, -1/8*E(8)+1/8*E(8)^3 ], [ 145, 153, 1/4 ], [ 145, 160, 1/8*E(8)-1/8*E(8)^3 ], [ 145, 172, -1/4*E(8)+1/4*E(8)^3 ],
  [ 145, 177, -1/4*E(8)+1/4*E(8)^3 ], [ 145, 183, -1/4 ], [ 146, 138, 1/4 ], [ 146, 145, 1/4 ], [ 146, 146, -1/4 ], [ 146, 152, -1/8*E(8)+1/8*E(8)^3 ], [ 146, 153, -1/4 ], 
  [ 146, 160, 1/8*E(8)-1/8*E(8)^3 ], [ 146, 172, -1/4*E(8)+1/4*E(8)^3 ], [ 146, 177, -1/4*E(8)+1/4*E(8)^3 ], [ 146, 183, 1/4 ], [ 147, 1, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 147, 9, 1/4*E(8)-1/4*E(8)^3 ], [ 147, 16, -1 ], [ 147, 125, 1/2*E(8)-1/2*E(8)^3 ], [ 147, 132, -1/4 ], [ 147, 133, 1/2*E(8)-1/2*E(8)^3 ], [ 147, 139, 1/4 ], [ 147, 140, -1/4 ], 
  [ 147, 147, 1/4 ], [ 148, 141, 1/2*E(8)-1/2*E(8)^3 ], [ 148, 148, 1/4 ], [ 148, 149, 3/2*E(8)-3/2*E(8)^3 ], [ 148, 155, -1/4 ], [ 148, 156, 3/4 ], [ 148, 163, -3/4 ], 
  [ 149, 148, 1/8*E(8)-1/8*E(8)^3 ], [ 149, 155, 1/8*E(8)-1/8*E(8)^3 ], [ 149, 156, -1/8*E(8)+1/8*E(8)^3 ], [ 149, 163, -1/8*E(8)+1/8*E(8)^3 ], [ 150, 130, 1/4 ], [ 150, 137, 1/4 ], 
  [ 150, 157, 1/8*E(8)-1/8*E(8)^3 ], [ 150, 164, 1/8*E(8)-1/8*E(8)^3 ], [ 150, 165, 1/8*E(8)-1/8*E(8)^3 ], [ 150, 171, 1/8*E(8)-1/8*E(8)^3 ], [ 151, 6, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 151, 24, -1/2 ], [ 151, 31, 1/2 ], [ 151, 126, 1/4*E(8)-1/4*E(8)^3 ], [ 151, 245, 1/4*E(8)-1/4*E(8)^3 ], [ 151, 246, -1/2*E(8)+1/2*E(8)^3 ], [ 151, 247, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 152, 138, 1/2*E(8)-1/2*E(8)^3 ], [ 152, 145, -1/2*E(8)+1/2*E(8)^3 ], [ 152, 146, -1/2*E(8)+1/2*E(8)^3 ], [ 152, 153, 1/2*E(8)-1/2*E(8)^3 ], [ 152, 172, 1 ], 
  [ 152, 177, -1 ], [ 153, 138, 1/4 ], [ 153, 145, 1/4 ], [ 153, 146, -1/4 ], [ 153, 152, 1/8*E(8)-1/8*E(8)^3 ], [ 153, 153, -1/4 ], [ 153, 160, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 153, 172, -1/4*E(8)+1/4*E(8)^3 ], [ 153, 177, -1/4*E(8)+1/4*E(8)^3 ], [ 153, 183, -1/4 ], [ 154, 179, -1/8*E(8)+1/8*E(8)^3 ], [ 154, 184, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 154, 186, 1/8*E(8)-1/8*E(8)^3 ], [ 154, 187, -3/8*E(8)+3/8*E(8)^3 ], [ 154, 190, 1/8 ], [ 154, 191, 1/8*E(8)-1/8*E(8)^3 ], [ 154, 192, -3/8*E(8)+3/8*E(8)^3 ], 
  [ 154, 193, 3/8*E(8)-3/8*E(8)^3 ], [ 154, 196, -1/8 ], [ 154, 197, 3/8 ], [ 154, 198, 3/8*E(8)-3/8*E(8)^3 ], [ 154, 203, -3/8 ], [ 155, 141, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 155, 148, -1/4 ], [ 155, 149, 3/2*E(8)-3/2*E(8)^3 ], [ 155, 155, 1/4 ], [ 155, 156, -3/4 ], [ 155, 163, 3/4 ], [ 156, 141, 1/2*E(8)-1/2*E(8)^3 ], [ 156, 148, 1/4 ], 
  [ 156, 149, -1/2*E(8)+1/2*E(8)^3 ], [ 156, 155, -1/4 ], [ 156, 156, -1/4 ], [ 156, 163, 1/4 ], [ 157, 122, 1 ], [ 157, 130, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 157, 137, 1/4*E(8)-1/4*E(8)^3 ], [ 157, 150, 1/2*E(8)-1/2*E(8)^3 ], [ 157, 157, -1/4 ], [ 157, 158, 1/2*E(8)-1/2*E(8)^3 ], [ 157, 164, 1/4 ], [ 157, 165, -1/4 ], [ 157, 171, 1/4 ], 
  [ 158, 130, -1/4 ], [ 158, 137, -1/4 ], [ 158, 157, 1/8*E(8)-1/8*E(8)^3 ], [ 158, 164, 1/8*E(8)-1/8*E(8)^3 ], [ 158, 165, 1/8*E(8)-1/8*E(8)^3 ], [ 158, 171, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 159, 127, -1/4*E(8)+1/4*E(8)^3 ], [ 159, 134, 1/4*E(8)-1/4*E(8)^3 ], [ 159, 135, -3/4*E(8)+3/4*E(8)^3 ], [ 159, 142, 3/4*E(8)-3/4*E(8)^3 ], [ 160, 138, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 160, 145, 1/2*E(8)-1/2*E(8)^3 ], [ 160, 146, 1/2*E(8)-1/2*E(8)^3 ], [ 160, 153, -1/2*E(8)+1/2*E(8)^3 ], [ 160, 172, 1 ], [ 160, 177, -1 ], [ 161, 179, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 161, 184, -1/8*E(8)+1/8*E(8)^3 ], [ 161, 186, 1/8*E(8)-1/8*E(8)^3 ], [ 161, 187, -3/8*E(8)+3/8*E(8)^3 ], [ 161, 190, -1/8 ], [ 161, 191, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 161, 192, -3/8*E(8)+3/8*E(8)^3 ], [ 161, 193, 3/8*E(8)-3/8*E(8)^3 ], [ 161, 196, 1/8 ], [ 161, 197, -3/8 ], [ 161, 198, 3/8*E(8)-3/8*E(8)^3 ], [ 161, 203, 3/8 ], 
  [ 162, 179, -1/8*E(8)+1/8*E(8)^3 ], [ 162, 184, -1/8*E(8)+1/8*E(8)^3 ], [ 162, 186, 1/8*E(8)-1/8*E(8)^3 ], [ 162, 187, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 162, 190, 1/8 ], [ 162, 191, 1/8*E(8)-1/8*E(8)^3 ], [ 162, 192, 1/8*E(8)-1/8*E(8)^3 ], [ 162, 193, -1/8*E(8)+1/8*E(8)^3 ], [ 162, 196, -1/8 ], [ 162, 197, -1/8 ], 
  [ 162, 198, -1/8*E(8)+1/8*E(8)^3 ], [ 162, 203, 1/8 ], [ 163, 141, 1/2*E(8)-1/2*E(8)^3 ], [ 163, 148, -1/4 ], [ 163, 149, -1/2*E(8)+1/2*E(8)^3 ], [ 163, 155, 1/4 ], 
  [ 163, 156, 1/4 ], [ 163, 163, -1/4 ], [ 164, 122, 1 ], [ 164, 130, 1/4*E(8)-1/4*E(8)^3 ], [ 164, 137, -1/4*E(8)+1/4*E(8)^3 ], [ 164, 150, 1/2*E(8)-1/2*E(8)^3 ], [ 164, 157, 1/4 ], 
  [ 164, 158, 1/2*E(8)-1/2*E(8)^3 ], [ 164, 164, -1/4 ], [ 164, 165, 1/4 ], [ 164, 171, -1/4 ], [ 165, 122, -1 ], [ 165, 130, 1/4*E(8)-1/4*E(8)^3 ], [ 165, 137, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 165, 150, 1/2*E(8)-1/2*E(8)^3 ], [ 165, 157, -1/4 ], [ 165, 158, 1/2*E(8)-1/2*E(8)^3 ], [ 165, 164, 1/4 ], [ 165, 165, -1/4 ], [ 165, 171, 1/4 ], [ 166, 173, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 166, 178, 1/8*E(8)-1/8*E(8)^3 ], [ 166, 180, 3/8*E(8)-3/8*E(8)^3 ], [ 166, 185, 3/8*E(8)-3/8*E(8)^3 ], [ 167, 127, -1/4*E(8)+1/4*E(8)^3 ], [ 167, 134, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 167, 135, 1/4*E(8)-1/4*E(8)^3 ], [ 167, 142, -1/4*E(8)+1/4*E(8)^3 ], [ 168, 168, -1 ], [ 169, 179, 1/2 ], [ 169, 184, -1/2 ], [ 169, 186, -1/2 ], [ 169, 187, 3/2 ], 
  [ 169, 191, 1/2 ], [ 169, 192, -3/2 ], [ 169, 193, -3/2 ], [ 169, 198, 3/2 ], [ 170, 179, -1/8*E(8)+1/8*E(8)^3 ], [ 170, 184, -1/8*E(8)+1/8*E(8)^3 ], [ 170, 186, 1/8*E(8)-1/8*E(8)^3 ],
  [ 170, 187, 1/8*E(8)-1/8*E(8)^3 ], [ 170, 190, -1/8 ], [ 170, 191, 1/8*E(8)-1/8*E(8)^3 ], [ 170, 192, 1/8*E(8)-1/8*E(8)^3 ], [ 170, 193, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 170, 196, 1/8 ], [ 170, 197, 1/8 ], [ 170, 198, -1/8*E(8)+1/8*E(8)^3 ], [ 170, 203, -1/8 ], [ 171, 122, -1 ], [ 171, 130, -1/4*E(8)+1/4*E(8)^3 ], [ 171, 137, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 171, 150, 1/2*E(8)-1/2*E(8)^3 ], [ 171, 157, 1/4 ], [ 171, 158, 1/2*E(8)-1/2*E(8)^3 ], [ 171, 164, -1/4 ], [ 171, 165, 1/4 ], [ 171, 171, -1/4 ], [ 172, 138, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 172, 145, -1/4*E(8)+1/4*E(8)^3 ], [ 172, 146, -1/4*E(8)+1/4*E(8)^3 ], [ 172, 152, 1/4 ], [ 172, 153, -1/4*E(8)+1/4*E(8)^3 ], [ 172, 160, 1/4 ], [ 173, 166, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 173, 173, -1/4 ], [ 173, 174, 3/2*E(8)-3/2*E(8)^3 ], [ 173, 178, 1/4 ], [ 173, 180, -3/4 ], [ 173, 185, 3/4 ], [ 174, 173, 1/8*E(8)-1/8*E(8)^3 ], [ 174, 178, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 174, 180, -1/8*E(8)+1/8*E(8)^3 ], [ 174, 185, -1/8*E(8)+1/8*E(8)^3 ], [ 175, 175, -1/4 ], [ 175, 181, 1/4 ], [ 175, 182, -3/4 ], [ 175, 188, 3/4 ], [ 175, 202, 1/4*E(8)-1/4*E(8)^3 ],
  [ 175, 207, 3/4*E(8)-3/4*E(8)^3 ], [ 176, 179, 1/2 ], [ 176, 184, -1/2 ], [ 176, 186, -1/2 ], [ 176, 187, -1/2 ], [ 176, 191, 1/2 ], [ 176, 192, 1/2 ], [ 176, 193, 1/2 ], 
  [ 176, 198, -1/2 ], [ 177, 138, -1/4*E(8)+1/4*E(8)^3 ], [ 177, 145, -1/4*E(8)+1/4*E(8)^3 ], [ 177, 146, -1/4*E(8)+1/4*E(8)^3 ], [ 177, 152, -1/4 ], [ 177, 153, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 177, 160, -1/4 ], [ 178, 166, 1/2*E(8)-1/2*E(8)^3 ], [ 178, 173, 1/4 ], [ 178, 174, 3/2*E(8)-3/2*E(8)^3 ], [ 178, 178, -1/4 ], [ 178, 180, 3/4 ], [ 178, 185, -3/4 ], 
  [ 179, 154, -1/8*E(8)+1/8*E(8)^3 ], [ 179, 161, -1/8*E(8)+1/8*E(8)^3 ], [ 179, 162, -3/8*E(8)+3/8*E(8)^3 ], [ 179, 169, 1/8 ], [ 179, 170, -3/8*E(8)+3/8*E(8)^3 ], 
  [ 179, 176, 3/8 ], [ 179, 179, 1/8 ], [ 179, 184, 1/8 ], [ 179, 186, 1/8 ], [ 179, 187, 3/8 ], [ 179, 190, -1/16*E(8)+1/16*E(8)^3 ], [ 179, 191, 1/8 ], [ 179, 192, 3/8 ], 
  [ 179, 193, 3/8 ], [ 179, 196, -1/16*E(8)+1/16*E(8)^3 ], [ 179, 197, -3/16*E(8)+3/16*E(8)^3 ], [ 179, 198, 3/8 ], [ 179, 203, -3/16*E(8)+3/16*E(8)^3 ], 
  [ 180, 166, 1/2*E(8)-1/2*E(8)^3 ], [ 180, 173, -1/4 ], [ 180, 174, -1/2*E(8)+1/2*E(8)^3 ], [ 180, 178, 1/4 ], [ 180, 180, 1/4 ], [ 180, 185, -1/4 ], [ 181, 175, 1/4 ], 
  [ 181, 181, -1/4 ], [ 181, 182, 3/4 ], [ 181, 188, -3/4 ], [ 181, 202, 1/4*E(8)-1/4*E(8)^3 ], [ 181, 207, 3/4*E(8)-3/4*E(8)^3 ], [ 182, 175, -1/4 ], [ 182, 181, 1/4 ], 
  [ 182, 182, 1/4 ], [ 182, 188, -1/4 ], [ 182, 202, 1/4*E(8)-1/4*E(8)^3 ], [ 182, 207, -1/4*E(8)+1/4*E(8)^3 ], [ 183, 138, 1 ], [ 183, 145, -1 ], [ 183, 146, 1 ], [ 183, 153, -1 ], 
  [ 184, 154, -1/8*E(8)+1/8*E(8)^3 ], [ 184, 161, -1/8*E(8)+1/8*E(8)^3 ], [ 184, 162, -3/8*E(8)+3/8*E(8)^3 ], [ 184, 169, -1/8 ], [ 184, 170, -3/8*E(8)+3/8*E(8)^3 ], 
  [ 184, 176, -3/8 ], [ 184, 179, 1/8 ], [ 184, 184, 1/8 ], [ 184, 186, 1/8 ], [ 184, 187, 3/8 ], [ 184, 190, 1/16*E(8)-1/16*E(8)^3 ], [ 184, 191, 1/8 ], [ 184, 192, 3/8 ], 
  [ 184, 193, 3/8 ], [ 184, 196, 1/16*E(8)-1/16*E(8)^3 ], [ 184, 197, 3/16*E(8)-3/16*E(8)^3 ], [ 184, 198, 3/8 ], [ 184, 203, 3/16*E(8)-3/16*E(8)^3 ], [ 185, 166, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 185, 173, 1/4 ], [ 185, 174, -1/2*E(8)+1/2*E(8)^3 ], [ 185, 178, -1/4 ], [ 185, 180, -1/4 ], [ 185, 185, 1/4 ], [ 186, 154, 1/8*E(8)-1/8*E(8)^3 ], [ 186, 161, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 186, 162, 3/8*E(8)-3/8*E(8)^3 ], [ 186, 169, -1/8 ], [ 186, 170, 3/8*E(8)-3/8*E(8)^3 ], [ 186, 176, -3/8 ], [ 186, 179, 1/8 ], [ 186, 184, 1/8 ], [ 186, 186, 1/8 ], 
  [ 186, 187, 3/8 ], [ 186, 190, -1/16*E(8)+1/16*E(8)^3 ], [ 186, 191, 1/8 ], [ 186, 192, 3/8 ], [ 186, 193, 3/8 ], [ 186, 196, -1/16*E(8)+1/16*E(8)^3 ], 
  [ 186, 197, -3/16*E(8)+3/16*E(8)^3 ], [ 186, 198, 3/8 ], [ 186, 203, -3/16*E(8)+3/16*E(8)^3 ], [ 187, 154, -1/8*E(8)+1/8*E(8)^3 ], [ 187, 161, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 187, 162, 1/8*E(8)-1/8*E(8)^3 ], [ 187, 169, 1/8 ], [ 187, 170, 1/8*E(8)-1/8*E(8)^3 ], [ 187, 176, -1/8 ], [ 187, 179, 1/8 ], [ 187, 184, 1/8 ], [ 187, 186, 1/8 ], 
  [ 187, 187, -1/8 ], [ 187, 190, -1/16*E(8)+1/16*E(8)^3 ], [ 187, 191, 1/8 ], [ 187, 192, -1/8 ], [ 187, 193, -1/8 ], [ 187, 196, -1/16*E(8)+1/16*E(8)^3 ], 
  [ 187, 197, 1/16*E(8)-1/16*E(8)^3 ], [ 187, 198, -1/8 ], [ 187, 203, 1/16*E(8)-1/16*E(8)^3 ], [ 188, 175, 1/4 ], [ 188, 181, -1/4 ], [ 188, 182, -1/4 ], [ 188, 188, 1/4 ], 
  [ 188, 202, 1/4*E(8)-1/4*E(8)^3 ], [ 188, 207, -1/4*E(8)+1/4*E(8)^3 ], [ 189, 189, 1 ], [ 190, 154, 1/2 ], [ 190, 161, -1/2 ], [ 190, 162, 3/2 ], [ 190, 170, -3/2 ], 
  [ 190, 179, -1/4*E(8)+1/4*E(8)^3 ], [ 190, 184, 1/4*E(8)-1/4*E(8)^3 ], [ 190, 186, -1/4*E(8)+1/4*E(8)^3 ], [ 190, 187, -3/4*E(8)+3/4*E(8)^3 ], [ 190, 191, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 190, 192, 3/4*E(8)-3/4*E(8)^3 ], [ 190, 193, -3/4*E(8)+3/4*E(8)^3 ], [ 190, 198, 3/4*E(8)-3/4*E(8)^3 ], [ 191, 154, 1/8*E(8)-1/8*E(8)^3 ], [ 191, 161, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 191, 162, 3/8*E(8)-3/8*E(8)^3 ], [ 191, 169, 1/8 ], [ 191, 170, 3/8*E(8)-3/8*E(8)^3 ], [ 191, 176, 3/8 ], [ 191, 179, 1/8 ], [ 191, 184, 1/8 ], [ 191, 186, 1/8 ], 
  [ 191, 187, 3/8 ], [ 191, 190, 1/16*E(8)-1/16*E(8)^3 ], [ 191, 191, 1/8 ], [ 191, 192, 3/8 ], [ 191, 193, 3/8 ], [ 191, 196, 1/16*E(8)-1/16*E(8)^3 ], 
  [ 191, 197, 3/16*E(8)-3/16*E(8)^3 ], [ 191, 198, 3/8 ], [ 191, 203, 3/16*E(8)-3/16*E(8)^3 ], [ 192, 154, -1/8*E(8)+1/8*E(8)^3 ], [ 192, 161, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 192, 162, 1/8*E(8)-1/8*E(8)^3 ], [ 192, 169, -1/8 ], [ 192, 170, 1/8*E(8)-1/8*E(8)^3 ], [ 192, 176, 1/8 ], [ 192, 179, 1/8 ], [ 192, 184, 1/8 ], [ 192, 186, 1/8 ], 
  [ 192, 187, -1/8 ], [ 192, 190, 1/16*E(8)-1/16*E(8)^3 ], [ 192, 191, 1/8 ], [ 192, 192, -1/8 ], [ 192, 193, -1/8 ], [ 192, 196, 1/16*E(8)-1/16*E(8)^3 ], 
  [ 192, 197, -1/16*E(8)+1/16*E(8)^3 ], [ 192, 198, -1/8 ], [ 192, 203, -1/16*E(8)+1/16*E(8)^3 ], [ 193, 154, 1/8*E(8)-1/8*E(8)^3 ], [ 193, 161, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 193, 162, -1/8*E(8)+1/8*E(8)^3 ], [ 193, 169, -1/8 ], [ 193, 170, -1/8*E(8)+1/8*E(8)^3 ], [ 193, 176, 1/8 ], [ 193, 179, 1/8 ], [ 193, 184, 1/8 ], 
  [ 193, 186, 1/8 ], [ 193, 187, -1/8 ], [ 193, 190, -1/16*E(8)+1/16*E(8)^3 ], [ 193, 191, 1/8 ], [ 193, 192, -1/8 ], [ 193, 193, -1/8 ], [ 193, 196, -1/16*E(8)+1/16*E(8)^3 ], 
  [ 193, 197, 1/16*E(8)-1/16*E(8)^3 ], [ 193, 198, -1/8 ], [ 193, 203, 1/16*E(8)-1/16*E(8)^3 ], [ 194, 212, -1/2*E(8)+1/2*E(8)^3 ], [ 194, 216, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 195, 195, 1/4 ], [ 195, 200, -1/4 ], [ 195, 201, 3/4 ], [ 195, 206, -3/4 ], [ 195, 217, 1/4*E(8)-1/4*E(8)^3 ], [ 195, 221, 3/4*E(8)-3/4*E(8)^3 ], [ 196, 154, -1/2 ], 
  [ 196, 161, 1/2 ], [ 196, 162, -3/2 ], [ 196, 170, 3/2 ], [ 196, 179, -1/4*E(8)+1/4*E(8)^3 ], [ 196, 184, 1/4*E(8)-1/4*E(8)^3 ], [ 196, 186, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 196, 187, -3/4*E(8)+3/4*E(8)^3 ], [ 196, 191, 1/4*E(8)-1/4*E(8)^3 ], [ 196, 192, 3/4*E(8)-3/4*E(8)^3 ], [ 196, 193, -3/4*E(8)+3/4*E(8)^3 ], [ 196, 198, 3/4*E(8)-3/4*E(8)^3 ], 
  [ 197, 154, 1/2 ], [ 197, 161, -1/2 ], [ 197, 162, -1/2 ], [ 197, 170, 1/2 ], [ 197, 179, -1/4*E(8)+1/4*E(8)^3 ], [ 197, 184, 1/4*E(8)-1/4*E(8)^3 ], [ 197, 186, -1/4*E(8)+1/4*E(8)^3 ],
  [ 197, 187, 1/4*E(8)-1/4*E(8)^3 ], [ 197, 191, 1/4*E(8)-1/4*E(8)^3 ], [ 197, 192, -1/4*E(8)+1/4*E(8)^3 ], [ 197, 193, 1/4*E(8)-1/4*E(8)^3 ], [ 197, 198, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 198, 154, 1/8*E(8)-1/8*E(8)^3 ], [ 198, 161, 1/8*E(8)-1/8*E(8)^3 ], [ 198, 162, -1/8*E(8)+1/8*E(8)^3 ], [ 198, 169, 1/8 ], [ 198, 170, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 198, 176, -1/8 ], [ 198, 179, 1/8 ], [ 198, 184, 1/8 ], [ 198, 186, 1/8 ], [ 198, 187, -1/8 ], [ 198, 190, 1/16*E(8)-1/16*E(8)^3 ], [ 198, 191, 1/8 ], [ 198, 192, -1/8 ], 
  [ 198, 193, -1/8 ], [ 198, 196, 1/16*E(8)-1/16*E(8)^3 ], [ 198, 197, -1/16*E(8)+1/16*E(8)^3 ], [ 198, 198, -1/8 ], [ 198, 203, -1/16*E(8)+1/16*E(8)^3 ], [ 199, 199, -1/2 ], 
  [ 199, 204, -1/2 ], [ 199, 208, 1/4*E(8)-1/4*E(8)^3 ], [ 200, 195, -1/4 ], [ 200, 200, 1/4 ], [ 200, 201, -3/4 ], [ 200, 206, 3/4 ], [ 200, 217, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 200, 221, 3/4*E(8)-3/4*E(8)^3 ], [ 201, 195, 1/4 ], [ 201, 200, -1/4 ], [ 201, 201, -1/4 ], [ 201, 206, 1/4 ], [ 201, 217, 1/4*E(8)-1/4*E(8)^3 ], [ 201, 221, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 202, 175, 1/4*E(8)-1/4*E(8)^3 ], [ 202, 181, 1/4*E(8)-1/4*E(8)^3 ], [ 202, 182, 3/4*E(8)-3/4*E(8)^3 ], [ 202, 188, 3/4*E(8)-3/4*E(8)^3 ], [ 203, 154, -1/2 ], 
  [ 203, 161, 1/2 ], [ 203, 162, 1/2 ], [ 203, 170, -1/2 ], [ 203, 179, -1/4*E(8)+1/4*E(8)^3 ], [ 203, 184, 1/4*E(8)-1/4*E(8)^3 ], [ 203, 186, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 203, 187, 1/4*E(8)-1/4*E(8)^3 ], [ 203, 191, 1/4*E(8)-1/4*E(8)^3 ], [ 203, 192, -1/4*E(8)+1/4*E(8)^3 ], [ 203, 193, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 203, 198, -1/4*E(8)+1/4*E(8)^3 ], [ 204, 199, -1/2 ], [ 204, 204, -1/2 ], [ 204, 208, -1/4*E(8)+1/4*E(8)^3 ], [ 205, 209, -1/8*E(8)+1/8*E(8)^3 ], [ 205, 213, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 205, 214, -3/8*E(8)+3/8*E(8)^3 ], [ 205, 218, 3/8*E(8)-3/8*E(8)^3 ], [ 206, 195, -1/4 ], [ 206, 200, 1/4 ], [ 206, 201, 1/4 ], [ 206, 206, -1/4 ], [ 206, 217, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 206, 221, -1/4*E(8)+1/4*E(8)^3 ], [ 207, 175, 1/4*E(8)-1/4*E(8)^3 ], [ 207, 181, 1/4*E(8)-1/4*E(8)^3 ], [ 207, 182, -1/4*E(8)+1/4*E(8)^3 ], [ 207, 188, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 208, 199, E(8)-E(8)^3 ], [ 208, 204, -E(8)+E(8)^3 ], [ 209, 205, -1/2*E(8)+1/2*E(8)^3 ], [ 209, 209, 1/4 ], [ 209, 210, -3/2*E(8)+3/2*E(8)^3 ], [ 209, 213, 1/4 ], 
  [ 209, 214, 3/4 ], [ 209, 218, 3/4 ], [ 210, 209, -1/8*E(8)+1/8*E(8)^3 ], [ 210, 213, 1/8*E(8)-1/8*E(8)^3 ], [ 210, 214, 1/8*E(8)-1/8*E(8)^3 ], [ 210, 218, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 211, 224, -1/2*E(8)+1/2*E(8)^3 ], [ 211, 227, 1/2*E(8)-1/2*E(8)^3 ], [ 212, 194, -1/2*E(8)+1/2*E(8)^3 ], [ 212, 212, -1/2 ], [ 212, 216, -1/2 ], [ 213, 205, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 213, 209, 1/4 ], [ 213, 210, 3/2*E(8)-3/2*E(8)^3 ], [ 213, 213, 1/4 ], [ 213, 214, 3/4 ], [ 213, 218, 3/4 ], [ 214, 205, -1/2*E(8)+1/2*E(8)^3 ], [ 214, 209, 1/4 ], 
  [ 214, 210, 1/2*E(8)-1/2*E(8)^3 ], [ 214, 213, 1/4 ], [ 214, 214, -1/4 ], [ 214, 218, -1/4 ], [ 215, 219, 1/8*E(8)-1/8*E(8)^3 ], [ 215, 222, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 215, 223, -1/8*E(8)+1/8*E(8)^3 ], [ 215, 225, 1/8*E(8)-1/8*E(8)^3 ], [ 215, 231, -1/4 ], [ 215, 234, 1/4 ], [ 216, 194, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 216, 212, -1/2 ], [ 216, 216, -1/2 ], [ 217, 195, 1/4*E(8)-1/4*E(8)^3 ], [ 217, 200, 1/4*E(8)-1/4*E(8)^3 ], [ 217, 201, 3/4*E(8)-3/4*E(8)^3 ], [ 217, 206, 3/4*E(8)-3/4*E(8)^3 ], 
  [ 218, 205, 1/2*E(8)-1/2*E(8)^3 ], [ 218, 209, 1/4 ], [ 218, 210, -1/2*E(8)+1/2*E(8)^3 ], [ 218, 213, 1/4 ], [ 218, 214, -1/4 ], [ 218, 218, -1/4 ], [ 219, 215, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 219, 219, -1/4 ], [ 219, 220, -1/2*E(8)+1/2*E(8)^3 ], [ 219, 222, -1/4 ], [ 219, 223, 1/4 ], [ 219, 225, 1/4 ], [ 219, 229, -1 ], [ 219, 231, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 219, 234, 1/4*E(8)-1/4*E(8)^3 ], [ 220, 219, -1/8*E(8)+1/8*E(8)^3 ], [ 220, 222, 1/8*E(8)-1/8*E(8)^3 ], [ 220, 223, 1/8*E(8)-1/8*E(8)^3 ], [ 220, 225, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 220, 231, -1/4 ], [ 220, 234, 1/4 ], [ 221, 195, 1/4*E(8)-1/4*E(8)^3 ], [ 221, 200, 1/4*E(8)-1/4*E(8)^3 ], [ 221, 201, -1/4*E(8)+1/4*E(8)^3 ], [ 221, 206, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 222, 215, -1/2*E(8)+1/2*E(8)^3 ], [ 222, 219, -1/4 ], [ 222, 220, 1/2*E(8)-1/2*E(8)^3 ], [ 222, 222, -1/4 ], [ 222, 223, 1/4 ], [ 222, 225, 1/4 ], [ 222, 229, 1 ], 
  [ 222, 231, 1/4*E(8)-1/4*E(8)^3 ], [ 222, 234, 1/4*E(8)-1/4*E(8)^3 ], [ 223, 215, -1/2*E(8)+1/2*E(8)^3 ], [ 223, 219, 1/4 ], [ 223, 220, 1/2*E(8)-1/2*E(8)^3 ], [ 223, 222, 1/4 ], 
  [ 223, 223, -1/4 ], [ 223, 225, -1/4 ], [ 223, 229, -1 ], [ 223, 231, 1/4*E(8)-1/4*E(8)^3 ], [ 223, 234, 1/4*E(8)-1/4*E(8)^3 ], [ 224, 211, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 224, 224, 1/2 ], [ 224, 227, 1/2 ], [ 225, 215, 1/2*E(8)-1/2*E(8)^3 ], [ 225, 219, 1/4 ], [ 225, 220, -1/2*E(8)+1/2*E(8)^3 ], [ 225, 222, 1/4 ], [ 225, 223, -1/4 ], 
  [ 225, 225, -1/4 ], [ 225, 229, 1 ], [ 225, 231, 1/4*E(8)-1/4*E(8)^3 ], [ 225, 234, 1/4*E(8)-1/4*E(8)^3 ], [ 226, 226, 1/2 ], [ 226, 228, -1/2 ], 
  [ 226, 230, 1/4*E(8)-1/4*E(8)^3 ], [ 227, 211, 1/2*E(8)-1/2*E(8)^3 ], [ 227, 224, 1/2 ], [ 227, 227, 1/2 ], [ 228, 226, -1/2 ], [ 228, 228, 1/2 ], [ 228, 230, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 229, 219, -1/4 ], [ 229, 222, 1/4 ], [ 229, 223, -1/4 ], [ 229, 225, 1/4 ], [ 230, 226, E(8)-E(8)^3 ], [ 230, 228, E(8)-E(8)^3 ], [ 231, 215, -1 ], [ 231, 219, 1/4*E(8)-1/4*E(8)^3 ],
  [ 231, 220, -1 ], [ 231, 222, 1/4*E(8)-1/4*E(8)^3 ], [ 231, 223, 1/4*E(8)-1/4*E(8)^3 ], [ 231, 225, 1/4*E(8)-1/4*E(8)^3 ], [ 232, 237, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 232, 238, -1/2*E(8)+1/2*E(8)^3 ], [ 233, 233, -1/2 ], [ 233, 235, 1/2 ], [ 233, 236, 1/4*E(8)-1/4*E(8)^3 ], [ 234, 215, 1 ], [ 234, 219, 1/4*E(8)-1/4*E(8)^3 ], [ 234, 220, 1 ], 
  [ 234, 222, 1/4*E(8)-1/4*E(8)^3 ], [ 234, 223, 1/4*E(8)-1/4*E(8)^3 ], [ 234, 225, 1/4*E(8)-1/4*E(8)^3 ], [ 235, 233, 1/2 ], [ 235, 235, -1/2 ], [ 235, 236, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 236, 233, E(8)-E(8)^3 ], [ 236, 235, E(8)-E(8)^3 ], [ 237, 232, -1/2*E(8)+1/2*E(8)^3 ], [ 237, 237, -1/2 ], [ 237, 238, 1/2 ], [ 238, 232, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 238, 237, 1/2 ], [ 238, 238, -1/2 ], [ 239, 239, -1/2 ], [ 239, 240, 3/2 ], [ 240, 239, 1/2 ], [ 240, 240, 1/2 ], [ 241, 6, 1/2 ], [ 241, 126, 1/2 ], [ 241, 242, 1 ], 
  [ 241, 245, -1/2 ], [ 241, 247, 1/2 ], [ 242, 242, 1 ], [ 243, 3, 1/4 ], [ 243, 4, 1/2*E(8)-1/2*E(8)^3 ], [ 243, 6, 1/2 ], [ 243, 11, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 243, 123, 1/4 ], [ 243, 124, 1/8*E(8)-1/8*E(8)^3 ], [ 243, 126, 1/2 ], [ 243, 131, -1/8*E(8)+1/8*E(8)^3 ], [ 243, 241, -1/4 ], [ 243, 242, 3/2 ], [ 243, 244, -1/4 ], 
  [ 243, 247, 1/2 ], [ 244, 3, 1/2 ], [ 244, 6, 1/2 ], [ 244, 123, 1/2 ], [ 244, 126, 1/2 ], [ 244, 241, -1/2 ], [ 244, 242, 2 ], [ 244, 244, -1/2 ], 
  [ 244, 245, 1/2 ], [ 244, 247, 1/2 ], [ 245, 6, 1/2 ], [ 245, 126, 1/2 ], [ 245, 241, -1 ], [ 245, 242, 1 ], [ 245, 245, 1/2 ], [ 245, 247, 1/2 ], [ 246, 6, 1/4 ], 
  [ 246, 24, -1/4*E(8)+1/4*E(8)^3 ], [ 246, 31, -1/4*E(8)+1/4*E(8)^3 ], [ 246, 126, 1/4 ], [ 246, 144, -1/4*E(8)+1/4*E(8)^3 ], [ 246, 151, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 246, 241, -1/2 ], [ 246, 242, 1/2 ], [ 246, 245, 1/4 ], [ 246, 247, 3/4 ], [ 247, 247, 1 ], [ 248, 8, -1/4 ], [ 248, 128, -3/4 ], [ 248, 247, 3/4 ], [ 248, 248, -1/2 ] ],
  
[ 248, [ 1, 4, E(4) ], [ 1, 11, -E(4) ], [ 1, 12, 1/4*E(8)-1/4*E(8)^3 ], [ 1, 19, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 1, 20, -1/4*E(8)+1/4*E(8)^3 ], [ 1, 27, 1/4*E(8)-1/4*E(8)^3 ], [ 2, 37, -1 ], [ 2, 44, -1 ], 
  [ 2, 45, 1 ], [ 2, 51, 1 ], [ 3, 5, 1/8*E(8)+1/8*E(8)^3 ], [ 3, 6, 1/4 ], 
  [ 3, 13, -1/8*E(8)-1/8*E(8)^3 ], [ 3, 125, 1/2*E(8)+1/2*E(8)^3 ], [ 3, 126, 1/4 ], 
  [ 3, 133, -1/2*E(8)-1/2*E(8)^3 ], [ 3, 244, 1/2 ], [ 3, 245, -3/4 ], [ 3, 247, 1/4 ], 
  [ 4, 5, -1/8*E(4) ], [ 4, 6, -1/8*E(8)+1/8*E(8)^3 ], [ 4, 13, -1/8*E(4) ], 
  [ 4, 126, 1/8*E(8)-1/8*E(8)^3 ], [ 4, 245, -1/8*E(8)+1/8*E(8)^3 ], [ 4, 246, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 4, 247, -1/8*E(8)+1/8*E(8)^3 ], [ 5, 132, -1/2*E(8)+1/2*E(8)^3 ], [ 5, 139, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 5, 140, -1/2*E(8)+1/2*E(8)^3 ], [ 5, 144, E(4) ], [ 5, 147, -1/2*E(8)+1/2*E(8)^3 ], [ 5, 151, E(4) ], 
  [ 6, 1, 1/4*E(8)+1/4*E(8)^3 ], [ 6, 3, 1/4 ], [ 6, 9, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 6, 121, 1/4*E(8)+1/4*E(8)^3 ], [ 6, 123, 1/4 ], [ 6, 129, 1/4*E(8)+1/4*E(8)^3 ], [ 6, 241, 3/4 ], 
  [ 6, 244, -1/4 ], [ 7, 28, 1/4*E(4) ], [ 7, 35, 1/4*E(4) ], [ 7, 36, -1/4*E(4) ], 
  [ 7, 39, 1/4*E(8)-1/4*E(8)^3 ], [ 7, 43, -1/4*E(4) ], [ 7, 47, -1/4*E(8)+1/4*E(8)^3 ], [ 8, 8, 1/4 ], 
  [ 8, 128, -9/4 ], [ 8, 247, 3/4 ], [ 8, 248, -3/2 ], [ 9, 4, E(4) ], [ 9, 11, -E(4) ], 
  [ 9, 12, -1/4*E(8)+1/4*E(8)^3 ], [ 9, 19, 1/4*E(8)-1/4*E(8)^3 ], [ 9, 20, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 9, 27, -1/4*E(8)+1/4*E(8)^3 ], [ 10, 37, 1/4*E(8)-1/4*E(8)^3 ], [ 10, 44, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 10, 45, 1/4*E(8)-1/4*E(8)^3 ], [ 10, 51, 1/4*E(8)-1/4*E(8)^3 ], [ 10, 52, -1/4*E(4) ], 
  [ 10, 57, -1/4*E(4) ], [ 11, 5, -1/8*E(4) ], [ 11, 6, 1/8*E(8)-1/8*E(8)^3 ], [ 11, 13, -1/8*E(4) ], 
  [ 11, 126, -1/8*E(8)+1/8*E(8)^3 ], [ 11, 245, 1/8*E(8)-1/8*E(8)^3 ], [ 11, 246, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 11, 247, 1/8*E(8)-1/8*E(8)^3 ], [ 12, 124, 1/8*E(8)+1/8*E(8)^3 ], [ 12, 131, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 12, 132, 1/4 ], [ 12, 136, 1/4 ], [ 12, 139, 1/4 ], [ 12, 140, -1/4 ], 
  [ 12, 144, -1/4*E(8)-1/4*E(8)^3 ], [ 12, 147, -1/4 ], [ 12, 151, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 13, 132, -1/2*E(8)+1/2*E(8)^3 ], [ 13, 139, -1/2*E(8)+1/2*E(8)^3 ], [ 13, 140, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 13, 144, -E(4) ], [ 13, 147, -1/2*E(8)+1/2*E(8)^3 ], [ 13, 151, -E(4) ], [ 14, 28, 1/4*E(4) ], 
  [ 14, 35, 1/4*E(4) ], [ 14, 36, -1/4*E(4) ], [ 14, 39, -1/4*E(8)+1/4*E(8)^3 ], [ 14, 43, -1/4*E(4) ], 
  [ 14, 47, 1/4*E(8)-1/4*E(8)^3 ], [ 15, 28, 3/4*E(4) ], [ 15, 35, 3/4*E(4) ], [ 15, 36, 1/4*E(4) ], 
  [ 15, 39, 3/4*E(8)-3/4*E(8)^3 ], [ 15, 43, 1/4*E(4) ], [ 15, 47, 1/4*E(8)-1/4*E(8)^3 ], [ 16, 12, 1/4 ], 
  [ 16, 19, -1/4 ], [ 16, 20, 1/4 ], [ 16, 27, -1/4 ], [ 17, 37, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 17, 44, -1/4*E(8)+1/4*E(8)^3 ], [ 17, 45, -1/4*E(8)+1/4*E(8)^3 ], [ 17, 51, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 17, 52, -1/4*E(4) ], [ 17, 57, -1/4*E(4) ], [ 18, 10, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 18, 17, 1/2*E(8)+1/2*E(8)^3 ], [ 18, 18, 1/4 ], [ 18, 23, 1 ], [ 18, 25, 1/4 ], [ 18, 26, -1/4 ], 
  [ 18, 30, -1/4*E(8)-1/4*E(8)^3 ], [ 18, 33, -1/4 ], [ 18, 38, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 19, 124, 1/8*E(8)+1/8*E(8)^3 ], [ 19, 131, 1/8*E(8)+1/8*E(8)^3 ], [ 19, 132, -1/4 ], [ 19, 136, 1/4 ], 
  [ 19, 139, -1/4 ], [ 19, 140, 1/4 ], [ 19, 144, 1/4*E(8)+1/4*E(8)^3 ], [ 19, 147, 1/4 ], 
  [ 19, 151, -1/4*E(8)-1/4*E(8)^3 ], [ 20, 124, 1/8*E(8)+1/8*E(8)^3 ], [ 20, 131, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 20, 132, 1/4 ], [ 20, 136, -1/4 ], [ 20, 139, 1/4 ], [ 20, 140, -1/4 ], 
  [ 20, 144, 1/4*E(8)+1/4*E(8)^3 ], [ 20, 147, -1/4 ], [ 20, 151, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 21, 7, -1/2*E(8)-1/2*E(8)^3 ], [ 21, 14, -1/2*E(8)-1/2*E(8)^3 ], [ 21, 15, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 21, 22, 1/2*E(8)+1/2*E(8)^3 ], [ 22, 28, 3/4*E(4) ], [ 22, 35, 3/4*E(4) ], [ 22, 36, 1/4*E(4) ], 
  [ 22, 39, -3/4*E(8)+3/4*E(8)^3 ], [ 22, 43, 1/4*E(4) ], [ 22, 47, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 23, 63, -1 ], [ 24, 3, 1/4*E(8)-1/4*E(8)^3 ], [ 24, 121, -1/2*E(4) ], [ 24, 123, -1/4*E(8)+1/4*E(8)^3 ]
    , [ 24, 129, 1/2*E(4) ], [ 24, 241, 1/4*E(8)-1/4*E(8)^3 ], [ 24, 243, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 24, 244, 1/4*E(8)-1/4*E(8)^3 ], [ 25, 10, -1/2*E(8)-1/2*E(8)^3 ], [ 25, 17, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 25, 18, 1/4 ], [ 25, 23, -1 ], [ 25, 25, 1/4 ], [ 25, 26, -1/4 ], [ 25, 30, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 25, 33, -1/4 ], [ 25, 38, 1/4*E(8)+1/4*E(8)^3 ], [ 26, 10, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 26, 17, -1/2*E(8)-1/2*E(8)^3 ], [ 26, 18, -1/4 ], [ 26, 23, 1 ], [ 26, 25, -1/4 ], [ 26, 26, 1/4 ], 
  [ 26, 30, -1/4*E(8)-1/4*E(8)^3 ], [ 26, 33, 1/4 ], [ 26, 38, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 27, 124, 1/8*E(8)+1/8*E(8)^3 ], [ 27, 131, 1/8*E(8)+1/8*E(8)^3 ], [ 27, 132, -1/4 ], [ 27, 136, -1/4 ],
  [ 27, 139, -1/4 ], [ 27, 140, 1/4 ], [ 27, 144, -1/4*E(8)-1/4*E(8)^3 ], [ 27, 147, 1/4 ], 
  [ 27, 151, 1/4*E(8)+1/4*E(8)^3 ], [ 28, 7, -1/4*E(4) ], [ 28, 14, 1/4*E(4) ], [ 28, 15, 1/4*E(4) ], 
  [ 28, 21, -1/8*E(8)+1/8*E(8)^3 ], [ 28, 22, -1/4*E(4) ], [ 28, 29, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 29, 7, -3/2*E(8)-3/2*E(8)^3 ], [ 29, 14, -3/2*E(8)-3/2*E(8)^3 ], [ 29, 15, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 29, 22, -1/2*E(8)-1/2*E(8)^3 ], [ 30, 32, -E(4) ], [ 30, 37, -1/2*E(8)+1/2*E(8)^3 ], [ 30, 40, E(4) ], 
  [ 30, 44, 1/2*E(8)-1/2*E(8)^3 ], [ 30, 45, 1/2*E(8)-1/2*E(8)^3 ], [ 30, 51, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 31, 3, 1/4*E(8)-1/4*E(8)^3 ], [ 31, 121, 1/2*E(4) ], [ 31, 123, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 31, 129, -1/2*E(4) ], [ 31, 241, 1/4*E(8)-1/4*E(8)^3 ], [ 31, 243, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 31, 244, 1/4*E(8)-1/4*E(8)^3 ], [ 32, 18, -1/8*E(8)+1/8*E(8)^3 ], [ 32, 25, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 32, 26, -1/8*E(8)+1/8*E(8)^3 ], [ 32, 30, 1/4*E(4) ], [ 32, 33, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 32, 38, 1/4*E(4) ], [ 33, 10, 1/2*E(8)+1/2*E(8)^3 ], [ 33, 17, 1/2*E(8)+1/2*E(8)^3 ], [ 33, 18, -1/4 ],
  [ 33, 23, -1 ], [ 33, 25, -1/4 ], [ 33, 26, 1/4 ], [ 33, 30, -1/4*E(8)-1/4*E(8)^3 ], [ 33, 33, 1/4 ], 
  [ 33, 38, 1/4*E(8)+1/4*E(8)^3 ], [ 34, 53, -1/2*E(4) ], [ 34, 58, -1/2*E(4) ], 
  [ 34, 59, -1/8*E(8)+1/8*E(8)^3 ], [ 34, 60, 1/2*E(4) ], [ 34, 64, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 34, 65, 1/2*E(4) ], [ 34, 66, 1/8*E(8)-1/8*E(8)^3 ], [ 34, 67, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 34, 71, 1/8*E(8)-1/8*E(8)^3 ], [ 34, 72, 1/8*E(8)-1/8*E(8)^3 ], [ 34, 73, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 34, 78, -1/8*E(8)+1/8*E(8)^3 ], [ 35, 7, 1/4*E(4) ], [ 35, 14, -1/4*E(4) ], [ 35, 15, -1/4*E(4) ], 
  [ 35, 21, -1/8*E(8)+1/8*E(8)^3 ], [ 35, 22, 1/4*E(4) ], [ 35, 29, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 36, 7, -3/4*E(4) ], [ 36, 14, 3/4*E(4) ], [ 36, 15, -1/4*E(4) ], [ 36, 21, -3/8*E(8)+3/8*E(8)^3 ], 
  [ 36, 22, 1/4*E(4) ], [ 36, 29, -1/8*E(8)+1/8*E(8)^3 ], [ 37, 32, 1/4*E(8)+1/4*E(8)^3 ], [ 37, 37, 1/4 ],
  [ 37, 40, 1/4*E(8)+1/4*E(8)^3 ], [ 37, 44, -1/4 ], [ 37, 45, 1/4 ], [ 37, 48, 1/4 ], [ 37, 51, -1/4 ], 
  [ 37, 52, -1/8*E(8)-1/8*E(8)^3 ], [ 37, 57, 1/8*E(8)+1/8*E(8)^3 ], [ 38, 32, E(4) ], 
  [ 38, 37, -1/2*E(8)+1/2*E(8)^3 ], [ 38, 40, -E(4) ], [ 38, 44, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 38, 45, 1/2*E(8)-1/2*E(8)^3 ], [ 38, 51, -1/2*E(8)+1/2*E(8)^3 ], [ 39, 28, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 39, 35, 1/4*E(8)+1/4*E(8)^3 ], [ 39, 36, 1/4*E(8)+1/4*E(8)^3 ], [ 39, 43, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 40, 18, 1/8*E(8)-1/8*E(8)^3 ], [ 40, 25, 1/8*E(8)-1/8*E(8)^3 ], [ 40, 26, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 40, 30, 1/4*E(4) ], [ 40, 33, 1/8*E(8)-1/8*E(8)^3 ], [ 40, 38, 1/4*E(4) ], [ 41, 53, 1/2*E(4) ], 
  [ 41, 58, 1/2*E(4) ], [ 41, 59, -1/8*E(8)+1/8*E(8)^3 ], [ 41, 60, -1/2*E(4) ], 
  [ 41, 64, -1/8*E(8)+1/8*E(8)^3 ], [ 41, 65, -1/2*E(4) ], [ 41, 66, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 41, 67, 1/8*E(8)-1/8*E(8)^3 ], [ 41, 71, 1/8*E(8)-1/8*E(8)^3 ], [ 41, 72, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 41, 73, -1/8*E(8)+1/8*E(8)^3 ], [ 41, 78, -1/8*E(8)+1/8*E(8)^3 ], [ 42, 53, -3/2*E(4) ], 
  [ 42, 58, -3/2*E(4) ], [ 42, 59, -3/8*E(8)+3/8*E(8)^3 ], [ 42, 60, -1/2*E(4) ], 
  [ 42, 64, -3/8*E(8)+3/8*E(8)^3 ], [ 42, 65, -1/2*E(4) ], [ 42, 66, 3/8*E(8)-3/8*E(8)^3 ], 
  [ 42, 67, -1/8*E(8)+1/8*E(8)^3 ], [ 42, 71, 3/8*E(8)-3/8*E(8)^3 ], [ 42, 72, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 42, 73, 1/8*E(8)-1/8*E(8)^3 ], [ 42, 78, 1/8*E(8)-1/8*E(8)^3 ], [ 43, 7, 3/4*E(4) ], 
  [ 43, 14, -3/4*E(4) ], [ 43, 15, 1/4*E(4) ], [ 43, 21, -3/8*E(8)+3/8*E(8)^3 ], [ 43, 22, -1/4*E(4) ], 
  [ 43, 29, -1/8*E(8)+1/8*E(8)^3 ], [ 44, 32, -1/4*E(8)-1/4*E(8)^3 ], [ 44, 37, -1/4 ], 
  [ 44, 40, -1/4*E(8)-1/4*E(8)^3 ], [ 44, 44, 1/4 ], [ 44, 45, -1/4 ], [ 44, 48, 1/4 ], [ 44, 51, 1/4 ], 
  [ 44, 52, -1/8*E(8)-1/8*E(8)^3 ], [ 44, 57, 1/8*E(8)+1/8*E(8)^3 ], [ 45, 32, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 45, 37, 1/4 ], [ 45, 40, -1/4*E(8)-1/4*E(8)^3 ], [ 45, 44, -1/4 ], [ 45, 45, 1/4 ], [ 45, 48, -1/4 ], 
  [ 45, 51, -1/4 ], [ 45, 52, -1/8*E(8)-1/8*E(8)^3 ], [ 45, 57, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 46, 70, 1/2*E(8)+1/2*E(8)^3 ], [ 46, 76, -1/2*E(8)-1/2*E(8)^3 ], [ 46, 77, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 46, 83, 1/2*E(8)+1/2*E(8)^3 ], [ 47, 28, -3/4*E(8)-3/4*E(8)^3 ], [ 47, 35, 3/4*E(8)+3/4*E(8)^3 ], 
  [ 47, 36, -1/4*E(8)-1/4*E(8)^3 ], [ 47, 43, 1/4*E(8)+1/4*E(8)^3 ], [ 48, 2, -1 ], [ 49, 59, 1/8 ], 
  [ 49, 64, 1/8 ], [ 49, 66, 1/8 ], [ 49, 67, -1/8 ], [ 49, 71, 1/8 ], [ 49, 72, -1/8 ], [ 49, 73, -1/8 ], 
  [ 49, 78, -1/8 ], [ 50, 53, 3/2*E(4) ], [ 50, 58, 3/2*E(4) ], [ 50, 59, -3/8*E(8)+3/8*E(8)^3 ], 
  [ 50, 60, 1/2*E(4) ], [ 50, 64, -3/8*E(8)+3/8*E(8)^3 ], [ 50, 65, 1/2*E(4) ], 
  [ 50, 66, 3/8*E(8)-3/8*E(8)^3 ], [ 50, 67, -1/8*E(8)+1/8*E(8)^3 ], [ 50, 71, 3/8*E(8)-3/8*E(8)^3 ], 
  [ 50, 72, -1/8*E(8)+1/8*E(8)^3 ], [ 50, 73, 1/8*E(8)-1/8*E(8)^3 ], [ 50, 78, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 51, 32, 1/4*E(8)+1/4*E(8)^3 ], [ 51, 37, -1/4 ], [ 51, 40, 1/4*E(8)+1/4*E(8)^3 ], [ 51, 44, 1/4 ], 
  [ 51, 45, -1/4 ], [ 51, 48, -1/4 ], [ 51, 51, 1/4 ], [ 51, 52, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 51, 57, 1/8*E(8)+1/8*E(8)^3 ], [ 52, 10, E(4) ], [ 52, 17, -E(4) ], [ 52, 18, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 52, 25, -1/4*E(8)+1/4*E(8)^3 ], [ 52, 26, -1/4*E(8)+1/4*E(8)^3 ], [ 52, 33, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 53, 70, -1/4*E(4) ], [ 53, 76, -1/4*E(4) ], [ 53, 77, 1/4*E(4) ], [ 53, 82, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 53, 83, 1/4*E(4) ], [ 53, 87, 1/8*E(8)-1/8*E(8)^3 ], [ 54, 70, 3/2*E(8)+3/2*E(8)^3 ], 
  [ 54, 76, -3/2*E(8)-3/2*E(8)^3 ], [ 54, 77, 1/2*E(8)+1/2*E(8)^3 ], [ 54, 83, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 55, 34, -1/4*E(4) ], [ 55, 41, -1/4*E(4) ], [ 55, 42, 1/4*E(4) ], [ 55, 46, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 55, 50, 1/4*E(4) ], [ 55, 54, 1/4*E(8)-1/4*E(8)^3 ], [ 56, 59, 3/8 ], [ 56, 64, 3/8 ], [ 56, 66, 3/8 ],
  [ 56, 67, 1/8 ], [ 56, 71, 3/8 ], [ 56, 72, 1/8 ], [ 56, 73, 1/8 ], [ 56, 78, 1/8 ], [ 57, 10, -E(4) ], 
  [ 57, 17, E(4) ], [ 57, 18, 1/4*E(8)-1/4*E(8)^3 ], [ 57, 25, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 57, 26, -1/4*E(8)+1/4*E(8)^3 ], [ 57, 33, 1/4*E(8)-1/4*E(8)^3 ], [ 58, 70, 1/4*E(4) ], 
  [ 58, 76, 1/4*E(4) ], [ 58, 77, -1/4*E(4) ], [ 58, 82, -1/8*E(8)+1/8*E(8)^3 ], [ 58, 83, -1/4*E(4) ], 
  [ 58, 87, 1/8*E(8)-1/8*E(8)^3 ], [ 59, 49, -1/2 ], [ 59, 53, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 59, 55, 1/8*E(8)+1/8*E(8)^3 ], [ 59, 56, 1/2 ], [ 59, 58, -1/4*E(8)-1/4*E(8)^3 ], [ 59, 59, 1/8 ], 
  [ 59, 60, -1/4*E(8)-1/4*E(8)^3 ], [ 59, 61, -1/8*E(8)-1/8*E(8)^3 ], [ 59, 62, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 59, 64, -1/8 ], [ 59, 65, 1/4*E(8)+1/4*E(8)^3 ], [ 59, 66, -1/8 ], [ 59, 67, -1/8 ], 
  [ 59, 68, 1/8*E(8)+1/8*E(8)^3 ], [ 59, 71, 1/8 ], [ 59, 72, 1/8 ], [ 59, 73, 1/8 ], [ 59, 78, -1/8 ], 
  [ 60, 70, -3/4*E(4) ], [ 60, 76, -3/4*E(4) ], [ 60, 77, -1/4*E(4) ], [ 60, 82, -3/8*E(8)+3/8*E(8)^3 ], 
  [ 60, 83, -1/4*E(4) ], [ 60, 87, -1/8*E(8)+1/8*E(8)^3 ], [ 61, 34, 1/4*E(4) ], [ 61, 41, 1/4*E(4) ], 
  [ 61, 42, -1/4*E(4) ], [ 61, 46, -1/4*E(8)+1/4*E(8)^3 ], [ 61, 50, -1/4*E(4) ], 
  [ 61, 54, 1/4*E(8)-1/4*E(8)^3 ], [ 62, 34, -3/4*E(4) ], [ 62, 41, -3/4*E(4) ], [ 62, 42, -1/4*E(4) ], 
  [ 62, 46, -3/4*E(8)+3/4*E(8)^3 ], [ 62, 50, -1/4*E(4) ], [ 62, 54, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 63, 18, -1/4 ], [ 63, 25, 1/4 ], [ 63, 26, -1/4 ], [ 63, 33, 1/4 ], [ 64, 49, 1/2 ], 
  [ 64, 53, -1/4*E(8)-1/4*E(8)^3 ], [ 64, 55, 1/8*E(8)+1/8*E(8)^3 ], [ 64, 56, -1/2 ], 
  [ 64, 58, 1/4*E(8)+1/4*E(8)^3 ], [ 64, 59, 1/8 ], [ 64, 60, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 64, 61, -1/8*E(8)-1/8*E(8)^3 ], [ 64, 62, -1/8*E(8)-1/8*E(8)^3 ], [ 64, 64, -1/8 ], 
  [ 64, 65, -1/4*E(8)-1/4*E(8)^3 ], [ 64, 66, -1/8 ], [ 64, 67, -1/8 ], [ 64, 68, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 64, 71, 1/8 ], [ 64, 72, 1/8 ], [ 64, 73, 1/8 ], [ 64, 78, -1/8 ], [ 65, 70, 3/4*E(4) ], 
  [ 65, 76, 3/4*E(4) ], [ 65, 77, 1/4*E(4) ], [ 65, 82, -3/8*E(8)+3/8*E(8)^3 ], [ 65, 83, 1/4*E(4) ], 
  [ 65, 87, -1/8*E(8)+1/8*E(8)^3 ], [ 66, 49, 1/2 ], [ 66, 53, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 66, 55, -1/8*E(8)-1/8*E(8)^3 ], [ 66, 56, -1/2 ], [ 66, 58, -1/4*E(8)-1/4*E(8)^3 ], [ 66, 59, 1/8 ], 
  [ 66, 60, -1/4*E(8)-1/4*E(8)^3 ], [ 66, 61, 1/8*E(8)+1/8*E(8)^3 ], [ 66, 62, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 66, 64, -1/8 ], [ 66, 65, 1/4*E(8)+1/4*E(8)^3 ], [ 66, 66, -1/8 ], [ 66, 67, -1/8 ], 
  [ 66, 68, -1/8*E(8)-1/8*E(8)^3 ], [ 66, 71, 1/8 ], [ 66, 72, 1/8 ], [ 66, 73, 1/8 ], [ 66, 78, -1/8 ], 
  [ 67, 49, -3/2 ], [ 67, 53, 3/4*E(8)+3/4*E(8)^3 ], [ 67, 55, 3/8*E(8)+3/8*E(8)^3 ], [ 67, 56, -1/2 ], 
  [ 67, 58, -3/4*E(8)-3/4*E(8)^3 ], [ 67, 59, 3/8 ], [ 67, 60, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 67, 61, -3/8*E(8)-3/8*E(8)^3 ], [ 67, 62, 1/8*E(8)+1/8*E(8)^3 ], [ 67, 64, -3/8 ], 
  [ 67, 65, -1/4*E(8)-1/4*E(8)^3 ], [ 67, 66, -3/8 ], [ 67, 67, 1/8 ], [ 67, 68, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 67, 71, 3/8 ], [ 67, 72, -1/8 ], [ 67, 73, -1/8 ], [ 67, 78, 1/8 ], [ 68, 34, 3/4*E(4) ], 
  [ 68, 41, 3/4*E(4) ], [ 68, 42, 1/4*E(4) ], [ 68, 46, -3/4*E(8)+3/4*E(8)^3 ], [ 68, 50, 1/4*E(4) ], 
  [ 68, 54, -1/4*E(8)+1/4*E(8)^3 ], [ 69, 69, 1 ], [ 70, 55, -1/8*E(4) ], 
  [ 70, 59, -1/16*E(8)+1/16*E(8)^3 ], [ 70, 61, -1/8*E(4) ], [ 70, 62, 1/8*E(4) ], 
  [ 70, 64, 1/16*E(8)-1/16*E(8)^3 ], [ 70, 66, -1/16*E(8)+1/16*E(8)^3 ], [ 70, 67, 1/16*E(8)-1/16*E(8)^3 ],
  [ 70, 68, 1/8*E(4) ], [ 70, 71, 1/16*E(8)-1/16*E(8)^3 ], [ 70, 72, -1/16*E(8)+1/16*E(8)^3 ], 
  [ 70, 73, 1/16*E(8)-1/16*E(8)^3 ], [ 70, 78, -1/16*E(8)+1/16*E(8)^3 ], [ 71, 49, -1/2 ], 
  [ 71, 53, -1/4*E(8)-1/4*E(8)^3 ], [ 71, 55, -1/8*E(8)-1/8*E(8)^3 ], [ 71, 56, 1/2 ], 
  [ 71, 58, 1/4*E(8)+1/4*E(8)^3 ], [ 71, 59, 1/8 ], [ 71, 60, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 71, 61, 1/8*E(8)+1/8*E(8)^3 ], [ 71, 62, 1/8*E(8)+1/8*E(8)^3 ], [ 71, 64, -1/8 ], 
  [ 71, 65, -1/4*E(8)-1/4*E(8)^3 ], [ 71, 66, -1/8 ], [ 71, 67, -1/8 ], [ 71, 68, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 71, 71, 1/8 ], [ 71, 72, 1/8 ], [ 71, 73, 1/8 ], [ 71, 78, -1/8 ], [ 72, 49, 3/2 ], 
  [ 72, 53, -3/4*E(8)-3/4*E(8)^3 ], [ 72, 55, 3/8*E(8)+3/8*E(8)^3 ], [ 72, 56, 1/2 ], 
  [ 72, 58, 3/4*E(8)+3/4*E(8)^3 ], [ 72, 59, 3/8 ], [ 72, 60, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 72, 61, -3/8*E(8)-3/8*E(8)^3 ], [ 72, 62, 1/8*E(8)+1/8*E(8)^3 ], [ 72, 64, -3/8 ], 
  [ 72, 65, 1/4*E(8)+1/4*E(8)^3 ], [ 72, 66, -3/8 ], [ 72, 67, 1/8 ], [ 72, 68, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 72, 71, 3/8 ], [ 72, 72, -1/8 ], [ 72, 73, -1/8 ], [ 72, 78, 1/8 ], [ 73, 49, 3/2 ], 
  [ 73, 53, 3/4*E(8)+3/4*E(8)^3 ], [ 73, 55, -3/8*E(8)-3/8*E(8)^3 ], [ 73, 56, 1/2 ], 
  [ 73, 58, -3/4*E(8)-3/4*E(8)^3 ], [ 73, 59, 3/8 ], [ 73, 60, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 73, 61, 3/8*E(8)+3/8*E(8)^3 ], [ 73, 62, -1/8*E(8)-1/8*E(8)^3 ], [ 73, 64, -3/8 ], 
  [ 73, 65, -1/4*E(8)-1/4*E(8)^3 ], [ 73, 66, -3/8 ], [ 73, 67, 1/8 ], [ 73, 68, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 73, 71, 3/8 ], [ 73, 72, -1/8 ], [ 73, 73, -1/8 ], [ 73, 78, 1/8 ], [ 74, 79, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 74, 84, -1/2*E(8)-1/2*E(8)^3 ], [ 75, 89, -1/4*E(4) ], [ 75, 93, 1/4*E(4) ], [ 75, 94, 1/4*E(4) ], 
  [ 75, 97, -1/4*E(8)+1/4*E(8)^3 ], [ 75, 98, -1/4*E(4) ], [ 75, 101, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 76, 55, 1/8*E(4) ], [ 76, 59, -1/16*E(8)+1/16*E(8)^3 ], [ 76, 61, 1/8*E(4) ], [ 76, 62, -1/8*E(4) ], 
  [ 76, 64, 1/16*E(8)-1/16*E(8)^3 ], [ 76, 66, -1/16*E(8)+1/16*E(8)^3 ], [ 76, 67, 1/16*E(8)-1/16*E(8)^3 ],
  [ 76, 68, -1/8*E(4) ], [ 76, 71, 1/16*E(8)-1/16*E(8)^3 ], [ 76, 72, -1/16*E(8)+1/16*E(8)^3 ], 
  [ 76, 73, 1/16*E(8)-1/16*E(8)^3 ], [ 76, 78, -1/16*E(8)+1/16*E(8)^3 ], [ 77, 55, -3/8*E(4) ], 
  [ 77, 59, -3/16*E(8)+3/16*E(8)^3 ], [ 77, 61, -3/8*E(4) ], [ 77, 62, -1/8*E(4) ], 
  [ 77, 64, 3/16*E(8)-3/16*E(8)^3 ], [ 77, 66, -3/16*E(8)+3/16*E(8)^3 ], [ 77, 67, -1/16*E(8)+1/16*E(8)^3 ]
    , [ 77, 68, -1/8*E(4) ], [ 77, 71, 3/16*E(8)-3/16*E(8)^3 ], [ 77, 72, 1/16*E(8)-1/16*E(8)^3 ], 
  [ 77, 73, -1/16*E(8)+1/16*E(8)^3 ], [ 77, 78, 1/16*E(8)-1/16*E(8)^3 ], [ 78, 49, -3/2 ], 
  [ 78, 53, -3/4*E(8)-3/4*E(8)^3 ], [ 78, 55, -3/8*E(8)-3/8*E(8)^3 ], [ 78, 56, -1/2 ], 
  [ 78, 58, 3/4*E(8)+3/4*E(8)^3 ], [ 78, 59, 3/8 ], [ 78, 60, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 78, 61, 3/8*E(8)+3/8*E(8)^3 ], [ 78, 62, -1/8*E(8)-1/8*E(8)^3 ], [ 78, 64, -3/8 ], 
  [ 78, 65, 1/4*E(8)+1/4*E(8)^3 ], [ 78, 66, -3/8 ], [ 78, 67, 1/8 ], [ 78, 68, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 78, 71, 3/8 ], [ 78, 72, -1/8 ], [ 78, 73, -1/8 ], [ 78, 78, 1/8 ], [ 79, 88, -E(8)+E(8)^3 ], 
  [ 79, 92, 1/2*E(4) ], [ 79, 96, -1/2*E(4) ], [ 80, 89, 1/4*E(4) ], [ 80, 93, -1/4*E(4) ], 
  [ 80, 94, -1/4*E(4) ], [ 80, 97, -1/4*E(8)+1/4*E(8)^3 ], [ 80, 98, 1/4*E(4) ], 
  [ 80, 101, 1/4*E(8)-1/4*E(8)^3 ], [ 81, 89, -3/4*E(4) ], [ 81, 93, 3/4*E(4) ], [ 81, 94, -1/4*E(4) ], 
  [ 81, 97, -3/4*E(8)+3/4*E(8)^3 ], [ 81, 98, 1/4*E(4) ], [ 81, 101, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 82, 34, 1/4*E(8)+1/4*E(8)^3 ], [ 82, 41, -1/4*E(8)-1/4*E(8)^3 ], [ 82, 42, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 82, 50, 1/4*E(8)+1/4*E(8)^3 ], [ 83, 55, 3/8*E(4) ], [ 83, 59, -3/16*E(8)+3/16*E(8)^3 ], 
  [ 83, 61, 3/8*E(4) ], [ 83, 62, 1/8*E(4) ], [ 83, 64, 3/16*E(8)-3/16*E(8)^3 ], 
  [ 83, 66, -3/16*E(8)+3/16*E(8)^3 ], [ 83, 67, -1/16*E(8)+1/16*E(8)^3 ], [ 83, 68, 1/8*E(4) ], 
  [ 83, 71, 3/16*E(8)-3/16*E(8)^3 ], [ 83, 72, 1/16*E(8)-1/16*E(8)^3 ], [ 83, 73, -1/16*E(8)+1/16*E(8)^3 ],
  [ 83, 78, 1/16*E(8)-1/16*E(8)^3 ], [ 84, 88, E(8)-E(8)^3 ], [ 84, 92, 1/2*E(4) ], [ 84, 96, -1/2*E(4) ], 
  [ 85, 75, -1/2*E(8)-1/2*E(8)^3 ], [ 85, 80, 1/2*E(8)+1/2*E(8)^3 ], [ 85, 81, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 85, 86, -1/2*E(8)-1/2*E(8)^3 ], [ 86, 89, 3/4*E(4) ], [ 86, 93, -3/4*E(4) ], [ 86, 94, 1/4*E(4) ], 
  [ 86, 97, -3/4*E(8)+3/4*E(8)^3 ], [ 86, 98, -1/4*E(4) ], [ 86, 101, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 87, 34, 3/4*E(8)+3/4*E(8)^3 ], [ 87, 41, -3/4*E(8)-3/4*E(8)^3 ], [ 87, 42, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 87, 50, -1/4*E(8)-1/4*E(8)^3 ], [ 88, 92, -1/4*E(8)-1/4*E(8)^3 ], [ 88, 96, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 89, 75, 1/4*E(4) ], [ 89, 80, 1/4*E(4) ], [ 89, 81, -1/4*E(4) ], [ 89, 85, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 89, 86, -1/4*E(4) ], [ 89, 90, -1/8*E(8)+1/8*E(8)^3 ], [ 90, 75, -3/2*E(8)-3/2*E(8)^3 ], 
  [ 90, 80, 3/2*E(8)+3/2*E(8)^3 ], [ 90, 81, -1/2*E(8)-1/2*E(8)^3 ], [ 90, 86, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 91, 111, 1/2*E(8)+1/2*E(8)^3 ], [ 91, 114, -1/2*E(8)-1/2*E(8)^3 ], [ 92, 74, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 92, 79, -1/2*E(4) ], [ 92, 84, 1/2*E(4) ], [ 93, 75, 1/4*E(4) ], [ 93, 80, 1/4*E(4) ], 
  [ 93, 81, -1/4*E(4) ], [ 93, 85, -1/8*E(8)+1/8*E(8)^3 ], [ 93, 86, -1/4*E(4) ], 
  [ 93, 90, 1/8*E(8)-1/8*E(8)^3 ], [ 94, 75, 3/4*E(4) ], [ 94, 80, 3/4*E(4) ], [ 94, 81, 1/4*E(4) ], 
  [ 94, 85, 3/8*E(8)-3/8*E(8)^3 ], [ 94, 86, 1/4*E(4) ], [ 94, 90, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 95, 99, 1/2*E(8)-1/2*E(8)^3 ], [ 95, 102, -1/2*E(8)+1/2*E(8)^3 ], [ 95, 103, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 95, 104, -E(4) ], [ 95, 105, 1/2*E(8)-1/2*E(8)^3 ], [ 95, 107, E(4) ], [ 96, 74, -1/2*E(8)+1/2*E(8)^3 ]
    , [ 96, 79, -1/2*E(4) ], [ 96, 84, 1/2*E(4) ], [ 97, 89, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 97, 93, -1/4*E(8)-1/4*E(8)^3 ], [ 97, 94, 1/4*E(8)+1/4*E(8)^3 ], [ 97, 98, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 98, 75, 3/4*E(4) ], [ 98, 80, 3/4*E(4) ], [ 98, 81, 1/4*E(4) ], [ 98, 85, -3/8*E(8)+3/8*E(8)^3 ], 
  [ 98, 86, 1/4*E(4) ], [ 98, 90, -1/8*E(8)+1/8*E(8)^3 ], [ 99, 99, -1/4 ], [ 99, 102, 1/4 ], 
  [ 99, 103, -1/4 ], [ 99, 104, 1/4*E(8)+1/4*E(8)^3 ], [ 99, 105, 1/4 ], [ 99, 106, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 99, 107, 1/4*E(8)+1/4*E(8)^3 ], [ 99, 108, -1/8*E(8)-1/8*E(8)^3 ], [ 99, 109, 1/4 ], 
  [ 100, 99, -1/2*E(8)+1/2*E(8)^3 ], [ 100, 102, 1/2*E(8)-1/2*E(8)^3 ], [ 100, 103, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 100, 104, -E(4) ], [ 100, 105, -1/2*E(8)+1/2*E(8)^3 ], [ 100, 107, E(4) ], 
  [ 101, 89, -3/4*E(8)-3/4*E(8)^3 ], [ 101, 93, -3/4*E(8)-3/4*E(8)^3 ], [ 101, 94, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 101, 98, -1/4*E(8)-1/4*E(8)^3 ], [ 102, 99, -1/4 ], [ 102, 102, 1/4 ], [ 102, 103, -1/4 ], 
  [ 102, 104, 1/4*E(8)+1/4*E(8)^3 ], [ 102, 105, 1/4 ], [ 102, 106, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 102, 107, 1/4*E(8)+1/4*E(8)^3 ], [ 102, 108, 1/8*E(8)+1/8*E(8)^3 ], [ 102, 109, -1/4 ], 
  [ 103, 99, 1/4 ], [ 103, 102, -1/4 ], [ 103, 103, 1/4 ], [ 103, 104, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 103, 105, -1/4 ], [ 103, 106, -1/8*E(8)-1/8*E(8)^3 ], [ 103, 107, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 103, 108, 1/8*E(8)+1/8*E(8)^3 ], [ 103, 109, 1/4 ], [ 104, 110, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 104, 111, -1/2*E(4) ], [ 104, 114, -1/2*E(4) ], [ 105, 99, 1/4 ], [ 105, 102, -1/4 ], 
  [ 105, 103, 1/4 ], [ 105, 104, 1/4*E(8)+1/4*E(8)^3 ], [ 105, 105, -1/4 ], 
  [ 105, 106, 1/8*E(8)+1/8*E(8)^3 ], [ 105, 107, 1/4*E(8)+1/4*E(8)^3 ], [ 105, 108, -1/8*E(8)-1/8*E(8)^3 ],
  [ 105, 109, -1/4 ], [ 106, 91, E(8)-E(8)^3 ], [ 106, 95, -1/2*E(4) ], [ 106, 100, 1/2*E(4) ], 
  [ 107, 110, 1/2*E(8)-1/2*E(8)^3 ], [ 107, 111, -1/2*E(4) ], [ 107, 114, -1/2*E(4) ], 
  [ 108, 91, E(8)-E(8)^3 ], [ 108, 95, 1/2*E(4) ], [ 108, 100, -1/2*E(4) ], [ 109, 99, -1 ], 
  [ 109, 102, -1 ], [ 109, 103, 1 ], [ 109, 105, 1 ], [ 110, 95, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 110, 100, -1/4*E(8)-1/4*E(8)^3 ], [ 111, 99, 1/4*E(8)-1/4*E(8)^3 ], [ 111, 102, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 111, 103, 1/4*E(8)-1/4*E(8)^3 ], [ 111, 105, 1/4*E(8)-1/4*E(8)^3 ], [ 111, 106, -1/4*E(4) ], 
  [ 111, 108, -1/4*E(4) ], [ 112, 113, 1/2*E(8)+1/2*E(8)^3 ], [ 112, 115, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 113, 116, -E(8)+E(8)^3 ], [ 113, 117, -1/2*E(4) ], [ 113, 118, -1/2*E(4) ], 
  [ 114, 99, 1/4*E(8)-1/4*E(8)^3 ], [ 114, 102, 1/4*E(8)-1/4*E(8)^3 ], [ 114, 103, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 114, 105, 1/4*E(8)-1/4*E(8)^3 ], [ 114, 106, 1/4*E(4) ], [ 114, 108, 1/4*E(4) ], 
  [ 115, 116, -E(8)+E(8)^3 ], [ 115, 117, 1/2*E(4) ], [ 115, 118, 1/2*E(4) ], 
  [ 116, 117, 1/4*E(8)+1/4*E(8)^3 ], [ 116, 118, -1/4*E(8)-1/4*E(8)^3 ], [ 117, 112, 1/2*E(8)-1/2*E(8)^3 ],
  [ 117, 113, 1/2*E(4) ], [ 117, 115, 1/2*E(4) ], [ 118, 112, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 118, 113, -1/2*E(4) ], [ 118, 115, -1/2*E(4) ], [ 119, 119, -1/2 ], [ 119, 120, -1/2 ], 
  [ 120, 119, 3/2 ], [ 120, 120, -1/2 ], [ 121, 124, -1/4*E(4) ], [ 121, 131, 1/4*E(4) ], 
  [ 121, 132, 1/4*E(8)-1/4*E(8)^3 ], [ 121, 139, -1/4*E(8)+1/4*E(8)^3 ], [ 121, 140, -1/4*E(8)+1/4*E(8)^3 ]
    , [ 121, 147, 1/4*E(8)-1/4*E(8)^3 ], [ 122, 157, -1/4 ], [ 122, 164, -1/4 ], [ 122, 165, 1/4 ], 
  [ 122, 171, 1/4 ], [ 123, 5, -1/8*E(8)-1/8*E(8)^3 ], [ 123, 6, 1/4 ], [ 123, 13, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 123, 125, -1/2*E(8)-1/2*E(8)^3 ], [ 123, 126, 1/4 ], [ 123, 133, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 123, 244, 1/2 ], [ 123, 245, -3/4 ], [ 123, 247, 1/4 ], [ 124, 6, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 124, 125, 2*E(4) ], [ 124, 126, -1/2*E(8)+1/2*E(8)^3 ], [ 124, 133, 2*E(4) ], 
  [ 124, 245, -1/2*E(8)+1/2*E(8)^3 ], [ 124, 246, E(8)-E(8)^3 ], [ 124, 247, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 125, 12, -1/8*E(8)+1/8*E(8)^3 ], [ 125, 19, -1/8*E(8)+1/8*E(8)^3 ], [ 125, 20, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 125, 24, -1/4*E(4) ], [ 125, 27, -1/8*E(8)+1/8*E(8)^3 ], [ 125, 31, -1/4*E(4) ], 
  [ 126, 1, -1/4*E(8)-1/4*E(8)^3 ], [ 126, 3, 1/4 ], [ 126, 9, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 126, 121, -1/4*E(8)-1/4*E(8)^3 ], [ 126, 123, 1/4 ], [ 126, 129, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 126, 241, 3/4 ], [ 126, 244, -1/4 ], [ 127, 148, -1/4*E(4) ], [ 127, 155, -1/4*E(4) ], 
  [ 127, 156, 3/4*E(4) ], [ 127, 159, 1/4*E(8)-1/4*E(8)^3 ], [ 127, 163, 3/4*E(4) ], 
  [ 127, 167, -3/4*E(8)+3/4*E(8)^3 ], [ 128, 8, -1/4 ], [ 128, 128, 1/4 ], [ 128, 247, 1/4 ], 
  [ 128, 248, -1/2 ], [ 129, 124, -1/4*E(4) ], [ 129, 131, 1/4*E(4) ], [ 129, 132, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 129, 139, 1/4*E(8)-1/4*E(8)^3 ], [ 129, 140, 1/4*E(8)-1/4*E(8)^3 ], [ 129, 147, -1/4*E(8)+1/4*E(8)^3 ],
  [ 130, 157, 1/4*E(8)-1/4*E(8)^3 ], [ 130, 164, 1/4*E(8)-1/4*E(8)^3 ], [ 130, 165, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 130, 171, 1/4*E(8)-1/4*E(8)^3 ], [ 130, 172, E(4) ], [ 130, 177, E(4) ], 
  [ 131, 6, -1/2*E(8)+1/2*E(8)^3 ], [ 131, 125, 2*E(4) ], [ 131, 126, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 131, 133, 2*E(4) ], [ 131, 245, 1/2*E(8)-1/2*E(8)^3 ], [ 131, 246, -E(8)+E(8)^3 ], 
  [ 131, 247, 1/2*E(8)-1/2*E(8)^3 ], [ 132, 4, -1/2*E(8)-1/2*E(8)^3 ], [ 132, 11, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 132, 12, 1/4 ], [ 132, 16, 1 ], [ 132, 19, 1/4 ], [ 132, 20, -1/4 ], [ 132, 24, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 132, 27, -1/4 ], [ 132, 31, -1/4*E(8)-1/4*E(8)^3 ], [ 133, 12, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 133, 19, -1/8*E(8)+1/8*E(8)^3 ], [ 133, 20, -1/8*E(8)+1/8*E(8)^3 ], [ 133, 24, 1/4*E(4) ], 
  [ 133, 27, -1/8*E(8)+1/8*E(8)^3 ], [ 133, 31, 1/4*E(4) ], [ 134, 148, -1/4*E(4) ], 
  [ 134, 155, -1/4*E(4) ], [ 134, 156, 3/4*E(4) ], [ 134, 159, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 134, 163, 3/4*E(4) ], [ 134, 167, 3/4*E(8)-3/4*E(8)^3 ], [ 135, 148, -1/4*E(4) ], 
  [ 135, 155, -1/4*E(4) ], [ 135, 156, -1/4*E(4) ], [ 135, 159, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 135, 163, -1/4*E(4) ], [ 135, 167, 1/4*E(8)-1/4*E(8)^3 ], [ 136, 132, 1 ], [ 136, 139, -1 ], 
  [ 136, 140, 1 ], [ 136, 147, -1 ], [ 137, 157, -1/4*E(8)+1/4*E(8)^3 ], [ 137, 164, -1/4*E(8)+1/4*E(8)^3 ]
    , [ 137, 165, -1/4*E(8)+1/4*E(8)^3 ], [ 137, 171, -1/4*E(8)+1/4*E(8)^3 ], [ 137, 172, E(4) ], 
  [ 137, 177, E(4) ], [ 138, 130, -1/8*E(8)-1/8*E(8)^3 ], [ 138, 137, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 138, 138, 1/4 ], [ 138, 143, 1/4 ], [ 138, 145, 1/4 ], [ 138, 146, -1/4 ], 
  [ 138, 150, 1/4*E(8)+1/4*E(8)^3 ], [ 138, 153, -1/4 ], [ 138, 158, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 139, 4, -1/2*E(8)-1/2*E(8)^3 ], [ 139, 11, -1/2*E(8)-1/2*E(8)^3 ], [ 139, 12, -1/4 ], [ 139, 16, 1 ], 
  [ 139, 19, -1/4 ], [ 139, 20, 1/4 ], [ 139, 24, -1/4*E(8)-1/4*E(8)^3 ], [ 139, 27, 1/4 ], 
  [ 139, 31, 1/4*E(8)+1/4*E(8)^3 ], [ 140, 4, -1/2*E(8)-1/2*E(8)^3 ], [ 140, 11, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 140, 12, 1/4 ], [ 140, 16, -1 ], [ 140, 19, 1/4 ], [ 140, 20, -1/4 ], [ 140, 24, -1/4*E(8)-1/4*E(8)^3 ]
    , [ 140, 27, -1/4 ], [ 140, 31, 1/4*E(8)+1/4*E(8)^3 ], [ 141, 127, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 141, 134, 1/8*E(8)+1/8*E(8)^3 ], [ 141, 135, -3/8*E(8)-3/8*E(8)^3 ], [ 141, 142, -3/8*E(8)-3/8*E(8)^3 ]
    , [ 142, 148, -1/4*E(4) ], [ 142, 155, -1/4*E(4) ], [ 142, 156, -1/4*E(4) ], 
  [ 142, 159, -1/4*E(8)+1/4*E(8)^3 ], [ 142, 163, -1/4*E(4) ], [ 142, 167, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 143, 183, -1 ], [ 144, 1, 1/2*E(4) ], [ 144, 3, -1/4*E(8)+1/4*E(8)^3 ], [ 144, 9, -1/2*E(4) ], 
  [ 144, 123, 1/4*E(8)-1/4*E(8)^3 ], [ 144, 241, 1/4*E(8)-1/4*E(8)^3 ], [ 144, 243, -1/2*E(8)+1/2*E(8)^3 ],
  [ 144, 244, 1/4*E(8)-1/4*E(8)^3 ], [ 145, 130, 1/8*E(8)+1/8*E(8)^3 ], [ 145, 137, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 145, 138, 1/4 ], [ 145, 143, -1/4 ], [ 145, 145, 1/4 ], [ 145, 146, -1/4 ], 
  [ 145, 150, 1/4*E(8)+1/4*E(8)^3 ], [ 145, 153, -1/4 ], [ 145, 158, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 146, 130, 1/8*E(8)+1/8*E(8)^3 ], [ 146, 137, 1/8*E(8)+1/8*E(8)^3 ], [ 146, 138, -1/4 ], 
  [ 146, 143, 1/4 ], [ 146, 145, -1/4 ], [ 146, 146, 1/4 ], [ 146, 150, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 146, 153, 1/4 ], [ 146, 158, -1/4*E(8)-1/4*E(8)^3 ], [ 147, 4, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 147, 11, -1/2*E(8)-1/2*E(8)^3 ], [ 147, 12, -1/4 ], [ 147, 16, -1 ], [ 147, 19, -1/4 ], 
  [ 147, 20, 1/4 ], [ 147, 24, 1/4*E(8)+1/4*E(8)^3 ], [ 147, 27, 1/4 ], [ 147, 31, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 148, 127, 1/4*E(4) ], [ 148, 134, -1/4*E(4) ], [ 148, 135, -3/4*E(4) ], 
  [ 148, 141, -1/2*E(8)+1/2*E(8)^3 ], [ 148, 142, 3/4*E(4) ], [ 148, 149, 3/2*E(8)-3/2*E(8)^3 ], 
  [ 149, 127, 1/8*E(8)+1/8*E(8)^3 ], [ 149, 134, 1/8*E(8)+1/8*E(8)^3 ], [ 149, 135, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 149, 142, 1/8*E(8)+1/8*E(8)^3 ], [ 150, 152, 1/4*E(4) ], [ 150, 157, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 150, 160, -1/4*E(4) ], [ 150, 164, 1/8*E(8)-1/8*E(8)^3 ], [ 150, 165, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 150, 171, -1/8*E(8)+1/8*E(8)^3 ], [ 151, 1, -1/2*E(4) ], [ 151, 3, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 151, 9, 1/2*E(4) ], [ 151, 123, 1/4*E(8)-1/4*E(8)^3 ], [ 151, 241, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 151, 243, -1/2*E(8)+1/2*E(8)^3 ], [ 151, 244, 1/4*E(8)-1/4*E(8)^3 ], [ 152, 138, -1/2*E(8)+1/2*E(8)^3 ]
    , [ 152, 145, -1/2*E(8)+1/2*E(8)^3 ], [ 152, 146, -1/2*E(8)+1/2*E(8)^3 ], [ 152, 150, -E(4) ], 
  [ 152, 153, -1/2*E(8)+1/2*E(8)^3 ], [ 152, 158, -E(4) ], [ 153, 130, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 153, 137, -1/8*E(8)-1/8*E(8)^3 ], [ 153, 138, -1/4 ], [ 153, 143, -1/4 ], [ 153, 145, -1/4 ], 
  [ 153, 146, 1/4 ], [ 153, 150, 1/4*E(8)+1/4*E(8)^3 ], [ 153, 153, 1/4 ], 
  [ 153, 158, -1/4*E(8)-1/4*E(8)^3 ], [ 154, 173, 1/8*E(4) ], [ 154, 178, 1/8*E(4) ], 
  [ 154, 179, -1/8*E(8)+1/8*E(8)^3 ], [ 154, 180, -3/8*E(4) ], [ 154, 184, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 154, 185, -3/8*E(4) ], [ 154, 186, 1/8*E(8)-1/8*E(8)^3 ], [ 154, 187, 3/8*E(8)-3/8*E(8)^3 ], 
  [ 154, 191, 1/8*E(8)-1/8*E(8)^3 ], [ 154, 192, 3/8*E(8)-3/8*E(8)^3 ], [ 154, 193, -3/8*E(8)+3/8*E(8)^3 ],
  [ 154, 198, -3/8*E(8)+3/8*E(8)^3 ], [ 155, 127, -1/4*E(4) ], [ 155, 134, 1/4*E(4) ], 
  [ 155, 135, 3/4*E(4) ], [ 155, 141, -1/2*E(8)+1/2*E(8)^3 ], [ 155, 142, -3/4*E(4) ], 
  [ 155, 149, 3/2*E(8)-3/2*E(8)^3 ], [ 156, 127, 1/4*E(4) ], [ 156, 134, -1/4*E(4) ], 
  [ 156, 135, 1/4*E(4) ], [ 156, 141, -1/2*E(8)+1/2*E(8)^3 ], [ 156, 142, -1/4*E(4) ], 
  [ 156, 149, -1/2*E(8)+1/2*E(8)^3 ], [ 157, 152, -1/4*E(8)-1/4*E(8)^3 ], [ 157, 157, 1/4 ], 
  [ 157, 160, -1/4*E(8)-1/4*E(8)^3 ], [ 157, 164, -1/4 ], [ 157, 165, 1/4 ], [ 157, 168, 1 ], 
  [ 157, 171, -1/4 ], [ 157, 172, 1/2*E(8)+1/2*E(8)^3 ], [ 157, 177, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 158, 152, -1/4*E(4) ], [ 158, 157, -1/8*E(8)+1/8*E(8)^3 ], [ 158, 160, 1/4*E(4) ], 
  [ 158, 164, 1/8*E(8)-1/8*E(8)^3 ], [ 158, 165, 1/8*E(8)-1/8*E(8)^3 ], [ 158, 171, -1/8*E(8)+1/8*E(8)^3 ],
  [ 159, 148, 1/4*E(8)+1/4*E(8)^3 ], [ 159, 155, -1/4*E(8)-1/4*E(8)^3 ], [ 159, 156, -3/4*E(8)-3/4*E(8)^3 ]
    , [ 159, 163, 3/4*E(8)+3/4*E(8)^3 ], [ 160, 138, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 160, 145, 1/2*E(8)-1/2*E(8)^3 ], [ 160, 146, 1/2*E(8)-1/2*E(8)^3 ], [ 160, 150, -E(4) ], 
  [ 160, 153, 1/2*E(8)-1/2*E(8)^3 ], [ 160, 158, -E(4) ], [ 161, 173, -1/8*E(4) ], [ 161, 178, -1/8*E(4) ],
  [ 161, 179, -1/8*E(8)+1/8*E(8)^3 ], [ 161, 180, 3/8*E(4) ], [ 161, 184, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 161, 185, 3/8*E(4) ], [ 161, 186, 1/8*E(8)-1/8*E(8)^3 ], [ 161, 187, 3/8*E(8)-3/8*E(8)^3 ], 
  [ 161, 191, 1/8*E(8)-1/8*E(8)^3 ], [ 161, 192, 3/8*E(8)-3/8*E(8)^3 ], [ 161, 193, -3/8*E(8)+3/8*E(8)^3 ],
  [ 161, 198, -3/8*E(8)+3/8*E(8)^3 ], [ 162, 173, 1/8*E(4) ], [ 162, 178, 1/8*E(4) ], 
  [ 162, 179, -1/8*E(8)+1/8*E(8)^3 ], [ 162, 180, 1/8*E(4) ], [ 162, 184, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 162, 185, 1/8*E(4) ], [ 162, 186, 1/8*E(8)-1/8*E(8)^3 ], [ 162, 187, -1/8*E(8)+1/8*E(8)^3 ], 
  [ 162, 191, 1/8*E(8)-1/8*E(8)^3 ], [ 162, 192, -1/8*E(8)+1/8*E(8)^3 ], [ 162, 193, 1/8*E(8)-1/8*E(8)^3 ],
  [ 162, 198, 1/8*E(8)-1/8*E(8)^3 ], [ 163, 127, -1/4*E(4) ], [ 163, 134, 1/4*E(4) ], 
  [ 163, 135, -1/4*E(4) ], [ 163, 141, -1/2*E(8)+1/2*E(8)^3 ], [ 163, 142, 1/4*E(4) ], 
  [ 163, 149, -1/2*E(8)+1/2*E(8)^3 ], [ 164, 152, 1/4*E(8)+1/4*E(8)^3 ], [ 164, 157, -1/4 ], 
  [ 164, 160, 1/4*E(8)+1/4*E(8)^3 ], [ 164, 164, 1/4 ], [ 164, 165, -1/4 ], [ 164, 168, 1 ], 
  [ 164, 171, 1/4 ], [ 164, 172, 1/2*E(8)+1/2*E(8)^3 ], [ 164, 177, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 165, 152, 1/4*E(8)+1/4*E(8)^3 ], [ 165, 157, 1/4 ], [ 165, 160, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 165, 164, -1/4 ], [ 165, 165, 1/4 ], [ 165, 168, -1 ], [ 165, 171, -1/4 ], 
  [ 165, 172, 1/2*E(8)+1/2*E(8)^3 ], [ 165, 177, -1/2*E(8)-1/2*E(8)^3 ], [ 166, 190, -1/8*E(8)-1/8*E(8)^3 ]
    , [ 166, 196, 1/8*E(8)+1/8*E(8)^3 ], [ 166, 197, 3/8*E(8)+3/8*E(8)^3 ], 
  [ 166, 203, -3/8*E(8)-3/8*E(8)^3 ], [ 167, 148, 1/4*E(8)+1/4*E(8)^3 ], [ 167, 155, -1/4*E(8)-1/4*E(8)^3 ]
    , [ 167, 156, 1/4*E(8)+1/4*E(8)^3 ], [ 167, 163, -1/4*E(8)-1/4*E(8)^3 ], [ 168, 122, -1 ], 
  [ 169, 179, 1/2 ], [ 169, 184, 1/2 ], [ 169, 186, 1/2 ], [ 169, 187, -3/2 ], [ 169, 191, 1/2 ], 
  [ 169, 192, -3/2 ], [ 169, 193, -3/2 ], [ 169, 198, -3/2 ], [ 170, 173, -1/8*E(4) ], 
  [ 170, 178, -1/8*E(4) ], [ 170, 179, -1/8*E(8)+1/8*E(8)^3 ], [ 170, 180, -1/8*E(4) ], 
  [ 170, 184, -1/8*E(8)+1/8*E(8)^3 ], [ 170, 185, -1/8*E(4) ], [ 170, 186, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 170, 187, -1/8*E(8)+1/8*E(8)^3 ], [ 170, 191, 1/8*E(8)-1/8*E(8)^3 ], [ 170, 192, -1/8*E(8)+1/8*E(8)^3 ]
    , [ 170, 193, 1/8*E(8)-1/8*E(8)^3 ], [ 170, 198, 1/8*E(8)-1/8*E(8)^3 ], 
  [ 171, 152, -1/4*E(8)-1/4*E(8)^3 ], [ 171, 157, -1/4 ], [ 171, 160, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 171, 164, 1/4 ], [ 171, 165, -1/4 ], [ 171, 168, -1 ], [ 171, 171, 1/4 ], 
  [ 171, 172, 1/2*E(8)+1/2*E(8)^3 ], [ 171, 177, -1/2*E(8)-1/2*E(8)^3 ], [ 172, 130, -1/4*E(4) ], 
  [ 172, 137, 1/4*E(4) ], [ 172, 138, 1/4*E(8)-1/4*E(8)^3 ], [ 172, 145, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 172, 146, -1/4*E(8)+1/4*E(8)^3 ], [ 172, 153, 1/4*E(8)-1/4*E(8)^3 ], [ 173, 190, 1/4*E(4) ], 
  [ 173, 196, 1/4*E(4) ], [ 173, 197, -3/4*E(4) ], [ 173, 202, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 173, 203, -3/4*E(4) ], [ 173, 207, 3/2*E(8)-3/2*E(8)^3 ], [ 174, 190, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 174, 196, 1/8*E(8)+1/8*E(8)^3 ], [ 174, 197, -1/8*E(8)-1/8*E(8)^3 ], [ 174, 203, 1/8*E(8)+1/8*E(8)^3 ],
  [ 175, 154, 1/4*E(4) ], [ 175, 161, 1/4*E(4) ], [ 175, 162, -3/4*E(4) ], 
  [ 175, 166, -1/4*E(8)+1/4*E(8)^3 ], [ 175, 170, -3/4*E(4) ], [ 175, 174, 3/4*E(8)-3/4*E(8)^3 ], 
  [ 176, 179, 1/2 ], [ 176, 184, 1/2 ], [ 176, 186, 1/2 ], [ 176, 187, 1/2 ], [ 176, 191, 1/2 ], 
  [ 176, 192, 1/2 ], [ 176, 193, 1/2 ], [ 176, 198, 1/2 ], [ 177, 130, 1/4*E(4) ], [ 177, 137, -1/4*E(4) ],
  [ 177, 138, 1/4*E(8)-1/4*E(8)^3 ], [ 177, 145, -1/4*E(8)+1/4*E(8)^3 ], [ 177, 146, -1/4*E(8)+1/4*E(8)^3 ]
    , [ 177, 153, 1/4*E(8)-1/4*E(8)^3 ], [ 178, 190, -1/4*E(4) ], [ 178, 196, -1/4*E(4) ], 
  [ 178, 197, 3/4*E(4) ], [ 178, 202, -1/2*E(8)+1/2*E(8)^3 ], [ 178, 203, 3/4*E(4) ], 
  [ 178, 207, 3/2*E(8)-3/2*E(8)^3 ], [ 179, 169, -1/8 ], [ 179, 173, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 179, 175, -1/8*E(8)-1/8*E(8)^3 ], [ 179, 176, 3/8 ], [ 179, 178, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 179, 179, 1/8 ], [ 179, 180, 3/16*E(8)+3/16*E(8)^3 ], [ 179, 181, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 179, 182, 3/8*E(8)+3/8*E(8)^3 ], [ 179, 184, -1/8 ], [ 179, 185, -3/16*E(8)-3/16*E(8)^3 ], 
  [ 179, 186, -1/8 ], [ 179, 187, -3/8 ], [ 179, 188, -3/8*E(8)-3/8*E(8)^3 ], [ 179, 191, 1/8 ], 
  [ 179, 192, 3/8 ], [ 179, 193, 3/8 ], [ 179, 198, -3/8 ], [ 180, 190, 1/4*E(4) ], [ 180, 196, 1/4*E(4) ],
  [ 180, 197, 1/4*E(4) ], [ 180, 202, -1/2*E(8)+1/2*E(8)^3 ], [ 180, 203, 1/4*E(4) ], 
  [ 180, 207, -1/2*E(8)+1/2*E(8)^3 ], [ 181, 154, -1/4*E(4) ], [ 181, 161, -1/4*E(4) ], 
  [ 181, 162, 3/4*E(4) ], [ 181, 166, -1/4*E(8)+1/4*E(8)^3 ], [ 181, 170, 3/4*E(4) ], 
  [ 181, 174, 3/4*E(8)-3/4*E(8)^3 ], [ 182, 154, 1/4*E(4) ], [ 182, 161, 1/4*E(4) ], 
  [ 182, 162, 1/4*E(4) ], [ 182, 166, -1/4*E(8)+1/4*E(8)^3 ], [ 182, 170, 1/4*E(4) ], 
  [ 182, 174, -1/4*E(8)+1/4*E(8)^3 ], [ 183, 138, -1 ], [ 183, 145, 1 ], [ 183, 146, -1 ], [ 183, 153, 1 ],
  [ 184, 169, 1/8 ], [ 184, 173, 1/16*E(8)+1/16*E(8)^3 ], [ 184, 175, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 184, 176, -3/8 ], [ 184, 178, -1/16*E(8)-1/16*E(8)^3 ], [ 184, 179, 1/8 ], 
  [ 184, 180, -3/16*E(8)-3/16*E(8)^3 ], [ 184, 181, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 184, 182, 3/8*E(8)+3/8*E(8)^3 ], [ 184, 184, -1/8 ], [ 184, 185, 3/16*E(8)+3/16*E(8)^3 ], 
  [ 184, 186, -1/8 ], [ 184, 187, -3/8 ], [ 184, 188, -3/8*E(8)-3/8*E(8)^3 ], [ 184, 191, 1/8 ], 
  [ 184, 192, 3/8 ], [ 184, 193, 3/8 ], [ 184, 198, -3/8 ], [ 185, 190, -1/4*E(4) ], 
  [ 185, 196, -1/4*E(4) ], [ 185, 197, -1/4*E(4) ], [ 185, 202, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 185, 203, -1/4*E(4) ], [ 185, 207, -1/2*E(8)+1/2*E(8)^3 ], [ 186, 169, 1/8 ], 
  [ 186, 173, -1/16*E(8)-1/16*E(8)^3 ], [ 186, 175, 1/8*E(8)+1/8*E(8)^3 ], [ 186, 176, -3/8 ], 
  [ 186, 178, 1/16*E(8)+1/16*E(8)^3 ], [ 186, 179, 1/8 ], [ 186, 180, 3/16*E(8)+3/16*E(8)^3 ], 
  [ 186, 181, -1/8*E(8)-1/8*E(8)^3 ], [ 186, 182, -3/8*E(8)-3/8*E(8)^3 ], [ 186, 184, -1/8 ], 
  [ 186, 185, -3/16*E(8)-3/16*E(8)^3 ], [ 186, 186, -1/8 ], [ 186, 187, -3/8 ], 
  [ 186, 188, 3/8*E(8)+3/8*E(8)^3 ], [ 186, 191, 1/8 ], [ 186, 192, 3/8 ], [ 186, 193, 3/8 ], 
  [ 186, 198, -3/8 ], [ 187, 169, -1/8 ], [ 187, 173, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 187, 175, -1/8*E(8)-1/8*E(8)^3 ], [ 187, 176, -1/8 ], [ 187, 178, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 187, 179, 1/8 ], [ 187, 180, -1/16*E(8)-1/16*E(8)^3 ], [ 187, 181, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 187, 182, -1/8*E(8)-1/8*E(8)^3 ], [ 187, 184, -1/8 ], [ 187, 185, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 187, 186, -1/8 ], [ 187, 187, 1/8 ], [ 187, 188, 1/8*E(8)+1/8*E(8)^3 ], [ 187, 191, 1/8 ], 
  [ 187, 192, -1/8 ], [ 187, 193, -1/8 ], [ 187, 198, 1/8 ], [ 188, 154, -1/4*E(4) ], 
  [ 188, 161, -1/4*E(4) ], [ 188, 162, -1/4*E(4) ], [ 188, 166, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 188, 170, -1/4*E(4) ], [ 188, 174, -1/4*E(8)+1/4*E(8)^3 ], [ 189, 189, 1 ], [ 190, 175, 1/2*E(4) ], 
  [ 190, 179, -1/4*E(8)+1/4*E(8)^3 ], [ 190, 181, 1/2*E(4) ], [ 190, 182, -3/2*E(4) ], 
  [ 190, 184, 1/4*E(8)-1/4*E(8)^3 ], [ 190, 186, -1/4*E(8)+1/4*E(8)^3 ], [ 190, 187, 3/4*E(8)-3/4*E(8)^3 ],
  [ 190, 188, -3/2*E(4) ], [ 190, 191, 1/4*E(8)-1/4*E(8)^3 ], [ 190, 192, -3/4*E(8)+3/4*E(8)^3 ], 
  [ 190, 193, 3/4*E(8)-3/4*E(8)^3 ], [ 190, 198, -3/4*E(8)+3/4*E(8)^3 ], [ 191, 169, -1/8 ], 
  [ 191, 173, 1/16*E(8)+1/16*E(8)^3 ], [ 191, 175, 1/8*E(8)+1/8*E(8)^3 ], [ 191, 176, 3/8 ], 
  [ 191, 178, -1/16*E(8)-1/16*E(8)^3 ], [ 191, 179, 1/8 ], [ 191, 180, -3/16*E(8)-3/16*E(8)^3 ], 
  [ 191, 181, -1/8*E(8)-1/8*E(8)^3 ], [ 191, 182, -3/8*E(8)-3/8*E(8)^3 ], [ 191, 184, -1/8 ], 
  [ 191, 185, 3/16*E(8)+3/16*E(8)^3 ], [ 191, 186, -1/8 ], [ 191, 187, -3/8 ], 
  [ 191, 188, 3/8*E(8)+3/8*E(8)^3 ], [ 191, 191, 1/8 ], [ 191, 192, 3/8 ], [ 191, 193, 3/8 ], 
  [ 191, 198, -3/8 ], [ 192, 169, 1/8 ], [ 192, 173, 1/16*E(8)+1/16*E(8)^3 ], 
  [ 192, 175, -1/8*E(8)-1/8*E(8)^3 ], [ 192, 176, 1/8 ], [ 192, 178, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 192, 179, 1/8 ], [ 192, 180, 1/16*E(8)+1/16*E(8)^3 ], [ 192, 181, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 192, 182, -1/8*E(8)-1/8*E(8)^3 ], [ 192, 184, -1/8 ], [ 192, 185, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 192, 186, -1/8 ], [ 192, 187, 1/8 ], [ 192, 188, 1/8*E(8)+1/8*E(8)^3 ], [ 192, 191, 1/8 ], 
  [ 192, 192, -1/8 ], [ 192, 193, -1/8 ], [ 192, 198, 1/8 ], [ 193, 169, 1/8 ], 
  [ 193, 173, -1/16*E(8)-1/16*E(8)^3 ], [ 193, 175, 1/8*E(8)+1/8*E(8)^3 ], [ 193, 176, 1/8 ], 
  [ 193, 178, 1/16*E(8)+1/16*E(8)^3 ], [ 193, 179, 1/8 ], [ 193, 180, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 193, 181, -1/8*E(8)-1/8*E(8)^3 ], [ 193, 182, 1/8*E(8)+1/8*E(8)^3 ], [ 193, 184, -1/8 ], 
  [ 193, 185, 1/16*E(8)+1/16*E(8)^3 ], [ 193, 186, -1/8 ], [ 193, 187, 1/8 ], 
  [ 193, 188, -1/8*E(8)-1/8*E(8)^3 ], [ 193, 191, 1/8 ], [ 193, 192, -1/8 ], [ 193, 193, -1/8 ], 
  [ 193, 198, 1/8 ], [ 194, 199, 1/2*E(8)+1/2*E(8)^3 ], [ 194, 204, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 195, 209, 1/4*E(4) ], [ 195, 213, -1/4*E(4) ], [ 195, 214, -3/4*E(4) ], 
  [ 195, 217, -1/4*E(8)+1/4*E(8)^3 ], [ 195, 218, 3/4*E(4) ], [ 195, 221, 3/4*E(8)-3/4*E(8)^3 ], 
  [ 196, 175, -1/2*E(4) ], [ 196, 179, -1/4*E(8)+1/4*E(8)^3 ], [ 196, 181, -1/2*E(4) ], 
  [ 196, 182, 3/2*E(4) ], [ 196, 184, 1/4*E(8)-1/4*E(8)^3 ], [ 196, 186, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 196, 187, 3/4*E(8)-3/4*E(8)^3 ], [ 196, 188, 3/2*E(4) ], [ 196, 191, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 196, 192, -3/4*E(8)+3/4*E(8)^3 ], [ 196, 193, 3/4*E(8)-3/4*E(8)^3 ], [ 196, 198, -3/4*E(8)+3/4*E(8)^3 ]
    , [ 197, 175, 1/2*E(4) ], [ 197, 179, -1/4*E(8)+1/4*E(8)^3 ], [ 197, 181, 1/2*E(4) ], 
  [ 197, 182, 1/2*E(4) ], [ 197, 184, 1/4*E(8)-1/4*E(8)^3 ], [ 197, 186, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 197, 187, -1/4*E(8)+1/4*E(8)^3 ], [ 197, 188, 1/2*E(4) ], [ 197, 191, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 197, 192, 1/4*E(8)-1/4*E(8)^3 ], [ 197, 193, -1/4*E(8)+1/4*E(8)^3 ], [ 197, 198, 1/4*E(8)-1/4*E(8)^3 ],
  [ 198, 169, -1/8 ], [ 198, 173, 1/16*E(8)+1/16*E(8)^3 ], [ 198, 175, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 198, 176, -1/8 ], [ 198, 178, -1/16*E(8)-1/16*E(8)^3 ], [ 198, 179, 1/8 ], 
  [ 198, 180, 1/16*E(8)+1/16*E(8)^3 ], [ 198, 181, -1/8*E(8)-1/8*E(8)^3 ], 
  [ 198, 182, 1/8*E(8)+1/8*E(8)^3 ], [ 198, 184, -1/8 ], [ 198, 185, -1/16*E(8)-1/16*E(8)^3 ], 
  [ 198, 186, -1/8 ], [ 198, 187, 1/8 ], [ 198, 188, -1/8*E(8)-1/8*E(8)^3 ], [ 198, 191, 1/8 ], 
  [ 198, 192, -1/8 ], [ 198, 193, -1/8 ], [ 198, 198, 1/8 ], [ 199, 208, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 199, 212, -1/2*E(4) ], [ 199, 216, 1/2*E(4) ], [ 200, 209, -1/4*E(4) ], [ 200, 213, 1/4*E(4) ], 
  [ 200, 214, 3/4*E(4) ], [ 200, 217, -1/4*E(8)+1/4*E(8)^3 ], [ 200, 218, -3/4*E(4) ], 
  [ 200, 221, 3/4*E(8)-3/4*E(8)^3 ], [ 201, 209, 1/4*E(4) ], [ 201, 213, -1/4*E(4) ], 
  [ 201, 214, 1/4*E(4) ], [ 201, 217, -1/4*E(8)+1/4*E(8)^3 ], [ 201, 218, -1/4*E(4) ], 
  [ 201, 221, -1/4*E(8)+1/4*E(8)^3 ], [ 202, 154, -1/4*E(8)-1/4*E(8)^3 ], [ 202, 161, 1/4*E(8)+1/4*E(8)^3 ]
    , [ 202, 162, 3/4*E(8)+3/4*E(8)^3 ], [ 202, 170, -3/4*E(8)-3/4*E(8)^3 ], [ 203, 175, -1/2*E(4) ], 
  [ 203, 179, -1/4*E(8)+1/4*E(8)^3 ], [ 203, 181, -1/2*E(4) ], [ 203, 182, -1/2*E(4) ], 
  [ 203, 184, 1/4*E(8)-1/4*E(8)^3 ], [ 203, 186, -1/4*E(8)+1/4*E(8)^3 ], [ 203, 187, -1/4*E(8)+1/4*E(8)^3 ]
    , [ 203, 188, -1/2*E(4) ], [ 203, 191, 1/4*E(8)-1/4*E(8)^3 ], [ 203, 192, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 203, 193, -1/4*E(8)+1/4*E(8)^3 ], [ 203, 198, 1/4*E(8)-1/4*E(8)^3 ], [ 204, 208, 1/4*E(8)-1/4*E(8)^3 ],
  [ 204, 212, -1/2*E(4) ], [ 204, 216, 1/2*E(4) ], [ 205, 195, 1/8*E(8)+1/8*E(8)^3 ], 
  [ 205, 200, -1/8*E(8)-1/8*E(8)^3 ], [ 205, 201, -3/8*E(8)-3/8*E(8)^3 ], [ 205, 206, 3/8*E(8)+3/8*E(8)^3 ]
    , [ 206, 209, -1/4*E(4) ], [ 206, 213, 1/4*E(4) ], [ 206, 214, -1/4*E(4) ], 
  [ 206, 217, -1/4*E(8)+1/4*E(8)^3 ], [ 206, 218, 1/4*E(4) ], [ 206, 221, -1/4*E(8)+1/4*E(8)^3 ], 
  [ 207, 154, -1/4*E(8)-1/4*E(8)^3 ], [ 207, 161, 1/4*E(8)+1/4*E(8)^3 ], [ 207, 162, -1/4*E(8)-1/4*E(8)^3 ]
    , [ 207, 170, 1/4*E(8)+1/4*E(8)^3 ], [ 208, 212, E(8)+E(8)^3 ], [ 208, 216, E(8)+E(8)^3 ], 
  [ 209, 195, -1/4*E(4) ], [ 209, 200, -1/4*E(4) ], [ 209, 201, 3/4*E(4) ], 
  [ 209, 205, 1/2*E(8)-1/2*E(8)^3 ], [ 209, 206, 3/4*E(4) ], [ 209, 210, -3/2*E(8)+3/2*E(8)^3 ], 
  [ 210, 195, 1/8*E(8)+1/8*E(8)^3 ], [ 210, 200, -1/8*E(8)-1/8*E(8)^3 ], [ 210, 201, 1/8*E(8)+1/8*E(8)^3 ],
  [ 210, 206, -1/8*E(8)-1/8*E(8)^3 ], [ 211, 231, -1/2*E(8)-1/2*E(8)^3 ], [ 211, 234, 1/2*E(8)+1/2*E(8)^3 ]
    , [ 212, 194, 1/2*E(8)-1/2*E(8)^3 ], [ 212, 199, 1/2*E(4) ], [ 212, 204, -1/2*E(4) ], 
  [ 213, 195, -1/4*E(4) ], [ 213, 200, -1/4*E(4) ], [ 213, 201, 3/4*E(4) ], 
  [ 213, 205, -1/2*E(8)+1/2*E(8)^3 ], [ 213, 206, 3/4*E(4) ], [ 213, 210, 3/2*E(8)-3/2*E(8)^3 ], 
  [ 214, 195, -1/4*E(4) ], [ 214, 200, -1/4*E(4) ], [ 214, 201, -1/4*E(4) ], 
  [ 214, 205, 1/2*E(8)-1/2*E(8)^3 ], [ 214, 206, -1/4*E(4) ], [ 214, 210, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 215, 219, 1/8*E(8)-1/8*E(8)^3 ], [ 215, 222, -1/8*E(8)+1/8*E(8)^3 ], [ 215, 223, -1/8*E(8)+1/8*E(8)^3 ]
    , [ 215, 224, 1/4*E(4) ], [ 215, 225, 1/8*E(8)-1/8*E(8)^3 ], [ 215, 227, -1/4*E(4) ], 
  [ 216, 194, -1/2*E(8)+1/2*E(8)^3 ], [ 216, 199, 1/2*E(4) ], [ 216, 204, -1/2*E(4) ], 
  [ 217, 209, 1/4*E(8)+1/4*E(8)^3 ], [ 217, 213, 1/4*E(8)+1/4*E(8)^3 ], [ 217, 214, -3/4*E(8)-3/4*E(8)^3 ],
  [ 217, 218, -3/4*E(8)-3/4*E(8)^3 ], [ 218, 195, -1/4*E(4) ], [ 218, 200, -1/4*E(4) ], 
  [ 218, 201, -1/4*E(4) ], [ 218, 205, -1/2*E(8)+1/2*E(8)^3 ], [ 218, 206, -1/4*E(4) ], 
  [ 218, 210, -1/2*E(8)+1/2*E(8)^3 ], [ 219, 219, -1/4 ], [ 219, 222, 1/4 ], [ 219, 223, -1/4 ], 
  [ 219, 224, -1/4*E(8)-1/4*E(8)^3 ], [ 219, 225, 1/4 ], [ 219, 226, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 219, 227, -1/4*E(8)-1/4*E(8)^3 ], [ 219, 228, 1/2*E(8)+1/2*E(8)^3 ], [ 219, 229, 1 ], 
  [ 220, 219, -1/8*E(8)+1/8*E(8)^3 ], [ 220, 222, 1/8*E(8)-1/8*E(8)^3 ], [ 220, 223, 1/8*E(8)-1/8*E(8)^3 ],
  [ 220, 224, 1/4*E(4) ], [ 220, 225, -1/8*E(8)+1/8*E(8)^3 ], [ 220, 227, -1/4*E(4) ], 
  [ 221, 209, 1/4*E(8)+1/4*E(8)^3 ], [ 221, 213, 1/4*E(8)+1/4*E(8)^3 ], [ 221, 214, 1/4*E(8)+1/4*E(8)^3 ], 
  [ 221, 218, 1/4*E(8)+1/4*E(8)^3 ], [ 222, 219, -1/4 ], [ 222, 222, 1/4 ], [ 222, 223, -1/4 ], 
  [ 222, 224, -1/4*E(8)-1/4*E(8)^3 ], [ 222, 225, 1/4 ], [ 222, 226, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 222, 227, -1/4*E(8)-1/4*E(8)^3 ], [ 222, 228, -1/2*E(8)-1/2*E(8)^3 ], [ 222, 229, -1 ], 
  [ 223, 219, 1/4 ], [ 223, 222, -1/4 ], [ 223, 223, 1/4 ], [ 223, 224, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 223, 225, -1/4 ], [ 223, 226, 1/2*E(8)+1/2*E(8)^3 ], [ 223, 227, -1/4*E(8)-1/4*E(8)^3 ], 
  [ 223, 228, -1/2*E(8)-1/2*E(8)^3 ], [ 223, 229, 1 ], [ 224, 230, -1/2*E(8)+1/2*E(8)^3 ], 
  [ 224, 231, 1/2*E(4) ], [ 224, 234, 1/2*E(4) ], [ 225, 219, 1/4 ], [ 225, 222, -1/4 ], [ 225, 223, 1/4 ],
  [ 225, 224, -1/4*E(8)-1/4*E(8)^3 ], [ 225, 225, -1/4 ], [ 225, 226, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 225, 227, -1/4*E(8)-1/4*E(8)^3 ], [ 225, 228, 1/2*E(8)+1/2*E(8)^3 ], [ 225, 229, -1 ], 
  [ 226, 211, 1/4*E(8)-1/4*E(8)^3 ], [ 226, 215, 1/2*E(4) ], [ 226, 220, -1/2*E(4) ], 
  [ 227, 230, 1/2*E(8)-1/2*E(8)^3 ], [ 227, 231, 1/2*E(4) ], [ 227, 234, 1/2*E(4) ], 
  [ 228, 211, 1/4*E(8)-1/4*E(8)^3 ], [ 228, 215, -1/2*E(4) ], [ 228, 220, 1/2*E(4) ], [ 229, 219, -1/4 ], 
  [ 229, 222, -1/4 ], [ 229, 223, 1/4 ], [ 229, 225, 1/4 ], [ 230, 215, E(8)+E(8)^3 ], 
  [ 230, 220, E(8)+E(8)^3 ], [ 231, 219, 1/4*E(8)-1/4*E(8)^3 ], [ 231, 222, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 231, 223, 1/4*E(8)-1/4*E(8)^3 ], [ 231, 225, 1/4*E(8)-1/4*E(8)^3 ], [ 231, 226, E(4) ], 
  [ 231, 228, E(4) ], [ 232, 233, -1/2*E(8)-1/2*E(8)^3 ], [ 232, 235, 1/2*E(8)+1/2*E(8)^3 ], 
  [ 233, 236, -1/4*E(8)+1/4*E(8)^3 ], [ 233, 237, 1/2*E(4) ], [ 233, 238, 1/2*E(4) ], 
  [ 234, 219, 1/4*E(8)-1/4*E(8)^3 ], [ 234, 222, 1/4*E(8)-1/4*E(8)^3 ], [ 234, 223, 1/4*E(8)-1/4*E(8)^3 ], 
  [ 234, 225, 1/4*E(8)-1/4*E(8)^3 ], [ 234, 226, -E(4) ], [ 234, 228, -E(4) ], 
  [ 235, 236, -1/4*E(8)+1/4*E(8)^3 ], [ 235, 237, -1/2*E(4) ], [ 235, 238, -1/2*E(4) ], 
  [ 236, 237, -E(8)-E(8)^3 ], [ 236, 238, E(8)+E(8)^3 ], [ 237, 232, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 237, 233, -1/2*E(4) ], [ 237, 235, -1/2*E(4) ], [ 238, 232, 1/2*E(8)-1/2*E(8)^3 ], 
  [ 238, 233, 1/2*E(4) ], [ 238, 235, 1/2*E(4) ], [ 239, 239, -1/2 ], [ 239, 240, -3/2 ], 
  [ 240, 239, 1/2 ], [ 240, 240, -1/2 ], [ 241, 3, 1/2 ], [ 241, 123, 1/2 ], [ 241, 241, -1/2 ], 
  [ 241, 244, 1/2 ], [ 242, 242, 1 ], [ 243, 3, 1/2 ], [ 243, 5, 1/8*E(8)+1/8*E(8)^3 ], [ 243, 6, 1/4 ], 
  [ 243, 13, -1/8*E(8)-1/8*E(8)^3 ], [ 243, 123, 1/2 ], [ 243, 125, -1/2*E(8)-1/2*E(8)^3 ], 
  [ 243, 126, 1/4 ], [ 243, 133, 1/2*E(8)+1/2*E(8)^3 ], [ 243, 241, -1/2 ], [ 243, 242, 1 ], 
  [ 243, 245, 1/4 ], [ 243, 247, 1/4 ], [ 244, 3, 1/2 ], [ 244, 6, 1/2 ], [ 244, 123, 1/2 ], 
  [ 244, 126, 1/2 ], [ 244, 241, -1/2 ], [ 244, 242, 2 ], [ 244, 244, -1/2 ], [ 244, 245, 1/2 ], 
  [ 244, 247, 1/2 ], [ 245, 3, 1/2 ], [ 245, 123, 1/2 ], [ 245, 241, -1/2 ], [ 245, 242, 2 ], 
  [ 245, 244, -1/2 ], [ 245, 247, 1 ], [ 246, 1, -1/4*E(8)-1/4*E(8)^3 ], [ 246, 3, 1/4 ], 
  [ 246, 9, -1/4*E(8)-1/4*E(8)^3 ], [ 246, 121, 1/4*E(8)+1/4*E(8)^3 ], [ 246, 123, 1/4 ], 
  [ 246, 129, 1/4*E(8)+1/4*E(8)^3 ], [ 246, 241, -1/4 ], [ 246, 242, 1 ], [ 246, 244, -1/4 ], 
  [ 246, 247, 1 ], [ 247, 247, 1 ], [ 248, 8, 1/4 ], [ 248, 128, 3/4 ], [ 248, 247, 3/4 ], 
  [ 248, 248, -1/2 ] ] ],


[ [ 1, 0, 0, 0, 0, 1, 0, 0 ],
  [ 248, [ 1, 9, -2 ], [ 2, 2, 1 ], [ 3, 123, 1/4 ], [ 4, 19, 1 ], [ 5, 125, 4 ], [ 6, 13, -1/2 ], 
  [ 7, 15, 4 ], [ 8, 128, 1/16 ], [ 9, 1, -1/2 ], [ 10, 25, 1 ], [ 11, 12, 1/4 ], [ 12, 11, 4 ], 
  [ 13, 6, -2 ], [ 14, 29, -2 ], [ 15, 7, 1/4 ], [ 16, 24, -1/2 ], [ 17, 18, 1/4 ], [ 18, 17, 4 ], 
  [ 19, 4, 1 ], [ 20, 27, -2 ], [ 21, 22, -8 ], [ 22, 21, -1/8 ], [ 23, 30, -1/2 ], [ 24, 16, -2 ], 
  [ 25, 10, 1 ], [ 26, 33, -2 ], [ 27, 20, -1/2 ], [ 28, 43, -8 ], [ 29, 14, -1/2 ], [ 30, 23, -2 ], 
  [ 31, 31, 1 ], [ 32, 32, 1 ], [ 33, 26, -1/2 ], [ 34, 50, -8 ], [ 35, 36, -2 ], [ 36, 35, -1/2 ], 
  [ 37, 44, 2 ], [ 38, 38, 1 ], [ 39, 47, 4 ], [ 40, 48, 1/2 ], [ 41, 42, -2 ], [ 42, 41, -1/2 ], 
  [ 43, 28, -1/8 ], [ 44, 37, 1/2 ], [ 45, 57, 1 ], [ 46, 54, 4 ], [ 47, 39, 1/4 ], [ 48, 40, 2 ], 
  [ 49, 62, 2 ], [ 50, 34, -1/8 ], [ 51, 52, 1/4 ], [ 52, 51, 4 ], [ 53, 72, 4 ], [ 54, 46, 1/4 ], 
  [ 55, 56, 8 ], [ 56, 55, 1/8 ], [ 57, 45, 1 ], [ 58, 67, 1 ], [ 59, 65, 16 ], [ 60, 64, 1/4 ], 
  [ 61, 68, -4 ], [ 62, 49, 1/2 ], [ 63, 63, 1 ], [ 64, 60, 4 ], [ 65, 59, 1/16 ], [ 66, 78, -8 ], 
  [ 67, 58, 1 ], [ 68, 61, -1/4 ], [ 69, 69, 1 ], [ 70, 77, 4 ], [ 71, 73, -2 ], [ 72, 53, 1/4 ], 
  [ 73, 71, -1/2 ], [ 74, 74, 1 ], [ 75, 81, 4 ], [ 76, 87, 2 ], [ 77, 70, 1/4 ], [ 78, 66, -1/8 ], 
  [ 79, 84, 2 ], [ 80, 90, 2 ], [ 81, 75, 1/4 ], [ 82, 83, 8 ], [ 83, 82, 1/8 ], [ 84, 79, 1/2 ], 
  [ 85, 86, 8 ], [ 86, 85, 1/8 ], [ 87, 76, 1/2 ], [ 88, 92, -1/2 ], [ 89, 98, 8 ], [ 90, 80, 1/2 ], 
  [ 91, 95, -1/2 ], [ 92, 88, -2 ], [ 93, 94, 2 ], [ 94, 93, 1/2 ], [ 95, 91, -2 ], [ 96, 96, 1 ], 
  [ 97, 101, 4 ], [ 98, 89, 1/8 ], [ 99, 102, -2 ], [ 100, 100, 1 ], [ 101, 97, 1/4 ], [ 102, 99, -1/2 ], 
  [ 103, 108, -1 ], [ 104, 104, -1 ], [ 105, 106, -1/4 ], [ 106, 105, -4 ], [ 107, 109, -1/2 ], 
  [ 108, 103, -1 ], [ 109, 107, -2 ], [ 110, 110, 1 ], [ 111, 114, -2 ], [ 112, 112, 1 ], [ 113, 115, -2 ],
  [ 114, 111, -1/2 ], [ 115, 113, -1/2 ], [ 116, 117, 1/2 ], [ 117, 116, 2 ], [ 118, 118, 1 ], 
  [ 119, 120, -4 ], [ 120, 119, -1/4 ], [ 121, 129, -1/2 ], [ 122, 122, 1 ], [ 123, 3, 4 ], 
  [ 124, 139, 1 ], [ 125, 5, 1/4 ], [ 126, 133, -2 ], [ 127, 135, 1/4 ], [ 128, 8, 16 ], [ 129, 121, -2 ], 
  [ 130, 145, 1 ], [ 131, 132, 4 ], [ 132, 131, 1/4 ], [ 133, 126, -1/2 ], [ 134, 149, -1/2 ], 
  [ 135, 127, 4 ], [ 136, 144, -2 ], [ 137, 138, 4 ], [ 138, 137, 1/4 ], [ 139, 124, 1 ], 
  [ 140, 147, -1/2 ], [ 141, 142, -1/8 ], [ 142, 141, -8 ], [ 143, 150, -2 ], [ 144, 136, -1/2 ], 
  [ 145, 130, 1 ], [ 146, 153, -1/2 ], [ 147, 140, -2 ], [ 148, 163, -1/8 ], [ 149, 134, -2 ], 
  [ 150, 143, -1/2 ], [ 151, 151, 1 ], [ 152, 152, 1 ], [ 153, 146, -2 ], [ 154, 170, -1/8 ], 
  [ 155, 156, -1/2 ], [ 156, 155, -2 ], [ 157, 164, 1/2 ], [ 158, 158, 1 ], [ 159, 167, 1/4 ], 
  [ 160, 168, 2 ], [ 161, 162, -1/2 ], [ 162, 161, -2 ], [ 163, 148, -8 ], [ 164, 157, 2 ], 
  [ 165, 177, 1 ], [ 166, 174, 1/4 ], [ 167, 159, 4 ], [ 168, 160, 1/2 ], [ 169, 182, 1/2 ], 
  [ 170, 154, -8 ], [ 171, 172, 4 ], [ 172, 171, 1/4 ], [ 173, 192, 1/4 ], [ 174, 166, 4 ], 
  [ 175, 176, 1/8 ], [ 176, 175, 8 ], [ 177, 165, 1 ], [ 178, 187, 1 ], [ 179, 185, 1/16 ], 
  [ 180, 184, 4 ], [ 181, 188, -1/4 ], [ 182, 169, 2 ], [ 183, 183, 1 ], [ 184, 180, 1/4 ], 
  [ 185, 179, 16 ], [ 186, 198, -1/8 ], [ 187, 178, 1 ], [ 188, 181, -4 ], [ 189, 189, 1 ], 
  [ 190, 197, 1/4 ], [ 191, 193, -1/2 ], [ 192, 173, 4 ], [ 193, 191, -2 ], [ 194, 194, 1 ], 
  [ 195, 201, 1/4 ], [ 196, 207, 1/2 ], [ 197, 190, 4 ], [ 198, 186, -8 ], [ 199, 204, 1/2 ], 
  [ 200, 210, 1/2 ], [ 201, 195, 4 ], [ 202, 203, 1/8 ], [ 203, 202, 8 ], [ 204, 199, 2 ], 
  [ 205, 206, 1/8 ], [ 206, 205, 8 ], [ 207, 196, 2 ], [ 208, 212, -2 ], [ 209, 218, 1/8 ], 
  [ 210, 200, 2 ], [ 211, 215, -2 ], [ 212, 208, -1/2 ], [ 213, 214, 1/2 ], [ 214, 213, 2 ], 
  [ 215, 211, -1/2 ], [ 216, 216, 1 ], [ 217, 221, 1/4 ], [ 218, 209, 8 ], [ 219, 222, -1/2 ], 
  [ 220, 220, 1 ], [ 221, 217, 4 ], [ 222, 219, -2 ], [ 223, 228, -1 ], [ 224, 224, -1 ], [ 225, 226, -4 ],
  [ 226, 225, -1/4 ], [ 227, 229, -2 ], [ 228, 223, -1 ], [ 229, 227, -1/2 ], [ 230, 230, 1 ], 
  [ 231, 234, -1/2 ], [ 232, 232, 1 ], [ 233, 235, -1/2 ], [ 234, 231, -2 ], [ 235, 233, -2 ], 
  [ 236, 237, 2 ], [ 237, 236, 1/2 ], [ 238, 238, 1 ], [ 239, 240, -1/4 ], [ 240, 239, -4 ], 
  [ 241, 241, 1 ], [ 242, 242, 1 ], [ 243, 241, 1 ], [ 243, 243, -1 ], [ 243, 244, 1 ], [ 244, 244, 1 ], 
  [ 245, 244, 1 ], [ 245, 245, -1 ], [ 245, 246, 1 ], [ 246, 246, 1 ], [ 247, 247, 1 ], [ 248, 247, 1 ], 
  [ 248, 248, -1 ] ] ],

[ [ 0, 0, 0, 0, 2, 0, 0, 0 ],
  [ 248, [ 1, 1, -1/12*E(3) ], [ 1, 2, -1/12*E(3) ], [ 1, 4, 2/3*E(3) ], 
  [ 1, 10, -4/9*E(3)-2/9*E(3)^2 ], [ 1, 11, 1/6*E(3)+1/12*E(3)^2 ], 
  [ 1, 16, 1/12 ], [ 1, 17, 1/12 ], [ 1, 23, 1/36*E(3)+1/18*E(3)^2 ], 
  [ 1, 121, -1/12*E(3)^2 ], [ 1, 122, -1/12*E(3)^2 ], [ 1, 123, -2/3*E(3)^2 ],
  [ 1, 129, -2/9*E(3)-4/9*E(3)^2 ], [ 1, 131, 1/12*E(3)+1/6*E(3)^2 ], 
  [ 1, 136, -1/12 ], [ 1, 137, -1/12 ], [ 1, 143, 1/18*E(3)+1/36*E(3)^2 ], 
  [ 1, 241, 1/9*E(3)-1/9*E(3)^2 ], [ 1, 242, 1/9*E(3)-1/9*E(3)^2 ], 
  [ 1, 243, -1/12*E(3)+1/12*E(3)^2 ], [ 1, 244, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 1, 245, 1/18*E(3)-1/18*E(3)^2 ], [ 2, 1, -3/4*E(3) ], 
  [ 2, 2, -1/12*E(3) ], [ 2, 3, -1/6*E(3) ], [ 2, 9, 1/3*E(3)+1/6*E(3)^2 ], 
  [ 2, 11, -1/2*E(3)-1/4*E(3)^2 ], [ 2, 16, -3/4 ], [ 2, 17, -1/12 ], 
  [ 2, 23, -1/12*E(3)-1/6*E(3)^2 ], [ 2, 121, -1/12*E(3)^2 ], 
  [ 2, 122, -3/4*E(3)^2 ], [ 2, 124, 1/6*E(3)^2 ], 
  [ 2, 130, 1/6*E(3)+1/3*E(3)^2 ], [ 2, 131, -1/4*E(3)-1/2*E(3)^2 ], 
  [ 2, 136, 1/12 ], [ 2, 137, 3/4 ], [ 2, 143, -1/6*E(3)-1/12*E(3)^2 ], 
  [ 2, 241, 1/3*E(3)-1/3*E(3)^2 ], [ 2, 242, 1/3*E(3)-1/3*E(3)^2 ], 
  [ 2, 243, -1/4*E(3)+1/4*E(3)^2 ], [ 2, 244, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 2, 245, 1/6*E(3)-1/6*E(3)^2 ], [ 3, 2, -2/3*E(3) ], [ 3, 3, 1/6*E(3) ], 
  [ 3, 9, 1/3*E(3)+1/6*E(3)^2 ], [ 3, 17, -2/3 ], 
  [ 3, 23, 2/3*E(3)+4/3*E(3)^2 ], [ 3, 121, -2/3*E(3)^2 ], 
  [ 3, 124, -1/6*E(3)^2 ], [ 3, 130, 1/6*E(3)+1/3*E(3)^2 ], [ 3, 136, 2/3 ], 
  [ 3, 143, 4/3*E(3)+2/3*E(3)^2 ], [ 3, 241, -2/3*E(3)+2/3*E(3)^2 ], 
  [ 3, 242, -2/3*E(3)+2/3*E(3)^2 ], [ 3, 245, 2/3*E(3)-2/3*E(3)^2 ], 
  [ 4, 1, 1/6*E(3) ], [ 4, 4, 1/6*E(3) ], [ 4, 10, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 4, 16, -1/6 ], [ 4, 23, 1/18*E(3)+1/9*E(3)^2 ], [ 4, 122, 1/6*E(3)^2 ], 
  [ 4, 123, -1/6*E(3)^2 ], [ 4, 129, 1/18*E(3)+1/9*E(3)^2 ], [ 4, 137, 1/6 ], 
  [ 4, 143, 1/9*E(3)+1/18*E(3)^2 ], [ 4, 241, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 4, 242, 1/18*E(3)-1/18*E(3)^2 ], [ 4, 245, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 5, 5, 2/9 ], [ 5, 12, -4/9*E(3)-8/9*E(3)^2 ], [ 5, 13, -1/9*E(3)^2 ], 
  [ 5, 18, 4/9*E(3)^2 ], [ 5, 19, 1/9*E(3)^2 ], 
  [ 5, 20, 2/9*E(3)-2/9*E(3)^2 ], [ 5, 24, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 5, 25, 1/9*E(3)-1/9*E(3)^2 ], [ 5, 26, -2/9*E(3) ], [ 5, 27, -1/18*E(3) ],
  [ 5, 29, 1/9*E(3) ], [ 5, 30, -1/9*E(3) ], [ 5, 31, -1/9*E(3)-1/18*E(3)^2 ],
  [ 5, 32, -4/9*E(3) ], [ 5, 33, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 5, 36, 4/9*E(3)+2/9*E(3)^2 ], [ 5, 37, 8/9*E(3)+4/9*E(3)^2 ], 
  [ 5, 38, 1/18 ], [ 5, 40, 2/9 ], [ 5, 42, 2/9 ], [ 5, 43, 1/18 ], 
  [ 5, 44, 2/9 ], [ 5, 45, 2/9*E(3)+4/9*E(3)^2 ], 
  [ 5, 47, -1/18*E(3)-1/9*E(3)^2 ], [ 5, 50, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 5, 51, -1/9*E(3)^2 ], [ 5, 54, -1/18*E(3)^2 ], [ 5, 56, -2/9*E(3)^2 ], 
  [ 5, 60, -2/9*E(3)+2/9*E(3)^2 ], [ 5, 65, 1/9*E(3) ], [ 6, 6, -2/9*E(3)^2 ],
  [ 6, 7, -5/12*E(3)-5/6*E(3)^2 ], [ 6, 14, 1/3*E(3)-1/3*E(3)^2 ], 
  [ 6, 15, -5/18*E(3)^2 ], [ 6, 22, 2/9*E(3) ], [ 6, 126, 25/36*E(3) ], 
  [ 6, 128, -5/6*E(3)-5/12*E(3)^2 ], [ 6, 135, -5/18*E(3) ], 
  [ 6, 142, -25/36*E(3)^2 ], [ 6, 245, -5/9 ], [ 6, 246, 5/6 ], 
  [ 6, 247, -5/18 ], [ 6, 248, 5/18 ], [ 7, 6, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 7, 8, -1/3*E(3) ], [ 7, 15, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 7, 22, 1/9*E(3)+2/9*E(3)^2 ], [ 7, 127, -1/3*E(3)^2 ], 
  [ 7, 135, -1/9*E(3)-2/9*E(3)^2 ], [ 7, 246, 1/9*E(3)-1/9*E(3)^2 ], 
  [ 7, 247, -1/9*E(3)+1/9*E(3)^2 ], [ 7, 248, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 8, 7, -3/4*E(3) ], [ 8, 15, 1/3*E(3)+1/6*E(3)^2 ], 
  [ 8, 126, 5/12*E(3)+5/6*E(3)^2 ], [ 8, 128, -3/4*E(3)^2 ], 
  [ 8, 135, 1/6*E(3)+1/3*E(3)^2 ], [ 8, 142, -5/6*E(3)-5/12*E(3)^2 ], 
  [ 8, 246, 1/6*E(3)-1/6*E(3)^2 ], [ 8, 247, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 8, 248, -1/6*E(3)+1/6*E(3)^2 ], [ 9, 2, 2/9*E(3)+4/9*E(3)^2 ], 
  [ 9, 3, 1/18*E(3)+1/9*E(3)^2 ], [ 9, 9, -1/18*E(3)^2 ], 
  [ 9, 10, 16/9*E(3)^2 ], [ 9, 17, -2/9*E(3)+2/9*E(3)^2 ], 
  [ 9, 23, 2/9*E(3) ], [ 9, 121, 4/9*E(3)+2/9*E(3)^2 ], 
  [ 9, 124, -1/9*E(3)-1/18*E(3)^2 ], [ 9, 129, 16/9*E(3) ], 
  [ 9, 130, -1/18*E(3) ], [ 9, 136, -2/9*E(3)+2/9*E(3)^2 ], 
  [ 9, 143, 2/9*E(3)^2 ], [ 9, 241, -2/9 ], [ 9, 242, 2/9 ], [ 9, 243, -4/9 ],
  [ 9, 244, 4/9 ], [ 9, 245, -2/9 ], [ 10, 1, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 10, 4, 1/6*E(3)+1/3*E(3)^2 ], [ 10, 9, 1/9*E(3)^2 ], 
  [ 10, 10, -1/18*E(3)^2 ], [ 10, 16, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 10, 23, 1/18*E(3) ], [ 10, 122, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 10, 123, -1/3*E(3)-1/6*E(3)^2 ], [ 10, 129, -1/18*E(3) ], 
  [ 10, 130, 1/9*E(3) ], [ 10, 137, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 10, 143, 1/18*E(3)^2 ], [ 10, 241, -1/18 ], [ 10, 242, 1/18 ], 
  [ 10, 243, -1/9 ], [ 10, 244, 1/9 ], [ 10, 245, -1/18 ], 
  [ 11, 1, 1/4*E(3)+1/2*E(3)^2 ], [ 11, 2, -1/12*E(3)-1/6*E(3)^2 ], 
  [ 11, 11, 1/4*E(3)^2 ], [ 11, 16, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 11, 17, -1/12*E(3)+1/12*E(3)^2 ], [ 11, 23, -1/4*E(3) ], 
  [ 11, 121, -1/6*E(3)-1/12*E(3)^2 ], [ 11, 122, 1/2*E(3)+1/4*E(3)^2 ], 
  [ 11, 131, 1/4*E(3) ], [ 11, 136, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 11, 137, -1/4*E(3)+1/4*E(3)^2 ], [ 11, 143, -1/4*E(3)^2 ], 
  [ 11, 241, -1/2 ], [ 11, 242, 1/2 ], [ 11, 243, 1/4 ], [ 11, 244, -1/4 ], 
  [ 12, 5, 1/27*E(3)-1/27*E(3)^2 ], [ 12, 12, 1/3*E(3) ], 
  [ 12, 13, 1/27*E(3)+1/54*E(3)^2 ], [ 12, 18, 2/27*E(3)+1/27*E(3)^2 ], 
  [ 12, 19, -2/27*E(3)-1/27*E(3)^2 ], [ 12, 20, -1/6 ], 
  [ 12, 26, 1/54*E(3)+1/27*E(3)^2 ], [ 12, 27, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 12, 29, 1/54*E(3)+1/27*E(3)^2 ], [ 12, 30, 1/27*E(3)+2/27*E(3)^2 ], 
  [ 12, 32, -1/27*E(3)-2/27*E(3)^2 ], [ 12, 36, 1/6*E(3)^2 ], 
  [ 12, 37, -1/3*E(3)^2 ], [ 12, 38, -1/54*E(3)+1/54*E(3)^2 ], 
  [ 12, 40, 1/54*E(3)-1/54*E(3)^2 ], [ 12, 42, -1/54*E(3)+1/54*E(3)^2 ], 
  [ 12, 43, 1/54*E(3)-1/54*E(3)^2 ], [ 12, 44, -1/27*E(3)+1/27*E(3)^2 ], 
  [ 12, 45, 1/6*E(3) ], [ 12, 51, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 12, 54, -1/27*E(3)-1/54*E(3)^2 ], [ 12, 56, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 12, 60, -1/6 ], [ 12, 65, -1/54*E(3)-1/27*E(3)^2 ], [ 13, 5, -5/18 ], 
  [ 13, 12, 5/9*E(3)+10/9*E(3)^2 ], [ 13, 13, -1/9*E(3)^2 ], 
  [ 13, 18, -5/9*E(3)^2 ], [ 13, 19, -5/36*E(3)^2 ], 
  [ 13, 20, 2/9*E(3)-2/9*E(3)^2 ], [ 13, 21, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 13, 24, 5/36*E(3)-5/36*E(3)^2 ], [ 13, 25, -5/36*E(3)+5/36*E(3)^2 ], 
  [ 13, 26, -2/9*E(3) ], [ 13, 27, -1/18*E(3) ], [ 13, 28, E(3) ], 
  [ 13, 29, 1/9*E(3) ], [ 13, 30, 5/36*E(3) ], 
  [ 13, 31, -1/9*E(3)-1/18*E(3)^2 ], [ 13, 32, 5/9*E(3) ], 
  [ 13, 33, 1/9*E(3)+1/18*E(3)^2 ], [ 13, 34, -2/3*E(3)-1/3*E(3)^2 ], 
  [ 13, 35, -1/6*E(3)-1/12*E(3)^2 ], [ 13, 36, 4/9*E(3)+2/9*E(3)^2 ], 
  [ 13, 37, -10/9*E(3)-5/9*E(3)^2 ], [ 13, 38, 1/18 ], [ 13, 39, 1/4 ], 
  [ 13, 40, 2/9 ], [ 13, 41, -1/4 ], [ 13, 42, 2/9 ], [ 13, 43, 1/18 ], 
  [ 13, 44, -5/18 ], [ 13, 45, 2/9*E(3)+4/9*E(3)^2 ], 
  [ 13, 46, -1/12*E(3)-1/6*E(3)^2 ], [ 13, 47, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 13, 49, -1/3*E(3)-2/3*E(3)^2 ], [ 13, 50, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 13, 51, -1/9*E(3)^2 ], [ 13, 53, E(3)^2 ], [ 13, 54, -1/18*E(3)^2 ], 
  [ 13, 56, -2/9*E(3)^2 ], [ 13, 58, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 13, 60, -2/9*E(3)+2/9*E(3)^2 ], [ 13, 65, 1/9*E(3) ], 
  [ 14, 6, 4/9*E(3)+2/9*E(3)^2 ], [ 14, 15, -5/9*E(3)-5/18*E(3)^2 ], 
  [ 14, 22, -2/9*E(3)-4/9*E(3)^2 ], [ 14, 135, -5/18*E(3)-5/9*E(3)^2 ], 
  [ 14, 246, 5/18*E(3)-5/18*E(3)^2 ], [ 14, 247, -5/18*E(3)+5/18*E(3)^2 ], 
  [ 14, 248, -5/18*E(3)+5/18*E(3)^2 ], [ 15, 6, -1/9*E(3)^2 ], 
  [ 15, 7, -1/6*E(3)-1/3*E(3)^2 ], [ 15, 8, 1/9*E(3)+2/9*E(3)^2 ], 
  [ 15, 14, -1/6*E(3)+1/6*E(3)^2 ], [ 15, 15, 1/9*E(3)^2 ], 
  [ 15, 22, 1/9*E(3) ], [ 15, 126, -5/18*E(3) ], 
  [ 15, 127, 2/9*E(3)+1/9*E(3)^2 ], [ 15, 128, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 15, 134, -5/18*E(3)+5/18*E(3)^2 ], [ 15, 135, 1/9*E(3) ], 
  [ 15, 142, 5/18*E(3)^2 ], [ 15, 245, -5/18 ], [ 15, 246, 1/6 ], 
  [ 15, 247, 11/18 ], [ 15, 248, -11/18 ], [ 16, 1, 1/12*E(3) ], 
  [ 16, 2, -1/12*E(3) ], [ 16, 4, -2/3*E(3) ], 
  [ 16, 10, -4/9*E(3)-2/9*E(3)^2 ], [ 16, 11, -1/6*E(3)-1/12*E(3)^2 ], 
  [ 16, 16, -1/12 ], [ 16, 17, 1/12 ], [ 16, 23, 1/36*E(3)+1/18*E(3)^2 ], 
  [ 16, 121, -1/12*E(3)^2 ], [ 16, 122, 1/12*E(3)^2 ], 
  [ 16, 123, 2/3*E(3)^2 ], [ 16, 129, -2/9*E(3)-4/9*E(3)^2 ], 
  [ 16, 131, -1/12*E(3)-1/6*E(3)^2 ], [ 16, 136, -1/12 ], [ 16, 137, 1/12 ], 
  [ 16, 143, 1/18*E(3)+1/36*E(3)^2 ], [ 16, 241, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 16, 242, -1/18*E(3)+1/18*E(3)^2 ], [ 16, 243, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 16, 244, 1/12*E(3)-1/12*E(3)^2 ], [ 16, 245, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 17, 1, 3/4*E(3) ], [ 17, 2, -1/12*E(3) ], [ 17, 3, -1/6*E(3) ], 
  [ 17, 9, -1/3*E(3)-1/6*E(3)^2 ], [ 17, 11, -1/2*E(3)-1/4*E(3)^2 ], 
  [ 17, 16, 3/4 ], [ 17, 17, -1/12 ], [ 17, 23, 1/12*E(3)+1/6*E(3)^2 ], 
  [ 17, 121, -1/12*E(3)^2 ], [ 17, 122, 3/4*E(3)^2 ], [ 17, 124, 1/6*E(3)^2 ],
  [ 17, 130, -1/6*E(3)-1/3*E(3)^2 ], [ 17, 131, -1/4*E(3)-1/2*E(3)^2 ], 
  [ 17, 136, 1/12 ], [ 17, 137, -3/4 ], [ 17, 143, 1/6*E(3)+1/12*E(3)^2 ], 
  [ 17, 241, 1/6*E(3)-1/6*E(3)^2 ], [ 17, 242, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 17, 243, -1/4*E(3)+1/4*E(3)^2 ], [ 17, 244, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 17, 245, 1/3*E(3)-1/3*E(3)^2 ], [ 18, 5, 1/9 ], 
  [ 18, 12, 1/9*E(3)+2/9*E(3)^2 ], [ 18, 13, -1/18*E(3)^2 ], 
  [ 18, 18, -1/9*E(3)^2 ], [ 18, 19, -1/9*E(3)^2 ], 
  [ 18, 20, -1/18*E(3)+1/18*E(3)^2 ], [ 18, 24, -2/9*E(3)+2/9*E(3)^2 ], 
  [ 18, 26, 1/18*E(3) ], [ 18, 27, 1/18*E(3) ], [ 18, 29, 1/18*E(3) ], 
  [ 18, 30, 1/9*E(3) ], [ 18, 31, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 18, 32, 1/9*E(3) ], [ 18, 36, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 18, 37, -2/9*E(3)-1/9*E(3)^2 ], [ 18, 38, -1/18 ], [ 18, 40, -1/18 ], 
  [ 18, 42, -1/18 ], [ 18, 43, -1/18 ], [ 18, 44, 1/9 ], 
  [ 18, 45, -1/18*E(3)-1/9*E(3)^2 ], [ 18, 47, -1/9*E(3)-2/9*E(3)^2 ], 
  [ 18, 51, -1/18*E(3)^2 ], [ 18, 54, 1/18*E(3)^2 ], [ 18, 56, 1/18*E(3)^2 ], 
  [ 18, 60, 1/18*E(3)-1/18*E(3)^2 ], [ 18, 65, 1/18*E(3) ], [ 19, 5, 1/9 ], 
  [ 19, 12, -4/9*E(3)-8/9*E(3)^2 ], [ 19, 13, -1/18*E(3)^2 ], 
  [ 19, 18, -4/9*E(3)^2 ], [ 19, 19, 2/9*E(3)^2 ], 
  [ 19, 20, 2/9*E(3)-2/9*E(3)^2 ], [ 19, 24, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 19, 25, -1/9*E(3)+1/9*E(3)^2 ], [ 19, 26, 2/9*E(3) ], 
  [ 19, 27, -1/9*E(3) ], [ 19, 29, 1/18*E(3) ], [ 19, 30, -2/9*E(3) ], 
  [ 19, 31, -1/9*E(3)-1/18*E(3)^2 ], [ 19, 32, 4/9*E(3) ], 
  [ 19, 33, -1/9*E(3)-1/18*E(3)^2 ], [ 19, 36, 4/9*E(3)+2/9*E(3)^2 ], 
  [ 19, 37, 8/9*E(3)+4/9*E(3)^2 ], [ 19, 38, 1/9 ], [ 19, 40, -2/9 ], 
  [ 19, 42, -2/9 ], [ 19, 43, 1/9 ], [ 19, 44, 1/9 ], 
  [ 19, 45, 2/9*E(3)+4/9*E(3)^2 ], [ 19, 47, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 19, 50, -1/18*E(3)-1/9*E(3)^2 ], [ 19, 51, -1/18*E(3)^2 ], 
  [ 19, 54, -1/9*E(3)^2 ], [ 19, 56, 2/9*E(3)^2 ], 
  [ 19, 60, -2/9*E(3)+2/9*E(3)^2 ], [ 19, 65, 1/18*E(3) ], 
  [ 20, 5, -5/108*E(3)+5/108*E(3)^2 ], [ 20, 12, -5/12*E(3) ], 
  [ 20, 13, 1/27*E(3)+1/54*E(3)^2 ], [ 20, 18, -5/54*E(3)-5/108*E(3)^2 ], 
  [ 20, 19, 5/54*E(3)+5/108*E(3)^2 ], [ 20, 20, -1/6 ], [ 20, 21, -1/12 ], 
  [ 20, 26, 1/54*E(3)+1/27*E(3)^2 ], [ 20, 27, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 20, 28, 1/4*E(3)+1/2*E(3)^2 ], [ 20, 29, 1/54*E(3)+1/27*E(3)^2 ], 
  [ 20, 30, -5/108*E(3)-5/54*E(3)^2 ], [ 20, 32, 5/108*E(3)+5/54*E(3)^2 ], 
  [ 20, 34, 1/12*E(3)^2 ], [ 20, 35, -1/12*E(3)^2 ], [ 20, 36, 1/6*E(3)^2 ], 
  [ 20, 37, 5/12*E(3)^2 ], [ 20, 38, -1/54*E(3)+1/54*E(3)^2 ], 
  [ 20, 40, 1/54*E(3)-1/54*E(3)^2 ], [ 20, 42, -1/54*E(3)+1/54*E(3)^2 ], 
  [ 20, 43, 1/54*E(3)-1/54*E(3)^2 ], [ 20, 44, 5/108*E(3)-5/108*E(3)^2 ], 
  [ 20, 45, 1/6*E(3) ], [ 20, 46, -1/12*E(3) ], [ 20, 49, 1/12*E(3) ], 
  [ 20, 51, -1/27*E(3)-1/54*E(3)^2 ], [ 20, 53, 1/2*E(3)+1/4*E(3)^2 ], 
  [ 20, 54, -1/27*E(3)-1/54*E(3)^2 ], [ 20, 56, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 20, 58, 1/12 ], [ 20, 60, -1/6 ], [ 20, 65, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 21, 13, 2/9*E(3)+1/9*E(3)^2 ], [ 21, 20, -2/3 ], 
  [ 21, 26, -2/9*E(3)-4/9*E(3)^2 ], [ 21, 27, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 21, 29, -1/9*E(3)-2/9*E(3)^2 ], [ 21, 31, -1/6*E(3)^2 ], 
  [ 21, 33, 1/6*E(3)^2 ], [ 21, 36, -2/3*E(3)^2 ], 
  [ 21, 38, 1/18*E(3)-1/18*E(3)^2 ], [ 21, 40, 2/9*E(3)-2/9*E(3)^2 ], 
  [ 21, 42, -2/9*E(3)+2/9*E(3)^2 ], [ 21, 43, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 21, 45, -2/3*E(3) ], [ 21, 47, -1/6*E(3) ], [ 21, 50, 1/6*E(3) ], 
  [ 21, 51, 2/9*E(3)+1/9*E(3)^2 ], [ 21, 54, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 21, 56, -4/9*E(3)-2/9*E(3)^2 ], [ 21, 60, -2/3 ], 
  [ 21, 65, -1/9*E(3)-2/9*E(3)^2 ], [ 22, 6, 2/9*E(3)^2 ], 
  [ 22, 7, -5/12*E(3)-5/6*E(3)^2 ], [ 22, 14, 1/3*E(3)-1/3*E(3)^2 ], 
  [ 22, 15, 5/18*E(3)^2 ], [ 22, 22, -2/9*E(3) ], [ 22, 126, -25/36*E(3) ], 
  [ 22, 128, -5/6*E(3)-5/12*E(3)^2 ], [ 22, 135, 5/18*E(3) ], 
  [ 22, 142, 25/36*E(3)^2 ], [ 22, 245, 5/9 ], [ 22, 246, -5/6 ], 
  [ 22, 247, 5/18 ], [ 22, 248, -5/18 ], [ 23, 1, -1/12*E(3)-1/6*E(3)^2 ], 
  [ 23, 2, 1/36*E(3)+1/18*E(3)^2 ], [ 23, 3, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 23, 4, -2/3*E(3)-4/3*E(3)^2 ], [ 23, 9, 1/18*E(3)^2 ], 
  [ 23, 10, 2/9*E(3)^2 ], [ 23, 11, -1/4*E(3)^2 ], 
  [ 23, 16, -1/12*E(3)+1/12*E(3)^2 ], [ 23, 17, -1/36*E(3)+1/36*E(3)^2 ], 
  [ 23, 23, 1/36*E(3) ], [ 23, 121, 1/18*E(3)+1/36*E(3)^2 ], 
  [ 23, 122, -1/6*E(3)-1/12*E(3)^2 ], [ 23, 123, 4/3*E(3)+2/3*E(3)^2 ], 
  [ 23, 124, 1/9*E(3)+1/18*E(3)^2 ], [ 23, 129, 2/9*E(3) ], 
  [ 23, 130, 1/18*E(3) ], [ 23, 131, -1/4*E(3) ], 
  [ 23, 136, -1/36*E(3)+1/36*E(3)^2 ], [ 23, 137, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 23, 143, 1/36*E(3)^2 ], [ 23, 241, -5/18 ], [ 23, 242, 5/18 ], 
  [ 23, 243, 25/36 ], [ 23, 244, -25/36 ], [ 23, 245, 2/9 ], 
  [ 24, 5, 1/27*E(3)-1/27*E(3)^2 ], [ 24, 13, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 24, 18, -16/27*E(3)-8/27*E(3)^2 ], [ 24, 19, -2/27*E(3)-1/27*E(3)^2 ], 
  [ 24, 26, -4/27*E(3)-8/27*E(3)^2 ], [ 24, 27, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 24, 29, 1/54*E(3)+1/27*E(3)^2 ], [ 24, 30, 1/27*E(3)+2/27*E(3)^2 ], 
  [ 24, 32, 8/27*E(3)+16/27*E(3)^2 ], [ 24, 38, -1/54*E(3)+1/54*E(3)^2 ], 
  [ 24, 40, -4/27*E(3)+4/27*E(3)^2 ], [ 24, 42, 4/27*E(3)-4/27*E(3)^2 ], 
  [ 24, 43, 1/54*E(3)-1/54*E(3)^2 ], [ 24, 44, -1/27*E(3)+1/27*E(3)^2 ], 
  [ 24, 51, -1/27*E(3)-1/54*E(3)^2 ], [ 24, 54, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 24, 56, -8/27*E(3)-4/27*E(3)^2 ], [ 24, 65, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 25, 5, -1/3*E(3)+1/3*E(3)^2 ], [ 25, 13, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 25, 19, -2/3*E(3)-1/3*E(3)^2 ], [ 25, 27, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 25, 29, -1/6*E(3)-1/3*E(3)^2 ], [ 25, 30, 1/3*E(3)+2/3*E(3)^2 ], 
  [ 25, 38, -1/6*E(3)+1/6*E(3)^2 ], [ 25, 43, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 25, 44, 1/3*E(3)-1/3*E(3)^2 ], [ 25, 51, 1/3*E(3)+1/6*E(3)^2 ], 
  [ 25, 54, -1/3*E(3)-1/6*E(3)^2 ], [ 25, 65, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 26, 5, -5/36 ], [ 26, 12, -5/36*E(3)-5/18*E(3)^2 ], 
  [ 26, 13, -1/18*E(3)^2 ], [ 26, 18, 5/36*E(3)^2 ], [ 26, 19, 5/36*E(3)^2 ], 
  [ 26, 20, -1/18*E(3)+1/18*E(3)^2 ], [ 26, 21, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 26, 24, 5/18*E(3)-5/18*E(3)^2 ], [ 26, 26, 1/18*E(3) ], 
  [ 26, 27, 1/18*E(3) ], [ 26, 28, -1/4*E(3) ], [ 26, 29, 1/18*E(3) ], 
  [ 26, 30, -5/36*E(3) ], [ 26, 31, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 26, 32, -5/36*E(3) ], [ 26, 34, 1/6*E(3)+1/12*E(3)^2 ], 
  [ 26, 35, 1/6*E(3)+1/12*E(3)^2 ], [ 26, 36, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 26, 37, 5/18*E(3)+5/36*E(3)^2 ], [ 26, 38, -1/18 ], [ 26, 39, 1/2 ], 
  [ 26, 40, -1/18 ], [ 26, 42, -1/18 ], [ 26, 43, -1/18 ], [ 26, 44, -5/36 ], 
  [ 26, 45, -1/18*E(3)-1/9*E(3)^2 ], [ 26, 46, 1/12*E(3)+1/6*E(3)^2 ], 
  [ 26, 47, -1/9*E(3)-2/9*E(3)^2 ], [ 26, 49, 1/12*E(3)+1/6*E(3)^2 ], 
  [ 26, 51, -1/18*E(3)^2 ], [ 26, 53, -1/4*E(3)^2 ], [ 26, 54, 1/18*E(3)^2 ], 
  [ 26, 56, 1/18*E(3)^2 ], [ 26, 58, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 26, 60, 1/18*E(3)-1/18*E(3)^2 ], [ 26, 65, 1/18*E(3) ], [ 27, 5, -5/36 ], 
  [ 27, 12, 5/9*E(3)+10/9*E(3)^2 ], [ 27, 13, -1/18*E(3)^2 ], 
  [ 27, 18, 5/9*E(3)^2 ], [ 27, 19, -5/18*E(3)^2 ], 
  [ 27, 20, 2/9*E(3)-2/9*E(3)^2 ], [ 27, 21, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 27, 24, 5/36*E(3)-5/36*E(3)^2 ], [ 27, 25, 5/36*E(3)-5/36*E(3)^2 ], 
  [ 27, 26, 2/9*E(3) ], [ 27, 27, -1/9*E(3) ], [ 27, 28, E(3) ], 
  [ 27, 29, 1/18*E(3) ], [ 27, 30, 5/18*E(3) ], 
  [ 27, 31, -1/9*E(3)-1/18*E(3)^2 ], [ 27, 32, -5/9*E(3) ], 
  [ 27, 33, -1/9*E(3)-1/18*E(3)^2 ], [ 27, 34, 2/3*E(3)+1/3*E(3)^2 ], 
  [ 27, 35, -1/3*E(3)-1/6*E(3)^2 ], [ 27, 36, 4/9*E(3)+2/9*E(3)^2 ], 
  [ 27, 37, -10/9*E(3)-5/9*E(3)^2 ], [ 27, 38, 1/9 ], [ 27, 39, 1/4 ], 
  [ 27, 40, -2/9 ], [ 27, 41, 1/4 ], [ 27, 42, -2/9 ], [ 27, 43, 1/9 ], 
  [ 27, 44, -5/36 ], [ 27, 45, 2/9*E(3)+4/9*E(3)^2 ], 
  [ 27, 46, -1/6*E(3)-1/3*E(3)^2 ], [ 27, 47, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 27, 49, 1/3*E(3)+2/3*E(3)^2 ], [ 27, 50, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 27, 51, -1/18*E(3)^2 ], [ 27, 53, E(3)^2 ], [ 27, 54, -1/9*E(3)^2 ], 
  [ 27, 56, 2/9*E(3)^2 ], [ 27, 58, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 27, 60, -2/9*E(3)+2/9*E(3)^2 ], [ 27, 65, 1/18*E(3) ], 
  [ 28, 13, 1/18*E(3)^2 ], [ 28, 20, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 28, 26, -1/18*E(3) ], [ 28, 27, 1/18*E(3) ], [ 28, 29, 1/18*E(3) ], 
  [ 28, 36, 1/3*E(3)+1/6*E(3)^2 ], [ 28, 38, 1/18 ], [ 28, 40, -1/18 ], 
  [ 28, 42, -1/18 ], [ 28, 43, 1/18 ], [ 28, 45, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 28, 51, -1/18*E(3)^2 ], [ 28, 54, 1/18*E(3)^2 ], [ 28, 56, -1/18*E(3)^2 ],
  [ 28, 60, 1/6*E(3)-1/6*E(3)^2 ], [ 28, 65, -1/18*E(3) ], [ 29, 5, 5/18 ], 
  [ 29, 12, -5/9*E(3)-10/9*E(3)^2 ], [ 29, 13, 1/9*E(3)^2 ], 
  [ 29, 18, 5/9*E(3)^2 ], [ 29, 19, 5/36*E(3)^2 ], 
  [ 29, 20, -2/9*E(3)+2/9*E(3)^2 ], [ 29, 21, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 29, 24, -5/36*E(3)+5/36*E(3)^2 ], [ 29, 25, 5/36*E(3)-5/36*E(3)^2 ], 
  [ 29, 26, 2/9*E(3) ], [ 29, 27, 1/18*E(3) ], [ 29, 28, E(3) ], 
  [ 29, 29, -1/9*E(3) ], [ 29, 30, -5/36*E(3) ], 
  [ 29, 31, 1/9*E(3)+1/18*E(3)^2 ], [ 29, 32, -5/9*E(3) ], 
  [ 29, 33, -1/9*E(3)-1/18*E(3)^2 ], [ 29, 34, -2/3*E(3)-1/3*E(3)^2 ], 
  [ 29, 35, -1/6*E(3)-1/12*E(3)^2 ], [ 29, 36, -4/9*E(3)-2/9*E(3)^2 ], 
  [ 29, 37, 10/9*E(3)+5/9*E(3)^2 ], [ 29, 38, -1/18 ], [ 29, 39, 1/4 ], 
  [ 29, 40, -2/9 ], [ 29, 41, -1/4 ], [ 29, 42, -2/9 ], [ 29, 43, -1/18 ], 
  [ 29, 44, 5/18 ], [ 29, 45, -2/9*E(3)-4/9*E(3)^2 ], 
  [ 29, 46, -1/12*E(3)-1/6*E(3)^2 ], [ 29, 47, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 29, 49, -1/3*E(3)-2/3*E(3)^2 ], [ 29, 50, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 29, 51, 1/9*E(3)^2 ], [ 29, 53, E(3)^2 ], [ 29, 54, 1/18*E(3)^2 ], 
  [ 29, 56, 2/9*E(3)^2 ], [ 29, 58, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 29, 60, 2/9*E(3)-2/9*E(3)^2 ], [ 29, 65, -1/9*E(3) ], [ 30, 5, -1/9 ], 
  [ 30, 12, -4/9*E(3)-8/9*E(3)^2 ], [ 30, 13, 1/18*E(3)^2 ], 
  [ 30, 18, 4/9*E(3)^2 ], [ 30, 19, -2/9*E(3)^2 ], 
  [ 30, 20, 2/9*E(3)-2/9*E(3)^2 ], [ 30, 24, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 30, 25, -1/9*E(3)+1/9*E(3)^2 ], [ 30, 26, -2/9*E(3) ], 
  [ 30, 27, 1/9*E(3) ], [ 30, 29, -1/18*E(3) ], [ 30, 30, 2/9*E(3) ], 
  [ 30, 31, -1/9*E(3)-1/18*E(3)^2 ], [ 30, 32, -4/9*E(3) ], 
  [ 30, 33, -1/9*E(3)-1/18*E(3)^2 ], [ 30, 36, 4/9*E(3)+2/9*E(3)^2 ], 
  [ 30, 37, 8/9*E(3)+4/9*E(3)^2 ], [ 30, 38, -1/9 ], [ 30, 40, 2/9 ], 
  [ 30, 42, 2/9 ], [ 30, 43, -1/9 ], [ 30, 44, -1/9 ], 
  [ 30, 45, 2/9*E(3)+4/9*E(3)^2 ], [ 30, 47, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 30, 50, -1/18*E(3)-1/9*E(3)^2 ], [ 30, 51, 1/18*E(3)^2 ], 
  [ 30, 54, 1/9*E(3)^2 ], [ 30, 56, -2/9*E(3)^2 ], 
  [ 30, 60, -2/9*E(3)+2/9*E(3)^2 ], [ 30, 65, -1/18*E(3) ], 
  [ 31, 5, -5/108*E(3)+5/108*E(3)^2 ], [ 31, 13, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 31, 18, 20/27*E(3)+10/27*E(3)^2 ], [ 31, 19, 5/54*E(3)+5/108*E(3)^2 ], 
  [ 31, 21, -1/12 ], [ 31, 26, -4/27*E(3)-8/27*E(3)^2 ], 
  [ 31, 27, -1/54*E(3)-1/27*E(3)^2 ], [ 31, 29, 1/54*E(3)+1/27*E(3)^2 ], 
  [ 31, 30, -5/108*E(3)-5/54*E(3)^2 ], [ 31, 32, -10/27*E(3)-20/27*E(3)^2 ], 
  [ 31, 34, -2/3*E(3)^2 ], [ 31, 35, -1/12*E(3)^2 ], 
  [ 31, 38, -1/54*E(3)+1/54*E(3)^2 ], [ 31, 40, -4/27*E(3)+4/27*E(3)^2 ], 
  [ 31, 42, 4/27*E(3)-4/27*E(3)^2 ], [ 31, 43, 1/54*E(3)-1/54*E(3)^2 ], 
  [ 31, 44, 5/108*E(3)-5/108*E(3)^2 ], [ 31, 46, -1/12*E(3) ], 
  [ 31, 49, -2/3*E(3) ], [ 31, 51, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 31, 54, -1/27*E(3)-1/54*E(3)^2 ], [ 31, 56, -8/27*E(3)-4/27*E(3)^2 ], 
  [ 31, 58, 1/12 ], [ 31, 65, -1/54*E(3)-1/27*E(3)^2 ], [ 32, 5, -1/9 ], 
  [ 32, 12, 1/9*E(3)+2/9*E(3)^2 ], [ 32, 13, 1/18*E(3)^2 ], 
  [ 32, 18, 1/9*E(3)^2 ], [ 32, 19, 1/9*E(3)^2 ], 
  [ 32, 20, -1/18*E(3)+1/18*E(3)^2 ], [ 32, 24, -2/9*E(3)+2/9*E(3)^2 ], 
  [ 32, 26, -1/18*E(3) ], [ 32, 27, -1/18*E(3) ], [ 32, 29, -1/18*E(3) ], 
  [ 32, 30, -1/9*E(3) ], [ 32, 31, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 32, 32, -1/9*E(3) ], [ 32, 36, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 32, 37, -2/9*E(3)-1/9*E(3)^2 ], [ 32, 38, 1/18 ], [ 32, 40, 1/18 ], 
  [ 32, 42, 1/18 ], [ 32, 43, 1/18 ], [ 32, 44, -1/9 ], 
  [ 32, 45, -1/18*E(3)-1/9*E(3)^2 ], [ 32, 47, -1/9*E(3)-2/9*E(3)^2 ], 
  [ 32, 51, 1/18*E(3)^2 ], [ 32, 54, -1/18*E(3)^2 ], [ 32, 56, -1/18*E(3)^2 ],
  [ 32, 60, 1/18*E(3)-1/18*E(3)^2 ], [ 32, 65, -1/18*E(3) ], 
  [ 33, 5, 5/12*E(3)-5/12*E(3)^2 ], [ 33, 13, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 33, 19, 5/6*E(3)+5/12*E(3)^2 ], [ 33, 21, 3/4 ], 
  [ 33, 27, -1/6*E(3)-1/3*E(3)^2 ], [ 33, 29, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 33, 30, -5/12*E(3)-5/6*E(3)^2 ], [ 33, 35, -3/4*E(3)^2 ], 
  [ 33, 38, -1/6*E(3)+1/6*E(3)^2 ], [ 33, 43, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 33, 44, -5/12*E(3)+5/12*E(3)^2 ], [ 33, 46, -3/4*E(3) ], 
  [ 33, 51, 1/3*E(3)+1/6*E(3)^2 ], [ 33, 54, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 33, 58, -3/4 ], [ 33, 65, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 34, 13, 1/9*E(3)+1/18*E(3)^2 ], [ 34, 20, 1/6 ], 
  [ 34, 26, 1/18*E(3)+1/9*E(3)^2 ], [ 34, 27, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 34, 29, -1/18*E(3)-1/9*E(3)^2 ], [ 34, 31, -1/3*E(3)^2 ], 
  [ 34, 36, 1/6*E(3)^2 ], [ 34, 38, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 34, 40, -1/18*E(3)+1/18*E(3)^2 ], [ 34, 42, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 34, 43, 1/18*E(3)-1/18*E(3)^2 ], [ 34, 45, 1/6*E(3) ], 
  [ 34, 47, -1/3*E(3) ], [ 34, 51, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 34, 54, 1/9*E(3)+1/18*E(3)^2 ], [ 34, 56, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 34, 60, 1/6 ], [ 34, 65, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 35, 13, 1/9*E(3)+1/18*E(3)^2 ], [ 35, 20, -2/3 ], 
  [ 35, 26, 2/9*E(3)+4/9*E(3)^2 ], [ 35, 27, -1/9*E(3)-2/9*E(3)^2 ], 
  [ 35, 29, -1/18*E(3)-1/9*E(3)^2 ], [ 35, 31, -1/6*E(3)^2 ], 
  [ 35, 33, -1/6*E(3)^2 ], [ 35, 36, -2/3*E(3)^2 ], 
  [ 35, 38, 1/9*E(3)-1/9*E(3)^2 ], [ 35, 40, -2/9*E(3)+2/9*E(3)^2 ], 
  [ 35, 42, 2/9*E(3)-2/9*E(3)^2 ], [ 35, 43, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 35, 45, -2/3*E(3) ], [ 35, 47, -1/6*E(3) ], [ 35, 50, -1/6*E(3) ], 
  [ 35, 51, 1/9*E(3)+1/18*E(3)^2 ], [ 35, 54, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 35, 56, 4/9*E(3)+2/9*E(3)^2 ], [ 35, 60, -2/3 ], 
  [ 35, 65, -1/18*E(3)-1/9*E(3)^2 ], [ 36, 5, 5/108*E(3)-5/108*E(3)^2 ], 
  [ 36, 12, 5/12*E(3) ], [ 36, 13, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 36, 18, 5/54*E(3)+5/108*E(3)^2 ], [ 36, 19, -5/54*E(3)-5/108*E(3)^2 ], 
  [ 36, 20, 1/6 ], [ 36, 21, -1/12 ], [ 36, 26, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 36, 27, 1/54*E(3)+1/27*E(3)^2 ], [ 36, 28, 1/4*E(3)+1/2*E(3)^2 ], 
  [ 36, 29, -1/54*E(3)-1/27*E(3)^2 ], [ 36, 30, 5/108*E(3)+5/54*E(3)^2 ], 
  [ 36, 32, -5/108*E(3)-5/54*E(3)^2 ], [ 36, 34, 1/12*E(3)^2 ], 
  [ 36, 35, -1/12*E(3)^2 ], [ 36, 36, -1/6*E(3)^2 ], [ 36, 37, -5/12*E(3)^2 ],
  [ 36, 38, 1/54*E(3)-1/54*E(3)^2 ], [ 36, 40, -1/54*E(3)+1/54*E(3)^2 ], 
  [ 36, 42, 1/54*E(3)-1/54*E(3)^2 ], [ 36, 43, -1/54*E(3)+1/54*E(3)^2 ], 
  [ 36, 44, -5/108*E(3)+5/108*E(3)^2 ], [ 36, 45, -1/6*E(3) ], 
  [ 36, 46, -1/12*E(3) ], [ 36, 49, 1/12*E(3) ], 
  [ 36, 51, 1/27*E(3)+1/54*E(3)^2 ], [ 36, 53, 1/2*E(3)+1/4*E(3)^2 ], 
  [ 36, 54, 1/27*E(3)+1/54*E(3)^2 ], [ 36, 56, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 36, 58, 1/12 ], [ 36, 60, 1/6 ], [ 36, 65, 1/54*E(3)+1/27*E(3)^2 ], 
  [ 37, 5, 1/27*E(3)-1/27*E(3)^2 ], [ 37, 12, -1/3*E(3) ], 
  [ 37, 13, 1/27*E(3)+1/54*E(3)^2 ], [ 37, 18, 2/27*E(3)+1/27*E(3)^2 ], 
  [ 37, 19, -2/27*E(3)-1/27*E(3)^2 ], [ 37, 20, 1/6 ], 
  [ 37, 26, 1/54*E(3)+1/27*E(3)^2 ], [ 37, 27, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 37, 29, 1/54*E(3)+1/27*E(3)^2 ], [ 37, 30, 1/27*E(3)+2/27*E(3)^2 ], 
  [ 37, 32, -1/27*E(3)-2/27*E(3)^2 ], [ 37, 36, -1/6*E(3)^2 ], 
  [ 37, 37, 1/3*E(3)^2 ], [ 37, 38, -1/54*E(3)+1/54*E(3)^2 ], 
  [ 37, 40, 1/54*E(3)-1/54*E(3)^2 ], [ 37, 42, -1/54*E(3)+1/54*E(3)^2 ], 
  [ 37, 43, 1/54*E(3)-1/54*E(3)^2 ], [ 37, 44, -1/27*E(3)+1/27*E(3)^2 ], 
  [ 37, 45, -1/6*E(3) ], [ 37, 51, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 37, 54, -1/27*E(3)-1/54*E(3)^2 ], [ 37, 56, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 37, 60, 1/6 ], [ 37, 65, -1/54*E(3)-1/27*E(3)^2 ], [ 38, 5, 5/36 ], 
  [ 38, 12, 5/9*E(3)+10/9*E(3)^2 ], [ 38, 13, 1/18*E(3)^2 ], 
  [ 38, 18, -5/9*E(3)^2 ], [ 38, 19, 5/18*E(3)^2 ], 
  [ 38, 20, 2/9*E(3)-2/9*E(3)^2 ], [ 38, 21, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 38, 24, 5/36*E(3)-5/36*E(3)^2 ], [ 38, 25, 5/36*E(3)-5/36*E(3)^2 ], 
  [ 38, 26, -2/9*E(3) ], [ 38, 27, 1/9*E(3) ], [ 38, 28, E(3) ], 
  [ 38, 29, -1/18*E(3) ], [ 38, 30, -5/18*E(3) ], 
  [ 38, 31, -1/9*E(3)-1/18*E(3)^2 ], [ 38, 32, 5/9*E(3) ], 
  [ 38, 33, -1/9*E(3)-1/18*E(3)^2 ], [ 38, 34, -2/3*E(3)-1/3*E(3)^2 ], 
  [ 38, 35, 1/3*E(3)+1/6*E(3)^2 ], [ 38, 36, 4/9*E(3)+2/9*E(3)^2 ], 
  [ 38, 37, -10/9*E(3)-5/9*E(3)^2 ], [ 38, 38, -1/9 ], [ 38, 39, 1/4 ], 
  [ 38, 40, 2/9 ], [ 38, 41, 1/4 ], [ 38, 42, 2/9 ], [ 38, 43, -1/9 ], 
  [ 38, 44, 5/36 ], [ 38, 45, 2/9*E(3)+4/9*E(3)^2 ], 
  [ 38, 46, 1/6*E(3)+1/3*E(3)^2 ], [ 38, 47, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 38, 49, -1/3*E(3)-2/3*E(3)^2 ], [ 38, 50, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 38, 51, 1/18*E(3)^2 ], [ 38, 53, E(3)^2 ], [ 38, 54, 1/9*E(3)^2 ], 
  [ 38, 56, -2/9*E(3)^2 ], [ 38, 58, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 38, 60, -2/9*E(3)+2/9*E(3)^2 ], [ 38, 65, -1/18*E(3) ], 
  [ 39, 13, 1/18*E(3)^2 ], [ 39, 26, 4/9*E(3) ], [ 39, 27, 1/18*E(3) ], 
  [ 39, 29, 1/18*E(3) ], [ 39, 38, 1/18 ], [ 39, 40, 4/9 ], [ 39, 42, 4/9 ], 
  [ 39, 43, 1/18 ], [ 39, 51, -1/18*E(3)^2 ], [ 39, 54, 1/18*E(3)^2 ], 
  [ 39, 56, 4/9*E(3)^2 ], [ 39, 65, -1/18*E(3) ], [ 40, 5, 5/36 ], 
  [ 40, 12, -5/36*E(3)-5/18*E(3)^2 ], [ 40, 13, 1/18*E(3)^2 ], 
  [ 40, 18, -5/36*E(3)^2 ], [ 40, 19, -5/36*E(3)^2 ], 
  [ 40, 20, -1/18*E(3)+1/18*E(3)^2 ], [ 40, 21, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 40, 24, 5/18*E(3)-5/18*E(3)^2 ], [ 40, 26, -1/18*E(3) ], 
  [ 40, 27, -1/18*E(3) ], [ 40, 28, -1/4*E(3) ], [ 40, 29, -1/18*E(3) ], 
  [ 40, 30, 5/36*E(3) ], [ 40, 31, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 40, 32, 5/36*E(3) ], [ 40, 34, -1/6*E(3)-1/12*E(3)^2 ], 
  [ 40, 35, -1/6*E(3)-1/12*E(3)^2 ], [ 40, 36, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 40, 37, 5/18*E(3)+5/36*E(3)^2 ], [ 40, 38, 1/18 ], [ 40, 39, 1/2 ], 
  [ 40, 40, 1/18 ], [ 40, 42, 1/18 ], [ 40, 43, 1/18 ], [ 40, 44, 5/36 ], 
  [ 40, 45, -1/18*E(3)-1/9*E(3)^2 ], [ 40, 46, -1/12*E(3)-1/6*E(3)^2 ], 
  [ 40, 47, -1/9*E(3)-2/9*E(3)^2 ], [ 40, 49, -1/12*E(3)-1/6*E(3)^2 ], 
  [ 40, 51, 1/18*E(3)^2 ], [ 40, 53, -1/4*E(3)^2 ], [ 40, 54, -1/18*E(3)^2 ], 
  [ 40, 56, -1/18*E(3)^2 ], [ 40, 58, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 40, 60, 1/18*E(3)-1/18*E(3)^2 ], [ 40, 65, -1/18*E(3) ], 
  [ 41, 13, -1/2*E(3)^2 ], [ 41, 27, 1/2*E(3) ], [ 41, 29, -1/2*E(3) ], 
  [ 41, 38, 1/2 ], [ 41, 43, 1/2 ], [ 41, 51, 1/2*E(3)^2 ], 
  [ 41, 54, 1/2*E(3)^2 ], [ 41, 65, 1/2*E(3) ], [ 42, 5, 5/36 ], 
  [ 42, 12, 5/36*E(3)+5/18*E(3)^2 ], [ 42, 13, 1/18*E(3)^2 ], 
  [ 42, 18, -5/36*E(3)^2 ], [ 42, 19, -5/36*E(3)^2 ], 
  [ 42, 20, 1/18*E(3)-1/18*E(3)^2 ], [ 42, 21, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 42, 24, -5/18*E(3)+5/18*E(3)^2 ], [ 42, 26, -1/18*E(3) ], 
  [ 42, 27, -1/18*E(3) ], [ 42, 28, -1/4*E(3) ], [ 42, 29, -1/18*E(3) ], 
  [ 42, 30, 5/36*E(3) ], [ 42, 31, 2/9*E(3)+1/9*E(3)^2 ], 
  [ 42, 32, 5/36*E(3) ], [ 42, 34, 1/6*E(3)+1/12*E(3)^2 ], 
  [ 42, 35, 1/6*E(3)+1/12*E(3)^2 ], [ 42, 36, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 42, 37, -5/18*E(3)-5/36*E(3)^2 ], [ 42, 38, 1/18 ], [ 42, 39, 1/2 ], 
  [ 42, 40, 1/18 ], [ 42, 42, 1/18 ], [ 42, 43, 1/18 ], [ 42, 44, 5/36 ], 
  [ 42, 45, 1/18*E(3)+1/9*E(3)^2 ], [ 42, 46, 1/12*E(3)+1/6*E(3)^2 ], 
  [ 42, 47, 1/9*E(3)+2/9*E(3)^2 ], [ 42, 49, 1/12*E(3)+1/6*E(3)^2 ], 
  [ 42, 51, 1/18*E(3)^2 ], [ 42, 53, -1/4*E(3)^2 ], [ 42, 54, -1/18*E(3)^2 ], 
  [ 42, 56, -1/18*E(3)^2 ], [ 42, 58, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 42, 60, -1/18*E(3)+1/18*E(3)^2 ], [ 42, 65, -1/18*E(3) ], [ 43, 5, 5/36 ],
  [ 43, 12, -5/9*E(3)-10/9*E(3)^2 ], [ 43, 13, 1/18*E(3)^2 ], 
  [ 43, 18, -5/9*E(3)^2 ], [ 43, 19, 5/18*E(3)^2 ], 
  [ 43, 20, -2/9*E(3)+2/9*E(3)^2 ], [ 43, 21, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 43, 24, -5/36*E(3)+5/36*E(3)^2 ], [ 43, 25, -5/36*E(3)+5/36*E(3)^2 ], 
  [ 43, 26, -2/9*E(3) ], [ 43, 27, 1/9*E(3) ], [ 43, 28, E(3) ], 
  [ 43, 29, -1/18*E(3) ], [ 43, 30, -5/18*E(3) ], 
  [ 43, 31, 1/9*E(3)+1/18*E(3)^2 ], [ 43, 32, 5/9*E(3) ], 
  [ 43, 33, 1/9*E(3)+1/18*E(3)^2 ], [ 43, 34, 2/3*E(3)+1/3*E(3)^2 ], 
  [ 43, 35, -1/3*E(3)-1/6*E(3)^2 ], [ 43, 36, -4/9*E(3)-2/9*E(3)^2 ], 
  [ 43, 37, 10/9*E(3)+5/9*E(3)^2 ], [ 43, 38, -1/9 ], [ 43, 39, 1/4 ], 
  [ 43, 40, 2/9 ], [ 43, 41, 1/4 ], [ 43, 42, 2/9 ], [ 43, 43, -1/9 ], 
  [ 43, 44, 5/36 ], [ 43, 45, -2/9*E(3)-4/9*E(3)^2 ], 
  [ 43, 46, -1/6*E(3)-1/3*E(3)^2 ], [ 43, 47, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 43, 49, 1/3*E(3)+2/3*E(3)^2 ], [ 43, 50, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 43, 51, 1/18*E(3)^2 ], [ 43, 53, E(3)^2 ], [ 43, 54, 1/9*E(3)^2 ], 
  [ 43, 56, -2/9*E(3)^2 ], [ 43, 58, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 43, 60, 2/9*E(3)-2/9*E(3)^2 ], [ 43, 65, -1/18*E(3) ], [ 44, 5, 2/9 ], 
  [ 44, 12, 4/9*E(3)+8/9*E(3)^2 ], [ 44, 13, -1/9*E(3)^2 ], 
  [ 44, 18, 4/9*E(3)^2 ], [ 44, 19, 1/9*E(3)^2 ], 
  [ 44, 20, -2/9*E(3)+2/9*E(3)^2 ], [ 44, 24, 1/9*E(3)-1/9*E(3)^2 ], 
  [ 44, 25, -1/9*E(3)+1/9*E(3)^2 ], [ 44, 26, -2/9*E(3) ], 
  [ 44, 27, -1/18*E(3) ], [ 44, 29, 1/9*E(3) ], [ 44, 30, -1/9*E(3) ], 
  [ 44, 31, 1/9*E(3)+1/18*E(3)^2 ], [ 44, 32, -4/9*E(3) ], 
  [ 44, 33, -1/9*E(3)-1/18*E(3)^2 ], [ 44, 36, -4/9*E(3)-2/9*E(3)^2 ], 
  [ 44, 37, -8/9*E(3)-4/9*E(3)^2 ], [ 44, 38, 1/18 ], [ 44, 40, 2/9 ], 
  [ 44, 42, 2/9 ], [ 44, 43, 1/18 ], [ 44, 44, 2/9 ], 
  [ 44, 45, -2/9*E(3)-4/9*E(3)^2 ], [ 44, 47, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 44, 50, -1/18*E(3)-1/9*E(3)^2 ], [ 44, 51, -1/9*E(3)^2 ], 
  [ 44, 54, -1/18*E(3)^2 ], [ 44, 56, -2/9*E(3)^2 ], 
  [ 44, 60, 2/9*E(3)-2/9*E(3)^2 ], [ 44, 65, 1/9*E(3) ], 
  [ 45, 5, -5/108*E(3)+5/108*E(3)^2 ], [ 45, 12, 5/12*E(3) ], 
  [ 45, 13, 1/27*E(3)+1/54*E(3)^2 ], [ 45, 18, -5/54*E(3)-5/108*E(3)^2 ], 
  [ 45, 19, 5/54*E(3)+5/108*E(3)^2 ], [ 45, 20, 1/6 ], [ 45, 21, -1/12 ], 
  [ 45, 26, 1/54*E(3)+1/27*E(3)^2 ], [ 45, 27, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 45, 28, -1/4*E(3)-1/2*E(3)^2 ], [ 45, 29, 1/54*E(3)+1/27*E(3)^2 ], 
  [ 45, 30, -5/108*E(3)-5/54*E(3)^2 ], [ 45, 32, 5/108*E(3)+5/54*E(3)^2 ], 
  [ 45, 34, 1/12*E(3)^2 ], [ 45, 35, -1/12*E(3)^2 ], [ 45, 36, -1/6*E(3)^2 ], 
  [ 45, 37, -5/12*E(3)^2 ], [ 45, 38, -1/54*E(3)+1/54*E(3)^2 ], 
  [ 45, 40, 1/54*E(3)-1/54*E(3)^2 ], [ 45, 42, -1/54*E(3)+1/54*E(3)^2 ], 
  [ 45, 43, 1/54*E(3)-1/54*E(3)^2 ], [ 45, 44, 5/108*E(3)-5/108*E(3)^2 ], 
  [ 45, 45, -1/6*E(3) ], [ 45, 46, -1/12*E(3) ], [ 45, 49, 1/12*E(3) ], 
  [ 45, 51, -1/27*E(3)-1/54*E(3)^2 ], [ 45, 53, -1/2*E(3)-1/4*E(3)^2 ], 
  [ 45, 54, -1/27*E(3)-1/54*E(3)^2 ], [ 45, 56, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 45, 58, 1/12 ], [ 45, 60, 1/6 ], [ 45, 65, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 46, 13, -1/9*E(3)-1/18*E(3)^2 ], [ 46, 20, -2/3 ], 
  [ 46, 26, -2/9*E(3)-4/9*E(3)^2 ], [ 46, 27, 1/9*E(3)+2/9*E(3)^2 ], 
  [ 46, 29, 1/18*E(3)+1/9*E(3)^2 ], [ 46, 31, -1/6*E(3)^2 ], 
  [ 46, 33, -1/6*E(3)^2 ], [ 46, 36, -2/3*E(3)^2 ], 
  [ 46, 38, -1/9*E(3)+1/9*E(3)^2 ], [ 46, 40, 2/9*E(3)-2/9*E(3)^2 ], 
  [ 46, 42, -2/9*E(3)+2/9*E(3)^2 ], [ 46, 43, 1/9*E(3)-1/9*E(3)^2 ], 
  [ 46, 45, -2/3*E(3) ], [ 46, 47, -1/6*E(3) ], [ 46, 50, -1/6*E(3) ], 
  [ 46, 51, -1/9*E(3)-1/18*E(3)^2 ], [ 46, 54, 2/9*E(3)+1/9*E(3)^2 ], 
  [ 46, 56, -4/9*E(3)-2/9*E(3)^2 ], [ 46, 60, -2/3 ], 
  [ 46, 65, 1/18*E(3)+1/9*E(3)^2 ], [ 47, 5, 5/108*E(3)-5/108*E(3)^2 ], 
  [ 47, 13, -1/27*E(3)-1/54*E(3)^2 ], [ 47, 18, -20/27*E(3)-10/27*E(3)^2 ], 
  [ 47, 19, -5/54*E(3)-5/108*E(3)^2 ], [ 47, 21, -1/12 ], 
  [ 47, 26, 4/27*E(3)+8/27*E(3)^2 ], [ 47, 27, 1/54*E(3)+1/27*E(3)^2 ], 
  [ 47, 29, -1/54*E(3)-1/27*E(3)^2 ], [ 47, 30, 5/108*E(3)+5/54*E(3)^2 ], 
  [ 47, 32, 10/27*E(3)+20/27*E(3)^2 ], [ 47, 34, -2/3*E(3)^2 ], 
  [ 47, 35, -1/12*E(3)^2 ], [ 47, 38, 1/54*E(3)-1/54*E(3)^2 ], 
  [ 47, 40, 4/27*E(3)-4/27*E(3)^2 ], [ 47, 42, -4/27*E(3)+4/27*E(3)^2 ], 
  [ 47, 43, -1/54*E(3)+1/54*E(3)^2 ], [ 47, 44, -5/108*E(3)+5/108*E(3)^2 ], 
  [ 47, 46, -1/12*E(3) ], [ 47, 49, -2/3*E(3) ], 
  [ 47, 51, 1/27*E(3)+1/54*E(3)^2 ], [ 47, 54, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 47, 56, 8/27*E(3)+4/27*E(3)^2 ], [ 47, 58, 1/12 ], 
  [ 47, 65, 1/54*E(3)+1/27*E(3)^2 ], [ 48, 48, 1/12 ], 
  [ 48, 52, -1/4*E(3)-1/2*E(3)^2 ], [ 48, 55, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 48, 57, 1/6*E(3)^2 ], [ 48, 59, -1/2*E(3)^2 ], 
  [ 48, 61, 1/36*E(3)-1/36*E(3)^2 ], [ 48, 62, -1/12*E(3)^2 ], 
  [ 48, 63, 1/4*E(3)-1/4*E(3)^2 ], [ 48, 64, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 48, 66, 1/4*E(3) ], [ 48, 67, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 48, 69, -1/12*E(3) ], [ 48, 70, 1/2*E(3) ], 
  [ 48, 71, -1/9*E(3)-1/18*E(3)^2 ], [ 48, 72, -1/6*E(3) ], 
  [ 48, 74, -1/18*E(3)-1/36*E(3)^2 ], [ 48, 75, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 48, 76, -1/4 ], [ 48, 77, 1/2*E(3)+1/4*E(3)^2 ], [ 48, 79, 1/4 ], 
  [ 48, 80, -1/36*E(3)-1/18*E(3)^2 ], [ 48, 81, 1/12 ], 
  [ 48, 84, 1/18*E(3)+1/9*E(3)^2 ], [ 48, 88, -1/4*E(3)^2 ], 
  [ 48, 91, -1/36*E(3)+1/36*E(3)^2 ], [ 49, 13, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 49, 20, 1/6 ], [ 49, 26, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 49, 27, -1/18*E(3)-1/9*E(3)^2 ], [ 49, 29, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 49, 31, -1/3*E(3)^2 ], [ 49, 36, 1/6*E(3)^2 ], 
  [ 49, 38, 1/18*E(3)-1/18*E(3)^2 ], [ 49, 40, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 49, 42, -1/18*E(3)+1/18*E(3)^2 ], [ 49, 43, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 49, 45, 1/6*E(3) ], [ 49, 47, -1/3*E(3) ], 
  [ 49, 51, -1/9*E(3)-1/18*E(3)^2 ], [ 49, 54, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 49, 56, -1/9*E(3)-1/18*E(3)^2 ], [ 49, 60, 1/6 ], 
  [ 49, 65, 1/18*E(3)+1/9*E(3)^2 ], [ 50, 5, -5/12*E(3)+5/12*E(3)^2 ], 
  [ 50, 13, 1/3*E(3)+1/6*E(3)^2 ], [ 50, 19, -5/6*E(3)-5/12*E(3)^2 ], 
  [ 50, 21, 3/4 ], [ 50, 27, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 50, 29, 1/6*E(3)+1/3*E(3)^2 ], [ 50, 30, 5/12*E(3)+5/6*E(3)^2 ], 
  [ 50, 35, -3/4*E(3)^2 ], [ 50, 38, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 50, 43, -1/6*E(3)+1/6*E(3)^2 ], [ 50, 44, 5/12*E(3)-5/12*E(3)^2 ], 
  [ 50, 46, -3/4*E(3) ], [ 50, 51, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 50, 54, 1/3*E(3)+1/6*E(3)^2 ], [ 50, 58, -3/4 ], 
  [ 50, 65, -1/6*E(3)-1/3*E(3)^2 ], [ 51, 5, -5/18 ], 
  [ 51, 12, -5/9*E(3)-10/9*E(3)^2 ], [ 51, 13, -1/9*E(3)^2 ], 
  [ 51, 18, -5/9*E(3)^2 ], [ 51, 19, -5/36*E(3)^2 ], 
  [ 51, 20, -2/9*E(3)+2/9*E(3)^2 ], [ 51, 21, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 51, 24, -5/36*E(3)+5/36*E(3)^2 ], [ 51, 25, 5/36*E(3)-5/36*E(3)^2 ], 
  [ 51, 26, -2/9*E(3) ], [ 51, 27, -1/18*E(3) ], [ 51, 28, -E(3) ], 
  [ 51, 29, 1/9*E(3) ], [ 51, 30, 5/36*E(3) ], 
  [ 51, 31, 1/9*E(3)+1/18*E(3)^2 ], [ 51, 32, 5/9*E(3) ], 
  [ 51, 33, -1/9*E(3)-1/18*E(3)^2 ], [ 51, 34, -2/3*E(3)-1/3*E(3)^2 ], 
  [ 51, 35, -1/6*E(3)-1/12*E(3)^2 ], [ 51, 36, -4/9*E(3)-2/9*E(3)^2 ], 
  [ 51, 37, 10/9*E(3)+5/9*E(3)^2 ], [ 51, 38, 1/18 ], [ 51, 39, -1/4 ], 
  [ 51, 40, 2/9 ], [ 51, 41, 1/4 ], [ 51, 42, 2/9 ], [ 51, 43, 1/18 ], 
  [ 51, 44, -5/18 ], [ 51, 45, -2/9*E(3)-4/9*E(3)^2 ], 
  [ 51, 46, -1/12*E(3)-1/6*E(3)^2 ], [ 51, 47, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 51, 49, -1/3*E(3)-2/3*E(3)^2 ], [ 51, 50, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 51, 51, -1/9*E(3)^2 ], [ 51, 53, -E(3)^2 ], [ 51, 54, -1/18*E(3)^2 ], 
  [ 51, 56, -2/9*E(3)^2 ], [ 51, 58, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 51, 60, 2/9*E(3)-2/9*E(3)^2 ], [ 51, 65, 1/9*E(3) ], 
  [ 52, 48, 1/12*E(3)-1/12*E(3)^2 ], [ 52, 52, -1/4*E(3) ], 
  [ 52, 55, 1/6*E(3) ], [ 52, 59, -1/3*E(3)-1/6*E(3)^2 ], [ 52, 61, -1/12 ], 
  [ 52, 62, 1/6*E(3)+1/12*E(3)^2 ], [ 52, 63, -1/4 ], 
  [ 52, 66, -1/12*E(3)-1/6*E(3)^2 ], [ 52, 67, 1/4 ], 
  [ 52, 69, 1/12*E(3)+1/6*E(3)^2 ], [ 52, 70, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 52, 74, -1/12*E(3)^2 ], [ 52, 75, 1/6*E(3)^2 ], 
  [ 52, 76, -1/12*E(3)+1/12*E(3)^2 ], [ 52, 77, 1/4*E(3)^2 ], 
  [ 52, 79, -1/12*E(3)+1/12*E(3)^2 ], [ 52, 80, -1/12*E(3) ], 
  [ 52, 81, -1/12*E(3)+1/12*E(3)^2 ], [ 52, 88, 1/6*E(3)+1/12*E(3)^2 ], 
  [ 52, 91, -1/12 ], [ 53, 13, 1/18*E(3)^2 ], [ 53, 20, 1/6*E(3)-1/6*E(3)^2 ],
  [ 53, 26, -1/18*E(3) ], [ 53, 27, 1/18*E(3) ], [ 53, 29, 1/18*E(3) ], 
  [ 53, 36, -1/3*E(3)-1/6*E(3)^2 ], [ 53, 38, 1/18 ], [ 53, 40, -1/18 ], 
  [ 53, 42, -1/18 ], [ 53, 43, 1/18 ], [ 53, 45, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 53, 51, -1/18*E(3)^2 ], [ 53, 54, 1/18*E(3)^2 ], [ 53, 56, -1/18*E(3)^2 ],
  [ 53, 60, -1/6*E(3)+1/6*E(3)^2 ], [ 53, 65, -1/18*E(3) ], [ 54, 5, -5/36 ], 
  [ 54, 12, -5/9*E(3)-10/9*E(3)^2 ], [ 54, 13, -1/18*E(3)^2 ], 
  [ 54, 18, 5/9*E(3)^2 ], [ 54, 19, -5/18*E(3)^2 ], 
  [ 54, 20, -2/9*E(3)+2/9*E(3)^2 ], [ 54, 21, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 54, 24, -5/36*E(3)+5/36*E(3)^2 ], [ 54, 25, -5/36*E(3)+5/36*E(3)^2 ], 
  [ 54, 26, 2/9*E(3) ], [ 54, 27, -1/9*E(3) ], [ 54, 28, E(3) ], 
  [ 54, 29, 1/18*E(3) ], [ 54, 30, 5/18*E(3) ], 
  [ 54, 31, 1/9*E(3)+1/18*E(3)^2 ], [ 54, 32, -5/9*E(3) ], 
  [ 54, 33, 1/9*E(3)+1/18*E(3)^2 ], [ 54, 34, -2/3*E(3)-1/3*E(3)^2 ], 
  [ 54, 35, 1/3*E(3)+1/6*E(3)^2 ], [ 54, 36, -4/9*E(3)-2/9*E(3)^2 ], 
  [ 54, 37, 10/9*E(3)+5/9*E(3)^2 ], [ 54, 38, 1/9 ], [ 54, 39, 1/4 ], 
  [ 54, 40, -2/9 ], [ 54, 41, 1/4 ], [ 54, 42, -2/9 ], [ 54, 43, 1/9 ], 
  [ 54, 44, -5/36 ], [ 54, 45, -2/9*E(3)-4/9*E(3)^2 ], 
  [ 54, 46, 1/6*E(3)+1/3*E(3)^2 ], [ 54, 47, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 54, 49, -1/3*E(3)-2/3*E(3)^2 ], [ 54, 50, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 54, 51, -1/18*E(3)^2 ], [ 54, 53, E(3)^2 ], [ 54, 54, -1/9*E(3)^2 ], 
  [ 54, 56, 2/9*E(3)^2 ], [ 54, 58, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 54, 60, 2/9*E(3)-2/9*E(3)^2 ], [ 54, 65, 1/18*E(3) ], 
  [ 55, 48, 1/27*E(3)-1/27*E(3)^2 ], [ 55, 52, 1/3*E(3) ], 
  [ 55, 57, -4/27*E(3)-2/27*E(3)^2 ], [ 55, 62, -2/27*E(3)-1/27*E(3)^2 ], 
  [ 55, 63, -1/3 ], [ 55, 67, 1/3 ], [ 55, 68, -1/27*E(3)-2/27*E(3)^2 ], 
  [ 55, 69, -1/27*E(3)-2/27*E(3)^2 ], [ 55, 72, 2/27*E(3)+4/27*E(3)^2 ], 
  [ 55, 73, -1/3*E(3)^2 ], [ 55, 77, -1/3*E(3)^2 ], 
  [ 55, 78, -2/27*E(3)+2/27*E(3)^2 ], [ 55, 81, -1/27*E(3)+1/27*E(3)^2 ], 
  [ 55, 83, 1/3*E(3) ], [ 55, 86, -2/27*E(3)-1/27*E(3)^2 ], [ 56, 5, -5/36 ], 
  [ 56, 12, 5/36*E(3)+5/18*E(3)^2 ], [ 56, 13, -1/18*E(3)^2 ], 
  [ 56, 18, 5/36*E(3)^2 ], [ 56, 19, 5/36*E(3)^2 ], 
  [ 56, 20, 1/18*E(3)-1/18*E(3)^2 ], [ 56, 21, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 56, 24, -5/18*E(3)+5/18*E(3)^2 ], [ 56, 26, 1/18*E(3) ], 
  [ 56, 27, 1/18*E(3) ], [ 56, 28, -1/4*E(3) ], [ 56, 29, 1/18*E(3) ], 
  [ 56, 30, -5/36*E(3) ], [ 56, 31, 2/9*E(3)+1/9*E(3)^2 ], 
  [ 56, 32, -5/36*E(3) ], [ 56, 34, -1/6*E(3)-1/12*E(3)^2 ], 
  [ 56, 35, -1/6*E(3)-1/12*E(3)^2 ], [ 56, 36, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 56, 37, -5/18*E(3)-5/36*E(3)^2 ], [ 56, 38, -1/18 ], [ 56, 39, 1/2 ], 
  [ 56, 40, -1/18 ], [ 56, 42, -1/18 ], [ 56, 43, -1/18 ], [ 56, 44, -5/36 ], 
  [ 56, 45, 1/18*E(3)+1/9*E(3)^2 ], [ 56, 46, -1/12*E(3)-1/6*E(3)^2 ], 
  [ 56, 47, 1/9*E(3)+2/9*E(3)^2 ], [ 56, 49, -1/12*E(3)-1/6*E(3)^2 ], 
  [ 56, 51, -1/18*E(3)^2 ], [ 56, 53, -1/4*E(3)^2 ], [ 56, 54, 1/18*E(3)^2 ], 
  [ 56, 56, 1/18*E(3)^2 ], [ 56, 58, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 56, 60, -1/18*E(3)+1/18*E(3)^2 ], [ 56, 65, 1/18*E(3) ], [ 57, 48, 2/3 ], 
  [ 57, 55, -4/9*E(3)-8/9*E(3)^2 ], [ 57, 57, -1/6*E(3)^2 ], 
  [ 57, 61, 2/9*E(3)-2/9*E(3)^2 ], [ 57, 62, -2/3*E(3)^2 ], 
  [ 57, 64, 1/9*E(3)-1/9*E(3)^2 ], [ 57, 69, -2/3*E(3) ], 
  [ 57, 71, 1/9*E(3)+1/18*E(3)^2 ], [ 57, 72, 1/6*E(3) ], 
  [ 57, 74, -4/9*E(3)-2/9*E(3)^2 ], [ 57, 75, -8/9*E(3)-4/9*E(3)^2 ], 
  [ 57, 80, -2/9*E(3)-4/9*E(3)^2 ], [ 57, 81, 2/3 ], 
  [ 57, 84, -1/18*E(3)-1/9*E(3)^2 ], [ 57, 91, -2/9*E(3)+2/9*E(3)^2 ], 
  [ 58, 13, 2/9*E(3)+1/9*E(3)^2 ], [ 58, 20, 2/3 ], 
  [ 58, 26, -2/9*E(3)-4/9*E(3)^2 ], [ 58, 27, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 58, 29, -1/9*E(3)-2/9*E(3)^2 ], [ 58, 31, 1/6*E(3)^2 ], 
  [ 58, 33, -1/6*E(3)^2 ], [ 58, 36, 2/3*E(3)^2 ], 
  [ 58, 38, 1/18*E(3)-1/18*E(3)^2 ], [ 58, 40, 2/9*E(3)-2/9*E(3)^2 ], 
  [ 58, 42, -2/9*E(3)+2/9*E(3)^2 ], [ 58, 43, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 58, 45, 2/3*E(3) ], [ 58, 47, 1/6*E(3) ], [ 58, 50, -1/6*E(3) ], 
  [ 58, 51, 2/9*E(3)+1/9*E(3)^2 ], [ 58, 54, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 58, 56, -4/9*E(3)-2/9*E(3)^2 ], [ 58, 60, 2/3 ], 
  [ 58, 65, -1/9*E(3)-2/9*E(3)^2 ], [ 59, 48, -1/9 ], 
  [ 59, 52, -1/9*E(3)-2/9*E(3)^2 ], [ 59, 62, -1/9*E(3)^2 ], 
  [ 59, 63, -1/9*E(3)+1/9*E(3)^2 ], [ 59, 67, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 59, 68, 1/9*E(3) ], [ 59, 69, -1/9*E(3) ], 
  [ 59, 73, -2/9*E(3)-1/9*E(3)^2 ], [ 59, 77, 2/9*E(3)+1/9*E(3)^2 ], 
  [ 59, 81, -1/9 ], [ 59, 83, 1/9*E(3)+2/9*E(3)^2 ], [ 59, 86, 1/9*E(3)^2 ], 
  [ 60, 5, 5/108*E(3)-5/108*E(3)^2 ], [ 60, 12, -5/12*E(3) ], 
  [ 60, 13, -1/27*E(3)-1/54*E(3)^2 ], [ 60, 18, 5/54*E(3)+5/108*E(3)^2 ], 
  [ 60, 19, -5/54*E(3)-5/108*E(3)^2 ], [ 60, 20, -1/6 ], [ 60, 21, -1/12 ], 
  [ 60, 26, -1/54*E(3)-1/27*E(3)^2 ], [ 60, 27, 1/54*E(3)+1/27*E(3)^2 ], 
  [ 60, 28, -1/4*E(3)-1/2*E(3)^2 ], [ 60, 29, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 60, 30, 5/108*E(3)+5/54*E(3)^2 ], [ 60, 32, -5/108*E(3)-5/54*E(3)^2 ], 
  [ 60, 34, 1/12*E(3)^2 ], [ 60, 35, -1/12*E(3)^2 ], [ 60, 36, 1/6*E(3)^2 ], 
  [ 60, 37, 5/12*E(3)^2 ], [ 60, 38, 1/54*E(3)-1/54*E(3)^2 ], 
  [ 60, 40, -1/54*E(3)+1/54*E(3)^2 ], [ 60, 42, 1/54*E(3)-1/54*E(3)^2 ], 
  [ 60, 43, -1/54*E(3)+1/54*E(3)^2 ], [ 60, 44, -5/108*E(3)+5/108*E(3)^2 ], 
  [ 60, 45, 1/6*E(3) ], [ 60, 46, -1/12*E(3) ], [ 60, 49, 1/12*E(3) ], 
  [ 60, 51, 1/27*E(3)+1/54*E(3)^2 ], [ 60, 53, -1/2*E(3)-1/4*E(3)^2 ], 
  [ 60, 54, 1/27*E(3)+1/54*E(3)^2 ], [ 60, 56, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 60, 58, 1/12 ], [ 60, 60, -1/6 ], [ 60, 65, 1/54*E(3)+1/27*E(3)^2 ], 
  [ 61, 48, -5/108*E(3)+5/108*E(3)^2 ], [ 61, 52, -5/12*E(3) ], 
  [ 61, 57, 5/27*E(3)+5/54*E(3)^2 ], [ 61, 61, 1/12 ], 
  [ 61, 62, 5/54*E(3)+5/108*E(3)^2 ], [ 61, 63, 5/12 ], 
  [ 61, 66, -1/4*E(3)-1/2*E(3)^2 ], [ 61, 67, -5/12 ], 
  [ 61, 68, -1/27*E(3)-2/27*E(3)^2 ], [ 61, 69, 5/108*E(3)+5/54*E(3)^2 ], 
  [ 61, 71, 1/6*E(3)^2 ], [ 61, 72, -5/54*E(3)-5/27*E(3)^2 ], 
  [ 61, 73, -1/3*E(3)^2 ], [ 61, 74, -1/12*E(3)^2 ], 
  [ 61, 76, 1/4*E(3)-1/4*E(3)^2 ], [ 61, 77, 5/12*E(3)^2 ], 
  [ 61, 78, -2/27*E(3)+2/27*E(3)^2 ], [ 61, 79, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 61, 80, -1/12*E(3) ], [ 61, 81, 5/108*E(3)-5/108*E(3)^2 ], 
  [ 61, 83, 1/3*E(3) ], [ 61, 84, -1/6*E(3) ], 
  [ 61, 86, -2/27*E(3)-1/27*E(3)^2 ], [ 61, 88, 1/2*E(3)+1/4*E(3)^2 ], 
  [ 61, 91, 1/12 ], [ 62, 48, -1/12 ], [ 62, 52, 1/4*E(3)+1/2*E(3)^2 ], 
  [ 62, 55, -1/18*E(3)-1/9*E(3)^2 ], [ 62, 57, -1/6*E(3)^2 ], 
  [ 62, 59, -1/2*E(3)^2 ], [ 62, 61, 1/36*E(3)-1/36*E(3)^2 ], 
  [ 62, 62, 1/12*E(3)^2 ], [ 62, 63, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 62, 64, -1/9*E(3)+1/9*E(3)^2 ], [ 62, 66, 1/4*E(3) ], 
  [ 62, 67, -1/4*E(3)+1/4*E(3)^2 ], [ 62, 69, 1/12*E(3) ], 
  [ 62, 70, 1/2*E(3) ], [ 62, 71, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 62, 72, 1/6*E(3) ], [ 62, 74, -1/18*E(3)-1/36*E(3)^2 ], 
  [ 62, 75, -1/9*E(3)-1/18*E(3)^2 ], [ 62, 76, -1/4 ], 
  [ 62, 77, -1/2*E(3)-1/4*E(3)^2 ], [ 62, 79, 1/4 ], 
  [ 62, 80, -1/36*E(3)-1/18*E(3)^2 ], [ 62, 81, -1/12 ], 
  [ 62, 84, 1/18*E(3)+1/9*E(3)^2 ], [ 62, 88, -1/4*E(3)^2 ], 
  [ 62, 91, -1/36*E(3)+1/36*E(3)^2 ], [ 63, 48, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 63, 52, -1/4*E(3) ], [ 63, 55, -1/6*E(3) ], 
  [ 63, 59, -1/3*E(3)-1/6*E(3)^2 ], [ 63, 61, 1/12 ], 
  [ 63, 62, -1/6*E(3)-1/12*E(3)^2 ], [ 63, 63, -1/4 ], 
  [ 63, 66, -1/12*E(3)-1/6*E(3)^2 ], [ 63, 67, 1/4 ], 
  [ 63, 69, -1/12*E(3)-1/6*E(3)^2 ], [ 63, 70, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 63, 74, 1/12*E(3)^2 ], [ 63, 75, -1/6*E(3)^2 ], 
  [ 63, 76, -1/12*E(3)+1/12*E(3)^2 ], [ 63, 77, 1/4*E(3)^2 ], 
  [ 63, 79, -1/12*E(3)+1/12*E(3)^2 ], [ 63, 80, 1/12*E(3) ], 
  [ 63, 81, 1/12*E(3)-1/12*E(3)^2 ], [ 63, 88, 1/6*E(3)+1/12*E(3)^2 ], 
  [ 63, 91, 1/12 ], [ 64, 48, 8/27*E(3)-8/27*E(3)^2 ], 
  [ 64, 57, 4/27*E(3)+2/27*E(3)^2 ], [ 64, 62, -16/27*E(3)-8/27*E(3)^2 ], 
  [ 64, 68, -8/27*E(3)-16/27*E(3)^2 ], [ 64, 69, -8/27*E(3)-16/27*E(3)^2 ], 
  [ 64, 72, -2/27*E(3)-4/27*E(3)^2 ], [ 64, 78, 2/27*E(3)-2/27*E(3)^2 ], 
  [ 64, 81, -8/27*E(3)+8/27*E(3)^2 ], [ 64, 86, -16/27*E(3)-8/27*E(3)^2 ], 
  [ 65, 5, 5/18 ], [ 65, 12, 5/9*E(3)+10/9*E(3)^2 ], [ 65, 13, 1/9*E(3)^2 ], 
  [ 65, 18, 5/9*E(3)^2 ], [ 65, 19, 5/36*E(3)^2 ], 
  [ 65, 20, 2/9*E(3)-2/9*E(3)^2 ], [ 65, 21, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 65, 24, 5/36*E(3)-5/36*E(3)^2 ], [ 65, 25, -5/36*E(3)+5/36*E(3)^2 ], 
  [ 65, 26, 2/9*E(3) ], [ 65, 27, 1/18*E(3) ], [ 65, 28, -E(3) ], 
  [ 65, 29, -1/9*E(3) ], [ 65, 30, -5/36*E(3) ], 
  [ 65, 31, -1/9*E(3)-1/18*E(3)^2 ], [ 65, 32, -5/9*E(3) ], 
  [ 65, 33, 1/9*E(3)+1/18*E(3)^2 ], [ 65, 34, -2/3*E(3)-1/3*E(3)^2 ], 
  [ 65, 35, -1/6*E(3)-1/12*E(3)^2 ], [ 65, 36, 4/9*E(3)+2/9*E(3)^2 ], 
  [ 65, 37, -10/9*E(3)-5/9*E(3)^2 ], [ 65, 38, -1/18 ], [ 65, 39, -1/4 ], 
  [ 65, 40, -2/9 ], [ 65, 41, 1/4 ], [ 65, 42, -2/9 ], [ 65, 43, -1/18 ], 
  [ 65, 44, 5/18 ], [ 65, 45, 2/9*E(3)+4/9*E(3)^2 ], 
  [ 65, 46, -1/12*E(3)-1/6*E(3)^2 ], [ 65, 47, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 65, 49, -1/3*E(3)-2/3*E(3)^2 ], [ 65, 50, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 65, 51, 1/9*E(3)^2 ], [ 65, 53, -E(3)^2 ], [ 65, 54, 1/18*E(3)^2 ], 
  [ 65, 56, 2/9*E(3)^2 ], [ 65, 58, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 65, 60, -2/9*E(3)+2/9*E(3)^2 ], [ 65, 65, -1/9*E(3) ], [ 66, 48, 5/36 ], 
  [ 66, 52, 5/36*E(3)+5/18*E(3)^2 ], [ 66, 61, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 66, 62, 5/36*E(3)^2 ], [ 66, 63, 5/36*E(3)-5/36*E(3)^2 ], 
  [ 66, 66, -1/4*E(3) ], [ 66, 67, 5/36*E(3)-5/36*E(3)^2 ], 
  [ 66, 68, 1/9*E(3) ], [ 66, 69, 5/36*E(3) ], 
  [ 66, 73, -2/9*E(3)-1/9*E(3)^2 ], [ 66, 74, 1/6*E(3)+1/12*E(3)^2 ], 
  [ 66, 76, -1/4 ], [ 66, 77, -5/18*E(3)-5/36*E(3)^2 ], [ 66, 79, 1/4 ], 
  [ 66, 80, 1/12*E(3)+1/6*E(3)^2 ], [ 66, 81, 5/36 ], 
  [ 66, 83, 1/9*E(3)+2/9*E(3)^2 ], [ 66, 86, 1/9*E(3)^2 ], 
  [ 66, 88, 1/4*E(3)^2 ], [ 66, 91, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 67, 48, -1/12*E(3)+1/12*E(3)^2 ], [ 67, 52, 1/4*E(3) ], 
  [ 67, 55, 1/6*E(3) ], [ 67, 59, -1/3*E(3)-1/6*E(3)^2 ], [ 67, 61, -1/12 ], 
  [ 67, 62, -1/6*E(3)-1/12*E(3)^2 ], [ 67, 63, 1/4 ], 
  [ 67, 66, -1/12*E(3)-1/6*E(3)^2 ], [ 67, 67, -1/4 ], 
  [ 67, 69, -1/12*E(3)-1/6*E(3)^2 ], [ 67, 70, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 67, 74, -1/12*E(3)^2 ], [ 67, 75, 1/6*E(3)^2 ], 
  [ 67, 76, -1/12*E(3)+1/12*E(3)^2 ], [ 67, 77, -1/4*E(3)^2 ], 
  [ 67, 79, -1/12*E(3)+1/12*E(3)^2 ], [ 67, 80, -1/12*E(3) ], 
  [ 67, 81, 1/12*E(3)-1/12*E(3)^2 ], [ 67, 88, 1/6*E(3)+1/12*E(3)^2 ], 
  [ 67, 91, -1/12 ], [ 68, 55, 5/36*E(3)+5/18*E(3)^2 ], 
  [ 68, 59, 5/4*E(3)^2 ], [ 68, 61, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 68, 64, 5/18*E(3)-5/18*E(3)^2 ], [ 68, 66, 1/2*E(3) ], 
  [ 68, 70, -5/4*E(3) ], [ 68, 71, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 68, 74, -1/9*E(3)-1/18*E(3)^2 ], [ 68, 75, 5/18*E(3)+5/36*E(3)^2 ], 
  [ 68, 76, -1/2 ], [ 68, 79, 1/2 ], [ 68, 80, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 68, 84, 1/9*E(3)+2/9*E(3)^2 ], [ 68, 88, -1/2*E(3)^2 ], 
  [ 68, 91, -1/18*E(3)+1/18*E(3)^2 ], [ 69, 48, -1/12 ], 
  [ 69, 52, -1/4*E(3)-1/2*E(3)^2 ], [ 69, 55, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 69, 57, -1/6*E(3)^2 ], [ 69, 59, -1/2*E(3)^2 ], 
  [ 69, 61, -1/36*E(3)+1/36*E(3)^2 ], [ 69, 62, 1/12*E(3)^2 ], 
  [ 69, 63, 1/4*E(3)-1/4*E(3)^2 ], [ 69, 64, 1/9*E(3)-1/9*E(3)^2 ], 
  [ 69, 66, 1/4*E(3) ], [ 69, 67, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 69, 69, 1/12*E(3) ], [ 69, 70, 1/2*E(3) ], 
  [ 69, 71, 1/9*E(3)+1/18*E(3)^2 ], [ 69, 72, 1/6*E(3) ], 
  [ 69, 74, 1/18*E(3)+1/36*E(3)^2 ], [ 69, 75, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 69, 76, -1/4 ], [ 69, 77, 1/2*E(3)+1/4*E(3)^2 ], [ 69, 79, 1/4 ], 
  [ 69, 80, 1/36*E(3)+1/18*E(3)^2 ], [ 69, 81, -1/12 ], 
  [ 69, 84, -1/18*E(3)-1/9*E(3)^2 ], [ 69, 88, -1/4*E(3)^2 ], 
  [ 69, 91, 1/36*E(3)-1/36*E(3)^2 ], [ 70, 48, 1/9 ], 
  [ 70, 52, -1/9*E(3)-2/9*E(3)^2 ], [ 70, 62, 1/9*E(3)^2 ], 
  [ 70, 63, -1/9*E(3)+1/9*E(3)^2 ], [ 70, 67, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 70, 68, -1/9*E(3) ], [ 70, 69, 1/9*E(3) ], 
  [ 70, 73, -2/9*E(3)-1/9*E(3)^2 ], [ 70, 77, 2/9*E(3)+1/9*E(3)^2 ], 
  [ 70, 81, 1/9 ], [ 70, 83, 1/9*E(3)+2/9*E(3)^2 ], [ 70, 86, -1/9*E(3)^2 ], 
  [ 71, 48, -10/27*E(3)+10/27*E(3)^2 ], [ 71, 57, -5/27*E(3)-5/54*E(3)^2 ], 
  [ 71, 61, 2/3 ], [ 71, 62, 20/27*E(3)+10/27*E(3)^2 ], 
  [ 71, 68, -8/27*E(3)-16/27*E(3)^2 ], [ 71, 69, 10/27*E(3)+20/27*E(3)^2 ], 
  [ 71, 71, -1/6*E(3)^2 ], [ 71, 72, 5/54*E(3)+5/27*E(3)^2 ], 
  [ 71, 74, -2/3*E(3)^2 ], [ 71, 78, 2/27*E(3)-2/27*E(3)^2 ], 
  [ 71, 80, -2/3*E(3) ], [ 71, 81, 10/27*E(3)-10/27*E(3)^2 ], 
  [ 71, 84, 1/6*E(3) ], [ 71, 86, -16/27*E(3)-8/27*E(3)^2 ], [ 71, 91, 2/3 ], 
  [ 72, 48, -2/3 ], [ 72, 55, -4/9*E(3)-8/9*E(3)^2 ], [ 72, 57, 1/6*E(3)^2 ], 
  [ 72, 61, 2/9*E(3)-2/9*E(3)^2 ], [ 72, 62, 2/3*E(3)^2 ], 
  [ 72, 64, 1/9*E(3)-1/9*E(3)^2 ], [ 72, 69, 2/3*E(3) ], 
  [ 72, 71, 1/9*E(3)+1/18*E(3)^2 ], [ 72, 72, -1/6*E(3) ], 
  [ 72, 74, -4/9*E(3)-2/9*E(3)^2 ], [ 72, 75, -8/9*E(3)-4/9*E(3)^2 ], 
  [ 72, 80, -2/9*E(3)-4/9*E(3)^2 ], [ 72, 81, -2/3 ], 
  [ 72, 84, -1/18*E(3)-1/9*E(3)^2 ], [ 72, 91, -2/9*E(3)+2/9*E(3)^2 ], 
  [ 73, 55, -5/12*E(3) ], [ 73, 59, 5/6*E(3)+5/12*E(3)^2 ], [ 73, 61, -1/6 ], 
  [ 73, 66, -1/6*E(3)-1/3*E(3)^2 ], [ 73, 70, -5/12*E(3)-5/6*E(3)^2 ], 
  [ 73, 74, -1/6*E(3)^2 ], [ 73, 75, -5/12*E(3)^2 ], 
  [ 73, 76, -1/6*E(3)+1/6*E(3)^2 ], [ 73, 79, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 73, 80, -1/6*E(3) ], [ 73, 88, 1/3*E(3)+1/6*E(3)^2 ], [ 73, 91, -1/6 ], 
  [ 74, 48, -5/108*E(3)+5/108*E(3)^2 ], [ 74, 52, -5/12*E(3) ], 
  [ 74, 57, 5/27*E(3)+5/54*E(3)^2 ], [ 74, 61, -1/12 ], 
  [ 74, 62, 5/54*E(3)+5/108*E(3)^2 ], [ 74, 63, 5/12 ], 
  [ 74, 66, 1/4*E(3)+1/2*E(3)^2 ], [ 74, 67, -5/12 ], 
  [ 74, 68, -1/27*E(3)-2/27*E(3)^2 ], [ 74, 69, 5/108*E(3)+5/54*E(3)^2 ], 
  [ 74, 71, -1/6*E(3)^2 ], [ 74, 72, -5/54*E(3)-5/27*E(3)^2 ], 
  [ 74, 73, -1/3*E(3)^2 ], [ 74, 74, 1/12*E(3)^2 ], 
  [ 74, 76, -1/4*E(3)+1/4*E(3)^2 ], [ 74, 77, 5/12*E(3)^2 ], 
  [ 74, 78, -2/27*E(3)+2/27*E(3)^2 ], [ 74, 79, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 74, 80, 1/12*E(3) ], [ 74, 81, 5/108*E(3)-5/108*E(3)^2 ], 
  [ 74, 83, 1/3*E(3) ], [ 74, 84, 1/6*E(3) ], 
  [ 74, 86, -2/27*E(3)-1/27*E(3)^2 ], [ 74, 88, -1/2*E(3)-1/4*E(3)^2 ], 
  [ 74, 91, -1/12 ], [ 75, 48, -1/27*E(3)+1/27*E(3)^2 ], [ 75, 52, 1/3*E(3) ],
  [ 75, 57, 4/27*E(3)+2/27*E(3)^2 ], [ 75, 62, 2/27*E(3)+1/27*E(3)^2 ], 
  [ 75, 63, -1/3 ], [ 75, 67, 1/3 ], [ 75, 68, 1/27*E(3)+2/27*E(3)^2 ], 
  [ 75, 69, 1/27*E(3)+2/27*E(3)^2 ], [ 75, 72, -2/27*E(3)-4/27*E(3)^2 ], 
  [ 75, 73, -1/3*E(3)^2 ], [ 75, 77, -1/3*E(3)^2 ], 
  [ 75, 78, 2/27*E(3)-2/27*E(3)^2 ], [ 75, 81, 1/27*E(3)-1/27*E(3)^2 ], 
  [ 75, 83, 1/3*E(3) ], [ 75, 86, 2/27*E(3)+1/27*E(3)^2 ], [ 76, 48, -5/36 ], 
  [ 76, 52, 5/36*E(3)+5/18*E(3)^2 ], [ 76, 61, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 76, 62, -5/36*E(3)^2 ], [ 76, 63, 5/36*E(3)-5/36*E(3)^2 ], 
  [ 76, 66, -1/4*E(3) ], [ 76, 67, 5/36*E(3)-5/36*E(3)^2 ], 
  [ 76, 68, -1/9*E(3) ], [ 76, 69, -5/36*E(3) ], 
  [ 76, 73, -2/9*E(3)-1/9*E(3)^2 ], [ 76, 74, -1/6*E(3)-1/12*E(3)^2 ], 
  [ 76, 76, -1/4 ], [ 76, 77, -5/18*E(3)-5/36*E(3)^2 ], [ 76, 79, 1/4 ], 
  [ 76, 80, -1/12*E(3)-1/6*E(3)^2 ], [ 76, 81, -5/36 ], 
  [ 76, 83, 1/9*E(3)+2/9*E(3)^2 ], [ 76, 86, -1/9*E(3)^2 ], 
  [ 76, 88, 1/4*E(3)^2 ], [ 76, 91, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 77, 48, 1/12*E(3)-1/12*E(3)^2 ], [ 77, 52, 1/4*E(3) ], 
  [ 77, 55, -1/6*E(3) ], [ 77, 59, -1/3*E(3)-1/6*E(3)^2 ], [ 77, 61, 1/12 ], 
  [ 77, 62, 1/6*E(3)+1/12*E(3)^2 ], [ 77, 63, 1/4 ], 
  [ 77, 66, -1/12*E(3)-1/6*E(3)^2 ], [ 77, 67, -1/4 ], 
  [ 77, 69, 1/12*E(3)+1/6*E(3)^2 ], [ 77, 70, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 77, 74, 1/12*E(3)^2 ], [ 77, 75, -1/6*E(3)^2 ], 
  [ 77, 76, -1/12*E(3)+1/12*E(3)^2 ], [ 77, 77, -1/4*E(3)^2 ], 
  [ 77, 79, -1/12*E(3)+1/12*E(3)^2 ], [ 77, 80, 1/12*E(3) ], 
  [ 77, 81, -1/12*E(3)+1/12*E(3)^2 ], [ 77, 88, 1/6*E(3)+1/12*E(3)^2 ], 
  [ 77, 91, 1/12 ], [ 78, 55, 10/9*E(3)+20/9*E(3)^2 ], 
  [ 78, 61, 4/9*E(3)-4/9*E(3)^2 ], [ 78, 64, -5/18*E(3)+5/18*E(3)^2 ], 
  [ 78, 71, 2/9*E(3)+1/9*E(3)^2 ], [ 78, 74, -8/9*E(3)-4/9*E(3)^2 ], 
  [ 78, 75, 20/9*E(3)+10/9*E(3)^2 ], [ 78, 80, -4/9*E(3)-8/9*E(3)^2 ], 
  [ 78, 84, -1/9*E(3)-2/9*E(3)^2 ], [ 78, 91, -4/9*E(3)+4/9*E(3)^2 ], 
  [ 79, 48, 5/36 ], [ 79, 52, 5/36*E(3)+5/18*E(3)^2 ], 
  [ 79, 61, -1/12*E(3)+1/12*E(3)^2 ], [ 79, 62, 5/36*E(3)^2 ], 
  [ 79, 63, 5/36*E(3)-5/36*E(3)^2 ], [ 79, 66, 1/4*E(3) ], 
  [ 79, 67, 5/36*E(3)-5/36*E(3)^2 ], [ 79, 68, 1/9*E(3) ], 
  [ 79, 69, 5/36*E(3) ], [ 79, 73, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 79, 74, -1/6*E(3)-1/12*E(3)^2 ], [ 79, 76, 1/4 ], 
  [ 79, 77, -5/18*E(3)-5/36*E(3)^2 ], [ 79, 79, -1/4 ], 
  [ 79, 80, -1/12*E(3)-1/6*E(3)^2 ], [ 79, 81, 5/36 ], 
  [ 79, 83, 1/9*E(3)+2/9*E(3)^2 ], [ 79, 86, 1/9*E(3)^2 ], 
  [ 79, 88, -1/4*E(3)^2 ], [ 79, 91, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 80, 48, 5/108*E(3)-5/108*E(3)^2 ], [ 80, 52, -5/12*E(3) ], 
  [ 80, 57, -5/27*E(3)-5/54*E(3)^2 ], [ 80, 61, -1/12 ], 
  [ 80, 62, -5/54*E(3)-5/108*E(3)^2 ], [ 80, 63, 5/12 ], 
  [ 80, 66, -1/4*E(3)-1/2*E(3)^2 ], [ 80, 67, -5/12 ], 
  [ 80, 68, 1/27*E(3)+2/27*E(3)^2 ], [ 80, 69, -5/108*E(3)-5/54*E(3)^2 ], 
  [ 80, 71, -1/6*E(3)^2 ], [ 80, 72, 5/54*E(3)+5/27*E(3)^2 ], 
  [ 80, 73, -1/3*E(3)^2 ], [ 80, 74, 1/12*E(3)^2 ], 
  [ 80, 76, 1/4*E(3)-1/4*E(3)^2 ], [ 80, 77, 5/12*E(3)^2 ], 
  [ 80, 78, 2/27*E(3)-2/27*E(3)^2 ], [ 80, 79, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 80, 80, 1/12*E(3) ], [ 80, 81, -5/108*E(3)+5/108*E(3)^2 ], 
  [ 80, 83, 1/3*E(3) ], [ 80, 84, 1/6*E(3) ], 
  [ 80, 86, 2/27*E(3)+1/27*E(3)^2 ], [ 80, 88, 1/2*E(3)+1/4*E(3)^2 ], 
  [ 80, 91, -1/12 ], [ 81, 48, 1/12 ], [ 81, 52, 1/4*E(3)+1/2*E(3)^2 ], 
  [ 81, 55, 1/18*E(3)+1/9*E(3)^2 ], [ 81, 57, 1/6*E(3)^2 ], 
  [ 81, 59, -1/2*E(3)^2 ], [ 81, 61, -1/36*E(3)+1/36*E(3)^2 ], 
  [ 81, 62, -1/12*E(3)^2 ], [ 81, 63, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 81, 64, 1/9*E(3)-1/9*E(3)^2 ], [ 81, 66, 1/4*E(3) ], 
  [ 81, 67, -1/4*E(3)+1/4*E(3)^2 ], [ 81, 69, -1/12*E(3) ], 
  [ 81, 70, 1/2*E(3) ], [ 81, 71, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 81, 72, -1/6*E(3) ], [ 81, 74, 1/18*E(3)+1/36*E(3)^2 ], 
  [ 81, 75, 1/9*E(3)+1/18*E(3)^2 ], [ 81, 76, -1/4 ], 
  [ 81, 77, -1/2*E(3)-1/4*E(3)^2 ], [ 81, 79, 1/4 ], 
  [ 81, 80, 1/36*E(3)+1/18*E(3)^2 ], [ 81, 81, 1/12 ], 
  [ 81, 84, -1/18*E(3)-1/9*E(3)^2 ], [ 81, 88, -1/4*E(3)^2 ], 
  [ 81, 91, 1/36*E(3)-1/36*E(3)^2 ], [ 82, 82, -1/18 ], 
  [ 82, 85, -1/18*E(3)-1/9*E(3)^2 ], [ 82, 87, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 82, 89, 4/9*E(3)^2 ], [ 82, 90, -1/6*E(3)^2 ], [ 82, 92, 1/18*E(3)^2 ], 
  [ 82, 93, -1/18*E(3)+1/18*E(3)^2 ], [ 82, 94, -4/9*E(3)+4/9*E(3)^2 ], 
  [ 82, 95, 1/18*E(3)-1/18*E(3)^2 ], [ 82, 96, -1/18*E(3) ], 
  [ 82, 97, 1/18*E(3) ], [ 82, 98, -1/6*E(3) ], [ 82, 99, -4/9*E(3) ], 
  [ 82, 100, 1/9*E(3)+1/18*E(3)^2 ], [ 82, 101, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 82, 102, -1/9*E(3)-1/18*E(3)^2 ], [ 82, 103, 4/9 ], [ 82, 104, -1/18 ], 
  [ 82, 105, -1/18*E(3)-1/9*E(3)^2 ], [ 82, 107, 1/18*E(3)^2 ], 
  [ 83, 55, 5/12*E(3) ], [ 83, 59, 5/6*E(3)+5/12*E(3)^2 ], [ 83, 61, 1/6 ], 
  [ 83, 66, -1/6*E(3)-1/3*E(3)^2 ], [ 83, 70, -5/12*E(3)-5/6*E(3)^2 ], 
  [ 83, 74, 1/6*E(3)^2 ], [ 83, 75, 5/12*E(3)^2 ], 
  [ 83, 76, -1/6*E(3)+1/6*E(3)^2 ], [ 83, 79, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 83, 80, 1/6*E(3) ], [ 83, 88, 1/3*E(3)+1/6*E(3)^2 ], [ 83, 91, 1/6 ], 
  [ 84, 48, -10/27*E(3)+10/27*E(3)^2 ], [ 84, 57, -5/27*E(3)-5/54*E(3)^2 ], 
  [ 84, 61, -2/3 ], [ 84, 62, 20/27*E(3)+10/27*E(3)^2 ], 
  [ 84, 68, -8/27*E(3)-16/27*E(3)^2 ], [ 84, 69, 10/27*E(3)+20/27*E(3)^2 ], 
  [ 84, 71, 1/6*E(3)^2 ], [ 84, 72, 5/54*E(3)+5/27*E(3)^2 ], 
  [ 84, 74, 2/3*E(3)^2 ], [ 84, 78, 2/27*E(3)-2/27*E(3)^2 ], 
  [ 84, 80, 2/3*E(3) ], [ 84, 81, 10/27*E(3)-10/27*E(3)^2 ], 
  [ 84, 84, -1/6*E(3) ], [ 84, 86, -16/27*E(3)-8/27*E(3)^2 ], 
  [ 84, 91, -2/3 ], [ 85, 82, 1/6*E(3)-1/6*E(3)^2 ], [ 85, 85, 1/6*E(3) ], 
  [ 85, 87, 1/2*E(3) ], [ 85, 90, 1/3*E(3)+1/6*E(3)^2 ], 
  [ 85, 92, 1/3*E(3)+1/6*E(3)^2 ], [ 85, 93, -1/6 ], [ 85, 95, -1/6 ], 
  [ 85, 96, 1/6*E(3)+1/3*E(3)^2 ], [ 85, 97, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 85, 98, 1/6*E(3)+1/3*E(3)^2 ], [ 85, 100, 1/6*E(3)^2 ], 
  [ 85, 101, 1/2*E(3)^2 ], [ 85, 102, 1/6*E(3)^2 ], 
  [ 85, 104, -1/6*E(3)+1/6*E(3)^2 ], [ 85, 105, -1/6*E(3) ], 
  [ 85, 107, -1/3*E(3)-1/6*E(3)^2 ], [ 86, 55, -5/36*E(3)-5/18*E(3)^2 ], 
  [ 86, 59, 5/4*E(3)^2 ], [ 86, 61, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 86, 64, -5/18*E(3)+5/18*E(3)^2 ], [ 86, 66, 1/2*E(3) ], 
  [ 86, 70, -5/4*E(3) ], [ 86, 71, 2/9*E(3)+1/9*E(3)^2 ], 
  [ 86, 74, 1/9*E(3)+1/18*E(3)^2 ], [ 86, 75, -5/18*E(3)-5/36*E(3)^2 ], 
  [ 86, 76, -1/2 ], [ 86, 79, 1/2 ], [ 86, 80, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 86, 84, -1/9*E(3)-2/9*E(3)^2 ], [ 86, 88, -1/2*E(3)^2 ], 
  [ 86, 91, 1/18*E(3)-1/18*E(3)^2 ], [ 87, 82, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 87, 85, 1/4*E(3) ], [ 87, 89, -4/3*E(3)-2/3*E(3)^2 ], 
  [ 87, 92, 1/6*E(3)+1/12*E(3)^2 ], [ 87, 93, 1/4 ], [ 87, 95, 1/4 ], 
  [ 87, 97, 1/12*E(3)+1/6*E(3)^2 ], [ 87, 99, 2/3*E(3)+4/3*E(3)^2 ], 
  [ 87, 102, 1/4*E(3)^2 ], [ 87, 104, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 88, 48, -5/36 ], [ 88, 52, 5/36*E(3)+5/18*E(3)^2 ], 
  [ 88, 61, 1/12*E(3)-1/12*E(3)^2 ], [ 88, 62, -5/36*E(3)^2 ], 
  [ 88, 63, 5/36*E(3)-5/36*E(3)^2 ], [ 88, 66, 1/4*E(3) ], 
  [ 88, 67, 5/36*E(3)-5/36*E(3)^2 ], [ 88, 68, -1/9*E(3) ], 
  [ 88, 69, -5/36*E(3) ], [ 88, 73, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 88, 74, 1/6*E(3)+1/12*E(3)^2 ], [ 88, 76, 1/4 ], 
  [ 88, 77, -5/18*E(3)-5/36*E(3)^2 ], [ 88, 79, -1/4 ], 
  [ 88, 80, 1/12*E(3)+1/6*E(3)^2 ], [ 88, 81, -5/36 ], 
  [ 88, 83, 1/9*E(3)+2/9*E(3)^2 ], [ 88, 86, -1/9*E(3)^2 ], 
  [ 88, 88, -1/4*E(3)^2 ], [ 88, 91, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 89, 82, 1/9 ], [ 89, 87, -1/9*E(3)-2/9*E(3)^2 ], [ 89, 89, 1/9*E(3)^2 ], 
  [ 89, 92, -1/9*E(3)^2 ], [ 89, 94, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 89, 96, 1/9*E(3) ], [ 89, 97, -1/9*E(3) ], [ 89, 99, -1/9*E(3) ], 
  [ 89, 101, -2/9*E(3)-1/9*E(3)^2 ], [ 89, 103, 1/9 ], [ 89, 104, 1/9 ], 
  [ 89, 107, -1/9*E(3)^2 ], [ 90, 82, -3/4 ], [ 90, 85, 1/4*E(3)+1/2*E(3)^2 ],
  [ 90, 92, -3/4*E(3)^2 ], [ 90, 93, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 90, 95, 1/4*E(3)-1/4*E(3)^2 ], [ 90, 97, -3/4*E(3) ], 
  [ 90, 102, 1/2*E(3)+1/4*E(3)^2 ], [ 90, 104, -3/4 ], 
  [ 91, 48, 5/108*E(3)-5/108*E(3)^2 ], [ 91, 52, -5/12*E(3) ], 
  [ 91, 57, -5/27*E(3)-5/54*E(3)^2 ], [ 91, 61, 1/12 ], 
  [ 91, 62, -5/54*E(3)-5/108*E(3)^2 ], [ 91, 63, 5/12 ], 
  [ 91, 66, 1/4*E(3)+1/2*E(3)^2 ], [ 91, 67, -5/12 ], 
  [ 91, 68, 1/27*E(3)+2/27*E(3)^2 ], [ 91, 69, -5/108*E(3)-5/54*E(3)^2 ], 
  [ 91, 71, 1/6*E(3)^2 ], [ 91, 72, 5/54*E(3)+5/27*E(3)^2 ], 
  [ 91, 73, -1/3*E(3)^2 ], [ 91, 74, -1/12*E(3)^2 ], 
  [ 91, 76, -1/4*E(3)+1/4*E(3)^2 ], [ 91, 77, 5/12*E(3)^2 ], 
  [ 91, 78, 2/27*E(3)-2/27*E(3)^2 ], [ 91, 79, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 91, 80, -1/12*E(3) ], [ 91, 81, -5/108*E(3)+5/108*E(3)^2 ], 
  [ 91, 83, 1/3*E(3) ], [ 91, 84, -1/6*E(3) ], 
  [ 91, 86, 2/27*E(3)+1/27*E(3)^2 ], [ 91, 88, -1/2*E(3)-1/4*E(3)^2 ], 
  [ 91, 91, 1/12 ], [ 92, 82, 1/18 ], [ 92, 85, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 92, 87, 1/18*E(3)+1/9*E(3)^2 ], [ 92, 89, -4/9*E(3)^2 ], 
  [ 92, 90, -1/6*E(3)^2 ], [ 92, 92, -1/18*E(3)^2 ], 
  [ 92, 93, 1/18*E(3)-1/18*E(3)^2 ], [ 92, 94, -4/9*E(3)+4/9*E(3)^2 ], 
  [ 92, 95, -1/18*E(3)+1/18*E(3)^2 ], [ 92, 96, 1/18*E(3) ], 
  [ 92, 97, -1/18*E(3) ], [ 92, 98, -1/6*E(3) ], [ 92, 99, 4/9*E(3) ], 
  [ 92, 100, -1/9*E(3)-1/18*E(3)^2 ], [ 92, 101, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 92, 102, 1/9*E(3)+1/18*E(3)^2 ], [ 92, 103, -4/9 ], [ 92, 104, 1/18 ], 
  [ 92, 105, 1/18*E(3)+1/9*E(3)^2 ], [ 92, 107, -1/18*E(3)^2 ], 
  [ 93, 82, 1/6*E(3)-1/6*E(3)^2 ], [ 93, 85, -1/6*E(3) ], 
  [ 93, 87, 1/2*E(3) ], [ 93, 90, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 93, 92, 1/3*E(3)+1/6*E(3)^2 ], [ 93, 93, 1/6 ], [ 93, 95, 1/6 ], 
  [ 93, 96, 1/6*E(3)+1/3*E(3)^2 ], [ 93, 97, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 93, 98, -1/6*E(3)-1/3*E(3)^2 ], [ 93, 100, -1/6*E(3)^2 ], 
  [ 93, 101, 1/2*E(3)^2 ], [ 93, 102, -1/6*E(3)^2 ], 
  [ 93, 104, -1/6*E(3)+1/6*E(3)^2 ], [ 93, 105, 1/6*E(3) ], 
  [ 93, 107, -1/3*E(3)-1/6*E(3)^2 ], [ 94, 82, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 94, 89, -1/3*E(3)-1/6*E(3)^2 ], [ 94, 92, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 94, 97, -1/6*E(3)-1/3*E(3)^2 ], [ 94, 99, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 94, 104, -1/6*E(3)+1/6*E(3)^2 ], [ 95, 82, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 95, 85, -1/6*E(3) ], [ 95, 87, 1/2*E(3) ], 
  [ 95, 90, 1/3*E(3)+1/6*E(3)^2 ], [ 95, 92, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 95, 93, 1/6 ], [ 95, 95, 1/6 ], [ 95, 96, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 95, 97, -1/6*E(3)-1/3*E(3)^2 ], [ 95, 98, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 95, 100, -1/6*E(3)^2 ], [ 95, 101, 1/2*E(3)^2 ], [ 95, 102, -1/6*E(3)^2 ],
  [ 95, 104, 1/6*E(3)-1/6*E(3)^2 ], [ 95, 105, 1/6*E(3) ], 
  [ 95, 107, 1/3*E(3)+1/6*E(3)^2 ], [ 96, 82, -5/36 ], 
  [ 96, 85, -5/36*E(3)-5/18*E(3)^2 ], [ 96, 89, 10/9*E(3)^2 ], 
  [ 96, 92, 5/36*E(3)^2 ], [ 96, 93, -5/36*E(3)+5/36*E(3)^2 ], 
  [ 96, 95, 5/36*E(3)-5/36*E(3)^2 ], [ 96, 96, 1/9*E(3) ], 
  [ 96, 97, 5/36*E(3) ], [ 96, 99, -10/9*E(3) ], 
  [ 96, 100, -2/9*E(3)-1/9*E(3)^2 ], [ 96, 102, -5/18*E(3)-5/36*E(3)^2 ], 
  [ 96, 103, -8/9 ], [ 96, 104, -5/36 ], [ 96, 105, 1/9*E(3)+2/9*E(3)^2 ], 
  [ 96, 107, -1/9*E(3)^2 ], [ 97, 82, 1/18 ], 
  [ 97, 85, -1/18*E(3)-1/9*E(3)^2 ], [ 97, 87, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 97, 89, -4/9*E(3)^2 ], [ 97, 90, -1/6*E(3)^2 ], [ 97, 92, -1/18*E(3)^2 ], 
  [ 97, 93, -1/18*E(3)+1/18*E(3)^2 ], [ 97, 94, 4/9*E(3)-4/9*E(3)^2 ], 
  [ 97, 95, 1/18*E(3)-1/18*E(3)^2 ], [ 97, 96, 1/18*E(3) ], 
  [ 97, 97, -1/18*E(3) ], [ 97, 98, -1/6*E(3) ], [ 97, 99, 4/9*E(3) ], 
  [ 97, 100, 1/9*E(3)+1/18*E(3)^2 ], [ 97, 101, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 97, 102, -1/9*E(3)-1/18*E(3)^2 ], [ 97, 103, -4/9 ], [ 97, 104, 1/18 ], 
  [ 97, 105, -1/18*E(3)-1/9*E(3)^2 ], [ 97, 107, -1/18*E(3)^2 ], 
  [ 98, 82, -3/4 ], [ 98, 85, -1/4*E(3)-1/2*E(3)^2 ], [ 98, 92, -3/4*E(3)^2 ],
  [ 98, 93, 1/4*E(3)-1/4*E(3)^2 ], [ 98, 95, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 98, 97, -3/4*E(3) ], [ 98, 102, -1/2*E(3)-1/4*E(3)^2 ], [ 98, 104, -3/4 ],
  [ 99, 82, -1/9 ], [ 99, 87, -1/9*E(3)-2/9*E(3)^2 ], [ 99, 89, -1/9*E(3)^2 ],
  [ 99, 92, 1/9*E(3)^2 ], [ 99, 94, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 99, 96, -1/9*E(3) ], [ 99, 97, 1/9*E(3) ], [ 99, 99, 1/9*E(3) ], 
  [ 99, 101, -2/9*E(3)-1/9*E(3)^2 ], [ 99, 103, -1/9 ], [ 99, 104, -1/9 ], 
  [ 99, 107, 1/9*E(3)^2 ], [ 100, 82, 5/12*E(3)-5/12*E(3)^2 ], 
  [ 100, 85, 5/12*E(3) ], [ 100, 92, 5/6*E(3)+5/12*E(3)^2 ], 
  [ 100, 93, -5/12 ], [ 100, 95, -5/12 ], [ 100, 96, -1/3*E(3)-2/3*E(3)^2 ], 
  [ 100, 97, 5/12*E(3)+5/6*E(3)^2 ], [ 100, 100, -1/3*E(3)^2 ], 
  [ 100, 102, 5/12*E(3)^2 ], [ 100, 104, -5/12*E(3)+5/12*E(3)^2 ], 
  [ 100, 105, 1/3*E(3) ], [ 100, 107, 2/3*E(3)+1/3*E(3)^2 ], 
  [ 101, 82, 1/12*E(3)-1/12*E(3)^2 ], [ 101, 85, 1/4*E(3) ], 
  [ 101, 89, 4/3*E(3)+2/3*E(3)^2 ], [ 101, 92, -1/6*E(3)-1/12*E(3)^2 ], 
  [ 101, 93, 1/4 ], [ 101, 95, 1/4 ], [ 101, 97, -1/12*E(3)-1/6*E(3)^2 ], 
  [ 101, 99, -2/3*E(3)-4/3*E(3)^2 ], [ 101, 102, 1/4*E(3)^2 ], 
  [ 101, 104, -1/12*E(3)+1/12*E(3)^2 ], [ 102, 82, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 102, 85, 1/6*E(3) ], [ 102, 87, 1/2*E(3) ], 
  [ 102, 90, -1/3*E(3)-1/6*E(3)^2 ], [ 102, 92, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 102, 93, -1/6 ], [ 102, 95, -1/6 ], [ 102, 96, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 102, 97, -1/6*E(3)-1/3*E(3)^2 ], [ 102, 98, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 102, 100, 1/6*E(3)^2 ], [ 102, 101, 1/2*E(3)^2 ], 
  [ 102, 102, 1/6*E(3)^2 ], [ 102, 104, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 102, 105, -1/6*E(3) ], [ 102, 107, 1/3*E(3)+1/6*E(3)^2 ], 
  [ 103, 82, 5/18 ], [ 103, 89, 5/18*E(3)^2 ], [ 103, 92, -5/18*E(3)^2 ], 
  [ 103, 96, -2/9*E(3) ], [ 103, 97, -5/18*E(3) ], [ 103, 99, -5/18*E(3) ], 
  [ 103, 103, -2/9 ], [ 103, 104, 5/18 ], [ 103, 107, 2/9*E(3)^2 ], 
  [ 104, 82, -1/18 ], [ 104, 85, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 104, 87, -1/18*E(3)-1/9*E(3)^2 ], [ 104, 89, 4/9*E(3)^2 ], 
  [ 104, 90, -1/6*E(3)^2 ], [ 104, 92, 1/18*E(3)^2 ], 
  [ 104, 93, 1/18*E(3)-1/18*E(3)^2 ], [ 104, 94, 4/9*E(3)-4/9*E(3)^2 ], 
  [ 104, 95, -1/18*E(3)+1/18*E(3)^2 ], [ 104, 96, -1/18*E(3) ], 
  [ 104, 97, 1/18*E(3) ], [ 104, 98, -1/6*E(3) ], [ 104, 99, -4/9*E(3) ], 
  [ 104, 100, -1/9*E(3)-1/18*E(3)^2 ], [ 104, 101, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 104, 102, 1/9*E(3)+1/18*E(3)^2 ], [ 104, 103, 4/9 ], [ 104, 104, -1/18 ], 
  [ 104, 105, 1/18*E(3)+1/9*E(3)^2 ], [ 104, 107, 1/18*E(3)^2 ], 
  [ 105, 82, 5/12*E(3)-5/12*E(3)^2 ], [ 105, 85, -5/12*E(3) ], 
  [ 105, 92, 5/6*E(3)+5/12*E(3)^2 ], [ 105, 93, 5/12 ], [ 105, 95, 5/12 ], 
  [ 105, 96, -1/3*E(3)-2/3*E(3)^2 ], [ 105, 97, 5/12*E(3)+5/6*E(3)^2 ], 
  [ 105, 100, 1/3*E(3)^2 ], [ 105, 102, -5/12*E(3)^2 ], 
  [ 105, 104, -5/12*E(3)+5/12*E(3)^2 ], [ 105, 105, -1/3*E(3) ], 
  [ 105, 107, 2/3*E(3)+1/3*E(3)^2 ], [ 106, 106, -1/3 ], 
  [ 106, 108, 1/18*E(3)+1/9*E(3)^2 ], [ 106, 109, 1/6*E(3)^2 ], 
  [ 106, 110, -1/6*E(3)^2 ], [ 106, 111, 1/2*E(3)-1/2*E(3)^2 ], 
  [ 106, 112, 1/18*E(3)-1/18*E(3)^2 ], [ 106, 113, -1/6*E(3) ], 
  [ 106, 114, 1/6*E(3) ], [ 106, 115, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 106, 116, 1/3 ], [ 107, 82, 5/36 ], [ 107, 85, -5/36*E(3)-5/18*E(3)^2 ], 
  [ 107, 89, -10/9*E(3)^2 ], [ 107, 92, -5/36*E(3)^2 ], 
  [ 107, 93, -5/36*E(3)+5/36*E(3)^2 ], [ 107, 95, 5/36*E(3)-5/36*E(3)^2 ], 
  [ 107, 96, -1/9*E(3) ], [ 107, 97, -5/36*E(3) ], [ 107, 99, 10/9*E(3) ], 
  [ 107, 100, -2/9*E(3)-1/9*E(3)^2 ], [ 107, 102, -5/18*E(3)-5/36*E(3)^2 ], 
  [ 107, 103, 8/9 ], [ 107, 104, 5/36 ], [ 107, 105, 1/9*E(3)+2/9*E(3)^2 ], 
  [ 107, 107, 1/9*E(3)^2 ], [ 108, 106, -2/3*E(3)+2/3*E(3)^2 ], 
  [ 108, 108, -1/2*E(3) ], [ 108, 109, 1/3*E(3)+1/6*E(3)^2 ], 
  [ 108, 110, 4/3*E(3)+2/3*E(3)^2 ], [ 108, 113, 2/3*E(3)+4/3*E(3)^2 ], 
  [ 108, 114, 1/6*E(3)+1/3*E(3)^2 ], [ 108, 115, 1/2*E(3)^2 ], 
  [ 108, 116, -2/3*E(3)+2/3*E(3)^2 ], [ 109, 106, 2/3 ], 
  [ 109, 108, 1/18*E(3)+1/9*E(3)^2 ], [ 109, 109, 1/6*E(3)^2 ], 
  [ 109, 110, -2/3*E(3)^2 ], [ 109, 112, -4/9*E(3)+4/9*E(3)^2 ], 
  [ 109, 113, -2/3*E(3) ], [ 109, 114, 1/6*E(3) ], 
  [ 109, 115, -1/9*E(3)-1/18*E(3)^2 ], [ 109, 116, -2/3 ], [ 110, 106, -1/6 ],
  [ 110, 108, 1/18*E(3)+1/9*E(3)^2 ], [ 110, 109, -1/6*E(3)^2 ], 
  [ 110, 110, -1/3*E(3)^2 ], [ 110, 111, -1/2*E(3)+1/2*E(3)^2 ], 
  [ 110, 112, 1/18*E(3)-1/18*E(3)^2 ], [ 110, 113, -1/3*E(3) ], 
  [ 110, 114, -1/6*E(3) ], [ 110, 115, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 110, 116, 1/6 ], [ 111, 106, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 111, 110, -1/3*E(3)-1/6*E(3)^2 ], [ 111, 113, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 111, 116, -1/6*E(3)+1/6*E(3)^2 ], [ 112, 106, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 112, 109, -2/3*E(3)-1/3*E(3)^2 ], [ 112, 110, 1/3*E(3)+1/6*E(3)^2 ], 
  [ 112, 113, 1/6*E(3)+1/3*E(3)^2 ], [ 112, 114, -1/3*E(3)-2/3*E(3)^2 ], 
  [ 112, 116, -1/6*E(3)+1/6*E(3)^2 ], [ 113, 106, -1/6 ], 
  [ 113, 108, -1/18*E(3)-1/9*E(3)^2 ], [ 113, 109, -1/6*E(3)^2 ], 
  [ 113, 110, -1/3*E(3)^2 ], [ 113, 111, 1/2*E(3)-1/2*E(3)^2 ], 
  [ 113, 112, -1/18*E(3)+1/18*E(3)^2 ], [ 113, 113, -1/3*E(3) ], 
  [ 113, 114, -1/6*E(3) ], [ 113, 115, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 113, 116, 1/6 ], [ 114, 106, 2/3 ], [ 114, 108, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 114, 109, 1/6*E(3)^2 ], [ 114, 110, -2/3*E(3)^2 ], 
  [ 114, 112, 4/9*E(3)-4/9*E(3)^2 ], [ 114, 113, -2/3*E(3) ], 
  [ 114, 114, 1/6*E(3) ], [ 114, 115, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 114, 116, -2/3 ], [ 115, 106, -2/3*E(3)+2/3*E(3)^2 ], 
  [ 115, 108, 1/2*E(3) ], [ 115, 109, 1/3*E(3)+1/6*E(3)^2 ], 
  [ 115, 110, 4/3*E(3)+2/3*E(3)^2 ], [ 115, 113, 2/3*E(3)+4/3*E(3)^2 ], 
  [ 115, 114, 1/6*E(3)+1/3*E(3)^2 ], [ 115, 115, -1/2*E(3)^2 ], 
  [ 115, 116, -2/3*E(3)+2/3*E(3)^2 ], [ 116, 106, 1/3 ], 
  [ 116, 108, 1/18*E(3)+1/9*E(3)^2 ], [ 116, 109, -1/6*E(3)^2 ], 
  [ 116, 110, 1/6*E(3)^2 ], [ 116, 111, 1/2*E(3)-1/2*E(3)^2 ], 
  [ 116, 112, 1/18*E(3)-1/18*E(3)^2 ], [ 116, 113, 1/6*E(3) ], 
  [ 116, 114, -1/6*E(3) ], [ 116, 115, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 116, 116, -1/3 ], [ 117, 117, -2/3 ], [ 117, 118, -1/3*E(3)^2 ], 
  [ 117, 120, 1/3*E(3) ], [ 118, 117, -5/6 ], [ 118, 118, 1/3*E(3)^2 ], 
  [ 118, 119, 1/2*E(3)-1/2*E(3)^2 ], [ 118, 120, -1/3*E(3) ], 
  [ 119, 118, 2/3*E(3)+1/3*E(3)^2 ], [ 119, 120, -1/3*E(3)-2/3*E(3)^2 ], 
  [ 120, 117, 5/6 ], [ 120, 118, -1/3*E(3)^2 ], 
  [ 120, 119, 1/2*E(3)-1/2*E(3)^2 ], [ 120, 120, 1/3*E(3) ], 
  [ 121, 1, -3/4*E(3) ], [ 121, 2, -1/12*E(3) ], [ 121, 3, -1/6*E(3) ], 
  [ 121, 9, -1/3*E(3)-1/6*E(3)^2 ], [ 121, 11, 1/2*E(3)+1/4*E(3)^2 ], 
  [ 121, 16, -3/4 ], [ 121, 17, -1/12 ], [ 121, 23, 1/12*E(3)+1/6*E(3)^2 ], 
  [ 121, 121, -1/12*E(3)^2 ], [ 121, 122, -3/4*E(3)^2 ], 
  [ 121, 124, 1/6*E(3)^2 ], [ 121, 130, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 121, 131, 1/4*E(3)+1/2*E(3)^2 ], [ 121, 136, 1/12 ], [ 121, 137, 3/4 ], 
  [ 121, 143, 1/6*E(3)+1/12*E(3)^2 ], [ 121, 241, -1/3*E(3)+1/3*E(3)^2 ], 
  [ 121, 242, -1/3*E(3)+1/3*E(3)^2 ], [ 121, 243, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 121, 244, 1/4*E(3)-1/4*E(3)^2 ], [ 121, 245, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 122, 1, -1/12*E(3) ], [ 122, 2, -1/12*E(3) ], [ 122, 4, 2/3*E(3) ], 
  [ 122, 10, 4/9*E(3)+2/9*E(3)^2 ], [ 122, 11, -1/6*E(3)-1/12*E(3)^2 ], 
  [ 122, 16, 1/12 ], [ 122, 17, 1/12 ], [ 122, 23, -1/36*E(3)-1/18*E(3)^2 ], 
  [ 122, 121, -1/12*E(3)^2 ], [ 122, 122, -1/12*E(3)^2 ], 
  [ 122, 123, -2/3*E(3)^2 ], [ 122, 129, 2/9*E(3)+4/9*E(3)^2 ], 
  [ 122, 131, -1/12*E(3)-1/6*E(3)^2 ], [ 122, 136, -1/12 ], 
  [ 122, 137, -1/12 ], [ 122, 143, -1/18*E(3)-1/36*E(3)^2 ], 
  [ 122, 241, -1/9*E(3)+1/9*E(3)^2 ], [ 122, 242, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 122, 243, 1/12*E(3)-1/12*E(3)^2 ], [ 122, 244, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 122, 245, -1/18*E(3)+1/18*E(3)^2 ], [ 123, 1, -1/6*E(3) ], 
  [ 123, 4, -1/6*E(3) ], [ 123, 10, 1/9*E(3)+1/18*E(3)^2 ], [ 123, 16, 1/6 ], 
  [ 123, 23, 1/18*E(3)+1/9*E(3)^2 ], [ 123, 122, -1/6*E(3)^2 ], 
  [ 123, 123, 1/6*E(3)^2 ], [ 123, 129, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 123, 137, -1/6 ], [ 123, 143, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 123, 241, 1/18*E(3)-1/18*E(3)^2 ], [ 123, 242, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 123, 245, -1/18*E(3)+1/18*E(3)^2 ], [ 124, 2, 2/3*E(3) ], 
  [ 124, 3, -1/6*E(3) ], [ 124, 9, 1/3*E(3)+1/6*E(3)^2 ], [ 124, 17, 2/3 ], 
  [ 124, 23, 2/3*E(3)+4/3*E(3)^2 ], [ 124, 121, 2/3*E(3)^2 ], 
  [ 124, 124, 1/6*E(3)^2 ], [ 124, 130, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 124, 136, -2/3 ], [ 124, 143, 4/3*E(3)+2/3*E(3)^2 ], 
  [ 124, 241, -2/3*E(3)+2/3*E(3)^2 ], [ 124, 242, -2/3*E(3)+2/3*E(3)^2 ], 
  [ 124, 245, 2/3*E(3)-2/3*E(3)^2 ], [ 125, 125, 2/9 ], 
  [ 125, 132, -2/27*E(3)-1/27*E(3)^2 ], [ 125, 133, -5/18*E(3) ], 
  [ 125, 138, 1/9*E(3) ], [ 125, 139, 1/9*E(3) ], 
  [ 125, 140, -5/108*E(3)+5/108*E(3)^2 ], [ 125, 144, 1/27*E(3)-1/27*E(3)^2 ],
  [ 125, 145, -1/3*E(3)+1/3*E(3)^2 ], [ 125, 146, -5/36*E(3)^2 ], 
  [ 125, 147, -5/36*E(3)^2 ], [ 125, 149, 5/18*E(3)^2 ], 
  [ 125, 150, -1/9*E(3)^2 ], [ 125, 151, -5/108*E(3)-5/54*E(3)^2 ], 
  [ 125, 152, -1/9*E(3)^2 ], [ 125, 153, 5/12*E(3)+5/6*E(3)^2 ], 
  [ 125, 156, 5/108*E(3)+5/54*E(3)^2 ], [ 125, 157, 1/27*E(3)+2/27*E(3)^2 ], 
  [ 125, 158, 5/36 ], [ 125, 160, 5/36 ], [ 125, 162, 5/36 ], 
  [ 125, 163, 5/36 ], [ 125, 164, 2/9 ], [ 125, 165, 5/54*E(3)+5/108*E(3)^2 ],
  [ 125, 167, -5/54*E(3)-5/108*E(3)^2 ], [ 125, 170, 5/6*E(3)+5/12*E(3)^2 ], 
  [ 125, 171, -5/18*E(3) ], [ 125, 174, -5/36*E(3) ], 
  [ 125, 176, -5/36*E(3) ], [ 125, 180, 5/108*E(3)-5/108*E(3)^2 ], 
  [ 125, 185, 5/18*E(3)^2 ], [ 126, 6, 1/9*E(3)^2 ], 
  [ 126, 8, -1/9*E(3)-2/9*E(3)^2 ], [ 126, 15, -1/9*E(3)^2 ], 
  [ 126, 22, -1/9*E(3) ], [ 126, 126, -2/9*E(3) ], 
  [ 126, 127, -2/9*E(3)-1/9*E(3)^2 ], [ 126, 134, -2/9*E(3)+2/9*E(3)^2 ], 
  [ 126, 135, -1/9*E(3) ], [ 126, 142, 2/9*E(3)^2 ], [ 126, 245, -2/9 ], 
  [ 126, 246, 1/3 ], [ 126, 247, -1/9 ], [ 126, 248, 1/9 ], 
  [ 127, 7, -3/4*E(3) ], [ 127, 15, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 127, 126, -5/12*E(3)-5/6*E(3)^2 ], [ 127, 128, -3/4*E(3)^2 ], 
  [ 127, 135, -1/6*E(3)-1/3*E(3)^2 ], [ 127, 142, 5/6*E(3)+5/12*E(3)^2 ], 
  [ 127, 246, -1/6*E(3)+1/6*E(3)^2 ], [ 127, 247, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 127, 248, 1/6*E(3)-1/6*E(3)^2 ], [ 128, 6, 2/9*E(3)+1/9*E(3)^2 ], 
  [ 128, 8, -1/3*E(3) ], [ 128, 15, 2/9*E(3)+1/9*E(3)^2 ], 
  [ 128, 22, -1/9*E(3)-2/9*E(3)^2 ], [ 128, 127, -1/3*E(3)^2 ], 
  [ 128, 135, 1/9*E(3)+2/9*E(3)^2 ], [ 128, 246, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 128, 247, 1/9*E(3)-1/9*E(3)^2 ], [ 128, 248, 1/9*E(3)-1/9*E(3)^2 ], 
  [ 129, 1, 1/6*E(3)+1/3*E(3)^2 ], [ 129, 4, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 129, 9, 1/9*E(3)^2 ], [ 129, 10, -1/18*E(3)^2 ], 
  [ 129, 16, 1/6*E(3)-1/6*E(3)^2 ], [ 129, 23, 1/18*E(3) ], 
  [ 129, 122, 1/3*E(3)+1/6*E(3)^2 ], [ 129, 123, 1/3*E(3)+1/6*E(3)^2 ], 
  [ 129, 129, -1/18*E(3) ], [ 129, 130, 1/9*E(3) ], 
  [ 129, 137, 1/6*E(3)-1/6*E(3)^2 ], [ 129, 143, 1/18*E(3)^2 ], 
  [ 129, 241, -1/18 ], [ 129, 242, 1/18 ], [ 129, 243, -1/9 ], 
  [ 129, 244, 1/9 ], [ 129, 245, -1/18 ], [ 130, 2, -2/9*E(3)-4/9*E(3)^2 ], 
  [ 130, 3, -1/18*E(3)-1/9*E(3)^2 ], [ 130, 9, -1/18*E(3)^2 ], 
  [ 130, 10, 16/9*E(3)^2 ], [ 130, 17, 2/9*E(3)-2/9*E(3)^2 ], 
  [ 130, 23, 2/9*E(3) ], [ 130, 121, -4/9*E(3)-2/9*E(3)^2 ], 
  [ 130, 124, 1/9*E(3)+1/18*E(3)^2 ], [ 130, 129, 16/9*E(3) ], 
  [ 130, 130, -1/18*E(3) ], [ 130, 136, 2/9*E(3)-2/9*E(3)^2 ], 
  [ 130, 143, 2/9*E(3)^2 ], [ 130, 241, -2/9 ], [ 130, 242, 2/9 ], 
  [ 130, 243, -4/9 ], [ 130, 244, 4/9 ], [ 130, 245, -2/9 ], 
  [ 131, 1, -1/4*E(3)-1/2*E(3)^2 ], [ 131, 2, 1/12*E(3)+1/6*E(3)^2 ], 
  [ 131, 11, 1/4*E(3)^2 ], [ 131, 16, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 131, 17, 1/12*E(3)-1/12*E(3)^2 ], [ 131, 23, -1/4*E(3) ], 
  [ 131, 121, 1/6*E(3)+1/12*E(3)^2 ], [ 131, 122, -1/2*E(3)-1/4*E(3)^2 ], 
  [ 131, 131, 1/4*E(3) ], [ 131, 136, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 131, 137, 1/4*E(3)-1/4*E(3)^2 ], [ 131, 143, -1/4*E(3)^2 ], 
  [ 131, 241, -1/2 ], [ 131, 242, 1/2 ], [ 131, 243, 1/4 ], 
  [ 131, 244, -1/4 ], [ 132, 125, -4/9*E(3)+4/9*E(3)^2 ], 
  [ 132, 132, 1/3*E(3)^2 ], [ 132, 133, 5/9*E(3)+10/9*E(3)^2 ], 
  [ 132, 138, 1/9*E(3)+2/9*E(3)^2 ], [ 132, 139, -4/9*E(3)-8/9*E(3)^2 ], 
  [ 132, 140, -5/12 ], [ 132, 146, 5/18*E(3)+5/36*E(3)^2 ], 
  [ 132, 147, -10/9*E(3)-5/9*E(3)^2 ], [ 132, 149, 10/9*E(3)+5/9*E(3)^2 ], 
  [ 132, 150, 8/9*E(3)+4/9*E(3)^2 ], [ 132, 152, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 132, 156, 5/12*E(3) ], [ 132, 157, -1/3*E(3) ], 
  [ 132, 158, 5/9*E(3)-5/9*E(3)^2 ], [ 132, 160, -5/36*E(3)+5/36*E(3)^2 ], 
  [ 132, 162, 5/36*E(3)-5/36*E(3)^2 ], [ 132, 163, -5/9*E(3)+5/9*E(3)^2 ], 
  [ 132, 164, 4/9*E(3)-4/9*E(3)^2 ], [ 132, 165, 5/12*E(3)^2 ], 
  [ 132, 171, -5/9*E(3)-10/9*E(3)^2 ], [ 132, 174, -5/9*E(3)-10/9*E(3)^2 ], 
  [ 132, 176, 5/36*E(3)+5/18*E(3)^2 ], [ 132, 180, -5/12 ], 
  [ 132, 185, -10/9*E(3)-5/9*E(3)^2 ], [ 133, 125, -1/9 ], 
  [ 133, 132, 1/27*E(3)+1/54*E(3)^2 ], [ 133, 133, -1/9*E(3) ], 
  [ 133, 138, -1/18*E(3) ], [ 133, 139, -1/18*E(3) ], 
  [ 133, 140, -1/54*E(3)+1/54*E(3)^2 ], [ 133, 141, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 133, 144, -1/54*E(3)+1/54*E(3)^2 ], [ 133, 145, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 133, 146, -1/18*E(3)^2 ], [ 133, 147, -1/18*E(3)^2 ], 
  [ 133, 148, 1/18*E(3)^2 ], [ 133, 149, 1/9*E(3)^2 ], 
  [ 133, 150, 1/18*E(3)^2 ], [ 133, 151, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 133, 152, 1/18*E(3)^2 ], [ 133, 153, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 133, 154, -1/18*E(3)-1/9*E(3)^2 ], [ 133, 155, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 133, 156, 1/54*E(3)+1/27*E(3)^2 ], [ 133, 157, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 133, 158, 1/18 ], [ 133, 159, 1/18 ], [ 133, 160, 1/18 ], 
  [ 133, 161, -1/2 ], [ 133, 162, 1/18 ], [ 133, 163, 1/18 ], 
  [ 133, 164, -1/9 ], [ 133, 165, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 133, 166, -1/9*E(3)-1/18*E(3)^2 ], [ 133, 167, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 133, 169, -1/9*E(3)-1/18*E(3)^2 ], [ 133, 170, 1/3*E(3)+1/6*E(3)^2 ], 
  [ 133, 171, -1/9*E(3) ], [ 133, 173, 1/18*E(3) ], [ 133, 174, -1/18*E(3) ], 
  [ 133, 176, -1/18*E(3) ], [ 133, 178, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 133, 180, 1/54*E(3)-1/54*E(3)^2 ], [ 133, 185, 1/9*E(3)^2 ], 
  [ 134, 15, -1/3*E(3)-1/6*E(3)^2 ], [ 134, 126, 1/3*E(3)+2/3*E(3)^2 ], 
  [ 134, 135, -1/6*E(3)-1/3*E(3)^2 ], [ 134, 142, -2/3*E(3)-1/3*E(3)^2 ], 
  [ 134, 246, -1/6*E(3)+1/6*E(3)^2 ], [ 134, 247, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 134, 248, 1/6*E(3)-1/6*E(3)^2 ], [ 135, 6, -1/9*E(3)^2 ], 
  [ 135, 7, 1/6*E(3)+1/3*E(3)^2 ], [ 135, 8, -1/9*E(3)-2/9*E(3)^2 ], 
  [ 135, 14, 1/6*E(3)-1/6*E(3)^2 ], [ 135, 15, 1/9*E(3)^2 ], 
  [ 135, 22, 1/9*E(3) ], [ 135, 126, -5/18*E(3) ], 
  [ 135, 127, -2/9*E(3)-1/9*E(3)^2 ], [ 135, 128, 1/3*E(3)+1/6*E(3)^2 ], 
  [ 135, 134, 5/18*E(3)-5/18*E(3)^2 ], [ 135, 135, 1/9*E(3) ], 
  [ 135, 142, 5/18*E(3)^2 ], [ 135, 245, -5/18 ], [ 135, 246, 1/6 ], 
  [ 135, 247, 11/18 ], [ 135, 248, -11/18 ], [ 136, 1, -3/4*E(3) ], 
  [ 136, 2, 1/12*E(3) ], [ 136, 3, 1/6*E(3) ], 
  [ 136, 9, -1/3*E(3)-1/6*E(3)^2 ], [ 136, 11, -1/2*E(3)-1/4*E(3)^2 ], 
  [ 136, 16, -3/4 ], [ 136, 17, 1/12 ], [ 136, 23, 1/12*E(3)+1/6*E(3)^2 ], 
  [ 136, 121, 1/12*E(3)^2 ], [ 136, 122, -3/4*E(3)^2 ], 
  [ 136, 124, -1/6*E(3)^2 ], [ 136, 130, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 136, 131, -1/4*E(3)-1/2*E(3)^2 ], [ 136, 136, -1/12 ], [ 136, 137, 3/4 ], 
  [ 136, 143, 1/6*E(3)+1/12*E(3)^2 ], [ 136, 241, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 136, 242, 1/6*E(3)-1/6*E(3)^2 ], [ 136, 243, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 136, 244, -1/4*E(3)+1/4*E(3)^2 ], [ 136, 245, 1/3*E(3)-1/3*E(3)^2 ], 
  [ 137, 1, -1/12*E(3) ], [ 137, 2, 1/12*E(3) ], [ 137, 4, 2/3*E(3) ], 
  [ 137, 10, -4/9*E(3)-2/9*E(3)^2 ], [ 137, 11, -1/6*E(3)-1/12*E(3)^2 ], 
  [ 137, 16, 1/12 ], [ 137, 17, -1/12 ], [ 137, 23, 1/36*E(3)+1/18*E(3)^2 ], 
  [ 137, 121, 1/12*E(3)^2 ], [ 137, 122, -1/12*E(3)^2 ], 
  [ 137, 123, -2/3*E(3)^2 ], [ 137, 129, -2/9*E(3)-4/9*E(3)^2 ], 
  [ 137, 131, -1/12*E(3)-1/6*E(3)^2 ], [ 137, 136, 1/12 ], 
  [ 137, 137, -1/12 ], [ 137, 143, 1/18*E(3)+1/36*E(3)^2 ], 
  [ 137, 241, -1/18*E(3)+1/18*E(3)^2 ], [ 137, 242, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 137, 243, 1/12*E(3)-1/12*E(3)^2 ], [ 137, 244, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 137, 245, -1/9*E(3)+1/9*E(3)^2 ], [ 138, 125, 4/9 ], 
  [ 138, 132, 2/27*E(3)+1/27*E(3)^2 ], [ 138, 133, -5/9*E(3) ], 
  [ 138, 138, -1/9*E(3) ], [ 138, 139, -4/9*E(3) ], 
  [ 138, 140, 5/108*E(3)-5/108*E(3)^2 ], [ 138, 144, 8/27*E(3)-8/27*E(3)^2 ], 
  [ 138, 146, 5/36*E(3)^2 ], [ 138, 147, 5/9*E(3)^2 ], 
  [ 138, 149, 5/9*E(3)^2 ], [ 138, 150, 4/9*E(3)^2 ], 
  [ 138, 151, -10/27*E(3)-20/27*E(3)^2 ], [ 138, 152, 1/9*E(3)^2 ], 
  [ 138, 156, -5/108*E(3)-5/54*E(3)^2 ], [ 138, 157, -1/27*E(3)-2/27*E(3)^2 ],
  [ 138, 158, -5/9 ], [ 138, 160, -5/36 ], [ 138, 162, -5/36 ], 
  [ 138, 163, -5/9 ], [ 138, 164, 4/9 ], [ 138, 165, -5/54*E(3)-5/108*E(3)^2 ]
    , [ 138, 167, -20/27*E(3)-10/27*E(3)^2 ], [ 138, 171, -5/9*E(3) ], 
  [ 138, 174, 5/9*E(3) ], [ 138, 176, 5/36*E(3) ], 
  [ 138, 180, -5/108*E(3)+5/108*E(3)^2 ], [ 138, 185, 5/9*E(3)^2 ], 
  [ 139, 125, 1/9 ], [ 139, 132, -2/27*E(3)-1/27*E(3)^2 ], 
  [ 139, 133, -5/36*E(3) ], [ 139, 138, -1/9*E(3) ], [ 139, 139, 2/9*E(3) ], 
  [ 139, 140, -5/108*E(3)+5/108*E(3)^2 ], [ 139, 144, 1/27*E(3)-1/27*E(3)^2 ],
  [ 139, 145, 1/3*E(3)-1/3*E(3)^2 ], [ 139, 146, 5/36*E(3)^2 ], 
  [ 139, 147, -5/18*E(3)^2 ], [ 139, 149, 5/36*E(3)^2 ], 
  [ 139, 150, -2/9*E(3)^2 ], [ 139, 151, -5/108*E(3)-5/54*E(3)^2 ], 
  [ 139, 152, 1/9*E(3)^2 ], [ 139, 153, -5/12*E(3)-5/6*E(3)^2 ], 
  [ 139, 156, 5/108*E(3)+5/54*E(3)^2 ], [ 139, 157, 1/27*E(3)+2/27*E(3)^2 ], 
  [ 139, 158, 5/18 ], [ 139, 160, -5/36 ], [ 139, 162, -5/36 ], 
  [ 139, 163, 5/18 ], [ 139, 164, 1/9 ], [ 139, 165, 5/54*E(3)+5/108*E(3)^2 ],
  [ 139, 167, -5/54*E(3)-5/108*E(3)^2 ], [ 139, 170, -5/6*E(3)-5/12*E(3)^2 ], 
  [ 139, 171, -5/36*E(3) ], [ 139, 174, -5/18*E(3) ], [ 139, 176, 5/36*E(3) ],
  [ 139, 180, 5/108*E(3)-5/108*E(3)^2 ], [ 139, 185, 5/36*E(3)^2 ], 
  [ 140, 125, 2/9*E(3)-2/9*E(3)^2 ], [ 140, 132, -1/6*E(3)^2 ], 
  [ 140, 133, 2/9*E(3)+4/9*E(3)^2 ], [ 140, 138, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 140, 139, 2/9*E(3)+4/9*E(3)^2 ], [ 140, 140, -1/6 ], [ 140, 141, -2/3 ], 
  [ 140, 146, 1/9*E(3)+1/18*E(3)^2 ], [ 140, 147, -4/9*E(3)-2/9*E(3)^2 ], 
  [ 140, 148, 1/3*E(3)+1/6*E(3)^2 ], [ 140, 149, 4/9*E(3)+2/9*E(3)^2 ], 
  [ 140, 150, -4/9*E(3)-2/9*E(3)^2 ], [ 140, 152, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 140, 154, 1/6*E(3) ], [ 140, 155, -2/3*E(3) ], [ 140, 156, 1/6*E(3) ], 
  [ 140, 157, 1/6*E(3) ], [ 140, 158, 2/9*E(3)-2/9*E(3)^2 ], 
  [ 140, 160, -1/18*E(3)+1/18*E(3)^2 ], [ 140, 162, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 140, 163, -2/9*E(3)+2/9*E(3)^2 ], [ 140, 164, -2/9*E(3)+2/9*E(3)^2 ], 
  [ 140, 165, 1/6*E(3)^2 ], [ 140, 166, -2/3*E(3)^2 ], 
  [ 140, 169, 1/6*E(3)^2 ], [ 140, 171, -2/9*E(3)-4/9*E(3)^2 ], 
  [ 140, 173, 1/6*E(3)+1/3*E(3)^2 ], [ 140, 174, -2/9*E(3)-4/9*E(3)^2 ], 
  [ 140, 176, 1/18*E(3)+1/9*E(3)^2 ], [ 140, 178, 2/3 ], [ 140, 180, -1/6 ], 
  [ 140, 185, -4/9*E(3)-2/9*E(3)^2 ], [ 141, 133, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 141, 140, -1/12 ], [ 141, 146, -1/6*E(3)-1/12*E(3)^2 ], 
  [ 141, 147, -1/6*E(3)-1/12*E(3)^2 ], [ 141, 149, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 141, 151, -1/12*E(3) ], [ 141, 153, 3/4*E(3) ], [ 141, 156, -1/12*E(3) ], 
  [ 141, 158, -1/12*E(3)+1/12*E(3)^2 ], [ 141, 160, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 141, 162, 1/12*E(3)-1/12*E(3)^2 ], [ 141, 163, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 141, 165, -1/12*E(3)^2 ], [ 141, 167, -1/12*E(3)^2 ], 
  [ 141, 170, 3/4*E(3)^2 ], [ 141, 171, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 141, 174, -1/12*E(3)-1/6*E(3)^2 ], [ 141, 176, -1/12*E(3)-1/6*E(3)^2 ], 
  [ 141, 180, -1/12 ], [ 141, 185, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 142, 6, -1/9*E(3)^2 ], [ 142, 8, -1/9*E(3)-2/9*E(3)^2 ], 
  [ 142, 15, 1/9*E(3)^2 ], [ 142, 22, 1/9*E(3) ], [ 142, 126, 2/9*E(3) ], 
  [ 142, 127, -2/9*E(3)-1/9*E(3)^2 ], [ 142, 134, -2/9*E(3)+2/9*E(3)^2 ], 
  [ 142, 135, 1/9*E(3) ], [ 142, 142, -2/9*E(3)^2 ], [ 142, 245, 2/9 ], 
  [ 142, 246, -1/3 ], [ 142, 247, 1/9 ], [ 142, 248, -1/9 ], 
  [ 143, 1, 1/12*E(3)+1/6*E(3)^2 ], [ 143, 2, -1/36*E(3)-1/18*E(3)^2 ], 
  [ 143, 3, 1/18*E(3)+1/9*E(3)^2 ], [ 143, 4, 2/3*E(3)+4/3*E(3)^2 ], 
  [ 143, 9, 1/18*E(3)^2 ], [ 143, 10, 2/9*E(3)^2 ], [ 143, 11, -1/4*E(3)^2 ], 
  [ 143, 16, 1/12*E(3)-1/12*E(3)^2 ], [ 143, 17, 1/36*E(3)-1/36*E(3)^2 ], 
  [ 143, 23, 1/36*E(3) ], [ 143, 121, -1/18*E(3)-1/36*E(3)^2 ], 
  [ 143, 122, 1/6*E(3)+1/12*E(3)^2 ], [ 143, 123, -4/3*E(3)-2/3*E(3)^2 ], 
  [ 143, 124, -1/9*E(3)-1/18*E(3)^2 ], [ 143, 129, 2/9*E(3) ], 
  [ 143, 130, 1/18*E(3) ], [ 143, 131, -1/4*E(3) ], 
  [ 143, 136, 1/36*E(3)-1/36*E(3)^2 ], [ 143, 137, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 143, 143, 1/36*E(3)^2 ], [ 143, 241, -5/18 ], [ 143, 242, 5/18 ], 
  [ 143, 243, 25/36 ], [ 143, 244, -25/36 ], [ 143, 245, 2/9 ], 
  [ 144, 125, -1/9*E(3)+1/9*E(3)^2 ], [ 144, 133, 5/36*E(3)+5/18*E(3)^2 ], 
  [ 144, 138, -2/9*E(3)-4/9*E(3)^2 ], [ 144, 139, -1/9*E(3)-2/9*E(3)^2 ], 
  [ 144, 146, -5/9*E(3)-5/18*E(3)^2 ], [ 144, 147, -5/18*E(3)-5/36*E(3)^2 ], 
  [ 144, 149, 5/18*E(3)+5/36*E(3)^2 ], [ 144, 150, 2/9*E(3)+1/9*E(3)^2 ], 
  [ 144, 152, 4/9*E(3)+2/9*E(3)^2 ], [ 144, 158, 5/36*E(3)-5/36*E(3)^2 ], 
  [ 144, 160, 5/18*E(3)-5/18*E(3)^2 ], [ 144, 162, -5/18*E(3)+5/18*E(3)^2 ], 
  [ 144, 163, -5/36*E(3)+5/36*E(3)^2 ], [ 144, 164, 1/9*E(3)-1/9*E(3)^2 ], 
  [ 144, 171, -5/36*E(3)-5/18*E(3)^2 ], [ 144, 174, -5/36*E(3)-5/18*E(3)^2 ], 
  [ 144, 176, -5/18*E(3)-5/9*E(3)^2 ], [ 144, 185, -5/18*E(3)-5/36*E(3)^2 ], 
  [ 145, 125, 1/9*E(3)-1/9*E(3)^2 ], [ 145, 133, -5/36*E(3)-5/18*E(3)^2 ], 
  [ 145, 139, -1/9*E(3)-2/9*E(3)^2 ], [ 145, 147, -5/18*E(3)-5/36*E(3)^2 ], 
  [ 145, 149, -5/18*E(3)-5/36*E(3)^2 ], [ 145, 150, 2/9*E(3)+1/9*E(3)^2 ], 
  [ 145, 158, 5/36*E(3)-5/36*E(3)^2 ], [ 145, 163, -5/36*E(3)+5/36*E(3)^2 ], 
  [ 145, 164, -1/9*E(3)+1/9*E(3)^2 ], [ 145, 171, 5/36*E(3)+5/18*E(3)^2 ], 
  [ 145, 174, -5/36*E(3)-5/18*E(3)^2 ], [ 145, 185, 5/18*E(3)+5/36*E(3)^2 ], 
  [ 146, 125, -2/9 ], [ 146, 132, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 146, 133, -2/9*E(3) ], [ 146, 138, 1/18*E(3) ], [ 146, 139, 2/9*E(3) ], 
  [ 146, 140, 1/54*E(3)-1/54*E(3)^2 ], [ 146, 141, -2/9*E(3)+2/9*E(3)^2 ], 
  [ 146, 144, -4/27*E(3)+4/27*E(3)^2 ], [ 146, 146, 1/18*E(3)^2 ], 
  [ 146, 147, 2/9*E(3)^2 ], [ 146, 148, -1/18*E(3)^2 ], 
  [ 146, 149, 2/9*E(3)^2 ], [ 146, 150, -2/9*E(3)^2 ], 
  [ 146, 151, -4/27*E(3)-8/27*E(3)^2 ], [ 146, 152, -1/18*E(3)^2 ], 
  [ 146, 154, 1/18*E(3)+1/9*E(3)^2 ], [ 146, 155, 2/9*E(3)+4/9*E(3)^2 ], 
  [ 146, 156, -1/54*E(3)-1/27*E(3)^2 ], [ 146, 157, 1/54*E(3)+1/27*E(3)^2 ], 
  [ 146, 158, -2/9 ], [ 146, 159, 4/9 ], [ 146, 160, -1/18 ], 
  [ 146, 162, -1/18 ], [ 146, 163, -2/9 ], [ 146, 164, -2/9 ], 
  [ 146, 165, -1/27*E(3)-1/54*E(3)^2 ], [ 146, 166, 4/9*E(3)+2/9*E(3)^2 ], 
  [ 146, 167, -8/27*E(3)-4/27*E(3)^2 ], [ 146, 169, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 146, 171, -2/9*E(3) ], [ 146, 173, -1/18*E(3) ], [ 146, 174, 2/9*E(3) ], 
  [ 146, 176, 1/18*E(3) ], [ 146, 178, -2/9*E(3)+2/9*E(3)^2 ], 
  [ 146, 180, -1/54*E(3)+1/54*E(3)^2 ], [ 146, 185, 2/9*E(3)^2 ], 
  [ 147, 125, -1/18 ], [ 147, 132, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 147, 133, -1/18*E(3) ], [ 147, 138, 1/18*E(3) ], [ 147, 139, -1/9*E(3) ], 
  [ 147, 140, -1/54*E(3)+1/54*E(3)^2 ], [ 147, 141, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 147, 144, -1/54*E(3)+1/54*E(3)^2 ], [ 147, 145, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 147, 146, 1/18*E(3)^2 ], [ 147, 147, -1/9*E(3)^2 ], 
  [ 147, 148, 1/18*E(3)^2 ], [ 147, 149, 1/18*E(3)^2 ], 
  [ 147, 150, 1/9*E(3)^2 ], [ 147, 151, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 147, 152, -1/18*E(3)^2 ], [ 147, 153, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 147, 154, 1/18*E(3)+1/9*E(3)^2 ], [ 147, 155, -1/9*E(3)-2/9*E(3)^2 ], 
  [ 147, 156, 1/54*E(3)+1/27*E(3)^2 ], [ 147, 157, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 147, 158, 1/9 ], [ 147, 159, 1/18 ], [ 147, 160, -1/18 ], 
  [ 147, 161, 1/2 ], [ 147, 162, -1/18 ], [ 147, 163, 1/9 ], 
  [ 147, 164, -1/18 ], [ 147, 165, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 147, 166, -2/9*E(3)-1/9*E(3)^2 ], [ 147, 167, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 147, 169, 1/9*E(3)+1/18*E(3)^2 ], [ 147, 170, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 147, 171, -1/18*E(3) ], [ 147, 173, 1/18*E(3) ], [ 147, 174, -1/9*E(3) ], 
  [ 147, 176, 1/18*E(3) ], [ 147, 178, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 147, 180, 1/54*E(3)-1/54*E(3)^2 ], [ 147, 185, 1/18*E(3)^2 ], 
  [ 148, 133, E(3) ], [ 148, 140, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 148, 146, -1/4*E(3)^2 ], [ 148, 147, E(3)^2 ], [ 148, 149, E(3)^2 ], 
  [ 148, 156, 1/4*E(3)+1/2*E(3)^2 ], [ 148, 158, 1 ], [ 148, 160, -1/4 ], 
  [ 148, 162, -1/4 ], [ 148, 163, 1 ], [ 148, 165, 1/2*E(3)+1/4*E(3)^2 ], 
  [ 148, 171, -E(3) ], [ 148, 174, E(3) ], [ 148, 176, -1/4*E(3) ], 
  [ 148, 180, -1/4*E(3)+1/4*E(3)^2 ], [ 148, 185, -E(3)^2 ], 
  [ 149, 125, 1/9 ], [ 149, 132, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 149, 133, 1/9*E(3) ], [ 149, 138, 1/18*E(3) ], [ 149, 139, 1/18*E(3) ], 
  [ 149, 140, 1/54*E(3)-1/54*E(3)^2 ], [ 149, 141, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 149, 144, 1/54*E(3)-1/54*E(3)^2 ], [ 149, 145, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 149, 146, 1/18*E(3)^2 ], [ 149, 147, 1/18*E(3)^2 ], 
  [ 149, 148, 1/18*E(3)^2 ], [ 149, 149, -1/9*E(3)^2 ], 
  [ 149, 150, -1/18*E(3)^2 ], [ 149, 151, 1/54*E(3)+1/27*E(3)^2 ], 
  [ 149, 152, -1/18*E(3)^2 ], [ 149, 153, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 149, 154, -1/18*E(3)-1/9*E(3)^2 ], [ 149, 155, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 149, 156, -1/54*E(3)-1/27*E(3)^2 ], [ 149, 157, 1/54*E(3)+1/27*E(3)^2 ], 
  [ 149, 158, -1/18 ], [ 149, 159, 1/18 ], [ 149, 160, -1/18 ], 
  [ 149, 161, -1/2 ], [ 149, 162, -1/18 ], [ 149, 163, -1/18 ], 
  [ 149, 164, 1/9 ], [ 149, 165, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 149, 166, -1/9*E(3)-1/18*E(3)^2 ], [ 149, 167, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 149, 169, -1/9*E(3)-1/18*E(3)^2 ], [ 149, 170, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 149, 171, 1/9*E(3) ], [ 149, 173, 1/18*E(3) ], [ 149, 174, 1/18*E(3) ], 
  [ 149, 176, 1/18*E(3) ], [ 149, 178, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 149, 180, -1/54*E(3)+1/54*E(3)^2 ], [ 149, 185, -1/9*E(3)^2 ], 
  [ 150, 125, -1/9 ], [ 150, 132, -2/27*E(3)-1/27*E(3)^2 ], 
  [ 150, 133, 5/36*E(3) ], [ 150, 138, 1/9*E(3) ], [ 150, 139, -2/9*E(3) ], 
  [ 150, 140, -5/108*E(3)+5/108*E(3)^2 ], [ 150, 144, 1/27*E(3)-1/27*E(3)^2 ],
  [ 150, 145, 1/3*E(3)-1/3*E(3)^2 ], [ 150, 146, -5/36*E(3)^2 ], 
  [ 150, 147, 5/18*E(3)^2 ], [ 150, 149, -5/36*E(3)^2 ], 
  [ 150, 150, 2/9*E(3)^2 ], [ 150, 151, -5/108*E(3)-5/54*E(3)^2 ], 
  [ 150, 152, -1/9*E(3)^2 ], [ 150, 153, -5/12*E(3)-5/6*E(3)^2 ], 
  [ 150, 156, 5/108*E(3)+5/54*E(3)^2 ], [ 150, 157, 1/27*E(3)+2/27*E(3)^2 ], 
  [ 150, 158, -5/18 ], [ 150, 160, 5/36 ], [ 150, 162, 5/36 ], 
  [ 150, 163, -5/18 ], [ 150, 164, -1/9 ], 
  [ 150, 165, 5/54*E(3)+5/108*E(3)^2 ], [ 150, 167, -5/54*E(3)-5/108*E(3)^2 ],
  [ 150, 170, -5/6*E(3)-5/12*E(3)^2 ], [ 150, 171, 5/36*E(3) ], 
  [ 150, 174, 5/18*E(3) ], [ 150, 176, -5/36*E(3) ], 
  [ 150, 180, 5/108*E(3)-5/108*E(3)^2 ], [ 150, 185, -5/36*E(3)^2 ], 
  [ 151, 125, 1/18*E(3)-1/18*E(3)^2 ], [ 151, 133, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 151, 138, 1/9*E(3)+2/9*E(3)^2 ], [ 151, 139, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 151, 141, -1/6 ], [ 151, 146, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 151, 147, -1/9*E(3)-1/18*E(3)^2 ], [ 151, 149, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 151, 150, -1/9*E(3)-1/18*E(3)^2 ], [ 151, 152, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 151, 154, -1/3*E(3) ], [ 151, 155, -1/6*E(3) ], 
  [ 151, 158, 1/18*E(3)-1/18*E(3)^2 ], [ 151, 160, 1/9*E(3)-1/9*E(3)^2 ], 
  [ 151, 162, -1/9*E(3)+1/9*E(3)^2 ], [ 151, 163, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 151, 164, -1/18*E(3)+1/18*E(3)^2 ], [ 151, 166, -1/6*E(3)^2 ], 
  [ 151, 169, -1/3*E(3)^2 ], [ 151, 171, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 151, 174, -1/18*E(3)-1/9*E(3)^2 ], [ 151, 176, -1/9*E(3)-2/9*E(3)^2 ], 
  [ 151, 178, 1/6 ], [ 151, 185, -1/9*E(3)-1/18*E(3)^2 ], [ 152, 125, -4/9 ], 
  [ 152, 132, 2/27*E(3)+1/27*E(3)^2 ], [ 152, 133, 5/9*E(3) ], 
  [ 152, 138, 1/9*E(3) ], [ 152, 139, 4/9*E(3) ], 
  [ 152, 140, 5/108*E(3)-5/108*E(3)^2 ], [ 152, 144, 8/27*E(3)-8/27*E(3)^2 ], 
  [ 152, 146, -5/36*E(3)^2 ], [ 152, 147, -5/9*E(3)^2 ], 
  [ 152, 149, -5/9*E(3)^2 ], [ 152, 150, -4/9*E(3)^2 ], 
  [ 152, 151, -10/27*E(3)-20/27*E(3)^2 ], [ 152, 152, -1/9*E(3)^2 ], 
  [ 152, 156, -5/108*E(3)-5/54*E(3)^2 ], [ 152, 157, -1/27*E(3)-2/27*E(3)^2 ],
  [ 152, 158, 5/9 ], [ 152, 160, 5/36 ], [ 152, 162, 5/36 ], 
  [ 152, 163, 5/9 ], [ 152, 164, -4/9 ], [ 152, 165, -5/54*E(3)-5/108*E(3)^2 ]
    , [ 152, 167, -20/27*E(3)-10/27*E(3)^2 ], [ 152, 171, 5/9*E(3) ], 
  [ 152, 174, -5/9*E(3) ], [ 152, 176, -5/36*E(3) ], 
  [ 152, 180, -5/108*E(3)+5/108*E(3)^2 ], [ 152, 185, -5/9*E(3)^2 ], 
  [ 153, 125, -1/18*E(3)+1/18*E(3)^2 ], [ 153, 133, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 153, 139, 1/18*E(3)+1/9*E(3)^2 ], [ 153, 141, 1/6 ], 
  [ 153, 147, -1/9*E(3)-1/18*E(3)^2 ], [ 153, 149, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 153, 150, -1/9*E(3)-1/18*E(3)^2 ], [ 153, 155, -1/6*E(3) ], 
  [ 153, 158, 1/18*E(3)-1/18*E(3)^2 ], [ 153, 163, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 153, 164, 1/18*E(3)-1/18*E(3)^2 ], [ 153, 166, -1/6*E(3)^2 ], 
  [ 153, 171, 1/18*E(3)+1/9*E(3)^2 ], [ 153, 174, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 153, 178, -1/6 ], [ 153, 185, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 154, 133, 1/3*E(3)+2/3*E(3)^2 ], [ 154, 140, 1/12 ], 
  [ 154, 146, 1/6*E(3)+1/12*E(3)^2 ], [ 154, 147, 2/3*E(3)+1/3*E(3)^2 ], 
  [ 154, 149, -2/3*E(3)-1/3*E(3)^2 ], [ 154, 151, -2/3*E(3) ], 
  [ 154, 156, 1/12*E(3) ], [ 154, 158, 1/3*E(3)-1/3*E(3)^2 ], 
  [ 154, 160, 1/12*E(3)-1/12*E(3)^2 ], [ 154, 162, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 154, 163, -1/3*E(3)+1/3*E(3)^2 ], [ 154, 165, 1/12*E(3)^2 ], 
  [ 154, 167, -2/3*E(3)^2 ], [ 154, 171, 1/3*E(3)+2/3*E(3)^2 ], 
  [ 154, 174, 1/3*E(3)+2/3*E(3)^2 ], [ 154, 176, 1/12*E(3)+1/6*E(3)^2 ], 
  [ 154, 180, 1/12 ], [ 154, 185, -2/3*E(3)-1/3*E(3)^2 ], 
  [ 155, 133, 1/12*E(3)+1/6*E(3)^2 ], [ 155, 140, -1/12 ], 
  [ 155, 146, 1/6*E(3)+1/12*E(3)^2 ], [ 155, 147, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 155, 149, -1/6*E(3)-1/12*E(3)^2 ], [ 155, 151, -1/12*E(3) ], 
  [ 155, 153, -3/4*E(3) ], [ 155, 156, -1/12*E(3) ], 
  [ 155, 158, -1/6*E(3)+1/6*E(3)^2 ], [ 155, 160, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 155, 162, -1/12*E(3)+1/12*E(3)^2 ], [ 155, 163, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 155, 165, -1/12*E(3)^2 ], [ 155, 167, -1/12*E(3)^2 ], 
  [ 155, 170, -3/4*E(3)^2 ], [ 155, 171, 1/12*E(3)+1/6*E(3)^2 ], 
  [ 155, 174, -1/6*E(3)-1/3*E(3)^2 ], [ 155, 176, 1/12*E(3)+1/6*E(3)^2 ], 
  [ 155, 180, -1/12 ], [ 155, 185, -1/6*E(3)-1/12*E(3)^2 ], 
  [ 156, 125, -2/9*E(3)+2/9*E(3)^2 ], [ 156, 132, 1/6*E(3)^2 ], 
  [ 156, 133, -2/9*E(3)-4/9*E(3)^2 ], [ 156, 138, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 156, 139, -2/9*E(3)-4/9*E(3)^2 ], [ 156, 140, 1/6 ], [ 156, 141, -2/3 ], 
  [ 156, 146, -1/9*E(3)-1/18*E(3)^2 ], [ 156, 147, 4/9*E(3)+2/9*E(3)^2 ], 
  [ 156, 148, 1/3*E(3)+1/6*E(3)^2 ], [ 156, 149, -4/9*E(3)-2/9*E(3)^2 ], 
  [ 156, 150, 4/9*E(3)+2/9*E(3)^2 ], [ 156, 152, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 156, 154, 1/6*E(3) ], [ 156, 155, -2/3*E(3) ], [ 156, 156, -1/6*E(3) ], 
  [ 156, 157, -1/6*E(3) ], [ 156, 158, -2/9*E(3)+2/9*E(3)^2 ], 
  [ 156, 160, 1/18*E(3)-1/18*E(3)^2 ], [ 156, 162, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 156, 163, 2/9*E(3)-2/9*E(3)^2 ], [ 156, 164, 2/9*E(3)-2/9*E(3)^2 ], 
  [ 156, 165, -1/6*E(3)^2 ], [ 156, 166, -2/3*E(3)^2 ], 
  [ 156, 169, 1/6*E(3)^2 ], [ 156, 171, 2/9*E(3)+4/9*E(3)^2 ], 
  [ 156, 173, 1/6*E(3)+1/3*E(3)^2 ], [ 156, 174, 2/9*E(3)+4/9*E(3)^2 ], 
  [ 156, 176, -1/18*E(3)-1/9*E(3)^2 ], [ 156, 178, 2/3 ], [ 156, 180, 1/6 ], 
  [ 156, 185, 4/9*E(3)+2/9*E(3)^2 ], [ 157, 125, -4/9*E(3)+4/9*E(3)^2 ], 
  [ 157, 132, -1/3*E(3)^2 ], [ 157, 133, 5/9*E(3)+10/9*E(3)^2 ], 
  [ 157, 138, 1/9*E(3)+2/9*E(3)^2 ], [ 157, 139, -4/9*E(3)-8/9*E(3)^2 ], 
  [ 157, 140, 5/12 ], [ 157, 146, 5/18*E(3)+5/36*E(3)^2 ], 
  [ 157, 147, -10/9*E(3)-5/9*E(3)^2 ], [ 157, 149, 10/9*E(3)+5/9*E(3)^2 ], 
  [ 157, 150, 8/9*E(3)+4/9*E(3)^2 ], [ 157, 152, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 157, 156, -5/12*E(3) ], [ 157, 157, 1/3*E(3) ], 
  [ 157, 158, 5/9*E(3)-5/9*E(3)^2 ], [ 157, 160, -5/36*E(3)+5/36*E(3)^2 ], 
  [ 157, 162, 5/36*E(3)-5/36*E(3)^2 ], [ 157, 163, -5/9*E(3)+5/9*E(3)^2 ], 
  [ 157, 164, 4/9*E(3)-4/9*E(3)^2 ], [ 157, 165, -5/12*E(3)^2 ], 
  [ 157, 171, -5/9*E(3)-10/9*E(3)^2 ], [ 157, 174, -5/9*E(3)-10/9*E(3)^2 ], 
  [ 157, 176, 5/36*E(3)+5/18*E(3)^2 ], [ 157, 180, 5/12 ], 
  [ 157, 185, -10/9*E(3)-5/9*E(3)^2 ], [ 158, 125, 1/18 ], 
  [ 158, 132, 1/27*E(3)+1/54*E(3)^2 ], [ 158, 133, 1/18*E(3) ], 
  [ 158, 138, -1/18*E(3) ], [ 158, 139, 1/9*E(3) ], 
  [ 158, 140, -1/54*E(3)+1/54*E(3)^2 ], [ 158, 141, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 158, 144, -1/54*E(3)+1/54*E(3)^2 ], [ 158, 145, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 158, 146, -1/18*E(3)^2 ], [ 158, 147, 1/9*E(3)^2 ], 
  [ 158, 148, 1/18*E(3)^2 ], [ 158, 149, -1/18*E(3)^2 ], 
  [ 158, 150, -1/9*E(3)^2 ], [ 158, 151, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 158, 152, 1/18*E(3)^2 ], [ 158, 153, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 158, 154, -1/18*E(3)-1/9*E(3)^2 ], [ 158, 155, 1/9*E(3)+2/9*E(3)^2 ], 
  [ 158, 156, 1/54*E(3)+1/27*E(3)^2 ], [ 158, 157, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 158, 158, -1/9 ], [ 158, 159, 1/18 ], [ 158, 160, 1/18 ], 
  [ 158, 161, 1/2 ], [ 158, 162, 1/18 ], [ 158, 163, -1/9 ], 
  [ 158, 164, 1/18 ], [ 158, 165, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 158, 166, 2/9*E(3)+1/9*E(3)^2 ], [ 158, 167, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 158, 169, -1/9*E(3)-1/18*E(3)^2 ], [ 158, 170, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 158, 171, 1/18*E(3) ], [ 158, 173, 1/18*E(3) ], [ 158, 174, 1/9*E(3) ], 
  [ 158, 176, -1/18*E(3) ], [ 158, 178, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 158, 180, 1/54*E(3)-1/54*E(3)^2 ], [ 158, 185, -1/18*E(3)^2 ], 
  [ 159, 133, 1/4*E(3) ], [ 159, 146, 1/2*E(3)^2 ], [ 159, 147, 1/4*E(3)^2 ], 
  [ 159, 149, 1/4*E(3)^2 ], [ 159, 158, 1/4 ], [ 159, 160, 1/2 ], 
  [ 159, 162, 1/2 ], [ 159, 163, 1/4 ], [ 159, 171, -1/4*E(3) ], 
  [ 159, 174, 1/4*E(3) ], [ 159, 176, 1/2*E(3) ], [ 159, 185, -1/4*E(3)^2 ], 
  [ 160, 125, 2/9 ], [ 160, 132, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 160, 133, 2/9*E(3) ], [ 160, 138, -1/18*E(3) ], [ 160, 139, -2/9*E(3) ], 
  [ 160, 140, 1/54*E(3)-1/54*E(3)^2 ], [ 160, 141, 2/9*E(3)-2/9*E(3)^2 ], 
  [ 160, 144, -4/27*E(3)+4/27*E(3)^2 ], [ 160, 146, -1/18*E(3)^2 ], 
  [ 160, 147, -2/9*E(3)^2 ], [ 160, 148, -1/18*E(3)^2 ], 
  [ 160, 149, -2/9*E(3)^2 ], [ 160, 150, 2/9*E(3)^2 ], 
  [ 160, 151, -4/27*E(3)-8/27*E(3)^2 ], [ 160, 152, 1/18*E(3)^2 ], 
  [ 160, 154, -1/18*E(3)-1/9*E(3)^2 ], [ 160, 155, -2/9*E(3)-4/9*E(3)^2 ], 
  [ 160, 156, -1/54*E(3)-1/27*E(3)^2 ], [ 160, 157, 1/54*E(3)+1/27*E(3)^2 ], 
  [ 160, 158, 2/9 ], [ 160, 159, 4/9 ], [ 160, 160, 1/18 ], 
  [ 160, 162, 1/18 ], [ 160, 163, 2/9 ], [ 160, 164, 2/9 ], 
  [ 160, 165, -1/27*E(3)-1/54*E(3)^2 ], [ 160, 166, -4/9*E(3)-2/9*E(3)^2 ], 
  [ 160, 167, -8/27*E(3)-4/27*E(3)^2 ], [ 160, 169, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 160, 171, 2/9*E(3) ], [ 160, 173, -1/18*E(3) ], [ 160, 174, -2/9*E(3) ], 
  [ 160, 176, -1/18*E(3) ], [ 160, 178, 2/9*E(3)-2/9*E(3)^2 ], 
  [ 160, 180, -1/54*E(3)+1/54*E(3)^2 ], [ 160, 185, -2/9*E(3)^2 ], 
  [ 161, 133, -1/4*E(3) ], [ 161, 147, 1/4*E(3)^2 ], 
  [ 161, 149, -1/4*E(3)^2 ], [ 161, 158, 1/4 ], [ 161, 163, 1/4 ], 
  [ 161, 171, 1/4*E(3) ], [ 161, 174, 1/4*E(3) ], [ 161, 185, 1/4*E(3)^2 ], 
  [ 162, 125, 2/9 ], [ 162, 132, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 162, 133, 2/9*E(3) ], [ 162, 138, -1/18*E(3) ], [ 162, 139, -2/9*E(3) ], 
  [ 162, 140, -1/54*E(3)+1/54*E(3)^2 ], [ 162, 141, -2/9*E(3)+2/9*E(3)^2 ], 
  [ 162, 144, 4/27*E(3)-4/27*E(3)^2 ], [ 162, 146, -1/18*E(3)^2 ], 
  [ 162, 147, -2/9*E(3)^2 ], [ 162, 148, -1/18*E(3)^2 ], 
  [ 162, 149, -2/9*E(3)^2 ], [ 162, 150, 2/9*E(3)^2 ], 
  [ 162, 151, 4/27*E(3)+8/27*E(3)^2 ], [ 162, 152, 1/18*E(3)^2 ], 
  [ 162, 154, 1/18*E(3)+1/9*E(3)^2 ], [ 162, 155, 2/9*E(3)+4/9*E(3)^2 ], 
  [ 162, 156, 1/54*E(3)+1/27*E(3)^2 ], [ 162, 157, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 162, 158, 2/9 ], [ 162, 159, 4/9 ], [ 162, 160, 1/18 ], 
  [ 162, 162, 1/18 ], [ 162, 163, 2/9 ], [ 162, 164, 2/9 ], 
  [ 162, 165, 1/27*E(3)+1/54*E(3)^2 ], [ 162, 166, 4/9*E(3)+2/9*E(3)^2 ], 
  [ 162, 167, 8/27*E(3)+4/27*E(3)^2 ], [ 162, 169, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 162, 171, 2/9*E(3) ], [ 162, 173, -1/18*E(3) ], [ 162, 174, -2/9*E(3) ], 
  [ 162, 176, -1/18*E(3) ], [ 162, 178, -2/9*E(3)+2/9*E(3)^2 ], 
  [ 162, 180, 1/54*E(3)-1/54*E(3)^2 ], [ 162, 185, -2/9*E(3)^2 ], 
  [ 163, 125, 1/18 ], [ 163, 132, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 163, 133, 1/18*E(3) ], [ 163, 138, -1/18*E(3) ], [ 163, 139, 1/9*E(3) ], 
  [ 163, 140, 1/54*E(3)-1/54*E(3)^2 ], [ 163, 141, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 163, 144, 1/54*E(3)-1/54*E(3)^2 ], [ 163, 145, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 163, 146, -1/18*E(3)^2 ], [ 163, 147, 1/9*E(3)^2 ], 
  [ 163, 148, 1/18*E(3)^2 ], [ 163, 149, -1/18*E(3)^2 ], 
  [ 163, 150, -1/9*E(3)^2 ], [ 163, 151, 1/54*E(3)+1/27*E(3)^2 ], 
  [ 163, 152, 1/18*E(3)^2 ], [ 163, 153, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 163, 154, 1/18*E(3)+1/9*E(3)^2 ], [ 163, 155, -1/9*E(3)-2/9*E(3)^2 ], 
  [ 163, 156, -1/54*E(3)-1/27*E(3)^2 ], [ 163, 157, 1/54*E(3)+1/27*E(3)^2 ], 
  [ 163, 158, -1/9 ], [ 163, 159, 1/18 ], [ 163, 160, 1/18 ], 
  [ 163, 161, 1/2 ], [ 163, 162, 1/18 ], [ 163, 163, -1/9 ], 
  [ 163, 164, 1/18 ], [ 163, 165, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 163, 166, -2/9*E(3)-1/9*E(3)^2 ], [ 163, 167, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 163, 169, 1/9*E(3)+1/18*E(3)^2 ], [ 163, 170, 1/3*E(3)+1/6*E(3)^2 ], 
  [ 163, 171, 1/18*E(3) ], [ 163, 173, 1/18*E(3) ], [ 163, 174, 1/9*E(3) ], 
  [ 163, 176, -1/18*E(3) ], [ 163, 178, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 163, 180, -1/54*E(3)+1/54*E(3)^2 ], [ 163, 185, -1/18*E(3)^2 ], 
  [ 164, 125, 2/9 ], [ 164, 132, 2/27*E(3)+1/27*E(3)^2 ], 
  [ 164, 133, -5/18*E(3) ], [ 164, 138, 1/9*E(3) ], [ 164, 139, 1/9*E(3) ], 
  [ 164, 140, 5/108*E(3)-5/108*E(3)^2 ], [ 164, 144, -1/27*E(3)+1/27*E(3)^2 ],
  [ 164, 145, 1/3*E(3)-1/3*E(3)^2 ], [ 164, 146, -5/36*E(3)^2 ], 
  [ 164, 147, -5/36*E(3)^2 ], [ 164, 149, 5/18*E(3)^2 ], 
  [ 164, 150, -1/9*E(3)^2 ], [ 164, 151, 5/108*E(3)+5/54*E(3)^2 ], 
  [ 164, 152, -1/9*E(3)^2 ], [ 164, 153, -5/12*E(3)-5/6*E(3)^2 ], 
  [ 164, 156, -5/108*E(3)-5/54*E(3)^2 ], [ 164, 157, -1/27*E(3)-2/27*E(3)^2 ],
  [ 164, 158, 5/36 ], [ 164, 160, 5/36 ], [ 164, 162, 5/36 ], 
  [ 164, 163, 5/36 ], [ 164, 164, 2/9 ], [ 164, 165, -5/54*E(3)-5/108*E(3)^2 ]
    , [ 164, 167, 5/54*E(3)+5/108*E(3)^2 ], 
  [ 164, 170, -5/6*E(3)-5/12*E(3)^2 ], [ 164, 171, -5/18*E(3) ], 
  [ 164, 174, -5/36*E(3) ], [ 164, 176, -5/36*E(3) ], 
  [ 164, 180, -5/108*E(3)+5/108*E(3)^2 ], [ 164, 185, 5/18*E(3)^2 ], 
  [ 165, 125, 2/9*E(3)-2/9*E(3)^2 ], [ 165, 132, 1/6*E(3)^2 ], 
  [ 165, 133, 2/9*E(3)+4/9*E(3)^2 ], [ 165, 138, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 165, 139, 2/9*E(3)+4/9*E(3)^2 ], [ 165, 140, 1/6 ], [ 165, 141, -2/3 ], 
  [ 165, 146, 1/9*E(3)+1/18*E(3)^2 ], [ 165, 147, -4/9*E(3)-2/9*E(3)^2 ], 
  [ 165, 148, -1/3*E(3)-1/6*E(3)^2 ], [ 165, 149, 4/9*E(3)+2/9*E(3)^2 ], 
  [ 165, 150, -4/9*E(3)-2/9*E(3)^2 ], [ 165, 152, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 165, 154, 1/6*E(3) ], [ 165, 155, -2/3*E(3) ], [ 165, 156, -1/6*E(3) ], 
  [ 165, 157, -1/6*E(3) ], [ 165, 158, 2/9*E(3)-2/9*E(3)^2 ], 
  [ 165, 160, -1/18*E(3)+1/18*E(3)^2 ], [ 165, 162, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 165, 163, -2/9*E(3)+2/9*E(3)^2 ], [ 165, 164, -2/9*E(3)+2/9*E(3)^2 ], 
  [ 165, 165, -1/6*E(3)^2 ], [ 165, 166, -2/3*E(3)^2 ], 
  [ 165, 169, 1/6*E(3)^2 ], [ 165, 171, -2/9*E(3)-4/9*E(3)^2 ], 
  [ 165, 173, -1/6*E(3)-1/3*E(3)^2 ], [ 165, 174, -2/9*E(3)-4/9*E(3)^2 ], 
  [ 165, 176, 1/18*E(3)+1/9*E(3)^2 ], [ 165, 178, 2/3 ], [ 165, 180, 1/6 ], 
  [ 165, 185, -4/9*E(3)-2/9*E(3)^2 ], [ 166, 133, -1/12*E(3)-1/6*E(3)^2 ], 
  [ 166, 140, -1/12 ], [ 166, 146, -1/6*E(3)-1/12*E(3)^2 ], 
  [ 166, 147, 1/3*E(3)+1/6*E(3)^2 ], [ 166, 149, 1/6*E(3)+1/12*E(3)^2 ], 
  [ 166, 151, -1/12*E(3) ], [ 166, 153, -3/4*E(3) ], [ 166, 156, -1/12*E(3) ],
  [ 166, 158, 1/6*E(3)-1/6*E(3)^2 ], [ 166, 160, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 166, 162, 1/12*E(3)-1/12*E(3)^2 ], [ 166, 163, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 166, 165, -1/12*E(3)^2 ], [ 166, 167, -1/12*E(3)^2 ], 
  [ 166, 170, -3/4*E(3)^2 ], [ 166, 171, -1/12*E(3)-1/6*E(3)^2 ], 
  [ 166, 174, 1/6*E(3)+1/3*E(3)^2 ], [ 166, 176, -1/12*E(3)-1/6*E(3)^2 ], 
  [ 166, 180, -1/12 ], [ 166, 185, 1/6*E(3)+1/12*E(3)^2 ], 
  [ 167, 125, -1/18*E(3)+1/18*E(3)^2 ], [ 167, 133, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 167, 138, -1/9*E(3)-2/9*E(3)^2 ], [ 167, 139, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 167, 141, -1/6 ], [ 167, 146, 2/9*E(3)+1/9*E(3)^2 ], 
  [ 167, 147, 1/9*E(3)+1/18*E(3)^2 ], [ 167, 149, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 167, 150, 1/9*E(3)+1/18*E(3)^2 ], [ 167, 152, 2/9*E(3)+1/9*E(3)^2 ], 
  [ 167, 154, -1/3*E(3) ], [ 167, 155, -1/6*E(3) ], 
  [ 167, 158, -1/18*E(3)+1/18*E(3)^2 ], [ 167, 160, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 167, 162, 1/9*E(3)-1/9*E(3)^2 ], [ 167, 163, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 167, 164, 1/18*E(3)-1/18*E(3)^2 ], [ 167, 166, -1/6*E(3)^2 ], 
  [ 167, 169, -1/3*E(3)^2 ], [ 167, 171, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 167, 174, 1/18*E(3)+1/9*E(3)^2 ], [ 167, 176, 1/9*E(3)+2/9*E(3)^2 ], 
  [ 167, 178, 1/6 ], [ 167, 185, 1/9*E(3)+1/18*E(3)^2 ], [ 168, 168, 1/12 ], 
  [ 168, 172, -1/6*E(3)-1/12*E(3)^2 ], [ 168, 175, -2/27*E(3)-1/27*E(3)^2 ], 
  [ 168, 177, 2/3*E(3) ], [ 168, 179, -1/9*E(3) ], 
  [ 168, 181, -5/108*E(3)+5/108*E(3)^2 ], [ 168, 182, -1/12*E(3) ], 
  [ 168, 183, -1/12*E(3)+1/12*E(3)^2 ], [ 168, 184, 8/27*E(3)-8/27*E(3)^2 ], 
  [ 168, 186, 5/36*E(3)^2 ], [ 168, 187, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 168, 189, -1/12*E(3)^2 ], [ 168, 190, 1/9*E(3)^2 ], 
  [ 168, 191, -10/27*E(3)-20/27*E(3)^2 ], [ 168, 192, -2/3*E(3)^2 ], 
  [ 168, 194, -5/108*E(3)-5/54*E(3)^2 ], [ 168, 195, -1/27*E(3)-2/27*E(3)^2 ],
  [ 168, 196, -5/36 ], [ 168, 197, 1/12*E(3)+1/6*E(3)^2 ], [ 168, 199, 5/36 ],
  [ 168, 200, -5/54*E(3)-5/108*E(3)^2 ], [ 168, 201, 1/12 ], 
  [ 168, 204, 20/27*E(3)+10/27*E(3)^2 ], [ 168, 208, -5/36*E(3) ], 
  [ 168, 211, 5/108*E(3)-5/108*E(3)^2 ], [ 169, 133, -1/3*E(3)-2/3*E(3)^2 ], 
  [ 169, 140, 1/12 ], [ 169, 146, -1/6*E(3)-1/12*E(3)^2 ], 
  [ 169, 147, -2/3*E(3)-1/3*E(3)^2 ], [ 169, 149, 2/3*E(3)+1/3*E(3)^2 ], 
  [ 169, 151, -2/3*E(3) ], [ 169, 156, 1/12*E(3) ], 
  [ 169, 158, -1/3*E(3)+1/3*E(3)^2 ], [ 169, 160, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 169, 162, 1/12*E(3)-1/12*E(3)^2 ], [ 169, 163, 1/3*E(3)-1/3*E(3)^2 ], 
  [ 169, 165, 1/12*E(3)^2 ], [ 169, 167, -2/3*E(3)^2 ], 
  [ 169, 171, -1/3*E(3)-2/3*E(3)^2 ], [ 169, 174, -1/3*E(3)-2/3*E(3)^2 ], 
  [ 169, 176, -1/12*E(3)-1/6*E(3)^2 ], [ 169, 180, 1/12 ], 
  [ 169, 185, 2/3*E(3)+1/3*E(3)^2 ], [ 170, 125, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 170, 133, 1/18*E(3)+1/9*E(3)^2 ], [ 170, 139, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 170, 141, 1/6 ], [ 170, 147, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 170, 149, 1/9*E(3)+1/18*E(3)^2 ], [ 170, 150, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 170, 155, -1/6*E(3) ], [ 170, 158, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 170, 163, 1/18*E(3)-1/18*E(3)^2 ], [ 170, 164, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 170, 166, -1/6*E(3)^2 ], [ 170, 171, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 170, 174, 1/18*E(3)+1/9*E(3)^2 ], [ 170, 178, -1/6 ], 
  [ 170, 185, -1/9*E(3)-1/18*E(3)^2 ], [ 171, 125, -1/9 ], 
  [ 171, 132, -1/27*E(3)-1/54*E(3)^2 ], [ 171, 133, -1/9*E(3) ], 
  [ 171, 138, -1/18*E(3) ], [ 171, 139, -1/18*E(3) ], 
  [ 171, 140, 1/54*E(3)-1/54*E(3)^2 ], [ 171, 141, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 171, 144, 1/54*E(3)-1/54*E(3)^2 ], [ 171, 145, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 171, 146, -1/18*E(3)^2 ], [ 171, 147, -1/18*E(3)^2 ], 
  [ 171, 148, -1/18*E(3)^2 ], [ 171, 149, 1/9*E(3)^2 ], 
  [ 171, 150, 1/18*E(3)^2 ], [ 171, 151, 1/54*E(3)+1/27*E(3)^2 ], 
  [ 171, 152, 1/18*E(3)^2 ], [ 171, 153, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 171, 154, -1/18*E(3)-1/9*E(3)^2 ], [ 171, 155, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 171, 156, -1/54*E(3)-1/27*E(3)^2 ], [ 171, 157, 1/54*E(3)+1/27*E(3)^2 ], 
  [ 171, 158, 1/18 ], [ 171, 159, -1/18 ], [ 171, 160, 1/18 ], 
  [ 171, 161, 1/2 ], [ 171, 162, 1/18 ], [ 171, 163, 1/18 ], 
  [ 171, 164, -1/9 ], [ 171, 165, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 171, 166, -1/9*E(3)-1/18*E(3)^2 ], [ 171, 167, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 171, 169, -1/9*E(3)-1/18*E(3)^2 ], [ 171, 170, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 171, 171, -1/9*E(3) ], [ 171, 173, -1/18*E(3) ], [ 171, 174, -1/18*E(3) ],
  [ 171, 176, -1/18*E(3) ], [ 171, 178, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 171, 180, -1/54*E(3)+1/54*E(3)^2 ], [ 171, 185, 1/9*E(3)^2 ], 
  [ 172, 168, -1/4*E(3)+1/4*E(3)^2 ], [ 172, 172, -1/4*E(3)^2 ], 
  [ 172, 175, 1/3*E(3)^2 ], [ 172, 179, -1/9*E(3)-2/9*E(3)^2 ], 
  [ 172, 181, -5/12 ], [ 172, 182, 1/4*E(3)+1/2*E(3)^2 ], [ 172, 183, -1/4 ], 
  [ 172, 186, -5/18*E(3)-5/36*E(3)^2 ], [ 172, 187, 1/4 ], 
  [ 172, 189, 1/2*E(3)+1/4*E(3)^2 ], [ 172, 190, 2/9*E(3)+1/9*E(3)^2 ], 
  [ 172, 194, -5/12*E(3) ], [ 172, 195, 1/3*E(3) ], 
  [ 172, 196, 5/36*E(3)-5/36*E(3)^2 ], [ 172, 197, 1/4*E(3) ], 
  [ 172, 199, 5/36*E(3)-5/36*E(3)^2 ], [ 172, 200, -5/12*E(3)^2 ], 
  [ 172, 201, 1/4*E(3)-1/4*E(3)^2 ], [ 172, 208, 5/36*E(3)+5/18*E(3)^2 ], 
  [ 172, 211, -5/12 ], [ 173, 133, E(3) ], [ 173, 140, -1/4*E(3)+1/4*E(3)^2 ],
  [ 173, 146, -1/4*E(3)^2 ], [ 173, 147, E(3)^2 ], [ 173, 149, E(3)^2 ], 
  [ 173, 156, -1/4*E(3)-1/2*E(3)^2 ], [ 173, 158, 1 ], [ 173, 160, -1/4 ], 
  [ 173, 162, -1/4 ], [ 173, 163, 1 ], [ 173, 165, -1/2*E(3)-1/4*E(3)^2 ], 
  [ 173, 171, -E(3) ], [ 173, 174, E(3) ], [ 173, 176, -1/4*E(3) ], 
  [ 173, 180, 1/4*E(3)-1/4*E(3)^2 ], [ 173, 185, -E(3)^2 ], 
  [ 174, 125, -1/18 ], [ 174, 132, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 174, 133, -1/18*E(3) ], [ 174, 138, 1/18*E(3) ], [ 174, 139, -1/9*E(3) ], 
  [ 174, 140, 1/54*E(3)-1/54*E(3)^2 ], [ 174, 141, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 174, 144, 1/54*E(3)-1/54*E(3)^2 ], [ 174, 145, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 174, 146, 1/18*E(3)^2 ], [ 174, 147, -1/9*E(3)^2 ], 
  [ 174, 148, 1/18*E(3)^2 ], [ 174, 149, 1/18*E(3)^2 ], 
  [ 174, 150, 1/9*E(3)^2 ], [ 174, 151, 1/54*E(3)+1/27*E(3)^2 ], 
  [ 174, 152, -1/18*E(3)^2 ], [ 174, 153, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 174, 154, -1/18*E(3)-1/9*E(3)^2 ], [ 174, 155, 1/9*E(3)+2/9*E(3)^2 ], 
  [ 174, 156, -1/54*E(3)-1/27*E(3)^2 ], [ 174, 157, 1/54*E(3)+1/27*E(3)^2 ], 
  [ 174, 158, 1/9 ], [ 174, 159, 1/18 ], [ 174, 160, -1/18 ], 
  [ 174, 161, 1/2 ], [ 174, 162, -1/18 ], [ 174, 163, 1/9 ], 
  [ 174, 164, -1/18 ], [ 174, 165, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 174, 166, 2/9*E(3)+1/9*E(3)^2 ], [ 174, 167, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 174, 169, -1/9*E(3)-1/18*E(3)^2 ], [ 174, 170, 1/3*E(3)+1/6*E(3)^2 ], 
  [ 174, 171, -1/18*E(3) ], [ 174, 173, 1/18*E(3) ], [ 174, 174, -1/9*E(3) ], 
  [ 174, 176, 1/18*E(3) ], [ 174, 178, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 174, 180, -1/54*E(3)+1/54*E(3)^2 ], [ 174, 185, 1/18*E(3)^2 ], 
  [ 175, 168, -1/18*E(3)+1/18*E(3)^2 ], [ 175, 172, 1/6*E(3)^2 ], 
  [ 175, 177, -4/9*E(3)-8/9*E(3)^2 ], [ 175, 182, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 175, 183, -1/6 ], [ 175, 187, 1/6 ], [ 175, 188, -5/18*E(3)-5/36*E(3)^2 ],
  [ 175, 189, -1/9*E(3)-1/18*E(3)^2 ], [ 175, 192, 8/9*E(3)+4/9*E(3)^2 ], 
  [ 175, 193, -5/12*E(3) ], [ 175, 197, -1/6*E(3) ], 
  [ 175, 198, 10/9*E(3)-10/9*E(3)^2 ], [ 175, 201, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 175, 203, 5/12*E(3)^2 ], [ 175, 206, -5/36*E(3)-5/18*E(3)^2 ], 
  [ 176, 125, -2/9 ], [ 176, 132, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 176, 133, -2/9*E(3) ], [ 176, 138, 1/18*E(3) ], [ 176, 139, 2/9*E(3) ], 
  [ 176, 140, -1/54*E(3)+1/54*E(3)^2 ], [ 176, 141, 2/9*E(3)-2/9*E(3)^2 ], 
  [ 176, 144, 4/27*E(3)-4/27*E(3)^2 ], [ 176, 146, 1/18*E(3)^2 ], 
  [ 176, 147, 2/9*E(3)^2 ], [ 176, 148, -1/18*E(3)^2 ], 
  [ 176, 149, 2/9*E(3)^2 ], [ 176, 150, -2/9*E(3)^2 ], 
  [ 176, 151, 4/27*E(3)+8/27*E(3)^2 ], [ 176, 152, -1/18*E(3)^2 ], 
  [ 176, 154, -1/18*E(3)-1/9*E(3)^2 ], [ 176, 155, -2/9*E(3)-4/9*E(3)^2 ], 
  [ 176, 156, 1/54*E(3)+1/27*E(3)^2 ], [ 176, 157, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 176, 158, -2/9 ], [ 176, 159, 4/9 ], [ 176, 160, -1/18 ], 
  [ 176, 162, -1/18 ], [ 176, 163, -2/9 ], [ 176, 164, -2/9 ], 
  [ 176, 165, 1/27*E(3)+1/54*E(3)^2 ], [ 176, 166, -4/9*E(3)-2/9*E(3)^2 ], 
  [ 176, 167, 8/27*E(3)+4/27*E(3)^2 ], [ 176, 169, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 176, 171, -2/9*E(3) ], [ 176, 173, -1/18*E(3) ], [ 176, 174, 2/9*E(3) ], 
  [ 176, 176, 1/18*E(3) ], [ 176, 178, 2/9*E(3)-2/9*E(3)^2 ], 
  [ 176, 180, 1/54*E(3)-1/54*E(3)^2 ], [ 176, 185, 2/9*E(3)^2 ], 
  [ 177, 168, 1/6 ], [ 177, 175, -4/27*E(3)-2/27*E(3)^2 ], 
  [ 177, 177, -1/6*E(3) ], [ 177, 181, -5/54*E(3)+5/54*E(3)^2 ], 
  [ 177, 182, -1/6*E(3) ], [ 177, 184, -2/27*E(3)+2/27*E(3)^2 ], 
  [ 177, 189, -1/6*E(3)^2 ], [ 177, 191, 5/54*E(3)+5/27*E(3)^2 ], 
  [ 177, 192, 1/6*E(3)^2 ], [ 177, 194, -5/54*E(3)-5/27*E(3)^2 ], 
  [ 177, 195, -2/27*E(3)-4/27*E(3)^2 ], [ 177, 200, -5/27*E(3)-5/54*E(3)^2 ], 
  [ 177, 201, 1/6 ], [ 177, 204, -5/27*E(3)-5/54*E(3)^2 ], 
  [ 177, 211, 5/54*E(3)-5/54*E(3)^2 ], [ 178, 133, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 178, 140, 1/12 ], [ 178, 146, -1/6*E(3)-1/12*E(3)^2 ], 
  [ 178, 147, -1/6*E(3)-1/12*E(3)^2 ], [ 178, 149, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 178, 151, 1/12*E(3) ], [ 178, 153, -3/4*E(3) ], [ 178, 156, 1/12*E(3) ], 
  [ 178, 158, -1/12*E(3)+1/12*E(3)^2 ], [ 178, 160, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 178, 162, 1/12*E(3)-1/12*E(3)^2 ], [ 178, 163, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 178, 165, 1/12*E(3)^2 ], [ 178, 167, 1/12*E(3)^2 ], 
  [ 178, 170, -3/4*E(3)^2 ], [ 178, 171, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 178, 174, -1/12*E(3)-1/6*E(3)^2 ], [ 178, 176, -1/12*E(3)-1/6*E(3)^2 ], 
  [ 178, 180, 1/12 ], [ 178, 185, -1/3*E(3)-1/6*E(3)^2 ], [ 179, 168, -1/2 ], 
  [ 179, 172, -1/3*E(3)-1/6*E(3)^2 ], [ 179, 182, -1/2*E(3) ], 
  [ 179, 183, 1/6*E(3)-1/6*E(3)^2 ], [ 179, 187, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 179, 188, 5/4*E(3)^2 ], [ 179, 189, -1/2*E(3)^2 ], 
  [ 179, 193, -5/12*E(3)-5/6*E(3)^2 ], [ 179, 197, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 179, 201, -1/2 ], [ 179, 203, 5/6*E(3)+5/12*E(3)^2 ], 
  [ 179, 206, 5/4*E(3) ], [ 180, 125, -2/9*E(3)+2/9*E(3)^2 ], 
  [ 180, 132, -1/6*E(3)^2 ], [ 180, 133, -2/9*E(3)-4/9*E(3)^2 ], 
  [ 180, 138, 1/18*E(3)+1/9*E(3)^2 ], [ 180, 139, -2/9*E(3)-4/9*E(3)^2 ], 
  [ 180, 140, -1/6 ], [ 180, 141, -2/3 ], [ 180, 146, -1/9*E(3)-1/18*E(3)^2 ],
  [ 180, 147, 4/9*E(3)+2/9*E(3)^2 ], [ 180, 148, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 180, 149, -4/9*E(3)-2/9*E(3)^2 ], [ 180, 150, 4/9*E(3)+2/9*E(3)^2 ], 
  [ 180, 152, -1/9*E(3)-1/18*E(3)^2 ], [ 180, 154, 1/6*E(3) ], 
  [ 180, 155, -2/3*E(3) ], [ 180, 156, 1/6*E(3) ], [ 180, 157, 1/6*E(3) ], 
  [ 180, 158, -2/9*E(3)+2/9*E(3)^2 ], [ 180, 160, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 180, 162, -1/18*E(3)+1/18*E(3)^2 ], [ 180, 163, 2/9*E(3)-2/9*E(3)^2 ], 
  [ 180, 164, 2/9*E(3)-2/9*E(3)^2 ], [ 180, 165, 1/6*E(3)^2 ], 
  [ 180, 166, -2/3*E(3)^2 ], [ 180, 169, 1/6*E(3)^2 ], 
  [ 180, 171, 2/9*E(3)+4/9*E(3)^2 ], [ 180, 173, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 180, 174, 2/9*E(3)+4/9*E(3)^2 ], [ 180, 176, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 180, 178, 2/3 ], [ 180, 180, -1/6 ], [ 180, 185, 4/9*E(3)+2/9*E(3)^2 ], 
  [ 181, 168, 1/36*E(3)-1/36*E(3)^2 ], [ 181, 172, -1/12*E(3)^2 ], 
  [ 181, 177, 2/9*E(3)+4/9*E(3)^2 ], [ 181, 181, 1/12 ], 
  [ 181, 182, 1/36*E(3)+1/18*E(3)^2 ], [ 181, 183, 1/12 ], 
  [ 181, 186, -1/6*E(3)-1/12*E(3)^2 ], [ 181, 187, -1/12 ], 
  [ 181, 188, -1/9*E(3)-1/18*E(3)^2 ], [ 181, 189, 1/18*E(3)+1/36*E(3)^2 ], 
  [ 181, 191, 2/3*E(3) ], [ 181, 192, -4/9*E(3)-2/9*E(3)^2 ], 
  [ 181, 193, -1/6*E(3) ], [ 181, 194, -1/12*E(3) ], 
  [ 181, 196, -1/12*E(3)+1/12*E(3)^2 ], [ 181, 197, 1/12*E(3) ], 
  [ 181, 198, 4/9*E(3)-4/9*E(3)^2 ], [ 181, 199, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 181, 200, -1/12*E(3)^2 ], [ 181, 201, -1/36*E(3)+1/36*E(3)^2 ], 
  [ 181, 203, 1/6*E(3)^2 ], [ 181, 204, -2/3*E(3)^2 ], 
  [ 181, 206, -1/18*E(3)-1/9*E(3)^2 ], [ 181, 208, 1/12*E(3)+1/6*E(3)^2 ], 
  [ 181, 211, 1/12 ], [ 182, 168, -1/12 ], [ 182, 172, 1/6*E(3)+1/12*E(3)^2 ],
  [ 182, 175, -2/27*E(3)-1/27*E(3)^2 ], [ 182, 177, -2/3*E(3) ], 
  [ 182, 179, -1/9*E(3) ], [ 182, 181, -5/108*E(3)+5/108*E(3)^2 ], 
  [ 182, 182, 1/12*E(3) ], [ 182, 183, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 182, 184, 8/27*E(3)-8/27*E(3)^2 ], [ 182, 186, 5/36*E(3)^2 ], 
  [ 182, 187, 1/12*E(3)-1/12*E(3)^2 ], [ 182, 189, 1/12*E(3)^2 ], 
  [ 182, 190, 1/9*E(3)^2 ], [ 182, 191, -10/27*E(3)-20/27*E(3)^2 ], 
  [ 182, 192, 2/3*E(3)^2 ], [ 182, 194, -5/108*E(3)-5/54*E(3)^2 ], 
  [ 182, 195, -1/27*E(3)-2/27*E(3)^2 ], [ 182, 196, -5/36 ], 
  [ 182, 197, -1/12*E(3)-1/6*E(3)^2 ], [ 182, 199, 5/36 ], 
  [ 182, 200, -5/54*E(3)-5/108*E(3)^2 ], [ 182, 201, -1/12 ], 
  [ 182, 204, 20/27*E(3)+10/27*E(3)^2 ], [ 182, 208, -5/36*E(3) ], 
  [ 182, 211, 5/108*E(3)-5/108*E(3)^2 ], [ 183, 168, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 183, 172, -1/4*E(3)^2 ], [ 183, 175, -1/3*E(3)^2 ], 
  [ 183, 179, -1/9*E(3)-2/9*E(3)^2 ], [ 183, 181, 5/12 ], 
  [ 183, 182, -1/4*E(3)-1/2*E(3)^2 ], [ 183, 183, -1/4 ], 
  [ 183, 186, -5/18*E(3)-5/36*E(3)^2 ], [ 183, 187, 1/4 ], 
  [ 183, 189, -1/2*E(3)-1/4*E(3)^2 ], [ 183, 190, 2/9*E(3)+1/9*E(3)^2 ], 
  [ 183, 194, 5/12*E(3) ], [ 183, 195, -1/3*E(3) ], 
  [ 183, 196, 5/36*E(3)-5/36*E(3)^2 ], [ 183, 197, 1/4*E(3) ], 
  [ 183, 199, 5/36*E(3)-5/36*E(3)^2 ], [ 183, 200, 5/12*E(3)^2 ], 
  [ 183, 201, -1/4*E(3)+1/4*E(3)^2 ], [ 183, 208, 5/36*E(3)+5/18*E(3)^2 ], 
  [ 183, 211, 5/12 ], [ 184, 168, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 184, 177, 1/9*E(3)+2/9*E(3)^2 ], [ 184, 182, -1/9*E(3)-2/9*E(3)^2 ], 
  [ 184, 188, -5/9*E(3)-5/18*E(3)^2 ], [ 184, 189, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 184, 192, -2/9*E(3)-1/9*E(3)^2 ], [ 184, 198, -5/18*E(3)+5/18*E(3)^2 ], 
  [ 184, 201, 1/9*E(3)-1/9*E(3)^2 ], [ 184, 206, -5/18*E(3)-5/9*E(3)^2 ], 
  [ 185, 125, 1/9 ], [ 185, 132, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 185, 133, 1/9*E(3) ], [ 185, 138, 1/18*E(3) ], [ 185, 139, 1/18*E(3) ], 
  [ 185, 140, -1/54*E(3)+1/54*E(3)^2 ], [ 185, 141, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 185, 144, -1/54*E(3)+1/54*E(3)^2 ], [ 185, 145, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 185, 146, 1/18*E(3)^2 ], [ 185, 147, 1/18*E(3)^2 ], 
  [ 185, 148, -1/18*E(3)^2 ], [ 185, 149, -1/9*E(3)^2 ], 
  [ 185, 150, -1/18*E(3)^2 ], [ 185, 151, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 185, 152, -1/18*E(3)^2 ], [ 185, 153, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 185, 154, -1/18*E(3)-1/9*E(3)^2 ], [ 185, 155, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 185, 156, 1/54*E(3)+1/27*E(3)^2 ], [ 185, 157, -1/54*E(3)-1/27*E(3)^2 ], 
  [ 185, 158, -1/18 ], [ 185, 159, -1/18 ], [ 185, 160, -1/18 ], 
  [ 185, 161, 1/2 ], [ 185, 162, -1/18 ], [ 185, 163, -1/18 ], 
  [ 185, 164, 1/9 ], [ 185, 165, 1/27*E(3)+1/54*E(3)^2 ], 
  [ 185, 166, -1/9*E(3)-1/18*E(3)^2 ], [ 185, 167, -1/27*E(3)-1/54*E(3)^2 ], 
  [ 185, 169, -1/9*E(3)-1/18*E(3)^2 ], [ 185, 170, 1/3*E(3)+1/6*E(3)^2 ], 
  [ 185, 171, 1/9*E(3) ], [ 185, 173, -1/18*E(3) ], [ 185, 174, 1/18*E(3) ], 
  [ 185, 176, 1/18*E(3) ], [ 185, 178, -1/9*E(3)+1/9*E(3)^2 ], 
  [ 185, 180, 1/54*E(3)-1/54*E(3)^2 ], [ 185, 185, -1/9*E(3)^2 ], 
  [ 186, 168, 1/4 ], [ 186, 172, 1/6*E(3)+1/12*E(3)^2 ], 
  [ 186, 181, -1/4*E(3)+1/4*E(3)^2 ], [ 186, 182, 1/4*E(3) ], 
  [ 186, 183, -1/12*E(3)+1/12*E(3)^2 ], [ 186, 186, -1/4*E(3)^2 ], 
  [ 186, 187, -1/12*E(3)+1/12*E(3)^2 ], [ 186, 188, 1/2*E(3)^2 ], 
  [ 186, 189, 1/4*E(3)^2 ], [ 186, 193, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 186, 194, 1/4*E(3)+1/2*E(3)^2 ], [ 186, 196, -1/4 ], 
  [ 186, 197, -1/12*E(3)-1/6*E(3)^2 ], [ 186, 199, 1/4 ], 
  [ 186, 200, 1/2*E(3)+1/4*E(3)^2 ], [ 186, 201, 1/4 ], 
  [ 186, 203, 1/3*E(3)+1/6*E(3)^2 ], [ 186, 206, 1/2*E(3) ], 
  [ 186, 208, 1/4*E(3) ], [ 186, 211, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 187, 168, 1/4*E(3)-1/4*E(3)^2 ], [ 187, 172, 1/4*E(3)^2 ], 
  [ 187, 175, 1/3*E(3)^2 ], [ 187, 179, -1/9*E(3)-2/9*E(3)^2 ], 
  [ 187, 181, -5/12 ], [ 187, 182, -1/4*E(3)-1/2*E(3)^2 ], [ 187, 183, 1/4 ], 
  [ 187, 186, -5/18*E(3)-5/36*E(3)^2 ], [ 187, 187, -1/4 ], 
  [ 187, 189, -1/2*E(3)-1/4*E(3)^2 ], [ 187, 190, 2/9*E(3)+1/9*E(3)^2 ], 
  [ 187, 194, -5/12*E(3) ], [ 187, 195, 1/3*E(3) ], 
  [ 187, 196, 5/36*E(3)-5/36*E(3)^2 ], [ 187, 197, -1/4*E(3) ], 
  [ 187, 199, 5/36*E(3)-5/36*E(3)^2 ], [ 187, 200, -5/12*E(3)^2 ], 
  [ 187, 201, -1/4*E(3)+1/4*E(3)^2 ], [ 187, 208, 5/36*E(3)+5/18*E(3)^2 ], 
  [ 187, 211, -5/12 ], [ 188, 175, 2/27*E(3)+1/27*E(3)^2 ], 
  [ 188, 179, 1/9*E(3) ], [ 188, 181, -1/27*E(3)+1/27*E(3)^2 ], 
  [ 188, 184, -8/27*E(3)+8/27*E(3)^2 ], [ 188, 186, 1/9*E(3)^2 ], 
  [ 188, 190, -1/9*E(3)^2 ], [ 188, 191, -8/27*E(3)-16/27*E(3)^2 ], 
  [ 188, 194, -1/27*E(3)-2/27*E(3)^2 ], [ 188, 195, 1/27*E(3)+2/27*E(3)^2 ], 
  [ 188, 196, -1/9 ], [ 188, 199, 1/9 ], [ 188, 200, -2/27*E(3)-1/27*E(3)^2 ],
  [ 188, 204, 16/27*E(3)+8/27*E(3)^2 ], [ 188, 208, -1/9*E(3) ], 
  [ 188, 211, 1/27*E(3)-1/27*E(3)^2 ], [ 189, 168, -1/12 ], 
  [ 189, 172, -1/6*E(3)-1/12*E(3)^2 ], [ 189, 175, 2/27*E(3)+1/27*E(3)^2 ], 
  [ 189, 177, -2/3*E(3) ], [ 189, 179, -1/9*E(3) ], 
  [ 189, 181, 5/108*E(3)-5/108*E(3)^2 ], [ 189, 182, 1/12*E(3) ], 
  [ 189, 183, -1/12*E(3)+1/12*E(3)^2 ], [ 189, 184, -8/27*E(3)+8/27*E(3)^2 ], 
  [ 189, 186, 5/36*E(3)^2 ], [ 189, 187, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 189, 189, 1/12*E(3)^2 ], [ 189, 190, 1/9*E(3)^2 ], 
  [ 189, 191, 10/27*E(3)+20/27*E(3)^2 ], [ 189, 192, 2/3*E(3)^2 ], 
  [ 189, 194, 5/108*E(3)+5/54*E(3)^2 ], [ 189, 195, 1/27*E(3)+2/27*E(3)^2 ], 
  [ 189, 196, -5/36 ], [ 189, 197, 1/12*E(3)+1/6*E(3)^2 ], [ 189, 199, 5/36 ],
  [ 189, 200, 5/54*E(3)+5/108*E(3)^2 ], [ 189, 201, -1/12 ], 
  [ 189, 204, -20/27*E(3)-10/27*E(3)^2 ], [ 189, 208, -5/36*E(3) ], 
  [ 189, 211, -5/108*E(3)+5/108*E(3)^2 ], [ 190, 168, 1/2 ], 
  [ 190, 172, -1/3*E(3)-1/6*E(3)^2 ], [ 190, 182, 1/2*E(3) ], 
  [ 190, 183, 1/6*E(3)-1/6*E(3)^2 ], [ 190, 187, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 190, 188, -5/4*E(3)^2 ], [ 190, 189, 1/2*E(3)^2 ], 
  [ 190, 193, -5/12*E(3)-5/6*E(3)^2 ], [ 190, 197, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 190, 201, 1/2 ], [ 190, 203, 5/6*E(3)+5/12*E(3)^2 ], 
  [ 190, 206, -5/4*E(3) ], [ 191, 168, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 191, 177, -1/18*E(3)-1/9*E(3)^2 ], [ 191, 181, 1/6 ], 
  [ 191, 182, 1/18*E(3)+1/9*E(3)^2 ], [ 191, 188, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 191, 189, 1/9*E(3)+1/18*E(3)^2 ], [ 191, 191, -1/6*E(3) ], 
  [ 191, 192, 1/9*E(3)+1/18*E(3)^2 ], [ 191, 194, -1/6*E(3) ], 
  [ 191, 198, -1/9*E(3)+1/9*E(3)^2 ], [ 191, 200, -1/6*E(3)^2 ], 
  [ 191, 201, -1/18*E(3)+1/18*E(3)^2 ], [ 191, 204, 1/6*E(3)^2 ], 
  [ 191, 206, -1/9*E(3)-2/9*E(3)^2 ], [ 191, 211, 1/6 ], [ 192, 168, -1/6 ], 
  [ 192, 175, -4/27*E(3)-2/27*E(3)^2 ], [ 192, 177, 1/6*E(3) ], 
  [ 192, 181, -5/54*E(3)+5/54*E(3)^2 ], [ 192, 182, 1/6*E(3) ], 
  [ 192, 184, -2/27*E(3)+2/27*E(3)^2 ], [ 192, 189, 1/6*E(3)^2 ], 
  [ 192, 191, 5/54*E(3)+5/27*E(3)^2 ], [ 192, 192, -1/6*E(3)^2 ], 
  [ 192, 194, -5/54*E(3)-5/27*E(3)^2 ], [ 192, 195, -2/27*E(3)-4/27*E(3)^2 ], 
  [ 192, 200, -5/27*E(3)-5/54*E(3)^2 ], [ 192, 201, -1/6 ], 
  [ 192, 204, -5/27*E(3)-5/54*E(3)^2 ], [ 192, 211, 5/54*E(3)-5/54*E(3)^2 ], 
  [ 193, 175, -1/3*E(3)^2 ], [ 193, 179, 1/9*E(3)+2/9*E(3)^2 ], 
  [ 193, 181, -1/3 ], [ 193, 186, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 193, 190, -2/9*E(3)-1/9*E(3)^2 ], [ 193, 194, -1/3*E(3) ], 
  [ 193, 195, -1/3*E(3) ], [ 193, 196, 1/9*E(3)-1/9*E(3)^2 ], 
  [ 193, 199, 1/9*E(3)-1/9*E(3)^2 ], [ 193, 200, -1/3*E(3)^2 ], 
  [ 193, 208, 1/9*E(3)+2/9*E(3)^2 ], [ 193, 211, -1/3 ], 
  [ 194, 168, 1/36*E(3)-1/36*E(3)^2 ], [ 194, 172, -1/12*E(3)^2 ], 
  [ 194, 177, 2/9*E(3)+4/9*E(3)^2 ], [ 194, 181, -1/12 ], 
  [ 194, 182, 1/36*E(3)+1/18*E(3)^2 ], [ 194, 183, 1/12 ], 
  [ 194, 186, 1/6*E(3)+1/12*E(3)^2 ], [ 194, 187, -1/12 ], 
  [ 194, 188, -1/9*E(3)-1/18*E(3)^2 ], [ 194, 189, 1/18*E(3)+1/36*E(3)^2 ], 
  [ 194, 191, -2/3*E(3) ], [ 194, 192, -4/9*E(3)-2/9*E(3)^2 ], 
  [ 194, 193, -1/6*E(3) ], [ 194, 194, 1/12*E(3) ], 
  [ 194, 196, 1/12*E(3)-1/12*E(3)^2 ], [ 194, 197, 1/12*E(3) ], 
  [ 194, 198, 4/9*E(3)-4/9*E(3)^2 ], [ 194, 199, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 194, 200, 1/12*E(3)^2 ], [ 194, 201, -1/36*E(3)+1/36*E(3)^2 ], 
  [ 194, 203, 1/6*E(3)^2 ], [ 194, 204, 2/3*E(3)^2 ], 
  [ 194, 206, -1/18*E(3)-1/9*E(3)^2 ], [ 194, 208, -1/12*E(3)-1/6*E(3)^2 ], 
  [ 194, 211, -1/12 ], [ 195, 168, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 195, 172, 1/6*E(3)^2 ], [ 195, 177, 4/9*E(3)+8/9*E(3)^2 ], 
  [ 195, 182, 1/18*E(3)+1/9*E(3)^2 ], [ 195, 183, -1/6 ], [ 195, 187, 1/6 ], 
  [ 195, 188, 5/18*E(3)+5/36*E(3)^2 ], [ 195, 189, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 195, 192, -8/9*E(3)-4/9*E(3)^2 ], [ 195, 193, -5/12*E(3) ], 
  [ 195, 197, -1/6*E(3) ], [ 195, 198, -10/9*E(3)+10/9*E(3)^2 ], 
  [ 195, 201, -1/18*E(3)+1/18*E(3)^2 ], [ 195, 203, 5/12*E(3)^2 ], 
  [ 195, 206, 5/36*E(3)+5/18*E(3)^2 ], [ 196, 168, -1/4 ], 
  [ 196, 172, 1/6*E(3)+1/12*E(3)^2 ], [ 196, 181, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 196, 182, -1/4*E(3) ], [ 196, 183, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 196, 186, -1/4*E(3)^2 ], [ 196, 187, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 196, 188, -1/2*E(3)^2 ], [ 196, 189, -1/4*E(3)^2 ], 
  [ 196, 193, -1/6*E(3)-1/3*E(3)^2 ], [ 196, 194, -1/4*E(3)-1/2*E(3)^2 ], 
  [ 196, 196, -1/4 ], [ 196, 197, -1/12*E(3)-1/6*E(3)^2 ], [ 196, 199, 1/4 ], 
  [ 196, 200, -1/2*E(3)-1/4*E(3)^2 ], [ 196, 201, -1/4 ], 
  [ 196, 203, 1/3*E(3)+1/6*E(3)^2 ], [ 196, 206, -1/2*E(3) ], 
  [ 196, 208, 1/4*E(3) ], [ 196, 211, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 197, 168, -1/4*E(3)+1/4*E(3)^2 ], [ 197, 172, 1/4*E(3)^2 ], 
  [ 197, 175, -1/3*E(3)^2 ], [ 197, 179, -1/9*E(3)-2/9*E(3)^2 ], 
  [ 197, 181, 5/12 ], [ 197, 182, 1/4*E(3)+1/2*E(3)^2 ], [ 197, 183, 1/4 ], 
  [ 197, 186, -5/18*E(3)-5/36*E(3)^2 ], [ 197, 187, -1/4 ], 
  [ 197, 189, 1/2*E(3)+1/4*E(3)^2 ], [ 197, 190, 2/9*E(3)+1/9*E(3)^2 ], 
  [ 197, 194, 5/12*E(3) ], [ 197, 195, -1/3*E(3) ], 
  [ 197, 196, 5/36*E(3)-5/36*E(3)^2 ], [ 197, 197, -1/4*E(3) ], 
  [ 197, 199, 5/36*E(3)-5/36*E(3)^2 ], [ 197, 200, 5/12*E(3)^2 ], 
  [ 197, 201, 1/4*E(3)-1/4*E(3)^2 ], [ 197, 208, 5/36*E(3)+5/18*E(3)^2 ], 
  [ 197, 211, 5/12 ], [ 198, 175, 4/27*E(3)+2/27*E(3)^2 ], 
  [ 198, 181, -2/27*E(3)+2/27*E(3)^2 ], [ 198, 184, 2/27*E(3)-2/27*E(3)^2 ], 
  [ 198, 191, 2/27*E(3)+4/27*E(3)^2 ], [ 198, 194, -2/27*E(3)-4/27*E(3)^2 ], 
  [ 198, 195, 2/27*E(3)+4/27*E(3)^2 ], [ 198, 200, -4/27*E(3)-2/27*E(3)^2 ], 
  [ 198, 204, -4/27*E(3)-2/27*E(3)^2 ], [ 198, 211, 2/27*E(3)-2/27*E(3)^2 ], 
  [ 199, 168, 1/4 ], [ 199, 172, 1/6*E(3)+1/12*E(3)^2 ], 
  [ 199, 181, 1/4*E(3)-1/4*E(3)^2 ], [ 199, 182, 1/4*E(3) ], 
  [ 199, 183, -1/12*E(3)+1/12*E(3)^2 ], [ 199, 186, 1/4*E(3)^2 ], 
  [ 199, 187, -1/12*E(3)+1/12*E(3)^2 ], [ 199, 188, 1/2*E(3)^2 ], 
  [ 199, 189, 1/4*E(3)^2 ], [ 199, 193, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 199, 194, -1/4*E(3)-1/2*E(3)^2 ], [ 199, 196, 1/4 ], 
  [ 199, 197, -1/12*E(3)-1/6*E(3)^2 ], [ 199, 199, -1/4 ], 
  [ 199, 200, -1/2*E(3)-1/4*E(3)^2 ], [ 199, 201, 1/4 ], 
  [ 199, 203, 1/3*E(3)+1/6*E(3)^2 ], [ 199, 206, 1/2*E(3) ], 
  [ 199, 208, -1/4*E(3) ], [ 199, 211, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 200, 168, -1/36*E(3)+1/36*E(3)^2 ], [ 200, 172, -1/12*E(3)^2 ], 
  [ 200, 177, -2/9*E(3)-4/9*E(3)^2 ], [ 200, 181, -1/12 ], 
  [ 200, 182, -1/36*E(3)-1/18*E(3)^2 ], [ 200, 183, 1/12 ], 
  [ 200, 186, -1/6*E(3)-1/12*E(3)^2 ], [ 200, 187, -1/12 ], 
  [ 200, 188, 1/9*E(3)+1/18*E(3)^2 ], [ 200, 189, -1/18*E(3)-1/36*E(3)^2 ], 
  [ 200, 191, -2/3*E(3) ], [ 200, 192, 4/9*E(3)+2/9*E(3)^2 ], 
  [ 200, 193, -1/6*E(3) ], [ 200, 194, 1/12*E(3) ], 
  [ 200, 196, -1/12*E(3)+1/12*E(3)^2 ], [ 200, 197, 1/12*E(3) ], 
  [ 200, 198, -4/9*E(3)+4/9*E(3)^2 ], [ 200, 199, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 200, 200, 1/12*E(3)^2 ], [ 200, 201, 1/36*E(3)-1/36*E(3)^2 ], 
  [ 200, 203, 1/6*E(3)^2 ], [ 200, 204, 2/3*E(3)^2 ], 
  [ 200, 206, 1/18*E(3)+1/9*E(3)^2 ], [ 200, 208, 1/12*E(3)+1/6*E(3)^2 ], 
  [ 200, 211, -1/12 ], [ 201, 168, 1/12 ], [ 201, 172, 1/6*E(3)+1/12*E(3)^2 ],
  [ 201, 175, 2/27*E(3)+1/27*E(3)^2 ], [ 201, 177, 2/3*E(3) ], 
  [ 201, 179, -1/9*E(3) ], [ 201, 181, 5/108*E(3)-5/108*E(3)^2 ], 
  [ 201, 182, -1/12*E(3) ], [ 201, 183, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 201, 184, -8/27*E(3)+8/27*E(3)^2 ], [ 201, 186, 5/36*E(3)^2 ], 
  [ 201, 187, 1/12*E(3)-1/12*E(3)^2 ], [ 201, 189, -1/12*E(3)^2 ], 
  [ 201, 190, 1/9*E(3)^2 ], [ 201, 191, 10/27*E(3)+20/27*E(3)^2 ], 
  [ 201, 192, -2/3*E(3)^2 ], [ 201, 194, 5/108*E(3)+5/54*E(3)^2 ], 
  [ 201, 195, 1/27*E(3)+2/27*E(3)^2 ], [ 201, 196, -5/36 ], 
  [ 201, 197, -1/12*E(3)-1/6*E(3)^2 ], [ 201, 199, 5/36 ], 
  [ 201, 200, 5/54*E(3)+5/108*E(3)^2 ], [ 201, 201, 1/12 ], 
  [ 201, 204, -20/27*E(3)-10/27*E(3)^2 ], [ 201, 208, -5/36*E(3) ], 
  [ 201, 211, -5/108*E(3)+5/108*E(3)^2 ], [ 202, 202, -1/18 ], 
  [ 202, 205, -1/3*E(3)-1/6*E(3)^2 ], [ 202, 207, 1/6*E(3)+1/12*E(3)^2 ], 
  [ 202, 209, 1/9*E(3) ], [ 202, 210, -3/4*E(3) ], [ 202, 212, 1/18*E(3) ], 
  [ 202, 213, 1/6*E(3)-1/6*E(3)^2 ], [ 202, 214, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 202, 215, -1/6*E(3)+1/6*E(3)^2 ], [ 202, 216, -5/36*E(3)^2 ], 
  [ 202, 217, 1/18*E(3)^2 ], [ 202, 218, -3/4*E(3)^2 ], 
  [ 202, 219, -1/9*E(3)^2 ], [ 202, 220, 5/12*E(3)+5/6*E(3)^2 ], 
  [ 202, 221, 1/12*E(3)+1/6*E(3)^2 ], [ 202, 222, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 202, 223, 5/18 ], [ 202, 224, -1/18 ], [ 202, 225, -5/6*E(3)-5/12*E(3)^2 ]
    , [ 202, 227, 5/36*E(3) ], [ 203, 175, 1/3*E(3)^2 ], 
  [ 203, 179, 1/9*E(3)+2/9*E(3)^2 ], [ 203, 181, 1/3 ], 
  [ 203, 186, -2/9*E(3)-1/9*E(3)^2 ], [ 203, 190, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 203, 194, 1/3*E(3) ], [ 203, 195, 1/3*E(3) ], 
  [ 203, 196, 1/9*E(3)-1/9*E(3)^2 ], [ 203, 199, 1/9*E(3)-1/9*E(3)^2 ], 
  [ 203, 200, 1/3*E(3)^2 ], [ 203, 208, 1/9*E(3)+2/9*E(3)^2 ], 
  [ 203, 211, 1/3 ], [ 204, 168, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 204, 177, -1/18*E(3)-1/9*E(3)^2 ], [ 204, 181, -1/6 ], 
  [ 204, 182, 1/18*E(3)+1/9*E(3)^2 ], [ 204, 188, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 204, 189, 1/9*E(3)+1/18*E(3)^2 ], [ 204, 191, 1/6*E(3) ], 
  [ 204, 192, 1/9*E(3)+1/18*E(3)^2 ], [ 204, 194, 1/6*E(3) ], 
  [ 204, 198, -1/9*E(3)+1/9*E(3)^2 ], [ 204, 200, 1/6*E(3)^2 ], 
  [ 204, 201, -1/18*E(3)+1/18*E(3)^2 ], [ 204, 204, -1/6*E(3)^2 ], 
  [ 204, 206, -1/9*E(3)-2/9*E(3)^2 ], [ 204, 211, -1/6 ], 
  [ 205, 202, -1/18*E(3)+1/18*E(3)^2 ], [ 205, 205, 1/6*E(3)^2 ], 
  [ 205, 207, 1/4*E(3)^2 ], [ 205, 210, 1/4*E(3)+1/2*E(3)^2 ], 
  [ 205, 212, 1/18*E(3)+1/9*E(3)^2 ], [ 205, 213, -1/6 ], [ 205, 215, -1/6 ], 
  [ 205, 216, 5/18*E(3)+5/36*E(3)^2 ], [ 205, 217, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 205, 218, 1/2*E(3)+1/4*E(3)^2 ], [ 205, 220, 5/12*E(3) ], 
  [ 205, 221, 1/4*E(3) ], [ 205, 222, 1/6*E(3) ], 
  [ 205, 224, 1/18*E(3)-1/18*E(3)^2 ], [ 205, 225, -5/12*E(3)^2 ], 
  [ 205, 227, -5/36*E(3)-5/18*E(3)^2 ], [ 206, 175, -2/27*E(3)-1/27*E(3)^2 ], 
  [ 206, 179, 1/9*E(3) ], [ 206, 181, 1/27*E(3)-1/27*E(3)^2 ], 
  [ 206, 184, 8/27*E(3)-8/27*E(3)^2 ], [ 206, 186, 1/9*E(3)^2 ], 
  [ 206, 190, -1/9*E(3)^2 ], [ 206, 191, 8/27*E(3)+16/27*E(3)^2 ], 
  [ 206, 194, 1/27*E(3)+2/27*E(3)^2 ], [ 206, 195, -1/27*E(3)-2/27*E(3)^2 ], 
  [ 206, 196, -1/9 ], [ 206, 199, 1/9 ], [ 206, 200, 2/27*E(3)+1/27*E(3)^2 ], 
  [ 206, 204, -16/27*E(3)-8/27*E(3)^2 ], [ 206, 208, -1/9*E(3) ], 
  [ 206, 211, -1/27*E(3)+1/27*E(3)^2 ], [ 207, 202, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 207, 205, 1/2*E(3)^2 ], [ 207, 209, -1/9*E(3)-2/9*E(3)^2 ], 
  [ 207, 212, 1/18*E(3)+1/9*E(3)^2 ], [ 207, 213, 1/2 ], [ 207, 215, 1/2 ], 
  [ 207, 217, 1/9*E(3)+1/18*E(3)^2 ], [ 207, 219, 2/9*E(3)+1/9*E(3)^2 ], 
  [ 207, 222, 1/2*E(3) ], [ 207, 224, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 208, 168, -1/4 ], [ 208, 172, 1/6*E(3)+1/12*E(3)^2 ], 
  [ 208, 181, -1/4*E(3)+1/4*E(3)^2 ], [ 208, 182, -1/4*E(3) ], 
  [ 208, 183, -1/12*E(3)+1/12*E(3)^2 ], [ 208, 186, 1/4*E(3)^2 ], 
  [ 208, 187, -1/12*E(3)+1/12*E(3)^2 ], [ 208, 188, -1/2*E(3)^2 ], 
  [ 208, 189, -1/4*E(3)^2 ], [ 208, 193, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 208, 194, 1/4*E(3)+1/2*E(3)^2 ], [ 208, 196, 1/4 ], 
  [ 208, 197, -1/12*E(3)-1/6*E(3)^2 ], [ 208, 199, -1/4 ], 
  [ 208, 200, 1/2*E(3)+1/4*E(3)^2 ], [ 208, 201, -1/4 ], 
  [ 208, 203, 1/3*E(3)+1/6*E(3)^2 ], [ 208, 206, -1/2*E(3) ], 
  [ 208, 208, -1/4*E(3) ], [ 208, 211, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 209, 202, 4/9 ], [ 209, 207, -4/3*E(3)-2/3*E(3)^2 ], 
  [ 209, 209, 1/9*E(3) ], [ 209, 212, -4/9*E(3) ], 
  [ 209, 214, 1/6*E(3)-1/6*E(3)^2 ], [ 209, 216, 10/9*E(3)^2 ], 
  [ 209, 217, -4/9*E(3)^2 ], [ 209, 219, -1/9*E(3)^2 ], 
  [ 209, 221, -2/3*E(3)-4/3*E(3)^2 ], [ 209, 223, 5/18 ], [ 209, 224, 4/9 ], 
  [ 209, 227, -10/9*E(3) ], [ 210, 202, -1/6 ], 
  [ 210, 205, 1/3*E(3)+1/6*E(3)^2 ], [ 210, 212, -1/6*E(3) ], 
  [ 210, 213, 1/6*E(3)-1/6*E(3)^2 ], [ 210, 215, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 210, 217, -1/6*E(3)^2 ], [ 210, 222, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 210, 224, -1/6 ], [ 211, 168, -1/36*E(3)+1/36*E(3)^2 ], 
  [ 211, 172, -1/12*E(3)^2 ], [ 211, 177, -2/9*E(3)-4/9*E(3)^2 ], 
  [ 211, 181, 1/12 ], [ 211, 182, -1/36*E(3)-1/18*E(3)^2 ], 
  [ 211, 183, 1/12 ], [ 211, 186, 1/6*E(3)+1/12*E(3)^2 ], [ 211, 187, -1/12 ],
  [ 211, 188, 1/9*E(3)+1/18*E(3)^2 ], [ 211, 189, -1/18*E(3)-1/36*E(3)^2 ], 
  [ 211, 191, 2/3*E(3) ], [ 211, 192, 4/9*E(3)+2/9*E(3)^2 ], 
  [ 211, 193, -1/6*E(3) ], [ 211, 194, -1/12*E(3) ], 
  [ 211, 196, 1/12*E(3)-1/12*E(3)^2 ], [ 211, 197, 1/12*E(3) ], 
  [ 211, 198, -4/9*E(3)+4/9*E(3)^2 ], [ 211, 199, 1/12*E(3)-1/12*E(3)^2 ], 
  [ 211, 200, -1/12*E(3)^2 ], [ 211, 201, 1/36*E(3)-1/36*E(3)^2 ], 
  [ 211, 203, 1/6*E(3)^2 ], [ 211, 204, -2/3*E(3)^2 ], 
  [ 211, 206, 1/18*E(3)+1/9*E(3)^2 ], [ 211, 208, -1/12*E(3)-1/6*E(3)^2 ], 
  [ 211, 211, 1/12 ], [ 212, 202, 1/18 ], [ 212, 205, 1/3*E(3)+1/6*E(3)^2 ], 
  [ 212, 207, 1/6*E(3)+1/12*E(3)^2 ], [ 212, 209, -1/9*E(3) ], 
  [ 212, 210, -3/4*E(3) ], [ 212, 212, -1/18*E(3) ], 
  [ 212, 213, -1/6*E(3)+1/6*E(3)^2 ], [ 212, 214, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 212, 215, 1/6*E(3)-1/6*E(3)^2 ], [ 212, 216, 5/36*E(3)^2 ], 
  [ 212, 217, -1/18*E(3)^2 ], [ 212, 218, -3/4*E(3)^2 ], 
  [ 212, 219, 1/9*E(3)^2 ], [ 212, 220, -5/12*E(3)-5/6*E(3)^2 ], 
  [ 212, 221, 1/12*E(3)+1/6*E(3)^2 ], [ 212, 222, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 212, 223, -5/18 ], [ 212, 224, 1/18 ], [ 212, 225, 5/6*E(3)+5/12*E(3)^2 ],
  [ 212, 227, -5/36*E(3) ], [ 213, 202, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 213, 205, -1/6*E(3)^2 ], [ 213, 207, 1/4*E(3)^2 ], 
  [ 213, 210, -1/4*E(3)-1/2*E(3)^2 ], [ 213, 212, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 213, 213, 1/6 ], [ 213, 215, 1/6 ], [ 213, 216, 5/18*E(3)+5/36*E(3)^2 ], 
  [ 213, 217, 1/9*E(3)+1/18*E(3)^2 ], [ 213, 218, -1/2*E(3)-1/4*E(3)^2 ], 
  [ 213, 220, -5/12*E(3) ], [ 213, 221, 1/4*E(3) ], [ 213, 222, -1/6*E(3) ], 
  [ 213, 224, 1/18*E(3)-1/18*E(3)^2 ], [ 213, 225, 5/12*E(3)^2 ], 
  [ 213, 227, -5/36*E(3)-5/18*E(3)^2 ], [ 214, 202, -4/9*E(3)+4/9*E(3)^2 ], 
  [ 214, 209, -1/9*E(3)-2/9*E(3)^2 ], [ 214, 212, -4/9*E(3)-8/9*E(3)^2 ], 
  [ 214, 217, -8/9*E(3)-4/9*E(3)^2 ], [ 214, 219, 2/9*E(3)+1/9*E(3)^2 ], 
  [ 214, 224, 4/9*E(3)-4/9*E(3)^2 ], [ 215, 202, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 215, 205, -1/6*E(3)^2 ], [ 215, 207, 1/4*E(3)^2 ], 
  [ 215, 210, 1/4*E(3)+1/2*E(3)^2 ], [ 215, 212, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 215, 213, 1/6 ], [ 215, 215, 1/6 ], [ 215, 216, -5/18*E(3)-5/36*E(3)^2 ], 
  [ 215, 217, -1/9*E(3)-1/18*E(3)^2 ], [ 215, 218, 1/2*E(3)+1/4*E(3)^2 ], 
  [ 215, 220, -5/12*E(3) ], [ 215, 221, 1/4*E(3) ], [ 215, 222, -1/6*E(3) ], 
  [ 215, 224, -1/18*E(3)+1/18*E(3)^2 ], [ 215, 225, 5/12*E(3)^2 ], 
  [ 215, 227, 5/36*E(3)+5/18*E(3)^2 ], [ 216, 202, -1/18 ], 
  [ 216, 205, -1/3*E(3)-1/6*E(3)^2 ], [ 216, 209, 1/9*E(3) ], 
  [ 216, 212, 1/18*E(3) ], [ 216, 213, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 216, 215, -1/6*E(3)+1/6*E(3)^2 ], [ 216, 216, 1/9*E(3)^2 ], 
  [ 216, 217, 1/18*E(3)^2 ], [ 216, 219, -1/9*E(3)^2 ], 
  [ 216, 220, -1/3*E(3)-2/3*E(3)^2 ], [ 216, 222, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 216, 223, -2/9 ], [ 216, 224, -1/18 ], [ 216, 225, 2/3*E(3)+1/3*E(3)^2 ], 
  [ 216, 227, -1/9*E(3) ], [ 217, 202, 1/18 ], 
  [ 217, 205, -1/3*E(3)-1/6*E(3)^2 ], [ 217, 207, -1/6*E(3)-1/12*E(3)^2 ], 
  [ 217, 209, -1/9*E(3) ], [ 217, 210, -3/4*E(3) ], [ 217, 212, -1/18*E(3) ], 
  [ 217, 213, 1/6*E(3)-1/6*E(3)^2 ], [ 217, 214, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 217, 215, -1/6*E(3)+1/6*E(3)^2 ], [ 217, 216, 5/36*E(3)^2 ], 
  [ 217, 217, -1/18*E(3)^2 ], [ 217, 218, -3/4*E(3)^2 ], 
  [ 217, 219, 1/9*E(3)^2 ], [ 217, 220, 5/12*E(3)+5/6*E(3)^2 ], 
  [ 217, 221, -1/12*E(3)-1/6*E(3)^2 ], [ 217, 222, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 217, 223, -5/18 ], [ 217, 224, 1/18 ], [ 217, 225, -5/6*E(3)-5/12*E(3)^2 ]
    , [ 217, 227, -5/36*E(3) ], [ 218, 202, -1/6 ], 
  [ 218, 205, -1/3*E(3)-1/6*E(3)^2 ], [ 218, 212, -1/6*E(3) ], 
  [ 218, 213, -1/6*E(3)+1/6*E(3)^2 ], [ 218, 215, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 218, 217, -1/6*E(3)^2 ], [ 218, 222, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 218, 224, -1/6 ], [ 219, 202, -4/9 ], [ 219, 207, -4/3*E(3)-2/3*E(3)^2 ], 
  [ 219, 209, -1/9*E(3) ], [ 219, 212, 4/9*E(3) ], 
  [ 219, 214, 1/6*E(3)-1/6*E(3)^2 ], [ 219, 216, -10/9*E(3)^2 ], 
  [ 219, 217, 4/9*E(3)^2 ], [ 219, 219, 1/9*E(3)^2 ], 
  [ 219, 221, -2/3*E(3)-4/3*E(3)^2 ], [ 219, 223, -5/18 ], [ 219, 224, -4/9 ],
  [ 219, 227, 10/9*E(3) ], [ 220, 202, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 220, 205, 1/6*E(3)^2 ], [ 220, 212, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 220, 213, -1/6 ], [ 220, 215, -1/6 ], [ 220, 216, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 220, 217, 1/9*E(3)+1/18*E(3)^2 ], [ 220, 220, -1/3*E(3) ], 
  [ 220, 222, 1/6*E(3) ], [ 220, 224, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 220, 225, 1/3*E(3)^2 ], [ 220, 227, 1/9*E(3)+2/9*E(3)^2 ], 
  [ 221, 202, -1/18*E(3)+1/18*E(3)^2 ], [ 221, 205, 1/2*E(3)^2 ], 
  [ 221, 209, 1/9*E(3)+2/9*E(3)^2 ], [ 221, 212, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 221, 213, 1/2 ], [ 221, 215, 1/2 ], [ 221, 217, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 221, 219, -2/9*E(3)-1/9*E(3)^2 ], [ 221, 222, 1/2*E(3) ], 
  [ 221, 224, 1/18*E(3)-1/18*E(3)^2 ], [ 222, 202, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 222, 205, 1/6*E(3)^2 ], [ 222, 207, 1/4*E(3)^2 ], 
  [ 222, 210, -1/4*E(3)-1/2*E(3)^2 ], [ 222, 212, -1/18*E(3)-1/9*E(3)^2 ], 
  [ 222, 213, -1/6 ], [ 222, 215, -1/6 ], [ 222, 216, -5/18*E(3)-5/36*E(3)^2 ]
    , [ 222, 217, -1/9*E(3)-1/18*E(3)^2 ], [ 222, 218, -1/2*E(3)-1/4*E(3)^2 ],
  [ 222, 220, 5/12*E(3) ], [ 222, 221, 1/4*E(3) ], [ 222, 222, 1/6*E(3) ], 
  [ 222, 224, -1/18*E(3)+1/18*E(3)^2 ], [ 222, 225, -5/12*E(3)^2 ], 
  [ 222, 227, 5/36*E(3)+5/18*E(3)^2 ], [ 223, 202, 4/9 ], 
  [ 223, 209, 1/9*E(3) ], [ 223, 212, -4/9*E(3) ], [ 223, 216, -8/9*E(3)^2 ], 
  [ 223, 217, -4/9*E(3)^2 ], [ 223, 219, -1/9*E(3)^2 ], [ 223, 223, -2/9 ], 
  [ 223, 224, 4/9 ], [ 223, 227, 8/9*E(3) ], [ 224, 202, -1/18 ], 
  [ 224, 205, 1/3*E(3)+1/6*E(3)^2 ], [ 224, 207, -1/6*E(3)-1/12*E(3)^2 ], 
  [ 224, 209, 1/9*E(3) ], [ 224, 210, -3/4*E(3) ], [ 224, 212, 1/18*E(3) ], 
  [ 224, 213, -1/6*E(3)+1/6*E(3)^2 ], [ 224, 214, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 224, 215, 1/6*E(3)-1/6*E(3)^2 ], [ 224, 216, -5/36*E(3)^2 ], 
  [ 224, 217, 1/18*E(3)^2 ], [ 224, 218, -3/4*E(3)^2 ], 
  [ 224, 219, -1/9*E(3)^2 ], [ 224, 220, -5/12*E(3)-5/6*E(3)^2 ], 
  [ 224, 221, -1/12*E(3)-1/6*E(3)^2 ], [ 224, 222, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 224, 223, 5/18 ], [ 224, 224, -1/18 ], [ 224, 225, 5/6*E(3)+5/12*E(3)^2 ],
  [ 224, 227, 5/36*E(3) ], [ 225, 202, -1/18*E(3)+1/18*E(3)^2 ], 
  [ 225, 205, -1/6*E(3)^2 ], [ 225, 212, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 225, 213, 1/6 ], [ 225, 215, 1/6 ], [ 225, 216, -2/9*E(3)-1/9*E(3)^2 ], 
  [ 225, 217, 1/9*E(3)+1/18*E(3)^2 ], [ 225, 220, 1/3*E(3) ], 
  [ 225, 222, -1/6*E(3) ], [ 225, 224, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 225, 225, -1/3*E(3)^2 ], [ 225, 227, 1/9*E(3)+2/9*E(3)^2 ], 
  [ 226, 226, -1/3 ], [ 226, 228, 4/3*E(3)+2/3*E(3)^2 ], 
  [ 226, 229, 2/3*E(3) ], [ 226, 230, -1/6*E(3) ], 
  [ 226, 231, -1/6*E(3)+1/6*E(3)^2 ], [ 226, 232, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 226, 233, -1/6*E(3)^2 ], [ 226, 234, 2/3*E(3)^2 ], 
  [ 226, 235, -2/3*E(3)-4/3*E(3)^2 ], [ 226, 236, 1/3 ], [ 227, 202, 1/18 ], 
  [ 227, 205, -1/3*E(3)-1/6*E(3)^2 ], [ 227, 209, -1/9*E(3) ], 
  [ 227, 212, -1/18*E(3) ], [ 227, 213, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 227, 215, -1/6*E(3)+1/6*E(3)^2 ], [ 227, 216, -1/9*E(3)^2 ], 
  [ 227, 217, -1/18*E(3)^2 ], [ 227, 219, 1/9*E(3)^2 ], 
  [ 227, 220, -1/3*E(3)-2/3*E(3)^2 ], [ 227, 222, -1/6*E(3)-1/3*E(3)^2 ], 
  [ 227, 223, 2/9 ], [ 227, 224, 1/18 ], [ 227, 225, 2/3*E(3)+1/3*E(3)^2 ], 
  [ 227, 227, 1/9*E(3) ], [ 228, 226, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 228, 228, -1/2*E(3)^2 ], [ 228, 229, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 228, 230, 1/18*E(3)+1/9*E(3)^2 ], [ 228, 233, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 228, 234, 1/9*E(3)+1/18*E(3)^2 ], [ 228, 235, 1/2*E(3) ], 
  [ 228, 236, 1/18*E(3)-1/18*E(3)^2 ], [ 229, 226, 1/6 ], 
  [ 229, 228, 1/3*E(3)+1/6*E(3)^2 ], [ 229, 229, 1/6*E(3) ], 
  [ 229, 230, -1/6*E(3) ], [ 229, 232, 1/3*E(3)-1/3*E(3)^2 ], 
  [ 229, 233, -1/6*E(3)^2 ], [ 229, 234, 1/6*E(3)^2 ], 
  [ 229, 235, -1/6*E(3)-1/3*E(3)^2 ], [ 229, 236, -1/6 ], [ 230, 226, -1/6 ], 
  [ 230, 228, 4/3*E(3)+2/3*E(3)^2 ], [ 230, 229, -2/3*E(3) ], 
  [ 230, 230, -1/3*E(3) ], [ 230, 231, 1/6*E(3)-1/6*E(3)^2 ], 
  [ 230, 232, -1/6*E(3)+1/6*E(3)^2 ], [ 230, 233, -1/3*E(3)^2 ], 
  [ 230, 234, -2/3*E(3)^2 ], [ 230, 235, -2/3*E(3)-4/3*E(3)^2 ], 
  [ 230, 236, 1/6 ], [ 231, 226, 1/2*E(3)-1/2*E(3)^2 ], 
  [ 231, 230, -1/2*E(3)-E(3)^2 ], [ 231, 233, -E(3)-1/2*E(3)^2 ], 
  [ 231, 236, 1/2*E(3)-1/2*E(3)^2 ], [ 232, 226, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 232, 229, -4/9*E(3)-8/9*E(3)^2 ], [ 232, 230, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 232, 233, 1/9*E(3)+1/18*E(3)^2 ], [ 232, 234, -8/9*E(3)-4/9*E(3)^2 ], 
  [ 232, 236, 1/18*E(3)-1/18*E(3)^2 ], [ 233, 226, -1/6 ], 
  [ 233, 228, -4/3*E(3)-2/3*E(3)^2 ], [ 233, 229, -2/3*E(3) ], 
  [ 233, 230, -1/3*E(3) ], [ 233, 231, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 233, 232, 1/6*E(3)-1/6*E(3)^2 ], [ 233, 233, -1/3*E(3)^2 ], 
  [ 233, 234, -2/3*E(3)^2 ], [ 233, 235, 2/3*E(3)+4/3*E(3)^2 ], 
  [ 233, 236, 1/6 ], [ 234, 226, 1/6 ], [ 234, 228, -1/3*E(3)-1/6*E(3)^2 ], 
  [ 234, 229, 1/6*E(3) ], [ 234, 230, -1/6*E(3) ], 
  [ 234, 232, -1/3*E(3)+1/3*E(3)^2 ], [ 234, 233, -1/6*E(3)^2 ], 
  [ 234, 234, 1/6*E(3)^2 ], [ 234, 235, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 234, 236, -1/6 ], [ 235, 226, 1/18*E(3)-1/18*E(3)^2 ], 
  [ 235, 228, 1/2*E(3)^2 ], [ 235, 229, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 235, 230, 1/18*E(3)+1/9*E(3)^2 ], [ 235, 233, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 235, 234, 1/9*E(3)+1/18*E(3)^2 ], [ 235, 235, -1/2*E(3) ], 
  [ 235, 236, 1/18*E(3)-1/18*E(3)^2 ], [ 236, 226, 1/3 ], 
  [ 236, 228, 4/3*E(3)+2/3*E(3)^2 ], [ 236, 229, -2/3*E(3) ], 
  [ 236, 230, 1/6*E(3) ], [ 236, 231, -1/6*E(3)+1/6*E(3)^2 ], 
  [ 236, 232, -1/6*E(3)+1/6*E(3)^2 ], [ 236, 233, 1/6*E(3)^2 ], 
  [ 236, 234, -2/3*E(3)^2 ], [ 236, 235, -2/3*E(3)-4/3*E(3)^2 ], 
  [ 236, 236, -1/3 ], [ 237, 237, -2/3 ], [ 237, 238, -5/6*E(3) ], 
  [ 237, 240, 5/6*E(3)^2 ], [ 238, 237, -1/3 ], [ 238, 238, 1/3*E(3) ], 
  [ 238, 239, -1/3*E(3)+1/3*E(3)^2 ], [ 238, 240, -1/3*E(3)^2 ], 
  [ 239, 238, 1/2*E(3)+E(3)^2 ], [ 239, 240, -E(3)-1/2*E(3)^2 ], 
  [ 240, 237, 1/3 ], [ 240, 238, -1/3*E(3) ], 
  [ 240, 239, -1/3*E(3)+1/3*E(3)^2 ], [ 240, 240, 1/3*E(3)^2 ], 
  [ 241, 1, -1/12*E(3)-1/6*E(3)^2 ], [ 241, 2, -1/36*E(3)-1/18*E(3)^2 ], 
  [ 241, 3, 1/18*E(3)+1/9*E(3)^2 ], [ 241, 4, -2/3*E(3)-4/3*E(3)^2 ], 
  [ 241, 9, -1/18*E(3)^2 ], [ 241, 10, -2/9*E(3)^2 ], 
  [ 241, 11, -1/4*E(3)^2 ], [ 241, 16, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 241, 17, 1/36*E(3)-1/36*E(3)^2 ], [ 241, 23, -1/36*E(3) ], 
  [ 241, 121, -1/18*E(3)-1/36*E(3)^2 ], [ 241, 122, -1/6*E(3)-1/12*E(3)^2 ], 
  [ 241, 123, 4/3*E(3)+2/3*E(3)^2 ], [ 241, 124, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 241, 129, -2/9*E(3) ], [ 241, 130, -1/18*E(3) ], [ 241, 131, -1/4*E(3) ], 
  [ 241, 136, 1/36*E(3)-1/36*E(3)^2 ], [ 241, 137, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 241, 143, -1/36*E(3)^2 ], [ 241, 241, -2/9 ], [ 241, 242, 2/9 ], 
  [ 241, 243, -7/36 ], [ 241, 244, 7/36 ], [ 241, 245, 5/18 ], 
  [ 242, 1, -1/12*E(3)-1/6*E(3)^2 ], [ 242, 2, -1/36*E(3)-1/18*E(3)^2 ], 
  [ 242, 3, 1/18*E(3)+1/9*E(3)^2 ], [ 242, 4, -2/3*E(3)-4/3*E(3)^2 ], 
  [ 242, 9, 1/18*E(3)^2 ], [ 242, 10, 2/9*E(3)^2 ], [ 242, 11, 1/4*E(3)^2 ], 
  [ 242, 16, -1/12*E(3)+1/12*E(3)^2 ], [ 242, 17, 1/36*E(3)-1/36*E(3)^2 ], 
  [ 242, 23, 1/36*E(3) ], [ 242, 121, -1/18*E(3)-1/36*E(3)^2 ], 
  [ 242, 122, -1/6*E(3)-1/12*E(3)^2 ], [ 242, 123, 4/3*E(3)+2/3*E(3)^2 ], 
  [ 242, 124, -1/9*E(3)-1/18*E(3)^2 ], [ 242, 129, 2/9*E(3) ], 
  [ 242, 130, 1/18*E(3) ], [ 242, 131, 1/4*E(3) ], 
  [ 242, 136, 1/36*E(3)-1/36*E(3)^2 ], [ 242, 137, -1/12*E(3)+1/12*E(3)^2 ], 
  [ 242, 143, 1/36*E(3)^2 ], [ 242, 241, 2/9 ], [ 242, 242, -2/9 ], 
  [ 242, 243, 7/36 ], [ 242, 244, -7/36 ], [ 242, 245, 13/18 ], 
  [ 243, 1, 1/6*E(3)+1/3*E(3)^2 ], [ 243, 2, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 243, 3, 1/18*E(3)+1/9*E(3)^2 ], [ 243, 4, -2/3*E(3)-4/3*E(3)^2 ], 
  [ 243, 9, -1/18*E(3)^2 ], [ 243, 10, -2/9*E(3)^2 ], 
  [ 243, 16, -1/3*E(3)+1/3*E(3)^2 ], [ 243, 17, 1/9*E(3)-1/9*E(3)^2 ], 
  [ 243, 23, 2/9*E(3) ], [ 243, 121, 1/9*E(3)+1/18*E(3)^2 ], 
  [ 243, 122, 1/3*E(3)+1/6*E(3)^2 ], [ 243, 123, 4/3*E(3)+2/3*E(3)^2 ], 
  [ 243, 124, -1/9*E(3)-1/18*E(3)^2 ], [ 243, 129, -2/9*E(3) ], 
  [ 243, 130, -1/18*E(3) ], [ 243, 136, 1/9*E(3)-1/9*E(3)^2 ], 
  [ 243, 137, -1/3*E(3)+1/3*E(3)^2 ], [ 243, 143, 2/9*E(3)^2 ], 
  [ 243, 241, -2/9 ], [ 243, 242, 2/9 ], [ 243, 243, 1/18 ], 
  [ 243, 244, -1/18 ], [ 243, 245, 7/9 ], [ 244, 1, 1/6*E(3)+1/3*E(3)^2 ], 
  [ 244, 2, 1/18*E(3)+1/9*E(3)^2 ], [ 244, 3, 1/18*E(3)+1/9*E(3)^2 ], 
  [ 244, 4, -2/3*E(3)-4/3*E(3)^2 ], [ 244, 9, 1/18*E(3)^2 ], 
  [ 244, 10, 2/9*E(3)^2 ], [ 244, 16, -1/3*E(3)+1/3*E(3)^2 ], 
  [ 244, 17, 1/9*E(3)-1/9*E(3)^2 ], [ 244, 23, -2/9*E(3) ], 
  [ 244, 121, 1/9*E(3)+1/18*E(3)^2 ], [ 244, 122, 1/3*E(3)+1/6*E(3)^2 ], 
  [ 244, 123, 4/3*E(3)+2/3*E(3)^2 ], [ 244, 124, -1/9*E(3)-1/18*E(3)^2 ], 
  [ 244, 129, 2/9*E(3) ], [ 244, 130, 1/18*E(3) ], 
  [ 244, 136, 1/9*E(3)-1/9*E(3)^2 ], [ 244, 137, -1/3*E(3)+1/3*E(3)^2 ], 
  [ 244, 143, -2/9*E(3)^2 ], [ 244, 241, 2/9 ], [ 244, 242, -2/9 ], 
  [ 244, 243, -1/18 ], [ 244, 244, 1/18 ], [ 244, 245, 11/9 ], 
  [ 245, 245, 1 ], [ 246, 6, 2/9*E(3)^2 ], [ 246, 15, 5/18*E(3)^2 ], 
  [ 246, 22, -2/9*E(3) ], [ 246, 126, 5/9*E(3) ], [ 246, 135, 5/18*E(3) ], 
  [ 246, 142, -5/9*E(3)^2 ], [ 246, 245, 5/9 ], [ 246, 246, 1/6 ], 
  [ 246, 247, 5/18 ], [ 246, 248, -5/18 ], [ 247, 6, 1/9*E(3)^2 ], 
  [ 247, 7, 1/6*E(3)+1/3*E(3)^2 ], [ 247, 8, 1/9*E(3)+2/9*E(3)^2 ], 
  [ 247, 14, 1/6*E(3)-1/6*E(3)^2 ], [ 247, 15, 7/18*E(3)^2 ], 
  [ 247, 22, -1/9*E(3) ], [ 247, 126, 5/18*E(3) ], 
  [ 247, 127, 2/9*E(3)+1/9*E(3)^2 ], [ 247, 128, 1/3*E(3)+1/6*E(3)^2 ], 
  [ 247, 134, -5/18*E(3)+5/18*E(3)^2 ], [ 247, 135, 7/18*E(3) ], 
  [ 247, 142, -5/18*E(3)^2 ], [ 247, 245, 5/18 ], [ 247, 246, 1/3 ], 
  [ 247, 247, -1/9 ], [ 247, 248, 1/9 ], [ 248, 6, 1/9*E(3)^2 ], 
  [ 248, 7, 1/6*E(3)+1/3*E(3)^2 ], [ 248, 8, 1/9*E(3)+2/9*E(3)^2 ], 
  [ 248, 14, 1/6*E(3)-1/6*E(3)^2 ], [ 248, 15, -1/9*E(3)^2 ], 
  [ 248, 22, -1/9*E(3) ], [ 248, 126, 5/18*E(3) ], 
  [ 248, 127, 2/9*E(3)+1/9*E(3)^2 ], [ 248, 128, 1/3*E(3)+1/6*E(3)^2 ], 
  [ 248, 134, -5/18*E(3)+5/18*E(3)^2 ], [ 248, 135, -1/9*E(3) ], 
  [ 248, 142, -5/18*E(3)^2 ], [ 248, 245, 5/18 ], [ 248, 246, -1/6 ], 
  [ 248, 247, 7/18 ], [ 248, 248, -7/18 ] ],

  [ 248, [ 1, 1, -1 ], [ 2, 2, -1 ], [ 3, 3, -1 ], [ 4, 4, -1 ], [ 5, 5, 1 ], 
  [ 6, 6, 1 ], [ 7, 7, -1 ], [ 8, 8, -1 ], [ 9, 9, 1 ], [ 10, 10, 1 ], 
  [ 11, 11, 1 ], [ 12, 12, -1 ], [ 13, 13, 1 ], [ 14, 14, -1 ], 
  [ 15, 15, 1 ], [ 16, 16, -1 ], [ 17, 17, -1 ], [ 18, 18, 1 ], 
  [ 19, 19, 1 ], [ 20, 20, -1 ], [ 21, 21, -1 ], [ 22, 22, 1 ], 
  [ 23, 23, 1 ], [ 24, 24, -1 ], [ 25, 25, -1 ], [ 26, 26, 1 ], 
  [ 27, 27, 1 ], [ 28, 28, 1 ], [ 29, 29, 1 ], [ 30, 30, 1 ], 
  [ 31, 31, -1 ], [ 32, 32, 1 ], [ 33, 33, -1 ], [ 34, 34, -1 ], 
  [ 35, 35, -1 ], [ 36, 36, -1 ], [ 37, 37, -1 ], [ 38, 38, 1 ], 
  [ 39, 39, 1 ], [ 40, 40, 1 ], [ 41, 41, 1 ], [ 42, 42, 1 ], [ 43, 43, 1 ], 
  [ 44, 44, 1 ], [ 45, 45, -1 ], [ 46, 46, -1 ], [ 47, 47, -1 ], 
  [ 48, 48, 1 ], [ 49, 49, -1 ], [ 50, 50, -1 ], [ 51, 51, 1 ], 
  [ 52, 52, -1 ], [ 53, 53, 1 ], [ 54, 54, 1 ], [ 55, 55, -1 ], 
  [ 56, 56, 1 ], [ 57, 57, 1 ], [ 58, 58, -1 ], [ 59, 59, 1 ], 
  [ 60, 60, -1 ], [ 61, 61, -1 ], [ 62, 62, 1 ], [ 63, 63, -1 ], 
  [ 64, 64, -1 ], [ 65, 65, 1 ], [ 66, 66, 1 ], [ 67, 67, -1 ], 
  [ 68, 68, 1 ], [ 69, 69, 1 ], [ 70, 70, 1 ], [ 71, 71, -1 ], 
  [ 72, 72, 1 ], [ 73, 73, -1 ], [ 74, 74, -1 ], [ 75, 75, -1 ], 
  [ 76, 76, 1 ], [ 77, 77, -1 ], [ 78, 78, 1 ], [ 79, 79, 1 ], 
  [ 80, 80, -1 ], [ 81, 81, 1 ], [ 82, 82, 1 ], [ 83, 83, -1 ], 
  [ 84, 84, -1 ], [ 85, 85, -1 ], [ 86, 86, 1 ], [ 87, 87, -1 ], 
  [ 88, 88, 1 ], [ 89, 89, 1 ], [ 90, 90, 1 ], [ 91, 91, -1 ], 
  [ 92, 92, 1 ], [ 93, 93, -1 ], [ 94, 94, -1 ], [ 95, 95, -1 ], 
  [ 96, 96, 1 ], [ 97, 97, 1 ], [ 98, 98, 1 ], [ 99, 99, 1 ], 
  [ 100, 100, -1 ], [ 101, 101, -1 ], [ 102, 102, -1 ], [ 103, 103, 1 ], 
  [ 104, 104, 1 ], [ 105, 105, -1 ], [ 106, 106, 1 ], [ 107, 107, 1 ], 
  [ 108, 108, -1 ], [ 109, 109, 1 ], [ 110, 110, 1 ], [ 111, 111, -1 ], 
  [ 112, 112, -1 ], [ 113, 113, 1 ], [ 114, 114, 1 ], [ 115, 115, -1 ], 
  [ 116, 116, 1 ], [ 117, 117, 1 ], [ 118, 118, 1 ], [ 119, 119, -1 ], 
  [ 120, 120, 1 ], [ 121, 121, -1 ], [ 122, 122, -1 ], [ 123, 123, -1 ], 
  [ 124, 124, -1 ], [ 125, 125, 1 ], [ 126, 126, 1 ], [ 127, 127, -1 ], 
  [ 128, 128, -1 ], [ 129, 129, 1 ], [ 130, 130, 1 ], [ 131, 131, 1 ], 
  [ 132, 132, -1 ], [ 133, 133, 1 ], [ 134, 134, -1 ], [ 135, 135, 1 ], 
  [ 136, 136, -1 ], [ 137, 137, -1 ], [ 138, 138, 1 ], [ 139, 139, 1 ], 
  [ 140, 140, -1 ], [ 141, 141, -1 ], [ 142, 142, 1 ], [ 143, 143, 1 ], 
  [ 144, 144, -1 ], [ 145, 145, -1 ], [ 146, 146, 1 ], [ 147, 147, 1 ], 
  [ 148, 148, 1 ], [ 149, 149, 1 ], [ 150, 150, 1 ], [ 151, 151, -1 ], 
  [ 152, 152, 1 ], [ 153, 153, -1 ], [ 154, 154, -1 ], [ 155, 155, -1 ], 
  [ 156, 156, -1 ], [ 157, 157, -1 ], [ 158, 158, 1 ], [ 159, 159, 1 ], 
  [ 160, 160, 1 ], [ 161, 161, 1 ], [ 162, 162, 1 ], [ 163, 163, 1 ], 
  [ 164, 164, 1 ], [ 165, 165, -1 ], [ 166, 166, -1 ], [ 167, 167, -1 ], 
  [ 168, 168, 1 ], [ 169, 169, -1 ], [ 170, 170, -1 ], [ 171, 171, 1 ], 
  [ 172, 172, -1 ], [ 173, 173, 1 ], [ 174, 174, 1 ], [ 175, 175, -1 ], 
  [ 176, 176, 1 ], [ 177, 177, 1 ], [ 178, 178, -1 ], [ 179, 179, 1 ], 
  [ 180, 180, -1 ], [ 181, 181, -1 ], [ 182, 182, 1 ], [ 183, 183, -1 ], 
  [ 184, 184, -1 ], [ 185, 185, 1 ], [ 186, 186, 1 ], [ 187, 187, -1 ], 
  [ 188, 188, 1 ], [ 189, 189, 1 ], [ 190, 190, 1 ], [ 191, 191, -1 ], 
  [ 192, 192, 1 ], [ 193, 193, -1 ], [ 194, 194, -1 ], [ 195, 195, -1 ], 
  [ 196, 196, 1 ], [ 197, 197, -1 ], [ 198, 198, 1 ], [ 199, 199, 1 ], 
  [ 200, 200, -1 ], [ 201, 201, 1 ], [ 202, 202, 1 ], [ 203, 203, -1 ], 
  [ 204, 204, -1 ], [ 205, 205, -1 ], [ 206, 206, 1 ], [ 207, 207, -1 ], 
  [ 208, 208, 1 ], [ 209, 209, 1 ], [ 210, 210, 1 ], [ 211, 211, -1 ], 
  [ 212, 212, 1 ], [ 213, 213, -1 ], [ 214, 214, -1 ], [ 215, 215, -1 ], 
  [ 216, 216, 1 ], [ 217, 217, 1 ], [ 218, 218, 1 ], [ 219, 219, 1 ], 
  [ 220, 220, -1 ], [ 221, 221, -1 ], [ 222, 222, -1 ], [ 223, 223, 1 ], 
  [ 224, 224, 1 ], [ 225, 225, -1 ], [ 226, 226, 1 ], [ 227, 227, 1 ], 
  [ 228, 228, -1 ], [ 229, 229, 1 ], [ 230, 230, 1 ], [ 231, 231, -1 ], 
  [ 232, 232, -1 ], [ 233, 233, 1 ], [ 234, 234, 1 ], [ 235, 235, -1 ], 
  [ 236, 236, 1 ], [ 237, 237, 1 ], [ 238, 238, 1 ], [ 239, 239, -1 ], 
  [ 240, 240, 1 ], [ 241, 241, 1 ], [ 242, 242, 1 ], [ 243, 243, 1 ], 
  [ 244, 244, 1 ], [ 245, 245, 1 ], [ 246, 246, 1 ], [ 247, 247, 1 ], 
  [ 248, 248, 1 ] ] ],

  [ [ 2, 0, 0, 0, 0, 0, 0, 2 ],
    [ 248, [ 1, 31, -E(5)^3 ], [ 2, 11, E(5)^4 ], [ 3, 10, E(5)^4 ], [ 4, 124, E(5)^2 ], [ 5, 28, E(5)^3 ], [ 6, 181, 1 ], [ 7, 48, -E(5)^2 ], [ 8, 22, E(5)^4 ], 
  [ 9, 45, E(5)^2 ], [ 10, 3, E(5) ], [ 11, 2, E(5) ], [ 12, 21, 1 ], [ 13, 153, -E(5)^3 ], [ 14, 134, E(5)^2 ], [ 15, 68, E(5) ], [ 16, 38, E(5)^4 ], [ 17, 17, -1 ], [ 18, 35, E(5)^4 ], 
  [ 19, 34, E(5)^4 ], [ 20, 160, -1 ], [ 21, 12, 1 ], [ 22, 8, E(5) ], [ 23, 51, -E(5)^3 ], [ 24, 66, -E(5)^2 ], [ 25, 49, E(5)^3 ], [ 26, 146, E(5)^4 ], 
  [ 27, 147, E(5)^4 ], [ 28, 5, E(5)^2 ], [ 29, 36, E(5)^4 ], [ 30, 76, -E(5) ], [ 31, 1, -E(5)^2 ], [ 32, 41, 1 ], [ 33, 133, -E(5)^3 ], [ 34, 19, E(5) ], [ 35, 18, E(5) ], 
  [ 36, 29, E(5) ], [ 37, 71, -E(5)^3 ], [ 38, 16, E(5) ], [ 39, 52, E(5)^4 ], [ 40, 140, -1 ], [ 41, 32, 1 ], [ 42, 43, 1 ], [ 43, 42, 1 ], 
  [ 44, 80, -E(5)^2 ], [ 45, 9, E(5)^3 ], [ 46, 63, E(5)^3 ], [ 47, 73, -E(5)^3 ], [ 48, 7, -E(5)^3 ], [ 49, 25, E(5)^2 ], [ 50, 56, E(5)^4 ], [ 51, 23, -E(5)^2 ], [ 52, 39, E(5) ], 
  [ 53, 57, 1 ], [ 54, 83, -E(5)^2 ], [ 55, 55, -1 ], [ 56, 50, E(5) ], [ 57, 53, 1 ], [ 58, 69, E(5)^4 ], [ 59, 82, E(5)^3 ], [ 60, 78, -E(5)^4 ], 
  [ 61, 126, 1 ], [ 62, 74, E(5)^4 ], [ 63, 46, E(5)^2 ], [ 64, 89, -E(5)^2 ], [ 65, 86, -E(5)^3 ], [ 66, 24, -E(5)^3 ], [ 67, 96, E(5)^2 ], [ 68, 15, E(5)^4 ], [ 69, 58, E(5) ], 
  [ 70, 85, -E(5)^4 ], [ 71, 37, -E(5)^2 ], [ 72, 103, -E(5) ], [ 73, 47, -E(5)^2 ], [ 74, 62, E(5) ], [ 75, 93, E(5)^3 ], [ 76, 30, -E(5)^4 ], [ 77, 100, -E(5)^3 ], 
  [ 78, 60, -E(5) ], [ 79, 87, -E(5)^4 ], [ 80, 44, -E(5)^3 ], [ 81, 105, E(5)^2 ], [ 82, 59, E(5)^2 ], [ 83, 54, -E(5)^3 ], [ 84, 94, E(5)^3 ], [ 85, 70, -E(5) ], [ 86, 65, -E(5)^2 ], 
  [ 87, 79, -E(5) ], [ 88, 90, 1 ], [ 89, 64, -E(5)^3 ], [ 90, 88, 1 ], [ 91, 98, -E(5)^4 ], [ 92, 106, E(5)^3 ], [ 93, 75, E(5)^2 ], [ 94, 84, E(5)^2 ], 
  [ 95, 110, E(5)^2 ], [ 96, 67, E(5)^3 ], [ 97, 97, -1 ], [ 98, 91, -E(5) ], [ 99, 108, E(5)^4 ], [ 100, 77, -E(5)^2 ], [ 101, 107, -E(5)^4 ], [ 102, 112, E(5)^3 ], [ 103, 72, -E(5)^4 ],
  [ 104, 118, E(5) ], [ 105, 81, E(5)^3 ], [ 106, 92, E(5)^2 ], [ 107, 101, -E(5) ], [ 108, 99, E(5) ], [ 109, 111, E(5)^4 ], [ 110, 95, E(5)^3 ], [ 111, 109, E(5) ], 
  [ 112, 102, E(5)^2 ], [ 113, 114, 1 ], [ 114, 113, 1 ], [ 115, 116, E(5)^4 ], [ 116, 115, E(5) ], [ 117, 119, -E(5)^4 ], [ 118, 104, E(5)^4 ], [ 119, 117, -E(5) ], [ 120, 120, -1 ], 
  [ 121, 151, -E(5)^2 ], [ 122, 131, E(5) ], [ 123, 130, E(5) ], [ 124, 4, E(5)^3 ], [ 125, 148, E(5)^2 ], [ 126, 61, 1 ], [ 127, 168, -E(5)^3 ], [ 128, 142, E(5) ], 
  [ 129, 165, E(5)^3 ], [ 130, 123, E(5)^4 ], [ 131, 122, E(5)^4 ], [ 132, 141, 1 ], [ 133, 33, -E(5)^2 ], [ 134, 14, E(5)^3 ], [ 135, 188, E(5)^4 ], [ 136, 158, E(5) ], [ 137, 137, -1 ],
  [ 138, 155, E(5) ], [ 139, 154, E(5) ], [ 140, 40, -1 ], [ 141, 132, 1 ], [ 142, 128, E(5)^4 ], [ 143, 171, -E(5)^2 ], [ 144, 186, -E(5)^3 ], [ 145, 169, E(5)^2 ], 
  [ 146, 26, E(5) ], [ 147, 27, E(5) ], [ 148, 125, E(5)^3 ], [ 149, 156, E(5) ], [ 150, 196, -E(5)^4 ], [ 151, 121, -E(5)^3 ], [ 152, 161, 1 ], [ 153, 13, -E(5)^2 ], 
  [ 154, 139, E(5)^4 ], [ 155, 138, E(5)^4 ], [ 156, 149, E(5)^4 ], [ 157, 191, -E(5)^2 ], [ 158, 136, E(5)^4 ], [ 159, 172, E(5) ], [ 160, 20, -1 ], [ 161, 152, 1 ], 
  [ 162, 163, 1 ], [ 163, 162, 1 ], [ 164, 200, -E(5)^3 ], [ 165, 129, E(5)^2 ], [ 166, 183, E(5)^2 ], [ 167, 193, -E(5)^2 ], [ 168, 127, -E(5)^2 ], [ 169, 145, E(5)^3 ], 
  [ 170, 176, E(5) ], [ 171, 143, -E(5)^3 ], [ 172, 159, E(5)^4 ], [ 173, 177, 1 ], [ 174, 203, -E(5)^3 ], [ 175, 175, -1 ], [ 176, 170, E(5)^4 ], [ 177, 173, 1 ], 
  [ 178, 189, E(5) ], [ 179, 202, E(5)^2 ], [ 180, 198, -E(5) ], [ 181, 6, 1 ], [ 182, 194, E(5) ], [ 183, 166, E(5)^3 ], [ 184, 209, -E(5)^3 ], [ 185, 206, -E(5)^2 ], 
  [ 186, 144, -E(5)^2 ], [ 187, 216, E(5)^3 ], [ 188, 135, E(5) ], [ 189, 178, E(5)^4 ], [ 190, 205, -E(5) ], [ 191, 157, -E(5)^3 ], [ 192, 223, -E(5)^4 ], [ 193, 167, -E(5)^3 ], 
  [ 194, 182, E(5)^4 ], [ 195, 213, E(5)^2 ], [ 196, 150, -E(5) ], [ 197, 220, -E(5)^2 ], [ 198, 180, -E(5)^4 ], [ 199, 207, -E(5) ], [ 200, 164, -E(5)^2 ], [ 201, 225, E(5)^3 ], 
  [ 202, 179, E(5)^3 ], [ 203, 174, -E(5)^2 ], [ 204, 214, E(5)^2 ], [ 205, 190, -E(5)^4 ], [ 206, 185, -E(5)^3 ], [ 207, 199, -E(5)^4 ], [ 208, 210, 1 ], [ 209, 184, -E(5)^2 ], 
  [ 210, 208, 1 ], [ 211, 218, -E(5) ], [ 212, 226, E(5)^2 ], [ 213, 195, E(5)^3 ], [ 214, 204, E(5)^3 ], [ 215, 230, E(5)^3 ], [ 216, 187, E(5)^2 ], [ 217, 217, -1 ], 
  [ 218, 211, -E(5)^4 ], [ 219, 228, E(5) ], [ 220, 197, -E(5)^3 ], [ 221, 227, -E(5) ], [ 222, 232, E(5)^2 ], [ 223, 192, -E(5) ], [ 224, 238, E(5)^4 ], [ 225, 201, E(5)^2 ], 
  [ 226, 212, E(5)^3 ], [ 227, 221, -E(5)^4 ], [ 228, 219, E(5)^4 ], [ 229, 231, E(5) ], [ 230, 215, E(5)^2 ], [ 231, 229, E(5)^4 ], [ 232, 222, E(5)^3 ], [ 233, 234, 1 ], 
  [ 234, 233, 1 ], [ 235, 236, E(5) ], [ 236, 235, E(5)^4 ], [ 237, 239, -E(5) ], [ 238, 224, E(5) ], [ 239, 237, -E(5)^4 ], [ 240, 240, -1 ], [ 241, 241, 1 ], [ 242, 243, 1 ], 
  [ 242, 246, -1 ], [ 242, 247, 1 ], [ 243, 241, 1 ], [ 243, 242, 1 ], [ 243, 246, -1 ], [ 243, 247, 1 ], [ 244, 241, 1 ], [ 244, 242, 1 ], [ 244, 243, 1 ], [ 244, 244, -1 ], 
  [ 244, 245, 1 ], [ 244, 246, -2 ], [ 244, 247, 2 ], [ 245, 241, 1 ], [ 245, 245, 1 ], [ 245, 246, -2 ], [ 245, 247, 2 ], [ 246, 241, 1 ], [ 246, 245, 1 ], [ 246, 246, -2 ], 
  [ 246, 247, 1 ], [ 246, 248, 1 ], [ 247, 245, 1 ], [ 247, 246, -1 ], [ 247, 248, 1 ], [ 248, 248, 1 ] ] ],

  [ [ 0, 2, 0, 0, 0, 0, 0, 0 ],
    [ 248, [ 1, 8, 1 ], [ 1, 15, -1 ], [ 1, 140, 1/8 ], [ 1, 147, -1/8 ], [ 1, 148, -1/8 ], [ 1, 155, 1/8 ], [ 2, 73, -8 ], [ 2, 78, -8 ], [ 2, 79, 8 ], 
  [ 2, 84, 8 ], [ 3, 3, 1/4 ], [ 3, 36, -2 ], [ 3, 43, 2 ], [ 3, 123, 1/4 ], [ 3, 156, 1/16 ], [ 3, 163, -1/16 ], [ 3, 241, -1/4 ], [ 3, 242, -1/2 ], [ 3, 244, 1/4 ], [ 3, 248, 1/2 ], 
  [ 4, 4, 1/4 ], [ 4, 11, -1/4 ], [ 4, 12, -3/4 ], [ 4, 19, 3/4 ], [ 4, 142, -3/16 ], [ 4, 149, 1/16 ], [ 5, 5, -1/4 ], [ 5, 125, 1/4 ], [ 5, 244, -1/4 ], 
  [ 5, 245, 1/2 ], [ 5, 246, -1/4 ], [ 6, 6, -1/4 ], [ 6, 13, -3/4 ], [ 6, 14, -1/4 ], [ 6, 21, -3/4 ], [ 6, 136, 3/8 ], [ 6, 144, 1/8 ], [ 7, 7, 1/4 ], [ 7, 31, -1 ], [ 7, 39, -1 ], 
  [ 7, 127, 1/4 ], [ 7, 151, 1/8 ], [ 7, 159, 1/8 ], [ 7, 241, 1/2 ], [ 7, 242, -1/2 ], [ 7, 246, 1/4 ], [ 7, 248, -1/4 ], [ 8, 1, 1/4 ], [ 8, 9, 1/4 ], 
  [ 8, 140, 1/16 ], [ 8, 147, 1/16 ], [ 8, 148, 1/16 ], [ 8, 155, 1/16 ], [ 9, 8, 1 ], [ 9, 15, -1 ], [ 9, 140, -1/8 ], [ 9, 147, 1/8 ], [ 9, 148, 1/8 ], [ 9, 155, -1/8 ], 
  [ 10, 45, 1/2 ], [ 10, 51, 1/2 ], [ 10, 52, -3/2 ], [ 10, 53, -1/2 ], [ 10, 57, -3/2 ], [ 10, 58, -1/2 ], [ 10, 59, 3/2 ], [ 10, 64, 3/2 ], 
  [ 10, 83, -4 ], [ 10, 87, 12 ], [ 10, 88, 4 ], [ 10, 92, -12 ], [ 11, 4, -1/4 ], [ 11, 11, 1/4 ], [ 11, 12, 3/4 ], [ 11, 19, -3/4 ], [ 11, 142, -3/16 ], [ 11, 149, 1/16 ], 
  [ 12, 4, -1/4 ], [ 12, 11, 1/4 ], [ 12, 12, -1/4 ], [ 12, 19, 1/4 ], [ 12, 142, -1/16 ], [ 12, 149, -1/16 ], [ 13, 6, -1/4 ], [ 13, 13, 1/4 ], [ 13, 14, -1/4 ], 
  [ 13, 21, 1/4 ], [ 13, 136, -1/8 ], [ 13, 144, 1/8 ], [ 14, 6, -1/4 ], [ 14, 13, -3/4 ], [ 14, 14, -1/4 ], [ 14, 21, -3/4 ], [ 14, 136, -3/8 ], [ 14, 144, -1/8 ], [ 15, 1, -1/4 ], 
  [ 15, 9, -1/4 ], [ 15, 140, 1/16 ], [ 15, 147, 1/16 ], [ 15, 148, 1/16 ], [ 15, 155, 1/16 ], [ 16, 126, 3/8 ], [ 16, 133, -1/8 ], [ 16, 134, -3/8 ], [ 16, 141, 1/8 ], 
  [ 17, 45, 1/2 ], [ 17, 51, 1/2 ], [ 17, 52, -3/2 ], [ 17, 53, -1/2 ], [ 17, 57, -3/2 ], [ 17, 58, -1/2 ], [ 17, 59, 3/2 ], [ 17, 64, 3/2 ], [ 17, 83, 4 ], [ 17, 87, -12 ], 
  [ 17, 88, -4 ], [ 17, 92, 12 ], [ 18, 45, -1/2 ], [ 18, 51, -1/2 ], [ 18, 52, -1/2 ], [ 18, 53, 1/2 ], [ 18, 57, -1/2 ], [ 18, 58, 1/2 ], [ 18, 59, 1/2 ], [ 18, 64, 1/2 ], 
  [ 18, 83, 4 ], [ 18, 87, 4 ], [ 18, 88, -4 ], [ 18, 92, -4 ], [ 19, 4, 1/4 ], [ 19, 11, -1/4 ], [ 19, 12, 1/4 ], [ 19, 19, -1/4 ], [ 19, 142, -1/16 ], [ 19, 149, -1/16 ], 
  [ 20, 20, 1/4 ], [ 20, 27, -1/4 ], [ 20, 28, 1/4 ], [ 20, 35, -1/4 ], [ 20, 121, 1/8 ], [ 20, 128, 1/16 ], [ 20, 129, -1/8 ], [ 20, 135, 1/16 ], [ 20, 167, -1/32 ], 
  [ 21, 6, -1/4 ], [ 21, 13, 1/4 ], [ 21, 14, -1/4 ], [ 21, 21, 1/4 ], [ 21, 136, 1/8 ], [ 21, 144, -1/8 ], [ 22, 124, -3/16 ], [ 22, 131, -3/16 ], [ 22, 132, -1/16 ], 
  [ 22, 139, -1/16 ], [ 23, 60, 2 ], [ 23, 65, 2 ], [ 23, 67, -6 ], [ 23, 72, -6 ], [ 24, 126, 1/8 ], [ 24, 133, 1/8 ], [ 24, 134, -1/8 ], [ 24, 141, -1/8 ], 
  [ 25, 45, -1/2 ], [ 25, 51, -1/2 ], [ 25, 52, -1/2 ], [ 25, 53, 1/2 ], [ 25, 57, -1/2 ], [ 25, 58, 1/2 ], [ 25, 59, 1/2 ], [ 25, 64, 1/2 ], [ 25, 83, -4 ], [ 25, 87, -4 ], 
  [ 25, 88, 4 ], [ 25, 92, 4 ], [ 26, 26, 1/4 ], [ 26, 33, 1/4 ], [ 26, 34, -1/4 ], [ 26, 41, -1/4 ], [ 26, 66, -1 ], [ 26, 68, 2 ], [ 26, 71, -1 ], 
  [ 26, 74, -2 ], [ 26, 96, 8 ], [ 27, 20, -1/4 ], [ 27, 27, 1/4 ], [ 27, 28, -1/4 ], [ 27, 35, 1/4 ], [ 27, 121, -1/8 ], [ 27, 128, 1/16 ], [ 27, 129, 1/8 ], [ 27, 135, 1/16 ], 
  [ 27, 167, -1/32 ], [ 28, 20, 1/4 ], [ 28, 27, -1/4 ], [ 28, 28, 1/4 ], [ 28, 35, -1/4 ], [ 28, 121, -1/8 ], [ 28, 128, 1/16 ], [ 28, 129, 1/8 ], [ 28, 135, 1/16 ], 
  [ 28, 167, 1/32 ], [ 29, 124, 1/16 ], [ 29, 131, 1/16 ], [ 29, 132, -1/16 ], [ 29, 139, -1/16 ], [ 30, 60, -2 ], [ 30, 65, -2 ], [ 30, 67, -2 ], [ 30, 72, -2 ], [ 31, 7, -1/8 ], 
  [ 31, 127, 1/8 ], [ 31, 151, 1/16 ], [ 31, 159, -1/16 ], [ 31, 246, -1/8 ], [ 31, 247, 1/4 ], [ 31, 248, -1/8 ], [ 32, 63, -2 ], [ 32, 70, 2 ], [ 33, 26, 1/4 ], 
  [ 33, 33, 1/4 ], [ 33, 34, -1/4 ], [ 33, 41, -1/4 ], [ 33, 66, -1 ], [ 33, 68, -2 ], [ 33, 71, -1 ], [ 33, 74, 2 ], [ 33, 96, -8 ], [ 34, 26, -1/4 ], [ 34, 33, -1/4 ], [ 34, 34, 1/4 ],
  [ 34, 41, 1/4 ], [ 34, 66, -1 ], [ 34, 68, -2 ], [ 34, 71, -1 ], [ 34, 74, 2 ], [ 34, 96, 8 ], [ 35, 20, -1/4 ], [ 35, 27, 1/4 ], [ 35, 28, -1/4 ], 
  [ 35, 35, 1/4 ], [ 35, 121, 1/8 ], [ 35, 128, 1/16 ], [ 35, 129, -1/8 ], [ 35, 135, 1/16 ], [ 35, 167, 1/32 ], [ 36, 3, -1/16 ], [ 36, 123, 1/16 ], [ 36, 156, 1/64 ], 
  [ 36, 163, 1/64 ], [ 36, 241, -1/16 ], [ 36, 243, 1/8 ], [ 36, 244, -1/16 ], [ 37, 37, 1/2 ], [ 37, 44, 1/2 ], [ 37, 77, -4 ], [ 38, 42, 1 ], [ 38, 50, 1 ], 
  [ 38, 73, 2 ], [ 38, 78, 2 ], [ 38, 79, 2 ], [ 38, 84, 2 ], [ 39, 7, -1/8 ], [ 39, 127, 1/8 ], [ 39, 151, -1/16 ], [ 39, 159, 1/16 ], [ 39, 246, -1/8 ], [ 39, 247, 1/4 ], 
  [ 39, 248, -1/8 ], [ 40, 40, 1/4 ], [ 40, 48, -3/4 ], [ 40, 49, -1/4 ], [ 40, 55, 3/4 ], [ 40, 76, -1 ], [ 40, 82, 3 ], [ 41, 26, -1/4 ], [ 41, 33, -1/4 ], 
  [ 41, 34, 1/4 ], [ 41, 41, 1/4 ], [ 41, 66, -1 ], [ 41, 68, 2 ], [ 41, 71, -1 ], [ 41, 74, -2 ], [ 41, 96, -8 ], [ 42, 38, 1/4 ], [ 42, 46, -1/4 ], [ 42, 73, 1 ], [ 42, 78, -1 ], 
  [ 42, 79, -1 ], [ 42, 84, 1 ], [ 43, 3, 1/16 ], [ 43, 123, -1/16 ], [ 43, 156, 1/64 ], [ 43, 163, 1/64 ], [ 43, 241, 1/16 ], [ 43, 243, -1/8 ], [ 43, 244, 1/16 ], 
  [ 44, 37, 1/2 ], [ 44, 44, 1/2 ], [ 44, 77, 4 ], [ 45, 10, 1/16 ], [ 45, 17, 1/16 ], [ 45, 18, -3/16 ], [ 45, 25, -3/16 ], [ 45, 45, -1/8 ], [ 45, 51, -1/8 ], [ 45, 52, 3/8 ], 
  [ 45, 53, -1/8 ], [ 45, 56, 1/2 ], [ 45, 57, 3/8 ], [ 45, 58, -1/8 ], [ 45, 59, 3/8 ], [ 45, 62, -3/2 ], [ 45, 64, 3/8 ], [ 45, 83, 1 ], [ 45, 87, -3 ], 
  [ 45, 88, 1 ], [ 45, 92, -3 ], [ 46, 42, -1 ], [ 46, 50, -1 ], [ 46, 73, 2 ], [ 46, 78, 2 ], [ 46, 79, 2 ], [ 46, 84, 2 ], [ 47, 140, -1/32 ], [ 47, 147, -1/32 ], [ 47, 148, 1/32 ], 
  [ 47, 155, 1/32 ], [ 48, 40, -1/4 ], [ 48, 48, -1/4 ], [ 48, 49, 1/4 ], [ 48, 55, 1/4 ], [ 48, 76, 1 ], [ 48, 82, 1 ], [ 49, 40, -1/4 ], [ 49, 48, 3/4 ], 
  [ 49, 49, 1/4 ], [ 49, 55, -3/4 ], [ 49, 76, -1 ], [ 49, 82, 3 ], [ 50, 38, 1/4 ], [ 50, 46, -1/4 ], [ 50, 73, -1 ], [ 50, 78, 1 ], [ 50, 79, 1 ], [ 50, 84, -1 ], [ 51, 10, 1/16 ], 
  [ 51, 17, 1/16 ], [ 51, 18, -3/16 ], [ 51, 25, -3/16 ], [ 51, 45, -1/8 ], [ 51, 51, -1/8 ], [ 51, 52, 3/8 ], [ 51, 53, -1/8 ], [ 51, 56, -1/2 ], [ 51, 57, 3/8 ], 
  [ 51, 58, -1/8 ], [ 51, 59, 3/8 ], [ 51, 62, 3/2 ], [ 51, 64, 3/8 ], [ 51, 83, -1 ], [ 51, 87, 3 ], [ 51, 88, -1 ], [ 51, 92, 3 ], [ 52, 10, -1/16 ], [ 52, 17, -1/16 ], 
  [ 52, 18, -1/16 ], [ 52, 25, -1/16 ], [ 52, 45, 1/8 ], [ 52, 51, 1/8 ], [ 52, 52, 1/8 ], [ 52, 53, 1/8 ], [ 52, 56, -1/2 ], [ 52, 57, 1/8 ], [ 52, 58, 1/8 ], 
  [ 52, 59, 1/8 ], [ 52, 62, -1/2 ], [ 52, 64, 1/8 ], [ 52, 83, -1 ], [ 52, 87, -1 ], [ 52, 88, -1 ], [ 52, 92, -1 ], [ 53, 10, -1/16 ], [ 53, 17, -1/16 ], [ 53, 18, 3/16 ], 
  [ 53, 25, 3/16 ], [ 53, 45, -1/8 ], [ 53, 51, -1/8 ], [ 53, 52, 3/8 ], [ 53, 53, -1/8 ], [ 53, 56, -1/2 ], [ 53, 57, 3/8 ], [ 53, 58, -1/8 ], [ 53, 59, 3/8 ], 
  [ 53, 62, 3/2 ], [ 53, 64, 3/8 ], [ 53, 83, 1 ], [ 53, 87, -3 ], [ 53, 88, 1 ], [ 53, 92, -3 ], [ 54, 54, 1 ], [ 55, 40, 1/4 ], [ 55, 48, 1/4 ], [ 55, 49, -1/4 ], [ 55, 55, -1/4 ], 
  [ 55, 76, 1 ], [ 55, 82, 1 ], [ 56, 45, 1/8 ], [ 56, 51, -1/8 ], [ 56, 52, -3/8 ], [ 56, 53, -1/8 ], [ 56, 57, 3/8 ], [ 56, 58, 1/8 ], [ 56, 59, 3/8 ], 
  [ 56, 64, -3/8 ], [ 57, 10, -1/16 ], [ 57, 17, -1/16 ], [ 57, 18, -1/16 ], [ 57, 25, -1/16 ], [ 57, 45, 1/8 ], [ 57, 51, 1/8 ], [ 57, 52, 1/8 ], [ 57, 53, 1/8 ], [ 57, 56, 1/2 ], 
  [ 57, 57, 1/8 ], [ 57, 58, 1/8 ], [ 57, 59, 1/8 ], [ 57, 62, 1/2 ], [ 57, 64, 1/8 ], [ 57, 83, 1 ], [ 57, 87, 1 ], [ 57, 88, 1 ], [ 57, 92, 1 ], 
  [ 58, 10, -1/16 ], [ 58, 17, -1/16 ], [ 58, 18, 3/16 ], [ 58, 25, 3/16 ], [ 58, 45, -1/8 ], [ 58, 51, -1/8 ], [ 58, 52, 3/8 ], [ 58, 53, -1/8 ], [ 58, 56, 1/2 ], [ 58, 57, 3/8 ], 
  [ 58, 58, -1/8 ], [ 58, 59, 3/8 ], [ 58, 62, -3/2 ], [ 58, 64, 3/8 ], [ 58, 83, -1 ], [ 58, 87, 3 ], [ 58, 88, -1 ], [ 58, 92, 3 ], [ 59, 10, 1/16 ], 
  [ 59, 17, 1/16 ], [ 59, 18, 1/16 ], [ 59, 25, 1/16 ], [ 59, 45, 1/8 ], [ 59, 51, 1/8 ], [ 59, 52, 1/8 ], [ 59, 53, 1/8 ], [ 59, 56, 1/2 ], [ 59, 57, 1/8 ], [ 59, 58, 1/8 ], 
  [ 59, 59, 1/8 ], [ 59, 62, 1/2 ], [ 59, 64, 1/8 ], [ 59, 83, -1 ], [ 59, 87, -1 ], [ 59, 88, -1 ], [ 59, 92, -1 ], [ 60, 23, 1/16 ], [ 60, 30, -3/16 ], 
  [ 60, 60, 1/4 ], [ 60, 65, -1/4 ], [ 60, 67, -3/4 ], [ 60, 72, 3/4 ], [ 61, 61, 1 ], [ 62, 45, -1/8 ], [ 62, 51, 1/8 ], [ 62, 52, -1/8 ], [ 62, 53, 1/8 ], [ 62, 57, 1/8 ], 
  [ 62, 58, -1/8 ], [ 62, 59, 1/8 ], [ 62, 64, -1/8 ], [ 63, 32, -1/4 ], [ 63, 63, 1/2 ], [ 63, 70, 1/2 ], [ 64, 10, 1/16 ], [ 64, 17, 1/16 ], [ 64, 18, 1/16 ], 
  [ 64, 25, 1/16 ], [ 64, 45, 1/8 ], [ 64, 51, 1/8 ], [ 64, 52, 1/8 ], [ 64, 53, 1/8 ], [ 64, 56, -1/2 ], [ 64, 57, 1/8 ], [ 64, 58, 1/8 ], [ 64, 59, 1/8 ], [ 64, 62, -1/2 ], 
  [ 64, 64, 1/8 ], [ 64, 83, 1 ], [ 64, 87, 1 ], [ 64, 88, 1 ], [ 64, 92, 1 ], [ 65, 23, 1/16 ], [ 65, 30, -3/16 ], [ 65, 60, -1/4 ], [ 65, 65, 1/4 ], [ 65, 67, 3/4 ], 
  [ 65, 72, -3/4 ], [ 66, 26, -1/8 ], [ 66, 33, -1/8 ], [ 66, 34, -1/8 ], [ 66, 41, -1/8 ], [ 66, 68, -1 ], [ 66, 74, -1 ], [ 67, 23, -1/16 ], [ 67, 30, -1/16 ], [ 67, 60, -1/4 ], 
  [ 67, 65, 1/4 ], [ 67, 67, -1/4 ], [ 67, 72, 1/4 ], [ 68, 26, 1/16 ], [ 68, 33, -1/16 ], [ 68, 34, -1/16 ], [ 68, 41, 1/16 ], [ 68, 66, -1/4 ], [ 68, 71, 1/4 ], 
  [ 69, 94, -2 ], [ 69, 98, 2 ], [ 69, 99, 2 ], [ 69, 102, -2 ], [ 69, 111, 8 ], [ 69, 114, -8 ], [ 70, 32, 1/4 ], [ 70, 63, 1/2 ], [ 70, 70, 1/2 ], [ 71, 26, -1/8 ], [ 71, 33, -1/8 ], 
  [ 71, 34, -1/8 ], [ 71, 41, -1/8 ], [ 71, 68, 1 ], [ 71, 74, 1 ], [ 72, 23, -1/16 ], [ 72, 30, -1/16 ], [ 72, 60, 1/4 ], [ 72, 65, -1/4 ], [ 72, 67, 1/4 ], 
  [ 72, 72, -1/4 ], [ 73, 2, -1/32 ], [ 73, 38, 1/16 ], [ 73, 42, 1/8 ], [ 73, 46, 1/16 ], [ 73, 50, -1/8 ], [ 73, 73, 1/4 ], [ 73, 78, -1/4 ], [ 73, 79, 1/4 ], [ 73, 84, -1/4 ], 
  [ 74, 26, -1/16 ], [ 74, 33, 1/16 ], [ 74, 34, 1/16 ], [ 74, 41, -1/16 ], [ 74, 66, -1/4 ], [ 74, 71, 1/4 ], [ 75, 94, 2 ], [ 75, 98, -2 ], [ 75, 99, -2 ], 
  [ 75, 102, 2 ], [ 75, 111, 8 ], [ 75, 114, -8 ], [ 76, 40, -1/8 ], [ 76, 48, 3/8 ], [ 76, 49, -1/8 ], [ 76, 55, 3/8 ], [ 77, 37, -1/8 ], [ 77, 44, 1/8 ], [ 78, 2, -1/32 ], 
  [ 78, 38, 1/16 ], [ 78, 42, -1/8 ], [ 78, 46, 1/16 ], [ 78, 50, 1/8 ], [ 78, 73, -1/4 ], [ 78, 78, 1/4 ], [ 78, 79, -1/4 ], [ 78, 84, 1/4 ], 
  [ 79, 2, 1/32 ], [ 79, 38, 1/16 ], [ 79, 42, -1/8 ], [ 79, 46, 1/16 ], [ 79, 50, 1/8 ], [ 79, 73, 1/4 ], [ 79, 78, -1/4 ], [ 79, 79, 1/4 ], [ 79, 84, -1/4 ], [ 80, 103, 2 ], 
  [ 80, 105, -2 ], [ 80, 106, -6 ], [ 80, 108, 6 ], [ 81, 101, -2 ], [ 81, 104, 2 ], [ 82, 40, 1/8 ], [ 82, 48, 1/8 ], [ 82, 49, 1/8 ], [ 82, 55, 1/8 ], 
  [ 83, 10, -1/64 ], [ 83, 17, 1/64 ], [ 83, 18, 3/64 ], [ 83, 25, -3/64 ], [ 83, 45, 1/32 ], [ 83, 51, -1/32 ], [ 83, 52, -3/32 ], [ 83, 53, 1/32 ], [ 83, 57, 3/32 ], [ 83, 58, -1/32 ],
  [ 83, 59, -3/32 ], [ 83, 64, 3/32 ], [ 84, 2, 1/32 ], [ 84, 38, 1/16 ], [ 84, 42, 1/8 ], [ 84, 46, 1/16 ], [ 84, 50, -1/8 ], [ 84, 73, -1/4 ], [ 84, 78, 1/4 ], 
  [ 84, 79, -1/4 ], [ 84, 84, 1/4 ], [ 85, 103, -2 ], [ 85, 105, 2 ], [ 85, 106, -2 ], [ 85, 108, 2 ], [ 86, 86, 1/4 ], [ 86, 90, -3/4 ], [ 86, 91, -1/4 ], [ 86, 95, 3/4 ], 
  [ 86, 107, -1 ], [ 86, 109, 3 ], [ 87, 10, 1/64 ], [ 87, 17, -1/64 ], [ 87, 18, 1/64 ], [ 87, 25, -1/64 ], [ 87, 45, -1/32 ], [ 87, 51, 1/32 ], [ 87, 52, -1/32 ], [ 87, 53, -1/32 ], 
  [ 87, 57, 1/32 ], [ 87, 58, 1/32 ], [ 87, 59, -1/32 ], [ 87, 64, 1/32 ], [ 88, 10, 1/64 ], [ 88, 17, -1/64 ], [ 88, 18, -3/64 ], [ 88, 25, 3/64 ], [ 88, 45, 1/32 ], 
  [ 88, 51, -1/32 ], [ 88, 52, -3/32 ], [ 88, 53, 1/32 ], [ 88, 57, 3/32 ], [ 88, 58, -1/32 ], [ 88, 59, -3/32 ], [ 88, 64, 3/32 ], [ 89, 89, 1/2 ], [ 89, 93, -1/2 ], [ 89, 110, 4 ], 
  [ 90, 86, -1/4 ], [ 90, 90, -1/4 ], [ 90, 91, 1/4 ], [ 90, 95, 1/4 ], [ 90, 107, 1 ], [ 90, 109, 1 ], [ 91, 86, -1/4 ], [ 91, 90, 3/4 ], [ 91, 91, 1/4 ], 
  [ 91, 95, -3/4 ], [ 91, 107, -1 ], [ 91, 109, 3 ], [ 92, 10, -1/64 ], [ 92, 17, 1/64 ], [ 92, 18, -1/64 ], [ 92, 25, 1/64 ], [ 92, 45, -1/32 ], [ 92, 51, 1/32 ], [ 92, 52, -1/32 ], 
  [ 92, 53, -1/32 ], [ 92, 57, 1/32 ], [ 92, 58, 1/32 ], [ 92, 59, -1/32 ], [ 92, 64, 1/32 ], [ 93, 89, -1/2 ], [ 93, 93, 1/2 ], [ 93, 110, 4 ], [ 94, 69, -1/16 ], 
  [ 94, 75, 1/16 ], [ 94, 94, -1/4 ], [ 94, 97, -1/4 ], [ 94, 98, -1/4 ], [ 94, 99, 1/4 ], [ 94, 102, 1/4 ], [ 94, 111, 1 ], [ 94, 114, 1 ], [ 95, 86, 1/4 ], [ 95, 90, 1/4 ], 
  [ 95, 91, -1/4 ], [ 95, 95, -1/4 ], [ 95, 107, 1 ], [ 95, 109, 1 ], [ 96, 26, 1/32 ], [ 96, 33, -1/32 ], [ 96, 34, 1/32 ], [ 96, 41, -1/32 ], [ 97, 94, -1 ], [ 97, 98, 1 ], 
  [ 97, 99, -1 ], [ 97, 102, 1 ], [ 98, 69, 1/16 ], [ 98, 75, -1/16 ], [ 98, 94, -1/4 ], [ 98, 97, 1/4 ], [ 98, 98, -1/4 ], [ 98, 99, 1/4 ], [ 98, 102, 1/4 ], [ 98, 111, 1 ], 
  [ 98, 114, 1 ], [ 99, 69, 1/16 ], [ 99, 75, -1/16 ], [ 99, 94, 1/4 ], [ 99, 97, -1/4 ], [ 99, 98, 1/4 ], [ 99, 99, -1/4 ], [ 99, 102, -1/4 ], [ 99, 111, 1 ], [ 99, 114, 1 ], 
  [ 100, 100, 1 ], [ 101, 81, -1/4 ], [ 101, 101, 1/2 ], [ 101, 104, 1/2 ], [ 102, 69, -1/16 ], [ 102, 75, 1/16 ], [ 102, 94, 1/4 ], [ 102, 97, 1/4 ], [ 102, 98, 1/4 ], 
  [ 102, 99, -1/4 ], [ 102, 102, -1/4 ], [ 102, 111, 1 ], [ 102, 114, 1 ], [ 103, 80, 1/16 ], [ 103, 85, -3/16 ], [ 103, 103, 1/4 ], [ 103, 105, 1/4 ], [ 103, 106, -3/4 ], 
  [ 103, 108, -3/4 ], [ 104, 81, 1/4 ], [ 104, 101, 1/2 ], [ 104, 104, 1/2 ], [ 105, 80, -1/16 ], [ 105, 85, 3/16 ], [ 105, 103, 1/4 ], [ 105, 105, 1/4 ], [ 105, 106, -3/4 ], 
  [ 105, 108, -3/4 ], [ 106, 80, -1/16 ], [ 106, 85, -1/16 ], [ 106, 103, -1/4 ], [ 106, 105, -1/4 ], [ 106, 106, -1/4 ], [ 106, 108, -1/4 ], [ 107, 86, -1/8 ], [ 107, 90, 3/8 ], 
  [ 107, 91, -1/8 ], [ 107, 95, 3/8 ], [ 108, 80, 1/16 ], [ 108, 85, 1/16 ], [ 108, 103, -1/4 ], [ 108, 105, -1/4 ], [ 108, 106, -1/4 ], [ 108, 108, -1/4 ], [ 109, 86, 1/8 ], 
  [ 109, 90, 1/8 ], [ 109, 91, 1/8 ], [ 109, 95, 1/8 ], [ 110, 89, 1/8 ], [ 110, 93, 1/8 ], [ 111, 69, 1/32 ], [ 111, 75, 1/32 ], [ 111, 94, 1/8 ], [ 111, 98, 1/8 ], 
  [ 111, 99, 1/8 ], [ 111, 102, 1/8 ], [ 112, 118, 2 ], [ 112, 119, 2 ], [ 113, 113, 1/2 ], [ 113, 115, -1/2 ], [ 113, 120, -4 ], [ 114, 69, -1/32 ], [ 114, 75, -1/32 ], 
  [ 114, 94, 1/8 ], [ 114, 98, 1/8 ], [ 114, 99, 1/8 ], [ 114, 102, 1/8 ], [ 115, 113, -1/2 ], [ 115, 115, 1/2 ], [ 115, 120, -4 ], [ 116, 116, 1/2 ], [ 116, 117, 3/2 ], 
  [ 117, 116, 1/2 ], [ 117, 117, -1/2 ], [ 118, 112, 1/4 ], [ 118, 118, 1/2 ], [ 118, 119, -1/2 ], [ 119, 112, 1/4 ], [ 119, 118, -1/2 ], [ 119, 119, 1/2 ], [ 120, 113, -1/8 ], 
  [ 120, 115, -1/8 ], [ 121, 20, 1 ], [ 121, 27, -1 ], [ 121, 28, -1 ], [ 121, 35, 1 ], [ 121, 128, 1/4 ], [ 121, 135, -1/4 ], [ 122, 193, -1/32 ], [ 122, 198, -1/32 ], 
  [ 122, 199, 1/32 ], [ 122, 204, 1/32 ], [ 123, 3, 1/4 ], [ 123, 36, 2 ], [ 123, 43, -2 ], [ 123, 123, 1/4 ], [ 123, 156, -1/16 ], [ 123, 163, 1/16 ], [ 123, 241, -1/4 ], 
  [ 123, 242, -1/2 ], [ 123, 244, 1/4 ], [ 123, 248, 1/2 ], [ 124, 22, -2 ], [ 124, 29, 2 ], [ 124, 124, 1/4 ], [ 124, 131, -1/4 ], [ 124, 132, -1/4 ], [ 124, 139, 1/4 ], 
  [ 125, 5, 9/4 ], [ 125, 125, -1/4 ], [ 125, 244, -3/4 ], [ 125, 245, 3/2 ], [ 125, 246, -3/4 ], [ 126, 16, 1 ], [ 126, 24, 1 ], [ 126, 126, -1/4 ], [ 126, 133, -1/4 ], 
  [ 126, 134, -1/4 ], [ 126, 141, -1/4 ], [ 127, 7, 1/4 ], [ 127, 31, 1 ], [ 127, 39, 1 ], [ 127, 127, 1/4 ], [ 127, 151, -1/8 ], [ 127, 159, -1/8 ], [ 127, 241, 1/2 ], 
  [ 127, 242, -1/2 ], [ 127, 246, 1/4 ], [ 127, 248, -1/4 ], [ 128, 20, 2 ], [ 128, 27, 2 ], [ 128, 28, 2 ], [ 128, 35, 2 ], [ 128, 121, 1 ], [ 128, 129, 1 ], [ 129, 20, -1 ], 
  [ 129, 27, 1 ], [ 129, 28, 1 ], [ 129, 35, -1 ], [ 129, 128, 1/4 ], [ 129, 135, -1/4 ], [ 130, 165, 1/16 ], [ 130, 171, 1/16 ], [ 130, 172, -1/16 ], [ 130, 173, -1/16 ], 
  [ 130, 177, -1/16 ], [ 130, 178, -1/16 ], [ 130, 179, 1/16 ], [ 130, 184, 1/16 ], [ 130, 203, -1/64 ], [ 130, 207, 1/64 ], [ 130, 208, 1/64 ], [ 130, 212, -1/64 ], [ 131, 22, -2 ], 
  [ 131, 29, 2 ], [ 131, 124, -1/4 ], [ 131, 131, 1/4 ], [ 131, 132, 1/4 ], [ 131, 139, -1/4 ], [ 132, 22, -2 ], [ 132, 29, -6 ], [ 132, 124, -3/4 ], [ 132, 131, 3/4 ], 
  [ 132, 132, -1/4 ], [ 132, 139, 1/4 ], [ 133, 16, -1 ], [ 133, 24, 3 ], [ 133, 126, -3/4 ], [ 133, 133, 1/4 ], [ 133, 134, -3/4 ], [ 133, 141, 1/4 ], [ 134, 16, -1 ], [ 134, 24, -1 ], 
  [ 134, 126, -1/4 ], [ 134, 133, -1/4 ], [ 134, 134, -1/4 ], [ 134, 141, -1/4 ], [ 135, 20, 2 ], [ 135, 27, 2 ], [ 135, 28, 2 ], [ 135, 35, 2 ], [ 135, 121, -1 ], 
  [ 135, 129, -1 ], [ 136, 6, 1 ], [ 136, 13, -1 ], [ 136, 14, -1 ], [ 136, 21, 1 ], [ 137, 165, 1/16 ], [ 137, 171, 1/16 ], [ 137, 172, -1/16 ], [ 137, 173, -1/16 ], 
  [ 137, 177, -1/16 ], [ 137, 178, -1/16 ], [ 137, 179, 1/16 ], [ 137, 184, 1/16 ], [ 137, 203, 1/64 ], [ 137, 207, -1/64 ], [ 137, 208, -1/64 ], [ 137, 212, 1/64 ], [ 138, 165, -3/16 ],
  [ 138, 171, -3/16 ], [ 138, 172, -1/16 ], [ 138, 173, 3/16 ], [ 138, 177, -1/16 ], [ 138, 178, 3/16 ], [ 138, 179, 1/16 ], [ 138, 184, 1/16 ], [ 138, 203, 3/64 ], [ 138, 207, 1/64 ], 
  [ 138, 208, -3/64 ], [ 138, 212, -1/64 ], [ 139, 22, -2 ], [ 139, 29, -6 ], [ 139, 124, 3/4 ], [ 139, 131, -3/4 ], [ 139, 132, 1/4 ], [ 139, 139, -1/4 ], [ 140, 1, 1 ], 
  [ 140, 8, 2 ], [ 140, 9, -1 ], [ 140, 15, 2 ], [ 140, 47, -8 ], [ 140, 140, 1/4 ], [ 140, 147, -1/4 ], [ 140, 148, 1/4 ], [ 140, 155, -1/4 ], [ 141, 16, 1 ], [ 141, 24, -3 ], 
  [ 141, 126, -3/4 ], [ 141, 133, 1/4 ], [ 141, 134, -3/4 ], [ 141, 141, 1/4 ], [ 142, 4, -2 ], [ 142, 11, -2 ], [ 142, 12, -2 ], [ 142, 19, -2 ], [ 143, 180, 1/16 ], 
  [ 143, 185, 1/16 ], [ 143, 187, -1/16 ], [ 143, 192, -1/16 ], [ 144, 6, 1 ], [ 144, 13, 3 ], [ 144, 14, -1 ], [ 144, 21, -3 ], [ 145, 165, -3/16 ], [ 145, 171, -3/16 ], 
  [ 145, 172, -1/16 ], [ 145, 173, 3/16 ], [ 145, 177, -1/16 ], [ 145, 178, 3/16 ], [ 145, 179, 1/16 ], [ 145, 184, 1/16 ], [ 145, 203, -3/64 ], [ 145, 207, -1/64 ], [ 145, 208, 3/64 ], 
  [ 145, 212, 1/64 ], [ 146, 146, 1/4 ], [ 146, 153, 1/4 ], [ 146, 154, -1/4 ], [ 146, 161, -1/4 ], [ 146, 186, -1/8 ], [ 146, 188, 1/16 ], [ 146, 191, -1/8 ], [ 146, 194, -1/16 ], 
  [ 146, 216, 1/32 ], [ 147, 1, -1 ], [ 147, 8, 2 ], [ 147, 9, 1 ], [ 147, 15, 2 ], [ 147, 47, -8 ], [ 147, 140, -1/4 ], [ 147, 147, 1/4 ], [ 147, 148, -1/4 ], 
  [ 147, 155, 1/4 ], [ 148, 1, -1 ], [ 148, 8, 2 ], [ 148, 9, 1 ], [ 148, 15, 2 ], [ 148, 47, 8 ], [ 148, 140, 1/4 ], [ 148, 147, -1/4 ], [ 148, 148, 1/4 ], [ 148, 155, -1/4 ], 
  [ 149, 4, 2 ], [ 149, 11, 2 ], [ 149, 12, -6 ], [ 149, 19, -6 ], [ 150, 180, -3/16 ], [ 150, 185, -3/16 ], [ 150, 187, -1/16 ], [ 150, 192, -1/16 ], [ 151, 7, 1 ], 
  [ 151, 31, 4 ], [ 151, 39, -4 ], [ 151, 127, -1 ], [ 151, 246, -1 ], [ 151, 247, 2 ], [ 151, 248, -1 ], [ 152, 183, -1/4 ], [ 152, 190, 1/4 ], [ 153, 146, 1/4 ], [ 153, 153, 1/4 ], 
  [ 153, 154, -1/4 ], [ 153, 161, -1/4 ], [ 153, 186, -1/8 ], [ 153, 188, -1/16 ], [ 153, 191, -1/8 ], [ 153, 194, 1/16 ], [ 153, 216, -1/32 ], [ 154, 146, -1/4 ], [ 154, 153, -1/4 ], 
  [ 154, 154, 1/4 ], [ 154, 161, 1/4 ], [ 154, 186, -1/8 ], [ 154, 188, -1/16 ], [ 154, 191, -1/8 ], [ 154, 194, 1/16 ], [ 154, 216, 1/32 ], [ 155, 1, 1 ], [ 155, 8, 2 ], 
  [ 155, 9, -1 ], [ 155, 15, 2 ], [ 155, 47, 8 ], [ 155, 140, -1/4 ], [ 155, 147, 1/4 ], [ 155, 148, -1/4 ], [ 155, 155, 1/4 ], [ 156, 3, 2 ], [ 156, 36, 16 ], [ 156, 43, 16 ], 
  [ 156, 123, -2 ], [ 156, 241, -2 ], [ 156, 243, 4 ], [ 156, 244, -2 ], [ 157, 157, 1/2 ], [ 157, 164, 1/2 ], [ 157, 197, -1/8 ], [ 158, 162, 1/4 ], [ 158, 170, 1/4 ], 
  [ 158, 193, 1/16 ], [ 158, 198, 1/16 ], [ 158, 199, 1/16 ], [ 158, 204, 1/16 ], [ 159, 7, 1 ], [ 159, 31, -4 ], [ 159, 39, 4 ], [ 159, 127, -1 ], [ 159, 246, -1 ], [ 159, 247, 2 ], 
  [ 159, 248, -1 ], [ 160, 160, 1/4 ], [ 160, 168, -1/4 ], [ 160, 169, -1/4 ], [ 160, 175, 1/4 ], [ 160, 196, -1/8 ], [ 160, 202, 1/8 ], [ 161, 146, -1/4 ], [ 161, 153, -1/4 ], 
  [ 161, 154, 1/4 ], [ 161, 161, 1/4 ], [ 161, 186, -1/8 ], [ 161, 188, 1/16 ], [ 161, 191, -1/8 ], [ 161, 194, -1/16 ], [ 161, 216, -1/32 ], [ 162, 158, 1 ], [ 162, 166, -1 ], 
  [ 162, 193, 1/8 ], [ 162, 198, -1/8 ], [ 162, 199, -1/8 ], [ 162, 204, 1/8 ], [ 163, 3, -2 ], [ 163, 36, 16 ], [ 163, 43, 16 ], [ 163, 123, 2 ], [ 163, 241, 2 ], [ 163, 243, -4 ], 
  [ 163, 244, 2 ], [ 164, 157, 1/2 ], [ 164, 164, 1/2 ], [ 164, 197, 1/8 ], [ 165, 130, 1/2 ], [ 165, 137, 1/2 ], [ 165, 138, -1/2 ], [ 165, 145, -1/2 ], [ 165, 165, -1/8 ], 
  [ 165, 171, -1/8 ], [ 165, 172, 1/8 ], [ 165, 173, -1/8 ], [ 165, 176, 1/8 ], [ 165, 177, 1/8 ], [ 165, 178, -1/8 ], [ 165, 179, 1/8 ], [ 165, 182, -1/8 ], [ 165, 184, 1/8 ], 
  [ 165, 203, 1/32 ], [ 165, 207, -1/32 ], [ 165, 208, 1/32 ], [ 165, 212, -1/32 ], [ 166, 162, -1/4 ], [ 166, 170, -1/4 ], [ 166, 193, 1/16 ], [ 166, 198, 1/16 ], [ 166, 199, 1/16 ], 
  [ 166, 204, 1/16 ], [ 167, 20, -8 ], [ 167, 27, -8 ], [ 167, 28, 8 ], [ 167, 35, 8 ], [ 168, 160, -3/4 ], [ 168, 168, -1/4 ], [ 168, 169, 3/4 ], [ 168, 175, 1/4 ], 
  [ 168, 196, 3/8 ], [ 168, 202, 1/8 ], [ 169, 160, -1/4 ], [ 169, 168, 1/4 ], [ 169, 169, 1/4 ], [ 169, 175, -1/4 ], [ 169, 196, -1/8 ], [ 169, 202, 1/8 ], [ 170, 158, 1 ], 
  [ 170, 166, -1 ], [ 170, 193, -1/8 ], [ 170, 198, 1/8 ], [ 170, 199, 1/8 ], [ 170, 204, -1/8 ], [ 171, 130, 1/2 ], [ 171, 137, 1/2 ], [ 171, 138, -1/2 ], [ 171, 145, -1/2 ], 
  [ 171, 165, -1/8 ], [ 171, 171, -1/8 ], [ 171, 172, 1/8 ], [ 171, 173, -1/8 ], [ 171, 176, -1/8 ], [ 171, 177, 1/8 ], [ 171, 178, -1/8 ], [ 171, 179, 1/8 ], [ 171, 182, 1/8 ], 
  [ 171, 184, 1/8 ], [ 171, 203, -1/32 ], [ 171, 207, 1/32 ], [ 171, 208, -1/32 ], [ 171, 212, 1/32 ], [ 172, 130, -3/2 ], [ 172, 137, -3/2 ], [ 172, 138, -1/2 ], [ 172, 145, -1/2 ], 
  [ 172, 165, 3/8 ], [ 172, 171, 3/8 ], [ 172, 172, 1/8 ], [ 172, 173, 3/8 ], [ 172, 176, -3/8 ], [ 172, 177, 1/8 ], [ 172, 178, 3/8 ], [ 172, 179, 1/8 ], [ 172, 182, -1/8 ], 
  [ 172, 184, 1/8 ], [ 172, 203, -3/32 ], [ 172, 207, -1/32 ], [ 172, 208, -3/32 ], [ 172, 212, -1/32 ], [ 173, 130, -1/2 ], [ 173, 137, -1/2 ], [ 173, 138, 1/2 ], [ 173, 145, 1/2 ], 
  [ 173, 165, -1/8 ], [ 173, 171, -1/8 ], [ 173, 172, 1/8 ], [ 173, 173, -1/8 ], [ 173, 176, -1/8 ], [ 173, 177, 1/8 ], [ 173, 178, -1/8 ], [ 173, 179, 1/8 ], [ 173, 182, 1/8 ], 
  [ 173, 184, 1/8 ], [ 173, 203, 1/32 ], [ 173, 207, -1/32 ], [ 173, 208, 1/32 ], [ 173, 212, -1/32 ], [ 174, 174, 1 ], [ 175, 160, 3/4 ], [ 175, 168, 1/4 ], [ 175, 169, -3/4 ], 
  [ 175, 175, -1/4 ], [ 175, 196, 3/8 ], [ 175, 202, 1/8 ], [ 176, 165, 1/2 ], [ 176, 171, -1/2 ], [ 176, 172, -1/2 ], [ 176, 173, -1/2 ], [ 176, 177, 1/2 ], [ 176, 178, 1/2 ], 
  [ 176, 179, 1/2 ], [ 176, 184, -1/2 ], [ 177, 130, -3/2 ], [ 177, 137, -3/2 ], [ 177, 138, -1/2 ], [ 177, 145, -1/2 ], [ 177, 165, 3/8 ], [ 177, 171, 3/8 ], [ 177, 172, 1/8 ], 
  [ 177, 173, 3/8 ], [ 177, 176, 3/8 ], [ 177, 177, 1/8 ], [ 177, 178, 3/8 ], [ 177, 179, 1/8 ], [ 177, 182, 1/8 ], [ 177, 184, 1/8 ], [ 177, 203, 3/32 ], [ 177, 207, 1/32 ], 
  [ 177, 208, 3/32 ], [ 177, 212, 1/32 ], [ 178, 130, -1/2 ], [ 178, 137, -1/2 ], [ 178, 138, 1/2 ], [ 178, 145, 1/2 ], [ 178, 165, -1/8 ], [ 178, 171, -1/8 ], [ 178, 172, 1/8 ], 
  [ 178, 173, -1/8 ], [ 178, 176, 1/8 ], [ 178, 177, 1/8 ], [ 178, 178, -1/8 ], [ 178, 179, 1/8 ], [ 178, 182, -1/8 ], [ 178, 184, 1/8 ], [ 178, 203, -1/32 ], [ 178, 207, 1/32 ], 
  [ 178, 208, -1/32 ], [ 178, 212, 1/32 ], [ 179, 130, 3/2 ], [ 179, 137, 3/2 ], [ 179, 138, 1/2 ], [ 179, 145, 1/2 ], [ 179, 165, 3/8 ], [ 179, 171, 3/8 ], [ 179, 172, 1/8 ], 
  [ 179, 173, 3/8 ], [ 179, 176, 3/8 ], [ 179, 177, 1/8 ], [ 179, 178, 3/8 ], [ 179, 179, 1/8 ], [ 179, 182, 1/8 ], [ 179, 184, 1/8 ], [ 179, 203, -3/32 ], [ 179, 207, -1/32 ], 
  [ 179, 208, -3/32 ], [ 179, 212, -1/32 ], [ 180, 143, 2 ], [ 180, 150, -2 ], [ 180, 180, 1/4 ], [ 180, 185, -1/4 ], [ 180, 187, -1/4 ], [ 180, 192, 1/4 ], [ 181, 181, 1 ], 
  [ 182, 165, -3/2 ], [ 182, 171, 3/2 ], [ 182, 172, -1/2 ], [ 182, 173, 3/2 ], [ 182, 177, 1/2 ], [ 182, 178, -3/2 ], [ 182, 179, 1/2 ], [ 182, 184, -1/2 ], [ 183, 152, -2 ], 
  [ 183, 183, 1/2 ], [ 183, 190, 1/2 ], [ 184, 130, 3/2 ], [ 184, 137, 3/2 ], [ 184, 138, 1/2 ], [ 184, 145, 1/2 ], [ 184, 165, 3/8 ], [ 184, 171, 3/8 ], [ 184, 172, 1/8 ], 
  [ 184, 173, 3/8 ], [ 184, 176, -3/8 ], [ 184, 177, 1/8 ], [ 184, 178, 3/8 ], [ 184, 179, 1/8 ], [ 184, 182, -1/8 ], [ 184, 184, 1/8 ], [ 184, 203, 3/32 ], [ 184, 207, 1/32 ], 
  [ 184, 208, 3/32 ], [ 184, 212, 1/32 ], [ 185, 143, 2 ], [ 185, 150, -2 ], [ 185, 180, -1/4 ], [ 185, 185, 1/4 ], [ 185, 187, 1/4 ], [ 185, 192, -1/4 ], [ 186, 146, -1 ], 
  [ 186, 153, -1 ], [ 186, 154, -1 ], [ 186, 161, -1 ], [ 186, 188, -1/4 ], [ 186, 194, -1/4 ], [ 187, 143, -6 ], [ 187, 150, -2 ], [ 187, 180, -3/4 ], [ 187, 185, 3/4 ], 
  [ 187, 187, -1/4 ], [ 187, 192, 1/4 ], [ 188, 146, 2 ], [ 188, 153, -2 ], [ 188, 154, -2 ], [ 188, 161, 2 ], [ 188, 186, -1 ], [ 188, 191, 1 ], [ 189, 214, -1/16 ], 
  [ 189, 218, 1/16 ], [ 189, 219, 1/16 ], [ 189, 222, -1/16 ], [ 189, 231, 1/32 ], [ 189, 234, -1/32 ], [ 190, 152, 2 ], [ 190, 183, 1/2 ], [ 190, 190, 1/2 ], [ 191, 146, -1 ], 
  [ 191, 153, -1 ], [ 191, 154, -1 ], [ 191, 161, -1 ], [ 191, 188, 1/4 ], [ 191, 194, 1/4 ], [ 192, 143, -6 ], [ 192, 150, -2 ], [ 192, 180, 3/4 ], [ 192, 185, -3/4 ], 
  [ 192, 187, 1/4 ], [ 192, 192, -1/4 ], [ 193, 122, -8 ], [ 193, 158, 2 ], [ 193, 162, 1 ], [ 193, 166, 2 ], [ 193, 170, -1 ], [ 193, 193, 1/4 ], [ 193, 198, -1/4 ], [ 193, 199, 1/4 ], 
  [ 193, 204, -1/4 ], [ 194, 146, -2 ], [ 194, 153, 2 ], [ 194, 154, 2 ], [ 194, 161, -2 ], [ 194, 186, -1 ], [ 194, 191, 1 ], [ 195, 214, 1/16 ], [ 195, 218, -1/16 ], 
  [ 195, 219, -1/16 ], [ 195, 222, 1/16 ], [ 195, 231, 1/32 ], [ 195, 234, -1/32 ], [ 196, 160, -1 ], [ 196, 168, 1 ], [ 196, 169, -1 ], [ 196, 175, 1 ], [ 197, 157, -4 ], 
  [ 197, 164, 4 ], [ 198, 122, -8 ], [ 198, 158, 2 ], [ 198, 162, -1 ], [ 198, 166, 2 ], [ 198, 170, 1 ], [ 198, 193, -1/4 ], [ 198, 198, 1/4 ], [ 198, 199, -1/4 ], 
  [ 198, 204, 1/4 ], [ 199, 122, 8 ], [ 199, 158, 2 ], [ 199, 162, -1 ], [ 199, 166, 2 ], [ 199, 170, 1 ], [ 199, 193, 1/4 ], [ 199, 198, -1/4 ], [ 199, 199, 1/4 ], [ 199, 204, -1/4 ], 
  [ 200, 223, 1/16 ], [ 200, 225, -1/16 ], [ 200, 226, -1/16 ], [ 200, 228, 1/16 ], [ 201, 221, -1/4 ], [ 201, 224, 1/4 ], [ 202, 160, 3 ], [ 202, 168, 1 ], [ 202, 169, 3 ], 
  [ 202, 175, 1 ], [ 203, 130, -4 ], [ 203, 137, 4 ], [ 203, 138, 4 ], [ 203, 145, -4 ], [ 203, 165, 1 ], [ 203, 171, -1 ], [ 203, 172, -1 ], [ 203, 173, 1 ], [ 203, 177, 1 ], 
  [ 203, 178, -1 ], [ 203, 179, -1 ], [ 203, 184, 1 ], [ 204, 122, 8 ], [ 204, 158, 2 ], [ 204, 162, 1 ], [ 204, 166, 2 ], [ 204, 170, -1 ], [ 204, 193, -1/4 ], [ 204, 198, 1/4 ], 
  [ 204, 199, -1/4 ], [ 204, 204, 1/4 ], [ 205, 223, -3/16 ], [ 205, 225, 3/16 ], [ 205, 226, -1/16 ], [ 205, 228, 1/16 ], [ 206, 206, 1/4 ], [ 206, 210, -1/4 ], [ 206, 211, -1/4 ], 
  [ 206, 215, 1/4 ], [ 206, 227, -1/8 ], [ 206, 229, 1/8 ], [ 207, 130, 12 ], [ 207, 137, -12 ], [ 207, 138, 4 ], [ 207, 145, -4 ], [ 207, 165, -3 ], [ 207, 171, 3 ], 
  [ 207, 172, -1 ], [ 207, 173, -3 ], [ 207, 177, 1 ], [ 207, 178, 3 ], [ 207, 179, -1 ], [ 207, 184, 1 ], [ 208, 130, 4 ], [ 208, 137, -4 ], [ 208, 138, -4 ], [ 208, 145, 4 ], 
  [ 208, 165, 1 ], [ 208, 171, -1 ], [ 208, 172, -1 ], [ 208, 173, 1 ], [ 208, 177, 1 ], [ 208, 178, -1 ], [ 208, 179, -1 ], [ 208, 184, 1 ], [ 209, 209, 1/2 ], 
  [ 209, 213, -1/2 ], [ 209, 230, 1/8 ], [ 210, 206, -3/4 ], [ 210, 210, -1/4 ], [ 210, 211, 3/4 ], [ 210, 215, 1/4 ], [ 210, 227, 3/8 ], [ 210, 229, 1/8 ], [ 211, 206, -1/4 ], 
  [ 211, 210, 1/4 ], [ 211, 211, 1/4 ], [ 211, 215, -1/4 ], [ 211, 227, -1/8 ], [ 211, 229, 1/8 ], [ 212, 130, -12 ], [ 212, 137, 12 ], [ 212, 138, -4 ], [ 212, 145, 4 ], 
  [ 212, 165, -3 ], [ 212, 171, 3 ], [ 212, 172, -1 ], [ 212, 173, -3 ], [ 212, 177, 1 ], [ 212, 178, 3 ], [ 212, 179, -1 ], [ 212, 184, 1 ], [ 213, 209, -1/2 ], [ 213, 213, 1/2 ], 
  [ 213, 230, 1/8 ], [ 214, 189, -2 ], [ 214, 195, 2 ], [ 214, 214, -1/4 ], [ 214, 217, -1 ], [ 214, 218, -1/4 ], [ 214, 219, 1/4 ], [ 214, 222, 1/4 ], [ 214, 231, 1/8 ], 
  [ 214, 234, 1/8 ], [ 215, 206, 3/4 ], [ 215, 210, 1/4 ], [ 215, 211, -3/4 ], [ 215, 215, -1/4 ], [ 215, 227, 3/8 ], [ 215, 229, 1/8 ], [ 216, 146, 8 ], [ 216, 153, -8 ], 
  [ 216, 154, 8 ], [ 216, 161, -8 ], [ 217, 214, -1/4 ], [ 217, 218, 1/4 ], [ 217, 219, -1/4 ], [ 217, 222, 1/4 ], [ 218, 189, 2 ], [ 218, 195, -2 ], [ 218, 214, -1/4 ], 
  [ 218, 217, 1 ], [ 218, 218, -1/4 ], [ 218, 219, 1/4 ], [ 218, 222, 1/4 ], [ 218, 231, 1/8 ], [ 218, 234, 1/8 ], [ 219, 189, 2 ], [ 219, 195, -2 ], [ 219, 214, 1/4 ], [ 219, 217, -1 ],
  [ 219, 218, 1/4 ], [ 219, 219, -1/4 ], [ 219, 222, -1/4 ], [ 219, 231, 1/8 ], [ 219, 234, 1/8 ], [ 220, 220, 1 ], [ 221, 201, -2 ], [ 221, 221, 1/2 ], [ 221, 224, 1/2 ], 
  [ 222, 189, -2 ], [ 222, 195, 2 ], [ 222, 214, 1/4 ], [ 222, 217, 1 ], [ 222, 218, 1/4 ], [ 222, 219, -1/4 ], [ 222, 222, -1/4 ], [ 222, 231, 1/8 ], [ 222, 234, 1/8 ], [ 223, 200, 2 ],
  [ 223, 205, -2 ], [ 223, 223, 1/4 ], [ 223, 225, 1/4 ], [ 223, 226, -1/4 ], [ 223, 228, -1/4 ], [ 224, 201, 2 ], [ 224, 221, 1/2 ], [ 224, 224, 1/2 ], [ 225, 200, -2 ], 
  [ 225, 205, 2 ], [ 225, 223, 1/4 ], [ 225, 225, 1/4 ], [ 225, 226, -1/4 ], [ 225, 228, -1/4 ], [ 226, 200, -6 ], [ 226, 205, -2 ], [ 226, 223, -3/4 ], [ 226, 225, -3/4 ], 
  [ 226, 226, -1/4 ], [ 226, 228, -1/4 ], [ 227, 206, -1 ], [ 227, 210, 1 ], [ 227, 211, -1 ], [ 227, 215, 1 ], [ 228, 200, 6 ], [ 228, 205, 2 ], [ 228, 223, -3/4 ], 
  [ 228, 225, -3/4 ], [ 228, 226, -1/4 ], [ 228, 228, -1/4 ], [ 229, 206, 3 ], [ 229, 210, 1 ], [ 229, 211, 3 ], [ 229, 215, 1 ], [ 230, 209, 4 ], [ 230, 213, 4 ], [ 231, 189, 8 ], 
  [ 231, 195, 8 ], [ 231, 214, 1 ], [ 231, 218, 1 ], [ 231, 219, 1 ], [ 231, 222, 1 ], [ 232, 238, 1/4 ], [ 232, 239, 1/4 ], [ 233, 233, 1/2 ], [ 233, 235, -1/2 ], 
  [ 233, 240, -1/8 ], [ 234, 189, -8 ], [ 234, 195, -8 ], [ 234, 214, 1 ], [ 234, 218, 1 ], [ 234, 219, 1 ], [ 234, 222, 1 ], [ 235, 233, -1/2 ], [ 235, 235, 1/2 ], [ 235, 240, -1/8 ], 
  [ 236, 236, 1/2 ], [ 236, 237, 1/2 ], [ 237, 236, 3/2 ], [ 237, 237, -1/2 ], [ 238, 232, 2 ], [ 238, 238, 1/2 ], [ 238, 239, -1/2 ], [ 239, 232, 2 ], [ 239, 238, -1/2 ], 
  [ 239, 239, 1/2 ], [ 240, 233, -4 ], [ 240, 235, -4 ], [ 241, 7, 1/2 ], [ 241, 127, 1/2 ], [ 241, 242, 1 ], [ 241, 246, -1/2 ], [ 241, 248, 1/2 ], [ 242, 242, 1 ], [ 243, 3, 1/4 ], 
  [ 243, 7, 1/2 ], [ 243, 36, 2 ], [ 243, 43, -2 ], [ 243, 123, 1/4 ], [ 243, 127, 1/2 ], [ 243, 156, 1/16 ], [ 243, 163, -1/16 ], [ 243, 241, -1/4 ], [ 243, 242, 3/2 ], 
  [ 243, 244, 1/4 ], [ 243, 246, -1/2 ], [ 244, 3, 1/2 ], [ 244, 7, 1/2 ], [ 244, 123, 1/2 ], [ 244, 127, 1/2 ], [ 244, 241, -1/2 ], [ 244, 242, 2 ], [ 244, 244, 1/2 ], 
  [ 244, 246, -1/2 ], [ 244, 248, -1/2 ], [ 245, 3, 1/2 ], [ 245, 5, 3/4 ], [ 245, 7, 1/2 ], [ 245, 123, 1/2 ], [ 245, 125, 1/4 ], [ 245, 127, 1/2 ], 
  [ 245, 241, -1/2 ], [ 245, 242, 2 ], [ 245, 244, 1/4 ], [ 245, 245, -1/2 ], [ 245, 246, 1/4 ], [ 245, 248, -1/2 ], [ 246, 3, 1/2 ], [ 246, 7, 1/2 ], [ 246, 123, 1/2 ], 
  [ 246, 127, 1/2 ], [ 246, 241, -1/2 ], [ 246, 242, 2 ], [ 246, 244, -1/2 ], [ 246, 246, 1/2 ], [ 246, 248, -1/2 ], [ 247, 3, 1/2 ], [ 247, 7, 1/4 ], [ 247, 31, 1 ], 
  [ 247, 39, 1 ], [ 247, 123, 1/2 ], [ 247, 127, 1/4 ], [ 247, 151, 1/8 ], [ 247, 159, 1/8 ], [ 247, 242, 3/2 ], [ 247, 244, -1/2 ], [ 247, 246, 1/4 ], [ 247, 248, -1/4 ], 
  [ 248, 3, 1/2 ], [ 248, 123, 1/2 ], [ 248, 241, 1/2 ], [ 248, 242, 1 ], [ 248, 244, -1/2 ] ] ],

  [ [ 1, 0, 0, 0, 0, 1, 0, 1 ],
    [ 248, [ 1, 44, -1 ], [ 2, 138, -1 ], [ 3, 139, -1 ], [ 4, 5, -1 ], [ 5, 4, -1 ], [ 6, 41, -1 ], [ 7, 127, -1 ], [ 8, 15, -1 ], [ 9, 23, -1 ], 
  [ 10, 130, -1 ], [ 11, 131, -1 ], [ 12, 12, -1 ], [ 13, 49, -1 ], [ 14, 33, 1 ], [ 15, 8, 1 ], [ 16, 30, 1 ], [ 17, 152, 1 ], [ 18, 122, -1 ], [ 19, 123, -1 ], [ 20, 55, -1 ], 
  [ 21, 40, 1 ], [ 22, 50, -1 ], [ 23, 9, 1 ], [ 24, 37, -1 ], [ 25, 145, 1 ], [ 26, 35, 1 ], [ 27, 34, 1 ], [ 28, 48, 1 ], [ 29, 56, -1 ], [ 30, 16, -1 ], 
  [ 31, 75, 1 ], [ 32, 137, 1 ], [ 33, 14, -1 ], [ 34, 27, -1 ], [ 35, 26, -1 ], [ 36, 62, -1 ], [ 37, 24, 1 ], [ 38, 58, 1 ], [ 39, 69, -1 ], [ 40, 21, -1 ], [ 41, 6, 1 ], 
  [ 42, 43, 1 ], [ 43, 42, 1 ], [ 44, 1, 1 ], [ 45, 64, -1 ], [ 46, 51, -1 ], [ 47, 81, 1 ], [ 48, 28, -1 ], [ 49, 13, 1 ], [ 50, 22, -1 ], [ 51, 46, -1 ], 
  [ 52, 70, 1 ], [ 53, 57, 1 ], [ 54, 65, 1 ], [ 55, 20, 1 ], [ 56, 29, -1 ], [ 57, 53, 1 ], [ 58, 38, 1 ], [ 59, 63, -1 ], [ 60, 72, -1 ], [ 61, 61, 1 ], [ 62, 36, -1 ], 
  [ 63, 59, -1 ], [ 64, 45, -1 ], [ 65, 54, -1 ], [ 66, 93, 1 ], [ 67, 77, 1 ], [ 68, 74, 1 ], [ 69, 39, -1 ], [ 70, 52, 1 ], [ 71, 80, 1 ], 
  [ 72, 60, 1 ], [ 73, 102, 1 ], [ 74, 68, -1 ], [ 75, 31, 1 ], [ 76, 85, -1 ], [ 77, 67, -1 ], [ 78, 91, 1 ], [ 79, 98, -1 ], [ 80, 71, -1 ], [ 81, 47, -1 ], [ 82, 89, 1 ], 
  [ 83, 95, -1 ], [ 84, 86, -1 ], [ 85, 76, 1 ], [ 86, 84, -1 ], [ 87, 99, 1 ], [ 88, 90, 1 ], [ 89, 82, -1 ], [ 90, 88, 1 ], [ 91, 78, 1 ], [ 92, 94, -1 ], 
  [ 93, 66, -1 ], [ 94, 92, -1 ], [ 95, 83, -1 ], [ 96, 112, -1 ], [ 97, 97, -1 ], [ 98, 79, -1 ], [ 99, 87, 1 ], [ 100, 105, -1 ], [ 101, 104, -1 ], [ 102, 73, 1 ], [ 103, 108, 1 ], 
  [ 104, 101, 1 ], [ 105, 100, 1 ], [ 106, 110, -1 ], [ 107, 115, -1 ], [ 108, 103, -1 ], [ 109, 116, -1 ], [ 110, 106, 1 ], [ 111, 117, -1 ], [ 112, 96, 1 ], [ 113, 114, 1 ], 
  [ 114, 113, 1 ], [ 115, 107, -1 ], [ 116, 109, -1 ], [ 117, 111, -1 ], [ 118, 119, 1 ], [ 119, 118, -1 ], [ 120, 120, -1 ], [ 121, 164, -1 ], [ 122, 18, -1 ], [ 123, 19, -1 ], 
  [ 124, 125, -1 ], [ 125, 124, -1 ], [ 126, 161, -1 ], [ 127, 7, -1 ], [ 128, 135, -1 ], [ 129, 143, -1 ], [ 130, 10, -1 ], [ 131, 11, -1 ], [ 132, 132, -1 ], [ 133, 169, -1 ], 
  [ 134, 153, 1 ], [ 135, 128, 1 ], [ 136, 150, 1 ], [ 137, 32, 1 ], [ 138, 2, -1 ], [ 139, 3, -1 ], [ 140, 175, -1 ], [ 141, 160, 1 ], [ 142, 170, -1 ], [ 143, 129, 1 ], 
  [ 144, 157, -1 ], [ 145, 25, 1 ], [ 146, 155, 1 ], [ 147, 154, 1 ], [ 148, 168, 1 ], [ 149, 176, -1 ], [ 150, 136, -1 ], [ 151, 195, 1 ], [ 152, 17, 1 ], [ 153, 134, -1 ], 
  [ 154, 147, -1 ], [ 155, 146, -1 ], [ 156, 182, -1 ], [ 157, 144, 1 ], [ 158, 178, 1 ], [ 159, 189, -1 ], [ 160, 141, -1 ], [ 161, 126, 1 ], [ 162, 163, 1 ], [ 163, 162, 1 ], 
  [ 164, 121, 1 ], [ 165, 184, -1 ], [ 166, 171, -1 ], [ 167, 201, 1 ], [ 168, 148, -1 ], [ 169, 133, 1 ], [ 170, 142, -1 ], [ 171, 166, -1 ], [ 172, 190, 1 ], [ 173, 177, 1 ], 
  [ 174, 185, 1 ], [ 175, 140, 1 ], [ 176, 149, -1 ], [ 177, 173, 1 ], [ 178, 158, 1 ], [ 179, 183, -1 ], [ 180, 192, -1 ], [ 181, 181, 1 ], [ 182, 156, -1 ], [ 183, 179, -1 ], 
  [ 184, 165, -1 ], [ 185, 174, -1 ], [ 186, 213, 1 ], [ 187, 197, 1 ], [ 188, 194, 1 ], [ 189, 159, -1 ], [ 190, 172, 1 ], [ 191, 200, 1 ], [ 192, 180, 1 ], [ 193, 222, 1 ], 
  [ 194, 188, -1 ], [ 195, 151, 1 ], [ 196, 205, -1 ], [ 197, 187, -1 ], [ 198, 211, 1 ], [ 199, 218, -1 ], [ 200, 191, -1 ], [ 201, 167, -1 ], [ 202, 209, 1 ], [ 203, 215, -1 ], 
  [ 204, 206, -1 ], [ 205, 196, 1 ], [ 206, 204, -1 ], [ 207, 219, 1 ], [ 208, 210, 1 ], [ 209, 202, -1 ], [ 210, 208, 1 ], [ 211, 198, 1 ], [ 212, 214, -1 ], [ 213, 186, -1 ], 
  [ 214, 212, -1 ], [ 215, 203, -1 ], [ 216, 232, -1 ], [ 217, 217, -1 ], [ 218, 199, -1 ], [ 219, 207, 1 ], [ 220, 225, -1 ], [ 221, 224, -1 ], [ 222, 193, 1 ], [ 223, 228, 1 ], 
  [ 224, 221, 1 ], [ 225, 220, 1 ], [ 226, 230, -1 ], [ 227, 235, -1 ], [ 228, 223, -1 ], [ 229, 236, -1 ], [ 230, 226, 1 ], [ 231, 237, -1 ], [ 232, 216, 1 ], [ 233, 234, 1 ], 
  [ 234, 233, 1 ], [ 235, 227, -1 ], [ 236, 229, -1 ], [ 237, 231, -1 ], [ 238, 239, 1 ], [ 239, 238, -1 ], [ 240, 240, -1 ], [ 241, 241, 1 ], [ 242, 241, 1 ], [ 242, 242, -1 ], 
  [ 242, 246, 1 ], [ 243, 241, 2 ], [ 243, 243, -1 ], [ 243, 246, 1 ], [ 244, 241, 2 ], [ 244, 242, -1 ], [ 244, 243, -1 ], [ 244, 245, 1 ], [ 244, 246, 1 ], [ 245, 241, 1 ], 
  [ 245, 242, -1 ], [ 245, 243, -1 ], [ 245, 244, 1 ], [ 245, 246, 1 ], [ 246, 246, 1 ], [ 247, 246, 1 ], [ 247, 247, -1 ], [ 247, 248, 1 ], [ 248, 248, 1 ] ]],

  [ [ 1, 0, 0, 0, 0, 1, 0, 2 ],
    [ 248, [ 1, 16, -1 ], [ 2, 32, 1 ], [ 3, 5, 1 ], [ 4, 139, 1 ], [ 5, 3, 1 ], [ 6, 20, -1 ], [ 7, 7, -1 ], [ 8, 8, -1 ], [ 9, 24, -1 ], 
  [ 10, 10, 1 ], [ 11, 131, 1 ], [ 12, 132, 1 ], [ 13, 27, -1 ], [ 14, 28, 1 ], [ 15, 15, 1 ], [ 16, 1, -1 ], [ 17, 18, -1 ], [ 18, 17, -1 ], [ 19, 124, 1 ], [ 20, 6, -1 ], 
  [ 21, 35, 1 ], [ 22, 36, -1 ], [ 23, 37, 1 ], [ 24, 9, -1 ], [ 25, 25, 1 ], [ 26, 40, -1 ], [ 27, 13, -1 ], [ 28, 14, 1 ], [ 29, 43, -1 ], [ 30, 44, -1 ], 
  [ 31, 31, 1 ], [ 32, 2, 1 ], [ 33, 48, 1 ], [ 34, 49, 1 ], [ 35, 21, 1 ], [ 36, 22, -1 ], [ 37, 23, 1 ], [ 38, 63, 1 ], [ 39, 39, -1 ], [ 40, 26, -1 ], [ 41, 55, -1 ], [ 42, 56, -1 ], 
  [ 43, 29, -1 ], [ 44, 30, -1 ], [ 45, 45, 1 ], [ 46, 70, -1 ], [ 47, 47, 1 ], [ 48, 33, 1 ], [ 49, 34, 1 ], [ 50, 62, 1 ], [ 51, 52, -1 ], [ 52, 51, -1 ], 
  [ 53, 53, -1 ], [ 54, 77, 1 ], [ 55, 41, -1 ], [ 56, 42, -1 ], [ 57, 57, 1 ], [ 58, 59, 1 ], [ 59, 58, 1 ], [ 60, 60, 1 ], [ 61, 61, 1 ], [ 62, 50, 1 ], [ 63, 38, 1 ], 
  [ 64, 64, -1 ], [ 65, 67, -1 ], [ 66, 76, 1 ], [ 67, 65, -1 ], [ 68, 68, -1 ], [ 69, 69, 1 ], [ 70, 46, -1 ], [ 71, 82, -1 ], [ 72, 72, 1 ], [ 73, 83, -1 ], [ 74, 74, 1 ], 
  [ 75, 75, -1 ], [ 76, 66, 1 ], [ 77, 54, 1 ], [ 78, 87, 1 ], [ 79, 88, 1 ], [ 80, 89, 1 ], [ 81, 81, 1 ], [ 82, 71, -1 ], [ 83, 73, -1 ], [ 84, 92, -1 ], 
  [ 85, 93, 1 ], [ 86, 94, -1 ], [ 87, 78, 1 ], [ 88, 79, 1 ], [ 89, 80, 1 ], [ 90, 98, -1 ], [ 91, 99, 1 ], [ 92, 84, -1 ], [ 93, 85, 1 ], [ 94, 86, -1 ], [ 95, 102, 1 ], 
  [ 96, 96, 1 ], [ 97, 97, -1 ], [ 98, 90, -1 ], [ 99, 91, 1 ], [ 100, 110, -1 ], [ 101, 101, 1 ], [ 102, 95, 1 ], [ 103, 103, 1 ], [ 104, 104, -1 ], [ 105, 106, 1 ], 
  [ 106, 105, 1 ], [ 107, 111, 1 ], [ 108, 108, 1 ], [ 109, 114, 1 ], [ 110, 100, -1 ], [ 111, 107, 1 ], [ 112, 112, 1 ], [ 113, 116, 1 ], [ 114, 109, 1 ], [ 115, 117, -1 ], 
  [ 116, 113, 1 ], [ 117, 115, -1 ], [ 118, 118, 1 ], [ 119, 119, -1 ], [ 120, 120, 1 ], [ 121, 136, -1 ], [ 122, 152, 1 ], [ 123, 125, 1 ], [ 124, 19, 1 ], 
  [ 125, 123, 1 ], [ 126, 140, -1 ], [ 127, 127, -1 ], [ 128, 128, -1 ], [ 129, 144, -1 ], [ 130, 130, 1 ], [ 131, 11, 1 ], [ 132, 12, 1 ], [ 133, 147, -1 ], [ 134, 148, 1 ], 
  [ 135, 135, 1 ], [ 136, 121, -1 ], [ 137, 138, -1 ], [ 138, 137, -1 ], [ 139, 4, 1 ], [ 140, 126, -1 ], [ 141, 155, 1 ], [ 142, 156, -1 ], [ 143, 157, 1 ], [ 144, 129, -1 ], 
  [ 145, 145, 1 ], [ 146, 160, -1 ], [ 147, 133, -1 ], [ 148, 134, 1 ], [ 149, 163, -1 ], [ 150, 164, -1 ], [ 151, 151, 1 ], [ 152, 122, 1 ], [ 153, 168, 1 ], [ 154, 169, 1 ], 
  [ 155, 141, 1 ], [ 156, 142, -1 ], [ 157, 143, 1 ], [ 158, 183, 1 ], [ 159, 159, -1 ], [ 160, 146, -1 ], [ 161, 175, -1 ], [ 162, 176, -1 ], [ 163, 149, -1 ], [ 164, 150, -1 ], 
  [ 165, 165, 1 ], [ 166, 190, -1 ], [ 167, 167, 1 ], [ 168, 153, 1 ], [ 169, 154, 1 ], [ 170, 182, 1 ], [ 171, 172, -1 ], [ 172, 171, -1 ], [ 173, 173, -1 ], [ 174, 197, 1 ], 
  [ 175, 161, -1 ], [ 176, 162, -1 ], [ 177, 177, 1 ], [ 178, 179, 1 ], [ 179, 178, 1 ], [ 180, 180, 1 ], [ 181, 181, 1 ], [ 182, 170, 1 ], [ 183, 158, 1 ], [ 184, 184, -1 ], 
  [ 185, 187, -1 ], [ 186, 196, 1 ], [ 187, 185, -1 ], [ 188, 188, -1 ], [ 189, 189, 1 ], [ 190, 166, -1 ], [ 191, 202, -1 ], [ 192, 192, 1 ], [ 193, 203, -1 ], [ 194, 194, 1 ], 
  [ 195, 195, -1 ], [ 196, 186, 1 ], [ 197, 174, 1 ], [ 198, 207, 1 ], [ 199, 208, 1 ], [ 200, 209, 1 ], [ 201, 201, 1 ], [ 202, 191, -1 ], [ 203, 193, -1 ], [ 204, 212, -1 ], 
  [ 205, 213, 1 ], [ 206, 214, -1 ], [ 207, 198, 1 ], [ 208, 199, 1 ], [ 209, 200, 1 ], [ 210, 218, -1 ], [ 211, 219, 1 ], [ 212, 204, -1 ], [ 213, 205, 1 ], [ 214, 206, -1 ], 
  [ 215, 222, 1 ], [ 216, 216, 1 ], [ 217, 217, -1 ], [ 218, 210, -1 ], [ 219, 211, 1 ], [ 220, 230, -1 ], [ 221, 221, 1 ], [ 222, 215, 1 ], [ 223, 223, 1 ], [ 224, 224, -1 ], 
  [ 225, 226, 1 ], [ 226, 225, 1 ], [ 227, 231, 1 ], [ 228, 228, 1 ], [ 229, 234, 1 ], [ 230, 220, -1 ], [ 231, 227, 1 ], [ 232, 232, 1 ], [ 233, 236, 1 ], [ 234, 229, 1 ], 
  [ 235, 237, -1 ], [ 236, 233, 1 ], [ 237, 235, -1 ], [ 238, 238, 1 ], [ 239, 239, -1 ], [ 240, 240, 1 ], [ 241, 241, 1 ], [ 242, 242, 1 ], [ 243, 241, 1 ], [ 243, 242, 1 ], 
  [ 243, 244, -1 ], [ 243, 245, 1 ], [ 244, 241, 1 ], [ 244, 242, 2 ], [ 244, 244, -1 ], [ 244, 246, 1 ], [ 245, 242, 1 ], [ 245, 243, 1 ], [ 245, 244, -1 ], [ 245, 246, 1 ], 
  [ 246, 246, 1 ], [ 247, 247, 1 ], [ 248, 248, 1 ] ] ],

  [ [ 0, 0, 0, 1, 0, 0, 0, 1 ],
    [ 248, [ 1, 121, E(5) ], [ 2, 122, E(5) ], [ 3, 9, -E(5)^2 ], [ 4, 18, E(5) ], [ 5, 125, E(5)^2 ], [ 6, 21, 1 ], [ 7, 127, E(5)^3 ], [ 8, 15, -E(5) ], 
  [ 9, 3, -E(5)^3 ], [ 10, 12, E(5)^2 ], [ 11, 30, -E(5)^3 ], [ 12, 10, E(5)^3 ], [ 13, 14, E(5)^2 ], [ 14, 13, E(5)^3 ], [ 15, 8, -E(5)^4 ], [ 16, 25, -E(5)^4 ], [ 17, 24, -E(5)^4 ], 
  [ 18, 4, E(5)^4 ], [ 19, 23, -1 ], [ 20, 34, E(5)^3 ], [ 21, 6, 1 ], [ 22, 29, -E(5)^4 ], [ 23, 19, -1 ], [ 24, 17, -E(5) ], [ 25, 16, -E(5) ], [ 26, 28, E(5)^4 ], 
  [ 27, 46, -1 ], [ 28, 26, E(5) ], [ 29, 22, -E(5) ], [ 30, 11, -E(5)^2 ], [ 31, 41, -E(5) ], [ 32, 37, -E(5)^2 ], [ 33, 39, -E(5) ], [ 34, 20, E(5)^2 ], [ 35, 38, -E(5)^3 ], 
  [ 36, 42, -E(5)^2 ], [ 37, 32, -E(5)^3 ], [ 38, 35, -E(5)^2 ], [ 39, 33, -E(5)^4 ], [ 40, 59, E(5)^2 ], [ 41, 31, -E(5)^4 ], [ 42, 36, -E(5)^3 ], [ 43, 54, E(5)^4 ], 
  [ 44, 44, -1 ], [ 45, 55, E(5)^3 ], [ 46, 27, -1 ], [ 47, 50, 1 ], [ 48, 53, E(5)^4 ], [ 49, 52, 1 ], [ 50, 47, 1 ], [ 51, 64, 1 ], [ 52, 49, 1 ], [ 53, 48, E(5) ], 
  [ 54, 43, E(5) ], [ 55, 45, E(5)^2 ], [ 56, 67, -E(5) ], [ 57, 58, E(5)^2 ], [ 58, 57, E(5)^3 ], [ 59, 40, E(5)^3 ], [ 60, 62, -E(5)^2 ], [ 61, 66, -E(5)^2 ], [ 62, 60, -E(5)^3 ], 
  [ 63, 75, E(5)^3 ], [ 64, 51, 1 ], [ 65, 72, -E(5)^4 ], [ 66, 61, -E(5)^3 ], [ 67, 56, -E(5)^4 ], [ 68, 79, -E(5)^3 ], [ 69, 70, E(5)^4 ], [ 70, 69, E(5) ], 
  [ 71, 71, -1 ], [ 72, 65, -E(5) ], [ 73, 74, -E(5)^4 ], [ 74, 73, -E(5) ], [ 75, 63, E(5)^2 ], [ 76, 85, E(5) ], [ 77, 81, -E(5)^2 ], [ 78, 84, -E(5) ], [ 79, 68, -E(5)^2 ], 
  [ 80, 82, E(5)^2 ], [ 81, 77, -E(5)^3 ], [ 82, 80, E(5)^3 ], [ 83, 95, E(5)^2 ], [ 84, 78, -E(5)^4 ], [ 85, 76, E(5)^4 ], [ 86, 92, E(5)^3 ], [ 87, 91, E(5)^4 ], 
  [ 88, 90, 1 ], [ 89, 89, -1 ], [ 90, 88, 1 ], [ 91, 87, E(5) ], [ 92, 86, E(5)^2 ], [ 93, 97, -E(5)^2 ], [ 94, 99, -E(5) ], [ 95, 83, E(5)^3 ], [ 96, 100, -E(5)^2 ], 
  [ 97, 93, -E(5)^3 ], [ 98, 104, -E(5)^3 ], [ 99, 94, -E(5)^4 ], [ 100, 96, -E(5)^3 ], [ 101, 102, -E(5)^4 ], [ 102, 101, -E(5) ], [ 103, 106, -E(5)^4 ], [ 104, 98, -E(5)^2 ], 
  [ 105, 109, -E(5) ], [ 106, 103, -E(5) ], [ 107, 108, -E(5)^2 ], [ 108, 107, -E(5)^3 ], [ 109, 105, -E(5)^4 ], [ 110, 113, E(5)^4 ], [ 111, 112, 1 ], [ 112, 111, 1 ], 
  [ 113, 110, E(5) ], [ 114, 115, -E(5)^2 ], [ 115, 114, -E(5)^3 ], [ 116, 117, -E(5)^4 ], [ 117, 116, -E(5) ], [ 118, 119, -E(5) ], [ 119, 118, -E(5)^4 ], [ 120, 120, -1 ], 
  [ 121, 1, E(5)^4 ], [ 122, 2, E(5)^4 ], [ 123, 129, -E(5)^3 ], [ 124, 138, E(5)^4 ], [ 125, 5, E(5)^3 ], [ 126, 141, 1 ], [ 127, 7, E(5)^2 ], [ 128, 135, -E(5)^4 ], 
  [ 129, 123, -E(5)^2 ], [ 130, 132, E(5)^3 ], [ 131, 150, -E(5)^2 ], [ 132, 130, E(5)^2 ], [ 133, 134, E(5)^3 ], [ 134, 133, E(5)^2 ], [ 135, 128, -E(5) ], [ 136, 145, -E(5) ], 
  [ 137, 144, -E(5) ], [ 138, 124, E(5) ], [ 139, 143, -1 ], [ 140, 154, E(5)^2 ], [ 141, 126, 1 ], [ 142, 149, -E(5) ], [ 143, 139, -1 ], [ 144, 137, -E(5)^4 ], [ 145, 136, -E(5)^4 ], 
  [ 146, 148, E(5) ], [ 147, 166, -1 ], [ 148, 146, E(5)^4 ], [ 149, 142, -E(5)^4 ], [ 150, 131, -E(5)^3 ], [ 151, 161, -E(5)^4 ], [ 152, 157, -E(5)^3 ], [ 153, 159, -E(5)^4 ], 
  [ 154, 140, E(5)^3 ], [ 155, 158, -E(5)^2 ], [ 156, 162, -E(5)^3 ], [ 157, 152, -E(5)^2 ], [ 158, 155, -E(5)^3 ], [ 159, 153, -E(5) ], [ 160, 179, E(5)^3 ], [ 161, 151, -E(5) ], 
  [ 162, 156, -E(5)^2 ], [ 163, 174, E(5) ], [ 164, 164, -1 ], [ 165, 175, E(5)^2 ], [ 166, 147, -1 ], [ 167, 170, 1 ], [ 168, 173, E(5) ], [ 169, 172, 1 ], [ 170, 167, 1 ], 
  [ 171, 184, 1 ], [ 172, 169, 1 ], [ 173, 168, E(5)^4 ], [ 174, 163, E(5)^4 ], [ 175, 165, E(5)^3 ], [ 176, 187, -E(5)^4 ], [ 177, 178, E(5)^3 ], [ 178, 177, E(5)^2 ], 
  [ 179, 160, E(5)^2 ], [ 180, 182, -E(5)^3 ], [ 181, 186, -E(5)^3 ], [ 182, 180, -E(5)^2 ], [ 183, 195, E(5)^2 ], [ 184, 171, 1 ], [ 185, 192, -E(5) ], 
  [ 186, 181, -E(5)^2 ], [ 187, 176, -E(5) ], [ 188, 199, -E(5)^2 ], [ 189, 190, E(5) ], [ 190, 189, E(5)^4 ], [ 191, 191, -1 ], [ 192, 185, -E(5)^4 ], [ 193, 194, -E(5) ], 
  [ 194, 193, -E(5)^4 ], [ 195, 183, E(5)^3 ], [ 196, 205, E(5)^4 ], [ 197, 201, -E(5)^3 ], [ 198, 204, -E(5)^4 ], [ 199, 188, -E(5)^3 ], [ 200, 202, E(5)^3 ], [ 201, 197, -E(5)^2 ], 
  [ 202, 200, E(5)^2 ], [ 203, 215, E(5)^3 ], [ 204, 198, -E(5) ], [ 205, 196, E(5) ], [ 206, 212, E(5)^2 ], [ 207, 211, E(5) ], [ 208, 210, 1 ], [ 209, 209, -1 ], 
  [ 210, 208, 1 ], [ 211, 207, E(5)^4 ], [ 212, 206, E(5)^3 ], [ 213, 217, -E(5)^3 ], [ 214, 219, -E(5)^4 ], [ 215, 203, E(5)^2 ], [ 216, 220, -E(5)^3 ], [ 217, 213, -E(5)^2 ], 
  [ 218, 224, -E(5)^2 ], [ 219, 214, -E(5) ], [ 220, 216, -E(5)^2 ], [ 221, 222, -E(5) ], [ 222, 221, -E(5)^4 ], [ 223, 226, -E(5) ], [ 224, 218, -E(5)^3 ], [ 225, 229, -E(5)^4 ], 
  [ 226, 223, -E(5)^4 ], [ 227, 228, -E(5)^3 ], [ 228, 227, -E(5)^2 ], [ 229, 225, -E(5) ], [ 230, 233, E(5) ], [ 231, 232, 1 ], [ 232, 231, 1 ], [ 233, 230, E(5)^4 ], 
  [ 234, 235, -E(5)^3 ], [ 235, 234, -E(5)^2 ], [ 236, 237, -E(5) ], [ 237, 236, -E(5)^4 ], [ 238, 239, -E(5)^4 ], [ 239, 238, -E(5) ], [ 240, 240, -1 ], [ 241, 241, -1 ], 
  [ 241, 243, 1 ], [ 242, 242, -1 ], [ 242, 244, 1 ], [ 243, 243, 1 ], [ 244, 244, 1 ], [ 245, 244, 1 ], [ 245, 245, -1 ], [ 245, 246, 1 ], [ 246, 246, 1 ], 
  [ 247, 246, 1 ], [ 247, 247, -1 ], [ 247, 248, 1 ], [ 248, 248, 1 ] ] ],

  [ [ 0, 2, 0, 0, 0, 0, 0, 2 ],
    [ 248, [ 1, 155, 1 ], [ 2, 53, -1 ], [ 3, 11, 1 ], [ 4, 124, 1 ], [ 5, 20, -1 ], [ 6, 126, 1 ], [ 7, 144, -1 ], [ 8, 47, 1 ], [ 9, 141, 1 ], 
  [ 10, 46, -1 ], [ 11, 3, 1 ], [ 12, 13, -1 ], [ 13, 12, -1 ], [ 14, 151, -1 ], [ 15, 22, -1 ], [ 16, 148, 1 ], [ 17, 58, 1 ], [ 18, 66, 1 ], [ 19, 27, -1 ], [ 20, 5, -1 ], 
  [ 21, 129, 1 ], [ 22, 15, -1 ], [ 23, 23, 1 ], [ 24, 127, -1 ], [ 25, 76, 1 ], [ 26, 59, 1 ], [ 27, 19, -1 ], [ 28, 136, 1 ], [ 29, 36, 1 ], [ 30, 45, 1 ], 
  [ 31, 134, -1 ], [ 32, 71, 1 ], [ 33, 70, 1 ], [ 34, 34, 1 ], [ 35, 121, 1 ], [ 36, 29, 1 ], [ 37, 38, 1 ], [ 38, 37, 1 ], [ 39, 159, 1 ], [ 40, 64, 1 ], [ 41, 49, 1 ], 
  [ 42, 79, 1 ], [ 43, 43, 1 ], [ 44, 51, -1 ], [ 45, 30, 1 ], [ 46, 10, -1 ], [ 47, 8, 1 ], [ 48, 82, 1 ], [ 49, 41, 1 ], [ 50, 88, 1 ], [ 51, 44, -1 ], [ 52, 52, 1 ], 
  [ 53, 2, -1 ], [ 54, 60, -1 ], [ 55, 61, -1 ], [ 56, 84, 1 ], [ 57, 63, 1 ], [ 58, 17, 1 ], [ 59, 26, 1 ], [ 60, 54, -1 ], [ 61, 55, -1 ], [ 62, 96, -1 ], [ 63, 57, 1 ], 
  [ 64, 40, 1 ], [ 65, 65, 1 ], [ 66, 18, 1 ], [ 67, 73, 1 ], [ 68, 92, -1 ], [ 69, 97, -1 ], [ 70, 33, 1 ], [ 71, 32, 1 ], [ 72, 83, 1 ], [ 73, 67, 1 ], 
  [ 74, 74, 1 ], [ 75, 80, -1 ], [ 76, 25, 1 ], [ 77, 78, 1 ], [ 78, 77, 1 ], [ 79, 42, 1 ], [ 80, 75, -1 ], [ 81, 107, 1 ], [ 82, 48, 1 ], [ 83, 72, 1 ], [ 84, 56, 1 ], [ 85, 89, 1 ], 
  [ 86, 104, 1 ], [ 87, 87, 1 ], [ 88, 50, 1 ], [ 89, 85, 1 ], [ 90, 111, -1 ], [ 91, 91, 1 ], [ 92, 68, -1 ], [ 93, 93, -1 ], [ 94, 109, -1 ], [ 95, 103, 1 ], 
  [ 96, 62, -1 ], [ 97, 69, -1 ], [ 98, 114, 1 ], [ 99, 100, 1 ], [ 100, 99, 1 ], [ 101, 101, 1 ], [ 102, 105, -1 ], [ 103, 95, 1 ], [ 104, 86, 1 ], [ 105, 102, -1 ], [ 106, 106, 1 ], 
  [ 107, 81, 1 ], [ 108, 110, 1 ], [ 109, 94, -1 ], [ 110, 108, 1 ], [ 111, 90, -1 ], [ 112, 119, 1 ], [ 113, 113, 1 ], [ 114, 98, 1 ], [ 115, 116, -1 ], [ 116, 115, -1 ], 
  [ 117, 118, 1 ], [ 118, 117, 1 ], [ 119, 112, 1 ], [ 120, 120, 1 ], [ 121, 35, 1 ], [ 122, 173, -1 ], [ 123, 131, 1 ], [ 124, 4, 1 ], [ 125, 140, -1 ], [ 126, 6, 1 ], 
  [ 127, 24, -1 ], [ 128, 167, 1 ], [ 129, 21, 1 ], [ 130, 166, -1 ], [ 131, 123, 1 ], [ 132, 133, -1 ], [ 133, 132, -1 ], [ 134, 31, -1 ], [ 135, 142, -1 ], [ 136, 28, 1 ], 
  [ 137, 178, 1 ], [ 138, 186, 1 ], [ 139, 147, -1 ], [ 140, 125, -1 ], [ 141, 9, 1 ], [ 142, 135, -1 ], [ 143, 143, 1 ], [ 144, 7, -1 ], [ 145, 196, 1 ], 
  [ 146, 179, 1 ], [ 147, 139, -1 ], [ 148, 16, 1 ], [ 149, 156, 1 ], [ 150, 165, 1 ], [ 151, 14, -1 ], [ 152, 191, 1 ], [ 153, 190, 1 ], [ 154, 154, 1 ], [ 155, 1, 1 ], [ 156, 149, 1 ],
  [ 157, 158, 1 ], [ 158, 157, 1 ], [ 159, 39, 1 ], [ 160, 184, 1 ], [ 161, 169, 1 ], [ 162, 199, 1 ], [ 163, 163, 1 ], [ 164, 171, -1 ], [ 165, 150, 1 ], [ 166, 130, -1 ], 
  [ 167, 128, 1 ], [ 168, 202, 1 ], [ 169, 161, 1 ], [ 170, 208, 1 ], [ 171, 164, -1 ], [ 172, 172, 1 ], [ 173, 122, -1 ], [ 174, 180, -1 ], [ 175, 181, -1 ], [ 176, 204, 1 ], 
  [ 177, 183, 1 ], [ 178, 137, 1 ], [ 179, 146, 1 ], [ 180, 174, -1 ], [ 181, 175, -1 ], [ 182, 216, -1 ], [ 183, 177, 1 ], [ 184, 160, 1 ], [ 185, 185, 1 ], [ 186, 138, 1 ], 
  [ 187, 193, 1 ], [ 188, 212, -1 ], [ 189, 217, -1 ], [ 190, 153, 1 ], [ 191, 152, 1 ], [ 192, 203, 1 ], [ 193, 187, 1 ], [ 194, 194, 1 ], [ 195, 200, -1 ], [ 196, 145, 1 ], 
  [ 197, 198, 1 ], [ 198, 197, 1 ], [ 199, 162, 1 ], [ 200, 195, -1 ], [ 201, 227, 1 ], [ 202, 168, 1 ], [ 203, 192, 1 ], [ 204, 176, 1 ], [ 205, 209, 1 ], [ 206, 224, 1 ], 
  [ 207, 207, 1 ], [ 208, 170, 1 ], [ 209, 205, 1 ], [ 210, 231, -1 ], [ 211, 211, 1 ], [ 212, 188, -1 ], [ 213, 213, -1 ], [ 214, 229, -1 ], [ 215, 223, 1 ], [ 216, 182, -1 ], 
  [ 217, 189, -1 ], [ 218, 234, 1 ], [ 219, 220, 1 ], [ 220, 219, 1 ], [ 221, 221, 1 ], [ 222, 225, -1 ], [ 223, 215, 1 ], [ 224, 206, 1 ], [ 225, 222, -1 ], [ 226, 226, 1 ], 
  [ 227, 201, 1 ], [ 228, 230, 1 ], [ 229, 214, -1 ], [ 230, 228, 1 ], [ 231, 210, -1 ], [ 232, 239, 1 ], [ 233, 233, 1 ], [ 234, 218, 1 ], [ 235, 236, -1 ], [ 236, 235, -1 ], 
  [ 237, 238, 1 ], [ 238, 237, 1 ], [ 239, 232, 1 ], [ 240, 240, 1 ], [ 241, 242, 1 ], [ 241, 247, -1 ], [ 241, 248, 1 ], [ 242, 242, 1 ], [ 243, 241, -1 ], [ 243, 242, 1 ], 
  [ 243, 243, 1 ], [ 243, 247, -1 ], [ 243, 248, 1 ], [ 244, 241, -1 ], [ 244, 242, 2 ], [ 244, 243, 1 ], [ 244, 244, -1 ], [ 244, 245, 1 ], [ 244, 247, -1 ], [ 244, 248, 1 ], 
  [ 245, 241, -1 ], [ 245, 242, 1 ], [ 245, 245, 1 ], [ 245, 247, -1 ], [ 245, 248, 1 ], [ 246, 241, -1 ], [ 246, 242, 1 ], [ 246, 245, 1 ], [ 246, 246, -1 ], [ 246, 248, 1 ], 
  [ 247, 241, -1 ], [ 247, 242, 1 ], [ 247, 248, 1 ], [ 248, 248, 1 ] ] ],

  [ [ 2, 0, 0, 0, 0, 0, 2, 0 ],
    [ 248, [ 1, 1, -1 ], [ 2, 2, -1 ], [ 3, 3, -1 ], [ 4, 4, -1 ], [ 5, 5, 1 ], [ 6, 6, 1 ], [ 7, 7, 1 ], [ 8, 8, -1 ], [ 9, 9, 1 ], 
  [ 10, 10, 1 ], [ 11, 11, 1 ], [ 12, 12, -1 ], [ 13, 13, 1 ], [ 14, 14, 1 ], [ 15, 15, -1 ], [ 16, 16, -1 ], [ 17, 17, -1 ], [ 18, 18, 1 ], [ 19, 19, 1 ], [ 20, 20, -1 ], 
  [ 21, 21, 1 ], [ 22, 22, -1 ], [ 23, 23, 1 ], [ 24, 24, -1 ], [ 25, 25, -1 ], [ 26, 26, 1 ], [ 27, 27, 1 ], [ 28, 28, -1 ], [ 29, 29, -1 ], [ 30, 30, 1 ], 
  [ 31, 31, -1 ], [ 32, 32, 1 ], [ 33, 33, -1 ], [ 34, 34, 1 ], [ 35, 35, 1 ], [ 36, 36, 1 ], [ 37, 37, -1 ], [ 38, 38, 1 ], [ 39, 39, -1 ], [ 40, 40, 1 ], [ 41, 41, -1 ], 
  [ 42, 42, -1 ], [ 43, 43, -1 ], [ 44, 44, 1 ], [ 45, 45, -1 ], [ 46, 46, 1 ], [ 47, 47, 1 ], [ 48, 48, 1 ], [ 49, 49, 1 ], [ 50, 50, 1 ], [ 51, 51, 1 ], 
  [ 52, 52, -1 ], [ 53, 53, -1 ], [ 54, 54, -1 ], [ 55, 55, 1 ], [ 56, 56, -1 ], [ 57, 57, 1 ], [ 58, 58, 1 ], [ 59, 59, -1 ], [ 60, 60, 1 ], [ 61, 61, 1 ], [ 62, 62, -1 ], 
  [ 63, 63, -1 ], [ 64, 64, 1 ], [ 65, 65, -1 ], [ 66, 66, -1 ], [ 67, 67, 1 ], [ 68, 68, -1 ], [ 69, 69, 1 ], [ 70, 70, -1 ], [ 71, 71, 1 ], [ 72, 72, -1 ], 
  [ 73, 73, 1 ], [ 74, 74, -1 ], [ 75, 75, 1 ], [ 76, 76, -1 ], [ 77, 77, 1 ], [ 78, 78, -1 ], [ 79, 79, 1 ], [ 80, 80, 1 ], [ 81, 81, -1 ], [ 82, 82, -1 ], [ 83, 83, 1 ], 
  [ 84, 84, -1 ], [ 85, 85, 1 ], [ 86, 86, -1 ], [ 87, 87, 1 ], [ 88, 88, 1 ], [ 89, 89, -1 ], [ 90, 90, -1 ], [ 91, 91, -1 ], [ 92, 92, 1 ], [ 93, 93, 1 ], 
  [ 94, 94, 1 ], [ 95, 95, -1 ], [ 96, 96, 1 ], [ 97, 97, -1 ], [ 98, 98, -1 ], [ 99, 99, 1 ], [ 100, 100, -1 ], [ 101, 101, 1 ], [ 102, 102, -1 ], [ 103, 103, 1 ], [ 104, 104, 1 ], 
  [ 105, 105, -1 ], [ 106, 106, 1 ], [ 107, 107, 1 ], [ 108, 108, -1 ], [ 109, 109, 1 ], [ 110, 110, 1 ], [ 111, 111, -1 ], [ 112, 112, -1 ], [ 113, 113, 1 ], [ 114, 114, 1 ], 
  [ 115, 115, -1 ], [ 116, 116, 1 ], [ 117, 117, 1 ], [ 118, 118, 1 ], [ 119, 119, 1 ], [ 120, 120, -1 ], [ 121, 121, -1 ], [ 122, 122, -1 ], [ 123, 123, -1 ], [ 124, 124, -1 ], 
  [ 125, 125, 1 ], [ 126, 126, 1 ], [ 127, 127, 1 ], [ 128, 128, -1 ], [ 129, 129, 1 ], [ 130, 130, 1 ], [ 131, 131, 1 ], [ 132, 132, -1 ], [ 133, 133, 1 ], [ 134, 134, 1 ], 
  [ 135, 135, -1 ], [ 136, 136, -1 ], [ 137, 137, -1 ], [ 138, 138, 1 ], [ 139, 139, 1 ], [ 140, 140, -1 ], [ 141, 141, 1 ], [ 142, 142, -1 ], [ 143, 143, 1 ], [ 144, 144, -1 ], 
  [ 145, 145, -1 ], [ 146, 146, 1 ], [ 147, 147, 1 ], [ 148, 148, -1 ], [ 149, 149, -1 ], [ 150, 150, 1 ], [ 151, 151, -1 ], [ 152, 152, 1 ], [ 153, 153, -1 ], [ 154, 154, 1 ], 
  [ 155, 155, 1 ], [ 156, 156, 1 ], [ 157, 157, -1 ], [ 158, 158, 1 ], [ 159, 159, -1 ], [ 160, 160, 1 ], [ 161, 161, -1 ], [ 162, 162, -1 ], [ 163, 163, -1 ], [ 164, 164, 1 ], 
  [ 165, 165, -1 ], [ 166, 166, 1 ], [ 167, 167, 1 ], [ 168, 168, 1 ], [ 169, 169, 1 ], [ 170, 170, 1 ], [ 171, 171, 1 ], [ 172, 172, -1 ], [ 173, 173, -1 ], [ 174, 174, -1 ], 
  [ 175, 175, 1 ], [ 176, 176, -1 ], [ 177, 177, 1 ], [ 178, 178, 1 ], [ 179, 179, -1 ], [ 180, 180, 1 ], [ 181, 181, 1 ], [ 182, 182, -1 ], [ 183, 183, -1 ], [ 184, 184, 1 ], 
  [ 185, 185, -1 ], [ 186, 186, -1 ], [ 187, 187, 1 ], [ 188, 188, -1 ], [ 189, 189, 1 ], [ 190, 190, -1 ], [ 191, 191, 1 ], [ 192, 192, -1 ], [ 193, 193, 1 ], [ 194, 194, -1 ], 
  [ 195, 195, 1 ], [ 196, 196, -1 ], [ 197, 197, 1 ], [ 198, 198, -1 ], [ 199, 199, 1 ], [ 200, 200, 1 ], [ 201, 201, -1 ], [ 202, 202, -1 ], [ 203, 203, 1 ], [ 204, 204, -1 ], 
  [ 205, 205, 1 ], [ 206, 206, -1 ], [ 207, 207, 1 ], [ 208, 208, 1 ], [ 209, 209, -1 ], [ 210, 210, -1 ], [ 211, 211, -1 ], [ 212, 212, 1 ], [ 213, 213, 1 ], [ 214, 214, 1 ], 
  [ 215, 215, -1 ], [ 216, 216, 1 ], [ 217, 217, -1 ], [ 218, 218, -1 ], [ 219, 219, 1 ], [ 220, 220, -1 ], [ 221, 221, 1 ], [ 222, 222, -1 ], [ 223, 223, 1 ], [ 224, 224, 1 ], 
  [ 225, 225, -1 ], [ 226, 226, 1 ], [ 227, 227, 1 ], [ 228, 228, -1 ], [ 229, 229, 1 ], [ 230, 230, 1 ], [ 231, 231, -1 ], [ 232, 232, -1 ], [ 233, 233, 1 ], [ 234, 234, 1 ], 
  [ 235, 235, -1 ], [ 236, 236, 1 ], [ 237, 237, 1 ], [ 238, 238, 1 ], [ 239, 239, 1 ], [ 240, 240, -1 ], [ 241, 241, 1 ], [ 242, 242, 1 ], [ 243, 243, 1 ], [ 244, 244, 1 ], 
  [ 245, 245, 1 ], [ 246, 246, 1 ], [ 247, 247, 1 ], [ 248, 248, 1 ] ] ],

  [ [ 0, 1, 1, 0, 0, 0, 1, 0 ],
    [ 248, [ 1, 1, -1 ], [ 2, 26, -E(4) ], [ 3, 27, E(4) ], [ 4, 126, -1 ], [ 5, 125, 1 ], [ 6, 124, -1 ], [ 7, 28, E(4) ], [ 8, 8, -1 ], [ 9, 31, -E(4) ], 
  [ 10, 18, E(4) ], [ 11, 19, -E(4) ], [ 12, 133, -1 ], [ 13, 132, -1 ], [ 14, 21, -E(4) ], [ 15, 36, -E(4) ], [ 16, 24, E(4) ], [ 17, 48, 1 ], [ 18, 10, E(4) ], [ 19, 11, -E(4) ], 
  [ 20, 140, 1 ], [ 21, 14, -E(4) ], [ 22, 29, E(4) ], [ 23, 52, -1 ], [ 24, 16, E(4) ], [ 25, 40, 1 ], [ 26, 2, -E(4) ], [ 27, 3, E(4) ], [ 28, 7, E(4) ], 
  [ 29, 22, E(4) ], [ 30, 45, -1 ], [ 31, 9, -E(4) ], [ 32, 32, -1 ], [ 33, 33, -1 ], [ 34, 34, 1 ], [ 35, 35, -1 ], [ 36, 15, -E(4) ], [ 37, 37, 1 ], [ 38, 38, 1 ], [ 39, 39, 1 ], 
  [ 40, 25, 1 ], [ 41, 61, E(4) ], [ 42, 42, -1 ], [ 43, 43, 1 ], [ 44, 63, E(4) ], [ 45, 30, -1 ], [ 46, 66, -E(4) ], [ 47, 47, -1 ], [ 48, 17, 1 ], 
  [ 49, 55, -E(4) ], [ 50, 68, -E(4) ], [ 51, 57, -E(4) ], [ 52, 23, -1 ], [ 53, 59, E(4) ], [ 54, 73, E(4) ], [ 55, 49, -E(4) ], [ 56, 62, E(4) ], [ 57, 51, -E(4) ], [ 58, 82, 1 ], 
  [ 59, 53, E(4) ], [ 60, 67, -E(4) ], [ 61, 41, E(4) ], [ 62, 56, E(4) ], [ 63, 44, E(4) ], [ 64, 76, 1 ], [ 65, 87, -1 ], [ 66, 46, -E(4) ], [ 67, 60, -E(4) ], 
  [ 68, 50, -E(4) ], [ 69, 69, -1 ], [ 70, 70, -1 ], [ 71, 71, -1 ], [ 72, 83, -1 ], [ 73, 54, E(4) ], [ 74, 74, 1 ], [ 75, 89, E(4) ], [ 76, 64, 1 ], [ 77, 77, 1 ], 
  [ 78, 78, 1 ], [ 79, 79, -1 ], [ 80, 85, -E(4) ], [ 81, 94, -E(4) ], [ 82, 58, 1 ], [ 83, 72, -1 ], [ 84, 96, -E(4) ], [ 85, 80, -E(4) ], [ 86, 90, E(4) ], [ 87, 65, -1 ], 
  [ 88, 92, E(4) ], [ 89, 75, E(4) ], [ 90, 86, E(4) ], [ 91, 106, -1 ], [ 92, 88, E(4) ], [ 93, 93, 1 ], [ 94, 81, -E(4) ], [ 95, 103, -1 ], [ 96, 84, -E(4) ], 
  [ 97, 97, -1 ], [ 98, 98, -1 ], [ 99, 99, 1 ], [ 100, 100, 1 ], [ 101, 101, 1 ], [ 102, 110, E(4) ], [ 103, 95, -1 ], [ 104, 111, -E(4) ], [ 105, 108, -E(4) ], [ 106, 91, -1 ], 
  [ 107, 109, E(4) ], [ 108, 105, -E(4) ], [ 109, 107, E(4) ], [ 110, 102, E(4) ], [ 111, 104, -E(4) ], [ 112, 112, -1 ], [ 113, 113, 1 ], [ 114, 114, -1 ], 
  [ 115, 118, -E(4) ], [ 116, 117, E(4) ], [ 117, 116, E(4) ], [ 118, 115, -E(4) ], [ 119, 119, -1 ], [ 120, 120, 1 ], [ 121, 121, -1 ], [ 122, 146, E(4) ], [ 123, 147, -E(4) ], 
  [ 124, 6, -1 ], [ 125, 5, 1 ], [ 126, 4, -1 ], [ 127, 148, -E(4) ], [ 128, 128, -1 ], [ 129, 151, E(4) ], [ 130, 138, -E(4) ], [ 131, 139, E(4) ], [ 132, 13, -1 ], 
  [ 133, 12, -1 ], [ 134, 141, E(4) ], [ 135, 156, E(4) ], [ 136, 144, -E(4) ], [ 137, 168, 1 ], [ 138, 130, -E(4) ], [ 139, 131, E(4) ], [ 140, 20, 1 ], [ 141, 134, E(4) ], 
  [ 142, 149, -E(4) ], [ 143, 172, -1 ], [ 144, 136, -E(4) ], [ 145, 160, 1 ], [ 146, 122, E(4) ], [ 147, 123, -E(4) ], [ 148, 127, -E(4) ], [ 149, 142, -E(4) ], 
  [ 150, 165, -1 ], [ 151, 129, E(4) ], [ 152, 152, -1 ], [ 153, 153, -1 ], [ 154, 154, 1 ], [ 155, 155, -1 ], [ 156, 135, E(4) ], [ 157, 157, 1 ], [ 158, 158, 1 ], [ 159, 159, 1 ], 
  [ 160, 145, 1 ], [ 161, 181, -E(4) ], [ 162, 162, -1 ], [ 163, 163, 1 ], [ 164, 183, -E(4) ], [ 165, 150, -1 ], [ 166, 186, E(4) ], [ 167, 167, -1 ], [ 168, 137, 1 ], 
  [ 169, 175, E(4) ], [ 170, 188, E(4) ], [ 171, 177, E(4) ], [ 172, 143, -1 ], [ 173, 179, -E(4) ], [ 174, 193, -E(4) ], [ 175, 169, E(4) ], [ 176, 182, -E(4) ], [ 177, 171, E(4) ], 
  [ 178, 202, 1 ], [ 179, 173, -E(4) ], [ 180, 187, E(4) ], [ 181, 161, -E(4) ], [ 182, 176, -E(4) ], [ 183, 164, -E(4) ], [ 184, 196, 1 ], [ 185, 207, -1 ], [ 186, 166, E(4) ], 
  [ 187, 180, E(4) ], [ 188, 170, E(4) ], [ 189, 189, -1 ], [ 190, 190, -1 ], [ 191, 191, -1 ], [ 192, 203, -1 ], [ 193, 174, -E(4) ], [ 194, 194, 1 ], [ 195, 209, -E(4) ], 
  [ 196, 184, 1 ], [ 197, 197, 1 ], [ 198, 198, 1 ], [ 199, 199, -1 ], [ 200, 205, E(4) ], [ 201, 214, E(4) ], [ 202, 178, 1 ], [ 203, 192, -1 ], [ 204, 216, E(4) ], [ 205, 200, E(4) ], 
  [ 206, 210, -E(4) ], [ 207, 185, -1 ], [ 208, 212, -E(4) ], [ 209, 195, -E(4) ], [ 210, 206, -E(4) ], [ 211, 226, -1 ], [ 212, 208, -E(4) ], [ 213, 213, 1 ], [ 214, 201, E(4) ], 
  [ 215, 223, -1 ], [ 216, 204, E(4) ], [ 217, 217, -1 ], [ 218, 218, -1 ], [ 219, 219, 1 ], [ 220, 220, 1 ], [ 221, 221, 1 ], [ 222, 230, -E(4) ], [ 223, 215, -1 ], 
  [ 224, 231, E(4) ], [ 225, 228, E(4) ], [ 226, 211, -1 ], [ 227, 229, -E(4) ], [ 228, 225, E(4) ], [ 229, 227, -E(4) ], [ 230, 222, -E(4) ], [ 231, 224, E(4) ], [ 232, 232, -1 ], 
  [ 233, 233, 1 ], [ 234, 234, -1 ], [ 235, 238, E(4) ], [ 236, 237, -E(4) ], [ 237, 236, -E(4) ], [ 238, 235, E(4) ], [ 239, 239, -1 ], [ 240, 240, 1 ], [ 241, 241, 1 ], 
  [ 242, 242, 1 ], [ 243, 243, 1 ], [ 244, 242, 1 ], [ 244, 243, 1 ], [ 244, 246, -1 ], [ 244, 247, 1 ], [ 245, 242, 1 ], [ 245, 243, 1 ], [ 245, 245, -1 ], [ 245, 247, 1 ], 
  [ 246, 242, 1 ], [ 246, 243, 1 ], [ 246, 244, -1 ], [ 246, 247, 1 ], [ 247, 247, 1 ], [ 248, 248, 1 ] ] ],

  [ [ 1, 0, 0, 0, 1, 0, 1, 0 ],
    [ 248, [ 1, 1, 1 ], [ 2, 2, -1 ], [ 3, 3, -1 ], [ 4, 4, -1 ], [ 5, 5, 1 ], [ 6, 6, 1 ], [ 7, 7, -1 ], [ 8, 8, -1 ], [ 9, 9, -1 ], 
  [ 10, 10, 1 ], [ 11, 11, 1 ], [ 12, 12, -1 ], [ 13, 13, 1 ], [ 14, 14, -1 ], [ 15, 15, 1 ], [ 16, 16, 1 ], [ 17, 17, -1 ], [ 18, 18, 1 ], [ 19, 19, 1 ], [ 20, 20, -1 ], 
  [ 21, 21, -1 ], [ 22, 22, 1 ], [ 23, 23, -1 ], [ 24, 24, 1 ], [ 25, 25, -1 ], [ 26, 26, 1 ], [ 27, 27, 1 ], [ 28, 28, 1 ], [ 29, 29, 1 ], [ 30, 30, -1 ], 
  [ 31, 31, 1 ], [ 32, 32, 1 ], [ 33, 33, -1 ], [ 34, 34, -1 ], [ 35, 35, -1 ], [ 36, 36, -1 ], [ 37, 37, 1 ], [ 38, 38, -1 ], [ 39, 39, -1 ], [ 40, 40, 1 ], [ 41, 41, 1 ], 
  [ 42, 42, 1 ], [ 43, 43, 1 ], [ 44, 44, -1 ], [ 45, 45, 1 ], [ 46, 46, 1 ], [ 47, 47, 1 ], [ 48, 48, 1 ], [ 49, 49, -1 ], [ 50, 50, -1 ], 
  [ 51, 51, -1 ], [ 52, 52, 1 ], [ 53, 53, -1 ], [ 54, 54, -1 ], [ 55, 55, -1 ], [ 56, 56, 1 ], [ 57, 57, -1 ], [ 58, 58, 1 ], [ 59, 59, -1 ], [ 60, 60, 1 ], [ 61, 61, -1 ], 
  [ 62, 62, 1 ], [ 63, 63, 1 ], [ 64, 64, 1 ], [ 65, 65, -1 ], [ 66, 66, -1 ], [ 67, 67, 1 ], [ 68, 68, 1 ], [ 69, 69, -1 ], [ 70, 70, -1 ], [ 71, 71, 1 ], 
  [ 72, 72, -1 ], [ 73, 73, 1 ], [ 74, 74, -1 ], [ 75, 75, 1 ], [ 76, 76, -1 ], [ 77, 77, 1 ], [ 78, 78, -1 ], [ 79, 79, -1 ], [ 80, 80, 1 ], [ 81, 81, -1 ], [ 82, 82, -1 ], 
  [ 83, 83, 1 ], [ 84, 84, 1 ], [ 85, 85, 1 ], [ 86, 86, -1 ], [ 87, 87, 1 ], [ 88, 88, -1 ], [ 89, 89, -1 ], [ 90, 90, -1 ], [ 91, 91, 1 ], [ 92, 92, -1 ], 
  [ 93, 93, 1 ], [ 94, 94, 1 ], [ 95, 95, 1 ], [ 96, 96, -1 ], [ 97, 97, 1 ], [ 98, 98, -1 ], [ 99, 99, -1 ], [ 100, 100, 1 ], [ 101, 101, -1 ], [ 102, 102, 1 ], [ 103, 103, -1 ], 
  [ 104, 104, 1 ], [ 105, 105, 1 ], [ 106, 106, -1 ], [ 107, 107, 1 ], [ 108, 108, 1 ], [ 109, 109, 1 ], [ 110, 110, -1 ], [ 111, 111, -1 ], [ 112, 112, 1 ], [ 113, 113, 1 ], 
  [ 114, 114, 1 ], [ 115, 115, -1 ], [ 116, 116, 1 ], [ 117, 117, 1 ], [ 118, 118, 1 ], [ 119, 119, -1 ], [ 120, 120, 1 ], [ 121, 121, 1 ], [ 122, 122, -1 ], [ 123, 123, -1 ], 
  [ 124, 124, -1 ], [ 125, 125, 1 ], [ 126, 126, 1 ], [ 127, 127, -1 ], [ 128, 128, -1 ], [ 129, 129, -1 ], [ 130, 130, 1 ], [ 131, 131, 1 ], [ 132, 132, -1 ], [ 133, 133, 1 ], 
  [ 134, 134, -1 ], [ 135, 135, 1 ], [ 136, 136, 1 ], [ 137, 137, -1 ], [ 138, 138, 1 ], [ 139, 139, 1 ], [ 140, 140, -1 ], [ 141, 141, -1 ], [ 142, 142, 1 ], [ 143, 143, -1 ], 
  [ 144, 144, 1 ], [ 145, 145, -1 ], [ 146, 146, 1 ], [ 147, 147, 1 ], [ 148, 148, 1 ], [ 149, 149, 1 ], [ 150, 150, -1 ], [ 151, 151, 1 ], [ 152, 152, 1 ], [ 153, 153, -1 ], 
  [ 154, 154, -1 ], [ 155, 155, -1 ], [ 156, 156, -1 ], [ 157, 157, 1 ], [ 158, 158, -1 ], [ 159, 159, -1 ], [ 160, 160, 1 ], [ 161, 161, 1 ], [ 162, 162, 1 ], [ 163, 163, 1 ], 
  [ 164, 164, -1 ], [ 165, 165, 1 ], [ 166, 166, 1 ], [ 167, 167, 1 ], [ 168, 168, 1 ], [ 169, 169, -1 ], [ 170, 170, -1 ], [ 171, 171, -1 ], [ 172, 172, 1 ], [ 173, 173, -1 ], 
  [ 174, 174, -1 ], [ 175, 175, -1 ], [ 176, 176, 1 ], [ 177, 177, -1 ], [ 178, 178, 1 ], [ 179, 179, -1 ], [ 180, 180, 1 ], [ 181, 181, -1 ], [ 182, 182, 1 ], [ 183, 183, 1 ], 
  [ 184, 184, 1 ], [ 185, 185, -1 ], [ 186, 186, -1 ], [ 187, 187, 1 ], [ 188, 188, 1 ], [ 189, 189, -1 ], [ 190, 190, -1 ], [ 191, 191, 1 ], [ 192, 192, -1 ], [ 193, 193, 1 ], 
  [ 194, 194, -1 ], [ 195, 195, 1 ], [ 196, 196, -1 ], [ 197, 197, 1 ], [ 198, 198, -1 ], [ 199, 199, -1 ], [ 200, 200, 1 ], [ 201, 201, -1 ], [ 202, 202, -1 ], [ 203, 203, 1 ], 
  [ 204, 204, 1 ], [ 205, 205, 1 ], [ 206, 206, -1 ], [ 207, 207, 1 ], [ 208, 208, -1 ], [ 209, 209, -1 ], [ 210, 210, -1 ], [ 211, 211, 1 ], [ 212, 212, -1 ], [ 213, 213, 1 ], 
  [ 214, 214, 1 ], [ 215, 215, 1 ], [ 216, 216, -1 ], [ 217, 217, 1 ], [ 218, 218, -1 ], [ 219, 219, -1 ], [ 220, 220, 1 ], [ 221, 221, -1 ], [ 222, 222, 1 ], [ 223, 223, -1 ], 
  [ 224, 224, 1 ], [ 225, 225, 1 ], [ 226, 226, -1 ], [ 227, 227, 1 ], [ 228, 228, 1 ], [ 229, 229, 1 ], [ 230, 230, -1 ], [ 231, 231, -1 ], [ 232, 232, 1 ], [ 233, 233, 1 ], 
  [ 234, 234, 1 ], [ 235, 235, -1 ], [ 236, 236, 1 ], [ 237, 237, 1 ], [ 238, 238, 1 ], [ 239, 239, -1 ], [ 240, 240, 1 ], [ 241, 241, 1 ], [ 242, 242, 1 ], [ 243, 243, 1 ], 
  [ 244, 244, 1 ], [ 245, 245, 1 ], [ 246, 246, 1 ], [ 247, 247, 1 ], [ 248, 248, 1 ] ] ],

  [ [ 0, 0, 0, 1, 0, 1, 0, 0 ],
    [ 248, [ 1, 123, 1 ], [ 2, 122, 1 ], [ 3, 121, 1 ], [ 4, 23, -1 ], [ 5, 5, 1 ], [ 6, 6, -1 ], [ 7, 15, 1 ], [ 8, 128, 1 ], [ 9, 129, 1 ], 
  [ 10, 16, -1 ], [ 11, 17, -1 ], [ 12, 30, -1 ], [ 13, 13, -1 ], [ 14, 22, -1 ], [ 15, 7, 1 ], [ 16, 10, -1 ], [ 17, 11, -1 ], [ 18, 24, -1 ], [ 19, 25, -1 ], [ 20, 38, 1 ], 
  [ 21, 29, -1 ], [ 22, 14, -1 ], [ 23, 4, -1 ], [ 24, 18, -1 ], [ 25, 19, -1 ], [ 26, 31, 1 ], [ 27, 33, 1 ], [ 28, 54, 1 ], [ 29, 21, -1 ], [ 30, 12, -1 ], 
  [ 31, 26, 1 ], [ 32, 44, 1 ], [ 33, 27, 1 ], [ 34, 47, 1 ], [ 35, 50, 1 ], [ 36, 46, 1 ], [ 37, 37, 1 ], [ 38, 20, 1 ], [ 39, 42, 1 ], [ 40, 51, -1 ], [ 41, 43, 1 ], 
  [ 42, 39, 1 ], [ 43, 41, 1 ], [ 44, 32, 1 ], [ 45, 45, -1 ], [ 46, 36, 1 ], [ 47, 34, 1 ], [ 48, 57, -1 ], [ 49, 65, -1 ], [ 50, 35, 1 ], [ 51, 40, -1 ], [ 52, 52, -1 ], 
  [ 53, 60, -1 ], [ 54, 28, 1 ], [ 55, 72, -1 ], [ 56, 58, -1 ], [ 57, 48, -1 ], [ 58, 56, -1 ], [ 59, 67, -1 ], [ 60, 53, -1 ], [ 61, 78, 1 ], [ 62, 64, -1 ], 
  [ 63, 69, -1 ], [ 64, 62, -1 ], [ 65, 49, -1 ], [ 66, 73, 1 ], [ 67, 59, -1 ], [ 68, 71, 1 ], [ 69, 63, -1 ], [ 70, 81, -1 ], [ 71, 68, 1 ], [ 72, 55, -1 ], [ 73, 66, 1 ], 
  [ 74, 84, -1 ], [ 75, 77, -1 ], [ 76, 86, 1 ], [ 77, 75, -1 ], [ 78, 61, 1 ], [ 79, 79, -1 ], [ 80, 83, 1 ], [ 81, 70, -1 ], [ 82, 90, 1 ], [ 83, 80, 1 ], 
  [ 84, 74, -1 ], [ 85, 87, 1 ], [ 86, 76, 1 ], [ 87, 85, 1 ], [ 88, 91, -1 ], [ 89, 101, -1 ], [ 90, 82, 1 ], [ 91, 88, -1 ], [ 92, 95, -1 ], [ 93, 98, -1 ], [ 94, 97, -1 ], 
  [ 95, 92, -1 ], [ 96, 100, 1 ], [ 97, 94, -1 ], [ 98, 93, -1 ], [ 99, 104, 1 ], [ 100, 96, 1 ], [ 101, 89, -1 ], [ 102, 102, 1 ], [ 103, 107, -1 ], [ 104, 99, 1 ], 
  [ 105, 105, -1 ], [ 106, 109, -1 ], [ 107, 103, -1 ], [ 108, 108, -1 ], [ 109, 106, -1 ], [ 110, 115, -1 ], [ 111, 113, -1 ], [ 112, 114, -1 ], [ 113, 111, -1 ], [ 114, 112, -1 ], 
  [ 115, 110, -1 ], [ 116, 116, -1 ], [ 117, 117, -1 ], [ 118, 118, 1 ], [ 119, 120, -1 ], [ 120, 119, -1 ], [ 121, 3, 1 ], [ 122, 2, 1 ], [ 123, 1, 1 ], [ 124, 143, -1 ], 
  [ 125, 125, 1 ], [ 126, 126, -1 ], [ 127, 135, 1 ], [ 128, 8, 1 ], [ 129, 9, 1 ], [ 130, 136, -1 ], [ 131, 137, -1 ], [ 132, 150, -1 ], [ 133, 133, -1 ], [ 134, 142, -1 ], 
  [ 135, 127, 1 ], [ 136, 130, -1 ], [ 137, 131, -1 ], [ 138, 144, -1 ], [ 139, 145, -1 ], [ 140, 158, 1 ], [ 141, 149, -1 ], [ 142, 134, -1 ], [ 143, 124, -1 ], [ 144, 138, -1 ], 
  [ 145, 139, -1 ], [ 146, 151, 1 ], [ 147, 153, 1 ], [ 148, 174, 1 ], [ 149, 141, -1 ], [ 150, 132, -1 ], [ 151, 146, 1 ], [ 152, 164, 1 ], [ 153, 147, 1 ], [ 154, 167, 1 ], 
  [ 155, 170, 1 ], [ 156, 166, 1 ], [ 157, 157, 1 ], [ 158, 140, 1 ], [ 159, 162, 1 ], [ 160, 171, -1 ], [ 161, 163, 1 ], [ 162, 159, 1 ], [ 163, 161, 1 ], [ 164, 152, 1 ], 
  [ 165, 165, -1 ], [ 166, 156, 1 ], [ 167, 154, 1 ], [ 168, 177, -1 ], [ 169, 185, -1 ], [ 170, 155, 1 ], [ 171, 160, -1 ], [ 172, 172, -1 ], [ 173, 180, -1 ], [ 174, 148, 1 ], 
  [ 175, 192, -1 ], [ 176, 178, -1 ], [ 177, 168, -1 ], [ 178, 176, -1 ], [ 179, 187, -1 ], [ 180, 173, -1 ], [ 181, 198, 1 ], [ 182, 184, -1 ], [ 183, 189, -1 ], [ 184, 182, -1 ], 
  [ 185, 169, -1 ], [ 186, 193, 1 ], [ 187, 179, -1 ], [ 188, 191, 1 ], [ 189, 183, -1 ], [ 190, 201, -1 ], [ 191, 188, 1 ], [ 192, 175, -1 ], [ 193, 186, 1 ], [ 194, 204, -1 ], 
  [ 195, 197, -1 ], [ 196, 206, 1 ], [ 197, 195, -1 ], [ 198, 181, 1 ], [ 199, 199, -1 ], [ 200, 203, 1 ], [ 201, 190, -1 ], [ 202, 210, 1 ], [ 203, 200, 1 ], [ 204, 194, -1 ], 
  [ 205, 207, 1 ], [ 206, 196, 1 ], [ 207, 205, 1 ], [ 208, 211, -1 ], [ 209, 221, -1 ], [ 210, 202, 1 ], [ 211, 208, -1 ], [ 212, 215, -1 ], [ 213, 218, -1 ], [ 214, 217, -1 ], 
  [ 215, 212, -1 ], [ 216, 220, 1 ], [ 217, 214, -1 ], [ 218, 213, -1 ], [ 219, 224, 1 ], [ 220, 216, 1 ], [ 221, 209, -1 ], [ 222, 222, 1 ], [ 223, 227, -1 ], [ 224, 219, 1 ], 
  [ 225, 225, -1 ], [ 226, 229, -1 ], [ 227, 223, -1 ], [ 228, 228, -1 ], [ 229, 226, -1 ], [ 230, 235, -1 ], [ 231, 233, -1 ], [ 232, 234, -1 ], [ 233, 231, -1 ], [ 234, 232, -1 ], 
  [ 235, 230, -1 ], [ 236, 236, -1 ], [ 237, 237, -1 ], [ 238, 238, 1 ], [ 239, 240, -1 ], [ 240, 239, -1 ], [ 241, 243, -1 ], [ 241, 244, 1 ], [ 242, 242, -1 ], [ 242, 244, 1 ], 
  [ 243, 241, -1 ], [ 243, 244, 1 ], [ 244, 244, 1 ], [ 245, 245, 1 ], [ 246, 246, 1 ], [ 247, 247, 1 ], [ 248, 247, 1 ], [ 248, 248, -1 ] ],
    [ 248, [ 1, 1, E(3)^2 ], [ 2, 2, E(3)^2 ], [ 3, 3, E(3)^2 ], [ 4, 4, 1 ], [ 5, 5, 1 ], [ 6, 6, 1 ], [ 7, 7, E(3)^2 ], [ 8, 8, E(3)^2 ], 
  [ 9, 9, E(3) ], [ 10, 10, E(3)^2 ], [ 11, 11, E(3)^2 ], [ 12, 12, 1 ], [ 13, 13, 1 ], [ 14, 14, E(3)^2 ], [ 15, 15, E(3) ], [ 16, 16, E(3) ], [ 17, 17, E(3) ], [ 18, 18, E(3)^2 ], 
  [ 19, 19, E(3)^2 ], [ 20, 20, 1 ], [ 21, 21, E(3)^2 ], [ 22, 22, E(3) ], [ 23, 23, 1 ], [ 24, 24, E(3) ], [ 25, 25, E(3) ], [ 26, 26, E(3)^2 ], 
  [ 27, 27, E(3)^2 ], [ 28, 28, E(3)^2 ], [ 29, 29, E(3) ], [ 30, 30, 1 ], [ 31, 31, E(3) ], [ 32, 32, E(3) ], [ 33, 33, E(3) ], [ 34, 34, E(3) ], [ 35, 35, E(3) ], [ 36, 36, E(3) ], 
  [ 37, 37, 1 ], [ 38, 38, 1 ], [ 39, 39, 1 ], [ 40, 40, E(3) ], [ 41, 41, 1 ], [ 42, 42, 1 ], [ 43, 43, 1 ], [ 44, 44, E(3)^2 ], [ 45, 45, 1 ], 
  [ 46, 46, E(3)^2 ], [ 47, 47, E(3)^2 ], [ 48, 48, E(3) ], [ 49, 49, 1 ], [ 50, 50, E(3)^2 ], [ 51, 51, E(3)^2 ], [ 52, 52, 1 ], [ 53, 53, E(3)^2 ], [ 54, 54, E(3) ], [ 55, 55, 1 ], 
  [ 56, 56, E(3)^2 ], [ 57, 57, E(3)^2 ], [ 58, 58, E(3) ], [ 59, 59, E(3)^2 ], [ 60, 60, E(3) ], [ 61, 61, 1 ], [ 62, 62, E(3)^2 ], [ 63, 63, E(3)^2 ], 
  [ 64, 64, E(3) ], [ 65, 65, 1 ], [ 66, 66, E(3)^2 ], [ 67, 67, E(3) ], [ 68, 68, E(3)^2 ], [ 69, 69, E(3) ], [ 70, 70, E(3) ], [ 71, 71, E(3) ], [ 72, 72, 1 ], [ 73, 73, E(3) ], 
  [ 74, 74, E(3) ], [ 75, 75, 1 ], [ 76, 76, E(3) ], [ 77, 77, 1 ], [ 78, 78, 1 ], [ 79, 79, 1 ], [ 80, 80, 1 ], [ 81, 81, E(3)^2 ], [ 82, 82, E(3) ], 
  [ 83, 83, 1 ], [ 84, 84, E(3)^2 ], [ 85, 85, 1 ], [ 86, 86, E(3)^2 ], [ 87, 87, 1 ], [ 88, 88, E(3)^2 ], [ 89, 89, 1 ], [ 90, 90, E(3)^2 ], [ 91, 91, E(3) ], [ 92, 92, E(3)^2 ], 
  [ 93, 93, E(3)^2 ], [ 94, 94, E(3)^2 ], [ 95, 95, E(3) ], [ 96, 96, E(3)^2 ], [ 97, 97, E(3) ], [ 98, 98, E(3) ], [ 99, 99, E(3) ], [ 100, 100, E(3) ], 
  [ 101, 101, 1 ], [ 102, 102, 1 ], [ 103, 103, E(3) ], [ 104, 104, E(3)^2 ], [ 105, 105, 1 ], [ 106, 106, E(3) ], [ 107, 107, E(3)^2 ], [ 108, 108, 1 ], [ 109, 109, E(3)^2 ], 
  [ 110, 110, 1 ], [ 111, 111, E(3)^2 ], [ 112, 112, E(3)^2 ], [ 113, 113, E(3) ], [ 114, 114, E(3) ], [ 115, 115, 1 ], [ 116, 116, 1 ], [ 117, 117, 1 ], [ 118, 118, 1 ], 
  [ 119, 119, E(3)^2 ], [ 120, 120, E(3) ], [ 121, 121, E(3) ], [ 122, 122, E(3) ], [ 123, 123, E(3) ], [ 124, 124, 1 ], [ 125, 125, 1 ], [ 126, 126, 1 ], [ 127, 127, E(3) ], 
  [ 128, 128, E(3) ], [ 129, 129, E(3)^2 ], [ 130, 130, E(3) ], [ 131, 131, E(3) ], [ 132, 132, 1 ], [ 133, 133, 1 ], [ 134, 134, E(3) ], [ 135, 135, E(3)^2 ], [ 136, 136, E(3)^2 ], 
  [ 137, 137, E(3)^2 ], [ 138, 138, E(3) ], [ 139, 139, E(3) ], [ 140, 140, 1 ], [ 141, 141, E(3) ], [ 142, 142, E(3)^2 ], [ 143, 143, 1 ], [ 144, 144, E(3)^2 ], 
  [ 145, 145, E(3)^2 ], [ 146, 146, E(3) ], [ 147, 147, E(3) ], [ 148, 148, E(3) ], [ 149, 149, E(3)^2 ], [ 150, 150, 1 ], [ 151, 151, E(3)^2 ], [ 152, 152, E(3)^2 ], 
  [ 153, 153, E(3)^2 ], [ 154, 154, E(3)^2 ], [ 155, 155, E(3)^2 ], [ 156, 156, E(3)^2 ], [ 157, 157, 1 ], [ 158, 158, 1 ], [ 159, 159, 1 ], [ 160, 160, E(3)^2 ], 
  [ 161, 161, 1 ], [ 162, 162, 1 ], [ 163, 163, 1 ], [ 164, 164, E(3) ], [ 165, 165, 1 ], [ 166, 166, E(3) ], [ 167, 167, E(3) ], [ 168, 168, E(3)^2 ], [ 169, 169, 1 ], 
  [ 170, 170, E(3) ], [ 171, 171, E(3) ], [ 172, 172, 1 ], [ 173, 173, E(3) ], [ 174, 174, E(3)^2 ], [ 175, 175, 1 ], [ 176, 176, E(3) ], [ 177, 177, E(3) ], 
  [ 178, 178, E(3)^2 ], [ 179, 179, E(3) ], [ 180, 180, E(3)^2 ], [ 181, 181, 1 ], [ 182, 182, E(3) ], [ 183, 183, E(3) ], [ 184, 184, E(3)^2 ], [ 185, 185, 1 ], [ 186, 186, E(3) ], 
  [ 187, 187, E(3)^2 ], [ 188, 188, E(3) ], [ 189, 189, E(3)^2 ], [ 190, 190, E(3)^2 ], [ 191, 191, E(3)^2 ], [ 192, 192, 1 ], [ 193, 193, E(3)^2 ], [ 194, 194, E(3)^2 ], 
  [ 195, 195, 1 ], [ 196, 196, E(3)^2 ], [ 197, 197, 1 ], [ 198, 198, 1 ], [ 199, 199, 1 ], [ 200, 200, 1 ], [ 201, 201, E(3) ], [ 202, 202, E(3)^2 ], [ 203, 203, 1 ], 
  [ 204, 204, E(3) ], [ 205, 205, 1 ], [ 206, 206, E(3) ], [ 207, 207, 1 ], [ 208, 208, E(3) ], [ 209, 209, 1 ], [ 210, 210, E(3) ], [ 211, 211, E(3)^2 ], 
  [ 212, 212, E(3) ], [ 213, 213, E(3) ], [ 214, 214, E(3) ], [ 215, 215, E(3)^2 ], [ 216, 216, E(3) ], [ 217, 217, E(3)^2 ], [ 218, 218, E(3)^2 ], [ 219, 219, E(3)^2 ], 
  [ 220, 220, E(3)^2 ], [ 221, 221, 1 ], [ 222, 222, 1 ], [ 223, 223, E(3)^2 ], [ 224, 224, E(3) ], [ 225, 225, 1 ], [ 226, 226, E(3)^2 ], [ 227, 227, E(3) ], 
  [ 228, 228, 1 ], [ 229, 229, E(3) ], [ 230, 230, 1 ], [ 231, 231, E(3) ], [ 232, 232, E(3) ], [ 233, 233, E(3)^2 ], [ 234, 234, E(3)^2 ], [ 235, 235, 1 ], [ 236, 236, 1 ], 
  [ 237, 237, 1 ], [ 238, 238, 1 ], [ 239, 239, E(3) ], [ 240, 240, E(3)^2 ], [ 241, 241, 1 ], [ 242, 242, 1 ], [ 243, 243, 1 ], [ 244, 244, 1 ], [ 245, 245, 1 ], 
  [ 246, 246, 1 ], [ 247, 247, 1 ], [ 248, 248, 1 ] ] ],

  [ [ 0, 1, 1, 0, 0, 0, 1, 2 ],
    [ 248, [ 1, 1, -1 ], [ 2, 26, -1 ], [ 3, 27, -1 ], [ 4, 126, -1 ], [ 5, 125, 1 ], [ 6, 124, -1 ], [ 7, 28, -1 ], [ 8, 8, 1 ], [ 9, 31, 1 ], 
  [ 10, 18, 1 ], [ 11, 19, 1 ], [ 12, 133, -1 ], [ 13, 132, -1 ], [ 14, 21, 1 ], [ 15, 36, -1 ], [ 16, 24, -1 ], [ 17, 48, 1 ], [ 18, 10, 1 ], [ 19, 11, 1 ], [ 20, 140, 1 ], 
  [ 21, 14, 1 ], [ 22, 29, 1 ], [ 23, 52, -1 ], [ 24, 16, -1 ], [ 25, 40, 1 ], [ 26, 2, -1 ], [ 27, 3, -1 ], [ 28, 7, -1 ], [ 29, 22, 1 ], [ 30, 45, -1 ], 
  [ 31, 9, 1 ], [ 32, 32, -1 ], [ 33, 33, -1 ], [ 34, 34, 1 ], [ 35, 35, 1 ], [ 36, 15, -1 ], [ 37, 37, 1 ], [ 38, 38, 1 ], [ 39, 39, -1 ], [ 40, 25, 1 ], [ 41, 61, -1 ], [ 42, 42, 1 ], 
  [ 43, 43, 1 ], [ 44, 63, -1 ], [ 45, 30, -1 ], [ 46, 66, 1 ], [ 47, 47, -1 ], [ 48, 17, 1 ], [ 49, 55, 1 ], [ 50, 68, -1 ], [ 51, 57, 1 ], [ 52, 23, -1 ], 
  [ 53, 59, -1 ], [ 54, 73, 1 ], [ 55, 49, 1 ], [ 56, 62, 1 ], [ 57, 51, 1 ], [ 58, 82, -1 ], [ 59, 53, -1 ], [ 60, 67, -1 ], [ 61, 41, -1 ], [ 62, 56, 1 ], [ 63, 44, -1 ], 
  [ 64, 76, -1 ], [ 65, 87, -1 ], [ 66, 46, 1 ], [ 67, 60, -1 ], [ 68, 50, -1 ], [ 69, 69, -1 ], [ 70, 70, 1 ], [ 71, 71, 1 ], [ 72, 83, -1 ], [ 73, 54, 1 ], 
  [ 74, 74, 1 ], [ 75, 89, -1 ], [ 76, 64, -1 ], [ 77, 77, 1 ], [ 78, 78, 1 ], [ 79, 79, -1 ], [ 80, 85, 1 ], [ 81, 94, -1 ], [ 82, 58, -1 ], [ 83, 72, -1 ], [ 84, 96, 1 ], 
  [ 85, 80, 1 ], [ 86, 90, 1 ], [ 87, 65, -1 ], [ 88, 92, -1 ], [ 89, 75, -1 ], [ 90, 86, 1 ], [ 91, 106, -1 ], [ 92, 88, -1 ], [ 93, 93, -1 ], [ 94, 81, -1 ], 
  [ 95, 103, -1 ], [ 96, 84, 1 ], [ 97, 97, 1 ], [ 98, 98, -1 ], [ 99, 99, 1 ], [ 100, 100, 1 ], [ 101, 101, 1 ], [ 102, 110, -1 ], [ 103, 95, -1 ], [ 104, 111, 1 ], [ 105, 108, 1 ], 
  [ 106, 91, -1 ], [ 107, 109, -1 ], [ 108, 105, 1 ], [ 109, 107, -1 ], [ 110, 102, -1 ], [ 111, 104, 1 ], [ 112, 112, -1 ], [ 113, 113, 1 ], [ 114, 114, 1 ], [ 115, 118, 1 ], 
  [ 116, 117, -1 ], [ 117, 116, -1 ], [ 118, 115, 1 ], [ 119, 119, 1 ], [ 120, 120, 1 ], [ 121, 121, -1 ], [ 122, 146, -1 ], [ 123, 147, -1 ], [ 124, 6, -1 ], [ 125, 5, 1 ], 
  [ 126, 4, -1 ], [ 127, 148, -1 ], [ 128, 128, 1 ], [ 129, 151, 1 ], [ 130, 138, 1 ], [ 131, 139, 1 ], [ 132, 13, -1 ], [ 133, 12, -1 ], [ 134, 141, 1 ], [ 135, 156, -1 ], 
  [ 136, 144, -1 ], [ 137, 168, 1 ], [ 138, 130, 1 ], [ 139, 131, 1 ], [ 140, 20, 1 ], [ 141, 134, 1 ], [ 142, 149, 1 ], [ 143, 172, -1 ], [ 144, 136, -1 ], [ 145, 160, 1 ], 
  [ 146, 122, -1 ], [ 147, 123, -1 ], [ 148, 127, -1 ], [ 149, 142, 1 ], [ 150, 165, -1 ], [ 151, 129, 1 ], [ 152, 152, -1 ], [ 153, 153, -1 ], [ 154, 154, 1 ], [ 155, 155, 1 ], 
  [ 156, 135, -1 ], [ 157, 157, 1 ], [ 158, 158, 1 ], [ 159, 159, -1 ], [ 160, 145, 1 ], [ 161, 181, -1 ], [ 162, 162, 1 ], [ 163, 163, 1 ], [ 164, 183, -1 ], [ 165, 150, -1 ], 
  [ 166, 186, 1 ], [ 167, 167, -1 ], [ 168, 137, 1 ], [ 169, 175, 1 ], [ 170, 188, -1 ], [ 171, 177, 1 ], [ 172, 143, -1 ], [ 173, 179, -1 ], [ 174, 193, 1 ], [ 175, 169, 1 ], 
  [ 176, 182, 1 ], [ 177, 171, 1 ], [ 178, 202, -1 ], [ 179, 173, -1 ], [ 180, 187, -1 ], [ 181, 161, -1 ], [ 182, 176, 1 ], [ 183, 164, -1 ], [ 184, 196, -1 ], [ 185, 207, -1 ], 
  [ 186, 166, 1 ], [ 187, 180, -1 ], [ 188, 170, -1 ], [ 189, 189, -1 ], [ 190, 190, 1 ], [ 191, 191, 1 ], [ 192, 203, -1 ], [ 193, 174, 1 ], [ 194, 194, 1 ], [ 195, 209, -1 ], 
  [ 196, 184, -1 ], [ 197, 197, 1 ], [ 198, 198, 1 ], [ 199, 199, -1 ], [ 200, 205, 1 ], [ 201, 214, -1 ], [ 202, 178, -1 ], [ 203, 192, -1 ], [ 204, 216, 1 ], [ 205, 200, 1 ], 
  [ 206, 210, 1 ], [ 207, 185, -1 ], [ 208, 212, -1 ], [ 209, 195, -1 ], [ 210, 206, 1 ], [ 211, 226, -1 ], [ 212, 208, -1 ], [ 213, 213, -1 ], [ 214, 201, -1 ], [ 215, 223, -1 ], 
  [ 216, 204, 1 ], [ 217, 217, 1 ], [ 218, 218, -1 ], [ 219, 219, 1 ], [ 220, 220, 1 ], [ 221, 221, 1 ], [ 222, 230, -1 ], [ 223, 215, -1 ], [ 224, 231, 1 ], [ 225, 228, 1 ], 
  [ 226, 211, -1 ], [ 227, 229, -1 ], [ 228, 225, 1 ], [ 229, 227, -1 ], [ 230, 222, -1 ], [ 231, 224, 1 ], [ 232, 232, -1 ], [ 233, 233, 1 ], [ 234, 234, 1 ], [ 235, 238, 1 ], 
  [ 236, 237, -1 ], [ 237, 236, -1 ], [ 238, 235, 1 ], [ 239, 239, 1 ], [ 240, 240, 1 ], [ 241, 241, 1 ], [ 242, 242, 1 ], [ 243, 243, 1 ], [ 244, 242, 1 ], [ 244, 243, 1 ], 
  [ 244, 246, -1 ], [ 244, 247, 1 ], [ 245, 242, 1 ], [ 245, 243, 1 ], [ 245, 245, -1 ], [ 245, 247, 1 ], [ 246, 242, 1 ], [ 246, 243, 1 ], [ 246, 244, -1 ], [ 246, 247, 1 ], 
  [ 247, 247, 1 ], [ 248, 248, 1 ] ] ],

  [ [ 0, 0, 0, 1, 0, 1, 0, 2 ],
    [ 248, [ 1, 1, 1 ], [ 2, 2, -1 ], [ 3, 3, -1 ], [ 4, 4, -1 ], [ 5, 5, 1 ], [ 6, 6, -1 ], [ 7, 7, -1 ], [ 8, 8, 1 ], [ 9, 9, -1 ], 
  [ 10, 10, 1 ], [ 11, 11, 1 ], [ 12, 12, -1 ], [ 13, 13, -1 ], [ 14, 14, 1 ], [ 15, 15, -1 ], [ 16, 16, 1 ], [ 17, 17, -1 ], [ 18, 18, 1 ], [ 19, 19, 1 ], [ 20, 20, 1 ], [ 21, 21, 1 ], 
  [ 22, 22, 1 ], [ 23, 23, -1 ], [ 24, 24, 1 ], [ 25, 25, -1 ], [ 26, 26, -1 ], [ 27, 27, -1 ], [ 28, 28, -1 ], [ 29, 29, 1 ], [ 30, 30, -1 ], [ 31, 31, -1 ], 
  [ 32, 32, 1 ], [ 33, 33, 1 ], [ 34, 34, 1 ], [ 35, 35, 1 ], [ 36, 36, -1 ], [ 37, 37, 1 ], [ 38, 38, 1 ], [ 39, 39, 1 ], [ 40, 40, -1 ], [ 41, 41, -1 ], [ 42, 42, 1 ], 
  [ 43, 43, 1 ], [ 44, 44, -1 ], [ 45, 45, -1 ], [ 46, 46, -1 ], [ 47, 47, 1 ], [ 48, 48, -1 ], [ 49, 49, 1 ], [ 50, 50, -1 ], [ 51, 51, 1 ], [ 52, 52, -1 ], [ 53, 53, 1 ], 
  [ 54, 54, -1 ], [ 55, 55, 1 ], [ 56, 56, 1 ], [ 57, 57, 1 ], [ 58, 58, -1 ], [ 59, 59, 1 ], [ 60, 60, 1 ], [ 61, 61, -1 ], [ 62, 62, 1 ], [ 63, 63, -1 ], 
  [ 64, 64, -1 ], [ 65, 65, -1 ], [ 66, 66, -1 ], [ 67, 67, 1 ], [ 68, 68, -1 ], [ 69, 69, 1 ], [ 70, 70, 1 ], [ 71, 71, 1 ], [ 72, 72, -1 ], [ 73, 73, -1 ], [ 74, 74, 1 ], 
  [ 75, 75, -1 ], [ 76, 76, -1 ], [ 77, 77, 1 ], [ 78, 78, 1 ], [ 79, 79, 1 ], [ 80, 80, 1 ], [ 81, 81, -1 ], [ 82, 82, -1 ], [ 83, 83, -1 ], [ 84, 84, -1 ], 
  [ 85, 85, 1 ], [ 86, 86, 1 ], [ 87, 87, -1 ], [ 88, 88, 1 ], [ 89, 89, -1 ], [ 90, 90, 1 ], [ 91, 91, -1 ], [ 92, 92, 1 ], [ 93, 93, 1 ], [ 94, 94, -1 ], [ 95, 95, -1 ], 
  [ 96, 96, -1 ], [ 97, 97, 1 ], [ 98, 98, 1 ], [ 99, 99, 1 ], [ 100, 100, 1 ], [ 101, 101, 1 ], [ 102, 102, -1 ], [ 103, 103, -1 ], [ 104, 104, -1 ], [ 105, 105, 1 ], 
  [ 106, 106, -1 ], [ 107, 107, 1 ], [ 108, 108, 1 ], [ 109, 109, 1 ], [ 110, 110, -1 ], [ 111, 111, -1 ], [ 112, 112, 1 ], [ 113, 113, 1 ], [ 114, 114, 1 ], [ 115, 115, -1 ], 
  [ 116, 116, 1 ], [ 117, 117, 1 ], [ 118, 118, -1 ], [ 119, 119, 1 ], [ 120, 120, 1 ], [ 121, 121, 1 ], [ 122, 122, -1 ], [ 123, 123, -1 ], [ 124, 124, -1 ], 
  [ 125, 125, 1 ], [ 126, 126, -1 ], [ 127, 127, -1 ], [ 128, 128, 1 ], [ 129, 129, -1 ], [ 130, 130, 1 ], [ 131, 131, 1 ], [ 132, 132, -1 ], [ 133, 133, -1 ], [ 134, 134, 1 ], 
  [ 135, 135, -1 ], [ 136, 136, 1 ], [ 137, 137, -1 ], [ 138, 138, 1 ], [ 139, 139, 1 ], [ 140, 140, 1 ], [ 141, 141, 1 ], [ 142, 142, 1 ], [ 143, 143, -1 ], [ 144, 144, 1 ], 
  [ 145, 145, -1 ], [ 146, 146, -1 ], [ 147, 147, -1 ], [ 148, 148, -1 ], [ 149, 149, 1 ], [ 150, 150, -1 ], [ 151, 151, -1 ], [ 152, 152, 1 ], [ 153, 153, 1 ], [ 154, 154, 1 ], 
  [ 155, 155, 1 ], [ 156, 156, -1 ], [ 157, 157, 1 ], [ 158, 158, 1 ], [ 159, 159, 1 ], [ 160, 160, -1 ], [ 161, 161, -1 ], [ 162, 162, 1 ], [ 163, 163, 1 ], [ 164, 164, -1 ], 
  [ 165, 165, -1 ], [ 166, 166, -1 ], [ 167, 167, 1 ], [ 168, 168, -1 ], [ 169, 169, 1 ], [ 170, 170, -1 ], [ 171, 171, 1 ], [ 172, 172, -1 ], [ 173, 173, 1 ], [ 174, 174, -1 ], 
  [ 175, 175, 1 ], [ 176, 176, 1 ], [ 177, 177, 1 ], [ 178, 178, -1 ], [ 179, 179, 1 ], [ 180, 180, 1 ], [ 181, 181, -1 ], [ 182, 182, 1 ], [ 183, 183, -1 ], [ 184, 184, -1 ], 
  [ 185, 185, -1 ], [ 186, 186, -1 ], [ 187, 187, 1 ], [ 188, 188, -1 ], [ 189, 189, 1 ], [ 190, 190, 1 ], [ 191, 191, 1 ], [ 192, 192, -1 ], [ 193, 193, -1 ], [ 194, 194, 1 ], 
  [ 195, 195, -1 ], [ 196, 196, -1 ], [ 197, 197, 1 ], [ 198, 198, 1 ], [ 199, 199, 1 ], [ 200, 200, 1 ], [ 201, 201, -1 ], [ 202, 202, -1 ], [ 203, 203, -1 ], [ 204, 204, -1 ], 
  [ 205, 205, 1 ], [ 206, 206, 1 ], [ 207, 207, -1 ], [ 208, 208, 1 ], [ 209, 209, -1 ], [ 210, 210, 1 ], [ 211, 211, -1 ], [ 212, 212, 1 ], [ 213, 213, 1 ], [ 214, 214, -1 ], 
  [ 215, 215, -1 ], [ 216, 216, -1 ], [ 217, 217, 1 ], [ 218, 218, 1 ], [ 219, 219, 1 ], [ 220, 220, 1 ], [ 221, 221, 1 ], [ 222, 222, -1 ], [ 223, 223, -1 ], [ 224, 224, -1 ], 
  [ 225, 225, 1 ], [ 226, 226, -1 ], [ 227, 227, 1 ], [ 228, 228, 1 ], [ 229, 229, 1 ], [ 230, 230, -1 ], [ 231, 231, -1 ], [ 232, 232, 1 ], [ 233, 233, 1 ], [ 234, 234, 1 ], 
  [ 235, 235, -1 ], [ 236, 236, 1 ], [ 237, 237, 1 ], [ 238, 238, -1 ], [ 239, 239, 1 ], [ 240, 240, 1 ], [ 241, 241, 1 ], [ 242, 242, 1 ], [ 243, 243, 1 ], [ 244, 244, 1 ], 
  [ 245, 245, 1 ], [ 246, 246, 1 ], [ 247, 247, 1 ], [ 248, 248, 1 ] ] ],

  [ [ 2, 0, 0, 0, 0, 2, 0, 2 ],
    [ 248, [ 1, 1, 1/2 ], [ 1, 16, -1/4 ], [ 1, 23, -1/4 ], [ 1, 24, 1/2 ], [ 1, 30, -1/2 ], [ 1, 44, 1/2 ], [ 2, 2, -1/4 ], [ 2, 3, -1/4 ], [ 2, 4, 1/8 ], 
  [ 2, 10, -1/8 ], [ 2, 11, -1/8 ], [ 2, 12, -1/4 ], [ 2, 17, 1/8 ], [ 2, 18, -1/4 ], [ 2, 19, -1/4 ], [ 2, 25, -1/4 ], [ 2, 32, -1/4 ], [ 2, 122, -1/4 ], [ 2, 123, -1/4 ], 
  [ 2, 124, -1/4 ], [ 2, 130, 1/4 ], [ 2, 131, 1/4 ], [ 2, 132, 1/8 ], [ 2, 137, -1/4 ], [ 2, 138, 1/8 ], [ 2, 139, 1/8 ], [ 2, 145, 1/8 ], [ 2, 152, -1/4 ], 
  [ 2, 244, -1/4 ], [ 2, 245, 1/2 ], [ 2, 246, -1/4 ], [ 3, 2, -1/4 ], [ 3, 3, -1/4 ], [ 3, 4, -1/8 ], [ 3, 10, -1/8 ], [ 3, 11, -1/8 ], [ 3, 12, 1/4 ], [ 3, 17, -1/8 ], 
  [ 3, 18, -1/4 ], [ 3, 19, -1/4 ], [ 3, 25, 1/4 ], [ 3, 32, 1/4 ], [ 3, 122, -1/4 ], [ 3, 123, -1/4 ], [ 3, 124, 1/4 ], [ 3, 130, 1/4 ], [ 3, 131, 1/4 ], [ 3, 132, -1/8 ], 
  [ 3, 137, 1/4 ], [ 3, 138, 1/8 ], [ 3, 139, 1/8 ], [ 3, 145, -1/8 ], [ 3, 152, 1/4 ], [ 3, 244, 1/4 ], [ 3, 245, -1/2 ], [ 3, 246, 1/4 ], [ 4, 2, 1/4 ], 
  [ 4, 3, -1/4 ], [ 4, 4, -1/8 ], [ 4, 5, -1 ], [ 4, 10, -1/8 ], [ 4, 11, 1/8 ], [ 4, 12, -3/4 ], [ 4, 17, 1/8 ], [ 4, 18, -1/4 ], [ 4, 19, 1/4 ], [ 4, 25, -1/4 ], [ 4, 32, 1/4 ], 
  [ 4, 122, -1/4 ], [ 4, 123, 1/4 ], [ 4, 124, 3/4 ], [ 4, 130, -1/4 ], [ 4, 131, 1/4 ], [ 4, 132, 1/8 ], [ 4, 137, 1/4 ], [ 4, 138, -1/8 ], [ 4, 139, 1/8 ], 
  [ 4, 145, -1/8 ], [ 4, 152, -1/4 ], [ 4, 242, 1/2 ], [ 4, 243, 1/2 ], [ 4, 244, -3/4 ], [ 4, 246, 1/4 ], [ 5, 4, -1/8 ], [ 5, 10, 1/8 ], [ 5, 11, -1/8 ], [ 5, 17, 1/8 ], 
  [ 5, 132, -1/8 ], [ 5, 138, -1/8 ], [ 5, 139, 1/8 ], [ 5, 145, 1/8 ], [ 6, 13, -1 ], [ 6, 20, 1/2 ], [ 6, 26, 1/2 ], [ 6, 27, -1/2 ], [ 6, 33, -1/2 ], [ 6, 48, -1 ], 
  [ 7, 7, -1 ], [ 8, 8, 1 ], [ 9, 9, -1/2 ], [ 9, 16, -1/4 ], [ 9, 23, -1/4 ], [ 9, 24, -1/2 ], [ 9, 30, 1/2 ], [ 9, 37, 1/2 ], [ 10, 2, -1/4 ], [ 10, 3, -1/4 ], 
  [ 10, 4, -1/8 ], [ 10, 5, 1 ], [ 10, 10, -1/8 ], [ 10, 11, 1/8 ], [ 10, 12, 1/4 ], [ 10, 17, 1/8 ], [ 10, 18, 3/4 ], [ 10, 19, 1/4 ], [ 10, 25, -1/4 ], [ 10, 32, 1/4 ], 
  [ 10, 122, 1/4 ], [ 10, 123, 1/4 ], [ 10, 124, -1/4 ], [ 10, 130, 3/4 ], [ 10, 131, 1/4 ], [ 10, 132, 1/8 ], [ 10, 137, 1/4 ], [ 10, 138, -1/8 ], [ 10, 139, 1/8 ], [ 10, 145, -1/8 ], 
  [ 10, 152, -1/4 ], [ 10, 242, -1/2 ], [ 10, 243, 1/2 ], [ 10, 244, -1/4 ], [ 10, 246, 1/4 ], [ 11, 2, -1/4 ], [ 11, 3, -1/4 ], [ 11, 4, 1/8 ], [ 11, 5, -1 ], 
  [ 11, 10, 1/8 ], [ 11, 11, -1/8 ], [ 11, 12, -1/4 ], [ 11, 17, -1/8 ], [ 11, 18, 1/4 ], [ 11, 19, 3/4 ], [ 11, 25, 1/4 ], [ 11, 32, -1/4 ], [ 11, 122, 1/4 ], [ 11, 123, 1/4 ], 
  [ 11, 124, 1/4 ], [ 11, 130, 1/4 ], [ 11, 131, 3/4 ], [ 11, 132, -1/8 ], [ 11, 137, -1/4 ], [ 11, 138, 1/8 ], [ 11, 139, -1/8 ], [ 11, 145, 1/8 ], [ 11, 152, 1/4 ], 
  [ 11, 241, -1/2 ], [ 11, 242, -1/2 ], [ 11, 243, 1/2 ], [ 11, 244, 1/4 ], [ 11, 246, -1/4 ], [ 12, 2, -1/8 ], [ 12, 3, 1/8 ], [ 12, 4, -3/16 ], [ 12, 10, 1/16 ], [ 12, 11, -1/16 ], 
  [ 12, 12, -1/8 ], [ 12, 17, -1/16 ], [ 12, 18, 1/8 ], [ 12, 19, -1/8 ], [ 12, 25, 1/8 ], [ 12, 32, -1/8 ], [ 12, 122, 1/8 ], [ 12, 123, -1/8 ], [ 12, 124, 1/8 ], 
  [ 12, 125, -1/8 ], [ 12, 130, 1/8 ], [ 12, 131, -1/8 ], [ 12, 132, 3/16 ], [ 12, 137, -1/8 ], [ 12, 138, 1/16 ], [ 12, 139, -1/16 ], [ 12, 145, 1/16 ], [ 12, 152, 1/8 ], 
  [ 12, 242, 1/4 ], [ 12, 243, 1/4 ], [ 12, 244, -3/8 ], [ 12, 246, 1/8 ], [ 13, 6, -1/4 ], [ 13, 20, 1/4 ], [ 13, 26, -1/4 ], [ 13, 27, 1/4 ], [ 13, 33, -1/4 ], 
  [ 13, 40, -1/4 ], [ 14, 21, 1 ], [ 14, 28, -1/2 ], [ 14, 34, -1/2 ], [ 14, 35, 1/2 ], [ 14, 41, 1/2 ], [ 14, 55, 1 ], [ 15, 15, -1 ], [ 16, 1, -1/2 ], [ 16, 9, -1/2 ], 
  [ 16, 24, 1 ], [ 16, 30, 1 ], [ 16, 37, -1/2 ], [ 16, 44, 1/2 ], [ 17, 2, 1/4 ], [ 17, 3, -1/4 ], [ 17, 4, 1/8 ], [ 17, 5, 1 ], [ 17, 10, 1/8 ], [ 17, 11, -1/8 ], 
  [ 17, 12, -1/4 ], [ 17, 17, -1/8 ], [ 17, 18, 1/4 ], [ 17, 19, -1/4 ], [ 17, 25, -3/4 ], [ 17, 32, -1/4 ], [ 17, 122, -1/4 ], [ 17, 123, 1/4 ], [ 17, 124, 1/4 ], [ 17, 130, 1/4 ], 
  [ 17, 131, -1/4 ], [ 17, 132, -1/8 ], [ 17, 137, 3/4 ], [ 17, 138, 1/8 ], [ 17, 139, -1/8 ], [ 17, 145, 1/8 ], [ 17, 152, 1/4 ], [ 17, 241, -1/2 ], [ 17, 242, 1/2 ], 
  [ 17, 243, 1/2 ], [ 17, 244, -1/4 ], [ 17, 246, -1/4 ], [ 18, 2, -1/8 ], [ 18, 3, -1/8 ], [ 18, 4, -1/16 ], [ 18, 10, 3/16 ], [ 18, 11, 1/16 ], [ 18, 12, 1/8 ], [ 18, 17, 1/16 ], 
  [ 18, 18, -1/8 ], [ 18, 19, 1/8 ], [ 18, 25, -1/8 ], [ 18, 32, 1/8 ], [ 18, 122, 1/8 ], [ 18, 123, 1/8 ], [ 18, 124, -1/8 ], [ 18, 125, -1/8 ], [ 18, 130, -1/8 ], 
  [ 18, 131, 1/8 ], [ 18, 132, 1/16 ], [ 18, 137, 1/8 ], [ 18, 138, 3/16 ], [ 18, 139, 1/16 ], [ 18, 145, -1/16 ], [ 18, 152, -1/8 ], [ 18, 242, 1/4 ], [ 18, 243, -1/4 ], 
  [ 18, 244, 1/8 ], [ 18, 246, -1/8 ], [ 19, 2, -1/8 ], [ 19, 3, -1/8 ], [ 19, 4, 1/16 ], [ 19, 10, 1/16 ], [ 19, 11, 3/16 ], [ 19, 12, -1/8 ], 
  [ 19, 17, -1/16 ], [ 19, 18, 1/8 ], [ 19, 19, -1/8 ], [ 19, 25, 1/8 ], [ 19, 32, -1/8 ], [ 19, 122, 1/8 ], [ 19, 123, 1/8 ], [ 19, 124, 1/8 ], [ 19, 125, 1/8 ], [ 19, 130, 1/8 ], 
  [ 19, 131, -1/8 ], [ 19, 132, -1/16 ], [ 19, 137, -1/8 ], [ 19, 138, 1/16 ], [ 19, 139, 3/16 ], [ 19, 145, 1/16 ], [ 19, 152, 1/8 ], [ 19, 241, 1/4 ], [ 19, 242, 1/4 ], 
  [ 19, 243, -1/4 ], [ 19, 244, -1/8 ], [ 19, 246, 1/8 ], [ 20, 6, 1/4 ], [ 20, 13, 1/2 ], [ 20, 20, 1/2 ], [ 20, 33, 1/2 ], [ 20, 40, -1/4 ], [ 20, 48, -1/2 ], 
  [ 21, 14, 1/4 ], [ 21, 28, -1/4 ], [ 21, 34, 1/4 ], [ 21, 35, -1/4 ], [ 21, 41, 1/4 ], [ 21, 49, 1/4 ], [ 22, 29, 1 ], [ 22, 36, -1/2 ], [ 22, 42, -1/2 ], [ 22, 43, 1/2 ], 
  [ 22, 50, 1/2 ], [ 22, 62, 1 ], [ 23, 1, -1/2 ], [ 23, 9, -1/2 ], [ 23, 24, -1 ], [ 23, 30, -1 ], [ 23, 37, -1/2 ], [ 23, 44, 1/2 ], [ 24, 1, 1/4 ], [ 24, 9, -1/4 ], 
  [ 24, 16, 1/4 ], [ 24, 23, -1/4 ], [ 24, 37, -1/4 ], [ 24, 44, -1/4 ], [ 25, 2, -1/8 ], [ 25, 3, 1/8 ], [ 25, 4, -1/16 ], [ 25, 10, -1/16 ], [ 25, 11, 1/16 ], [ 25, 12, 1/8 ], 
  [ 25, 17, -3/16 ], [ 25, 18, -1/8 ], [ 25, 19, 1/8 ], [ 25, 25, -1/8 ], [ 25, 32, 1/8 ], [ 25, 122, 1/8 ], [ 25, 123, -1/8 ], [ 25, 124, -1/8 ], [ 25, 125, 1/8 ], 
  [ 25, 130, -1/8 ], [ 25, 131, 1/8 ], [ 25, 132, 1/16 ], [ 25, 137, 1/8 ], [ 25, 138, -1/16 ], [ 25, 139, 1/16 ], [ 25, 145, 3/16 ], [ 25, 152, -1/8 ], [ 25, 241, -1/4 ], 
  [ 25, 242, 1/4 ], [ 25, 243, 1/4 ], [ 25, 244, -1/8 ], [ 25, 246, -1/8 ], [ 26, 6, 1/4 ], [ 26, 13, -1/2 ], [ 26, 26, -1/2 ], [ 26, 27, -1/2 ], [ 26, 40, -1/4 ], 
  [ 26, 48, 1/2 ], [ 27, 6, -1/4 ], [ 27, 13, 1/2 ], [ 27, 26, -1/2 ], [ 27, 27, -1/2 ], [ 27, 40, 1/4 ], [ 27, 48, -1/2 ], [ 28, 14, -1/4 ], [ 28, 21, -1/2 ], [ 28, 28, -1/2 ], 
  [ 28, 41, -1/2 ], [ 28, 49, 1/4 ], [ 28, 55, 1/2 ], [ 29, 22, 1/4 ], [ 29, 36, -1/4 ], [ 29, 42, 1/4 ], [ 29, 43, -1/4 ], [ 29, 50, 1/4 ], [ 29, 56, 1/4 ], 
  [ 30, 1, -1/4 ], [ 30, 9, 1/4 ], [ 30, 16, 1/4 ], [ 30, 23, -1/4 ], [ 30, 37, 1/4 ], [ 30, 44, 1/4 ], [ 31, 31, -1/2 ], [ 31, 45, 1/4 ], [ 31, 51, -1/4 ], [ 31, 52, -1/2 ], 
  [ 31, 57, -1/2 ], [ 31, 69, 1/2 ], [ 32, 2, -1/4 ], [ 32, 3, 1/4 ], [ 32, 4, 1/8 ], [ 32, 10, 1/8 ], [ 32, 11, -1/8 ], [ 32, 12, -1/4 ], [ 32, 17, -1/8 ], 
  [ 32, 18, 1/4 ], [ 32, 19, -1/4 ], [ 32, 25, 1/4 ], [ 32, 32, -1/4 ], [ 32, 122, -1/4 ], [ 32, 123, 1/4 ], [ 32, 124, -1/4 ], [ 32, 130, -1/4 ], [ 32, 131, 1/4 ], [ 32, 132, 1/8 ], 
  [ 32, 137, 1/4 ], [ 32, 138, -1/8 ], [ 32, 139, 1/8 ], [ 32, 145, -1/8 ], [ 32, 152, -1/4 ], [ 32, 244, 1/4 ], [ 32, 245, -1/2 ], [ 32, 246, 1/4 ], [ 33, 6, -1/4 ], 
  [ 33, 13, -1/2 ], [ 33, 20, 1/2 ], [ 33, 33, 1/2 ], [ 33, 40, 1/4 ], [ 33, 48, 1/2 ], [ 34, 14, -1/4 ], [ 34, 21, 1/2 ], [ 34, 34, 1/2 ], [ 34, 35, 1/2 ], [ 34, 49, 1/4 ], 
  [ 34, 55, -1/2 ], [ 35, 14, 1/4 ], [ 35, 21, -1/2 ], [ 35, 34, 1/2 ], [ 35, 35, 1/2 ], [ 35, 49, -1/4 ], [ 35, 55, 1/2 ], [ 36, 22, -1/4 ], [ 36, 29, -1/2 ], 
  [ 36, 36, -1/2 ], [ 36, 50, -1/2 ], [ 36, 56, 1/4 ], [ 36, 62, 1/2 ], [ 37, 9, 1/2 ], [ 37, 16, -1/4 ], [ 37, 23, -1/4 ], [ 37, 24, -1/2 ], [ 37, 30, 1/2 ], [ 37, 37, -1/2 ], 
  [ 38, 38, 1/2 ], [ 38, 45, 1/4 ], [ 38, 51, -1/4 ], [ 38, 52, 1/2 ], [ 38, 57, 1/2 ], [ 38, 63, 1/2 ], [ 39, 39, 1/2 ], [ 39, 53, -1/4 ], [ 39, 58, 1/4 ], 
  [ 39, 59, 1/2 ], [ 39, 64, 1/2 ], [ 39, 75, -1/2 ], [ 40, 13, -1 ], [ 40, 20, -1/2 ], [ 40, 26, -1/2 ], [ 40, 27, 1/2 ], [ 40, 33, 1/2 ], [ 40, 48, -1 ], [ 41, 14, 1/4 ], 
  [ 41, 21, 1/2 ], [ 41, 28, -1/2 ], [ 41, 41, -1/2 ], [ 41, 49, -1/4 ], [ 41, 55, -1/2 ], [ 42, 22, -1/4 ], [ 42, 29, 1/2 ], [ 42, 42, 1/2 ], [ 42, 43, 1/2 ], 
  [ 42, 56, 1/4 ], [ 42, 62, -1/2 ], [ 43, 22, 1/4 ], [ 43, 29, -1/2 ], [ 43, 42, 1/2 ], [ 43, 43, 1/2 ], [ 43, 56, -1/4 ], [ 43, 62, 1/2 ], [ 44, 1, 1/2 ], [ 44, 16, 1/4 ], 
  [ 44, 23, 1/4 ], [ 44, 24, -1/2 ], [ 44, 30, 1/2 ], [ 44, 44, 1/2 ], [ 45, 31, 1/2 ], [ 45, 38, 1/2 ], [ 45, 52, -1 ], [ 45, 57, 1 ], [ 45, 63, -1/2 ], [ 45, 69, 1/2 ], 
  [ 46, 46, -1/2 ], [ 46, 53, -1/4 ], [ 46, 58, 1/4 ], [ 46, 59, -1/2 ], [ 46, 64, -1/2 ], [ 46, 70, -1/2 ], [ 47, 47, 1/2 ], [ 47, 60, -1/4 ], [ 47, 65, 1/4 ], [ 47, 67, 1/2 ], 
  [ 47, 72, 1/2 ], [ 47, 81, -1/2 ], [ 48, 6, -1/4 ], [ 48, 20, -1/4 ], [ 48, 26, 1/4 ], [ 48, 27, -1/4 ], [ 48, 33, 1/4 ], [ 48, 40, -1/4 ], [ 49, 21, 1 ], 
  [ 49, 28, 1/2 ], [ 49, 34, 1/2 ], [ 49, 35, -1/2 ], [ 49, 41, -1/2 ], [ 49, 55, 1 ], [ 50, 22, 1/4 ], [ 50, 29, 1/2 ], [ 50, 36, -1/2 ], [ 50, 50, -1/2 ], [ 50, 56, -1/4 ], 
  [ 50, 62, -1/2 ], [ 51, 31, -1/2 ], [ 51, 38, -1/2 ], [ 51, 52, -1 ], [ 51, 57, 1 ], [ 51, 63, 1/2 ], [ 51, 69, -1/2 ], [ 52, 31, -1/4 ], [ 52, 38, 1/4 ], 
  [ 52, 45, -1/4 ], [ 52, 51, -1/4 ], [ 52, 63, -1/4 ], [ 52, 69, -1/4 ], [ 53, 39, -1/2 ], [ 53, 46, -1/2 ], [ 53, 59, 1 ], [ 53, 64, -1 ], [ 53, 70, 1/2 ], [ 53, 75, -1/2 ], 
  [ 54, 54, -1/2 ], [ 54, 60, -1/4 ], [ 54, 65, 1/4 ], [ 54, 67, -1/2 ], [ 54, 72, -1/2 ], [ 54, 77, -1/2 ], [ 55, 14, 1/4 ], [ 55, 28, 1/4 ], [ 55, 34, -1/4 ], 
  [ 55, 35, 1/4 ], [ 55, 41, -1/4 ], [ 55, 49, 1/4 ], [ 56, 29, 1 ], [ 56, 36, 1/2 ], [ 56, 42, 1/2 ], [ 56, 43, -1/2 ], [ 56, 50, -1/2 ], [ 56, 62, 1 ], [ 57, 31, -1/4 ], 
  [ 57, 38, 1/4 ], [ 57, 45, 1/4 ], [ 57, 51, 1/4 ], [ 57, 63, -1/4 ], [ 57, 69, -1/4 ], [ 58, 39, 1/2 ], [ 58, 46, 1/2 ], [ 58, 59, 1 ], [ 58, 64, -1 ], [ 58, 70, -1/2 ], 
  [ 58, 75, 1/2 ], [ 59, 39, 1/4 ], [ 59, 46, -1/4 ], [ 59, 53, 1/4 ], [ 59, 58, 1/4 ], [ 59, 70, 1/4 ], [ 59, 75, 1/4 ], [ 60, 47, -1/2 ], [ 60, 54, -1/2 ], [ 60, 67, 1 ], 
  [ 60, 72, -1 ], [ 60, 77, 1/2 ], [ 60, 81, -1/2 ], [ 61, 61, -1 ], [ 62, 22, 1/4 ], [ 62, 36, 1/4 ], [ 62, 42, -1/4 ], [ 62, 43, 1/4 ], [ 62, 50, -1/4 ], [ 62, 56, 1/4 ], 
  [ 63, 38, 1/2 ], [ 63, 45, -1/4 ], [ 63, 51, 1/4 ], [ 63, 52, -1/2 ], [ 63, 57, -1/2 ], [ 63, 63, 1/2 ], [ 64, 39, 1/4 ], [ 64, 46, -1/4 ], [ 64, 53, -1/4 ], 
  [ 64, 58, -1/4 ], [ 64, 70, 1/4 ], [ 64, 75, 1/4 ], [ 65, 47, 1/2 ], [ 65, 54, 1/2 ], [ 65, 67, 1 ], [ 65, 72, -1 ], [ 65, 77, -1/2 ], [ 65, 81, 1/2 ], [ 66, 66, -1/2 ], 
  [ 66, 76, -1/4 ], [ 66, 80, 1/4 ], [ 66, 82, 1/2 ], [ 66, 85, 1/2 ], [ 66, 93, 1/2 ], [ 67, 47, 1/4 ], [ 67, 54, -1/4 ], [ 67, 60, 1/4 ], [ 67, 65, 1/4 ], 
  [ 67, 77, 1/4 ], [ 67, 81, 1/4 ], [ 68, 68, -1 ], [ 69, 31, 1/2 ], [ 69, 45, 1/4 ], [ 69, 51, -1/4 ], [ 69, 52, -1/2 ], [ 69, 57, -1/2 ], [ 69, 69, -1/2 ], [ 70, 46, -1/2 ], 
  [ 70, 53, 1/4 ], [ 70, 58, -1/4 ], [ 70, 59, 1/2 ], [ 70, 64, 1/2 ], [ 70, 70, -1/2 ], [ 71, 71, 1/2 ], [ 71, 76, 1/4 ], [ 71, 80, -1/4 ], [ 71, 82, 1/2 ], 
  [ 71, 85, 1/2 ], [ 71, 89, 1/2 ], [ 72, 47, 1/4 ], [ 72, 54, -1/4 ], [ 72, 60, -1/4 ], [ 72, 65, -1/4 ], [ 72, 77, 1/4 ], [ 72, 81, 1/4 ], [ 73, 73, -1/2 ], [ 73, 83, -1/4 ], 
  [ 73, 86, 1/4 ], [ 73, 87, 1/2 ], [ 73, 90, 1/2 ], [ 73, 98, 1/2 ], [ 74, 74, 1 ], [ 75, 39, -1/2 ], [ 75, 53, -1/4 ], [ 75, 58, 1/4 ], [ 75, 59, 1/2 ], 
  [ 75, 64, 1/2 ], [ 75, 75, 1/2 ], [ 76, 66, -1/2 ], [ 76, 71, 1/2 ], [ 76, 82, -1 ], [ 76, 85, 1 ], [ 76, 89, -1/2 ], [ 76, 93, -1/2 ], [ 77, 54, -1/2 ], [ 77, 60, 1/4 ], 
  [ 77, 65, -1/4 ], [ 77, 67, 1/2 ], [ 77, 72, 1/2 ], [ 77, 77, -1/2 ], [ 78, 78, 1/2 ], [ 78, 83, 1/4 ], [ 78, 86, -1/4 ], [ 78, 87, 1/2 ], [ 78, 90, 1/2 ], 
  [ 78, 94, 1/2 ], [ 79, 79, 1/2 ], [ 79, 88, 1/4 ], [ 79, 91, -1/4 ], [ 79, 92, -1/2 ], [ 79, 95, -1/2 ], [ 79, 102, -1/2 ], [ 80, 66, 1/2 ], [ 80, 71, -1/2 ], [ 80, 82, -1 ], 
  [ 80, 85, 1 ], [ 80, 89, 1/2 ], [ 80, 93, 1/2 ], [ 81, 47, -1/2 ], [ 81, 60, -1/4 ], [ 81, 65, 1/4 ], [ 81, 67, 1/2 ], [ 81, 72, 1/2 ], [ 81, 81, 1/2 ], 
  [ 82, 66, 1/4 ], [ 82, 71, 1/4 ], [ 82, 76, -1/4 ], [ 82, 80, -1/4 ], [ 82, 89, -1/4 ], [ 82, 93, 1/4 ], [ 83, 73, -1/2 ], [ 83, 78, 1/2 ], [ 83, 87, -1 ], [ 83, 90, 1 ], 
  [ 83, 94, -1/2 ], [ 83, 98, -1/2 ], [ 84, 84, -1/2 ], [ 84, 88, -1/4 ], [ 84, 91, 1/4 ], [ 84, 92, -1/2 ], [ 84, 95, -1/2 ], [ 84, 99, -1/2 ], [ 85, 66, 1/4 ], 
  [ 85, 71, 1/4 ], [ 85, 76, 1/4 ], [ 85, 80, 1/4 ], [ 85, 89, -1/4 ], [ 85, 93, 1/4 ], [ 86, 73, 1/2 ], [ 86, 78, -1/2 ], [ 86, 87, -1 ], [ 86, 90, 1 ], [ 86, 94, 1/2 ], 
  [ 86, 98, 1/2 ], [ 87, 73, 1/4 ], [ 87, 78, 1/4 ], [ 87, 83, -1/4 ], [ 87, 86, -1/4 ], [ 87, 94, -1/4 ], [ 87, 98, 1/4 ], [ 88, 79, 1/2 ], [ 88, 84, -1/2 ], [ 88, 92, 1 ], 
  [ 88, 95, -1 ], [ 88, 99, 1/2 ], [ 88, 102, 1/2 ], [ 89, 71, 1/2 ], [ 89, 76, -1/4 ], [ 89, 80, 1/4 ], [ 89, 82, -1/2 ], [ 89, 85, -1/2 ], [ 89, 89, 1/2 ], 
  [ 90, 73, 1/4 ], [ 90, 78, 1/4 ], [ 90, 83, 1/4 ], [ 90, 86, 1/4 ], [ 90, 94, -1/4 ], [ 90, 98, 1/4 ], [ 91, 79, -1/2 ], [ 91, 84, 1/2 ], [ 91, 92, 1 ], [ 91, 95, -1 ], 
  [ 91, 99, -1/2 ], [ 91, 102, -1/2 ], [ 92, 79, -1/4 ], [ 92, 84, -1/4 ], [ 92, 88, 1/4 ], [ 92, 91, 1/4 ], [ 92, 99, 1/4 ], [ 92, 102, -1/4 ], 
  [ 93, 66, 1/2 ], [ 93, 76, -1/4 ], [ 93, 80, 1/4 ], [ 93, 82, 1/2 ], [ 93, 85, 1/2 ], [ 93, 93, -1/2 ], [ 94, 78, 1/2 ], [ 94, 83, -1/4 ], [ 94, 86, 1/4 ], [ 94, 87, -1/2 ], 
  [ 94, 90, -1/2 ], [ 94, 94, 1/2 ], [ 95, 79, -1/4 ], [ 95, 84, -1/4 ], [ 95, 88, -1/4 ], [ 95, 91, -1/4 ], [ 95, 99, 1/4 ], [ 95, 102, -1/4 ], [ 96, 96, -1/2 ], 
  [ 96, 103, -1/4 ], [ 96, 105, -1/4 ], [ 96, 106, 1/2 ], [ 96, 108, -1/2 ], [ 96, 112, -1/2 ], [ 97, 97, -1 ], [ 98, 73, 1/2 ], [ 98, 83, -1/4 ], [ 98, 86, 1/4 ], [ 98, 87, 1/2 ], 
  [ 98, 90, 1/2 ], [ 98, 98, -1/2 ], [ 99, 84, -1/2 ], [ 99, 88, 1/4 ], [ 99, 91, -1/4 ], [ 99, 92, 1/2 ], [ 99, 95, 1/2 ], [ 99, 99, -1/2 ], [ 100, 100, 1/2 ], 
  [ 100, 103, 1/4 ], [ 100, 105, 1/4 ], [ 100, 106, 1/2 ], [ 100, 108, -1/2 ], [ 100, 110, -1/2 ], [ 101, 101, -1 ], [ 102, 79, -1/2 ], [ 102, 88, 1/4 ], [ 102, 91, -1/4 ], 
  [ 102, 92, -1/2 ], [ 102, 95, -1/2 ], [ 102, 102, 1/2 ], [ 103, 96, -1/2 ], [ 103, 100, 1/2 ], [ 103, 106, -1 ], [ 103, 108, -1 ], [ 103, 110, 1/2 ], [ 103, 112, 1/2 ], 
  [ 104, 104, 1 ], [ 105, 96, -1/2 ], [ 105, 100, 1/2 ], [ 105, 106, 1 ], [ 105, 108, 1 ], [ 105, 110, 1/2 ], [ 105, 112, 1/2 ], [ 106, 96, 1/4 ], [ 106, 100, 1/4 ], [ 106, 103, -1/4 ], 
  [ 106, 105, 1/4 ], [ 106, 110, 1/4 ], [ 106, 112, -1/4 ], [ 107, 109, 1 ], [ 107, 111, 1/2 ], [ 107, 113, 1/2 ], [ 107, 114, 1/2 ], [ 107, 115, 1/2 ], [ 107, 117, 1 ], 
  [ 108, 96, -1/4 ], [ 108, 100, -1/4 ], [ 108, 103, -1/4 ], [ 108, 105, 1/4 ], [ 108, 110, -1/4 ], [ 108, 112, 1/4 ], [ 109, 107, 1/4 ], [ 109, 111, -1/4 ], [ 109, 113, 1/4 ], 
  [ 109, 114, 1/4 ], [ 109, 115, -1/4 ], [ 109, 116, 1/4 ], [ 110, 100, -1/2 ], [ 110, 103, 1/4 ], [ 110, 105, 1/4 ], [ 110, 106, 1/2 ], [ 110, 108, -1/2 ], [ 110, 110, 1/2 ], 
  [ 111, 107, 1/4 ], [ 111, 109, -1/2 ], [ 111, 111, 1/2 ], [ 111, 115, -1/2 ], [ 111, 116, -1/4 ], [ 111, 117, 1/2 ], [ 112, 96, -1/2 ], [ 112, 103, 1/4 ], [ 112, 105, 1/4 ], 
  [ 112, 106, -1/2 ], [ 112, 108, 1/2 ], [ 112, 112, -1/2 ], [ 113, 107, 1/4 ], [ 113, 109, 1/2 ], [ 113, 113, -1/2 ], [ 113, 114, 1/2 ], [ 113, 116, -1/4 ], [ 113, 117, -1/2 ], 
  [ 114, 107, 1/4 ], [ 114, 109, 1/2 ], [ 114, 113, 1/2 ], [ 114, 114, -1/2 ], [ 114, 116, -1/4 ], [ 114, 117, -1/2 ], [ 115, 107, 1/4 ], [ 115, 109, -1/2 ], [ 115, 111, -1/2 ], 
  [ 115, 115, 1/2 ], [ 115, 116, -1/4 ], [ 115, 117, 1/2 ], [ 116, 109, 1 ], [ 116, 111, -1/2 ], [ 116, 113, -1/2 ], [ 116, 114, -1/2 ], [ 116, 115, -1/2 ], [ 116, 117, 1 ], 
  [ 117, 107, 1/4 ], [ 117, 111, 1/4 ], [ 117, 113, -1/4 ], [ 117, 114, -1/4 ], [ 117, 115, 1/4 ], [ 117, 116, 1/4 ], [ 118, 118, 1 ], [ 119, 119, -1 ], [ 120, 120, -1 ], 
  [ 121, 121, 1/2 ], [ 121, 136, -1/2 ], [ 121, 143, -1/2 ], [ 121, 144, 1/4 ], [ 121, 150, -1/4 ], [ 121, 164, 1/2 ], [ 122, 2, -1/4 ], [ 122, 3, -1/4 ], [ 122, 4, -1/8 ], 
  [ 122, 10, 1/8 ], [ 122, 11, 1/8 ], [ 122, 12, 1/4 ], [ 122, 17, -1/8 ], [ 122, 18, 1/4 ], [ 122, 19, 1/4 ], [ 122, 25, 1/4 ], [ 122, 32, -1/4 ], [ 122, 122, -1/4 ], 
  [ 122, 123, -1/4 ], [ 122, 124, 1/4 ], [ 122, 130, -1/4 ], [ 122, 131, -1/4 ], [ 122, 132, -1/8 ], [ 122, 137, 1/4 ], [ 122, 138, -1/8 ], [ 122, 139, -1/8 ], [ 122, 145, -1/8 ], 
  [ 122, 152, -1/4 ], [ 122, 244, -1/4 ], [ 122, 245, 1/2 ], [ 122, 246, -1/4 ], [ 123, 2, -1/4 ], [ 123, 3, -1/4 ], [ 123, 4, 1/8 ], [ 123, 10, 1/8 ], [ 123, 11, 1/8 ], 
  [ 123, 12, -1/4 ], [ 123, 17, 1/8 ], [ 123, 18, 1/4 ], [ 123, 19, 1/4 ], [ 123, 25, -1/4 ], [ 123, 32, 1/4 ], [ 123, 122, -1/4 ], [ 123, 123, -1/4 ], [ 123, 124, -1/4 ], 
  [ 123, 130, -1/4 ], [ 123, 131, -1/4 ], [ 123, 132, 1/8 ], [ 123, 137, -1/4 ], [ 123, 138, -1/8 ], [ 123, 139, -1/8 ], [ 123, 145, 1/8 ], [ 123, 152, 1/4 ], [ 123, 244, 1/4 ], 
  [ 123, 245, -1/2 ], [ 123, 246, 1/4 ], [ 124, 2, -1/8 ], [ 124, 3, 1/8 ], [ 124, 4, 3/16 ], [ 124, 10, -1/16 ], [ 124, 11, 1/16 ], [ 124, 12, 1/8 ], [ 124, 17, 1/16 ], 
  [ 124, 18, -1/8 ], [ 124, 19, 1/8 ], [ 124, 25, -1/8 ], [ 124, 32, -1/8 ], [ 124, 122, 1/8 ], [ 124, 123, -1/8 ], [ 124, 124, -1/8 ], [ 124, 125, -1/8 ], [ 124, 130, -1/8 ], 
  [ 124, 131, 1/8 ], [ 124, 132, -3/16 ], [ 124, 137, 1/8 ], [ 124, 138, -1/16 ], [ 124, 139, 1/16 ], [ 124, 145, -1/16 ], [ 124, 152, 1/8 ], [ 124, 242, 1/4 ], [ 124, 243, 1/4 ], 
  [ 124, 244, -3/8 ], [ 124, 246, 1/8 ], [ 125, 12, -1 ], [ 125, 18, -1 ], [ 125, 19, 1 ], [ 125, 25, 1 ], [ 125, 124, -1 ], [ 125, 130, 1 ], [ 125, 131, -1 ], [ 125, 137, 1 ], 
  [ 126, 133, -1/4 ], [ 126, 140, 1/4 ], [ 126, 146, 1/4 ], [ 126, 147, -1/4 ], [ 126, 153, -1/4 ], [ 126, 168, -1/4 ], [ 127, 127, -1 ], [ 128, 128, 1 ], [ 129, 129, -1/2 ], 
  [ 129, 136, -1/2 ], [ 129, 143, -1/2 ], [ 129, 144, -1/4 ], [ 129, 150, 1/4 ], [ 129, 157, 1/2 ], [ 130, 2, 1/8 ], [ 130, 3, 1/8 ], [ 130, 4, -1/16 ], [ 130, 10, 3/16 ], 
  [ 130, 11, 1/16 ], [ 130, 12, 1/8 ], [ 130, 17, 1/16 ], [ 130, 18, -1/8 ], [ 130, 19, 1/8 ], [ 130, 25, -1/8 ], [ 130, 32, -1/8 ], [ 130, 122, -1/8 ], [ 130, 123, -1/8 ], 
  [ 130, 124, -1/8 ], [ 130, 125, 1/8 ], [ 130, 130, -1/8 ], [ 130, 131, 1/8 ], [ 130, 132, 1/16 ], [ 130, 137, 1/8 ], [ 130, 138, 3/16 ], [ 130, 139, 1/16 ], [ 130, 145, -1/16 ], 
  [ 130, 152, 1/8 ], [ 130, 242, -1/4 ], [ 130, 243, 1/4 ], [ 130, 244, -1/8 ], [ 130, 246, 1/8 ], [ 131, 2, 1/8 ], [ 131, 3, 1/8 ], [ 131, 4, 1/16 ], [ 131, 10, 1/16 ], 
  [ 131, 11, 3/16 ], [ 131, 12, -1/8 ], [ 131, 17, -1/16 ], [ 131, 18, 1/8 ], [ 131, 19, -1/8 ], [ 131, 25, 1/8 ], [ 131, 32, 1/8 ], [ 131, 122, -1/8 ], [ 131, 123, -1/8 ], 
  [ 131, 124, 1/8 ], [ 131, 125, -1/8 ], [ 131, 130, 1/8 ], [ 131, 131, -1/8 ], [ 131, 132, -1/16 ], [ 131, 137, -1/8 ], [ 131, 138, 1/16 ], [ 131, 139, 3/16 ], [ 131, 145, 1/16 ], 
  [ 131, 152, -1/8 ], [ 131, 241, -1/4 ], [ 131, 242, -1/4 ], [ 131, 243, 1/4 ], [ 131, 244, 1/8 ], [ 131, 246, -1/8 ], [ 132, 2, 1/4 ], [ 132, 3, -1/4 ], [ 132, 4, 1/8 ], 
  [ 132, 5, -1 ], [ 132, 10, 1/8 ], [ 132, 11, -1/8 ], [ 132, 12, 3/4 ], [ 132, 17, -1/8 ], [ 132, 18, 1/4 ], [ 132, 19, -1/4 ], [ 132, 25, 1/4 ], [ 132, 32, 1/4 ], [ 132, 122, -1/4 ], 
  [ 132, 123, 1/4 ], [ 132, 124, -3/4 ], [ 132, 130, 1/4 ], [ 132, 131, -1/4 ], [ 132, 132, -1/8 ], [ 132, 137, -1/4 ], [ 132, 138, 1/8 ], [ 132, 139, -1/8 ], [ 132, 145, 1/8 ], 
  [ 132, 152, -1/4 ], [ 132, 242, 1/2 ], [ 132, 243, 1/2 ], [ 132, 244, -3/4 ], [ 132, 246, 1/4 ], [ 133, 126, -1 ], [ 133, 140, 1/2 ], [ 133, 146, -1/2 ], [ 133, 147, 1/2 ], 
  [ 133, 153, -1/2 ], [ 133, 160, -1 ], [ 134, 141, 1/4 ], [ 134, 148, -1/4 ], [ 134, 154, -1/4 ], [ 134, 155, 1/4 ], [ 134, 161, 1/4 ], [ 134, 175, 1/4 ], [ 135, 135, -1 ], 
  [ 136, 121, -1/4 ], [ 136, 129, -1/4 ], [ 136, 144, 1/4 ], [ 136, 150, 1/4 ], [ 136, 157, -1/4 ], [ 136, 164, 1/4 ], [ 137, 2, -1/8 ], [ 137, 3, 1/8 ], [ 137, 4, 1/16 ], 
  [ 137, 10, 1/16 ], [ 137, 11, -1/16 ], [ 137, 12, -1/8 ], [ 137, 17, 3/16 ], [ 137, 18, 1/8 ], [ 137, 19, -1/8 ], [ 137, 25, 1/8 ], [ 137, 32, 1/8 ], [ 137, 122, 1/8 ], 
  [ 137, 123, -1/8 ], [ 137, 124, 1/8 ], [ 137, 125, 1/8 ], [ 137, 130, 1/8 ], [ 137, 131, -1/8 ], [ 137, 132, -1/16 ], [ 137, 137, -1/8 ], [ 137, 138, 1/16 ], [ 137, 139, -1/16 ], 
  [ 137, 145, -3/16 ], [ 137, 152, -1/8 ], [ 137, 241, -1/4 ], [ 137, 242, 1/4 ], [ 137, 243, 1/4 ], [ 137, 244, -1/8 ], [ 137, 246, -1/8 ], [ 138, 2, 1/4 ], [ 138, 3, 1/4 ], 
  [ 138, 4, -1/8 ], [ 138, 5, -1 ], [ 138, 10, -1/8 ], [ 138, 11, 1/8 ], [ 138, 12, 1/4 ], [ 138, 17, 1/8 ], [ 138, 18, 3/4 ], [ 138, 19, 1/4 ], [ 138, 25, -1/4 ], 
  [ 138, 32, -1/4 ], [ 138, 122, -1/4 ], [ 138, 123, -1/4 ], [ 138, 124, -1/4 ], [ 138, 130, 3/4 ], [ 138, 131, 1/4 ], [ 138, 132, 1/8 ], [ 138, 137, 1/4 ], [ 138, 138, -1/8 ], 
  [ 138, 139, 1/8 ], [ 138, 145, -1/8 ], [ 138, 152, 1/4 ], [ 138, 242, 1/2 ], [ 138, 243, -1/2 ], [ 138, 244, 1/4 ], [ 138, 246, -1/4 ], [ 139, 2, 1/4 ], [ 139, 3, 1/4 ], 
  [ 139, 4, 1/8 ], [ 139, 5, 1 ], [ 139, 10, 1/8 ], [ 139, 11, -1/8 ], [ 139, 12, -1/4 ], [ 139, 17, -1/8 ], [ 139, 18, 1/4 ], [ 139, 19, 3/4 ], [ 139, 25, 1/4 ], [ 139, 32, 1/4 ], 
  [ 139, 122, -1/4 ], [ 139, 123, -1/4 ], [ 139, 124, 1/4 ], [ 139, 130, 1/4 ], [ 139, 131, 3/4 ], [ 139, 132, -1/8 ], [ 139, 137, -1/4 ], [ 139, 138, 1/8 ], [ 139, 139, -1/8 ], 
  [ 139, 145, 1/8 ], [ 139, 152, -1/4 ], [ 139, 241, 1/2 ], [ 139, 242, 1/2 ], [ 139, 243, -1/2 ], [ 139, 244, -1/4 ], [ 139, 246, 1/4 ], [ 140, 126, 1/2 ], [ 140, 133, 1/4 ], 
  [ 140, 140, 1/2 ], [ 140, 153, 1/2 ], [ 140, 160, -1/2 ], [ 140, 168, -1/4 ], [ 141, 134, 1 ], [ 141, 148, -1/2 ], [ 141, 154, 1/2 ], [ 141, 155, -1/2 ], [ 141, 161, 1/2 ], 
  [ 141, 169, 1 ], [ 142, 149, 1/4 ], [ 142, 156, -1/4 ], [ 142, 162, -1/4 ], [ 142, 163, 1/4 ], [ 142, 170, 1/4 ], [ 142, 182, 1/4 ], [ 143, 121, -1/4 ], [ 143, 129, -1/4 ], 
  [ 143, 144, -1/4 ], [ 143, 150, -1/4 ], [ 143, 157, -1/4 ], [ 143, 164, 1/4 ], [ 144, 121, 1/2 ], [ 144, 129, -1/2 ], [ 144, 136, 1 ], [ 144, 143, -1 ], [ 144, 157, -1/2 ], 
  [ 144, 164, -1/2 ], [ 145, 2, 1/4 ], [ 145, 3, -1/4 ], [ 145, 4, -1/8 ], [ 145, 5, 1 ], [ 145, 10, -1/8 ], [ 145, 11, 1/8 ], [ 145, 12, 1/4 ], [ 145, 17, 1/8 ], 
  [ 145, 18, -1/4 ], [ 145, 19, 1/4 ], [ 145, 25, 3/4 ], [ 145, 32, -1/4 ], [ 145, 122, -1/4 ], [ 145, 123, 1/4 ], [ 145, 124, -1/4 ], [ 145, 130, -1/4 ], [ 145, 131, 1/4 ], 
  [ 145, 132, 1/8 ], [ 145, 137, -3/4 ], [ 145, 138, -1/8 ], [ 145, 139, 1/8 ], [ 145, 145, -1/8 ], [ 145, 152, 1/4 ], [ 145, 241, -1/2 ], [ 145, 242, 1/2 ], [ 145, 243, 1/2 ], 
  [ 145, 244, -1/4 ], [ 145, 246, -1/4 ], [ 146, 126, 1/2 ], [ 146, 133, -1/4 ], [ 146, 146, -1/2 ], [ 146, 147, -1/2 ], [ 146, 160, -1/2 ], [ 146, 168, 1/4 ], [ 147, 126, -1/2 ], 
  [ 147, 133, 1/4 ], [ 147, 146, -1/2 ], [ 147, 147, -1/2 ], [ 147, 160, 1/2 ], [ 147, 168, -1/4 ], [ 148, 134, -1/2 ], [ 148, 141, -1/4 ], [ 148, 148, -1/2 ], [ 148, 161, -1/2 ], 
  [ 148, 169, 1/2 ], [ 148, 175, 1/4 ], [ 149, 142, 1 ], [ 149, 156, -1/2 ], [ 149, 162, 1/2 ], [ 149, 163, -1/2 ], [ 149, 170, 1/2 ], [ 149, 176, 1 ], [ 150, 121, -1/2 ], 
  [ 150, 129, 1/2 ], [ 150, 136, 1 ], [ 150, 143, -1 ], [ 150, 157, 1/2 ], [ 150, 164, 1/2 ], [ 151, 151, -1/2 ], [ 151, 165, 1/2 ], [ 151, 171, -1/2 ], [ 151, 172, -1/4 ], 
  [ 151, 177, -1/4 ], [ 151, 189, 1/2 ], [ 152, 2, -1/4 ], [ 152, 3, 1/4 ], [ 152, 4, -1/8 ], [ 152, 10, -1/8 ], [ 152, 11, 1/8 ], [ 152, 12, 1/4 ], [ 152, 17, 1/8 ], 
  [ 152, 18, -1/4 ], [ 152, 19, 1/4 ], [ 152, 25, -1/4 ], [ 152, 32, -1/4 ], [ 152, 122, -1/4 ], [ 152, 123, 1/4 ], [ 152, 124, 1/4 ], [ 152, 130, 1/4 ], [ 152, 131, -1/4 ], 
  [ 152, 132, -1/8 ], [ 152, 137, -1/4 ], [ 152, 138, 1/8 ], [ 152, 139, -1/8 ], [ 152, 145, 1/8 ], [ 152, 152, -1/4 ], [ 152, 244, 1/4 ], [ 152, 245, -1/2 ], [ 152, 246, 1/4 ], 
  [ 153, 126, -1/2 ], [ 153, 133, -1/4 ], [ 153, 140, 1/2 ], [ 153, 153, 1/2 ], [ 153, 160, 1/2 ], [ 153, 168, 1/4 ], [ 154, 134, -1/2 ], [ 154, 141, 1/4 ], [ 154, 154, 1/2 ], 
  [ 154, 155, 1/2 ], [ 154, 169, 1/2 ], [ 154, 175, -1/4 ], [ 155, 134, 1/2 ], [ 155, 141, -1/4 ], [ 155, 154, 1/2 ], [ 155, 155, 1/2 ], [ 155, 169, -1/2 ], [ 155, 175, 1/4 ], 
  [ 156, 142, -1/2 ], [ 156, 149, -1/4 ], [ 156, 156, -1/2 ], [ 156, 170, -1/2 ], [ 156, 176, 1/2 ], [ 156, 182, 1/4 ], [ 157, 129, 1/2 ], [ 157, 136, -1/2 ], [ 157, 143, -1/2 ], 
  [ 157, 144, -1/4 ], [ 157, 150, 1/4 ], [ 157, 157, -1/2 ], [ 158, 158, 1/2 ], [ 158, 165, 1/2 ], [ 158, 171, -1/2 ], [ 158, 172, 1/4 ], [ 158, 177, 1/4 ], [ 158, 183, 1/2 ], 
  [ 159, 159, 1/2 ], [ 159, 173, -1/2 ], [ 159, 178, 1/2 ], [ 159, 179, 1/4 ], [ 159, 184, 1/4 ], [ 159, 195, -1/2 ], [ 160, 133, -1/4 ], [ 160, 140, -1/4 ], [ 160, 146, -1/4 ], 
  [ 160, 147, 1/4 ], [ 160, 153, 1/4 ], [ 160, 168, -1/4 ], [ 161, 134, 1/2 ], [ 161, 141, 1/4 ], [ 161, 148, -1/2 ], [ 161, 161, -1/2 ], [ 161, 169, -1/2 ], [ 161, 175, -1/4 ], 
  [ 162, 142, -1/2 ], [ 162, 149, 1/4 ], [ 162, 162, 1/2 ], [ 162, 163, 1/2 ], [ 162, 176, 1/2 ], [ 162, 182, -1/4 ], [ 163, 142, 1/2 ], [ 163, 149, -1/4 ], [ 163, 162, 1/2 ], 
  [ 163, 163, 1/2 ], [ 163, 176, -1/2 ], [ 163, 182, 1/4 ], [ 164, 121, 1/2 ], [ 164, 136, 1/2 ], [ 164, 143, 1/2 ], [ 164, 144, -1/4 ], [ 164, 150, 1/4 ], [ 164, 164, 1/2 ], 
  [ 165, 151, 1/4 ], [ 165, 158, 1/4 ], [ 165, 172, -1/4 ], [ 165, 177, 1/4 ], [ 165, 183, -1/4 ], [ 165, 189, 1/4 ], [ 166, 166, -1/2 ], [ 166, 173, -1/2 ], [ 166, 178, 1/2 ], 
  [ 166, 179, -1/4 ], [ 166, 184, -1/4 ], [ 166, 190, -1/2 ], [ 167, 167, 1/2 ], [ 167, 180, -1/2 ], [ 167, 185, 1/2 ], [ 167, 187, 1/4 ], [ 167, 192, 1/4 ], [ 167, 201, -1/2 ], 
  [ 168, 126, -1 ], [ 168, 140, -1/2 ], [ 168, 146, 1/2 ], [ 168, 147, -1/2 ], [ 168, 153, 1/2 ], [ 168, 160, -1 ], [ 169, 141, 1/4 ], [ 169, 148, 1/4 ], [ 169, 154, 1/4 ], 
  [ 169, 155, -1/4 ], [ 169, 161, -1/4 ], [ 169, 175, 1/4 ], [ 170, 142, 1/2 ], [ 170, 149, 1/4 ], [ 170, 156, -1/2 ], [ 170, 170, -1/2 ], [ 170, 176, -1/2 ], [ 170, 182, -1/4 ], 
  [ 171, 151, -1/4 ], [ 171, 158, -1/4 ], [ 171, 172, -1/4 ], [ 171, 177, 1/4 ], [ 171, 183, 1/4 ], [ 171, 189, -1/4 ], [ 172, 151, -1/2 ], [ 172, 158, 1/2 ], [ 172, 165, -1 ], 
  [ 172, 171, -1 ], [ 172, 183, -1/2 ], [ 172, 189, -1/2 ], [ 173, 159, -1/4 ], [ 173, 166, -1/4 ], [ 173, 179, 1/4 ], [ 173, 184, -1/4 ], [ 173, 190, 1/4 ], [ 173, 195, -1/4 ], 
  [ 174, 174, -1/2 ], [ 174, 180, -1/2 ], [ 174, 185, 1/2 ], [ 174, 187, -1/4 ], [ 174, 192, -1/4 ], [ 174, 197, -1/2 ], [ 175, 134, 1 ], [ 175, 148, 1/2 ], [ 175, 154, -1/2 ], 
  [ 175, 155, 1/2 ], [ 175, 161, -1/2 ], [ 175, 169, 1 ], [ 176, 149, 1/4 ], [ 176, 156, 1/4 ], [ 176, 162, 1/4 ], [ 176, 163, -1/4 ], [ 176, 170, -1/4 ], [ 176, 182, 1/4 ], 
  [ 177, 151, -1/2 ], [ 177, 158, 1/2 ], [ 177, 165, 1 ], [ 177, 171, 1 ], [ 177, 183, -1/2 ], [ 177, 189, -1/2 ], [ 178, 159, 1/4 ], [ 178, 166, 1/4 ], [ 178, 179, 1/4 ], 
  [ 178, 184, -1/4 ], [ 178, 190, -1/4 ], [ 178, 195, 1/4 ], [ 179, 159, 1/2 ], [ 179, 166, -1/2 ], [ 179, 173, 1 ], [ 179, 178, 1 ], [ 179, 190, 1/2 ], [ 179, 195, 1/2 ], 
  [ 180, 167, -1/4 ], [ 180, 174, -1/4 ], [ 180, 187, 1/4 ], [ 180, 192, -1/4 ], [ 180, 197, 1/4 ], [ 180, 201, -1/4 ], [ 181, 181, -1 ], [ 182, 142, 1 ], [ 182, 156, 1/2 ], 
  [ 182, 162, -1/2 ], [ 182, 163, 1/2 ], [ 182, 170, -1/2 ], [ 182, 176, 1 ], [ 183, 158, 1/2 ], [ 183, 165, -1/2 ], [ 183, 171, 1/2 ], [ 183, 172, -1/4 ], [ 183, 177, -1/4 ], 
  [ 183, 183, 1/2 ], [ 184, 159, 1/2 ], [ 184, 166, -1/2 ], [ 184, 173, -1 ], [ 184, 178, -1 ], [ 184, 190, 1/2 ], [ 184, 195, 1/2 ], [ 185, 167, 1/4 ], [ 185, 174, 1/4 ], 
  [ 185, 187, 1/4 ], [ 185, 192, -1/4 ], [ 185, 197, -1/4 ], [ 185, 201, 1/4 ], [ 186, 186, -1/2 ], [ 186, 196, -1/2 ], [ 186, 200, 1/2 ], [ 186, 202, 1/4 ], [ 186, 205, 1/4 ], 
  [ 186, 213, 1/2 ], [ 187, 167, 1/2 ], [ 187, 174, -1/2 ], [ 187, 180, 1 ], [ 187, 185, 1 ], [ 187, 197, 1/2 ], [ 187, 201, 1/2 ], [ 188, 188, -1 ], [ 189, 151, 1/2 ], 
  [ 189, 165, 1/2 ], [ 189, 171, -1/2 ], [ 189, 172, -1/4 ], [ 189, 177, -1/4 ], [ 189, 189, -1/2 ], [ 190, 166, -1/2 ], [ 190, 173, 1/2 ], [ 190, 178, -1/2 ], [ 190, 179, 1/4 ], 
  [ 190, 184, 1/4 ], [ 190, 190, -1/2 ], [ 191, 191, 1/2 ], [ 191, 196, 1/2 ], [ 191, 200, -1/2 ], [ 191, 202, 1/4 ], [ 191, 205, 1/4 ], [ 191, 209, 1/2 ], [ 192, 167, 1/2 ], 
  [ 192, 174, -1/2 ], [ 192, 180, -1 ], [ 192, 185, -1 ], [ 192, 197, 1/2 ], [ 192, 201, 1/2 ], [ 193, 193, -1/2 ], [ 193, 203, -1/2 ], [ 193, 206, 1/2 ], [ 193, 207, 1/4 ], 
  [ 193, 210, 1/4 ], [ 193, 218, 1/2 ], [ 194, 194, 1 ], [ 195, 159, -1/2 ], [ 195, 173, -1/2 ], [ 195, 178, 1/2 ], [ 195, 179, 1/4 ], [ 195, 184, 1/4 ], [ 195, 195, 1/2 ], 
  [ 196, 186, -1/4 ], [ 196, 191, 1/4 ], [ 196, 202, -1/4 ], [ 196, 205, 1/4 ], [ 196, 209, -1/4 ], [ 196, 213, -1/4 ], [ 197, 174, -1/2 ], [ 197, 180, 1/2 ], [ 197, 185, -1/2 ], 
  [ 197, 187, 1/4 ], [ 197, 192, 1/4 ], [ 197, 197, -1/2 ], [ 198, 198, 1/2 ], [ 198, 203, 1/2 ], [ 198, 206, -1/2 ], [ 198, 207, 1/4 ], [ 198, 210, 1/4 ], [ 198, 214, 1/2 ], 
  [ 199, 199, 1/2 ], [ 199, 208, 1/2 ], [ 199, 211, -1/2 ], [ 199, 212, -1/4 ], [ 199, 215, -1/4 ], [ 199, 222, -1/2 ], [ 200, 186, 1/4 ], [ 200, 191, -1/4 ], [ 200, 202, -1/4 ], 
  [ 200, 205, 1/4 ], [ 200, 209, 1/4 ], [ 200, 213, 1/4 ], [ 201, 167, -1/2 ], [ 201, 180, -1/2 ], [ 201, 185, 1/2 ], [ 201, 187, 1/4 ], [ 201, 192, 1/4 ], [ 201, 201, 1/2 ], 
  [ 202, 186, 1/2 ], [ 202, 191, 1/2 ], [ 202, 196, -1 ], [ 202, 200, -1 ], [ 202, 209, -1/2 ], [ 202, 213, 1/2 ], [ 203, 193, -1/4 ], [ 203, 198, 1/4 ], [ 203, 207, -1/4 ], 
  [ 203, 210, 1/4 ], [ 203, 214, -1/4 ], [ 203, 218, -1/4 ], [ 204, 204, -1/2 ], [ 204, 208, -1/2 ], [ 204, 211, 1/2 ], [ 204, 212, -1/4 ], [ 204, 215, -1/4 ], [ 204, 219, -1/2 ], 
  [ 205, 186, 1/2 ], [ 205, 191, 1/2 ], [ 205, 196, 1 ], [ 205, 200, 1 ], [ 205, 209, -1/2 ], [ 205, 213, 1/2 ], [ 206, 193, 1/4 ], [ 206, 198, -1/4 ], [ 206, 207, -1/4 ], 
  [ 206, 210, 1/4 ], [ 206, 214, 1/4 ], [ 206, 218, 1/4 ], [ 207, 193, 1/2 ], [ 207, 198, 1/2 ], [ 207, 203, -1 ], [ 207, 206, -1 ], [ 207, 214, -1/2 ], [ 207, 218, 1/2 ], 
  [ 208, 199, 1/4 ], [ 208, 204, -1/4 ], [ 208, 212, 1/4 ], [ 208, 215, -1/4 ], [ 208, 219, 1/4 ], [ 208, 222, 1/4 ], [ 209, 191, 1/2 ], [ 209, 196, -1/2 ], [ 209, 200, 1/2 ], 
  [ 209, 202, -1/4 ], [ 209, 205, -1/4 ], [ 209, 209, 1/2 ], [ 210, 193, 1/2 ], [ 210, 198, 1/2 ], [ 210, 203, 1 ], [ 210, 206, 1 ], [ 210, 214, -1/2 ], [ 210, 218, 1/2 ], 
  [ 211, 199, -1/4 ], [ 211, 204, 1/4 ], [ 211, 212, 1/4 ], [ 211, 215, -1/4 ], [ 211, 219, -1/4 ], [ 211, 222, -1/4 ], [ 212, 199, -1/2 ], [ 212, 204, -1/2 ], [ 212, 208, 1 ], 
  [ 212, 211, 1 ], [ 212, 219, 1/2 ], [ 212, 222, -1/2 ], [ 213, 186, 1/2 ], [ 213, 196, -1/2 ], [ 213, 200, 1/2 ], [ 213, 202, 1/4 ], [ 213, 205, 1/4 ], [ 213, 213, -1/2 ], 
  [ 214, 198, 1/2 ], [ 214, 203, -1/2 ], [ 214, 206, 1/2 ], [ 214, 207, -1/4 ], [ 214, 210, -1/4 ], [ 214, 214, 1/2 ], [ 215, 199, -1/2 ], [ 215, 204, -1/2 ], [ 215, 208, -1 ], 
  [ 215, 211, -1 ], [ 215, 219, 1/2 ], [ 215, 222, -1/2 ], [ 216, 216, -1/2 ], [ 216, 223, -1/2 ], [ 216, 225, -1/2 ], [ 216, 226, 1/4 ], [ 216, 228, -1/4 ], [ 216, 232, -1/2 ], 
  [ 217, 217, -1 ], [ 218, 193, 1/2 ], [ 218, 203, -1/2 ], [ 218, 206, 1/2 ], [ 218, 207, 1/4 ], [ 218, 210, 1/4 ], [ 218, 218, -1/2 ], [ 219, 204, -1/2 ], [ 219, 208, 1/2 ], 
  [ 219, 211, -1/2 ], [ 219, 212, 1/4 ], [ 219, 215, 1/4 ], [ 219, 219, -1/2 ], [ 220, 220, 1/2 ], [ 220, 223, 1/2 ], [ 220, 225, 1/2 ], [ 220, 226, 1/4 ], [ 220, 228, -1/4 ], 
  [ 220, 230, -1/2 ], [ 221, 221, -1 ], [ 222, 199, -1/2 ], [ 222, 208, 1/2 ], [ 222, 211, -1/2 ], [ 222, 212, -1/4 ], [ 222, 215, -1/4 ], [ 222, 222, 1/2 ], [ 223, 216, -1/4 ], 
  [ 223, 220, 1/4 ], [ 223, 226, -1/4 ], [ 223, 228, -1/4 ], [ 223, 230, 1/4 ], [ 223, 232, 1/4 ], [ 224, 224, 1 ], [ 225, 216, -1/4 ], [ 225, 220, 1/4 ], [ 225, 226, 1/4 ], 
  [ 225, 228, 1/4 ], [ 225, 230, 1/4 ], [ 225, 232, 1/4 ], [ 226, 216, 1/2 ], [ 226, 220, 1/2 ], [ 226, 223, -1 ], [ 226, 225, 1 ], [ 226, 230, 1/2 ], [ 226, 232, -1/2 ], 
  [ 227, 229, 1/4 ], [ 227, 231, 1/4 ], [ 227, 233, 1/4 ], [ 227, 234, 1/4 ], [ 227, 235, 1/4 ], [ 227, 237, 1/4 ], [ 228, 216, -1/2 ], [ 228, 220, -1/2 ], 
  [ 228, 223, -1 ], [ 228, 225, 1 ], [ 228, 230, -1/2 ], [ 228, 232, 1/2 ], [ 229, 227, 1 ], [ 229, 231, -1/2 ], [ 229, 233, 1/2 ], [ 229, 234, 1/2 ], [ 229, 235, -1/2 ], 
  [ 229, 236, 1 ], [ 230, 220, -1/2 ], [ 230, 223, 1/2 ], [ 230, 225, 1/2 ], [ 230, 226, 1/4 ], [ 230, 228, -1/4 ], [ 230, 230, 1/2 ], [ 231, 227, 1/2 ], [ 231, 229, -1/4 ], 
  [ 231, 231, 1/2 ], [ 231, 235, -1/2 ], [ 231, 236, -1/2 ], [ 231, 237, 1/4 ], [ 232, 216, -1/2 ], [ 232, 223, 1/2 ], [ 232, 225, 1/2 ], [ 232, 226, -1/4 ], [ 232, 228, 1/4 ], 
  [ 232, 232, -1/2 ], [ 233, 227, 1/2 ], [ 233, 229, 1/4 ], [ 233, 233, -1/2 ], [ 233, 234, 1/2 ], [ 233, 236, -1/2 ], [ 233, 237, -1/4 ], [ 234, 227, 1/2 ], [ 234, 229, 1/4 ], 
  [ 234, 233, 1/2 ], [ 234, 234, -1/2 ], [ 234, 236, -1/2 ], [ 234, 237, -1/4 ], [ 235, 227, 1/2 ], [ 235, 229, -1/4 ], [ 235, 231, -1/2 ], [ 235, 235, 1/2 ], [ 235, 236, -1/2 ], 
  [ 235, 237, 1/4 ], [ 236, 229, 1/4 ], [ 236, 231, -1/4 ], [ 236, 233, -1/4 ], [ 236, 234, -1/4 ], [ 236, 235, -1/4 ], [ 236, 237, 1/4 ], [ 237, 227, 1 ], [ 237, 231, 1/2 ], 
  [ 237, 233, -1/2 ], [ 237, 234, -1/2 ], [ 237, 235, 1/2 ], [ 237, 236, 1 ], [ 238, 238, 1 ], [ 239, 239, -1 ], [ 240, 240, -1 ], [ 241, 241, 1 ], [ 242, 10, -1/4 ], [ 242, 17, 1/4 ], 
  [ 242, 18, 1/2 ], [ 242, 25, 1/2 ], [ 242, 130, -1/2 ], [ 242, 137, 1/2 ], [ 242, 138, 1/4 ], [ 242, 145, 1/4 ], [ 242, 241, 1/2 ], [ 242, 246, 1/2 ], [ 243, 11, 1/4 ], 
  [ 243, 17, 1/4 ], [ 243, 19, -1/2 ], [ 243, 25, 1/2 ], [ 243, 131, 1/2 ], [ 243, 137, 1/2 ], [ 243, 139, -1/4 ], [ 243, 145, 1/4 ], [ 243, 241, 1 ], [ 243, 246, 1/2 ], 
  [ 244, 4, -1/4 ], [ 244, 10, -1/4 ], [ 244, 11, 1/4 ], [ 244, 12, -1/2 ], [ 244, 17, 1/4 ], [ 244, 18, 1/2 ], [ 244, 19, -1/2 ], [ 244, 25, 1/2 ], 
  [ 244, 124, -1/2 ], [ 244, 130, -1/2 ], [ 244, 131, 1/2 ], [ 244, 132, -1/4 ], [ 244, 137, 1/2 ], [ 244, 138, 1/4 ], [ 244, 139, -1/4 ], [ 244, 145, 1/4 ], [ 244, 241, 1 ], 
  [ 244, 246, 1 ], [ 245, 2, 1/4 ], [ 245, 3, -1/4 ], [ 245, 4, -1/8 ], [ 245, 10, -1/8 ], [ 245, 11, 1/8 ], [ 245, 12, -1/4 ], [ 245, 17, 1/8 ], [ 245, 18, 1/4 ], 
  [ 245, 19, -1/4 ], [ 245, 25, 1/4 ], [ 245, 32, -1/4 ], [ 245, 122, 1/4 ], [ 245, 123, -1/4 ], [ 245, 124, -1/4 ], [ 245, 130, -1/4 ], [ 245, 131, 1/4 ], [ 245, 132, -1/8 ], 
  [ 245, 137, 1/4 ], [ 245, 138, 1/8 ], [ 245, 139, -1/8 ], [ 245, 145, 1/8 ], [ 245, 152, -1/4 ], [ 245, 241, 1/2 ], [ 245, 244, 1/4 ], [ 245, 245, -1/2 ], [ 245, 246, 5/4 ], 
  [ 246, 246, 1 ], [ 247, 247, 1 ], [ 248, 248, 1 ] ] ],

  [ [ 0, 0, 0, 0, 2, 0, 0, 2 ],
    [ 248, [ 1, 122, -1 ], [ 2, 121, -1 ], [ 3, 124, 1 ], [ 4, 123, 1 ], [ 5, 51, -1 ], [ 6, 126, 1 ], [ 7, 14, -1 ], [ 8, 8, 1 ], [ 9, 130, 1 ], 
  [ 10, 129, 1 ], [ 11, 131, 1 ], [ 12, 45, -1 ], [ 13, 44, -1 ], [ 14, 7, -1 ], [ 15, 22, -1 ], [ 16, 137, 1 ], [ 17, 136, 1 ], [ 18, 40, 1 ], [ 19, 38, 1 ], [ 20, 37, -1 ], 
  [ 21, 58, 1 ], [ 22, 15, -1 ], [ 23, 143, 1 ], [ 24, 31, -1 ], [ 25, 33, -1 ], [ 26, 32, 1 ], [ 27, 30, 1 ], [ 28, 53, 1 ], [ 29, 65, 1 ], [ 30, 27, 1 ], 
  [ 31, 24, -1 ], [ 32, 26, 1 ], [ 33, 25, -1 ], [ 34, 49, -1 ], [ 35, 46, -1 ], [ 36, 60, 1 ], [ 37, 20, -1 ], [ 38, 19, 1 ], [ 39, 39, 1 ], [ 40, 18, 1 ], [ 41, 41, 1 ], 
  [ 42, 56, -1 ], [ 43, 54, -1 ], [ 44, 13, -1 ], [ 45, 12, -1 ], [ 46, 35, -1 ], [ 47, 47, 1 ], [ 48, 69, 1 ], [ 49, 34, -1 ], [ 50, 50, 1 ], [ 51, 5, -1 ], 
  [ 52, 63, 1 ], [ 53, 28, 1 ], [ 54, 43, -1 ], [ 55, 80, 1 ], [ 56, 42, -1 ], [ 57, 57, -1 ], [ 58, 21, 1 ], [ 59, 76, 1 ], [ 60, 36, 1 ], [ 61, 75, 1 ], [ 62, 86, 1 ], 
  [ 63, 52, 1 ], [ 64, 71, -1 ], [ 65, 29, 1 ], [ 66, 70, 1 ], [ 67, 83, 1 ], [ 68, 81, 1 ], [ 69, 48, 1 ], [ 70, 66, 1 ], [ 71, 64, -1 ], [ 72, 78, -1 ], [ 73, 77, 1 ], 
  [ 74, 91, -1 ], [ 75, 61, 1 ], [ 76, 59, 1 ], [ 77, 73, 1 ], [ 78, 72, -1 ], [ 79, 88, -1 ], [ 80, 55, 1 ], [ 81, 68, 1 ], [ 82, 97, -1 ], [ 83, 67, 1 ], [ 84, 84, 1 ], 
  [ 85, 93, 1 ], [ 86, 62, 1 ], [ 87, 101, -1 ], [ 88, 79, -1 ], [ 89, 89, 1 ], [ 90, 98, 1 ], [ 91, 74, -1 ], [ 92, 107, -1 ], [ 93, 85, 1 ], [ 94, 94, 1 ], [ 95, 105, 1 ], 
  [ 96, 104, -1 ], [ 97, 82, -1 ], [ 98, 90, 1 ], [ 99, 103, 1 ], [ 100, 102, 1 ], [ 101, 87, -1 ], [ 102, 100, 1 ], [ 103, 99, 1 ], [ 104, 96, -1 ], [ 105, 95, 1 ], 
  [ 106, 116, -1 ], [ 107, 92, -1 ], [ 108, 115, 1 ], [ 109, 114, 1 ], [ 110, 113, 1 ], [ 111, 111, 1 ], [ 112, 112, 1 ], [ 113, 110, 1 ], [ 114, 109, 1 ], [ 115, 108, 1 ], 
  [ 116, 106, -1 ], [ 117, 118, 1 ], [ 118, 117, 1 ], [ 119, 119, 1 ], [ 120, 120, 1 ], [ 121, 2, -1 ], [ 122, 1, -1 ], [ 123, 4, 1 ], [ 124, 3, 1 ], 
  [ 125, 171, -1 ], [ 126, 6, 1 ], [ 127, 134, -1 ], [ 128, 128, 1 ], [ 129, 10, 1 ], [ 130, 9, 1 ], [ 131, 11, 1 ], [ 132, 165, -1 ], [ 133, 164, -1 ], [ 134, 127, -1 ], 
  [ 135, 142, -1 ], [ 136, 17, 1 ], [ 137, 16, 1 ], [ 138, 160, 1 ], [ 139, 158, 1 ], [ 140, 157, -1 ], [ 141, 178, 1 ], [ 142, 135, -1 ], [ 143, 23, 1 ], [ 144, 151, -1 ], 
  [ 145, 153, -1 ], [ 146, 152, 1 ], [ 147, 150, 1 ], [ 148, 173, 1 ], [ 149, 185, 1 ], [ 150, 147, 1 ], [ 151, 144, -1 ], [ 152, 146, 1 ], [ 153, 145, -1 ], [ 154, 169, -1 ], 
  [ 155, 166, -1 ], [ 156, 180, 1 ], [ 157, 140, -1 ], [ 158, 139, 1 ], [ 159, 159, 1 ], [ 160, 138, 1 ], [ 161, 161, 1 ], [ 162, 176, -1 ], [ 163, 174, -1 ], [ 164, 133, -1 ], 
  [ 165, 132, -1 ], [ 166, 155, -1 ], [ 167, 167, 1 ], [ 168, 189, 1 ], [ 169, 154, -1 ], [ 170, 170, 1 ], [ 171, 125, -1 ], [ 172, 183, 1 ], [ 173, 148, 1 ], [ 174, 163, -1 ], 
  [ 175, 200, 1 ], [ 176, 162, -1 ], [ 177, 177, -1 ], [ 178, 141, 1 ], [ 179, 196, 1 ], [ 180, 156, 1 ], [ 181, 195, 1 ], [ 182, 206, 1 ], [ 183, 172, 1 ], [ 184, 191, -1 ], 
  [ 185, 149, 1 ], [ 186, 190, 1 ], [ 187, 203, 1 ], [ 188, 201, 1 ], [ 189, 168, 1 ], [ 190, 186, 1 ], [ 191, 184, -1 ], [ 192, 198, -1 ], [ 193, 197, 1 ], [ 194, 211, -1 ], 
  [ 195, 181, 1 ], [ 196, 179, 1 ], [ 197, 193, 1 ], [ 198, 192, -1 ], [ 199, 208, -1 ], [ 200, 175, 1 ], [ 201, 188, 1 ], [ 202, 217, -1 ], [ 203, 187, 1 ], [ 204, 204, 1 ], 
  [ 205, 213, 1 ], [ 206, 182, 1 ], [ 207, 221, -1 ], [ 208, 199, -1 ], [ 209, 209, 1 ], [ 210, 218, 1 ], [ 211, 194, -1 ], [ 212, 227, -1 ], [ 213, 205, 1 ], [ 214, 214, 1 ], 
  [ 215, 225, 1 ], [ 216, 224, -1 ], [ 217, 202, -1 ], [ 218, 210, 1 ], [ 219, 223, 1 ], [ 220, 222, 1 ], [ 221, 207, -1 ], [ 222, 220, 1 ], [ 223, 219, 1 ], [ 224, 216, -1 ], 
  [ 225, 215, 1 ], [ 226, 236, -1 ], [ 227, 212, -1 ], [ 228, 235, 1 ], [ 229, 234, 1 ], [ 230, 233, 1 ], [ 231, 231, 1 ], [ 232, 232, 1 ], [ 233, 230, 1 ], [ 234, 229, 1 ], 
  [ 235, 228, 1 ], [ 236, 226, -1 ], [ 237, 238, 1 ], [ 238, 237, 1 ], [ 239, 239, 1 ], [ 240, 240, 1 ], [ 241, 242, -1 ], [ 241, 245, 1 ], [ 242, 241, -1 ], [ 242, 245, 1 ], 
  [ 243, 244, -1 ], [ 243, 245, 2 ], [ 244, 243, -1 ], [ 244, 245, 2 ], [ 245, 245, 1 ], [ 246, 245, 1 ], [ 246, 246, -1 ], [ 246, 247, 1 ], [ 247, 247, 1 ], [ 248, 248, 1 ] ] ],

  [ [ 1, 0, 0, 1, 0, 1, 0, 1 ],
   [ 248, [ 1, 9, -1 ], [ 2, 2, -1 ], [ 3, 123, -1 ], [ 4, 11, -1 ], [ 5, 5, -1 ], [ 6, 14, -1 ], [ 7, 127, -1 ], [ 8, 15, -1 ], [ 9, 1, 1 ], 
  [ 10, 17, 1 ], [ 11, 4, 1 ], [ 12, 19, 1 ], [ 13, 21, 1 ], [ 14, 6, 1 ], [ 15, 8, 1 ], [ 16, 16, -1 ], [ 17, 10, -1 ], [ 18, 25, -1 ], [ 19, 12, -1 ], [ 20, 35, -1 ], [ 21, 13, -1 ], 
  [ 22, 22, -1 ], [ 23, 23, 1 ], [ 24, 24, 1 ], [ 25, 18, 1 ], [ 26, 41, 1 ], [ 27, 28, 1 ], [ 28, 27, 1 ], [ 29, 29, 1 ], [ 30, 30, -1 ], [ 31, 39, -1 ], 
  [ 32, 32, 1 ], [ 33, 34, -1 ], [ 34, 33, -1 ], [ 35, 20, -1 ], [ 36, 43, -1 ], [ 37, 44, 1 ], [ 38, 46, 1 ], [ 39, 31, 1 ], [ 40, 49, -1 ], [ 41, 26, 1 ], [ 42, 50, 1 ], 
  [ 43, 36, 1 ], [ 44, 37, -1 ], [ 45, 58, -1 ], [ 46, 38, -1 ], [ 47, 47, -1 ], [ 48, 55, 1 ], [ 49, 40, 1 ], [ 50, 42, -1 ], [ 51, 53, 1 ], 
  [ 52, 64, 1 ], [ 53, 51, 1 ], [ 54, 54, 1 ], [ 55, 48, -1 ], [ 56, 56, -1 ], [ 57, 59, -1 ], [ 58, 45, -1 ], [ 59, 57, -1 ], [ 60, 65, -1 ], [ 61, 61, -1 ], [ 62, 62, 1 ], 
  [ 63, 70, -1 ], [ 64, 52, 1 ], [ 65, 60, 1 ], [ 66, 71, -1 ], [ 67, 72, 1 ], [ 68, 74, -1 ], [ 69, 75, 1 ], [ 70, 63, 1 ], [ 71, 66, 1 ], [ 72, 67, -1 ], 
  [ 73, 84, -1 ], [ 74, 68, 1 ], [ 75, 69, -1 ], [ 76, 76, 1 ], [ 77, 77, -1 ], [ 78, 79, 1 ], [ 79, 78, 1 ], [ 80, 80, -1 ], [ 81, 81, 1 ], [ 82, 82, -1 ], [ 83, 88, 1 ], 
  [ 84, 73, -1 ], [ 85, 85, 1 ], [ 86, 91, -1 ], [ 87, 92, -1 ], [ 88, 83, -1 ], [ 89, 93, 1 ], [ 90, 95, 1 ], [ 91, 86, 1 ], [ 92, 87, 1 ], [ 93, 89, -1 ], 
  [ 94, 102, 1 ], [ 95, 90, -1 ], [ 96, 96, 1 ], [ 97, 97, -1 ], [ 98, 99, -1 ], [ 99, 98, -1 ], [ 100, 100, -1 ], [ 101, 104, -1 ], [ 102, 94, 1 ], [ 103, 105, -1 ], [ 104, 101, 1 ], 
  [ 105, 103, 1 ], [ 106, 108, 1 ], [ 107, 107, 1 ], [ 108, 106, -1 ], [ 109, 109, -1 ], [ 110, 110, 1 ], [ 111, 114, -1 ], [ 112, 112, -1 ], [ 113, 115, 1 ], [ 114, 111, 1 ], 
  [ 115, 113, -1 ], [ 116, 116, 1 ], [ 117, 117, -1 ], [ 118, 119, -1 ], [ 119, 118, 1 ], [ 120, 120, 1 ], [ 121, 129, -1 ], [ 122, 122, -1 ], [ 123, 3, -1 ], [ 124, 131, -1 ], 
  [ 125, 125, -1 ], [ 126, 134, -1 ], [ 127, 7, -1 ], [ 128, 135, -1 ], [ 129, 121, 1 ], [ 130, 137, 1 ], [ 131, 124, 1 ], [ 132, 139, 1 ], [ 133, 141, 1 ], [ 134, 126, 1 ], 
  [ 135, 128, 1 ], [ 136, 136, -1 ], [ 137, 130, -1 ], [ 138, 145, -1 ], [ 139, 132, -1 ], [ 140, 155, -1 ], [ 141, 133, -1 ], [ 142, 142, -1 ], [ 143, 143, 1 ], [ 144, 144, 1 ], 
  [ 145, 138, 1 ], [ 146, 161, 1 ], [ 147, 148, 1 ], [ 148, 147, 1 ], [ 149, 149, 1 ], [ 150, 150, -1 ], [ 151, 159, -1 ], [ 152, 152, 1 ], [ 153, 154, -1 ], [ 154, 153, -1 ], 
  [ 155, 140, -1 ], [ 156, 163, -1 ], [ 157, 164, 1 ], [ 158, 166, 1 ], [ 159, 151, 1 ], [ 160, 169, -1 ], [ 161, 146, 1 ], [ 162, 170, 1 ], [ 163, 156, 1 ], [ 164, 157, -1 ], 
  [ 165, 178, -1 ], [ 166, 158, -1 ], [ 167, 167, -1 ], [ 168, 175, 1 ], [ 169, 160, 1 ], [ 170, 162, -1 ], [ 171, 173, 1 ], [ 172, 184, 1 ], [ 173, 171, 1 ], [ 174, 174, 1 ], 
  [ 175, 168, -1 ], [ 176, 176, -1 ], [ 177, 179, -1 ], [ 178, 165, -1 ], [ 179, 177, -1 ], [ 180, 185, -1 ], [ 181, 181, -1 ], [ 182, 182, 1 ], [ 183, 190, -1 ], [ 184, 172, 1 ], 
  [ 185, 180, 1 ], [ 186, 191, -1 ], [ 187, 192, 1 ], [ 188, 194, -1 ], [ 189, 195, 1 ], [ 190, 183, 1 ], [ 191, 186, 1 ], [ 192, 187, -1 ], [ 193, 204, -1 ], [ 194, 188, 1 ], 
  [ 195, 189, -1 ], [ 196, 196, 1 ], [ 197, 197, -1 ], [ 198, 199, 1 ], [ 199, 198, 1 ], [ 200, 200, -1 ], [ 201, 201, 1 ], [ 202, 202, -1 ], [ 203, 208, 1 ], [ 204, 193, -1 ], 
  [ 205, 205, 1 ], [ 206, 211, -1 ], [ 207, 212, -1 ], [ 208, 203, -1 ], [ 209, 213, 1 ], [ 210, 215, 1 ], [ 211, 206, 1 ], [ 212, 207, 1 ], [ 213, 209, -1 ], [ 214, 222, 1 ], 
  [ 215, 210, -1 ], [ 216, 216, 1 ], [ 217, 217, -1 ], [ 218, 219, -1 ], [ 219, 218, -1 ], [ 220, 220, -1 ], [ 221, 224, -1 ], [ 222, 214, 1 ], [ 223, 225, -1 ], [ 224, 221, 1 ], 
  [ 225, 223, 1 ], [ 226, 228, 1 ], [ 227, 227, 1 ], [ 228, 226, -1 ], [ 229, 229, -1 ], [ 230, 230, 1 ], [ 231, 234, -1 ], [ 232, 232, -1 ], [ 233, 235, 1 ], [ 234, 231, 1 ], 
  [ 235, 233, -1 ], [ 236, 236, 1 ], [ 237, 237, -1 ], [ 238, 239, -1 ], [ 239, 238, 1 ], [ 240, 240, 1 ], [ 241, 241, 1 ], [ 242, 242, 1 ], [ 243, 241, 1 ], [ 243, 243, -1 ], 
  [ 243, 244, 1 ], [ 244, 244, 1 ], [ 245, 245, 1 ], [ 246, 246, 1 ], [ 247, 246, 1 ], [ 247, 247, -1 ], [ 247, 248, 1 ], [ 248, 248, 1 ] ] ],

  [ [ 1, 0, 0, 1, 0, 1, 0, 2 ],
    
    [ 248, [ 1, 9, -1 ], [ 2, 122, -1 ], [ 3, 123, -1 ], [ 4, 25, 1 ], [ 5, 125, -1 ], [ 6, 13, 1 ], [ 7, 7, -1 ], [ 8, 8, -1 ], [ 9, 1, 1 ], 
  [ 10, 19, -1 ], [ 11, 18, -1 ], [ 12, 17, -1 ], [ 13, 6, -1 ], [ 14, 21, -1 ], [ 15, 15, 1 ], [ 16, 30, 1 ], [ 17, 12, 1 ], [ 18, 11, 1 ], [ 19, 10, 1 ], [ 20, 33, -1 ], 
  [ 21, 14, 1 ], [ 22, 29, 1 ], [ 23, 24, -1 ], [ 24, 23, -1 ], [ 25, 4, -1 ], [ 26, 27, 1 ], [ 27, 26, 1 ], [ 28, 41, 1 ], [ 29, 22, -1 ], [ 30, 16, 1 ], 
  [ 31, 38, -1 ], [ 32, 32, 1 ], [ 33, 20, -1 ], [ 34, 35, -1 ], [ 35, 34, -1 ], [ 36, 50, -1 ], [ 37, 44, 1 ], [ 38, 31, 1 ], [ 39, 46, 1 ], [ 40, 48, -1 ], [ 41, 28, 1 ], 
  [ 42, 43, 1 ], [ 43, 42, 1 ], [ 44, 37, -1 ], [ 45, 57, -1 ], [ 46, 39, -1 ], [ 47, 54, -1 ], [ 48, 40, 1 ], [ 49, 55, 1 ], [ 50, 36, -1 ], 
  [ 51, 52, 1 ], [ 52, 51, 1 ], [ 53, 64, 1 ], [ 54, 47, 1 ], [ 55, 49, -1 ], [ 56, 62, -1 ], [ 57, 45, -1 ], [ 58, 59, -1 ], [ 59, 58, -1 ], [ 60, 72, -1 ], [ 61, 61, -1 ], 
  [ 62, 56, 1 ], [ 63, 69, 1 ], [ 64, 53, 1 ], [ 65, 67, 1 ], [ 66, 71, -1 ], [ 67, 65, 1 ], [ 68, 68, 1 ], [ 69, 63, -1 ], [ 70, 75, -1 ], [ 71, 66, 1 ], 
  [ 72, 60, -1 ], [ 73, 78, 1 ], [ 74, 74, -1 ], [ 75, 70, 1 ], [ 76, 85, 1 ], [ 77, 81, 1 ], [ 78, 73, -1 ], [ 79, 84, -1 ], [ 80, 82, -1 ], [ 81, 77, -1 ], [ 82, 80, -1 ], 
  [ 83, 90, -1 ], [ 84, 79, 1 ], [ 85, 76, 1 ], [ 86, 87, 1 ], [ 87, 86, 1 ], [ 88, 95, 1 ], [ 89, 93, 1 ], [ 90, 83, -1 ], [ 91, 92, -1 ], 
  [ 92, 91, -1 ], [ 93, 89, -1 ], [ 94, 98, -1 ], [ 95, 88, 1 ], [ 96, 100, -1 ], [ 97, 97, -1 ], [ 98, 94, 1 ], [ 99, 102, 1 ], [ 100, 96, 1 ], [ 101, 101, 1 ], [ 102, 99, -1 ], 
  [ 103, 108, -1 ], [ 104, 104, -1 ], [ 105, 106, 1 ], [ 106, 105, 1 ], [ 107, 109, 1 ], [ 108, 103, -1 ], [ 109, 107, -1 ], [ 110, 112, 1 ], [ 111, 115, -1 ], [ 112, 110, -1 ], 
  [ 113, 114, 1 ], [ 114, 113, 1 ], [ 115, 111, -1 ], [ 116, 117, -1 ], [ 117, 116, 1 ], [ 118, 118, -1 ], [ 119, 119, 1 ], [ 120, 120, -1 ], [ 121, 129, -1 ], [ 122, 2, -1 ], 
  [ 123, 3, -1 ], [ 124, 145, 1 ], [ 125, 5, -1 ], [ 126, 133, 1 ], [ 127, 127, -1 ], [ 128, 128, -1 ], [ 129, 121, 1 ], [ 130, 139, -1 ], [ 131, 138, -1 ], [ 132, 137, -1 ], 
  [ 133, 126, -1 ], [ 134, 141, -1 ], [ 135, 135, 1 ], [ 136, 150, 1 ], [ 137, 132, 1 ], [ 138, 131, 1 ], [ 139, 130, 1 ], [ 140, 153, -1 ], [ 141, 134, 1 ], [ 142, 149, 1 ], 
  [ 143, 144, -1 ], [ 144, 143, -1 ], [ 145, 124, -1 ], [ 146, 147, 1 ], [ 147, 146, 1 ], [ 148, 161, 1 ], [ 149, 142, -1 ], [ 150, 136, 1 ], [ 151, 158, -1 ], [ 152, 152, 1 ], 
  [ 153, 140, -1 ], [ 154, 155, -1 ], [ 155, 154, -1 ], [ 156, 170, -1 ], [ 157, 164, 1 ], [ 158, 151, 1 ], [ 159, 166, 1 ], [ 160, 168, -1 ], [ 161, 148, 1 ], [ 162, 163, 1 ], 
  [ 163, 162, 1 ], [ 164, 157, -1 ], [ 165, 177, -1 ], [ 166, 159, -1 ], [ 167, 174, -1 ], [ 168, 160, 1 ], [ 169, 175, 1 ], [ 170, 156, -1 ], [ 171, 172, 1 ], [ 172, 171, 1 ], 
  [ 173, 184, 1 ], [ 174, 167, 1 ], [ 175, 169, -1 ], [ 176, 182, -1 ], [ 177, 165, -1 ], [ 178, 179, -1 ], [ 179, 178, -1 ], [ 180, 192, -1 ], [ 181, 181, -1 ], [ 182, 176, 1 ], 
  [ 183, 189, 1 ], [ 184, 173, 1 ], [ 185, 187, 1 ], [ 186, 191, -1 ], [ 187, 185, 1 ], [ 188, 188, 1 ], [ 189, 183, -1 ], [ 190, 195, -1 ], [ 191, 186, 1 ], [ 192, 180, -1 ], 
  [ 193, 198, 1 ], [ 194, 194, -1 ], [ 195, 190, 1 ], [ 196, 205, 1 ], [ 197, 201, 1 ], [ 198, 193, -1 ], [ 199, 204, -1 ], [ 200, 202, -1 ], [ 201, 197, -1 ], [ 202, 200, -1 ], 
  [ 203, 210, -1 ], [ 204, 199, 1 ], [ 205, 196, 1 ], [ 206, 207, 1 ], [ 207, 206, 1 ], [ 208, 215, 1 ], [ 209, 213, 1 ], [ 210, 203, -1 ], [ 211, 212, -1 ], [ 212, 211, -1 ], 
  [ 213, 209, -1 ], [ 214, 218, -1 ], [ 215, 208, 1 ], [ 216, 220, -1 ], [ 217, 217, -1 ], [ 218, 214, 1 ], [ 219, 222, 1 ], [ 220, 216, 1 ], [ 221, 221, 1 ], [ 222, 219, -1 ], 
  [ 223, 228, -1 ], [ 224, 224, -1 ], [ 225, 226, 1 ], [ 226, 225, 1 ], [ 227, 229, 1 ], [ 228, 223, -1 ], [ 229, 227, -1 ], [ 230, 232, 1 ], [ 231, 235, -1 ], [ 232, 230, -1 ], 
  [ 233, 234, 1 ], [ 234, 233, 1 ], [ 235, 231, -1 ], [ 236, 237, -1 ], [ 237, 236, 1 ], [ 238, 238, -1 ], [ 239, 239, 1 ], [ 240, 240, -1 ], [ 241, 241, 1 ], [ 242, 242, -1 ], 
  [ 242, 244, 1 ], [ 243, 241, 1 ], [ 243, 243, -1 ], [ 243, 244, 1 ], [ 244, 244, 1 ], [ 245, 244, 1 ], [ 245, 245, -1 ], [ 245, 246, 1 ], [ 246, 246, 1 ], [ 247, 247, 1 ], 
  [ 248, 248, 1 ] ] ],

  [ [ 2, 0, 0, 1, 0, 1, 0, 2 ],
    [ 248, [ 1, 1, -1 ], [ 2, 2, -1 ], [ 3, 3, -1 ], [ 4, 4, 1 ], [ 5, 5, 1 ], [ 6, 6, -1 ], [ 7, 7, -1 ], [ 8, 8, -1 ], [ 9, 9, 1 ], 
  [ 10, 10, -1 ], [ 11, 11, -1 ], [ 12, 12, 1 ], [ 13, 13, -1 ], [ 14, 14, 1 ], [ 15, 15, 1 ], [ 16, 16, 1 ], [ 17, 17, 1 ], [ 18, 18, -1 ], [ 19, 19, -1 ], [ 20, 20, -1 ], 
  [ 21, 21, 1 ], [ 22, 22, -1 ], [ 23, 23, -1 ], [ 24, 24, 1 ], [ 25, 25, 1 ], [ 26, 26, 1 ], [ 27, 27, 1 ], [ 28, 28, 1 ], [ 29, 29, -1 ], [ 30, 30, -1 ], 
  [ 31, 31, -1 ], [ 32, 32, 1 ], [ 33, 33, -1 ], [ 34, 34, -1 ], [ 35, 35, -1 ], [ 36, 36, -1 ], [ 37, 37, -1 ], [ 38, 38, 1 ], [ 39, 39, 1 ], [ 40, 40, -1 ], [ 41, 41, 1 ], 
  [ 42, 42, 1 ], [ 43, 43, 1 ], [ 44, 44, 1 ], [ 45, 45, 1 ], [ 46, 46, -1 ], [ 47, 47, -1 ], [ 48, 48, -1 ], [ 49, 49, 1 ], [ 50, 50, -1 ], [ 51, 51, -1 ], 
  [ 52, 52, 1 ], [ 53, 53, -1 ], [ 54, 54, 1 ], [ 55, 55, 1 ], [ 56, 56, -1 ], [ 57, 57, -1 ], [ 58, 58, 1 ], [ 59, 59, -1 ], [ 60, 60, 1 ], [ 61, 61, -1 ], [ 62, 62, -1 ], 
  [ 63, 63, -1 ], [ 64, 64, 1 ], [ 65, 65, -1 ], [ 66, 66, 1 ], [ 67, 67, 1 ], [ 68, 68, 1 ], [ 69, 69, 1 ], [ 70, 70, 1 ], [ 71, 71, -1 ], [ 72, 72, -1 ], 
  [ 73, 73, -1 ], [ 74, 74, -1 ], [ 75, 75, -1 ], [ 76, 76, -1 ], [ 77, 77, -1 ], [ 78, 78, 1 ], [ 79, 79, 1 ], [ 80, 80, 1 ], [ 81, 81, 1 ], [ 82, 82, -1 ], [ 83, 83, 1 ], 
  [ 84, 84, -1 ], [ 85, 85, 1 ], [ 86, 86, -1 ], [ 87, 87, 1 ], [ 88, 88, -1 ], [ 89, 89, 1 ], [ 90, 90, -1 ], [ 91, 91, 1 ], [ 92, 92, -1 ], [ 93, 93, -1 ], 
  [ 94, 94, -1 ], [ 95, 95, 1 ], [ 96, 96, 1 ], [ 97, 97, 1 ], [ 98, 98, 1 ], [ 99, 99, 1 ], [ 100, 100, -1 ], [ 101, 101, -1 ], [ 102, 102, -1 ], [ 103, 103, -1 ], [ 104, 104, 1 ], 
  [ 105, 105, 1 ], [ 106, 106, -1 ], [ 107, 107, -1 ], [ 108, 108, 1 ], [ 109, 109, -1 ], [ 110, 110, 1 ], [ 111, 111, -1 ], [ 112, 112, -1 ], [ 113, 113, 1 ], [ 114, 114, 1 ], 
  [ 115, 115, -1 ], [ 116, 116, -1 ], [ 117, 117, -1 ], [ 118, 118, 1 ], [ 119, 119, -1 ], [ 120, 120, 1 ], [ 121, 121, -1 ], [ 122, 122, -1 ], [ 123, 123, -1 ], [ 124, 124, 1 ], 
  [ 125, 125, 1 ], [ 126, 126, -1 ], [ 127, 127, -1 ], [ 128, 128, -1 ], [ 129, 129, 1 ], [ 130, 130, -1 ], [ 131, 131, -1 ], [ 132, 132, 1 ], [ 133, 133, -1 ], [ 134, 134, 1 ], 
  [ 135, 135, 1 ], [ 136, 136, 1 ], [ 137, 137, 1 ], [ 138, 138, -1 ], [ 139, 139, -1 ], [ 140, 140, -1 ], [ 141, 141, 1 ], [ 142, 142, -1 ], [ 143, 143, -1 ], [ 144, 144, 1 ], 
  [ 145, 145, 1 ], [ 146, 146, 1 ], [ 147, 147, 1 ], [ 148, 148, 1 ], [ 149, 149, -1 ], [ 150, 150, -1 ], [ 151, 151, -1 ], [ 152, 152, 1 ], [ 153, 153, -1 ], [ 154, 154, -1 ], 
  [ 155, 155, -1 ], [ 156, 156, -1 ], [ 157, 157, -1 ], [ 158, 158, 1 ], [ 159, 159, 1 ], [ 160, 160, -1 ], [ 161, 161, 1 ], [ 162, 162, 1 ], [ 163, 163, 1 ], [ 164, 164, 1 ], 
  [ 165, 165, 1 ], [ 166, 166, -1 ], [ 167, 167, -1 ], [ 168, 168, -1 ], [ 169, 169, 1 ], [ 170, 170, -1 ], [ 171, 171, -1 ], [ 172, 172, 1 ], [ 173, 173, -1 ], [ 174, 174, 1 ], 
  [ 175, 175, 1 ], [ 176, 176, -1 ], [ 177, 177, -1 ], [ 178, 178, 1 ], [ 179, 179, -1 ], [ 180, 180, 1 ], [ 181, 181, -1 ], [ 182, 182, -1 ], [ 183, 183, -1 ], [ 184, 184, 1 ], 
  [ 185, 185, -1 ], [ 186, 186, 1 ], [ 187, 187, 1 ], [ 188, 188, 1 ], [ 189, 189, 1 ], [ 190, 190, 1 ], [ 191, 191, -1 ], [ 192, 192, -1 ], [ 193, 193, -1 ], [ 194, 194, -1 ], 
  [ 195, 195, -1 ], [ 196, 196, -1 ], [ 197, 197, -1 ], [ 198, 198, 1 ], [ 199, 199, 1 ], [ 200, 200, 1 ], [ 201, 201, 1 ], [ 202, 202, -1 ], [ 203, 203, 1 ], [ 204, 204, -1 ], 
  [ 205, 205, 1 ], [ 206, 206, -1 ], [ 207, 207, 1 ], [ 208, 208, -1 ], [ 209, 209, 1 ], [ 210, 210, -1 ], [ 211, 211, 1 ], [ 212, 212, -1 ], [ 213, 213, -1 ], [ 214, 214, -1 ], 
  [ 215, 215, 1 ], [ 216, 216, 1 ], [ 217, 217, 1 ], [ 218, 218, 1 ], [ 219, 219, 1 ], [ 220, 220, -1 ], [ 221, 221, -1 ], [ 222, 222, -1 ], [ 223, 223, -1 ], [ 224, 224, 1 ], 
  [ 225, 225, 1 ], [ 226, 226, -1 ], [ 227, 227, -1 ], [ 228, 228, 1 ], [ 229, 229, -1 ], [ 230, 230, 1 ], [ 231, 231, -1 ], [ 232, 232, -1 ], [ 233, 233, 1 ], [ 234, 234, 1 ], 
  [ 235, 235, -1 ], [ 236, 236, -1 ], [ 237, 237, -1 ], [ 238, 238, 1 ], [ 239, 239, -1 ], [ 240, 240, 1 ], [ 241, 241, 1 ], [ 242, 242, 1 ], [ 243, 243, 1 ], [ 244, 244, 1 ], 
  [ 245, 245, 1 ], [ 246, 246, 1 ], [ 247, 247, 1 ], [ 248, 248, 1 ] ] ],

  [ [ 2, 0, 0, 0, 2, 0, 0, 2 ],
    [ 248, [ 1, 1, -1 ], [ 2, 10, -1 ], [ 3, 11, -1 ], [ 4, 124, 1 ], [ 5, 12, -1 ], [ 6, 14, -1 ], [ 7, 127, 1 ], [ 8, 15, -1 ], [ 9, 16, 1 ], 
  [ 10, 2, -1 ], [ 11, 3, -1 ], [ 12, 5, -1 ], [ 13, 28, 1 ], [ 14, 6, -1 ], [ 15, 8, -1 ], [ 16, 9, 1 ], [ 17, 17, -1 ], [ 18, 18, 1 ], [ 19, 19, 1 ], [ 20, 21, 1 ], [ 21, 20, 1 ], 
  [ 22, 22, 1 ], [ 23, 23, 1 ], [ 24, 24, -1 ], [ 25, 32, -1 ], [ 26, 34, -1 ], [ 27, 35, -1 ], [ 28, 13, 1 ], [ 29, 36, -1 ], [ 30, 37, 1 ], [ 31, 39, 1 ], 
  [ 32, 25, -1 ], [ 33, 49, 1 ], [ 34, 26, -1 ], [ 35, 27, -1 ], [ 36, 29, -1 ], [ 37, 30, 1 ], [ 38, 53, -1 ], [ 39, 31, 1 ], [ 40, 41, 1 ], [ 41, 40, 1 ], [ 42, 42, 1 ], 
  [ 43, 43, 1 ], [ 44, 44, 1 ], [ 45, 46, -1 ], [ 46, 45, -1 ], [ 47, 47, -1 ], [ 48, 55, -1 ], [ 49, 33, 1 ], [ 50, 56, -1 ], [ 51, 58, -1 ], [ 52, 59, 1 ], 
  [ 53, 38, -1 ], [ 54, 60, 1 ], [ 55, 48, -1 ], [ 56, 50, -1 ], [ 57, 70, -1 ], [ 58, 51, -1 ], [ 59, 52, 1 ], [ 60, 54, 1 ], [ 61, 61, -1 ], [ 62, 62, 1 ], [ 63, 64, -1 ], 
  [ 64, 63, -1 ], [ 65, 65, 1 ], [ 66, 66, 1 ], [ 67, 67, -1 ], [ 68, 74, -1 ], [ 69, 75, -1 ], [ 70, 57, -1 ], [ 71, 76, -1 ], [ 72, 77, 1 ], [ 73, 79, 1 ], 
  [ 74, 68, -1 ], [ 75, 69, -1 ], [ 76, 71, -1 ], [ 77, 72, 1 ], [ 78, 88, -1 ], [ 79, 73, 1 ], [ 80, 80, -1 ], [ 81, 81, 1 ], [ 82, 82, 1 ], [ 83, 84, -1 ], [ 84, 83, -1 ], 
  [ 85, 89, -1 ], [ 86, 91, -1 ], [ 87, 92, 1 ], [ 88, 78, -1 ], [ 89, 85, -1 ], [ 90, 99, -1 ], [ 91, 86, -1 ], [ 92, 87, 1 ], [ 93, 93, -1 ], [ 94, 95, -1 ], 
  [ 95, 94, -1 ], [ 96, 96, 1 ], [ 97, 97, 1 ], [ 98, 102, -1 ], [ 99, 90, -1 ], [ 100, 103, -1 ], [ 101, 104, 1 ], [ 102, 98, -1 ], [ 103, 100, -1 ], [ 104, 101, 1 ], [ 105, 105, -1 ], 
  [ 106, 106, 1 ], [ 107, 107, 1 ], [ 108, 110, -1 ], [ 109, 111, 1 ], [ 110, 108, -1 ], [ 111, 109, 1 ], [ 112, 112, -1 ], [ 113, 113, 1 ], [ 114, 114, 1 ], [ 115, 116, 1 ], 
  [ 116, 115, 1 ], [ 117, 117, 1 ], [ 118, 119, 1 ], [ 119, 118, 1 ], [ 120, 120, 1 ], [ 121, 121, -1 ], [ 122, 130, -1 ], [ 123, 131, -1 ], [ 124, 4, 1 ], [ 125, 132, -1 ], 
  [ 126, 134, -1 ], [ 127, 7, 1 ], [ 128, 135, -1 ], [ 129, 136, 1 ], [ 130, 122, -1 ], [ 131, 123, -1 ], [ 132, 125, -1 ], [ 133, 148, 1 ], [ 134, 126, -1 ], [ 135, 128, -1 ], 
  [ 136, 129, 1 ], [ 137, 137, -1 ], [ 138, 138, 1 ], [ 139, 139, 1 ], [ 140, 141, 1 ], [ 141, 140, 1 ], [ 142, 142, 1 ], [ 143, 143, 1 ], [ 144, 144, -1 ], [ 145, 152, -1 ], 
  [ 146, 154, -1 ], [ 147, 155, -1 ], [ 148, 133, 1 ], [ 149, 156, -1 ], [ 150, 157, 1 ], [ 151, 159, 1 ], [ 152, 145, -1 ], [ 153, 169, 1 ], [ 154, 146, -1 ], [ 155, 147, -1 ], 
  [ 156, 149, -1 ], [ 157, 150, 1 ], [ 158, 173, -1 ], [ 159, 151, 1 ], [ 160, 161, 1 ], [ 161, 160, 1 ], [ 162, 162, 1 ], [ 163, 163, 1 ], [ 164, 164, 1 ], [ 165, 166, -1 ], 
  [ 166, 165, -1 ], [ 167, 167, -1 ], [ 168, 175, -1 ], [ 169, 153, 1 ], [ 170, 176, -1 ], [ 171, 178, -1 ], [ 172, 179, 1 ], [ 173, 158, -1 ], [ 174, 180, 1 ], [ 175, 168, -1 ], 
  [ 176, 170, -1 ], [ 177, 190, -1 ], [ 178, 171, -1 ], [ 179, 172, 1 ], [ 180, 174, 1 ], [ 181, 181, -1 ], [ 182, 182, 1 ], [ 183, 184, -1 ], [ 184, 183, -1 ], [ 185, 185, 1 ], 
  [ 186, 186, 1 ], [ 187, 187, -1 ], [ 188, 194, -1 ], [ 189, 195, -1 ], [ 190, 177, -1 ], [ 191, 196, -1 ], [ 192, 197, 1 ], [ 193, 199, 1 ], [ 194, 188, -1 ], [ 195, 189, -1 ], 
  [ 196, 191, -1 ], [ 197, 192, 1 ], [ 198, 208, -1 ], [ 199, 193, 1 ], [ 200, 200, -1 ], [ 201, 201, 1 ], [ 202, 202, 1 ], [ 203, 204, -1 ], [ 204, 203, -1 ], [ 205, 209, -1 ], 
  [ 206, 211, -1 ], [ 207, 212, 1 ], [ 208, 198, -1 ], [ 209, 205, -1 ], [ 210, 219, -1 ], [ 211, 206, -1 ], [ 212, 207, 1 ], [ 213, 213, -1 ], [ 214, 215, -1 ], [ 215, 214, -1 ], 
  [ 216, 216, 1 ], [ 217, 217, 1 ], [ 218, 222, -1 ], [ 219, 210, -1 ], [ 220, 223, -1 ], [ 221, 224, 1 ], [ 222, 218, -1 ], [ 223, 220, -1 ], [ 224, 221, 1 ], [ 225, 225, -1 ], 
  [ 226, 226, 1 ], [ 227, 227, 1 ], [ 228, 230, -1 ], [ 229, 231, 1 ], [ 230, 228, -1 ], [ 231, 229, 1 ], [ 232, 232, -1 ], [ 233, 233, 1 ], [ 234, 234, 1 ], [ 235, 236, 1 ], 
  [ 236, 235, 1 ], [ 237, 237, 1 ], [ 238, 239, 1 ], [ 239, 238, 1 ], [ 240, 240, 1 ], [ 241, 241, 1 ], [ 242, 242, 1 ], [ 243, 243, 1 ], [ 244, 242, 1 ], [ 244, 243, 1 ], 
  [ 244, 244, -1 ], [ 244, 245, 1 ], [ 245, 245, 1 ], [ 246, 246, 1 ], [ 247, 246, 1 ], [ 247, 247, -1 ], [ 247, 248, 1 ], [ 248, 248, 1 ] ] ],

  [ [ 0, 0, 0, 2, 0, 0, 2, 0 ],
    [ 248, [ 1, 123, 1 ], [ 2, 122, 1 ], [ 3, 121, 1 ], [ 4, 23, 1 ], [ 5, 13, 1 ], [ 6, 126, 1 ], [ 7, 22, 1 ], [ 8, 128, 1 ], [ 9, 129, 1 ], 
  [ 10, 16, 1 ], [ 11, 17, 1 ], [ 12, 38, 1 ], [ 13, 5, 1 ], [ 14, 15, 1 ], [ 15, 14, 1 ], [ 16, 10, 1 ], [ 17, 11, 1 ], [ 18, 31, 1 ], [ 19, 33, 1 ], [ 20, 30, 1 ], [ 21, 29, 1 ], 
  [ 22, 7, 1 ], [ 23, 4, 1 ], [ 24, 26, 1 ], [ 25, 27, 1 ], [ 26, 24, 1 ], [ 27, 25, 1 ], [ 28, 54, 1 ], [ 29, 21, 1 ], [ 30, 20, 1 ], [ 31, 18, 1 ], 
  [ 32, 51, 1 ], [ 33, 19, 1 ], [ 34, 47, 1 ], [ 35, 50, 1 ], [ 36, 46, 1 ], [ 37, 45, 1 ], [ 38, 12, 1 ], [ 39, 42, 1 ], [ 40, 44, 1 ], [ 41, 43, 1 ], [ 42, 39, 1 ], [ 43, 41, 1 ], 
  [ 44, 40, 1 ], [ 45, 37, 1 ], [ 46, 36, 1 ], [ 47, 34, 1 ], [ 48, 57, -1 ], [ 49, 65, 1 ], [ 50, 35, 1 ], [ 51, 32, 1 ], [ 52, 52, -1 ], [ 53, 60, 1 ], 
  [ 54, 28, 1 ], [ 55, 78, 1 ], [ 56, 58, 1 ], [ 57, 48, -1 ], [ 58, 56, 1 ], [ 59, 73, 1 ], [ 60, 53, 1 ], [ 61, 72, 1 ], [ 62, 71, 1 ], [ 63, 69, 1 ], [ 64, 68, 1 ], [ 65, 49, 1 ], 
  [ 66, 67, 1 ], [ 67, 66, 1 ], [ 68, 64, 1 ], [ 69, 63, 1 ], [ 70, 86, -1 ], [ 71, 62, 1 ], [ 72, 61, 1 ], [ 73, 59, 1 ], [ 74, 84, -1 ], [ 75, 83, -1 ], 
  [ 76, 81, -1 ], [ 77, 80, -1 ], [ 78, 55, 1 ], [ 79, 79, -1 ], [ 80, 77, -1 ], [ 81, 76, -1 ], [ 82, 90, 1 ], [ 83, 75, -1 ], [ 84, 74, -1 ], [ 85, 87, 1 ], [ 86, 70, -1 ], 
  [ 87, 85, 1 ], [ 88, 91, 1 ], [ 89, 101, 1 ], [ 90, 82, 1 ], [ 91, 88, 1 ], [ 92, 100, 1 ], [ 93, 98, 1 ], [ 94, 97, 1 ], [ 95, 96, 1 ], [ 96, 95, 1 ], 
  [ 97, 94, 1 ], [ 98, 93, 1 ], [ 99, 107, 1 ], [ 100, 92, 1 ], [ 101, 89, 1 ], [ 102, 105, 1 ], [ 103, 104, 1 ], [ 104, 103, 1 ], [ 105, 102, 1 ], [ 106, 109, -1 ], [ 107, 99, 1 ], 
  [ 108, 108, -1 ], [ 109, 106, -1 ], [ 110, 115, 1 ], [ 111, 113, 1 ], [ 112, 114, 1 ], [ 113, 111, 1 ], [ 114, 112, 1 ], [ 115, 110, 1 ], [ 116, 116, -1 ], [ 117, 118, 1 ], 
  [ 118, 117, 1 ], [ 119, 120, 1 ], [ 120, 119, 1 ], [ 121, 3, 1 ], [ 122, 2, 1 ], [ 123, 1, 1 ], [ 124, 143, 1 ], [ 125, 133, 1 ], [ 126, 6, 1 ], [ 127, 142, 1 ], 
  [ 128, 8, 1 ], [ 129, 9, 1 ], [ 130, 136, 1 ], [ 131, 137, 1 ], [ 132, 158, 1 ], [ 133, 125, 1 ], [ 134, 135, 1 ], [ 135, 134, 1 ], [ 136, 130, 1 ], [ 137, 131, 1 ], [ 138, 151, 1 ], 
  [ 139, 153, 1 ], [ 140, 150, 1 ], [ 141, 149, 1 ], [ 142, 127, 1 ], [ 143, 124, 1 ], [ 144, 146, 1 ], [ 145, 147, 1 ], [ 146, 144, 1 ], [ 147, 145, 1 ], [ 148, 174, 1 ], 
  [ 149, 141, 1 ], [ 150, 140, 1 ], [ 151, 138, 1 ], [ 152, 171, 1 ], [ 153, 139, 1 ], [ 154, 167, 1 ], [ 155, 170, 1 ], [ 156, 166, 1 ], [ 157, 165, 1 ], [ 158, 132, 1 ], 
  [ 159, 162, 1 ], [ 160, 164, 1 ], [ 161, 163, 1 ], [ 162, 159, 1 ], [ 163, 161, 1 ], [ 164, 160, 1 ], [ 165, 157, 1 ], [ 166, 156, 1 ], [ 167, 154, 1 ], [ 168, 177, -1 ], 
  [ 169, 185, 1 ], [ 170, 155, 1 ], [ 171, 152, 1 ], [ 172, 172, -1 ], [ 173, 180, 1 ], [ 174, 148, 1 ], [ 175, 198, 1 ], [ 176, 178, 1 ], [ 177, 168, -1 ], [ 178, 176, 1 ], 
  [ 179, 193, 1 ], [ 180, 173, 1 ], [ 181, 192, 1 ], [ 182, 191, 1 ], [ 183, 189, 1 ], [ 184, 188, 1 ], [ 185, 169, 1 ], [ 186, 187, 1 ], [ 187, 186, 1 ], [ 188, 184, 1 ], 
  [ 189, 183, 1 ], [ 190, 206, -1 ], [ 191, 182, 1 ], [ 192, 181, 1 ], [ 193, 179, 1 ], [ 194, 204, -1 ], [ 195, 203, -1 ], [ 196, 201, -1 ], [ 197, 200, -1 ], [ 198, 175, 1 ], 
  [ 199, 199, -1 ], [ 200, 197, -1 ], [ 201, 196, -1 ], [ 202, 210, 1 ], [ 203, 195, -1 ], [ 204, 194, -1 ], [ 205, 207, 1 ], [ 206, 190, -1 ], [ 207, 205, 1 ], [ 208, 211, 1 ], 
  [ 209, 221, 1 ], [ 210, 202, 1 ], [ 211, 208, 1 ], [ 212, 220, 1 ], [ 213, 218, 1 ], [ 214, 217, 1 ], [ 215, 216, 1 ], [ 216, 215, 1 ], [ 217, 214, 1 ], [ 218, 213, 1 ], 
  [ 219, 227, 1 ], [ 220, 212, 1 ], [ 221, 209, 1 ], [ 222, 225, 1 ], [ 223, 224, 1 ], [ 224, 223, 1 ], [ 225, 222, 1 ], [ 226, 229, -1 ], [ 227, 219, 1 ], [ 228, 228, -1 ], 
  [ 229, 226, -1 ], [ 230, 235, 1 ], [ 231, 233, 1 ], [ 232, 234, 1 ], [ 233, 231, 1 ], [ 234, 232, 1 ], [ 235, 230, 1 ], [ 236, 236, -1 ], [ 237, 238, 1 ], [ 238, 237, 1 ], 
  [ 239, 240, 1 ], [ 240, 239, 1 ], [ 241, 243, -1 ], [ 241, 244, 1 ], [ 242, 242, -1 ], [ 242, 244, 1 ], [ 243, 241, -1 ], [ 243, 244, 1 ], [ 244, 244, 1 ], [ 245, 245, 1 ], 
  [ 246, 245, 1 ], [ 246, 246, -1 ], [ 246, 247, 1 ], [ 247, 247, 1 ], [ 248, 247, 1 ], [ 248, 248, -1 ] ],
    [ 248, [ 1, 1, E(3) ], [ 2, 2, E(3) ], [ 3, 3, E(3) ], [ 4, 4, 1 ], [ 5, 5, E(3) ], [ 6, 6, E(3) ], [ 7, 7, E(3)^2 ], [ 8, 8, E(3) ], 
  [ 9, 9, E(3)^2 ], [ 10, 10, E(3) ], [ 11, 11, E(3) ], [ 12, 12, E(3) ], [ 13, 13, E(3)^2 ], [ 14, 14, 1 ], [ 15, 15, 1 ], [ 16, 16, E(3)^2 ], [ 17, 17, E(3)^2 ], [ 18, 18, E(3)^2 ], 
  [ 19, 19, E(3)^2 ], [ 20, 20, E(3)^2 ], [ 21, 21, E(3) ], [ 22, 22, E(3) ], [ 23, 23, 1 ], [ 24, 24, 1 ], [ 25, 25, 1 ], [ 26, 26, 1 ], [ 27, 27, 1 ], 
  [ 28, 28, E(3) ], [ 29, 29, E(3)^2 ], [ 30, 30, E(3) ], [ 31, 31, E(3) ], [ 32, 32, 1 ], [ 33, 33, E(3) ], [ 34, 34, E(3)^2 ], [ 35, 35, E(3)^2 ], [ 36, 36, E(3)^2 ], [ 37, 37, E(3) ],
  [ 38, 38, E(3)^2 ], [ 39, 39, 1 ], [ 40, 40, E(3) ], [ 41, 41, 1 ], [ 42, 42, 1 ], [ 43, 43, 1 ], [ 44, 44, E(3)^2 ], [ 45, 45, E(3)^2 ], [ 46, 46, E(3) ], 
  [ 47, 47, E(3) ], [ 48, 48, E(3)^2 ], [ 49, 49, 1 ], [ 50, 50, E(3) ], [ 51, 51, 1 ], [ 52, 52, 1 ], [ 53, 53, E(3) ], [ 54, 54, E(3)^2 ], [ 55, 55, E(3) ], [ 56, 56, E(3) ], 
  [ 57, 57, E(3) ], [ 58, 58, E(3)^2 ], [ 59, 59, E(3)^2 ], [ 60, 60, E(3)^2 ], [ 61, 61, E(3)^2 ], [ 62, 62, E(3)^2 ], [ 63, 63, E(3) ], [ 64, 64, 1 ], 
  [ 65, 65, 1 ], [ 66, 66, 1 ], [ 67, 67, 1 ], [ 68, 68, 1 ], [ 69, 69, E(3)^2 ], [ 70, 70, 1 ], [ 71, 71, E(3) ], [ 72, 72, E(3) ], [ 73, 73, E(3) ], [ 74, 74, E(3)^2 ], 
  [ 75, 75, E(3) ], [ 76, 76, E(3) ], [ 77, 77, E(3) ], [ 78, 78, E(3)^2 ], [ 79, 79, 1 ], [ 80, 80, E(3)^2 ], [ 81, 81, E(3)^2 ], 
  [ 82, 82, E(3)^2 ], [ 83, 83, E(3)^2 ], [ 84, 84, E(3) ], [ 85, 85, 1 ], [ 86, 86, 1 ], [ 87, 87, 1 ], [ 88, 88, E(3) ], [ 89, 89, 1 ], [ 90, 90, E(3) ], [ 91, 91, E(3)^2 ], 
  [ 92, 92, E(3)^2 ], [ 93, 93, E(3) ], [ 94, 94, E(3) ], [ 95, 95, 1 ], [ 96, 96, 1 ], [ 97, 97, E(3)^2 ], [ 98, 98, E(3)^2 ], [ 99, 99, 1 ], 
  [ 100, 100, E(3) ], [ 101, 101, 1 ], [ 102, 102, E(3) ], [ 103, 103, E(3) ], [ 104, 104, E(3)^2 ], [ 105, 105, E(3)^2 ], [ 106, 106, E(3)^2 ], [ 107, 107, 1 ], [ 108, 108, 1 ], 
  [ 109, 109, E(3) ], [ 110, 110, 1 ], [ 111, 111, E(3) ], [ 112, 112, E(3) ], [ 113, 113, E(3)^2 ], [ 114, 114, E(3)^2 ], [ 115, 115, 1 ], [ 116, 116, 1 ], 
  [ 117, 117, E(3) ], [ 118, 118, E(3)^2 ], [ 119, 119, E(3) ], [ 120, 120, E(3)^2 ], [ 121, 121, E(3)^2 ], [ 122, 122, E(3)^2 ], [ 123, 123, E(3)^2 ], [ 124, 124, 1 ], 
  [ 125, 125, E(3)^2 ], [ 126, 126, E(3)^2 ], [ 127, 127, E(3) ], [ 128, 128, E(3)^2 ], [ 129, 129, E(3) ], [ 130, 130, E(3)^2 ], [ 131, 131, E(3)^2 ], [ 132, 132, E(3)^2 ], 
  [ 133, 133, E(3) ], [ 134, 134, 1 ], [ 135, 135, 1 ], [ 136, 136, E(3) ], [ 137, 137, E(3) ], [ 138, 138, E(3) ], [ 139, 139, E(3) ], [ 140, 140, E(3) ], [ 141, 141, E(3)^2 ], 
  [ 142, 142, E(3)^2 ], [ 143, 143, 1 ], [ 144, 144, 1 ], [ 145, 145, 1 ], [ 146, 146, 1 ], [ 147, 147, 1 ], [ 148, 148, E(3)^2 ], [ 149, 149, E(3) ], [ 150, 150, E(3)^2 ], 
  [ 151, 151, E(3)^2 ], [ 152, 152, 1 ], [ 153, 153, E(3)^2 ], [ 154, 154, E(3) ], [ 155, 155, E(3) ], [ 156, 156, E(3) ], [ 157, 157, E(3)^2 ], [ 158, 158, E(3) ], [ 159, 159, 1 ], 
  [ 160, 160, E(3)^2 ], [ 161, 161, 1 ], [ 162, 162, 1 ], [ 163, 163, 1 ], [ 164, 164, E(3) ], [ 165, 165, E(3) ], [ 166, 166, E(3)^2 ], [ 167, 167, E(3)^2 ], [ 168, 168, E(3) ], 
  [ 169, 169, 1 ], [ 170, 170, E(3)^2 ], [ 171, 171, 1 ], [ 172, 172, 1 ], [ 173, 173, E(3)^2 ], [ 174, 174, E(3) ], [ 175, 175, E(3)^2 ], [ 176, 176, E(3)^2 ], [ 177, 177, E(3)^2 ], 
  [ 178, 178, E(3) ], [ 179, 179, E(3) ], [ 180, 180, E(3) ], [ 181, 181, E(3) ], [ 182, 182, E(3) ], [ 183, 183, E(3)^2 ], [ 184, 184, 1 ], [ 185, 185, 1 ], [ 186, 186, 1 ], 
  [ 187, 187, 1 ], [ 188, 188, 1 ], [ 189, 189, E(3) ], [ 190, 190, 1 ], [ 191, 191, E(3)^2 ], [ 192, 192, E(3)^2 ], [ 193, 193, E(3)^2 ], [ 194, 194, E(3) ], [ 195, 195, E(3)^2 ], 
  [ 196, 196, E(3)^2 ], [ 197, 197, E(3)^2 ], [ 198, 198, E(3) ], [ 199, 199, 1 ], [ 200, 200, E(3) ], [ 201, 201, E(3) ], [ 202, 202, E(3) ], [ 203, 203, E(3) ], 
  [ 204, 204, E(3)^2 ], [ 205, 205, 1 ], [ 206, 206, 1 ], [ 207, 207, 1 ], [ 208, 208, E(3)^2 ], [ 209, 209, 1 ], [ 210, 210, E(3)^2 ], [ 211, 211, E(3) ], [ 212, 212, E(3) ], 
  [ 213, 213, E(3)^2 ], [ 214, 214, E(3)^2 ], [ 215, 215, 1 ], [ 216, 216, 1 ], [ 217, 217, E(3) ], [ 218, 218, E(3) ], [ 219, 219, 1 ], [ 220, 220, E(3)^2 ], 
  [ 221, 221, 1 ], [ 222, 222, E(3)^2 ], [ 223, 223, E(3)^2 ], [ 224, 224, E(3) ], [ 225, 225, E(3) ], [ 226, 226, E(3) ], [ 227, 227, 1 ], [ 228, 228, 1 ], [ 229, 229, E(3)^2 ], 
  [ 230, 230, 1 ], [ 231, 231, E(3)^2 ], [ 232, 232, E(3)^2 ], [ 233, 233, E(3) ], [ 234, 234, E(3) ], [ 235, 235, 1 ], [ 236, 236, 1 ], [ 237, 237, E(3)^2 ], 
  [ 238, 238, E(3) ], [ 239, 239, E(3)^2 ], [ 240, 240, E(3) ], [ 241, 241, 1 ], [ 242, 242, 1 ], [ 243, 243, 1 ], [ 244, 244, 1 ], [ 245, 245, 1 ], [ 246, 246, 1 ], [ 247, 247, 1 ], 
  [ 248, 248, 1 ] ]],


  [ [ 0, 0, 0, 2, 0, 0, 2, 2 ],
    [ 248, [ 1, 1, E(3) ], [ 2, 2, E(3) ], [ 3, 3, E(3) ], [ 4, 4, 1 ], [ 5, 5, E(3) ], [ 6, 6, E(3) ], [ 7, 7, E(3)^2 ], [ 8, 8, 1 ], 
  [ 9, 9, E(3)^2 ], [ 10, 10, E(3) ], [ 11, 11, E(3) ], [ 12, 12, E(3) ], [ 13, 13, E(3)^2 ], [ 14, 14, 1 ], [ 15, 15, E(3)^2 ], [ 16, 16, E(3)^2 ], [ 17, 17, E(3)^2 ], 
  [ 18, 18, E(3)^2 ], [ 19, 19, E(3)^2 ], [ 20, 20, E(3)^2 ], [ 21, 21, E(3) ], [ 22, 22, 1 ], [ 23, 23, 1 ], [ 24, 24, 1 ], [ 25, 25, 1 ], [ 26, 26, 1 ], 
  [ 27, 27, 1 ], [ 28, 28, E(3) ], [ 29, 29, E(3) ], [ 30, 30, E(3) ], [ 31, 31, E(3) ], [ 32, 32, 1 ], [ 33, 33, E(3) ], [ 34, 34, E(3)^2 ], [ 35, 35, E(3)^2 ], [ 36, 36, E(3) ], 
  [ 37, 37, E(3) ], [ 38, 38, E(3)^2 ], [ 39, 39, 1 ], [ 40, 40, E(3) ], [ 41, 41, 1 ], [ 42, 42, E(3)^2 ], [ 43, 43, E(3)^2 ], [ 44, 44, E(3)^2 ], 
  [ 45, 45, E(3)^2 ], [ 46, 46, E(3) ], [ 47, 47, 1 ], [ 48, 48, E(3)^2 ], [ 49, 49, 1 ], [ 50, 50, 1 ], [ 51, 51, 1 ], [ 52, 52, 1 ], [ 53, 53, E(3) ], [ 54, 54, E(3) ], 
  [ 55, 55, E(3) ], [ 56, 56, 1 ], [ 57, 57, E(3) ], [ 58, 58, E(3)^2 ], [ 59, 59, E(3)^2 ], [ 60, 60, E(3) ], [ 61, 61, E(3)^2 ], [ 62, 62, E(3) ], 
  [ 63, 63, E(3) ], [ 64, 64, 1 ], [ 65, 65, E(3)^2 ], [ 66, 66, 1 ], [ 67, 67, E(3)^2 ], [ 68, 68, E(3)^2 ], [ 69, 69, E(3)^2 ], [ 70, 70, 1 ], [ 71, 71, E(3) ], [ 72, 72, 1 ], 
  [ 73, 73, 1 ], [ 74, 74, E(3) ], [ 75, 75, E(3) ], [ 76, 76, E(3) ], [ 77, 77, 1 ], [ 78, 78, E(3) ], [ 79, 79, E(3)^2 ], [ 80, 80, E(3)^2 ], [ 81, 81, E(3) ], 
  [ 82, 82, E(3)^2 ], [ 83, 83, E(3) ], [ 84, 84, 1 ], [ 85, 85, 1 ], [ 86, 86, E(3)^2 ], [ 87, 87, E(3)^2 ], [ 88, 88, 1 ], [ 89, 89, 1 ], [ 90, 90, 1 ], [ 91, 91, E(3) ], 
  [ 92, 92, E(3) ], [ 93, 93, E(3) ], [ 94, 94, 1 ], [ 95, 95, E(3)^2 ], [ 96, 96, E(3)^2 ], [ 97, 97, E(3)^2 ], [ 98, 98, E(3) ], [ 99, 99, E(3)^2 ], [ 100, 100, 1 ], 
  [ 101, 101, E(3)^2 ], [ 102, 102, 1 ], [ 103, 103, 1 ], [ 104, 104, E(3) ], [ 105, 105, E(3) ], [ 106, 106, E(3) ], [ 107, 107, E(3)^2 ], [ 108, 108, E(3)^2 ], [ 109, 109, 1 ], 
  [ 110, 110, E(3)^2 ], [ 111, 111, 1 ], [ 112, 112, 1 ], [ 113, 113, E(3) ], [ 114, 114, E(3) ], [ 115, 115, E(3)^2 ], [ 116, 116, E(3)^2 ], [ 117, 117, 1 ], [ 118, 118, E(3) ], 
  [ 119, 119, 1 ], [ 120, 120, 1 ], [ 121, 121, E(3)^2 ], [ 122, 122, E(3)^2 ], [ 123, 123, E(3)^2 ], [ 124, 124, 1 ], [ 125, 125, E(3)^2 ], [ 126, 126, E(3)^2 ], [ 127, 127, E(3) ], 
  [ 128, 128, 1 ], [ 129, 129, E(3) ], [ 130, 130, E(3)^2 ], [ 131, 131, E(3)^2 ], [ 132, 132, E(3)^2 ], [ 133, 133, E(3) ], [ 134, 134, 1 ], [ 135, 135, E(3) ], 
  [ 136, 136, E(3) ], [ 137, 137, E(3) ], [ 138, 138, E(3) ], [ 139, 139, E(3) ], [ 140, 140, E(3) ], [ 141, 141, E(3)^2 ], [ 142, 142, 1 ], [ 143, 143, 1 ], [ 144, 144, 1 ], 
  [ 145, 145, 1 ], [ 146, 146, 1 ], [ 147, 147, 1 ], [ 148, 148, E(3)^2 ], [ 149, 149, E(3)^2 ], [ 150, 150, E(3)^2 ], [ 151, 151, E(3)^2 ], [ 152, 152, 1 ], 
  [ 153, 153, E(3)^2 ], [ 154, 154, E(3) ], [ 155, 155, E(3) ], [ 156, 156, E(3)^2 ], [ 157, 157, E(3)^2 ], [ 158, 158, E(3) ], [ 159, 159, 1 ], [ 160, 160, E(3)^2 ], [ 161, 161, 1 ], 
  [ 162, 162, E(3) ], [ 163, 163, E(3) ], [ 164, 164, E(3) ], [ 165, 165, E(3) ], [ 166, 166, E(3)^2 ], [ 167, 167, 1 ], [ 168, 168, E(3) ], [ 169, 169, 1 ], [ 170, 170, 1 ], 
  [ 171, 171, 1 ], [ 172, 172, 1 ], [ 173, 173, E(3)^2 ], [ 174, 174, E(3)^2 ], [ 175, 175, E(3)^2 ], [ 176, 176, 1 ], [ 177, 177, E(3)^2 ], [ 178, 178, E(3) ], [ 179, 179, E(3) ], 
  [ 180, 180, E(3)^2 ], [ 181, 181, E(3) ], [ 182, 182, E(3)^2 ], [ 183, 183, E(3)^2 ], [ 184, 184, 1 ], [ 185, 185, E(3) ], [ 186, 186, 1 ], [ 187, 187, E(3) ], 
  [ 188, 188, E(3) ], [ 189, 189, E(3) ], [ 190, 190, 1 ], [ 191, 191, E(3)^2 ], [ 192, 192, 1 ], [ 193, 193, 1 ], [ 194, 194, E(3)^2 ], [ 195, 195, E(3)^2 ], [ 196, 196, E(3)^2 ], 
  [ 197, 197, 1 ], [ 198, 198, E(3)^2 ], [ 199, 199, E(3) ], [ 200, 200, E(3) ], [ 201, 201, E(3)^2 ], [ 202, 202, E(3) ], [ 203, 203, E(3)^2 ], [ 204, 204, 1 ], 
  [ 205, 205, 1 ], [ 206, 206, E(3) ], [ 207, 207, E(3) ], [ 208, 208, 1 ], [ 209, 209, 1 ], [ 210, 210, 1 ], [ 211, 211, E(3)^2 ], [ 212, 212, E(3)^2 ], [ 213, 213, E(3)^2 ], 
  [ 214, 214, 1 ], [ 215, 215, E(3) ], [ 216, 216, E(3) ], [ 217, 217, E(3) ], [ 218, 218, E(3)^2 ], [ 219, 219, E(3) ], [ 220, 220, 1 ], [ 221, 221, E(3) ], 
  [ 222, 222, 1 ], [ 223, 223, 1 ], [ 224, 224, E(3)^2 ], [ 225, 225, E(3)^2 ], [ 226, 226, E(3)^2 ], [ 227, 227, E(3) ], [ 228, 228, E(3) ], [ 229, 229, 1 ], [ 230, 230, E(3) ], 
  [ 231, 231, 1 ], [ 232, 232, 1 ], [ 233, 233, E(3)^2 ], [ 234, 234, E(3)^2 ], [ 235, 235, E(3) ], [ 236, 236, E(3) ], [ 237, 237, 1 ], [ 238, 238, E(3)^2 ], 
  [ 239, 239, 1 ], [ 240, 240, 1 ], [ 241, 241, 1 ], [ 242, 242, 1 ], [ 243, 243, 1 ], [ 244, 244, 1 ], [ 245, 245, 1 ], [ 246, 246, 1 ], [ 247, 247, 1 ], [ 248, 248, 1 ] ],
[ 248, [ 1, 121, 1 ], [ 2, 122, 1 ], [ 3, 9, 1 ], [ 4, 26, 1 ], [ 5, 126, 1 ], [ 6, 125, 1 ], [ 7, 21, 1 ], [ 8, 8, 1 ], [ 9, 3, 1 ], 
  [ 10, 20, 1 ], [ 11, 38, 1 ], [ 12, 18, 1 ], [ 13, 133, 1 ], [ 14, 14, 1 ], [ 15, 29, 1 ], [ 16, 33, 1 ], [ 17, 31, 1 ], [ 18, 12, 1 ], [ 19, 30, 1 ], [ 20, 10, 1 ], [ 21, 7, 1 ], 
  [ 22, 22, 1 ], [ 23, 27, 1 ], [ 24, 25, 1 ], [ 25, 24, 1 ], [ 26, 4, 1 ], [ 27, 23, 1 ], [ 28, 34, 1 ], [ 29, 15, 1 ], [ 30, 19, 1 ], [ 31, 17, 1 ], 
  [ 32, 52, 1 ], [ 33, 16, 1 ], [ 34, 28, 1 ], [ 35, 46, 1 ], [ 36, 42, 1 ], [ 37, 48, 1 ], [ 38, 11, 1 ], [ 39, 41, 1 ], [ 40, 45, 1 ], [ 41, 39, 1 ], [ 42, 36, 1 ], [ 43, 54, 1 ], 
  [ 44, 57, -1 ], [ 45, 40, 1 ], [ 46, 35, 1 ], [ 47, 50, 1 ], [ 48, 37, 1 ], [ 49, 66, -1 ], [ 50, 47, 1 ], [ 51, 51, -1 ], [ 52, 32, 1 ], [ 53, 61, -1 ], 
  [ 54, 43, 1 ], [ 55, 59, -1 ], [ 56, 73, -1 ], [ 57, 44, -1 ], [ 58, 71, 1 ], [ 59, 55, -1 ], [ 60, 68, -1 ], [ 61, 53, -1 ], [ 62, 67, -1 ], [ 63, 69, 1 ], [ 64, 64, 1 ], 
  [ 65, 78, 1 ], [ 66, 49, -1 ], [ 67, 62, -1 ], [ 68, 60, -1 ], [ 69, 63, 1 ], [ 70, 85, 1 ], [ 71, 58, 1 ], [ 72, 72, 1 ], [ 73, 56, -1 ], [ 74, 79, -1 ], 
  [ 75, 82, 1 ], [ 76, 80, 1 ], [ 77, 90, 1 ], [ 78, 65, 1 ], [ 79, 74, -1 ], [ 80, 76, 1 ], [ 81, 87, 1 ], [ 82, 75, 1 ], [ 83, 86, 1 ], [ 84, 84, 1 ], [ 85, 70, 1 ], 
  [ 86, 83, 1 ], [ 87, 81, 1 ], [ 88, 100, -1 ], [ 89, 89, 1 ], [ 90, 77, 1 ], [ 91, 96, -1 ], [ 92, 95, -1 ], [ 93, 97, -1 ], [ 94, 94, 1 ], [ 95, 92, -1 ], [ 96, 91, -1 ], 
  [ 97, 93, -1 ], [ 98, 101, -1 ], [ 99, 106, 1 ], [ 100, 88, -1 ], [ 101, 98, -1 ], [ 102, 109, -1 ], [ 103, 103, 1 ], [ 104, 108, -1 ], [ 105, 107, -1 ], [ 106, 99, 1 ], 
  [ 107, 105, -1 ], [ 108, 104, -1 ], [ 109, 102, -1 ], [ 110, 113, -1 ], [ 111, 112, -1 ], [ 112, 111, -1 ], [ 113, 110, -1 ], [ 114, 115, -1 ], [ 115, 114, -1 ], [ 116, 118, 1 ], 
  [ 117, 117, 1 ], [ 118, 116, 1 ], [ 119, 119, 1 ], [ 120, 120, 1 ], [ 121, 1, 1 ], [ 122, 2, 1 ], [ 123, 129, 1 ], [ 124, 146, 1 ], [ 125, 6, 1 ], [ 126, 5, 1 ], 
  [ 127, 141, 1 ], [ 128, 128, 1 ], [ 129, 123, 1 ], [ 130, 140, 1 ], [ 131, 158, 1 ], [ 132, 138, 1 ], [ 133, 13, 1 ], [ 134, 134, 1 ], [ 135, 149, 1 ], [ 136, 153, 1 ], 
  [ 137, 151, 1 ], [ 138, 132, 1 ], [ 139, 150, 1 ], [ 140, 130, 1 ], [ 141, 127, 1 ], [ 142, 142, 1 ], [ 143, 147, 1 ], [ 144, 145, 1 ], [ 145, 144, 1 ], 
  [ 146, 124, 1 ], [ 147, 143, 1 ], [ 148, 154, 1 ], [ 149, 135, 1 ], [ 150, 139, 1 ], [ 151, 137, 1 ], [ 152, 172, 1 ], [ 153, 136, 1 ], [ 154, 148, 1 ], [ 155, 166, 1 ], 
  [ 156, 162, 1 ], [ 157, 168, 1 ], [ 158, 131, 1 ], [ 159, 161, 1 ], [ 160, 165, 1 ], [ 161, 159, 1 ], [ 162, 156, 1 ], [ 163, 174, 1 ], [ 164, 177, -1 ], [ 165, 160, 1 ], 
  [ 166, 155, 1 ], [ 167, 170, 1 ], [ 168, 157, 1 ], [ 169, 186, -1 ], [ 170, 167, 1 ], [ 171, 171, -1 ], [ 172, 152, 1 ], [ 173, 181, -1 ], [ 174, 163, 1 ], [ 175, 179, -1 ], 
  [ 176, 193, -1 ], [ 177, 164, -1 ], [ 178, 191, 1 ], [ 179, 175, -1 ], [ 180, 188, -1 ], [ 181, 173, -1 ], [ 182, 187, -1 ], [ 183, 189, 1 ], [ 184, 184, 1 ], [ 185, 198, 1 ], 
  [ 186, 169, -1 ], [ 187, 182, -1 ], [ 188, 180, -1 ], [ 189, 183, 1 ], [ 190, 205, 1 ], [ 191, 178, 1 ], [ 192, 192, 1 ], [ 193, 176, -1 ], [ 194, 199, -1 ], [ 195, 202, 1 ], 
  [ 196, 200, 1 ], [ 197, 210, 1 ], [ 198, 185, 1 ], [ 199, 194, -1 ], [ 200, 196, 1 ], [ 201, 207, 1 ], [ 202, 195, 1 ], [ 203, 206, 1 ], [ 204, 204, 1 ], [ 205, 190, 1 ], 
  [ 206, 203, 1 ], [ 207, 201, 1 ], [ 208, 220, -1 ], [ 209, 209, 1 ], [ 210, 197, 1 ], [ 211, 216, -1 ], [ 212, 215, -1 ], [ 213, 217, -1 ], [ 214, 214, 1 ], [ 215, 212, -1 ], 
  [ 216, 211, -1 ], [ 217, 213, -1 ], [ 218, 221, -1 ], [ 219, 226, 1 ], [ 220, 208, -1 ], [ 221, 218, -1 ], [ 222, 229, -1 ], [ 223, 223, 1 ], [ 224, 228, -1 ], [ 225, 227, -1 ], 
  [ 226, 219, 1 ], [ 227, 225, -1 ], [ 228, 224, -1 ], [ 229, 222, -1 ], [ 230, 233, -1 ], [ 231, 232, -1 ], [ 232, 231, -1 ], [ 233, 230, -1 ], [ 234, 235, -1 ], [ 235, 234, -1 ], 
  [ 236, 238, 1 ], [ 237, 237, 1 ], [ 238, 236, 1 ], [ 239, 239, 1 ], [ 240, 240, 1 ], [ 241, 241, -1 ], [ 241, 243, 1 ], [ 242, 242, -1 ], [ 242, 244, 1 ], [ 243, 243, 1 ], 
  [ 244, 244, 1 ], [ 245, 244, 1 ], [ 245, 246, -1 ], [ 245, 247, 1 ], [ 246, 244, 1 ], [ 246, 245, -1 ], [ 246, 247, 1 ], [ 247, 247, 1 ], [ 248, 248, 1 ] ]],

  [ [ 2, 0, 0, 2, 0, 0, 2, 0 ],
    [ 248, [ 1, 1, 1 ], [ 2, 2, -1 ], [ 3, 3, -1 ], [ 4, 4, 1 ], [ 5, 5, -1 ], [ 6, 6, 1 ], [ 7, 7, -1 ], [ 8, 8, -1 ], [ 9, 9, -1 ], 
  [ 10, 10, -1 ], [ 11, 11, -1 ], [ 12, 12, -1 ], [ 13, 13, -1 ], [ 14, 14, -1 ], [ 15, 15, 1 ], [ 16, 16, -1 ], [ 17, 17, 1 ], [ 18, 18, 1 ], [ 19, 19, 1 ], [ 20, 20, -1 ], 
  [ 21, 21, 1 ], [ 22, 22, 1 ], [ 23, 23, 1 ], [ 24, 24, 1 ], [ 25, 25, -1 ], [ 26, 26, 1 ], [ 27, 27, 1 ], [ 28, 28, 1 ], [ 29, 29, -1 ], [ 30, 30, -1 ], 
  [ 31, 31, 1 ], [ 32, 32, -1 ], [ 33, 33, -1 ], [ 34, 34, -1 ], [ 35, 35, -1 ], [ 36, 36, -1 ], [ 37, 37, -1 ], [ 38, 38, -1 ], [ 39, 39, -1 ], [ 40, 40, -1 ], [ 41, 41, 1 ], 
  [ 42, 42, 1 ], [ 43, 43, 1 ], [ 44, 44, 1 ], [ 45, 45, -1 ], [ 46, 46, 1 ], [ 47, 47, 1 ], [ 48, 48, 1 ], [ 49, 49, 1 ], [ 50, 50, -1 ], 
  [ 51, 51, 1 ], [ 52, 52, 1 ], [ 53, 53, 1 ], [ 54, 54, -1 ], [ 55, 55, -1 ], [ 56, 56, -1 ], [ 57, 57, -1 ], [ 58, 58, -1 ], [ 59, 59, -1 ], [ 60, 60, -1 ], [ 61, 61, -1 ], 
  [ 62, 62, 1 ], [ 63, 63, -1 ], [ 64, 64, 1 ], [ 65, 65, 1 ], [ 66, 66, -1 ], [ 67, 67, 1 ], [ 68, 68, 1 ], [ 69, 69, 1 ], [ 70, 70, 1 ], [ 71, 71, 1 ], 
  [ 72, 72, -1 ], [ 73, 73, 1 ], [ 74, 74, -1 ], [ 75, 75, -1 ], [ 76, 76, 1 ], [ 77, 77, -1 ], [ 78, 78, -1 ], [ 79, 79, -1 ], [ 80, 80, -1 ], [ 81, 81, 1 ], [ 82, 82, -1 ], 
  [ 83, 83, -1 ], [ 84, 84, 1 ], [ 85, 85, 1 ], [ 86, 86, 1 ], [ 87, 87, 1 ], [ 88, 88, 1 ], [ 89, 89, 1 ], [ 90, 90, -1 ], [ 91, 91, -1 ], 
  [ 92, 92, -1 ], [ 93, 93, -1 ], [ 94, 94, -1 ], [ 95, 95, 1 ], [ 96, 96, -1 ], [ 97, 97, -1 ], [ 98, 98, 1 ], [ 99, 99, 1 ], [ 100, 100, 1 ], [ 101, 101, 1 ], [ 102, 102, -1 ], 
  [ 103, 103, 1 ], [ 104, 104, -1 ], [ 105, 105, -1 ], [ 106, 106, -1 ], [ 107, 107, -1 ], [ 108, 108, 1 ], [ 109, 109, 1 ], [ 110, 110, 1 ], [ 111, 111, 1 ], [ 112, 112, -1 ], 
  [ 113, 113, -1 ], [ 114, 114, -1 ], [ 115, 115, 1 ], [ 116, 116, 1 ], [ 117, 117, -1 ], [ 118, 118, -1 ], [ 119, 119, 1 ], [ 120, 120, -1 ], [ 121, 121, 1 ], [ 122, 122, -1 ], 
  [ 123, 123, -1 ], [ 124, 124, 1 ], [ 125, 125, -1 ], [ 126, 126, 1 ], [ 127, 127, -1 ], [ 128, 128, -1 ], [ 129, 129, -1 ], [ 130, 130, -1 ], [ 131, 131, -1 ], [ 132, 132, -1 ], 
  [ 133, 133, -1 ], [ 134, 134, -1 ], [ 135, 135, 1 ], [ 136, 136, -1 ], [ 137, 137, 1 ], [ 138, 138, 1 ], [ 139, 139, 1 ], [ 140, 140, -1 ], [ 141, 141, 1 ], [ 142, 142, 1 ], 
  [ 143, 143, 1 ], [ 144, 144, 1 ], [ 145, 145, -1 ], [ 146, 146, 1 ], [ 147, 147, 1 ], [ 148, 148, 1 ], [ 149, 149, -1 ], [ 150, 150, -1 ], [ 151, 151, 1 ], [ 152, 152, -1 ], 
  [ 153, 153, -1 ], [ 154, 154, -1 ], [ 155, 155, -1 ], [ 156, 156, -1 ], [ 157, 157, -1 ], [ 158, 158, -1 ], [ 159, 159, -1 ], [ 160, 160, -1 ], [ 161, 161, 1 ], [ 162, 162, 1 ], 
  [ 163, 163, 1 ], [ 164, 164, 1 ], [ 165, 165, -1 ], [ 166, 166, 1 ], [ 167, 167, 1 ], [ 168, 168, 1 ], [ 169, 169, 1 ], [ 170, 170, -1 ], [ 171, 171, 1 ], [ 172, 172, 1 ], 
  [ 173, 173, 1 ], [ 174, 174, -1 ], [ 175, 175, -1 ], [ 176, 176, -1 ], [ 177, 177, -1 ], [ 178, 178, -1 ], [ 179, 179, -1 ], [ 180, 180, -1 ], [ 181, 181, -1 ], [ 182, 182, 1 ], 
  [ 183, 183, -1 ], [ 184, 184, 1 ], [ 185, 185, 1 ], [ 186, 186, -1 ], [ 187, 187, 1 ], [ 188, 188, 1 ], [ 189, 189, 1 ], [ 190, 190, 1 ], [ 191, 191, 1 ], [ 192, 192, -1 ], 
  [ 193, 193, 1 ], [ 194, 194, -1 ], [ 195, 195, -1 ], [ 196, 196, 1 ], [ 197, 197, -1 ], [ 198, 198, -1 ], [ 199, 199, -1 ], [ 200, 200, -1 ], [ 201, 201, 1 ], [ 202, 202, -1 ], 
  [ 203, 203, -1 ], [ 204, 204, 1 ], [ 205, 205, 1 ], [ 206, 206, 1 ], [ 207, 207, 1 ], [ 208, 208, 1 ], [ 209, 209, 1 ], [ 210, 210, -1 ], [ 211, 211, -1 ], [ 212, 212, -1 ], 
  [ 213, 213, -1 ], [ 214, 214, -1 ], [ 215, 215, 1 ], [ 216, 216, -1 ], [ 217, 217, -1 ], [ 218, 218, 1 ], [ 219, 219, 1 ], [ 220, 220, 1 ], [ 221, 221, 1 ], [ 222, 222, -1 ], 
  [ 223, 223, 1 ], [ 224, 224, -1 ], [ 225, 225, -1 ], [ 226, 226, -1 ], [ 227, 227, -1 ], [ 228, 228, 1 ], [ 229, 229, 1 ], [ 230, 230, 1 ], [ 231, 231, 1 ], [ 232, 232, -1 ], 
  [ 233, 233, -1 ], [ 234, 234, -1 ], [ 235, 235, 1 ], [ 236, 236, 1 ], [ 237, 237, -1 ], [ 238, 238, -1 ], [ 239, 239, 1 ], [ 240, 240, -1 ], [ 241, 241, 1 ], [ 242, 242, 1 ], 
  [ 243, 243, 1 ], [ 244, 244, 1 ], [ 245, 245, 1 ], [ 246, 246, 1 ], [ 247, 247, 1 ], [ 248, 248, 1 ] ] ],

  [[ 2, 0, 0, 2, 0, 0, 2, 2 ],
   [ 248, [ 1, 1, -1 ], [ 2, 2, -1 ], [ 3, 3, -1 ], [ 4, 4, 1 ], [ 5, 5, -1 ], [ 6, 6, 1 ], [ 7, 7, 1 ], [ 8, 8, 1 ], [ 9, 9, 1 ], 
  [ 10, 10, -1 ], [ 11, 11, -1 ], [ 12, 12, -1 ], [ 13, 13, -1 ], [ 14, 14, 1 ], [ 15, 15, 1 ], [ 16, 16, 1 ], [ 17, 17, 1 ], [ 18, 18, 1 ], [ 19, 19, 1 ], [ 20, 20, -1 ], 
  [ 21, 21, -1 ], [ 22, 22, 1 ], [ 23, 23, -1 ], [ 24, 24, -1 ], [ 25, 25, -1 ], [ 26, 26, 1 ], [ 27, 27, 1 ], [ 28, 28, -1 ], [ 29, 29, -1 ], [ 30, 30, 1 ], 
  [ 31, 31, -1 ], [ 32, 32, -1 ], [ 33, 33, -1 ], [ 34, 34, 1 ], [ 35, 35, 1 ], [ 36, 36, -1 ], [ 37, 37, 1 ], [ 38, 38, 1 ], [ 39, 39, -1 ], [ 40, 40, -1 ], [ 41, 41, -1 ], 
  [ 42, 42, 1 ], [ 43, 43, 1 ], [ 44, 44, -1 ], [ 45, 45, 1 ], [ 46, 46, 1 ], [ 47, 47, -1 ], [ 48, 48, 1 ], [ 49, 49, -1 ], [ 50, 50, -1 ], [ 51, 51, -1 ], 
  [ 52, 52, -1 ], [ 53, 53, 1 ], [ 54, 54, 1 ], [ 55, 55, 1 ], [ 56, 56, -1 ], [ 57, 57, 1 ], [ 58, 58, -1 ], [ 59, 59, -1 ], [ 60, 60, 1 ], [ 61, 61, 1 ], [ 62, 62, 1 ], 
  [ 63, 63, 1 ], [ 64, 64, 1 ], [ 65, 65, -1 ], [ 66, 66, -1 ], [ 67, 67, -1 ], [ 68, 68, 1 ], [ 69, 69, -1 ], [ 70, 70, 1 ], [ 71, 71, 1 ], [ 72, 72, 1 ], [ 73, 73, -1 ], 
  [ 74, 74, 1 ], [ 75, 75, -1 ], [ 76, 76, 1 ], [ 77, 77, 1 ], [ 78, 78, 1 ], [ 79, 79, -1 ], [ 80, 80, -1 ], [ 81, 81, -1 ], [ 82, 82, -1 ], [ 83, 83, 1 ], 
  [ 84, 84, 1 ], [ 85, 85, 1 ], [ 86, 86, -1 ], [ 87, 87, -1 ], [ 88, 88, 1 ], [ 89, 89, 1 ], [ 90, 90, 1 ], [ 91, 91, -1 ], [ 92, 92, -1 ], [ 93, 93, -1 ], [ 94, 94, 1 ], 
  [ 95, 95, 1 ], [ 96, 96, -1 ], [ 97, 97, 1 ], [ 98, 98, -1 ], [ 99, 99, 1 ], [ 100, 100, 1 ], [ 101, 101, 1 ], [ 102, 102, -1 ], [ 103, 103, 1 ], [ 104, 104, 1 ], 
  [ 105, 105, -1 ], [ 106, 106, -1 ], [ 107, 107, 1 ], [ 108, 108, 1 ], [ 109, 109, -1 ], [ 110, 110, 1 ], [ 111, 111, -1 ], [ 112, 112, -1 ], [ 113, 113, 1 ], [ 114, 114, 1 ], 
  [ 115, 115, -1 ], [ 116, 116, -1 ], [ 117, 117, 1 ], [ 118, 118, 1 ], [ 119, 119, 1 ], [ 120, 120, 1 ], [ 121, 121, -1 ], [ 122, 122, -1 ], [ 123, 123, -1 ], 
  [ 124, 124, 1 ], [ 125, 125, -1 ], [ 126, 126, 1 ], [ 127, 127, 1 ], [ 128, 128, 1 ], [ 129, 129, 1 ], [ 130, 130, -1 ], [ 131, 131, -1 ], [ 132, 132, -1 ], [ 133, 133, -1 ], 
  [ 134, 134, 1 ], [ 135, 135, 1 ], [ 136, 136, 1 ], [ 137, 137, 1 ], [ 138, 138, 1 ], [ 139, 139, 1 ], [ 140, 140, -1 ], [ 141, 141, -1 ], [ 142, 142, 1 ], [ 143, 143, -1 ], 
  [ 144, 144, -1 ], [ 145, 145, -1 ], [ 146, 146, 1 ], [ 147, 147, 1 ], [ 148, 148, -1 ], [ 149, 149, -1 ], [ 150, 150, 1 ], [ 151, 151, -1 ], [ 152, 152, -1 ], [ 153, 153, -1 ], 
  [ 154, 154, 1 ], [ 155, 155, 1 ], [ 156, 156, -1 ], [ 157, 157, 1 ], [ 158, 158, 1 ], [ 159, 159, -1 ], [ 160, 160, -1 ], [ 161, 161, -1 ], [ 162, 162, 1 ], [ 163, 163, 1 ], 
  [ 164, 164, -1 ], [ 165, 165, 1 ], [ 166, 166, 1 ], [ 167, 167, -1 ], [ 168, 168, 1 ], [ 169, 169, -1 ], [ 170, 170, -1 ], [ 171, 171, -1 ], [ 172, 172, -1 ], [ 173, 173, 1 ], 
  [ 174, 174, 1 ], [ 175, 175, 1 ], [ 176, 176, -1 ], [ 177, 177, 1 ], [ 178, 178, -1 ], [ 179, 179, -1 ], [ 180, 180, 1 ], [ 181, 181, 1 ], [ 182, 182, 1 ], [ 183, 183, 1 ], 
  [ 184, 184, 1 ], [ 185, 185, -1 ], [ 186, 186, -1 ], [ 187, 187, -1 ], [ 188, 188, 1 ], [ 189, 189, -1 ], [ 190, 190, 1 ], [ 191, 191, 1 ], [ 192, 192, 1 ], [ 193, 193, -1 ], 
  [ 194, 194, 1 ], [ 195, 195, -1 ], [ 196, 196, 1 ], [ 197, 197, 1 ], [ 198, 198, 1 ], [ 199, 199, -1 ], [ 200, 200, -1 ], [ 201, 201, -1 ], [ 202, 202, -1 ], [ 203, 203, 1 ], 
  [ 204, 204, 1 ], [ 205, 205, 1 ], [ 206, 206, -1 ], [ 207, 207, -1 ], [ 208, 208, 1 ], [ 209, 209, 1 ], [ 210, 210, 1 ], [ 211, 211, -1 ], [ 212, 212, -1 ], [ 213, 213, -1 ], 
  [ 214, 214, 1 ], [ 215, 215, 1 ], [ 216, 216, -1 ], [ 217, 217, 1 ], [ 218, 218, -1 ], [ 219, 219, 1 ], [ 220, 220, 1 ], [ 221, 221, 1 ], [ 222, 222, -1 ], [ 223, 223, 1 ], 
  [ 224, 224, 1 ], [ 225, 225, -1 ], [ 226, 226, -1 ], [ 227, 227, 1 ], [ 228, 228, 1 ], [ 229, 229, -1 ], [ 230, 230, 1 ], [ 231, 231, -1 ], [ 232, 232, -1 ], [ 233, 233, 1 ], 
  [ 234, 234, 1 ], [ 235, 235, -1 ], [ 236, 236, -1 ], [ 237, 237, 1 ], [ 238, 238, 1 ], [ 239, 239, 1 ], [ 240, 240, 1 ], [ 241, 241, 1 ], [ 242, 242, 1 ], [ 243, 243, 1 ], 
  [ 244, 244, 1 ], [ 245, 245, 1 ], [ 246, 246, 1 ], [ 247, 247, 1 ], [ 248, 248, 1 ] ] ],

  [ [ 2, 0, 0, 2, 0, 2, 0, 2 ],
    [ 248, [ 1, 1, 1 ], [ 2, 2, -1 ], [ 3, 3, -1 ], [ 4, 4, 1 ], [ 5, 5, -1 ], [ 6, 6, -1 ], [ 7, 7, -1 ], [ 8, 8, 1 ], [ 9, 9, -1 ], 
  [ 10, 10, -1 ], [ 11, 11, -1 ], [ 12, 12, -1 ], [ 13, 13, 1 ], [ 14, 14, 1 ], [ 15, 15, -1 ], [ 16, 16, -1 ], [ 17, 17, 1 ], [ 18, 18, 1 ], [ 19, 19, 1 ], [ 20, 20, 1 ], 
  [ 21, 21, -1 ], [ 22, 22, 1 ], [ 23, 23, 1 ], [ 24, 24, 1 ], [ 25, 25, -1 ], [ 26, 26, -1 ], [ 27, 27, -1 ], [ 28, 28, -1 ], [ 29, 29, -1 ], [ 30, 30, -1 ], 
  [ 31, 31, -1 ], [ 32, 32, -1 ], [ 33, 33, 1 ], [ 34, 34, 1 ], [ 35, 35, 1 ], [ 36, 36, -1 ], [ 37, 37, -1 ], [ 38, 38, 1 ], [ 39, 39, 1 ], [ 40, 40, 1 ], [ 41, 41, -1 ], 
  [ 42, 42, 1 ], [ 43, 43, 1 ], [ 44, 44, 1 ], [ 45, 45, 1 ], [ 46, 46, -1 ], [ 47, 47, 1 ], [ 48, 48, -1 ], [ 49, 49, -1 ], [ 50, 50, -1 ], [ 51, 51, -1 ], 
  [ 52, 52, -1 ], [ 53, 53, -1 ], [ 54, 54, -1 ], [ 55, 55, 1 ], [ 56, 56, -1 ], [ 57, 57, 1 ], [ 58, 58, 1 ], [ 59, 59, 1 ], [ 60, 60, -1 ], [ 61, 61, -1 ], [ 62, 62, 1 ], 
  [ 63, 63, 1 ], [ 64, 64, -1 ], [ 65, 65, 1 ], [ 66, 66, -1 ], [ 67, 67, 1 ], [ 68, 68, -1 ], [ 69, 69, -1 ], [ 70, 70, -1 ], [ 71, 71, 1 ], [ 72, 72, -1 ], 
  [ 73, 73, -1 ], [ 74, 74, 1 ], [ 75, 75, 1 ], [ 76, 76, 1 ], [ 77, 77, -1 ], [ 78, 78, 1 ], [ 79, 79, 1 ], [ 80, 80, -1 ], [ 81, 81, 1 ], [ 82, 82, -1 ], [ 83, 83, 1 ], 
  [ 84, 84, -1 ], [ 85, 85, 1 ], [ 86, 86, -1 ], [ 87, 87, -1 ], [ 88, 88, -1 ], [ 89, 89, 1 ], [ 90, 90, 1 ], [ 91, 91, 1 ], [ 92, 92, 1 ], [ 93, 93, -1 ], 
  [ 94, 94, 1 ], [ 95, 95, -1 ], [ 96, 96, -1 ], [ 97, 97, -1 ], [ 98, 98, -1 ], [ 99, 99, -1 ], [ 100, 100, 1 ], [ 101, 101, -1 ], [ 102, 102, 1 ], [ 103, 103, 1 ], [ 104, 104, 1 ], 
  [ 105, 105, -1 ], [ 106, 106, -1 ], [ 107, 107, -1 ], [ 108, 108, 1 ], [ 109, 109, 1 ], [ 110, 110, 1 ], [ 111, 111, 1 ], [ 112, 112, -1 ], [ 113, 113, -1 ], [ 114, 114, -1 ], 
  [ 115, 115, 1 ], [ 116, 116, 1 ], [ 117, 117, -1 ], [ 118, 118, 1 ], [ 119, 119, -1 ], [ 120, 120, -1 ], [ 121, 121, 1 ], [ 122, 122, -1 ], [ 123, 123, -1 ], [ 124, 124, 1 ], 
  [ 125, 125, -1 ], [ 126, 126, -1 ], [ 127, 127, -1 ], [ 128, 128, 1 ], [ 129, 129, -1 ], [ 130, 130, -1 ], [ 131, 131, -1 ], [ 132, 132, -1 ], [ 133, 133, 1 ], [ 134, 134, 1 ], 
  [ 135, 135, -1 ], [ 136, 136, -1 ], [ 137, 137, 1 ], [ 138, 138, 1 ], [ 139, 139, 1 ], [ 140, 140, 1 ], [ 141, 141, -1 ], [ 142, 142, 1 ], [ 143, 143, 1 ], [ 144, 144, 1 ], 
  [ 145, 145, -1 ], [ 146, 146, -1 ], [ 147, 147, -1 ], [ 148, 148, -1 ], [ 149, 149, -1 ], [ 150, 150, -1 ], [ 151, 151, -1 ], [ 152, 152, -1 ], [ 153, 153, 1 ], [ 154, 154, 1 ], 
  [ 155, 155, 1 ], [ 156, 156, -1 ], [ 157, 157, -1 ], [ 158, 158, 1 ], [ 159, 159, 1 ], [ 160, 160, 1 ], [ 161, 161, -1 ], [ 162, 162, 1 ], [ 163, 163, 1 ], [ 164, 164, 1 ], 
  [ 165, 165, 1 ], [ 166, 166, -1 ], [ 167, 167, 1 ], [ 168, 168, -1 ], [ 169, 169, -1 ], [ 170, 170, -1 ], [ 171, 171, -1 ], [ 172, 172, -1 ], [ 173, 173, -1 ], [ 174, 174, -1 ], 
  [ 175, 175, 1 ], [ 176, 176, -1 ], [ 177, 177, 1 ], [ 178, 178, 1 ], [ 179, 179, 1 ], [ 180, 180, -1 ], [ 181, 181, -1 ], [ 182, 182, 1 ], [ 183, 183, 1 ], [ 184, 184, -1 ], 
  [ 185, 185, 1 ], [ 186, 186, -1 ], [ 187, 187, 1 ], [ 188, 188, -1 ], [ 189, 189, -1 ], [ 190, 190, -1 ], [ 191, 191, 1 ], [ 192, 192, -1 ], [ 193, 193, -1 ], [ 194, 194, 1 ], 
  [ 195, 195, 1 ], [ 196, 196, 1 ], [ 197, 197, -1 ], [ 198, 198, 1 ], [ 199, 199, 1 ], [ 200, 200, -1 ], [ 201, 201, 1 ], [ 202, 202, -1 ], [ 203, 203, 1 ], [ 204, 204, -1 ], 
  [ 205, 205, 1 ], [ 206, 206, -1 ], [ 207, 207, -1 ], [ 208, 208, -1 ], [ 209, 209, 1 ], [ 210, 210, 1 ], [ 211, 211, 1 ], [ 212, 212, 1 ], [ 213, 213, -1 ], [ 214, 214, 1 ], 
  [ 215, 215, -1 ], [ 216, 216, -1 ], [ 217, 217, -1 ], [ 218, 218, -1 ], [ 219, 219, -1 ], [ 220, 220, 1 ], [ 221, 221, -1 ], [ 222, 222, 1 ], [ 223, 223, 1 ], [ 224, 224, 1 ], 
  [ 225, 225, -1 ], [ 226, 226, -1 ], [ 227, 227, -1 ], [ 228, 228, 1 ], [ 229, 229, 1 ], [ 230, 230, 1 ], [ 231, 231, 1 ], [ 232, 232, -1 ], [ 233, 233, -1 ], [ 234, 234, -1 ], 
  [ 235, 235, 1 ], [ 236, 236, 1 ], [ 237, 237, -1 ], [ 238, 238, 1 ], [ 239, 239, -1 ], [ 240, 240, -1 ], [ 241, 241, 1 ], [ 242, 242, 1 ], [ 243, 243, 1 ], [ 244, 244, 1 ], 
  [ 245, 245, 1 ], [ 246, 246, 1 ], [ 247, 247, 1 ], [ 248, 248, 1 ] ]],

  [ [ 2, 0, 0, 2, 0, 2, 2, 2 ],
    [ 248, [ 1, 1, -1 ], [ 2, 2, -1 ], [ 3, 3, -1 ], [ 4, 4, -1 ], [ 5, 5, -1 ], [ 6, 6, 1 ], [ 7, 7, 1 ], [ 8, 8, 1 ], [ 9, 9, 1 ], 
  [ 10, 10, 1 ], [ 11, 11, 1 ], [ 12, 12, 1 ], [ 13, 13, -1 ], [ 14, 14, 1 ], [ 15, 15, 1 ], [ 16, 16, -1 ], [ 17, 17, -1 ], [ 18, 18, -1 ], [ 19, 19, -1 ], [ 20, 20, 1 ], 
  [ 21, 21, -1 ], [ 22, 22, 1 ], [ 23, 23, 1 ], [ 24, 24, 1 ], [ 25, 25, 1 ], [ 26, 26, -1 ], [ 27, 27, -1 ], [ 28, 28, 1 ], [ 29, 29, -1 ], [ 30, 30, -1 ], 
  [ 31, 31, 1 ], [ 32, 32, -1 ], [ 33, 33, 1 ], [ 34, 34, -1 ], [ 35, 35, -1 ], [ 36, 36, 1 ], [ 37, 37, 1 ], [ 38, 38, -1 ], [ 39, 39, 1 ], [ 40, 40, -1 ], [ 41, 41, 1 ], 
  [ 42, 42, -1 ], [ 43, 43, -1 ], [ 44, 44, -1 ], [ 45, 45, 1 ], [ 46, 46, -1 ], [ 47, 47, 1 ], [ 48, 48, 1 ], [ 49, 49, -1 ], [ 50, 50, 1 ], [ 51, 51, -1 ], 
  [ 52, 52, -1 ], [ 53, 53, 1 ], [ 54, 54, -1 ], [ 55, 55, 1 ], [ 56, 56, -1 ], [ 57, 57, 1 ], [ 58, 58, -1 ], [ 59, 59, -1 ], [ 60, 60, 1 ], [ 61, 61, 1 ], [ 62, 62, 1 ], 
  [ 63, 63, -1 ], [ 64, 64, 1 ], [ 65, 65, -1 ], [ 66, 66, -1 ], [ 67, 67, -1 ], [ 68, 68, 1 ], [ 69, 69, 1 ], [ 70, 70, -1 ], [ 71, 71, 1 ], [ 72, 72, 1 ], 
  [ 73, 73, -1 ], [ 74, 74, 1 ], [ 75, 75, 1 ], [ 76, 76, -1 ], [ 77, 77, -1 ], [ 78, 78, 1 ], [ 79, 79, -1 ], [ 80, 80, 1 ], [ 81, 81, 1 ], [ 82, 82, 1 ], [ 83, 83, -1 ], 
  [ 84, 84, 1 ], [ 85, 85, -1 ], [ 86, 86, 1 ], [ 87, 87, 1 ], [ 88, 88, -1 ], [ 89, 89, 1 ], [ 90, 90, -1 ], [ 91, 91, 1 ], [ 92, 92, 1 ], [ 93, 93, -1 ], 
  [ 94, 94, 1 ], [ 95, 95, -1 ], [ 96, 96, 1 ], [ 97, 97, 1 ], [ 98, 98, -1 ], [ 99, 99, 1 ], [ 100, 100, -1 ], [ 101, 101, 1 ], [ 102, 102, -1 ], [ 103, 103, 1 ], [ 104, 104, 1 ], 
  [ 105, 105, -1 ], [ 106, 106, -1 ], [ 107, 107, 1 ], [ 108, 108, 1 ], [ 109, 109, -1 ], [ 110, 110, -1 ], [ 111, 111, 1 ], [ 112, 112, 1 ], [ 113, 113, -1 ], [ 114, 114, -1 ], 
  [ 115, 115, 1 ], [ 116, 116, -1 ], [ 117, 117, 1 ], [ 118, 118, 1 ], [ 119, 119, 1 ], [ 120, 120, 1 ], [ 121, 121, -1 ], [ 122, 122, -1 ], [ 123, 123, -1 ], [ 124, 124, -1 ], 
  [ 125, 125, -1 ], [ 126, 126, 1 ], [ 127, 127, 1 ], [ 128, 128, 1 ], [ 129, 129, 1 ], [ 130, 130, 1 ], [ 131, 131, 1 ], [ 132, 132, 1 ], [ 133, 133, -1 ], [ 134, 134, 1 ], 
  [ 135, 135, 1 ], [ 136, 136, -1 ], [ 137, 137, -1 ], [ 138, 138, -1 ], [ 139, 139, -1 ], [ 140, 140, 1 ], [ 141, 141, -1 ], [ 142, 142, 1 ], [ 143, 143, 1 ], [ 144, 144, 1 ], 
  [ 145, 145, 1 ], [ 146, 146, -1 ], [ 147, 147, -1 ], [ 148, 148, 1 ], [ 149, 149, -1 ], [ 150, 150, -1 ], [ 151, 151, 1 ], [ 152, 152, -1 ], [ 153, 153, 1 ], [ 154, 154, -1 ], 
  [ 155, 155, -1 ], [ 156, 156, 1 ], [ 157, 157, 1 ], [ 158, 158, -1 ], [ 159, 159, 1 ], [ 160, 160, -1 ], [ 161, 161, 1 ], [ 162, 162, -1 ], [ 163, 163, -1 ], [ 164, 164, -1 ], 
  [ 165, 165, 1 ], [ 166, 166, -1 ], [ 167, 167, 1 ], [ 168, 168, 1 ], [ 169, 169, -1 ], [ 170, 170, 1 ], [ 171, 171, -1 ], [ 172, 172, -1 ], [ 173, 173, 1 ], [ 174, 174, -1 ], 
  [ 175, 175, 1 ], [ 176, 176, -1 ], [ 177, 177, 1 ], [ 178, 178, -1 ], [ 179, 179, -1 ], [ 180, 180, 1 ], [ 181, 181, 1 ], [ 182, 182, 1 ], [ 183, 183, -1 ], [ 184, 184, 1 ], 
  [ 185, 185, -1 ], [ 186, 186, -1 ], [ 187, 187, -1 ], [ 188, 188, 1 ], [ 189, 189, 1 ], [ 190, 190, -1 ], [ 191, 191, 1 ], [ 192, 192, 1 ], [ 193, 193, -1 ], [ 194, 194, 1 ], 
  [ 195, 195, 1 ], [ 196, 196, -1 ], [ 197, 197, -1 ], [ 198, 198, 1 ], [ 199, 199, -1 ], [ 200, 200, 1 ], [ 201, 201, 1 ], [ 202, 202, 1 ], [ 203, 203, -1 ], [ 204, 204, 1 ], 
  [ 205, 205, -1 ], [ 206, 206, 1 ], [ 207, 207, 1 ], [ 208, 208, -1 ], [ 209, 209, 1 ], [ 210, 210, -1 ], [ 211, 211, 1 ], [ 212, 212, 1 ], [ 213, 213, -1 ], [ 214, 214, 1 ], 
  [ 215, 215, -1 ], [ 216, 216, 1 ], [ 217, 217, 1 ], [ 218, 218, -1 ], [ 219, 219, 1 ], [ 220, 220, -1 ], [ 221, 221, 1 ], [ 222, 222, -1 ], [ 223, 223, 1 ], [ 224, 224, 1 ], 
  [ 225, 225, -1 ], [ 226, 226, -1 ], [ 227, 227, 1 ], [ 228, 228, 1 ], [ 229, 229, -1 ], [ 230, 230, -1 ], [ 231, 231, 1 ], [ 232, 232, 1 ], [ 233, 233, -1 ], [ 234, 234, -1 ], 
  [ 235, 235, 1 ], [ 236, 236, -1 ], [ 237, 237, 1 ], [ 238, 238, 1 ], [ 239, 239, 1 ], [ 240, 240, 1 ], [ 241, 241, 1 ], [ 242, 242, 1 ], [ 243, 243, 1 ], [ 244, 244, 1 ], 
  [ 245, 245, 1 ], [ 246, 246, 1 ], [ 247, 247, 1 ], [ 248, 248, 1 ] ]],

  [ [ 0, 0, 0, 2, 0, 0, 0, 2 ],
    [ 248, [ 1, 1, 1 ], [ 2, 2, -1 ], [ 3, 3, -1 ], [ 4, 4, 1 ], [ 5, 5, 1 ], [ 6, 6, -1 ], [ 7, 7, 1 ], [ 8, 8, 1 ], [ 9, 9, -1 ], 
  [ 10, 10, -1 ], [ 11, 11, -1 ], [ 12, 12, 1 ], [ 13, 13, -1 ], [ 14, 14, -1 ], [ 15, 15, 1 ], [ 16, 16, -1 ], [ 17, 17, 1 ], [ 18, 18, -1 ], [ 19, 19, -1 ], [ 20, 20, -1 ], 
  [ 21, 21, -1 ], [ 22, 22, -1 ], [ 23, 23, 1 ], [ 24, 24, -1 ], [ 25, 25, 1 ], [ 26, 26, 1 ], [ 27, 27, 1 ], [ 28, 28, -1 ], [ 29, 29, -1 ], [ 30, 30, 1 ], 
  [ 31, 31, 1 ], [ 32, 32, 1 ], [ 33, 33, -1 ], [ 34, 34, 1 ], [ 35, 35, 1 ], [ 36, 36, -1 ], [ 37, 37, 1 ], [ 38, 38, -1 ], [ 39, 39, 1 ], [ 40, 40, -1 ], [ 41, 41, -1 ], 
  [ 42, 42, 1 ], [ 43, 43, 1 ], [ 44, 44, -1 ], [ 45, 45, -1 ], [ 46, 46, -1 ], [ 47, 47, 1 ], [ 48, 48, -1 ], [ 49, 49, -1 ], [ 50, 50, -1 ], [ 51, 51, 1 ], 
  [ 52, 52, -1 ], [ 53, 53, -1 ], [ 54, 54, -1 ], [ 55, 55, -1 ], [ 56, 56, -1 ], [ 57, 57, 1 ], [ 58, 58, 1 ], [ 59, 59, -1 ], [ 60, 60, -1 ], [ 61, 61, 1 ], [ 62, 62, -1 ], 
  [ 63, 63, 1 ], [ 64, 64, 1 ], [ 65, 65, 1 ], [ 66, 66, 1 ], [ 67, 67, -1 ], [ 68, 68, 1 ], [ 69, 69, -1 ], [ 70, 70, 1 ], [ 71, 71, -1 ], [ 72, 72, 1 ], 
  [ 73, 73, 1 ], [ 74, 74, 1 ], [ 75, 75, -1 ], [ 76, 76, -1 ], [ 77, 77, 1 ], [ 78, 78, -1 ], [ 79, 79, 1 ], [ 80, 80, 1 ], [ 81, 81, -1 ], [ 82, 82, -1 ], [ 83, 83, -1 ], 
  [ 84, 84, -1 ], [ 85, 85, 1 ], [ 86, 86, 1 ], [ 87, 87, -1 ], [ 88, 88, -1 ], [ 89, 89, 1 ], [ 90, 90, 1 ], [ 91, 91, 1 ], [ 92, 92, -1 ], 
  [ 93, 93, -1 ], [ 94, 94, 1 ], [ 95, 95, 1 ], [ 96, 96, 1 ], [ 97, 97, -1 ], [ 98, 98, -1 ], [ 99, 99, 1 ], [ 100, 100, -1 ], [ 101, 101, -1 ], [ 102, 102, -1 ], [ 103, 103, -1 ], 
  [ 104, 104, -1 ], [ 105, 105, 1 ], [ 106, 106, -1 ], [ 107, 107, 1 ], [ 108, 108, 1 ], [ 109, 109, 1 ], [ 110, 110, 1 ], [ 111, 111, 1 ], [ 112, 112, -1 ], [ 113, 113, -1 ], 
  [ 114, 114, -1 ], [ 115, 115, 1 ], [ 116, 116, 1 ], [ 117, 117, 1 ], [ 118, 118, -1 ], [ 119, 119, -1 ], [ 120, 120, -1 ], [ 121, 121, 1 ], [ 122, 122, -1 ], [ 123, 123, -1 ], 
  [ 124, 124, 1 ], [ 125, 125, 1 ], [ 126, 126, -1 ], [ 127, 127, 1 ], [ 128, 128, 1 ], [ 129, 129, -1 ], [ 130, 130, -1 ], [ 131, 131, -1 ], [ 132, 132, 1 ], [ 133, 133, -1 ], 
  [ 134, 134, -1 ], [ 135, 135, 1 ], [ 136, 136, -1 ], [ 137, 137, 1 ], [ 138, 138, -1 ], [ 139, 139, -1 ], [ 140, 140, -1 ], [ 141, 141, -1 ], [ 142, 142, -1 ], [ 143, 143, 1 ], 
  [ 144, 144, -1 ], [ 145, 145, 1 ], [ 146, 146, 1 ], [ 147, 147, 1 ], [ 148, 148, -1 ], [ 149, 149, -1 ], [ 150, 150, 1 ], [ 151, 151, 1 ], [ 152, 152, 1 ], [ 153, 153, -1 ], 
  [ 154, 154, 1 ], [ 155, 155, 1 ], [ 156, 156, -1 ], [ 157, 157, 1 ], [ 158, 158, -1 ], [ 159, 159, 1 ], [ 160, 160, -1 ], [ 161, 161, -1 ], [ 162, 162, 1 ], [ 163, 163, 1 ], 
  [ 164, 164, -1 ], [ 165, 165, -1 ], [ 166, 166, -1 ], [ 167, 167, 1 ], [ 168, 168, -1 ], [ 169, 169, -1 ], [ 170, 170, -1 ], [ 171, 171, 1 ], [ 172, 172, -1 ], [ 173, 173, -1 ], 
  [ 174, 174, -1 ], [ 175, 175, -1 ], [ 176, 176, -1 ], [ 177, 177, 1 ], [ 178, 178, 1 ], [ 179, 179, -1 ], [ 180, 180, -1 ], [ 181, 181, 1 ], [ 182, 182, -1 ], [ 183, 183, 1 ], 
  [ 184, 184, 1 ], [ 185, 185, 1 ], [ 186, 186, 1 ], [ 187, 187, -1 ], [ 188, 188, 1 ], [ 189, 189, -1 ], [ 190, 190, 1 ], [ 191, 191, -1 ], [ 192, 192, 1 ], [ 193, 193, 1 ], 
  [ 194, 194, 1 ], [ 195, 195, -1 ], [ 196, 196, -1 ], [ 197, 197, 1 ], [ 198, 198, -1 ], [ 199, 199, 1 ], [ 200, 200, 1 ], [ 201, 201, -1 ], [ 202, 202, -1 ], [ 203, 203, -1 ], 
  [ 204, 204, -1 ], [ 205, 205, 1 ], [ 206, 206, 1 ], [ 207, 207, -1 ], [ 208, 208, -1 ], [ 209, 209, 1 ], [ 210, 210, 1 ], [ 211, 211, 1 ], [ 212, 212, -1 ], [ 213, 213, -1 ], 
  [ 214, 214, 1 ], [ 215, 215, 1 ], [ 216, 216, 1 ], [ 217, 217, -1 ], [ 218, 218, -1 ], [ 219, 219, 1 ], [ 220, 220, -1 ], [ 221, 221, -1 ], [ 222, 222, -1 ], [ 223, 223, -1 ], 
  [ 224, 224, -1 ], [ 225, 225, 1 ], [ 226, 226, -1 ], [ 227, 227, 1 ], [ 228, 228, 1 ], [ 229, 229, 1 ], [ 230, 230, 1 ], [ 231, 231, 1 ], [ 232, 232, -1 ], [ 233, 233, -1 ], 
  [ 234, 234, -1 ], [ 235, 235, 1 ], [ 236, 236, 1 ], [ 237, 237, 1 ], [ 238, 238, -1 ], [ 239, 239, -1 ], [ 240, 240, -1 ], [ 241, 241, 1 ], [ 242, 242, 1 ], [ 243, 243, 1 ], 
  [ 244, 244, 1 ], [ 245, 245, 1 ], [ 246, 246, 1 ], [ 247, 247, 1 ], [ 248, 248, 1 ] ],

[ 248, [ 1, 1, -1/2 ], [ 1, 3, 3/16*E(3)-3/16*E(3)^2 ], [ 1, 9, 1/8*E(3)-1/8*E(3)^2 ], [ 1, 121, 9/16 ], [ 1, 123, 1/2*E(3)-1/2*E(3)^2 ], [ 1, 129, -3/8*E(3)+3/8*E(3)^2 ], 
  [ 1, 241, -9/8 ], [ 1, 243, 9/8 ], [ 1, 244, -3/8 ], [ 2, 2, 1/4 ], [ 2, 122, 3/4 ], [ 2, 242, 1/2*E(3)-1/2*E(3)^2 ], [ 2, 244, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 3, 3, -1/2 ], [ 3, 121, -1/2*E(3)+1/2*E(3)^2 ], [ 4, 4, 1/4 ], [ 4, 10, 1/4*E(3)-1/4*E(3)^2 ], [ 4, 11, -1/8*E(3)+1/8*E(3)^2 ], [ 4, 16, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 4, 17, 3/8 ], [ 4, 23, 3/4 ], [ 5, 5, -1/2 ], [ 5, 6, -9/64*E(3)+9/64*E(3)^2 ], [ 5, 13, -3/8*E(3)+3/8*E(3)^2 ], [ 5, 14, 3/16*E(3)-3/16*E(3)^2 ], [ 5, 21, 1/2*E(3)-1/2*E(3)^2 ], 
  [ 5, 125, 9/64 ], [ 5, 127, 9/64 ], [ 5, 134, -1/16*E(3)+1/16*E(3)^2 ], [ 5, 141, 3/64*E(3)-3/64*E(3)^2 ], [ 5, 244, 3/8 ], [ 5, 245, -9/16 ], [ 5, 246, 3/16 ], 
  [ 5, 247, -3/16 ], [ 5, 248, 3/16 ], [ 6, 6, -1/2 ], [ 6, 127, -1/2*E(3)+1/2*E(3)^2 ], [ 7, 6, 27/64*E(3)-27/64*E(3)^2 ], [ 7, 7, -1/2 ], [ 7, 14, -3/16*E(3)+3/16*E(3)^2 ], 
  [ 7, 125, 9/64 ], [ 7, 126, 1/2*E(3)-1/2*E(3)^2 ], [ 7, 127, 81/64 ], [ 7, 133, -1/8*E(3)+1/8*E(3)^2 ], [ 7, 134, 9/16*E(3)-9/16*E(3)^2 ], [ 7, 141, -9/64*E(3)+9/64*E(3)^2 ], 
  [ 7, 245, -9/16 ], [ 7, 246, 27/16 ], [ 7, 247, -27/16 ], [ 7, 248, 9/16 ], [ 8, 8, -1/2 ], [ 8, 22, -1/2*E(3)+1/2*E(3)^2 ], [ 9, 3, 3/8 ], [ 9, 9, 1/4 ], 
  [ 9, 121, 3/8*E(3)-3/8*E(3)^2 ], [ 9, 129, 3/4 ], [ 9, 241, -1/4*E(3)+1/4*E(3)^2 ], [ 9, 243, -1/4*E(3)+1/4*E(3)^2 ], [ 9, 244, 1/4*E(3)-1/4*E(3)^2 ], [ 10, 4, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 10, 10, 1/4 ], [ 10, 11, 3/8 ], [ 10, 16, 3/4 ], [ 10, 17, -1/8*E(3)+1/8*E(3)^2 ], [ 10, 23, -1/4*E(3)+1/4*E(3)^2 ], [ 11, 11, -1/2 ], [ 11, 17, -1/2*E(3)+1/2*E(3)^2 ], 
  [ 12, 4, 3/32 ], [ 12, 10, 3/32*E(3)-3/32*E(3)^2 ], [ 12, 11, -3/64*E(3)+3/64*E(3)^2 ], [ 12, 12, -1/8 ], [ 12, 16, -3/32*E(3)+3/32*E(3)^2 ], [ 12, 17, 9/64 ], 
  [ 12, 18, -1/8*E(3)+1/8*E(3)^2 ], [ 12, 19, 1/16*E(3)-1/16*E(3)^2 ], [ 12, 20, -3/32*E(3)+3/32*E(3)^2 ], [ 12, 23, 9/32 ], [ 12, 24, 1/8*E(3)-1/8*E(3)^2 ], [ 12, 25, -3/16 ], 
  [ 12, 26, 9/32 ], [ 12, 27, -9/64 ], [ 12, 28, 1/8*E(3)-1/8*E(3)^2 ], [ 12, 30, -3/8 ], [ 12, 31, -9/32 ], [ 12, 33, -9/64*E(3)+9/64*E(3)^2 ], [ 12, 34, -3/8 ], 
  [ 12, 35, 3/16 ], [ 12, 38, -9/32*E(3)+9/32*E(3)^2 ], [ 12, 39, 3/8 ], [ 12, 41, 3/16*E(3)-3/16*E(3)^2 ], [ 12, 46, 3/8*E(3)-3/8*E(3)^2 ], [ 13, 6, 3/8 ], [ 13, 13, 1 ], 
  [ 13, 127, 1/8*E(3)-1/8*E(3)^2 ], [ 14, 6, -9/16 ], [ 14, 14, 1/4 ], [ 14, 125, 1/16*E(3)-1/16*E(3)^2 ], [ 14, 127, -9/16*E(3)+9/16*E(3)^2 ], [ 14, 134, 3/4 ], [ 14, 141, -3/16 ], 
  [ 14, 245, -1/4*E(3)+1/4*E(3)^2 ], [ 14, 246, 1/4*E(3)-1/4*E(3)^2 ], [ 14, 247, 1/4*E(3)-1/4*E(3)^2 ], [ 14, 248, -1/4*E(3)+1/4*E(3)^2 ], [ 15, 8, 9/8 ], 
  [ 15, 15, 1 ], [ 15, 22, 3/8*E(3)-3/8*E(3)^2 ], [ 16, 4, -1/4*E(3)+1/4*E(3)^2 ], [ 16, 10, 3/4 ], [ 16, 11, 3/8 ], [ 16, 16, 1/4 ], [ 16, 17, 3/8*E(3)-3/8*E(3)^2 ], 
  [ 16, 23, 1/4*E(3)-1/4*E(3)^2 ], [ 17, 11, -1/2*E(3)+1/2*E(3)^2 ], [ 17, 17, -1/2 ], [ 18, 4, 3/32*E(3)-3/32*E(3)^2 ], [ 18, 10, 3/32 ], 
  [ 18, 11, 9/64 ], [ 18, 12, -1/8*E(3)+1/8*E(3)^2 ], [ 18, 16, 9/32 ], [ 18, 17, -3/64*E(3)+3/64*E(3)^2 ], [ 18, 18, -1/8 ], [ 18, 19, -3/16 ], [ 18, 20, 9/32 ], 
  [ 18, 23, -3/32*E(3)+3/32*E(3)^2 ], [ 18, 24, -3/8 ], [ 18, 25, 1/16*E(3)-1/16*E(3)^2 ], [ 18, 26, -3/32*E(3)+3/32*E(3)^2 ], [ 18, 27, -9/64*E(3)+9/64*E(3)^2 ], 
  [ 18, 28, -3/8 ], [ 18, 30, 1/8*E(3)-1/8*E(3)^2 ], [ 18, 31, -9/32*E(3)+9/32*E(3)^2 ], [ 18, 33, -9/64 ], [ 18, 34, 1/8*E(3)-1/8*E(3)^2 ], [ 18, 35, 3/16*E(3)-3/16*E(3)^2 ], 
  [ 18, 38, -9/32 ], [ 18, 39, 3/8*E(3)-3/8*E(3)^2 ], [ 18, 41, 3/16 ], [ 18, 46, 3/8 ], [ 19, 11, -3/16 ], [ 19, 17, -3/16*E(3)+3/16*E(3)^2 ], [ 19, 19, 1/4 ], 
  [ 19, 25, 1/4*E(3)-1/4*E(3)^2 ], [ 19, 27, 3/16*E(3)-3/16*E(3)^2 ], [ 19, 33, -9/16 ], [ 19, 35, -1/4*E(3)+1/4*E(3)^2 ], [ 19, 41, 3/4 ], [ 20, 20, 1/4 ], 
  [ 20, 26, 1/4*E(3)-1/4*E(3)^2 ], [ 20, 27, -1/8*E(3)+1/8*E(3)^2 ], [ 20, 31, -1/4*E(3)+1/4*E(3)^2 ], [ 20, 33, 3/8 ], [ 20, 38, 3/4 ], 
  [ 21, 5, 1/2*E(3)-1/2*E(3)^2 ], [ 21, 6, 27/64 ], [ 21, 13, 9/8 ], [ 21, 14, -3/16 ], [ 21, 21, -1/2 ], [ 21, 125, -3/64*E(3)+3/64*E(3)^2 ], [ 21, 127, 9/64*E(3)-9/64*E(3)^2 ], 
  [ 21, 134, -3/16 ], [ 21, 141, 3/64 ], [ 21, 244, -1/8*E(3)+1/8*E(3)^2 ], [ 21, 245, 5/16*E(3)-5/16*E(3)^2 ], [ 21, 246, -3/16*E(3)+3/16*E(3)^2 ], [ 21, 247, -1/16*E(3)+1/16*E(3)^2 ], 
  [ 21, 248, 1/16*E(3)-1/16*E(3)^2 ], [ 22, 8, -1/2*E(3)+1/2*E(3)^2 ], [ 22, 22, -1/2 ], [ 23, 4, 3/4 ], [ 23, 10, -1/4*E(3)+1/4*E(3)^2 ], [ 23, 11, 3/8*E(3)-3/8*E(3)^2 ], 
  [ 23, 16, 1/4*E(3)-1/4*E(3)^2 ], [ 23, 17, 3/8 ], [ 23, 23, 1/4 ], [ 24, 4, -3/32*E(3)+3/32*E(3)^2 ], [ 24, 10, 9/32 ], [ 24, 11, 9/64 ], [ 24, 12, 1/8*E(3)-1/8*E(3)^2 ], 
  [ 24, 16, 3/32 ], [ 24, 17, 9/64*E(3)-9/64*E(3)^2 ], [ 24, 18, -3/8 ], [ 24, 19, -3/16 ], [ 24, 20, -9/32 ], [ 24, 23, 3/32*E(3)-3/32*E(3)^2 ], [ 24, 24, -1/8 ], 
  [ 24, 25, -3/16*E(3)+3/16*E(3)^2 ], [ 24, 26, -9/32*E(3)+9/32*E(3)^2 ], [ 24, 27, -9/64*E(3)+9/64*E(3)^2 ], [ 24, 28, 3/8 ], [ 24, 30, -1/8*E(3)+1/8*E(3)^2 ], 
  [ 24, 31, -3/32*E(3)+3/32*E(3)^2 ], [ 24, 33, 27/64 ], [ 24, 34, 3/8*E(3)-3/8*E(3)^2 ], [ 24, 35, 3/16*E(3)-3/16*E(3)^2 ], 
  [ 24, 38, 9/32 ], [ 24, 39, 1/8*E(3)-1/8*E(3)^2 ], [ 24, 41, -9/16 ], [ 24, 46, -3/8 ], [ 25, 11, -3/16*E(3)+3/16*E(3)^2 ], [ 25, 17, -3/16 ], [ 25, 19, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 25, 25, 1/4 ], [ 25, 27, -9/16 ], [ 25, 33, 3/16*E(3)-3/16*E(3)^2 ], [ 25, 35, 3/4 ], [ 25, 41, -1/4*E(3)+1/4*E(3)^2 ], [ 26, 20, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 26, 26, 1/4 ], [ 26, 27, 3/8 ], [ 26, 31, 3/4 ], [ 26, 33, -1/8*E(3)+1/8*E(3)^2 ], [ 26, 38, -1/4*E(3)+1/4*E(3)^2 ], [ 27, 27, -1/2 ], [ 27, 33, -1/2*E(3)+1/2*E(3)^2 ], 
  [ 28, 4, -1/32*E(3)+1/32*E(3)^2 ], [ 28, 10, 3/32 ], [ 28, 11, -3/64 ], [ 28, 12, 1/8*E(3)-1/8*E(3)^2 ], [ 28, 16, -3/32 ], [ 28, 17, -3/64*E(3)+3/64*E(3)^2 ], 
  [ 28, 18, -3/8 ], [ 28, 19, 3/16 ], [ 28, 20, 9/32 ], [ 28, 23, -3/32*E(3)+3/32*E(3)^2 ], [ 28, 24, 3/8 ], [ 28, 25, 3/16*E(3)-3/16*E(3)^2 ], [ 28, 26, 9/32*E(3)-9/32*E(3)^2 ], 
  [ 28, 27, -9/64*E(3)+9/64*E(3)^2 ], [ 28, 28, -1/8 ], [ 28, 30, 3/8*E(3)-3/8*E(3)^2 ], [ 28, 31, -9/32*E(3)+9/32*E(3)^2 ], [ 28, 33, 27/64 ], [ 28, 34, -1/8*E(3)+1/8*E(3)^2 ], 
  [ 28, 35, 1/16*E(3)-1/16*E(3)^2 ], [ 28, 38, 27/32 ], [ 28, 39, 1/8*E(3)-1/8*E(3)^2 ], [ 28, 41, -3/16 ], [ 28, 46, -3/8 ], [ 29, 8, 1/8*E(3)-1/8*E(3)^2 ], 
  [ 29, 22, 3/8 ], [ 29, 29, 1 ], [ 30, 4, 9/32 ], [ 30, 10, -3/32*E(3)+3/32*E(3)^2 ], [ 30, 11, 9/64*E(3)-9/64*E(3)^2 ], [ 30, 12, -3/8 ], [ 30, 16, 3/32*E(3)-3/32*E(3)^2 ], 
  [ 30, 17, 9/64 ], [ 30, 18, 1/8*E(3)-1/8*E(3)^2 ], [ 30, 19, -3/16*E(3)+3/16*E(3)^2 ], [ 30, 20, -9/32*E(3)+9/32*E(3)^2 ], [ 30, 23, 3/32 ], [ 30, 24, -1/8*E(3)+1/8*E(3)^2 ], 
  [ 30, 25, -3/16 ], [ 30, 26, -9/32 ], [ 30, 27, 27/64 ], [ 30, 28, 3/8*E(3)-3/8*E(3)^2 ], [ 30, 30, -1/8 ], [ 30, 31, 9/32 ], [ 30, 33, -9/64*E(3)+9/64*E(3)^2 ], 
  [ 30, 34, 3/8 ], [ 30, 35, -9/16 ], [ 30, 38, -3/32*E(3)+3/32*E(3)^2 ], [ 30, 39, -3/8 ], [ 30, 41, 3/16*E(3)-3/16*E(3)^2 ], [ 30, 46, 1/8*E(3)-1/8*E(3)^2 ], 
  [ 31, 20, -1/4*E(3)+1/4*E(3)^2 ], [ 31, 26, 3/4 ], [ 31, 27, 3/8 ], [ 31, 31, 1/4 ], [ 31, 33, 3/8*E(3)-3/8*E(3)^2 ], [ 31, 38, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 32, 32, 1/4 ], [ 32, 40, 3/16*E(3)-3/16*E(3)^2 ], [ 32, 44, 1/4*E(3)-1/4*E(3)^2 ], [ 32, 49, -1/4*E(3)+1/4*E(3)^2 ], [ 32, 51, -9/16 ], [ 32, 58, 3/4 ], 
  [ 33, 27, -1/2*E(3)+1/2*E(3)^2 ], [ 33, 33, -1/2 ], [ 34, 4, 3/32 ], [ 34, 10, -1/32*E(3)+1/32*E(3)^2 ], [ 34, 11, -3/64*E(3)+3/64*E(3)^2 ], 
  [ 34, 12, -3/8 ], [ 34, 16, -3/32*E(3)+3/32*E(3)^2 ], [ 34, 17, -3/64 ], [ 34, 18, 1/8*E(3)-1/8*E(3)^2 ], [ 34, 19, 3/16*E(3)-3/16*E(3)^2 ], [ 34, 20, 9/32*E(3)-9/32*E(3)^2 ], 
  [ 34, 23, -3/32 ], [ 34, 24, 3/8*E(3)-3/8*E(3)^2 ], [ 34, 25, 3/16 ], [ 34, 26, 9/32 ], [ 34, 27, 27/64 ], [ 34, 28, -1/8*E(3)+1/8*E(3)^2 ], [ 34, 30, 3/8 ], 
  [ 34, 31, 27/32 ], [ 34, 33, -9/64*E(3)+9/64*E(3)^2 ], [ 34, 34, -1/8 ], [ 34, 35, -3/16 ], [ 34, 38, -9/32*E(3)+9/32*E(3)^2 ], [ 34, 39, -3/8 ], [ 34, 41, 1/16*E(3)-1/16*E(3)^2 ], 
  [ 34, 46, 1/8*E(3)-1/8*E(3)^2 ], [ 35, 11, 1/16*E(3)-1/16*E(3)^2 ], [ 35, 17, -3/16 ], [ 35, 19, -1/4*E(3)+1/4*E(3)^2 ], [ 35, 25, 3/4 ], [ 35, 27, -9/16 ], 
  [ 35, 33, -9/16*E(3)+9/16*E(3)^2 ], [ 35, 35, 1/4 ], [ 35, 41, 1/4*E(3)-1/4*E(3)^2 ], [ 36, 36, 1/4 ], [ 36, 42, 1/4*E(3)-1/4*E(3)^2 ], [ 36, 43, -1/8*E(3)+1/8*E(3)^2 ], 
  [ 36, 47, -1/4*E(3)+1/4*E(3)^2 ], [ 36, 50, 3/8 ], [ 36, 54, 3/4 ], [ 37, 32, -3/8 ], [ 37, 37, -1/2 ], [ 37, 40, -9/32*E(3)+9/32*E(3)^2 ], [ 37, 44, -1/8*E(3)+1/8*E(3)^2 ], 
  [ 37, 45, -3/8*E(3)+3/8*E(3)^2 ], [ 37, 49, 3/8*E(3)-3/8*E(3)^2 ], [ 37, 51, 9/32 ], [ 37, 53, 1/2*E(3)-1/2*E(3)^2 ], [ 37, 58, -3/8 ], [ 38, 20, 3/4 ], 
  [ 38, 26, -1/4*E(3)+1/4*E(3)^2 ], [ 38, 27, 3/8*E(3)-3/8*E(3)^2 ], [ 38, 31, 1/4*E(3)-1/4*E(3)^2 ], [ 38, 33, 3/8 ], [ 38, 38, 1/4 ], [ 39, 4, -3/32 ], 
  [ 39, 10, -3/32*E(3)+3/32*E(3)^2 ], [ 39, 11, -3/64*E(3)+3/64*E(3)^2 ], [ 39, 12, 3/8 ], [ 39, 16, -1/32*E(3)+1/32*E(3)^2 ], [ 39, 17, 9/64 ], 
  [ 39, 18, 3/8*E(3)-3/8*E(3)^2 ], [ 39, 19, 3/16*E(3)-3/16*E(3)^2 ], [ 39, 20, -9/32*E(3)+9/32*E(3)^2 ], [ 39, 23, 3/32 ], [ 39, 24, 1/8*E(3)-1/8*E(3)^2 ], [ 39, 25, -9/16 ], 
  [ 39, 26, 27/32 ], [ 39, 27, 27/64 ], [ 39, 28, 1/8*E(3)-1/8*E(3)^2 ], [ 39, 30, -3/8 ], [ 39, 31, 9/32 ], [ 39, 33, 27/64*E(3)-27/64*E(3)^2 ], [ 39, 34, -3/8 ], 
  [ 39, 35, -3/16 ], [ 39, 38, 9/32*E(3)-9/32*E(3)^2 ], [ 39, 39, -1/8 ], [ 39, 41, -3/16*E(3)+3/16*E(3)^2 ], [ 39, 46, -1/8*E(3)+1/8*E(3)^2 ], [ 40, 40, -1/2 ], 
  [ 40, 51, -1/2*E(3)+1/2*E(3)^2 ], [ 41, 11, -3/16 ], [ 41, 17, 1/16*E(3)-1/16*E(3)^2 ], [ 41, 19, 3/4 ], [ 41, 25, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 41, 27, -9/16*E(3)+9/16*E(3)^2 ], [ 41, 33, -9/16 ], [ 41, 35, 1/4*E(3)-1/4*E(3)^2 ], [ 41, 41, 1/4 ], [ 42, 36, 1/4*E(3)-1/4*E(3)^2 ], [ 42, 42, 1/4 ], [ 42, 43, 3/8 ], 
  [ 42, 47, 3/4 ], [ 42, 50, -1/8*E(3)+1/8*E(3)^2 ], [ 42, 54, -1/4*E(3)+1/4*E(3)^2 ], [ 43, 43, -1/2 ], [ 43, 50, -1/2*E(3)+1/2*E(3)^2 ], [ 44, 32, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 44, 40, -9/16 ], [ 44, 44, 1/4 ], [ 44, 49, 3/4 ], [ 44, 51, 3/16*E(3)-3/16*E(3)^2 ], [ 44, 58, -1/4*E(3)+1/4*E(3)^2 ], [ 45, 40, 3/4 ], [ 45, 45, 1 ], 
  [ 45, 51, 1/4*E(3)-1/4*E(3)^2 ], [ 46, 4, -3/32*E(3)+3/32*E(3)^2 ], [ 46, 10, -3/32 ], [ 46, 11, 9/64 ], [ 46, 12, 3/8*E(3)-3/8*E(3)^2 ], [ 46, 16, 3/32 ], 
  [ 46, 17, -3/64*E(3)+3/64*E(3)^2 ], [ 46, 18, 3/8 ], [ 46, 19, -9/16 ], [ 46, 20, 27/32 ], [ 46, 23, -1/32*E(3)+1/32*E(3)^2 ], [ 46, 24, -3/8 ], 
  [ 46, 25, 3/16*E(3)-3/16*E(3)^2 ], [ 46, 26, -9/32*E(3)+9/32*E(3)^2 ], [ 46, 27, 27/64*E(3)-27/64*E(3)^2 ], [ 46, 28, -3/8 ], [ 46, 30, 1/8*E(3)-1/8*E(3)^2 ], 
  [ 46, 31, 9/32*E(3)-9/32*E(3)^2 ], [ 46, 33, 27/64 ], [ 46, 34, 1/8*E(3)-1/8*E(3)^2 ], [ 46, 35, -3/16*E(3)+3/16*E(3)^2 ], [ 46, 38, 9/32 ], [ 46, 39, -1/8*E(3)+1/8*E(3)^2 ], 
  [ 46, 41, -3/16 ], [ 46, 46, -1/8 ], [ 47, 36, -1/4*E(3)+1/4*E(3)^2 ], [ 47, 42, 3/4 ], [ 47, 43, 3/8 ], [ 47, 47, 1/4 ], [ 47, 50, 3/8*E(3)-3/8*E(3)^2 ], 
  [ 47, 54, 1/4*E(3)-1/4*E(3)^2 ], [ 48, 40, -3/16 ], [ 48, 48, 1/4 ], [ 48, 51, -3/16*E(3)+3/16*E(3)^2 ], [ 48, 57, 1/4*E(3)-1/4*E(3)^2 ], [ 48, 61, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 48, 71, -3/4 ], [ 49, 32, -1/4*E(3)+1/4*E(3)^2 ], [ 49, 40, -9/16 ], [ 49, 44, 3/4 ], [ 49, 49, 1/4 ], [ 49, 51, -9/16*E(3)+9/16*E(3)^2 ], [ 49, 58, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 50, 43, -1/2*E(3)+1/2*E(3)^2 ], [ 50, 50, -1/2 ], [ 51, 40, -1/2*E(3)+1/2*E(3)^2 ], [ 51, 51, -1/2 ], [ 52, 40, 9/32 ], [ 52, 45, 3/8 ], [ 52, 48, -3/8 ], 
  [ 52, 51, 3/32*E(3)-3/32*E(3)^2 ], [ 52, 52, -1/2 ], [ 52, 57, -1/8*E(3)+1/8*E(3)^2 ], [ 52, 61, -3/8*E(3)+3/8*E(3)^2 ], [ 52, 66, -1/2*E(3)+1/2*E(3)^2 ], [ 52, 71, 3/8 ], 
  [ 53, 32, 3/8*E(3)-3/8*E(3)^2 ], [ 53, 37, 1/2*E(3)-1/2*E(3)^2 ], [ 53, 40, 27/32 ], [ 53, 44, -3/8 ], [ 53, 45, 9/8 ], [ 53, 49, -3/8 ], [ 53, 51, 9/32*E(3)-9/32*E(3)^2 ], 
  [ 53, 53, -1/2 ], [ 53, 58, -1/8*E(3)+1/8*E(3)^2 ], [ 54, 36, 3/4 ], [ 54, 42, -1/4*E(3)+1/4*E(3)^2 ], [ 54, 43, 3/8*E(3)-3/8*E(3)^2 ], [ 54, 47, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 54, 50, 3/8 ], [ 54, 54, 1/4 ], [ 55, 32, -1/16*E(3)+1/16*E(3)^2 ], [ 55, 40, -9/32 ], [ 55, 44, 3/16 ], [ 55, 48, 9/16 ], [ 55, 49, 3/16 ], [ 55, 51, -9/32*E(3)+9/32*E(3)^2 ], 
  [ 55, 55, -1/2 ], [ 55, 57, 9/16*E(3)-9/16*E(3)^2 ], [ 55, 58, 3/16*E(3)-3/16*E(3)^2 ], [ 55, 61, 3/16*E(3)-3/16*E(3)^2 ], [ 55, 64, -1/2*E(3)+1/2*E(3)^2 ], 
  [ 55, 71, -9/16 ], [ 56, 56, -1/2 ], [ 56, 65, -1/2*E(3)+1/2*E(3)^2 ], [ 57, 40, -3/16*E(3)+3/16*E(3)^2 ], [ 57, 48, 1/4*E(3)-1/4*E(3)^2 ], [ 57, 51, -3/16 ], [ 57, 57, 1/4 ], 
  [ 57, 61, -3/4 ], [ 57, 71, 1/4*E(3)-1/4*E(3)^2 ], [ 58, 32, 3/4 ], [ 58, 40, -9/16*E(3)+9/16*E(3)^2 ], [ 58, 44, -1/4*E(3)+1/4*E(3)^2 ], [ 58, 49, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 58, 51, -9/16 ], [ 58, 58, 1/4 ], [ 59, 32, 3/32*E(3)-3/32*E(3)^2 ], [ 59, 37, 1/8*E(3)-1/8*E(3)^2 ], [ 59, 40, 27/64 ], [ 59, 44, -3/32 ], [ 59, 45, 9/16 ], 
  [ 59, 48, -27/32 ], [ 59, 49, -9/32 ], [ 59, 51, 9/64*E(3)-9/64*E(3)^2 ], [ 59, 52, -9/8 ], [ 59, 53, -3/8 ], [ 59, 55, 3/4 ], [ 59, 57, -9/32*E(3)+9/32*E(3)^2 ], 
  [ 59, 58, -3/32*E(3)+3/32*E(3)^2 ], [ 59, 59, 1 ], [ 59, 61, -9/32*E(3)+9/32*E(3)^2 ], [ 59, 64, 1/4*E(3)-1/4*E(3)^2 ], [ 59, 66, -3/8*E(3)+3/8*E(3)^2 ], 
  [ 59, 71, 9/32 ], [ 60, 56, 3/4 ], [ 60, 60, 1 ], [ 60, 65, 1/4*E(3)-1/4*E(3)^2 ], [ 61, 40, -1/16*E(3)+1/16*E(3)^2 ], [ 61, 48, 1/4*E(3)-1/4*E(3)^2 ], [ 61, 51, 3/16 ], 
  [ 61, 57, -3/4 ], [ 61, 61, 1/4 ], [ 61, 71, 1/4*E(3)-1/4*E(3)^2 ], [ 62, 56, -3/16 ], [ 62, 62, 1/4 ], [ 62, 65, -3/16*E(3)+3/16*E(3)^2 ], [ 62, 68, 3/16*E(3)-3/16*E(3)^2 ], 
  [ 62, 72, 1/4*E(3)-1/4*E(3)^2 ], [ 62, 74, -1/4*E(3)+1/4*E(3)^2 ], [ 62, 78, -9/16 ], [ 62, 84, 3/4 ], [ 63, 63, 1/4 ], [ 63, 69, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 63, 76, 1/4*E(3)-1/4*E(3)^2 ], [ 63, 80, 3/4 ], [ 64, 32, 3/16 ], [ 64, 40, -9/32*E(3)+9/32*E(3)^2 ], [ 64, 44, -1/16*E(3)+1/16*E(3)^2 ], [ 64, 48, 9/16*E(3)-9/16*E(3)^2 ], 
  [ 64, 49, 3/16*E(3)-3/16*E(3)^2 ], [ 64, 51, -9/32 ], [ 64, 55, -1/2*E(3)+1/2*E(3)^2 ], [ 64, 57, 9/16 ], [ 64, 58, 3/16 ], [ 64, 61, -9/16 ], [ 64, 64, -1/2 ], 
  [ 64, 71, 3/16*E(3)-3/16*E(3)^2 ], [ 65, 56, -1/2*E(3)+1/2*E(3)^2 ], [ 65, 65, -1/2 ], [ 66, 40, 3/32*E(3)-3/32*E(3)^2 ], [ 66, 45, 1/8*E(3)-1/8*E(3)^2 ], 
  [ 66, 48, -3/8*E(3)+3/8*E(3)^2 ], [ 66, 51, -3/32 ], [ 66, 52, -1/2*E(3)+1/2*E(3)^2 ], [ 66, 57, 3/8 ], [ 66, 61, -3/8 ], 
  [ 66, 66, -1/2 ], [ 66, 71, -1/8*E(3)+1/8*E(3)^2 ], [ 67, 56, 9/32 ], [ 67, 60, 3/8 ], [ 67, 62, -3/8 ], [ 67, 65, 3/32*E(3)-3/32*E(3)^2 ], [ 67, 67, -1/2 ], 
  [ 67, 68, -9/32*E(3)+9/32*E(3)^2 ], [ 67, 72, -1/8*E(3)+1/8*E(3)^2 ], [ 67, 73, -3/8*E(3)+3/8*E(3)^2 ], [ 67, 74, 3/8*E(3)-3/8*E(3)^2 ], [ 67, 78, 9/32 ], 
  [ 67, 79, 1/2*E(3)-1/2*E(3)^2 ], [ 67, 84, -3/8 ], [ 68, 68, -1/2 ], [ 68, 78, -1/2*E(3)+1/2*E(3)^2 ], [ 69, 63, -1/4*E(3)+1/4*E(3)^2 ], [ 69, 69, 1/4 ], [ 69, 76, 3/4 ], 
  [ 69, 80, 1/4*E(3)-1/4*E(3)^2 ], [ 70, 63, 9/16 ], [ 70, 69, -9/16*E(3)+9/16*E(3)^2 ], [ 70, 70, -1/2 ], [ 70, 75, 1/2*E(3)-1/2*E(3)^2 ], [ 70, 76, 3/16*E(3)-3/16*E(3)^2 ], 
  [ 70, 80, 9/16 ], [ 71, 40, 3/16 ], [ 71, 48, -3/4 ], [ 71, 51, -1/16*E(3)+1/16*E(3)^2 ], [ 71, 57, 1/4*E(3)-1/4*E(3)^2 ], [ 71, 61, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 71, 71, 1/4 ], [ 72, 56, -3/16*E(3)+3/16*E(3)^2 ], [ 72, 62, 1/4*E(3)-1/4*E(3)^2 ], [ 72, 65, -3/16 ], [ 72, 68, -9/16 ], [ 72, 72, 1/4 ], [ 72, 74, 3/4 ], 
  [ 72, 78, 3/16*E(3)-3/16*E(3)^2 ], [ 72, 84, -1/4*E(3)+1/4*E(3)^2 ], [ 73, 68, 3/4 ], [ 73, 73, 1 ], [ 73, 78, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 74, 56, 1/16*E(3)-1/16*E(3)^2 ], [ 74, 62, -1/4*E(3)+1/4*E(3)^2 ], [ 74, 65, -3/16 ], [ 74, 68, -9/16 ], [ 74, 72, 3/4 ], [ 74, 74, 1/4 ], [ 74, 78, -9/16*E(3)+9/16*E(3)^2 ], 
  [ 74, 84, 1/4*E(3)-1/4*E(3)^2 ], [ 75, 63, -9/16*E(3)+9/16*E(3)^2 ], [ 75, 69, 9/16 ], [ 75, 70, 1/2*E(3)-1/2*E(3)^2 ], [ 75, 75, -1/2 ], [ 75, 76, 9/16 ], 
  [ 75, 80, 3/16*E(3)-3/16*E(3)^2 ], [ 76, 63, 1/4*E(3)-1/4*E(3)^2 ], [ 76, 69, 3/4 ], [ 76, 76, 1/4 ], [ 76, 80, -1/4*E(3)+1/4*E(3)^2 ], [ 77, 77, 1/4 ], 
  [ 77, 81, -1/4*E(3)+1/4*E(3)^2 ], [ 77, 83, 3/16*E(3)-3/16*E(3)^2 ], [ 77, 86, 9/16 ], [ 77, 88, -1/4*E(3)+1/4*E(3)^2 ], [ 77, 91, -3/4 ], 
  [ 78, 68, -1/2*E(3)+1/2*E(3)^2 ], [ 78, 78, -1/2 ], [ 79, 56, -3/32*E(3)+3/32*E(3)^2 ], [ 79, 60, -1/8*E(3)+1/8*E(3)^2 ], [ 79, 62, 3/8*E(3)-3/8*E(3)^2 ], [ 79, 65, 3/32 ], 
  [ 79, 67, 1/2*E(3)-1/2*E(3)^2 ], [ 79, 68, 27/32 ], [ 79, 72, -3/8 ], [ 79, 73, 9/8 ], [ 79, 74, -3/8 ], [ 79, 78, 9/32*E(3)-9/32*E(3)^2 ], [ 79, 79, -1/2 ], 
  [ 79, 84, -1/8*E(3)+1/8*E(3)^2 ], [ 80, 63, 3/4 ], [ 80, 69, 1/4*E(3)-1/4*E(3)^2 ], [ 80, 76, -1/4*E(3)+1/4*E(3)^2 ], [ 80, 80, 1/4 ], [ 81, 77, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 81, 81, 1/4 ], [ 81, 83, 9/16 ], [ 81, 86, 3/16*E(3)-3/16*E(3)^2 ], [ 81, 88, -3/4 ], [ 81, 91, -1/4*E(3)+1/4*E(3)^2 ], [ 82, 63, 1/16*E(3)-1/16*E(3)^2 ], [ 82, 69, 3/16 ], 
  [ 82, 76, 3/16 ], [ 82, 80, -3/16*E(3)+3/16*E(3)^2 ], [ 82, 82, -1/2 ], [ 82, 85, 1/2*E(3)-1/2*E(3)^2 ], [ 83, 83, -1/2 ], [ 83, 86, 1/2*E(3)-1/2*E(3)^2 ], 
  [ 84, 56, -3/16 ], [ 84, 62, 3/4 ], [ 84, 65, 1/16*E(3)-1/16*E(3)^2 ], [ 84, 68, -9/16*E(3)+9/16*E(3)^2 ], [ 84, 72, -1/4*E(3)+1/4*E(3)^2 ], [ 84, 74, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 84, 78, -9/16 ], [ 84, 84, 1/4 ], [ 85, 63, 3/16 ], [ 85, 69, 1/16*E(3)-1/16*E(3)^2 ], [ 85, 76, -3/16*E(3)+3/16*E(3)^2 ], [ 85, 80, 3/16 ], [ 85, 82, 1/2*E(3)-1/2*E(3)^2 ], 
  [ 85, 85, -1/2 ], [ 86, 83, 1/2*E(3)-1/2*E(3)^2 ], [ 86, 86, -1/2 ], [ 87, 83, -3/16 ], [ 87, 86, 3/16*E(3)-3/16*E(3)^2 ], [ 87, 87, 1/4 ], [ 87, 90, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 87, 96, 1/4*E(3)-1/4*E(3)^2 ], [ 87, 100, 3/4 ], [ 88, 77, -1/4*E(3)+1/4*E(3)^2 ], [ 88, 81, -3/4 ], [ 88, 83, -9/16 ], [ 88, 86, 9/16*E(3)-9/16*E(3)^2 ], 
  [ 88, 88, 1/4 ], [ 88, 91, -1/4*E(3)+1/4*E(3)^2 ], [ 89, 89, -1/2 ], [ 89, 93, -1/4*E(3)+1/4*E(3)^2 ], [ 89, 97, 1/2*E(3)-1/2*E(3)^2 ], [ 90, 83, 3/16*E(3)-3/16*E(3)^2 ], 
  [ 90, 86, -3/16 ], [ 90, 87, -1/4*E(3)+1/4*E(3)^2 ], [ 90, 90, 1/4 ], [ 90, 96, 3/4 ], [ 90, 100, 1/4*E(3)-1/4*E(3)^2 ], [ 91, 77, -3/4 ], [ 91, 81, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 91, 83, 9/16*E(3)-9/16*E(3)^2 ], [ 91, 86, -9/16 ], [ 91, 88, -1/4*E(3)+1/4*E(3)^2 ], [ 91, 91, 1/4 ], [ 92, 77, -1/16*E(3)+1/16*E(3)^2 ], [ 92, 81, -3/16 ], 
  [ 92, 83, -9/32 ], [ 92, 86, 9/32*E(3)-9/32*E(3)^2 ], [ 92, 87, 9/16 ], [ 92, 88, 3/16 ], [ 92, 90, -9/16*E(3)+9/16*E(3)^2 ], [ 92, 91, -3/16*E(3)+3/16*E(3)^2 ], [ 92, 92, -1/2 ], 
  [ 92, 95, 1/2*E(3)-1/2*E(3)^2 ], [ 92, 96, 3/16*E(3)-3/16*E(3)^2 ], [ 92, 100, 9/16 ], [ 93, 93, 1 ], [ 94, 94, 1/4 ], [ 94, 98, 1/8*E(3)-1/8*E(3)^2 ], 
  [ 94, 101, -1/4*E(3)+1/4*E(3)^2 ], [ 94, 103, 1/4*E(3)-1/4*E(3)^2 ], [ 94, 105, -3/8 ], [ 94, 107, 3/4 ], [ 95, 77, -3/16 ], [ 95, 81, -1/16*E(3)+1/16*E(3)^2 ], 
  [ 95, 83, 9/32*E(3)-9/32*E(3)^2 ], [ 95, 86, -9/32 ], [ 95, 87, -9/16*E(3)+9/16*E(3)^2 ], [ 95, 88, -3/16*E(3)+3/16*E(3)^2 ], [ 95, 90, 9/16 ], 
  [ 95, 91, 3/16 ], [ 95, 92, 1/2*E(3)-1/2*E(3)^2 ], [ 95, 95, -1/2 ], [ 95, 96, 9/16 ], [ 95, 100, 3/16*E(3)-3/16*E(3)^2 ], [ 96, 83, -1/16*E(3)+1/16*E(3)^2 ], [ 96, 86, -3/16 ], 
  [ 96, 87, 1/4*E(3)-1/4*E(3)^2 ], [ 96, 90, 3/4 ], [ 96, 96, 1/4 ], [ 96, 100, -1/4*E(3)+1/4*E(3)^2 ], [ 97, 89, 1/2*E(3)-1/2*E(3)^2 ], [ 97, 93, 3/4 ], 
  [ 97, 97, -1/2 ], [ 98, 98, -1/2 ], [ 98, 105, -1/2*E(3)+1/2*E(3)^2 ], [ 99, 94, 9/16 ], [ 99, 98, 9/32*E(3)-9/32*E(3)^2 ], [ 99, 99, -1/2 ], [ 99, 101, -9/16*E(3)+9/16*E(3)^2 ], 
  [ 99, 102, -1/4*E(3)+1/4*E(3)^2 ], [ 99, 103, 3/16*E(3)-3/16*E(3)^2 ], [ 99, 104, 1/2*E(3)-1/2*E(3)^2 ], [ 99, 105, -9/32 ], [ 99, 107, 9/16 ], [ 100, 83, -3/16 ], 
  [ 100, 86, -1/16*E(3)+1/16*E(3)^2 ], [ 100, 87, 3/4 ], [ 100, 90, 1/4*E(3)-1/4*E(3)^2 ], [ 100, 96, -1/4*E(3)+1/4*E(3)^2 ], [ 100, 100, 1/4 ], [ 101, 94, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 101, 98, -3/8 ], [ 101, 101, 1/4 ], [ 101, 103, 3/4 ], [ 101, 105, -3/8*E(3)+3/8*E(3)^2 ], [ 101, 107, 1/4*E(3)-1/4*E(3)^2 ], [ 102, 98, -9/8 ], [ 102, 102, 1 ], 
  [ 102, 105, -3/8*E(3)+3/8*E(3)^2 ], [ 103, 94, 1/4*E(3)-1/4*E(3)^2 ], [ 103, 98, -3/8 ], [ 103, 101, 3/4 ], [ 103, 103, 1/4 ], [ 103, 105, 1/8*E(3)-1/8*E(3)^2 ], 
  [ 103, 107, -1/4*E(3)+1/4*E(3)^2 ], [ 104, 94, -9/16*E(3)+9/16*E(3)^2 ], [ 104, 98, -27/32 ], [ 104, 99, 1/2*E(3)-1/2*E(3)^2 ], [ 104, 101, 9/16 ], 
  [ 104, 102, 3/4 ], [ 104, 103, 9/16 ], [ 104, 104, -1/2 ], [ 104, 105, -9/32*E(3)+9/32*E(3)^2 ], [ 104, 107, 3/16*E(3)-3/16*E(3)^2 ], [ 105, 98, -1/2*E(3)+1/2*E(3)^2 ], 
  [ 105, 105, -1/2 ], [ 106, 94, 1/16*E(3)-1/16*E(3)^2 ], [ 106, 98, -3/32 ], [ 106, 101, 3/16 ], [ 106, 103, 3/16 ], [ 106, 105, 3/32*E(3)-3/32*E(3)^2 ], [ 106, 106, -1/2 ], 
  [ 106, 107, -3/16*E(3)+3/16*E(3)^2 ], [ 106, 108, -1/4*E(3)+1/4*E(3)^2 ], [ 106, 109, 1/2*E(3)-1/2*E(3)^2 ], [ 107, 94, 3/4 ], [ 107, 98, -3/8*E(3)+3/8*E(3)^2 ], 
  [ 107, 101, 1/4*E(3)-1/4*E(3)^2 ], [ 107, 103, -1/4*E(3)+1/4*E(3)^2 ], [ 107, 105, -3/8 ], [ 107, 107, 1/4 ], [ 108, 98, -1/8*E(3)+1/8*E(3)^2 ], [ 108, 105, -3/8 ], [ 108, 108, 1 ], 
  [ 109, 94, 3/16 ], [ 109, 98, -3/32*E(3)+3/32*E(3)^2 ], [ 109, 101, 1/16*E(3)-1/16*E(3)^2 ], [ 109, 103, -3/16*E(3)+3/16*E(3)^2 ], [ 109, 105, -9/32 ], 
  [ 109, 106, 1/2*E(3)-1/2*E(3)^2 ], [ 109, 107, 3/16 ], [ 109, 108, 3/4 ], [ 109, 109, -1/2 ], [ 110, 110, 1/4 ], [ 110, 112, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 110, 114, 1/4*E(3)-1/4*E(3)^2 ], [ 110, 115, -3/4 ], [ 111, 110, 3/8 ], [ 111, 111, -1/2 ], [ 111, 112, 3/8*E(3)-3/8*E(3)^2 ], [ 111, 113, -1/2*E(3)+1/2*E(3)^2 ], 
  [ 111, 114, 1/8*E(3)-1/8*E(3)^2 ], [ 111, 115, -3/8 ], [ 112, 110, 1/4*E(3)-1/4*E(3)^2 ], [ 112, 112, 1/4 ], [ 112, 114, -3/4 ], [ 112, 115, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 113, 110, 3/8*E(3)-3/8*E(3)^2 ], [ 113, 111, -1/2*E(3)+1/2*E(3)^2 ], [ 113, 112, 3/8 ], [ 113, 113, -1/2 ], [ 113, 114, -3/8 ], [ 113, 115, 1/8*E(3)-1/8*E(3)^2 ], 
  [ 114, 110, 1/4*E(3)-1/4*E(3)^2 ], [ 114, 112, -3/4 ], [ 114, 114, 1/4 ], [ 114, 115, 1/4*E(3)-1/4*E(3)^2 ], [ 115, 110, -3/4 ], [ 115, 112, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 115, 114, 1/4*E(3)-1/4*E(3)^2 ], [ 115, 115, 1/4 ], [ 116, 116, 1 ], [ 117, 116, -3/8 ], [ 117, 117, -1/2 ], [ 117, 118, 3/8*E(3)-3/8*E(3)^2 ], [ 117, 119, 1/2*E(3)-1/2*E(3)^2 ], 
  [ 118, 118, 1 ], [ 119, 116, 1/8*E(3)-1/8*E(3)^2 ], [ 119, 117, 1/2*E(3)-1/2*E(3)^2 ], [ 119, 118, -9/8 ], [ 119, 119, -1/2 ], [ 120, 120, 1 ], [ 121, 3, -1/2*E(3)+1/2*E(3)^2 ], 
  [ 121, 121, -1/2 ], [ 122, 2, 3/4 ], [ 122, 122, 1/4 ], [ 122, 242, -1/2*E(3)+1/2*E(3)^2 ], [ 122, 244, 1/4*E(3)-1/4*E(3)^2 ], [ 123, 1, 1/2*E(3)-1/2*E(3)^2 ], 
  [ 123, 3, 3/16 ], [ 123, 9, 3/8 ], [ 123, 121, -3/16*E(3)+3/16*E(3)^2 ], [ 123, 123, -1/2 ], [ 123, 129, 3/8 ], [ 123, 241, 5/8*E(3)-5/8*E(3)^2 ], [ 123, 243, -1/8*E(3)+1/8*E(3)^2 ], 
  [ 123, 244, -1/8*E(3)+1/8*E(3)^2 ], [ 124, 124, 1/4 ], [ 124, 130, -1/4*E(3)+1/4*E(3)^2 ], [ 124, 132, 3/32 ], [ 124, 136, 1/4*E(3)-1/4*E(3)^2 ], [ 124, 138, -3/32*E(3)+3/32*E(3)^2 ], 
  [ 124, 143, 3/4 ], [ 124, 144, 3/32*E(3)-3/32*E(3)^2 ], [ 124, 148, 1/32*E(3)-1/32*E(3)^2 ], [ 124, 150, 9/32 ], [ 124, 154, 3/32 ], [ 124, 159, -3/32 ], 
  [ 124, 166, 3/32*E(3)-3/32*E(3)^2 ], [ 125, 125, -1/2 ], [ 125, 141, -1/2*E(3)+1/2*E(3)^2 ], [ 126, 6, 27/64 ], [ 126, 7, 1/2*E(3)-1/2*E(3)^2 ], 
  [ 126, 14, -9/16 ], [ 126, 125, 9/64*E(3)-9/64*E(3)^2 ], [ 126, 126, -1/2 ], [ 126, 127, -27/64*E(3)+27/64*E(3)^2 ], [ 126, 133, 3/8 ], [ 126, 134, -9/16 ], [ 126, 141, 27/64 ], 
  [ 126, 245, -3/16*E(3)+3/16*E(3)^2 ], [ 126, 246, -3/16*E(3)+3/16*E(3)^2 ], [ 126, 247, 15/16*E(3)-15/16*E(3)^2 ], [ 126, 248, -9/16*E(3)+9/16*E(3)^2 ], 
  [ 127, 6, -1/2*E(3)+1/2*E(3)^2 ], [ 127, 127, -1/2 ], [ 128, 128, -1/2 ], [ 128, 135, 9/8 ], [ 128, 142, 1/2*E(3)-1/2*E(3)^2 ], 
  [ 128, 149, -1/8*E(3)+1/8*E(3)^2 ], [ 129, 3, 3/8 ], [ 129, 9, 3/4 ], [ 129, 121, -1/8*E(3)+1/8*E(3)^2 ], [ 129, 129, 1/4 ], [ 129, 241, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 129, 243, 1/4*E(3)-1/4*E(3)^2 ], [ 129, 244, -1/4*E(3)+1/4*E(3)^2 ], [ 130, 124, -1/4*E(3)+1/4*E(3)^2 ], [ 130, 130, 1/4 ], [ 130, 132, -3/32*E(3)+3/32*E(3)^2 ], 
  [ 130, 136, 3/4 ], [ 130, 138, 3/32 ], [ 130, 143, 1/4*E(3)-1/4*E(3)^2 ], [ 130, 144, 9/32 ], [ 130, 148, 3/32 ], [ 130, 150, 3/32*E(3)-3/32*E(3)^2 ], 
  [ 130, 154, 1/32*E(3)-1/32*E(3)^2 ], [ 130, 159, 3/32*E(3)-3/32*E(3)^2 ], [ 130, 166, -3/32 ], [ 131, 124, 1/8*E(3)-1/8*E(3)^2 ], [ 131, 130, 3/8 ], [ 131, 131, -1/2 ], 
  [ 131, 132, 3/64*E(3)-3/64*E(3)^2 ], [ 131, 136, 3/8 ], [ 131, 137, 1/2*E(3)-1/2*E(3)^2 ], [ 131, 138, 9/64 ], [ 131, 139, -3/16 ], [ 131, 143, -3/8*E(3)+3/8*E(3)^2 ], 
  [ 131, 144, 9/64 ], [ 131, 145, 3/16*E(3)-3/16*E(3)^2 ], [ 131, 148, -3/64 ], [ 131, 150, -9/64*E(3)+9/64*E(3)^2 ], [ 131, 154, 3/64*E(3)-3/64*E(3)^2 ], 
  [ 131, 155, -1/16*E(3)+1/16*E(3)^2 ], [ 131, 159, 3/64*E(3)-3/64*E(3)^2 ], [ 131, 161, -3/16 ], [ 131, 166, 9/64 ], [ 132, 132, -1/8 ], [ 132, 138, 1/8*E(3)-1/8*E(3)^2 ], 
  [ 132, 144, -1/8*E(3)+1/8*E(3)^2 ], [ 132, 148, -1/8*E(3)+1/8*E(3)^2 ], [ 132, 150, -3/8 ], [ 132, 154, -3/8 ], [ 132, 159, 3/8 ], [ 132, 166, -3/8*E(3)+3/8*E(3)^2 ], 
  [ 133, 125, 3/8*E(3)-3/8*E(3)^2 ], [ 133, 133, 1 ], [ 133, 141, 9/8 ], [ 134, 6, -9/16 ], [ 134, 14, 3/4 ], [ 134, 125, -3/16*E(3)+3/16*E(3)^2 ], [ 134, 127, 3/16*E(3)-3/16*E(3)^2 ], 
  [ 134, 134, 1/4 ], [ 134, 141, -3/16 ], [ 134, 245, 1/4*E(3)-1/4*E(3)^2 ], [ 134, 246, -1/4*E(3)+1/4*E(3)^2 ], [ 134, 247, -1/4*E(3)+1/4*E(3)^2 ], [ 134, 248, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 135, 135, 1 ], [ 136, 124, 1/4*E(3)-1/4*E(3)^2 ], [ 136, 130, 3/4 ], [ 136, 132, 3/32*E(3)-3/32*E(3)^2 ], [ 136, 136, 1/4 ], [ 136, 138, 9/32 ], [ 136, 143, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 136, 144, 3/32 ], [ 136, 148, -3/32 ], [ 136, 150, -3/32*E(3)+3/32*E(3)^2 ], [ 136, 154, 3/32*E(3)-3/32*E(3)^2 ], [ 136, 159, 1/32*E(3)-1/32*E(3)^2 ], [ 136, 166, 3/32 ], 
  [ 137, 124, 3/8 ], [ 137, 130, 1/8*E(3)-1/8*E(3)^2 ], [ 137, 131, 1/2*E(3)-1/2*E(3)^2 ], [ 137, 132, 9/64 ], [ 137, 136, -3/8*E(3)+3/8*E(3)^2 ], [ 137, 137, -1/2 ], 
  [ 137, 138, 3/64*E(3)-3/64*E(3)^2 ], [ 137, 139, 3/16*E(3)-3/16*E(3)^2 ], [ 137, 143, 3/8 ], [ 137, 144, -9/64*E(3)+9/64*E(3)^2 ], [ 137, 145, -3/16 ], 
  [ 137, 148, 3/64*E(3)-3/64*E(3)^2 ], [ 137, 150, 9/64 ], [ 137, 154, -3/64 ], [ 137, 155, -3/16 ], [ 137, 159, 9/64 ], [ 137, 161, -1/16*E(3)+1/16*E(3)^2 ], 
  [ 137, 166, 3/64*E(3)-3/64*E(3)^2 ], [ 138, 132, 1/8*E(3)-1/8*E(3)^2 ], [ 138, 138, -1/8 ], [ 138, 144, -3/8 ], [ 138, 148, -3/8 ], [ 138, 150, -1/8*E(3)+1/8*E(3)^2 ], 
  [ 138, 154, -1/8*E(3)+1/8*E(3)^2 ], [ 138, 159, -3/8*E(3)+3/8*E(3)^2 ], [ 138, 166, 3/8 ], [ 139, 132, -1/16*E(3)+1/16*E(3)^2 ], [ 139, 138, -3/16 ], 
  [ 139, 139, 1/4 ], [ 139, 144, -3/16 ], [ 139, 145, -1/4*E(3)+1/4*E(3)^2 ], [ 139, 148, 3/16 ], [ 139, 150, 3/16*E(3)-3/16*E(3)^2 ], [ 139, 154, -3/16*E(3)+3/16*E(3)^2 ], 
  [ 139, 155, 1/4*E(3)-1/4*E(3)^2 ], [ 139, 159, -3/16*E(3)+3/16*E(3)^2 ], [ 139, 161, 3/4 ], [ 139, 166, -9/16 ], [ 140, 132, 3/32*E(3)-3/32*E(3)^2 ], [ 140, 138, 9/32 ], 
  [ 140, 140, 1/4 ], [ 140, 144, -9/32 ], [ 140, 146, -1/4*E(3)+1/4*E(3)^2 ], [ 140, 148, 9/32 ], [ 140, 150, 9/32*E(3)-9/32*E(3)^2 ], [ 140, 151, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 140, 154, -9/32*E(3)+9/32*E(3)^2 ], [ 140, 158, 3/4 ], [ 140, 159, 9/32*E(3)-9/32*E(3)^2 ], [ 140, 166, 27/32 ], [ 141, 125, -1/2*E(3)+1/2*E(3)^2 ], [ 141, 141, -1/2 ], 
  [ 142, 128, 1/2*E(3)-1/2*E(3)^2 ], [ 142, 135, -3/8*E(3)+3/8*E(3)^2 ], [ 142, 142, -1/2 ], [ 142, 149, 3/8 ], [ 143, 124, 3/4 ], [ 143, 130, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 143, 132, 9/32 ], [ 143, 136, -1/4*E(3)+1/4*E(3)^2 ], [ 143, 138, 3/32*E(3)-3/32*E(3)^2 ], [ 143, 143, 1/4 ], [ 143, 144, -3/32*E(3)+3/32*E(3)^2 ], 
  [ 143, 148, 3/32*E(3)-3/32*E(3)^2 ], [ 143, 150, 3/32 ], [ 143, 154, -3/32 ], [ 143, 159, 3/32 ], [ 143, 166, 1/32*E(3)-1/32*E(3)^2 ], [ 144, 132, -1/8*E(3)+1/8*E(3)^2 ], 
  [ 144, 138, -3/8 ], [ 144, 144, -1/8 ], [ 144, 148, 3/8 ], [ 144, 150, 1/8*E(3)-1/8*E(3)^2 ], [ 144, 154, -3/8*E(3)+3/8*E(3)^2 ], [ 144, 159, -1/8*E(3)+1/8*E(3)^2 ], 
  [ 144, 166, -3/8 ], [ 145, 132, -3/16 ], [ 145, 138, -1/16*E(3)+1/16*E(3)^2 ], [ 145, 139, -1/4*E(3)+1/4*E(3)^2 ], [ 145, 144, 3/16*E(3)-3/16*E(3)^2 ], [ 145, 145, 1/4 ], 
  [ 145, 148, -3/16*E(3)+3/16*E(3)^2 ], [ 145, 150, -3/16 ], [ 145, 154, 3/16 ], [ 145, 155, 3/4 ], [ 145, 159, -9/16 ], [ 145, 161, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 145, 166, -3/16*E(3)+3/16*E(3)^2 ], [ 146, 132, 9/32 ], [ 146, 138, 3/32*E(3)-3/32*E(3)^2 ], [ 146, 140, -1/4*E(3)+1/4*E(3)^2 ], [ 146, 144, 9/32*E(3)-9/32*E(3)^2 ], 
  [ 146, 146, 1/4 ], [ 146, 148, -9/32*E(3)+9/32*E(3)^2 ], [ 146, 150, -9/32 ], [ 146, 151, 3/4 ], [ 146, 154, 9/32 ], [ 146, 158, 1/4*E(3)-1/4*E(3)^2 ], [ 146, 159, 27/32 ], 
  [ 146, 166, 9/32*E(3)-9/32*E(3)^2 ], [ 147, 132, -9/64 ], [ 147, 138, 9/64*E(3)-9/64*E(3)^2 ], [ 147, 139, -3/16*E(3)+3/16*E(3)^2 ], [ 147, 140, 1/8*E(3)-1/8*E(3)^2 ], 
  [ 147, 144, 9/64*E(3)-9/64*E(3)^2 ], [ 147, 145, -9/16 ], [ 147, 146, 3/8 ], [ 147, 147, -1/2 ], [ 147, 148, 9/64*E(3)-9/64*E(3)^2 ], [ 147, 150, 27/64 ], [ 147, 151, 3/8 ], 
  [ 147, 153, 1/2*E(3)-1/2*E(3)^2 ], [ 147, 154, 27/64 ], [ 147, 155, -9/16 ], [ 147, 158, -3/8*E(3)+3/8*E(3)^2 ], [ 147, 159, 27/64 ], [ 147, 161, 9/16*E(3)-9/16*E(3)^2 ], 
  [ 147, 166, -27/64*E(3)+27/64*E(3)^2 ], [ 148, 132, -1/8*E(3)+1/8*E(3)^2 ], [ 148, 138, -3/8 ], [ 148, 144, 3/8 ], [ 148, 148, -1/8 ], [ 148, 150, -3/8*E(3)+3/8*E(3)^2 ], 
  [ 148, 154, 1/8*E(3)-1/8*E(3)^2 ], [ 148, 159, -1/8*E(3)+1/8*E(3)^2 ], [ 148, 166, -3/8 ], [ 149, 149, 1 ], [ 150, 132, -3/8 ], [ 150, 138, -1/8*E(3)+1/8*E(3)^2 ], 
  [ 150, 144, 1/8*E(3)-1/8*E(3)^2 ], [ 150, 148, -3/8*E(3)+3/8*E(3)^2 ], [ 150, 150, -1/8 ], [ 150, 154, 3/8 ], [ 150, 159, -3/8 ], [ 150, 166, -1/8*E(3)+1/8*E(3)^2 ], 
  [ 151, 132, -9/32 ], [ 151, 138, 9/32*E(3)-9/32*E(3)^2 ], [ 151, 140, 1/4*E(3)-1/4*E(3)^2 ], [ 151, 144, 3/32*E(3)-3/32*E(3)^2 ], [ 151, 146, 3/4 ], 
  [ 151, 148, 9/32*E(3)-9/32*E(3)^2 ], [ 151, 150, 9/32 ], [ 151, 151, 1/4 ], [ 151, 154, 27/32 ], [ 151, 158, -1/4*E(3)+1/4*E(3)^2 ], [ 151, 159, 9/32 ], 
  [ 151, 166, -9/32*E(3)+9/32*E(3)^2 ], [ 152, 152, 1/4 ], [ 152, 157, -3/8 ], [ 152, 164, -1/4*E(3)+1/4*E(3)^2 ], [ 152, 169, 1/4*E(3)-1/4*E(3)^2 ], [ 152, 173, -3/8*E(3)+3/8*E(3)^2 ], 
  [ 152, 175, 1/16*E(3)-1/16*E(3)^2 ], [ 152, 178, 3/4 ], [ 152, 179, -3/32*E(3)+3/32*E(3)^2 ], [ 152, 184, 3/16 ], [ 153, 132, 9/64*E(3)-9/64*E(3)^2 ], [ 153, 138, -9/64 ], 
  [ 153, 139, -9/16 ], [ 153, 140, 3/8 ], [ 153, 144, 27/64 ], [ 153, 145, -3/16*E(3)+3/16*E(3)^2 ], [ 153, 146, 1/8*E(3)-1/8*E(3)^2 ], [ 153, 147, 1/2*E(3)-1/2*E(3)^2 ], 
  [ 153, 148, 27/64 ], [ 153, 150, 9/64*E(3)-9/64*E(3)^2 ], [ 153, 151, -3/8*E(3)+3/8*E(3)^2 ], [ 153, 153, -1/2 ], [ 153, 154, 9/64*E(3)-9/64*E(3)^2 ], 
  [ 153, 155, 9/16*E(3)-9/16*E(3)^2 ], [ 153, 158, 3/8 ], [ 153, 159, -27/64*E(3)+27/64*E(3)^2 ], [ 153, 161, -9/16 ], [ 153, 166, 27/64 ], [ 154, 132, -3/8 ], 
  [ 154, 138, -1/8*E(3)+1/8*E(3)^2 ], [ 154, 144, -3/8*E(3)+3/8*E(3)^2 ], [ 154, 148, 1/8*E(3)-1/8*E(3)^2 ], [ 154, 150, 3/8 ], [ 154, 154, -1/8 ], [ 154, 159, -3/8 ], 
  [ 154, 166, -1/8*E(3)+1/8*E(3)^2 ], [ 155, 132, 3/16 ], [ 155, 138, -3/16*E(3)+3/16*E(3)^2 ], [ 155, 139, 1/4*E(3)-1/4*E(3)^2 ], [ 155, 144, -3/16*E(3)+3/16*E(3)^2 ], 
  [ 155, 145, 3/4 ], [ 155, 148, -1/16*E(3)+1/16*E(3)^2 ], [ 155, 150, -9/16 ], [ 155, 154, -3/16 ], [ 155, 155, 1/4 ], [ 155, 159, -3/16 ], [ 155, 161, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 155, 166, 3/16*E(3)-3/16*E(3)^2 ], [ 156, 156, 1/4 ], [ 156, 162, -1/4*E(3)+1/4*E(3)^2 ], [ 156, 167, 1/4*E(3)-1/4*E(3)^2 ], [ 156, 174, 3/4 ], [ 157, 157, -1/2 ], 
  [ 157, 173, -1/2*E(3)+1/2*E(3)^2 ], [ 157, 179, -1/8*E(3)+1/8*E(3)^2 ], [ 158, 132, 9/32*E(3)-9/32*E(3)^2 ], [ 158, 138, -9/32 ], [ 158, 140, 3/4 ], [ 158, 144, 9/32 ], 
  [ 158, 146, 1/4*E(3)-1/4*E(3)^2 ], [ 158, 148, 27/32 ], [ 158, 150, 3/32*E(3)-3/32*E(3)^2 ], [ 158, 151, -1/4*E(3)+1/4*E(3)^2 ], [ 158, 154, 9/32*E(3)-9/32*E(3)^2 ], 
  [ 158, 158, 1/4 ], [ 158, 159, -9/32*E(3)+9/32*E(3)^2 ], [ 158, 166, 9/32 ], [ 159, 132, 3/8 ], [ 159, 138, -3/8*E(3)+3/8*E(3)^2 ], [ 159, 144, -1/8*E(3)+1/8*E(3)^2 ], 
  [ 159, 148, -1/8*E(3)+1/8*E(3)^2 ], [ 159, 150, -3/8 ], [ 159, 154, -3/8 ], [ 159, 159, -1/8 ], [ 159, 166, 1/8*E(3)-1/8*E(3)^2 ], [ 160, 152, -3/16*E(3)+3/16*E(3)^2 ], 
  [ 160, 157, 9/32*E(3)-9/32*E(3)^2 ], [ 160, 160, -1/2 ], [ 160, 164, -9/16 ], [ 160, 165, 3/4 ], [ 160, 168, -3/16 ], [ 160, 169, -9/16 ], [ 160, 171, 1/2*E(3)-1/2*E(3)^2 ], 
  [ 160, 172, 9/32 ], [ 160, 173, 27/32 ], [ 160, 175, -9/32 ], [ 160, 177, 3/16*E(3)-3/16*E(3)^2 ], [ 160, 178, 9/16*E(3)-9/16*E(3)^2 ], [ 160, 179, 27/64 ], 
  [ 160, 181, 1/16*E(3)-1/16*E(3)^2 ], [ 160, 184, 9/32*E(3)-9/32*E(3)^2 ], [ 160, 186, -3/32*E(3)+3/32*E(3)^2 ], [ 160, 191, 3/16 ], 
  [ 161, 132, -3/16*E(3)+3/16*E(3)^2 ], [ 161, 138, 3/16 ], [ 161, 139, 3/4 ], [ 161, 144, -9/16 ], [ 161, 145, 1/4*E(3)-1/4*E(3)^2 ], [ 161, 148, -3/16 ], 
  [ 161, 150, -3/16*E(3)+3/16*E(3)^2 ], [ 161, 154, -1/16*E(3)+1/16*E(3)^2 ], [ 161, 155, -1/4*E(3)+1/4*E(3)^2 ], [ 161, 159, 3/16*E(3)-3/16*E(3)^2 ], [ 161, 161, 1/4 ], 
  [ 161, 166, -3/16 ], [ 162, 156, -1/4*E(3)+1/4*E(3)^2 ], [ 162, 162, 1/4 ], [ 162, 167, 3/4 ], [ 162, 174, 1/4*E(3)-1/4*E(3)^2 ], [ 163, 156, 1/8*E(3)-1/8*E(3)^2 ], 
  [ 163, 162, 3/8 ], [ 163, 163, -1/2 ], [ 163, 167, 3/8 ], [ 163, 170, 1/2*E(3)-1/2*E(3)^2 ], [ 163, 174, -3/8*E(3)+3/8*E(3)^2 ], [ 164, 152, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 164, 157, 1/8*E(3)-1/8*E(3)^2 ], [ 164, 164, 1/4 ], [ 164, 169, 3/4 ], [ 164, 173, -3/8 ], [ 164, 175, 3/16 ], [ 164, 178, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 164, 179, -3/32 ], [ 164, 184, 1/16*E(3)-1/16*E(3)^2 ], [ 165, 157, 3/8*E(3)-3/8*E(3)^2 ], [ 165, 165, 1 ], [ 165, 172, 3/8 ], [ 165, 173, 9/8 ], [ 165, 179, 9/16 ], 
  [ 165, 186, -1/8*E(3)+1/8*E(3)^2 ], [ 166, 132, -3/8*E(3)+3/8*E(3)^2 ], [ 166, 138, 3/8 ], [ 166, 144, -3/8 ], [ 166, 148, -3/8 ], [ 166, 150, -1/8*E(3)+1/8*E(3)^2 ], 
  [ 166, 154, -1/8*E(3)+1/8*E(3)^2 ], [ 166, 159, 1/8*E(3)-1/8*E(3)^2 ], [ 166, 166, -1/8 ], [ 167, 156, 1/4*E(3)-1/4*E(3)^2 ], [ 167, 162, 3/4 ], [ 167, 167, 1/4 ], 
  [ 167, 174, -1/4*E(3)+1/4*E(3)^2 ], [ 168, 168, 1/4 ], [ 168, 172, -3/8 ], [ 168, 175, 9/16 ], [ 168, 177, -1/4*E(3)+1/4*E(3)^2 ], [ 168, 179, -27/32 ], 
  [ 168, 181, -1/4*E(3)+1/4*E(3)^2 ], [ 168, 184, -9/16*E(3)+9/16*E(3)^2 ], [ 168, 186, 3/8*E(3)-3/8*E(3)^2 ], [ 168, 191, -3/4 ], [ 169, 152, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 169, 157, -3/8*E(3)+3/8*E(3)^2 ], [ 169, 164, 3/4 ], [ 169, 169, 1/4 ], [ 169, 173, -3/8 ], [ 169, 175, 3/16 ], [ 169, 178, -1/4*E(3)+1/4*E(3)^2 ], [ 169, 179, -9/32 ], 
  [ 169, 184, -3/16*E(3)+3/16*E(3)^2 ], [ 170, 156, 3/8 ], [ 170, 162, 1/8*E(3)-1/8*E(3)^2 ], [ 170, 163, 1/2*E(3)-1/2*E(3)^2 ], [ 170, 167, -3/8*E(3)+3/8*E(3)^2 ], 
  [ 170, 170, -1/2 ], [ 170, 174, 3/8 ], [ 171, 152, -9/16 ], [ 171, 157, 9/32 ], [ 171, 160, 1/2*E(3)-1/2*E(3)^2 ], [ 171, 164, -3/16*E(3)+3/16*E(3)^2 ], 
  [ 171, 165, -1/4*E(3)+1/4*E(3)^2 ], [ 171, 168, 3/16*E(3)-3/16*E(3)^2 ], [ 171, 169, 9/16*E(3)-9/16*E(3)^2 ], [ 171, 171, -1/2 ], [ 171, 172, -3/32*E(3)+3/32*E(3)^2 ], 
  [ 171, 173, -9/32*E(3)+9/32*E(3)^2 ], [ 171, 175, 9/32*E(3)-9/32*E(3)^2 ], [ 171, 177, -3/16 ], [ 171, 178, -9/16 ], [ 171, 179, -9/64*E(3)+9/64*E(3)^2 ], [ 171, 181, 3/16 ], 
  [ 171, 184, -9/32 ], [ 171, 186, -3/32 ], [ 171, 191, 1/16*E(3)-1/16*E(3)^2 ], [ 172, 172, -1/2 ], [ 172, 179, -9/8 ], [ 172, 186, 1/2*E(3)-1/2*E(3)^2 ], 
  [ 173, 157, -1/2*E(3)+1/2*E(3)^2 ], [ 173, 173, -1/2 ], [ 173, 179, -3/8 ], [ 174, 156, 3/4 ], [ 174, 162, 1/4*E(3)-1/4*E(3)^2 ], [ 174, 167, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 174, 174, 1/4 ], [ 175, 175, -1/2 ], [ 175, 179, 3/4 ], [ 175, 184, 1/2*E(3)-1/2*E(3)^2 ], [ 176, 176, -1/2 ], [ 176, 180, 3/4 ], [ 176, 182, -3/16 ], 
  [ 176, 185, 1/2*E(3)-1/2*E(3)^2 ], [ 176, 187, 9/32 ], [ 176, 192, 3/16*E(3)-3/16*E(3)^2 ], [ 176, 194, -1/16*E(3)+1/16*E(3)^2 ], [ 176, 199, 3/32*E(3)-3/32*E(3)^2 ], 
  [ 176, 204, -3/16 ], [ 177, 168, -1/4*E(3)+1/4*E(3)^2 ], [ 177, 172, 1/8*E(3)-1/8*E(3)^2 ], [ 177, 175, -9/16*E(3)+9/16*E(3)^2 ], [ 177, 177, 1/4 ], [ 177, 179, 9/32*E(3)-9/32*E(3)^2 ]
    , [ 177, 181, -3/4 ], [ 177, 184, 9/16 ], [ 177, 186, 3/8 ], [ 177, 191, -1/4*E(3)+1/4*E(3)^2 ], [ 178, 152, 3/4 ], [ 178, 157, -3/8 ], [ 178, 164, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 178, 169, -1/4*E(3)+1/4*E(3)^2 ], [ 178, 173, 1/8*E(3)-1/8*E(3)^2 ], [ 178, 175, -3/16*E(3)+3/16*E(3)^2 ], [ 178, 178, 1/4 ], [ 178, 179, 3/32*E(3)-3/32*E(3)^2 ], 
  [ 178, 184, 3/16 ], [ 179, 179, 1 ], [ 180, 180, 1 ], [ 180, 187, 3/8 ], [ 180, 199, 1/8*E(3)-1/8*E(3)^2 ], [ 181, 168, -1/4*E(3)+1/4*E(3)^2 ], [ 181, 172, 3/8*E(3)-3/8*E(3)^2 ], 
  [ 181, 175, -3/16*E(3)+3/16*E(3)^2 ], [ 181, 177, -3/4 ], [ 181, 179, 9/32*E(3)-9/32*E(3)^2 ], [ 181, 181, 1/4 ], [ 181, 184, -9/16 ], [ 181, 186, -3/8 ], 
  [ 181, 191, -1/4*E(3)+1/4*E(3)^2 ], [ 182, 182, 1/4 ], [ 182, 187, -3/8 ], [ 182, 192, -1/4*E(3)+1/4*E(3)^2 ], [ 182, 194, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 182, 199, -3/8*E(3)+3/8*E(3)^2 ], [ 182, 204, 3/4 ], [ 183, 183, 1/4 ], [ 183, 189, 1/4*E(3)-1/4*E(3)^2 ], [ 183, 190, 9/16 ], [ 183, 195, 9/16*E(3)-9/16*E(3)^2 ], 
  [ 183, 196, -1/4*E(3)+1/4*E(3)^2 ], [ 183, 200, 3/4 ], [ 183, 202, -1/16*E(3)+1/16*E(3)^2 ], [ 183, 205, 3/16 ], [ 184, 175, 1/2*E(3)-1/2*E(3)^2 ], [ 184, 179, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 184, 184, -1/2 ], [ 185, 176, 1/2*E(3)-1/2*E(3)^2 ], [ 185, 180, -1/4*E(3)+1/4*E(3)^2 ], [ 185, 182, 3/16*E(3)-3/16*E(3)^2 ], [ 185, 185, -1/2 ], [ 185, 187, -3/32*E(3)+3/32*E(3)^2 ]
    , [ 185, 192, -3/16 ], [ 185, 194, -3/16 ], [ 185, 199, 3/32 ], [ 185, 204, -1/16*E(3)+1/16*E(3)^2 ], [ 186, 172, 1/2*E(3)-1/2*E(3)^2 ], [ 186, 179, 3/8*E(3)-3/8*E(3)^2 ], 
  [ 186, 186, -1/2 ], [ 187, 187, -1/2 ], [ 187, 199, -1/2*E(3)+1/2*E(3)^2 ], [ 188, 182, -3/16*E(3)+3/16*E(3)^2 ], [ 188, 187, 9/32*E(3)-9/32*E(3)^2 ], [ 188, 188, -1/2 ], 
  [ 188, 192, -9/16 ], [ 188, 193, 3/4 ], [ 188, 194, -9/16 ], [ 188, 198, 1/2*E(3)-1/2*E(3)^2 ], [ 188, 199, 27/32 ], [ 188, 204, 9/16*E(3)-9/16*E(3)^2 ], 
  [ 189, 183, 1/4*E(3)-1/4*E(3)^2 ], [ 189, 189, 1/4 ], [ 189, 190, 9/16*E(3)-9/16*E(3)^2 ], [ 189, 195, 9/16 ], [ 189, 196, 3/4 ], [ 189, 200, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 189, 202, 3/16 ], [ 189, 205, -1/16*E(3)+1/16*E(3)^2 ], [ 190, 190, -1/2 ], [ 190, 195, -1/2*E(3)+1/2*E(3)^2 ], [ 191, 168, -3/4 ], [ 191, 172, 3/8 ], [ 191, 175, -9/16 ], 
  [ 191, 177, -1/4*E(3)+1/4*E(3)^2 ], [ 191, 179, 9/32 ], [ 191, 181, -1/4*E(3)+1/4*E(3)^2 ], [ 191, 184, -3/16*E(3)+3/16*E(3)^2 ], [ 191, 186, 1/8*E(3)-1/8*E(3)^2 ], 
  [ 191, 191, 1/4 ], [ 192, 182, -1/4*E(3)+1/4*E(3)^2 ], [ 192, 187, 1/8*E(3)-1/8*E(3)^2 ], [ 192, 192, 1/4 ], [ 192, 194, 3/4 ], [ 192, 199, -3/8 ], [ 192, 204, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 193, 187, 3/8*E(3)-3/8*E(3)^2 ], [ 193, 193, 1 ], [ 193, 199, 9/8 ], [ 194, 182, 1/4*E(3)-1/4*E(3)^2 ], [ 194, 187, -3/8*E(3)+3/8*E(3)^2 ], [ 194, 192, 3/4 ], 
  [ 194, 194, 1/4 ], [ 194, 199, -3/8 ], [ 194, 204, -1/4*E(3)+1/4*E(3)^2 ], [ 195, 190, -1/2*E(3)+1/2*E(3)^2 ], [ 195, 195, -1/2 ], [ 196, 183, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 196, 189, 3/4 ], [ 196, 190, -3/16*E(3)+3/16*E(3)^2 ], [ 196, 195, 9/16 ], [ 196, 196, 1/4 ], [ 196, 200, 1/4*E(3)-1/4*E(3)^2 ], [ 196, 202, 3/16 ], 
  [ 196, 205, 3/16*E(3)-3/16*E(3)^2 ], [ 197, 197, 1/4 ], [ 197, 201, 1/4*E(3)-1/4*E(3)^2 ], [ 197, 208, 1/4*E(3)-1/4*E(3)^2 ], [ 197, 211, -3/4 ], [ 197, 212, 1/16*E(3)-1/16*E(3)^2 ], 
  [ 197, 215, -3/16 ], [ 198, 182, -9/16 ], [ 198, 187, 9/32 ], [ 198, 188, 1/2*E(3)-1/2*E(3)^2 ], [ 198, 192, -3/16*E(3)+3/16*E(3)^2 ], [ 198, 193, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 198, 194, 9/16*E(3)-9/16*E(3)^2 ], [ 198, 198, -1/2 ], [ 198, 199, -9/32*E(3)+9/32*E(3)^2 ], [ 198, 204, -9/16 ], [ 199, 187, -1/2*E(3)+1/2*E(3)^2 ], [ 199, 199, -1/2 ], 
  [ 200, 183, 3/4 ], [ 200, 189, -1/4*E(3)+1/4*E(3)^2 ], [ 200, 190, 9/16 ], [ 200, 195, -3/16*E(3)+3/16*E(3)^2 ], [ 200, 196, 1/4*E(3)-1/4*E(3)^2 ], [ 200, 200, 1/4 ], 
  [ 200, 202, 3/16*E(3)-3/16*E(3)^2 ], [ 200, 205, 3/16 ], [ 201, 197, 1/4*E(3)-1/4*E(3)^2 ], [ 201, 201, 1/4 ], [ 201, 208, -3/4 ], [ 201, 211, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 201, 212, -3/16 ], [ 201, 215, 1/16*E(3)-1/16*E(3)^2 ], [ 202, 202, -1/2 ], [ 202, 205, -1/2*E(3)+1/2*E(3)^2 ], [ 203, 197, -3/16*E(3)+3/16*E(3)^2 ], 
  [ 203, 201, 9/16 ], [ 203, 203, -1/2 ], [ 203, 206, -1/2*E(3)+1/2*E(3)^2 ], [ 203, 207, -3/16 ], [ 203, 208, -9/16 ], [ 203, 210, -3/16*E(3)+3/16*E(3)^2 ], 
  [ 203, 211, -9/16*E(3)+9/16*E(3)^2 ], [ 203, 212, -9/32 ], [ 203, 215, -9/32*E(3)+9/32*E(3)^2 ], [ 203, 216, 1/16*E(3)-1/16*E(3)^2 ], [ 203, 220, -3/16 ], [ 204, 182, 3/4 ], 
  [ 204, 187, -3/8 ], [ 204, 192, 1/4*E(3)-1/4*E(3)^2 ], [ 204, 194, -1/4*E(3)+1/4*E(3)^2 ], [ 204, 199, 1/8*E(3)-1/8*E(3)^2 ], [ 204, 204, 1/4 ], [ 205, 202, -1/2*E(3)+1/2*E(3)^2 ], 
  [ 205, 205, -1/2 ], [ 206, 197, 9/16 ], [ 206, 201, -3/16*E(3)+3/16*E(3)^2 ], [ 206, 203, -1/2*E(3)+1/2*E(3)^2 ], [ 206, 206, -1/2 ], [ 206, 207, -3/16*E(3)+3/16*E(3)^2 ], 
  [ 206, 208, -9/16*E(3)+9/16*E(3)^2 ], [ 206, 210, -3/16 ], [ 206, 211, -9/16 ], [ 206, 212, -9/32*E(3)+9/32*E(3)^2 ], [ 206, 215, -9/32 ], [ 206, 216, -3/16 ], 
  [ 206, 220, 1/16*E(3)-1/16*E(3)^2 ], [ 207, 207, 1/4 ], [ 207, 210, 1/4*E(3)-1/4*E(3)^2 ], [ 207, 212, 9/16 ], [ 207, 215, 9/16*E(3)-9/16*E(3)^2 ], 
  [ 207, 216, -1/4*E(3)+1/4*E(3)^2 ], [ 207, 220, 3/4 ], [ 208, 197, 1/4*E(3)-1/4*E(3)^2 ], [ 208, 201, -3/4 ], [ 208, 208, 1/4 ], [ 208, 211, 1/4*E(3)-1/4*E(3)^2 ], [ 208, 212, 3/16 ], 
  [ 208, 215, 3/16*E(3)-3/16*E(3)^2 ], [ 209, 209, -1/2 ], [ 209, 217, -1/2*E(3)+1/2*E(3)^2 ], [ 210, 207, 1/4*E(3)-1/4*E(3)^2 ], [ 210, 210, 1/4 ], [ 210, 212, 9/16*E(3)-9/16*E(3)^2 ], 
  [ 210, 215, 9/16 ], [ 210, 216, 3/4 ], [ 210, 220, -1/4*E(3)+1/4*E(3)^2 ], [ 211, 197, -3/4 ], [ 211, 201, 1/4*E(3)-1/4*E(3)^2 ], [ 211, 208, 1/4*E(3)-1/4*E(3)^2 ], 
  [ 211, 211, 1/4 ], [ 211, 212, 3/16*E(3)-3/16*E(3)^2 ], [ 211, 215, 3/16 ], [ 212, 212, -1/2 ], [ 212, 215, -1/2*E(3)+1/2*E(3)^2 ], [ 213, 209, 1/4*E(3)-1/4*E(3)^2 ], [ 213, 213, 1 ], 
  [ 213, 217, 3/4 ], [ 214, 214, 1/4 ], [ 214, 219, 9/16 ], [ 214, 221, 1/4*E(3)-1/4*E(3)^2 ], [ 214, 223, -1/4*E(3)+1/4*E(3)^2 ], [ 214, 224, 9/16*E(3)-9/16*E(3)^2 ], 
  [ 214, 226, -1/16*E(3)+1/16*E(3)^2 ], [ 214, 227, 3/4 ], [ 214, 229, 3/16 ], [ 215, 212, -1/2*E(3)+1/2*E(3)^2 ], [ 215, 215, -1/2 ], [ 216, 207, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 216, 210, 3/4 ], [ 216, 212, -3/16*E(3)+3/16*E(3)^2 ], [ 216, 215, 9/16 ], [ 216, 216, 1/4 ], [ 216, 220, 1/4*E(3)-1/4*E(3)^2 ], [ 217, 209, -1/2*E(3)+1/2*E(3)^2 ], 
  [ 217, 217, -1/2 ], [ 218, 214, -1/8*E(3)+1/8*E(3)^2 ], [ 218, 218, -1/2 ], [ 218, 219, -9/32*E(3)+9/32*E(3)^2 ], [ 218, 221, -3/8 ], [ 218, 222, -9/8 ], [ 218, 223, -3/8 ], 
  [ 218, 224, -27/32 ], [ 218, 225, 1/2*E(3)-1/2*E(3)^2 ], [ 218, 226, -3/32 ], [ 218, 227, 3/8*E(3)-3/8*E(3)^2 ], [ 218, 228, 1/8*E(3)-1/8*E(3)^2 ], [ 218, 229, 3/32*E(3)-3/32*E(3)^2 ],
  [ 219, 219, -1/2 ], [ 219, 224, -1/2*E(3)+1/2*E(3)^2 ], [ 220, 207, 3/4 ], [ 220, 210, -1/4*E(3)+1/4*E(3)^2 ], [ 220, 212, 9/16 ], [ 220, 215, -3/16*E(3)+3/16*E(3)^2 ], 
  [ 220, 216, 1/4*E(3)-1/4*E(3)^2 ], [ 220, 220, 1/4 ], [ 221, 214, 1/4*E(3)-1/4*E(3)^2 ], [ 221, 219, 9/16*E(3)-9/16*E(3)^2 ], [ 221, 221, 1/4 ], [ 221, 223, 3/4 ], [ 221, 224, 9/16 ], 
  [ 221, 226, 3/16 ], [ 221, 227, -1/4*E(3)+1/4*E(3)^2 ], [ 221, 229, -1/16*E(3)+1/16*E(3)^2 ], [ 222, 219, 1/4*E(3)-1/4*E(3)^2 ], [ 222, 222, 1 ], [ 222, 224, 3/4 ], 
  [ 223, 214, -1/4*E(3)+1/4*E(3)^2 ], [ 223, 219, -3/16*E(3)+3/16*E(3)^2 ], [ 223, 221, 3/4 ], [ 223, 223, 1/4 ], [ 223, 224, 9/16 ], [ 223, 226, 3/16 ], 
  [ 223, 227, 1/4*E(3)-1/4*E(3)^2 ], [ 223, 229, 3/16*E(3)-3/16*E(3)^2 ], [ 224, 219, -1/2*E(3)+1/2*E(3)^2 ], [ 224, 224, -1/2 ], [ 225, 214, -3/8 ], 
  [ 225, 218, 1/2*E(3)-1/2*E(3)^2 ], [ 225, 219, -9/32 ], [ 225, 221, 3/8*E(3)-3/8*E(3)^2 ], [ 225, 222, 3/8*E(3)-3/8*E(3)^2 ], [ 225, 223, -1/8*E(3)+1/8*E(3)^2 ], 
  [ 225, 224, 9/32*E(3)-9/32*E(3)^2 ], [ 225, 225, -1/2 ], [ 225, 226, -3/32*E(3)+3/32*E(3)^2 ], [ 225, 227, -3/8 ], [ 225, 228, -3/8 ], [ 225, 229, -9/32 ], [ 226, 226, -1/2 ], 
  [ 226, 229, -1/2*E(3)+1/2*E(3)^2 ], [ 227, 214, 3/4 ], [ 227, 219, 9/16 ], [ 227, 221, -1/4*E(3)+1/4*E(3)^2 ], [ 227, 223, 1/4*E(3)-1/4*E(3)^2 ], [ 227, 224, -3/16*E(3)+3/16*E(3)^2 ], 
  [ 227, 226, 3/16*E(3)-3/16*E(3)^2 ], [ 227, 227, 1/4 ], [ 227, 229, 3/16 ], [ 228, 226, 1/4*E(3)-1/4*E(3)^2 ], [ 228, 228, 1 ], [ 228, 229, 3/4 ], [ 229, 226, -1/2*E(3)+1/2*E(3)^2 ], 
  [ 229, 229, -1/2 ], [ 230, 230, 1/4 ], [ 230, 231, 3/8 ], [ 230, 232, -1/4*E(3)+1/4*E(3)^2 ], [ 230, 233, -3/8*E(3)+3/8*E(3)^2 ], [ 230, 234, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 230, 235, -3/4 ], [ 231, 231, -1/2 ], [ 231, 233, 1/2*E(3)-1/2*E(3)^2 ], [ 232, 230, -1/4*E(3)+1/4*E(3)^2 ], [ 232, 231, -3/8*E(3)+3/8*E(3)^2 ], [ 232, 232, 1/4 ], 
  [ 232, 233, 3/8 ], [ 232, 234, -3/4 ], [ 232, 235, -1/4*E(3)+1/4*E(3)^2 ], [ 233, 231, 1/2*E(3)-1/2*E(3)^2 ], [ 233, 233, -1/2 ], [ 234, 230, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 234, 231, -1/8*E(3)+1/8*E(3)^2 ], [ 234, 232, -3/4 ], [ 234, 233, -3/8 ], [ 234, 234, 1/4 ], [ 234, 235, -1/4*E(3)+1/4*E(3)^2 ], [ 235, 230, -3/4 ], 
  [ 235, 231, -3/8 ], [ 235, 232, -1/4*E(3)+1/4*E(3)^2 ], [ 235, 233, -1/8*E(3)+1/8*E(3)^2 ], [ 235, 234, -1/4*E(3)+1/4*E(3)^2 ], [ 235, 235, 1/4 ], [ 236, 236, 1 ], [ 236, 237, -3/8 ], 
  [ 236, 239, -1/8*E(3)+1/8*E(3)^2 ], [ 237, 237, -1/2 ], [ 237, 239, -1/2*E(3)+1/2*E(3)^2 ], [ 238, 237, -3/8*E(3)+3/8*E(3)^2 ], [ 238, 238, 1 ], [ 238, 239, -9/8 ], 
  [ 239, 237, -1/2*E(3)+1/2*E(3)^2 ], [ 239, 239, -1/2 ], [ 240, 240, 1 ], [ 241, 3, -3/8*E(3)+3/8*E(3)^2 ], [ 241, 9, -1/4*E(3)+1/4*E(3)^2 ], [ 241, 121, -3/8 ], 
  [ 241, 129, 1/4*E(3)-1/4*E(3)^2 ], [ 241, 241, 1/4 ], [ 241, 243, -3/4 ], [ 241, 244, 3/4 ], [ 242, 2, 1/4*E(3)-1/4*E(3)^2 ], [ 242, 122, -1/4*E(3)+1/4*E(3)^2 ], 
  [ 242, 242, -1/2 ], [ 242, 244, 3/4 ], [ 243, 3, -1/8*E(3)+1/8*E(3)^2 ], [ 243, 9, -1/4*E(3)+1/4*E(3)^2 ], [ 243, 121, 3/8 ], [ 243, 129, 1/4*E(3)-1/4*E(3)^2 ], [ 243, 241, -3/4 ], 
  [ 243, 243, 1/4 ], [ 243, 244, 3/4 ], [ 244, 244, 1 ], [ 245, 125, -3/8 ], [ 245, 141, -1/8*E(3)+1/8*E(3)^2 ], [ 245, 245, 1 ], [ 246, 6, -3/16*E(3)+3/16*E(3)^2 ], 
  [ 246, 14, 1/4*E(3)-1/4*E(3)^2 ], [ 246, 125, -3/16 ], [ 246, 127, 9/16 ], [ 246, 134, -1/4*E(3)+1/4*E(3)^2 ], [ 246, 141, 1/16*E(3)-1/16*E(3)^2 ], [ 246, 245, 3/4 ], 
  [ 246, 246, 1/4 ], [ 246, 247, -3/4 ], [ 246, 248, 3/4 ], [ 247, 6, -9/16*E(3)+9/16*E(3)^2 ], [ 247, 14, 1/4*E(3)-1/4*E(3)^2 ], [ 247, 125, -3/16 ], 
  [ 247, 127, -9/16 ], [ 247, 134, -1/4*E(3)+1/4*E(3)^2 ], [ 247, 141, 1/16*E(3)-1/16*E(3)^2 ], [ 247, 245, 3/4 ], [ 247, 246, -3/4 ], [ 247, 247, 1/4 ], [ 247, 248, 3/4 ], 
  [ 248, 248, 1 ] ]]

]  

];


InstallMethod( ComponentGroup,
   "for a nilpotent orbit",
   true, [ IsNilpotentOrbit ], 0,
   function( o )


      local L, ct, pos1, pos2, len, unsparse;


unsparse:= function( q )

        local n,m,k;

        n:= q[1];
        m:= NullMat( n, n );
        for k in [2..Length(q)] do
            m[q[k][1]][q[k][2]]:= q[k][3];
        od;
        return m;
end;


      L:= AmbientLieAlgebra( o );
      ct:= CartanType( CartanMatrix( RootSystem(L) ) ).types;
      if Length( ct ) > 1 then
         Error("currently ComponentGroup( <o> ) only works for nilpotent orbits in simple Lie algebras");
      fi;

      if ct[1][1] = "A" then
         return Group( IdentityMat( Dimension(L), LeftActingDomain(L) ) );
      elif ct[1][1] in ["B","C","D"] then
         return SLAfcts.compgrpB_D( o );
      else
         pos1:= PositionProperty( SLAfcts.compgrps, u -> u[1] = ct[1][1] and
	           u[2] = ct[1][2] );
         pos2:= PositionProperty( SLAfcts.compgrps[pos1], u -> IsList(u) and
	           u[1] = WeightedDynkinDiagram( o ) );
         if pos2 = fail then
	    return Group( IdentityMat( Dimension(L), LeftActingDomain(L) ) );
	 else
	    len:= Length( SLAfcts.compgrps[pos1][pos2] );
	    return Group( List( SLAfcts.compgrps[pos1][pos2]{[2..len]},
	              unsparse ) );
	 fi;
      fi;
	              
end );
