"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.ParseHeaderStream = void 0;
// @ts-ignore
const readable_stream_1 = require("readable-stream");
const material_management_node_1 = require("@aws-crypto/material-management-node");
const serialize_1 = require("@aws-crypto/serialize");
const toUtf8 = (input) => Buffer.from(input.buffer, input.byteOffset, input.byteLength).toString('utf8');
const deserialize = (0, serialize_1.deserializeFactory)(toUtf8, material_management_node_1.NodeAlgorithmSuite);
const PortableTransformWithType = readable_stream_1.Transform;
class ParseHeaderStream extends PortableTransformWithType {
    materialsManager;
    commitmentPolicy;
    signaturePolicy;
    maxEncryptedDataKeys;
    _headerState;
    constructor(signaturePolicy, { commitmentPolicy, maxEncryptedDataKeys }, cmm) {
        super();
        /* Precondition: ParseHeaderStream needs a valid commitmentPolicy. */
        (0, material_management_node_1.needs)(material_management_node_1.CommitmentPolicy[commitmentPolicy], 'Invalid commitment policy.');
        /* Precondition: ParseHeaderStream needs a valid signaturePolicy. */
        (0, material_management_node_1.needs)(material_management_node_1.SignaturePolicy[signaturePolicy], 'Invalid signature policy.');
        // buildDecrypt defaults this to false for backwards compatibility, so this is satisfied
        /* Precondition: ParseHeaderStream needs a valid maxEncryptedDataKeys. */
        (0, material_management_node_1.needs)(maxEncryptedDataKeys === false || maxEncryptedDataKeys >= 1, 'Invalid maxEncryptedDataKeys value.');
        Object.defineProperty(this, 'materialsManager', {
            value: cmm,
            enumerable: true,
        });
        Object.defineProperty(this, 'commitmentPolicy', {
            value: commitmentPolicy,
            enumerable: true,
        });
        Object.defineProperty(this, 'maxEncryptedDataKeys', {
            value: maxEncryptedDataKeys,
            enumerable: true,
        });
        this._headerState = {
            buffer: Buffer.alloc(0),
            headerParsed: false,
        };
        Object.defineProperty(this, 'signaturePolicy', {
            value: signaturePolicy,
            enumerable: true,
        });
    }
    _transform(
    // eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types
    chunk, encoding, callback) {
        try {
            const { _headerState, commitmentPolicy, materialsManager, signaturePolicy, maxEncryptedDataKeys, } = this;
            const { buffer } = _headerState;
            const headerBuffer = Buffer.concat([buffer, chunk]);
            const headerInfo = deserialize.deserializeMessageHeader(headerBuffer, {
                maxEncryptedDataKeys,
            });
            if (!headerInfo) {
                _headerState.buffer = headerBuffer;
                return callback();
            }
            const { messageHeader, algorithmSuite } = headerInfo;
            const messageIDStr = Buffer.from(messageHeader.messageId).toString('hex');
            /* Precondition: The parsed header algorithmSuite from ParseHeaderStream must be supported by the commitmentPolicy. */
            material_management_node_1.CommitmentPolicySuites.isDecryptEnabled(commitmentPolicy, algorithmSuite, messageIDStr);
            /* Precondition: The parsed header algorithmSuite from ParseHeaderStream must be supported by the signaturePolicy. */
            material_management_node_1.SignaturePolicySuites.isDecryptEnabled(signaturePolicy, algorithmSuite, messageIDStr);
            const { rawHeader, headerAuth } = headerInfo;
            const { headerIv, headerAuthTag, headerAuthLength } = headerAuth;
            const suite = new material_management_node_1.NodeAlgorithmSuite(algorithmSuite.id);
            const { messageId, encryptionContext, encryptedDataKeys } = messageHeader;
            materialsManager
                .decryptMaterials({ suite, encryptionContext, encryptedDataKeys })
                .then((material) => {
                /* Precondition: The material algorithmSuite returned to ParseHeaderStream must be supported by the commitmentPolicy. */
                material_management_node_1.CommitmentPolicySuites.isDecryptEnabled(commitmentPolicy, material.suite, messageIDStr);
                /* Precondition: The material algorithmSuite returned to ParseHeaderStream must be supported by the signaturePolicy. */
                material_management_node_1.SignaturePolicySuites.isDecryptEnabled(signaturePolicy, material.suite, messageIDStr);
                _headerState.buffer = Buffer.alloc(0); // clear the Buffer...
                const { getDecipherInfo, getVerify, dispose } = (0, material_management_node_1.getDecryptionHelper)(material);
                const getDecipher = getDecipherInfo(messageId, 
                /* This is sub-optimal.
                 * Ideally I could pluck the `suiteData`
                 * right off the header
                 * and in such a way that may be undefined.
                 * But that has other consequences
                 * that are beyond the scope of this course.
                 */
                messageHeader.suiteData);
                const headerAuth = getDecipher(headerIv);
                headerAuth.setAAD(Buffer.from(rawHeader.buffer, rawHeader.byteOffset, rawHeader.byteLength));
                headerAuth.setAuthTag(Buffer.from(headerAuthTag.buffer, headerAuthTag.byteOffset, headerAuthTag.byteLength));
                headerAuth.update(Buffer.alloc(0));
                headerAuth.final(); // will throw if invalid
                const verify = getVerify ? getVerify() : void 0;
                const verifyInfo = {
                    headerInfo,
                    getDecipher,
                    verify,
                    dispose,
                };
                this.emit('VerifyInfo', verifyInfo);
                this.emit('MessageHeader', headerInfo.messageHeader);
                _headerState.headerParsed = true;
                // The header is parsed, pass control
                const readPos = rawHeader.byteLength + headerAuthLength;
                const tail = headerBuffer.slice(readPos);
                /* needs calls in downstream _transform streams will throw.
                 * But streams are async.
                 * So this error should be turned into an `.emit('error', ex)`.
                 */
                this._transform = (chunk, _enc, cb) => {
                    try {
                        cb(null, chunk);
                    }
                    catch (ex) {
                        this.emit('error', ex);
                    }
                };
                // flush the tail.  Stream control is now in the verify and decrypt streams
                return setImmediate(() => this._transform(tail, encoding, callback));
            })
                .catch((err) => callback(err));
        }
        catch (ex) {
            /* Exceptional Postcondition: An error MUST be emitted or this would be an unhandled exception. */
            this.emit('error', ex);
        }
    }
    _flush(callback) {
        /* Postcondition: A completed header MUST have been processed.
         * callback is an errBack function,
         * so it expects either an error OR undefined
         */
        callback(this._headerState.headerParsed
            ? undefined
            : new Error('Incomplete Header'));
    }
}
exports.ParseHeaderStream = ParseHeaderStream;
//# sourceMappingURL=data:application/json;base64,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