// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
// @ts-ignore
import { Transform as PortableTransform } from 'readable-stream';
import { needs, } from '@aws-crypto/material-management-node';
import { deserializeSignature, decodeBodyHeader, serializeMessageHeaderAuth, } from '@aws-crypto/serialize';
import { ParseHeaderStream } from './parse_header_stream';
const PortableTransformWithType = PortableTransform;
export class VerifyStream extends PortableTransformWithType {
    _headerInfo;
    _verifyState = {
        buffer: Buffer.alloc(0),
        authTagBuffer: Buffer.alloc(0),
        signatureInfo: Buffer.alloc(0),
        sequenceNumber: 0,
        finalAuthTagReceived: false,
    };
    _verify;
    _maxBodySize;
    constructor({ maxBodySize }) {
        super();
        /* Precondition: VerifyStream requires maxBodySize must be falsey or a number. */
        needs(!maxBodySize || typeof maxBodySize === 'number', 'Unsupported MaxBodySize.');
        Object.defineProperty(this, '_maxBodySize', {
            value: maxBodySize,
            enumerable: true,
        });
        this.on('pipe', (source) => {
            /* Precondition: The source must a ParseHeaderStream emit the required events. */
            needs(source instanceof ParseHeaderStream, 'Unsupported source');
            source.once('VerifyInfo', (verifyInfo) => {
                const { getDecipher, verify, headerInfo, dispose } = verifyInfo;
                const { messageId, contentType } = headerInfo.messageHeader;
                /* If I have a verify, the header needs to be flushed through.
                 * I do it here for initialize the verifier before I even
                 * add the element to the object.
                 */
                if (verify) {
                    const { rawHeader, headerAuth, messageHeader } = headerInfo;
                    const { headerIv, headerAuthTag } = headerAuth;
                    verify.update(rawHeader);
                    verify.update(serializeMessageHeaderAuth({
                        headerIv,
                        headerAuthTag,
                        messageHeader,
                    }));
                }
                Object.defineProperty(this, '_headerInfo', {
                    value: headerInfo,
                    enumerable: true,
                });
                Object.defineProperty(this, '_verify', {
                    value: verify,
                    enumerable: true,
                });
                const decipherInfo = {
                    messageId: Buffer.from(messageId.buffer || messageId, messageId.byteOffset || 0, messageId.byteLength),
                    contentType,
                    getDecipher,
                    dispose,
                };
                this.emit('DecipherInfo', decipherInfo);
            });
        });
    }
    _transform(chunk, enc, callback) {
        /* Precondition: VerifyInfo must have initialized the stream. */
        needs(this._headerInfo, 'VerifyStream not configured, VerifyInfo event not yet received.');
        // BodyHeader
        const state = this._verifyState;
        const { currentFrame } = state;
        if (!currentFrame) {
            const { buffer } = state;
            // Buffer.concat can be expensive. If buffer is empty, just use the chunk.
            const frameBuffer = buffer.length > 0 ? Buffer.concat([buffer, chunk]) : chunk;
            const frameHeader = decodeBodyHeader(frameBuffer, this._headerInfo, 0);
            if (!frameHeader) {
                // Need more data
                state.buffer = frameBuffer;
                return callback();
            }
            /* Precondition: If maxBodySize was set I can not buffer more data than maxBodySize.
             * Before returning *any* cleartext, the stream **MUST** verify the decryption.
             * This means that I must buffer the message until the AuthTag is reached.
             */
            needs(!this._maxBodySize || this._maxBodySize >= frameHeader.contentLength, 'maxBodySize exceeded.');
            /* Keeping track of the sequence number myself. */
            state.sequenceNumber += 1;
            /* Precondition: The sequence number is required to monotonically increase, starting from 1.
             * This is to avoid a bad actor from abusing the sequence number on un-signed algorithm suites.
             * If the frame size matched the data format (say NDJSON),
             * then the data could be significantly altered just by rearranging the frames.
             * Non-framed data returns a sequenceNumber of 1.
             */
            needs(frameHeader.sequenceNumber === state.sequenceNumber, 'Encrypted body sequence out of order.');
            if (this._verify) {
                this._verify.update(frameBuffer.slice(0, frameHeader.readPos));
            }
            const tail = frameBuffer.slice(frameHeader.readPos);
            this.emit('BodyInfo', frameHeader);
            state.currentFrame = frameHeader;
            return setImmediate(() => this._transform(tail, enc, callback));
        }
        // Content
        const { contentLength } = currentFrame;
        if (chunk.length && contentLength > 0) {
            if (contentLength > chunk.length) {
                currentFrame.contentLength -= chunk.length;
                this.push(chunk);
                return callback();
            }
            else {
                const content = chunk.slice(0, contentLength);
                const tail = chunk.slice(content.length);
                this.push(content);
                currentFrame.contentLength = 0;
                return setImmediate(() => this._transform(tail, enc, callback));
            }
        }
        // AuthTag
        const { tagLength } = currentFrame;
        const tagLengthBytes = tagLength / 8;
        const { authTagBuffer } = state;
        if (chunk.length && tagLengthBytes > authTagBuffer.length) {
            const left = tagLengthBytes - authTagBuffer.length;
            if (left > chunk.length) {
                // Buffer.concat can be expensive. If buffer is empty, just use the chunk.
                state.authTagBuffer =
                    authTagBuffer.length > 0
                        ? Buffer.concat([authTagBuffer, chunk])
                        : chunk;
                return callback();
            }
            else {
                // Buffer.concat can be expensive. If buffer is empty, just use the chunk.
                const finalAuthTagBuffer = authTagBuffer.length > 0
                    ? Buffer.concat([authTagBuffer, chunk], tagLengthBytes)
                    : chunk.slice(0, tagLengthBytes);
                if (this._verify) {
                    this._verify.update(finalAuthTagBuffer);
                }
                /* Reset state.
                 * Ciphertext buffers and authTag buffers need to be cleared.
                 */
                state.buffer = Buffer.alloc(0);
                state.currentFrame = undefined;
                state.authTagBuffer = Buffer.alloc(0);
                /* After the final frame the file format is _much_ simpler.
                 * Making sure the cascading if blocks fall to the signature can be tricky and brittle.
                 * After the final frame, just moving on to concatenate the signature is much simpler.
                 */
                if (currentFrame.isFinalFrame) {
                    /* Signal that the we are at the end of the ciphertext.
                     * See decodeBodyHeader, non-framed will set isFinalFrame
                     * for the single frame.
                     */
                    this._verifyState.finalAuthTagReceived = true;
                    /* Overwriting the _transform function.
                     * Data flow control is now handled here.
                     */
                    this._transform = (chunk, _enc, callback) => {
                        if (chunk.length) {
                            state.signatureInfo = Buffer.concat([state.signatureInfo, chunk]);
                        }
                        callback();
                    };
                }
                const tail = chunk.slice(left);
                /* The decipher_stream uses the `AuthTag` event to flush the accumulated frame.
                 * This is because ciphertext should never be returned until it is verified.
                 * i.e. the auth tag checked.
                 * This can create an issue if the chucks and frame size are small.
                 * If the verify stream continues processing and sends the next auth tag,
                 * before the current auth tag has been completed.
                 * This is basically a back pressure issue.
                 * Since the frame size, and consequently the high water mark,
                 * can not be know when the stream is created,
                 * the internal stream state would need to be modified.
                 * I assert that a simple callback is a simpler way to handle this.
                 */
                const next = () => this._transform(tail, enc, callback);
                return this.emit('AuthTag', finalAuthTagBuffer, next);
            }
        }
        callback();
    }
    push(chunk, encoding) {
        // Typescript???? this._verify instanceof Verify is better....
        if (this._verify && chunk) {
            this._verify.update(chunk);
        }
        return super.push(chunk, encoding);
    }
    _flush(callback) {
        const { finalAuthTagReceived } = this._verifyState;
        /* Precondition: All ciphertext MUST have been received.
         * The verify stream has ended,
         * there will be no more data.
         * Therefore we MUST have reached the end.
         */
        if (!finalAuthTagReceived)
            return callback(new Error('Incomplete message'));
        /* Check for early return (Postcondition): If there is no verify stream do not attempt to verify. */
        if (!this._verify)
            return callback();
        try {
            const { signatureInfo } = this._verifyState;
            /* Precondition: The signature must be well formed. */
            const { buffer, byteOffset, byteLength } = deserializeSignature(signatureInfo);
            const signature = Buffer.from(buffer, byteOffset, byteLength);
            const isVerified = this._verify.awsCryptoVerify(signature);
            /* Postcondition: The signature must be valid. */
            needs(isVerified, 'Invalid Signature');
            callback();
        }
        catch (e) {
            callback(e);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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