/*
 * Copyright (c) 2018-2020 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __aarch64__

#include <algorithm>

#include "arm_gemm.hpp"
#include <cstdint>
#include "../../asmlib.hpp"
#include "../../utils.hpp"

namespace arm_gemm {

void a64_hybrid_s8s32_dot_16x4(const int8_t *A, int lda, const int8_t *B, int32_t *C, int ldc, int M, int N, int K, const int32_t *, Activation , bool accumulate) {
    const int K_stride = ((K + 3) / 4) * 4;
    const long loops_count = ((K + 16) / 32) - 1;
    K -= loops_count * 32;
    const long regs_count = (K / 16) - 1;
    K -= (regs_count + 1) * 16;
    const long blocks_count = K / 4;
    const long odds_count = K - (blocks_count * 4);

    int rows_to_compute;

    for (int y=0; y<M; y+=rows_to_compute) {
        const int8_t * const a_ptr0_base = A + (y * lda);
        const unsigned long ldab = lda * sizeof(int8_t);

        int32_t *c_ptr0 = C + (y * ldc);

        rows_to_compute = M-y;
        if (rows_to_compute > 4) {
            if (rows_to_compute % 4) {
                rows_to_compute = 4 - 1;
            } else {
                rows_to_compute = 4;
            }
        }

        for (int x0=0; x0<N; x0+=16ul) {
            const long width = std::min((unsigned long)N-x0, 16ul);
            long loops = loops_count;
            long regs = regs_count;
            long blocks = blocks_count;
            long odds = odds_count;
            const int8_t *a_ptr0 = a_ptr0_base;
            const int8_t *b_ptr0 = B + (K_stride * x0);
            const bool use_result_buffer = (width < 16);
            int32_t result_buffer[64];
            const unsigned long ldcb = (use_result_buffer ? 16 : ldc) * sizeof(int32_t);
            int32_t *c_ptr_real = c_ptr0;
            if (use_result_buffer && accumulate) {
                for(int cy=0; cy<std::min(M-y, 4); cy++) {
                    for(unsigned int cx=0; cx<width; cx++) {
                        result_buffer[cy * 16 + cx] = c_ptr_real[cy * ldc + cx];
                    }
                }
            }
            if (use_result_buffer) {
                c_ptr0 = result_buffer;
            }

            switch(rows_to_compute) {
                case 1:
                    __asm __volatile (
                        "cbnz %[accumulate], 1f\n"
                        "movi v16.4s, #0\n"
                        "ldr q0, [%[a_ptr0]]\n"
                        "movi v17.4s, #0\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        "movi v18.4s, #0\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        "movi v19.4s, #0\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x80\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ldr q16, [%[c_ptr0]]\n"
                        "ldr q17, [%[c_ptr0], #0x10]\n"
                        "ldr q18, [%[c_ptr0], #0x20]\n"
                        "ldr q19, [%[c_ptr0], #0x30]\n"
                        "ldr q0, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x80\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        "ldr q15, [%[b_ptr0], #-0x10]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        "ldr q4, [%[a_ptr0]]\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4fa0e190 // sdot v16.4s, v12.16b, v0.4b[1]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4fa0e1b1 // sdot v17.4s, v13.16b, v0.4b[1]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa0e1d2 // sdot v18.4s, v14.16b, v0.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa0e1f3 // sdot v19.4s, v15.16b, v0.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4f80e910 // sdot v16.4s, v8.16b, v0.4b[2]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4f80e931 // sdot v17.4s, v9.16b, v0.4b[2]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f80e952 // sdot v18.4s, v10.16b, v0.4b[2]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        ".inst 0x4f80e973 // sdot v19.4s, v11.16b, v0.4b[2]\n"
                        "prfm PLDL1KEEP, [%[a_ptr0], #0x40]\n"
                        ".inst 0x4fa0e990 // sdot v16.4s, v12.16b, v0.4b[3]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x100\n"
                        ".inst 0x4fa0e9b1 // sdot v17.4s, v13.16b, v0.4b[3]\n"
                        "ldr q8, [%[b_ptr0], #-0x80]\n"
                        ".inst 0x4fa0e9d2 // sdot v18.4s, v14.16b, v0.4b[3]\n"
                        "ldr q9, [%[b_ptr0], #-0x70]\n"
                        ".inst 0x4fa0e9f3 // sdot v19.4s, v15.16b, v0.4b[3]\n"
                        "ldr q10, [%[b_ptr0], #-0x60]\n"
                        "ldr q11, [%[b_ptr0], #-0x50]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x4f84e110 // sdot v16.4s, v8.16b, v4.4b[0]\n"
                        "ldr q12, [%[b_ptr0], #-0x40]\n"
                        ".inst 0x4f84e131 // sdot v17.4s, v9.16b, v4.4b[0]\n"
                        "ldr q13, [%[b_ptr0], #-0x30]\n"
                        ".inst 0x4f84e152 // sdot v18.4s, v10.16b, v4.4b[0]\n"
                        "ldr q14, [%[b_ptr0], #-0x20]\n"
                        ".inst 0x4f84e173 // sdot v19.4s, v11.16b, v4.4b[0]\n"
                        "ldr q15, [%[b_ptr0], #-0x10]\n"
                        ".inst 0x4fa4e190 // sdot v16.4s, v12.16b, v4.4b[1]\n"
                        "ldr q0, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x4fa4e1b1 // sdot v17.4s, v13.16b, v4.4b[1]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4fa4e1d2 // sdot v18.4s, v14.16b, v4.4b[1]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4fa4e1f3 // sdot v19.4s, v15.16b, v4.4b[1]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4f84e910 // sdot v16.4s, v8.16b, v4.4b[2]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4f84e931 // sdot v17.4s, v9.16b, v4.4b[2]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4f84e952 // sdot v18.4s, v10.16b, v4.4b[2]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4f84e973 // sdot v19.4s, v11.16b, v4.4b[2]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4fa4e990 // sdot v16.4s, v12.16b, v4.4b[3]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x100\n"
                        ".inst 0x4fa4e9b1 // sdot v17.4s, v13.16b, v4.4b[3]\n"
                        "ldr q8, [%[b_ptr0], #-0x80]\n"
                        ".inst 0x4fa4e9d2 // sdot v18.4s, v14.16b, v4.4b[3]\n"
                        "ldr q9, [%[b_ptr0], #-0x70]\n"
                        ".inst 0x4fa4e9f3 // sdot v19.4s, v15.16b, v4.4b[3]\n"
                        "ldr q10, [%[b_ptr0], #-0x60]\n"
                        "ldr q11, [%[b_ptr0], #-0x50]\n"
                        "ldr q12, [%[b_ptr0], #-0x40]\n"
                        "ldr q13, [%[b_ptr0], #-0x30]\n"
                        "ldr q14, [%[b_ptr0], #-0x20]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "ldr q15, [%[b_ptr0], #-0x10]\n"
                        "prfm PSTL1KEEP, [%[c_ptr0]]\n"
                        "cbz %[regs], 4f\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        "ldr q4, [%[a_ptr0]]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4fa0e190 // sdot v16.4s, v12.16b, v0.4b[1]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa0e1b1 // sdot v17.4s, v13.16b, v0.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa0e1d2 // sdot v18.4s, v14.16b, v0.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4fa0e1f3 // sdot v19.4s, v15.16b, v0.4b[1]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4f80e910 // sdot v16.4s, v8.16b, v0.4b[2]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f80e931 // sdot v17.4s, v9.16b, v0.4b[2]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x100\n"
                        ".inst 0x4f80e952 // sdot v18.4s, v10.16b, v0.4b[2]\n"
                        "ldr q8, [%[b_ptr0], #-0x80]\n"
                        ".inst 0x4f80e973 // sdot v19.4s, v11.16b, v0.4b[2]\n"
                        "ldr q9, [%[b_ptr0], #-0x70]\n"
                        ".inst 0x4fa0e990 // sdot v16.4s, v12.16b, v0.4b[3]\n"
                        "ldr q10, [%[b_ptr0], #-0x60]\n"
                        ".inst 0x4fa0e9b1 // sdot v17.4s, v13.16b, v0.4b[3]\n"
                        "ldr q11, [%[b_ptr0], #-0x50]\n"
                        ".inst 0x4fa0e9d2 // sdot v18.4s, v14.16b, v0.4b[3]\n"
                        "ldr q12, [%[b_ptr0], #-0x40]\n"
                        ".inst 0x4fa0e9f3 // sdot v19.4s, v15.16b, v0.4b[3]\n"
                        "ldr q13, [%[b_ptr0], #-0x30]\n"
                        ".inst 0x4f84e110 // sdot v16.4s, v8.16b, v4.4b[0]\n"
                        "ldr q14, [%[b_ptr0], #-0x20]\n"
                        ".inst 0x4f84e131 // sdot v17.4s, v9.16b, v4.4b[0]\n"
                        "ldr q15, [%[b_ptr0], #-0x10]\n"
                        ".inst 0x4f84e152 // sdot v18.4s, v10.16b, v4.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f84e173 // sdot v19.4s, v11.16b, v4.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4fa4e190 // sdot v16.4s, v12.16b, v4.4b[1]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4fa4e1b1 // sdot v17.4s, v13.16b, v4.4b[1]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa4e1d2 // sdot v18.4s, v14.16b, v4.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa4e1f3 // sdot v19.4s, v15.16b, v4.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4f84e910 // sdot v16.4s, v8.16b, v4.4b[2]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4f84e931 // sdot v17.4s, v9.16b, v4.4b[2]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f84e952 // sdot v18.4s, v10.16b, v4.4b[2]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x80\n"
                        ".inst 0x4f84e973 // sdot v19.4s, v11.16b, v4.4b[2]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        ".inst 0x4fa4e990 // sdot v16.4s, v12.16b, v4.4b[3]\n"
                        ".inst 0x4fa4e9b1 // sdot v17.4s, v13.16b, v4.4b[3]\n"
                        ".inst 0x4fa4e9d2 // sdot v18.4s, v14.16b, v4.4b[3]\n"
                        ".inst 0x4fa4e9f3 // sdot v19.4s, v15.16b, v4.4b[3]\n"
                        "b 5f\n"
                        "4:\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa0e190 // sdot v16.4s, v12.16b, v0.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa0e1b1 // sdot v17.4s, v13.16b, v0.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4fa0e1d2 // sdot v18.4s, v14.16b, v0.4b[1]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4fa0e1f3 // sdot v19.4s, v15.16b, v0.4b[1]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f80e910 // sdot v16.4s, v8.16b, v0.4b[2]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x80\n"
                        ".inst 0x4f80e931 // sdot v17.4s, v9.16b, v0.4b[2]\n"
                        ".inst 0x4f80e952 // sdot v18.4s, v10.16b, v0.4b[2]\n"
                        ".inst 0x4f80e973 // sdot v19.4s, v11.16b, v0.4b[2]\n"
                        ".inst 0x4fa0e990 // sdot v16.4s, v12.16b, v0.4b[3]\n"
                        ".inst 0x4fa0e9b1 // sdot v17.4s, v13.16b, v0.4b[3]\n"
                        ".inst 0x4fa0e9d2 // sdot v18.4s, v14.16b, v0.4b[3]\n"
                        ".inst 0x4fa0e9f3 // sdot v19.4s, v15.16b, v0.4b[3]\n"
                        "5:\n"
                        "cbz %[blocks], 6f\n"
                        "7:\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        "ldr s0, [%[a_ptr0]]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x4\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x40\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        "b.ne 7b\n"
                        "6:\n"
                        "cbz %[odds], 8f\n"
                        "ld1 {v0.b}[0], [%[a_ptr0]], #1\n"
                        "subs %[odds], %[odds], #0x1\n"
                        "b.eq 9f\n"
                        "ld1 {v0.b}[1], [%[a_ptr0]], #1\n"
                        "subs %[odds], %[odds], #0x1\n"
                        "b.eq 9f\n"
                        "ld1 {v0.b}[2], [%[a_ptr0]]\n"
                        "9:\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        "8:\n"
                        "str q16, [%[c_ptr0]]\n"
                        "str q17, [%[c_ptr0], #0x10]\n"
                        "str q18, [%[c_ptr0], #0x20]\n"
                        "str q19, [%[c_ptr0], #0x30]\n"
                        "add %[c_ptr0], %[c_ptr0], #0x40\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [blocks] "+r" (blocks), [odds] "+r" (odds)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb)
                        : "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                    );
                    break;
                case 2:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "c_ptr1 .req X1\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "cbnz %[accumulate], 1f\n"
                        "movi v16.4s, #0\n"
                        "ldr q0, [%[a_ptr0]]\n"
                        "movi v17.4s, #0\n"
                        "ldr q1, [a_ptr1]\n"
                        "movi v18.4s, #0\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        "movi v19.4s, #0\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        "movi v20.4s, #0\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        "movi v21.4s, #0\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        "movi v22.4s, #0\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        "movi v23.4s, #0\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "add %[b_ptr0], %[b_ptr0], #0x80\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ldr q16, [%[c_ptr0]]\n"
                        "ldr q17, [%[c_ptr0], #0x10]\n"
                        "ldr q18, [%[c_ptr0], #0x20]\n"
                        "ldr q19, [%[c_ptr0], #0x30]\n"
                        "ldr q20, [c_ptr1]\n"
                        "ldr q21, [c_ptr1, #0x10]\n"
                        "ldr q22, [c_ptr1, #0x20]\n"
                        "ldr q23, [c_ptr1, #0x30]\n"
                        "ldr q0, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ldr q1, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x80\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        "ldr q15, [%[b_ptr0], #-0x10]\n"
                        ".inst 0x4f81e114 // sdot v20.4s, v8.16b, v1.4b[0]\n"
                        "ldr q4, [%[a_ptr0]]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        "ldr q5, [a_ptr1]\n"
                        ".inst 0x4f81e135 // sdot v21.4s, v9.16b, v1.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4f81e156 // sdot v22.4s, v10.16b, v1.4b[0]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        ".inst 0x4f81e177 // sdot v23.4s, v11.16b, v1.4b[0]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa0e190 // sdot v16.4s, v12.16b, v0.4b[1]\n"
                        "prfm PLDL1KEEP, [%[a_ptr0], #0x40]\n"
                        ".inst 0x4fa1e194 // sdot v20.4s, v12.16b, v1.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa0e1b1 // sdot v17.4s, v13.16b, v0.4b[1]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x4fa1e1b5 // sdot v21.4s, v13.16b, v1.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4fa0e1d2 // sdot v18.4s, v14.16b, v0.4b[1]\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        ".inst 0x4fa1e1d6 // sdot v22.4s, v14.16b, v1.4b[1]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4fa0e1f3 // sdot v19.4s, v15.16b, v0.4b[1]\n"
                        "prfm PLDL1KEEP, [a_ptr1, #0x40]\n"
                        ".inst 0x4fa1e1f7 // sdot v23.4s, v15.16b, v1.4b[1]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f80e910 // sdot v16.4s, v8.16b, v0.4b[2]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x100\n"
                        ".inst 0x4f81e914 // sdot v20.4s, v8.16b, v1.4b[2]\n"
                        "ldr q8, [%[b_ptr0], #-0x80]\n"
                        ".inst 0x4f80e931 // sdot v17.4s, v9.16b, v0.4b[2]\n"
                        ".inst 0x4f81e935 // sdot v21.4s, v9.16b, v1.4b[2]\n"
                        "ldr q9, [%[b_ptr0], #-0x70]\n"
                        ".inst 0x4f80e952 // sdot v18.4s, v10.16b, v0.4b[2]\n"
                        ".inst 0x4f81e956 // sdot v22.4s, v10.16b, v1.4b[2]\n"
                        "ldr q10, [%[b_ptr0], #-0x60]\n"
                        ".inst 0x4f80e973 // sdot v19.4s, v11.16b, v0.4b[2]\n"
                        ".inst 0x4f81e977 // sdot v23.4s, v11.16b, v1.4b[2]\n"
                        "ldr q11, [%[b_ptr0], #-0x50]\n"
                        ".inst 0x4fa0e990 // sdot v16.4s, v12.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e994 // sdot v20.4s, v12.16b, v1.4b[3]\n"
                        "ldr q12, [%[b_ptr0], #-0x40]\n"
                        ".inst 0x4fa0e9b1 // sdot v17.4s, v13.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9b5 // sdot v21.4s, v13.16b, v1.4b[3]\n"
                        "ldr q13, [%[b_ptr0], #-0x30]\n"
                        ".inst 0x4fa0e9d2 // sdot v18.4s, v14.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9d6 // sdot v22.4s, v14.16b, v1.4b[3]\n"
                        "ldr q14, [%[b_ptr0], #-0x20]\n"
                        ".inst 0x4fa0e9f3 // sdot v19.4s, v15.16b, v0.4b[3]\n"
                        "ldr q0, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x4fa1e9f7 // sdot v23.4s, v15.16b, v1.4b[3]\n"
                        "ldr q15, [%[b_ptr0], #-0x10]\n"
                        ".inst 0x4f84e110 // sdot v16.4s, v8.16b, v4.4b[0]\n"
                        "ldr q1, [a_ptr1, #-0x10]\n"
                        ".inst 0x4f85e114 // sdot v20.4s, v8.16b, v5.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f84e131 // sdot v17.4s, v9.16b, v4.4b[0]\n"
                        ".inst 0x4f85e135 // sdot v21.4s, v9.16b, v5.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4f84e152 // sdot v18.4s, v10.16b, v4.4b[0]\n"
                        ".inst 0x4f85e156 // sdot v22.4s, v10.16b, v5.4b[0]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4f84e173 // sdot v19.4s, v11.16b, v4.4b[0]\n"
                        ".inst 0x4f85e177 // sdot v23.4s, v11.16b, v5.4b[0]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa4e190 // sdot v16.4s, v12.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e194 // sdot v20.4s, v12.16b, v5.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa4e1b1 // sdot v17.4s, v13.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e1b5 // sdot v21.4s, v13.16b, v5.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4fa4e1d2 // sdot v18.4s, v14.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e1d6 // sdot v22.4s, v14.16b, v5.4b[1]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4fa4e1f3 // sdot v19.4s, v15.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e1f7 // sdot v23.4s, v15.16b, v5.4b[1]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f84e910 // sdot v16.4s, v8.16b, v4.4b[2]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x100\n"
                        ".inst 0x4f85e914 // sdot v20.4s, v8.16b, v5.4b[2]\n"
                        "ldr q8, [%[b_ptr0], #-0x80]\n"
                        ".inst 0x4f84e931 // sdot v17.4s, v9.16b, v4.4b[2]\n"
                        ".inst 0x4f85e935 // sdot v21.4s, v9.16b, v5.4b[2]\n"
                        "ldr q9, [%[b_ptr0], #-0x70]\n"
                        ".inst 0x4f84e952 // sdot v18.4s, v10.16b, v4.4b[2]\n"
                        ".inst 0x4f85e956 // sdot v22.4s, v10.16b, v5.4b[2]\n"
                        "ldr q10, [%[b_ptr0], #-0x60]\n"
                        ".inst 0x4f84e973 // sdot v19.4s, v11.16b, v4.4b[2]\n"
                        ".inst 0x4f85e977 // sdot v23.4s, v11.16b, v5.4b[2]\n"
                        "ldr q11, [%[b_ptr0], #-0x50]\n"
                        ".inst 0x4fa4e990 // sdot v16.4s, v12.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e994 // sdot v20.4s, v12.16b, v5.4b[3]\n"
                        "ldr q12, [%[b_ptr0], #-0x40]\n"
                        ".inst 0x4fa4e9b1 // sdot v17.4s, v13.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e9b5 // sdot v21.4s, v13.16b, v5.4b[3]\n"
                        "ldr q13, [%[b_ptr0], #-0x30]\n"
                        ".inst 0x4fa4e9d2 // sdot v18.4s, v14.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e9d6 // sdot v22.4s, v14.16b, v5.4b[3]\n"
                        "ldr q14, [%[b_ptr0], #-0x20]\n"
                        ".inst 0x4fa4e9f3 // sdot v19.4s, v15.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e9f7 // sdot v23.4s, v15.16b, v5.4b[3]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "ldr q15, [%[b_ptr0], #-0x10]\n"
                        "prfm PSTL1KEEP, [%[c_ptr0]]\n"
                        "prfm PSTL1KEEP, [c_ptr1]\n"
                        "cbz %[regs], 4f\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        "ldr q4, [%[a_ptr0]]\n"
                        ".inst 0x4f81e114 // sdot v20.4s, v8.16b, v1.4b[0]\n"
                        "ldr q5, [a_ptr1]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f81e135 // sdot v21.4s, v9.16b, v1.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        ".inst 0x4f81e156 // sdot v22.4s, v10.16b, v1.4b[0]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        ".inst 0x4f81e177 // sdot v23.4s, v11.16b, v1.4b[0]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa0e190 // sdot v16.4s, v12.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e194 // sdot v20.4s, v12.16b, v1.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa0e1b1 // sdot v17.4s, v13.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1b5 // sdot v21.4s, v13.16b, v1.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4fa0e1d2 // sdot v18.4s, v14.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1d6 // sdot v22.4s, v14.16b, v1.4b[1]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4fa0e1f3 // sdot v19.4s, v15.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1f7 // sdot v23.4s, v15.16b, v1.4b[1]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f80e910 // sdot v16.4s, v8.16b, v0.4b[2]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x100\n"
                        ".inst 0x4f81e914 // sdot v20.4s, v8.16b, v1.4b[2]\n"
                        "ldr q8, [%[b_ptr0], #-0x80]\n"
                        ".inst 0x4f80e931 // sdot v17.4s, v9.16b, v0.4b[2]\n"
                        ".inst 0x4f81e935 // sdot v21.4s, v9.16b, v1.4b[2]\n"
                        "ldr q9, [%[b_ptr0], #-0x70]\n"
                        ".inst 0x4f80e952 // sdot v18.4s, v10.16b, v0.4b[2]\n"
                        ".inst 0x4f81e956 // sdot v22.4s, v10.16b, v1.4b[2]\n"
                        "ldr q10, [%[b_ptr0], #-0x60]\n"
                        ".inst 0x4f80e973 // sdot v19.4s, v11.16b, v0.4b[2]\n"
                        ".inst 0x4f81e977 // sdot v23.4s, v11.16b, v1.4b[2]\n"
                        "ldr q11, [%[b_ptr0], #-0x50]\n"
                        ".inst 0x4fa0e990 // sdot v16.4s, v12.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e994 // sdot v20.4s, v12.16b, v1.4b[3]\n"
                        "ldr q12, [%[b_ptr0], #-0x40]\n"
                        ".inst 0x4fa0e9b1 // sdot v17.4s, v13.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9b5 // sdot v21.4s, v13.16b, v1.4b[3]\n"
                        "ldr q13, [%[b_ptr0], #-0x30]\n"
                        ".inst 0x4fa0e9d2 // sdot v18.4s, v14.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9d6 // sdot v22.4s, v14.16b, v1.4b[3]\n"
                        "ldr q14, [%[b_ptr0], #-0x20]\n"
                        ".inst 0x4fa0e9f3 // sdot v19.4s, v15.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9f7 // sdot v23.4s, v15.16b, v1.4b[3]\n"
                        "ldr q15, [%[b_ptr0], #-0x10]\n"
                        ".inst 0x4f84e110 // sdot v16.4s, v8.16b, v4.4b[0]\n"
                        ".inst 0x4f85e114 // sdot v20.4s, v8.16b, v5.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f84e131 // sdot v17.4s, v9.16b, v4.4b[0]\n"
                        ".inst 0x4f85e135 // sdot v21.4s, v9.16b, v5.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4f84e152 // sdot v18.4s, v10.16b, v4.4b[0]\n"
                        ".inst 0x4f85e156 // sdot v22.4s, v10.16b, v5.4b[0]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4f84e173 // sdot v19.4s, v11.16b, v4.4b[0]\n"
                        ".inst 0x4f85e177 // sdot v23.4s, v11.16b, v5.4b[0]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa4e190 // sdot v16.4s, v12.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e194 // sdot v20.4s, v12.16b, v5.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa4e1b1 // sdot v17.4s, v13.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e1b5 // sdot v21.4s, v13.16b, v5.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4fa4e1d2 // sdot v18.4s, v14.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e1d6 // sdot v22.4s, v14.16b, v5.4b[1]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4fa4e1f3 // sdot v19.4s, v15.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e1f7 // sdot v23.4s, v15.16b, v5.4b[1]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f84e910 // sdot v16.4s, v8.16b, v4.4b[2]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x80\n"
                        ".inst 0x4f85e914 // sdot v20.4s, v8.16b, v5.4b[2]\n"
                        ".inst 0x4f84e931 // sdot v17.4s, v9.16b, v4.4b[2]\n"
                        ".inst 0x4f85e935 // sdot v21.4s, v9.16b, v5.4b[2]\n"
                        ".inst 0x4f84e952 // sdot v18.4s, v10.16b, v4.4b[2]\n"
                        ".inst 0x4f85e956 // sdot v22.4s, v10.16b, v5.4b[2]\n"
                        ".inst 0x4f84e973 // sdot v19.4s, v11.16b, v4.4b[2]\n"
                        ".inst 0x4f85e977 // sdot v23.4s, v11.16b, v5.4b[2]\n"
                        ".inst 0x4fa4e990 // sdot v16.4s, v12.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e994 // sdot v20.4s, v12.16b, v5.4b[3]\n"
                        ".inst 0x4fa4e9b1 // sdot v17.4s, v13.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e9b5 // sdot v21.4s, v13.16b, v5.4b[3]\n"
                        ".inst 0x4fa4e9d2 // sdot v18.4s, v14.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e9d6 // sdot v22.4s, v14.16b, v5.4b[3]\n"
                        ".inst 0x4fa4e9f3 // sdot v19.4s, v15.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e9f7 // sdot v23.4s, v15.16b, v5.4b[3]\n"
                        "b 5f\n"
                        "4:\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        ".inst 0x4f81e114 // sdot v20.4s, v8.16b, v1.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        ".inst 0x4f81e135 // sdot v21.4s, v9.16b, v1.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        ".inst 0x4f81e156 // sdot v22.4s, v10.16b, v1.4b[0]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        ".inst 0x4f81e177 // sdot v23.4s, v11.16b, v1.4b[0]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa0e190 // sdot v16.4s, v12.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e194 // sdot v20.4s, v12.16b, v1.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa0e1b1 // sdot v17.4s, v13.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1b5 // sdot v21.4s, v13.16b, v1.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4fa0e1d2 // sdot v18.4s, v14.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1d6 // sdot v22.4s, v14.16b, v1.4b[1]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4fa0e1f3 // sdot v19.4s, v15.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1f7 // sdot v23.4s, v15.16b, v1.4b[1]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f80e910 // sdot v16.4s, v8.16b, v0.4b[2]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x80\n"
                        ".inst 0x4f81e914 // sdot v20.4s, v8.16b, v1.4b[2]\n"
                        ".inst 0x4f80e931 // sdot v17.4s, v9.16b, v0.4b[2]\n"
                        ".inst 0x4f81e935 // sdot v21.4s, v9.16b, v1.4b[2]\n"
                        ".inst 0x4f80e952 // sdot v18.4s, v10.16b, v0.4b[2]\n"
                        ".inst 0x4f81e956 // sdot v22.4s, v10.16b, v1.4b[2]\n"
                        ".inst 0x4f80e973 // sdot v19.4s, v11.16b, v0.4b[2]\n"
                        ".inst 0x4f81e977 // sdot v23.4s, v11.16b, v1.4b[2]\n"
                        ".inst 0x4fa0e990 // sdot v16.4s, v12.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e994 // sdot v20.4s, v12.16b, v1.4b[3]\n"
                        ".inst 0x4fa0e9b1 // sdot v17.4s, v13.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9b5 // sdot v21.4s, v13.16b, v1.4b[3]\n"
                        ".inst 0x4fa0e9d2 // sdot v18.4s, v14.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9d6 // sdot v22.4s, v14.16b, v1.4b[3]\n"
                        ".inst 0x4fa0e9f3 // sdot v19.4s, v15.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9f7 // sdot v23.4s, v15.16b, v1.4b[3]\n"
                        "5:\n"
                        "cbz %[blocks], 6f\n"
                        "7:\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        "ldr s0, [%[a_ptr0]]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x4\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x40\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        "ldr s1, [a_ptr1]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        "add a_ptr1, a_ptr1, #0x4\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        ".inst 0x4f81e114 // sdot v20.4s, v8.16b, v1.4b[0]\n"
                        ".inst 0x4f81e135 // sdot v21.4s, v9.16b, v1.4b[0]\n"
                        ".inst 0x4f81e156 // sdot v22.4s, v10.16b, v1.4b[0]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        ".inst 0x4f81e177 // sdot v23.4s, v11.16b, v1.4b[0]\n"
                        "b.ne 7b\n"
                        "6:\n"
                        "cbz %[odds], 8f\n"
                        "ld1 {v0.b}[0], [%[a_ptr0]], #1\n"
                        "ld1 {v1.b}[0], [a_ptr1], #1\n"
                        "subs %[odds], %[odds], #0x1\n"
                        "b.eq 9f\n"
                        "ld1 {v0.b}[1], [%[a_ptr0]], #1\n"
                        "ld1 {v1.b}[1], [a_ptr1], #1\n"
                        "subs %[odds], %[odds], #0x1\n"
                        "b.eq 9f\n"
                        "ld1 {v0.b}[2], [%[a_ptr0]]\n"
                        "ld1 {v1.b}[2], [a_ptr1]\n"
                        "9:\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        ".inst 0x4f81e114 // sdot v20.4s, v8.16b, v1.4b[0]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        ".inst 0x4f81e135 // sdot v21.4s, v9.16b, v1.4b[0]\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        ".inst 0x4f81e156 // sdot v22.4s, v10.16b, v1.4b[0]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        ".inst 0x4f81e177 // sdot v23.4s, v11.16b, v1.4b[0]\n"
                        "8:\n"
                        "str q16, [%[c_ptr0]]\n"
                        "str q17, [%[c_ptr0], #0x10]\n"
                        "str q18, [%[c_ptr0], #0x20]\n"
                        "str q19, [%[c_ptr0], #0x30]\n"
                        "add %[c_ptr0], %[c_ptr0], #0x40\n"
                        "str q20, [c_ptr1]\n"
                        "str q21, [c_ptr1, #0x10]\n"
                        "str q22, [c_ptr1, #0x20]\n"
                        "str q23, [c_ptr1, #0x30]\n"
                        ".unreq a_ptr1\n"
                        ".unreq c_ptr1\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [blocks] "+r" (blocks), [odds] "+r" (odds)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb)
                        : "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x0", "x1", "cc", "memory"
                    );
                    break;
                case 3:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "a_ptr2 .req X1\n"
                        "c_ptr1 .req X2\n"
                        "c_ptr2 .req X3\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "add a_ptr2, a_ptr1, %[lda]\n"
                        "add c_ptr2, c_ptr1, %[ldc]\n"
                        "cbnz %[accumulate], 1f\n"
                        "movi v16.4s, #0\n"
                        "ldr q0, [%[a_ptr0]]\n"
                        "movi v17.4s, #0\n"
                        "ldr q1, [a_ptr1]\n"
                        "movi v18.4s, #0\n"
                        "ldr q2, [a_ptr2]\n"
                        "movi v19.4s, #0\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        "movi v20.4s, #0\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        "movi v21.4s, #0\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        "movi v22.4s, #0\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        "movi v23.4s, #0\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        "movi v24.4s, #0\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        "movi v25.4s, #0\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        "movi v26.4s, #0\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "movi v27.4s, #0\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "add %[b_ptr0], %[b_ptr0], #0x80\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ldr q16, [%[c_ptr0]]\n"
                        "ldr q17, [%[c_ptr0], #0x10]\n"
                        "ldr q18, [%[c_ptr0], #0x20]\n"
                        "ldr q19, [%[c_ptr0], #0x30]\n"
                        "ldr q20, [c_ptr1]\n"
                        "ldr q21, [c_ptr1, #0x10]\n"
                        "ldr q22, [c_ptr1, #0x20]\n"
                        "ldr q23, [c_ptr1, #0x30]\n"
                        "ldr q24, [c_ptr2]\n"
                        "ldr q25, [c_ptr2, #0x10]\n"
                        "ldr q26, [c_ptr2, #0x20]\n"
                        "ldr q27, [c_ptr2, #0x30]\n"
                        "ldr q0, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ldr q1, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "ldr q2, [a_ptr2]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x80\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        "ldr q15, [%[b_ptr0], #-0x10]\n"
                        ".inst 0x4f81e114 // sdot v20.4s, v8.16b, v1.4b[0]\n"
                        "ldr q4, [%[a_ptr0]]\n"
                        ".inst 0x4f82e118 // sdot v24.4s, v8.16b, v2.4b[0]\n"
                        "ldr q5, [a_ptr1]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        "ldr q6, [a_ptr2]\n"
                        ".inst 0x4f81e135 // sdot v21.4s, v9.16b, v1.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f82e139 // sdot v25.4s, v9.16b, v2.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        ".inst 0x4f81e156 // sdot v22.4s, v10.16b, v1.4b[0]\n"
                        "prfm PLDL1KEEP, [%[a_ptr0], #0x40]\n"
                        ".inst 0x4f82e15a // sdot v26.4s, v10.16b, v2.4b[0]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x4f81e177 // sdot v23.4s, v11.16b, v1.4b[0]\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        ".inst 0x4f82e17b // sdot v27.4s, v11.16b, v2.4b[0]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa0e190 // sdot v16.4s, v12.16b, v0.4b[1]\n"
                        "add a_ptr2, a_ptr2, #0x20\n"
                        ".inst 0x4fa1e194 // sdot v20.4s, v12.16b, v1.4b[1]\n"
                        "prfm PLDL1KEEP, [a_ptr1, #0x40]\n"
                        ".inst 0x4fa2e198 // sdot v24.4s, v12.16b, v2.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa0e1b1 // sdot v17.4s, v13.16b, v0.4b[1]\n"
                        "prfm PLDL1KEEP, [a_ptr2, #0x40]\n"
                        ".inst 0x4fa1e1b5 // sdot v21.4s, v13.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e1b9 // sdot v25.4s, v13.16b, v2.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4fa0e1d2 // sdot v18.4s, v14.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1d6 // sdot v22.4s, v14.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e1da // sdot v26.4s, v14.16b, v2.4b[1]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4fa0e1f3 // sdot v19.4s, v15.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1f7 // sdot v23.4s, v15.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e1fb // sdot v27.4s, v15.16b, v2.4b[1]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f80e910 // sdot v16.4s, v8.16b, v0.4b[2]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x100\n"
                        ".inst 0x4f81e914 // sdot v20.4s, v8.16b, v1.4b[2]\n"
                        ".inst 0x4f82e918 // sdot v24.4s, v8.16b, v2.4b[2]\n"
                        "ldr q8, [%[b_ptr0], #-0x80]\n"
                        ".inst 0x4f80e931 // sdot v17.4s, v9.16b, v0.4b[2]\n"
                        ".inst 0x4f81e935 // sdot v21.4s, v9.16b, v1.4b[2]\n"
                        ".inst 0x4f82e939 // sdot v25.4s, v9.16b, v2.4b[2]\n"
                        "ldr q9, [%[b_ptr0], #-0x70]\n"
                        ".inst 0x4f80e952 // sdot v18.4s, v10.16b, v0.4b[2]\n"
                        ".inst 0x4f81e956 // sdot v22.4s, v10.16b, v1.4b[2]\n"
                        ".inst 0x4f82e95a // sdot v26.4s, v10.16b, v2.4b[2]\n"
                        "ldr q10, [%[b_ptr0], #-0x60]\n"
                        ".inst 0x4f80e973 // sdot v19.4s, v11.16b, v0.4b[2]\n"
                        ".inst 0x4f81e977 // sdot v23.4s, v11.16b, v1.4b[2]\n"
                        ".inst 0x4f82e97b // sdot v27.4s, v11.16b, v2.4b[2]\n"
                        "ldr q11, [%[b_ptr0], #-0x50]\n"
                        ".inst 0x4fa0e990 // sdot v16.4s, v12.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e994 // sdot v20.4s, v12.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e998 // sdot v24.4s, v12.16b, v2.4b[3]\n"
                        "ldr q12, [%[b_ptr0], #-0x40]\n"
                        ".inst 0x4fa0e9b1 // sdot v17.4s, v13.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9b5 // sdot v21.4s, v13.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e9b9 // sdot v25.4s, v13.16b, v2.4b[3]\n"
                        "ldr q13, [%[b_ptr0], #-0x30]\n"
                        ".inst 0x4fa0e9d2 // sdot v18.4s, v14.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9d6 // sdot v22.4s, v14.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e9da // sdot v26.4s, v14.16b, v2.4b[3]\n"
                        "ldr q14, [%[b_ptr0], #-0x20]\n"
                        ".inst 0x4fa0e9f3 // sdot v19.4s, v15.16b, v0.4b[3]\n"
                        "ldr q0, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x4fa1e9f7 // sdot v23.4s, v15.16b, v1.4b[3]\n"
                        "ldr q1, [a_ptr1, #-0x10]\n"
                        ".inst 0x4fa2e9fb // sdot v27.4s, v15.16b, v2.4b[3]\n"
                        "ldr q15, [%[b_ptr0], #-0x10]\n"
                        ".inst 0x4f84e110 // sdot v16.4s, v8.16b, v4.4b[0]\n"
                        "ldr q2, [a_ptr2, #-0x10]\n"
                        ".inst 0x4f85e114 // sdot v20.4s, v8.16b, v5.4b[0]\n"
                        ".inst 0x4f86e118 // sdot v24.4s, v8.16b, v6.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f84e131 // sdot v17.4s, v9.16b, v4.4b[0]\n"
                        ".inst 0x4f85e135 // sdot v21.4s, v9.16b, v5.4b[0]\n"
                        ".inst 0x4f86e139 // sdot v25.4s, v9.16b, v6.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4f84e152 // sdot v18.4s, v10.16b, v4.4b[0]\n"
                        ".inst 0x4f85e156 // sdot v22.4s, v10.16b, v5.4b[0]\n"
                        ".inst 0x4f86e15a // sdot v26.4s, v10.16b, v6.4b[0]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4f84e173 // sdot v19.4s, v11.16b, v4.4b[0]\n"
                        ".inst 0x4f85e177 // sdot v23.4s, v11.16b, v5.4b[0]\n"
                        ".inst 0x4f86e17b // sdot v27.4s, v11.16b, v6.4b[0]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa4e190 // sdot v16.4s, v12.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e194 // sdot v20.4s, v12.16b, v5.4b[1]\n"
                        ".inst 0x4fa6e198 // sdot v24.4s, v12.16b, v6.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa4e1b1 // sdot v17.4s, v13.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e1b5 // sdot v21.4s, v13.16b, v5.4b[1]\n"
                        ".inst 0x4fa6e1b9 // sdot v25.4s, v13.16b, v6.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4fa4e1d2 // sdot v18.4s, v14.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e1d6 // sdot v22.4s, v14.16b, v5.4b[1]\n"
                        ".inst 0x4fa6e1da // sdot v26.4s, v14.16b, v6.4b[1]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4fa4e1f3 // sdot v19.4s, v15.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e1f7 // sdot v23.4s, v15.16b, v5.4b[1]\n"
                        ".inst 0x4fa6e1fb // sdot v27.4s, v15.16b, v6.4b[1]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f84e910 // sdot v16.4s, v8.16b, v4.4b[2]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x100\n"
                        ".inst 0x4f85e914 // sdot v20.4s, v8.16b, v5.4b[2]\n"
                        ".inst 0x4f86e918 // sdot v24.4s, v8.16b, v6.4b[2]\n"
                        "ldr q8, [%[b_ptr0], #-0x80]\n"
                        ".inst 0x4f84e931 // sdot v17.4s, v9.16b, v4.4b[2]\n"
                        ".inst 0x4f85e935 // sdot v21.4s, v9.16b, v5.4b[2]\n"
                        ".inst 0x4f86e939 // sdot v25.4s, v9.16b, v6.4b[2]\n"
                        "ldr q9, [%[b_ptr0], #-0x70]\n"
                        ".inst 0x4f84e952 // sdot v18.4s, v10.16b, v4.4b[2]\n"
                        ".inst 0x4f85e956 // sdot v22.4s, v10.16b, v5.4b[2]\n"
                        ".inst 0x4f86e95a // sdot v26.4s, v10.16b, v6.4b[2]\n"
                        "ldr q10, [%[b_ptr0], #-0x60]\n"
                        ".inst 0x4f84e973 // sdot v19.4s, v11.16b, v4.4b[2]\n"
                        ".inst 0x4f85e977 // sdot v23.4s, v11.16b, v5.4b[2]\n"
                        ".inst 0x4f86e97b // sdot v27.4s, v11.16b, v6.4b[2]\n"
                        "ldr q11, [%[b_ptr0], #-0x50]\n"
                        ".inst 0x4fa4e990 // sdot v16.4s, v12.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e994 // sdot v20.4s, v12.16b, v5.4b[3]\n"
                        ".inst 0x4fa6e998 // sdot v24.4s, v12.16b, v6.4b[3]\n"
                        "ldr q12, [%[b_ptr0], #-0x40]\n"
                        ".inst 0x4fa4e9b1 // sdot v17.4s, v13.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e9b5 // sdot v21.4s, v13.16b, v5.4b[3]\n"
                        ".inst 0x4fa6e9b9 // sdot v25.4s, v13.16b, v6.4b[3]\n"
                        "ldr q13, [%[b_ptr0], #-0x30]\n"
                        ".inst 0x4fa4e9d2 // sdot v18.4s, v14.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e9d6 // sdot v22.4s, v14.16b, v5.4b[3]\n"
                        ".inst 0x4fa6e9da // sdot v26.4s, v14.16b, v6.4b[3]\n"
                        "ldr q14, [%[b_ptr0], #-0x20]\n"
                        ".inst 0x4fa4e9f3 // sdot v19.4s, v15.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e9f7 // sdot v23.4s, v15.16b, v5.4b[3]\n"
                        ".inst 0x4fa6e9fb // sdot v27.4s, v15.16b, v6.4b[3]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "ldr q15, [%[b_ptr0], #-0x10]\n"
                        "prfm PSTL1KEEP, [%[c_ptr0]]\n"
                        "prfm PSTL1KEEP, [c_ptr1]\n"
                        "prfm PSTL1KEEP, [c_ptr2]\n"
                        "cbz %[regs], 4f\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        "ldr q4, [%[a_ptr0]]\n"
                        ".inst 0x4f81e114 // sdot v20.4s, v8.16b, v1.4b[0]\n"
                        "ldr q5, [a_ptr1]\n"
                        ".inst 0x4f82e118 // sdot v24.4s, v8.16b, v2.4b[0]\n"
                        "ldr q6, [a_ptr2]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f81e135 // sdot v21.4s, v9.16b, v1.4b[0]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        ".inst 0x4f82e139 // sdot v25.4s, v9.16b, v2.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        ".inst 0x4f81e156 // sdot v22.4s, v10.16b, v1.4b[0]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        ".inst 0x4f82e15a // sdot v26.4s, v10.16b, v2.4b[0]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        ".inst 0x4f81e177 // sdot v23.4s, v11.16b, v1.4b[0]\n"
                        ".inst 0x4f82e17b // sdot v27.4s, v11.16b, v2.4b[0]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa0e190 // sdot v16.4s, v12.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e194 // sdot v20.4s, v12.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e198 // sdot v24.4s, v12.16b, v2.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa0e1b1 // sdot v17.4s, v13.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1b5 // sdot v21.4s, v13.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e1b9 // sdot v25.4s, v13.16b, v2.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4fa0e1d2 // sdot v18.4s, v14.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1d6 // sdot v22.4s, v14.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e1da // sdot v26.4s, v14.16b, v2.4b[1]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4fa0e1f3 // sdot v19.4s, v15.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1f7 // sdot v23.4s, v15.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e1fb // sdot v27.4s, v15.16b, v2.4b[1]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f80e910 // sdot v16.4s, v8.16b, v0.4b[2]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x100\n"
                        ".inst 0x4f81e914 // sdot v20.4s, v8.16b, v1.4b[2]\n"
                        ".inst 0x4f82e918 // sdot v24.4s, v8.16b, v2.4b[2]\n"
                        "ldr q8, [%[b_ptr0], #-0x80]\n"
                        ".inst 0x4f80e931 // sdot v17.4s, v9.16b, v0.4b[2]\n"
                        ".inst 0x4f81e935 // sdot v21.4s, v9.16b, v1.4b[2]\n"
                        ".inst 0x4f82e939 // sdot v25.4s, v9.16b, v2.4b[2]\n"
                        "ldr q9, [%[b_ptr0], #-0x70]\n"
                        ".inst 0x4f80e952 // sdot v18.4s, v10.16b, v0.4b[2]\n"
                        ".inst 0x4f81e956 // sdot v22.4s, v10.16b, v1.4b[2]\n"
                        ".inst 0x4f82e95a // sdot v26.4s, v10.16b, v2.4b[2]\n"
                        "ldr q10, [%[b_ptr0], #-0x60]\n"
                        ".inst 0x4f80e973 // sdot v19.4s, v11.16b, v0.4b[2]\n"
                        ".inst 0x4f81e977 // sdot v23.4s, v11.16b, v1.4b[2]\n"
                        ".inst 0x4f82e97b // sdot v27.4s, v11.16b, v2.4b[2]\n"
                        "ldr q11, [%[b_ptr0], #-0x50]\n"
                        ".inst 0x4fa0e990 // sdot v16.4s, v12.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e994 // sdot v20.4s, v12.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e998 // sdot v24.4s, v12.16b, v2.4b[3]\n"
                        "ldr q12, [%[b_ptr0], #-0x40]\n"
                        ".inst 0x4fa0e9b1 // sdot v17.4s, v13.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9b5 // sdot v21.4s, v13.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e9b9 // sdot v25.4s, v13.16b, v2.4b[3]\n"
                        "ldr q13, [%[b_ptr0], #-0x30]\n"
                        ".inst 0x4fa0e9d2 // sdot v18.4s, v14.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9d6 // sdot v22.4s, v14.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e9da // sdot v26.4s, v14.16b, v2.4b[3]\n"
                        "ldr q14, [%[b_ptr0], #-0x20]\n"
                        ".inst 0x4fa0e9f3 // sdot v19.4s, v15.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9f7 // sdot v23.4s, v15.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e9fb // sdot v27.4s, v15.16b, v2.4b[3]\n"
                        "ldr q15, [%[b_ptr0], #-0x10]\n"
                        ".inst 0x4f84e110 // sdot v16.4s, v8.16b, v4.4b[0]\n"
                        ".inst 0x4f85e114 // sdot v20.4s, v8.16b, v5.4b[0]\n"
                        ".inst 0x4f86e118 // sdot v24.4s, v8.16b, v6.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f84e131 // sdot v17.4s, v9.16b, v4.4b[0]\n"
                        ".inst 0x4f85e135 // sdot v21.4s, v9.16b, v5.4b[0]\n"
                        ".inst 0x4f86e139 // sdot v25.4s, v9.16b, v6.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4f84e152 // sdot v18.4s, v10.16b, v4.4b[0]\n"
                        ".inst 0x4f85e156 // sdot v22.4s, v10.16b, v5.4b[0]\n"
                        ".inst 0x4f86e15a // sdot v26.4s, v10.16b, v6.4b[0]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4f84e173 // sdot v19.4s, v11.16b, v4.4b[0]\n"
                        ".inst 0x4f85e177 // sdot v23.4s, v11.16b, v5.4b[0]\n"
                        ".inst 0x4f86e17b // sdot v27.4s, v11.16b, v6.4b[0]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa4e190 // sdot v16.4s, v12.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e194 // sdot v20.4s, v12.16b, v5.4b[1]\n"
                        ".inst 0x4fa6e198 // sdot v24.4s, v12.16b, v6.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa4e1b1 // sdot v17.4s, v13.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e1b5 // sdot v21.4s, v13.16b, v5.4b[1]\n"
                        ".inst 0x4fa6e1b9 // sdot v25.4s, v13.16b, v6.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4fa4e1d2 // sdot v18.4s, v14.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e1d6 // sdot v22.4s, v14.16b, v5.4b[1]\n"
                        ".inst 0x4fa6e1da // sdot v26.4s, v14.16b, v6.4b[1]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4fa4e1f3 // sdot v19.4s, v15.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e1f7 // sdot v23.4s, v15.16b, v5.4b[1]\n"
                        ".inst 0x4fa6e1fb // sdot v27.4s, v15.16b, v6.4b[1]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f84e910 // sdot v16.4s, v8.16b, v4.4b[2]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x80\n"
                        ".inst 0x4f85e914 // sdot v20.4s, v8.16b, v5.4b[2]\n"
                        ".inst 0x4f86e918 // sdot v24.4s, v8.16b, v6.4b[2]\n"
                        ".inst 0x4f84e931 // sdot v17.4s, v9.16b, v4.4b[2]\n"
                        ".inst 0x4f85e935 // sdot v21.4s, v9.16b, v5.4b[2]\n"
                        ".inst 0x4f86e939 // sdot v25.4s, v9.16b, v6.4b[2]\n"
                        ".inst 0x4f84e952 // sdot v18.4s, v10.16b, v4.4b[2]\n"
                        ".inst 0x4f85e956 // sdot v22.4s, v10.16b, v5.4b[2]\n"
                        ".inst 0x4f86e95a // sdot v26.4s, v10.16b, v6.4b[2]\n"
                        ".inst 0x4f84e973 // sdot v19.4s, v11.16b, v4.4b[2]\n"
                        ".inst 0x4f85e977 // sdot v23.4s, v11.16b, v5.4b[2]\n"
                        ".inst 0x4f86e97b // sdot v27.4s, v11.16b, v6.4b[2]\n"
                        ".inst 0x4fa4e990 // sdot v16.4s, v12.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e994 // sdot v20.4s, v12.16b, v5.4b[3]\n"
                        ".inst 0x4fa6e998 // sdot v24.4s, v12.16b, v6.4b[3]\n"
                        ".inst 0x4fa4e9b1 // sdot v17.4s, v13.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e9b5 // sdot v21.4s, v13.16b, v5.4b[3]\n"
                        ".inst 0x4fa6e9b9 // sdot v25.4s, v13.16b, v6.4b[3]\n"
                        ".inst 0x4fa4e9d2 // sdot v18.4s, v14.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e9d6 // sdot v22.4s, v14.16b, v5.4b[3]\n"
                        ".inst 0x4fa6e9da // sdot v26.4s, v14.16b, v6.4b[3]\n"
                        ".inst 0x4fa4e9f3 // sdot v19.4s, v15.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e9f7 // sdot v23.4s, v15.16b, v5.4b[3]\n"
                        ".inst 0x4fa6e9fb // sdot v27.4s, v15.16b, v6.4b[3]\n"
                        "b 5f\n"
                        "4:\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        ".inst 0x4f81e114 // sdot v20.4s, v8.16b, v1.4b[0]\n"
                        ".inst 0x4f82e118 // sdot v24.4s, v8.16b, v2.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        ".inst 0x4f81e135 // sdot v21.4s, v9.16b, v1.4b[0]\n"
                        ".inst 0x4f82e139 // sdot v25.4s, v9.16b, v2.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        ".inst 0x4f81e156 // sdot v22.4s, v10.16b, v1.4b[0]\n"
                        ".inst 0x4f82e15a // sdot v26.4s, v10.16b, v2.4b[0]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        ".inst 0x4f81e177 // sdot v23.4s, v11.16b, v1.4b[0]\n"
                        ".inst 0x4f82e17b // sdot v27.4s, v11.16b, v2.4b[0]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa0e190 // sdot v16.4s, v12.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e194 // sdot v20.4s, v12.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e198 // sdot v24.4s, v12.16b, v2.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa0e1b1 // sdot v17.4s, v13.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1b5 // sdot v21.4s, v13.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e1b9 // sdot v25.4s, v13.16b, v2.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4fa0e1d2 // sdot v18.4s, v14.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1d6 // sdot v22.4s, v14.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e1da // sdot v26.4s, v14.16b, v2.4b[1]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4fa0e1f3 // sdot v19.4s, v15.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1f7 // sdot v23.4s, v15.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e1fb // sdot v27.4s, v15.16b, v2.4b[1]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f80e910 // sdot v16.4s, v8.16b, v0.4b[2]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x80\n"
                        ".inst 0x4f81e914 // sdot v20.4s, v8.16b, v1.4b[2]\n"
                        ".inst 0x4f82e918 // sdot v24.4s, v8.16b, v2.4b[2]\n"
                        ".inst 0x4f80e931 // sdot v17.4s, v9.16b, v0.4b[2]\n"
                        ".inst 0x4f81e935 // sdot v21.4s, v9.16b, v1.4b[2]\n"
                        ".inst 0x4f82e939 // sdot v25.4s, v9.16b, v2.4b[2]\n"
                        ".inst 0x4f80e952 // sdot v18.4s, v10.16b, v0.4b[2]\n"
                        ".inst 0x4f81e956 // sdot v22.4s, v10.16b, v1.4b[2]\n"
                        ".inst 0x4f82e95a // sdot v26.4s, v10.16b, v2.4b[2]\n"
                        ".inst 0x4f80e973 // sdot v19.4s, v11.16b, v0.4b[2]\n"
                        ".inst 0x4f81e977 // sdot v23.4s, v11.16b, v1.4b[2]\n"
                        ".inst 0x4f82e97b // sdot v27.4s, v11.16b, v2.4b[2]\n"
                        ".inst 0x4fa0e990 // sdot v16.4s, v12.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e994 // sdot v20.4s, v12.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e998 // sdot v24.4s, v12.16b, v2.4b[3]\n"
                        ".inst 0x4fa0e9b1 // sdot v17.4s, v13.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9b5 // sdot v21.4s, v13.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e9b9 // sdot v25.4s, v13.16b, v2.4b[3]\n"
                        ".inst 0x4fa0e9d2 // sdot v18.4s, v14.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9d6 // sdot v22.4s, v14.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e9da // sdot v26.4s, v14.16b, v2.4b[3]\n"
                        ".inst 0x4fa0e9f3 // sdot v19.4s, v15.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9f7 // sdot v23.4s, v15.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e9fb // sdot v27.4s, v15.16b, v2.4b[3]\n"
                        "5:\n"
                        "cbz %[blocks], 6f\n"
                        "7:\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        "ldr s0, [%[a_ptr0]]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x4\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x40\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        "ldr s1, [a_ptr1]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        "add a_ptr1, a_ptr1, #0x4\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        "ldr s2, [a_ptr2]\n"
                        ".inst 0x4f81e114 // sdot v20.4s, v8.16b, v1.4b[0]\n"
                        "add a_ptr2, a_ptr2, #0x4\n"
                        ".inst 0x4f81e135 // sdot v21.4s, v9.16b, v1.4b[0]\n"
                        ".inst 0x4f82e118 // sdot v24.4s, v8.16b, v2.4b[0]\n"
                        ".inst 0x4f82e139 // sdot v25.4s, v9.16b, v2.4b[0]\n"
                        ".inst 0x4f81e156 // sdot v22.4s, v10.16b, v1.4b[0]\n"
                        ".inst 0x4f82e15a // sdot v26.4s, v10.16b, v2.4b[0]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        ".inst 0x4f81e177 // sdot v23.4s, v11.16b, v1.4b[0]\n"
                        ".inst 0x4f82e17b // sdot v27.4s, v11.16b, v2.4b[0]\n"
                        "b.ne 7b\n"
                        "6:\n"
                        "cbz %[odds], 8f\n"
                        "ld1 {v0.b}[0], [%[a_ptr0]], #1\n"
                        "ld1 {v1.b}[0], [a_ptr1], #1\n"
                        "ld1 {v2.b}[0], [a_ptr2], #1\n"
                        "subs %[odds], %[odds], #0x1\n"
                        "b.eq 9f\n"
                        "ld1 {v0.b}[1], [%[a_ptr0]], #1\n"
                        "ld1 {v1.b}[1], [a_ptr1], #1\n"
                        "ld1 {v2.b}[1], [a_ptr2], #1\n"
                        "subs %[odds], %[odds], #0x1\n"
                        "b.eq 9f\n"
                        "ld1 {v0.b}[2], [%[a_ptr0]]\n"
                        "ld1 {v1.b}[2], [a_ptr1]\n"
                        "ld1 {v2.b}[2], [a_ptr2]\n"
                        "9:\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        ".inst 0x4f81e114 // sdot v20.4s, v8.16b, v1.4b[0]\n"
                        ".inst 0x4f82e118 // sdot v24.4s, v8.16b, v2.4b[0]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        ".inst 0x4f81e135 // sdot v21.4s, v9.16b, v1.4b[0]\n"
                        ".inst 0x4f82e139 // sdot v25.4s, v9.16b, v2.4b[0]\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        ".inst 0x4f81e156 // sdot v22.4s, v10.16b, v1.4b[0]\n"
                        ".inst 0x4f82e15a // sdot v26.4s, v10.16b, v2.4b[0]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        ".inst 0x4f81e177 // sdot v23.4s, v11.16b, v1.4b[0]\n"
                        ".inst 0x4f82e17b // sdot v27.4s, v11.16b, v2.4b[0]\n"
                        "8:\n"
                        "str q16, [%[c_ptr0]]\n"
                        "str q17, [%[c_ptr0], #0x10]\n"
                        "str q18, [%[c_ptr0], #0x20]\n"
                        "str q19, [%[c_ptr0], #0x30]\n"
                        "add %[c_ptr0], %[c_ptr0], #0x40\n"
                        "str q20, [c_ptr1]\n"
                        "str q21, [c_ptr1, #0x10]\n"
                        "str q22, [c_ptr1, #0x20]\n"
                        "str q23, [c_ptr1, #0x30]\n"
                        "str q24, [c_ptr2]\n"
                        "str q25, [c_ptr2, #0x10]\n"
                        "str q26, [c_ptr2, #0x20]\n"
                        "str q27, [c_ptr2, #0x30]\n"
                        ".unreq a_ptr1\n"
                        ".unreq a_ptr2\n"
                        ".unreq c_ptr1\n"
                        ".unreq c_ptr2\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [blocks] "+r" (blocks), [odds] "+r" (odds)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb)
                        : "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x0", "x1", "x2", "x3", "cc", "memory"
                    );
                    break;
                default:
                case 4:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "a_ptr2 .req X1\n"
                        "a_ptr3 .req X2\n"
                        "c_ptr1 .req X3\n"
                        "c_ptr2 .req X4\n"
                        "c_ptr3 .req X5\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "add a_ptr2, a_ptr1, %[lda]\n"
                        "add c_ptr2, c_ptr1, %[ldc]\n"
                        "add a_ptr3, a_ptr2, %[lda]\n"
                        "add c_ptr3, c_ptr2, %[ldc]\n"
                        "cbnz %[accumulate], 1f\n"
                        "movi v16.4s, #0\n"
                        "ldr q0, [%[a_ptr0]]\n"
                        "movi v17.4s, #0\n"
                        "ldr q1, [a_ptr1]\n"
                        "movi v18.4s, #0\n"
                        "ldr q2, [a_ptr2]\n"
                        "movi v19.4s, #0\n"
                        "ldr q3, [a_ptr3]\n"
                        "movi v20.4s, #0\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        "movi v21.4s, #0\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        "movi v22.4s, #0\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        "movi v23.4s, #0\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        "movi v24.4s, #0\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        "movi v25.4s, #0\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        "movi v26.4s, #0\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        "movi v27.4s, #0\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "movi v28.4s, #0\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "movi v29.4s, #0\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "movi v30.4s, #0\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "movi v31.4s, #0\n"
                        "add %[b_ptr0], %[b_ptr0], #0x80\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ldr q16, [%[c_ptr0]]\n"
                        "ldr q17, [%[c_ptr0], #0x10]\n"
                        "ldr q18, [%[c_ptr0], #0x20]\n"
                        "ldr q19, [%[c_ptr0], #0x30]\n"
                        "ldr q20, [c_ptr1]\n"
                        "ldr q21, [c_ptr1, #0x10]\n"
                        "ldr q22, [c_ptr1, #0x20]\n"
                        "ldr q23, [c_ptr1, #0x30]\n"
                        "ldr q24, [c_ptr2]\n"
                        "ldr q25, [c_ptr2, #0x10]\n"
                        "ldr q26, [c_ptr2, #0x20]\n"
                        "ldr q27, [c_ptr2, #0x30]\n"
                        "ldr q28, [c_ptr3]\n"
                        "ldr q29, [c_ptr3, #0x10]\n"
                        "ldr q30, [c_ptr3, #0x20]\n"
                        "ldr q31, [c_ptr3, #0x30]\n"
                        "ldr q0, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ldr q1, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "ldr q2, [a_ptr2]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "ldr q3, [a_ptr3]\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x80\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        "ldr q15, [%[b_ptr0], #-0x10]\n"
                        ".inst 0x4f81e114 // sdot v20.4s, v8.16b, v1.4b[0]\n"
                        "ldr q4, [%[a_ptr0]]\n"
                        ".inst 0x4f82e118 // sdot v24.4s, v8.16b, v2.4b[0]\n"
                        "ldr q5, [a_ptr1]\n"
                        ".inst 0x4f83e11c // sdot v28.4s, v8.16b, v3.4b[0]\n"
                        "ldr q6, [a_ptr2]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        "ldr q7, [a_ptr3]\n"
                        ".inst 0x4f81e135 // sdot v21.4s, v9.16b, v1.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f82e139 // sdot v25.4s, v9.16b, v2.4b[0]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        ".inst 0x4f83e13d // sdot v29.4s, v9.16b, v3.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        "prfm PLDL1KEEP, [%[a_ptr0], #0x40]\n"
                        ".inst 0x4f81e156 // sdot v22.4s, v10.16b, v1.4b[0]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x4f82e15a // sdot v26.4s, v10.16b, v2.4b[0]\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        ".inst 0x4f83e15e // sdot v30.4s, v10.16b, v3.4b[0]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        "add a_ptr2, a_ptr2, #0x20\n"
                        ".inst 0x4f81e177 // sdot v23.4s, v11.16b, v1.4b[0]\n"
                        "add a_ptr3, a_ptr3, #0x20\n"
                        ".inst 0x4f82e17b // sdot v27.4s, v11.16b, v2.4b[0]\n"
                        "prfm PLDL1KEEP, [a_ptr1, #0x40]\n"
                        ".inst 0x4f83e17f // sdot v31.4s, v11.16b, v3.4b[0]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa0e190 // sdot v16.4s, v12.16b, v0.4b[1]\n"
                        "prfm PLDL1KEEP, [a_ptr2, #0x40]\n"
                        ".inst 0x4fa1e194 // sdot v20.4s, v12.16b, v1.4b[1]\n"
                        "prfm PLDL1KEEP, [a_ptr3, #0x40]\n"
                        ".inst 0x4fa2e198 // sdot v24.4s, v12.16b, v2.4b[1]\n"
                        ".inst 0x4fa3e19c // sdot v28.4s, v12.16b, v3.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa0e1b1 // sdot v17.4s, v13.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1b5 // sdot v21.4s, v13.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e1b9 // sdot v25.4s, v13.16b, v2.4b[1]\n"
                        ".inst 0x4fa3e1bd // sdot v29.4s, v13.16b, v3.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4fa0e1d2 // sdot v18.4s, v14.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1d6 // sdot v22.4s, v14.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e1da // sdot v26.4s, v14.16b, v2.4b[1]\n"
                        ".inst 0x4fa3e1de // sdot v30.4s, v14.16b, v3.4b[1]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4fa0e1f3 // sdot v19.4s, v15.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1f7 // sdot v23.4s, v15.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e1fb // sdot v27.4s, v15.16b, v2.4b[1]\n"
                        ".inst 0x4fa3e1ff // sdot v31.4s, v15.16b, v3.4b[1]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f80e910 // sdot v16.4s, v8.16b, v0.4b[2]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x100\n"
                        ".inst 0x4f81e914 // sdot v20.4s, v8.16b, v1.4b[2]\n"
                        ".inst 0x4f82e918 // sdot v24.4s, v8.16b, v2.4b[2]\n"
                        ".inst 0x4f83e91c // sdot v28.4s, v8.16b, v3.4b[2]\n"
                        "ldr q8, [%[b_ptr0], #-0x80]\n"
                        ".inst 0x4f80e931 // sdot v17.4s, v9.16b, v0.4b[2]\n"
                        ".inst 0x4f81e935 // sdot v21.4s, v9.16b, v1.4b[2]\n"
                        ".inst 0x4f82e939 // sdot v25.4s, v9.16b, v2.4b[2]\n"
                        ".inst 0x4f83e93d // sdot v29.4s, v9.16b, v3.4b[2]\n"
                        "ldr q9, [%[b_ptr0], #-0x70]\n"
                        ".inst 0x4f80e952 // sdot v18.4s, v10.16b, v0.4b[2]\n"
                        ".inst 0x4f81e956 // sdot v22.4s, v10.16b, v1.4b[2]\n"
                        ".inst 0x4f82e95a // sdot v26.4s, v10.16b, v2.4b[2]\n"
                        ".inst 0x4f83e95e // sdot v30.4s, v10.16b, v3.4b[2]\n"
                        "ldr q10, [%[b_ptr0], #-0x60]\n"
                        ".inst 0x4f80e973 // sdot v19.4s, v11.16b, v0.4b[2]\n"
                        ".inst 0x4f81e977 // sdot v23.4s, v11.16b, v1.4b[2]\n"
                        ".inst 0x4f82e97b // sdot v27.4s, v11.16b, v2.4b[2]\n"
                        ".inst 0x4f83e97f // sdot v31.4s, v11.16b, v3.4b[2]\n"
                        "ldr q11, [%[b_ptr0], #-0x50]\n"
                        ".inst 0x4fa0e990 // sdot v16.4s, v12.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e994 // sdot v20.4s, v12.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e998 // sdot v24.4s, v12.16b, v2.4b[3]\n"
                        ".inst 0x4fa3e99c // sdot v28.4s, v12.16b, v3.4b[3]\n"
                        "ldr q12, [%[b_ptr0], #-0x40]\n"
                        ".inst 0x4fa0e9b1 // sdot v17.4s, v13.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9b5 // sdot v21.4s, v13.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e9b9 // sdot v25.4s, v13.16b, v2.4b[3]\n"
                        ".inst 0x4fa3e9bd // sdot v29.4s, v13.16b, v3.4b[3]\n"
                        "ldr q13, [%[b_ptr0], #-0x30]\n"
                        ".inst 0x4fa0e9d2 // sdot v18.4s, v14.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9d6 // sdot v22.4s, v14.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e9da // sdot v26.4s, v14.16b, v2.4b[3]\n"
                        ".inst 0x4fa3e9de // sdot v30.4s, v14.16b, v3.4b[3]\n"
                        "ldr q14, [%[b_ptr0], #-0x20]\n"
                        ".inst 0x4fa0e9f3 // sdot v19.4s, v15.16b, v0.4b[3]\n"
                        "ldr q0, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x4fa1e9f7 // sdot v23.4s, v15.16b, v1.4b[3]\n"
                        "ldr q1, [a_ptr1, #-0x10]\n"
                        ".inst 0x4fa2e9fb // sdot v27.4s, v15.16b, v2.4b[3]\n"
                        "ldr q2, [a_ptr2, #-0x10]\n"
                        ".inst 0x4fa3e9ff // sdot v31.4s, v15.16b, v3.4b[3]\n"
                        "ldr q15, [%[b_ptr0], #-0x10]\n"
                        ".inst 0x4f84e110 // sdot v16.4s, v8.16b, v4.4b[0]\n"
                        "ldr q3, [a_ptr3, #-0x10]\n"
                        ".inst 0x4f85e114 // sdot v20.4s, v8.16b, v5.4b[0]\n"
                        ".inst 0x4f86e118 // sdot v24.4s, v8.16b, v6.4b[0]\n"
                        ".inst 0x4f87e11c // sdot v28.4s, v8.16b, v7.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f84e131 // sdot v17.4s, v9.16b, v4.4b[0]\n"
                        ".inst 0x4f85e135 // sdot v21.4s, v9.16b, v5.4b[0]\n"
                        ".inst 0x4f86e139 // sdot v25.4s, v9.16b, v6.4b[0]\n"
                        ".inst 0x4f87e13d // sdot v29.4s, v9.16b, v7.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4f84e152 // sdot v18.4s, v10.16b, v4.4b[0]\n"
                        ".inst 0x4f85e156 // sdot v22.4s, v10.16b, v5.4b[0]\n"
                        ".inst 0x4f86e15a // sdot v26.4s, v10.16b, v6.4b[0]\n"
                        ".inst 0x4f87e15e // sdot v30.4s, v10.16b, v7.4b[0]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4f84e173 // sdot v19.4s, v11.16b, v4.4b[0]\n"
                        ".inst 0x4f85e177 // sdot v23.4s, v11.16b, v5.4b[0]\n"
                        ".inst 0x4f86e17b // sdot v27.4s, v11.16b, v6.4b[0]\n"
                        ".inst 0x4f87e17f // sdot v31.4s, v11.16b, v7.4b[0]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa4e190 // sdot v16.4s, v12.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e194 // sdot v20.4s, v12.16b, v5.4b[1]\n"
                        ".inst 0x4fa6e198 // sdot v24.4s, v12.16b, v6.4b[1]\n"
                        ".inst 0x4fa7e19c // sdot v28.4s, v12.16b, v7.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa4e1b1 // sdot v17.4s, v13.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e1b5 // sdot v21.4s, v13.16b, v5.4b[1]\n"
                        ".inst 0x4fa6e1b9 // sdot v25.4s, v13.16b, v6.4b[1]\n"
                        ".inst 0x4fa7e1bd // sdot v29.4s, v13.16b, v7.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4fa4e1d2 // sdot v18.4s, v14.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e1d6 // sdot v22.4s, v14.16b, v5.4b[1]\n"
                        ".inst 0x4fa6e1da // sdot v26.4s, v14.16b, v6.4b[1]\n"
                        ".inst 0x4fa7e1de // sdot v30.4s, v14.16b, v7.4b[1]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4fa4e1f3 // sdot v19.4s, v15.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e1f7 // sdot v23.4s, v15.16b, v5.4b[1]\n"
                        ".inst 0x4fa6e1fb // sdot v27.4s, v15.16b, v6.4b[1]\n"
                        ".inst 0x4fa7e1ff // sdot v31.4s, v15.16b, v7.4b[1]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f84e910 // sdot v16.4s, v8.16b, v4.4b[2]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x100\n"
                        ".inst 0x4f85e914 // sdot v20.4s, v8.16b, v5.4b[2]\n"
                        ".inst 0x4f86e918 // sdot v24.4s, v8.16b, v6.4b[2]\n"
                        ".inst 0x4f87e91c // sdot v28.4s, v8.16b, v7.4b[2]\n"
                        "ldr q8, [%[b_ptr0], #-0x80]\n"
                        ".inst 0x4f84e931 // sdot v17.4s, v9.16b, v4.4b[2]\n"
                        ".inst 0x4f85e935 // sdot v21.4s, v9.16b, v5.4b[2]\n"
                        ".inst 0x4f86e939 // sdot v25.4s, v9.16b, v6.4b[2]\n"
                        ".inst 0x4f87e93d // sdot v29.4s, v9.16b, v7.4b[2]\n"
                        "ldr q9, [%[b_ptr0], #-0x70]\n"
                        ".inst 0x4f84e952 // sdot v18.4s, v10.16b, v4.4b[2]\n"
                        ".inst 0x4f85e956 // sdot v22.4s, v10.16b, v5.4b[2]\n"
                        ".inst 0x4f86e95a // sdot v26.4s, v10.16b, v6.4b[2]\n"
                        ".inst 0x4f87e95e // sdot v30.4s, v10.16b, v7.4b[2]\n"
                        "ldr q10, [%[b_ptr0], #-0x60]\n"
                        ".inst 0x4f84e973 // sdot v19.4s, v11.16b, v4.4b[2]\n"
                        ".inst 0x4f85e977 // sdot v23.4s, v11.16b, v5.4b[2]\n"
                        ".inst 0x4f86e97b // sdot v27.4s, v11.16b, v6.4b[2]\n"
                        ".inst 0x4f87e97f // sdot v31.4s, v11.16b, v7.4b[2]\n"
                        "ldr q11, [%[b_ptr0], #-0x50]\n"
                        ".inst 0x4fa4e990 // sdot v16.4s, v12.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e994 // sdot v20.4s, v12.16b, v5.4b[3]\n"
                        ".inst 0x4fa6e998 // sdot v24.4s, v12.16b, v6.4b[3]\n"
                        ".inst 0x4fa7e99c // sdot v28.4s, v12.16b, v7.4b[3]\n"
                        "ldr q12, [%[b_ptr0], #-0x40]\n"
                        ".inst 0x4fa4e9b1 // sdot v17.4s, v13.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e9b5 // sdot v21.4s, v13.16b, v5.4b[3]\n"
                        ".inst 0x4fa6e9b9 // sdot v25.4s, v13.16b, v6.4b[3]\n"
                        ".inst 0x4fa7e9bd // sdot v29.4s, v13.16b, v7.4b[3]\n"
                        "ldr q13, [%[b_ptr0], #-0x30]\n"
                        ".inst 0x4fa4e9d2 // sdot v18.4s, v14.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e9d6 // sdot v22.4s, v14.16b, v5.4b[3]\n"
                        ".inst 0x4fa6e9da // sdot v26.4s, v14.16b, v6.4b[3]\n"
                        ".inst 0x4fa7e9de // sdot v30.4s, v14.16b, v7.4b[3]\n"
                        "ldr q14, [%[b_ptr0], #-0x20]\n"
                        ".inst 0x4fa4e9f3 // sdot v19.4s, v15.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e9f7 // sdot v23.4s, v15.16b, v5.4b[3]\n"
                        ".inst 0x4fa6e9fb // sdot v27.4s, v15.16b, v6.4b[3]\n"
                        ".inst 0x4fa7e9ff // sdot v31.4s, v15.16b, v7.4b[3]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "ldr q15, [%[b_ptr0], #-0x10]\n"
                        "prfm PSTL1KEEP, [%[c_ptr0]]\n"
                        "prfm PSTL1KEEP, [c_ptr1]\n"
                        "prfm PSTL1KEEP, [c_ptr2]\n"
                        "prfm PSTL1KEEP, [c_ptr3]\n"
                        "cbz %[regs], 4f\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        "ldr q4, [%[a_ptr0]]\n"
                        ".inst 0x4f81e114 // sdot v20.4s, v8.16b, v1.4b[0]\n"
                        "ldr q5, [a_ptr1]\n"
                        ".inst 0x4f82e118 // sdot v24.4s, v8.16b, v2.4b[0]\n"
                        "ldr q6, [a_ptr2]\n"
                        ".inst 0x4f83e11c // sdot v28.4s, v8.16b, v3.4b[0]\n"
                        "ldr q7, [a_ptr3]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f81e135 // sdot v21.4s, v9.16b, v1.4b[0]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        ".inst 0x4f82e139 // sdot v25.4s, v9.16b, v2.4b[0]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        ".inst 0x4f83e13d // sdot v29.4s, v9.16b, v3.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        ".inst 0x4f81e156 // sdot v22.4s, v10.16b, v1.4b[0]\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        ".inst 0x4f82e15a // sdot v26.4s, v10.16b, v2.4b[0]\n"
                        ".inst 0x4f83e15e // sdot v30.4s, v10.16b, v3.4b[0]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        ".inst 0x4f81e177 // sdot v23.4s, v11.16b, v1.4b[0]\n"
                        ".inst 0x4f82e17b // sdot v27.4s, v11.16b, v2.4b[0]\n"
                        ".inst 0x4f83e17f // sdot v31.4s, v11.16b, v3.4b[0]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa0e190 // sdot v16.4s, v12.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e194 // sdot v20.4s, v12.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e198 // sdot v24.4s, v12.16b, v2.4b[1]\n"
                        ".inst 0x4fa3e19c // sdot v28.4s, v12.16b, v3.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa0e1b1 // sdot v17.4s, v13.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1b5 // sdot v21.4s, v13.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e1b9 // sdot v25.4s, v13.16b, v2.4b[1]\n"
                        ".inst 0x4fa3e1bd // sdot v29.4s, v13.16b, v3.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4fa0e1d2 // sdot v18.4s, v14.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1d6 // sdot v22.4s, v14.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e1da // sdot v26.4s, v14.16b, v2.4b[1]\n"
                        ".inst 0x4fa3e1de // sdot v30.4s, v14.16b, v3.4b[1]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4fa0e1f3 // sdot v19.4s, v15.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1f7 // sdot v23.4s, v15.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e1fb // sdot v27.4s, v15.16b, v2.4b[1]\n"
                        ".inst 0x4fa3e1ff // sdot v31.4s, v15.16b, v3.4b[1]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f80e910 // sdot v16.4s, v8.16b, v0.4b[2]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x100\n"
                        ".inst 0x4f81e914 // sdot v20.4s, v8.16b, v1.4b[2]\n"
                        ".inst 0x4f82e918 // sdot v24.4s, v8.16b, v2.4b[2]\n"
                        ".inst 0x4f83e91c // sdot v28.4s, v8.16b, v3.4b[2]\n"
                        "ldr q8, [%[b_ptr0], #-0x80]\n"
                        ".inst 0x4f80e931 // sdot v17.4s, v9.16b, v0.4b[2]\n"
                        ".inst 0x4f81e935 // sdot v21.4s, v9.16b, v1.4b[2]\n"
                        ".inst 0x4f82e939 // sdot v25.4s, v9.16b, v2.4b[2]\n"
                        ".inst 0x4f83e93d // sdot v29.4s, v9.16b, v3.4b[2]\n"
                        "ldr q9, [%[b_ptr0], #-0x70]\n"
                        ".inst 0x4f80e952 // sdot v18.4s, v10.16b, v0.4b[2]\n"
                        ".inst 0x4f81e956 // sdot v22.4s, v10.16b, v1.4b[2]\n"
                        ".inst 0x4f82e95a // sdot v26.4s, v10.16b, v2.4b[2]\n"
                        ".inst 0x4f83e95e // sdot v30.4s, v10.16b, v3.4b[2]\n"
                        "ldr q10, [%[b_ptr0], #-0x60]\n"
                        ".inst 0x4f80e973 // sdot v19.4s, v11.16b, v0.4b[2]\n"
                        ".inst 0x4f81e977 // sdot v23.4s, v11.16b, v1.4b[2]\n"
                        ".inst 0x4f82e97b // sdot v27.4s, v11.16b, v2.4b[2]\n"
                        ".inst 0x4f83e97f // sdot v31.4s, v11.16b, v3.4b[2]\n"
                        "ldr q11, [%[b_ptr0], #-0x50]\n"
                        ".inst 0x4fa0e990 // sdot v16.4s, v12.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e994 // sdot v20.4s, v12.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e998 // sdot v24.4s, v12.16b, v2.4b[3]\n"
                        ".inst 0x4fa3e99c // sdot v28.4s, v12.16b, v3.4b[3]\n"
                        "ldr q12, [%[b_ptr0], #-0x40]\n"
                        ".inst 0x4fa0e9b1 // sdot v17.4s, v13.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9b5 // sdot v21.4s, v13.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e9b9 // sdot v25.4s, v13.16b, v2.4b[3]\n"
                        ".inst 0x4fa3e9bd // sdot v29.4s, v13.16b, v3.4b[3]\n"
                        "ldr q13, [%[b_ptr0], #-0x30]\n"
                        ".inst 0x4fa0e9d2 // sdot v18.4s, v14.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9d6 // sdot v22.4s, v14.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e9da // sdot v26.4s, v14.16b, v2.4b[3]\n"
                        ".inst 0x4fa3e9de // sdot v30.4s, v14.16b, v3.4b[3]\n"
                        "ldr q14, [%[b_ptr0], #-0x20]\n"
                        ".inst 0x4fa0e9f3 // sdot v19.4s, v15.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9f7 // sdot v23.4s, v15.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e9fb // sdot v27.4s, v15.16b, v2.4b[3]\n"
                        ".inst 0x4fa3e9ff // sdot v31.4s, v15.16b, v3.4b[3]\n"
                        "ldr q15, [%[b_ptr0], #-0x10]\n"
                        ".inst 0x4f84e110 // sdot v16.4s, v8.16b, v4.4b[0]\n"
                        ".inst 0x4f85e114 // sdot v20.4s, v8.16b, v5.4b[0]\n"
                        ".inst 0x4f86e118 // sdot v24.4s, v8.16b, v6.4b[0]\n"
                        ".inst 0x4f87e11c // sdot v28.4s, v8.16b, v7.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f84e131 // sdot v17.4s, v9.16b, v4.4b[0]\n"
                        ".inst 0x4f85e135 // sdot v21.4s, v9.16b, v5.4b[0]\n"
                        ".inst 0x4f86e139 // sdot v25.4s, v9.16b, v6.4b[0]\n"
                        ".inst 0x4f87e13d // sdot v29.4s, v9.16b, v7.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4f84e152 // sdot v18.4s, v10.16b, v4.4b[0]\n"
                        ".inst 0x4f85e156 // sdot v22.4s, v10.16b, v5.4b[0]\n"
                        ".inst 0x4f86e15a // sdot v26.4s, v10.16b, v6.4b[0]\n"
                        ".inst 0x4f87e15e // sdot v30.4s, v10.16b, v7.4b[0]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4f84e173 // sdot v19.4s, v11.16b, v4.4b[0]\n"
                        ".inst 0x4f85e177 // sdot v23.4s, v11.16b, v5.4b[0]\n"
                        ".inst 0x4f86e17b // sdot v27.4s, v11.16b, v6.4b[0]\n"
                        ".inst 0x4f87e17f // sdot v31.4s, v11.16b, v7.4b[0]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa4e190 // sdot v16.4s, v12.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e194 // sdot v20.4s, v12.16b, v5.4b[1]\n"
                        ".inst 0x4fa6e198 // sdot v24.4s, v12.16b, v6.4b[1]\n"
                        ".inst 0x4fa7e19c // sdot v28.4s, v12.16b, v7.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa4e1b1 // sdot v17.4s, v13.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e1b5 // sdot v21.4s, v13.16b, v5.4b[1]\n"
                        ".inst 0x4fa6e1b9 // sdot v25.4s, v13.16b, v6.4b[1]\n"
                        ".inst 0x4fa7e1bd // sdot v29.4s, v13.16b, v7.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4fa4e1d2 // sdot v18.4s, v14.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e1d6 // sdot v22.4s, v14.16b, v5.4b[1]\n"
                        ".inst 0x4fa6e1da // sdot v26.4s, v14.16b, v6.4b[1]\n"
                        ".inst 0x4fa7e1de // sdot v30.4s, v14.16b, v7.4b[1]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4fa4e1f3 // sdot v19.4s, v15.16b, v4.4b[1]\n"
                        ".inst 0x4fa5e1f7 // sdot v23.4s, v15.16b, v5.4b[1]\n"
                        ".inst 0x4fa6e1fb // sdot v27.4s, v15.16b, v6.4b[1]\n"
                        ".inst 0x4fa7e1ff // sdot v31.4s, v15.16b, v7.4b[1]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f84e910 // sdot v16.4s, v8.16b, v4.4b[2]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x80\n"
                        ".inst 0x4f85e914 // sdot v20.4s, v8.16b, v5.4b[2]\n"
                        ".inst 0x4f86e918 // sdot v24.4s, v8.16b, v6.4b[2]\n"
                        ".inst 0x4f87e91c // sdot v28.4s, v8.16b, v7.4b[2]\n"
                        ".inst 0x4f84e931 // sdot v17.4s, v9.16b, v4.4b[2]\n"
                        ".inst 0x4f85e935 // sdot v21.4s, v9.16b, v5.4b[2]\n"
                        ".inst 0x4f86e939 // sdot v25.4s, v9.16b, v6.4b[2]\n"
                        ".inst 0x4f87e93d // sdot v29.4s, v9.16b, v7.4b[2]\n"
                        ".inst 0x4f84e952 // sdot v18.4s, v10.16b, v4.4b[2]\n"
                        ".inst 0x4f85e956 // sdot v22.4s, v10.16b, v5.4b[2]\n"
                        ".inst 0x4f86e95a // sdot v26.4s, v10.16b, v6.4b[2]\n"
                        ".inst 0x4f87e95e // sdot v30.4s, v10.16b, v7.4b[2]\n"
                        ".inst 0x4f84e973 // sdot v19.4s, v11.16b, v4.4b[2]\n"
                        ".inst 0x4f85e977 // sdot v23.4s, v11.16b, v5.4b[2]\n"
                        ".inst 0x4f86e97b // sdot v27.4s, v11.16b, v6.4b[2]\n"
                        ".inst 0x4f87e97f // sdot v31.4s, v11.16b, v7.4b[2]\n"
                        ".inst 0x4fa4e990 // sdot v16.4s, v12.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e994 // sdot v20.4s, v12.16b, v5.4b[3]\n"
                        ".inst 0x4fa6e998 // sdot v24.4s, v12.16b, v6.4b[3]\n"
                        ".inst 0x4fa7e99c // sdot v28.4s, v12.16b, v7.4b[3]\n"
                        ".inst 0x4fa4e9b1 // sdot v17.4s, v13.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e9b5 // sdot v21.4s, v13.16b, v5.4b[3]\n"
                        ".inst 0x4fa6e9b9 // sdot v25.4s, v13.16b, v6.4b[3]\n"
                        ".inst 0x4fa7e9bd // sdot v29.4s, v13.16b, v7.4b[3]\n"
                        ".inst 0x4fa4e9d2 // sdot v18.4s, v14.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e9d6 // sdot v22.4s, v14.16b, v5.4b[3]\n"
                        ".inst 0x4fa6e9da // sdot v26.4s, v14.16b, v6.4b[3]\n"
                        ".inst 0x4fa7e9de // sdot v30.4s, v14.16b, v7.4b[3]\n"
                        ".inst 0x4fa4e9f3 // sdot v19.4s, v15.16b, v4.4b[3]\n"
                        ".inst 0x4fa5e9f7 // sdot v23.4s, v15.16b, v5.4b[3]\n"
                        ".inst 0x4fa6e9fb // sdot v27.4s, v15.16b, v6.4b[3]\n"
                        ".inst 0x4fa7e9ff // sdot v31.4s, v15.16b, v7.4b[3]\n"
                        "b 5f\n"
                        "4:\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        ".inst 0x4f81e114 // sdot v20.4s, v8.16b, v1.4b[0]\n"
                        ".inst 0x4f82e118 // sdot v24.4s, v8.16b, v2.4b[0]\n"
                        ".inst 0x4f83e11c // sdot v28.4s, v8.16b, v3.4b[0]\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        ".inst 0x4f81e135 // sdot v21.4s, v9.16b, v1.4b[0]\n"
                        ".inst 0x4f82e139 // sdot v25.4s, v9.16b, v2.4b[0]\n"
                        ".inst 0x4f83e13d // sdot v29.4s, v9.16b, v3.4b[0]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        ".inst 0x4f81e156 // sdot v22.4s, v10.16b, v1.4b[0]\n"
                        ".inst 0x4f82e15a // sdot v26.4s, v10.16b, v2.4b[0]\n"
                        ".inst 0x4f83e15e // sdot v30.4s, v10.16b, v3.4b[0]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        ".inst 0x4f81e177 // sdot v23.4s, v11.16b, v1.4b[0]\n"
                        ".inst 0x4f82e17b // sdot v27.4s, v11.16b, v2.4b[0]\n"
                        ".inst 0x4f83e17f // sdot v31.4s, v11.16b, v3.4b[0]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4fa0e190 // sdot v16.4s, v12.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e194 // sdot v20.4s, v12.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e198 // sdot v24.4s, v12.16b, v2.4b[1]\n"
                        ".inst 0x4fa3e19c // sdot v28.4s, v12.16b, v3.4b[1]\n"
                        "ldr q12, [%[b_ptr0], #0x40]\n"
                        ".inst 0x4fa0e1b1 // sdot v17.4s, v13.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1b5 // sdot v21.4s, v13.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e1b9 // sdot v25.4s, v13.16b, v2.4b[1]\n"
                        ".inst 0x4fa3e1bd // sdot v29.4s, v13.16b, v3.4b[1]\n"
                        "ldr q13, [%[b_ptr0], #0x50]\n"
                        ".inst 0x4fa0e1d2 // sdot v18.4s, v14.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1d6 // sdot v22.4s, v14.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e1da // sdot v26.4s, v14.16b, v2.4b[1]\n"
                        ".inst 0x4fa3e1de // sdot v30.4s, v14.16b, v3.4b[1]\n"
                        "ldr q14, [%[b_ptr0], #0x60]\n"
                        ".inst 0x4fa0e1f3 // sdot v19.4s, v15.16b, v0.4b[1]\n"
                        ".inst 0x4fa1e1f7 // sdot v23.4s, v15.16b, v1.4b[1]\n"
                        ".inst 0x4fa2e1fb // sdot v27.4s, v15.16b, v2.4b[1]\n"
                        ".inst 0x4fa3e1ff // sdot v31.4s, v15.16b, v3.4b[1]\n"
                        "ldr q15, [%[b_ptr0], #0x70]\n"
                        ".inst 0x4f80e910 // sdot v16.4s, v8.16b, v0.4b[2]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x80\n"
                        ".inst 0x4f81e914 // sdot v20.4s, v8.16b, v1.4b[2]\n"
                        ".inst 0x4f82e918 // sdot v24.4s, v8.16b, v2.4b[2]\n"
                        ".inst 0x4f83e91c // sdot v28.4s, v8.16b, v3.4b[2]\n"
                        ".inst 0x4f80e931 // sdot v17.4s, v9.16b, v0.4b[2]\n"
                        ".inst 0x4f81e935 // sdot v21.4s, v9.16b, v1.4b[2]\n"
                        ".inst 0x4f82e939 // sdot v25.4s, v9.16b, v2.4b[2]\n"
                        ".inst 0x4f83e93d // sdot v29.4s, v9.16b, v3.4b[2]\n"
                        ".inst 0x4f80e952 // sdot v18.4s, v10.16b, v0.4b[2]\n"
                        ".inst 0x4f81e956 // sdot v22.4s, v10.16b, v1.4b[2]\n"
                        ".inst 0x4f82e95a // sdot v26.4s, v10.16b, v2.4b[2]\n"
                        ".inst 0x4f83e95e // sdot v30.4s, v10.16b, v3.4b[2]\n"
                        ".inst 0x4f80e973 // sdot v19.4s, v11.16b, v0.4b[2]\n"
                        ".inst 0x4f81e977 // sdot v23.4s, v11.16b, v1.4b[2]\n"
                        ".inst 0x4f82e97b // sdot v27.4s, v11.16b, v2.4b[2]\n"
                        ".inst 0x4f83e97f // sdot v31.4s, v11.16b, v3.4b[2]\n"
                        ".inst 0x4fa0e990 // sdot v16.4s, v12.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e994 // sdot v20.4s, v12.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e998 // sdot v24.4s, v12.16b, v2.4b[3]\n"
                        ".inst 0x4fa3e99c // sdot v28.4s, v12.16b, v3.4b[3]\n"
                        ".inst 0x4fa0e9b1 // sdot v17.4s, v13.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9b5 // sdot v21.4s, v13.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e9b9 // sdot v25.4s, v13.16b, v2.4b[3]\n"
                        ".inst 0x4fa3e9bd // sdot v29.4s, v13.16b, v3.4b[3]\n"
                        ".inst 0x4fa0e9d2 // sdot v18.4s, v14.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9d6 // sdot v22.4s, v14.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e9da // sdot v26.4s, v14.16b, v2.4b[3]\n"
                        ".inst 0x4fa3e9de // sdot v30.4s, v14.16b, v3.4b[3]\n"
                        ".inst 0x4fa0e9f3 // sdot v19.4s, v15.16b, v0.4b[3]\n"
                        ".inst 0x4fa1e9f7 // sdot v23.4s, v15.16b, v1.4b[3]\n"
                        ".inst 0x4fa2e9fb // sdot v27.4s, v15.16b, v2.4b[3]\n"
                        ".inst 0x4fa3e9ff // sdot v31.4s, v15.16b, v3.4b[3]\n"
                        "5:\n"
                        "cbz %[blocks], 6f\n"
                        "7:\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        "ldr s0, [%[a_ptr0]]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x4\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        "add %[b_ptr0], %[b_ptr0], #0x40\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        "ldr s1, [a_ptr1]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        "add a_ptr1, a_ptr1, #0x4\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        "ldr s2, [a_ptr2]\n"
                        ".inst 0x4f81e114 // sdot v20.4s, v8.16b, v1.4b[0]\n"
                        "add a_ptr2, a_ptr2, #0x4\n"
                        ".inst 0x4f81e135 // sdot v21.4s, v9.16b, v1.4b[0]\n"
                        "ldr s3, [a_ptr3]\n"
                        ".inst 0x4f82e118 // sdot v24.4s, v8.16b, v2.4b[0]\n"
                        "add a_ptr3, a_ptr3, #0x4\n"
                        ".inst 0x4f82e139 // sdot v25.4s, v9.16b, v2.4b[0]\n"
                        ".inst 0x4f83e11c // sdot v28.4s, v8.16b, v3.4b[0]\n"
                        ".inst 0x4f83e13d // sdot v29.4s, v9.16b, v3.4b[0]\n"
                        ".inst 0x4f81e156 // sdot v22.4s, v10.16b, v1.4b[0]\n"
                        ".inst 0x4f82e15a // sdot v26.4s, v10.16b, v2.4b[0]\n"
                        ".inst 0x4f83e15e // sdot v30.4s, v10.16b, v3.4b[0]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        ".inst 0x4f81e177 // sdot v23.4s, v11.16b, v1.4b[0]\n"
                        ".inst 0x4f82e17b // sdot v27.4s, v11.16b, v2.4b[0]\n"
                        ".inst 0x4f83e17f // sdot v31.4s, v11.16b, v3.4b[0]\n"
                        "b.ne 7b\n"
                        "6:\n"
                        "cbz %[odds], 8f\n"
                        "ld1 {v0.b}[0], [%[a_ptr0]], #1\n"
                        "ld1 {v1.b}[0], [a_ptr1], #1\n"
                        "ld1 {v2.b}[0], [a_ptr2], #1\n"
                        "ld1 {v3.b}[0], [a_ptr3], #1\n"
                        "subs %[odds], %[odds], #0x1\n"
                        "b.eq 9f\n"
                        "ld1 {v0.b}[1], [%[a_ptr0]], #1\n"
                        "ld1 {v1.b}[1], [a_ptr1], #1\n"
                        "ld1 {v2.b}[1], [a_ptr2], #1\n"
                        "ld1 {v3.b}[1], [a_ptr3], #1\n"
                        "subs %[odds], %[odds], #0x1\n"
                        "b.eq 9f\n"
                        "ld1 {v0.b}[2], [%[a_ptr0]]\n"
                        "ld1 {v1.b}[2], [a_ptr1]\n"
                        "ld1 {v2.b}[2], [a_ptr2]\n"
                        "ld1 {v3.b}[2], [a_ptr3]\n"
                        "9:\n"
                        "ldr q8, [%[b_ptr0]]\n"
                        "ldr q9, [%[b_ptr0], #0x10]\n"
                        "ldr q10, [%[b_ptr0], #0x20]\n"
                        "ldr q11, [%[b_ptr0], #0x30]\n"
                        ".inst 0x4f80e110 // sdot v16.4s, v8.16b, v0.4b[0]\n"
                        ".inst 0x4f81e114 // sdot v20.4s, v8.16b, v1.4b[0]\n"
                        ".inst 0x4f82e118 // sdot v24.4s, v8.16b, v2.4b[0]\n"
                        ".inst 0x4f83e11c // sdot v28.4s, v8.16b, v3.4b[0]\n"
                        ".inst 0x4f80e131 // sdot v17.4s, v9.16b, v0.4b[0]\n"
                        ".inst 0x4f81e135 // sdot v21.4s, v9.16b, v1.4b[0]\n"
                        ".inst 0x4f82e139 // sdot v25.4s, v9.16b, v2.4b[0]\n"
                        ".inst 0x4f83e13d // sdot v29.4s, v9.16b, v3.4b[0]\n"
                        ".inst 0x4f80e152 // sdot v18.4s, v10.16b, v0.4b[0]\n"
                        ".inst 0x4f81e156 // sdot v22.4s, v10.16b, v1.4b[0]\n"
                        ".inst 0x4f82e15a // sdot v26.4s, v10.16b, v2.4b[0]\n"
                        ".inst 0x4f83e15e // sdot v30.4s, v10.16b, v3.4b[0]\n"
                        ".inst 0x4f80e173 // sdot v19.4s, v11.16b, v0.4b[0]\n"
                        ".inst 0x4f81e177 // sdot v23.4s, v11.16b, v1.4b[0]\n"
                        ".inst 0x4f82e17b // sdot v27.4s, v11.16b, v2.4b[0]\n"
                        ".inst 0x4f83e17f // sdot v31.4s, v11.16b, v3.4b[0]\n"
                        "8:\n"
                        "str q16, [%[c_ptr0]]\n"
                        "str q17, [%[c_ptr0], #0x10]\n"
                        "str q18, [%[c_ptr0], #0x20]\n"
                        "str q19, [%[c_ptr0], #0x30]\n"
                        "add %[c_ptr0], %[c_ptr0], #0x40\n"
                        "str q20, [c_ptr1]\n"
                        "str q21, [c_ptr1, #0x10]\n"
                        "str q22, [c_ptr1, #0x20]\n"
                        "str q23, [c_ptr1, #0x30]\n"
                        "str q24, [c_ptr2]\n"
                        "str q25, [c_ptr2, #0x10]\n"
                        "str q26, [c_ptr2, #0x20]\n"
                        "str q27, [c_ptr2, #0x30]\n"
                        "str q28, [c_ptr3]\n"
                        "str q29, [c_ptr3, #0x10]\n"
                        "str q30, [c_ptr3, #0x20]\n"
                        "str q31, [c_ptr3, #0x30]\n"
                        ".unreq a_ptr1\n"
                        ".unreq a_ptr2\n"
                        ".unreq a_ptr3\n"
                        ".unreq c_ptr1\n"
                        ".unreq c_ptr2\n"
                        ".unreq c_ptr3\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [blocks] "+r" (blocks), [odds] "+r" (odds)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb)
                        : "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x0", "x1", "x2", "x3", "x4", "x5", "cc", "memory"
                    );
                    break;
            }
            if (use_result_buffer) {
                for(int cy=0; cy<std::min(M-y, 4); cy++) {
                    for(unsigned int cx=0; cx<width; cx++) {
                        c_ptr_real[cy * ldc + cx] = result_buffer[cy * 16 + cx];
                    }
                }
            }
        }
    }
}

} // namespace arm_gemm

#endif // __aarch64__
