
/******************************************************************************
 *
 *  This file is part of seqrequester, a tool for summarizing, extracting,
 *  generating and modifying DNA sequences.
 *
 *  This software is based on:
 *    'Canu' v2.0              (https://github.com/marbl/canu)
 *  which is based on:
 *    'Celera Assembler' r4587 (http://wgs-assembler.sourceforge.net)
 *    the 'kmer package' r1994 (http://kmer.sourceforge.net)
 *
 *  Except as indicated otherwise, this is a 'United States Government Work',
 *  and is released in the public domain.
 *
 *  File 'README.licenses' in the root directory of this distribution
 *  contains full conditions and disclaimers.
 */

#include "seqrequester.H"
#include "files.H"
#include "mt19937ar.H"



void
doGenerate(generateParameters &genPar) {
  mtRandom   MT;

  uint64  nSeqs  = 0;
  uint64  nBases = 0;

  uint64  seqLen = 0;
  uint64  seqMax = 65536;
  char   *seq    = new char  [seqMax + 1];
  uint8  *qlt    = new uint8 [seqMax + 1];

  double Athresh = genPar.aFreq;
  double Cthresh = genPar.aFreq + genPar.cFreq;
  double Gthresh = genPar.aFreq + genPar.cFreq + genPar.gFreq;
  double Tthresh = genPar.aFreq + genPar.cFreq + genPar.gFreq + genPar.tFreq;

  if (genPar.randomSeedValid)
    MT.mtSetSeed(genPar.randomSeed);

  if (genPar.ident == nullptr)
    genPar.ident = duplicateString("random%08lu");

  while ((nSeqs  < genPar.nSeqs) &&
         (nBases < genPar.nBases)) {
    double   len = MT.mtRandomGaussian(genPar.gMean, genPar.gStdDev);

    while (len < -0.5)
      len = MT.mtRandomGaussian(genPar.gMean, genPar.gStdDev);

    seqLen = (uint64)round(len);

    if (seqLen+1 >= seqMax)
      resizeArrayPair(seq, qlt, 0, seqMax, seqLen+1);

    for (uint64 ii=0; ii<seqLen; ii++) {
      double  bp = MT.mtRandomRealOpen();

      if        (bp < Athresh) {
        seq[ii] = 'A';
        qlt[ii] = 0;

      } else if (bp < Cthresh) {
        seq[ii] = 'C';
        qlt[ii] = 0;

      } else if (bp < Gthresh) {
        seq[ii] = 'G';
        qlt[ii] = 0;

      } else {
        seq[ii] = 'T';
        qlt[ii] = 0;
      }
    }

    seq[seqLen] = 0;
    qlt[seqLen] = 0;

    outputFASTA(stdout, seq, seqLen, 0, genPar.ident, nSeqs);

    nSeqs  += 1;
    nBases += seqLen;
  }

  delete [] seq;
  delete [] qlt;
}

