/*
 *  Copyright (C) 2002-2021  The DOSBox Team
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */


#include "dosbox.h"
#include "logging.h"
#include "mem.h"
#include "cpu.h"
#include "inout.h"
#include "int10.h"
#include "callback.h"
#include "jfont.h"

bool int10_vga_bios_vector = true;
bool rom_bios_8x8_cga_font = true;
bool VGA_BIOS_dont_duplicate_CGA_first_half = false;
bool VIDEO_BIOS_always_carry_14_high_font = true;
bool VIDEO_BIOS_always_carry_16_high_font = true;
bool VIDEO_BIOS_enable_CGA_8x8_second_half = false;
bool VIDEO_BIOS_disable = false;

static uint8_t static_functionality[0x10]=
{
 /* 0 */ 0xff,  // All modes supported #1
 /* 1 */ 0xff,  // All modes supported #2
 /* 2 */ 0x0f,  // All modes supported #3
 /* 3 */ 0x00, 0x00, 0x00, 0x00,  // reserved
 /* 7 */ 0x07,  // 200, 350, 400 scan lines
 /* 8 */ 0x04,  // total number of character blocks available in text modes
 /* 9 */ 0x02,  // maximum number of active character blocks in text modes
 /* a */ 0xff,  // Misc Flags Everything supported 
 /* b */ 0x0e,  // Support for Display combination, intensity/blinking and video state saving/restoring
 /* c */ 0x00,  // reserved
 /* d */ 0x00,  // reserved
 /* e */ 0x00,  // Change to add new functions
 /* f */ 0x00   // reserved
};

static uint16_t map_offset[8]={
	0x0000,0x4000,0x8000,0xc000,
	0x2000,0x6000,0xa000,0xe000
};

void INT10_LoadFont(PhysPt font,bool reload,uint16_t count,Bitu offset,Bitu map,uint8_t height) {
    unsigned char m64k;

	if (IS_VGA_ARCH || (IS_EGA_ARCH && vga.mem.memsize >= 0x20000))
        m64k=0x02;
    else
        m64k = 0x00;

    PhysPt ftwhere = PhysMake(0xa000, map_offset[map & 0x7] + (uint16_t)(offset * 32));
	uint16_t base=real_readw(BIOSMEM_SEG,BIOSMEM_CRTC_ADDRESS);
	bool mono=(base==VGAREG_MDA_CRTC_ADDRESS);
	
	//Put video adapter in planar mode
	IO_Write(0x3c4,0x02);IO_Write(0x3c5,0x04); // select plane 2 for writing
	IO_Write(0x3c4,0x04);IO_Write(0x3c5,0x05|m64k); // odd/even off in SEQ
	IO_Write(0x3ce,0x04);IO_Write(0x3cf,0x02); // select plane 2 for reading
	IO_Write(0x3ce,0x05);IO_Write(0x3cf,0x00); // write mode 0, odd/even off in GFX
	IO_Write(0x3ce,0x06);IO_Write(0x3cf,0x04); // CPU memory window A0000-AFFFF
	
	//Load character patterns
	for (uint16_t i=0;i<count;i++) {
		MEM_BlockCopy(ftwhere+i*32u,font,height);
		font+=height;
	}
	//Load alternate character patterns
	if (map & 0x80) {
		while (Bitu chr=(Bitu)mem_readb(font++)) {
			MEM_BlockCopy(ftwhere+(PhysPt)chr*32u,font,height);
			font+=height;
		}
	}

	//Return to normal text mode
	IO_Write(0x3c4,0x02);IO_Write(0x3c5,0x03); // select planes 0&1 for writing
	IO_Write(0x3c4,0x04);IO_Write(0x3c5,0x01|m64k); // odd/even on in SEQ
	IO_Write(0x3ce,0x04);IO_Write(0x3cf,0x00); // select plane 0 for reading
	IO_Write(0x3ce,0x05);IO_Write(0x3cf,0x10); // write mode 0, odd/even on in GFX
	IO_Write(0x3ce,0x06);IO_Write(0x3cf,mono?0x0a:0x0e); // Bx000-BxFFF, odd/even on

	/* Reload tables and registers with new values based on this height */
	if (reload) {
		//Max scanline 
		IO_Write(base,0x9);
		IO_Write(base+1u,(IO_Read(base+1u) & 0xe0u)|((unsigned int)height-1u));
		//Vertical display end
		Bitu rows=CurMode->sheight/height;
		Bitu vdend=rows*height*((CurMode->sheight==200 && IS_VGA_ARCH)?2:1)-1;
		IO_Write(base,0x12);
		IO_Write(base+1u,(uint8_t)vdend); // FIXME: Does not change overflow bits
		//Underline location
		if (CurMode->mode==7) {
			IO_Write(base,0x14);
			IO_Write(base+1u,(IO_Read(base+1u) & ~0x1fu)|(height-1u));
		}
		//Rows setting in bios segment
		real_writeb(BIOSMEM_SEG,BIOSMEM_NB_ROWS,(uint8_t)(rows-1));
		real_writeb(BIOSMEM_SEG,BIOSMEM_CHAR_HEIGHT,(uint8_t)height);
		//Page size
		Bitu pagesize=rows*real_readb(BIOSMEM_SEG,BIOSMEM_NB_COLS)*2;
		pagesize+=0x100; // bios adds extra on reload
		real_writew(BIOSMEM_SEG,BIOSMEM_PAGE_SIZE,(uint16_t)pagesize);
		//Cursor shape
		if (height>=14) height--; // move up one line on 14+ line fonts
		INT10_SetCursorShape(height-2,height-1);
        //Clip the cursor within range in case it is now beyond the new height. To fix 28.COM and 50.COM (issue #1324 and #1325).
        //FIXME: Remove this code *IF* it turns out BIOSes do not clip the row value when calling INT 10h AX=1111h/1113h
        //       If BIOSes are inconsistent about it, note which ones do and don't and try to match behavior with machine= type.
        //       At the very least, if the IBM PS/2 VGA BIOS does not, then this code should be coded NOT to clip the cursor
        //       when machine=vgaonly.
        {
            uint8_t page=real_readb(BIOSMEM_SEG,BIOSMEM_CURRENT_PAGE);
            uint8_t cur_row=CURSOR_POS_ROW(page);
            uint8_t cur_col=CURSOR_POS_COL(page);

            if (cur_row >= rows)
                INT10_SetCursorPos((uint8_t)(rows-1),cur_col,page);
        }
	}
}

// FIXME: ripsaw8080 noted that while the svgaCard==SVGA_None conditions are appropriate for DOSBox SVN
// because the vgaonly machine type is alone in displaying 9-dot character cells, but in DOSBox-X (as in
// DOSBox SVN Daum) the char9= setting should be respected. As it is now, users will not get the wider
// symbols when char9= is true and machine= is an SVGA machine type.
// SVN commit reference: https://sourceforge.net/p/dosbox/code-0/3932/
void INT10_ReloadFont(void) {
	Bitu map=0;
	switch(CurMode->cheight) {
	case 8:
		INT10_LoadFont(Real2Phys(int10.rom.font_8_first),false,256,0,map,8);
		break;
	case 14:
		if (IS_VGA_ARCH && svgaCard==SVGA_None && CurMode->mode==7) map=0x80;
		INT10_LoadFont(Real2Phys(int10.rom.font_14),false,256,0,map,14);
		break;
	case 19:
		if (IS_EGA_ARCH && svgaCard == SVGA_None) map = 0x80;
		INT10_LoadFont(Real2Phys(int10.rom.font_19), false, 256, 0, map, 19);
		break;
	case 16:
	default:
		if (IS_VGA_ARCH && svgaCard==SVGA_None) map=0x80;
		INT10_LoadFont(Real2Phys(int10.rom.font_16),false,256,0,map,16);
		break;
	}
}

extern Bitu VGA_BIOS_Size;

static Bitu VGA_ROM_BIOS_ENTRY_cb = 0;

Bitu VGA_ROM_BIOS_ENTRY_callback_func(void) {
    LOG_MSG("VGA ROM BIOS init callback");

    reg_eax = 3;		// 80x25 text
    CALLBACK_RunRealInt(0x10);

    return CBRET_NONE;
}

bool MEM_unmap_physmem(Bitu start,Bitu end);

void INT10_RemoveVGABIOS(void) { /* PC-98 does not have VGA BIOS */
    if (VGA_BIOS_Size != 0) {
        for (unsigned int i=0;i < VGA_BIOS_Size;i++)
            phys_writeb(0xC0000+i,0xFF);

        MEM_unmap_physmem(0xC0000,0xC0000+VGA_BIOS_Size-1);
        VGA_BIOS_Size = 0;
    }
}

RealPt GetSystemBiosINT10Vector(void);
Bitu ROMBIOS_GetMemory(Bitu bytes,const char *who,Bitu alignment,Bitu must_be_at);

extern bool VGA_BIOS_use_rom;

void INT10_SetupRomMemory(void) {
	/* if no space allocated for video BIOS (such as machine=cga) then return immediately */
	if (VGA_BIOS_Size == 0) {
		int10.rom.font_8_first=RealMake(0xF000,0xFA6E); /* why duplicate data? use the copy in the ROM BIOS */
		int10.rom.font_8_second=0;
		int10.rom.static_state=0;
		int10.rom.font_14=0;
		int10.rom.font_16=0;

		/* ref: [http://www.ctyme.com/intr/rb-6173.htm] */
		if (IS_TANDY_ARCH)
			RealSetVec(0x44,int10.rom.font_8_first);
		else
			RealSetVec(0x43,int10.rom.font_8_first);

		if (IS_EGAVGA_ARCH || VIDEO_BIOS_enable_CGA_8x8_second_half) {
			Bitu rom = ROMBIOS_GetMemory((8*128),"CGA 8x8 second half",1,0u);
			if (rom == 0) E_Exit("Unable to alloc CGA 8x8 second half");
			int10.rom.font_8_second = RealMake((unsigned int)rom >> 4u,(unsigned int)rom & 0xFu);
			for (unsigned i=0;i<128*8;i++) {
				phys_writeb(rom+i,int10_font_08[i+128*8]);
			}
		}

		RealSetVec(0x1F,int10.rom.font_8_second);

        if (machine == MCH_MCGA) {
            Bitu base = ROMBIOS_GetMemory((Bitu)(256*16),"MCGA 16-line font",1,0u);
            if (base == 0) E_Exit("Unable to alloc MCGA 16x font");

            for (unsigned int i=0;i<256*16;i++)
                phys_writeb((PhysPt)base+i,int10_font_16[i]);

            int10.rom.font_16 = RealMake((uint16_t)(base >> 4u),(uint16_t)(base & 0xF));

            // MCGA has the pointer at 40:A8 (BIOSMEM_VS_POINTER), confirmed on real hardware.
            // It points into the BIOS, because MCGA systems do not have a BIOS at C000:0000
            Bitu vptr = ROMBIOS_GetMemory((Bitu)(0x600),"MCGA video save pointer and structs",1,0u);
            Bitu vptrseg = vptr >> 4;
            Bitu vptroff = vptr & 0xF;
            vptr -= vptroff;
            Bitu vptroff_limit = vptroff + 0x600;

            int10.rom.video_parameter_table=RealMake((uint16_t)vptrseg, (uint16_t)vptroff);
            vptroff+=INT10_SetupVideoParameterTable((PhysPt)(vptr+vptroff));

            // The dynamic save area should be in RAM, it cannot exist in ROM
            int10.rom.video_dynamic_save_area=0;

            int10.rom.video_save_pointers=RealMake((uint16_t)vptrseg, (uint16_t)vptroff);
            phys_writed((PhysPt)(vptr+vptroff),int10.rom.video_parameter_table);
            vptroff+=4;
            phys_writed((PhysPt)(vptr+vptroff),int10.rom.video_dynamic_save_area);		// dynamic save area pointer
            vptroff+=4;
            phys_writed((PhysPt)(vptr+vptroff),0);		// alphanumeric character set override
            vptroff+=4;
            phys_writed((PhysPt)(vptr+vptroff),0);		// graphics character set override
            vptroff+=4;
            phys_writed((PhysPt)(vptr+vptroff),0);		// secondary save pointer table
            vptroff+=4;
            phys_writed((PhysPt)(vptr+vptroff),0); vptroff+=4;
            phys_writed((PhysPt)(vptr+vptroff),0); vptroff+=4;

            if (vptroff > vptroff_limit) E_Exit("MCGA ptr overrun");
        }

		return;
	}

/* This should fill up certain structures inside the Video Bios Rom Area */
	PhysPt rom_base=PhysMake(0xc000,0);
	Bitu i;
	int10.rom.used=3;
	if (IS_EGAVGA_ARCH) {
		// set up the start of the ROM

		// we must make valid boot code at seg:3. return value is callback index
		if (VGA_ROM_BIOS_ENTRY_cb == 0) {
			VGA_ROM_BIOS_ENTRY_cb = CALLBACK_Allocate();
			CALLBACK_Setup(VGA_ROM_BIOS_ENTRY_cb,VGA_ROM_BIOS_ENTRY_callback_func,CB_RETF,"VGA ROM BIOS boot up entry point");
		}

		// ROM signature
		phys_writew(rom_base+0,0xaa55);
		phys_writeb(rom_base+2,(uint8_t)(VGA_BIOS_Size >> 9u));
		// entry point
		phys_writeb(rom_base+3,0xFE); // Callback instruction
		phys_writeb(rom_base+4,0x38);
		phys_writew(rom_base+5,(uint16_t)VGA_ROM_BIOS_ENTRY_cb);
		phys_writeb(rom_base+7,0xCB); // RETF

		// VGA BIOS copyright
		if (IS_VGA_ARCH) {
			if (svgaCard == SVGA_ATI)
				phys_writes(rom_base+0x1e, "IBM", 4/*length of string+NUL*/);
			else
				phys_writes(rom_base+0x1e, "IBM compatible VGA BIOS", 24/*length of string+NUL*/);
		}
		else {
			phys_writes(rom_base+0x1e, "IBM compatible EGA BIOS", 24);
		}

		if (IS_VGA_ARCH) {
			// SVGA card-specific ROM signatures
			switch (svgaCard) {
				case SVGA_S3Trio:
					if(!VGA_BIOS_use_rom) {
						// S3 ROM signature
						phys_writes(rom_base+0x003f, "S3 86C764", 10);
					}
					break;
				case SVGA_TsengET4K:
				case SVGA_TsengET3K:
					if(!VGA_BIOS_use_rom) {
						// Tseng ROM signature
						phys_writes(rom_base+0x0075, " Tseng ", 8);
					}
					break;
				case SVGA_ParadisePVGA1A:
					phys_writeb(rom_base+0x0048,' ');
					phys_writeb(rom_base+0x0049,'W');
					phys_writeb(rom_base+0x004a,'E');
					phys_writeb(rom_base+0x004b,'S');
					phys_writeb(rom_base+0x004c,'T');
					phys_writeb(rom_base+0x004d,'E');
					phys_writeb(rom_base+0x004e,'R');
					phys_writeb(rom_base+0x004f,'N');
					phys_writeb(rom_base+0x0050,' ');
					phys_writeb(rom_base+0x007d,'V');
					phys_writeb(rom_base+0x007e,'G');
					phys_writeb(rom_base+0x007f,'A');
					phys_writeb(rom_base+0x0080,'=');
					break;
				case SVGA_ATI:
					/* Visit the PCem project and browse their ROM collection, examine the ATI BIOS images and
					 * notice they only put "IBM" up there. See also this ATI VGA Wonder XL VGA ROM BIOS image
					 * here: [https://archive.org/details/ati_vgawonder_xl_rom]
					 *
					 * See also this documentation: [http://hackipedia.org/browse.cgi/Computer/Platform/PC%2c%20IBM%20compatible/Video/VGA/SVGA/ATI%2c%20Array%20Technology%20Inc/ATI%20Technologies%20Super%20VGA%20Chipsets%20%28WHATVGA%20v2%2e00%29%20%281995%2d09%2d29%29%2etxt]
					 *
					 * See also the WHATVGA.EXE source code (written in Pascal) */
					/* FIXME: There is a "atimach64vt2pci.bin" image where the signature at 0x31 is "3?" instead of "31" and byte 0x43 is 0x40 instead of 0x20. Why? */
					phys_writew(rom_base+0x10, 0x1CE); /*ATI extended registers I/O base port*/
					phys_writes(rom_base+0x31, "761295520", 9/*length of string*/); /* ATI product ID */
					phys_writes(rom_base+0x40, "31", 2/*length of string*/); /* ATI Wonder/Mach series */
					switch (atiCard) {
						case ATI_EGAVGAWonder: // ATI 18800 EGA/VGA Wonder
							phys_writeb(rom_base+0x43,0x31); // NTS: the 18800-1 uses code 0x32 according to WHATVGA
							break;
						case ATI_VGAWonder: // ATI 28800-1 VGA Wonder
							phys_writeb(rom_base+0x43,0x32); // NTS: 28800-2 is 0x33, what is 28800-1?
							break;
						case ATI_VGAWonderPlus: // ATI 28800-2 VGA Wonder+
							phys_writeb(rom_base+0x43,0x33);
							break;
						case ATI_VGAWonderXL: // ATI 28800-4 VGA WonderXL
							phys_writeb(rom_base+0x43,0x34);
							break;
						case ATI_VGAWonderXL24: // ATI 28800-6 VGA Wonder
							phys_writeb(rom_base+0x43,0x35);
							break;
						case ATI_Mach8: // ATI 38800-1
							phys_writeb(rom_base+0x43,0x38); // FIXME: GUESS because '8' might make sense for Mach8, I dunno
							break;
						case ATI_Mach32: // ATI 68800-3
							phys_writeb(rom_base+0x43,0x61);
							break;
						case ATI_Mach64: // ATI 88800GX
							phys_writeb(rom_base+0x43,0x20); // confirmed by PCem rom image mach64g/bios.bin
							break;
						default:
							break;
					};
					break;
				case SVGA_None:
					break;
			}
		}

		// JMP to INT 10h in the system BIOS.
		//
		// SuperCalc 3 and 4 fails to detect EGA/VGA if INT 10h points at a location higher than segment 0xFE00 in the system BIOS.
		// It will also not attempt EGA/VGA detect unless there is an adapter ROM signature (AA55h) at 0xC000:0x0000.
		// The check does not attempt EGA/VGA detection if ((segment >> 4) + offset) >= 0xFE00.
		//
		// The idea is to point INT 10h at this JMP so that DOS programs like SuperCalc will think there is in fact an EGA/VGA BIOS
		// and that INT 10h is provided by the EGA/VGA BIOS so it can function normally.
		//
		// [https://github.com/joncampbell123/dosbox-x/issues/1473]
		if (int10_vga_bios_vector) {
			const RealPt biosint10 = GetSystemBiosINT10Vector();

			if (biosint10 != 0) {
				LOG(LOG_MISC,LOG_DEBUG)("Redirecting INT 10h to point at the VGA BIOS");

				phys_writeb(rom_base+0xEE,0xEA); // JMP FAR
				phys_writew(rom_base+0xEF,(uint16_t)(biosint10 & 0xFFFFu));
				phys_writew(rom_base+0xF1,(uint16_t)((biosint10 >> 16u) & 0xFFFFu));

				/* WARNING: This overwrites the INT 10 startup code's vector successfully only because this
				 *          code is called AFTER it has initialized the INT 10h vector. If initialization
				 *          order changes this could stop working. */
				RealSetVec(0x10,RealMake(0xC000,0x00EE));
			}
		}

        // and then other data follows
		int10.rom.used=0x100;
	}
	
	if (IS_VGA_ARCH && svgaCard==SVGA_S3Trio) INT10_SetupVESA();
	
	if (VGA_BIOS_dont_duplicate_CGA_first_half) {
		int10.rom.font_8_first=RealMake(0xF000,0xFA6E); /* why duplicate data? use the copy in the ROM BIOS */
	}
	else {
		int10.rom.font_8_first=RealMake(0xC000,int10.rom.used);
		for (i=0;i<128*8;i++) {
			phys_writeb(rom_base+int10.rom.used++,int10_font_08[i]);
		}
	}
	if (IS_EGAVGA_ARCH || VIDEO_BIOS_enable_CGA_8x8_second_half) {
		int10.rom.font_8_second=RealMake(0xC000,int10.rom.used);
		for (i=0;i<128*8;i++) {
			phys_writeb(rom_base+int10.rom.used++,int10_font_08[i+128*8]);
		}
	}
	else {
		int10.rom.font_8_second=0;
	}
	if (IS_EGAVGA_ARCH || VIDEO_BIOS_always_carry_14_high_font) {
		int10.rom.font_14=RealMake(0xC000,int10.rom.used);
		for (i=0;i<256*14;i++) {
			phys_writeb(rom_base+int10.rom.used++,int10_font_14[i]);
		}
		int10.rom.font_14_alternate=RealMake(0xC000,int10.rom.used);
		for (i=0;i<20*15+1;i++) {
			phys_writeb(rom_base+int10.rom.used++,int10_font_14_alternate[i]);
	}
	}
	else {
		int10.rom.font_14=0; /* why write the 14-high version if not emulating EGA/VGA? */
		int10.rom.font_14_alternate=0;
	}	
	if (IS_VGA_ARCH || VIDEO_BIOS_always_carry_16_high_font) {
		int10.rom.font_16=RealMake(0xC000,int10.rom.used);
		for (i=0;i<256*16;i++) {
			phys_writeb(rom_base+int10.rom.used++,int10_font_16[i]);
		}
		int10.rom.font_16_alternate=RealMake(0xC000,int10.rom.used);
		for (i=0;i<19*17+1;i++) {
			phys_writeb(rom_base+int10.rom.used++,int10_font_16_alternate[i]);
		}
	}
    else if (IS_JEGA_ARCH) {
        int10.rom.font_19 = RealMake(0xC000, int10.rom.used);
        for (i = 0; i < 256 * 19; i++)
            phys_writeb(rom_base + int10.rom.used++, jfont_sbcs_19[i]);
    }
	else {
		int10.rom.font_16=0; /* why write the 16-high version if not emulating VGA? */
		int10.rom.font_16_alternate=0;
		/* FIXME: Does the EGA BIOS have the 16-high font, or just 14-high? */
	}
	if (IS_VGA_ARCH) { /* according to sources this static state stuff only applies to VGA, right? */
		int10.rom.static_state=RealMake(0xC000,int10.rom.used);
		for (i=0;i<0x10;i++) {
			phys_writeb(rom_base+int10.rom.used++,static_functionality[i]);
		}
	}
	else {
		int10.rom.static_state=0;
	}
	RealSetVec(0x1F,int10.rom.font_8_second);

	if (IS_EGAVGA_ARCH) {
		int10.rom.video_parameter_table=RealMake(0xC000,int10.rom.used);
		int10.rom.used+=INT10_SetupVideoParameterTable(rom_base+int10.rom.used);

		if (IS_VGA_ARCH) {
			int10.rom.video_dcc_table=RealMake(0xC000,int10.rom.used);
			phys_writeb(rom_base+int10.rom.used++,0x10);	// number of entries
			phys_writeb(rom_base+int10.rom.used++,1);		// version number
			phys_writeb(rom_base+int10.rom.used++,8);		// maximum display code
			phys_writeb(rom_base+int10.rom.used++,0);		// reserved
			// display combination codes
			phys_writew(rom_base+int10.rom.used,0x0000);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0100);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0200);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0102);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0400);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0104);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0500);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0502);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0600);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0601);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0605);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0800);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0801);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0700);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0702);	int10.rom.used+=2;
			phys_writew(rom_base+int10.rom.used,0x0706);	int10.rom.used+=2;

			int10.rom.video_save_pointer_table=RealMake(0xC000,int10.rom.used);
			phys_writew(rom_base+int10.rom.used,0x1a);	// length of table
			int10.rom.used+=2;
			phys_writed(rom_base+int10.rom.used,int10.rom.video_dcc_table);
			int10.rom.used+=4;
			phys_writed(rom_base+int10.rom.used,0);		// alphanumeric charset override
			int10.rom.used+=4;
			phys_writed(rom_base+int10.rom.used,0);		// user palette table
			int10.rom.used+=4;
			phys_writed(rom_base+int10.rom.used,0);		int10.rom.used+=4;
			phys_writed(rom_base+int10.rom.used,0);		int10.rom.used+=4;
			phys_writed(rom_base+int10.rom.used,0);		int10.rom.used+=4;
		}

        // The dynamic save area should be in RAM, it cannot exist in ROM
        int10.rom.video_dynamic_save_area=0;

		int10.rom.video_save_pointers=RealMake(0xC000,int10.rom.used);
		phys_writed(rom_base+int10.rom.used,int10.rom.video_parameter_table);
		int10.rom.used+=4;
		phys_writed(rom_base+int10.rom.used,int10.rom.video_dynamic_save_area);		// dynamic save area pointer
		int10.rom.used+=4;
		phys_writed(rom_base+int10.rom.used,0);		// alphanumeric character set override
		int10.rom.used+=4;
		phys_writed(rom_base+int10.rom.used,0);		// graphics character set override
		int10.rom.used+=4;
		if (IS_VGA_ARCH) {
			phys_writed(rom_base+int10.rom.used,int10.rom.video_save_pointer_table);
		} else {
			phys_writed(rom_base+int10.rom.used,0);		// secondary save pointer table
		}
		int10.rom.used+=4;
		phys_writed(rom_base+int10.rom.used,0);		int10.rom.used+=4;
		phys_writed(rom_base+int10.rom.used,0);		int10.rom.used+=4;
	}
	
	INT10_SetupRomMemoryChecksum();

	if (IS_TANDY_ARCH) {
		RealSetVec(0x44,RealMake(0xf000,0xfa6e));
	}
}

void INT10_ReloadRomFonts(void) {
	// 16x8 font
	PhysPt font16pt=Real2Phys(int10.rom.font_16);
	for (unsigned int i=0;i<256*16;i++) {
		phys_writeb(font16pt+i,int10_font_16[i]);
	}
	phys_writeb(Real2Phys(int10.rom.font_16_alternate),0x1d);
	// 14x8 font
	PhysPt font14pt=Real2Phys(int10.rom.font_14);
	for (unsigned int i=0;i<256*14;i++) {
		phys_writeb(font14pt+i,int10_font_14[i]);
	}
	phys_writeb(Real2Phys(int10.rom.font_14_alternate),0x1d);
	// 8x8 fonts
	PhysPt font8pt=Real2Phys(int10.rom.font_8_first);
	for (unsigned int i=0;i<128*8;i++) {
		phys_writeb(font8pt+i,int10_font_08[i]);
	}
	font8pt=Real2Phys(int10.rom.font_8_second);
	for (unsigned int i=0;i<128*8;i++) {
		phys_writeb(font8pt+i,int10_font_08[i+128*8]);
	}
	if (IS_JEGA_ARCH) {
		PhysPt font19pt = Real2Phys(int10.rom.font_19);
		for (Bitu i = 0; i < 256 * 19; i++) {
			phys_writeb(font19pt + i, jfont_sbcs_19[i]);
		}
	}
	INT10_SetupRomMemoryChecksum();
}

void INT10_SetupRomMemoryChecksum(void) {
	if (IS_EGAVGA_ARCH) { //EGA/VGA. Just to be safe
		/* Sum of all bytes in rom module 256 should be 0 */
		uint8_t sum = 0;
		PhysPt rom_base = PhysMake(0xc000,0);
		unsigned int last_rombyte = (unsigned int)VGA_BIOS_Size - 1;		//32 KB romsize
		for (unsigned int i = 0;i < last_rombyte;i++)
			sum += phys_readb(rom_base + i);	//OVERFLOW IS OKAY
		sum = (uint8_t)((256 - (Bitu)sum)&0xff);
		phys_writeb(rom_base + last_rombyte,sum);
	}
}


uint8_t int10_font_08[256 * 8] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7e, 0x81, 0xa5, 0x81, 0xbd, 0x99, 0x81, 0x7e,
  0x7e, 0xff, 0xdb, 0xff, 0xc3, 0xe7, 0xff, 0x7e,
  0x6c, 0xfe, 0xfe, 0xfe, 0x7c, 0x38, 0x10, 0x00,
  0x10, 0x38, 0x7c, 0xfe, 0x7c, 0x38, 0x10, 0x00,
  0x38, 0x7c, 0x38, 0xfe, 0xfe, 0x7c, 0x38, 0x7c,
  0x10, 0x10, 0x38, 0x7c, 0xfe, 0x7c, 0x38, 0x7c,
  0x00, 0x00, 0x18, 0x3c, 0x3c, 0x18, 0x00, 0x00,
  0xff, 0xff, 0xe7, 0xc3, 0xc3, 0xe7, 0xff, 0xff,
  0x00, 0x3c, 0x66, 0x42, 0x42, 0x66, 0x3c, 0x00,
  0xff, 0xc3, 0x99, 0xbd, 0xbd, 0x99, 0xc3, 0xff,
  0x0f, 0x07, 0x0f, 0x7d, 0xcc, 0xcc, 0xcc, 0x78,
  0x3c, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18,
  0x3f, 0x33, 0x3f, 0x30, 0x30, 0x70, 0xf0, 0xe0,
  0x7f, 0x63, 0x7f, 0x63, 0x63, 0x67, 0xe6, 0xc0,
  0x99, 0x5a, 0x3c, 0xe7, 0xe7, 0x3c, 0x5a, 0x99,
  0x80, 0xe0, 0xf8, 0xfe, 0xf8, 0xe0, 0x80, 0x00,
  0x02, 0x0e, 0x3e, 0xfe, 0x3e, 0x0e, 0x02, 0x00,
  0x18, 0x3c, 0x7e, 0x18, 0x18, 0x7e, 0x3c, 0x18,
  0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x66, 0x00,
  0x7f, 0xdb, 0xdb, 0x7b, 0x1b, 0x1b, 0x1b, 0x00,
  0x3e, 0x63, 0x38, 0x6c, 0x6c, 0x38, 0xcc, 0x78,
  0x00, 0x00, 0x00, 0x00, 0x7e, 0x7e, 0x7e, 0x00,
  0x18, 0x3c, 0x7e, 0x18, 0x7e, 0x3c, 0x18, 0xff,
  0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x00,
  0x00, 0x18, 0x0c, 0xfe, 0x0c, 0x18, 0x00, 0x00,
  0x00, 0x30, 0x60, 0xfe, 0x60, 0x30, 0x00, 0x00,
  0x00, 0x00, 0xc0, 0xc0, 0xc0, 0xfe, 0x00, 0x00,
  0x00, 0x24, 0x66, 0xff, 0x66, 0x24, 0x00, 0x00,
  0x00, 0x18, 0x3c, 0x7e, 0xff, 0xff, 0x00, 0x00,
  0x00, 0xff, 0xff, 0x7e, 0x3c, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x30, 0x78, 0x78, 0x30, 0x30, 0x00, 0x30, 0x00,
  0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x6c, 0x6c, 0xfe, 0x6c, 0xfe, 0x6c, 0x6c, 0x00,
  0x30, 0x7c, 0xc0, 0x78, 0x0c, 0xf8, 0x30, 0x00,
  0x00, 0xc6, 0xcc, 0x18, 0x30, 0x66, 0xc6, 0x00,
  0x38, 0x6c, 0x38, 0x76, 0xdc, 0xcc, 0x76, 0x00,
  0x60, 0x60, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x30, 0x60, 0x60, 0x60, 0x30, 0x18, 0x00,
  0x60, 0x30, 0x18, 0x18, 0x18, 0x30, 0x60, 0x00,
  0x00, 0x66, 0x3c, 0xff, 0x3c, 0x66, 0x00, 0x00,
  0x00, 0x30, 0x30, 0xfc, 0x30, 0x30, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x60,
  0x00, 0x00, 0x00, 0xfc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00,
  0x06, 0x0c, 0x18, 0x30, 0x60, 0xc0, 0x80, 0x00,
  0x7c, 0xc6, 0xce, 0xde, 0xf6, 0xe6, 0x7c, 0x00,
  0x30, 0x70, 0x30, 0x30, 0x30, 0x30, 0xfc, 0x00,
  0x78, 0xcc, 0x0c, 0x38, 0x60, 0xcc, 0xfc, 0x00,
  0x78, 0xcc, 0x0c, 0x38, 0x0c, 0xcc, 0x78, 0x00,
  0x1c, 0x3c, 0x6c, 0xcc, 0xfe, 0x0c, 0x1e, 0x00,
  0xfc, 0xc0, 0xf8, 0x0c, 0x0c, 0xcc, 0x78, 0x00,
  0x38, 0x60, 0xc0, 0xf8, 0xcc, 0xcc, 0x78, 0x00,
  0xfc, 0xcc, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x00,
  0x78, 0xcc, 0xcc, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x78, 0xcc, 0xcc, 0x7c, 0x0c, 0x18, 0x70, 0x00,
  0x00, 0x30, 0x30, 0x00, 0x00, 0x30, 0x30, 0x00,
  0x00, 0x30, 0x30, 0x00, 0x00, 0x30, 0x30, 0x60,
  0x18, 0x30, 0x60, 0xc0, 0x60, 0x30, 0x18, 0x00,
  0x00, 0x00, 0xfc, 0x00, 0x00, 0xfc, 0x00, 0x00,
  0x60, 0x30, 0x18, 0x0c, 0x18, 0x30, 0x60, 0x00,
  0x78, 0xcc, 0x0c, 0x18, 0x30, 0x00, 0x30, 0x00,
  0x7c, 0xc6, 0xde, 0xde, 0xde, 0xc0, 0x78, 0x00,
  0x30, 0x78, 0xcc, 0xcc, 0xfc, 0xcc, 0xcc, 0x00,
  0xfc, 0x66, 0x66, 0x7c, 0x66, 0x66, 0xfc, 0x00,
  0x3c, 0x66, 0xc0, 0xc0, 0xc0, 0x66, 0x3c, 0x00,
  0xf8, 0x6c, 0x66, 0x66, 0x66, 0x6c, 0xf8, 0x00,
  0xfe, 0x62, 0x68, 0x78, 0x68, 0x62, 0xfe, 0x00,
  0xfe, 0x62, 0x68, 0x78, 0x68, 0x60, 0xf0, 0x00,
  0x3c, 0x66, 0xc0, 0xc0, 0xce, 0x66, 0x3e, 0x00,
  0xcc, 0xcc, 0xcc, 0xfc, 0xcc, 0xcc, 0xcc, 0x00,
  0x78, 0x30, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x1e, 0x0c, 0x0c, 0x0c, 0xcc, 0xcc, 0x78, 0x00,
  0xe6, 0x66, 0x6c, 0x78, 0x6c, 0x66, 0xe6, 0x00,
  0xf0, 0x60, 0x60, 0x60, 0x62, 0x66, 0xfe, 0x00,
  0xc6, 0xee, 0xfe, 0xfe, 0xd6, 0xc6, 0xc6, 0x00,
  0xc6, 0xe6, 0xf6, 0xde, 0xce, 0xc6, 0xc6, 0x00,
  0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0x6c, 0x38, 0x00,
  0xfc, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xf0, 0x00,
  0x78, 0xcc, 0xcc, 0xcc, 0xdc, 0x78, 0x1c, 0x00,
  0xfc, 0x66, 0x66, 0x7c, 0x6c, 0x66, 0xe6, 0x00,
  0x78, 0xcc, 0xe0, 0x70, 0x1c, 0xcc, 0x78, 0x00,
  0xfc, 0xb4, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00,
  0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xfc, 0x00,
  0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x78, 0x30, 0x00,
  0xc6, 0xc6, 0xc6, 0xd6, 0xfe, 0xee, 0xc6, 0x00,
  0xc6, 0xc6, 0x6c, 0x38, 0x38, 0x6c, 0xc6, 0x00,
  0xcc, 0xcc, 0xcc, 0x78, 0x30, 0x30, 0x78, 0x00,
  0xfe, 0xc6, 0x8c, 0x18, 0x32, 0x66, 0xfe, 0x00,
  0x78, 0x60, 0x60, 0x60, 0x60, 0x60, 0x78, 0x00,
  0xc0, 0x60, 0x30, 0x18, 0x0c, 0x06, 0x02, 0x00,
  0x78, 0x18, 0x18, 0x18, 0x18, 0x18, 0x78, 0x00,
  0x10, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x76, 0x00,
  0xe0, 0x60, 0x60, 0x7c, 0x66, 0x66, 0xdc, 0x00,
  0x00, 0x00, 0x78, 0xcc, 0xc0, 0xcc, 0x78, 0x00,
  0x1c, 0x0c, 0x0c, 0x7c, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00,
  0x38, 0x6c, 0x60, 0xf0, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x76, 0xcc, 0xcc, 0x7c, 0x0c, 0xf8,
  0xe0, 0x60, 0x6c, 0x76, 0x66, 0x66, 0xe6, 0x00,
  0x30, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x0c, 0x00, 0x0c, 0x0c, 0x0c, 0xcc, 0xcc, 0x78,
  0xe0, 0x60, 0x66, 0x6c, 0x78, 0x6c, 0xe6, 0x00,
  0x70, 0x30, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x00, 0x00, 0xcc, 0xfe, 0xfe, 0xd6, 0xc6, 0x00,
  0x00, 0x00, 0xf8, 0xcc, 0xcc, 0xcc, 0xcc, 0x00,
  0x00, 0x00, 0x78, 0xcc, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0xdc, 0x66, 0x66, 0x7c, 0x60, 0xf0,
  0x00, 0x00, 0x76, 0xcc, 0xcc, 0x7c, 0x0c, 0x1e,
  0x00, 0x00, 0xdc, 0x76, 0x66, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x7c, 0xc0, 0x78, 0x0c, 0xf8, 0x00,
  0x10, 0x30, 0x7c, 0x30, 0x30, 0x34, 0x18, 0x00,
  0x00, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0xcc, 0xcc, 0xcc, 0x78, 0x30, 0x00,
  0x00, 0x00, 0xc6, 0xd6, 0xfe, 0xfe, 0x6c, 0x00,
  0x00, 0x00, 0xc6, 0x6c, 0x38, 0x6c, 0xc6, 0x00,
  0x00, 0x00, 0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0xf8,
  0x00, 0x00, 0xfc, 0x98, 0x30, 0x64, 0xfc, 0x00,
  0x1c, 0x30, 0x30, 0xe0, 0x30, 0x30, 0x1c, 0x00,
  0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x00,
  0xe0, 0x30, 0x30, 0x1c, 0x30, 0x30, 0xe0, 0x00,
  0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0x00,
  0x78, 0xcc, 0xc0, 0xcc, 0x78, 0x18, 0x0c, 0x78,
  0x00, 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00,
  0x1c, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00,
  0x7e, 0xc3, 0x3c, 0x06, 0x3e, 0x66, 0x3f, 0x00,
  0xcc, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00,
  0xe0, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00,
  0x30, 0x30, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00,
  0x00, 0x00, 0x78, 0xc0, 0xc0, 0x78, 0x0c, 0x38,
  0x7e, 0xc3, 0x3c, 0x66, 0x7e, 0x60, 0x3c, 0x00,
  0xcc, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00,
  0xe0, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00,
  0xcc, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x7c, 0xc6, 0x38, 0x18, 0x18, 0x18, 0x3c, 0x00,
  0xe0, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
  0xc6, 0x38, 0x6c, 0xc6, 0xfe, 0xc6, 0xc6, 0x00,
  0x30, 0x30, 0x00, 0x78, 0xcc, 0xfc, 0xcc, 0x00,
  0x1c, 0x00, 0xfc, 0x60, 0x78, 0x60, 0xfc, 0x00,
  0x00, 0x00, 0x7f, 0x0c, 0x7f, 0xcc, 0x7f, 0x00,
  0x3e, 0x6c, 0xcc, 0xfe, 0xcc, 0xcc, 0xce, 0x00,
  0x78, 0xcc, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0xcc, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0xe0, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x78, 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00,
  0x00, 0xe0, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00,
  0x00, 0xcc, 0x00, 0xcc, 0xcc, 0x7c, 0x0c, 0xf8,
  0xc3, 0x18, 0x3c, 0x66, 0x66, 0x3c, 0x18, 0x00,
  0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x78, 0x00,
  0x18, 0x18, 0x7e, 0xc0, 0xc0, 0x7e, 0x18, 0x18,
  0x38, 0x6c, 0x64, 0xf0, 0x60, 0xe6, 0xfc, 0x00,
  0xcc, 0xcc, 0x78, 0xfc, 0x30, 0xfc, 0x30, 0x30,
  0xf8, 0xcc, 0xcc, 0xfa, 0xc6, 0xcf, 0xc6, 0xc7,
  0x0e, 0x1b, 0x18, 0x3c, 0x18, 0x18, 0xd8, 0x70,
  0x1c, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00,
  0x38, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x00, 0x1c, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0x1c, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00,
  0x00, 0xf8, 0x00, 0xf8, 0xcc, 0xcc, 0xcc, 0x00,
  0xfc, 0x00, 0xcc, 0xec, 0xfc, 0xdc, 0xcc, 0x00,
  0x3c, 0x6c, 0x6c, 0x3e, 0x00, 0x7e, 0x00, 0x00,
  0x38, 0x6c, 0x6c, 0x38, 0x00, 0x7c, 0x00, 0x00,
  0x30, 0x00, 0x30, 0x60, 0xc0, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0x00, 0xfc, 0xc0, 0xc0, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xfc, 0x0c, 0x0c, 0x00, 0x00,
  0xc3, 0xc6, 0xcc, 0xde, 0x33, 0x66, 0xcc, 0x0f,
  0xc3, 0xc6, 0xcc, 0xdb, 0x37, 0x6f, 0xcf, 0x03,
  0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00,
  0x00, 0x33, 0x66, 0xcc, 0x66, 0x33, 0x00, 0x00,
  0x00, 0xcc, 0x66, 0x33, 0x66, 0xcc, 0x00, 0x00,
  0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 0x22, 0x88,
  0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
  0xdb, 0x77, 0xdb, 0xee, 0xdb, 0x77, 0xdb, 0xee,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0x18, 0x18,
  0x18, 0x18, 0xf8, 0x18, 0xf8, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0xf6, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x36, 0x36, 0x36,
  0x00, 0x00, 0xf8, 0x18, 0xf8, 0x18, 0x18, 0x18,
  0x36, 0x36, 0xf6, 0x06, 0xf6, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0xfe, 0x06, 0xf6, 0x36, 0x36, 0x36,
  0x36, 0x36, 0xf6, 0x06, 0xfe, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0xfe, 0x00, 0x00, 0x00,
  0x18, 0x18, 0xf8, 0x18, 0xf8, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x1f, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xff, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0x37, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x37, 0x30, 0x3f, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3f, 0x30, 0x37, 0x36, 0x36, 0x36,
  0x36, 0x36, 0xf7, 0x00, 0xff, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xff, 0x00, 0xf7, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x37, 0x30, 0x37, 0x36, 0x36, 0x36,
  0x00, 0x00, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00,
  0x36, 0x36, 0xf7, 0x00, 0xf7, 0x36, 0x36, 0x36,
  0x18, 0x18, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0xff, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xff, 0x00, 0xff, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0xff, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x3f, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x1f, 0x18, 0x1f, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x3f, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0xff, 0x36, 0x36, 0x36,
  0x18, 0x18, 0xff, 0x18, 0xff, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0xf8, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x18, 0x18,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
  0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
  0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
  0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x76, 0xdc, 0xc8, 0xdc, 0x76, 0x00,
  0x00, 0x78, 0xcc, 0xf8, 0xcc, 0xf8, 0xc0, 0xc0,
  0x00, 0xfc, 0xcc, 0xc0, 0xc0, 0xc0, 0xc0, 0x00,
  0x00, 0xfe, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00,
  0xfc, 0xcc, 0x60, 0x30, 0x60, 0xcc, 0xfc, 0x00,
  0x00, 0x00, 0x7e, 0xd8, 0xd8, 0xd8, 0x70, 0x00,
  0x00, 0x66, 0x66, 0x66, 0x66, 0x7c, 0x60, 0xc0,
  0x00, 0x76, 0xdc, 0x18, 0x18, 0x18, 0x18, 0x00,
  0xfc, 0x30, 0x78, 0xcc, 0xcc, 0x78, 0x30, 0xfc,
  0x38, 0x6c, 0xc6, 0xfe, 0xc6, 0x6c, 0x38, 0x00,
  0x38, 0x6c, 0xc6, 0xc6, 0x6c, 0x6c, 0xee, 0x00,
  0x1c, 0x30, 0x18, 0x7c, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0x7e, 0xdb, 0xdb, 0x7e, 0x00, 0x00,
  0x06, 0x0c, 0x7e, 0xdb, 0xdb, 0x7e, 0x60, 0xc0,
  0x38, 0x60, 0xc0, 0xf8, 0xc0, 0x60, 0x38, 0x00,
  0x78, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x00,
  0x00, 0xfc, 0x00, 0xfc, 0x00, 0xfc, 0x00, 0x00,
  0x30, 0x30, 0xfc, 0x30, 0x30, 0x00, 0xfc, 0x00,
  0x60, 0x30, 0x18, 0x30, 0x60, 0x00, 0xfc, 0x00,
  0x18, 0x30, 0x60, 0x30, 0x18, 0x00, 0xfc, 0x00,
  0x0e, 0x1b, 0x1b, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xd8, 0xd8, 0x70,
  0x30, 0x30, 0x00, 0xfc, 0x00, 0x30, 0x30, 0x00,
  0x00, 0x76, 0xdc, 0x00, 0x76, 0xdc, 0x00, 0x00,
  0x38, 0x6c, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00,
  0x0f, 0x0c, 0x0c, 0x0c, 0xec, 0x6c, 0x3c, 0x1c,
  0x78, 0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00,
  0x70, 0x18, 0x30, 0x60, 0x78, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x3c, 0x3c, 0x3c, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

uint8_t int10_font_14[256 * 14] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7e, 0x81, 0xa5, 0x81, 0x81, 0xbd, 0x99, 0x81,
  0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xff,
  0xdb, 0xff, 0xff, 0xc3, 0xe7, 0xff, 0x7e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x6c, 0xfe, 0xfe,
  0xfe, 0xfe, 0x7c, 0x38, 0x10, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x10, 0x38, 0x7c, 0xfe, 0x7c,
  0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x3c, 0x3c, 0xe7, 0xe7, 0xe7, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c,
  0x7e, 0xff, 0xff, 0x7e, 0x18, 0x18, 0x3c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
  0x3c, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xe7, 0xc3, 0xc3,
  0xe7, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0x42, 0x42, 0x66, 0x3c,
  0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
  0xc3, 0x99, 0xbd, 0xbd, 0x99, 0xc3, 0xff, 0xff,
  0xff, 0xff, 0x00, 0x00, 0x1e, 0x0e, 0x1a, 0x32,
  0x78, 0xcc, 0xcc, 0xcc, 0x78, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0x66, 0x66, 0x3c, 0x18,
  0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x3f, 0x33, 0x3f, 0x30, 0x30, 0x30, 0x70, 0xf0,
  0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0x63,
  0x7f, 0x63, 0x63, 0x63, 0x67, 0xe7, 0xe6, 0xc0,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0xdb, 0x3c,
  0xe7, 0x3c, 0xdb, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x80, 0xc0, 0xe0, 0xf8, 0xfe, 0xf8,
  0xe0, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x02, 0x06, 0x0e, 0x3e, 0xfe, 0x3e, 0x0e, 0x06,
  0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c,
  0x7e, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x66, 0x00, 0x66, 0x66, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7f, 0xdb, 0xdb, 0xdb, 0x7b, 0x1b,
  0x1b, 0x1b, 0x1b, 0x00, 0x00, 0x00, 0x00, 0x7c,
  0xc6, 0x60, 0x38, 0x6c, 0xc6, 0xc6, 0x6c, 0x38,
  0x0c, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0xfe, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18,
  0x18, 0x18, 0x7e, 0x3c, 0x18, 0x7e, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x7e, 0x3c,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x0c, 0xfe, 0x0c, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x60,
  0xfe, 0x60, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xc0, 0xc0,
  0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x28, 0x6c, 0xfe, 0x6c, 0x28, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
  0x38, 0x38, 0x7c, 0x7c, 0xfe, 0xfe, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xfe, 0x7c,
  0x7c, 0x38, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x3c, 0x3c, 0x3c, 0x18, 0x18, 0x00, 0x18,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66,
  0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x6c, 0x6c, 0xfe, 0x6c,
  0x6c, 0x6c, 0xfe, 0x6c, 0x6c, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x7c, 0xc6, 0xc2, 0xc0, 0x7c, 0x06,
  0x86, 0xc6, 0x7c, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc2, 0xc6, 0x0c, 0x18, 0x30, 0x66,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c,
  0x6c, 0x38, 0x76, 0xdc, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0x00, 0x30, 0x30, 0x30, 0x60, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x30,
  0x30, 0x18, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x30, 0x18, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x18,
  0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x66, 0x3c, 0xff, 0x3c, 0x66, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18,
  0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x02, 0x06, 0x0c, 0x18,
  0x30, 0x60, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xce, 0xde, 0xf6, 0xe6,
  0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x38, 0x78, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6,
  0x06, 0x0c, 0x18, 0x30, 0x60, 0xc6, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0x06, 0x06,
  0x3c, 0x06, 0x06, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0c, 0x1c, 0x3c, 0x6c, 0xcc, 0xfe,
  0x0c, 0x0c, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xfe, 0xc0, 0xc0, 0xc0, 0xfc, 0x06, 0x06, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x60,
  0xc0, 0xc0, 0xfc, 0xc6, 0xc6, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0xc6, 0x06, 0x0c,
  0x18, 0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7c, 0xc6,
  0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7c, 0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x06, 0x0c,
  0x78, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
  0x18, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x06, 0x0c, 0x18, 0x30, 0x60, 0x30,
  0x18, 0x0c, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7e, 0x00, 0x00, 0x7e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x30,
  0x18, 0x0c, 0x06, 0x0c, 0x18, 0x30, 0x60, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x0c,
  0x18, 0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xde, 0xde, 0xde,
  0xdc, 0xc0, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0x66,
  0x66, 0x66, 0x7c, 0x66, 0x66, 0x66, 0xfc, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0,
  0xc0, 0xc0, 0xc2, 0x66, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xf8, 0x6c, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x6c, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xfe, 0x66, 0x62, 0x68, 0x78, 0x68, 0x62, 0x66,
  0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x66,
  0x62, 0x68, 0x78, 0x68, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0,
  0xc0, 0xde, 0xc6, 0x66, 0x3a, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xfe, 0xc6,
  0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x3c, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x0c,
  0x0c, 0x0c, 0x0c, 0x0c, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xe6, 0x66, 0x6c, 0x6c,
  0x78, 0x6c, 0x6c, 0x66, 0xe6, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xf0, 0x60, 0x60, 0x60, 0x60, 0x60,
  0x62, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xc6, 0xee, 0xfe, 0xfe, 0xd6, 0xc6, 0xc6, 0xc6,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xe6,
  0xf6, 0xfe, 0xde, 0xce, 0xc6, 0xc6, 0xc6, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x6c, 0x38, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x60,
  0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xd6, 0xde, 0x7c,
  0x0c, 0x0e, 0x00, 0x00, 0x00, 0x00, 0xfc, 0x66,
  0x66, 0x66, 0x7c, 0x6c, 0x66, 0x66, 0xe6, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x60,
  0x38, 0x0c, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0x7e, 0x5a, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0x6c, 0x38, 0x10, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6,
  0xd6, 0xd6, 0xfe, 0x7c, 0x6c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0x6c, 0x38, 0x38, 0x38,
  0x6c, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x66, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xc6,
  0x8c, 0x18, 0x30, 0x60, 0xc2, 0xc6, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x3c, 0x30, 0x30, 0x30,
  0x30, 0x30, 0x30, 0x30, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x80, 0xc0, 0xe0, 0x70, 0x38, 0x1c,
  0x0e, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x3c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
  0x3c, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00,
  0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc,
  0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe0, 0x60,
  0x60, 0x78, 0x6c, 0x66, 0x66, 0x66, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c,
  0xc6, 0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1c, 0x0c, 0x0c, 0x3c, 0x6c, 0xcc,
  0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c,
  0x64, 0x60, 0xf0, 0x60, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76,
  0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x6c, 0x76, 0x66,
  0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x06,
  0x00, 0x0e, 0x06, 0x06, 0x06, 0x06, 0x66, 0x66,
  0x3c, 0x00, 0x00, 0x00, 0xe0, 0x60, 0x60, 0x66,
  0x6c, 0x78, 0x6c, 0x66, 0xe6, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xec, 0xfe, 0xd6, 0xd6, 0xd6,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xdc, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c,
  0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x7c, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x76, 0xcc, 0xcc, 0xcc, 0x7c,
  0x0c, 0x0c, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xdc, 0x76, 0x66, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c,
  0xc6, 0x70, 0x1c, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x10, 0x30, 0x30, 0xfc, 0x30, 0x30,
  0x30, 0x36, 0x1c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
  0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x66, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6,
  0xc6, 0xd6, 0xd6, 0xfe, 0x6c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x6c, 0x38,
  0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0x7e,
  0x06, 0x0c, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xfe, 0xcc, 0x18, 0x30, 0x66, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x0e, 0x18, 0x18, 0x18,
  0x70, 0x18, 0x18, 0x18, 0x0e, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x18,
  0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x70, 0x18, 0x18, 0x18, 0x0e, 0x18, 0x18, 0x18,
  0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x38,
  0x6c, 0xc6, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xc2,
  0x66, 0x3c, 0x0c, 0x06, 0x7c, 0x00, 0x00, 0x00,
  0xcc, 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
  0x76, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x18, 0x30,
  0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0x00, 0x78,
  0x0c, 0x7c, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xcc, 0xcc, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x60,
  0x30, 0x18, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc,
  0x76, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c, 0x38,
  0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x66,
  0x60, 0x66, 0x3c, 0x0c, 0x06, 0x3c, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xcc, 0xcc, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x60, 0x30, 0x18,
  0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x00, 0x38,
  0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x3c, 0x66, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x60,
  0x30, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0x10,
  0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0x38, 0x00, 0x38, 0x6c,
  0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0x00, 0x00, 0x00,
  0x18, 0x30, 0x60, 0x00, 0xfe, 0x66, 0x60, 0x7c,
  0x60, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xcc, 0x76, 0x36, 0x7e, 0xd8, 0xd8,
  0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x6c,
  0xcc, 0xcc, 0xfe, 0xcc, 0xcc, 0xcc, 0xce, 0x00,
  0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c,
  0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x60,
  0x30, 0x18, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x30, 0x78, 0xcc,
  0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0x00, 0x60, 0x30, 0x18, 0x00, 0xcc,
  0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0x00, 0xc6, 0xc6, 0xc6,
  0xc6, 0x7e, 0x06, 0x0c, 0x78, 0x00, 0x00, 0xc6,
  0xc6, 0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0xc6, 0x6c,
  0x38, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0x00,
  0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x18, 0x3c, 0x66, 0x60,
  0x60, 0x66, 0x3c, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60, 0x60,
  0x60, 0xe6, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18, 0x7e, 0x18,
  0x18, 0x00, 0x00, 0x00, 0x00, 0xf8, 0xcc, 0xcc,
  0xf8, 0xc4, 0xcc, 0xde, 0xcc, 0xcc, 0xc6, 0x00,
  0x00, 0x00, 0x00, 0x0e, 0x1b, 0x18, 0x18, 0x18,
  0x7e, 0x18, 0x18, 0x18, 0x18, 0xd8, 0x70, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x0c,
  0x18, 0x30, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x18, 0x30, 0x60,
  0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x30, 0x60, 0x00, 0xcc,
  0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x76, 0xdc, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x76, 0xdc,
  0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde, 0xce, 0xc6,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x6c, 0x6c,
  0x3e, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x60,
  0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0xc0, 0xc0, 0xc0,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0x06, 0x06, 0x06, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xc0, 0xc0, 0xc6, 0xcc, 0xd8,
  0x30, 0x60, 0xdc, 0x86, 0x0c, 0x18, 0x3e, 0x00,
  0x00, 0xc0, 0xc0, 0xc6, 0xcc, 0xd8, 0x30, 0x66,
  0xce, 0x9e, 0x3e, 0x06, 0x06, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x00, 0x18, 0x18, 0x3c, 0x3c, 0x3c,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x6c, 0xd8, 0x6c, 0x36, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd8, 0x6c,
  0x36, 0x6c, 0xd8, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44,
  0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x55, 0xaa,
  0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
  0x55, 0xaa, 0x55, 0xaa, 0xdd, 0x77, 0xdd, 0x77,
  0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77,
  0xdd, 0x77, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0xf6, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xfe, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x36, 0x36,
  0x36, 0x36, 0x36, 0xf6, 0x06, 0xf6, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe,
  0x06, 0xf6, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x06, 0xfe,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xfe, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18,
  0x18, 0xf8, 0x18, 0xf8, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xf8, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xff, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x1f, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x37, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x3f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x3f, 0x30, 0x37, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0xf7, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x00, 0xf7, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x37,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x36, 0x36, 0x36, 0x36,
  0x36, 0xf7, 0x00, 0xf7, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff,
  0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xff, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x3f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x1f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x1f, 0x18, 0x1f, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x3f, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0xff, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x1f, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0,
  0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
  0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f,
  0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
  0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0xd8,
  0xd8, 0xdc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xfc, 0xc6, 0xc6, 0xfc,
  0xc0, 0xc0, 0x40, 0x00, 0x00, 0x00, 0xfe, 0xc6,
  0xc6, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x6c,
  0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc6, 0x60, 0x30, 0x18, 0x30,
  0x60, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7e, 0xd8, 0xd8, 0xd8, 0xd8,
  0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x66, 0x66, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xc0,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0x18, 0x3c, 0x66, 0x66, 0x66,
  0x3c, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0x6c,
  0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c,
  0xc6, 0xc6, 0xc6, 0x6c, 0x6c, 0x6c, 0xee, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x1e, 0x30, 0x18, 0x0c,
  0x3e, 0x66, 0x66, 0x66, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xdb, 0xdb,
  0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x03, 0x06, 0x7e, 0xdb, 0xdb, 0xf3, 0x7e, 0x60,
  0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1c, 0x30,
  0x60, 0x60, 0x7c, 0x60, 0x60, 0x30, 0x1c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0xfe, 0x00,
  0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x7e, 0x18, 0x18, 0x00, 0x00,
  0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x18,
  0x0c, 0x06, 0x0c, 0x18, 0x30, 0x00, 0x7e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 0x60,
  0x30, 0x18, 0x0c, 0x00, 0x7e, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0e, 0x1b, 0x1b, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xd8, 0xd8,
  0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
  0x18, 0x00, 0x7e, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc,
  0x00, 0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x0f, 0x0c, 0x0c, 0x0c, 0x0c,
  0x0c, 0xec, 0x6c, 0x3c, 0x1c, 0x00, 0x00, 0x00,
  0x00, 0xd8, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x70,
  0xd8, 0x30, 0x60, 0xc8, 0xf8, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

uint8_t int10_font_16[256 * 16] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0x81, 0xa5, 0x81, 0x81, 0xbd,
  0x99, 0x81, 0x81, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0xff, 0xdb, 0xff, 0xff, 0xc3,
  0xe7, 0xff, 0xff, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x6c, 0xfe, 0xfe, 0xfe,
  0xfe, 0x7c, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x7c, 0xfe,
  0x7c, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x3c, 0x3c, 0xe7, 0xe7,
  0xe7, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x3c, 0x7e, 0xff, 0xff,
  0x7e, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c,
  0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe7, 0xc3,
  0xc3, 0xe7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0x42,
  0x42, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xc3, 0x99, 0xbd,
  0xbd, 0x99, 0xc3, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x1e, 0x0e, 0x1a, 0x32, 0x78, 0xcc,
  0xcc, 0xcc, 0xcc, 0x78, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0x66, 0x66, 0x66, 0x3c,
  0x18, 0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3f, 0x33, 0x3f, 0x30, 0x30, 0x30,
  0x30, 0x70, 0xf0, 0xe0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7f, 0x63, 0x7f, 0x63, 0x63, 0x63,
  0x63, 0x67, 0xe7, 0xe6, 0xc0, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x18, 0xdb, 0x3c, 0xe7,
  0x3c, 0xdb, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfe, 0xf8,
  0xf0, 0xe0, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x02, 0x06, 0x0e, 0x1e, 0x3e, 0xfe, 0x3e,
  0x1e, 0x0e, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18,
  0x7e, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x00, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7f, 0xdb, 0xdb, 0xdb, 0x7b, 0x1b,
  0x1b, 0x1b, 0x1b, 0x1b, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x7c, 0xc6, 0x60, 0x38, 0x6c, 0xc6, 0xc6,
  0x6c, 0x38, 0x0c, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xfe, 0xfe, 0xfe, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18,
  0x7e, 0x3c, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x7e, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x0c, 0xfe,
  0x0c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x60, 0xfe,
  0x60, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xc0,
  0xc0, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x6c, 0xfe,
  0x6c, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x38, 0x7c,
  0x7c, 0xfe, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0xfe, 0x7c, 0x7c,
  0x38, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x3c, 0x3c, 0x18, 0x18,
  0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x66, 0x66, 0x66, 0x24, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x6c, 0x6c, 0xfe, 0x6c, 0x6c,
  0x6c, 0xfe, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x7c, 0xc6, 0xc2, 0xc0, 0x7c, 0x06,
  0x06, 0x86, 0xc6, 0x7c, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xc2, 0xc6, 0x0c, 0x18,
  0x30, 0x60, 0xc6, 0x86, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0x6c, 0x38, 0x76, 0xdc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x30, 0x30, 0x30, 0x60, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x30,
  0x30, 0x30, 0x18, 0x0c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x30, 0x18, 0x0c, 0x0c, 0x0c, 0x0c,
  0x0c, 0x0c, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x3c, 0xff,
  0x3c, 0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7e,
  0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x02, 0x06, 0x0c, 0x18,
  0x30, 0x60, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xd6, 0xd6,
  0xc6, 0xc6, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x38, 0x78, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0x06, 0x0c, 0x18, 0x30,
  0x60, 0xc0, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0x06, 0x06, 0x3c, 0x06,
  0x06, 0x06, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0c, 0x1c, 0x3c, 0x6c, 0xcc, 0xfe,
  0x0c, 0x0c, 0x0c, 0x1e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc0, 0xc0, 0xc0, 0xfc, 0x06,
  0x06, 0x06, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x60, 0xc0, 0xc0, 0xfc, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc6, 0x06, 0x06, 0x0c, 0x18,
  0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7c, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7e, 0x06,
  0x06, 0x06, 0x0c, 0x78, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x06, 0x0c, 0x18, 0x30, 0x60,
  0x30, 0x18, 0x0c, 0x06, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0x00, 0x00,
  0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x60, 0x30, 0x18, 0x0c, 0x06,
  0x0c, 0x18, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x0c, 0x18, 0x18,
  0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xde, 0xde,
  0xde, 0xdc, 0xc0, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xfe,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x66,
  0x66, 0x66, 0x66, 0xfc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xc0,
  0xc0, 0xc2, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xf8, 0x6c, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x66, 0x6c, 0xf8, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0x66, 0x62, 0x68, 0x78, 0x68,
  0x60, 0x62, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0x66, 0x62, 0x68, 0x78, 0x68,
  0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xde,
  0xc6, 0xc6, 0x66, 0x3a, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xfe, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1e, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
  0xcc, 0xcc, 0xcc, 0x78, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xe6, 0x66, 0x66, 0x6c, 0x78, 0x78,
  0x6c, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xf0, 0x60, 0x60, 0x60, 0x60, 0x60,
  0x60, 0x62, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xee, 0xfe, 0xfe, 0xd6, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde, 0xce,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x60,
  0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xd6, 0xde, 0x7c, 0x0c, 0x0e, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x6c,
  0x66, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x60, 0x38, 0x0c,
  0x06, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0x7e, 0x5a, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0x6c, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xd6, 0xd6,
  0xd6, 0xfe, 0xee, 0x6c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0x6c, 0x7c, 0x38, 0x38,
  0x7c, 0x6c, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x3c, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc6, 0x86, 0x0c, 0x18, 0x30,
  0x60, 0xc2, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x30, 0x30, 0x30, 0x30, 0x30,
  0x30, 0x30, 0x30, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x80, 0xc0, 0xe0, 0x70, 0x38,
  0x1c, 0x0e, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
  0x0c, 0x0c, 0x0c, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x10, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00,
  0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x78, 0x6c, 0x66,
  0x66, 0x66, 0x66, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc0,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1c, 0x0c, 0x0c, 0x3c, 0x6c, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60,
  0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x6c, 0x76, 0x66,
  0x66, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x06, 0x06, 0x00, 0x0e, 0x06, 0x06,
  0x06, 0x06, 0x06, 0x06, 0x66, 0x66, 0x3c, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x66, 0x6c, 0x78,
  0x78, 0x6c, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xec, 0xfe, 0xd6,
  0xd6, 0xd6, 0xd6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0x0c, 0x1e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x76, 0x66,
  0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0x60,
  0x38, 0x0c, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x10, 0x30, 0x30, 0xfc, 0x30, 0x30,
  0x30, 0x30, 0x36, 0x1c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66,
  0x66, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xd6,
  0xd6, 0xd6, 0xfe, 0x6c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x6c, 0x38,
  0x38, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x0c, 0xf8, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xcc, 0x18,
  0x30, 0x60, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0e, 0x18, 0x18, 0x18, 0x70, 0x18,
  0x18, 0x18, 0x18, 0x0e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x70, 0x18, 0x18, 0x18, 0x0e, 0x18,
  0x18, 0x18, 0x18, 0x70, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x76, 0xdc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6,
  0xc6, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xc0,
  0xc2, 0x66, 0x3c, 0x0c, 0x06, 0x7c, 0x00, 0x00,
  0x00, 0x00, 0xcc, 0x00, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0c, 0x18, 0x30, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xcc, 0x00, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x38, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0x60, 0x60,
  0x66, 0x3c, 0x0c, 0x06, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0x00, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x66, 0x00, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x3c, 0x66, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xc6, 0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6,
  0xfe, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x38, 0x6c, 0x38, 0x00, 0x38, 0x6c, 0xc6, 0xc6,
  0xfe, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x30, 0x60, 0x00, 0xfe, 0x66, 0x60, 0x7c,
  0x60, 0x60, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xcc, 0x76, 0x36,
  0x7e, 0xd8, 0xd8, 0x6e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3e, 0x6c, 0xcc, 0xcc, 0xfe, 0xcc,
  0xcc, 0xcc, 0xcc, 0xce, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0x00, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x30, 0x78, 0xcc, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0x00, 0x00, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x0c, 0x78, 0x00,
  0x00, 0xc6, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xc6, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x3c, 0x66, 0x60, 0x60, 0x60,
  0x66, 0x3c, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60, 0x60,
  0x60, 0x60, 0xe6, 0xfc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18,
  0x7e, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xf8, 0xcc, 0xcc, 0xf8, 0xc4, 0xcc, 0xde,
  0xcc, 0xcc, 0xcc, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0e, 0x1b, 0x18, 0x18, 0x18, 0x7e, 0x18,
  0x18, 0x18, 0x18, 0x18, 0xd8, 0x70, 0x00, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0c, 0x18, 0x30, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x76, 0xdc, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x76, 0xdc, 0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde,
  0xce, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x3c, 0x6c, 0x6c, 0x3e, 0x00, 0x7e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x60,
  0xc0, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xc0,
  0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x06,
  0x06, 0x06, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30,
  0x60, 0xdc, 0x86, 0x0c, 0x18, 0x3e, 0x00, 0x00,
  0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30,
  0x66, 0xce, 0x9e, 0x3e, 0x06, 0x06, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18,
  0x3c, 0x3c, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x6c, 0xd8,
  0x6c, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xd8, 0x6c, 0x36,
  0x6c, 0xd8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44,
  0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44,
  0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
  0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
  0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77,
  0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xf6,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x06, 0xf6,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x06, 0xf6,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x06, 0xfe,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xfe,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x1f,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x37,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x3f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0x30, 0x37,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf7, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xf7,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x37,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf7, 0x00, 0xf7,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x3f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x1f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x1f,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
  0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
  0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
  0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0xd8,
  0xd8, 0xd8, 0xdc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x78, 0xcc, 0xcc, 0xcc, 0xd8, 0xcc,
  0xc6, 0xc6, 0xc6, 0xcc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc6, 0xc6, 0xc0, 0xc0, 0xc0,
  0xc0, 0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x6c, 0x6c, 0x6c,
  0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xfe, 0xc6, 0x60, 0x30, 0x18,
  0x30, 0x60, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xd8, 0xd8,
  0xd8, 0xd8, 0xd8, 0x70, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x7c, 0x60, 0x60, 0xc0, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7e, 0x18, 0x3c, 0x66, 0x66,
  0x66, 0x3c, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xfe,
  0xc6, 0xc6, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0x6c,
  0x6c, 0x6c, 0x6c, 0xee, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1e, 0x30, 0x18, 0x0c, 0x3e, 0x66,
  0x66, 0x66, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xdb, 0xdb,
  0xdb, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x03, 0x06, 0x7e, 0xdb, 0xdb,
  0xf3, 0x7e, 0x60, 0xc0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1c, 0x30, 0x60, 0x60, 0x7c, 0x60,
  0x60, 0x60, 0x30, 0x1c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0xfe,
  0x00, 0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7e, 0x18,
  0x18, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x30, 0x18, 0x0c, 0x06, 0x0c,
  0x18, 0x30, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 0x60, 0x30,
  0x18, 0x0c, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0e, 0x1b, 0x1b, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0xd8, 0xd8, 0xd8, 0x70, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x7e,
  0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0x00,
  0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0f, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0xec,
  0x6c, 0x6c, 0x3c, 0x1c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xd8, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x70, 0xd8, 0x30, 0x60, 0xc8, 0xf8, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x7c, 0x7c, 0x7c, 0x7c,
  0x7c, 0x7c, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

uint8_t int10_font_19[256 * 19] = {
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x7E,0x81,0x81,0xA5,0x81,0x81,0x81,0xBD,0x99,0x81,0x81,
	0x7E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7E,0xFF,0xFF,
	0xDB,0xFF,0xFF,0xFF,0xC3,0xE7,0xFF,0xFF,0x7E,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x6C,0xFE,0xFE,
	0xFE,0xFE,0xFE,0x7C,0x38,0x10,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x10,0x38,0x7C,0xFE,0xFE,0x7C,
	0x38,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x18,0x3C,0x3C,0x3C,0xE7,0xE7,0xE7,0xE7,0x18,0x18,0x3C,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x3C,
	0x7E,0xFF,0xFF,0xFF,0x7E,0x18,0x18,0x3C,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x3C,0x3C,
	0x3C,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xFF,
	0xFF,0xFF,0xFF,0xFF,0xFF,0xE7,0xC3,0xC3,0xC3,0xE7,0xFF,
	0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x00,0x00,0x00,0x00,0x00,
	0x00,0x3C,0x66,0x42,0x42,0x42,0x66,0x3C,0x00,0x00,0x00,
	0x00,0x00,0x00,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xC3,0x99,
	0xBD,0xBD,0xBD,0x99,0xC3,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
	0x00,0x00,0x00,0x1E,0x06,0x0E,0x1A,0x30,0x78,0xCC,0xCC,
	0xCC,0xCC,0xCC,0x78,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x3C,0x66,0x66,0x66,0x66,0x66,0x66,0x3C,0x18,0x7E,0x18,
	0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3F,0x33,0x33,
	0x3F,0x30,0x30,0x30,0x30,0x30,0x70,0xF0,0xE0,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x7F,0x63,0x63,0x7F,0x63,0x63,
	0x63,0x63,0x63,0x67,0xE7,0xE6,0xC0,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x18,0x18,0xDB,0x3C,0xE7,0xE7,0x3C,
	0xDB,0x18,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,
	0xC0,0xE0,0xF0,0xF8,0xFE,0xFE,0xF8,0xF0,0xE0,0xC0,0x80,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x06,0x0E,0x1E,
	0x3E,0xFE,0xFE,0x3E,0x1E,0x0E,0x06,0x02,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x18,0x3C,0x7E,0x18,0x18,0x18,0x18,
	0x18,0x18,0x7E,0x3C,0x18,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x00,
	0x66,0x66,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7F,0xDB,
	0xDB,0xDB,0xDB,0x7B,0x1B,0x1B,0x1B,0x1B,0x1B,0x1B,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x7C,0xC6,0x60,0x38,0x6C,
	0xC6,0xC6,0x6C,0x38,0x0C,0xC6,0x7C,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0xFE,0xFE,0xFE,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x18,0x3C,0x7E,0x18,0x18,0x18,0x18,0x18,0x18,0x7E,0x3C,
	0x18,0x7E,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x3C,0x7E,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x7E,0x3C,0x18,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x0C,0xFE,0x0C,0x18,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x30,0x60,0xFE,0x60,0x30,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0xC0,0xC0,0xC0,0xC0,0xC0,0xFE,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x24,0x66,0xFF,
	0x66,0x24,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x10,0x10,0x38,0x38,0x7C,0x7C,0xFE,0xFE,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0xFE,0xFE,0x7C,0x7C,0x38,0x38,0x10,0x10,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x18,0x3C,0x3C,0x3C,0x3C,0x18,0x18,0x18,
	0x18,0x00,0x18,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x66,
	0x66,0x66,0x24,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x6C,0x6C,0x6C,
	0xFE,0x6C,0x6C,0x6C,0x6C,0xFE,0x6C,0x6C,0x6C,0x00,0x00,
	0x00,0x00,0x00,0x18,0x18,0x7C,0xC6,0xC2,0xC0,0xC0,0x7C,
	0x06,0x06,0x06,0x86,0xC6,0x7C,0x18,0x18,0x00,0x00,0x00,
	0x00,0x00,0xC6,0xC6,0xCC,0x0C,0x18,0x18,0x30,0x30,0x60,
	0x66,0xC6,0xC6,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,
	0x6C,0x6C,0x6C,0x38,0x76,0xDC,0xDC,0xCC,0xCC,0xCC,0x76,
	0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x18,0x30,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x0C,0x18,0x30,0x30,0x30,0x30,0x30,
	0x30,0x30,0x30,0x18,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x30,0x18,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,
	0x18,0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x66,0x3C,0xFF,0x3C,0x66,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,
	0x18,0xFF,0x18,0x18,0x18,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x18,0x18,0x18,0x30,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x06,0x06,0x0C,0x0C,0x18,0x18,
	0x30,0x30,0x60,0x60,0xC0,0xC0,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x7C,0xC6,0xC6,0xC6,0xD6,0xD6,0xD6,0xD6,0xC6,
	0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,
	0x38,0x78,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x7E,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7C,0xC6,0x06,0x06,
	0x0C,0x18,0x30,0x60,0xC0,0xC0,0xC6,0xFE,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x7C,0xC6,0x06,0x06,0x06,0x3C,0x06,
	0x06,0x06,0x06,0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x1C,0x1C,0x3C,0x3C,0x6C,0x6C,0xCC,0xFE,0x0C,0x0C,
	0x0C,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0xC0,
	0xC0,0xC0,0xC0,0xFC,0x06,0x06,0x06,0x06,0xC6,0x7C,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x60,0xC0,0xC0,0xC0,
	0xFC,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0xFE,0xC6,0x06,0x06,0x06,0x0C,0x18,0x18,
	0x30,0x30,0x30,0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x7C,0xC6,0xC6,0xC6,0xC6,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,
	0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7C,0xC6,0xC6,
	0xC6,0xC6,0x7E,0x06,0x06,0x06,0x06,0x0C,0x78,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x00,0x00,
	0x00,0x00,0x00,0x18,0x18,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x18,0x18,0x00,0x00,0x00,0x00,0x00,
	0x18,0x18,0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x06,0x0C,0x18,0x30,0x60,0x60,0x30,0x18,0x0C,0x06,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0xFE,0x00,0x00,0xFE,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x60,0x30,0x18,0x0C,0x06,0x06,
	0x0C,0x18,0x30,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x7C,0xC6,0xC6,0x06,0x06,0x0C,0x18,0x18,0x18,0x00,
	0x18,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7C,
	0xC6,0xC6,0xC6,0xDE,0xDE,0xDE,0xDC,0xC0,0xC0,0x7C,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x38,0x6C,0xC6,0xC6,
	0xC6,0xFE,0xC6,0xC6,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0xFC,0x66,0x66,0x66,0x66,0x7C,0x66,0x66,
	0x66,0x66,0x66,0xFC,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x3C,0x66,0xC2,0xC0,0xC0,0xC0,0xC0,0xC0,0xC0,0xC2,0x66,
	0x3C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xF8,0x6C,0x66,
	0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x6C,0xF8,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0xFE,0x66,0x62,0x60,0x68,0x78,
	0x68,0x60,0x60,0x62,0x66,0xFE,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0xFE,0x66,0x62,0x60,0x68,0x78,0x68,0x60,0x60,
	0x60,0x60,0xF0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3C,
	0x66,0xC2,0xC0,0xC0,0xC0,0xDE,0xC6,0xC6,0xC6,0x66,0x3A,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC6,0xC6,0xC6,0xC6,
	0xC6,0xFE,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x3C,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x0F,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0xC6,
	0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xE6,0x66,
	0x66,0x6C,0x6C,0x78,0x7C,0x6C,0x6C,0x66,0x66,0xE6,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0xF0,0x60,0x60,0x60,0x60,
	0x60,0x60,0x60,0x60,0x62,0x66,0xFE,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0xC6,0xEE,0xFE,0xFE,0xD6,0xD6,0xC6,0xC6,
	0xC6,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0xC6,0xC6,0xE6,0xE6,0xF6,0xF6,0xDE,0xDE,0xCE,0xCE,0xC6,
	0xC6,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x6C,0xC6,
	0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x6C,0x38,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0xFC,0x66,0x66,0x66,0x66,0x7C,
	0x60,0x60,0x60,0x60,0x60,0xF0,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,
	0xD6,0xDE,0x7C,0x0C,0x0E,0x00,0x00,0x00,0x00,0x00,0xFC,
	0x66,0x66,0x66,0x66,0x7C,0x6C,0x6C,0x66,0x66,0x66,0xE6,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7C,0xC6,0xC6,0xC0,
	0x60,0x38,0x0C,0x06,0x06,0xC6,0xC6,0x7C,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x7E,0x7E,0x5A,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,
	0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC6,0xC6,
	0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x6C,0x38,0x10,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0xC6,0xC6,0xC6,0xC6,0xC6,
	0xC6,0xD6,0xD6,0xD6,0xFE,0x6C,0x6C,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0xC6,0xC6,0xC6,0x6C,0x6C,0x38,0x38,0x6C,
	0x6C,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x66,0x66,0x66,0x66,0x66,0x3C,0x18,0x18,0x18,0x18,0x18,
	0x3C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0xC6,0x86,
	0x06,0x0C,0x18,0x30,0x60,0xC0,0xC2,0xC6,0xFE,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x3C,0x30,0x30,0x30,0x30,0x30,
	0x30,0x30,0x30,0x30,0x30,0x3C,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0xC0,0xC0,0x60,0x60,0x30,0x30,0x18,0x18,0x0C,
	0x0C,0x06,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3C,
	0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x3C,
	0x00,0x00,0x00,0x00,0x00,0x10,0x38,0x6C,0xC6,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0x00,0x00,0x00,0x00,
	0x30,0x18,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x78,0x0C,0x0C,0x7C,0xCC,0xCC,0xCC,0x76,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0xE0,0x60,0x60,0x60,0x78,
	0x6C,0x66,0x66,0x66,0x66,0x66,0xDC,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7C,0xC6,0xC0,0xC0,
	0xC0,0xC0,0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x1C,0x0C,0x0C,0x0C,0x3C,0x6C,0xCC,0xCC,0xCC,0xCC,0xCC,
	0x76,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x7C,0xC6,0xC6,0xFE,0xC0,0xC0,0xC6,0x7C,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x38,0x6C,0x64,0x60,0x60,0xF0,
	0x60,0x60,0x60,0x60,0x60,0xF0,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x76,0xCC,0xCC,0xCC,0xCC,
	0xCC,0x7C,0x0C,0x0C,0xCC,0x78,0x00,0x00,0x00,0x00,0xE0,
	0x60,0x60,0x60,0x6C,0x76,0x66,0x66,0x66,0x66,0x66,0xE6,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x00,0x00,
	0x38,0x18,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x06,0x06,0x00,0x00,0x0E,0x06,0x06,
	0x06,0x06,0x06,0x06,0x06,0x66,0x66,0x3C,0x00,0x00,0x00,
	0x00,0xE0,0x60,0x60,0x60,0x66,0x66,0x6C,0x78,0x78,0x6C,
	0x66,0xE6,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xEC,
	0xFE,0xD6,0xD6,0xD6,0xD6,0xC6,0xC6,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xDC,0x66,0x66,0x66,
	0x66,0x66,0x66,0x66,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,
	0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0xDC,0x66,0x66,0x66,0x66,0x66,0x7C,0x60,0x60,0x60,
	0xF0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x76,0xCC,
	0xCC,0xCC,0xCC,0xCC,0x7C,0x0C,0x0C,0x0C,0x1E,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0xDC,0x76,0x66,0x60,0x60,
	0x60,0x60,0xF0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x7C,0xC6,0x60,0x38,0x0C,0x06,0xC6,0x7C,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x30,0x30,0x30,
	0xFC,0x30,0x30,0x30,0x30,0x30,0x36,0x1C,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xCC,0xCC,0xCC,
	0xCC,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x66,0x66,0x66,0x66,0x66,0x66,
	0x3C,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0xC6,0xC6,0xC6,0xD6,0xD6,0xD6,0xFE,0x6C,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC6,
	0xC6,0x6C,0x38,0x38,0x6C,0xC6,0xC6,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC6,0xC6,0xC6,0xC6,
	0xC6,0xC6,0x7E,0x06,0x06,0x0C,0x78,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0xFE,0xC6,0x0C,0x18,0x30,0x60,0xC6,
	0xFE,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0E,0x18,0x18,
	0x18,0x18,0x70,0x70,0x18,0x18,0x18,0x18,0x0E,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x70,0x18,0x18,0x18,0x18,0x0E,0x0E,0x18,0x18,
	0x18,0x18,0x70,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x76,
	0xDC,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,
	0x38,0x6C,0xC6,0xC6,0xC6,0xC6,0xFE,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x3C,0x66,0xC2,0xC0,0xC0,0xC0,0xC0,
	0xC0,0xC0,0xC2,0x66,0x3C,0x18,0x0C,0x38,0x00,0x00,0x00,
	0x00,0x00,0xCC,0xCC,0x00,0xCC,0xCC,0xCC,0xCC,0xCC,0xCC,
	0xCC,0x76,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x18,
	0x30,0x00,0x7C,0xC6,0xC6,0xFE,0xC0,0xC0,0xC6,0x7C,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x38,0x6C,0x00,0x78,
	0x0C,0x0C,0x7C,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0xCC,0xCC,0x00,0x78,0x0C,0x0C,0x7C,
	0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x60,0x30,0x18,0x00,0x78,0x0C,0x0C,0x7C,0xCC,0xCC,0xCC,
	0x76,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x6C,0x38,
	0x00,0x78,0x0C,0x0C,0x7C,0xCC,0xCC,0xCC,0x76,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7C,0xC6,
	0xC0,0xC0,0xC0,0xC0,0xC6,0x7C,0x18,0x0C,0x38,0x00,0x00,
	0x00,0x00,0x10,0x38,0x6C,0x00,0x7C,0xC6,0xC6,0xFE,0xC0,
	0xC0,0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0xCC,0xCC,0x00,0x7C,0xC6,0xC6,0xFE,0xC0,0xC0,0xC6,0x7C,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x30,0x18,0x00,
	0x7C,0xC6,0xC6,0xFE,0xC0,0xC0,0xC6,0x7C,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x66,0x66,0x00,0x38,0x18,0x18,
	0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x18,0x3C,0x66,0x00,0x38,0x18,0x18,0x18,0x18,0x18,
	0x18,0x3C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x30,
	0x18,0x00,0x38,0x18,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,
	0x00,0x00,0x00,0xC6,0xC6,0x00,0x10,0x38,0x6C,0xC6,0xC6,
	0xC6,0xFE,0xC6,0xC6,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00,
	0x38,0x6C,0x38,0x00,0x38,0x6C,0xC6,0xC6,0xC6,0xFE,0xC6,
	0xC6,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00,0x0C,0x18,0x00,
	0xFE,0x66,0x62,0x60,0x68,0x78,0x68,0x60,0x60,0x62,0x66,
	0xFE,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0xEC,0x36,0x36,0x7E,0xD8,0xD8,0xD8,0x6E,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x3E,0x6C,0xCC,0xCC,0xCC,0xFE,
	0xCC,0xCC,0xCC,0xCC,0xCC,0xCE,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x10,0x38,0x6C,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,
	0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0xC6,0xC6,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x30,0x18,0x00,
	0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x30,0x78,0xCC,0x00,0xCC,0xCC,0xCC,
	0xCC,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x60,0x30,0x18,0x00,0xCC,0xCC,0xCC,0xCC,0xCC,0xCC,
	0xCC,0x76,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC6,
	0xC6,0x00,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7E,0x06,0x06,
	0x0C,0x78,0x00,0xC6,0xC6,0x00,0x38,0x6C,0xC6,0xC6,0xC6,
	0xC6,0xC6,0xC6,0xC6,0xC6,0x6C,0x38,0x00,0x00,0x00,0x00,
	0xC6,0xC6,0x00,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,
	0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x18,0x18,0x3C,0x66,0x60,0x60,0x60,0x60,0x66,0x3C,0x18,
	0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x6C,0x64,0x60,
	0x60,0xF0,0x60,0x60,0x60,0x60,0x60,0xE6,0xFC,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x66,0x66,0x66,0x3C,0x18,0x7E,
	0x18,0x18,0x7E,0x18,0x18,0x18,0x00,0x00,0x00,0x00,0x00,
	0x00,0xF8,0xCC,0xCC,0xCC,0xF8,0xC4,0xCC,0xDE,0xCC,0xCC,
	0xCC,0xCC,0xC6,0x00,0x00,0x00,0x00,0x00,0x00,0x0E,0x1B,
	0x18,0x18,0x18,0x18,0x7E,0x18,0x18,0x18,0x18,0x18,0x18,
	0xD8,0x70,0x00,0x00,0x00,0x00,0x00,0x0C,0x18,0x30,0x00,
	0x78,0x0C,0x0C,0x7C,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x0C,0x18,0x30,0x00,0x38,0x18,0x18,
	0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x0C,0x18,0x30,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,
	0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x18,
	0x30,0x00,0xCC,0xCC,0xCC,0xCC,0xCC,0xCC,0xCC,0x76,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x76,0xDC,0x00,0xDC,
	0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x00,0x00,0x00,0x00,
	0x76,0xDC,0x00,0xC6,0xC6,0xE6,0xE6,0xF6,0xF6,0xDE,0xDE,
	0xCE,0xCE,0xC6,0xC6,0x00,0x00,0x00,0x00,0x00,0x00,0x3C,
	0x6C,0x6C,0x6C,0x3E,0x00,0x7E,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x6C,0x6C,0x6C,
	0x38,0x00,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x30,0x30,0x30,
	0x60,0xC0,0xC0,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0xC0,0xC0,0xC0,
	0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0xFE,0x06,0x06,0x06,0x06,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0xC0,0xC0,0xC0,0xC2,0xC6,
	0xCC,0x18,0x30,0x60,0xDC,0xA6,0x0C,0x18,0x30,0x3E,0x00,
	0x00,0x00,0x00,0xC0,0xC0,0xC0,0xC2,0xC6,0xCC,0x18,0x30,
	0x60,0xCC,0x9C,0x3C,0x7E,0x0C,0x0C,0x00,0x00,0x00,0x00,
	0x00,0x18,0x18,0x00,0x00,0x18,0x18,0x18,0x18,0x3C,0x3C,
	0x3C,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x33,0x66,0xCC,0xCC,0x66,0x33,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xCC,
	0x66,0x33,0x33,0x66,0xCC,0x00,0x00,0x00,0x00,0x00,0x00,
	0x11,0x44,0x11,0x44,0x11,0x44,0x11,0x44,0x11,0x44,0x11,
	0x44,0x11,0x44,0x11,0x44,0x11,0x44,0x11,0x55,0xAA,0x55,
	0xAA,0x55,0xAA,0x55,0xAA,0x55,0xAA,0x55,0xAA,0x55,0xAA,
	0x55,0xAA,0x55,0xAA,0x55,0xDD,0x77,0xDD,0x77,0xDD,0x77,
	0xDD,0x77,0xDD,0x77,0xDD,0x77,0xDD,0x77,0xDD,0x77,0xDD,
	0x77,0xDD,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0xF8,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0xF8,0x18,0xF8,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0xF6,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFE,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0xF8,0x18,0xF8,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0xF6,0x06,0xF6,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0x06,0xF6,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0xF6,0x06,0xFE,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0xFE,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0xF8,
	0x18,0xF8,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xF8,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x1F,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0xFF,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x1F,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0xFF,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0xFF,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x1F,0x18,0x1F,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x37,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x37,0x30,0x3F,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3F,
	0x30,0x37,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0xF7,0x00,0xFF,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0xFF,0x00,0xF7,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x37,0x30,0x37,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0x00,
	0xFF,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0xF7,0x00,0xF7,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0xFF,0x00,0xFF,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0xFF,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0x00,0xFF,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x3F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x1F,
	0x18,0x1F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1F,0x18,0x1F,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x3F,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x36,0xFF,0x36,0x36,0x36,0x36,0x36,0x36,0x36,0x36,
	0x36,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0xFF,0x18,
	0xFF,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0xF8,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x1F,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
	0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xFF,
	0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xF0,0xF0,0xF0,
	0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,
	0xF0,0xF0,0xF0,0xF0,0xF0,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x0F,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x76,0xDC,0xD8,0xD8,0xD8,
	0xD8,0xDC,0x76,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x78,
	0xCC,0xCC,0xCC,0xCC,0xD8,0xCC,0xC6,0xC6,0xC6,0xC6,0xDC,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0xC6,0xC6,0xC0,
	0xC0,0xC0,0xC0,0xC0,0xC0,0xC0,0xC0,0xC0,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0x6C,0x6C,
	0x6C,0x6C,0x6C,0x6C,0x6C,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0xFE,0xC6,0xC0,0x60,0x30,0x18,0x18,0x30,0x60,0xC0,
	0xC6,0xFE,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x7E,0xD8,0xD8,0xD8,0xD8,0xD8,0xD8,0x70,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x66,
	0x66,0x66,0x66,0x66,0x66,0x66,0x7C,0x60,0x60,0xC0,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x76,0xDC,0x18,0x18,
	0x18,0x18,0x18,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x3C,0x18,0x3C,0x66,0x66,0x66,0x66,0x66,0x66,0x3C,0x18,
	0x3C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x6C,0xC6,
	0xC6,0xC6,0xFE,0xC6,0xC6,0xC6,0xC6,0x6C,0x38,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x38,0x6C,0xC6,0xC6,0xC6,0xC6,
	0xC6,0xC6,0x6C,0x6C,0x6C,0xEE,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x1E,0x30,0x18,0x0C,0x3E,0x66,0x66,0x66,0x66,
	0x66,0x66,0x3C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x7E,0xDB,0xDB,0xDB,0xDB,0x7E,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x03,0x06,
	0x7E,0xCF,0xDB,0xDB,0xF3,0x7E,0x60,0xC0,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x1C,0x30,0x60,0x60,0x60,0x7C,0x60,
	0x60,0x60,0x60,0x30,0x1C,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,
	0xC6,0xC6,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0xFE,0x00,0x00,0x00,0xFE,0x00,0x00,0x00,0xFE,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x18,
	0x7E,0x18,0x18,0x18,0x00,0x00,0x7E,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x60,0x30,0x18,0x0C,0x06,0x0C,0x18,
	0x30,0x60,0x00,0x7E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x06,0x0C,0x18,0x30,0x60,0x30,0x18,0x0C,0x06,0x00,
	0x7E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0E,0x1B,0x1B,
	0x1B,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,
	0x18,0x18,0xD8,0xD8,0xD8,0x70,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x18,0x18,0x00,0x00,0x7E,0x00,0x00,
	0x18,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x76,0xDC,0x00,0x00,0x76,0xDC,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x6C,0x6C,0x6C,0x38,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0F,0x0C,0x0C,
	0x0C,0x0C,0x0C,0x0C,0x0C,0xEC,0x6C,0x6C,0x3C,0x1C,0x00,
	0x00,0x00,0x00,0x00,0x00,0xD8,0x6C,0x6C,0x6C,0x6C,0x6C,
	0x6C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x38,0x6C,0x0C,0x18,0x30,0x64,0x7C,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x7C,0x7C,0x7C,0x7C,0x7C,0x7C,0x7C,0x7C,0x7C,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,
};

uint8_t int10_font_24[256 * 48] = {
    /*
     * code=0, hex=0x00, ascii="^@"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=1, hex=0x01, ascii="^A"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x20,0x10,  /* 001000000001 */
    0x20,0x10,  /* 001000000001 */
    0x2C,0xD0,  /* 001011001101 */
    0x2C,0xD0,  /* 001011001101 */
    0x20,0x10,  /* 001000000001 */
    0x20,0x10,  /* 001000000001 */
    0x20,0x10,  /* 001000000001 */
    0x2C,0xD0,  /* 001011001101 */
    0x27,0x90,  /* 001001111001 */
    0x23,0x10,  /* 001000110001 */
    0x20,0x10,  /* 001000000001 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=2, hex=0x02, ascii="^B"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1F,0xE0,  /* 000111111110 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x33,0x30,  /* 001100110011 */
    0x38,0x70,  /* 001110000111 */
    0x3C,0xF0,  /* 001111001111 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x1F,0xE0,  /* 000111111110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=3, hex=0x03, ascii="^C"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x08,0x40,  /* 000010000100 */
    0x1C,0xE0,  /* 000111001110 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x1F,0xE0,  /* 000111111110 */
    0x1F,0xE0,  /* 000111111110 */
    0x1F,0xE0,  /* 000111111110 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=4, hex=0x04, ascii="^D"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x1F,0xE0,  /* 000111111110 */
    0x1F,0xE0,  /* 000111111110 */
    0x3F,0xF0,  /* 001111111111 */
    0x1F,0xE0,  /* 000111111110 */
    0x1F,0xE0,  /* 000111111110 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=5, hex=0x05, ascii="^E"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x07,0x80,  /* 000001111000 */
    0x03,0x00,  /* 000000110000 */
    0x1B,0x60,  /* 000110110110 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x1B,0x60,  /* 000110110110 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x0F,0xC0,  /* 000011111100 */
    0x1F,0xE0,  /* 000111111110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=6, hex=0x06, ascii="^F"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x1F,0xE0,  /* 000111111110 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x1F,0xE0,  /* 000111111110 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x0F,0xC0,  /* 000011111100 */
    0x1F,0xE0,  /* 000111111110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=7, hex=0x07, ascii="^G"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x07,0x80,  /* 000001111000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=8, hex=0x08, ascii="^H"
     */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFC,0xF0,  /* 111111001111 */
    0xF8,0x70,  /* 111110000111 */
    0xF0,0x30,  /* 111100000011 */
    0xF0,0x30,  /* 111100000011 */
    0xF8,0x70,  /* 111110000111 */
    0xFC,0xF0,  /* 111111001111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */

    /*
     * code=9, hex=0x09, ascii="^I"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x0C,0xC0,  /* 000011001100 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=10, hex=0x0A, ascii="^J"
     */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xF8,0x70,  /* 111110000111 */
    0xF3,0x30,  /* 111100110011 */
    0xE7,0x90,  /* 111001111001 */
    0xE7,0x90,  /* 111001111001 */
    0xF3,0x30,  /* 111100110011 */
    0xF8,0x70,  /* 111110000111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */

    /*
     * code=11, hex=0x0B, ascii="^K"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0xF0,  /* 000000111111 */
    0x00,0x70,  /* 000000000111 */
    0x00,0xF0,  /* 000000001111 */
    0x01,0xB0,  /* 000000011011 */
    0x03,0x30,  /* 000000110011 */
    0x06,0x00,  /* 000001100000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=12, hex=0x0C, ascii="^L"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x1F,0xE0,  /* 000111111110 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=13, hex=0x0D, ascii="^M"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0E,0x00,  /* 000011100000 */
    0x0F,0x00,  /* 000011110000 */
    0x0D,0x80,  /* 000011011000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0x60,  /* 000011000110 */
    0x0C,0x60,  /* 000011000110 */
    0x0C,0x60,  /* 000011000110 */
    0x0C,0x60,  /* 000011000110 */
    0x0C,0xC0,  /* 000011001100 */
    0x0F,0x80,  /* 000011111000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x3C,0x00,  /* 001111000000 */
    0x3C,0x00,  /* 001111000000 */
    0x38,0x00,  /* 001110000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=14, hex=0x0E, ascii="^N"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0F,0xF0,  /* 000011111111 */
    0x0C,0x30,  /* 000011000011 */
    0x0F,0xF0,  /* 000011111111 */
    0x0C,0x30,  /* 000011000011 */
    0x0C,0x30,  /* 000011000011 */
    0x0C,0x30,  /* 000011000011 */
    0x0C,0x30,  /* 000011000011 */
    0x0C,0x30,  /* 000011000011 */
    0x0C,0x30,  /* 000011000011 */
    0x0C,0x30,  /* 000011000011 */
    0x0C,0x30,  /* 000011000011 */
    0x0C,0x30,  /* 000011000011 */
    0x0C,0x30,  /* 000011000011 */
    0x0C,0xF0,  /* 000011001111 */
    0x3C,0xF0,  /* 001111001111 */
    0x3C,0xE0,  /* 001111001110 */
    0x38,0x00,  /* 001110000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=15, hex=0x0F, ascii="^O"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x33,0x30,  /* 001100110011 */
    0x1F,0xE0,  /* 000111111110 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x38,0x70,  /* 001110000111 */
    0x18,0x60,  /* 000110000110 */
    0x0C,0xC0,  /* 000011001100 */
    0x1F,0xE0,  /* 000111111110 */
    0x33,0x30,  /* 001100110011 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=16, hex=0x10, ascii="^P"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x00,  /* 001100000000 */
    0x38,0x00,  /* 001110000000 */
    0x3C,0x00,  /* 001111000000 */
    0x3E,0x00,  /* 001111100000 */
    0x3F,0x00,  /* 001111110000 */
    0x3F,0x80,  /* 001111111000 */
    0x3F,0xC0,  /* 001111111100 */
    0x3F,0xE0,  /* 001111111110 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xE0,  /* 001111111110 */
    0x3F,0xC0,  /* 001111111100 */
    0x3F,0x80,  /* 001111111000 */
    0x3F,0x00,  /* 001111110000 */
    0x3E,0x00,  /* 001111100000 */
    0x3C,0x00,  /* 001111000000 */
    0x38,0x00,  /* 001110000000 */
    0x30,0x00,  /* 001100000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=17, hex=0x11, ascii="^Q"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x70,  /* 000000000111 */
    0x00,0xF0,  /* 000000001111 */
    0x01,0xF0,  /* 000000011111 */
    0x03,0xF0,  /* 000000111111 */
    0x07,0xF0,  /* 000001111111 */
    0x0F,0xF0,  /* 000011111111 */
    0x1F,0xF0,  /* 000111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x1F,0xF0,  /* 000111111111 */
    0x0F,0xF0,  /* 000011111111 */
    0x07,0xF0,  /* 000001111111 */
    0x03,0xF0,  /* 000000111111 */
    0x01,0xF0,  /* 000000011111 */
    0x00,0xF0,  /* 000000001111 */
    0x00,0x70,  /* 000000000111 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=18, hex=0x12, ascii="^R"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x01,0x80,  /* 000000011000 */
    0x03,0xC0,  /* 000000111100 */
    0x07,0xE0,  /* 000001111110 */
    0x0F,0xF0,  /* 000011111111 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x0F,0xF0,  /* 000011111111 */
    0x07,0xE0,  /* 000001111110 */
    0x03,0xC0,  /* 000000111100 */
    0x01,0x80,  /* 000000011000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=19, hex=0x13, ascii="^S"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=20, hex=0x14, ascii="^T"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0xF0,  /* 000001111111 */
    0x1D,0xB0,  /* 000111011011 */
    0x39,0xB0,  /* 001110011011 */
    0x31,0xB0,  /* 001100011011 */
    0x31,0xB0,  /* 001100011011 */
    0x31,0xB0,  /* 001100011011 */
    0x19,0xB0,  /* 000110011011 */
    0x1D,0xB0,  /* 000111011011 */
    0x07,0xB0,  /* 000001111011 */
    0x01,0xB0,  /* 000000011011 */
    0x01,0xB0,  /* 000000011011 */
    0x01,0xB0,  /* 000000011011 */
    0x01,0xB0,  /* 000000011011 */
    0x01,0xB0,  /* 000000011011 */
    0x01,0xB0,  /* 000000011011 */
    0x01,0xB0,  /* 000000011011 */
    0x01,0xB0,  /* 000000011011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=21, hex=0x15, ascii="^U"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x00,  /* 001100000000 */
    0x1F,0x80,  /* 000111111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x0C,0xC0,  /* 000011001100 */
    0x07,0xE0,  /* 000001111110 */
    0x00,0x30,  /* 000000000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=22, hex=0x16, ascii="^V"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x1F,0xF0,  /* 000111111111 */
    0x1F,0xF0,  /* 000111111111 */
    0x1F,0xF0,  /* 000111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=23, hex=0x17, ascii="^W"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x0F,0xC0,  /* 000011111100 */
    0x1F,0xE0,  /* 000111111110 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x1F,0xE0,  /* 000111111110 */
    0x0F,0xC0,  /* 000011111100 */
    0x07,0x80,  /* 000001111000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=24, hex=0x18, ascii="^X"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x01,0x80,  /* 000000011000 */
    0x03,0xC0,  /* 000000111100 */
    0x07,0xE0,  /* 000001111110 */
    0x0F,0xF0,  /* 000011111111 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=25, hex=0x19, ascii="^Y"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x0F,0xF0,  /* 000011111111 */
    0x07,0xE0,  /* 000001111110 */
    0x03,0xC0,  /* 000000111100 */
    0x01,0x80,  /* 000000011000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=26, hex=0x1A, ascii="^Z"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x01,0x80,  /* 000000011000 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0x60,  /* 000000000110 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x60,  /* 000000000110 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=27, hex=0x1B, ascii="^["
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x18,0x00,  /* 000110000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x18,0x00,  /* 000110000000 */
    0x0C,0x00,  /* 000011000000 */
    0x06,0x00,  /* 000001100000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=28, hex=0x1C, ascii="^\"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0F,0xF0,  /* 000011111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=29, hex=0x1D, ascii="^]"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x04,0x80,  /* 000001001000 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x3F,0xF0,  /* 001111111111 */
    0x18,0x60,  /* 000110000110 */
    0x0C,0xC0,  /* 000011001100 */
    0x04,0x80,  /* 000001001000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=30, hex=0x1E, ascii="^^"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x1F,0xE0,  /* 000111111110 */
    0x1F,0xE0,  /* 000111111110 */
    0x1F,0xE0,  /* 000111111110 */
    0x1F,0xE0,  /* 000111111110 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=31, hex=0x1F, ascii="^_"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x3F,0xF0,  /* 001111111111 */
    0x1F,0xE0,  /* 000111111110 */
    0x1F,0xE0,  /* 000111111110 */
    0x1F,0xE0,  /* 000111111110 */
    0x1F,0xE0,  /* 000111111110 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=32, hex=0x20, ascii=" "
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=33, hex=0x21, ascii="!"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=34, hex=0x22, ascii="""
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=35, hex=0x23, ascii="#"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x3F,0xF0,  /* 001111111111 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x3F,0xF0,  /* 001111111111 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=36, hex=0x24, ascii="$"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x1F,0xE0,  /* 000111111110 */
    0x1B,0x60,  /* 000110110110 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x00,  /* 001100110000 */
    0x1B,0x00,  /* 000110110000 */
    0x1F,0x00,  /* 000111110000 */
    0x07,0x80,  /* 000001111000 */
    0x03,0xE0,  /* 000000111110 */
    0x03,0x60,  /* 000000110110 */
    0x03,0x30,  /* 000000110011 */
    0x33,0x30,  /* 001100110011 */
    0x1B,0x60,  /* 000110110110 */
    0x1F,0xE0,  /* 000111111110 */
    0x07,0x80,  /* 000001111000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=37, hex=0x25, ascii="%"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x1C,0x30,  /* 000111000011 */
    0x36,0x60,  /* 001101100110 */
    0x36,0x60,  /* 001101100110 */
    0x1C,0xC0,  /* 000111001100 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x18,0xE0,  /* 000110001110 */
    0x19,0xB0,  /* 000110011011 */
    0x31,0xB0,  /* 001100011011 */
    0x30,0xE0,  /* 001100001110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=38, hex=0x26, ascii="&"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0E,0x00,  /* 000011100000 */
    0x1B,0x00,  /* 000110110000 */
    0x31,0x80,  /* 001100011000 */
    0x31,0x80,  /* 001100011000 */
    0x31,0x80,  /* 001100011000 */
    0x1B,0x00,  /* 000110110000 */
    0x0E,0x00,  /* 000011100000 */
    0x0C,0x00,  /* 000011000000 */
    0x1E,0x30,  /* 000111100011 */
    0x33,0x30,  /* 001100110011 */
    0x31,0xE0,  /* 001100011110 */
    0x30,0xC0,  /* 001100001100 */
    0x31,0xC0,  /* 001100011100 */
    0x33,0x60,  /* 001100110110 */
    0x1E,0x30,  /* 000111100011 */
    0x0C,0x10,  /* 000011000001 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=39, hex=0x27, ascii="'"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=40, hex=0x28, ascii="("
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=41, hex=0x29, ascii=")"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=42, hex=0x2A, ascii="*"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x18,0x60,  /* 000110000110 */
    0x0C,0xC0,  /* 000011001100 */
    0x07,0x80,  /* 000001111000 */
    0x3F,0xF0,  /* 001111111111 */
    0x07,0x80,  /* 000001111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=43, hex=0x2B, ascii="+"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x3F,0xF0,  /* 001111111111 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=44, hex=0x2C, ascii=","
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=45, hex=0x2D, ascii="-"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=46, hex=0x2E, ascii="."
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=47, hex=0x2F, ascii="/"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=48, hex=0x30, ascii="0"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=49, hex=0x31, ascii="1"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x00,  /* 000001110000 */
    0x0F,0x00,  /* 000011110000 */
    0x1B,0x00,  /* 000110110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=50, hex=0x32, ascii="2"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x60,  /* 000000000110 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x18,0x00,  /* 000110000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=51, hex=0x33, ascii="3"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x60,  /* 000000000110 */
    0x00,0xC0,  /* 000000001100 */
    0x07,0xC0,  /* 000001111100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=52, hex=0x34, ascii="4"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0xC0,  /* 000000011100 */
    0x01,0xC0,  /* 000000011100 */
    0x03,0xC0,  /* 000000111100 */
    0x03,0xC0,  /* 000000111100 */
    0x06,0xC0,  /* 000001101100 */
    0x06,0xC0,  /* 000001101100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0xC0,  /* 000110001100 */
    0x18,0xC0,  /* 000110001100 */
    0x30,0xC0,  /* 001100001100 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=53, hex=0x35, ascii="5"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x3F,0x80,  /* 001111111000 */
    0x30,0xE0,  /* 001100001110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=54, hex=0x36, ascii="6"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x01,0xC0,  /* 000000011100 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x3F,0x80,  /* 001111111000 */
    0x38,0xE0,  /* 001110001110 */
    0x30,0x60,  /* 001100000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=55, hex=0x37, ascii="7"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=56, hex=0x38, ascii="8"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x0C,0xC0,  /* 000011001100 */
    0x0F,0xC0,  /* 000011111100 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=57, hex=0x39, ascii="9"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x30,  /* 000110000011 */
    0x1C,0x70,  /* 000111000111 */
    0x07,0xF0,  /* 000001111111 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x0E,0x00,  /* 000011100000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=58, hex=0x3A, ascii=":"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x00,  /* 000001110000 */
    0x07,0x00,  /* 000001110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x00,  /* 000001110000 */
    0x07,0x00,  /* 000001110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=59, hex=0x3B, ascii=";"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x00,  /* 000001110000 */
    0x07,0x00,  /* 000001110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x00,  /* 000001110000 */
    0x07,0x00,  /* 000001110000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=60, hex=0x3C, ascii="<"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x60,  /* 000000000110 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x18,0x00,  /* 000110000000 */
    0x30,0x00,  /* 001100000000 */
    0x18,0x00,  /* 000110000000 */
    0x0C,0x00,  /* 000011000000 */
    0x06,0x00,  /* 000001100000 */
    0x03,0x00,  /* 000000110000 */
    0x01,0x80,  /* 000000011000 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=61, hex=0x3D, ascii="="
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=62, hex=0x3E, ascii=">"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x00,  /* 001100000000 */
    0x18,0x00,  /* 000110000000 */
    0x0C,0x00,  /* 000011000000 */
    0x06,0x00,  /* 000001100000 */
    0x03,0x00,  /* 000000110000 */
    0x01,0x80,  /* 000000011000 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x60,  /* 000000000110 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x18,0x00,  /* 000110000000 */
    0x30,0x00,  /* 001100000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=63, hex=0x3F, ascii="?"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x60,  /* 000000000110 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=64, hex=0x40, ascii="@"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x31,0xF0,  /* 001100011111 */
    0x33,0x70,  /* 001100110111 */
    0x36,0x30,  /* 001101100011 */
    0x36,0x30,  /* 001101100011 */
    0x36,0x30,  /* 001101100011 */
    0x36,0x30,  /* 001101100011 */
    0x33,0x30,  /* 001100110011 */
    0x31,0xF0,  /* 001100011111 */
    0x30,0x00,  /* 001100000000 */
    0x18,0x00,  /* 000110000000 */
    0x1C,0x00,  /* 000111000000 */
    0x07,0xF0,  /* 000001111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=65, hex=0x41, ascii="A"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x1F,0xE0,  /* 000111111110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=66, hex=0x42, ascii="B"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0x80,  /* 001111111000 */
    0x30,0xE0,  /* 001100001110 */
    0x30,0x60,  /* 001100000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x60,  /* 001100000110 */
    0x30,0xC0,  /* 001100001100 */
    0x3F,0xC0,  /* 001111111100 */
    0x30,0xC0,  /* 001100001100 */
    0x30,0x60,  /* 001100000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x60,  /* 001100000110 */
    0x30,0xE0,  /* 001100001110 */
    0x3F,0x80,  /* 001111111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=67, hex=0x43, ascii="C"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0xC0,  /* 000001111100 */
    0x1C,0x60,  /* 000111000110 */
    0x18,0x30,  /* 000110000011 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x18,0x30,  /* 000110000011 */
    0x1C,0x60,  /* 000111000110 */
    0x07,0xC0,  /* 000001111100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=68, hex=0x44, ascii="D"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0x80,  /* 001111111000 */
    0x30,0xE0,  /* 001100001110 */
    0x30,0x60,  /* 001100000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x60,  /* 001100000110 */
    0x30,0xE0,  /* 001100001110 */
    0x3F,0x80,  /* 001111111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=69, hex=0x45, ascii="E"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x3F,0x80,  /* 001111111000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=70, hex=0x46, ascii="F"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x3F,0x80,  /* 001111111000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=71, hex=0x47, ascii="G"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0xC0,  /* 000001111100 */
    0x1C,0x60,  /* 000111000110 */
    0x18,0x30,  /* 000110000011 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x31,0xF0,  /* 001100011111 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x30,  /* 000110000011 */
    0x1C,0x30,  /* 000111000011 */
    0x07,0xF0,  /* 000001111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=72, hex=0x48, ascii="H"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x3F,0xF0,  /* 001111111111 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=73, hex=0x49, ascii="I"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x1F,0xE0,  /* 000111111110 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x1F,0xE0,  /* 000111111110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=74, hex=0x4A, ascii="J"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0xF0,  /* 000000111111 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x30,0xC0,  /* 001100001100 */
    0x30,0xC0,  /* 001100001100 */
    0x19,0x80,  /* 000110011000 */
    0x1F,0x80,  /* 000111111000 */
    0x06,0x00,  /* 000001100000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=75, hex=0x4B, ascii="K"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x10,  /* 001100000001 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x60,  /* 001100000110 */
    0x30,0xC0,  /* 001100001100 */
    0x31,0x80,  /* 001100011000 */
    0x33,0x00,  /* 001100110000 */
    0x36,0x00,  /* 001101100000 */
    0x3C,0x00,  /* 001111000000 */
    0x38,0x00,  /* 001110000000 */
    0x3C,0x00,  /* 001111000000 */
    0x36,0x00,  /* 001101100000 */
    0x33,0x00,  /* 001100110000 */
    0x31,0x80,  /* 001100011000 */
    0x30,0xC0,  /* 001100001100 */
    0x30,0x60,  /* 001100000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x10,  /* 001100000001 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=76, hex=0x4C, ascii="L"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=77, hex=0x4D, ascii="M"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x38,0x70,  /* 001110000111 */
    0x3C,0xF0,  /* 001111001111 */
    0x37,0xB0,  /* 001101111011 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=78, hex=0x4E, ascii="N"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x38,0x30,  /* 001110000011 */
    0x38,0x30,  /* 001110000011 */
    0x3C,0x30,  /* 001111000011 */
    0x3C,0x30,  /* 001111000011 */
    0x36,0x30,  /* 001101100011 */
    0x36,0x30,  /* 001101100011 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x31,0xB0,  /* 001100011011 */
    0x31,0xB0,  /* 001100011011 */
    0x30,0xF0,  /* 001100001111 */
    0x30,0xF0,  /* 001100001111 */
    0x30,0x70,  /* 001100000111 */
    0x30,0x70,  /* 001100000111 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=79, hex=0x4F, ascii="O"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=80, hex=0x50, ascii="P"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0x80,  /* 001111111000 */
    0x30,0xE0,  /* 001100001110 */
    0x30,0x60,  /* 001100000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x60,  /* 001100000110 */
    0x30,0xE0,  /* 001100001110 */
    0x3F,0x80,  /* 001111111000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=81, hex=0x51, ascii="Q"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x31,0xB0,  /* 001100011011 */
    0x18,0xE0,  /* 000110001110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0xB0,  /* 000001111011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=82, hex=0x52, ascii="R"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0x80,  /* 001111111000 */
    0x30,0xE0,  /* 001100001110 */
    0x30,0x60,  /* 001100000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x60,  /* 001100000110 */
    0x30,0xE0,  /* 001100001110 */
    0x3F,0x80,  /* 001111111000 */
    0x33,0x00,  /* 001100110000 */
    0x33,0x00,  /* 001100110000 */
    0x31,0x80,  /* 001100011000 */
    0x30,0xC0,  /* 001100001100 */
    0x30,0xC0,  /* 001100001100 */
    0x30,0x60,  /* 001100000110 */
    0x30,0x60,  /* 001100000110 */
    0x30,0x30,  /* 001100000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=83, hex=0x53, ascii="S"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x18,0x00,  /* 000110000000 */
    0x1C,0x00,  /* 000111000000 */
    0x07,0x80,  /* 000001111000 */
    0x00,0xE0,  /* 000000001110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=84, hex=0x54, ascii="T"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=85, hex=0x55, ascii="U"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=86, hex=0x56, ascii="V"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=87, hex=0x57, ascii="W"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x3F,0xF0,  /* 001111111111 */
    0x1C,0xE0,  /* 000111001110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=88, hex=0x58, ascii="X"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=89, hex=0x59, ascii="Y"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=90, hex=0x5A, ascii="Z"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x30,0x00,  /* 001100000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=91, hex=0x5B, ascii="["
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x1F,0xC0,  /* 000111111100 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x1F,0xC0,  /* 000111111100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=92, hex=0x5C, ascii="\"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x00,  /* 001100000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=93, hex=0x5D, ascii="]"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0F,0xE0,  /* 000011111110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x0F,0xE0,  /* 000011111110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=94, hex=0x5E, ascii="^"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=95, hex=0x5F, ascii="_"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0xFF,0xF0,  /* 111111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=96, hex=0x60, ascii="`"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=97, hex=0x61, ascii="a"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0F,0xC0,  /* 000011111100 */
    0x18,0x60,  /* 000110000110 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x0F,0xF0,  /* 000011111111 */
    0x18,0x30,  /* 000110000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x70,  /* 001100000111 */
    0x18,0xF0,  /* 000110001111 */
    0x0F,0xB0,  /* 000011111011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=98, hex=0x62, ascii="b"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x37,0x80,  /* 001101111000 */
    0x3C,0xE0,  /* 001111001110 */
    0x38,0x60,  /* 001110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x38,0x60,  /* 001110000110 */
    0x3C,0xE0,  /* 001111001110 */
    0x37,0x80,  /* 001101111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=99, hex=0x63, ascii="c"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0xC0,  /* 000001111100 */
    0x1C,0x60,  /* 000111000110 */
    0x18,0x30,  /* 000110000011 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x18,0x30,  /* 000110000011 */
    0x1C,0x60,  /* 000111000110 */
    0x07,0xC0,  /* 000001111100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=100, hex=0x64, ascii="d"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x07,0xB0,  /* 000001111011 */
    0x1C,0xF0,  /* 000111001111 */
    0x18,0x70,  /* 000110000111 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x70,  /* 000110000111 */
    0x1C,0xF0,  /* 000111001111 */
    0x07,0xB0,  /* 000001111011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=101, hex=0x65, ascii="e"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x3F,0xF0,  /* 001111111111 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x18,0x30,  /* 000110000011 */
    0x1C,0x60,  /* 000111000110 */
    0x07,0xC0,  /* 000001111100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=102, hex=0x66, ascii="f"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x01,0xE0,  /* 000000011110 */
    0x03,0x30,  /* 000000110011 */
    0x03,0x30,  /* 000000110011 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x0F,0xC0,  /* 000011111100 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=103, hex=0x67, ascii="g"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0xB0,  /* 000001111011 */
    0x1C,0xF0,  /* 000111001111 */
    0x18,0x70,  /* 000110000111 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x70,  /* 000110000111 */
    0x1C,0xF0,  /* 000111001111 */
    0x07,0xB0,  /* 000001111011 */
    0x00,0x30,  /* 000000000011 */
    0x30,0x60,  /* 001100000110 */
    0x18,0xE0,  /* 000110001110 */
    0x0F,0x80,  /* 000011111000 */

    /*
     * code=104, hex=0x68, ascii="h"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x37,0x80,  /* 001101111000 */
    0x3C,0xE0,  /* 001111001110 */
    0x38,0x60,  /* 001110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=105, hex=0x69, ascii="i"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x00,  /* 000001110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x0F,0xC0,  /* 000011111100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=106, hex=0x6A, ascii="j"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x80,  /* 000000111000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x31,0x80,  /* 001100011000 */
    0x31,0x80,  /* 001100011000 */
    0x1B,0x80,  /* 000110111000 */
    0x0F,0x00,  /* 000011110000 */

    /*
     * code=107, hex=0x6B, ascii="k"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x60,  /* 001100000110 */
    0x30,0xC0,  /* 001100001100 */
    0x31,0x80,  /* 001100011000 */
    0x33,0x00,  /* 001100110000 */
    0x36,0x00,  /* 001101100000 */
    0x3F,0x00,  /* 001111110000 */
    0x39,0x80,  /* 001110011000 */
    0x30,0xC0,  /* 001100001100 */
    0x30,0x60,  /* 001100000110 */
    0x30,0x30,  /* 001100000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=108, hex=0x6C, ascii="l"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x80,  /* 000000111000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x07,0xE0,  /* 000001111110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=109, hex=0x6D, ascii="m"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x36,0xC0,  /* 001101101100 */
    0x3F,0xE0,  /* 001111111110 */
    0x3B,0x70,  /* 001110110111 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=110, hex=0x6E, ascii="n"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x37,0x80,  /* 001101111000 */
    0x3C,0xE0,  /* 001111001110 */
    0x38,0x60,  /* 001110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=111, hex=0x6F, ascii="o"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=112, hex=0x70, ascii="p"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x37,0x80,  /* 001101111000 */
    0x3C,0xE0,  /* 001111001110 */
    0x38,0x60,  /* 001110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x38,0x60,  /* 001110000110 */
    0x3C,0xE0,  /* 001111001110 */
    0x37,0x80,  /* 001101111000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */

    /*
     * code=113, hex=0x71, ascii="q"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0xB0,  /* 000001111011 */
    0x1C,0xF0,  /* 000111001111 */
    0x18,0x70,  /* 000110000111 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x70,  /* 000110000111 */
    0x1C,0xF0,  /* 000111001111 */
    0x07,0xB0,  /* 000001111011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */

    /*
     * code=114, hex=0x72, ascii="r"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x37,0xC0,  /* 001101111100 */
    0x3C,0x60,  /* 001111000110 */
    0x38,0x30,  /* 001110000011 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=115, hex=0x73, ascii="s"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0F,0xC0,  /* 000011111100 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x00,  /* 001100000000 */
    0x18,0x00,  /* 000110000000 */
    0x0F,0xC0,  /* 000011111100 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x30,  /* 000000000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x0F,0xC0,  /* 000011111100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=116, hex=0x74, ascii="t"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x1F,0x80,  /* 000111111000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x60,  /* 000001100110 */
    0x06,0x60,  /* 000001100110 */
    0x03,0xC0,  /* 000000111100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=117, hex=0x75, ascii="u"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x70,  /* 000110000111 */
    0x1C,0xF0,  /* 000111001111 */
    0x07,0xB0,  /* 000001111011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=118, hex=0x76, ascii="v"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=119, hex=0x77, ascii="w"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x3F,0xF0,  /* 001111111111 */
    0x1C,0xE0,  /* 000111001110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=120, hex=0x78, ascii="x"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x0C,0xC0,  /* 000011001100 */
    0x07,0x80,  /* 000001111000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=121, hex=0x79, ascii="y"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x70,  /* 000110000111 */
    0x1C,0xF0,  /* 000111001111 */
    0x07,0xB0,  /* 000001111011 */
    0x00,0x30,  /* 000000000011 */
    0x30,0x60,  /* 001100000110 */
    0x18,0xE0,  /* 000110001110 */
    0x0F,0x80,  /* 000011111000 */

    /*
     * code=122, hex=0x7A, ascii="z"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x60,  /* 000000000110 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x18,0x00,  /* 000110000000 */
    0x30,0x00,  /* 001100000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=123, hex=0x7B, ascii="{"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0xE0,  /* 000000001110 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x06,0x00,  /* 000001100000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x01,0x80,  /* 000000011000 */
    0x00,0xE0,  /* 000000001110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=124, hex=0x7C, ascii="|"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=125, hex=0x7D, ascii="}"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0E,0x00,  /* 000011100000 */
    0x03,0x00,  /* 000000110000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0x60,  /* 000000000110 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x0E,0x00,  /* 000011100000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=126, hex=0x7E, ascii="~"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x1C,0x00,  /* 000111000000 */
    0x36,0x30,  /* 001101100011 */
    0x63,0x60,  /* 011000110110 */
    0x01,0xC0,  /* 000000011100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=127, hex=0x7F, ascii="^?"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=128, hex=0x80, ascii="!^@"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0xC0,  /* 000001111100 */
    0x1C,0x60,  /* 000111000110 */
    0x18,0x30,  /* 000110000011 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x18,0x30,  /* 000110000011 */
    0x1C,0x60,  /* 000111000110 */
    0x07,0xC0,  /* 000001111100 */
    0x01,0x80,  /* 000000011000 */
    0x00,0xC0,  /* 000000001100 */
    0x18,0xC0,  /* 000110001100 */
    0x0F,0x80,  /* 000011111000 */

    /*
     * code=129, hex=0x81, ascii="!^A"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x70,  /* 000110000111 */
    0x1C,0xF0,  /* 000111001111 */
    0x07,0xB0,  /* 000001111011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=130, hex=0x82, ascii="!^B"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x3F,0xF0,  /* 001111111111 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x18,0x30,  /* 000110000011 */
    0x1C,0x60,  /* 000111000110 */
    0x07,0xC0,  /* 000001111100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=131, hex=0x83, ascii="!^C"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0F,0xC0,  /* 000011111100 */
    0x18,0x60,  /* 000110000110 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x0F,0xF0,  /* 000011111111 */
    0x18,0x30,  /* 000110000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x70,  /* 001100000111 */
    0x18,0xF0,  /* 000110001111 */
    0x0F,0xB0,  /* 000011111011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=132, hex=0x84, ascii="!^D"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0F,0xC0,  /* 000011111100 */
    0x18,0x60,  /* 000110000110 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x0F,0xF0,  /* 000011111111 */
    0x18,0x30,  /* 000110000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x70,  /* 001100000111 */
    0x18,0xF0,  /* 000110001111 */
    0x0F,0xB0,  /* 000011111011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=133, hex=0x85, ascii="!^E"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0x00,  /* 000011000000 */
    0x06,0x00,  /* 000001100000 */
    0x03,0x00,  /* 000000110000 */
    0x01,0x80,  /* 000000011000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0F,0xC0,  /* 000011111100 */
    0x18,0x60,  /* 000110000110 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x0F,0xF0,  /* 000011111111 */
    0x18,0x30,  /* 000110000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x70,  /* 001100000111 */
    0x18,0xF0,  /* 000110001111 */
    0x0F,0xB0,  /* 000011111011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=134, hex=0x86, ascii="!^F"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x0C,0xC0,  /* 000011001100 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x0F,0xC0,  /* 000011111100 */
    0x18,0x60,  /* 000110000110 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x0F,0xF0,  /* 000011111111 */
    0x18,0x30,  /* 000110000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x70,  /* 001100000111 */
    0x18,0xF0,  /* 000110001111 */
    0x0F,0xB0,  /* 000011111011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=135, hex=0x87, ascii="!^G"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0xC0,  /* 000001111100 */
    0x1C,0x60,  /* 000111000110 */
    0x18,0x30,  /* 000110000011 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x18,0x30,  /* 000110000011 */
    0x1C,0x60,  /* 000111000110 */
    0x07,0xC0,  /* 000001111100 */
    0x01,0x80,  /* 000000011000 */
    0x00,0xC0,  /* 000000001100 */
    0x18,0xC0,  /* 000110001100 */
    0x0F,0x80,  /* 000011111000 */

    /*
     * code=136, hex=0x88, ascii="!^H"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x3F,0xF0,  /* 001111111111 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x18,0x30,  /* 000110000011 */
    0x1C,0x60,  /* 000111000110 */
    0x07,0xC0,  /* 000001111100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=137, hex=0x89, ascii="!^I"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x3F,0xF0,  /* 001111111111 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x18,0x30,  /* 000110000011 */
    0x1C,0x60,  /* 000111000110 */
    0x07,0xC0,  /* 000001111100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=138, hex=0x8A, ascii="!^J"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0x00,  /* 000011000000 */
    0x06,0x00,  /* 000001100000 */
    0x03,0x00,  /* 000000110000 */
    0x01,0x80,  /* 000000011000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x3F,0xF0,  /* 001111111111 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x18,0x30,  /* 000110000011 */
    0x1C,0x60,  /* 000111000110 */
    0x07,0xC0,  /* 000001111100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=139, hex=0x8B, ascii="!^K"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x00,  /* 000001110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x0F,0xC0,  /* 000011111100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=140, hex=0x8C, ascii="!^L"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x00,  /* 000001110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x0F,0xC0,  /* 000011111100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=141, hex=0x8D, ascii="!^M"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0x00,  /* 000011000000 */
    0x06,0x00,  /* 000001100000 */
    0x03,0x00,  /* 000000110000 */
    0x01,0x80,  /* 000000011000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x00,  /* 000001110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x0F,0xC0,  /* 000011111100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=142, hex=0x8E, ascii="!^N"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x1F,0xE0,  /* 000111111110 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=143, hex=0x8F, ascii="!^O"
     */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x0C,0xC0,  /* 000011001100 */
    0x07,0x80,  /* 000001111000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x1F,0xE0,  /* 000111111110 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=144, hex=0x90, ascii="!^P"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x3F,0xF0,  /* 001111111111 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x3F,0x80,  /* 001111111000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=145, hex=0x91, ascii="!^Q"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x1C,0xE0,  /* 000111001110 */
    0x37,0xB0,  /* 001101111011 */
    0x03,0x30,  /* 000000110011 */
    0x03,0x30,  /* 000000110011 */
    0x03,0x30,  /* 000000110011 */
    0x1F,0xE0,  /* 000111111110 */
    0x33,0x00,  /* 001100110000 */
    0x33,0x00,  /* 001100110000 */
    0x33,0x00,  /* 001100110000 */
    0x37,0xB0,  /* 001101111011 */
    0x1E,0xE0,  /* 000111101110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=146, hex=0x92, ascii="!^R"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x01,0xF0,  /* 000000011111 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x80,  /* 000000111000 */
    0x03,0x80,  /* 000000111000 */
    0x03,0x80,  /* 000000111000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x05,0x80,  /* 000001011000 */
    0x0D,0xE0,  /* 000011011110 */
    0x0D,0x80,  /* 000011011000 */
    0x0F,0x80,  /* 000011111000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x31,0x80,  /* 001100011000 */
    0x31,0x80,  /* 001100011000 */
    0x31,0xF0,  /* 001100011111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=147, hex=0x93, ascii="!^S"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=148, hex=0x94, ascii="!^T"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=149, hex=0x95, ascii="!^U"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0x00,  /* 000011000000 */
    0x06,0x00,  /* 000001100000 */
    0x03,0x00,  /* 000000110000 */
    0x01,0x80,  /* 000000011000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=150, hex=0x96, ascii="!^V"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x70,  /* 000110000111 */
    0x1C,0xF0,  /* 000111001111 */
    0x07,0xB0,  /* 000001111011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=151, hex=0x97, ascii="!^W"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0x00,  /* 000011000000 */
    0x06,0x00,  /* 000001100000 */
    0x03,0x00,  /* 000000110000 */
    0x01,0x80,  /* 000000011000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x70,  /* 000110000111 */
    0x1C,0xF0,  /* 000111001111 */
    0x07,0xB0,  /* 000001111011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=152, hex=0x98, ascii="!^X"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x70,  /* 000110000111 */
    0x1C,0xF0,  /* 000111001111 */
    0x07,0xB0,  /* 000001111011 */
    0x00,0x30,  /* 000000000011 */
    0x30,0x60,  /* 001100000110 */
    0x18,0xE0,  /* 000110001110 */
    0x0F,0x80,  /* 000011111000 */

    /*
     * code=153, hex=0x99, ascii="!^Y"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=154, hex=0x9A, ascii="!^Z"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=155, hex=0x9B, ascii="!^["
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0xC0,  /* 000001111100 */
    0x1C,0x60,  /* 000111000110 */
    0x18,0x30,  /* 000110000011 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x18,0x30,  /* 000110000011 */
    0x1C,0x60,  /* 000111000110 */
    0x07,0xC0,  /* 000001111100 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=156, hex=0x9C, ascii="!^\"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0xE0,  /* 000000111110 */
    0x06,0x30,  /* 000001100011 */
    0x06,0x30,  /* 000001100011 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x1F,0x80,  /* 000111111000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x1E,0x00,  /* 000111100000 */
    0x1F,0xF0,  /* 000111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=157, hex=0x9D, ascii="!^]"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x03,0x00,  /* 000000110000 */
    0x1F,0xE0,  /* 000111111110 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x1F,0xE0,  /* 000111111110 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=158, hex=0x9E, ascii="!^^"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0x80,  /* 001111111000 */
    0x30,0xE0,  /* 001100001110 */
    0x30,0x60,  /* 001100000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x60,  /* 001100000110 */
    0x30,0xE0,  /* 001100001110 */
    0x3F,0x80,  /* 001111111000 */
    0x30,0x80,  /* 001100001000 */
    0x31,0x80,  /* 001100011000 */
    0x37,0xE0,  /* 001101111110 */
    0x31,0x80,  /* 001100011000 */
    0x31,0x80,  /* 001100011000 */
    0x31,0x80,  /* 001100011000 */
    0x31,0xB0,  /* 001100011011 */
    0x30,0xE0,  /* 001100001110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=159, hex=0x9F, ascii="!^_"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x01,0xE0,  /* 000000011110 */
    0x03,0x30,  /* 000000110011 */
    0x03,0x30,  /* 000000110011 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x0F,0xC0,  /* 000011111100 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x33,0x00,  /* 001100110000 */
    0x33,0x00,  /* 001100110000 */
    0x1E,0x00,  /* 000111100000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=160, hex=0xA0, ascii="! "
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0F,0xC0,  /* 000011111100 */
    0x18,0x60,  /* 000110000110 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x0F,0xF0,  /* 000011111111 */
    0x18,0x30,  /* 000110000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x70,  /* 001100000111 */
    0x18,0xF0,  /* 000110001111 */
    0x0F,0xB0,  /* 000011111011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=161, hex=0xA1, ascii="!!"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x00,  /* 000001110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x0F,0xC0,  /* 000011111100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=162, hex=0xA2, ascii="!""
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=163, hex=0xA3, ascii="!#"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x70,  /* 000110000111 */
    0x1C,0xF0,  /* 000111001111 */
    0x07,0xB0,  /* 000001111011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=164, hex=0xA4, ascii="!$"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x1C,0x00,  /* 000111000000 */
    0x36,0x30,  /* 001101100011 */
    0x63,0x60,  /* 011000110110 */
    0x01,0xC0,  /* 000000011100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x37,0x80,  /* 001101111000 */
    0x3C,0xE0,  /* 001111001110 */
    0x38,0x60,  /* 001110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=165, hex=0xA5, ascii="!%"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x1C,0x00,  /* 000111000000 */
    0x36,0x30,  /* 001101100011 */
    0x63,0x60,  /* 011000110110 */
    0x01,0xC0,  /* 000000011100 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x38,0x30,  /* 001110000011 */
    0x3C,0x30,  /* 001111000011 */
    0x36,0x30,  /* 001101100011 */
    0x33,0x30,  /* 001100110011 */
    0x31,0xB0,  /* 001100011011 */
    0x30,0xF0,  /* 001100001111 */
    0x30,0x70,  /* 001100000111 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=166, hex=0xA6, ascii="!&"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0E,0x80,  /* 000011101000 */
    0x1B,0x80,  /* 000110111000 */
    0x31,0x80,  /* 001100011000 */
    0x31,0x80,  /* 001100011000 */
    0x1B,0x80,  /* 000110111000 */
    0x0E,0xC0,  /* 000011101100 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xC0,  /* 001111111100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=167, hex=0xA7, ascii="!'"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0F,0x00,  /* 000011110000 */
    0x19,0x80,  /* 000110011000 */
    0x30,0xC0,  /* 001100001100 */
    0x30,0xC0,  /* 001100001100 */
    0x19,0x80,  /* 000110011000 */
    0x0F,0x00,  /* 000011110000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xC0,  /* 001111111100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=168, hex=0xA8, ascii="!("
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x18,0x00,  /* 000110000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=169, hex=0xA9, ascii="!)"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0F,0xF0,  /* 000011111111 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=170, hex=0xAA, ascii="!*"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0F,0xF0,  /* 000011111111 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=171, hex=0xAB, ascii="!+"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x18,0x30,  /* 000110000011 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x18,0xC0,  /* 000110001100 */
    0x18,0xC0,  /* 000110001100 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0xE0,  /* 000001101110 */
    0x0D,0xB0,  /* 000011011011 */
    0x0C,0x30,  /* 000011000011 */
    0x18,0x60,  /* 000110000110 */
    0x18,0xC0,  /* 000110001100 */
    0x31,0x80,  /* 001100011000 */
    0x31,0xF0,  /* 001100011111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=172, hex=0xAC, ascii="!,"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x18,0x30,  /* 000110000011 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x18,0xC0,  /* 000110001100 */
    0x18,0xC0,  /* 000110001100 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x60,  /* 000001100110 */
    0x06,0xE0,  /* 000001101110 */
    0x0D,0xE0,  /* 000011011110 */
    0x0D,0x60,  /* 000011010110 */
    0x1B,0x60,  /* 000110110110 */
    0x1B,0xF0,  /* 000110111111 */
    0x30,0x60,  /* 001100000110 */
    0x30,0x60,  /* 001100000110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=173, hex=0xAD, ascii="!-"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=174, hex=0xAE, ascii="!."
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x30,  /* 000000110011 */
    0x06,0x60,  /* 000001100110 */
    0x0C,0xC0,  /* 000011001100 */
    0x19,0x80,  /* 000110011000 */
    0x33,0x00,  /* 001100110000 */
    0x19,0x80,  /* 000110011000 */
    0x0C,0xC0,  /* 000011001100 */
    0x06,0x60,  /* 000001100110 */
    0x03,0x30,  /* 000000110011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=175, hex=0xAF, ascii="!/"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x33,0x00,  /* 001100110000 */
    0x19,0x80,  /* 000110011000 */
    0x0C,0xC0,  /* 000011001100 */
    0x06,0x60,  /* 000001100110 */
    0x03,0x30,  /* 000000110011 */
    0x06,0x60,  /* 000001100110 */
    0x0C,0xC0,  /* 000011001100 */
    0x19,0x80,  /* 000110011000 */
    0x33,0x00,  /* 001100110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=176, hex=0xB0, ascii="!0"
     */
    0x22,0x20,  /* 001000100010 */
    0x88,0x80,  /* 100010001000 */
    0x22,0x20,  /* 001000100010 */
    0x88,0x80,  /* 100010001000 */
    0x22,0x20,  /* 001000100010 */
    0x88,0x80,  /* 100010001000 */
    0x22,0x20,  /* 001000100010 */
    0x88,0x80,  /* 100010001000 */
    0x22,0x20,  /* 001000100010 */
    0x88,0x80,  /* 100010001000 */
    0x22,0x20,  /* 001000100010 */
    0x88,0x80,  /* 100010001000 */
    0x22,0x20,  /* 001000100010 */
    0x88,0x80,  /* 100010001000 */
    0x22,0x20,  /* 001000100010 */
    0x88,0x80,  /* 100010001000 */
    0x22,0x20,  /* 001000100010 */
    0x88,0x80,  /* 100010001000 */
    0x22,0x20,  /* 001000100010 */
    0x88,0x80,  /* 100010001000 */
    0x22,0x20,  /* 001000100010 */
    0x88,0x80,  /* 100010001000 */
    0x22,0x20,  /* 001000100010 */
    0x88,0x80,  /* 100010001000 */

    /*
     * code=177, hex=0xB1, ascii="!1"
     */
    0x55,0x50,  /* 010101010101 */
    0xAA,0xA0,  /* 101010101010 */
    0x55,0x50,  /* 010101010101 */
    0xAA,0xA0,  /* 101010101010 */
    0x55,0x50,  /* 010101010101 */
    0xAA,0xA0,  /* 101010101010 */
    0x55,0x50,  /* 010101010101 */
    0xAA,0xA0,  /* 101010101010 */
    0x55,0x50,  /* 010101010101 */
    0xAA,0xA0,  /* 101010101010 */
    0x55,0x50,  /* 010101010101 */
    0xAA,0xA0,  /* 101010101010 */
    0x55,0x50,  /* 010101010101 */
    0xAA,0xA0,  /* 101010101010 */
    0x55,0x50,  /* 010101010101 */
    0xAA,0xA0,  /* 101010101010 */
    0x55,0x50,  /* 010101010101 */
    0xAA,0xA0,  /* 101010101010 */
    0x55,0x50,  /* 010101010101 */
    0xAA,0xA0,  /* 101010101010 */
    0x55,0x50,  /* 010101010101 */
    0xAA,0xA0,  /* 101010101010 */
    0x55,0x50,  /* 010101010101 */
    0xAA,0xA0,  /* 101010101010 */

    /*
     * code=178, hex=0xB2, ascii="!2"
     */
    0x77,0x70,  /* 011101110111 */
    0xDD,0xD0,  /* 110111011101 */
    0x77,0x70,  /* 011101110111 */
    0xDD,0xD0,  /* 110111011101 */
    0x77,0x70,  /* 011101110111 */
    0xDD,0xD0,  /* 110111011101 */
    0x77,0x70,  /* 011101110111 */
    0xDD,0xD0,  /* 110111011101 */
    0x77,0x70,  /* 011101110111 */
    0xDD,0xD0,  /* 110111011101 */
    0x77,0x70,  /* 011101110111 */
    0xDD,0xD0,  /* 110111011101 */
    0x77,0x70,  /* 011101110111 */
    0xDD,0xD0,  /* 110111011101 */
    0x77,0x70,  /* 011101110111 */
    0xDD,0xD0,  /* 110111011101 */
    0x77,0x70,  /* 011101110111 */
    0xDD,0xD0,  /* 110111011101 */
    0x77,0x70,  /* 011101110111 */
    0xDD,0xD0,  /* 110111011101 */
    0x77,0x70,  /* 011101110111 */
    0xDD,0xD0,  /* 110111011101 */
    0x77,0x70,  /* 011101110111 */
    0xDD,0xD0,  /* 110111011101 */

    /*
     * code=179, hex=0xB3, ascii="!3"
     */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */

    /*
     * code=180, hex=0xB4, ascii="!4"
     */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0xFE,0x00,  /* 111111100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */

    /*
     * code=181, hex=0xB5, ascii="!5"
     */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0xFE,0x00,  /* 111111100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0xFE,0x00,  /* 111111100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */

    /*
     * code=182, hex=0xB6, ascii="!6"
     */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0xF9,0x80,  /* 111110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */

    /*
     * code=183, hex=0xB7, ascii="!7"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0xFF,0x80,  /* 111111111000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */

    /*
     * code=184, hex=0xB8, ascii="!8"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0xFE,0x00,  /* 111111100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0xFE,0x00,  /* 111111100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */

    /*
     * code=185, hex=0xB9, ascii="!9"
     */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0xF9,0x80,  /* 111110011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0xF9,0x80,  /* 111110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */

    /*
     * code=186, hex=0xBA, ascii="!:"
     */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */

    /*
     * code=187, hex=0xBB, ascii="!;"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0xFF,0x80,  /* 111111111000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0xF9,0x80,  /* 111110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */

    /*
     * code=188, hex=0xBC, ascii="!<"
     */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0xF9,0x80,  /* 111110011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0xFF,0x80,  /* 111111111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=189, hex=0xBD, ascii="!="
     */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0xFF,0x80,  /* 111111111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=190, hex=0xBE, ascii="!>"
     */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0xFE,0x00,  /* 111111100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0xFE,0x00,  /* 111111100000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=191, hex=0xBF, ascii="!?"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0xFE,0x00,  /* 111111100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */

    /*
     * code=192, hex=0xC0, ascii="!@"
     */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x07,0xF0,  /* 000001111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=193, hex=0xC1, ascii="!A"
     */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0xFF,0xF0,  /* 111111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=194, hex=0xC2, ascii="!B"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0xFF,0xF0,  /* 111111111111 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */

    /*
     * code=195, hex=0xC3, ascii="!C"
     */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x07,0xF0,  /* 000001111111 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */

    /*
     * code=196, hex=0xC4, ascii="!D"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0xFF,0xF0,  /* 111111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=197, hex=0xC5, ascii="!E"
     */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0xFF,0xF0,  /* 111111111111 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */

    /*
     * code=198, hex=0xC6, ascii="!F"
     */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x07,0xF0,  /* 000001111111 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x07,0xF0,  /* 000001111111 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */

    /*
     * code=199, hex=0xC7, ascii="!G"
     */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0xF0,  /* 000110011111 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */

    /*
     * code=200, hex=0xC8, ascii="!H"
     */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0xF0,  /* 000110011111 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x1F,0xF0,  /* 000111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=201, hex=0xC9, ascii="!I"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x1F,0xF0,  /* 000111111111 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x19,0xF0,  /* 000110011111 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */

    /*
     * code=202, hex=0xCA, ascii="!J"
     */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0xF9,0xF0,  /* 111110011111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0xFF,0xF0,  /* 111111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=203, hex=0xCB, ascii="!K"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0xFF,0xF0,  /* 111111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0xF9,0xF0,  /* 111110011111 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */

    /*
     * code=204, hex=0xCC, ascii="!L"
     */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0xF0,  /* 000110011111 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x19,0xF0,  /* 000110011111 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */

    /*
     * code=205, hex=0xCD, ascii="!M"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0xFF,0xF0,  /* 111111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0xFF,0xF0,  /* 111111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=206, hex=0xCE, ascii="!N"
     */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0xF9,0xF0,  /* 111110011111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0xF9,0xF0,  /* 111110011111 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */

    /*
     * code=207, hex=0xCF, ascii="!O"
     */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0xFF,0xF0,  /* 111111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0xFF,0xF0,  /* 111111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=208, hex=0xD0, ascii="!P"
     */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0xFF,0xF0,  /* 111111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=209, hex=0xD1, ascii="!Q"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0xFF,0xF0,  /* 111111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0xFF,0xF0,  /* 111111111111 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */

    /*
     * code=210, hex=0xD2, ascii="!R"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0xFF,0xF0,  /* 111111111111 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */

    /*
     * code=211, hex=0xD3, ascii="!S"
     */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x1F,0xF0,  /* 000111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=212, hex=0xD4, ascii="!T"
     */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x07,0xF0,  /* 000001111111 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x07,0xF0,  /* 000001111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=213, hex=0xD5, ascii="!U"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0xF0,  /* 000001111111 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x07,0xF0,  /* 000001111111 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */

    /*
     * code=214, hex=0xD6, ascii="!V"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x1F,0xF0,  /* 000111111111 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */

    /*
     * code=215, hex=0xD7, ascii="!W"
     */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0xFF,0xF0,  /* 111111111111 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */
    0x19,0x80,  /* 000110011000 */

    /*
     * code=216, hex=0xD8, ascii="!X"
     */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0xFF,0xF0,  /* 111111111111 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0xFF,0xF0,  /* 111111111111 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */

    /*
     * code=217, hex=0xD9, ascii="!Y"
     */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0xFE,0x00,  /* 111111100000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=218, hex=0xDA, ascii="!Z"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0xF0,  /* 000001111111 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */

    /*
     * code=219, hex=0xDB, ascii="!["
     */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */

    /*
     * code=220, hex=0xDC, ascii="!\"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */

    /*
     * code=221, hex=0xDD, ascii="!]"
     */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */
    0xFC,0x00,  /* 111111000000 */

    /*
     * code=222, hex=0xDE, ascii="!^"
     */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */
    0x03,0xF0,  /* 000000111111 */

    /*
     * code=223, hex=0xDF, ascii="!_"
     */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0xFF,0xF0,  /* 111111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=224, hex=0xE0, ascii="!`"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0F,0x30,  /* 000011110011 */
    0x19,0xE0,  /* 000110011110 */
    0x30,0xC0,  /* 001100001100 */
    0x30,0xC0,  /* 001100001100 */
    0x30,0xC0,  /* 001100001100 */
    0x30,0xC0,  /* 001100001100 */
    0x30,0xC0,  /* 001100001100 */
    0x30,0xC0,  /* 001100001100 */
    0x30,0xC0,  /* 001100001100 */
    0x19,0xE0,  /* 000110011110 */
    0x0F,0x30,  /* 000011110011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=225, hex=0xE1, ascii="!a"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x60,  /* 001100000110 */
    0x30,0xC0,  /* 001100001100 */
    0x37,0xC0,  /* 001101111100 */
    0x30,0xC0,  /* 001100001100 */
    0x30,0x60,  /* 001100000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x60,  /* 001100000110 */
    0x30,0xE0,  /* 001100001110 */
    0x37,0x80,  /* 001101111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=226, hex=0xE2, ascii="!b"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=227, hex=0xE3, ascii="!c"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0E,0x30,  /* 000011100011 */
    0x3F,0xE0,  /* 001111111110 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=228, hex=0xE4, ascii="!d"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x30,  /* 000110000011 */
    0x18,0x00,  /* 000110000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x30,  /* 000110000011 */
    0x30,0x30,  /* 001100000011 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=229, hex=0xE5, ascii="!e"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0xF0,  /* 000001111111 */
    0x1D,0xC0,  /* 000111011100 */
    0x18,0xC0,  /* 000110001100 */
    0x30,0x60,  /* 001100000110 */
    0x30,0x60,  /* 001100000110 */
    0x30,0x60,  /* 001100000110 */
    0x30,0x60,  /* 001100000110 */
    0x30,0x60,  /* 001100000110 */
    0x18,0xC0,  /* 000110001100 */
    0x1D,0xC0,  /* 000111011100 */
    0x07,0x00,  /* 000001110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=230, hex=0xE6, ascii="!f"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x18,0x30,  /* 000110000011 */
    0x18,0x30,  /* 000110000011 */
    0x18,0x30,  /* 000110000011 */
    0x18,0x30,  /* 000110000011 */
    0x18,0x30,  /* 000110000011 */
    0x18,0x30,  /* 000110000011 */
    0x18,0x30,  /* 000110000011 */
    0x18,0x30,  /* 000110000011 */
    0x1C,0x60,  /* 000111000110 */
    0x1E,0xE0,  /* 000111101110 */
    0x1B,0x80,  /* 000110111000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x38,0x00,  /* 001110000000 */
    0x30,0x00,  /* 001100000000 */

    /*
     * code=231, hex=0xE7, ascii="!g"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x0C,0xC0,  /* 000011001100 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=232, hex=0xE8, ascii="!h"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x0F,0xC0,  /* 000011111100 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x0F,0xC0,  /* 000011111100 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=233, hex=0xE9, ascii="!i"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x3F,0xF0,  /* 001111111111 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x1C,0xE0,  /* 000111001110 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=234, hex=0xEA, ascii="!j"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x0C,0xC0,  /* 000011001100 */
    0x3C,0xF0,  /* 001111001111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=235, hex=0xEB, ascii="!k"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0xC0,  /* 000000111100 */
    0x06,0x70,  /* 000001100111 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x06,0x00,  /* 000001100000 */
    0x03,0x00,  /* 000000110000 */
    0x07,0x80,  /* 000001111000 */
    0x0C,0xC0,  /* 000011001100 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x18,0x60,  /* 000110000110 */
    0x0C,0xC0,  /* 000011001100 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=236, hex=0xEC, ascii="!l"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x1C,0xE0,  /* 000111001110 */
    0x37,0xB0,  /* 001101111011 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x37,0xB0,  /* 001101111011 */
    0x1C,0xE0,  /* 000111001110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=237, hex=0xED, ascii="!m"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x60,  /* 000000000110 */
    0x00,0xC0,  /* 000000001100 */
    0x0F,0xC0,  /* 000011111100 */
    0x19,0xE0,  /* 000110011110 */
    0x31,0xB0,  /* 001100011011 */
    0x33,0x30,  /* 001100110011 */
    0x33,0x30,  /* 001100110011 */
    0x36,0x30,  /* 001101100011 */
    0x1E,0x60,  /* 000111100110 */
    0x0F,0xC0,  /* 000011111100 */
    0x0C,0x00,  /* 000011000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x30,0x00,  /* 001100000000 */
    0x30,0x00,  /* 001100000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=238, hex=0xEE, ascii="!n"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x01,0xE0,  /* 000000011110 */
    0x07,0x00,  /* 000001110000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x1F,0xC0,  /* 000111111100 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x0C,0x00,  /* 000011000000 */
    0x0C,0x00,  /* 000011000000 */
    0x06,0x00,  /* 000001100000 */
    0x07,0x00,  /* 000001110000 */
    0x01,0xE0,  /* 000000011110 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=239, hex=0xEF, ascii="!o"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x1C,0xE0,  /* 000111001110 */
    0x18,0x60,  /* 000110000110 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x30,0x30,  /* 001100000011 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=240, hex=0xF0, ascii="!p"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=241, hex=0xF1, ascii="!q"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x3F,0xF0,  /* 001111111111 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=242, hex=0xF2, ascii="!r"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0C,0x00,  /* 000011000000 */
    0x06,0x00,  /* 000001100000 */
    0x03,0x00,  /* 000000110000 */
    0x01,0x80,  /* 000000011000 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0x60,  /* 000000000110 */
    0x00,0x30,  /* 000000000011 */
    0x00,0x60,  /* 000000000110 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=243, hex=0xF3, ascii="!s"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x18,0x00,  /* 000110000000 */
    0x30,0x00,  /* 001100000000 */
    0x18,0x00,  /* 000110000000 */
    0x0C,0x00,  /* 000011000000 */
    0x06,0x00,  /* 000001100000 */
    0x03,0x00,  /* 000000110000 */
    0x01,0x80,  /* 000000011000 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=244, hex=0xF4, ascii="!t"
     */
    0x03,0xC0,  /* 000000111100 */
    0x06,0x60,  /* 000001100110 */
    0x06,0x60,  /* 000001100110 */
    0x06,0x60,  /* 000001100110 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */

    /*
     * code=245, hex=0xF5, ascii="!u"
     */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x06,0x00,  /* 000001100000 */
    0x66,0x00,  /* 011001100000 */
    0x66,0x00,  /* 011001100000 */
    0x66,0x00,  /* 011001100000 */
    0x3C,0x00,  /* 001111000000 */

    /*
     * code=246, hex=0xF6, ascii="!v"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0xF0,  /* 001111111111 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=247, hex=0xF7, ascii="!w"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x1C,0x00,  /* 000111000000 */
    0x36,0x30,  /* 001101100011 */
    0x63,0x60,  /* 011000110110 */
    0x01,0xC0,  /* 000000011100 */
    0x1C,0x00,  /* 000111000000 */
    0x36,0x30,  /* 001101100011 */
    0x63,0x60,  /* 011000110110 */
    0x01,0xC0,  /* 000000011100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=248, hex=0xF8, ascii="!x"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0F,0x00,  /* 000011110000 */
    0x19,0x80,  /* 000110011000 */
    0x30,0xC0,  /* 001100001100 */
    0x30,0xC0,  /* 001100001100 */
    0x19,0x80,  /* 000110011000 */
    0x0F,0x00,  /* 000011110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=249, hex=0xF9, ascii="!y"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=250, hex=0xFA, ascii="!z"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x07,0x80,  /* 000001111000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=251, hex=0xFB, ascii="!{"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0xF0,  /* 000000001111 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x00,0xC0,  /* 000000001100 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x01,0x80,  /* 000000011000 */
    0x03,0x00,  /* 000000110000 */
    0x03,0x00,  /* 000000110000 */
    0xC3,0x00,  /* 110000110000 */
    0xC6,0x00,  /* 110001100000 */
    0x66,0x00,  /* 011001100000 */
    0x66,0x00,  /* 011001100000 */
    0x24,0x00,  /* 001001000000 */
    0x3C,0x00,  /* 001111000000 */
    0x3C,0x00,  /* 001111000000 */
    0x18,0x00,  /* 000110000000 */
    0x18,0x00,  /* 000110000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=252, hex=0xFC, ascii="!|"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x3F,0x00,  /* 001111110000 */
    0x39,0x80,  /* 001110011000 */
    0x31,0x80,  /* 001100011000 */
    0x31,0x80,  /* 001100011000 */
    0x31,0x80,  /* 001100011000 */
    0x31,0x80,  /* 001100011000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=253, hex=0xFD, ascii="!}"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x1E,0x00,  /* 000111100000 */
    0x33,0x00,  /* 001100110000 */
    0x06,0x00,  /* 000001100000 */
    0x0C,0x00,  /* 000011000000 */
    0x18,0x00,  /* 000110000000 */
    0x3F,0x00,  /* 001111110000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=254, hex=0xFE, ascii="!~"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x0F,0xC0,  /* 000011111100 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */

    /*
     * code=255, hex=0xFF, ascii="!^?"
     */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
    0x00,0x00,  /* 000000000000 */
};


uint8_t int10_font_14_alternate[20 * 15 + 1] = {
  0x1d,
  0x00, 0x00, 0x00, 0x00, 0x24, 0x66, 0xff,
  0x66, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x22,
  0x00, 0x63, 0x63, 0x63, 0x22, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x2b,
  0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0xff,
  0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x2d,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x4d,
  0x00, 0x00, 0xc3, 0xe7, 0xff, 0xdb, 0xc3,
  0xc3, 0xc3, 0xc3, 0xc3, 0x00, 0x00, 0x00,
  0x54,
  0x00, 0x00, 0xff, 0xdb, 0x99, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00,
  0x56,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3,
  0xc3, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00,
  0x57,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xdb,
  0xdb, 0xff, 0x66, 0x66, 0x00, 0x00, 0x00,
  0x58,
  0x00, 0x00, 0xc3, 0xc3, 0x66, 0x3c, 0x18,
  0x3c, 0x66, 0xc3, 0xc3, 0x00, 0x00, 0x00,
  0x59,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0x66, 0x3c,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00,
  0x5a,
  0x00, 0x00, 0xff, 0xc3, 0x86, 0x0c, 0x18,
  0x30, 0x61, 0xc3, 0xff, 0x00, 0x00, 0x00,
  0x6d,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xe6, 0xff,
  0xdb, 0xdb, 0xdb, 0xdb, 0x00, 0x00, 0x00,
  0x76,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3,
  0xc3, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00,
  0x77,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3,
  0xdb, 0xdb, 0xff, 0x66, 0x00, 0x00, 0x00,
  0x91,
  0x00, 0x00, 0x00, 0x00, 0x6e, 0x3b, 0x1b,
  0x7e, 0xd8, 0xdc, 0x77, 0x00, 0x00, 0x00,
  0x9b,
  0x00, 0x18, 0x18, 0x7e, 0xc3, 0xc0, 0xc0,
  0xc3, 0x7e, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x9d,
  0x00, 0x00, 0xc3, 0x66, 0x3c, 0x18, 0xff,
  0x18, 0xff, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x9e,
  0x00, 0xfc, 0x66, 0x66, 0x7c, 0x62, 0x66,
  0x6f, 0x66, 0x66, 0xf3, 0x00, 0x00, 0x00,
  0xf1,
  0x00, 0x00, 0x18, 0x18, 0x18, 0xff, 0x18,
  0x18, 0x18, 0x00, 0xff, 0x00, 0x00, 0x00,
  0xf6,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00
};

uint8_t int10_font_16_alternate[19 * 17 + 1] = {
  0x1d,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x66, 0xff,
  0x66, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x30,
  0x00, 0x00, 0x3c, 0x66, 0xc3, 0xc3, 0xdb, 0xdb,
  0xc3, 0xc3, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x4d,
  0x00, 0x00, 0xc3, 0xe7, 0xff, 0xff, 0xdb, 0xc3,
  0xc3, 0xc3, 0xc3, 0xc3, 0x00, 0x00, 0x00, 0x00,
  0x54,
  0x00, 0x00, 0xff, 0xdb, 0x99, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x56,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3,
  0xc3, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x57,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xdb,
  0xdb, 0xff, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x58,
  0x00, 0x00, 0xc3, 0xc3, 0x66, 0x3c, 0x18, 0x18,
  0x3c, 0x66, 0xc3, 0xc3, 0x00, 0x00, 0x00, 0x00,
  0x59,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0x66, 0x3c, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x5a,
  0x00, 0x00, 0xff, 0xc3, 0x86, 0x0c, 0x18, 0x30,
  0x60, 0xc1, 0xc3, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x6d,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xe6, 0xff, 0xdb,
  0xdb, 0xdb, 0xdb, 0xdb, 0x00, 0x00, 0x00, 0x00,
  0x76,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3, 0xc3,
  0xc3, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x77,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3, 0xc3,
  0xdb, 0xdb, 0xff, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x78,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0x66, 0x3c,
  0x18, 0x3c, 0x66, 0xc3, 0x00, 0x00, 0x00, 0x00,
  0x91,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x6e, 0x3b, 0x1b,
  0x7e, 0xd8, 0xdc, 0x77, 0x00, 0x00, 0x00, 0x00,
  0x9b,
  0x00, 0x18, 0x18, 0x7e, 0xc3, 0xc0, 0xc0, 0xc0,
  0xc3, 0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x9d,
  0x00, 0x00, 0xc3, 0x66, 0x3c, 0x18, 0xff, 0x18,
  0xff, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x9e,
  0x00, 0xfc, 0x66, 0x66, 0x7c, 0x62, 0x66, 0x6f,
  0x66, 0x66, 0x66, 0xf3, 0x00, 0x00, 0x00, 0x00,
  0xab,
  0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30,
  0x60, 0xce, 0x9b, 0x06, 0x0c, 0x1f, 0x00, 0x00,
  0xac,
  0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30,
  0x66, 0xce, 0x96, 0x3e, 0x06, 0x06, 0x00, 0x00,
  0x00
};
