//go:build go1.18
// +build go1.18

// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License. See License.txt in the project root for license information.
// Code generated by Microsoft (R) AutoRest Code Generator.
// Changes may cause incorrect behavior and will be lost if the code is regenerated.

package blob

import (
	"context"
	"encoding/base64"
	"net/http"
	"strconv"
	"strings"
	"time"

	"github.com/Azure/azure-sdk-for-go/sdk/azcore/policy"
	"github.com/Azure/azure-sdk-for-go/sdk/azcore/runtime"
)

type blobClient struct {
	endpoint string
	pl       runtime.Pipeline
}

// newBlobClient creates a new instance of blobClient with the specified values.
// endpoint - The URL of the service account, container, or blob that is the target of the desired operation.
// pl - the pipeline used for sending requests and handling responses.
func newBlobClient(endpoint string, pl runtime.Pipeline) *blobClient {
	client := &blobClient{
		endpoint: endpoint,
		pl:       pl,
	}
	return client
}

// AbortCopyFromURL - The Abort Copy From URL operation aborts a pending Copy From URL operation, and leaves a destination
// blob with zero length and full metadata.
// If the operation fails it returns an *azcore.ResponseError type.
// copyID - The copy identifier provided in the x-ms-copy-id header of the original Copy Blob operation.
// blobClientAbortCopyFromURLOptions - blobClientAbortCopyFromURLOptions contains the optional parameters for the blobClient.AbortCopyFromURL
// method.
// LeaseAccessConditions - LeaseAccessConditions contains a group of parameters for the containerClient.GetProperties method.
func (client *blobClient) AbortCopyFromURL(ctx context.Context, copyID string, blobClientAbortCopyFromURLOptions *blobClientAbortCopyFromURLOptions, leaseAccessConditions *LeaseAccessConditions) (blobClientAbortCopyFromURLResponse, error) {
	req, err := client.abortCopyFromURLCreateRequest(ctx, copyID, blobClientAbortCopyFromURLOptions, leaseAccessConditions)
	if err != nil {
		return blobClientAbortCopyFromURLResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientAbortCopyFromURLResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusNoContent) {
		return blobClientAbortCopyFromURLResponse{}, runtime.NewResponseError(resp)
	}
	return client.abortCopyFromURLHandleResponse(resp)
}

// abortCopyFromURLCreateRequest creates the AbortCopyFromURL request.
func (client *blobClient) abortCopyFromURLCreateRequest(ctx context.Context, copyID string, blobClientAbortCopyFromURLOptions *blobClientAbortCopyFromURLOptions, leaseAccessConditions *LeaseAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodPut, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("comp", "copy")
	reqQP.Set("copyid", copyID)
	if blobClientAbortCopyFromURLOptions != nil && blobClientAbortCopyFromURLOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientAbortCopyFromURLOptions.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	req.Raw().Header.Set("x-ms-copy-action", "abort")
	if leaseAccessConditions != nil && leaseAccessConditions.LeaseID != nil {
		req.Raw().Header.Set("x-ms-lease-id", *leaseAccessConditions.LeaseID)
	}
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientAbortCopyFromURLOptions != nil && blobClientAbortCopyFromURLOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientAbortCopyFromURLOptions.RequestID)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// abortCopyFromURLHandleResponse handles the AbortCopyFromURL response.
func (client *blobClient) abortCopyFromURLHandleResponse(resp *http.Response) (blobClientAbortCopyFromURLResponse, error) {
	result := blobClientAbortCopyFromURLResponse{RawResponse: resp}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientAbortCopyFromURLResponse{}, err
		}
		result.Date = &date
	}
	return result, nil
}

// AcquireLease - [Update] The Lease Blob operation establishes and manages a lock on a blob for write and delete operations
// If the operation fails it returns an *azcore.ResponseError type.
// blobClientAcquireLeaseOptions - blobClientAcquireLeaseOptions contains the optional parameters for the blobClient.AcquireLease
// method.
// ModifiedAccessConditions - ModifiedAccessConditions contains a group of parameters for the containerClient.Delete method.
func (client *blobClient) AcquireLease(ctx context.Context, blobClientAcquireLeaseOptions *blobClientAcquireLeaseOptions, modifiedAccessConditions *ModifiedAccessConditions) (blobClientAcquireLeaseResponse, error) {
	req, err := client.acquireLeaseCreateRequest(ctx, blobClientAcquireLeaseOptions, modifiedAccessConditions)
	if err != nil {
		return blobClientAcquireLeaseResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientAcquireLeaseResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusCreated) {
		return blobClientAcquireLeaseResponse{}, runtime.NewResponseError(resp)
	}
	return client.acquireLeaseHandleResponse(resp)
}

// acquireLeaseCreateRequest creates the AcquireLease request.
func (client *blobClient) acquireLeaseCreateRequest(ctx context.Context, blobClientAcquireLeaseOptions *blobClientAcquireLeaseOptions, modifiedAccessConditions *ModifiedAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodPut, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("comp", "lease")
	if blobClientAcquireLeaseOptions != nil && blobClientAcquireLeaseOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientAcquireLeaseOptions.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	req.Raw().Header.Set("x-ms-lease-action", "acquire")
	if blobClientAcquireLeaseOptions != nil && blobClientAcquireLeaseOptions.Duration != nil {
		req.Raw().Header.Set("x-ms-lease-duration", strconv.FormatInt(int64(*blobClientAcquireLeaseOptions.Duration), 10))
	}
	if blobClientAcquireLeaseOptions != nil && blobClientAcquireLeaseOptions.ProposedLeaseID != nil {
		req.Raw().Header.Set("x-ms-proposed-lease-id", *blobClientAcquireLeaseOptions.ProposedLeaseID)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfModifiedSince != nil {
		req.Raw().Header.Set("If-Modified-Since", modifiedAccessConditions.IfModifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfUnmodifiedSince != nil {
		req.Raw().Header.Set("If-Unmodified-Since", modifiedAccessConditions.IfUnmodifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfMatch != nil {
		req.Raw().Header.Set("If-Match", *modifiedAccessConditions.IfMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfNoneMatch != nil {
		req.Raw().Header.Set("If-None-Match", *modifiedAccessConditions.IfNoneMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfTags != nil {
		req.Raw().Header.Set("x-ms-if-tags", *modifiedAccessConditions.IfTags)
	}
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientAcquireLeaseOptions != nil && blobClientAcquireLeaseOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientAcquireLeaseOptions.RequestID)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// acquireLeaseHandleResponse handles the AcquireLease response.
func (client *blobClient) acquireLeaseHandleResponse(resp *http.Response) (blobClientAcquireLeaseResponse, error) {
	result := blobClientAcquireLeaseResponse{RawResponse: resp}
	if val := resp.Header.Get("ETag"); val != "" {
		result.ETag = &val
	}
	if val := resp.Header.Get("Last-Modified"); val != "" {
		lastModified, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientAcquireLeaseResponse{}, err
		}
		result.LastModified = &lastModified
	}
	if val := resp.Header.Get("x-ms-lease-id"); val != "" {
		result.LeaseID = &val
	}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientAcquireLeaseResponse{}, err
		}
		result.Date = &date
	}
	return result, nil
}

// BreakLease - [Update] The Lease Blob operation establishes and manages a lock on a blob for write and delete operations
// If the operation fails it returns an *azcore.ResponseError type.
// blobClientBreakLeaseOptions - blobClientBreakLeaseOptions contains the optional parameters for the blobClient.BreakLease
// method.
// ModifiedAccessConditions - ModifiedAccessConditions contains a group of parameters for the containerClient.Delete method.
func (client *blobClient) BreakLease(ctx context.Context, blobClientBreakLeaseOptions *blobClientBreakLeaseOptions, modifiedAccessConditions *ModifiedAccessConditions) (blobClientBreakLeaseResponse, error) {
	req, err := client.breakLeaseCreateRequest(ctx, blobClientBreakLeaseOptions, modifiedAccessConditions)
	if err != nil {
		return blobClientBreakLeaseResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientBreakLeaseResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusAccepted) {
		return blobClientBreakLeaseResponse{}, runtime.NewResponseError(resp)
	}
	return client.breakLeaseHandleResponse(resp)
}

// breakLeaseCreateRequest creates the BreakLease request.
func (client *blobClient) breakLeaseCreateRequest(ctx context.Context, blobClientBreakLeaseOptions *blobClientBreakLeaseOptions, modifiedAccessConditions *ModifiedAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodPut, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("comp", "lease")
	if blobClientBreakLeaseOptions != nil && blobClientBreakLeaseOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientBreakLeaseOptions.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	req.Raw().Header.Set("x-ms-lease-action", "break")
	if blobClientBreakLeaseOptions != nil && blobClientBreakLeaseOptions.BreakPeriod != nil {
		req.Raw().Header.Set("x-ms-lease-break-period", strconv.FormatInt(int64(*blobClientBreakLeaseOptions.BreakPeriod), 10))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfModifiedSince != nil {
		req.Raw().Header.Set("If-Modified-Since", modifiedAccessConditions.IfModifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfUnmodifiedSince != nil {
		req.Raw().Header.Set("If-Unmodified-Since", modifiedAccessConditions.IfUnmodifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfMatch != nil {
		req.Raw().Header.Set("If-Match", *modifiedAccessConditions.IfMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfNoneMatch != nil {
		req.Raw().Header.Set("If-None-Match", *modifiedAccessConditions.IfNoneMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfTags != nil {
		req.Raw().Header.Set("x-ms-if-tags", *modifiedAccessConditions.IfTags)
	}
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientBreakLeaseOptions != nil && blobClientBreakLeaseOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientBreakLeaseOptions.RequestID)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// breakLeaseHandleResponse handles the BreakLease response.
func (client *blobClient) breakLeaseHandleResponse(resp *http.Response) (blobClientBreakLeaseResponse, error) {
	result := blobClientBreakLeaseResponse{RawResponse: resp}
	if val := resp.Header.Get("ETag"); val != "" {
		result.ETag = &val
	}
	if val := resp.Header.Get("Last-Modified"); val != "" {
		lastModified, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientBreakLeaseResponse{}, err
		}
		result.LastModified = &lastModified
	}
	if val := resp.Header.Get("x-ms-lease-time"); val != "" {
		leaseTime32, err := strconv.ParseInt(val, 10, 32)
		leaseTime := int32(leaseTime32)
		if err != nil {
			return blobClientBreakLeaseResponse{}, err
		}
		result.LeaseTime = &leaseTime
	}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientBreakLeaseResponse{}, err
		}
		result.Date = &date
	}
	return result, nil
}

// ChangeLease - [Update] The Lease Blob operation establishes and manages a lock on a blob for write and delete operations
// If the operation fails it returns an *azcore.ResponseError type.
// leaseID - Specifies the current lease ID on the resource.
// proposedLeaseID - Proposed lease ID, in a GUID string format. The Blob service returns 400 (Invalid request) if the proposed
// lease ID is not in the correct format. See Guid Constructor (String) for a list of valid GUID
// string formats.
// blobClientChangeLeaseOptions - blobClientChangeLeaseOptions contains the optional parameters for the blobClient.ChangeLease
// method.
// ModifiedAccessConditions - ModifiedAccessConditions contains a group of parameters for the containerClient.Delete method.
func (client *blobClient) ChangeLease(ctx context.Context, leaseID string, proposedLeaseID string, blobClientChangeLeaseOptions *blobClientChangeLeaseOptions, modifiedAccessConditions *ModifiedAccessConditions) (blobClientChangeLeaseResponse, error) {
	req, err := client.changeLeaseCreateRequest(ctx, leaseID, proposedLeaseID, blobClientChangeLeaseOptions, modifiedAccessConditions)
	if err != nil {
		return blobClientChangeLeaseResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientChangeLeaseResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusOK) {
		return blobClientChangeLeaseResponse{}, runtime.NewResponseError(resp)
	}
	return client.changeLeaseHandleResponse(resp)
}

// changeLeaseCreateRequest creates the ChangeLease request.
func (client *blobClient) changeLeaseCreateRequest(ctx context.Context, leaseID string, proposedLeaseID string, blobClientChangeLeaseOptions *blobClientChangeLeaseOptions, modifiedAccessConditions *ModifiedAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodPut, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("comp", "lease")
	if blobClientChangeLeaseOptions != nil && blobClientChangeLeaseOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientChangeLeaseOptions.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	req.Raw().Header.Set("x-ms-lease-action", "change")
	req.Raw().Header.Set("x-ms-lease-id", leaseID)
	req.Raw().Header.Set("x-ms-proposed-lease-id", proposedLeaseID)
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfModifiedSince != nil {
		req.Raw().Header.Set("If-Modified-Since", modifiedAccessConditions.IfModifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfUnmodifiedSince != nil {
		req.Raw().Header.Set("If-Unmodified-Since", modifiedAccessConditions.IfUnmodifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfMatch != nil {
		req.Raw().Header.Set("If-Match", *modifiedAccessConditions.IfMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfNoneMatch != nil {
		req.Raw().Header.Set("If-None-Match", *modifiedAccessConditions.IfNoneMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfTags != nil {
		req.Raw().Header.Set("x-ms-if-tags", *modifiedAccessConditions.IfTags)
	}
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientChangeLeaseOptions != nil && blobClientChangeLeaseOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientChangeLeaseOptions.RequestID)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// changeLeaseHandleResponse handles the ChangeLease response.
func (client *blobClient) changeLeaseHandleResponse(resp *http.Response) (blobClientChangeLeaseResponse, error) {
	result := blobClientChangeLeaseResponse{RawResponse: resp}
	if val := resp.Header.Get("ETag"); val != "" {
		result.ETag = &val
	}
	if val := resp.Header.Get("Last-Modified"); val != "" {
		lastModified, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientChangeLeaseResponse{}, err
		}
		result.LastModified = &lastModified
	}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-lease-id"); val != "" {
		result.LeaseID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientChangeLeaseResponse{}, err
		}
		result.Date = &date
	}
	return result, nil
}

// CopyFromURL - The Copy From URL operation copies a blob or an internet resource to a new blob. It will not return a response
// until the copy is complete.
// If the operation fails it returns an *azcore.ResponseError type.
// copySource - Specifies the name of the source page blob snapshot. This value is a URL of up to 2 KB in length that specifies
// a page blob snapshot. The value should be URL-encoded as it would appear in a request
// URI. The source blob must either be public or must be authenticated via a shared access signature.
// blobClientCopyFromURLOptions - blobClientCopyFromURLOptions contains the optional parameters for the blobClient.CopyFromURL
// method.
// SourceModifiedAccessConditions - SourceModifiedAccessConditions contains a group of parameters for the blobClient.StartCopyFromURL
// method.
// ModifiedAccessConditions - ModifiedAccessConditions contains a group of parameters for the containerClient.Delete method.
// LeaseAccessConditions - LeaseAccessConditions contains a group of parameters for the containerClient.GetProperties method.
func (client *blobClient) CopyFromURL(ctx context.Context, copySource string, blobClientCopyFromURLOptions *blobClientCopyFromURLOptions, sourceModifiedAccessConditions *SourceModifiedAccessConditions, modifiedAccessConditions *ModifiedAccessConditions, leaseAccessConditions *LeaseAccessConditions) (blobClientCopyFromURLResponse, error) {
	req, err := client.copyFromURLCreateRequest(ctx, copySource, blobClientCopyFromURLOptions, sourceModifiedAccessConditions, modifiedAccessConditions, leaseAccessConditions)
	if err != nil {
		return blobClientCopyFromURLResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientCopyFromURLResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusAccepted) {
		return blobClientCopyFromURLResponse{}, runtime.NewResponseError(resp)
	}
	return client.copyFromURLHandleResponse(resp)
}

// copyFromURLCreateRequest creates the CopyFromURL request.
func (client *blobClient) copyFromURLCreateRequest(ctx context.Context, copySource string, blobClientCopyFromURLOptions *blobClientCopyFromURLOptions, sourceModifiedAccessConditions *SourceModifiedAccessConditions, modifiedAccessConditions *ModifiedAccessConditions, leaseAccessConditions *LeaseAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodPut, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	if blobClientCopyFromURLOptions != nil && blobClientCopyFromURLOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientCopyFromURLOptions.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	req.Raw().Header.Set("x-ms-requires-sync", "true")
	if blobClientCopyFromURLOptions != nil && blobClientCopyFromURLOptions.Metadata != nil {
		for k, v := range blobClientCopyFromURLOptions.Metadata {
			req.Raw().Header.Set("x-ms-meta-"+k, v)
		}
	}
	if blobClientCopyFromURLOptions != nil && blobClientCopyFromURLOptions.Tier != nil {
		req.Raw().Header.Set("x-ms-access-tier", string(*blobClientCopyFromURLOptions.Tier))
	}
	if sourceModifiedAccessConditions != nil && sourceModifiedAccessConditions.SourceIfModifiedSince != nil {
		req.Raw().Header.Set("x-ms-source-if-modified-since", sourceModifiedAccessConditions.SourceIfModifiedSince.Format(time.RFC1123))
	}
	if sourceModifiedAccessConditions != nil && sourceModifiedAccessConditions.SourceIfUnmodifiedSince != nil {
		req.Raw().Header.Set("x-ms-source-if-unmodified-since", sourceModifiedAccessConditions.SourceIfUnmodifiedSince.Format(time.RFC1123))
	}
	if sourceModifiedAccessConditions != nil && sourceModifiedAccessConditions.SourceIfMatch != nil {
		req.Raw().Header.Set("x-ms-source-if-match", *sourceModifiedAccessConditions.SourceIfMatch)
	}
	if sourceModifiedAccessConditions != nil && sourceModifiedAccessConditions.SourceIfNoneMatch != nil {
		req.Raw().Header.Set("x-ms-source-if-none-match", *sourceModifiedAccessConditions.SourceIfNoneMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfModifiedSince != nil {
		req.Raw().Header.Set("If-Modified-Since", modifiedAccessConditions.IfModifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfUnmodifiedSince != nil {
		req.Raw().Header.Set("If-Unmodified-Since", modifiedAccessConditions.IfUnmodifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfMatch != nil {
		req.Raw().Header.Set("If-Match", *modifiedAccessConditions.IfMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfNoneMatch != nil {
		req.Raw().Header.Set("If-None-Match", *modifiedAccessConditions.IfNoneMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfTags != nil {
		req.Raw().Header.Set("x-ms-if-tags", *modifiedAccessConditions.IfTags)
	}
	req.Raw().Header.Set("x-ms-copy-source", copySource)
	if leaseAccessConditions != nil && leaseAccessConditions.LeaseID != nil {
		req.Raw().Header.Set("x-ms-lease-id", *leaseAccessConditions.LeaseID)
	}
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientCopyFromURLOptions != nil && blobClientCopyFromURLOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientCopyFromURLOptions.RequestID)
	}
	if blobClientCopyFromURLOptions != nil && blobClientCopyFromURLOptions.SourceContentMD5 != nil {
		req.Raw().Header.Set("x-ms-source-content-md5", base64.StdEncoding.EncodeToString(blobClientCopyFromURLOptions.SourceContentMD5))
	}
	if blobClientCopyFromURLOptions != nil && blobClientCopyFromURLOptions.BlobTagsString != nil {
		req.Raw().Header.Set("x-ms-tags", *blobClientCopyFromURLOptions.BlobTagsString)
	}
	if blobClientCopyFromURLOptions != nil && blobClientCopyFromURLOptions.ImmutabilityPolicyExpiry != nil {
		req.Raw().Header.Set("x-ms-immutability-policy-until-date", blobClientCopyFromURLOptions.ImmutabilityPolicyExpiry.Format(time.RFC1123))
	}
	if blobClientCopyFromURLOptions != nil && blobClientCopyFromURLOptions.ImmutabilityPolicyMode != nil {
		req.Raw().Header.Set("x-ms-immutability-policy-mode", string(*blobClientCopyFromURLOptions.ImmutabilityPolicyMode))
	}
	if blobClientCopyFromURLOptions != nil && blobClientCopyFromURLOptions.LegalHold != nil {
		req.Raw().Header.Set("x-ms-legal-hold", strconv.FormatBool(*blobClientCopyFromURLOptions.LegalHold))
	}
	if blobClientCopyFromURLOptions != nil && blobClientCopyFromURLOptions.CopySourceAuthorization != nil {
		req.Raw().Header.Set("x-ms-copy-source-authorization", *blobClientCopyFromURLOptions.CopySourceAuthorization)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// copyFromURLHandleResponse handles the CopyFromURL response.
func (client *blobClient) copyFromURLHandleResponse(resp *http.Response) (blobClientCopyFromURLResponse, error) {
	result := blobClientCopyFromURLResponse{RawResponse: resp}
	if val := resp.Header.Get("ETag"); val != "" {
		result.ETag = &val
	}
	if val := resp.Header.Get("Last-Modified"); val != "" {
		lastModified, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientCopyFromURLResponse{}, err
		}
		result.LastModified = &lastModified
	}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("x-ms-version-id"); val != "" {
		result.VersionID = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientCopyFromURLResponse{}, err
		}
		result.Date = &date
	}
	if val := resp.Header.Get("x-ms-copy-id"); val != "" {
		result.CopyID = &val
	}
	if val := resp.Header.Get("x-ms-copy-status"); val != "" {
		result.CopyStatus = &val
	}
	if val := resp.Header.Get("Content-MD5"); val != "" {
		contentMD5, err := base64.StdEncoding.DecodeString(val)
		if err != nil {
			return blobClientCopyFromURLResponse{}, err
		}
		result.ContentMD5 = contentMD5
	}
	if val := resp.Header.Get("x-ms-content-crc64"); val != "" {
		xMSContentCRC64, err := base64.StdEncoding.DecodeString(val)
		if err != nil {
			return blobClientCopyFromURLResponse{}, err
		}
		result.XMSContentCRC64 = xMSContentCRC64
	}
	return result, nil
}

// CreateSnapshot - The Create Snapshot operation creates a read-only snapshot of a blob
// If the operation fails it returns an *azcore.ResponseError type.
// blobClientCreateSnapshotOptions - blobClientCreateSnapshotOptions contains the optional parameters for the blobClient.CreateSnapshot
// method.
// CpkInfo - CpkInfo contains a group of parameters for the blobClient.Download method.
// CpkScopeInfo - CpkScopeInfo contains a group of parameters for the blobClient.SetMetadata method.
// ModifiedAccessConditions - ModifiedAccessConditions contains a group of parameters for the containerClient.Delete method.
// LeaseAccessConditions - LeaseAccessConditions contains a group of parameters for the containerClient.GetProperties method.
func (client *blobClient) CreateSnapshot(ctx context.Context, blobClientCreateSnapshotOptions *blobClientCreateSnapshotOptions, cpkInfo *CpkInfo, cpkScopeInfo *CpkScopeInfo, modifiedAccessConditions *ModifiedAccessConditions, leaseAccessConditions *LeaseAccessConditions) (blobClientCreateSnapshotResponse, error) {
	req, err := client.createSnapshotCreateRequest(ctx, blobClientCreateSnapshotOptions, cpkInfo, cpkScopeInfo, modifiedAccessConditions, leaseAccessConditions)
	if err != nil {
		return blobClientCreateSnapshotResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientCreateSnapshotResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusCreated) {
		return blobClientCreateSnapshotResponse{}, runtime.NewResponseError(resp)
	}
	return client.createSnapshotHandleResponse(resp)
}

// createSnapshotCreateRequest creates the CreateSnapshot request.
func (client *blobClient) createSnapshotCreateRequest(ctx context.Context, blobClientCreateSnapshotOptions *blobClientCreateSnapshotOptions, cpkInfo *CpkInfo, cpkScopeInfo *CpkScopeInfo, modifiedAccessConditions *ModifiedAccessConditions, leaseAccessConditions *LeaseAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodPut, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("comp", "snapshot")
	if blobClientCreateSnapshotOptions != nil && blobClientCreateSnapshotOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientCreateSnapshotOptions.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	if blobClientCreateSnapshotOptions != nil && blobClientCreateSnapshotOptions.Metadata != nil {
		for k, v := range blobClientCreateSnapshotOptions.Metadata {
			req.Raw().Header.Set("x-ms-meta-"+k, v)
		}
	}
	if cpkInfo != nil && cpkInfo.EncryptionKey != nil {
		req.Raw().Header.Set("x-ms-encryption-key", *cpkInfo.EncryptionKey)
	}
	if cpkInfo != nil && cpkInfo.EncryptionKeySHA256 != nil {
		req.Raw().Header.Set("x-ms-encryption-key-sha256", *cpkInfo.EncryptionKeySHA256)
	}
	if cpkInfo != nil && cpkInfo.EncryptionAlgorithm != nil {
		req.Raw().Header.Set("x-ms-encryption-algorithm", string(*cpkInfo.EncryptionAlgorithm))
	}
	if cpkScopeInfo != nil && cpkScopeInfo.EncryptionScope != nil {
		req.Raw().Header.Set("x-ms-encryption-scope", *cpkScopeInfo.EncryptionScope)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfModifiedSince != nil {
		req.Raw().Header.Set("If-Modified-Since", modifiedAccessConditions.IfModifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfUnmodifiedSince != nil {
		req.Raw().Header.Set("If-Unmodified-Since", modifiedAccessConditions.IfUnmodifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfMatch != nil {
		req.Raw().Header.Set("If-Match", *modifiedAccessConditions.IfMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfNoneMatch != nil {
		req.Raw().Header.Set("If-None-Match", *modifiedAccessConditions.IfNoneMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfTags != nil {
		req.Raw().Header.Set("x-ms-if-tags", *modifiedAccessConditions.IfTags)
	}
	if leaseAccessConditions != nil && leaseAccessConditions.LeaseID != nil {
		req.Raw().Header.Set("x-ms-lease-id", *leaseAccessConditions.LeaseID)
	}
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientCreateSnapshotOptions != nil && blobClientCreateSnapshotOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientCreateSnapshotOptions.RequestID)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// createSnapshotHandleResponse handles the CreateSnapshot response.
func (client *blobClient) createSnapshotHandleResponse(resp *http.Response) (blobClientCreateSnapshotResponse, error) {
	result := blobClientCreateSnapshotResponse{RawResponse: resp}
	if val := resp.Header.Get("x-ms-snapshot"); val != "" {
		result.Snapshot = &val
	}
	if val := resp.Header.Get("ETag"); val != "" {
		result.ETag = &val
	}
	if val := resp.Header.Get("Last-Modified"); val != "" {
		lastModified, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientCreateSnapshotResponse{}, err
		}
		result.LastModified = &lastModified
	}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("x-ms-version-id"); val != "" {
		result.VersionID = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientCreateSnapshotResponse{}, err
		}
		result.Date = &date
	}
	if val := resp.Header.Get("x-ms-request-server-encrypted"); val != "" {
		isServerEncrypted, err := strconv.ParseBool(val)
		if err != nil {
			return blobClientCreateSnapshotResponse{}, err
		}
		result.IsServerEncrypted = &isServerEncrypted
	}
	return result, nil
}

// Delete - If the storage account's soft delete feature is disabled then, when a blob is deleted, it is permanently removed
// from the storage account. If the storage account's soft delete feature is enabled,
// then, when a blob is deleted, it is marked for deletion and becomes inaccessible immediately. However, the blob service
// retains the blob or snapshot for the number of days specified by the
// DeleteRetentionPolicy section of Storage service properties [Set-Blob-Service-Properties.md]. After the specified number
// of days has passed, the blob's data is permanently removed from the storage
// account. Note that you continue to be charged for the soft-deleted blob's storage until it is permanently removed. Use
// the List Blobs API and specify the "include=deleted" query parameter to discover
// which blobs and snapshots have been soft deleted. You can then use the Undelete Blob API to restore a soft-deleted blob.
// All other operations on a soft-deleted blob or snapshot causes the service to
// return an HTTP status code of 404 (ResourceNotFound).
// If the operation fails it returns an *azcore.ResponseError type.
// blobClientDeleteOptions - blobClientDeleteOptions contains the optional parameters for the blobClient.Delete method.
// LeaseAccessConditions - LeaseAccessConditions contains a group of parameters for the containerClient.GetProperties method.
// ModifiedAccessConditions - ModifiedAccessConditions contains a group of parameters for the containerClient.Delete method.
func (client *blobClient) Delete(ctx context.Context, blobClientDeleteOptions *blobClientDeleteOptions, leaseAccessConditions *LeaseAccessConditions, modifiedAccessConditions *ModifiedAccessConditions) (blobClientDeleteResponse, error) {
	req, err := client.deleteCreateRequest(ctx, blobClientDeleteOptions, leaseAccessConditions, modifiedAccessConditions)
	if err != nil {
		return blobClientDeleteResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientDeleteResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusAccepted) {
		return blobClientDeleteResponse{}, runtime.NewResponseError(resp)
	}
	return client.deleteHandleResponse(resp)
}

// deleteCreateRequest creates the Delete request.
func (client *blobClient) deleteCreateRequest(ctx context.Context, blobClientDeleteOptions *blobClientDeleteOptions, leaseAccessConditions *LeaseAccessConditions, modifiedAccessConditions *ModifiedAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodDelete, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	if blobClientDeleteOptions != nil && blobClientDeleteOptions.Snapshot != nil {
		reqQP.Set("snapshot", *blobClientDeleteOptions.Snapshot)
	}
	if blobClientDeleteOptions != nil && blobClientDeleteOptions.VersionID != nil {
		reqQP.Set("versionid", *blobClientDeleteOptions.VersionID)
	}
	if blobClientDeleteOptions != nil && blobClientDeleteOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientDeleteOptions.Timeout), 10))
	}
	if blobClientDeleteOptions != nil && blobClientDeleteOptions.BlobDeleteType != nil {
		reqQP.Set("deletetype", string(*blobClientDeleteOptions.BlobDeleteType))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	if leaseAccessConditions != nil && leaseAccessConditions.LeaseID != nil {
		req.Raw().Header.Set("x-ms-lease-id", *leaseAccessConditions.LeaseID)
	}
	if blobClientDeleteOptions != nil && blobClientDeleteOptions.DeleteSnapshots != nil {
		req.Raw().Header.Set("x-ms-delete-snapshots", string(*blobClientDeleteOptions.DeleteSnapshots))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfModifiedSince != nil {
		req.Raw().Header.Set("If-Modified-Since", modifiedAccessConditions.IfModifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfUnmodifiedSince != nil {
		req.Raw().Header.Set("If-Unmodified-Since", modifiedAccessConditions.IfUnmodifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfMatch != nil {
		req.Raw().Header.Set("If-Match", *modifiedAccessConditions.IfMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfNoneMatch != nil {
		req.Raw().Header.Set("If-None-Match", *modifiedAccessConditions.IfNoneMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfTags != nil {
		req.Raw().Header.Set("x-ms-if-tags", *modifiedAccessConditions.IfTags)
	}
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientDeleteOptions != nil && blobClientDeleteOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientDeleteOptions.RequestID)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// deleteHandleResponse handles the Delete response.
func (client *blobClient) deleteHandleResponse(resp *http.Response) (blobClientDeleteResponse, error) {
	result := blobClientDeleteResponse{RawResponse: resp}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientDeleteResponse{}, err
		}
		result.Date = &date
	}
	return result, nil
}

// DeleteImmutabilityPolicy - The Delete Immutability Policy operation deletes the immutability policy on the blob
// If the operation fails it returns an *azcore.ResponseError type.
// options - blobClientDeleteImmutabilityPolicyOptions contains the optional parameters for the blobClient.DeleteImmutabilityPolicy
// method.
func (client *blobClient) DeleteImmutabilityPolicy(ctx context.Context, options *blobClientDeleteImmutabilityPolicyOptions) (blobClientDeleteImmutabilityPolicyResponse, error) {
	req, err := client.deleteImmutabilityPolicyCreateRequest(ctx, options)
	if err != nil {
		return blobClientDeleteImmutabilityPolicyResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientDeleteImmutabilityPolicyResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusOK) {
		return blobClientDeleteImmutabilityPolicyResponse{}, runtime.NewResponseError(resp)
	}
	return client.deleteImmutabilityPolicyHandleResponse(resp)
}

// deleteImmutabilityPolicyCreateRequest creates the DeleteImmutabilityPolicy request.
func (client *blobClient) deleteImmutabilityPolicyCreateRequest(ctx context.Context, options *blobClientDeleteImmutabilityPolicyOptions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodDelete, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("comp", "immutabilityPolicies")
	if options != nil && options.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*options.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if options != nil && options.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *options.RequestID)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// deleteImmutabilityPolicyHandleResponse handles the DeleteImmutabilityPolicy response.
func (client *blobClient) deleteImmutabilityPolicyHandleResponse(resp *http.Response) (blobClientDeleteImmutabilityPolicyResponse, error) {
	result := blobClientDeleteImmutabilityPolicyResponse{RawResponse: resp}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientDeleteImmutabilityPolicyResponse{}, err
		}
		result.Date = &date
	}
	return result, nil
}

// Download - The Download operation reads or downloads a blob from the system, including its metadata and properties. You
// can also call Download to read a snapshot.
// If the operation fails it returns an *azcore.ResponseError type.
// blobClientDownloadOptions - blobClientDownloadOptions contains the optional parameters for the blobClient.Download method.
// LeaseAccessConditions - LeaseAccessConditions contains a group of parameters for the containerClient.GetProperties method.
// CpkInfo - CpkInfo contains a group of parameters for the blobClient.Download method.
// ModifiedAccessConditions - ModifiedAccessConditions contains a group of parameters for the containerClient.Delete method.
func (client *blobClient) Download(ctx context.Context, blobClientDownloadOptions *blobClientDownloadOptions, leaseAccessConditions *LeaseAccessConditions, cpkInfo *CpkInfo, modifiedAccessConditions *ModifiedAccessConditions) (blobClientDownloadResponse, error) {
	req, err := client.downloadCreateRequest(ctx, blobClientDownloadOptions, leaseAccessConditions, cpkInfo, modifiedAccessConditions)
	if err != nil {
		return blobClientDownloadResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientDownloadResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusOK, http.StatusPartialContent, http.StatusNotModified) {
		return blobClientDownloadResponse{}, runtime.NewResponseError(resp)
	}
	return client.downloadHandleResponse(resp)
}

// downloadCreateRequest creates the Download request.
func (client *blobClient) downloadCreateRequest(ctx context.Context, blobClientDownloadOptions *blobClientDownloadOptions, leaseAccessConditions *LeaseAccessConditions, cpkInfo *CpkInfo, modifiedAccessConditions *ModifiedAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodGet, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	if blobClientDownloadOptions != nil && blobClientDownloadOptions.Snapshot != nil {
		reqQP.Set("snapshot", *blobClientDownloadOptions.Snapshot)
	}
	if blobClientDownloadOptions != nil && blobClientDownloadOptions.VersionID != nil {
		reqQP.Set("versionid", *blobClientDownloadOptions.VersionID)
	}
	if blobClientDownloadOptions != nil && blobClientDownloadOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientDownloadOptions.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	runtime.SkipBodyDownload(req)
	if blobClientDownloadOptions != nil && blobClientDownloadOptions.Range != nil {
		req.Raw().Header.Set("x-ms-range", *blobClientDownloadOptions.Range)
	}
	if leaseAccessConditions != nil && leaseAccessConditions.LeaseID != nil {
		req.Raw().Header.Set("x-ms-lease-id", *leaseAccessConditions.LeaseID)
	}
	if blobClientDownloadOptions != nil && blobClientDownloadOptions.RangeGetContentMD5 != nil {
		req.Raw().Header.Set("x-ms-range-get-content-md5", strconv.FormatBool(*blobClientDownloadOptions.RangeGetContentMD5))
	}
	if blobClientDownloadOptions != nil && blobClientDownloadOptions.RangeGetContentCRC64 != nil {
		req.Raw().Header.Set("x-ms-range-get-content-crc64", strconv.FormatBool(*blobClientDownloadOptions.RangeGetContentCRC64))
	}
	if cpkInfo != nil && cpkInfo.EncryptionKey != nil {
		req.Raw().Header.Set("x-ms-encryption-key", *cpkInfo.EncryptionKey)
	}
	if cpkInfo != nil && cpkInfo.EncryptionKeySHA256 != nil {
		req.Raw().Header.Set("x-ms-encryption-key-sha256", *cpkInfo.EncryptionKeySHA256)
	}
	if cpkInfo != nil && cpkInfo.EncryptionAlgorithm != nil {
		req.Raw().Header.Set("x-ms-encryption-algorithm", string(*cpkInfo.EncryptionAlgorithm))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfModifiedSince != nil {
		req.Raw().Header.Set("If-Modified-Since", modifiedAccessConditions.IfModifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfUnmodifiedSince != nil {
		req.Raw().Header.Set("If-Unmodified-Since", modifiedAccessConditions.IfUnmodifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfMatch != nil {
		req.Raw().Header.Set("If-Match", *modifiedAccessConditions.IfMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfNoneMatch != nil {
		req.Raw().Header.Set("If-None-Match", *modifiedAccessConditions.IfNoneMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfTags != nil {
		req.Raw().Header.Set("x-ms-if-tags", *modifiedAccessConditions.IfTags)
	}
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientDownloadOptions != nil && blobClientDownloadOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientDownloadOptions.RequestID)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// downloadHandleResponse handles the Download response.
func (client *blobClient) downloadHandleResponse(resp *http.Response) (blobClientDownloadResponse, error) {
	result := blobClientDownloadResponse{RawResponse: resp}
	if val := resp.Header.Get("Last-Modified"); val != "" {
		lastModified, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientDownloadResponse{}, err
		}
		result.LastModified = &lastModified
	}
	for hh := range resp.Header {
		if len(hh) > len("x-ms-meta-") && strings.EqualFold(hh[:len("x-ms-meta-")], "x-ms-meta-") {
			if result.Metadata == nil {
				result.Metadata = map[string]string{}
			}
			result.Metadata[hh[len("x-ms-meta-"):]] = resp.Header.Get(hh)
		}
	}
	if val := resp.Header.Get("x-ms-or-policy-id"); val != "" {
		result.ObjectReplicationPolicyID = &val
	}
	for hh := range resp.Header {
		if len(hh) > len("x-ms-or-") && strings.EqualFold(hh[:len("x-ms-or-")], "x-ms-or-") {
			if result.Metadata == nil {
				result.Metadata = map[string]string{}
			}
			result.Metadata[hh[len("x-ms-or-"):]] = resp.Header.Get(hh)
		}
	}
	if val := resp.Header.Get("Content-Length"); val != "" {
		contentLength, err := strconv.ParseInt(val, 10, 64)
		if err != nil {
			return blobClientDownloadResponse{}, err
		}
		result.ContentLength = &contentLength
	}
	if val := resp.Header.Get("Content-Type"); val != "" {
		result.ContentType = &val
	}
	if val := resp.Header.Get("Content-Range"); val != "" {
		result.ContentRange = &val
	}
	if val := resp.Header.Get("ETag"); val != "" {
		result.ETag = &val
	}
	if val := resp.Header.Get("Content-MD5"); val != "" {
		contentMD5, err := base64.StdEncoding.DecodeString(val)
		if err != nil {
			return blobClientDownloadResponse{}, err
		}
		result.ContentMD5 = contentMD5
	}
	if val := resp.Header.Get("Content-Encoding"); val != "" {
		result.ContentEncoding = &val
	}
	if val := resp.Header.Get("Cache-Control"); val != "" {
		result.CacheControl = &val
	}
	if val := resp.Header.Get("Content-Disposition"); val != "" {
		result.ContentDisposition = &val
	}
	if val := resp.Header.Get("Content-Language"); val != "" {
		result.ContentLanguage = &val
	}
	if val := resp.Header.Get("x-ms-blob-sequence-number"); val != "" {
		blobSequenceNumber, err := strconv.ParseInt(val, 10, 64)
		if err != nil {
			return blobClientDownloadResponse{}, err
		}
		result.BlobSequenceNumber = &blobSequenceNumber
	}
	if val := resp.Header.Get("x-ms-blob-type"); val != "" {
		result.BlobType = (*BlobType)(&val)
	}
	if val := resp.Header.Get("x-ms-copy-completion-time"); val != "" {
		copyCompletionTime, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientDownloadResponse{}, err
		}
		result.CopyCompletionTime = &copyCompletionTime
	}
	if val := resp.Header.Get("x-ms-copy-status-description"); val != "" {
		result.CopyStatusDescription = &val
	}
	if val := resp.Header.Get("x-ms-copy-id"); val != "" {
		result.CopyID = &val
	}
	if val := resp.Header.Get("x-ms-copy-progress"); val != "" {
		result.CopyProgress = &val
	}
	if val := resp.Header.Get("x-ms-copy-source"); val != "" {
		result.CopySource = &val
	}
	if val := resp.Header.Get("x-ms-copy-status"); val != "" {
		result.CopyStatus = (*CopyStatusType)(&val)
	}
	if val := resp.Header.Get("x-ms-lease-duration"); val != "" {
		result.LeaseDuration = (*LeaseDurationType)(&val)
	}
	if val := resp.Header.Get("x-ms-lease-state"); val != "" {
		result.LeaseState = (*LeaseStateType)(&val)
	}
	if val := resp.Header.Get("x-ms-lease-status"); val != "" {
		result.LeaseStatus = (*LeaseStatusType)(&val)
	}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("x-ms-version-id"); val != "" {
		result.VersionID = &val
	}
	if val := resp.Header.Get("x-ms-is-current-version"); val != "" {
		isCurrentVersion, err := strconv.ParseBool(val)
		if err != nil {
			return blobClientDownloadResponse{}, err
		}
		result.IsCurrentVersion = &isCurrentVersion
	}
	if val := resp.Header.Get("Accept-Ranges"); val != "" {
		result.AcceptRanges = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientDownloadResponse{}, err
		}
		result.Date = &date
	}
	if val := resp.Header.Get("x-ms-blob-committed-block-count"); val != "" {
		blobCommittedBlockCount32, err := strconv.ParseInt(val, 10, 32)
		blobCommittedBlockCount := int32(blobCommittedBlockCount32)
		if err != nil {
			return blobClientDownloadResponse{}, err
		}
		result.BlobCommittedBlockCount = &blobCommittedBlockCount
	}
	if val := resp.Header.Get("x-ms-server-encrypted"); val != "" {
		isServerEncrypted, err := strconv.ParseBool(val)
		if err != nil {
			return blobClientDownloadResponse{}, err
		}
		result.IsServerEncrypted = &isServerEncrypted
	}
	if val := resp.Header.Get("x-ms-encryption-key-sha256"); val != "" {
		result.EncryptionKeySHA256 = &val
	}
	if val := resp.Header.Get("x-ms-encryption-scope"); val != "" {
		result.EncryptionScope = &val
	}
	if val := resp.Header.Get("x-ms-blob-content-md5"); val != "" {
		blobContentMD5, err := base64.StdEncoding.DecodeString(val)
		if err != nil {
			return blobClientDownloadResponse{}, err
		}
		result.BlobContentMD5 = blobContentMD5
	}
	if val := resp.Header.Get("x-ms-tag-count"); val != "" {
		tagCount, err := strconv.ParseInt(val, 10, 64)
		if err != nil {
			return blobClientDownloadResponse{}, err
		}
		result.TagCount = &tagCount
	}
	if val := resp.Header.Get("x-ms-blob-sealed"); val != "" {
		isSealed, err := strconv.ParseBool(val)
		if err != nil {
			return blobClientDownloadResponse{}, err
		}
		result.IsSealed = &isSealed
	}
	if val := resp.Header.Get("x-ms-last-access-time"); val != "" {
		lastAccessed, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientDownloadResponse{}, err
		}
		result.LastAccessed = &lastAccessed
	}
	if val := resp.Header.Get("x-ms-immutability-policy-until-date"); val != "" {
		immutabilityPolicyExpiresOn, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientDownloadResponse{}, err
		}
		result.ImmutabilityPolicyExpiresOn = &immutabilityPolicyExpiresOn
	}
	if val := resp.Header.Get("x-ms-immutability-policy-mode"); val != "" {
		result.ImmutabilityPolicyMode = (*BlobImmutabilityPolicyMode)(&val)
	}
	if val := resp.Header.Get("x-ms-legal-hold"); val != "" {
		legalHold, err := strconv.ParseBool(val)
		if err != nil {
			return blobClientDownloadResponse{}, err
		}
		result.LegalHold = &legalHold
	}
	if val := resp.Header.Get("x-ms-content-crc64"); val != "" {
		contentCRC64, err := base64.StdEncoding.DecodeString(val)
		if err != nil {
			return blobClientDownloadResponse{}, err
		}
		result.ContentCRC64 = contentCRC64
	}
	if val := resp.Header.Get("x-ms-error-code"); val != "" {
		result.ErrorCode = &val
	}
	return result, nil
}

// GetAccountInfo - Returns the sku name and account kind
// If the operation fails it returns an *azcore.ResponseError type.
// options - blobClientGetAccountInfoOptions contains the optional parameters for the blobClient.GetAccountInfo method.
func (client *blobClient) GetAccountInfo(ctx context.Context, options *blobClientGetAccountInfoOptions) (blobClientGetAccountInfoResponse, error) {
	req, err := client.getAccountInfoCreateRequest(ctx, options)
	if err != nil {
		return blobClientGetAccountInfoResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientGetAccountInfoResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusOK) {
		return blobClientGetAccountInfoResponse{}, runtime.NewResponseError(resp)
	}
	return client.getAccountInfoHandleResponse(resp)
}

// getAccountInfoCreateRequest creates the GetAccountInfo request.
func (client *blobClient) getAccountInfoCreateRequest(ctx context.Context, options *blobClientGetAccountInfoOptions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodGet, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("restype", "account")
	reqQP.Set("comp", "properties")
	req.Raw().URL.RawQuery = reqQP.Encode()
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// getAccountInfoHandleResponse handles the GetAccountInfo response.
func (client *blobClient) getAccountInfoHandleResponse(resp *http.Response) (blobClientGetAccountInfoResponse, error) {
	result := blobClientGetAccountInfoResponse{RawResponse: resp}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientGetAccountInfoResponse{}, err
		}
		result.Date = &date
	}
	if val := resp.Header.Get("x-ms-sku-name"); val != "" {
		result.SKUName = (*SKUName)(&val)
	}
	if val := resp.Header.Get("x-ms-account-kind"); val != "" {
		result.AccountKind = (*AccountKind)(&val)
	}
	return result, nil
}

// GetProperties - The Get Properties operation returns all user-defined metadata, standard HTTP properties, and system properties
// for the blob. It does not return the content of the blob.
// If the operation fails it returns an *azcore.ResponseError type.
// blobClientGetPropertiesOptions - blobClientGetPropertiesOptions contains the optional parameters for the blobClient.GetProperties
// method.
// LeaseAccessConditions - LeaseAccessConditions contains a group of parameters for the containerClient.GetProperties method.
// CpkInfo - CpkInfo contains a group of parameters for the blobClient.Download method.
// ModifiedAccessConditions - ModifiedAccessConditions contains a group of parameters for the containerClient.Delete method.
func (client *blobClient) GetProperties(ctx context.Context, blobClientGetPropertiesOptions *blobClientGetPropertiesOptions, leaseAccessConditions *LeaseAccessConditions, cpkInfo *CpkInfo, modifiedAccessConditions *ModifiedAccessConditions) (blobClientGetPropertiesResponse, error) {
	req, err := client.getPropertiesCreateRequest(ctx, blobClientGetPropertiesOptions, leaseAccessConditions, cpkInfo, modifiedAccessConditions)
	if err != nil {
		return blobClientGetPropertiesResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientGetPropertiesResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusOK) {
		return blobClientGetPropertiesResponse{}, runtime.NewResponseError(resp)
	}
	return client.getPropertiesHandleResponse(resp)
}

// getPropertiesCreateRequest creates the GetProperties request.
func (client *blobClient) getPropertiesCreateRequest(ctx context.Context, blobClientGetPropertiesOptions *blobClientGetPropertiesOptions, leaseAccessConditions *LeaseAccessConditions, cpkInfo *CpkInfo, modifiedAccessConditions *ModifiedAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodHead, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	if blobClientGetPropertiesOptions != nil && blobClientGetPropertiesOptions.Snapshot != nil {
		reqQP.Set("snapshot", *blobClientGetPropertiesOptions.Snapshot)
	}
	if blobClientGetPropertiesOptions != nil && blobClientGetPropertiesOptions.VersionID != nil {
		reqQP.Set("versionid", *blobClientGetPropertiesOptions.VersionID)
	}
	if blobClientGetPropertiesOptions != nil && blobClientGetPropertiesOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientGetPropertiesOptions.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	if leaseAccessConditions != nil && leaseAccessConditions.LeaseID != nil {
		req.Raw().Header.Set("x-ms-lease-id", *leaseAccessConditions.LeaseID)
	}
	if cpkInfo != nil && cpkInfo.EncryptionKey != nil {
		req.Raw().Header.Set("x-ms-encryption-key", *cpkInfo.EncryptionKey)
	}
	if cpkInfo != nil && cpkInfo.EncryptionKeySHA256 != nil {
		req.Raw().Header.Set("x-ms-encryption-key-sha256", *cpkInfo.EncryptionKeySHA256)
	}
	if cpkInfo != nil && cpkInfo.EncryptionAlgorithm != nil {
		req.Raw().Header.Set("x-ms-encryption-algorithm", string(*cpkInfo.EncryptionAlgorithm))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfModifiedSince != nil {
		req.Raw().Header.Set("If-Modified-Since", modifiedAccessConditions.IfModifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfUnmodifiedSince != nil {
		req.Raw().Header.Set("If-Unmodified-Since", modifiedAccessConditions.IfUnmodifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfMatch != nil {
		req.Raw().Header.Set("If-Match", *modifiedAccessConditions.IfMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfNoneMatch != nil {
		req.Raw().Header.Set("If-None-Match", *modifiedAccessConditions.IfNoneMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfTags != nil {
		req.Raw().Header.Set("x-ms-if-tags", *modifiedAccessConditions.IfTags)
	}
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientGetPropertiesOptions != nil && blobClientGetPropertiesOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientGetPropertiesOptions.RequestID)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// getPropertiesHandleResponse handles the GetProperties response.
func (client *blobClient) getPropertiesHandleResponse(resp *http.Response) (blobClientGetPropertiesResponse, error) {
	result := blobClientGetPropertiesResponse{RawResponse: resp}
	if val := resp.Header.Get("Last-Modified"); val != "" {
		lastModified, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.LastModified = &lastModified
	}
	if val := resp.Header.Get("x-ms-creation-time"); val != "" {
		creationTime, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.CreationTime = &creationTime
	}
	for hh := range resp.Header {
		if len(hh) > len("x-ms-meta-") && strings.EqualFold(hh[:len("x-ms-meta-")], "x-ms-meta-") {
			if result.Metadata == nil {
				result.Metadata = map[string]string{}
			}
			result.Metadata[hh[len("x-ms-meta-"):]] = resp.Header.Get(hh)
		}
	}
	if val := resp.Header.Get("x-ms-or-policy-id"); val != "" {
		result.ObjectReplicationPolicyID = &val
	}
	for hh := range resp.Header {
		if len(hh) > len("x-ms-or-") && strings.EqualFold(hh[:len("x-ms-or-")], "x-ms-or-") {
			if result.Metadata == nil {
				result.Metadata = map[string]string{}
			}
			result.Metadata[hh[len("x-ms-or-"):]] = resp.Header.Get(hh)
		}
	}
	if val := resp.Header.Get("x-ms-blob-type"); val != "" {
		result.BlobType = (*BlobType)(&val)
	}
	if val := resp.Header.Get("x-ms-copy-completion-time"); val != "" {
		copyCompletionTime, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.CopyCompletionTime = &copyCompletionTime
	}
	if val := resp.Header.Get("x-ms-copy-status-description"); val != "" {
		result.CopyStatusDescription = &val
	}
	if val := resp.Header.Get("x-ms-copy-id"); val != "" {
		result.CopyID = &val
	}
	if val := resp.Header.Get("x-ms-copy-progress"); val != "" {
		result.CopyProgress = &val
	}
	if val := resp.Header.Get("x-ms-copy-source"); val != "" {
		result.CopySource = &val
	}
	if val := resp.Header.Get("x-ms-copy-status"); val != "" {
		result.CopyStatus = (*CopyStatusType)(&val)
	}
	if val := resp.Header.Get("x-ms-incremental-copy"); val != "" {
		isIncrementalCopy, err := strconv.ParseBool(val)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.IsIncrementalCopy = &isIncrementalCopy
	}
	if val := resp.Header.Get("x-ms-copy-destination-snapshot"); val != "" {
		result.DestinationSnapshot = &val
	}
	if val := resp.Header.Get("x-ms-lease-duration"); val != "" {
		result.LeaseDuration = (*LeaseDurationType)(&val)
	}
	if val := resp.Header.Get("x-ms-lease-state"); val != "" {
		result.LeaseState = (*LeaseStateType)(&val)
	}
	if val := resp.Header.Get("x-ms-lease-status"); val != "" {
		result.LeaseStatus = (*LeaseStatusType)(&val)
	}
	if val := resp.Header.Get("Content-Length"); val != "" {
		contentLength, err := strconv.ParseInt(val, 10, 64)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.ContentLength = &contentLength
	}
	if val := resp.Header.Get("Content-Type"); val != "" {
		result.ContentType = &val
	}
	if val := resp.Header.Get("ETag"); val != "" {
		result.ETag = &val
	}
	if val := resp.Header.Get("Content-MD5"); val != "" {
		contentMD5, err := base64.StdEncoding.DecodeString(val)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.ContentMD5 = contentMD5
	}
	if val := resp.Header.Get("Content-Encoding"); val != "" {
		result.ContentEncoding = &val
	}
	if val := resp.Header.Get("Content-Disposition"); val != "" {
		result.ContentDisposition = &val
	}
	if val := resp.Header.Get("Content-Language"); val != "" {
		result.ContentLanguage = &val
	}
	if val := resp.Header.Get("Cache-Control"); val != "" {
		result.CacheControl = &val
	}
	if val := resp.Header.Get("x-ms-blob-sequence-number"); val != "" {
		blobSequenceNumber, err := strconv.ParseInt(val, 10, 64)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.BlobSequenceNumber = &blobSequenceNumber
	}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.Date = &date
	}
	if val := resp.Header.Get("Accept-Ranges"); val != "" {
		result.AcceptRanges = &val
	}
	if val := resp.Header.Get("x-ms-blob-committed-block-count"); val != "" {
		blobCommittedBlockCount32, err := strconv.ParseInt(val, 10, 32)
		blobCommittedBlockCount := int32(blobCommittedBlockCount32)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.BlobCommittedBlockCount = &blobCommittedBlockCount
	}
	if val := resp.Header.Get("x-ms-server-encrypted"); val != "" {
		isServerEncrypted, err := strconv.ParseBool(val)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.IsServerEncrypted = &isServerEncrypted
	}
	if val := resp.Header.Get("x-ms-encryption-key-sha256"); val != "" {
		result.EncryptionKeySHA256 = &val
	}
	if val := resp.Header.Get("x-ms-encryption-scope"); val != "" {
		result.EncryptionScope = &val
	}
	if val := resp.Header.Get("x-ms-access-tier"); val != "" {
		result.AccessTier = &val
	}
	if val := resp.Header.Get("x-ms-access-tier-inferred"); val != "" {
		accessTierInferred, err := strconv.ParseBool(val)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.AccessTierInferred = &accessTierInferred
	}
	if val := resp.Header.Get("x-ms-archive-status"); val != "" {
		result.ArchiveStatus = &val
	}
	if val := resp.Header.Get("x-ms-access-tier-change-time"); val != "" {
		accessTierChangeTime, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.AccessTierChangeTime = &accessTierChangeTime
	}
	if val := resp.Header.Get("x-ms-version-id"); val != "" {
		result.VersionID = &val
	}
	if val := resp.Header.Get("x-ms-is-current-version"); val != "" {
		isCurrentVersion, err := strconv.ParseBool(val)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.IsCurrentVersion = &isCurrentVersion
	}
	if val := resp.Header.Get("x-ms-tag-count"); val != "" {
		tagCount, err := strconv.ParseInt(val, 10, 64)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.TagCount = &tagCount
	}
	if val := resp.Header.Get("x-ms-expiry-time"); val != "" {
		expiresOn, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.ExpiresOn = &expiresOn
	}
	if val := resp.Header.Get("x-ms-blob-sealed"); val != "" {
		isSealed, err := strconv.ParseBool(val)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.IsSealed = &isSealed
	}
	if val := resp.Header.Get("x-ms-rehydrate-priority"); val != "" {
		result.RehydratePriority = &val
	}
	if val := resp.Header.Get("x-ms-last-access-time"); val != "" {
		lastAccessed, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.LastAccessed = &lastAccessed
	}
	if val := resp.Header.Get("x-ms-immutability-policy-until-date"); val != "" {
		immutabilityPolicyExpiresOn, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.ImmutabilityPolicyExpiresOn = &immutabilityPolicyExpiresOn
	}
	if val := resp.Header.Get("x-ms-immutability-policy-mode"); val != "" {
		result.ImmutabilityPolicyMode = (*BlobImmutabilityPolicyMode)(&val)
	}
	if val := resp.Header.Get("x-ms-legal-hold"); val != "" {
		legalHold, err := strconv.ParseBool(val)
		if err != nil {
			return blobClientGetPropertiesResponse{}, err
		}
		result.LegalHold = &legalHold
	}
	return result, nil
}

// GetTags - The Get Tags operation enables users to get the tags associated with a blob.
// If the operation fails it returns an *azcore.ResponseError type.
// blobClientGetTagsOptions - blobClientGetTagsOptions contains the optional parameters for the blobClient.GetTags method.
// ModifiedAccessConditions - ModifiedAccessConditions contains a group of parameters for the containerClient.Delete method.
// LeaseAccessConditions - LeaseAccessConditions contains a group of parameters for the containerClient.GetProperties method.
func (client *blobClient) GetTags(ctx context.Context, blobClientGetTagsOptions *blobClientGetTagsOptions, modifiedAccessConditions *ModifiedAccessConditions, leaseAccessConditions *LeaseAccessConditions) (blobClientGetTagsResponse, error) {
	req, err := client.getTagsCreateRequest(ctx, blobClientGetTagsOptions, modifiedAccessConditions, leaseAccessConditions)
	if err != nil {
		return blobClientGetTagsResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientGetTagsResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusOK) {
		return blobClientGetTagsResponse{}, runtime.NewResponseError(resp)
	}
	return client.getTagsHandleResponse(resp)
}

// getTagsCreateRequest creates the GetTags request.
func (client *blobClient) getTagsCreateRequest(ctx context.Context, blobClientGetTagsOptions *blobClientGetTagsOptions, modifiedAccessConditions *ModifiedAccessConditions, leaseAccessConditions *LeaseAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodGet, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("comp", "tags")
	if blobClientGetTagsOptions != nil && blobClientGetTagsOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientGetTagsOptions.Timeout), 10))
	}
	if blobClientGetTagsOptions != nil && blobClientGetTagsOptions.Snapshot != nil {
		reqQP.Set("snapshot", *blobClientGetTagsOptions.Snapshot)
	}
	if blobClientGetTagsOptions != nil && blobClientGetTagsOptions.VersionID != nil {
		reqQP.Set("versionid", *blobClientGetTagsOptions.VersionID)
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientGetTagsOptions != nil && blobClientGetTagsOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientGetTagsOptions.RequestID)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfTags != nil {
		req.Raw().Header.Set("x-ms-if-tags", *modifiedAccessConditions.IfTags)
	}
	if leaseAccessConditions != nil && leaseAccessConditions.LeaseID != nil {
		req.Raw().Header.Set("x-ms-lease-id", *leaseAccessConditions.LeaseID)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// getTagsHandleResponse handles the GetTags response.
func (client *blobClient) getTagsHandleResponse(resp *http.Response) (blobClientGetTagsResponse, error) {
	result := blobClientGetTagsResponse{RawResponse: resp}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientGetTagsResponse{}, err
		}
		result.Date = &date
	}
	if err := runtime.UnmarshalAsXML(resp, &result.BlobTags); err != nil {
		return blobClientGetTagsResponse{}, err
	}
	return result, nil
}

// Query - The Query operation enables users to select/project on blob data by providing simple query expressions.
// If the operation fails it returns an *azcore.ResponseError type.
// blobClientQueryOptions - blobClientQueryOptions contains the optional parameters for the blobClient.Query method.
// LeaseAccessConditions - LeaseAccessConditions contains a group of parameters for the containerClient.GetProperties method.
// CpkInfo - CpkInfo contains a group of parameters for the blobClient.Download method.
// ModifiedAccessConditions - ModifiedAccessConditions contains a group of parameters for the containerClient.Delete method.
func (client *blobClient) Query(ctx context.Context, blobClientQueryOptions *blobClientQueryOptions, leaseAccessConditions *LeaseAccessConditions, cpkInfo *CpkInfo, modifiedAccessConditions *ModifiedAccessConditions) (blobClientQueryResponse, error) {
	req, err := client.queryCreateRequest(ctx, blobClientQueryOptions, leaseAccessConditions, cpkInfo, modifiedAccessConditions)
	if err != nil {
		return blobClientQueryResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientQueryResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusOK, http.StatusPartialContent) {
		return blobClientQueryResponse{}, runtime.NewResponseError(resp)
	}
	return client.queryHandleResponse(resp)
}

// queryCreateRequest creates the Query request.
func (client *blobClient) queryCreateRequest(ctx context.Context, blobClientQueryOptions *blobClientQueryOptions, leaseAccessConditions *LeaseAccessConditions, cpkInfo *CpkInfo, modifiedAccessConditions *ModifiedAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodPost, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("comp", "query")
	if blobClientQueryOptions != nil && blobClientQueryOptions.Snapshot != nil {
		reqQP.Set("snapshot", *blobClientQueryOptions.Snapshot)
	}
	if blobClientQueryOptions != nil && blobClientQueryOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientQueryOptions.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	runtime.SkipBodyDownload(req)
	if leaseAccessConditions != nil && leaseAccessConditions.LeaseID != nil {
		req.Raw().Header.Set("x-ms-lease-id", *leaseAccessConditions.LeaseID)
	}
	if cpkInfo != nil && cpkInfo.EncryptionKey != nil {
		req.Raw().Header.Set("x-ms-encryption-key", *cpkInfo.EncryptionKey)
	}
	if cpkInfo != nil && cpkInfo.EncryptionKeySHA256 != nil {
		req.Raw().Header.Set("x-ms-encryption-key-sha256", *cpkInfo.EncryptionKeySHA256)
	}
	if cpkInfo != nil && cpkInfo.EncryptionAlgorithm != nil {
		req.Raw().Header.Set("x-ms-encryption-algorithm", string(*cpkInfo.EncryptionAlgorithm))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfModifiedSince != nil {
		req.Raw().Header.Set("If-Modified-Since", modifiedAccessConditions.IfModifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfUnmodifiedSince != nil {
		req.Raw().Header.Set("If-Unmodified-Since", modifiedAccessConditions.IfUnmodifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfMatch != nil {
		req.Raw().Header.Set("If-Match", *modifiedAccessConditions.IfMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfNoneMatch != nil {
		req.Raw().Header.Set("If-None-Match", *modifiedAccessConditions.IfNoneMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfTags != nil {
		req.Raw().Header.Set("x-ms-if-tags", *modifiedAccessConditions.IfTags)
	}
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientQueryOptions != nil && blobClientQueryOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientQueryOptions.RequestID)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	if blobClientQueryOptions != nil && blobClientQueryOptions.QueryRequest != nil {
		return req, runtime.MarshalAsXML(req, *blobClientQueryOptions.QueryRequest)
	}
	return req, nil
}

// queryHandleResponse handles the Query response.
func (client *blobClient) queryHandleResponse(resp *http.Response) (blobClientQueryResponse, error) {
	result := blobClientQueryResponse{RawResponse: resp}
	if val := resp.Header.Get("Last-Modified"); val != "" {
		lastModified, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientQueryResponse{}, err
		}
		result.LastModified = &lastModified
	}
	for hh := range resp.Header {
		if len(hh) > len("x-ms-meta-") && strings.EqualFold(hh[:len("x-ms-meta-")], "x-ms-meta-") {
			if result.Metadata == nil {
				result.Metadata = map[string]string{}
			}
			result.Metadata[hh[len("x-ms-meta-"):]] = resp.Header.Get(hh)
		}
	}
	if val := resp.Header.Get("Content-Length"); val != "" {
		contentLength, err := strconv.ParseInt(val, 10, 64)
		if err != nil {
			return blobClientQueryResponse{}, err
		}
		result.ContentLength = &contentLength
	}
	if val := resp.Header.Get("Content-Type"); val != "" {
		result.ContentType = &val
	}
	if val := resp.Header.Get("Content-Range"); val != "" {
		result.ContentRange = &val
	}
	if val := resp.Header.Get("ETag"); val != "" {
		result.ETag = &val
	}
	if val := resp.Header.Get("Content-MD5"); val != "" {
		contentMD5, err := base64.StdEncoding.DecodeString(val)
		if err != nil {
			return blobClientQueryResponse{}, err
		}
		result.ContentMD5 = contentMD5
	}
	if val := resp.Header.Get("Content-Encoding"); val != "" {
		result.ContentEncoding = &val
	}
	if val := resp.Header.Get("Cache-Control"); val != "" {
		result.CacheControl = &val
	}
	if val := resp.Header.Get("Content-Disposition"); val != "" {
		result.ContentDisposition = &val
	}
	if val := resp.Header.Get("Content-Language"); val != "" {
		result.ContentLanguage = &val
	}
	if val := resp.Header.Get("x-ms-blob-sequence-number"); val != "" {
		blobSequenceNumber, err := strconv.ParseInt(val, 10, 64)
		if err != nil {
			return blobClientQueryResponse{}, err
		}
		result.BlobSequenceNumber = &blobSequenceNumber
	}
	if val := resp.Header.Get("x-ms-blob-type"); val != "" {
		result.BlobType = (*BlobType)(&val)
	}
	if val := resp.Header.Get("x-ms-copy-completion-time"); val != "" {
		copyCompletionTime, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientQueryResponse{}, err
		}
		result.CopyCompletionTime = &copyCompletionTime
	}
	if val := resp.Header.Get("x-ms-copy-status-description"); val != "" {
		result.CopyStatusDescription = &val
	}
	if val := resp.Header.Get("x-ms-copy-id"); val != "" {
		result.CopyID = &val
	}
	if val := resp.Header.Get("x-ms-copy-progress"); val != "" {
		result.CopyProgress = &val
	}
	if val := resp.Header.Get("x-ms-copy-source"); val != "" {
		result.CopySource = &val
	}
	if val := resp.Header.Get("x-ms-copy-status"); val != "" {
		result.CopyStatus = (*CopyStatusType)(&val)
	}
	if val := resp.Header.Get("x-ms-lease-duration"); val != "" {
		result.LeaseDuration = (*LeaseDurationType)(&val)
	}
	if val := resp.Header.Get("x-ms-lease-state"); val != "" {
		result.LeaseState = (*LeaseStateType)(&val)
	}
	if val := resp.Header.Get("x-ms-lease-status"); val != "" {
		result.LeaseStatus = (*LeaseStatusType)(&val)
	}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("Accept-Ranges"); val != "" {
		result.AcceptRanges = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientQueryResponse{}, err
		}
		result.Date = &date
	}
	if val := resp.Header.Get("x-ms-blob-committed-block-count"); val != "" {
		blobCommittedBlockCount32, err := strconv.ParseInt(val, 10, 32)
		blobCommittedBlockCount := int32(blobCommittedBlockCount32)
		if err != nil {
			return blobClientQueryResponse{}, err
		}
		result.BlobCommittedBlockCount = &blobCommittedBlockCount
	}
	if val := resp.Header.Get("x-ms-server-encrypted"); val != "" {
		isServerEncrypted, err := strconv.ParseBool(val)
		if err != nil {
			return blobClientQueryResponse{}, err
		}
		result.IsServerEncrypted = &isServerEncrypted
	}
	if val := resp.Header.Get("x-ms-encryption-key-sha256"); val != "" {
		result.EncryptionKeySHA256 = &val
	}
	if val := resp.Header.Get("x-ms-encryption-scope"); val != "" {
		result.EncryptionScope = &val
	}
	if val := resp.Header.Get("x-ms-blob-content-md5"); val != "" {
		blobContentMD5, err := base64.StdEncoding.DecodeString(val)
		if err != nil {
			return blobClientQueryResponse{}, err
		}
		result.BlobContentMD5 = blobContentMD5
	}
	if val := resp.Header.Get("x-ms-content-crc64"); val != "" {
		contentCRC64, err := base64.StdEncoding.DecodeString(val)
		if err != nil {
			return blobClientQueryResponse{}, err
		}
		result.ContentCRC64 = contentCRC64
	}
	return result, nil
}

// ReleaseLease - [Update] The Lease Blob operation establishes and manages a lock on a blob for write and delete operations
// If the operation fails it returns an *azcore.ResponseError type.
// leaseID - Specifies the current lease ID on the resource.
// blobClientReleaseLeaseOptions - blobClientReleaseLeaseOptions contains the optional parameters for the blobClient.ReleaseLease
// method.
// ModifiedAccessConditions - ModifiedAccessConditions contains a group of parameters for the containerClient.Delete method.
func (client *blobClient) ReleaseLease(ctx context.Context, leaseID string, blobClientReleaseLeaseOptions *blobClientReleaseLeaseOptions, modifiedAccessConditions *ModifiedAccessConditions) (blobClientReleaseLeaseResponse, error) {
	req, err := client.releaseLeaseCreateRequest(ctx, leaseID, blobClientReleaseLeaseOptions, modifiedAccessConditions)
	if err != nil {
		return blobClientReleaseLeaseResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientReleaseLeaseResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusOK) {
		return blobClientReleaseLeaseResponse{}, runtime.NewResponseError(resp)
	}
	return client.releaseLeaseHandleResponse(resp)
}

// releaseLeaseCreateRequest creates the ReleaseLease request.
func (client *blobClient) releaseLeaseCreateRequest(ctx context.Context, leaseID string, blobClientReleaseLeaseOptions *blobClientReleaseLeaseOptions, modifiedAccessConditions *ModifiedAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodPut, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("comp", "lease")
	if blobClientReleaseLeaseOptions != nil && blobClientReleaseLeaseOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientReleaseLeaseOptions.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	req.Raw().Header.Set("x-ms-lease-action", "release")
	req.Raw().Header.Set("x-ms-lease-id", leaseID)
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfModifiedSince != nil {
		req.Raw().Header.Set("If-Modified-Since", modifiedAccessConditions.IfModifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfUnmodifiedSince != nil {
		req.Raw().Header.Set("If-Unmodified-Since", modifiedAccessConditions.IfUnmodifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfMatch != nil {
		req.Raw().Header.Set("If-Match", *modifiedAccessConditions.IfMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfNoneMatch != nil {
		req.Raw().Header.Set("If-None-Match", *modifiedAccessConditions.IfNoneMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfTags != nil {
		req.Raw().Header.Set("x-ms-if-tags", *modifiedAccessConditions.IfTags)
	}
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientReleaseLeaseOptions != nil && blobClientReleaseLeaseOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientReleaseLeaseOptions.RequestID)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// releaseLeaseHandleResponse handles the ReleaseLease response.
func (client *blobClient) releaseLeaseHandleResponse(resp *http.Response) (blobClientReleaseLeaseResponse, error) {
	result := blobClientReleaseLeaseResponse{RawResponse: resp}
	if val := resp.Header.Get("ETag"); val != "" {
		result.ETag = &val
	}
	if val := resp.Header.Get("Last-Modified"); val != "" {
		lastModified, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientReleaseLeaseResponse{}, err
		}
		result.LastModified = &lastModified
	}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientReleaseLeaseResponse{}, err
		}
		result.Date = &date
	}
	return result, nil
}

// RenewLease - [Update] The Lease Blob operation establishes and manages a lock on a blob for write and delete operations
// If the operation fails it returns an *azcore.ResponseError type.
// leaseID - Specifies the current lease ID on the resource.
// blobClientRenewLeaseOptions - blobClientRenewLeaseOptions contains the optional parameters for the blobClient.RenewLease
// method.
// ModifiedAccessConditions - ModifiedAccessConditions contains a group of parameters for the containerClient.Delete method.
func (client *blobClient) RenewLease(ctx context.Context, leaseID string, blobClientRenewLeaseOptions *blobClientRenewLeaseOptions, modifiedAccessConditions *ModifiedAccessConditions) (blobClientRenewLeaseResponse, error) {
	req, err := client.renewLeaseCreateRequest(ctx, leaseID, blobClientRenewLeaseOptions, modifiedAccessConditions)
	if err != nil {
		return blobClientRenewLeaseResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientRenewLeaseResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusOK) {
		return blobClientRenewLeaseResponse{}, runtime.NewResponseError(resp)
	}
	return client.renewLeaseHandleResponse(resp)
}

// renewLeaseCreateRequest creates the RenewLease request.
func (client *blobClient) renewLeaseCreateRequest(ctx context.Context, leaseID string, blobClientRenewLeaseOptions *blobClientRenewLeaseOptions, modifiedAccessConditions *ModifiedAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodPut, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("comp", "lease")
	if blobClientRenewLeaseOptions != nil && blobClientRenewLeaseOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientRenewLeaseOptions.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	req.Raw().Header.Set("x-ms-lease-action", "renew")
	req.Raw().Header.Set("x-ms-lease-id", leaseID)
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfModifiedSince != nil {
		req.Raw().Header.Set("If-Modified-Since", modifiedAccessConditions.IfModifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfUnmodifiedSince != nil {
		req.Raw().Header.Set("If-Unmodified-Since", modifiedAccessConditions.IfUnmodifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfMatch != nil {
		req.Raw().Header.Set("If-Match", *modifiedAccessConditions.IfMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfNoneMatch != nil {
		req.Raw().Header.Set("If-None-Match", *modifiedAccessConditions.IfNoneMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfTags != nil {
		req.Raw().Header.Set("x-ms-if-tags", *modifiedAccessConditions.IfTags)
	}
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientRenewLeaseOptions != nil && blobClientRenewLeaseOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientRenewLeaseOptions.RequestID)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// renewLeaseHandleResponse handles the RenewLease response.
func (client *blobClient) renewLeaseHandleResponse(resp *http.Response) (blobClientRenewLeaseResponse, error) {
	result := blobClientRenewLeaseResponse{RawResponse: resp}
	if val := resp.Header.Get("ETag"); val != "" {
		result.ETag = &val
	}
	if val := resp.Header.Get("Last-Modified"); val != "" {
		lastModified, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientRenewLeaseResponse{}, err
		}
		result.LastModified = &lastModified
	}
	if val := resp.Header.Get("x-ms-lease-id"); val != "" {
		result.LeaseID = &val
	}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientRenewLeaseResponse{}, err
		}
		result.Date = &date
	}
	return result, nil
}

// SetExpiry - Sets the time a blob will expire and be deleted.
// If the operation fails it returns an *azcore.ResponseError type.
// expiryOptions - Required. Indicates mode of the expiry time
// options - blobClientSetExpiryOptions contains the optional parameters for the blobClient.SetExpiry method.
func (client *blobClient) SetExpiry(ctx context.Context, expiryOptions BlobExpiryOptions, options *blobClientSetExpiryOptions) (blobClientSetExpiryResponse, error) {
	req, err := client.setExpiryCreateRequest(ctx, expiryOptions, options)
	if err != nil {
		return blobClientSetExpiryResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientSetExpiryResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusOK) {
		return blobClientSetExpiryResponse{}, runtime.NewResponseError(resp)
	}
	return client.setExpiryHandleResponse(resp)
}

// setExpiryCreateRequest creates the SetExpiry request.
func (client *blobClient) setExpiryCreateRequest(ctx context.Context, expiryOptions BlobExpiryOptions, options *blobClientSetExpiryOptions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodPut, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("comp", "expiry")
	if options != nil && options.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*options.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if options != nil && options.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *options.RequestID)
	}
	req.Raw().Header.Set("x-ms-expiry-option", string(expiryOptions))
	if options != nil && options.ExpiresOn != nil {
		req.Raw().Header.Set("x-ms-expiry-time", *options.ExpiresOn)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// setExpiryHandleResponse handles the SetExpiry response.
func (client *blobClient) setExpiryHandleResponse(resp *http.Response) (blobClientSetExpiryResponse, error) {
	result := blobClientSetExpiryResponse{RawResponse: resp}
	if val := resp.Header.Get("ETag"); val != "" {
		result.ETag = &val
	}
	if val := resp.Header.Get("Last-Modified"); val != "" {
		lastModified, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientSetExpiryResponse{}, err
		}
		result.LastModified = &lastModified
	}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientSetExpiryResponse{}, err
		}
		result.Date = &date
	}
	return result, nil
}

// SetHTTPHeaders - The Set HTTP Headers operation sets system properties on the blob
// If the operation fails it returns an *azcore.ResponseError type.
// blobClientSetHTTPHeadersOptions - blobClientSetHTTPHeadersOptions contains the optional parameters for the blobClient.SetHTTPHeaders
// method.
// BlobHTTPHeaders - BlobHTTPHeaders contains a group of parameters for the blobClient.SetHTTPHeaders method.
// LeaseAccessConditions - LeaseAccessConditions contains a group of parameters for the containerClient.GetProperties method.
// ModifiedAccessConditions - ModifiedAccessConditions contains a group of parameters for the containerClient.Delete method.
func (client *blobClient) SetHTTPHeaders(ctx context.Context, blobClientSetHTTPHeadersOptions *blobClientSetHTTPHeadersOptions, blobHTTPHeaders *BlobHTTPHeaders, leaseAccessConditions *LeaseAccessConditions, modifiedAccessConditions *ModifiedAccessConditions) (blobClientSetHTTPHeadersResponse, error) {
	req, err := client.setHTTPHeadersCreateRequest(ctx, blobClientSetHTTPHeadersOptions, blobHTTPHeaders, leaseAccessConditions, modifiedAccessConditions)
	if err != nil {
		return blobClientSetHTTPHeadersResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientSetHTTPHeadersResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusOK) {
		return blobClientSetHTTPHeadersResponse{}, runtime.NewResponseError(resp)
	}
	return client.setHTTPHeadersHandleResponse(resp)
}

// setHTTPHeadersCreateRequest creates the SetHTTPHeaders request.
func (client *blobClient) setHTTPHeadersCreateRequest(ctx context.Context, blobClientSetHTTPHeadersOptions *blobClientSetHTTPHeadersOptions, blobHTTPHeaders *BlobHTTPHeaders, leaseAccessConditions *LeaseAccessConditions, modifiedAccessConditions *ModifiedAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodPut, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("comp", "properties")
	if blobClientSetHTTPHeadersOptions != nil && blobClientSetHTTPHeadersOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientSetHTTPHeadersOptions.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	if blobHTTPHeaders != nil && blobHTTPHeaders.BlobCacheControl != nil {
		req.Raw().Header.Set("x-ms-blob-cache-control", *blobHTTPHeaders.BlobCacheControl)
	}
	if blobHTTPHeaders != nil && blobHTTPHeaders.BlobContentType != nil {
		req.Raw().Header.Set("x-ms-blob-content-type", *blobHTTPHeaders.BlobContentType)
	}
	if blobHTTPHeaders != nil && blobHTTPHeaders.BlobContentMD5 != nil {
		req.Raw().Header.Set("x-ms-blob-content-md5", base64.StdEncoding.EncodeToString(blobHTTPHeaders.BlobContentMD5))
	}
	if blobHTTPHeaders != nil && blobHTTPHeaders.BlobContentEncoding != nil {
		req.Raw().Header.Set("x-ms-blob-content-encoding", *blobHTTPHeaders.BlobContentEncoding)
	}
	if blobHTTPHeaders != nil && blobHTTPHeaders.BlobContentLanguage != nil {
		req.Raw().Header.Set("x-ms-blob-content-language", *blobHTTPHeaders.BlobContentLanguage)
	}
	if leaseAccessConditions != nil && leaseAccessConditions.LeaseID != nil {
		req.Raw().Header.Set("x-ms-lease-id", *leaseAccessConditions.LeaseID)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfModifiedSince != nil {
		req.Raw().Header.Set("If-Modified-Since", modifiedAccessConditions.IfModifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfUnmodifiedSince != nil {
		req.Raw().Header.Set("If-Unmodified-Since", modifiedAccessConditions.IfUnmodifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfMatch != nil {
		req.Raw().Header.Set("If-Match", *modifiedAccessConditions.IfMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfNoneMatch != nil {
		req.Raw().Header.Set("If-None-Match", *modifiedAccessConditions.IfNoneMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfTags != nil {
		req.Raw().Header.Set("x-ms-if-tags", *modifiedAccessConditions.IfTags)
	}
	if blobHTTPHeaders != nil && blobHTTPHeaders.BlobContentDisposition != nil {
		req.Raw().Header.Set("x-ms-blob-content-disposition", *blobHTTPHeaders.BlobContentDisposition)
	}
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientSetHTTPHeadersOptions != nil && blobClientSetHTTPHeadersOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientSetHTTPHeadersOptions.RequestID)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// setHTTPHeadersHandleResponse handles the SetHTTPHeaders response.
func (client *blobClient) setHTTPHeadersHandleResponse(resp *http.Response) (blobClientSetHTTPHeadersResponse, error) {
	result := blobClientSetHTTPHeadersResponse{RawResponse: resp}
	if val := resp.Header.Get("ETag"); val != "" {
		result.ETag = &val
	}
	if val := resp.Header.Get("Last-Modified"); val != "" {
		lastModified, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientSetHTTPHeadersResponse{}, err
		}
		result.LastModified = &lastModified
	}
	if val := resp.Header.Get("x-ms-blob-sequence-number"); val != "" {
		blobSequenceNumber, err := strconv.ParseInt(val, 10, 64)
		if err != nil {
			return blobClientSetHTTPHeadersResponse{}, err
		}
		result.BlobSequenceNumber = &blobSequenceNumber
	}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientSetHTTPHeadersResponse{}, err
		}
		result.Date = &date
	}
	return result, nil
}

// SetImmutabilityPolicy - The Set Immutability Policy operation sets the immutability policy on the blob
// If the operation fails it returns an *azcore.ResponseError type.
// blobClientSetImmutabilityPolicyOptions - blobClientSetImmutabilityPolicyOptions contains the optional parameters for the
// blobClient.SetImmutabilityPolicy method.
// ModifiedAccessConditions - ModifiedAccessConditions contains a group of parameters for the containerClient.Delete method.
func (client *blobClient) SetImmutabilityPolicy(ctx context.Context, blobClientSetImmutabilityPolicyOptions *blobClientSetImmutabilityPolicyOptions, modifiedAccessConditions *ModifiedAccessConditions) (blobClientSetImmutabilityPolicyResponse, error) {
	req, err := client.setImmutabilityPolicyCreateRequest(ctx, blobClientSetImmutabilityPolicyOptions, modifiedAccessConditions)
	if err != nil {
		return blobClientSetImmutabilityPolicyResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientSetImmutabilityPolicyResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusOK) {
		return blobClientSetImmutabilityPolicyResponse{}, runtime.NewResponseError(resp)
	}
	return client.setImmutabilityPolicyHandleResponse(resp)
}

// setImmutabilityPolicyCreateRequest creates the SetImmutabilityPolicy request.
func (client *blobClient) setImmutabilityPolicyCreateRequest(ctx context.Context, blobClientSetImmutabilityPolicyOptions *blobClientSetImmutabilityPolicyOptions, modifiedAccessConditions *ModifiedAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodPut, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("comp", "immutabilityPolicies")
	if blobClientSetImmutabilityPolicyOptions != nil && blobClientSetImmutabilityPolicyOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientSetImmutabilityPolicyOptions.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientSetImmutabilityPolicyOptions != nil && blobClientSetImmutabilityPolicyOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientSetImmutabilityPolicyOptions.RequestID)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfUnmodifiedSince != nil {
		req.Raw().Header.Set("If-Unmodified-Since", modifiedAccessConditions.IfUnmodifiedSince.Format(time.RFC1123))
	}
	if blobClientSetImmutabilityPolicyOptions != nil && blobClientSetImmutabilityPolicyOptions.ImmutabilityPolicyExpiry != nil {
		req.Raw().Header.Set("x-ms-immutability-policy-until-date", blobClientSetImmutabilityPolicyOptions.ImmutabilityPolicyExpiry.Format(time.RFC1123))
	}
	if blobClientSetImmutabilityPolicyOptions != nil && blobClientSetImmutabilityPolicyOptions.ImmutabilityPolicyMode != nil {
		req.Raw().Header.Set("x-ms-immutability-policy-mode", string(*blobClientSetImmutabilityPolicyOptions.ImmutabilityPolicyMode))
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// setImmutabilityPolicyHandleResponse handles the SetImmutabilityPolicy response.
func (client *blobClient) setImmutabilityPolicyHandleResponse(resp *http.Response) (blobClientSetImmutabilityPolicyResponse, error) {
	result := blobClientSetImmutabilityPolicyResponse{RawResponse: resp}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientSetImmutabilityPolicyResponse{}, err
		}
		result.Date = &date
	}
	if val := resp.Header.Get("x-ms-immutability-policy-until-date"); val != "" {
		immutabilityPolicyExpiry, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientSetImmutabilityPolicyResponse{}, err
		}
		result.ImmutabilityPolicyExpiry = &immutabilityPolicyExpiry
	}
	if val := resp.Header.Get("x-ms-immutability-policy-mode"); val != "" {
		result.ImmutabilityPolicyMode = (*BlobImmutabilityPolicyMode)(&val)
	}
	return result, nil
}

// SetLegalHold - The Set Legal Hold operation sets a legal hold on the blob.
// If the operation fails it returns an *azcore.ResponseError type.
// legalHold - Specified if a legal hold should be set on the blob.
// options - blobClientSetLegalHoldOptions contains the optional parameters for the blobClient.SetLegalHold method.
func (client *blobClient) SetLegalHold(ctx context.Context, legalHold bool, options *blobClientSetLegalHoldOptions) (blobClientSetLegalHoldResponse, error) {
	req, err := client.setLegalHoldCreateRequest(ctx, legalHold, options)
	if err != nil {
		return blobClientSetLegalHoldResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientSetLegalHoldResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusOK) {
		return blobClientSetLegalHoldResponse{}, runtime.NewResponseError(resp)
	}
	return client.setLegalHoldHandleResponse(resp)
}

// setLegalHoldCreateRequest creates the SetLegalHold request.
func (client *blobClient) setLegalHoldCreateRequest(ctx context.Context, legalHold bool, options *blobClientSetLegalHoldOptions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodPut, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("comp", "legalhold")
	if options != nil && options.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*options.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if options != nil && options.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *options.RequestID)
	}
	req.Raw().Header.Set("x-ms-legal-hold", strconv.FormatBool(legalHold))
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// setLegalHoldHandleResponse handles the SetLegalHold response.
func (client *blobClient) setLegalHoldHandleResponse(resp *http.Response) (blobClientSetLegalHoldResponse, error) {
	result := blobClientSetLegalHoldResponse{RawResponse: resp}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientSetLegalHoldResponse{}, err
		}
		result.Date = &date
	}
	if val := resp.Header.Get("x-ms-legal-hold"); val != "" {
		legalHold, err := strconv.ParseBool(val)
		if err != nil {
			return blobClientSetLegalHoldResponse{}, err
		}
		result.LegalHold = &legalHold
	}
	return result, nil
}

// SetMetadata - The Set Blob Metadata operation sets user-defined metadata for the specified blob as one or more name-value
// pairs
// If the operation fails it returns an *azcore.ResponseError type.
// blobClientSetMetadataOptions - blobClientSetMetadataOptions contains the optional parameters for the blobClient.SetMetadata
// method.
// LeaseAccessConditions - LeaseAccessConditions contains a group of parameters for the containerClient.GetProperties method.
// CpkInfo - CpkInfo contains a group of parameters for the blobClient.Download method.
// CpkScopeInfo - CpkScopeInfo contains a group of parameters for the blobClient.SetMetadata method.
// ModifiedAccessConditions - ModifiedAccessConditions contains a group of parameters for the containerClient.Delete method.
func (client *blobClient) SetMetadata(ctx context.Context, blobClientSetMetadataOptions *blobClientSetMetadataOptions, leaseAccessConditions *LeaseAccessConditions, cpkInfo *CpkInfo, cpkScopeInfo *CpkScopeInfo, modifiedAccessConditions *ModifiedAccessConditions) (blobClientSetMetadataResponse, error) {
	req, err := client.setMetadataCreateRequest(ctx, blobClientSetMetadataOptions, leaseAccessConditions, cpkInfo, cpkScopeInfo, modifiedAccessConditions)
	if err != nil {
		return blobClientSetMetadataResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientSetMetadataResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusOK) {
		return blobClientSetMetadataResponse{}, runtime.NewResponseError(resp)
	}
	return client.setMetadataHandleResponse(resp)
}

// setMetadataCreateRequest creates the SetMetadata request.
func (client *blobClient) setMetadataCreateRequest(ctx context.Context, blobClientSetMetadataOptions *blobClientSetMetadataOptions, leaseAccessConditions *LeaseAccessConditions, cpkInfo *CpkInfo, cpkScopeInfo *CpkScopeInfo, modifiedAccessConditions *ModifiedAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodPut, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("comp", "metadata")
	if blobClientSetMetadataOptions != nil && blobClientSetMetadataOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientSetMetadataOptions.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	if blobClientSetMetadataOptions != nil && blobClientSetMetadataOptions.Metadata != nil {
		for k, v := range blobClientSetMetadataOptions.Metadata {
			req.Raw().Header.Set("x-ms-meta-"+k, v)
		}
	}
	if leaseAccessConditions != nil && leaseAccessConditions.LeaseID != nil {
		req.Raw().Header.Set("x-ms-lease-id", *leaseAccessConditions.LeaseID)
	}
	if cpkInfo != nil && cpkInfo.EncryptionKey != nil {
		req.Raw().Header.Set("x-ms-encryption-key", *cpkInfo.EncryptionKey)
	}
	if cpkInfo != nil && cpkInfo.EncryptionKeySHA256 != nil {
		req.Raw().Header.Set("x-ms-encryption-key-sha256", *cpkInfo.EncryptionKeySHA256)
	}
	if cpkInfo != nil && cpkInfo.EncryptionAlgorithm != nil {
		req.Raw().Header.Set("x-ms-encryption-algorithm", string(*cpkInfo.EncryptionAlgorithm))
	}
	if cpkScopeInfo != nil && cpkScopeInfo.EncryptionScope != nil {
		req.Raw().Header.Set("x-ms-encryption-scope", *cpkScopeInfo.EncryptionScope)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfModifiedSince != nil {
		req.Raw().Header.Set("If-Modified-Since", modifiedAccessConditions.IfModifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfUnmodifiedSince != nil {
		req.Raw().Header.Set("If-Unmodified-Since", modifiedAccessConditions.IfUnmodifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfMatch != nil {
		req.Raw().Header.Set("If-Match", *modifiedAccessConditions.IfMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfNoneMatch != nil {
		req.Raw().Header.Set("If-None-Match", *modifiedAccessConditions.IfNoneMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfTags != nil {
		req.Raw().Header.Set("x-ms-if-tags", *modifiedAccessConditions.IfTags)
	}
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientSetMetadataOptions != nil && blobClientSetMetadataOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientSetMetadataOptions.RequestID)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// setMetadataHandleResponse handles the SetMetadata response.
func (client *blobClient) setMetadataHandleResponse(resp *http.Response) (blobClientSetMetadataResponse, error) {
	result := blobClientSetMetadataResponse{RawResponse: resp}
	if val := resp.Header.Get("ETag"); val != "" {
		result.ETag = &val
	}
	if val := resp.Header.Get("Last-Modified"); val != "" {
		lastModified, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientSetMetadataResponse{}, err
		}
		result.LastModified = &lastModified
	}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("x-ms-version-id"); val != "" {
		result.VersionID = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientSetMetadataResponse{}, err
		}
		result.Date = &date
	}
	if val := resp.Header.Get("x-ms-request-server-encrypted"); val != "" {
		isServerEncrypted, err := strconv.ParseBool(val)
		if err != nil {
			return blobClientSetMetadataResponse{}, err
		}
		result.IsServerEncrypted = &isServerEncrypted
	}
	if val := resp.Header.Get("x-ms-encryption-key-sha256"); val != "" {
		result.EncryptionKeySHA256 = &val
	}
	if val := resp.Header.Get("x-ms-encryption-scope"); val != "" {
		result.EncryptionScope = &val
	}
	return result, nil
}

// SetTags - The Set Tags operation enables users to set tags on a blob.
// If the operation fails it returns an *azcore.ResponseError type.
// blobClientSetTagsOptions - blobClientSetTagsOptions contains the optional parameters for the blobClient.SetTags method.
// ModifiedAccessConditions - ModifiedAccessConditions contains a group of parameters for the containerClient.Delete method.
// LeaseAccessConditions - LeaseAccessConditions contains a group of parameters for the containerClient.GetProperties method.
func (client *blobClient) SetTags(ctx context.Context, blobClientSetTagsOptions *blobClientSetTagsOptions, modifiedAccessConditions *ModifiedAccessConditions, leaseAccessConditions *LeaseAccessConditions) (blobClientSetTagsResponse, error) {
	req, err := client.setTagsCreateRequest(ctx, blobClientSetTagsOptions, modifiedAccessConditions, leaseAccessConditions)
	if err != nil {
		return blobClientSetTagsResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientSetTagsResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusNoContent) {
		return blobClientSetTagsResponse{}, runtime.NewResponseError(resp)
	}
	return client.setTagsHandleResponse(resp)
}

// setTagsCreateRequest creates the SetTags request.
func (client *blobClient) setTagsCreateRequest(ctx context.Context, blobClientSetTagsOptions *blobClientSetTagsOptions, modifiedAccessConditions *ModifiedAccessConditions, leaseAccessConditions *LeaseAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodPut, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("comp", "tags")
	if blobClientSetTagsOptions != nil && blobClientSetTagsOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientSetTagsOptions.Timeout), 10))
	}
	if blobClientSetTagsOptions != nil && blobClientSetTagsOptions.VersionID != nil {
		reqQP.Set("versionid", *blobClientSetTagsOptions.VersionID)
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientSetTagsOptions != nil && blobClientSetTagsOptions.TransactionalContentMD5 != nil {
		req.Raw().Header.Set("Content-MD5", base64.StdEncoding.EncodeToString(blobClientSetTagsOptions.TransactionalContentMD5))
	}
	if blobClientSetTagsOptions != nil && blobClientSetTagsOptions.TransactionalContentCRC64 != nil {
		req.Raw().Header.Set("x-ms-content-crc64", base64.StdEncoding.EncodeToString(blobClientSetTagsOptions.TransactionalContentCRC64))
	}
	if blobClientSetTagsOptions != nil && blobClientSetTagsOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientSetTagsOptions.RequestID)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfTags != nil {
		req.Raw().Header.Set("x-ms-if-tags", *modifiedAccessConditions.IfTags)
	}
	if leaseAccessConditions != nil && leaseAccessConditions.LeaseID != nil {
		req.Raw().Header.Set("x-ms-lease-id", *leaseAccessConditions.LeaseID)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	if blobClientSetTagsOptions != nil && blobClientSetTagsOptions.Tags != nil {
		return req, runtime.MarshalAsXML(req, *blobClientSetTagsOptions.Tags)
	}
	return req, nil
}

// setTagsHandleResponse handles the SetTags response.
func (client *blobClient) setTagsHandleResponse(resp *http.Response) (blobClientSetTagsResponse, error) {
	result := blobClientSetTagsResponse{RawResponse: resp}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientSetTagsResponse{}, err
		}
		result.Date = &date
	}
	return result, nil
}

// SetTier - The Set Tier operation sets the tier on a blob. The operation is allowed on a page blob in a premium storage
// account and on a block blob in a blob storage account (locally redundant storage only). A
// premium page blob's tier determines the allowed size, IOPS, and bandwidth of the blob. A block blob's tier determines Hot/Cool/Archive
// storage type. This operation does not update the blob's ETag.
// If the operation fails it returns an *azcore.ResponseError type.
// tier - Indicates the tier to be set on the blob.
// blobClientSetTierOptions - blobClientSetTierOptions contains the optional parameters for the blobClient.SetTier method.
// LeaseAccessConditions - LeaseAccessConditions contains a group of parameters for the containerClient.GetProperties method.
// ModifiedAccessConditions - ModifiedAccessConditions contains a group of parameters for the containerClient.Delete method.
func (client *blobClient) SetTier(ctx context.Context, tier AccessTier, blobClientSetTierOptions *blobClientSetTierOptions, leaseAccessConditions *LeaseAccessConditions, modifiedAccessConditions *ModifiedAccessConditions) (blobClientSetTierResponse, error) {
	req, err := client.setTierCreateRequest(ctx, tier, blobClientSetTierOptions, leaseAccessConditions, modifiedAccessConditions)
	if err != nil {
		return blobClientSetTierResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientSetTierResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusOK, http.StatusAccepted) {
		return blobClientSetTierResponse{}, runtime.NewResponseError(resp)
	}
	return client.setTierHandleResponse(resp)
}

// setTierCreateRequest creates the SetTier request.
func (client *blobClient) setTierCreateRequest(ctx context.Context, tier AccessTier, blobClientSetTierOptions *blobClientSetTierOptions, leaseAccessConditions *LeaseAccessConditions, modifiedAccessConditions *ModifiedAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodPut, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("comp", "tier")
	if blobClientSetTierOptions != nil && blobClientSetTierOptions.Snapshot != nil {
		reqQP.Set("snapshot", *blobClientSetTierOptions.Snapshot)
	}
	if blobClientSetTierOptions != nil && blobClientSetTierOptions.VersionID != nil {
		reqQP.Set("versionid", *blobClientSetTierOptions.VersionID)
	}
	if blobClientSetTierOptions != nil && blobClientSetTierOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientSetTierOptions.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	req.Raw().Header.Set("x-ms-access-tier", string(tier))
	if blobClientSetTierOptions != nil && blobClientSetTierOptions.RehydratePriority != nil {
		req.Raw().Header.Set("x-ms-rehydrate-priority", string(*blobClientSetTierOptions.RehydratePriority))
	}
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientSetTierOptions != nil && blobClientSetTierOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientSetTierOptions.RequestID)
	}
	if leaseAccessConditions != nil && leaseAccessConditions.LeaseID != nil {
		req.Raw().Header.Set("x-ms-lease-id", *leaseAccessConditions.LeaseID)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfTags != nil {
		req.Raw().Header.Set("x-ms-if-tags", *modifiedAccessConditions.IfTags)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// setTierHandleResponse handles the SetTier response.
func (client *blobClient) setTierHandleResponse(resp *http.Response) (blobClientSetTierResponse, error) {
	result := blobClientSetTierResponse{RawResponse: resp}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	return result, nil
}

// StartCopyFromURL - The Start Copy From URL operation copies a blob or an internet resource to a new blob.
// If the operation fails it returns an *azcore.ResponseError type.
// copySource - Specifies the name of the source page blob snapshot. This value is a URL of up to 2 KB in length that specifies
// a page blob snapshot. The value should be URL-encoded as it would appear in a request
// URI. The source blob must either be public or must be authenticated via a shared access signature.
// blobClientStartCopyFromURLOptions - blobClientStartCopyFromURLOptions contains the optional parameters for the blobClient.StartCopyFromURL
// method.
// SourceModifiedAccessConditions - SourceModifiedAccessConditions contains a group of parameters for the blobClient.StartCopyFromURL
// method.
// ModifiedAccessConditions - ModifiedAccessConditions contains a group of parameters for the containerClient.Delete method.
// LeaseAccessConditions - LeaseAccessConditions contains a group of parameters for the containerClient.GetProperties method.
func (client *blobClient) StartCopyFromURL(ctx context.Context, copySource string, blobClientStartCopyFromURLOptions *blobClientStartCopyFromURLOptions, sourceModifiedAccessConditions *SourceModifiedAccessConditions, modifiedAccessConditions *ModifiedAccessConditions, leaseAccessConditions *LeaseAccessConditions) (blobClientStartCopyFromURLResponse, error) {
	req, err := client.startCopyFromURLCreateRequest(ctx, copySource, blobClientStartCopyFromURLOptions, sourceModifiedAccessConditions, modifiedAccessConditions, leaseAccessConditions)
	if err != nil {
		return blobClientStartCopyFromURLResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientStartCopyFromURLResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusAccepted) {
		return blobClientStartCopyFromURLResponse{}, runtime.NewResponseError(resp)
	}
	return client.startCopyFromURLHandleResponse(resp)
}

// startCopyFromURLCreateRequest creates the StartCopyFromURL request.
func (client *blobClient) startCopyFromURLCreateRequest(ctx context.Context, copySource string, blobClientStartCopyFromURLOptions *blobClientStartCopyFromURLOptions, sourceModifiedAccessConditions *SourceModifiedAccessConditions, modifiedAccessConditions *ModifiedAccessConditions, leaseAccessConditions *LeaseAccessConditions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodPut, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	if blobClientStartCopyFromURLOptions != nil && blobClientStartCopyFromURLOptions.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*blobClientStartCopyFromURLOptions.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	if blobClientStartCopyFromURLOptions != nil && blobClientStartCopyFromURLOptions.Metadata != nil {
		for k, v := range blobClientStartCopyFromURLOptions.Metadata {
			req.Raw().Header.Set("x-ms-meta-"+k, v)
		}
	}
	if blobClientStartCopyFromURLOptions != nil && blobClientStartCopyFromURLOptions.Tier != nil {
		req.Raw().Header.Set("x-ms-access-tier", string(*blobClientStartCopyFromURLOptions.Tier))
	}
	if blobClientStartCopyFromURLOptions != nil && blobClientStartCopyFromURLOptions.RehydratePriority != nil {
		req.Raw().Header.Set("x-ms-rehydrate-priority", string(*blobClientStartCopyFromURLOptions.RehydratePriority))
	}
	if sourceModifiedAccessConditions != nil && sourceModifiedAccessConditions.SourceIfModifiedSince != nil {
		req.Raw().Header.Set("x-ms-source-if-modified-since", sourceModifiedAccessConditions.SourceIfModifiedSince.Format(time.RFC1123))
	}
	if sourceModifiedAccessConditions != nil && sourceModifiedAccessConditions.SourceIfUnmodifiedSince != nil {
		req.Raw().Header.Set("x-ms-source-if-unmodified-since", sourceModifiedAccessConditions.SourceIfUnmodifiedSince.Format(time.RFC1123))
	}
	if sourceModifiedAccessConditions != nil && sourceModifiedAccessConditions.SourceIfMatch != nil {
		req.Raw().Header.Set("x-ms-source-if-match", *sourceModifiedAccessConditions.SourceIfMatch)
	}
	if sourceModifiedAccessConditions != nil && sourceModifiedAccessConditions.SourceIfNoneMatch != nil {
		req.Raw().Header.Set("x-ms-source-if-none-match", *sourceModifiedAccessConditions.SourceIfNoneMatch)
	}
	if sourceModifiedAccessConditions != nil && sourceModifiedAccessConditions.SourceIfTags != nil {
		req.Raw().Header.Set("x-ms-source-if-tags", *sourceModifiedAccessConditions.SourceIfTags)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfModifiedSince != nil {
		req.Raw().Header.Set("If-Modified-Since", modifiedAccessConditions.IfModifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfUnmodifiedSince != nil {
		req.Raw().Header.Set("If-Unmodified-Since", modifiedAccessConditions.IfUnmodifiedSince.Format(time.RFC1123))
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfMatch != nil {
		req.Raw().Header.Set("If-Match", *modifiedAccessConditions.IfMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfNoneMatch != nil {
		req.Raw().Header.Set("If-None-Match", *modifiedAccessConditions.IfNoneMatch)
	}
	if modifiedAccessConditions != nil && modifiedAccessConditions.IfTags != nil {
		req.Raw().Header.Set("x-ms-if-tags", *modifiedAccessConditions.IfTags)
	}
	req.Raw().Header.Set("x-ms-copy-source", copySource)
	if leaseAccessConditions != nil && leaseAccessConditions.LeaseID != nil {
		req.Raw().Header.Set("x-ms-lease-id", *leaseAccessConditions.LeaseID)
	}
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if blobClientStartCopyFromURLOptions != nil && blobClientStartCopyFromURLOptions.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *blobClientStartCopyFromURLOptions.RequestID)
	}
	if blobClientStartCopyFromURLOptions != nil && blobClientStartCopyFromURLOptions.BlobTagsString != nil {
		req.Raw().Header.Set("x-ms-tags", *blobClientStartCopyFromURLOptions.BlobTagsString)
	}
	if blobClientStartCopyFromURLOptions != nil && blobClientStartCopyFromURLOptions.SealBlob != nil {
		req.Raw().Header.Set("x-ms-seal-blob", strconv.FormatBool(*blobClientStartCopyFromURLOptions.SealBlob))
	}
	if blobClientStartCopyFromURLOptions != nil && blobClientStartCopyFromURLOptions.ImmutabilityPolicyExpiry != nil {
		req.Raw().Header.Set("x-ms-immutability-policy-until-date", blobClientStartCopyFromURLOptions.ImmutabilityPolicyExpiry.Format(time.RFC1123))
	}
	if blobClientStartCopyFromURLOptions != nil && blobClientStartCopyFromURLOptions.ImmutabilityPolicyMode != nil {
		req.Raw().Header.Set("x-ms-immutability-policy-mode", string(*blobClientStartCopyFromURLOptions.ImmutabilityPolicyMode))
	}
	if blobClientStartCopyFromURLOptions != nil && blobClientStartCopyFromURLOptions.LegalHold != nil {
		req.Raw().Header.Set("x-ms-legal-hold", strconv.FormatBool(*blobClientStartCopyFromURLOptions.LegalHold))
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// startCopyFromURLHandleResponse handles the StartCopyFromURL response.
func (client *blobClient) startCopyFromURLHandleResponse(resp *http.Response) (blobClientStartCopyFromURLResponse, error) {
	result := blobClientStartCopyFromURLResponse{RawResponse: resp}
	if val := resp.Header.Get("ETag"); val != "" {
		result.ETag = &val
	}
	if val := resp.Header.Get("Last-Modified"); val != "" {
		lastModified, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientStartCopyFromURLResponse{}, err
		}
		result.LastModified = &lastModified
	}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("x-ms-version-id"); val != "" {
		result.VersionID = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientStartCopyFromURLResponse{}, err
		}
		result.Date = &date
	}
	if val := resp.Header.Get("x-ms-copy-id"); val != "" {
		result.CopyID = &val
	}
	if val := resp.Header.Get("x-ms-copy-status"); val != "" {
		result.CopyStatus = (*CopyStatusType)(&val)
	}
	return result, nil
}

// Undelete - Undelete a blob that was previously soft deleted
// If the operation fails it returns an *azcore.ResponseError type.
// options - blobClientUndeleteOptions contains the optional parameters for the blobClient.Undelete method.
func (client *blobClient) Undelete(ctx context.Context, options *blobClientUndeleteOptions) (blobClientUndeleteResponse, error) {
	req, err := client.undeleteCreateRequest(ctx, options)
	if err != nil {
		return blobClientUndeleteResponse{}, err
	}
	resp, err := client.pl.Do(req)
	if err != nil {
		return blobClientUndeleteResponse{}, err
	}
	if !runtime.HasStatusCode(resp, http.StatusOK) {
		return blobClientUndeleteResponse{}, runtime.NewResponseError(resp)
	}
	return client.undeleteHandleResponse(resp)
}

// undeleteCreateRequest creates the Undelete request.
func (client *blobClient) undeleteCreateRequest(ctx context.Context, options *blobClientUndeleteOptions) (*policy.Request, error) {
	req, err := runtime.NewRequest(ctx, http.MethodPut, client.endpoint)
	if err != nil {
		return nil, err
	}
	reqQP := req.Raw().URL.Query()
	reqQP.Set("comp", "undelete")
	if options != nil && options.Timeout != nil {
		reqQP.Set("timeout", strconv.FormatInt(int64(*options.Timeout), 10))
	}
	req.Raw().URL.RawQuery = reqQP.Encode()
	req.Raw().Header.Set("x-ms-version", "2020-10-02")
	if options != nil && options.RequestID != nil {
		req.Raw().Header.Set("x-ms-client-request-id", *options.RequestID)
	}
	req.Raw().Header.Set("Accept", "application/xml")
	return req, nil
}

// undeleteHandleResponse handles the Undelete response.
func (client *blobClient) undeleteHandleResponse(resp *http.Response) (blobClientUndeleteResponse, error) {
	result := blobClientUndeleteResponse{RawResponse: resp}
	if val := resp.Header.Get("x-ms-client-request-id"); val != "" {
		result.ClientRequestID = &val
	}
	if val := resp.Header.Get("x-ms-request-id"); val != "" {
		result.RequestID = &val
	}
	if val := resp.Header.Get("x-ms-version"); val != "" {
		result.Version = &val
	}
	if val := resp.Header.Get("Date"); val != "" {
		date, err := time.Parse(time.RFC1123, val)
		if err != nil {
			return blobClientUndeleteResponse{}, err
		}
		result.Date = &date
	}
	return result, nil
}
