static char rcsid[] = "$Id: transcript.c 224642 2021-08-25 22:03:29Z twu $";
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "transcript.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>		/* For qsort */
#include <ctype.h>		/* For isalpha */

#include "assert.h"
#include "mem.h"
#include "sense.h"
#include "exon.h"


/* Transcript_new */
/* Also need to turn on DEBUG1 in exon.c */
#ifdef DEBUG0
#define debug0(x) x
#else
#define debug0(x)
#endif

/* Printing */
#ifdef DEBUG1
#define debug1(x) x
#else
#define debug1(x)
#endif

/* Concordance */
#ifdef DEBUG2
#define debug2(x) x
#else
#define debug2(x)
#endif

/* Transcript_trim */
#ifdef DEBUG5
#define debug5(x) x
#else
#define debug5(x)
#endif

/* Transcript exons */
#ifdef DEBUG6
#define debug6(x) x
#else
#define debug6(x)
#endif


static int pairmax_transcriptome;
static int expected_pairlength;
static Outputtype_T output_type;
static Transcriptome_T transcriptome;

/* For debugging */
static Univ_IIT_T transcript_iit;


void
Transcript_setup (int pairmax_transcriptome_in, int expected_pairlength_in,
		  Outputtype_T output_type_in, Transcriptome_T transcriptome_in,
		  Univ_IIT_T transcript_iit_in) {

  pairmax_transcriptome = pairmax_transcriptome_in;
  expected_pairlength = expected_pairlength_in;
  output_type = output_type_in;
  transcriptome = transcriptome_in;
  
  /* For debugging */
  transcript_iit = transcript_iit_in;

  return;
}


/* Exons are always stored in transcript order: exon1 to exonN */
#define T Transcript_T
struct T {
  int num;	      /* Index in transcriptome */
  int nexons;	      /* nexons in transcript */
  int genestrand;	/* gplusp: > 0 if on plus strand and < 0 if on minus strand */
  int start;	      /* Starting coordinate on transcript, corresponding to querystart.  0-based */
  int end; /* Ending coordinate, corresponding to queryend.  If transcript_plusp is false, trend > trstart */
  int transcript_length;

  int trim_low;
  int trim_high;

  List_T exons;
  Bitvector_T cryptic_introns;
  Bitvector_T retained_introns;
  Bitvector_T spliced_introns;
  bool utr5p;
  bool utr3p;

  Transcript_velocity_T velocity;
  Transcript_velocity_T pair_velocity;
  int insertlength;		/* For concordant transcripts */
};



/* Modified from Stage3end_remap_transcriptome */
static List_T
compute_exons (Bitvector_T spliced_introns, bool *utr5p, bool *utr3p,
	       int trnum, int transcript_genestrand,
	       int trlow, int trhigh, int trlength,
	       int trim_low, int trim_high, Listpool_T listpool) {
  List_T exons = NULL;
  Chrpos_T *exonstarts_array;
  int *exonbounds;
  int start_exoni, end_exoni, exoni, nexons;
  char firstchar, lastchar;
#ifdef DEBUG6
  char *label;
  bool allocp;
#endif

#ifdef DEBUG6
  label = Univ_IIT_label(transcript_iit,trnum,&allocp);
  printf("%s\n",label);
  if (allocp) {
    FREE(label);
  }
#endif

  nexons = Transcriptome_exons(&exonbounds,&exonstarts_array,transcriptome,trnum);
#ifdef DEBUG6
  for (exoni = 0; exoni < nexons; exoni++) {
    printf("%d\n",exonbounds[exoni]);
  }
#endif


  start_exoni = 0;
  while (start_exoni < nexons && exonbounds[start_exoni] <= trlow /* same as < trlow + 1*/) {
    start_exoni++;
  }
    
  end_exoni = start_exoni;
  while (end_exoni < nexons && exonbounds[end_exoni] < trhigh) {
    end_exoni++;
  }
  if (end_exoni == nexons) {
    end_exoni--;
  }

  debug6(printf("genestrand %d.  trlow %d, trhigh %d\n",transcript_genestrand,trlow,trhigh));
  debug6(printf("start_exon %d, end_exon %d\n",start_exoni,end_exoni));

  *utr5p = *utr3p = false;

  if (start_exoni == end_exoni) {
    /* Single exon */
    if (start_exoni == 0) {
      /* Extension is possible from Transcript_trim */
      if (trlow < 0) {
	firstchar = 'u'; *utr5p = true; abort();
      } else {
	firstchar = '.';
      }
    } else if (trlow != exonbounds[start_exoni - 1]) {
      firstchar = '.';
    } else if (transcript_genestrand > 0 && trim_low > 0) {
      firstchar = 's';
      Bitvector_set(spliced_introns,/*introni*/start_exoni - 1);
    } else if (transcript_genestrand < 0 && trim_high > 0) {
      firstchar = 's';
      Bitvector_set(spliced_introns,/*introni*/start_exoni - 1);
    } else {	    
      firstchar = '.';
    }
      
    if (end_exoni == nexons - 1) {
      /* Extension is possible from Transcript_trim */
      if (trhigh > trlength) {
	lastchar = 'u'; *utr3p = true; abort();
      } else {
	lastchar = '.';
      }
    } else if (trhigh != exonbounds[end_exoni]) {
      lastchar = '.';
    } else if (transcript_genestrand > 0 && trim_high > 0) {
      lastchar = 's';
      Bitvector_set(spliced_introns,/*introni*/end_exoni);
    } else if (transcript_genestrand < 0 && trim_low > 0) {
      lastchar = 's';
      Bitvector_set(spliced_introns,/*introni*/end_exoni);
    } else {
      lastchar = '.';
    }
      
    exons = Listpool_push(exons,listpool,Exon_new(firstchar,start_exoni,lastchar));

  } else {
    /* First exon */
    if (start_exoni == 0) {
      /* Extension is possible from Transcript_trim */
      if (trlow < 0) {
	firstchar = 'u'; *utr5p = true; abort();
      } else {
	firstchar = '.';
      }
    } else if (trlow != exonbounds[start_exoni - 1]) {
      firstchar = '.';
    } else if (transcript_genestrand > 0 && trim_low > 0) {
      firstchar = 's';
      Bitvector_set(spliced_introns,/*introni*/start_exoni - 1);
    } else if (transcript_genestrand < 0 && trim_high > 0) {
      firstchar = 's';
      Bitvector_set(spliced_introns,/*introni*/start_exoni - 1);
    } else {	    
      firstchar = '.';
    }
    exons = Listpool_push(exons,listpool,Exon_new(firstchar,start_exoni,'s'));

    for (exoni = start_exoni + 1; exoni < end_exoni; exoni++) {
      /* Middle exons */
      Bitvector_set(spliced_introns,/*introni*/exoni - 1);
      exons = Listpool_push(exons,listpool,Exon_new('s',exoni,'s'));
    }

    /* Last exon */
    Bitvector_set(spliced_introns,/*introni*/exoni - 1);
    if (end_exoni == nexons - 1) {
      /* Extension is possible from Transcript_trim */
      if (trhigh > trlength) {
	lastchar = 'u'; *utr3p = true; abort();
      } else {
	lastchar = '.';
      }
    } else if (trhigh != exonbounds[end_exoni]) {
      lastchar = '.';
    } else if (transcript_genestrand > 0 && trim_high > 0) {
      lastchar = 's';
      Bitvector_set(spliced_introns,/*introni*/end_exoni);
    } else if (transcript_genestrand < 0 && trim_low > 0) {
      lastchar = 's';
      Bitvector_set(spliced_introns,/*introni*/end_exoni);
    } else {
      lastchar = '.';
    }

    exons = Listpool_push(exons,listpool,Exon_new('s',end_exoni,lastchar));
  }

  debug6(printf("\n"));
  return List_reverse(exons);
}  


#if defined(DEBUG0) || defined(DEBUG1)
static void
introns_print_stdout (T this) {
  bool firstp = true;

  if (Bitvector_null_p(this->cryptic_introns) == false) {
    firstp = false;
    putchar('X');
    Bitvector_print_file(stdout,this->cryptic_introns,/*sep*/'-');
  }

  if (this->utr5p == true && this->utr3p == true) {
    if (firstp == true) {
      firstp = false;
    } else {
      putchar('/');
    }
    printf("U5-3");		/* Cannot use ',', which separates transcripts */

  } else if (this->utr5p == true) {
    if (firstp == true) {
      firstp = false;
    } else {
      putchar('/');
    }
    printf("U5");

  } else if (this->utr3p == true) {
    if (firstp == true) {
      firstp = false;
    } else {
      putchar('/');
    }
    printf("U3");
  }

  if (Bitvector_null_p(this->retained_introns) == false) {
    if (firstp == true) {
      firstp = false;
    } else {
      putchar('/');
    }
    putchar('I');
    Bitvector_print_file(stdout,this->retained_introns,/*sep*/'-');
  }

  if (firstp == true) {
    if (Bitvector_null_p(this->spliced_introns) == true) {
      putchar('A');
    } else {
      putchar('S');
      Bitvector_print_file(stdout,this->spliced_introns,/*sep*/'-');
    }
  }

  return;
}



static int
print_one_stdout (T this, Univ_IIT_T transcript_iit, bool invertp) {
  char *label;
  bool allocp;

  label = Univ_IIT_label(transcript_iit,this->num,&allocp);

  if (this->genestrand > 0) {
    if (this->start < this->end) {
      printf("+%s:%d..%d:",label,
	     this->start < 0 ? this->start : this->start + 1,
	     this->end);
    } else {
      printf("+%s:%d..%d:",label,
	     this->end < 0 ? this->end : this->end + 1,
	     this->start);
    }
    introns_print_stdout(this);
    Exon_list_print_fwd_stdout(this->exons);

  } else {
    if (this->start < this->end) {
      printf("-%s:%d..%d:",label,
	     this->end,
	     this->start < 0 ? this->start : this->start + 1);
    } else {
      printf("-%s:%d..%d:",label,
	     this->start,
	     this->end < 0 ? this->end : this->end + 1);
    }
    introns_print_stdout(this);
    Exon_list_print_rev_stdout(this->exons);
  }

  if (allocp) {
    FREE(label);
  }

  return this->genestrand;
}

void
Transcript_dump_list_stdout (List_T transcripts) {
  T transcript;
  List_T p;

  if (transcripts != NULL) {
    transcript = List_head(transcripts);
    print_one_stdout(transcript,transcript_iit,/*invertp*/false);

    for (p = List_next(transcripts); p != NULL; p = List_next(p)) {
      transcript = List_head(p);
      putchar(',');
      print_one_stdout(transcript,transcript_iit,/*invertp*/false);
    }
  }
  printf("\n");

  return;
}
#endif



int
Transcript_num (T this) {
  return this->num;
}

int
Transcript_start (T this) {
  return this->start;
}

int
Transcript_end (T this) {
  return this->end;
}

int
Transcript_length (T this) {
  return this->transcript_length;
}

int
Transcript_genestrand (T this) {
  return this->genestrand;
}

Transcript_velocity_T
Transcript_velocity (T this) {
  return this->velocity;
}

void
Transcript_free (T *old) {
  debug0(printf("Freeing %p\n",*old));
  Exon_list_gc(&(*old)->exons);
  Bitvector_free(&(*old)->cryptic_introns);
  Bitvector_free(&(*old)->retained_introns);
  Bitvector_free(&(*old)->spliced_introns);
  FREE(*old);

  return;
}

void
Transcript_gc (List_T *list) {
  List_T p;
  T this;

  for (p = *list; p != NULL; p = List_next(p)) {
    this = (T) List_head(p);
    Transcript_free(&this);
  }
  /* List_free(&(*list)); -- allocated by Listpool_push */

  return;
}


/* querystart and queryend are for the read */
T
Transcript_new_need_exons (int num, int transcript_genestrand, int nexons,
			   int trstart, int trend, int trlength,
			   int trim_low, int trim_high, Listpool_T listpool) {
#ifdef DEBUG0
  char *label;
  bool allocp;
#endif

  T new = (T) MALLOC(sizeof(*new));

#ifdef DEBUG0
  label = Univ_IIT_label(transcript_iit,num,&allocp);
  printf("%p: Making transcript %s with trnum %d, genestrand %d, trstart %d, trend %d, trlength %d.  Need exons\n",
	 new,label,num,transcript_genestrand,trstart,trend,trlength);
  if (allocp) {
    FREE(label);
  }
#endif

  new->num = num;
  new->genestrand = transcript_genestrand;
  new->nexons = nexons;

  new->start = trstart;
  new->end = trend;
  new->transcript_length = trlength;

  new->trim_low = trim_low;
  new->trim_high = trim_high;

  /* Cannot have any cryptic or retained introns from alignment to a transcript */
  new->cryptic_introns = (Bitvector_T) NULL;
  new->retained_introns = (Bitvector_T) NULL;
  new->spliced_introns = Bitvector_new(/*nintrons*/nexons - 1);

  if (trstart < trend) {
    new->exons = compute_exons(new->spliced_introns,&new->utr5p,&new->utr3p,
			       num,transcript_genestrand,
			       /*trlow*/trstart,/*trhigh*/trend,trlength,
			       trim_low,trim_high,listpool);

  } else {
    new->exons = compute_exons(new->spliced_introns,&new->utr5p,&new->utr3p,
			       num,transcript_genestrand,
			       /*trlow*/trend,/*trhigh*/trstart,trlength,
			       trim_low,trim_high,listpool);
  }

  if (new->utr5p == true || new->utr3p == true) {
    new->velocity = new->pair_velocity = V_UTR;
  } else if (Bitvector_null_p(new->spliced_introns) == false) {
    new->velocity = new->pair_velocity = V_SPLICED;
  } else {
    new->velocity = new->pair_velocity = V_AMBIGUOUS;
  }
  new->insertlength = 0;

#ifdef DEBUG0
  print_one_stdout(new,transcript_iit,/*invertp*/false);
  printf("\n");
#endif

  return new;
}


T
Transcript_new_with_exons (int num, int transcript_genestrand, int nexons,
			    int trstart, int trend, int trlength,
			    int querystart, int queryend, int querylength,
			    List_T exons, Bitvector_T cryptic_introns,
			    Bitvector_T retained_introns, Bitvector_T spliced_introns,
			    bool utr5p, bool utr3p) {
  T new = (T) MALLOC(sizeof(*new));
#ifdef DEBUG0
  char *label;
  bool allocp;
#endif
#ifdef DEBUG6
  List_T p;
#endif


#ifdef DEBUG0
  label = Univ_IIT_label(transcript_iit,num,&allocp);
  printf("%p: Making transcript %s with trnum %d, transcript_genestrand %d, start %d, end %d, trlength %d.  With exons\n",
	 new,label,num,transcript_genestrand,trstart,trend,trlength);
  if (allocp) {
    FREE(label);
  }
#endif
  
#ifdef DEBUG6
  Exon_list_print_fwd_stdout(exons);
  printf("\n");
#endif


  new->num = num;
  new->genestrand = transcript_genestrand;
  new->nexons = nexons;

  new->start = trstart;
  new->end = trend;
  new->transcript_length = trlength;

  if (transcript_genestrand > 0) {
    if (trstart < trend) {
      new->trim_low = querystart;
      new->trim_high = querylength - queryend;
    } else {
      new->trim_low = querylength - queryend;
      new->trim_high = querystart;
    }

  } else {
    if (trstart < trend) {
      new->trim_low = querylength - queryend;
      new->trim_high = querystart;
    } else {
      new->trim_low = querystart;
      new->trim_high = querylength - queryend;
    }
  }
  
  new->exons = exons;

  new->cryptic_introns = cryptic_introns;
  new->retained_introns = retained_introns;
  new->spliced_introns = spliced_introns;
  new->utr5p = utr5p;
  new->utr3p = utr3p;

  if (Bitvector_null_p(cryptic_introns) == false) {
    new->velocity = new->pair_velocity = V_CRYPTIC;
  } else if (utr5p == true || utr3p == true) {
    new->velocity = new->pair_velocity = V_UTR;
  } else if (Bitvector_null_p(retained_introns) == false) {
    new->velocity = new->pair_velocity = V_RETAINED;
  } else if (Bitvector_null_p(spliced_introns) == false) {
    new->velocity = new->pair_velocity = V_SPLICED;
  } else {
    new->velocity = new->pair_velocity = V_AMBIGUOUS;
  }
    
  new->insertlength = 0;

#ifdef DEBUG0
  print_one_stdout(new,transcript_iit,/*invertp*/false);
  printf("\n");
#endif

  return new;
}


T
Transcript_copy (T old, Listpool_T listpool) {
  T new = (T) MALLOC(sizeof(*new));

  debug0(printf("Copying %p from %p\n",new,old));

  new->num = old->num;
  new->genestrand = old->genestrand;
  new->nexons = old->nexons;
  
  new->start = old->start;
  new->end = old->end;
  new->transcript_length = old->transcript_length;

  new->trim_low = old->trim_low;
  new->trim_high = old->trim_high;

  new->exons = Exon_list_copy(old->exons,listpool);

  new->cryptic_introns = Bitvector_copy(old->cryptic_introns);
  new->retained_introns = Bitvector_copy(old->retained_introns);
  new->spliced_introns = Bitvector_copy(old->spliced_introns);

  new->utr5p = old->utr5p;
  new->utr3p = old->utr3p;
  
  new->velocity = old->velocity;
  new->pair_velocity = old->pair_velocity;
  new->insertlength = old->insertlength;

  return new;
}

List_T
Transcript_copy_list (List_T old, Listpool_T listpool) {
  List_T new = NULL, p;
  T this;

  for (p = old; p != NULL; p = List_next(p)) {
    this = (T) List_head(p);
    new = Listpool_push(new,listpool,(void *) Transcript_copy(this,listpool));
  }
  return List_reverse(new);
}

/* Returns true if spliced */
bool
Transcript_trim (T this, int trim_low_adj, int trim_high_adj, Listpool_T listpool) {
#ifdef DEBUG5
  char *label;
  bool allocp;
#endif

#ifdef DEBUG5
  label = Univ_IIT_label(transcript_iit,this->num,&allocp);
  printf("Transcript_trim %s: trim_low_adj %d, trim_high_adj %d\n",
	 label,trim_low_adj,trim_high_adj);
  if (allocp) {
    FREE(label);
  }
#endif

  if (trim_low_adj == 0 && trim_high_adj == 0) {
    /* Skip making changes */
  } else {
    Exon_list_gc(&this->exons);
    Bitvector_clear_all(this->spliced_introns);

    this->trim_low += trim_low_adj;
    this->trim_high += trim_high_adj;

    if (this->genestrand > 0) {
      if (this->start < this->end) {
	if (trim_low_adj != 0) {
	  this->start += trim_low_adj;
	}
	if (trim_high_adj != 0) {
	  this->end -= trim_high_adj;
	}
	this->exons = compute_exons(this->spliced_introns,&this->utr5p,&this->utr3p,
				      this->num,this->genestrand,
				      /*trlow*/this->start,/*trhigh*/this->end,this->transcript_length,
				      this->trim_low,this->trim_high,listpool);

      } else {
	if (trim_low_adj != 0) {
	  this->end += trim_low_adj;
	}
	if (trim_high_adj != 0) {
	  this->start -= trim_high_adj;
	}
	this->exons = compute_exons(this->spliced_introns,&this->utr5p,&this->utr3p,
				      this->num,this->genestrand,
				      /*trlow*/this->end,/*trhigh*/this->start,this->transcript_length,
				      this->trim_low,this->trim_high,listpool);
      }

    } else {
      if (this->start < this->end) {
	if (trim_low_adj != 0) {
	  this->end -= trim_low_adj;
	}
	if (trim_high_adj != 0) {
	  this->start += trim_high_adj;
	}
	this->exons = compute_exons(this->spliced_introns,&this->utr5p,&this->utr3p,
				      this->num,this->genestrand,
				      /*trlow*/this->start,/*trhigh*/this->end,this->transcript_length,
				      this->trim_low,this->trim_high,listpool);

      } else {
	if (trim_low_adj != 0) {
	  this->start -= trim_low_adj;
	}
	if (trim_high_adj != 0) {
	  this->end += trim_high_adj;
	}
	this->exons = compute_exons(this->spliced_introns,&this->utr5p,&this->utr3p,
				      this->num,this->genestrand,
				      /*trlow*/this->end,/*trhigh*/this->start,this->transcript_length,
				      this->trim_low,this->trim_high,listpool);
      }
    }

    if (this->utr5p == true || this->utr3p == true) {
      this->velocity = this->pair_velocity = V_UTR;
    } else if (Bitvector_null_p(this->spliced_introns) == false) {
      this->velocity = this->pair_velocity = V_SPLICED;
    } else {
      this->velocity = this->pair_velocity = V_AMBIGUOUS;
    }
  }

  if (this->velocity == V_SPLICED || this->velocity == V_AMBIGUOUS) {
    return true;
  } else {
    return false;
  }
}


bool
Transcript_extend_utr5p (List_T transcripts) {
  bool extend5p = true;
  List_T p;
  T this;

  for (p = transcripts; p != NULL; p = List_next(p)) {
    this = (T) List_head(p);
    if (this->start < this->end) {
      if (this->start != 0) {
	extend5p = false;
      }
    } else {
      if (this->end != 0) {
	extend5p = false;
      }
    }
  }

  return extend5p;
}


bool
Transcript_extend_utr3p (List_T transcripts) {
  bool extend3p = true;
  List_T p;
  T this;

  for (p = transcripts; p != NULL; p = List_next(p)) {
    this = (T) List_head(p);
    if (this->start < this->end) {
      if (this->end != this->transcript_length) {
	extend3p = false;
      }
    } else {
      if (this->start != this->transcript_length) {
	extend3p = false;
      }
    }
  }

  return extend3p;
}



#if 0
static bool
Transcript_equal (T x, T y) {
  if (x->num == y->num &&
      x->start == y->start &&
      x->end == y->end) {
    return true;
  } else {
    return false;
  }
}
#endif

bool
Transcript_in_list_p (T x, List_T list) {
  List_T p;
  T y;

  for (p = list; p != NULL; p = List_next(p)) {
    y = (T) List_head(p);
    if (x->num == y->num &&
	x->start == y->start &&
	x->end == y->end) {
      return true;
    }
  }
  return false;
}


#ifdef DEBUG1
void
Transcript_print_nums (List_T list) {
  List_T p;
  T this;

  printf(" Trnums:");
  for (p = list; p != NULL; p = List_next(p)) {
    this = (T) List_head(p);
    printf(" %d:%d",this->num,this->start);
  }
  return;
}
#endif

#ifdef DEBUG1
void
Transcript_print_list_debug (List_T list) {
  List_T p;
  T this;

  printf(" Trnums:");
  for (p = list; p != NULL; p = List_next(p)) {
    this = (T) List_head(p);
    printf(" %d",this->num);
  }
  printf("\n");

  printf(" Trstarts:");
  for (p = list; p != NULL; p = List_next(p)) {
    this = (T) List_head(p);
    printf(" %d",this->start);
  }
  printf("\n");

  printf(" Trends:");
  for (p = list; p != NULL; p = List_next(p)) {
    this = (T) List_head(p);
    printf(" %d",this->end);
  }
  printf("\n");

  return;
}
#endif

#if 0
int
Transcript_list_sensedir (List_T transcripts) {
  int sensedir = SENSE_NULL;
  bool tplusp, conflictp = false;
  List_T p;
  T transcript;
  
  for (p = transcripts; p != NULL; p = List_next(p)) {
    transcript = (T) List_head(p);
    tplusp = transcript->start < transcript->end ? true : false;

    if (tplusp == true) {
      if (sensedir == SENSE_NULL) {
	sensedir = SENSE_FORWARD;
      } else if (sensedir == SENSE_ANTI) {
	conflictp = true;
      }

    } else {
      if (sensedir == SENSE_NULL) {
	sensedir = SENSE_ANTI;
      } else if (sensedir == SENSE_FORWARD) {
	conflictp = true;
      }
    }
  }

  if (conflictp == true) {
    return SENSE_NULL;
  } else {
    return sensedir;
  }
}
#endif


static int
Transcript_cmp (const void *a, const void *b) {
  T x = * (T *) a;
  T y = * (T *) b;

  if (x->num < y->num) {
    return -1;
  } else if (y->num < x->num) {
    return +1;
  } else if (x->start < y->start) {
    return -1;
  } else if (y->start < x->start) {
    return +1;
  } else if (x->end < y->end) {
    return -1;
  } else if (y->end < x->end) {
    return +1;
  } else {
    return 0;
  }
}


static Transcript_velocity_T
compute_pair_velocity (Transcript_velocity_T velocity5, Transcript_velocity_T velocity3) {
  if (velocity5 == V_CRYPTIC || velocity3 == V_CRYPTIC) {
    return V_CRYPTIC;

  } else if (velocity5 == V_UTR || velocity3 == V_UTR) {
    return V_UTR;

  } else if (velocity5 == V_RETAINED || velocity3 == V_RETAINED) {
    return V_RETAINED;

  } else if (velocity5 == V_SPLICED || velocity3 == V_SPLICED) {
    return V_SPLICED;

  } else {
    /* Both must be V_AMBIGUOUS */
    return V_AMBIGUOUS;
  }
}


static void
merge_pair_introns (T dest, T source) {
  dest->cryptic_introns = Bitvector_union(dest->cryptic_introns,source->cryptic_introns,/*copyp*/false);
  dest->retained_introns = Bitvector_union(dest->retained_introns,source->retained_introns,/*copyp*/false);
  dest->spliced_introns = Bitvector_union(dest->spliced_introns,source->spliced_introns,/*copyp*/false);
  dest->utr5p |= source->utr5p;
  dest->utr3p |= source->utr3p;
  return;
}

static void
add_between_introns (T this, List_T read1_exons, List_T read2_exons,
		     bool plusp, int transcript_genestrand) {
  Exon_T start_exon, end_exon;
  unsigned int start_exoni, end_exoni, introni;

  if (plusp == true) {
    if (transcript_genestrand > 0) {
      start_exon = (Exon_T) List_last_value(read1_exons,NULL);
      end_exon = (Exon_T) List_head(read2_exons);
    } else {
      start_exon = (Exon_T) List_last_value(read2_exons,NULL);
      end_exon = (Exon_T) List_head(read1_exons);
    }
  } else {
    if (transcript_genestrand > 0) {
      start_exon = (Exon_T) List_last_value(read2_exons,NULL);
      end_exon = (Exon_T) List_head(read1_exons);
    } else {
      start_exon = (Exon_T) List_last_value(read1_exons,NULL);
      end_exon = (Exon_T) List_head(read2_exons);
    }
  }

  if (start_exon->lastchar == '.' && end_exon->firstchar == '.') {
    start_exoni = start_exon->exoni;
    end_exoni = end_exon->exoni;
    if (start_exoni < end_exoni) {
      if (this->spliced_introns == NULL) {
	this->spliced_introns = Bitvector_new(/*nintrons*/this->nexons - 1);
      }
      for (introni = start_exoni; introni < end_exoni; introni++) {
	Bitvector_set(this->spliced_introns,introni);
      }
    }
  }

  return;
}



#if 0
bool
Transcript_check_ascending (List_T transcripts) {
  List_T p, q;

  if ((p = transcripts) == NULL) {
    return true;
  } else if ((q = List_next(p)) == NULL) {
    return true;
  } else {
    while (q != NULL) {
      transcript1 = List_head(p);
      transcript2 = List_head(q);
      if (transcript1->num > transcript2->num) {
	abort();
      } else {
	p = List_next(p);
	q = List_next(q);
      }
    }
  }

  return true;
}
#endif



/* transcripts are for the first read and then the second read */
List_T
Transcript_concordant_consistent (List_T transcripts5, List_T transcripts3,
				  int querylength5, int querylength3,
				  Listpool_T listpool, bool plusp, bool first_read_p) {
  List_T concordant = NULL;
  
  T *array5, *array3, transcript5, transcript3;
  int ntranscripts5, ntranscripts3;

  int i, j, k, l;
  int trnum5, trnum3;
  int insertlength;
  Transcript_velocity_T pair_velocity;

#ifdef CHECK_ASSERTIONS
  int a;
#endif  

#ifdef DEBUG2
  List_T p;
  T transcript;
  char *label;
  bool allocp;
#endif


  debug2(printf("Entered Transcript_concordant_consistent with %d transcripts5, %d transcripts3, first_read_p %d\n",
		List_length(transcripts5),List_length(transcripts3),first_read_p));

#ifdef DEBUG2
  for (p = transcripts5; p != NULL; p = List_next(p)) {
    transcript = (T) List_head(p);
    label = Univ_IIT_label(transcript_iit,transcript->num,&allocp);
    printf("5' Transcript %s (%p): %d..%d, velocity %d: ",
	   label,transcript,transcript->start,transcript->end,transcript->velocity);
    Exon_list_print_fwd_stdout(transcript->exons);
    printf("\n");
  }
    
  for (p = transcripts3; p != NULL; p = List_next(p)) {
    transcript = (T) List_head(p);
    label = Univ_IIT_label(transcript_iit,transcript->num,&allocp);
    printf("3' Transcript %s (%p): %d..%d, velocity %d: ",
	   label,transcript,transcript->start,transcript->end,transcript->velocity);
    Exon_list_print_fwd_stdout(transcript->exons);
    printf("\n");
  }
#endif


  if ((ntranscripts5 = List_length(transcripts5)) == 0 ||
      (ntranscripts3 = List_length(transcripts3)) == 0) {
    return (List_T) NULL;

  } else {
    array5 = (T *) List_to_array(transcripts5,NULL);
    array3 = (T *) List_to_array(transcripts3,NULL);
    qsort(array5,ntranscripts5,sizeof(T),Transcript_cmp);
    qsort(array3,ntranscripts3,sizeof(T),Transcript_cmp);
  }


  i = k = 0;
  while (i < ntranscripts5 && k < ntranscripts3) {
    trnum5 = array5[i]->num;
    j = i+1;
    while (j < ntranscripts5 && array5[j]->num == trnum5) {
      j++;
    }
      
    trnum3 = array3[k]->num;
    l = k+1;
    while (l < ntranscripts3 && array3[l]->num == trnum3) {
      l++;
    }
      
    if (trnum5 < trnum3) {
      i = j;
	
    } else if (trnum3 < trnum5) {
      k = l;
	
    } else {
      /* Previously, we could have a mixture of transcript endpoints
	 from transfer_transcripts, but now, Transcript_trim should
	 make them all equivalent */
#ifdef CHECK_ASSERTIONS
      for (a = i+1; a < j; a++) {
	assert(array5[a]->start == array5[a-1]->start);
	assert(array5[a]->end == array5[a-1]->end);
      }
      for (a = k+1; a < l; a++) {
	assert(array3[a]->start == array3[a-1]->start);
	assert(array3[a]->end == array3[a-1]->end);
      }
#endif

      transcript5 = array5[i];
      transcript3 = array3[k];
	  
      if (transcript5->start < transcript5->end) {
	/* Plus on transcript */
	debug2(printf("(5) Checking for concordance on trnum %d: start5 %d, end3 %d\n",
		      trnum5,transcript5->start,transcript3->end));
	if (transcript5->start < transcript3->end && transcript3->end <= transcript5->start + pairmax_transcriptome) {
	  pair_velocity = compute_pair_velocity(transcript5->velocity,transcript3->velocity);
	  insertlength = transcript3->start - transcript5->end + querylength5 + querylength3;
	  if (first_read_p == true) {
	    transcript5->pair_velocity = pair_velocity;
	    transcript5->insertlength = insertlength;
	    merge_pair_introns(transcript5,transcript3);
	    add_between_introns(transcript5,transcript5->exons,transcript3->exons,plusp,transcript5->genestrand);
	    concordant = Listpool_push(concordant,listpool,(void *) transcript5);
	  } else {
	    transcript3->pair_velocity = pair_velocity;
	    transcript3->insertlength = insertlength;
	    merge_pair_introns(transcript3,transcript5);
	    add_between_introns(transcript3,transcript5->exons,transcript3->exons,plusp,transcript3->genestrand);
	    concordant = Listpool_push(concordant,listpool,(void *) transcript3);
	  }
	}
	    
      } else {
	/* Minus on transcript */
	debug2(printf("(6) Checking for concordance on trnum %d: end3 %d, start5 %d\n",
		      trnum5,transcript3->end,transcript5->start));
	if (transcript3->end < transcript5->start && transcript5->start <= transcript3->end + pairmax_transcriptome) {
	  pair_velocity = compute_pair_velocity(transcript5->velocity,transcript3->velocity);
	  insertlength = transcript5->end - transcript3->start + querylength5 + querylength3;
	  if (first_read_p == true) {
	    transcript5->pair_velocity = pair_velocity;
	    transcript5->insertlength = insertlength;
	    merge_pair_introns(transcript5,transcript3);
	    add_between_introns(transcript5,transcript5->exons,transcript3->exons,plusp,transcript5->genestrand);
	    concordant = Listpool_push(concordant,listpool,(void *) transcript5);
	  } else {
	    transcript3->pair_velocity = pair_velocity;
	    transcript3->insertlength = insertlength;
	    merge_pair_introns(transcript3,transcript5);
	    add_between_introns(transcript3,transcript5->exons,transcript3->exons,plusp,transcript3->genestrand);
	    concordant = Listpool_push(concordant,listpool,(void *) transcript3);
	  }
	}
      }
	
      i = j;
      k = l;
    }
  }

  FREE(array3);
  FREE(array5);

  debug2(printf("Returning %d concordant spliced transcripts\n",List_length(concordant)));
  return List_reverse(concordant);
}


static void
exons_inside_chars (char *insidechar5, char *insidechar3,
		    List_T exons_first, List_T exons_second,
		    int transcript_genestrand, bool plusp) {
  Exon_T exon;

  if (transcript_genestrand > 0) {
    if (plusp == true) {
      exon = (Exon_T) List_last_value(exons_first,NULL);
      *insidechar5 = exon->lastchar;
      exon = (Exon_T) List_head(exons_second);
      *insidechar3 = exon->firstchar;
    } else {
      exon = (Exon_T) List_head(exons_first);
      *insidechar5 = exon->firstchar;
      exon = (Exon_T) List_last_value(exons_second,NULL);
      *insidechar3 = exon->lastchar;
    }
  } else {
    if (plusp == true) {
      exon = (Exon_T) List_head(exons_first);
      *insidechar5 = exon->firstchar;
      exon = (Exon_T) List_last_value(exons_second,NULL);
      *insidechar3 = exon->lastchar;
    } else {
      exon = (Exon_T) List_last_value(exons_first,NULL);
      *insidechar5 = exon->lastchar;
      exon = (Exon_T) List_head(exons_second);
      *insidechar3 = exon->firstchar;
    }
  }

  return;
}


/* pairedlength is the genomic insertlength */
/* transcripts are for the first read and then the second read */
List_T
Transcript_concordant_inconsistent (List_T transcripts5_spliced, List_T transcripts5_unspliced,
				    List_T transcripts3_spliced, List_T transcripts3_unspliced,
				    int querylength5, int querylength3, int pairedlength,
				    Listpool_T listpool, bool plusp, bool first_read_p, bool consistent_found_p) {
  List_T concordant = NULL;
  
  T *array5_spliced, *array3_spliced, *array5_unspliced, *array3_unspliced, transcript5, transcript3;
  int ntranscripts5_spliced, ntranscripts3_spliced, ntranscripts5_unspliced, ntranscripts3_unspliced;

  int i, j, k, l;
  int trnum5, trnum3;
  Transcript_velocity_T pair_velocity;
  int insertlength;
  char insidechar5, insidechar3;

#ifdef CHECK_ASSERTIONS
  int a;
#endif  


  debug2(printf("Entered Transcript_concordant_inconsistent with first_read_p %d\n",
		first_read_p));

  ntranscripts5_unspliced = List_length(transcripts5_unspliced);
  ntranscripts3_unspliced = List_length(transcripts3_unspliced);

  if (ntranscripts5_unspliced == 0 && ntranscripts3_unspliced == 0) {
    return (List_T) NULL;
  } else {
    if (ntranscripts5_unspliced > 0) {
      array5_unspliced = (T *) List_to_array(transcripts5_unspliced,NULL);
      qsort(array5_unspliced,ntranscripts5_unspliced,sizeof(T),Transcript_cmp);
    }
    if (ntranscripts3_unspliced > 0) {
      array3_unspliced = (T *) List_to_array(transcripts3_unspliced,NULL);
      qsort(array3_unspliced,ntranscripts3_unspliced,sizeof(T),Transcript_cmp);
    }
  }

  ntranscripts5_spliced = List_length(transcripts5_spliced);
  ntranscripts3_spliced = List_length(transcripts3_spliced);


  /* (1) Unspliced5 vs Spliced3 */
  if (ntranscripts5_unspliced == 0 || ntranscripts3_spliced == 0) {
    /* Skip */
  } else {
    array3_spliced = (T *) List_to_array(transcripts3_spliced,NULL);
    qsort(array3_spliced,ntranscripts3_spliced,sizeof(T),Transcript_cmp);

    i = k = 0;
    while (i < ntranscripts5_unspliced && k < ntranscripts3_spliced) {
      trnum5 = array5_unspliced[i]->num;
      j = i+1;
      while (j < ntranscripts5_unspliced && array5_unspliced[j]->num == trnum5) {
	j++;
      }
      
      trnum3 = array3_spliced[k]->num;
      l = k+1;
      while (l < ntranscripts3_spliced && array3_spliced[l]->num == trnum3) {
	l++;
      }
      
      if (trnum5 < trnum3) {
	i = j;
	
      } else if (trnum3 < trnum5) {
	k = l;
	
      } else {
	/* Previously, we could have a mixture of transcript endpoints
	   from transfer_transcripts, but now, Transcript_trim should
	   make them all equivalent */
#ifdef CHECK_ASSERTIONS
	for (a = i+1; a < j; a++) {
	  assert(array5_unspliced[a]->start == array5_unspliced[a-1]->start);
	  assert(array5_unspliced[a]->end == array5_unspliced[a-1]->end);
	}
	for (a = k+1; a < l; a++) {
	  assert(array3_spliced[a]->start == array3_spliced[a-1]->start);
	  assert(array3_spliced[a]->end == array3_spliced[a-1]->end);
	}
#endif

	transcript5 = array5_unspliced[i];
	transcript3 = array3_spliced[k];
	pair_velocity = compute_pair_velocity(transcript5->velocity,transcript3->velocity);
	exons_inside_chars(&insidechar5,&insidechar3,transcript5->exons,transcript3->exons,
			   transcript5->genestrand,plusp);
	if (insidechar5 == 'i' || insidechar3 == 'i') {
	  insertlength = pairedlength;
	} else if (transcript5->start < transcript5->end) {
	  insertlength = transcript3->start - transcript5->end + querylength5 + querylength3;
	} else {
	  insertlength = transcript5->end - transcript3->start + querylength5 + querylength3;
	}

#ifdef DEBUG2
	Exon_list_print_fwd_stdout(array5_unspliced[i]->exons);
	printf(" and ");
	Exon_list_print_fwd_stdout(array3_spliced[k]->exons);
	printf(" => %c and %c\n",insidechar5,insidechar3);
#endif

	if (consistent_found_p == true && (pair_velocity == V_CRYPTIC || pair_velocity == V_UTR)) {
	  /* Skip: already have good consistent solutions, so skip invalid ones in secondary */
	  debug2(printf("Skipping pair_velocity %d\n",pair_velocity));
	} else if (first_read_p == true) {
	  transcript5->pair_velocity = pair_velocity;
	  transcript5->insertlength = insertlength;
	  merge_pair_introns(transcript5,transcript3);
	  add_between_introns(transcript5,transcript5->exons,transcript3->exons,plusp,transcript5->genestrand);
	  concordant = Listpool_push(concordant,listpool,(void *) transcript5);
	} else {
	  transcript3->pair_velocity = pair_velocity;
	  transcript3->insertlength = insertlength;
	  merge_pair_introns(transcript3,transcript5);
	  add_between_introns(transcript3,transcript5->exons,transcript3->exons,plusp,transcript3->genestrand);
	  concordant = Listpool_push(concordant,listpool,(void *) transcript3);
	}
      
	i = j;
	k = l;
      }
    }

    FREE(array3_spliced);
  }


  /* (2) Spliced5 vs Unspliced3 */
  if (ntranscripts5_spliced == 0 || ntranscripts3_unspliced == 0) {
    /* Skip */
  } else {
    array5_spliced = (T *) List_to_array(transcripts5_spliced,NULL);
    qsort(array5_spliced,ntranscripts5_spliced,sizeof(T),Transcript_cmp);


    i = k = 0;
    while (i < ntranscripts5_spliced && k < ntranscripts3_unspliced) {
      trnum5 = array5_spliced[i]->num;
      j = i+1;
      while (j < ntranscripts5_spliced && array5_spliced[j]->num == trnum5) {
	j++;
      }
      
      trnum3 = array3_unspliced[k]->num;
      l = k+1;
      while (l < ntranscripts3_unspliced && array3_unspliced[l]->num == trnum3) {
	l++;
      }
      
      if (trnum5 < trnum3) {
	i = j;
	
      } else if (trnum3 < trnum5) {
	k = l;
	
      } else {
	/* Previously, we could have a mixture of transcript endpoints
	   from transfer_transcripts, but now, Transcript_trim should
	   make them all equivalent */
#ifdef CHECK_ASSERTIONS
	for (a = i+1; a < j; a++) {
	  assert(array5_spliced[a]->start == array5_spliced[a-1]->start);
	  assert(array5_spliced[a]->end == array5_spliced[a-1]->end);
	}
	for (a = k+1; a < l; a++) {
	  assert(array3_unspliced[a]->start == array3_unspliced[a-1]->start);
	  assert(array3_unspliced[a]->end == array3_unspliced[a-1]->end);
	}
#endif

	transcript5 = array5_spliced[i];
	transcript3 = array3_unspliced[k];
	pair_velocity = compute_pair_velocity(transcript5->velocity,transcript3->velocity);
	exons_inside_chars(&insidechar5,&insidechar3,transcript5->exons,transcript3->exons,
			   transcript5->genestrand,plusp);
	if (insidechar5 == 'i' || insidechar3 == 'i') {
	  insertlength = pairedlength;
	} else if (transcript5->start < transcript5->end) {
	  insertlength = transcript3->start - transcript5->end + querylength5 + querylength3;
	} else {
	  insertlength = transcript5->end - transcript3->start + querylength5 + querylength3;
	}

#ifdef DEBUG2
	Exon_list_print_fwd_stdout(array5_spliced[i]->exons);
	printf(" and ");
	Exon_list_print_fwd_stdout(array3_unspliced[k]->exons);
	printf(" => %c and %c\n",insidechar5,insidechar3);
#endif

	if (consistent_found_p == true && (pair_velocity == V_CRYPTIC || pair_velocity == V_UTR)) {
	  /* Skip: already have good consistent solutions, so skip invalid ones in secondary */
	  debug2(printf("Skipping pair_velocity %d\n",pair_velocity));
	} else if (first_read_p == true) {
	  transcript5->pair_velocity = pair_velocity;
	  transcript5->insertlength = insertlength;
	  merge_pair_introns(transcript5,transcript3);
	  add_between_introns(transcript5,transcript5->exons,transcript3->exons,plusp,transcript5->genestrand);
	  concordant = Listpool_push(concordant,listpool,(void *) transcript5);
	} else {
	  transcript3->pair_velocity = pair_velocity;
	  transcript3->insertlength = insertlength;
	  merge_pair_introns(transcript3,transcript5);
	  add_between_introns(transcript3,transcript5->exons,transcript3->exons,plusp,transcript3->genestrand);
	  concordant = Listpool_push(concordant,listpool,(void *) transcript3);
	}
      
	i = j;
	k = l;
      }
    }

    FREE(array5_spliced);
  }


  /* (3) Unspliced5 vs Unspliced3 */
  if (ntranscripts5_unspliced == 0 || ntranscripts3_unspliced == 0) {
    /* Skip */
  } else {
    i = k = 0;
    while (i < ntranscripts5_unspliced && k < ntranscripts3_unspliced) {
      trnum5 = array5_unspliced[i]->num;
      j = i+1;
      while (j < ntranscripts5_unspliced && array5_unspliced[j]->num == trnum5) {
	j++;
      }
      
      trnum3 = array3_unspliced[k]->num;
      l = k+1;
      while (l < ntranscripts3_unspliced && array3_unspliced[l]->num == trnum3) {
	l++;
      }
      
      if (trnum5 < trnum3) {
	i = j;
	
      } else if (trnum3 < trnum5) {
	k = l;
	
      } else {
	/* Previously, we could have a mixture of transcript endpoints
	   from transfer_transcripts, but now, Transcript_trim should
	   make them all equivalent */
#ifdef CHECK_ASSERTIONS
	for (a = i+1; a < j; a++) {
	  assert(array5_unspliced[a]->start == array5_unspliced[a-1]->start);
	  assert(array5_unspliced[a]->end == array5_unspliced[a-1]->end);
	}
	for (a = k+1; a < l; a++) {
	  assert(array3_unspliced[a]->start == array3_unspliced[a-1]->start);
	  assert(array3_unspliced[a]->end == array3_unspliced[a-1]->end);
	}
#endif

	transcript5 = array5_unspliced[i];
	transcript3 = array3_unspliced[k];
	pair_velocity = compute_pair_velocity(transcript5->velocity,transcript3->velocity);
	exons_inside_chars(&insidechar5,&insidechar3,transcript5->exons,transcript3->exons,
			   transcript5->genestrand,plusp);
	if (insidechar5 == 'i' || insidechar3 == 'i') {
	  insertlength = pairedlength;
	} else if (transcript5->start < transcript5->end) {
	  insertlength = transcript3->start - transcript5->end + querylength5 + querylength3;
	} else {
	  insertlength = transcript5->end - transcript3->start + querylength5 + querylength3;
	}

#ifdef DEBUG2
	Exon_list_print_fwd_stdout(array5_unspliced[i]->exons);
	printf(" and ");
	Exon_list_print_fwd_stdout(array3_unspliced[k]->exons);
	printf(" => %c and %c\n",insidechar5,insidechar3);
#endif

	if (consistent_found_p == true && (pair_velocity == V_CRYPTIC || pair_velocity == V_UTR)) {
	  /* Skip: already have good consistent solutions, so skip invalid ones in secondary */
	  debug2(printf("Skipping pair_velocity %d\n",pair_velocity));
	} else if (first_read_p == true) {
	  transcript5->pair_velocity = pair_velocity;
	  transcript5->insertlength = insertlength;
	  merge_pair_introns(transcript5,transcript3);
	  add_between_introns(transcript5,transcript5->exons,transcript3->exons,plusp,transcript5->genestrand);
	  concordant = Listpool_push(concordant,listpool,(void *) transcript5);
	} else {
	  transcript3->pair_velocity = pair_velocity;
	  transcript3->insertlength = insertlength;
	  array3_unspliced[k]->pair_velocity = pair_velocity;
	  merge_pair_introns(transcript3,transcript5);
	  add_between_introns(transcript3,transcript5->exons,transcript3->exons,plusp,transcript3->genestrand);
	  concordant = Listpool_push(concordant,listpool,(void *) transcript3);
	}
      
	i = j;
	k = l;
      }
    }
  }

  if (ntranscripts5_unspliced > 0) {
    FREE(array5_unspliced);
  }
  if (ntranscripts3_unspliced > 0) {
    FREE(array3_unspliced);
  }

  debug2(printf("Returning %d concordant unspliced transcripts\n",List_length(concordant)));
  return List_reverse(concordant);
}


List_T
Transcript_singleend_inconsistent (List_T transcripts, Listpool_T listpool, bool consistent_found_p) {
  List_T singleend = NULL, p;
  T transcript;

  for (p = transcripts; p != NULL; p = List_next(p)) {
    transcript = (T) List_head(p);
    if (consistent_found_p == true && (transcript->velocity == V_CRYPTIC || transcript->velocity == V_UTR)) {
      /* Skip: already have good consistent solutions, so skip invalid ones in secondary */
    } else {
      singleend = Listpool_push(singleend,listpool,(void *) transcript);
    }
  }

  return List_reverse(singleend);
}


bool
Transcript_concordant_p (List_T transcripts5, List_T transcripts3) {
  bool foundp;
  T *array5, *array3;
  int ntranscripts5, ntranscripts3;
  int i, j, k, l;
  int a, b;
  int trnum5, trnum3;
  int start5, end5, end3;

#ifdef DEBUG2
  char *label;
  bool allocp;
#endif


  if ((ntranscripts5 = List_length(transcripts5)) == 0 ||
      (ntranscripts3 = List_length(transcripts3)) == 0) {
    return false;
  } else {
    debug2(printf("Entered concordant_p with %d and %d transcripts\n",ntranscripts5,ntranscripts3));
    foundp = false;

    array5 = (T *) List_to_array(transcripts5,NULL);
    array3 = (T *) List_to_array(transcripts3,NULL);
    qsort(array5,ntranscripts5,sizeof(T),Transcript_cmp);
    qsort(array3,ntranscripts3,sizeof(T),Transcript_cmp);
  }

#ifdef DEBUG2
  for (i = 0; i < ntranscripts5; i++) {
    label = Univ_IIT_label(transcript_iit,array5[i]->num,&allocp);
    printf("5' Transcript %s, number %d\n",label,array5[i]->num);
  }
  for (i = 0; i < ntranscripts3; i++) {
    label = Univ_IIT_label(transcript_iit,array3[i]->num,&allocp);
    printf("3' Transcript %s, number %d\n",label,array3[i]->num);
  }
#endif

  i = k = 0;
  while (i < ntranscripts5 && k < ntranscripts3) {
    trnum5 = array5[i]->num;
    j = i+1;
    while (j < ntranscripts5 && array5[j]->num == trnum5) {
      j++;
    }

    trnum3 = array3[k]->num;
    l = k+1;
    while (l < ntranscripts3 && array3[l]->num == trnum3) {
      l++;
    }
    
    if (trnum5 < trnum3) {
      i = j;
      
    } else if (trnum3 < trnum5) {
      k = l;
      
    } else {
      /* Check only for non-intronic transcripts */
      for (a = i; a < j; a++) {
	start5 = array5[a]->start;
	end5 = array5[a]->end;
	
	if (start5 < end5) {
	  /* Plus on transcript */
	  for (b = k; b < l; b++) {
	    end3 = array3[b]->end;
#ifdef DEBUG2
	    label = Univ_IIT_label(transcript_iit,array5[a]->num,&allocp);
	    printf("plus: checking for concordance on %s, trnum %d: start5 %d, end3 %d, pairmax %d\n",
		   label,trnum5,start5,end3,pairmax_transcriptome);
#endif
	    if (start5 < end3 && end3 <= start5 + pairmax_transcriptome) {
	      foundp = true;
	    }
	  }
	  
	} else {
	  /* Minus on transcript */
	  for (b = k; b < l; b++) {
	    end3 = array3[b]->end;
#ifdef DEBUG2
	    label = Univ_IIT_label(transcript_iit,array3[b]->num,&allocp);
	    printf("minus: checking for concordance on %s, trnum %d: end3 %d, start5 %d, pairmax %d\n",
		   label,trnum5,end3,start5,pairmax_transcriptome);
#endif
	    if (end3 < start5 && start5 <= end3 + pairmax_transcriptome) {
	      foundp = true;
	    }
	  }
	}
      }
      
      i = j;
      k = l;
    }
  }

  FREE(array3);
  FREE(array5);

  debug2(printf("Returning foundp %d\n\n",foundp));
  return foundp;
}


static void
introns_print (Filestring_T fp, T this) {
  bool firstp = true;

  if (Bitvector_null_p(this->cryptic_introns) == false) {
    firstp = false;
    PUTC('X',fp);
    Bitvector_print(fp,this->cryptic_introns,/*sep*/'-');
  }

  if (this->utr5p == true && this->utr3p == true) {
    if (firstp == true) {
      firstp = false;
    } else {
      PUTC('/',fp);
    }
    FPRINTF(fp,"U5-3");		/* Cannot use ',', which separates transcripts */

  } else if (this->utr5p == true) {
    if (firstp == true) {
      firstp = false;
    } else {
      PUTC('/',fp);
    }
    FPRINTF(fp,"U5");

  } else if (this->utr3p == true) {
    if (firstp == true) {
      firstp = false;
    } else {
      PUTC('/',fp);
    }
    FPRINTF(fp,"U3");
  }

  if (Bitvector_null_p(this->retained_introns) == false) {
    if (firstp == true) {
      firstp = false;
    } else {
      PUTC('/',fp);
    }
    PUTC('I',fp);
    Bitvector_print(fp,this->retained_introns,/*sep*/'-');
  }

  if (Bitvector_null_p(this->spliced_introns) == true) {
    if (firstp == true) {
      PUTC('A',fp);
    }
  } else {
    if (firstp == true) {
      firstp = false;
    } else {
      PUTC('/',fp);
    }
    PUTC('S',fp);
    Bitvector_print(fp,this->spliced_introns,/*sep*/'-');
  }

  PUTC(':',fp);

  return;
}


static void
print_one (Filestring_T fp, T this, Univ_IIT_T transcript_iit) {
  char *label;
  bool allocp;

  label = Univ_IIT_label(transcript_iit,this->num,&allocp);

#if 0
  /* Printing only in SAM format */
  if (output_type == STD_OUTPUT) {
    /* Follow query */
    if (invertp == false) {
      if (this->start < this->end) {
	FPRINTF(fp,"+%s:%d..%d:",label,
		this->start < 0 ? this->start : this->start + 1,
		this->end);
	introns_print(fp,this);
	Exon_list_print_fwd(fp,this->exons);
      } else {
	FPRINTF(fp,"-%s:%d..%d:",label,
		this->start,
		this->end < 0 ? this->end : this->end + 1);
	introns_print(fp,this);
	Exon_list_print_rev(fp,this->exons);
      }
    } else {
      if (this->start < this->end) {
	FPRINTF(fp,"-%s:%d..%d:",label,
		this->end,
		this->start < 0 ? this->start : this->start + 1);
	introns_print(fp,this);
	Exon_list_print_rev(fp,this->exons);
      } else {
	FPRINTF(fp,"+%s:%d..%d:",label,
		this->end < 0 ? this->end : this->end + 1,
		this->start);
	introns_print(fp,this);
	Exon_list_print_fwd(fp,this->exons);
      }
    }
  }
#endif

  if (this->genestrand > 0) {
    if (this->start < this->end) {
      FPRINTF(fp,"+%s:%d..%d:",label,
	      this->start < 0 ? this->start : this->start + 1,
	      this->end);
    } else {
      FPRINTF(fp,"+%s:%d..%d:",label,
	      this->end < 0 ? this->end : this->end + 1,
	      this->start);
    }
    introns_print(fp,this);
    Exon_list_print_fwd(fp,this->exons);

  } else {
    if (this->start < this->end) {
      FPRINTF(fp,"-%s:%d..%d:",label,
	      this->end,
	      this->start < 0 ? this->start : this->start + 1);
    } else {
      FPRINTF(fp,"-%s:%d..%d:",label,
	      this->start,
	      this->end < 0 ? this->end : this->end + 1);
    }
    introns_print(fp,this);
    Exon_list_print_rev(fp,this->exons);
  }

  if (this->insertlength > 0) {
    FPRINTF(fp,":%d",this->insertlength);
  }

  if (allocp) {
    FREE(label);
  }

  return;
}


void
Transcript_print_list (Filestring_T fp, List_T transcripts,
		       Univ_IIT_T transcript_iit, char *header) {
  T transcript;
  List_T p;

  if (transcripts != NULL) {
    transcript = List_head(transcripts);
    FPRINTF(fp,"%s",header);
    print_one(fp,transcript,transcript_iit);

    for (p = List_next(transcripts); p != NULL; p = List_next(p)) {
      transcript = List_head(p);
      PUTC(',',fp);
      print_one(fp,transcript,transcript_iit);
    }
  }

  return;
}

