// Copyright 2025 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package slhdsa

import (
	"crypto/rand"
	"slices"
	"testing"
)

func TestHypertreeSignKat(t *testing.T) {
	par := SLH_DSA_SHA2_128s
	msg := []byte{0x87, 0x56, 0x63, 0x8b, 0xfa, 0xa, 0x94, 0xb8, 0x3d, 0x93, 0x98, 0x25, 0x8f, 0xe8, 0x90, 0x93}
	skSeed := []byte{0x8e, 0x37, 0x36, 0x28, 0x86, 0x12, 0xfb, 0xd, 0xff, 0xd5, 0xa6, 0xa1, 0x6b, 0xe0, 0x89, 0xfb}
	pkSeed := []byte{0xa9, 0xb5, 0xe3, 0xd9, 0x99, 0xf8, 0x18, 0xc3, 0x41, 0x51, 0x56, 0x92, 0x61, 0xe8, 0x39, 0x2b}
	idxTree := uint64(0xe0d7a2b48b675)
	idxLeaf := uint32(0x1c3)
	wantSig := []byte{0x72, 0x78, 0x2d, 0xdd, 0x9d, 0x6e, 0xde, 0xc7, 0x3a, 0x61, 0x6b, 0x8, 0x19, 0x4f, 0x5f, 0xd, 0xb9, 0xea, 0xa3, 0x88, 0x28, 0xfe, 0xb7, 0xf5, 0x8e, 0x34, 0xa1, 0xa6, 0xa2, 0xb1, 0x83, 0x5a, 0x52, 0x0, 0x9, 0x20, 0xdf, 0x7a, 0xfb, 0x2b, 0xe6, 0x5c, 0xf5, 0x5, 0x9f, 0x71, 0x15, 0xe9, 0xa0, 0x11, 0x4, 0x6e, 0x87, 0xfd, 0x9f, 0xe0, 0x13, 0x9f, 0x43, 0xd3, 0xeb, 0xa1, 0x3b, 0xcd, 0x2, 0xf0, 0x36, 0xf, 0x92, 0xeb, 0x3, 0x28, 0x6d, 0xfa, 0xd0, 0x85, 0x7f, 0xb3, 0x7c, 0x3, 0x27, 0xa2, 0x3, 0xec, 0xa9, 0x56, 0x9d, 0xa0, 0xcf, 0xc1, 0x17, 0x1e, 0x51, 0x43, 0xbb, 0x3b, 0x1e, 0x58, 0xfa, 0x9, 0xe9, 0x56, 0x2, 0x8f, 0x73, 0x4e, 0x42, 0xb3, 0xc, 0xeb, 0xf2, 0xa5, 0x97, 0x1, 0x71, 0x24, 0xab, 0x6, 0x90, 0xae, 0x32, 0x14, 0x5d, 0x72, 0x3e, 0x30, 0x14, 0x9d, 0x6b, 0x2, 0xfe, 0x10, 0x7f, 0x47, 0x7d, 0xf5, 0x78, 0x79, 0xef, 0x91, 0xd5, 0xc9, 0x4b, 0xde, 0xa9, 0x38, 0x36, 0x1e, 0x8, 0x68, 0x94, 0xec, 0x48, 0x5a, 0xe1, 0xd2, 0x1c, 0x6c, 0x31, 0x50, 0xcd, 0x17, 0x6d, 0x14, 0x11, 0x65, 0x76, 0x8e, 0xb4, 0x4c, 0xe5, 0x8b, 0x41, 0xc5, 0xef, 0x70, 0x2a, 0x3f, 0x8, 0xc3, 0xb4, 0x66, 0xad, 0x22, 0xbc, 0x3c, 0x6e, 0xec, 0x13, 0xb9, 0xe9, 0x67, 0xd, 0x66, 0xe4, 0xf8, 0x53, 0x5d, 0xc2, 0x1a, 0x1c, 0x84, 0x10, 0xb, 0xfe, 0xd7, 0xb5, 0x1e, 0x28, 0x5a, 0xcc, 0xe3, 0x85, 0x8c, 0x34, 0x31, 0xac, 0xa9, 0x82, 0x87, 0xa, 0x27, 0xc8, 0x59, 0xc2, 0x7c, 0xa1, 0xd5, 0x4f, 0x9f, 0x23, 0xe1, 0x4, 0xa9, 0x44, 0x77, 0x4b, 0x26, 0xcd, 0x92, 0xf0, 0x71, 0xbf, 0x9, 0x5d, 0xfc, 0x71, 0xca, 0x39, 0x42, 0xa3, 0x5f, 0x3b, 0x12, 0xd4, 0x61, 0x64, 0xff, 0x8e, 0xe9, 0xb, 0xf9, 0x49, 0xd2, 0x18, 0xb0, 0x9f, 0x40, 0xaa, 0x18, 0x54, 0x20, 0xb9, 0xb7, 0x5f, 0xfd, 0xa7, 0x37, 0x98, 0xcf, 0x2, 0xd4, 0x6a, 0x12, 0xb, 0x77, 0x46, 0x13, 0x18, 0xee, 0xcf, 0xa7, 0xbc, 0x4a, 0x70, 0x4b, 0xdb, 0x4d, 0x65, 0x71, 0x4d, 0x15, 0x7, 0x90, 0xed, 0x39, 0x28, 0x99, 0x53, 0x8c, 0xe7, 0x40, 0xfc, 0x51, 0x35, 0xcf, 0x77, 0x49, 0xcd, 0x4, 0x34, 0xb6, 0x5d, 0x9f, 0x6, 0xca, 0xe5, 0x44, 0x98, 0x61, 0x7d, 0x2a, 0xab, 0x31, 0x8, 0x31, 0xee, 0xf5, 0x81, 0x8, 0xbe, 0xe3, 0x83, 0x38, 0x3d, 0xf6, 0xd2, 0xe6, 0x22, 0xbf, 0xdf, 0x59, 0x4, 0xbe, 0x5a, 0xba, 0x45, 0x63, 0x71, 0xa2, 0x36, 0xb4, 0x39, 0xb7, 0x7a, 0xad, 0x33, 0x39, 0x40, 0x48, 0x94, 0x9d, 0x12, 0xc0, 0x97, 0xaa, 0xf, 0x93, 0x2a, 0x99, 0xe1, 0x1, 0xa6, 0xaa, 0x2e, 0x27, 0x3e, 0x84, 0x78, 0xe2, 0xa7, 0xab, 0x63, 0xe0, 0xcd, 0x36, 0x35, 0x8b, 0x8a, 0xab, 0xd, 0x26, 0xc7, 0x91, 0x37, 0x34, 0xfd, 0x81, 0xc3, 0xc0, 0x65, 0x66, 0x30, 0xef, 0x69, 0xf7, 0x2a, 0x80, 0xbd, 0xec, 0x5, 0x93, 0x6b, 0x60, 0x1a, 0x3e, 0xa6, 0xc6, 0x52, 0x1d, 0x11, 0x2c, 0x7d, 0xde, 0xc2, 0xfe, 0xc0, 0xce, 0x8d, 0xe9, 0x30, 0x38, 0x63, 0x56, 0xd2, 0x93, 0xe1, 0x89, 0x76, 0x5f, 0x84, 0xe7, 0x4b, 0x79, 0xa3, 0x87, 0x34, 0xba, 0x55, 0xda, 0x29, 0xd0, 0x3e, 0x53, 0x77, 0x7b, 0xb9, 0xa7, 0x45, 0xd5, 0xd7, 0xfc, 0xe, 0x55, 0x3d, 0x80, 0xc, 0x7a, 0x3e, 0x4f, 0x17, 0x28, 0x4c, 0xe3, 0x1d, 0x51, 0x1f, 0xe4, 0x7, 0x4b, 0xf0, 0xdd, 0x1, 0x0, 0xca, 0x4b, 0xdb, 0x2b, 0x79, 0x57, 0xcb, 0xe8, 0x7, 0xd5, 0xac, 0x49, 0xbf, 0xaf, 0xcb, 0x63, 0xb5, 0x67, 0xbc, 0x28, 0xb0, 0xd5, 0x3b, 0x60, 0xa8, 0x8f, 0x9f, 0xea, 0x92, 0x67, 0x3f, 0xe4, 0xd8, 0xff, 0x8e, 0x2d, 0xea, 0x5b, 0x46, 0xd8, 0x3, 0x34, 0x1d, 0xf5, 0x19, 0xd5, 0xd9, 0x47, 0xe0, 0x93, 0x90, 0xd4, 0xfb, 0x2c, 0x66, 0x33, 0x9a, 0xe6, 0x6, 0x25, 0xec, 0xa1, 0xa, 0x5, 0xb8, 0x71, 0x9d, 0x8e, 0xb3, 0xd, 0x62, 0x40, 0x6c, 0x0, 0x6a, 0x8f, 0x9e, 0x47, 0xa1, 0x35, 0x5e, 0xd1, 0xd8, 0xad, 0x7f, 0x4a, 0x65, 0x2c, 0x9c, 0x1a, 0x4, 0xa9, 0x60, 0xa1, 0x9, 0x13, 0xf, 0x8a, 0x7c, 0x17, 0x93, 0xac, 0x69, 0x5f, 0x1c, 0xab, 0x45, 0x1f, 0xd1, 0x64, 0xa0, 0xb1, 0x64, 0xfb, 0x1b, 0x8a, 0xe3, 0x67, 0x61, 0x12, 0xf5, 0x88, 0xe3, 0x8, 0x4b, 0x8d, 0x63, 0xfd, 0xec, 0xb2, 0x44, 0x8f, 0xbd, 0x69, 0xef, 0x7, 0x51, 0x87, 0x17, 0x38, 0x5f, 0x45, 0x5, 0x9d, 0xf, 0x22, 0xf4, 0xde, 0x7, 0x3d, 0x94, 0x70, 0xfe, 0xa4, 0x2b, 0x80, 0xeb, 0x7f, 0xab, 0x2a, 0x2a, 0xb0, 0xe3, 0xa8, 0x86, 0x68, 0x8e, 0x67, 0x64, 0xa7, 0xc2, 0x4b, 0x6, 0x9c, 0x3f, 0x74, 0xc6, 0x70, 0xc1, 0x3a, 0x85, 0x14, 0x76, 0x13, 0x7, 0xd7, 0xd1, 0x8e, 0x27, 0x8, 0x52, 0x9b, 0x1d, 0xd, 0xeb, 0xec, 0x5c, 0xc7, 0x6b, 0x43, 0x84, 0xb9, 0xdb, 0x7f, 0x11, 0x2, 0xea, 0xea, 0xa, 0x49, 0x69, 0xd6, 0x20, 0x83, 0xe6, 0xbc, 0x2, 0x2c, 0x6d, 0x8d, 0x6c, 0xdd, 0xb3, 0x4e, 0xb4, 0xdf, 0x54, 0x65, 0x8d, 0x2, 0x19, 0xc3, 0x6a, 0x3, 0x57, 0xf9, 0x88, 0xc1, 0xbc, 0x15, 0x58, 0x71, 0x74, 0xdb, 0x79, 0xab, 0x2b, 0x18, 0xfe, 0x5d, 0xfe, 0x96, 0xf6, 0x7e, 0x8a, 0x6, 0x1a, 0xef, 0x87, 0x7, 0x63, 0x21, 0x9c, 0x37, 0xfe, 0x76, 0x2c, 0xb3, 0xac, 0xc6, 0x61, 0x3c, 0xd6, 0x70, 0x61, 0xae, 0xe8, 0x4c, 0xf7, 0x2, 0xe3, 0xca, 0x63, 0x38, 0xb, 0xba, 0x3d, 0x39, 0x7f, 0xe8, 0x68, 0xeb, 0xf8, 0x28, 0xf5, 0x66, 0xe8, 0xa8, 0xd1, 0xa6, 0xc5, 0xab, 0xb2, 0x1a, 0x69, 0x52, 0xd5, 0x30, 0x3, 0xb3, 0x20, 0xc1, 0x51, 0x96, 0xd8, 0xc, 0x38, 0x29, 0x1e, 0xa6, 0x59, 0xeb, 0xe0, 0x5, 0x74, 0xe0, 0x3a, 0xee, 0x8d, 0xec, 0x7f, 0xc2, 0xac, 0x62, 0x82, 0x20, 0xf7, 0xbf, 0xb, 0x30, 0x3f, 0x6f, 0xcf, 0xb9, 0xfd, 0x99, 0x26, 0xba, 0xf1, 0x39, 0x0, 0x5d, 0xb, 0x90, 0x5a, 0x86, 0xce, 0xfa, 0x8c, 0x40, 0x77, 0x6, 0x35, 0xe6, 0xb1, 0xd, 0x60, 0xd5, 0x6, 0xcd, 0x2f, 0x15, 0xaf, 0xa4, 0xf4, 0xf2, 0xe6, 0xda, 0x67, 0x68, 0x60, 0xf5, 0xed, 0xd6, 0x54, 0xc3, 0x26, 0xb2, 0x52, 0x5a, 0x0, 0x8b, 0x2, 0xa5, 0x78, 0xfa, 0xe4, 0x22, 0x71, 0x3a, 0xad, 0x2e, 0x19, 0x2a, 0x8d, 0xf0, 0xe1, 0xb9, 0x70, 0x79, 0xc9, 0xfe, 0x32, 0xe4, 0xc2, 0xaf, 0xba, 0x11, 0xe, 0xb0, 0xe1, 0x20, 0x5c, 0xac, 0x8c, 0xb, 0x57, 0xb5, 0x48, 0xb3, 0x35, 0x22, 0xf1, 0x44, 0xfe, 0x8, 0x25, 0x10, 0x41, 0x90, 0x70, 0x3b, 0x33, 0xf7, 0x4d, 0xa3, 0xb2, 0x22, 0x8, 0xe, 0xbe, 0xcf, 0x54, 0x21, 0x31, 0x24, 0xea, 0x96, 0xe7, 0x33, 0x62, 0x20, 0xf, 0x92, 0x7f, 0xe1, 0x39, 0xaf, 0x5b, 0xdd, 0xbb, 0x9f, 0xc7, 0x31, 0x6a, 0x4a, 0x4d, 0x0, 0xae, 0x51, 0x5f, 0xdd, 0xe3, 0x6a, 0x33, 0xef, 0x9b, 0x81, 0x8c, 0xab, 0xc1, 0x82, 0xb8, 0x80, 0xef, 0x88, 0x88, 0x2b, 0x58, 0x6d, 0x5a, 0x88, 0x68, 0x6c, 0x31, 0x71, 0xc2, 0x47, 0xce, 0x3b, 0x47, 0x77, 0x8c, 0xc0, 0x72, 0xd1, 0x9d, 0xa3, 0xc0, 0x2b, 0xdd, 0x12, 0x42, 0xbb, 0xd8, 0x36, 0x18, 0x69, 0x26, 0x43, 0xdb, 0xda, 0xac, 0x5d, 0x9d, 0x7b, 0xad, 0x64, 0x6f, 0x2b, 0x57, 0x33, 0x89, 0x2c, 0xdc, 0x8d, 0xb0, 0xe0, 0xfd, 0xbf, 0x6, 0x10, 0x24, 0xc1, 0x2, 0xeb, 0x18, 0xad, 0xff, 0xc4, 0x2, 0x2a, 0x2a, 0xf4, 0x4d, 0xb1, 0x89, 0x68, 0x19, 0x95, 0x61, 0xa5, 0x89, 0xdf, 0x82, 0xc6, 0xd0, 0x79, 0xc9, 0xc1, 0x93, 0x31, 0x95, 0x6f, 0xf9, 0x7c, 0xe1, 0x49, 0xe0, 0xe0, 0xe1, 0xd9, 0x33, 0x41, 0x7e, 0x6c, 0x37, 0x2f, 0xb3, 0x78, 0xf1, 0x8b, 0x46, 0x22, 0x44, 0xc9, 0x34, 0xd2, 0x67, 0x38, 0xcf, 0x6a, 0x11, 0xbd, 0xaa, 0xcc, 0x1f, 0xdb, 0x0, 0x48, 0xe1, 0x26, 0xb9, 0x53, 0x20, 0x2d, 0xd6, 0x2e, 0xef, 0x1c, 0xdd, 0x50, 0xfc, 0x37, 0x6b, 0x31, 0xd5, 0xa5, 0xe0, 0xf1, 0xe4, 0x91, 0xb1, 0xe7, 0x39, 0x55, 0x5a, 0x50, 0x57, 0x3e, 0x2d, 0xdb, 0xc6, 0x7c, 0x66, 0x22, 0x48, 0x36, 0xe6, 0x8a, 0xf1, 0x6, 0x2f, 0xc5, 0x5d, 0x8c, 0x5b, 0x74, 0x73, 0xe0, 0x1b, 0xb5, 0x52, 0x2a, 0x56, 0x84, 0x22, 0xa9, 0xa4, 0x8c, 0x1a, 0x89, 0xaa, 0x3e, 0x81, 0xee, 0x5e, 0xa1, 0x64, 0x2f, 0x11, 0xf4, 0x4b, 0xe4, 0x83, 0xc7, 0xef, 0xf, 0xec, 0xcb, 0x35, 0x23, 0x1f, 0xb6, 0xc5, 0x8e, 0xb3, 0xf8, 0xaf, 0x7a, 0xc2, 0xf4, 0x68, 0xde, 0x24, 0xee, 0x3c, 0x66, 0x86, 0x9e, 0xa6, 0x63, 0x66, 0xdd, 0xfd, 0x8b, 0xd6, 0xd3, 0x9f, 0x24, 0xfd, 0x22, 0x2c, 0x57, 0x18, 0x3e, 0xc4, 0xe0, 0xb1, 0x8d, 0x64, 0x5a, 0x7, 0x62, 0x65, 0xfb, 0x72, 0x30, 0xc8, 0x9f, 0x3c, 0xc3, 0xbb, 0xaa, 0xef, 0x13, 0xb4, 0x22, 0xd8, 0x57, 0xe4, 0xfd, 0xe9, 0xe, 0x93, 0x83, 0x78, 0x65, 0x6a, 0x49, 0xdc, 0x58, 0x16, 0x4, 0xf8, 0xe9, 0x6f, 0x60, 0x4c, 0x1a, 0x25, 0x1c, 0x88, 0xde, 0xf7, 0x5a, 0x30, 0x37, 0x60, 0xe, 0xa3, 0xc5, 0x26, 0x97, 0x2e, 0x20, 0xd6, 0x8e, 0xf3, 0x1b, 0x5, 0x24, 0xe, 0x7d, 0x7e, 0xf3, 0x12, 0x7f, 0xd1, 0x31, 0xed, 0x93, 0x2d, 0xe8, 0xd, 0xeb, 0x19, 0xa6, 0xe1, 0x20, 0x62, 0x9e, 0x5a, 0xf6, 0x1, 0xb7, 0xad, 0xae, 0xd3, 0x28, 0xcb, 0x91, 0x99, 0x35, 0x85, 0xd2, 0xfd, 0x2d, 0x52, 0xc2, 0x79, 0x6f, 0x16, 0x90, 0x56, 0xdf, 0x2e, 0x2a, 0x34, 0x25, 0x6d, 0xec, 0x97, 0xcd, 0xdf, 0x52, 0x3, 0xeb, 0xf, 0xb3, 0x55, 0xd8, 0xf8, 0x52, 0x95, 0xaf, 0xa2, 0xdc, 0xee, 0xb0, 0xd3, 0xcf, 0x8d, 0xc8, 0xa5, 0xc6, 0x20, 0x21, 0x79, 0x54, 0x7e, 0x64, 0x55, 0x24, 0xfc, 0x48, 0xce, 0x70, 0x7a, 0xa8, 0xba, 0xc9, 0xcc, 0x27, 0x40, 0xde, 0x34, 0x99, 0xc3, 0xd8, 0xaa, 0xf8, 0xdd, 0x9e, 0xef, 0x6d, 0x58, 0x46, 0x76, 0xd0, 0x6, 0x54, 0xb6, 0x38, 0x91, 0x67, 0x8d, 0xf3, 0x6e, 0x43, 0x93, 0x40, 0x70, 0x4c, 0xbd, 0x9e, 0xf1, 0x22, 0x49, 0x82, 0x40, 0xc9, 0x5, 0x10, 0xe3, 0x1c, 0xc4, 0x9f, 0x44, 0x39, 0xa6, 0x4f, 0xbe, 0x7, 0x6d, 0x32, 0x4b, 0x68, 0x70, 0x25, 0x8d, 0x43, 0x3b, 0x93, 0x53, 0x65, 0xe1, 0xfe, 0x5e, 0x34, 0xba, 0xc9, 0xa7, 0x2d, 0xfc, 0xd9, 0xe5, 0xf8, 0x1, 0x84, 0x6d, 0xe, 0x9, 0xa9, 0xa9, 0x3e, 0x57, 0xd2, 0xdd, 0x9a, 0xcd, 0x21, 0xec, 0xfc, 0xa, 0xfc, 0x32, 0xd9, 0x9e, 0xe2, 0xca, 0xe7, 0xda, 0x9, 0x21, 0x17, 0xdc, 0x66, 0x1f, 0xfa, 0x1c, 0xcd, 0xe3, 0xf1, 0xee, 0x7a, 0x8a, 0x5e, 0xb7, 0x31, 0x82, 0xab, 0xa4, 0x4d, 0xfd, 0x62, 0x7c, 0xf8, 0x9f, 0xa9, 0x42, 0xab, 0xa3, 0xd0, 0xa, 0x10, 0x8, 0x44, 0x7b, 0x39, 0x3a, 0x82, 0x96, 0x26, 0x39, 0xae, 0x21, 0x40, 0x6c, 0xcc, 0xbf, 0xb8, 0xe5, 0xa5, 0xa0, 0x24, 0x3c, 0x3e, 0xc2, 0x0, 0xfa, 0xf4, 0x14, 0xd4, 0xa9, 0xb7, 0xc1, 0xab, 0xbf, 0xe3, 0xda, 0xc3, 0x1f, 0xb3, 0x21, 0x60, 0x56, 0xfb, 0x5f, 0xf4, 0x33, 0xe6, 0xb1, 0xad, 0x1e, 0x53, 0x2a, 0x65, 0x47, 0xa6, 0x26, 0x20, 0x33, 0xed, 0xab, 0xc4, 0x3f, 0xe1, 0xdd, 0xcc, 0xf7, 0xe3, 0x8, 0x0, 0x58, 0x68, 0x9f, 0xf9, 0x41, 0x8d, 0xfc, 0x45, 0x18, 0x8a, 0xd9, 0x5d, 0x5f, 0xaf, 0xeb, 0xcc, 0xec, 0x88, 0xf3, 0x7a, 0x37, 0x78, 0x9f, 0x63, 0x77, 0x5d, 0xbc, 0x3d, 0xd7, 0xb7, 0x72, 0x16, 0xfe, 0x36, 0xac, 0x45, 0xe8, 0xbb, 0x47, 0xec, 0x99, 0x92, 0x44, 0x21, 0xfd, 0x5f, 0xb9, 0x1f, 0xeb, 0xf5, 0xab, 0xee, 0x25, 0xea, 0xb5, 0x92, 0x6e, 0xf2, 0x89, 0x4c, 0x63, 0x2c, 0xb3, 0x75, 0x74, 0xea, 0x3, 0x5c, 0x44, 0x9f, 0x69, 0x5b, 0xdc, 0x58, 0xde, 0xdd, 0xb3, 0xa0, 0x80, 0xcd, 0xf5, 0xd3, 0x99, 0xcf, 0xe1, 0xed, 0xca, 0xb, 0x27, 0xd7, 0xc, 0xbb, 0x19, 0x5c, 0x2, 0xca, 0x5, 0xc9, 0x8c, 0xa2, 0x47, 0xc9, 0x2d, 0x39, 0xf2, 0xa9, 0x7e, 0x2, 0x48, 0x97, 0x9d, 0x4, 0xcf, 0xbb, 0x57, 0x9f, 0xbd, 0xfd, 0x1d, 0x31, 0xe0, 0x43, 0x34, 0x1, 0xb6, 0xec, 0xa0, 0x3e, 0xe6, 0x96, 0xaf, 0x8f, 0x99, 0xf9, 0xb7, 0xad, 0xd4, 0x7, 0xe3, 0xea, 0xb5, 0x72, 0x7c, 0x9d, 0x1d, 0x66, 0xfb, 0x4f, 0x70, 0xa, 0x3e, 0x26, 0x5a, 0x20, 0x4f, 0x4e, 0x9f, 0x2c, 0xa4, 0x91, 0x91, 0xbd, 0xe7, 0x97, 0xf1, 0x5c, 0xa8, 0x94, 0x7c, 0x72, 0xee, 0xa1, 0x16, 0xcc, 0x81, 0x92, 0x7e, 0xbf, 0xd4, 0x40, 0x19, 0x69, 0xcc, 0x2f, 0x45, 0x99, 0x8, 0xfd, 0x22, 0x42, 0xfe, 0x8e, 0xa6, 0x7e, 0xec, 0x66, 0x62, 0x7, 0x21, 0xb3, 0x37, 0xa3, 0xf4, 0x31, 0xd9, 0xbd, 0xa5, 0x31, 0x37, 0x24, 0x48, 0xc0, 0xfe, 0xb6, 0x84, 0x88, 0x89, 0xe1, 0x41, 0x88, 0x99, 0xed, 0x8b, 0x3f, 0x23, 0xda, 0x96, 0x7b, 0xf, 0xa0, 0xb8, 0x59, 0x20, 0xbe, 0xc4, 0xe6, 0x56, 0xb1, 0x6a, 0x1e, 0x78, 0x22, 0x5b, 0x4f, 0x66, 0x80, 0xfc, 0x4e, 0x85, 0x6f, 0x60, 0x8b, 0x4b, 0x27, 0xd1, 0xd2, 0xb8, 0x12, 0x7d, 0x79, 0xc9, 0xfb, 0x92, 0xd4, 0x58, 0x40, 0xe, 0x8d, 0x7e, 0x7b, 0xf2, 0xda, 0x5c, 0x96, 0x7d, 0x23, 0x13, 0x87, 0xef, 0x84, 0x2a, 0x2e, 0xc6, 0xf3, 0xc2, 0xe2, 0x6, 0x42, 0x4d, 0x14, 0xb0, 0x60, 0x41, 0x3f, 0xfd, 0xb9, 0x6e, 0xb2, 0x5d, 0xc5, 0x65, 0xd9, 0x56, 0x9a, 0xb9, 0x1d, 0x3c, 0x96, 0xb9, 0xd5, 0x92, 0x28, 0x32, 0x7a, 0x6c, 0x4f, 0xdd, 0xb6, 0x7b, 0x7c, 0xec, 0xca, 0xc0, 0xf4, 0xf1, 0x81, 0x4a, 0x95, 0x4a, 0xff, 0x19, 0xd3, 0xf4, 0xcd, 0x11, 0x6, 0x1d, 0x96, 0xe9, 0x2f, 0x82, 0x40, 0x5a, 0x4c, 0xa9, 0x1b, 0x53, 0xc5, 0xff, 0xa2, 0x49, 0x2f, 0x6d, 0xb3, 0xf7, 0x4a, 0x43, 0xb3, 0xaa, 0x7b, 0xea, 0x99, 0x4a, 0xf8, 0x85, 0xf0, 0x59, 0xbd, 0xa3, 0xc4, 0xc9, 0x5a, 0x2c, 0xf9, 0x1, 0x68, 0x4c, 0xd5, 0x28, 0x67, 0x76, 0x45, 0x2d, 0xa2, 0xef, 0xfb, 0x5, 0x98, 0x1f, 0x9e, 0xa3, 0x6e, 0x76, 0x2e, 0x13, 0x72, 0x43, 0xdf, 0x35, 0x72, 0x92, 0xee, 0x9e, 0xf2, 0x89, 0x1b, 0xeb, 0xbc, 0xbd, 0x2c, 0x5, 0x5f, 0xa, 0xf9, 0x9e, 0x49, 0xf1, 0x5d, 0x9d, 0x6c, 0x96, 0x66, 0xdb, 0xa7, 0xd5, 0x39, 0xa1, 0xf7, 0x2d, 0x8c, 0x4c, 0x22, 0x1f, 0x4d, 0x64, 0xf2, 0xa9, 0xbd, 0x56, 0xdd, 0x62, 0xad, 0x68, 0x56, 0x7d, 0x9f, 0x1b, 0x33, 0x8f, 0x69, 0x57, 0x70, 0xcc, 0x78, 0xec, 0xa3, 0xf5, 0xe9, 0xa7, 0xe8, 0x9a, 0xd2, 0xf8, 0xd7, 0xa6, 0x20, 0x76, 0x78, 0x2a, 0xf1, 0x8d, 0xd0, 0xf, 0xa5, 0x17, 0x19, 0xa1, 0xfc, 0x58, 0xac, 0xe0, 0x14, 0x5a, 0x5e, 0x47, 0x83, 0xaf, 0x51, 0xed, 0x44, 0x9a, 0x57, 0x3d, 0xda, 0x11, 0x4a, 0x30, 0x6b, 0x1d, 0xa5, 0x7f, 0xd6, 0x45, 0x4b, 0x8f, 0xb9, 0xe, 0x36, 0xa2, 0x54, 0x55, 0x92, 0x33, 0x2c, 0x91, 0x1f, 0x72, 0x63, 0x3d, 0x6a, 0x58, 0x9e, 0xef, 0xb3, 0x9d, 0x48, 0xa0, 0x4d, 0x2c, 0x27, 0xe1, 0xd2, 0x27, 0x45, 0x36, 0x35, 0x92, 0x19, 0xfc, 0x11, 0x8f, 0x38, 0xac, 0x88, 0x8c, 0x3a, 0xe1, 0x32, 0xd5, 0xea, 0x5a, 0xa0, 0x6a, 0x67, 0x8a, 0x17, 0x55, 0x74, 0x1e, 0xfc, 0xae, 0x25, 0x7a, 0x72, 0x8b, 0xf3, 0xec, 0xea, 0xe9, 0x2b, 0xec, 0xf6, 0x97, 0xdc, 0x86, 0xa3, 0x1b, 0x9b, 0x7, 0x11, 0x43, 0xbd, 0xd4, 0x30, 0x67, 0xac, 0xaa, 0x3c, 0x47, 0xfd, 0x10, 0xaa, 0xa4, 0xaa, 0x3d, 0x2a, 0xa6, 0x48, 0x19, 0x2c, 0x3a, 0x4b, 0xf, 0xf9, 0xaf, 0x76, 0x69, 0xf6, 0xe7, 0xbf, 0x15, 0xf1, 0x69, 0x5, 0x9b, 0x78, 0x2a, 0xef, 0xc2, 0xe5, 0x3e, 0x75, 0xb, 0xfd, 0x4a, 0x2d, 0xf6, 0x3b, 0xf8, 0x26, 0x70, 0x62, 0x1, 0xba, 0x4e, 0x3c, 0x40, 0xf3, 0x78, 0xb9, 0xe9, 0xdb, 0xd1, 0xbd, 0xba, 0x43, 0x4, 0xd5, 0x3c, 0xa7, 0xe9, 0x10, 0x3b, 0x14, 0xcc, 0x8, 0xc5, 0x26, 0xce, 0xdc, 0xd7, 0xcc, 0x3e, 0xb8, 0xa9, 0x5, 0xdc, 0x35, 0xac, 0x3f, 0x1c, 0x23, 0x7d, 0xfa, 0x26, 0x69, 0xc0, 0x7a, 0x2f, 0x9e, 0x72, 0x5b, 0x5e, 0xe0, 0x84, 0xf, 0xff, 0x96, 0xb1, 0xb1, 0x3e, 0xeb, 0x2e, 0xbe, 0x20, 0x23, 0x64, 0x8d, 0xb9, 0xc8, 0x4c, 0x28, 0x1, 0x38, 0xdd, 0x85, 0xfb, 0xc4, 0xee, 0x26, 0xf7, 0xa9, 0x71, 0x94, 0xb2, 0xb4, 0x4b, 0x45, 0xaf, 0xe, 0xa1, 0x24, 0x7d, 0xfa, 0x82, 0xc5, 0xff, 0x39, 0xf6, 0x9c, 0xce, 0xb7, 0xec, 0xc3, 0xb9, 0x73, 0x55, 0x93, 0x3c, 0xc7, 0x41, 0x3b, 0xc8, 0xd8, 0x4b, 0xdc, 0x74, 0xb7, 0x81, 0x74, 0x4, 0x48, 0xaf, 0x15, 0x2a, 0x4d, 0xad, 0x32, 0x1a, 0x3, 0x94, 0x7a, 0xa8, 0x7d, 0x74, 0x6a, 0xed, 0x91, 0x5f, 0xdc, 0x70, 0x7c, 0xfd, 0xd4, 0xd9, 0xc4, 0x2, 0x1d, 0xdc, 0xd0, 0x1, 0x79, 0x3, 0xfb, 0x30, 0xce, 0xd5, 0x83, 0xd0, 0x59, 0x5f, 0xeb, 0x32, 0xab, 0xc8, 0xde, 0x2a, 0x30, 0x93, 0xb5, 0xaf, 0x9f, 0xfc, 0x4e, 0x2d, 0xd6, 0xfb, 0xee, 0x36, 0xa2, 0xaa, 0x65, 0x7e, 0xb8, 0xbc, 0x2f, 0xbd, 0x75, 0x21, 0xd7, 0x50, 0x9b, 0xe0, 0xc, 0x3e, 0x4e, 0x7, 0x9f, 0xc2, 0x7f, 0x34, 0x5b, 0xa8, 0x3e, 0x1d, 0x40, 0x31, 0xca, 0x60, 0x1b, 0xdc, 0x71, 0xb9, 0xd1, 0x74, 0x95, 0x97, 0x9f, 0x7e, 0xfa, 0x48, 0xbe, 0xb1, 0x95, 0xc6, 0x81, 0xdb, 0x6a, 0xa, 0x26, 0x2f, 0x5d, 0xe9, 0x69, 0x94, 0x54, 0xb2, 0x98, 0x8e, 0x99, 0x3c, 0x42, 0x3, 0xc0, 0xe5, 0x4, 0x51, 0x3d, 0xb3, 0xfd, 0xca, 0xe, 0x2d, 0xb, 0x6c, 0x3d, 0x5, 0xcc, 0x69, 0x28, 0x29, 0x3, 0xbf, 0x41, 0xf3, 0x64, 0x48, 0x41, 0x22, 0xdf, 0xb6, 0x6a, 0x53, 0xc4, 0xdd, 0xf4, 0x96, 0xbc, 0x86, 0x58, 0x38, 0x3e, 0xec, 0x3c, 0x30, 0x1a, 0xb2, 0x75, 0x3d, 0x23, 0x9c, 0xdb, 0xe1, 0x3c, 0x26, 0xf4, 0xa0, 0x40, 0x41, 0x32, 0x59, 0x2b, 0x75, 0x8c, 0x2d, 0xb7, 0x80, 0xe0, 0x7e, 0x93, 0xdd, 0x88, 0x98, 0xde, 0xd2, 0xbf, 0x21, 0xc6, 0x1d, 0xc0, 0x64, 0x8b, 0x29, 0xe0, 0x28, 0x88, 0xdf, 0x1e, 0x4, 0xe9, 0x8, 0x9, 0x2e, 0x60, 0xbd, 0x4, 0xac, 0x63, 0xa8, 0x73, 0xe9, 0x8c, 0xbe, 0x7, 0x3f, 0xdc, 0xcc, 0xea, 0x88, 0x46, 0xc9, 0x48, 0x98, 0xfe, 0xac, 0x8f, 0x3b, 0x91, 0x8a, 0xa1, 0xea, 0x11, 0x24, 0x70, 0xfb, 0x25, 0x2, 0x54, 0x44, 0xba, 0xe6, 0x2a, 0xe8, 0x60, 0x3, 0xcc, 0x40, 0xf5, 0xb6, 0x3f, 0x2b, 0xc5, 0x55, 0x5f, 0x40, 0xd9, 0xb1, 0xcd, 0xc7, 0x7c, 0xef, 0x80, 0xe8, 0xbe, 0x11, 0x17, 0x9b, 0x85, 0x8, 0xd3, 0xdf, 0x6f, 0xf4, 0x6a, 0x71, 0x5f, 0x60, 0x18, 0x39, 0xe5, 0x80, 0x4c, 0x67, 0xa4, 0x2a, 0x5d, 0xf, 0x2d, 0x2b, 0x31, 0x85, 0xb2, 0xa, 0x5d, 0xbe, 0x5b, 0x6a, 0x7b, 0xe8, 0x21, 0x10, 0x11, 0x7d, 0xe3, 0xc2, 0xe6, 0x34, 0xf6, 0xc7, 0xd, 0xf3, 0xf9, 0xc4, 0xa, 0xdc, 0x2a, 0xbe, 0xf9, 0xe7, 0x10, 0xc9, 0x3b, 0xa3, 0x2c, 0x17, 0x7e, 0x9f, 0x87, 0x1, 0x93, 0xd8, 0xb8, 0x60, 0xe6, 0x24, 0x89, 0xa1, 0x70, 0x43, 0xd2, 0x59, 0xfb, 0x4b, 0x6f, 0x7d, 0xac, 0xf5, 0x15, 0x6a, 0x8f, 0x26, 0x1a, 0x7e, 0xfb, 0xcd, 0x34, 0xb4, 0x64, 0xd4, 0xf2, 0xbf, 0x43, 0x6f, 0xba, 0x7f, 0x90, 0xa9, 0x2b, 0x2c, 0x31, 0xa9, 0xb4, 0x8c, 0x95, 0x97, 0x2e, 0x75, 0xb4, 0x5c, 0xd6, 0xc7, 0x2c, 0x11, 0xf3, 0x22, 0xe6, 0xb5, 0xca, 0xab, 0x13, 0x94, 0x4e, 0x7d, 0x99, 0x9a, 0xeb, 0x32, 0xa2, 0xf, 0x1a, 0x4b, 0x59, 0x29, 0x56, 0xd4, 0x36, 0x7e, 0xe, 0x8d, 0x13, 0x30, 0xcb, 0x67, 0xb6, 0x64, 0x6b, 0x16, 0xd2, 0x46, 0x9b, 0x40, 0x84, 0x62, 0xba, 0xaa, 0xc2, 0x9d, 0xc6, 0x8d, 0xe6, 0xf8, 0x5b, 0x6e, 0xb8, 0x83, 0x28, 0x87, 0xc, 0x9f, 0x3c, 0xc3, 0x74, 0xeb, 0xe8, 0x7e, 0x34, 0x5f, 0x2d, 0x48, 0x63, 0x57, 0xe3, 0x3c, 0xce, 0x53, 0x7d, 0x71, 0xd7, 0x6d, 0x50, 0xfa, 0x9e, 0x2f, 0xdf, 0xa6, 0x33, 0x40, 0x57, 0x9e, 0xf0, 0x17, 0xf3, 0x97, 0x29, 0xbb, 0x24, 0x58, 0x4f, 0x66, 0x92, 0x86, 0x84, 0x1f, 0xb7, 0x69, 0xa5, 0xf0, 0x89, 0xe8, 0x11, 0x64, 0x20, 0x4b, 0x78, 0xb5, 0x20, 0x2f, 0xc4, 0xc4, 0xdf, 0x72, 0x65, 0xfa, 0x73, 0xd9, 0x45, 0xcb, 0x16, 0xdb, 0xd3, 0x48, 0xa5, 0xc5, 0x92, 0x1, 0xff, 0x8c, 0xd, 0xdf, 0x13, 0x45, 0x62, 0xf9, 0xfd, 0x34, 0x17, 0xd, 0x8c, 0x7d, 0xd6, 0x6b, 0x18, 0xb6, 0xab, 0xdd, 0x62, 0x9a, 0x88, 0xff, 0xfe, 0x2c, 0x27, 0x1, 0x68, 0xac, 0x25, 0x85, 0x94, 0x98, 0xf5, 0x90, 0x92, 0xf1, 0x34, 0x89, 0x27, 0xdd, 0xe4, 0x8c, 0x48, 0x93, 0xb5, 0x66, 0x4d, 0xdc, 0xde, 0x58, 0x1, 0x98, 0x18, 0xb1, 0x49, 0x1, 0x57, 0x86, 0x2b, 0x10, 0xe0, 0xd3, 0xb3, 0xde, 0xa, 0x8e, 0xa4, 0xe0, 0xbc, 0xbe, 0xa5, 0xbf, 0xb6, 0x6a, 0x48, 0xda, 0x7a, 0xca, 0x29, 0x24, 0xef, 0xfe, 0xa3, 0xa5, 0x14, 0xda, 0xcb, 0xbb, 0xfb, 0x96, 0xcb, 0x95, 0x3c, 0xb, 0x55, 0xe4, 0xc, 0x3e, 0x5e, 0xa5, 0xd5, 0x95, 0xcc, 0x16, 0xd1, 0xa3, 0x8f, 0x14, 0xef, 0x83, 0x6f, 0x7, 0x59, 0xec, 0x0, 0x30, 0xee, 0x77, 0x27, 0xd0, 0xf4, 0xf8, 0xb3, 0xe3, 0x63, 0x6f, 0x27, 0x18, 0x1a, 0x13, 0xc, 0x77, 0xbd, 0xe9, 0x5a, 0xe4, 0xe2, 0xbd, 0x4f, 0x68, 0x7c, 0x70, 0x85, 0x3, 0xf9, 0xf5, 0x67, 0xed, 0x5, 0x27, 0x83, 0xf4, 0x11, 0xc9, 0x53, 0x90, 0x4c, 0xd9, 0x9f, 0x43, 0xa3, 0xf7, 0xef, 0x69, 0xdd, 0x86, 0xdb, 0x53, 0xaa, 0xba, 0x31, 0x39, 0x22, 0x5c, 0x54, 0xb, 0xfc, 0x74, 0xab, 0x2f, 0xe5, 0x10, 0x10, 0xac, 0xe9, 0x8c, 0x30, 0xc3, 0x42, 0xc2, 0x96, 0x40, 0x89, 0x28, 0x5c, 0xa0, 0xed, 0xd2, 0xbd, 0x62, 0x36, 0x67, 0x58, 0xa, 0xfb, 0xf4, 0x3, 0xf2, 0xe5, 0x38, 0x2c, 0x2e, 0x16, 0xdc, 0x7a, 0x2d, 0x29, 0x63, 0x46, 0x89, 0xac, 0xe5, 0x87, 0xb5, 0xca, 0x97, 0xd7, 0x5b, 0x3, 0xe, 0x90, 0x2a, 0x7, 0xbf, 0xf1, 0xc2, 0xbc, 0xf6, 0xd7, 0x3a, 0xe, 0x70, 0xbe, 0x82, 0xdf, 0xab, 0xb2, 0x58, 0x12, 0xb9, 0x65, 0x1d, 0xb, 0x7c, 0x5b, 0xcc, 0x27, 0xe2, 0x41, 0xee, 0x68, 0xf5, 0xf8, 0xa0, 0x51, 0xae, 0x55, 0x9f, 0x1c, 0xa9, 0xbd, 0x79, 0x98, 0x2, 0x1, 0x84, 0x66, 0x8, 0xfb, 0x1c, 0x7, 0x3d, 0x5f, 0xb3, 0xb8, 0x7e, 0xcd, 0xc, 0xce, 0xdc, 0x9, 0x30, 0xb6, 0xca, 0xaa, 0x14, 0xb1, 0x2d, 0x11, 0xea, 0xd9, 0xbf, 0x26, 0xc5, 0x76, 0x89, 0x8a, 0x2b, 0x2a, 0xe3, 0xf0, 0x4c, 0x36, 0x4b, 0x31, 0xb5, 0x95, 0x7b, 0xf, 0x4e, 0xba, 0x8e, 0xf2, 0x4a, 0x80, 0xab, 0xe1, 0x7, 0x84, 0x5e, 0x8f, 0xc2, 0xb0, 0x6e, 0x8d, 0x3d, 0xf7, 0x63, 0xf5, 0x3c, 0x75, 0x5b, 0x46, 0x21, 0xad, 0xb1, 0xb8, 0x5d, 0x51, 0x34, 0xbf, 0x5, 0x15, 0x0, 0x27, 0x26, 0x65, 0x73, 0x8e, 0x8, 0x93, 0xa0, 0xde, 0x92, 0x52, 0xc7, 0xb2, 0xe9, 0xa9, 0x51, 0xd, 0x1e, 0x8c, 0xb0, 0xd0, 0xe, 0x9d, 0xf5, 0x4e, 0x9e, 0x31, 0xee, 0xaf, 0x45, 0x76, 0xbc, 0x14, 0xbc, 0xfe, 0x52, 0x7a, 0x4e, 0x5e, 0x77, 0x8d, 0x55, 0x55, 0xe6, 0x8a, 0xa2, 0x49, 0x8a, 0x49, 0x53, 0xcc, 0xd, 0x4b, 0xe3, 0x8e, 0x44, 0x13, 0x97, 0x92, 0x43, 0x65, 0xe3, 0x85, 0xf4, 0x3d, 0xf1, 0xe0, 0x7, 0x30, 0x58, 0x12, 0x10, 0xa4, 0x3d, 0xb5, 0x1a, 0xb7, 0xf, 0x28, 0x70, 0x92, 0x8, 0xb4, 0x2d, 0x4a, 0xac, 0x48, 0xd7, 0xde, 0xb1, 0x39, 0xdd, 0x4a, 0x32, 0xf4, 0x7f, 0x9, 0x9, 0xb9, 0xa5, 0x9f, 0xee, 0xc8, 0x93, 0xb5, 0x61, 0x23, 0x5a, 0xc8, 0x2d, 0x52, 0x4c, 0xf3, 0xfc, 0x56, 0x19, 0x48, 0xe0, 0xba, 0x5f, 0xfa, 0xa4, 0xa4, 0x58, 0x4b, 0xa0, 0x46, 0x46, 0xd3, 0xea, 0x5f, 0xa4, 0xe4, 0x27, 0x8d, 0x81, 0xe0, 0xc2, 0xe9, 0x6f, 0xdd, 0x29, 0x1b, 0xb5, 0xb6, 0xcf, 0xde, 0x45, 0x92, 0x62, 0xb3, 0xf2, 0x47, 0x55, 0x19, 0x8a, 0xb8, 0xcb, 0xbd, 0x6c, 0x2f, 0x58, 0x7e, 0xc3, 0xbf, 0x81, 0xf8, 0x83, 0x16, 0x11, 0x21, 0x4b, 0x52, 0x80, 0x18, 0xe7, 0x35, 0xdd, 0x2c, 0xe7, 0xc5, 0xfb, 0xbc, 0x5a, 0xc5, 0x51, 0x9a, 0x3d, 0xd3, 0x2d, 0x68, 0xe7, 0x55, 0x43, 0x19, 0xd6, 0x58, 0x7d, 0xc3, 0x0, 0x4, 0xcc, 0x3a, 0xf1, 0xf4, 0x57, 0x31, 0x3e, 0xd0, 0x24, 0x6c, 0x66, 0xbd, 0xc3, 0xc0, 0xfc, 0x16, 0x6e, 0xfa, 0xc8, 0x9b, 0xe3, 0xf0, 0xce, 0x4c, 0xe, 0x99, 0x55, 0x9e, 0x55, 0x67, 0xd0, 0xe8, 0xa3, 0xb6, 0x26, 0x68, 0x36, 0x1c, 0x83, 0x36, 0x64, 0xdd, 0xd6, 0x79, 0x84, 0x1c, 0x2e, 0x24, 0xe4, 0x21, 0xca, 0x22, 0xf3, 0x8, 0x13, 0x6d, 0xba, 0xfe, 0xd1, 0xf0, 0xe1, 0x2, 0x29, 0x53, 0x5a, 0x42, 0xfe, 0xf1, 0x1f, 0x3c, 0xae, 0xbf, 0x5, 0x8e, 0x3e, 0x6b, 0x2, 0xb8, 0xc3, 0xdd, 0x39, 0xcc, 0x87, 0x90, 0x2, 0xa9, 0x88, 0x0, 0x75, 0xaf, 0xe3, 0x2b, 0xe6, 0xdf, 0x30, 0x5, 0x86, 0xea, 0x46, 0xd7, 0x42, 0xf0, 0x2a, 0x6c, 0x81, 0x63, 0x8c, 0x8a, 0x2, 0x27, 0x8, 0x5d, 0x70, 0x88, 0x2b, 0x27, 0xeb, 0x4, 0x66, 0x72, 0x96, 0xa1, 0xd0, 0xa6, 0x76, 0x68, 0xc1, 0xce, 0xd, 0x8e, 0xe4, 0x75, 0xe0, 0xf6, 0xba, 0xd6, 0xe2, 0x82, 0x10, 0xdf, 0xd1, 0xd8, 0x3c, 0x15, 0x2c, 0x6e, 0xb5, 0x8a, 0x1a, 0x6d, 0x35, 0x89, 0xef, 0x75, 0xee, 0x6d, 0x19, 0x44, 0x23, 0xb9, 0xeb, 0xc9, 0x19, 0xcc, 0x0, 0x3a, 0xbf, 0xf9, 0x89, 0xf7, 0x48, 0x7f, 0xca, 0xdd, 0x28, 0xc0, 0x45, 0x56, 0x1f, 0xef, 0x2a, 0x62, 0x7, 0x3c, 0xc8, 0x1e, 0xbc, 0x2f, 0xf4, 0x70, 0x11, 0x83, 0x1f, 0x58, 0xc3, 0xf9, 0xda, 0xc9, 0x80, 0xdf, 0x66, 0xda, 0x96, 0xe0, 0xa4, 0xfc, 0xaa, 0x66, 0xdf, 0xbe, 0xe8, 0x32, 0xdb, 0x1, 0x39, 0x55, 0x4c, 0xb3, 0xeb, 0x6, 0x35, 0xa4, 0x15, 0xcc, 0x25, 0xd6, 0x6, 0xff, 0x5c, 0x62, 0xc3, 0x53, 0x87, 0x1c, 0x68, 0x82, 0xc7, 0x52, 0x40, 0x8, 0x4c, 0x32, 0x73, 0x69, 0xa6, 0x5f, 0x30, 0xd0, 0xe8, 0xda, 0x9b, 0x2, 0x9, 0x90, 0x29, 0xb6, 0xb3, 0xa0, 0x2, 0x87, 0xc6, 0x5e, 0x65, 0xd1, 0x86, 0xb8, 0x9e, 0x15, 0xc4, 0x6b, 0xa6, 0xcf, 0x78, 0x5c, 0xe3, 0x44, 0x4f, 0x43, 0x11, 0x9a, 0x9d, 0x81, 0xe6, 0x50, 0x52, 0xa3, 0x24, 0xbc, 0x6, 0x5, 0x1d, 0x53, 0x6c, 0x4a, 0x40, 0x78, 0xae, 0x7c, 0x8c, 0x5, 0x62, 0xd3, 0x46, 0xf3, 0x3d, 0xa6, 0x88, 0x13, 0x65, 0x54, 0x5f, 0x97, 0x46, 0xbd, 0x34, 0xe5, 0x55, 0x56, 0xab, 0xfc, 0x7e, 0xd3, 0xc1, 0xed, 0x9f, 0x34, 0xf6, 0x96, 0x5e, 0x8d, 0x8f, 0xd4, 0x46, 0x4, 0x9f, 0x51, 0xea, 0x7e, 0x9f, 0x77, 0x6b, 0x7f, 0x5b, 0xed, 0xce, 0xaf, 0x5c, 0xad, 0x88, 0xe4, 0xef, 0x77, 0x84, 0x55, 0x45, 0x89, 0x24, 0x20, 0x49, 0x48, 0x87, 0x45, 0x8a, 0x7, 0x9a, 0xb9, 0xaf, 0x90, 0x65, 0x83, 0x6b, 0xe, 0x38, 0xfb, 0x79, 0xf6, 0x9, 0x3d, 0xcb, 0x35, 0x91, 0xca, 0x7e, 0x96, 0x1a, 0x29, 0x1e, 0x22, 0xa8, 0xdb, 0xb7, 0x20, 0x47, 0xe0, 0xf2, 0xa3, 0x96, 0x8, 0xf4, 0x9f, 0x6c, 0x57, 0x1c, 0x1a, 0xa6, 0x13, 0xdb, 0x13, 0xc0, 0x15, 0xb, 0xa3, 0x5b, 0xa, 0xa8, 0x38, 0xf1, 0xdb, 0xe9, 0x3d, 0x30, 0x8b, 0x6e, 0x1f, 0xd5, 0xa3, 0xac, 0xa6, 0x41, 0xf0, 0x3c, 0x4a, 0xd3, 0x94, 0x78, 0xd9, 0xe, 0x38, 0xe6, 0x39, 0xe8, 0xac, 0xf5, 0xd9, 0xa8, 0xd3, 0x85, 0xcf, 0xab, 0xf, 0xa4, 0x24, 0xd1, 0x47, 0xcd, 0x2b, 0x80, 0xca, 0x2d, 0x6b, 0x87, 0x4b, 0x9a, 0xe2, 0x82, 0xf3, 0x8a, 0xb, 0xfd, 0x36, 0x5c, 0x8f, 0x59, 0x4d, 0x9, 0xed, 0x46, 0xed, 0xc4, 0xba, 0xed, 0xb7, 0xfe, 0x6f, 0x55, 0xf8, 0xf7, 0x34, 0xc6, 0x7b, 0xa5, 0xc9, 0x37, 0xd3, 0xc8, 0xf4, 0xd, 0x1d, 0x1, 0xbc, 0xb4, 0x74, 0xed, 0xde, 0x2b, 0xd5, 0xe, 0x33, 0x6f, 0x2b, 0xbb, 0xd7, 0x51, 0x81, 0xed, 0xf, 0x24, 0xe7, 0xb6, 0x82, 0x3, 0x82, 0x8e, 0x5c, 0xa7, 0xb7, 0xc7, 0xc2, 0x2d, 0x76, 0xde, 0xfd, 0xd2, 0x8d, 0xf3, 0x88, 0xf4, 0x48, 0xa, 0xff, 0x1e, 0xc0, 0x45, 0x97, 0x11, 0x44, 0x43, 0x3a, 0x66, 0x5, 0x9, 0x7e, 0x73, 0x46, 0x59, 0x3d, 0xf2, 0x18, 0xd7, 0x57, 0x6c, 0x6f, 0x3, 0x46, 0xfe, 0xf, 0x29, 0xff, 0xc5, 0xab, 0x40, 0x6d, 0x97, 0x1c, 0xc7, 0x9, 0xbd, 0x58, 0x99, 0x4c, 0x51, 0xc4, 0x70, 0xed, 0x9f, 0xe7, 0x51, 0x31, 0x4, 0xff, 0x4f, 0x2f, 0xea, 0xf0, 0x98, 0xd4, 0x62, 0x1a, 0x7c, 0xa4, 0x6f, 0x7b, 0x46, 0xff, 0x26, 0xb6, 0xaf, 0x4, 0x0, 0xaa, 0x80, 0xbd, 0x11, 0xca, 0x85, 0x3b, 0x44, 0x91, 0xcd, 0x1b, 0x9f, 0x8d, 0x15, 0xa2, 0x89, 0x80, 0xad, 0x63, 0x8c, 0x41, 0x85, 0x50, 0x7b, 0xed, 0x68, 0x70, 0x5a, 0x4b, 0xf0, 0x57, 0xff, 0x46, 0x72, 0xbb, 0xf4, 0x86, 0x80, 0x8b, 0x75, 0xc1, 0xfe, 0xa4, 0x2a, 0x8, 0x65, 0xd9, 0x5a, 0x69, 0x86, 0xa2, 0x37, 0x91, 0xd8, 0x21, 0x8c, 0x68, 0x34, 0x5, 0x84, 0x9f, 0xdb, 0x35, 0xd7, 0x85, 0x3d, 0xfe, 0xec, 0xfa, 0x1a, 0xe3, 0x89, 0x1d, 0xf2, 0x7d, 0xe8, 0xac, 0x8e, 0x34, 0x56, 0x48, 0xf2, 0x43, 0xa3, 0x98, 0xc, 0xa0, 0xff, 0xbb, 0x24, 0x4d, 0xa7, 0x2c, 0x3f, 0xed, 0x24, 0xc5, 0x9c, 0x8c, 0x51, 0x22, 0x97, 0xa1, 0x2a, 0xac, 0xc9, 0xc8, 0x65, 0x1c, 0x84, 0xe, 0x1b, 0xb1, 0x5f, 0xd2, 0xad, 0xe4, 0xe3, 0x21, 0x16, 0xbc, 0x31, 0x5c, 0xef, 0x4b, 0xd4, 0x85, 0x96, 0xb9, 0xc2, 0x4b, 0x79, 0x6, 0xd4, 0x67, 0x99, 0xde, 0xe0, 0x38, 0xe8, 0xf2, 0xdd, 0xe5, 0x33, 0x30, 0xc4, 0xc6, 0xc1, 0xc5, 0x3d, 0x50, 0xa2, 0xef, 0xc6, 0xe4, 0xa1, 0xaa, 0x7e, 0xe0, 0x2, 0x17, 0x16, 0xdc, 0x8d, 0x22, 0xf4, 0x4d, 0x7f, 0x5e, 0xc2, 0x63, 0x8, 0x10, 0xff, 0xe1, 0x14, 0xc8, 0x14, 0x87, 0xf8, 0x74, 0x8e, 0x1, 0x4b, 0xf3, 0xbc, 0x46, 0x67, 0x9, 0x54, 0x2c, 0x82, 0x92, 0xa2, 0x47, 0xf6, 0x36, 0xc3, 0x16, 0x1e, 0xac, 0x37, 0x22, 0x3, 0x20, 0x75, 0x8c, 0xf0, 0x48, 0xe0, 0x70, 0x54, 0x24, 0xbe, 0xa5, 0x39, 0x5d, 0x3e, 0x57, 0xcb, 0x35, 0x8a, 0x35, 0xb8, 0x7b, 0x83, 0xa9, 0x4c, 0x7a, 0xe, 0xca, 0xb4, 0x90, 0x2a, 0xd3, 0xf4, 0xf0, 0x93, 0x9b, 0xba, 0xb9, 0x68, 0x54, 0xf5, 0x21, 0xf5, 0x44, 0x95, 0x2e, 0xbd, 0x20, 0x2, 0xe3, 0xe6, 0xaf, 0x2f, 0x8b, 0x84, 0xa, 0x7c, 0x8d, 0xa6, 0x84, 0x80, 0xac, 0x9e, 0x3a, 0x65, 0x2b, 0x8, 0x26, 0x48, 0xa0, 0xdb, 0x54, 0xca, 0x9c, 0x69, 0x4e, 0xd0, 0x0, 0x91, 0xc8, 0x2e, 0xe0, 0x75, 0xc0, 0x9c, 0xd9, 0x9, 0xd5, 0x34, 0xff, 0xc, 0x15, 0x2c, 0xf0, 0xa9, 0xf4, 0xa4, 0x14, 0xca, 0x1a, 0x33, 0x32, 0x8a, 0x45, 0x5c, 0xa7, 0x3, 0x97, 0x68, 0xc0, 0x8d, 0xe6, 0x28, 0x18, 0x8d, 0xe7, 0x5b, 0x86, 0x9a, 0x1d, 0xfb, 0xec, 0x15, 0xc3, 0xd8, 0x2f, 0x91, 0x3, 0xfd, 0x56, 0xc3, 0xa, 0x68, 0x71, 0xc7, 0x98, 0x5, 0x4b, 0xcd, 0x6b, 0x19, 0x9c, 0xbe, 0x81, 0x22, 0x7d, 0xf1, 0xbd, 0x9e, 0xf4, 0x6, 0x93, 0xa5, 0xfd, 0x30, 0xaf, 0xc7, 0x42, 0x3d, 0x6f, 0xee, 0x18, 0x48, 0xea, 0x1f, 0xd5, 0xd6, 0xe2, 0xe6, 0xb9, 0x75, 0xd1, 0x7c, 0x2d, 0x99, 0x81, 0x37, 0xce, 0x7d, 0x3e, 0x38, 0xb7, 0xa8, 0x16, 0xe3, 0x8a, 0x68, 0x56, 0xe6, 0x4f, 0xed, 0x34, 0xb3, 0x6f, 0x4b, 0x84, 0x10, 0x9d, 0xd1, 0x1e, 0x99, 0x47, 0x1b, 0xb3, 0x59, 0xa, 0x7c, 0x32, 0xd0, 0x3d, 0x6d, 0x5c, 0x58, 0xb4, 0xae, 0x2b, 0x22, 0x7b, 0x63, 0x36, 0x80, 0x20, 0xd6, 0x3d, 0x1b, 0x5a, 0x45, 0x62, 0x92, 0xc1, 0x5f, 0x11, 0xdc, 0xbf, 0xb1, 0x27, 0xf6, 0xeb, 0x5b, 0x3d, 0x39, 0x7f, 0xbd, 0x4c, 0xc2, 0xac, 0x6, 0xe8, 0x21, 0x34, 0x8b, 0xb0, 0x4c, 0x34, 0xf2, 0x53, 0xbf, 0x77, 0x40, 0x3c, 0x76, 0xae, 0xd4, 0xd6, 0x12, 0xfd, 0x5f, 0x34, 0xc3, 0xdf, 0x4f, 0xb0, 0x87, 0x52, 0xce, 0xdc, 0xdc, 0xa2, 0xbf, 0x9c, 0xef, 0xa7, 0x42, 0x48, 0x4d, 0xcc, 0x69, 0x43, 0xc5, 0x31, 0x44, 0xc5, 0x23, 0xed, 0xcf, 0xa8, 0x7f, 0xed, 0x62, 0x28, 0x31, 0x61, 0x25, 0x0, 0x88, 0xe8, 0xb1, 0xcb, 0x1e, 0xa8, 0x53, 0xd1, 0x3b, 0x31, 0xbd, 0xa4, 0xb2, 0xa8, 0xc1, 0xf8, 0x13, 0xad, 0x81, 0x7, 0xfb, 0x69, 0xc, 0xf7, 0x0, 0x96, 0x57, 0xf3, 0xa, 0x73, 0xe2, 0x62, 0x2f, 0x40, 0xcd, 0x96, 0xdb, 0x2, 0x4, 0x5f, 0xfa, 0x9d, 0x9f, 0x61, 0x1d, 0x6a, 0xb5, 0x40, 0x35, 0xd3, 0x8d, 0x4a, 0xb1, 0x68, 0x90, 0xa2, 0x3, 0x9d, 0xbf, 0x95, 0x58, 0x14, 0x1f, 0xf6, 0x2e, 0xae, 0xee, 0x1d, 0xc, 0x4c, 0xa1, 0x55, 0xc5, 0xc4, 0x8b, 0xc1, 0xff, 0x9d, 0x65, 0x65, 0xef, 0xc4, 0x48, 0x1c, 0x77, 0x37, 0xf2, 0x79, 0x9b, 0xfb, 0xcf, 0x69, 0xcc, 0xe7, 0xac, 0x1d, 0x70, 0x88, 0x8a, 0xfe, 0xe6, 0xc2, 0x5b, 0xba, 0xa9, 0x70, 0x16, 0x5b, 0xa0, 0x4e, 0x1f, 0x7b, 0xbd, 0x55, 0x61, 0xd8, 0xb2, 0xd, 0x8e, 0x53, 0x9c, 0x14, 0xaf, 0xae, 0xf1, 0x53, 0x44, 0xdd, 0xf2, 0x48, 0xca, 0x1, 0x66, 0x54, 0x92, 0x76, 0xc5, 0x9, 0x68, 0xcc, 0xa3, 0x32, 0x93, 0x92, 0xbb, 0xb9, 0x5, 0x23, 0x1c, 0xb, 0xfe, 0xab, 0xa3, 0x52, 0xd3, 0x9d, 0xa1, 0xa2, 0x3b, 0x8a, 0xa, 0x66, 0x1, 0x23, 0x6a, 0xe1, 0xf2, 0x8c, 0xbd, 0xec, 0x3a, 0x79, 0xd1, 0x1d, 0xf8, 0x7c, 0x49, 0x2e, 0xf0, 0x26, 0x3e, 0xc4, 0x71, 0x6a, 0x3c, 0x7a, 0x4d, 0x9d, 0xbe, 0x9, 0x7c, 0xe0, 0x63, 0x61, 0xd4, 0xbc, 0xce, 0xfd, 0x42, 0x64, 0x76, 0x93, 0xca, 0x15, 0x69, 0xf7, 0x5c, 0xfa, 0x1c, 0x7a, 0x3e, 0x86, 0x53, 0xd, 0x19, 0xab, 0x94, 0x42, 0xf, 0xe4, 0xb9, 0xf5, 0xdf, 0xa6, 0x1, 0x78, 0xab, 0x9d, 0xe8, 0x85, 0x90, 0x3e, 0xd2, 0x98, 0xa5, 0xae, 0x84, 0x88, 0xd6, 0x0, 0x5a, 0xe7, 0xf9, 0x63, 0x2c, 0x11, 0xaf, 0xc3, 0xb8, 0x7a, 0x1d, 0x76, 0x59, 0x54, 0xeb, 0xc0, 0x49, 0x19, 0xc5, 0xa6, 0x14, 0x3, 0x60, 0xd9, 0x7d, 0x8, 0xdb, 0xec, 0xc2, 0x52, 0xa0, 0x3a, 0x3b, 0x4, 0x1d, 0x4c, 0xa2, 0x53, 0x20, 0xff, 0x5a, 0x33, 0xce, 0xd, 0x6e, 0x1c, 0x9f, 0xbe, 0x71, 0x7, 0xe4, 0x36, 0x43, 0xe3, 0xf5, 0xe6, 0x99, 0x1e, 0x8e, 0x2c, 0xcc, 0xf, 0xa0, 0xca, 0x93, 0x9, 0x32, 0xc1, 0x8a, 0x81, 0x34, 0x98, 0xba, 0xbc, 0xe6, 0x74, 0xa7, 0xd8, 0x98, 0x6f, 0x2f, 0x99, 0xa9, 0x2, 0x23, 0x9f, 0x8f, 0x9a, 0xf2, 0xae, 0xe6, 0xe8, 0x4c, 0x80, 0xc4, 0xbc, 0xac, 0xd2, 0xa, 0x2c, 0xea, 0xac, 0x5f, 0xa2, 0x9c, 0x6b, 0xa2, 0xd5, 0x2e, 0x21, 0x42, 0xa6, 0x4f, 0xa9, 0xb2, 0x60, 0x6, 0xd2, 0x41, 0x4c, 0xb, 0xdc, 0x72, 0x61, 0xdc, 0x28, 0xfb, 0x17, 0x99, 0x6a, 0xa1, 0x30, 0xc8, 0xe8, 0xf5, 0xfb, 0xc9, 0xf2, 0xda, 0xe8, 0x72, 0x87, 0x1f, 0xc6, 0x15, 0xf4, 0x6c, 0xf6, 0xdc, 0xf2, 0xc3, 0xc3}
	if gotSig := par.htSign(msg, skSeed, pkSeed, idxTree, idxLeaf); !slices.Equal(gotSig, wantSig) {
		t.Errorf("par.htSign() = %v, want %v", gotSig, wantSig)
	}
}

func TestHypertreeSignVerify(t *testing.T) {
	par := SLH_DSA_SHA2_128s
	var msg [16]byte
	rand.Read(msg[:])
	skSeed := []byte{0x8e, 0x37, 0x36, 0x28, 0x86, 0x12, 0xfb, 0xd, 0xff, 0xd5, 0xa6, 0xa1, 0x6b, 0xe0, 0x89, 0xfb}
	pkSeed := []byte{0xa9, 0xb5, 0xe3, 0xd9, 0x99, 0xf8, 0x18, 0xc3, 0x41, 0x51, 0x56, 0x92, 0x61, 0xe8, 0x39, 0x2b}
	idxTree := uint64(0xe0d7a2b48b675)
	idxLeaf := uint32(0x1c3)
	pkRoot := []byte{0xf3, 0x39, 0x8c, 0x44, 0x4d, 0xa3, 0xc6, 0xfc, 0x62, 0x5b, 0x86, 0x35, 0x98, 0xd3, 0xd7, 0x6a}
	sig := par.htSign(msg[:], skSeed, pkSeed, idxTree, idxLeaf)
	if !par.htVerify(msg[:], sig, pkSeed, idxTree, idxLeaf, pkRoot) {
		t.Fatalf("par.htVerify() failed")
	}
	sig[0] ^= 1 // Corrupt the signature.
	if par.htVerify(msg[:], sig, pkSeed, idxTree, idxLeaf, pkRoot) {
		t.Errorf("par.htVerify() did not fail with invalid signature")
	}
}
