#!/usr/bin/env python

"""
Mail preparation support.

Copyright (C) 2014, 2015, 2016, 2017 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from imiptools.config import settings
from email.mime.message import MIMEMessage
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from smtplib import LMTP, SMTP

LOCAL_PREFIX = settings["LOCAL_PREFIX"]
MESSAGE_SENDER = settings["MESSAGE_SENDER"]
OUTGOING_PREFIX = settings["OUTGOING_PREFIX"]

# Fake gettext function for strings to be translated later.

_ = lambda s: s

MESSAGE_SUBJECT = _("Calendar system message")

PREAMBLE_TEXT = _("""\
This message contains several different parts, one of which will contain
calendar information that will only be understood by a suitable program.
""")

class Messenger:

    "Sending of outgoing messages."

    def __init__(self, lmtp_socket=None, local_smtp=False, sender=None, subject=None, preamble_text=None):

        """
        Deliver to a local mail system using LMTP if 'lmtp_socket' is provided
        or if 'local_smtp' is given as a true value.
        """

        self.lmtp_socket = lmtp_socket
        self.local_smtp = local_smtp
        self.sender = sender or MESSAGE_SENDER
        self.subject = subject
        self.preamble_text = preamble_text

        # The translation method is set by the client once locale information is
        # known.

        self.gettext = None

    def local_delivery(self):

        "Return whether local delivery is performed using this messenger."

        return self.lmtp_socket is not None or self.local_smtp

    def sendmail(self, recipients, data, sender=None, outgoing_bcc=None):

        """
        Send a mail to the given 'recipients' consisting of the given 'data',
        using the given 'sender' identity if indicated, indicating an
        'outgoing_bcc' identity if indicated.

        The 'outgoing_bcc' argument is required when sending on behalf of a user
        from the calendar@domain address, since this will not be detected as a
        valid participant and handled using the outgoing transport.
        """

        if self.lmtp_socket:
            smtp = LMTP(self.lmtp_socket)
        else:
            smtp = SMTP("localhost")

        if outgoing_bcc:
            recipients = list(recipients) + ["%s+%s" % (OUTGOING_PREFIX, outgoing_bcc)]
        elif self.local_smtp:
            recipients = [self.make_local(recipient) for recipient in recipients]

        smtp.sendmail(sender or self.sender, recipients, data)
        smtp.quit()

    def make_local(self, recipient):

        """
        Make the 'recipient' an address for local delivery. For this to function
        correctly, a virtual alias or equivalent must be defined for addresses
        of the following form:

        local+NAME@DOMAIN

        Such aliases should direct delivery to the local recipient.
        """

        parts = recipient.split("+", 1)
        return "%s+%s" % (LOCAL_PREFIX, parts[-1])

    def make_outgoing_message(self, parts, recipients, sender=None, outgoing_bcc=None):

        """
        Make a message from the given 'parts' for the given 'recipients', using
        the given 'sender' identity if indicated, indicating an 'outgoing_bcc'
        identity if indicated.
        """

        message = self._make_summary_for_parts(parts)

        message["From"] = sender or self.sender
        for recipient in recipients:
            message["To"] = recipient
        if outgoing_bcc:
            message["Bcc"] = "%s+%s" % (OUTGOING_PREFIX, outgoing_bcc)
        message["Subject"] = self.subject or \
            self.gettext and self.gettext(MESSAGE_SUBJECT) or MESSAGE_SUBJECT

        return message

    def make_summary_message(self, msg, parts):

        """
        Return a simple summary using details from 'msg' and the given 'parts'.
        Information messages provided amongst the parts by the handlers will be
        merged into the preamble so that mail programs will show them
        immediately.
        """

        message = self._make_summary_for_parts(parts, True)
        self._copy_headers(message, msg)
        return message

    def wrap_message(self, msg, parts):

        """
        Wrap 'msg' and provide the given 'parts' as the primary content.
        Information messages provided amongst the parts by the handlers will be
        merged into the preamble so that mail programs will show them
        immediately.
        """

        message = self._make_container_for_parts(parts, True)
        payload = message.get_payload()
        payload.append(MIMEMessage(msg))
        self._copy_headers(message, msg)
        return message

    def _make_summary_for_parts(self, parts, merge=False):

        """
        Return a simple summary for the given 'parts', merging information parts if
        'merge' is specified and set to a true value.
        """

        if len(parts) == 1:
            return parts[0]
        else:
            return self._make_container_for_parts(parts, merge)

    def _make_container_for_parts(self, parts, merge=False):

        """
        Return a container for the given 'parts', merging information parts if
        'merge' is specified and set to a true value.
        """

        # Merge calendar information if requested.

        if merge:
            info, parts = self._merge_calendar_info_parts(parts)
        else:
            info = []

        # Insert a preamble message before any calendar information messages.

        info.insert(0, self.preamble_text or
                       self.gettext and self.gettext(PREAMBLE_TEXT) or PREAMBLE_TEXT)

        message = MIMEMultipart("mixed", _subparts=parts)
        message.preamble = "\n\n".join(info)
        return message

    def _merge_calendar_info_parts(self, parts):

        """
        Return a collection of plain text calendar information messages from
        'parts', together with a collection of the remaining parts.
        """

        info = []
        remaining = []

        for part in parts:

            # Attempt to acquire informational messages.

            if part.get("X-IMIP-Agent") == "info":

                # Ignore the preamble of any multipart message and just
                # collect its parts.

                if part.is_multipart():
                    i, r = self._merge_calendar_info_parts(part.get_payload())
                    remaining += r

                # Obtain any single-part messages.

                else:
                    info.append(part.get_payload(decode=True))

            # Accumulate other parts regardless of their purpose.

            else:
                remaining.append(part)

        return info, remaining

    def _copy_headers(self, message, msg):

        "Copy to 'message' certain headers from 'msg'."

        message["From"] = msg["From"]
        message["To"] = msg["To"]
        message["Subject"] = msg["Subject"]

# vim: tabstop=4 expandtab shiftwidth=4
