/*
	File                 : nsl_pcm.c
	Project              : LabPlot
	Description          : Constants for process monitoring and control
	--------------------------------------------------------------------
	SPDX-FileCopyrightText: 2024 Alexander Semke <alexander.semke@web.de>
	SPDX-License-Identifier: GPL-2.0-or-later
*/

#include "nsl_pcm.h"
#include "nsl_common.h"
#include <gsl/gsl_math.h>
#include <gsl/gsl_sf_gamma.h>

//*************************************************************
//*********** Scaling factors for charts for Ranges ***********
//*************************************************************

double nsl_pcm_D3(unsigned int n) {
	const double d2 = nsl_pcm_d2(n);
	if (d2 != 0) {
		const double d3 = nsl_pcm_d3(n);
		return 1 - 3 * d3 / d2;
	} else
		return 0.;
}

double nsl_pcm_D4(unsigned int n) {
	const double d2 = nsl_pcm_d2(n);
	if (d2 != 0.) {
		const double d3 = nsl_pcm_d3(n);
		return 1 + 3 * d3 / d2;
	} else
		return 0.;
}

double nsl_pcm_D5(unsigned int n) {
	const double d4 = nsl_pcm_d4(n);
	if (d4 != 0.) {
		const double d2 = nsl_pcm_d2(n);
		const double d3 = nsl_pcm_d3(n);
		return (d2 - 3 * d3) / d4;
	} else
		return 0.;
}

double nsl_pcm_D6(unsigned int n) {
	const double d4 = nsl_pcm_d4(n);
	if (d4 != 0.) {
		const double d2 = nsl_pcm_d2(n);
		const double d3 = nsl_pcm_d3(n);
		return (d2 + 3 * d3) / d4;
	} else
		return 0.;
}

//*************************************************************
//********* Scaling factors for charts for Averages ***********
//*************************************************************
double nsl_pcm_A2(unsigned int n) {
	const double d2 = nsl_pcm_d2(n);
	if (d2 != 0. && n != 0)
		return 3 / d2 / sqrt(n);
	else
		return 0.;
}

double nsl_pcm_A3(unsigned int n) {
	const double c4 = nsl_pcm_c4(n);
	if (c4 != 0. && n != 0)
		return 3. / c4 / sqrt(n);
	else
		return 0.;
}

double nsl_pcm_A4(unsigned int n) {
	const double d4 = nsl_pcm_d4(n);
	if (d4 != 0 && n != 0)
		return 3. / d4 / sqrt(n);
	else
		return 0.;
}

//*************************************************************
//**** Scaling factors for charts for Standard Deviations *****
//*************************************************************
double nsl_pcm_B3(unsigned int n) {
	const double c4 = nsl_pcm_c4(n);
	if (c4 != 0.)
		return 1 - 3 / c4 * sqrt(1 - pow(c4, 2));
	else
		return 0.;
}

double nsl_pcm_B4(unsigned int n) {
	const double c4 = nsl_pcm_c4(n);
	if (c4 != 0.)
		return 1 + 3 / c4 * sqrt(1 - pow(c4, 2));
	else
		return 0.;
}

double nsl_pcm_B5(unsigned int n) {
	const double c4 = nsl_pcm_c4(n);
	return c4 - 3 * sqrt(1 - pow(c4, 2));
}

double nsl_pcm_B6(unsigned int n) {
	const double c4 = nsl_pcm_c4(n);
	return c4 + 3 * sqrt(1 - pow(c4, 2));
}

//*************************************************************
//************************** Scale factors ********************
//*************************************************************
/*
Mathematica code to compute d2, d3 and d4:
a[x1_] := 1/2 (1+Erf[x1/Sqrt[2]])
d2[n_] := NIntegrate[1-(1-a[y])^n-a[y]^n, {y, -Infinity, Infinity}]
d3[n_] := Sqrt[2 NIntegrate[1-(1-a[z])^n-a[y]^n+(a[y]-a[z])^n, {y, -Infinity, Infinity}, {z, -Infinity, y}] - d2[n]^2]

f[x_, w_] := Integrate[Exp[-u^2/2], {u, x, x+w}]
ff[w_?NumericQ, n_?NumericQ] := n(n-1)NIntegrate[f[x,w]^(n-2)Exp[-(x+w)^2/2]Exp[-x^2/2], {x, -Infinity, Infinity}]/(Sqrt[2\[Pi]])^n
Integ[n_?NumericQ, d4_?NumericQ] := NIntegrate[ff[y, n], {y, 0, d4}]
d4f[n_?NumberQ] := Abs[d4]/. FindRoot[Integ[n, Abs[d4]] == 0.5, {d4, Log[n]}]

Export["factors.dat", Table[{n, d2[n], d3[n], d4f[n]}, {n, 2, 100}]]
*/

// Precomputed d2 values for subgroup sizes from 2 to 100
static const double d2_values[] = {
	0.0, // Placeholder for index 0 (not used)
	0.0, // Placeholder for index 1 (not used)
	1.128379167095197, // n = 2
	1.6925687506427953, // n = 3
	2.058750746008131, // n = 4
	2.325928947280891, // n = 5
	2.5344127212223113, // n = 6
	2.7043567512130715, // n = 7
	2.84720061208971, // n = 8
	2.970026324417524, // n = 9
	3.0775054616692854, // n = 10
	3.172872703795761, // n = 11
	3.2584552797180555, // n = 12
	3.335980354037362, // n = 13
	3.406763108130005, // n = 14
	3.471826889880469, // n = 15
	3.5319827861078563, // n = 16
	3.5878839617635503, // n = 17
	3.640063757935497, // n = 18
	3.688963023205587, // n = 19
	3.73495011959446, // n = 20
	3.7783358298403242, // n = 21
	3.819384643360417, // n = 22
	3.8583234232824726, // n = 23
	3.8953481484487007, // n = 24
	3.9306292195043384, // n = 25
	3.9643156795197294, // n = 26
	3.9965386040101434, // n = 27
	4.0274138482434, // n = 28
	4.057044292091938, // n = 29
	4.085521688339652, // n = 30
	4.112928195272904, // n = 31
	4.139337655854215, // n = 32
	4.164816671936556, // n = 33
	4.189425511533138, // n = 34
	4.213218879203962, // n = 35
	4.236246573508922, // n = 36
	4.258554050742269, // n = 37
	4.280182910466118, // n = 38
	4.301171315453116, // n = 39
	4.3215543563455086, // n = 40
	4.341364369502283, // n = 41
	4.360631215034061, // n = 42
	4.379382520837778, // n = 43
	4.397643897479931, // n = 44
	4.41543912799172, // n = 45
	4.432790335995703, // n = 46
	4.449718135053523, // n = 47
	4.4662417616860175, // n = 48
	4.482379193968969, // n = 49
	4.498147258564613, // n = 50
	4.5135617250934565, // n = 51
	4.528637392564921, // n = 52
	4.5433881665252445, // n = 53
	4.55782712886117, // n = 54
	4.571966600960647, // n = 55
	4.585818200969639, // n = 56
	4.599392895793089, // n = 57
	4.612701048405993, // n = 58
	4.625752460971057, // n = 59
	4.638556414199233, // n = 60
	4.651121703337651, // n = 61
	4.663456671124449, // n = 62
	4.675569238010803, // n = 63
	4.68746692991663, // n = 64
	4.6991569037568315, // n = 65
	4.710645970946825, // n = 66
	4.721940619080131, // n = 67
	4.733047031939316, // n = 68
	4.743971107995398, // n = 69
	4.7547184775284626, // n = 70
	4.7652945184907445, // n = 71
	4.775704371220947, // n = 72
	4.785952952107797, // n = 73
	4.796044966291241, // n = 74
	4.805984919481121, // n = 75
	4.8157771289656885, // n = 76
	4.825425733875301, // n = 77
	4.834934704760835, // n = 78
	4.844307852540693, // n = 79
	4.853548836865534, // n = 80
	4.862661173945435, // n = 81
	4.871648243880189, // n = 82
	4.880513297530069, // n = 83
	4.889259462960919, // n = 84
	4.897889751494851, // n = 85
	4.906407063394893, // n = 86
	4.914814193209873, // n = 87
	4.923113834803415, // n = 88
	4.931308586089161, // n = 89
	4.939400953492473, // n = 90
	4.947393356157264, // n = 91
	4.95528812991519, // n = 92
	4.963087531033057, // n = 93
	4.970793739753079, // n = 94
	4.978408863639524, // n = 95
	4.985934940744299, // n = 96
	4.993373942602997, // n = 97
	5.000727777072205, // n = 98
	5.007998291017959, // n = 99
	5.015187272864651, // n = 100
};

// Precomputed d3 values for subgroup sizes from 2 to 100
static const double d3_values[] = {
	0.0, // Placeholder for index 0 (not used)
	0.0, // Placeholder for index 1 (not used)
	0.8525024593148771, // n = 2
	0.8883679999626635, // n = 3
	0.8798081990580747, // n = 4
	0.8640819435612683, // n = 5
	0.8480397420352023, // n = 6
	0.8332053582614541, // n = 7
	0.8198314978366432, // n = 8
	0.8078342899243331, // n = 9
	0.7970507564359406, // n = 10
	0.7873146270099187, // n = 11
	0.7784782958912122, // n = 12
	0.7704161921986292, // n = 13
	0.7630230552672753, // n = 14
	0.7562114294281701, // n = 15
	0.7499080263230599, // n = 16
	0.7440517434080416, // n = 17
	0.7385908764470441, // n = 18
	0.7334815288697892, // n = 19
	0.7286863760848462, // n = 20
	0.7241733633893577, // n = 21
	0.7199148998128939, // n = 22
	0.7158867570622821, // n = 23
	0.7120681564913107, // n = 24
	0.7084407214471241, // n = 25
	0.7049883264785365, // n = 26
	0.7016964756023472, // n = 27
	0.6985527795371705, // n = 28
	0.6955453158281097, // n = 29
	0.692665061732523, // n = 30
	0.6899019064406943, // n = 31
	0.6872479286394707, // n = 32
	0.6846958337783061, // n = 33
	0.6822373486289512, // n = 34
	0.6798707626895025, // n = 35
	0.6775862795646065, // n = 36
	0.6753799959648209, // n = 37
	0.6732475497714754, // n = 38
	0.6711845260181102, // n = 39
	0.6691871257929164, // n = 40
	0.6672519361040411, // n = 41
	0.6653748115619257, // n = 42
	0.6635534263861155, // n = 43
	0.6617847536018923, // n = 44
	0.6600661320816961, // n = 45
	0.6583950512389808, // n = 46
	0.6567691819817087, // n = 47
	0.6551864293109941, // n = 48
	0.6536448456347774, // n = 49
	0.6521425778346468, // n = 50
	0.6506776880771042, // n = 51
	0.6492478713029549, // n = 52
	0.647854113318744, // n = 53
	0.646492393284731, // n = 54
	0.6451621923677606, // n = 55
	0.6438622335666184, // n = 56
	0.6425913475268711, // n = 57
	0.6413483247243467, // n = 58
	0.6401313277093479, // n = 59
	0.6389418390415028, // n = 60
	0.6377762118588484, // n = 61
	0.6366346212222858, // n = 62
	0.6355159506164537, // n = 63
	0.634419710113454, // n = 64
	0.633346356766867, // n = 65
	0.6322922457718249, // n = 66
	0.6312581870437196, // n = 67
	0.6302419128062672, // n = 68
	0.6292458826904189, // n = 69
	0.6282676601686616, // n = 70
	0.6273075248463464, // n = 71
	0.6263641385835967, // n = 72
	0.6254371069518693, // n = 73
	0.6245259486675449, // n = 74
	0.6236302932391317, // n = 75
	0.6227496519921646, // n = 76
	0.6218835054743108, // n = 77
	0.6210314466576278, // n = 78
	0.6201929920791063, // n = 79
	0.6193681171464972, // n = 80
	0.6185559818048052, // n = 81
	0.6177565412438544, // n = 82
	0.6169692325884243, // n = 83
	0.6161939106691174, // n = 84
	0.6154300339860344, // n = 85
	0.6146774917434993, // n = 86
	0.6139359707804884, // n = 87
	0.613205177897005, // n = 88
	0.6124847828628188, // n = 89
	0.6117743785538406, // n = 90
	0.6110742108727337, // n = 91
	0.6103834999411003, // n = 92
	0.6097022507380908, // n = 93
	0.6090301470125982, // n = 94
	0.6083669237574164, // n = 95
	0.6077124972088769, // n = 96
	0.6070672771163781, // n = 97
	0.6064299472286251, // n = 98
	0.6058005907023666, // n = 99
	0.6051791909962189 // n = 100
};

// Precomputed d4 values for subgroup sizes from 2 to 100
static const double d4_values[] = {
	0.0, // Placeholder for index 0 (not used)
	0.0, // Placeholder for index 1 (not used)
	0.9538725532193055, // d4 for n = 2
	1.5877877504022864, // d4 for n = 3
	1.9783204855470817, // d4 for n = 4
	2.256882493008741, // d4 for n = 5
	2.471651604798986, // d4 for n = 6
	2.645452036662346, // d4 for n = 7
	2.7908407587807083, // d4 for n = 8
	2.9154379134007966, // d4 for n = 9
	3.024201573065575, // d4 for n = 10
	3.1205308924267423, // d4 for n = 11
	3.206853249924588, // d4 for n = 12
	3.2849597396428334, // d4 for n = 13
	3.3562079660525823, // d4 for n = 14
	3.421650354969399, // d4 for n = 15
	3.482118481122798, // d4 for n = 16
	3.538280289849273, // d4 for n = 17
	3.590680006456775, // d4 for n = 18
	3.6397666406156217, // d4 for n = 19
	3.6859147717940353, // d4 for n = 20
	3.729439987285306, // d4 for n = 21
	3.7706105366621876, // d4 for n = 22
	3.8096562600362365, // d4 for n = 23
	3.8467755171286946, // d4 for n = 24
	3.8821406334603856, // d4 for n = 25
	3.9159022173092843, // d4 for n = 26
	3.94819263117792, // d4 for n = 27
	3.979128794518547, // d4 for n = 28
	4.008814473475817, // d4 for n = 29
	4.037342163587272, // d4 for n = 30
	4.064794649997647, // d4 for n = 31
	4.091246305116032, // d4 for n = 32
	4.1167641826239665, // d4 for n = 33
	4.141408939310546, // d4 for n = 34
	4.165235611762549, // d4 for n = 35
	4.188294285948355, // d4 for n = 36
	4.210630667338881, // d4 for n = 37
	4.232286572342689, // d4 for n = 38
	4.253300353115184, // d4 for n = 39
	4.273707266257725, // d4 for n = 40
	4.293539794026014, // d4 for n = 41
	4.3128279251852275, // d4 for n = 42
	4.331599401408163, // d4 for n = 43
	4.349879934156739, // d4 for n = 44
	4.3676933961864775, // d4 for n = 45
	4.385061991089481, // d4 for n = 46
	4.402006403886551, // d4 for n = 47
	4.418545935102548, // d4 for n = 48
	4.434698620470257, // d4 for n = 49
	4.450481337921391, // d4 for n = 50
	4.465909903882364, // d4 for n = 51
	4.480999159391772, // d4 for n = 52
	4.49576304812399, // d4 for n = 53
	4.510214686420372, // d4 for n = 54
	4.524366434946402, // d4 for n = 55
	4.538229916610786, // d4 for n = 56
	4.551816148046175, // d4 for n = 57
	4.565135507973099, // d4 for n = 58
	4.578197819729887, // d4 for n = 59
	4.591012384997295, // d4 for n = 60
	4.603588016624896, // d4 for n = 61
	4.615933074550778, // d4 for n = 62
	4.628055493587272, // d4 for n = 63
	4.6399628142123746, // d4 for n = 64
	4.651662206483535, // d4 for n = 65
	4.663160493714933, // d4 for n = 66
	4.674464174728598, // d4 for n = 67
	4.68557944351571, // d4 for n = 68
	4.696512208067015, // d4 for n = 69
	4.707268107439607, // d4 for n = 70
	4.717852529334247, // d4 for n = 71
	4.728270617178354, // d4 for n = 72
	4.738527298986588, // d4 for n = 73
	4.748627284941135, // d4 for n = 74
	4.758575086880681, // d4 for n = 75
	4.7683750279711035, // d4 for n = 76
	4.778031252484882, // d4 for n = 77
	4.787547736105561, // d4 for n = 78
	4.7969282944459355, // d4 for n = 79
	4.806176591515496, // d4 for n = 80
	4.8152961476374525, // d4 for n = 81
	4.824290346689857, // d4 for n = 82
	4.83316244333441, // d4 for n = 83
	4.841915568910414, // d4 for n = 84
	4.8505527380334525, // d4 for n = 85
	4.859076853938725, // d4 for n = 86
	4.867490714019012, // d4 for n = 87
	4.875797014819841, // d4 for n = 88
	4.8839983567614755, // d4 for n = 89
	4.892097248711164, // d4 for n = 90
	4.900096112044636, // d4 for n = 91
	4.907997284730678, // d4 for n = 92
	4.915803025043119, // d4 for n = 93
	4.923515515173285, // d4 for n = 94
	4.931136864486147, // d4 for n = 95
	4.938669112757336, // d4 for n = 96
	4.946114233148186, // d4 for n = 97
	4.953474135091711, // d4 for n = 98
	4.960750666918383, // d4 for n = 99
	4.967945618451404, // d4 for n = 100
};

double nsl_pcm_d2(unsigned int n) {
	if (n > 100) // outside the precomputed range
		return 0.0;

	return d2_values[n];
}

double nsl_pcm_d3(unsigned int n) {
	if (n > 100) // outside the precomputed range
		return 0.0;

	return d3_values[n];
}

double nsl_pcm_d4(unsigned int n) {
	if (n > 25) // outside the precomputed range
		return 0.0;

	return d4_values[n];
}

double nsl_pcm_c4(unsigned int n) {
	return sqrt(2. / (n - 1)) * gsl_sf_gamma((n - 2.) / 2. + 1) / gsl_sf_gamma((n - 3.) / 2. + 1);
}
