/*
 * Copyright (c) 2008 INRIA
 *
 * SPDX-License-Identifier: GPL-2.0-only
 *
 * Author: Mathieu Lacage <mathieu.lacage@sophia.inria.fr>
 */

#ifndef YANS_WIFI_HELPER_H
#define YANS_WIFI_HELPER_H

#include "wifi-helper.h"

#include "ns3/yans-wifi-channel.h"

namespace ns3
{

/**
 * \brief manage and create wifi channel objects for the YANS model.
 *
 * The intent of this class is to make it easy to create a channel object
 * which implements the YANS channel model. The YANS channel model is described
 * in "Yet Another Network Simulator"; an author-prepared version of this paper  * is at:
 * https://hal.inria.fr/file/index/docid/78318/filename/yans-rr.pdf
 */
class YansWifiChannelHelper
{
  public:
    /**
     * Create a channel helper without any parameter set. The user must set
     * them all to be able to call Create later.
     */
    YansWifiChannelHelper();

    /**
     * Create a channel helper in a default working state. By default, we create
     * a channel model with a propagation delay equal to a constant, the speed of light,
     * and a propagation loss based on a log distance model with a reference loss of 46.6777 dB
     * at reference distance of 1m.
     * \returns YansWifiChannelHelper
     */
    static YansWifiChannelHelper Default();

    /**
     * \tparam Ts \deduced Argument types
     * \param name the name of the model to add
     * \param [in] args Name and AttributeValue pairs to set.
     *
     * Add a propagation loss model to the set of currently-configured loss models.
     * This method is additive to allow you to construct complex propagation loss models
     * such as a log distance + Jakes model, etc.
     *
     * The order in which PropagationLossModels are added may be significant. Some
     * propagation models are dependent of the "txPower" (e.g. Nakagami model), and
     * are therefore not commutative. The final receive power (excluding receiver
     * gains) are calculated in the order the models are added.
     */
    template <typename... Ts>
    void AddPropagationLoss(std::string name, Ts&&... args);
    /**
     * \tparam Ts \deduced Argument types
     * \param name the name of the model to set
     * \param [in] args Name and AttributeValue pairs to set.
     *
     * Configure a propagation delay for this channel.
     */
    template <typename... Ts>
    void SetPropagationDelay(std::string name, Ts&&... args);

    /**
     * \returns a new channel
     *
     * Create a channel based on the configuration parameters set previously.
     */
    Ptr<YansWifiChannel> Create() const;

    /**
     * Assign a fixed random variable stream number to the random variables
     * used by the channel.  Typically this corresponds to random variables
     * used in the propagation loss models.  Return the number of streams
     * (possibly zero) that have been assigned.
     *
     * \param c NetDeviceContainer of the set of net devices for which the
     *          WifiNetDevice should be modified to use fixed streams
     * \param stream first stream index to use
     *
     * \return the number of stream indices assigned by this helper
     */
    int64_t AssignStreams(Ptr<YansWifiChannel> c, int64_t stream);

  private:
    std::vector<ObjectFactory> m_propagationLoss; ///< vector of propagation loss models
    ObjectFactory m_propagationDelay;             ///< propagation delay model
};

/**
 * \brief Make it easy to create and manage PHY objects for the YANS model.
 *
 * The YANS PHY model is described in "Yet Another Network Simulator",
 * published in WNS2 2006; an author-prepared version of this paper
 * is at: https://hal.inria.fr/file/index/docid/78318/filename/yans-rr.pdf
 *
 * The Pcap and ASCII traces generated by the EnableAscii and EnablePcap methods defined
 * in this class correspond to PHY-level traces and come to us via WifiPhyHelper
 *
 */
class YansWifiPhyHelper : public WifiPhyHelper
{
  public:
    /**
     * Create a PHY helper.
     */
    YansWifiPhyHelper();

    /**
     * \param channel the channel to associate to this helper
     *
     * Every PHY created by a call to Install is associated to this channel.
     */
    void SetChannel(Ptr<YansWifiChannel> channel);
    /**
     * \param channelName The name of the channel to associate to this helper
     *
     * Every PHY created by a call to Install is associated to this channel.
     */
    void SetChannel(std::string channelName);

  private:
    /**
     * \param node the node on which we wish to create a wifi PHY
     * \param device the device within which this PHY will be created
     * \returns newly-created PHY objects.
     *
     * This method implements the pure virtual method defined in \ref ns3::WifiPhyHelper.
     */
    std::vector<Ptr<WifiPhy>> Create(Ptr<Node> node, Ptr<WifiNetDevice> device) const override;

    Ptr<YansWifiChannel> m_channel; ///< YANS wifi channel
};

/***************************************************************
 *  Implementation of the templates declared above.
 ***************************************************************/

template <typename... Ts>
void
YansWifiChannelHelper::AddPropagationLoss(std::string name, Ts&&... args)
{
    m_propagationLoss.push_back(ObjectFactory(name, std::forward<Ts>(args)...));
}

template <typename... Ts>
void
YansWifiChannelHelper::SetPropagationDelay(std::string name, Ts&&... args)
{
    m_propagationDelay = ObjectFactory(name, std::forward<Ts>(args)...);
}

} // namespace ns3

#endif /* YANS_WIFI_HELPER_H */
