<?php
/**
 * This file is part of the static reflection component.
 *
 * PHP Version 5
 *
 * Copyright (c) 2008-2009, Manuel Pichler <mapi@pdepend.org>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *   * Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 *
 *   * Neither the name of Manuel Pichler nor the names of his
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @category  PHP
 * @package   org\pdepend\reflection\queries
 * @author    Manuel Pichler <mapi@pdepend.org>
 * @copyright 2008-2009 Manuel Pichler. All rights reserved.
 * @license   http://www.opensource.org/licenses/bsd-license.php  BSD License
 * @version   SVN: $Id$
 * @link      http://pdepend.org/
 */

namespace org\pdepend\reflection\queries;

/**
 * This query class provides access to a single class. It can be called with the
 * full qualified name of a class and it returns a <b>\ReflectionClass</b>
 * instance.
 *
 * <code>
 * $query = $session->createClassQuery();
 * $class = $query->find( '\org\pdepend\reflection\parser\Parser' );
 *
 * echo 'Class: ', $class->getName(), PHP_EOL,
 *      'File:  ', $class->getFileName(), PHP_EOL;
 * </code>
 *
 * @category  PHP
 * @package   org\pdepend\reflection\queries
 * @author    Manuel Pichler <mapi@pdepend.org>
 * @copyright 2008-2009 Manuel Pichler. All rights reserved.
 * @license   http://www.opensource.org/licenses/bsd-license.php  BSD License
 * @version   Release: @package_version@
 * @link      http://pdepend.org/
 */
class ReflectionClassQuery
{
    /**
     * The type of the class.
     */
    const TYPE = __CLASS__;

    /**
     * Stack of valid reflection class factories
     *
     * @var array(\org\pdepend\reflection\interfaces\ReflectionClassFactory)
     */
    private $_classFactories = array();

    /**
     * Constructs a new reflection class query instance.
     *
     * @param array(\org\pdepend\reflection\interfaces\ReflectionClassFactory) $classFactories List
     *        of allowed reflection class factories.
     */
    public function __construct( array $classFactories )
    {
        $this->_classFactories = $classFactories;
    }

    /**
     * This method tries to find an interface or class for the given and will
     * return a <b>\ReflectionClass</b> instance for it. It will throw an
     * exception when no matching class exists.
     *
     * @param string $className Full qualified class name.
     *
     * @return \ReflectionClass
     * @throws \ReflectionException When no class or interface exists for the
     *         given class name.
     */
    public function find( $className )
    {
        foreach ( $this->_classFactories as $factory )
        {
            if ( $factory->hasClass( $className ) )
            {
                return $factory->createClass( $className );
            }
        }
        throw new \ReflectionException( 'Class ' . $className . ' does not exist' );
    }
}