"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.KmsHierarchicalKeyRingNode = void 0;
/**
 * This class is the KMS H-keyring. This class is within the kms-keyring-node
 * module because it is a KMS keyring variation. However, the KDF used in this
 * keyring's operations will only work in Node.js runtimes and not browser JS.
 * Thus, this H-keyring implementation is only Node compatible, and thus,
 * resides in a node module, not a browser module
 */
const material_management_1 = require("@aws-crypto/material-management");
const cache_material_1 = require("@aws-crypto/cache-material");
const kms_hkeyring_node_helpers_1 = require("./kms_hkeyring_node_helpers");
const branch_keystore_node_1 = require("@aws-crypto/branch-keystore-node");
const kms_keyring_1 = require("@aws-crypto/kms-keyring");
const crypto_1 = require("crypto");
class KmsHierarchicalKeyRingNode extends material_management_1.KeyringNode {
    constructor({ branchKeyId, branchKeyIdSupplier, keyStore, cacheLimitTtl, cache, maxCacheSize, partitionId, }) {
        super();
        (0, material_management_1.needs)(!partitionId || typeof partitionId === 'string', 'Partition id must be a string.');
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#partition-id
        //= type=implication
        //# The Partition ID MUST NOT be changed after initialization.
        (0, material_management_1.readOnlyProperty)(this, '_partition', 
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#partition-id-1
        //# It can either be a String provided by the user, which MUST be interpreted as the bytes of
        //# UTF-8 Encoding of the String, or a v4 UUID, which SHOULD be interpreted as the 16 byte representation of the UUID.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#partition-id-1
        //# The constructor of the Hierarchical Keyring MUST record these bytes at construction time.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#partition-id
        //# If provided, it MUST be interpreted as UTF8 bytes.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#partition-id
        //= type=exception
        //# If the PartitionId is NOT provided by the user, it MUST be set to the 16 byte representation of a v4 UUID.
        partitionId ? (0, kms_hkeyring_node_helpers_1.stringToUtf8Bytes)(partitionId) : (0, crypto_1.randomBytes)(64));
        /* Precondition: The branch key id must be a string */
        if (branchKeyId) {
            (0, material_management_1.needs)(typeof branchKeyId === 'string', 'The branch key id must be a string');
        }
        else {
            branchKeyId = undefined;
        }
        /* Precondition: The branch key id supplier must be a BranchKeyIdSupplier */
        if (branchKeyIdSupplier) {
            (0, material_management_1.needs)((0, kms_keyring_1.isBranchKeyIdSupplier)(branchKeyIdSupplier), 'The branch key id supplier must be a BranchKeyIdSupplier');
        }
        else {
            branchKeyIdSupplier = undefined;
        }
        /* Precondition: The keystore must be a BranchKeyStore */
        (0, material_management_1.needs)((0, branch_keystore_node_1.isIBranchKeyStoreNode)(keyStore), 'The keystore must be a BranchKeyStore');
        (0, material_management_1.readOnlyProperty)(this, '_logicalKeyStoreName', 
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#logical-key-store-name
        //# Logical Key Store Name MUST be converted to UTF8 Bytes to be used in
        //# the cache identifiers.
        (0, kms_hkeyring_node_helpers_1.stringToUtf8Bytes)(keyStore.getKeyStoreInfo().logicalKeyStoreName));
        /* Precondition: The cache limit TTL must be a number */
        (0, material_management_1.needs)(typeof cacheLimitTtl === 'number', 'The cache limit TTL must be a number');
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#cache-limit-ttl
        //# The maximum amount of time in seconds that an entry within the cache may be used before it MUST be evicted.
        //# The client MUST set a time-to-live (TTL) for [branch key materials](../structures.md#branch-key-materials) in the underlying cache.
        //# This value MUST be greater than zero.
        /* Precondition: Cache limit TTL must be non-negative and less than or equal to (Number.MAX_SAFE_INTEGER / 1000) seconds */
        // In the MPL, TTL can be a non-negative signed 64-bit integer.
        // However, JavaScript numbers cannot safely represent integers beyond
        // Number.MAX_SAFE_INTEGER. Thus, we will cap TTL in seconds such that TTL
        // in ms is <= Number.MAX_SAFE_INTEGER. TTL could be a BigInt type but this
        // would require casting back to a number in order to configure the CMC,
        // which leads to a lossy conversion
        (0, material_management_1.needs)(0 <= cacheLimitTtl && cacheLimitTtl * 1000 <= Number.MAX_SAFE_INTEGER, 'Cache limit TTL must be non-negative and less than or equal to (Number.MAX_SAFE_INTEGER / 1000) seconds');
        /* Precondition: Must provide a branch key identifier or supplier */
        (0, material_management_1.needs)(branchKeyId || branchKeyIdSupplier, 'Must provide a branch key identifier or supplier');
        (0, material_management_1.readOnlyProperty)(this, 'keyStore', Object.freeze(keyStore));
        /* Postcondition: The keystore object is frozen */
        // convert seconds to milliseconds
        (0, material_management_1.readOnlyProperty)(this, 'cacheLimitTtl', cacheLimitTtl * 1000);
        (0, material_management_1.readOnlyProperty)(this, 'branchKeyId', branchKeyId);
        (0, material_management_1.readOnlyProperty)(this, 'branchKeyIdSupplier', branchKeyIdSupplier
            ? Object.freeze(branchKeyIdSupplier)
            : branchKeyIdSupplier);
        /* Postcondition: Provided branch key supplier must be frozen */
        if (cache) {
            (0, material_management_1.needs)(!maxCacheSize, 'Max cache size not supported when passing a cache.');
        }
        else {
            /* Precondition: The max cache size must be a number */
            (0, material_management_1.needs)(
            // Order is important, 0 is a number but also false.
            typeof maxCacheSize === 'number' || !maxCacheSize, 'The max cache size must be a number');
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#initialization
            //# If no max cache size is provided, the cryptographic materials cache MUST be configured to a
            //# max cache size of 1000.
            maxCacheSize = maxCacheSize === 0 || maxCacheSize ? maxCacheSize : 1000;
            /* Precondition: Max cache size must be non-negative and less than or equal Number.MAX_SAFE_INTEGER */
            (0, material_management_1.needs)(0 <= maxCacheSize && maxCacheSize <= Number.MAX_SAFE_INTEGER, 'Max cache size must be non-negative and less than or equal Number.MAX_SAFE_INTEGER');
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#initialization
            //# On initialization the Hierarchical Keyring MUST initialize a [cryptographic-materials-cache](../local-cryptographic-materials-cache.md) with the configured cache limit TTL and the max cache size.
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#initialization
            //# If the Hierarchical Keyring does NOT get a `Shared` cache on initialization,
            //# it MUST initialize a [cryptographic-materials-cache](../local-cryptographic-materials-cache.md)
            //# with the user provided cache limit TTL and the entry capacity.
            cache = (0, cache_material_1.getLocalCryptographicMaterialsCache)(maxCacheSize);
        }
        (0, material_management_1.readOnlyProperty)(this, 'maxCacheSize', maxCacheSize);
        (0, material_management_1.readOnlyProperty)(this, '_cmc', cache);
        Object.freeze(this);
        /* Postcondition: The HKR object must be frozen */
    }
    async _onEncrypt(
    //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
    //= type=implication
    //# OnEncrypt MUST take [encryption materials](../structures.md#encryption-materials) as input.
    encryptionMaterial) {
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
        //# The `branchKeyId` used in this operation is either the configured branchKeyId, if supplied, or the result of the `branchKeySupplier`'s
        //# `getBranchKeyId` operation, using the encryption material's encryption context as input.
        const branchKeyId = (0, kms_hkeyring_node_helpers_1.getBranchKeyId)(this, encryptionMaterial);
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
        //# The hierarchical keyring MUST use the formulas specified in [Appendix A](#appendix-a-cache-entry-identifier-formulas)
        //# to compute the [cache entry identifier](../cryptographic-materials-cache.md#cache-identifier).
        const cacheEntryId = (0, kms_hkeyring_node_helpers_1.getCacheEntryId)(this._logicalKeyStoreName, this._partition, branchKeyId);
        const branchKeyMaterials = await (0, kms_hkeyring_node_helpers_1.getBranchKeyMaterials)(this, this._cmc, branchKeyId, cacheEntryId);
        // get a pdk (generate it if not already set)
        const pdk = (0, kms_hkeyring_node_helpers_1.getPlaintextDataKey)(encryptionMaterial);
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
        //# If the keyring is unable to wrap a plaintext data key, OnEncrypt MUST fail
        //# and MUST NOT modify the [decryption materials](structures.md#decryption-materials).
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
        //# - MUST wrap a data key with the branch key materials according to the [branch key wrapping](#branch-key-wrapping) section.
        const edk = (0, kms_hkeyring_node_helpers_1.wrapPlaintextDataKey)(pdk, branchKeyMaterials, encryptionMaterial);
        // return the modified encryption material with the new edk and newly
        // generated pdk (if applicable)
        return (0, kms_hkeyring_node_helpers_1.modifyEncryptionMaterial)(encryptionMaterial, pdk, edk, branchKeyId);
    }
    async onDecrypt(
    //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
    //= type=implication
    //# OnDecrypt MUST take [decryption materials](../structures.md#decryption-materials) and a list of [encrypted data keys](../structures.md#encrypted-data-keys) as input.
    material, encryptedDataKeys) {
        (0, material_management_1.needs)((0, material_management_1.isDecryptionMaterial)(material), 'Unsupported material type.');
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
        //# If the decryption materials already contain a `PlainTextDataKey`, OnDecrypt MUST fail.
        /* Precondition: If the decryption materials already contain a PlainTextDataKey, OnDecrypt MUST fail */
        (0, material_management_1.needs)(!material.hasUnencryptedDataKey, 'Decryption materials already contain a plaintext data key');
        (0, material_management_1.needs)(encryptedDataKeys.every((edk) => edk instanceof material_management_1.EncryptedDataKey), 'Unsupported EncryptedDataKey type');
        const _material = await this._onDecrypt(material, encryptedDataKeys);
        (0, material_management_1.needs)(material === _material, 'New DecryptionMaterial instances can not be created.');
        return material;
    }
    cacheEntryHasExceededLimits({ now }) {
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
        //# There MUST be a check (cacheEntryWithinLimits) to make sure that for the cache entry found, who's TTL has NOT expired,
        //# `time.now() - cacheEntryCreationTime <= ttlSeconds` is true and
        //# valid for TTL of the Hierarchical Keyring getting the cache entry.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
        //# There MUST be a check (cacheEntryWithinLimits) to make sure that for the cache entry found, who's TTL has NOT expired,
        //# `time.now() - cacheEntryCreationTime <= ttlSeconds` is true and
        //# valid for TTL of the Hierarchical Keyring getting the cache entry.
        const age = Date.now() - now;
        return age > this.cacheLimitTtl;
    }
    async _onDecrypt(decryptionMaterial, encryptedDataKeyObjs) {
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
        //# The `branchKeyId` used in this operation is either the configured branchKeyId, if supplied, or the result of the `branchKeySupplier`'s
        //# `getBranchKeyId` operation, using the decryption material's encryption context as input.
        const branchKeyId = (0, kms_hkeyring_node_helpers_1.getBranchKeyId)(this, decryptionMaterial);
        // filter out edk objects that don't match this keyring's configuration
        const filteredEdkObjs = encryptedDataKeyObjs.filter((edkObj) => (0, kms_hkeyring_node_helpers_1.filterEdk)(branchKeyId, edkObj));
        /* Precondition: There must be an encrypted data key that matches this keyring configuration */
        (0, material_management_1.needs)(filteredEdkObjs.length > 0, "There must be an encrypted data key that matches this keyring's configuration");
        const errors = [];
        for (const { encryptedDataKey: ciphertext } of filteredEdkObjs) {
            let udk = undefined;
            try {
                //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#getitem-branch-keystore-ondecrypt
                //# - Deserialize the UUID string representation of the `version` from the [encrypted data key](../structures.md#encrypted-data-key) [ciphertext](#ciphertext).
                // get the branch key version (as compressed bytes) from the
                // destructured ciphertext of the edk
                const { branchKeyVersionAsBytesCompressed } = (0, kms_hkeyring_node_helpers_1.destructureCiphertext)(ciphertext, decryptionMaterial.suite);
                //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#getitem-branch-keystore-ondecrypt
                //# - The deserialized UUID string representation of the `version`
                // uncompress the branch key version into regular utf8 bytes
                const branchKeyVersionAsBytes = (0, kms_hkeyring_node_helpers_1.stringToUtf8Bytes)((0, kms_hkeyring_node_helpers_1.decompressBytesToUuidv4)(branchKeyVersionAsBytesCompressed));
                //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
                //# The hierarchical keyring MUST use the OnDecrypt formula specified in [Appendix A](#decryption-materials)
                //# in order to compute the [cache entry identifier](cryptographic-materials-cache.md#cache-identifier).
                const cacheEntryId = (0, kms_hkeyring_node_helpers_1.getCacheEntryId)(this._logicalKeyStoreName, this._partition, 
                //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#getitem-branch-keystore-ondecrypt
                //# OnDecrypt MUST calculate the following values:
                branchKeyId, branchKeyVersionAsBytes);
                // get the string representation of the branch key version
                const branchKeyVersionAsString = (0, kms_hkeyring_node_helpers_1.decompressBytesToUuidv4)(branchKeyVersionAsBytesCompressed);
                //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
                //# To decrypt each encrypted data key in the filtered set, the hierarchical keyring MUST attempt
                //# to find the corresponding [branch key materials](../structures.md#branch-key-materials)
                //# from the underlying [cryptographic materials cache](../local-cryptographic-materials-cache.md).
                const branchKeyMaterials = await (0, kms_hkeyring_node_helpers_1.getBranchKeyMaterials)(this, this._cmc, branchKeyId, cacheEntryId, branchKeyVersionAsString);
                //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
                //# - MUST unwrap the encrypted data key with the branch key materials according to the [branch key unwrapping](#branch-key-unwrapping) section.
                udk = (0, kms_hkeyring_node_helpers_1.unwrapEncryptedDataKey)(ciphertext, branchKeyMaterials, decryptionMaterial);
            }
            catch (e) {
                //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
                //# For each encrypted data key in the filtered set, one at a time, OnDecrypt MUST attempt to decrypt the encrypted data key.
                //# If this attempt results in an error, then these errors MUST be collected.
                errors.push({ errPlus: e });
            }
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
            //# If a decryption succeeds, this keyring MUST
            //# add the resulting plaintext data key to the decryption materials and return the modified materials.
            if (udk) {
                return (0, kms_hkeyring_node_helpers_1.modifyDencryptionMaterial)(decryptionMaterial, udk, branchKeyId);
            }
        }
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
        //# If OnDecrypt fails to successfully decrypt any [encrypted data key](../structures.md#encrypted-data-key),
        //# then it MUST yield an error that includes all the collected errors
        //# and MUST NOT modify the [decryption materials](structures.md#decryption-materials).
        throw new Error(errors.reduce((m, e, i) => `${m} Error #${i + 1} \n ${e.errPlus.stack} \n`, 'Unable to decrypt data key'));
    }
}
exports.KmsHierarchicalKeyRingNode = KmsHierarchicalKeyRingNode;
(0, material_management_1.immutableClass)(KmsHierarchicalKeyRingNode);
// The JS version has not been released with a Storm Tracking CMC
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#initialization
//= type=exception
//# If the cache to initialize is a [Storm Tracking Cryptographic Materials Cache](../storm-tracking-cryptographic-materials-cache.md#overview)
//# then the [Grace Period](../storm-tracking-cryptographic-materials-cache.md#grace-period) MUST be less than the [cache limit TTL](#cache-limit-ttl).
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#initialization
//= type=exception
//# If no `cache` is provided, a `DefaultCache` MUST be configured with entry capacity of 1000.
// These are not something we can enforce
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#logical-key-store-name
//= type=exception
//# > Note: Users MUST NEVER have two different physical Key Stores with the same Logical Key Store Name.
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#shared-cache-considerations
//= type=exception
//# Any keyring that has access to the `Shared` cache MAY be able to use materials
//# that it MAY or MAY NOT have direct access to.
//#
//# Users MUST make sure that all of Partition ID, Logical Key Store Name of the Key Store for the Hierarchical Keyring
//# and Branch Key ID are set to be the same for two Hierarchical Keyrings if and only they want the keyrings to share
//# cache entries.
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia21zX2hrZXlyaW5nX25vZGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMva21zX2hrZXlyaW5nX25vZGUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLG9FQUFvRTtBQUNwRSxzQ0FBc0M7OztBQUV0Qzs7Ozs7O0dBTUc7QUFFSCx5RUFZd0M7QUFDeEMsK0RBSW1DO0FBQ25DLDJFQWFvQztBQUNwQywyRUFHeUM7QUFDekMseURBR2dDO0FBQ2hDLG1DQUFvQztBQXdDcEMsTUFBYSwwQkFDWCxTQUFRLGlDQUFXO0lBZW5CLFlBQVksRUFDVixXQUFXLEVBQ1gsbUJBQW1CLEVBQ25CLFFBQVEsRUFDUixhQUFhLEVBQ2IsS0FBSyxFQUNMLFlBQVksRUFDWixXQUFXLEdBQ3FCO1FBQ2hDLEtBQUssRUFBRSxDQUFBO1FBRVAsSUFBQSwyQkFBSyxFQUNILENBQUMsV0FBVyxJQUFJLE9BQU8sV0FBVyxLQUFLLFFBQVEsRUFDL0MsZ0NBQWdDLENBQ2pDLENBQUE7UUFFRCxtR0FBbUc7UUFDbkcsb0JBQW9CO1FBQ3BCLDhEQUE4RDtRQUM5RCxJQUFBLHNDQUFnQixFQUNkLElBQUksRUFDSixZQUFZO1FBRVoscUdBQXFHO1FBQ3JHLDZGQUE2RjtRQUM3RixzSEFBc0g7UUFFdEgscUdBQXFHO1FBQ3JHLDZGQUE2RjtRQUU3RixtR0FBbUc7UUFDbkcsc0RBQXNEO1FBRXRELG1HQUFtRztRQUNuRyxrQkFBa0I7UUFDbEIsOEdBQThHO1FBQzlHLFdBQVcsQ0FBQyxDQUFDLENBQUMsSUFBQSw2Q0FBaUIsRUFBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBQSxvQkFBVyxFQUFDLEVBQUUsQ0FBQyxDQUMvRCxDQUFBO1FBRUQsc0RBQXNEO1FBQ3RELElBQUksV0FBVyxFQUFFO1lBQ2YsSUFBQSwyQkFBSyxFQUNILE9BQU8sV0FBVyxLQUFLLFFBQVEsRUFDL0Isb0NBQW9DLENBQ3JDLENBQUE7U0FDRjthQUFNO1lBQ0wsV0FBVyxHQUFHLFNBQVMsQ0FBQTtTQUN4QjtRQUVELDRFQUE0RTtRQUM1RSxJQUFJLG1CQUFtQixFQUFFO1lBQ3ZCLElBQUEsMkJBQUssRUFDSCxJQUFBLG1DQUFxQixFQUFDLG1CQUFtQixDQUFDLEVBQzFDLDBEQUEwRCxDQUMzRCxDQUFBO1NBQ0Y7YUFBTTtZQUNMLG1CQUFtQixHQUFHLFNBQVMsQ0FBQTtTQUNoQztRQUVELHlEQUF5RDtRQUN6RCxJQUFBLDJCQUFLLEVBQ0gsSUFBQSw0Q0FBcUIsRUFBQyxRQUFRLENBQUMsRUFDL0IsdUNBQXVDLENBQ3hDLENBQUE7UUFFRCxJQUFBLHNDQUFnQixFQUNkLElBQUksRUFDSixzQkFBc0I7UUFDdEIsNkdBQTZHO1FBQzdHLHdFQUF3RTtRQUN4RSwwQkFBMEI7UUFDMUIsSUFBQSw2Q0FBaUIsRUFBQyxRQUFRLENBQUMsZUFBZSxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FDbEUsQ0FBQTtRQUVELHdEQUF3RDtRQUN4RCxJQUFBLDJCQUFLLEVBQ0gsT0FBTyxhQUFhLEtBQUssUUFBUSxFQUNqQyxzQ0FBc0MsQ0FDdkMsQ0FBQTtRQUVELHNHQUFzRztRQUN0RywrR0FBK0c7UUFDL0csdUlBQXVJO1FBQ3ZJLHlDQUF5QztRQUN6QywySEFBMkg7UUFDM0gsK0RBQStEO1FBQy9ELHNFQUFzRTtRQUN0RSwwRUFBMEU7UUFDMUUsMkVBQTJFO1FBQzNFLHdFQUF3RTtRQUN4RSxvQ0FBb0M7UUFDcEMsSUFBQSwyQkFBSyxFQUNILENBQUMsSUFBSSxhQUFhLElBQUksYUFBYSxHQUFHLElBQUksSUFBSSxNQUFNLENBQUMsZ0JBQWdCLEVBQ3JFLHlHQUF5RyxDQUMxRyxDQUFBO1FBRUQsb0VBQW9FO1FBQ3BFLElBQUEsMkJBQUssRUFDSCxXQUFXLElBQUksbUJBQW1CLEVBQ2xDLGtEQUFrRCxDQUNuRCxDQUFBO1FBRUQsSUFBQSxzQ0FBZ0IsRUFBQyxJQUFJLEVBQUUsVUFBVSxFQUFFLE1BQU0sQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQTtRQUMzRCxrREFBa0Q7UUFFbEQsa0NBQWtDO1FBQ2xDLElBQUEsc0NBQWdCLEVBQUMsSUFBSSxFQUFFLGVBQWUsRUFBRSxhQUFhLEdBQUcsSUFBSSxDQUFDLENBQUE7UUFFN0QsSUFBQSxzQ0FBZ0IsRUFBQyxJQUFJLEVBQUUsYUFBYSxFQUFFLFdBQVcsQ0FBQyxDQUFBO1FBRWxELElBQUEsc0NBQWdCLEVBQ2QsSUFBSSxFQUNKLHFCQUFxQixFQUNyQixtQkFBbUI7WUFDakIsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsbUJBQW1CLENBQUM7WUFDcEMsQ0FBQyxDQUFDLG1CQUFtQixDQUN4QixDQUFBO1FBQ0QsZ0VBQWdFO1FBRWhFLElBQUksS0FBSyxFQUFFO1lBQ1QsSUFBQSwyQkFBSyxFQUFDLENBQUMsWUFBWSxFQUFFLG9EQUFvRCxDQUFDLENBQUE7U0FDM0U7YUFBTTtZQUNMLHVEQUF1RDtZQUN2RCxJQUFBLDJCQUFLO1lBQ0gsb0RBQW9EO1lBQ3BELE9BQU8sWUFBWSxLQUFLLFFBQVEsSUFBSSxDQUFDLFlBQVksRUFDakQscUNBQXFDLENBQ3RDLENBQUE7WUFFRCxxR0FBcUc7WUFDckcsK0ZBQStGO1lBQy9GLDJCQUEyQjtZQUMzQixZQUFZLEdBQUcsWUFBWSxLQUFLLENBQUMsSUFBSSxZQUFZLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFBO1lBQ3ZFLHNHQUFzRztZQUN0RyxJQUFBLDJCQUFLLEVBQ0gsQ0FBQyxJQUFJLFlBQVksSUFBSSxZQUFZLElBQUksTUFBTSxDQUFDLGdCQUFnQixFQUM1RCxvRkFBb0YsQ0FDckYsQ0FBQTtZQUVELHFHQUFxRztZQUNyRyx1TUFBdU07WUFFdk0scUdBQXFHO1lBQ3JHLGdGQUFnRjtZQUNoRixtR0FBbUc7WUFDbkcsa0VBQWtFO1lBQ2xFLEtBQUssR0FBRyxJQUFBLG9EQUFtQyxFQUFDLFlBQVksQ0FBQyxDQUFBO1NBQzFEO1FBQ0QsSUFBQSxzQ0FBZ0IsRUFBQyxJQUFJLEVBQUUsY0FBYyxFQUFFLFlBQVksQ0FBQyxDQUFBO1FBQ3BELElBQUEsc0NBQWdCLEVBQUMsSUFBSSxFQUFFLE1BQU0sRUFBRSxLQUFLLENBQUMsQ0FBQTtRQUVyQyxNQUFNLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFBO1FBQ25CLGtEQUFrRDtJQUNwRCxDQUFDO0lBRUQsS0FBSyxDQUFDLFVBQVU7SUFDZCxnR0FBZ0c7SUFDaEcsb0JBQW9CO0lBQ3BCLCtGQUErRjtJQUMvRixrQkFBMEM7UUFFMUMsZ0dBQWdHO1FBQ2hHLDBJQUEwSTtRQUMxSSw0RkFBNEY7UUFDNUYsTUFBTSxXQUFXLEdBQUcsSUFBQSwwQ0FBYyxFQUFDLElBQUksRUFBRSxrQkFBa0IsQ0FBQyxDQUFBO1FBRTVELGdHQUFnRztRQUNoRyx5SEFBeUg7UUFDekgsa0dBQWtHO1FBQ2xHLE1BQU0sWUFBWSxHQUFHLElBQUEsMkNBQWUsRUFDbEMsSUFBSSxDQUFDLG9CQUFvQixFQUN6QixJQUFJLENBQUMsVUFBVSxFQUNmLFdBQVcsQ0FDWixDQUFBO1FBRUQsTUFBTSxrQkFBa0IsR0FBRyxNQUFNLElBQUEsaURBQXFCLEVBQ3BELElBQUksRUFDSixJQUFJLENBQUMsSUFBSSxFQUNULFdBQVcsRUFDWCxZQUFZLENBQ2IsQ0FBQTtRQUVELDZDQUE2QztRQUM3QyxNQUFNLEdBQUcsR0FBRyxJQUFBLCtDQUFtQixFQUFDLGtCQUFrQixDQUFDLENBQUE7UUFFbkQsZ0dBQWdHO1FBQ2hHLDhFQUE4RTtRQUM5RSx1RkFBdUY7UUFFdkYsZ0dBQWdHO1FBQ2hHLDhIQUE4SDtRQUM5SCxNQUFNLEdBQUcsR0FBRyxJQUFBLGdEQUFvQixFQUM5QixHQUFHLEVBQ0gsa0JBQWtCLEVBQ2xCLGtCQUFrQixDQUNuQixDQUFBO1FBRUQscUVBQXFFO1FBQ3JFLGdDQUFnQztRQUNoQyxPQUFPLElBQUEsb0RBQXdCLEVBQUMsa0JBQWtCLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxXQUFXLENBQUMsQ0FBQTtJQUM1RSxDQUFDO0lBRUQsS0FBSyxDQUFDLFNBQVM7SUFDYixnR0FBZ0c7SUFDaEcsb0JBQW9CO0lBQ3BCLHlLQUF5SztJQUN6SyxRQUFnQyxFQUNoQyxpQkFBcUM7UUFFckMsSUFBQSwyQkFBSyxFQUFDLElBQUEsMENBQW9CLEVBQUMsUUFBUSxDQUFDLEVBQUUsNEJBQTRCLENBQUMsQ0FBQTtRQUVuRSxnR0FBZ0c7UUFDaEcsMEZBQTBGO1FBQzFGLHVHQUF1RztRQUN2RyxJQUFBLDJCQUFLLEVBQ0gsQ0FBQyxRQUFRLENBQUMscUJBQXFCLEVBQy9CLDJEQUEyRCxDQUM1RCxDQUFBO1FBRUQsSUFBQSwyQkFBSyxFQUNILGlCQUFpQixDQUFDLEtBQUssQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsR0FBRyxZQUFZLHNDQUFnQixDQUFDLEVBQ2pFLG1DQUFtQyxDQUNwQyxDQUFBO1FBRUQsTUFBTSxTQUFTLEdBQUcsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLFFBQVEsRUFBRSxpQkFBaUIsQ0FBQyxDQUFBO1FBRXBFLElBQUEsMkJBQUssRUFDSCxRQUFRLEtBQUssU0FBUyxFQUN0QixzREFBc0QsQ0FDdkQsQ0FBQTtRQUVELE9BQU8sUUFBUSxDQUFBO0lBQ2pCLENBQUM7SUFFRCwyQkFBMkIsQ0FBQyxFQUFFLEdBQUcsRUFBMEI7UUFDekQsZ0dBQWdHO1FBQ2hHLDBIQUEwSDtRQUMxSCxtRUFBbUU7UUFDbkUsc0VBQXNFO1FBRXRFLGdHQUFnRztRQUNoRywwSEFBMEg7UUFDMUgsbUVBQW1FO1FBQ25FLHNFQUFzRTtRQUV0RSxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsR0FBRyxFQUFFLEdBQUcsR0FBRyxDQUFBO1FBQzVCLE9BQU8sR0FBRyxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUE7SUFDakMsQ0FBQztJQUVELEtBQUssQ0FBQyxVQUFVLENBQ2Qsa0JBQTBDLEVBQzFDLG9CQUF3QztRQUV4QyxnR0FBZ0c7UUFDaEcsMElBQTBJO1FBQzFJLDRGQUE0RjtRQUM1RixNQUFNLFdBQVcsR0FBRyxJQUFBLDBDQUFjLEVBQUMsSUFBSSxFQUFFLGtCQUFrQixDQUFDLENBQUE7UUFFNUQsdUVBQXVFO1FBQ3ZFLE1BQU0sZUFBZSxHQUFHLG9CQUFvQixDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQzdELElBQUEscUNBQVMsRUFBQyxXQUFXLEVBQUUsTUFBTSxDQUFDLENBQy9CLENBQUE7UUFFRCwrRkFBK0Y7UUFDL0YsSUFBQSwyQkFBSyxFQUNILGVBQWUsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUMxQiwrRUFBK0UsQ0FDaEYsQ0FBQTtRQUVELE1BQU0sTUFBTSxHQUFnQixFQUFFLENBQUE7UUFDOUIsS0FBSyxNQUFNLEVBQUUsZ0JBQWdCLEVBQUUsVUFBVSxFQUFFLElBQUksZUFBZSxFQUFFO1lBQzlELElBQUksR0FBRyxHQUEyQixTQUFTLENBQUE7WUFDM0MsSUFBSTtnQkFDRix3SEFBd0g7Z0JBQ3hILCtKQUErSjtnQkFDL0osNERBQTREO2dCQUM1RCxxQ0FBcUM7Z0JBQ3JDLE1BQU0sRUFBRSxpQ0FBaUMsRUFBRSxHQUFHLElBQUEsaURBQXFCLEVBQ2pFLFVBQVUsRUFDVixrQkFBa0IsQ0FBQyxLQUFLLENBQ3pCLENBQUE7Z0JBRUQsd0hBQXdIO2dCQUN4SCxrRUFBa0U7Z0JBQ2xFLDREQUE0RDtnQkFDNUQsTUFBTSx1QkFBdUIsR0FBRyxJQUFBLDZDQUFpQixFQUMvQyxJQUFBLG1EQUF1QixFQUFDLGlDQUFpQyxDQUFDLENBQzNELENBQUE7Z0JBRUQsZ0dBQWdHO2dCQUNoRyw0R0FBNEc7Z0JBQzVHLHdHQUF3RztnQkFDeEcsTUFBTSxZQUFZLEdBQUcsSUFBQSwyQ0FBZSxFQUNsQyxJQUFJLENBQUMsb0JBQW9CLEVBQ3pCLElBQUksQ0FBQyxVQUFVO2dCQUNmLHdIQUF3SDtnQkFDeEgsa0RBQWtEO2dCQUNsRCxXQUFXLEVBQ1gsdUJBQXVCLENBQ3hCLENBQUE7Z0JBRUQsMERBQTBEO2dCQUMxRCxNQUFNLHdCQUF3QixHQUFHLElBQUEsbURBQXVCLEVBQ3RELGlDQUFpQyxDQUNsQyxDQUFBO2dCQUVELGdHQUFnRztnQkFDaEcsaUdBQWlHO2dCQUNqRywyRkFBMkY7Z0JBQzNGLG1HQUFtRztnQkFDbkcsTUFBTSxrQkFBa0IsR0FBRyxNQUFNLElBQUEsaURBQXFCLEVBQ3BELElBQUksRUFDSixJQUFJLENBQUMsSUFBSSxFQUNULFdBQVcsRUFDWCxZQUFZLEVBQ1osd0JBQXdCLENBQ3pCLENBQUE7Z0JBRUQsZ0dBQWdHO2dCQUNoRyxnSkFBZ0o7Z0JBQ2hKLEdBQUcsR0FBRyxJQUFBLGtEQUFzQixFQUMxQixVQUFVLEVBQ1Ysa0JBQWtCLEVBQ2xCLGtCQUFrQixDQUNuQixDQUFBO2FBQ0Y7WUFBQyxPQUFPLENBQUMsRUFBRTtnQkFDVixnR0FBZ0c7Z0JBQ2hHLDZIQUE2SDtnQkFDN0gsNkVBQTZFO2dCQUM3RSxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsT0FBTyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUE7YUFDNUI7WUFFRCxnR0FBZ0c7WUFDaEcsK0NBQStDO1lBQy9DLHVHQUF1RztZQUN2RyxJQUFJLEdBQUcsRUFBRTtnQkFDUCxPQUFPLElBQUEscURBQXlCLEVBQUMsa0JBQWtCLEVBQUUsR0FBRyxFQUFFLFdBQVcsQ0FBQyxDQUFBO2FBQ3ZFO1NBQ0Y7UUFFRCxnR0FBZ0c7UUFDaEcsNkdBQTZHO1FBQzdHLHNFQUFzRTtRQUN0RSx1RkFBdUY7UUFDdkYsTUFBTSxJQUFJLEtBQUssQ0FDYixNQUFNLENBQUMsTUFBTSxDQUNYLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEtBQUssRUFDNUQsNEJBQTRCLENBQzdCLENBQ0YsQ0FBQTtJQUNILENBQUM7Q0FDRjtBQS9XRCxnRUErV0M7QUFFRCxJQUFBLG9DQUFjLEVBQUMsMEJBQTBCLENBQUMsQ0FBQTtBQUUxQyxpRUFBaUU7QUFFakUscUdBQXFHO0FBQ3JHLGtCQUFrQjtBQUNsQiwrSUFBK0k7QUFDL0ksdUpBQXVKO0FBRXZKLHFHQUFxRztBQUNyRyxrQkFBa0I7QUFDbEIsK0ZBQStGO0FBRS9GLHlDQUF5QztBQUV6Qyw2R0FBNkc7QUFDN0csa0JBQWtCO0FBQ2xCLHlHQUF5RztBQUV6RyxrSEFBa0g7QUFDbEgsa0JBQWtCO0FBQ2xCLGtGQUFrRjtBQUNsRixpREFBaUQ7QUFDakQsR0FBRztBQUNILHVIQUF1SDtBQUN2SCxzSEFBc0g7QUFDdEgsa0JBQWtCIn0=