"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.determineMapping = void 0;
var _lodash = _interopRequireDefault(require("lodash"));
var _common = require("../../common");
var _util = require("./util");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const determineMapping = (document, nestedObjectsLimit) => {
  return {
    dynamic: true,
    date_detection: true,
    ...determineType(document, 1, nestedObjectsLimit)
  };
};
exports.determineMapping = determineMapping;
const determineType = (value, currentNestedCount, nestedObjectsLimit) => {
  if (currentNestedCount >= nestedObjectsLimit) {
    throw Error(`Current document exceeds nested object limit of ${nestedObjectsLimit}`);
  }
  const defaultType = {
    type: _common.DYNAMIC_MAPPING_TYPES.TEXT,
    fields: {
      keyword: {
        type: 'keyword',
        ignore_above: 256
      }
    }
  };
  switch (true) {
    case Array.isArray(value) && value.length < 1:
    case value == null:
      return {
        type: _common.DYNAMIC_MAPPING_TYPES.NULL
      };
    case typeof value === 'string' && value.length === 0:
      return defaultType;
    case (0, _util.isBooleanType)(value):
      return {
        type: _common.DYNAMIC_MAPPING_TYPES.BOOLEAN
      };
    case (0, _util.isNumericType)(value):
      return {
        type: determineExactNumberType(Number(value))
      };
    case (0, _util.isValidDate)(_lodash.default.toString(value)):
      // TODO Dates need further parsing since OpenSearch expects a certain timestamp format
      return defaultType;
    case Array.isArray(value) && value.length > 0:
      return determineType(value[0], currentNestedCount, nestedObjectsLimit);
    case value && typeof value === 'object' && !Array.isArray(value):
      const properties = {};
      Object.keys(value).forEach(key => {
        properties[key] = determineType(value[key], currentNestedCount + 1, nestedObjectsLimit);
      });
      return {
        properties
      };
    default:
      return defaultType;
  }
};
const determineExactNumberType = value => {
  if (Number.isSafeInteger(value)) {
    return _common.DYNAMIC_MAPPING_TYPES.INTEGER;
  } else if (!Number.isInteger(value)) {
    return _common.DYNAMIC_MAPPING_TYPES.FLOAT;
  } else {
    return _common.DYNAMIC_MAPPING_TYPES.DOUBLE;
  }
};