"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WorkspaceUiSettingsClientWrapper = void 0;
var _utils = require("../../../../core/server/utils");
var _server = require("../../../../core/server");
var _constants = require("../../common/constants");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * Copyright OpenSearch Contributors
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          */
/**
 * This saved object client wrapper offers methods to get and update UI settings considering
 * the context of the current workspace.
 */
class WorkspaceUiSettingsClientWrapper {
  constructor(logger, env) {
    this.logger = logger;
    this.env = env;
    _defineProperty(this, "getScopedClient", void 0);
    _defineProperty(this, "asScopedUISettingsClient", void 0);
    _defineProperty(this, "wrapperFactory", wrapperOptions => {
      const getUiSettingsWithWorkspace = async (type, id, options = {}) => {
        const {
          requestWorkspaceId
        } = (0, _utils.getWorkspaceState)(wrapperOptions.request);

        /**
         * When getting ui settings within a workspace, it will combine the workspace ui settings with
         * the global ui settings and workspace ui settings will override global settings attribute
         */
        if (type === 'config' && id.startsWith(_server.CURRENT_WORKSPACE_PLACEHOLDER)) {
          var _workspaceObject;
          // if not in a workspace and try to get workspace level settings
          // it should return NotFoundError
          if (!requestWorkspaceId) {
            throw _server.SavedObjectsErrorHelpers.createGenericNotFoundError();
          }
          const normalizeDocId = id.replace(`${_server.CURRENT_WORKSPACE_PLACEHOLDER}_`, '');
          const configObject = await wrapperOptions.client.get('config', normalizeDocId, options);
          let workspaceObject = null;
          const workspaceTypeEnabledClient = this.getWorkspaceTypeEnabledClient(wrapperOptions.request);
          try {
            workspaceObject = await workspaceTypeEnabledClient.get(_server.WORKSPACE_TYPE, requestWorkspaceId);
          } catch (e) {
            this.logger.error(`Unable to get workspaceObject with id: ${requestWorkspaceId}`);
          }
          const UISettingsClient = this.getUISettingsClient(workspaceTypeEnabledClient);
          const registeredConfigs = UISettingsClient.getRegistered();
          const workspaceScopeKeys = Object.entries(registeredConfigs).filter(([, config]) => Array().concat(config.scope || []).includes(_server.UiSettingScope.WORKSPACE)).map(([key]) => key);
          const workspaceSettings = ((_workspaceObject = workspaceObject) === null || _workspaceObject === void 0 || (_workspaceObject = _workspaceObject.attributes) === null || _workspaceObject === void 0 ? void 0 : _workspaceObject.uiSettings) || {};
          workspaceScopeKeys.forEach(key => {
            workspaceSettings[key] = workspaceSettings[key];
          });
          configObject.attributes = workspaceSettings;
          return configObject;
        }
        return wrapperOptions.client.get(type, id, options);
      };
      const updateUiSettingsWithWorkspace = async (type, id, attributes, options = {}) => {
        const {
          requestWorkspaceId
        } = (0, _utils.getWorkspaceState)(wrapperOptions.request);
        const updateWorkspaceSettings = async (configDocId, workspaceId, workspaceAttributes) => {
          const savedObjectsClient = this.getWorkspaceTypeEnabledClient(wrapperOptions.request);
          const configObject = await wrapperOptions.client.get('config', configDocId, options);
          const workspaceObject = await savedObjectsClient.get(_server.WORKSPACE_TYPE, workspaceId);
          const workspaceUpdateResult = await savedObjectsClient.update(_server.WORKSPACE_TYPE, workspaceId, {
            ...workspaceObject.attributes,
            uiSettings: {
              ...workspaceObject.attributes.uiSettings,
              ...workspaceAttributes
            }
          }, options);
          configObject.attributes = workspaceUpdateResult.attributes.uiSettings || {};
          return configObject;
        };

        /**
         * When updating ui settings within a workspace, it will update the workspace ui settings,
         * the global ui settings will remain unchanged.
         * Skip updating workspace level setting if the request is updating user level setting specifically or global workspace level setting.
         */
        if (type === 'config') {
          if (id.startsWith(_server.CURRENT_WORKSPACE_PLACEHOLDER)) {
            // if not in a workspace and try to update workspace level settings
            // it should return 400 BadRequestError
            if (!requestWorkspaceId) {
              throw _server.SavedObjectsErrorHelpers.createBadRequestError();
            }
            const normalizeDocId = id.replace(`${_server.CURRENT_WORKSPACE_PLACEHOLDER}_`, '');
            return updateWorkspaceSettings(normalizeDocId, requestWorkspaceId, attributes);
          } else if (requestWorkspaceId && id === this.env.packageInfo.version) {
            // The code below maintains backward compatibility for UI setting updates in version 3.0.0.
            // Remove if no external code is modifying these settings through the global scope.
            this.logger.warn('Deprecation warning: updating workspace settings through global scope will no longer be supported.');
            return updateWorkspaceSettings(id, requestWorkspaceId, attributes);
          }
        }
        return wrapperOptions.client.update(type, id, attributes, options);
      };
      return {
        ...wrapperOptions.client,
        checkConflicts: wrapperOptions.client.checkConflicts,
        errors: wrapperOptions.client.errors,
        addToNamespaces: wrapperOptions.client.addToNamespaces,
        deleteFromNamespaces: wrapperOptions.client.deleteFromNamespaces,
        find: wrapperOptions.client.find,
        bulkGet: wrapperOptions.client.bulkGet,
        create: wrapperOptions.client.create,
        bulkCreate: wrapperOptions.client.bulkCreate,
        delete: wrapperOptions.client.delete,
        bulkUpdate: wrapperOptions.client.bulkUpdate,
        deleteByWorkspace: wrapperOptions.client.deleteByWorkspace,
        get: getUiSettingsWithWorkspace,
        update: updateUiSettingsWithWorkspace
      };
    });
  }
  /**
   * WORKSPACE_TYPE is a hidden type, regular saved object client won't return hidden types.
   * To access workspace uiSettings which is defined as a property of workspace object, the
   * WORKSPACE_TYPE needs to be excluded.
   */
  getWorkspaceTypeEnabledClient(request) {
    var _this$getScopedClient;
    return (_this$getScopedClient = this.getScopedClient) === null || _this$getScopedClient === void 0 ? void 0 : _this$getScopedClient.call(this, request, {
      includedHiddenTypes: [_server.WORKSPACE_TYPE],
      excludedWrappers: [_constants.WORKSPACE_UI_SETTINGS_CLIENT_WRAPPER_ID]
    });
  }
  getUISettingsClient(savedObjectClient) {
    var _this$asScopedUISetti;
    return (_this$asScopedUISetti = this.asScopedUISettingsClient) === null || _this$asScopedUISetti === void 0 ? void 0 : _this$asScopedUISetti.call(this, savedObjectClient);
  }
  setScopedClient(getScopedClient) {
    this.getScopedClient = getScopedClient;
  }
  setAsScopedUISettingsClient(asScopedToClient) {
    this.asScopedUISettingsClient = asScopedToClient;
  }
}
exports.WorkspaceUiSettingsClientWrapper = WorkspaceUiSettingsClientWrapper;