
#! @Chapter Functions for calculating with Majorana representations
#! @Section Basic functions

#! @Arguments rep
#! @Returns true is all algebra products have been found, otherwise returns false
#! @Description Takes a Majorana representation <A>rep</A>, as outputted by
#! <Ref Func="MajoranaRepresentation"/>. If the representation is complete, that is
#! to say, if the vector space spanned by the basis vectors indexed by the elements
#! in <A>rep.setup.coords</A> is closed under the algebra product given by
#! <A>rep.algebraproducts</A>, return true. Otherwise, if some products are not known
#! then return false.
DeclareGlobalFunction( "MAJORANA_IsComplete" );

#! @Arguments rep
#! @Returns the dimension of the representation <A>rep</A> as an integer
#! @Description Takes a Majorana representation <A>rep</A>, as outputted by
#! <Ref Func="MajoranaRepresentation"/> and returns its dimension as a
#! vector space. If the representation is not complete
#! (cf. <Ref Func="MAJORANA_IsComplete"/> ) then this value might not be
#! the true dimension of the algebra.
DeclareGlobalFunction( "MAJORANA_Dimension" );

#! @Arguments index, eval, rep
#! @Returns a basis of the eigenspace of the axis as position <A>index</A> with
#! eigenvalue <A>eval</A> as a sparse matrix
DeclareGlobalFunction( "MAJORANA_Eigenvectors" );

#! @Arguments rep
#! @Returns a sparse matrix that gives a basis of the algebra
DeclareGlobalFunction( "MAJORANA_Basis" );

#! @Arguments axis, basis, rep
#! @Returns a sparse matrix representing the adjoint action of <A>axis</A> on
#! <A>basis</A>
#! @Description Takes a Majorana representation <A>rep</A>, as outputted by
#! <Ref Func="MajoranaRepresentation"/>, a row vector <A>axis</A> in sparse
#! matrix format and a set of basis vectors, also in sparse matrix format.
#! Returns a matrix, also in sparse matrix format, that represents the
#! adjoint action of <A>axis</A> on <A>basis</A>.
DeclareGlobalFunction( "MAJORANA_AdjointAction" );

#! @Section The subalgebra structure

#! @Arguments vecs, rep
#! @Returns the subalgebra of the representation <A>rep</A> that is generated by
#! <A>vecs</A>
#! @Description Takes a Majorana representation <A>rep</A>, as outputted by
#! <Ref Func="MajoranaRepresentation"/> and a set of vectors <A>vecs</A> in sparse
#! matrix format and returns the subalgebra generated by <A>vecs</A>, also
#! in sparse matrix format.
DeclareGlobalFunction( "MAJORANA_Subalgebra" );

#! @Arguments subalg, rep
#! @Returns true if the subalgebra <A>subalg</A> is a Jordan algebra, otherwise
#! returns false
#! @Description Takes a Majorana representation <A>rep</A>, as outputted by
#! <Ref Func="MajoranaRepresentation"/> and a subalgebra <A>subalg</A> of rep.
#! If this subalgebra is a Jordan algebra then function returns true, otherwise
#! returns false.
DeclareGlobalFunction( "MAJORANA_IsJordanAlgebra" );

DeclareGlobalFunction( "MAJORANA_ConvertToBasis" );

DeclareGlobalFunction( "MAJORANA_NaiveProduct" );

#! @BeginExampleSession
#! gap> G := G := AlternatingGroup(5);;
#! gap> T := AsList( ConjugacyClass(G, (1,2)(3,4)));;
#! gap> input := ShapesOfMajoranaRepresentation(G,T);;
#! gap> rep := MajoranaRepresentation(input, 2);;
#! gap> MAJORANA_IsComplete(rep);
#! false
#! gap> NClosedMajoranaRepresentation(rep);;
#! gap> MAJORANA_IsComplete(rep);
#! true
#! gap> MAJORANA_Dimension(rep);
#! 46
#! gap> basis := MAJORANA_Basis(rep);
#! <a 46 x 61 sparse matrix over Rationals>
#! gap> subalg := MAJORANA_Subalgebra(basis, rep);
#! <a 46 x 61 sparse matrix over Rationals>
#! gap> MAJORANA_IsJordanAlgebra(subalg, rep);
#! false
#! @EndExampleSession
