"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.getAwsKmsMrkAwareDiscoveryMultiKeyringBuilder = void 0;
const material_management_1 = require("@aws-crypto/material-management");
function getAwsKmsMrkAwareDiscoveryMultiKeyringBuilder(MrkAwareDiscoveryKeyring, MultiKeyring, defaultClientProvider) {
    //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.5
    //# The caller MUST provide:
    return function buildAwsKmsMrkAwareDiscoveryMultiKeyringNode({ regions, discoveryFilter, 
    //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.5
    //# If a regional client supplier is not passed,
    //# then a default MUST be created that takes a region string and
    //# generates a default AWS SDK client for the given region.
    clientProvider = defaultClientProvider, grantTokens, }) {
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.5
        //# If an empty set of Region is provided this function MUST fail.
        (0, material_management_1.needs)(regions.length, 'Configured regions must contain at least one region.');
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.5
        //# If
        //# any element of the set of regions is null or an empty string this
        //# function MUST fail.
        (0, material_management_1.needs)(regions.every((region) => typeof region === 'string' && !!region), 'Configured regions must not contain a null or empty string as a region.');
        const children = regions
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.5
            //# A set of AWS KMS clients MUST be created by calling regional client
            //# supplier for each region in the input set of regions.
            .map(clientProvider)
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.5
            //# Then a set of AWS KMS MRK Aware Symmetric Region Discovery Keyring
            //# (aws-kms-mrk-aware-symmetric-region-discovery-keyring.md) MUST be
            //# created for each AWS KMS client by initializing each keyring with
            .map((client) => {
            /* Postcondition: If the configured clientProvider is not able to create a client for a defined region, throw an error. */
            (0, material_management_1.needs)(client, 'Configured clientProvider is unable to create a client for a configured region.');
            return new MrkAwareDiscoveryKeyring({
                client,
                discoveryFilter,
                grantTokens,
            });
        });
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.5
        //# Then a Multi-Keyring (../multi-keyring.md#inputs) MUST be initialize
        //# by using this set of discovery keyrings as the child keyrings
        //# (../multi-keyring.md#child-keyrings).
        //
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.5
        //# This Multi-Keyring MUST be
        //# this functions output.
        return new MultiKeyring({ children });
    };
}
exports.getAwsKmsMrkAwareDiscoveryMultiKeyringBuilder = getAwsKmsMrkAwareDiscoveryMultiKeyringBuilder;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia21zX21ya19kaXNjb3ZlcnlfbXVsdGlfa2V5cmluZy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9rbXNfbXJrX2Rpc2NvdmVyeV9tdWx0aV9rZXlyaW5nLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSxvRUFBb0U7QUFDcEUsc0NBQXNDOzs7QUFFdEMseUVBS3dDO0FBdUJ4QyxTQUFnQiw2Q0FBNkMsQ0FJM0Qsd0JBRUMsRUFDRCxZQUFzQyxFQUN0QyxxQkFBZ0Q7SUFFaEQseUVBQXlFO0lBQ3pFLDRCQUE0QjtJQUM1QixPQUFPLFNBQVMsNENBQTRDLENBQUMsRUFDM0QsT0FBTyxFQUNQLGVBQWU7SUFDZix5RUFBeUU7SUFDekUsZ0RBQWdEO0lBQ2hELGlFQUFpRTtJQUNqRSw0REFBNEQ7SUFDNUQsY0FBYyxHQUFHLHFCQUFxQixFQUN0QyxXQUFXLEdBQ3NDO1FBQ2pELHlFQUF5RTtRQUN6RSxrRUFBa0U7UUFDbEUsSUFBQSwyQkFBSyxFQUNILE9BQU8sQ0FBQyxNQUFNLEVBQ2Qsc0RBQXNELENBQ3ZELENBQUE7UUFFRCx5RUFBeUU7UUFDekUsTUFBTTtRQUNOLHFFQUFxRTtRQUNyRSx1QkFBdUI7UUFDdkIsSUFBQSwyQkFBSyxFQUNILE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUFDLE9BQU8sTUFBTSxLQUFLLFFBQVEsSUFBSSxDQUFDLENBQUMsTUFBTSxDQUFDLEVBQ2pFLHlFQUF5RSxDQUMxRSxDQUFBO1FBRUQsTUFBTSxRQUFRLEdBQ1osT0FBTztZQUNMLHlFQUF5RTtZQUN6RSx1RUFBdUU7WUFDdkUseURBQXlEO2FBQ3hELEdBQUcsQ0FBQyxjQUFjLENBQUM7WUFDcEIseUVBQXlFO1lBQ3pFLHNFQUFzRTtZQUN0RSxxRUFBcUU7WUFDckUscUVBQXFFO2FBQ3BFLEdBQUcsQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFO1lBQ2QsMEhBQTBIO1lBQzFILElBQUEsMkJBQUssRUFDSCxNQUFNLEVBQ04saUZBQWlGLENBQ2xGLENBQUE7WUFDRCxPQUFPLElBQUksd0JBQXdCLENBQUM7Z0JBQ2xDLE1BQU07Z0JBQ04sZUFBZTtnQkFDZixXQUFXO2FBQ1osQ0FBQyxDQUFBO1FBQ0osQ0FBQyxDQUFDLENBQUE7UUFFTix5RUFBeUU7UUFDekUsd0VBQXdFO1FBQ3hFLGlFQUFpRTtRQUNqRSx5Q0FBeUM7UUFDekMsRUFBRTtRQUNGLHlFQUF5RTtRQUN6RSw4QkFBOEI7UUFDOUIsMEJBQTBCO1FBQzFCLE9BQU8sSUFBSSxZQUFZLENBQUMsRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFBO0lBQ3ZDLENBQUMsQ0FBQTtBQUNILENBQUM7QUF2RUQsc0dBdUVDIn0=