"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.VerifyStream = void 0;
// @ts-ignore
const readable_stream_1 = require("readable-stream");
const material_management_node_1 = require("@aws-crypto/material-management-node");
const serialize_1 = require("@aws-crypto/serialize");
const parse_header_stream_1 = require("./parse_header_stream");
const PortableTransformWithType = readable_stream_1.Transform;
class VerifyStream extends PortableTransformWithType {
    _headerInfo;
    _verifyState = {
        buffer: Buffer.alloc(0),
        authTagBuffer: Buffer.alloc(0),
        signatureInfo: Buffer.alloc(0),
        sequenceNumber: 0,
        finalAuthTagReceived: false,
    };
    _verify;
    _maxBodySize;
    constructor({ maxBodySize }) {
        super();
        /* Precondition: VerifyStream requires maxBodySize must be falsey or a number. */
        (0, material_management_node_1.needs)(!maxBodySize || typeof maxBodySize === 'number', 'Unsupported MaxBodySize.');
        Object.defineProperty(this, '_maxBodySize', {
            value: maxBodySize,
            enumerable: true,
        });
        this.on('pipe', (source) => {
            /* Precondition: The source must a ParseHeaderStream emit the required events. */
            (0, material_management_node_1.needs)(source instanceof parse_header_stream_1.ParseHeaderStream, 'Unsupported source');
            source.once('VerifyInfo', (verifyInfo) => {
                const { getDecipher, verify, headerInfo, dispose } = verifyInfo;
                const { messageId, contentType } = headerInfo.messageHeader;
                /* If I have a verify, the header needs to be flushed through.
                 * I do it here for initialize the verifier before I even
                 * add the element to the object.
                 */
                if (verify) {
                    const { rawHeader, headerAuth, messageHeader } = headerInfo;
                    const { headerIv, headerAuthTag } = headerAuth;
                    verify.update(rawHeader);
                    verify.update((0, serialize_1.serializeMessageHeaderAuth)({
                        headerIv,
                        headerAuthTag,
                        messageHeader,
                    }));
                }
                Object.defineProperty(this, '_headerInfo', {
                    value: headerInfo,
                    enumerable: true,
                });
                Object.defineProperty(this, '_verify', {
                    value: verify,
                    enumerable: true,
                });
                const decipherInfo = {
                    messageId: Buffer.from(messageId.buffer || messageId, messageId.byteOffset || 0, messageId.byteLength),
                    contentType,
                    getDecipher,
                    dispose,
                };
                this.emit('DecipherInfo', decipherInfo);
            });
        });
    }
    _transform(chunk, enc, callback) {
        /* Precondition: VerifyInfo must have initialized the stream. */
        (0, material_management_node_1.needs)(this._headerInfo, 'VerifyStream not configured, VerifyInfo event not yet received.');
        // BodyHeader
        const state = this._verifyState;
        const { currentFrame } = state;
        if (!currentFrame) {
            const { buffer } = state;
            // Buffer.concat can be expensive. If buffer is empty, just use the chunk.
            const frameBuffer = buffer.length > 0 ? Buffer.concat([buffer, chunk]) : chunk;
            const frameHeader = (0, serialize_1.decodeBodyHeader)(frameBuffer, this._headerInfo, 0);
            if (!frameHeader) {
                // Need more data
                state.buffer = frameBuffer;
                return callback();
            }
            /* Precondition: If maxBodySize was set I can not buffer more data than maxBodySize.
             * Before returning *any* cleartext, the stream **MUST** verify the decryption.
             * This means that I must buffer the message until the AuthTag is reached.
             */
            (0, material_management_node_1.needs)(!this._maxBodySize || this._maxBodySize >= frameHeader.contentLength, 'maxBodySize exceeded.');
            /* Keeping track of the sequence number myself. */
            state.sequenceNumber += 1;
            /* Precondition: The sequence number is required to monotonically increase, starting from 1.
             * This is to avoid a bad actor from abusing the sequence number on un-signed algorithm suites.
             * If the frame size matched the data format (say NDJSON),
             * then the data could be significantly altered just by rearranging the frames.
             * Non-framed data returns a sequenceNumber of 1.
             */
            (0, material_management_node_1.needs)(frameHeader.sequenceNumber === state.sequenceNumber, 'Encrypted body sequence out of order.');
            if (this._verify) {
                this._verify.update(frameBuffer.slice(0, frameHeader.readPos));
            }
            const tail = frameBuffer.slice(frameHeader.readPos);
            this.emit('BodyInfo', frameHeader);
            state.currentFrame = frameHeader;
            return setImmediate(() => this._transform(tail, enc, callback));
        }
        // Content
        const { contentLength } = currentFrame;
        if (chunk.length && contentLength > 0) {
            if (contentLength > chunk.length) {
                currentFrame.contentLength -= chunk.length;
                this.push(chunk);
                return callback();
            }
            else {
                const content = chunk.slice(0, contentLength);
                const tail = chunk.slice(content.length);
                this.push(content);
                currentFrame.contentLength = 0;
                return setImmediate(() => this._transform(tail, enc, callback));
            }
        }
        // AuthTag
        const { tagLength } = currentFrame;
        const tagLengthBytes = tagLength / 8;
        const { authTagBuffer } = state;
        if (chunk.length && tagLengthBytes > authTagBuffer.length) {
            const left = tagLengthBytes - authTagBuffer.length;
            if (left > chunk.length) {
                // Buffer.concat can be expensive. If buffer is empty, just use the chunk.
                state.authTagBuffer =
                    authTagBuffer.length > 0
                        ? Buffer.concat([authTagBuffer, chunk])
                        : chunk;
                return callback();
            }
            else {
                // Buffer.concat can be expensive. If buffer is empty, just use the chunk.
                const finalAuthTagBuffer = authTagBuffer.length > 0
                    ? Buffer.concat([authTagBuffer, chunk], tagLengthBytes)
                    : chunk.slice(0, tagLengthBytes);
                if (this._verify) {
                    this._verify.update(finalAuthTagBuffer);
                }
                /* Reset state.
                 * Ciphertext buffers and authTag buffers need to be cleared.
                 */
                state.buffer = Buffer.alloc(0);
                state.currentFrame = undefined;
                state.authTagBuffer = Buffer.alloc(0);
                /* After the final frame the file format is _much_ simpler.
                 * Making sure the cascading if blocks fall to the signature can be tricky and brittle.
                 * After the final frame, just moving on to concatenate the signature is much simpler.
                 */
                if (currentFrame.isFinalFrame) {
                    /* Signal that the we are at the end of the ciphertext.
                     * See decodeBodyHeader, non-framed will set isFinalFrame
                     * for the single frame.
                     */
                    this._verifyState.finalAuthTagReceived = true;
                    /* Overwriting the _transform function.
                     * Data flow control is now handled here.
                     */
                    this._transform = (chunk, _enc, callback) => {
                        if (chunk.length) {
                            state.signatureInfo = Buffer.concat([state.signatureInfo, chunk]);
                        }
                        callback();
                    };
                }
                const tail = chunk.slice(left);
                /* The decipher_stream uses the `AuthTag` event to flush the accumulated frame.
                 * This is because ciphertext should never be returned until it is verified.
                 * i.e. the auth tag checked.
                 * This can create an issue if the chucks and frame size are small.
                 * If the verify stream continues processing and sends the next auth tag,
                 * before the current auth tag has been completed.
                 * This is basically a back pressure issue.
                 * Since the frame size, and consequently the high water mark,
                 * can not be know when the stream is created,
                 * the internal stream state would need to be modified.
                 * I assert that a simple callback is a simpler way to handle this.
                 */
                const next = () => this._transform(tail, enc, callback);
                return this.emit('AuthTag', finalAuthTagBuffer, next);
            }
        }
        callback();
    }
    push(chunk, encoding) {
        // Typescript???? this._verify instanceof Verify is better....
        if (this._verify && chunk) {
            this._verify.update(chunk);
        }
        return super.push(chunk, encoding);
    }
    _flush(callback) {
        const { finalAuthTagReceived } = this._verifyState;
        /* Precondition: All ciphertext MUST have been received.
         * The verify stream has ended,
         * there will be no more data.
         * Therefore we MUST have reached the end.
         */
        if (!finalAuthTagReceived)
            return callback(new Error('Incomplete message'));
        /* Check for early return (Postcondition): If there is no verify stream do not attempt to verify. */
        if (!this._verify)
            return callback();
        try {
            const { signatureInfo } = this._verifyState;
            /* Precondition: The signature must be well formed. */
            const { buffer, byteOffset, byteLength } = (0, serialize_1.deserializeSignature)(signatureInfo);
            const signature = Buffer.from(buffer, byteOffset, byteLength);
            const isVerified = this._verify.awsCryptoVerify(signature);
            /* Postcondition: The signature must be valid. */
            (0, material_management_node_1.needs)(isVerified, 'Invalid Signature');
            callback();
        }
        catch (e) {
            callback(e);
        }
    }
}
exports.VerifyStream = VerifyStream;
//# sourceMappingURL=data:application/json;base64,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