// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { EncryptedDataKey } from './encrypted_data_key';
import { SignatureKey, VerificationKey } from './signature_key';
import { frozenClass, readOnlyProperty } from './immutable_class';
import { KeyringTraceFlag } from './keyring_trace';
import { NodeAlgorithmSuite } from './node_algorithms';
import { WebCryptoAlgorithmSuite } from './web_crypto_algorithms';
import { needs } from './needs';
import { validate, version } from 'uuid';
export const supportsKeyObject = (function () {
    try {
        const { KeyObject, createSecretKey } = require('crypto'); // eslint-disable-line @typescript-eslint/no-var-requires
        if (!KeyObject || !createSecretKey)
            return false;
        return { KeyObject, createSecretKey };
    }
    catch (ex) {
        return false;
    }
})();
/*
 * This public interface to the CryptographicMaterial object is provided for
 * developers of CMMs and keyrings only. If you are a user of the AWS Encryption
 * SDK and you are not developing your own CMMs and/or keyrings, you do not
 * need to use it and you should not do so.
 *
 * The CryptographicMaterial's purpose is to bind together all the required elements for
 * encrypting or decrypting a payload.
 * The functional data key (unencrypted or CryptoKey) is the most sensitive data and needs to
 * be protected.  The longer this data persists in memory the
 * greater the opportunity to be invalidated.  Because
 * a Caching CMM exists it is important to ensure that the
 * unencrypted data key and its meta data can not be manipulated,
 * and that the unencrypted data key can be zeroed when
 * it is no longer needed.
 */
const timingSafeEqual = (function () {
    try {
        /* It is possible for `require` to return an empty object, or an object
         * that does not implement `timingSafeEqual`.
         * in this case I need a fallback
         */
        const { timingSafeEqual: nodeTimingSafeEqual } = require('crypto'); // eslint-disable-line @typescript-eslint/no-var-requires
        return nodeTimingSafeEqual || portableTimingSafeEqual;
    }
    catch (e) {
        return portableTimingSafeEqual;
    }
    /* https://codahale.com/a-lesson-in-timing-attacks/ */
    function portableTimingSafeEqual(a, b) {
        /* It is *possible* that a runtime could optimize this constant time function.
         * Adding `eval` could prevent the optimization, but this is no guarantee.
         * The eval below is commented out
         * because if a browser is using a Content Security Policy with `'unsafe-eval'`
         * it would fail on this eval.
         * The value in attempting to ensure that this function is not optimized
         * is not worth the cost of making customers allow `'unsafe-eval'`.
         * If you want to copy this function for your own use,
         * please review the timing-attack link above.
         * Side channel attacks are pernicious and subtle.
         */
        // eval('') // eslint-disable-line no-eval
        /* Check for early return (Postcondition) UNTESTED: Size is well-know information
         * and does not leak information about contents.
         */
        if (a.byteLength !== b.byteLength)
            return false;
        let diff = 0;
        for (let i = 0; i < b.length; i++) {
            diff |= a[i] ^ b[i];
        }
        return diff === 0;
    }
})();
export class NodeBranchKeyMaterial {
    // all attributes are readonly so they are accessible outside the class but
    // they cannot be modified
    // since all fields are objects, keep them immutable from external changes via
    // shared memory
    // we want the branch key to be mutable within the class but immutable outside
    // the class
    _branchKey;
    constructor(branchKey, branchKeyIdentifier, branchKeyVersion, encryptionContext) {
        /* Precondition: Branch key must be a Buffer */
        needs(branchKey instanceof Buffer, 'Branch key must be a Buffer');
        /* Precondition: Branch key id must be a string */
        needs(typeof branchKeyIdentifier === 'string', 'Branch key id must be a string');
        /* Precondition: Branch key version must be a string */
        needs(typeof branchKeyVersion === 'string', 'Branch key version must be a string');
        /* Precondition: encryptionContext must be an object, even if it is empty */
        needs(encryptionContext && typeof encryptionContext === 'object', 'Encryption context must be an object');
        /* Precondition: branchKey must be a 32 byte-long buffer */
        needs(branchKey.length === 32, 'Branch key must be 32 bytes long');
        /* Precondition: branch key ID is required */
        needs(branchKeyIdentifier, 'Empty branch key ID');
        /* Precondition: branch key version must be valid version 4 uuid */
        needs(validate(branchKeyVersion) && version(branchKeyVersion) === 4, 'Branch key version must be valid version 4 uuid');
        /* Postcondition: branch key is immutable */
        this._branchKey = Buffer.from(branchKey);
        /* Postconditon: encryption context is immutable */
        this.encryptionContext = Object.freeze({ ...encryptionContext });
        this.branchKeyIdentifier = branchKeyIdentifier;
        this.branchKeyVersion = Buffer.from(branchKeyVersion, 'utf-8');
        Object.setPrototypeOf(this, NodeBranchKeyMaterial.prototype);
        /* Postcondition: instance is frozen */
        // preventing any modifications to its properties or methods.
        Object.freeze(this);
    }
    // makes the branch key public to users wrapped in immutable access
    branchKey() {
        return this._branchKey;
    }
    // this capability is not required of branch key materials according to the
    // specification. Using this is a good security practice so that the data
    // key's bytes are not preserved even in free memory
    zeroUnencryptedDataKey() {
        this._branchKey.fill(0);
        return this;
    }
}
// make the class immutable
frozenClass(NodeBranchKeyMaterial);
export class NodeEncryptionMaterial {
    suite;
    setUnencryptedDataKey;
    getUnencryptedDataKey;
    zeroUnencryptedDataKey;
    hasUnencryptedDataKey;
    keyringTrace = [];
    encryptedDataKeys;
    addEncryptedDataKey;
    setSignatureKey;
    signatureKey;
    encryptionContext;
    constructor(suite, encryptionContext) {
        /* Precondition: NodeEncryptionMaterial suite must be NodeAlgorithmSuite. */
        needs(suite instanceof NodeAlgorithmSuite, 'Suite must be a NodeAlgorithmSuite');
        this.suite = suite;
        /* Precondition: NodeEncryptionMaterial encryptionContext must be an object, even if it is empty. */
        needs(encryptionContext && typeof encryptionContext === 'object', 'Encryption context must be set');
        this.encryptionContext = Object.freeze({ ...encryptionContext });
        // EncryptionMaterial have generated a data key on setUnencryptedDataKey
        const setFlags = KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY;
        decorateCryptographicMaterial(this, setFlags);
        decorateEncryptionMaterial(this);
        Object.setPrototypeOf(this, NodeEncryptionMaterial.prototype);
        Object.freeze(this);
    }
    hasValidKey() {
        return this.hasUnencryptedDataKey;
    }
}
frozenClass(NodeEncryptionMaterial);
export class NodeDecryptionMaterial {
    suite;
    setUnencryptedDataKey;
    getUnencryptedDataKey;
    zeroUnencryptedDataKey;
    hasUnencryptedDataKey;
    keyringTrace = [];
    setVerificationKey;
    verificationKey;
    encryptionContext;
    constructor(suite, encryptionContext) {
        /* Precondition: NodeDecryptionMaterial suite must be NodeAlgorithmSuite. */
        needs(suite instanceof NodeAlgorithmSuite, 'Suite must be a NodeAlgorithmSuite');
        this.suite = suite;
        /* Precondition: NodeDecryptionMaterial encryptionContext must be an object, even if it is empty. */
        needs(encryptionContext && typeof encryptionContext === 'object', 'Encryption context must be set');
        this.encryptionContext = Object.freeze({ ...encryptionContext });
        // DecryptionMaterial have decrypted a data key on setUnencryptedDataKey
        const setFlags = KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY;
        decorateCryptographicMaterial(this, setFlags);
        decorateDecryptionMaterial(this);
        Object.setPrototypeOf(this, NodeDecryptionMaterial.prototype);
        Object.freeze(this);
    }
    hasValidKey() {
        return this.hasUnencryptedDataKey;
    }
}
frozenClass(NodeDecryptionMaterial);
export class WebCryptoEncryptionMaterial {
    suite;
    setUnencryptedDataKey;
    getUnencryptedDataKey;
    zeroUnencryptedDataKey;
    hasUnencryptedDataKey;
    keyringTrace = [];
    encryptedDataKeys;
    addEncryptedDataKey;
    setSignatureKey;
    signatureKey;
    setCryptoKey;
    getCryptoKey;
    hasCryptoKey;
    validUsages;
    encryptionContext;
    constructor(suite, encryptionContext) {
        /* Precondition: WebCryptoEncryptionMaterial suite must be WebCryptoAlgorithmSuite. */
        needs(suite instanceof WebCryptoAlgorithmSuite, 'Suite must be a WebCryptoAlgorithmSuite');
        this.suite = suite;
        this.validUsages = Object.freeze([
            'deriveKey',
            'encrypt',
        ]);
        /* Precondition: WebCryptoEncryptionMaterial encryptionContext must be an object, even if it is empty. */
        needs(encryptionContext && typeof encryptionContext === 'object', 'Encryption context must be set');
        this.encryptionContext = Object.freeze({ ...encryptionContext });
        // EncryptionMaterial have generated a data key on setUnencryptedDataKey
        const setFlag = KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY;
        decorateCryptographicMaterial(this, setFlag);
        decorateEncryptionMaterial(this);
        decorateWebCryptoMaterial(this, setFlag);
        Object.setPrototypeOf(this, WebCryptoEncryptionMaterial.prototype);
        Object.freeze(this);
    }
    hasValidKey() {
        return this.hasUnencryptedDataKey && this.hasCryptoKey;
    }
}
frozenClass(WebCryptoEncryptionMaterial);
export class WebCryptoDecryptionMaterial {
    suite;
    setUnencryptedDataKey;
    getUnencryptedDataKey;
    zeroUnencryptedDataKey;
    hasUnencryptedDataKey;
    keyringTrace = [];
    setVerificationKey;
    verificationKey;
    setCryptoKey;
    getCryptoKey;
    hasCryptoKey;
    validUsages;
    encryptionContext;
    constructor(suite, encryptionContext) {
        /* Precondition: WebCryptoDecryptionMaterial suite must be WebCryptoAlgorithmSuite. */
        needs(suite instanceof WebCryptoAlgorithmSuite, 'Suite must be a WebCryptoAlgorithmSuite');
        this.suite = suite;
        this.validUsages = Object.freeze([
            'deriveKey',
            'decrypt',
        ]);
        /* Precondition: WebCryptoDecryptionMaterial encryptionContext must be an object, even if it is empty. */
        needs(encryptionContext && typeof encryptionContext === 'object', 'Encryption context must be set');
        this.encryptionContext = Object.freeze({ ...encryptionContext });
        // DecryptionMaterial have decrypted a data key on setUnencryptedDataKey
        const setFlag = KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY;
        decorateCryptographicMaterial(this, setFlag);
        decorateDecryptionMaterial(this);
        decorateWebCryptoMaterial(this, setFlag);
        Object.setPrototypeOf(this, WebCryptoDecryptionMaterial.prototype);
        Object.freeze(this);
    }
    hasValidKey() {
        return this.hasCryptoKey;
    }
}
frozenClass(WebCryptoDecryptionMaterial);
export function isEncryptionMaterial(obj) {
    return (obj instanceof WebCryptoEncryptionMaterial ||
        obj instanceof NodeEncryptionMaterial);
}
export function isDecryptionMaterial(obj) {
    return (obj instanceof WebCryptoDecryptionMaterial ||
        obj instanceof NodeDecryptionMaterial);
}
export function isBranchKeyMaterial(obj) {
    return obj instanceof NodeBranchKeyMaterial;
}
export function decorateCryptographicMaterial(material, setFlag) {
    /* Precondition: setFlag must be in the set of KeyringTraceFlag.SET_FLAGS. */
    needs(setFlag & KeyringTraceFlag.SET_FLAGS, 'Invalid setFlag');
    /* When a KeyringTraceFlag is passed to setUnencryptedDataKey,
     * it must be valid for the type of material.
     * It is invalid to claim that EncryptionMaterial were decrypted.
     */
    const deniedSetFlags = (KeyringTraceFlag.SET_FLAGS ^ setFlag) |
        (setFlag === KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY
            ? KeyringTraceFlag.DECRYPT_FLAGS
            : setFlag === KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY
                ? KeyringTraceFlag.ENCRYPT_FLAGS
                : 0);
    let unencryptedDataKeyZeroed = false;
    let unencryptedDataKey;
    // This copy of the unencryptedDataKey is stored to insure that the
    // unencrypted data key is *never* modified.  Since the
    // unencryptedDataKey is returned by reference, any change
    // to it would be propagated to any cached versions.
    let udkForVerification;
    const setUnencryptedDataKey = (dataKey, trace) => {
        /* Avoid making unnecessary copies of the dataKey. */
        const tempUdk = dataKey instanceof Uint8Array ? dataKey : unwrapDataKey(dataKey);
        /* All security conditions are tested here and failures will throw. */
        verifyUnencryptedDataKeyForSet(tempUdk, trace);
        unencryptedDataKey = wrapWithKeyObjectIfSupported(dataKey);
        udkForVerification = new Uint8Array(tempUdk);
        material.keyringTrace.push(trace);
        return material;
    };
    const getUnencryptedDataKey = () => {
        /* Precondition: unencryptedDataKey must be set before we can return it. */
        needs(unencryptedDataKey, 'unencryptedDataKey has not been set');
        /* Precondition: unencryptedDataKey must not be Zeroed out.
         * Returning a null key would be incredibly bad.
         */
        needs(!unencryptedDataKeyZeroed, 'unencryptedDataKey has been zeroed.');
        /* Precondition: The unencryptedDataKey must not have been modified.
         * If the unencryptedDataKey is a KeyObject,
         * then the security around modification is handled in C.
         * Do not duplicate the secret just to check...
         */
        needs(!(unencryptedDataKey instanceof Uint8Array) ||
            timingSafeEqual(udkForVerification, unwrapDataKey(unencryptedDataKey)), 'unencryptedDataKey has been corrupted.');
        return unencryptedDataKey;
    };
    Object.defineProperty(material, 'hasUnencryptedDataKey', {
        // Check that we have both not zeroed AND that we have not set
        get: () => !!unencryptedDataKey && !unencryptedDataKeyZeroed,
        enumerable: true,
    });
    const zeroUnencryptedDataKey = () => {
        /* These checks are separated on purpose.  It should be impossible to have only one unset.
         * *But* if it was the case, I *must* make sure I zero out the set one, and not leave it up to GC.
         * If I only checked on say unencryptedDataKey, and udkForVerification was somehow set,
         * doing the simplest thing would be to set both to new Uint8Array.
         * Leaving udkForVerification to be garbage collected.
         * This level of insanity is due to the fact that we are dealing with the unencrypted data key.
         */
        let unsetCount = 0;
        /* Precondition: If the unencryptedDataKey has not been set, it should not be settable later. */
        if (!unencryptedDataKey) {
            unencryptedDataKey = new Uint8Array();
            unsetCount += 1;
        }
        /* Precondition: If the udkForVerification has not been set, it should not be settable later. */
        if (!udkForVerification) {
            udkForVerification = new Uint8Array();
            unsetCount += 1;
        }
        /* The KeyObject manages its own ref counter.
         * Once there are no more users, it will clean the memory.
         */
        if (!(unencryptedDataKey instanceof Uint8Array)) {
            unencryptedDataKey = new Uint8Array();
        }
        unencryptedDataKey.fill(0);
        udkForVerification.fill(0);
        unencryptedDataKeyZeroed = true;
        /* Postcondition UNTESTED: Both unencryptedDataKey and udkForVerification must be either set or unset.
         * If it is ever the case that only one was unset, then something is wrong in a profound way.
         * It is not clear how this could ever happen, unless someone is manipulating the OS...
         */
        needs(unsetCount === 0 || unsetCount === 2, 'Either unencryptedDataKey or udkForVerification was not set.');
        return material;
    };
    readOnlyProperty(material, 'setUnencryptedDataKey', setUnencryptedDataKey);
    readOnlyProperty(material, 'getUnencryptedDataKey', getUnencryptedDataKey);
    readOnlyProperty(material, 'zeroUnencryptedDataKey', zeroUnencryptedDataKey);
    return material;
    function verifyUnencryptedDataKeyForSet(dataKey, trace) {
        /* Precondition: unencryptedDataKey must not be set.  Modifying the unencryptedDataKey is denied */
        needs(!unencryptedDataKey, 'unencryptedDataKey has already been set');
        /* Precondition: dataKey must be Binary Data */
        needs(dataKey instanceof Uint8Array, 'dataKey must be a Uint8Array');
        /* Precondition: dataKey should have an ArrayBuffer that *only* stores the key.
         * This is a simple check to make sure that the key is not stored on
         * a large potentially shared ArrayBuffer.
         * If this was the case, it may be possible to find or manipulate.
         */
        needs(dataKey.byteOffset === 0, 'Unencrypted Master Key must be an isolated buffer.');
        /* Precondition: The data key length must agree with algorithm specification.
         * If this is not the case, it either means ciphertext was tampered
         * with or the keyring implementation is not setting the length properly.
         */
        needs(dataKey.byteLength === material.suite.keyLengthBytes, 'Key length does not agree with the algorithm specification.');
        /* Precondition: Trace must be set, and the flag must indicate that the data key was generated. */
        needs(trace && trace.keyName && trace.keyNamespace, 'Malformed KeyringTrace');
        /* Precondition: On set the required KeyringTraceFlag must be set. */
        needs(trace.flags & setFlag, 'Required KeyringTraceFlag not set');
        /* Precondition: Only valid flags are allowed.
         * An unencrypted data key can not be both generated and decrypted.
         */
        needs(!(trace.flags & deniedSetFlags), 'Invalid KeyringTraceFlags set.');
    }
}
export function decorateEncryptionMaterial(material) {
    const deniedEncryptFlags = KeyringTraceFlag.SET_FLAGS | KeyringTraceFlag.DECRYPT_FLAGS;
    const encryptedDataKeys = [];
    let signatureKey;
    const addEncryptedDataKey = (edk, flags) => {
        /* Precondition: If a data key has not already been generated, there must be no EDKs.
         * Pushing EDKs on the list before the data key has been generated may cause the list of
         * EDKs to be inconsistent. (i.e., they would decrypt to different data keys.)
         */
        needs(material.hasUnencryptedDataKey, 'Unencrypted data key not set.');
        /* Precondition: Edk must be EncryptedDataKey
         * Putting things onto the list that are not EncryptedDataKey
         * may cause the list of EDKs to be inconsistent. (i.e. they may not serialize, or be mutable)
         */
        needs(edk instanceof EncryptedDataKey, 'Unsupported instance of encryptedDataKey');
        /* Precondition: flags must indicate that the key was encrypted. */
        needs(flags & KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY, 'Encrypted data key flag must be set.');
        /* Precondition: flags must not include a setFlag or a decrypt flag.
         * The setFlag is reserved for setting the unencrypted data key
         * and must only occur once in the set of KeyringTrace flags.
         * The two setFlags in use are:
         * KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY
         * KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY
         *
         * KeyringTraceFlag.WRAPPING_KEY_VERIFIED_ENC_CTX is reserved for the decrypt path
         */
        needs(!(flags & deniedEncryptFlags), 'Invalid flag for EncryptedDataKey.');
        material.keyringTrace.push({
            keyName: edk.providerInfo,
            keyNamespace: edk.providerId,
            flags,
        });
        encryptedDataKeys.push(edk);
        return material;
    };
    readOnlyProperty(material, 'addEncryptedDataKey', addEncryptedDataKey);
    Object.defineProperty(material, 'encryptedDataKeys', {
        // I only want EDKs added through addEncryptedDataKey
        // so I return a new array
        get: () => [...encryptedDataKeys],
        enumerable: true,
    });
    const setSignatureKey = (key) => {
        /* Precondition: The SignatureKey stored must agree with the algorithm specification.
         * If this is not the case it means the MaterialManager or Keyring is not setting
         * the SignatureKey correctly
         */
        needs(material.suite.signatureCurve, 'Algorithm specification does not support signatures.');
        /* Precondition: signatureKey must not be set.  Modifying the signatureKey is denied. */
        needs(!signatureKey, 'Signature key has already been set.');
        /* Precondition: key must be a SignatureKey. */
        needs(key instanceof SignatureKey, 'Unsupported instance of key');
        signatureKey = key;
        return material;
    };
    readOnlyProperty(material, 'setSignatureKey', setSignatureKey);
    Object.defineProperty(material, 'signatureKey', {
        get: () => {
            /* Precondition: The SignatureKey requested must agree with the algorithm specification.
             * If this is not the case it means the MaterialManager or Keyring is not setting
             * the SignatureKey correctly
             */
            needs(!!material.suite.signatureCurve === !!signatureKey, 'Algorithm specification not satisfied.');
            return signatureKey;
        },
        enumerable: true,
    });
    return material;
}
export function decorateDecryptionMaterial(material) {
    // Verification Key
    let verificationKey;
    const setVerificationKey = (key) => {
        /* Precondition: The VerificationKey stored must agree with the algorithm specification.
         * If this is not the case it means the MaterialManager or Keyring is not setting
         * the VerificationKey correctly
         */
        needs(material.suite.signatureCurve, 'Algorithm specification does not support signatures.');
        /* Precondition: verificationKey must not be set.  Modifying the verificationKey is denied. */
        needs(!verificationKey, 'Verification key has already been set.');
        /* Precondition: key must be a VerificationKey. */
        needs(key instanceof VerificationKey, 'Unsupported instance of key');
        verificationKey = key;
        return material;
    };
    readOnlyProperty(material, 'setVerificationKey', setVerificationKey);
    Object.defineProperty(material, 'verificationKey', {
        get: () => {
            /* Precondition: The VerificationKey requested must agree with the algorithm specification.
             * If this is not the case it means the MaterialManager or Keyring is not setting
             * the VerificationKey correctly
             */
            needs(!!material.suite.signatureCurve === !!verificationKey, 'Algorithm specification not satisfied.');
            return verificationKey;
        },
        enumerable: true,
    });
    return material;
}
export function decorateWebCryptoMaterial(material, setFlags) {
    let cryptoKey;
    const setCryptoKey = (dataKey, trace) => {
        /* Precondition: cryptoKey must not be set.  Modifying the cryptoKey is denied */
        needs(!cryptoKey, 'cryptoKey is already set.');
        /* Precondition: dataKey must be a supported type. */
        needs(isCryptoKey(dataKey) || isMixedBackendCryptoKey(dataKey), 'Unsupported dataKey type.');
        /* Precondition: The CryptoKey must match the algorithm suite specification. */
        needs(isValidCryptoKey(dataKey, material), 'CryptoKey settings not acceptable.');
        /* If the material does not have an unencrypted data key,
         * then we are setting the crypto key here and need a keyring trace .
         */
        if (!material.hasUnencryptedDataKey) {
            /* Precondition: If the CryptoKey is the only version, the trace information must be set here. */
            needs(trace && trace.keyName && trace.keyNamespace, 'Malformed KeyringTrace');
            /* Precondition: On setting the CryptoKey the required KeyringTraceFlag must be set. */
            needs(trace.flags & setFlags, 'Required KeyringTraceFlag not set');
            /* If I a setting a cryptoKey without an unencrypted data key,
             * an unencrypted data should never be set.
             * The expectation is if you are setting the cryptoKey *first* then
             * the unencrypted data key has already been "handled".
             * This ensures that a cryptoKey and an unencrypted data key always match.
             */
            material.zeroUnencryptedDataKey();
            material.keyringTrace.push(trace);
        }
        if (isCryptoKey(dataKey)) {
            cryptoKey = dataKey;
        }
        else {
            const { zeroByteCryptoKey, nonZeroByteCryptoKey } = dataKey;
            cryptoKey = Object.freeze({ zeroByteCryptoKey, nonZeroByteCryptoKey });
        }
        return material;
    };
    readOnlyProperty(material, 'setCryptoKey', setCryptoKey);
    const getCryptoKey = () => {
        /* Precondition: The cryptoKey must be set before we can return it. */
        needs(cryptoKey, 'Crypto key is not set.');
        // In the case of MixedBackendCryptoKey the object
        // has already been frozen above so it is safe to return
        return cryptoKey;
    };
    readOnlyProperty(material, 'getCryptoKey', getCryptoKey);
    Object.defineProperty(material, 'hasCryptoKey', {
        get: () => !!cryptoKey,
        enumerable: true,
    });
    return material;
}
export function isCryptoKey(dataKey) {
    return (dataKey &&
        'algorithm' in dataKey &&
        'type' in dataKey &&
        'usages' in dataKey &&
        'extractable' in dataKey);
}
export function isValidCryptoKey(dataKey, material) {
    if (!isCryptoKey(dataKey)) {
        const { zeroByteCryptoKey, nonZeroByteCryptoKey } = dataKey;
        return (isValidCryptoKey(zeroByteCryptoKey, material) &&
            isValidCryptoKey(nonZeroByteCryptoKey, material));
    }
    const { suite, validUsages } = material;
    const { encryption, keyLength, kdf } = suite;
    /* See:
     * https://developer.mozilla.org/en-US/docs/Web/API/CryptoKey
     * https://developer.mozilla.org/en-US/docs/Web/API/AesKeyGenParams
     */
    const { type, algorithm, usages, extractable } = dataKey;
    // @ts-ignore length is an optional value...
    const { name, length } = algorithm;
    /* MSRCrypto, for legacy reasons,
     * normalizes the algorithm name
     * to lower case.
     * https://github.com/microsoft/MSR-JavaScript-Crypto/issues/1
     * For now, I'm going to upper case the name.
     */
    // Only symmetric algorithms
    return (type === 'secret' &&
        // Must match the suite
        ((kdf && name.toUpperCase() === kdf) ||
            (name.toUpperCase() === encryption && length === keyLength)) &&
        /* Only valid usage are: encrypt|decrypt|deriveKey
         * The complexity between deriveKey and suite.kdf should be handled in the Material class.
         */
        usages.some((u) => validUsages.includes(u)) &&
        // Since CryptoKey can not be zeroized, not extractable is the next best thing
        !extractable);
}
function isMixedBackendCryptoKey(dataKey) {
    const { zeroByteCryptoKey, nonZeroByteCryptoKey } = dataKey;
    return isCryptoKey(zeroByteCryptoKey) && isCryptoKey(nonZeroByteCryptoKey);
}
export function keyUsageForMaterial(material) {
    const { suite } = material;
    if (suite.kdf)
        return 'deriveKey';
    return subtleFunctionForMaterial(material);
}
export function subtleFunctionForMaterial(material) {
    if (material instanceof WebCryptoEncryptionMaterial)
        return 'encrypt';
    if (material instanceof WebCryptoDecryptionMaterial)
        return 'decrypt';
    throw new Error('Unsupported material');
}
export function unwrapDataKey(dataKey) {
    if (dataKey instanceof Uint8Array)
        return dataKey;
    if (supportsKeyObject && dataKey instanceof supportsKeyObject.KeyObject)
        return dataKey.export();
    throw new Error('Unsupported dataKey type');
}
export function wrapWithKeyObjectIfSupported(dataKey) {
    if (supportsKeyObject) {
        if (dataKey instanceof Uint8Array) {
            const ko = supportsKeyObject.createSecretKey(dataKey);
            /* Postcondition: Zero the secret.  It is now inside the KeyObject. */
            dataKey.fill(0);
            return ko;
        }
        if (dataKey instanceof supportsKeyObject.KeyObject)
            return dataKey;
    }
    else if (dataKey instanceof Uint8Array) {
        return dataKey;
    }
    throw new Error('Unsupported dataKey type');
}
//# sourceMappingURL=data:application/json;base64,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