// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import BN from 'bn.js';
import { needs } from './needs';
const prime256v1 = eccEncodeCompressedPoint(32);
const secp384r1 = eccEncodeCompressedPoint(48);
export const encodeNamedCurves = Object.freeze({
    // NodeJS/OpenSSL names
    prime256v1,
    secp384r1,
    // WebCrypto/Browser names
    'P-256': prime256v1,
    'P-384': secp384r1,
});
/*
 * 1. This only works for prime curves
 * 2. This will not handle the point at infinity
 */
function eccEncodeCompressedPoint(keyLength) {
    return function encode(publicKey) {
        /* Precondition: publicKey must be the right length.
         * The format for the public key is [type, ...keyLength, ...keyLength]
         */
        needs(publicKey.byteLength === 1 + keyLength * 2, 'Malformed public key.');
        // const type = publicKey[0]
        const x = publicKey.slice(1, keyLength + 1);
        const y = publicKey.slice(keyLength + 1, keyLength * 2 + 1);
        const yOrder = new BN([...y]).mod(new BN(2)).toNumber() + 2;
        const compressPoint = new Uint8Array(1 + x.length);
        compressPoint.set([yOrder], 0);
        compressPoint.set(x, 1);
        return compressPoint;
    };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZWNjX2VuY29kZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9lY2NfZW5jb2RlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBLG9FQUFvRTtBQUNwRSxzQ0FBc0M7QUFFdEMsT0FBTyxFQUFFLE1BQU0sT0FBTyxDQUFBO0FBRXRCLE9BQU8sRUFBRSxLQUFLLEVBQUUsTUFBTSxTQUFTLENBQUE7QUFFL0IsTUFBTSxVQUFVLEdBQUcsd0JBQXdCLENBQUMsRUFBRSxDQUFDLENBQUE7QUFDL0MsTUFBTSxTQUFTLEdBQUcsd0JBQXdCLENBQUMsRUFBRSxDQUFDLENBQUE7QUFROUMsTUFBTSxDQUFDLE1BQU0saUJBQWlCLEdBQWdDLE1BQU0sQ0FBQyxNQUFNLENBQUM7SUFDMUUsdUJBQXVCO0lBQ3ZCLFVBQVU7SUFDVixTQUFTO0lBQ1QsMEJBQTBCO0lBQzFCLE9BQU8sRUFBRSxVQUFVO0lBQ25CLE9BQU8sRUFBRSxTQUFTO0NBQ25CLENBQUMsQ0FBQTtBQUVGOzs7R0FHRztBQUNILFNBQVMsd0JBQXdCLENBQUMsU0FBaUI7SUFDakQsT0FBTyxTQUFTLE1BQU0sQ0FBQyxTQUFxQjtRQUMxQzs7V0FFRztRQUNILEtBQUssQ0FBQyxTQUFTLENBQUMsVUFBVSxLQUFLLENBQUMsR0FBRyxTQUFTLEdBQUcsQ0FBQyxFQUFFLHVCQUF1QixDQUFDLENBQUE7UUFFMUUsNEJBQTRCO1FBQzVCLE1BQU0sQ0FBQyxHQUFHLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLFNBQVMsR0FBRyxDQUFDLENBQUMsQ0FBQTtRQUMzQyxNQUFNLENBQUMsR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDLFNBQVMsR0FBRyxDQUFDLEVBQUUsU0FBUyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQTtRQUUzRCxNQUFNLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxRQUFRLEVBQUUsR0FBRyxDQUFDLENBQUE7UUFFM0QsTUFBTSxhQUFhLEdBQUcsSUFBSSxVQUFVLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQTtRQUNsRCxhQUFhLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUE7UUFDOUIsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUE7UUFFdkIsT0FBTyxhQUFhLENBQUE7SUFDdEIsQ0FBQyxDQUFBO0FBQ0gsQ0FBQyJ9