"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SamlAuthRoutes = void 0;
var _configSchema = require("@osd/config-schema");
var _next_url = require("../../../utils/next_url");
var _common = require("../../../../common");
var _cookie_splitter = require("../../../session/cookie_splitter");
/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */

class SamlAuthRoutes {
  constructor(router,
  // @ts-ignore: unused variable
  config, sessionStorageFactory, securityClient, coreSetup) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.securityClient = securityClient;
    this.coreSetup = coreSetup;
  }
  getExtraAuthStorageOptions(logger) {
    // If we're here, we will always have the openid configuration
    return {
      cookiePrefix: this.config.saml.extra_storage.cookie_prefix,
      additionalCookies: this.config.saml.extra_storage.additional_cookies,
      logger
    };
  }
  setupRoutes() {
    this.router.get({
      path: _common.SAML_AUTH_LOGIN,
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: nexturl => {
              return (0, _next_url.validateNextUrl)(nexturl, this.coreSetup.http.basePath.serverBasePath);
            }
          })),
          redirectHash: _configSchema.schema.string()
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      if (request.auth.isAuthenticated) {
        return response.redirected({
          headers: {
            location: `${this.coreSetup.http.basePath.serverBasePath}/app/opensearch-dashboards`
          }
        });
      }
      try {
        const samlHeader = await this.securityClient.getSamlHeader(request);
        // const { nextUrl = '/' } = request.query;
        const cookie = {
          saml: {
            nextUrl: request.query.nextUrl,
            requestId: samlHeader.requestId,
            redirectHash: request.query.redirectHash === 'true'
          }
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location: samlHeader.location
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`Failed to get saml header: ${error}`);
        return response.internalError(); // TODO: redirect to error page?
      }
    });

    this.router.post({
      path: `/_opendistro/_security/saml/acs`,
      validate: {
        body: _configSchema.schema.any()
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      let requestId = '';
      let nextUrl = '/';
      let redirectHash = false;
      try {
        const cookie = await this.sessionStorageFactory.asScoped(request).get();
        if (cookie) {
          var _cookie$saml, _cookie$saml2, _cookie$saml3;
          requestId = ((_cookie$saml = cookie.saml) === null || _cookie$saml === void 0 ? void 0 : _cookie$saml.requestId) || '';
          nextUrl = ((_cookie$saml2 = cookie.saml) === null || _cookie$saml2 === void 0 ? void 0 : _cookie$saml2.nextUrl) || `${this.coreSetup.http.basePath.serverBasePath}/app/opensearch-dashboards`;
          redirectHash = ((_cookie$saml3 = cookie.saml) === null || _cookie$saml3 === void 0 ? void 0 : _cookie$saml3.redirectHash) || false;
        }
        if (!requestId) {
          return response.badRequest({
            body: 'Invalid requestId'
          });
        }
      } catch (error) {
        context.security_plugin.logger.error(`Failed to parse cookie: ${error}`);
        return response.badRequest();
      }
      try {
        const credentials = await this.securityClient.authToken({
          requestId,
          samlResponse: request.body.SAMLResponse,
          acsEndpoint: undefined,
          authRequestType: _common.AuthType.SAML
        });
        const user = await this.securityClient.authenticateWithHeader(request, 'authorization', credentials.authorization);
        let expiryTime = Date.now() + this.config.session.ttl;
        const [headerEncoded, payloadEncoded, signature] = credentials.authorization.split('.');
        if (!payloadEncoded) {
          context.security_plugin.logger.error('JWT token payload not found');
        }
        const tokenPayload = JSON.parse(Buffer.from(payloadEncoded, 'base64').toString());
        if (tokenPayload.exp) {
          expiryTime = parseInt(tokenPayload.exp, 10) * 1000;
        }
        const cookie = {
          username: user.username,
          credentials: {
            authHeaderValueExtra: true
          },
          authType: _common.AuthType.SAML,
          // TODO: create constant
          expiryTime
        };
        (0, _cookie_splitter.setExtraAuthStorage)(request, credentials.authorization, this.getExtraAuthStorageOptions(context.security_plugin.logger));
        this.sessionStorageFactory.asScoped(request).set(cookie);
        if (redirectHash) {
          return response.redirected({
            headers: {
              location: `${this.coreSetup.http.basePath.serverBasePath}/auth/saml/redirectUrlFragment?nextUrl=${escape(nextUrl)}`
            }
          });
        } else {
          return response.redirected({
            headers: {
              location: nextUrl
            }
          });
        }
      } catch (error) {
        context.security_plugin.logger.error(`SAML SP initiated authentication workflow failed: ${error}`);
      }
      return response.internalError();
    });
    this.router.post({
      path: `/_opendistro/_security/saml/acs/idpinitiated`,
      validate: {
        body: _configSchema.schema.any()
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      const acsEndpoint = `${this.coreSetup.http.basePath.serverBasePath}/_opendistro/_security/saml/acs/idpinitiated`;
      try {
        const credentials = await this.securityClient.authToken({
          requestId: undefined,
          samlResponse: request.body.SAMLResponse,
          acsEndpoint,
          authRequestType: _common.AuthType.SAML
        });
        const user = await this.securityClient.authenticateWithHeader(request, 'authorization', credentials.authorization);
        let expiryTime = Date.now() + this.config.session.ttl;
        const [headerEncoded, payloadEncoded, signature] = credentials.authorization.split('.');
        if (!payloadEncoded) {
          context.security_plugin.logger.error('JWT token payload not found');
        }
        const tokenPayload = JSON.parse(Buffer.from(payloadEncoded, 'base64').toString());
        if (tokenPayload.exp) {
          expiryTime = parseInt(tokenPayload.exp, 10) * 1000;
        }
        const cookie = {
          username: user.username,
          credentials: {
            authHeaderValueExtra: true
          },
          authType: _common.AuthType.SAML,
          // TODO: create constant
          expiryTime
        };
        (0, _cookie_splitter.setExtraAuthStorage)(request, credentials.authorization, this.getExtraAuthStorageOptions(context.security_plugin.logger));
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location: `${this.coreSetup.http.basePath.serverBasePath}/app/opensearch-dashboards`
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`SAML IDP initiated authentication workflow failed: ${error}`);
      }
      return response.internalError();
    });

    // captureUrlFragment is the first route that will be invoked in the SP initiated login.
    // This route will execute the captureUrlFragment.js script.
    this.coreSetup.http.resources.register({
      path: '/auth/saml/captureUrlFragment',
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: nexturl => {
              return (0, _next_url.validateNextUrl)(nexturl, this.coreSetup.http.basePath.serverBasePath);
            }
          }))
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      const serverBasePath = this.coreSetup.http.basePath.serverBasePath;
      return response.renderHtml({
        body: `
            <!DOCTYPE html>
            <title>OSD SAML Capture</title>
            <link rel="icon" href="data:,">
            <script src="${serverBasePath}/auth/saml/captureUrlFragment.js"></script>
          `
      });
    });

    // This script will store the URL Hash in browser's local storage.
    this.coreSetup.http.resources.register({
      path: '/auth/saml/captureUrlFragment.js',
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      return response.renderJs({
        body: `let samlHash=window.location.hash.toString();
                 let redirectHash = false;
                 if (samlHash !== "") {
                    window.localStorage.removeItem('samlHash');
                    window.localStorage.setItem('samlHash', samlHash);
                     redirectHash = true;
                  }
                 let params = new URLSearchParams(window.location.search);
                 let nextUrl = params.get("nextUrl");
                 finalUrl = "login?redirectHash=" + encodeURIComponent(redirectHash);
                 if (!!nextUrl) {
                   finalUrl += "&nextUrl=" + encodeURIComponent(nextUrl);
                 }
                 window.location.replace(finalUrl);
                `
      });
    });

    //  Once the User is authenticated via the '_opendistro/_security/saml/acs' route,
    //  the browser will be redirected to '/auth/saml/redirectUrlFragment' route,
    //  which will execute the redirectUrlFragment.js.
    this.coreSetup.http.resources.register({
      path: '/auth/saml/redirectUrlFragment',
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.any()
        })
      },
      options: {
        authRequired: true
      }
    }, async (context, request, response) => {
      const serverBasePath = this.coreSetup.http.basePath.serverBasePath;
      return response.renderHtml({
        body: `
            <!DOCTYPE html>
            <title>OSD SAML Success</title>
            <link rel="icon" href="data:,">
            <script src="${serverBasePath}/auth/saml/redirectUrlFragment.js"></script>
          `
      });
    });

    // This script will pop the Hash from local storage if it exists.
    // And forward the browser to the next url.
    this.coreSetup.http.resources.register({
      path: '/auth/saml/redirectUrlFragment.js',
      validate: false,
      options: {
        authRequired: true
      }
    }, async (context, request, response) => {
      return response.renderJs({
        body: `let samlHash=window.localStorage.getItem('samlHash');
                 window.localStorage.removeItem('samlHash');
                 let params = new URLSearchParams(window.location.search);
                 let nextUrl = params.get("nextUrl");
                 finalUrl = nextUrl + samlHash;
                 window.location.replace(finalUrl);
                `
      });
    });
    this.router.get({
      path: _common.SAML_AUTH_LOGOUT,
      validate: false
    }, async (context, request, response) => {
      try {
        const authInfo = await this.securityClient.authinfo(request);
        await (0, _cookie_splitter.clearSplitCookies)(request, this.getExtraAuthStorageOptions(context.security_plugin.logger));
        this.sessionStorageFactory.asScoped(request).clear();
        // TODO: need a default logout page
        const redirectUrl = authInfo.sso_logout_url || this.coreSetup.http.basePath.serverBasePath || '/';
        return response.redirected({
          headers: {
            location: redirectUrl
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`SAML logout failed: ${error}`);
        return response.badRequest();
      }
    });
  }
}
exports.SamlAuthRoutes = SamlAuthRoutes;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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