"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkErrorType = exports.callCluster = void 0;
exports.errorResponse = errorResponse;
exports.getFileName = getFileName;
exports.joinRequestParams = void 0;
exports.parseOpenSearchErrorResponse = parseOpenSearchErrorResponse;
var _uuid = require("uuid");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * OpenSearch error response body:
 *  {
 *   error: {
 *     root_cause: [{ type: 'status_exception', reason: 'test exception' }],
 *     type: 'status_exception',
 *     reason: 'test exception',
 *   },
 *   status: 404,
 * };
 *
 */
function parseOpenSearchErrorResponse(error) {
  if (error.response) {
    try {
      const opensearchErrorResponse = JSON.parse(error.response);
      return opensearchErrorResponse.error.reason || error.response;
    } catch (parsingError) {
      return error.response;
    }
  }
  return error.message;
}
function errorResponse(response, error) {
  return response.custom({
    statusCode: error.statusCode || 500,
    body: parseOpenSearchErrorResponse(error)
  });
}

/**
 * Generate report file name based on name and timestamp.
 * @param itemName      report item name
 * @param timeCreated   timestamp when this is being created
 */
function getFileName(itemName, timeCreated) {
  return `${itemName}_${timeCreated.toISOString()}_${(0, _uuid.v1)()}`;
}

/**
 * Call OpenSearch cluster function.
 * @param client    OpenSearch client
 * @param endpoint  OpenSearch API method
 * @param params    OpenSearch API parameters
 */
const callCluster = async (client, endpoint, params, isScheduledTask) => {
  let opensearchResp;
  if (isScheduledTask) {
    opensearchResp = await client.callAsInternalUser(endpoint, params);
  } else {
    opensearchResp = await client.callAsCurrentUser(endpoint, params);
  }
  return opensearchResp;
};
exports.callCluster = callCluster;
const checkErrorType = error => {
  if (error.statusCode && Math.floor(error.statusCode / 100) === 4) {
    return 'user_error';
  } else {
    return 'system_error';
  }
};
exports.checkErrorType = checkErrorType;
const joinRequestParams = queryParams => {
  if (Array.isArray(queryParams)) return queryParams.join(',');
  if (typeof queryParams === 'string') return queryParams;
  return '';
};
exports.joinRequestParams = joinRequestParams;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfdXVpZCIsInJlcXVpcmUiLCJwYXJzZU9wZW5TZWFyY2hFcnJvclJlc3BvbnNlIiwiZXJyb3IiLCJyZXNwb25zZSIsIm9wZW5zZWFyY2hFcnJvclJlc3BvbnNlIiwiSlNPTiIsInBhcnNlIiwicmVhc29uIiwicGFyc2luZ0Vycm9yIiwibWVzc2FnZSIsImVycm9yUmVzcG9uc2UiLCJjdXN0b20iLCJzdGF0dXNDb2RlIiwiYm9keSIsImdldEZpbGVOYW1lIiwiaXRlbU5hbWUiLCJ0aW1lQ3JlYXRlZCIsInRvSVNPU3RyaW5nIiwidXVpZHYxIiwiY2FsbENsdXN0ZXIiLCJjbGllbnQiLCJlbmRwb2ludCIsInBhcmFtcyIsImlzU2NoZWR1bGVkVGFzayIsIm9wZW5zZWFyY2hSZXNwIiwiY2FsbEFzSW50ZXJuYWxVc2VyIiwiY2FsbEFzQ3VycmVudFVzZXIiLCJleHBvcnRzIiwiY2hlY2tFcnJvclR5cGUiLCJNYXRoIiwiZmxvb3IiLCJqb2luUmVxdWVzdFBhcmFtcyIsInF1ZXJ5UGFyYW1zIiwiQXJyYXkiLCJpc0FycmF5Iiwiam9pbiJdLCJzb3VyY2VzIjpbImhlbHBlcnMudHMiXSwic291cmNlc0NvbnRlbnQiOlsiLypcbiAqIENvcHlyaWdodCBPcGVuU2VhcmNoIENvbnRyaWJ1dG9yc1xuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiAqL1xuXG5pbXBvcnQgeyBPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlRmFjdG9yeSB9IGZyb20gJy4uLy4uLy4uLy4uLy4uL3NyYy9jb3JlL3NlcnZlcic7XG5pbXBvcnQgeyB2MSBhcyB1dWlkdjEgfSBmcm9tICd1dWlkJztcbmltcG9ydCB7XG4gIElMZWdhY3lDbHVzdGVyQ2xpZW50LFxuICBJTGVnYWN5U2NvcGVkQ2x1c3RlckNsaWVudCxcbn0gZnJvbSAnLi4vLi4vLi4vLi4vLi4vc3JjL2NvcmUvc2VydmVyJztcblxuLyoqXG4gKiBPcGVuU2VhcmNoIGVycm9yIHJlc3BvbnNlIGJvZHk6XG4gKiAge1xuICogICBlcnJvcjoge1xuICogICAgIHJvb3RfY2F1c2U6IFt7IHR5cGU6ICdzdGF0dXNfZXhjZXB0aW9uJywgcmVhc29uOiAndGVzdCBleGNlcHRpb24nIH1dLFxuICogICAgIHR5cGU6ICdzdGF0dXNfZXhjZXB0aW9uJyxcbiAqICAgICByZWFzb246ICd0ZXN0IGV4Y2VwdGlvbicsXG4gKiAgIH0sXG4gKiAgIHN0YXR1czogNDA0LFxuICogfTtcbiAqXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBwYXJzZU9wZW5TZWFyY2hFcnJvclJlc3BvbnNlKGVycm9yOiBhbnkpIHtcbiAgaWYgKGVycm9yLnJlc3BvbnNlKSB7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IG9wZW5zZWFyY2hFcnJvclJlc3BvbnNlID0gSlNPTi5wYXJzZShlcnJvci5yZXNwb25zZSk7XG4gICAgICByZXR1cm4gb3BlbnNlYXJjaEVycm9yUmVzcG9uc2UuZXJyb3IucmVhc29uIHx8IGVycm9yLnJlc3BvbnNlO1xuICAgIH0gY2F0Y2ggKHBhcnNpbmdFcnJvcikge1xuICAgICAgcmV0dXJuIGVycm9yLnJlc3BvbnNlO1xuICAgIH1cbiAgfVxuICByZXR1cm4gZXJyb3IubWVzc2FnZTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGVycm9yUmVzcG9uc2UocmVzcG9uc2U6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2VGYWN0b3J5LCBlcnJvcjogYW55KSB7XG4gIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgIHN0YXR1c0NvZGU6IGVycm9yLnN0YXR1c0NvZGUgfHwgNTAwLFxuICAgIGJvZHk6IHBhcnNlT3BlblNlYXJjaEVycm9yUmVzcG9uc2UoZXJyb3IpLFxuICB9KTtcbn1cblxuLyoqXG4gKiBHZW5lcmF0ZSByZXBvcnQgZmlsZSBuYW1lIGJhc2VkIG9uIG5hbWUgYW5kIHRpbWVzdGFtcC5cbiAqIEBwYXJhbSBpdGVtTmFtZSAgICAgIHJlcG9ydCBpdGVtIG5hbWVcbiAqIEBwYXJhbSB0aW1lQ3JlYXRlZCAgIHRpbWVzdGFtcCB3aGVuIHRoaXMgaXMgYmVpbmcgY3JlYXRlZFxuICovXG5leHBvcnQgZnVuY3Rpb24gZ2V0RmlsZU5hbWUoaXRlbU5hbWU6IHN0cmluZywgdGltZUNyZWF0ZWQ6IERhdGUpOiBzdHJpbmcge1xuICByZXR1cm4gYCR7aXRlbU5hbWV9XyR7dGltZUNyZWF0ZWQudG9JU09TdHJpbmcoKX1fJHt1dWlkdjEoKX1gO1xufVxuXG4vKipcbiAqIENhbGwgT3BlblNlYXJjaCBjbHVzdGVyIGZ1bmN0aW9uLlxuICogQHBhcmFtIGNsaWVudCAgICBPcGVuU2VhcmNoIGNsaWVudFxuICogQHBhcmFtIGVuZHBvaW50ICBPcGVuU2VhcmNoIEFQSSBtZXRob2RcbiAqIEBwYXJhbSBwYXJhbXMgICAgT3BlblNlYXJjaCBBUEkgcGFyYW1ldGVyc1xuICovXG5leHBvcnQgY29uc3QgY2FsbENsdXN0ZXIgPSBhc3luYyAoXG4gIGNsaWVudDogSUxlZ2FjeUNsdXN0ZXJDbGllbnQgfCBJTGVnYWN5U2NvcGVkQ2x1c3RlckNsaWVudCxcbiAgZW5kcG9pbnQ6IHN0cmluZyxcbiAgcGFyYW1zOiBhbnksXG4gIGlzU2NoZWR1bGVkVGFzazogYm9vbGVhblxuKSA9PiB7XG4gIGxldCBvcGVuc2VhcmNoUmVzcDtcbiAgaWYgKGlzU2NoZWR1bGVkVGFzaykge1xuICAgIG9wZW5zZWFyY2hSZXNwID0gYXdhaXQgKGNsaWVudCBhcyBJTGVnYWN5Q2x1c3RlckNsaWVudCkuY2FsbEFzSW50ZXJuYWxVc2VyKFxuICAgICAgZW5kcG9pbnQsXG4gICAgICBwYXJhbXNcbiAgICApO1xuICB9IGVsc2Uge1xuICAgIG9wZW5zZWFyY2hSZXNwID0gYXdhaXQgKGNsaWVudCBhcyBJTGVnYWN5U2NvcGVkQ2x1c3RlckNsaWVudCkuY2FsbEFzQ3VycmVudFVzZXIoXG4gICAgICBlbmRwb2ludCxcbiAgICAgIHBhcmFtc1xuICAgICk7XG4gIH1cbiAgcmV0dXJuIG9wZW5zZWFyY2hSZXNwO1xufTtcblxuZXhwb3J0IGNvbnN0IGNoZWNrRXJyb3JUeXBlID0gKGVycm9yOiBhbnkpID0+IHtcbiAgaWYgKGVycm9yLnN0YXR1c0NvZGUgJiYgTWF0aC5mbG9vcihlcnJvci5zdGF0dXNDb2RlIC8gMTAwKSA9PT0gNCkge1xuICAgIHJldHVybiAndXNlcl9lcnJvcic7XG4gIH0gZWxzZSB7XG4gICAgcmV0dXJuICdzeXN0ZW1fZXJyb3InO1xuICB9XG59O1xuXG5leHBvcnQgY29uc3Qgam9pblJlcXVlc3RQYXJhbXMgPSAoXG4gIHF1ZXJ5UGFyYW1zOiBzdHJpbmcgfCBzdHJpbmdbXSB8IHVuZGVmaW5lZFxuKSA9PiB7XG4gIGlmIChBcnJheS5pc0FycmF5KHF1ZXJ5UGFyYW1zKSkgcmV0dXJuIHF1ZXJ5UGFyYW1zLmpvaW4oJywnKTtcbiAgaWYgKHR5cGVvZiBxdWVyeVBhcmFtcyA9PT0gJ3N0cmluZycpIHJldHVybiBxdWVyeVBhcmFtcztcbiAgcmV0dXJuICcnO1xufTsiXSwibWFwcGluZ3MiOiI7Ozs7Ozs7Ozs7QUFNQSxJQUFBQSxLQUFBLEdBQUFDLE9BQUE7QUFOQTtBQUNBO0FBQ0E7QUFDQTs7QUFTQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDTyxTQUFTQyw0QkFBNEJBLENBQUNDLEtBQVUsRUFBRTtFQUN2RCxJQUFJQSxLQUFLLENBQUNDLFFBQVEsRUFBRTtJQUNsQixJQUFJO01BQ0YsTUFBTUMsdUJBQXVCLEdBQUdDLElBQUksQ0FBQ0MsS0FBSyxDQUFDSixLQUFLLENBQUNDLFFBQVEsQ0FBQztNQUMxRCxPQUFPQyx1QkFBdUIsQ0FBQ0YsS0FBSyxDQUFDSyxNQUFNLElBQUlMLEtBQUssQ0FBQ0MsUUFBUTtJQUMvRCxDQUFDLENBQUMsT0FBT0ssWUFBWSxFQUFFO01BQ3JCLE9BQU9OLEtBQUssQ0FBQ0MsUUFBUTtJQUN2QjtFQUNGO0VBQ0EsT0FBT0QsS0FBSyxDQUFDTyxPQUFPO0FBQ3RCO0FBRU8sU0FBU0MsYUFBYUEsQ0FBQ1AsUUFBNkMsRUFBRUQsS0FBVSxFQUFFO0VBQ3ZGLE9BQU9DLFFBQVEsQ0FBQ1EsTUFBTSxDQUFDO0lBQ3JCQyxVQUFVLEVBQUVWLEtBQUssQ0FBQ1UsVUFBVSxJQUFJLEdBQUc7SUFDbkNDLElBQUksRUFBRVosNEJBQTRCLENBQUNDLEtBQUs7RUFDMUMsQ0FBQyxDQUFDO0FBQ0o7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNPLFNBQVNZLFdBQVdBLENBQUNDLFFBQWdCLEVBQUVDLFdBQWlCLEVBQVU7RUFDdkUsT0FBUSxHQUFFRCxRQUFTLElBQUdDLFdBQVcsQ0FBQ0MsV0FBVyxDQUFDLENBQUUsSUFBRyxJQUFBQyxRQUFNLEVBQUMsQ0FBRSxFQUFDO0FBQy9EOztBQUVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNPLE1BQU1DLFdBQVcsR0FBRyxNQUFBQSxDQUN6QkMsTUFBeUQsRUFDekRDLFFBQWdCLEVBQ2hCQyxNQUFXLEVBQ1hDLGVBQXdCLEtBQ3JCO0VBQ0gsSUFBSUMsY0FBYztFQUNsQixJQUFJRCxlQUFlLEVBQUU7SUFDbkJDLGNBQWMsR0FBRyxNQUFPSixNQUFNLENBQTBCSyxrQkFBa0IsQ0FDeEVKLFFBQVEsRUFDUkMsTUFDRixDQUFDO0VBQ0gsQ0FBQyxNQUFNO0lBQ0xFLGNBQWMsR0FBRyxNQUFPSixNQUFNLENBQWdDTSxpQkFBaUIsQ0FDN0VMLFFBQVEsRUFDUkMsTUFDRixDQUFDO0VBQ0g7RUFDQSxPQUFPRSxjQUFjO0FBQ3ZCLENBQUM7QUFBQ0csT0FBQSxDQUFBUixXQUFBLEdBQUFBLFdBQUE7QUFFSyxNQUFNUyxjQUFjLEdBQUkxQixLQUFVLElBQUs7RUFDNUMsSUFBSUEsS0FBSyxDQUFDVSxVQUFVLElBQUlpQixJQUFJLENBQUNDLEtBQUssQ0FBQzVCLEtBQUssQ0FBQ1UsVUFBVSxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRTtJQUNoRSxPQUFPLFlBQVk7RUFDckIsQ0FBQyxNQUFNO0lBQ0wsT0FBTyxjQUFjO0VBQ3ZCO0FBQ0YsQ0FBQztBQUFDZSxPQUFBLENBQUFDLGNBQUEsR0FBQUEsY0FBQTtBQUVLLE1BQU1HLGlCQUFpQixHQUM1QkMsV0FBMEMsSUFDdkM7RUFDSCxJQUFJQyxLQUFLLENBQUNDLE9BQU8sQ0FBQ0YsV0FBVyxDQUFDLEVBQUUsT0FBT0EsV0FBVyxDQUFDRyxJQUFJLENBQUMsR0FBRyxDQUFDO0VBQzVELElBQUksT0FBT0gsV0FBVyxLQUFLLFFBQVEsRUFBRSxPQUFPQSxXQUFXO0VBQ3ZELE9BQU8sRUFBRTtBQUNYLENBQUM7QUFBQ0wsT0FBQSxDQUFBSSxpQkFBQSxHQUFBQSxpQkFBQSJ9