"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CommandRunner = void 0;
const vscode_uri_1 = require("vscode-uri");
const misc_1 = require("./misc");
const execPath_1 = require("./execPath");
class CommandRunner {
    connection;
    context;
    settings;
    constructor(connection, context, settings) {
        this.connection = connection;
        this.context = context;
        this.settings = settings;
    }
    async runCommand(executable, args, workingDirectory, mountPaths) {
        let executablePath;
        let command;
        let runEnv;
        const isEEEnabled = this.settings.executionEnvironment.enabled;
        let interpreterPathFromConfig = this.settings.python.interpreterPath;
        if (interpreterPathFromConfig.includes("${workspaceFolder}")) {
            const workspaceFolder = vscode_uri_1.URI.parse(this.context.workspaceFolder.uri).path;
            interpreterPathFromConfig = interpreterPathFromConfig.replace("${workspaceFolder}", workspaceFolder);
        }
        const interpreterPath = isEEEnabled ? "python3" : interpreterPathFromConfig;
        if (executable.startsWith("ansible")) {
            executablePath = isEEEnabled
                ? executable
                : (0, execPath_1.getAnsibleCommandExecPath)(executable, this.settings);
        }
        else {
            executablePath = executable;
        }
        // prepare command and env for local run
        if (!isEEEnabled) {
            [command, runEnv] = (0, misc_1.withInterpreter)(executablePath, args, interpreterPath, this.settings.python.activationScript);
        }
        else {
            // prepare command and env for execution environment run
            const executionEnvironment = await this.context.executionEnvironment;
            command = executionEnvironment.wrapContainerArgs(`${executable} ${args}`, mountPaths);
            runEnv = undefined;
        }
        if (command === undefined) {
            return { stdout: "", stderr: "" };
        }
        const currentWorkingDirectory = workingDirectory
            ? workingDirectory
            : vscode_uri_1.URI.parse(this.context.workspaceFolder.uri).path;
        const result = await (0, misc_1.asyncExec)(command, {
            encoding: "utf-8",
            cwd: currentWorkingDirectory,
            env: runEnv,
            maxBuffer: 10 * 1000 * 1000,
        });
        return result;
    }
    /**
     * A method to return the path to the provided executable
     * @param executable - String representing the name of the executable
     * @returns Complete path of the executable (string) or undefined depending upon the presence of the executable
     */
    async getExecutablePath(executable) {
        try {
            const executablePath = await this.runCommand("command", `-v ${executable}`);
            return executablePath.stdout.trim();
        }
        catch (error) {
            console.log(error);
        }
        try {
            const executablePath = await this.runCommand("whereis", executable);
            const outParts = executablePath.stdout.split(":");
            return outParts.length >= 2 ? outParts[1].trim() : undefined;
        }
        catch (error) {
            console.log(error);
        }
    }
}
exports.CommandRunner = CommandRunner;
