﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/panorama/PanoramaRequest.h>
#include <aws/panorama/Panorama_EXPORTS.h>
#include <aws/panorama/model/NodeCategory.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace Panorama {
namespace Model {

/**
 */
class ListNodesRequest : public PanoramaRequest {
 public:
  AWS_PANORAMA_API ListNodesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListNodes"; }

  AWS_PANORAMA_API Aws::String SerializePayload() const override;

  AWS_PANORAMA_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>Search for nodes by category.</p>
   */
  inline NodeCategory GetCategory() const { return m_category; }
  inline bool CategoryHasBeenSet() const { return m_categoryHasBeenSet; }
  inline void SetCategory(NodeCategory value) {
    m_categoryHasBeenSet = true;
    m_category = value;
  }
  inline ListNodesRequest& WithCategory(NodeCategory value) {
    SetCategory(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of nodes to return in one page of results.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListNodesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify the pagination token from a previous request to retrieve the next
   * page of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListNodesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Search for nodes by the account ID of the nodes' owner.</p>
   */
  inline const Aws::String& GetOwnerAccount() const { return m_ownerAccount; }
  inline bool OwnerAccountHasBeenSet() const { return m_ownerAccountHasBeenSet; }
  template <typename OwnerAccountT = Aws::String>
  void SetOwnerAccount(OwnerAccountT&& value) {
    m_ownerAccountHasBeenSet = true;
    m_ownerAccount = std::forward<OwnerAccountT>(value);
  }
  template <typename OwnerAccountT = Aws::String>
  ListNodesRequest& WithOwnerAccount(OwnerAccountT&& value) {
    SetOwnerAccount(std::forward<OwnerAccountT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Search for nodes by name.</p>
   */
  inline const Aws::String& GetPackageName() const { return m_packageName; }
  inline bool PackageNameHasBeenSet() const { return m_packageNameHasBeenSet; }
  template <typename PackageNameT = Aws::String>
  void SetPackageName(PackageNameT&& value) {
    m_packageNameHasBeenSet = true;
    m_packageName = std::forward<PackageNameT>(value);
  }
  template <typename PackageNameT = Aws::String>
  ListNodesRequest& WithPackageName(PackageNameT&& value) {
    SetPackageName(std::forward<PackageNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Search for nodes by version.</p>
   */
  inline const Aws::String& GetPackageVersion() const { return m_packageVersion; }
  inline bool PackageVersionHasBeenSet() const { return m_packageVersionHasBeenSet; }
  template <typename PackageVersionT = Aws::String>
  void SetPackageVersion(PackageVersionT&& value) {
    m_packageVersionHasBeenSet = true;
    m_packageVersion = std::forward<PackageVersionT>(value);
  }
  template <typename PackageVersionT = Aws::String>
  ListNodesRequest& WithPackageVersion(PackageVersionT&& value) {
    SetPackageVersion(std::forward<PackageVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Search for nodes by patch version.</p>
   */
  inline const Aws::String& GetPatchVersion() const { return m_patchVersion; }
  inline bool PatchVersionHasBeenSet() const { return m_patchVersionHasBeenSet; }
  template <typename PatchVersionT = Aws::String>
  void SetPatchVersion(PatchVersionT&& value) {
    m_patchVersionHasBeenSet = true;
    m_patchVersion = std::forward<PatchVersionT>(value);
  }
  template <typename PatchVersionT = Aws::String>
  ListNodesRequest& WithPatchVersion(PatchVersionT&& value) {
    SetPatchVersion(std::forward<PatchVersionT>(value));
    return *this;
  }
  ///@}
 private:
  NodeCategory m_category{NodeCategory::NOT_SET};
  bool m_categoryHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  Aws::String m_ownerAccount;
  bool m_ownerAccountHasBeenSet = false;

  Aws::String m_packageName;
  bool m_packageNameHasBeenSet = false;

  Aws::String m_packageVersion;
  bool m_packageVersionHasBeenSet = false;

  Aws::String m_patchVersion;
  bool m_patchVersionHasBeenSet = false;
};

}  // namespace Model
}  // namespace Panorama
}  // namespace Aws
