﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/redshift-data/RedshiftDataAPIServiceRequest.h>
#include <aws/redshift-data/RedshiftDataAPIService_EXPORTS.h>

#include <utility>

namespace Aws {
namespace RedshiftDataAPIService {
namespace Model {

/**
 */
class ListTablesRequest : public RedshiftDataAPIServiceRequest {
 public:
  AWS_REDSHIFTDATAAPISERVICE_API ListTablesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListTables"; }

  AWS_REDSHIFTDATAAPISERVICE_API Aws::String SerializePayload() const override;

  AWS_REDSHIFTDATAAPISERVICE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The cluster identifier. This parameter is required when connecting to a
   * cluster and authenticating using either Secrets Manager or temporary
   * credentials. </p>
   */
  inline const Aws::String& GetClusterIdentifier() const { return m_clusterIdentifier; }
  inline bool ClusterIdentifierHasBeenSet() const { return m_clusterIdentifierHasBeenSet; }
  template <typename ClusterIdentifierT = Aws::String>
  void SetClusterIdentifier(ClusterIdentifierT&& value) {
    m_clusterIdentifierHasBeenSet = true;
    m_clusterIdentifier = std::forward<ClusterIdentifierT>(value);
  }
  template <typename ClusterIdentifierT = Aws::String>
  ListTablesRequest& WithClusterIdentifier(ClusterIdentifierT&& value) {
    SetClusterIdentifier(std::forward<ClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name or ARN of the secret that enables access to the database. This
   * parameter is required when authenticating using Secrets Manager. </p>
   */
  inline const Aws::String& GetSecretArn() const { return m_secretArn; }
  inline bool SecretArnHasBeenSet() const { return m_secretArnHasBeenSet; }
  template <typename SecretArnT = Aws::String>
  void SetSecretArn(SecretArnT&& value) {
    m_secretArnHasBeenSet = true;
    m_secretArn = std::forward<SecretArnT>(value);
  }
  template <typename SecretArnT = Aws::String>
  ListTablesRequest& WithSecretArn(SecretArnT&& value) {
    SetSecretArn(std::forward<SecretArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The database user name. This parameter is required when connecting to a
   * cluster as a database user and authenticating using temporary credentials. </p>
   */
  inline const Aws::String& GetDbUser() const { return m_dbUser; }
  inline bool DbUserHasBeenSet() const { return m_dbUserHasBeenSet; }
  template <typename DbUserT = Aws::String>
  void SetDbUser(DbUserT&& value) {
    m_dbUserHasBeenSet = true;
    m_dbUser = std::forward<DbUserT>(value);
  }
  template <typename DbUserT = Aws::String>
  ListTablesRequest& WithDbUser(DbUserT&& value) {
    SetDbUser(std::forward<DbUserT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the database that contains the tables to list. If
   * <code>ConnectedDatabase</code> is not specified, this is also the database to
   * connect to with your authentication credentials.</p>
   */
  inline const Aws::String& GetDatabase() const { return m_database; }
  inline bool DatabaseHasBeenSet() const { return m_databaseHasBeenSet; }
  template <typename DatabaseT = Aws::String>
  void SetDatabase(DatabaseT&& value) {
    m_databaseHasBeenSet = true;
    m_database = std::forward<DatabaseT>(value);
  }
  template <typename DatabaseT = Aws::String>
  ListTablesRequest& WithDatabase(DatabaseT&& value) {
    SetDatabase(std::forward<DatabaseT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A database name. The connected database is specified when you connect with
   * your authentication credentials. </p>
   */
  inline const Aws::String& GetConnectedDatabase() const { return m_connectedDatabase; }
  inline bool ConnectedDatabaseHasBeenSet() const { return m_connectedDatabaseHasBeenSet; }
  template <typename ConnectedDatabaseT = Aws::String>
  void SetConnectedDatabase(ConnectedDatabaseT&& value) {
    m_connectedDatabaseHasBeenSet = true;
    m_connectedDatabase = std::forward<ConnectedDatabaseT>(value);
  }
  template <typename ConnectedDatabaseT = Aws::String>
  ListTablesRequest& WithConnectedDatabase(ConnectedDatabaseT&& value) {
    SetConnectedDatabase(std::forward<ConnectedDatabaseT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A pattern to filter results by schema name. Within a schema pattern, "%"
   * means match any substring of 0 or more characters and "_" means match any one
   * character. Only schema name entries matching the search pattern are returned. If
   * <code>SchemaPattern</code> is not specified, then all tables that match
   * <code>TablePattern</code> are returned. If neither <code>SchemaPattern</code> or
   * <code>TablePattern</code> are specified, then all tables are returned. </p>
   */
  inline const Aws::String& GetSchemaPattern() const { return m_schemaPattern; }
  inline bool SchemaPatternHasBeenSet() const { return m_schemaPatternHasBeenSet; }
  template <typename SchemaPatternT = Aws::String>
  void SetSchemaPattern(SchemaPatternT&& value) {
    m_schemaPatternHasBeenSet = true;
    m_schemaPattern = std::forward<SchemaPatternT>(value);
  }
  template <typename SchemaPatternT = Aws::String>
  ListTablesRequest& WithSchemaPattern(SchemaPatternT&& value) {
    SetSchemaPattern(std::forward<SchemaPatternT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A pattern to filter results by table name. Within a table pattern, "%" means
   * match any substring of 0 or more characters and "_" means match any one
   * character. Only table name entries matching the search pattern are returned. If
   * <code>TablePattern</code> is not specified, then all tables that match
   * <code>SchemaPattern</code>are returned. If neither <code>SchemaPattern</code> or
   * <code>TablePattern</code> are specified, then all tables are returned. </p>
   */
  inline const Aws::String& GetTablePattern() const { return m_tablePattern; }
  inline bool TablePatternHasBeenSet() const { return m_tablePatternHasBeenSet; }
  template <typename TablePatternT = Aws::String>
  void SetTablePattern(TablePatternT&& value) {
    m_tablePatternHasBeenSet = true;
    m_tablePattern = std::forward<TablePatternT>(value);
  }
  template <typename TablePatternT = Aws::String>
  ListTablesRequest& WithTablePattern(TablePatternT&& value) {
    SetTablePattern(std::forward<TablePatternT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value that indicates the starting point for the next set of response
   * records in a subsequent request. If a value is returned in a response, you can
   * retrieve the next set of records by providing this returned NextToken value in
   * the next NextToken parameter and retrying the command. If the NextToken field is
   * empty, all response records have been retrieved for the request. </p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListTablesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of tables to return in the response. If more tables exist
   * than fit in one response, then <code>NextToken</code> is returned to page
   * through the results. </p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListTablesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The serverless workgroup name or Amazon Resource Name (ARN). This parameter
   * is required when connecting to a serverless workgroup and authenticating using
   * either Secrets Manager or temporary credentials.</p>
   */
  inline const Aws::String& GetWorkgroupName() const { return m_workgroupName; }
  inline bool WorkgroupNameHasBeenSet() const { return m_workgroupNameHasBeenSet; }
  template <typename WorkgroupNameT = Aws::String>
  void SetWorkgroupName(WorkgroupNameT&& value) {
    m_workgroupNameHasBeenSet = true;
    m_workgroupName = std::forward<WorkgroupNameT>(value);
  }
  template <typename WorkgroupNameT = Aws::String>
  ListTablesRequest& WithWorkgroupName(WorkgroupNameT&& value) {
    SetWorkgroupName(std::forward<WorkgroupNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clusterIdentifier;
  bool m_clusterIdentifierHasBeenSet = false;

  Aws::String m_secretArn;
  bool m_secretArnHasBeenSet = false;

  Aws::String m_dbUser;
  bool m_dbUserHasBeenSet = false;

  Aws::String m_database;
  bool m_databaseHasBeenSet = false;

  Aws::String m_connectedDatabase;
  bool m_connectedDatabaseHasBeenSet = false;

  Aws::String m_schemaPattern;
  bool m_schemaPatternHasBeenSet = false;

  Aws::String m_tablePattern;
  bool m_tablePatternHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_workgroupName;
  bool m_workgroupNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace RedshiftDataAPIService
}  // namespace Aws
