﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace SageMaker {
namespace Model {

/**
 * <p>Defines a container that provides the runtime environment for a model that
 * you deploy with an inference component.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/sagemaker-2017-07-24/InferenceComponentContainerSpecification">AWS
 * API Reference</a></p>
 */
class InferenceComponentContainerSpecification {
 public:
  AWS_SAGEMAKER_API InferenceComponentContainerSpecification() = default;
  AWS_SAGEMAKER_API InferenceComponentContainerSpecification(Aws::Utils::Json::JsonView jsonValue);
  AWS_SAGEMAKER_API InferenceComponentContainerSpecification& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_SAGEMAKER_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Elastic Container Registry (Amazon ECR) path where the Docker
   * image for the model is stored.</p>
   */
  inline const Aws::String& GetImage() const { return m_image; }
  inline bool ImageHasBeenSet() const { return m_imageHasBeenSet; }
  template <typename ImageT = Aws::String>
  void SetImage(ImageT&& value) {
    m_imageHasBeenSet = true;
    m_image = std::forward<ImageT>(value);
  }
  template <typename ImageT = Aws::String>
  InferenceComponentContainerSpecification& WithImage(ImageT&& value) {
    SetImage(std::forward<ImageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon S3 path where the model artifacts, which result from model
   * training, are stored. This path must point to a single gzip compressed tar
   * archive (.tar.gz suffix).</p>
   */
  inline const Aws::String& GetArtifactUrl() const { return m_artifactUrl; }
  inline bool ArtifactUrlHasBeenSet() const { return m_artifactUrlHasBeenSet; }
  template <typename ArtifactUrlT = Aws::String>
  void SetArtifactUrl(ArtifactUrlT&& value) {
    m_artifactUrlHasBeenSet = true;
    m_artifactUrl = std::forward<ArtifactUrlT>(value);
  }
  template <typename ArtifactUrlT = Aws::String>
  InferenceComponentContainerSpecification& WithArtifactUrl(ArtifactUrlT&& value) {
    SetArtifactUrl(std::forward<ArtifactUrlT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The environment variables to set in the Docker container. Each key and value
   * in the Environment string-to-string map can have length of up to 1024. We
   * support up to 16 entries in the map.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetEnvironment() const { return m_environment; }
  inline bool EnvironmentHasBeenSet() const { return m_environmentHasBeenSet; }
  template <typename EnvironmentT = Aws::Map<Aws::String, Aws::String>>
  void SetEnvironment(EnvironmentT&& value) {
    m_environmentHasBeenSet = true;
    m_environment = std::forward<EnvironmentT>(value);
  }
  template <typename EnvironmentT = Aws::Map<Aws::String, Aws::String>>
  InferenceComponentContainerSpecification& WithEnvironment(EnvironmentT&& value) {
    SetEnvironment(std::forward<EnvironmentT>(value));
    return *this;
  }
  template <typename EnvironmentKeyT = Aws::String, typename EnvironmentValueT = Aws::String>
  InferenceComponentContainerSpecification& AddEnvironment(EnvironmentKeyT&& key, EnvironmentValueT&& value) {
    m_environmentHasBeenSet = true;
    m_environment.emplace(std::forward<EnvironmentKeyT>(key), std::forward<EnvironmentValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_image;
  bool m_imageHasBeenSet = false;

  Aws::String m_artifactUrl;
  bool m_artifactUrlHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_environment;
  bool m_environmentHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
