"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationsManager = void 0;
var _path = _interopRequireDefault(require("path"));
var _metrics_helper = require("../../common/metrics/metrics_helper");
var _integrations_builder = require("./integrations_builder");
var _fs_data_adaptor = require("./repository/fs_data_adaptor");
var _index_data_adaptor = require("./repository/index_data_adaptor");
var _repository = require("./repository/repository");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * Copyright OpenSearch Contributors
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          */
class IntegrationsManager {
  constructor(client, repository) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "instanceBuilder", void 0);
    _defineProperty(this, "repository", void 0);
    _defineProperty(this, "deleteIntegrationInstance", async id => {
      let children;
      try {
        children = await this.client.get('integration-instance', id);
      } catch (err) {
        var _err$output;
        return ((_err$output = err.output) === null || _err$output === void 0 ? void 0 : _err$output.statusCode) === 404 ? Promise.resolve([id]) : Promise.reject(err);
      }
      const toDelete = children.attributes.assets.filter(i => i.assetId).map(i => {
        return {
          id: i.assetId,
          type: i.assetType
        };
      });
      toDelete.push({
        id,
        type: 'integration-instance'
      });
      const result = Promise.all(toDelete.map(async asset => {
        try {
          await this.client.delete(asset.type, asset.id);
          return Promise.resolve(asset.id);
        } catch (err) {
          var _err$output2;
          (0, _metrics_helper.addRequestToMetric)('integrations', 'delete', err);
          return ((_err$output2 = err.output) === null || _err$output2 === void 0 ? void 0 : _err$output2.statusCode) === 404 ? Promise.resolve(asset.id) : Promise.reject(err);
        }
      }));
      (0, _metrics_helper.addRequestToMetric)('integrations', 'delete', 'count');
      return result;
    });
    // Internal; use getIntegrationTemplates.
    _defineProperty(this, "_getAllIntegrationTemplates", async () => {
      const integrationList = await this.repository.getIntegrationList();
      const configResults = await Promise.all(integrationList.map(x => x.getConfig()));
      const configs = configResults.filter(cfg => cfg.ok);
      return Promise.resolve({
        hits: configs.map(cfg => cfg.value)
      });
    });
    // Internal; use getIntegrationTemplates.
    _defineProperty(this, "_getIntegrationTemplatesByName", async name => {
      const integration = await this.repository.getIntegration(name);
      const config = await (integration === null || integration === void 0 ? void 0 : integration.getConfig());
      if (!config || !config.ok) {
        return Promise.resolve({
          hits: []
        });
      }
      return Promise.resolve({
        hits: [config.value]
      });
    });
    _defineProperty(this, "getIntegrationTemplates", async query => {
      if (query !== null && query !== void 0 && query.name) {
        return this._getIntegrationTemplatesByName(query.name);
      }
      return this._getAllIntegrationTemplates();
    });
    _defineProperty(this, "getIntegrationInstances", async _query => {
      var _result$saved_objects;
      (0, _metrics_helper.addRequestToMetric)('integrations', 'get', 'count');
      const result = await this.client.find({
        type: 'integration-instance',
        perPage: 1000
      });
      return Promise.resolve({
        total: result.total,
        hits: (_result$saved_objects = result.saved_objects) === null || _result$saved_objects === void 0 ? void 0 : _result$saved_objects.map(x => ({
          ...x.attributes,
          id: x.id
        }))
      });
    });
    _defineProperty(this, "getIntegrationInstance", async query => {
      (0, _metrics_helper.addRequestToMetric)('integrations', 'get', 'count');
      const result = await this.client.get('integration-instance', `${query.id}`);
      return Promise.resolve(this.buildInstanceResponse(result));
    });
    _defineProperty(this, "buildInstanceResponse", async savedObj => {
      const assets = savedObj.attributes.assets;
      const status = assets ? await this.getAssetStatus(assets) : 'available';
      return {
        id: savedObj.id,
        status,
        ...savedObj.attributes
      };
    });
    _defineProperty(this, "getAssetStatus", async assets => {
      const statuses = await Promise.all(assets.map(async asset => {
        try {
          await this.client.get(asset.assetType, asset.assetId);
          return {
            id: asset.assetId,
            status: 'available'
          };
        } catch (err) {
          var _err$output3;
          const statusCode = (_err$output3 = err.output) === null || _err$output3 === void 0 ? void 0 : _err$output3.statusCode;
          if (statusCode && 400 <= statusCode && statusCode < 500) {
            return {
              id: asset.assetId,
              status: 'unavailable'
            };
          }
          console.error('Failed to get asset status', err);
          return {
            id: asset.assetId,
            status: 'unknown'
          };
        }
      }));
      const [available, unavailable, unknown] = [statuses.filter(x => x.status === 'available').length, statuses.filter(x => x.status === 'unavailable').length, statuses.filter(x => x.status === 'unknown').length];
      if (unknown > 0) return 'unknown';
      if (unavailable > 0 && available > 0) return 'partially-available';
      if (unavailable > 0) return 'unavailable';
      return 'available';
    });
    _defineProperty(this, "loadIntegrationInstance", async (templateName, name, indexPattern, dataSourceMDSId, dataSourceMDSLabel, workflows, dataSource, tableName) => {
      const template = await this.repository.getIntegration(templateName);
      if (template === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      try {
        (0, _metrics_helper.addRequestToMetric)('integrations', 'create', 'count');
        const result = await this.instanceBuilder.build(template, {
          name,
          indexPattern,
          dataSourceMDSId,
          dataSourceMDSLabel,
          workflows,
          dataSource,
          tableName
        });
        const test = await this.client.create('integration-instance', result);
        return Promise.resolve({
          ...result,
          id: test.id
        });
      } catch (err) {
        (0, _metrics_helper.addRequestToMetric)('integrations', 'create', err);
        return Promise.reject({
          message: err.message,
          statusCode: 500
        });
      }
    });
    _defineProperty(this, "getStatic", async (templateName, staticPath) => {
      const integration = await this.repository.getIntegration(templateName);
      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      const data = await integration.getStatic(staticPath);
      if (data.ok) {
        return data.value;
      }
      const is404 = data.error.code === 'ENOENT';
      return Promise.reject({
        message: data.error.message,
        statusCode: is404 ? 404 : 500
      });
    });
    _defineProperty(this, "getSchemas", async templateName => {
      const integration = await this.repository.getIntegration(templateName);
      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      const result = await integration.getSchemas();
      if (result.ok) {
        return result.value;
      }
      const is404 = result.error.code === 'ENOENT';
      return Promise.reject({
        message: result.error.message,
        statusCode: is404 ? 404 : 500
      });
    });
    _defineProperty(this, "getAssets", async templateName => {
      const integration = await this.repository.getIntegration(templateName);
      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      const assets = await integration.getAssets();
      if (assets.ok) {
        return assets.value;
      }
      const is404 = assets.error.code === 'ENOENT';
      return Promise.reject({
        message: assets.error.message,
        statusCode: is404 ? 404 : 500
      });
    });
    _defineProperty(this, "getSampleData", async templateName => {
      const integration = await this.repository.getIntegration(templateName);
      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      const sampleData = await integration.getSampleData();
      if (sampleData.ok) {
        return sampleData.value;
      }
      const is404 = sampleData.error.code === 'ENOENT';
      return Promise.reject({
        message: sampleData.error.message,
        statusCode: is404 ? 404 : 500
      });
    });
    this.client = client;
    this.repository = repository !== null && repository !== void 0 ? repository : new _repository.TemplateManager([new _index_data_adaptor.IndexDataAdaptor(this.client), new _fs_data_adaptor.FileSystemDataAdaptor(_path.default.join(__dirname, '__data__/repository'))]);
    this.instanceBuilder = new _integrations_builder.IntegrationInstanceBuilder(this.client);
  }
}
exports.IntegrationsManager = IntegrationsManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfcGF0aCIsIl9pbnRlcm9wUmVxdWlyZURlZmF1bHQiLCJyZXF1aXJlIiwiX21ldHJpY3NfaGVscGVyIiwiX2ludGVncmF0aW9uc19idWlsZGVyIiwiX2ZzX2RhdGFfYWRhcHRvciIsIl9pbmRleF9kYXRhX2FkYXB0b3IiLCJfcmVwb3NpdG9yeSIsImUiLCJfX2VzTW9kdWxlIiwiZGVmYXVsdCIsIl9kZWZpbmVQcm9wZXJ0eSIsInIiLCJ0IiwiX3RvUHJvcGVydHlLZXkiLCJPYmplY3QiLCJkZWZpbmVQcm9wZXJ0eSIsInZhbHVlIiwiZW51bWVyYWJsZSIsImNvbmZpZ3VyYWJsZSIsIndyaXRhYmxlIiwiaSIsIl90b1ByaW1pdGl2ZSIsIlN5bWJvbCIsInRvUHJpbWl0aXZlIiwiY2FsbCIsIlR5cGVFcnJvciIsIlN0cmluZyIsIk51bWJlciIsIkludGVncmF0aW9uc01hbmFnZXIiLCJjb25zdHJ1Y3RvciIsImNsaWVudCIsInJlcG9zaXRvcnkiLCJpZCIsImNoaWxkcmVuIiwiZ2V0IiwiZXJyIiwiX2VyciRvdXRwdXQiLCJvdXRwdXQiLCJzdGF0dXNDb2RlIiwiUHJvbWlzZSIsInJlc29sdmUiLCJyZWplY3QiLCJ0b0RlbGV0ZSIsImF0dHJpYnV0ZXMiLCJhc3NldHMiLCJmaWx0ZXIiLCJhc3NldElkIiwibWFwIiwidHlwZSIsImFzc2V0VHlwZSIsInB1c2giLCJyZXN1bHQiLCJhbGwiLCJhc3NldCIsImRlbGV0ZSIsIl9lcnIkb3V0cHV0MiIsImFkZFJlcXVlc3RUb01ldHJpYyIsImludGVncmF0aW9uTGlzdCIsImdldEludGVncmF0aW9uTGlzdCIsImNvbmZpZ1Jlc3VsdHMiLCJ4IiwiZ2V0Q29uZmlnIiwiY29uZmlncyIsImNmZyIsIm9rIiwiaGl0cyIsIm5hbWUiLCJpbnRlZ3JhdGlvbiIsImdldEludGVncmF0aW9uIiwiY29uZmlnIiwicXVlcnkiLCJfZ2V0SW50ZWdyYXRpb25UZW1wbGF0ZXNCeU5hbWUiLCJfZ2V0QWxsSW50ZWdyYXRpb25UZW1wbGF0ZXMiLCJfcXVlcnkiLCJfcmVzdWx0JHNhdmVkX29iamVjdHMiLCJmaW5kIiwicGVyUGFnZSIsInRvdGFsIiwic2F2ZWRfb2JqZWN0cyIsImJ1aWxkSW5zdGFuY2VSZXNwb25zZSIsInNhdmVkT2JqIiwic3RhdHVzIiwiZ2V0QXNzZXRTdGF0dXMiLCJzdGF0dXNlcyIsIl9lcnIkb3V0cHV0MyIsImNvbnNvbGUiLCJlcnJvciIsImF2YWlsYWJsZSIsInVuYXZhaWxhYmxlIiwidW5rbm93biIsImxlbmd0aCIsInRlbXBsYXRlTmFtZSIsImluZGV4UGF0dGVybiIsImRhdGFTb3VyY2VNRFNJZCIsImRhdGFTb3VyY2VNRFNMYWJlbCIsIndvcmtmbG93cyIsImRhdGFTb3VyY2UiLCJ0YWJsZU5hbWUiLCJ0ZW1wbGF0ZSIsIm1lc3NhZ2UiLCJpbnN0YW5jZUJ1aWxkZXIiLCJidWlsZCIsInRlc3QiLCJjcmVhdGUiLCJzdGF0aWNQYXRoIiwiZGF0YSIsImdldFN0YXRpYyIsImlzNDA0IiwiY29kZSIsImdldFNjaGVtYXMiLCJnZXRBc3NldHMiLCJzYW1wbGVEYXRhIiwiZ2V0U2FtcGxlRGF0YSIsIlRlbXBsYXRlTWFuYWdlciIsIkluZGV4RGF0YUFkYXB0b3IiLCJGaWxlU3lzdGVtRGF0YUFkYXB0b3IiLCJwYXRoIiwiam9pbiIsIl9fZGlybmFtZSIsIkludGVncmF0aW9uSW5zdGFuY2VCdWlsZGVyIiwiZXhwb3J0cyJdLCJzb3VyY2VzIjpbImludGVncmF0aW9uc19tYW5hZ2VyLnRzIl0sInNvdXJjZXNDb250ZW50IjpbIi8qXG4gKiBDb3B5cmlnaHQgT3BlblNlYXJjaCBDb250cmlidXRvcnNcbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuaW1wb3J0IHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBTYXZlZE9iamVjdCwgU2F2ZWRPYmplY3RzQ2xpZW50Q29udHJhY3QgfSBmcm9tICcuLi8uLi8uLi8uLi8uLi9zcmMvY29yZS9zZXJ2ZXIvdHlwZXMnO1xuaW1wb3J0IHsgYWRkUmVxdWVzdFRvTWV0cmljIH0gZnJvbSAnLi4vLi4vY29tbW9uL21ldHJpY3MvbWV0cmljc19oZWxwZXInO1xuaW1wb3J0IHsgSW50ZWdyYXRpb25JbnN0YW5jZUJ1aWxkZXIgfSBmcm9tICcuL2ludGVncmF0aW9uc19idWlsZGVyJztcbmltcG9ydCB7IEZpbGVTeXN0ZW1EYXRhQWRhcHRvciB9IGZyb20gJy4vcmVwb3NpdG9yeS9mc19kYXRhX2FkYXB0b3InO1xuaW1wb3J0IHsgSW5kZXhEYXRhQWRhcHRvciB9IGZyb20gJy4vcmVwb3NpdG9yeS9pbmRleF9kYXRhX2FkYXB0b3InO1xuaW1wb3J0IHsgVGVtcGxhdGVNYW5hZ2VyIH0gZnJvbSAnLi9yZXBvc2l0b3J5L3JlcG9zaXRvcnknO1xuXG5leHBvcnQgY2xhc3MgSW50ZWdyYXRpb25zTWFuYWdlciB7XG4gIGNsaWVudDogU2F2ZWRPYmplY3RzQ2xpZW50Q29udHJhY3Q7XG4gIGluc3RhbmNlQnVpbGRlcjogSW50ZWdyYXRpb25JbnN0YW5jZUJ1aWxkZXI7XG4gIHJlcG9zaXRvcnk6IFRlbXBsYXRlTWFuYWdlcjtcblxuICBjb25zdHJ1Y3RvcihjbGllbnQ6IFNhdmVkT2JqZWN0c0NsaWVudENvbnRyYWN0LCByZXBvc2l0b3J5PzogVGVtcGxhdGVNYW5hZ2VyKSB7XG4gICAgdGhpcy5jbGllbnQgPSBjbGllbnQ7XG4gICAgdGhpcy5yZXBvc2l0b3J5ID1cbiAgICAgIHJlcG9zaXRvcnkgPz9cbiAgICAgIG5ldyBUZW1wbGF0ZU1hbmFnZXIoW1xuICAgICAgICBuZXcgSW5kZXhEYXRhQWRhcHRvcih0aGlzLmNsaWVudCksXG4gICAgICAgIG5ldyBGaWxlU3lzdGVtRGF0YUFkYXB0b3IocGF0aC5qb2luKF9fZGlybmFtZSwgJ19fZGF0YV9fL3JlcG9zaXRvcnknKSksXG4gICAgICBdKTtcbiAgICB0aGlzLmluc3RhbmNlQnVpbGRlciA9IG5ldyBJbnRlZ3JhdGlvbkluc3RhbmNlQnVpbGRlcih0aGlzLmNsaWVudCk7XG4gIH1cblxuICBkZWxldGVJbnRlZ3JhdGlvbkluc3RhbmNlID0gYXN5bmMgKGlkOiBzdHJpbmcpOiBQcm9taXNlPHN0cmluZ1tdPiA9PiB7XG4gICAgbGV0IGNoaWxkcmVuOiBTYXZlZE9iamVjdDxJbnRlZ3JhdGlvbkluc3RhbmNlPjtcbiAgICB0cnkge1xuICAgICAgY2hpbGRyZW4gPSBhd2FpdCB0aGlzLmNsaWVudC5nZXQoJ2ludGVncmF0aW9uLWluc3RhbmNlJywgaWQpO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgcmV0dXJuIGVyci5vdXRwdXQ/LnN0YXR1c0NvZGUgPT09IDQwNCA/IFByb21pc2UucmVzb2x2ZShbaWRdKSA6IFByb21pc2UucmVqZWN0KGVycik7XG4gICAgfVxuXG4gICAgY29uc3QgdG9EZWxldGUgPSBjaGlsZHJlbi5hdHRyaWJ1dGVzLmFzc2V0c1xuICAgICAgLmZpbHRlcigoaTogQXNzZXRSZWZlcmVuY2UpID0+IGkuYXNzZXRJZClcbiAgICAgIC5tYXAoKGk6IEFzc2V0UmVmZXJlbmNlKSA9PiB7XG4gICAgICAgIHJldHVybiB7IGlkOiBpLmFzc2V0SWQsIHR5cGU6IGkuYXNzZXRUeXBlIH07XG4gICAgICB9KTtcbiAgICB0b0RlbGV0ZS5wdXNoKHsgaWQsIHR5cGU6ICdpbnRlZ3JhdGlvbi1pbnN0YW5jZScgfSk7XG5cbiAgICBjb25zdCByZXN1bHQgPSBQcm9taXNlLmFsbChcbiAgICAgIHRvRGVsZXRlLm1hcChcbiAgICAgICAgYXN5bmMgKGFzc2V0OiB7IHR5cGU6IHN0cmluZzsgaWQ6IHN0cmluZyB9KTogUHJvbWlzZTxzdHJpbmc+ID0+IHtcbiAgICAgICAgICB0cnkge1xuICAgICAgICAgICAgYXdhaXQgdGhpcy5jbGllbnQuZGVsZXRlKGFzc2V0LnR5cGUsIGFzc2V0LmlkKTtcbiAgICAgICAgICAgIHJldHVybiBQcm9taXNlLnJlc29sdmUoYXNzZXQuaWQpO1xuICAgICAgICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgICAgICAgYWRkUmVxdWVzdFRvTWV0cmljKCdpbnRlZ3JhdGlvbnMnLCAnZGVsZXRlJywgZXJyKTtcbiAgICAgICAgICAgIHJldHVybiBlcnIub3V0cHV0Py5zdGF0dXNDb2RlID09PSA0MDQgPyBQcm9taXNlLnJlc29sdmUoYXNzZXQuaWQpIDogUHJvbWlzZS5yZWplY3QoZXJyKTtcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIClcbiAgICApO1xuICAgIGFkZFJlcXVlc3RUb01ldHJpYygnaW50ZWdyYXRpb25zJywgJ2RlbGV0ZScsICdjb3VudCcpO1xuICAgIHJldHVybiByZXN1bHQ7XG4gIH07XG5cbiAgLy8gSW50ZXJuYWw7IHVzZSBnZXRJbnRlZ3JhdGlvblRlbXBsYXRlcy5cbiAgX2dldEFsbEludGVncmF0aW9uVGVtcGxhdGVzID0gYXN5bmMgKCk6IFByb21pc2U8SW50ZWdyYXRpb25UZW1wbGF0ZVNlYXJjaFJlc3VsdD4gPT4ge1xuICAgIGNvbnN0IGludGVncmF0aW9uTGlzdCA9IGF3YWl0IHRoaXMucmVwb3NpdG9yeS5nZXRJbnRlZ3JhdGlvbkxpc3QoKTtcbiAgICBjb25zdCBjb25maWdSZXN1bHRzID0gYXdhaXQgUHJvbWlzZS5hbGwoaW50ZWdyYXRpb25MaXN0Lm1hcCgoeCkgPT4geC5nZXRDb25maWcoKSkpO1xuICAgIGNvbnN0IGNvbmZpZ3MgPSBjb25maWdSZXN1bHRzLmZpbHRlcigoY2ZnKSA9PiBjZmcub2spIGFzIEFycmF5PHsgdmFsdWU6IEludGVncmF0aW9uQ29uZmlnIH0+O1xuICAgIHJldHVybiBQcm9taXNlLnJlc29sdmUoeyBoaXRzOiBjb25maWdzLm1hcCgoY2ZnKSA9PiBjZmcudmFsdWUpIH0pO1xuICB9O1xuXG4gIC8vIEludGVybmFsOyB1c2UgZ2V0SW50ZWdyYXRpb25UZW1wbGF0ZXMuXG4gIF9nZXRJbnRlZ3JhdGlvblRlbXBsYXRlc0J5TmFtZSA9IGFzeW5jIChcbiAgICBuYW1lOiBzdHJpbmdcbiAgKTogUHJvbWlzZTxJbnRlZ3JhdGlvblRlbXBsYXRlU2VhcmNoUmVzdWx0PiA9PiB7XG4gICAgY29uc3QgaW50ZWdyYXRpb24gPSBhd2FpdCB0aGlzLnJlcG9zaXRvcnkuZ2V0SW50ZWdyYXRpb24obmFtZSk7XG4gICAgY29uc3QgY29uZmlnID0gYXdhaXQgaW50ZWdyYXRpb24/LmdldENvbmZpZygpO1xuICAgIGlmICghY29uZmlnIHx8ICFjb25maWcub2spIHtcbiAgICAgIHJldHVybiBQcm9taXNlLnJlc29sdmUoeyBoaXRzOiBbXSB9KTtcbiAgICB9XG4gICAgcmV0dXJuIFByb21pc2UucmVzb2x2ZSh7IGhpdHM6IFtjb25maWcudmFsdWVdIH0pO1xuICB9O1xuXG4gIGdldEludGVncmF0aW9uVGVtcGxhdGVzID0gYXN5bmMgKFxuICAgIHF1ZXJ5PzogSW50ZWdyYXRpb25UZW1wbGF0ZVF1ZXJ5XG4gICk6IFByb21pc2U8SW50ZWdyYXRpb25UZW1wbGF0ZVNlYXJjaFJlc3VsdD4gPT4ge1xuICAgIGlmIChxdWVyeT8ubmFtZSkge1xuICAgICAgcmV0dXJuIHRoaXMuX2dldEludGVncmF0aW9uVGVtcGxhdGVzQnlOYW1lKHF1ZXJ5Lm5hbWUpO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5fZ2V0QWxsSW50ZWdyYXRpb25UZW1wbGF0ZXMoKTtcbiAgfTtcblxuICBnZXRJbnRlZ3JhdGlvbkluc3RhbmNlcyA9IGFzeW5jIChcbiAgICBfcXVlcnk/OiBJbnRlZ3JhdGlvbkluc3RhbmNlUXVlcnlcbiAgKTogUHJvbWlzZTxJbnRlZ3JhdGlvbkluc3RhbmNlc1NlYXJjaFJlc3VsdD4gPT4ge1xuICAgIGFkZFJlcXVlc3RUb01ldHJpYygnaW50ZWdyYXRpb25zJywgJ2dldCcsICdjb3VudCcpO1xuICAgIGNvbnN0IHJlc3VsdCA9IGF3YWl0IHRoaXMuY2xpZW50LmZpbmQoeyB0eXBlOiAnaW50ZWdyYXRpb24taW5zdGFuY2UnLCBwZXJQYWdlOiAxMDAwIH0pO1xuICAgIHJldHVybiBQcm9taXNlLnJlc29sdmUoe1xuICAgICAgdG90YWw6IHJlc3VsdC50b3RhbCxcbiAgICAgIGhpdHM6IHJlc3VsdC5zYXZlZF9vYmplY3RzPy5tYXAoKHgpID0+ICh7XG4gICAgICAgIC4uLnguYXR0cmlidXRlcyEsXG4gICAgICAgIGlkOiB4LmlkLFxuICAgICAgfSkpIGFzIEludGVncmF0aW9uSW5zdGFuY2VSZXN1bHRbXSxcbiAgICB9KTtcbiAgfTtcblxuICBnZXRJbnRlZ3JhdGlvbkluc3RhbmNlID0gYXN5bmMgKFxuICAgIHF1ZXJ5PzogSW50ZWdyYXRpb25JbnN0YW5jZVF1ZXJ5XG4gICk6IFByb21pc2U8SW50ZWdyYXRpb25JbnN0YW5jZVJlc3VsdD4gPT4ge1xuICAgIGFkZFJlcXVlc3RUb01ldHJpYygnaW50ZWdyYXRpb25zJywgJ2dldCcsICdjb3VudCcpO1xuICAgIGNvbnN0IHJlc3VsdCA9IChhd2FpdCB0aGlzLmNsaWVudC5nZXQoJ2ludGVncmF0aW9uLWluc3RhbmNlJywgYCR7cXVlcnkhLmlkfWApKSBhcyBTYXZlZE9iamVjdDxcbiAgICAgIEludGVncmF0aW9uSW5zdGFuY2VcbiAgICA+O1xuICAgIHJldHVybiBQcm9taXNlLnJlc29sdmUodGhpcy5idWlsZEluc3RhbmNlUmVzcG9uc2UocmVzdWx0KSk7XG4gIH07XG5cbiAgYnVpbGRJbnN0YW5jZVJlc3BvbnNlID0gYXN5bmMgKFxuICAgIHNhdmVkT2JqOiBTYXZlZE9iamVjdDxJbnRlZ3JhdGlvbkluc3RhbmNlPlxuICApOiBQcm9taXNlPEludGVncmF0aW9uSW5zdGFuY2VSZXN1bHQ+ID0+IHtcbiAgICBjb25zdCBhc3NldHM6IEFzc2V0UmVmZXJlbmNlW10gfCB1bmRlZmluZWQgPSBzYXZlZE9iai5hdHRyaWJ1dGVzLmFzc2V0cztcbiAgICBjb25zdCBzdGF0dXM6IHN0cmluZyA9IGFzc2V0cyA/IGF3YWl0IHRoaXMuZ2V0QXNzZXRTdGF0dXMoYXNzZXRzKSA6ICdhdmFpbGFibGUnO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIGlkOiBzYXZlZE9iai5pZCxcbiAgICAgIHN0YXR1cyxcbiAgICAgIC4uLnNhdmVkT2JqLmF0dHJpYnV0ZXMsXG4gICAgfTtcbiAgfTtcblxuICBnZXRBc3NldFN0YXR1cyA9IGFzeW5jIChhc3NldHM6IEFzc2V0UmVmZXJlbmNlW10pOiBQcm9taXNlPHN0cmluZz4gPT4ge1xuICAgIGNvbnN0IHN0YXR1c2VzOiBBcnJheTx7IGlkOiBzdHJpbmc7IHN0YXR1czogc3RyaW5nIH0+ID0gYXdhaXQgUHJvbWlzZS5hbGwoXG4gICAgICBhc3NldHMubWFwKGFzeW5jIChhc3NldCkgPT4ge1xuICAgICAgICB0cnkge1xuICAgICAgICAgIGF3YWl0IHRoaXMuY2xpZW50LmdldChhc3NldC5hc3NldFR5cGUsIGFzc2V0LmFzc2V0SWQpO1xuICAgICAgICAgIHJldHVybiB7IGlkOiBhc3NldC5hc3NldElkLCBzdGF0dXM6ICdhdmFpbGFibGUnIH07XG4gICAgICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgICAgIGNvbnN0IHN0YXR1c0NvZGUgPSBlcnIub3V0cHV0Py5zdGF0dXNDb2RlO1xuICAgICAgICAgIGlmIChzdGF0dXNDb2RlICYmIDQwMCA8PSBzdGF0dXNDb2RlICYmIHN0YXR1c0NvZGUgPCA1MDApIHtcbiAgICAgICAgICAgIHJldHVybiB7IGlkOiBhc3NldC5hc3NldElkLCBzdGF0dXM6ICd1bmF2YWlsYWJsZScgfTtcbiAgICAgICAgICB9XG4gICAgICAgICAgY29uc29sZS5lcnJvcignRmFpbGVkIHRvIGdldCBhc3NldCBzdGF0dXMnLCBlcnIpO1xuICAgICAgICAgIHJldHVybiB7IGlkOiBhc3NldC5hc3NldElkLCBzdGF0dXM6ICd1bmtub3duJyB9O1xuICAgICAgICB9XG4gICAgICB9KVxuICAgICk7XG5cbiAgICBjb25zdCBbYXZhaWxhYmxlLCB1bmF2YWlsYWJsZSwgdW5rbm93bl0gPSBbXG4gICAgICBzdGF0dXNlcy5maWx0ZXIoKHgpID0+IHguc3RhdHVzID09PSAnYXZhaWxhYmxlJykubGVuZ3RoLFxuICAgICAgc3RhdHVzZXMuZmlsdGVyKCh4KSA9PiB4LnN0YXR1cyA9PT0gJ3VuYXZhaWxhYmxlJykubGVuZ3RoLFxuICAgICAgc3RhdHVzZXMuZmlsdGVyKCh4KSA9PiB4LnN0YXR1cyA9PT0gJ3Vua25vd24nKS5sZW5ndGgsXG4gICAgXTtcbiAgICBpZiAodW5rbm93biA+IDApIHJldHVybiAndW5rbm93bic7XG4gICAgaWYgKHVuYXZhaWxhYmxlID4gMCAmJiBhdmFpbGFibGUgPiAwKSByZXR1cm4gJ3BhcnRpYWxseS1hdmFpbGFibGUnO1xuICAgIGlmICh1bmF2YWlsYWJsZSA+IDApIHJldHVybiAndW5hdmFpbGFibGUnO1xuICAgIHJldHVybiAnYXZhaWxhYmxlJztcbiAgfTtcblxuICBsb2FkSW50ZWdyYXRpb25JbnN0YW5jZSA9IGFzeW5jIChcbiAgICB0ZW1wbGF0ZU5hbWU6IHN0cmluZyxcbiAgICBuYW1lOiBzdHJpbmcsXG4gICAgaW5kZXhQYXR0ZXJuOiBzdHJpbmcsXG4gICAgZGF0YVNvdXJjZU1EU0lkPzogc3RyaW5nLFxuICAgIGRhdGFTb3VyY2VNRFNMYWJlbD86IHN0cmluZyxcbiAgICB3b3JrZmxvd3M/OiBzdHJpbmdbXSxcbiAgICBkYXRhU291cmNlPzogc3RyaW5nLFxuICAgIHRhYmxlTmFtZT86IHN0cmluZ1xuICApOiBQcm9taXNlPEludGVncmF0aW9uSW5zdGFuY2U+ID0+IHtcbiAgICBjb25zdCB0ZW1wbGF0ZSA9IGF3YWl0IHRoaXMucmVwb3NpdG9yeS5nZXRJbnRlZ3JhdGlvbih0ZW1wbGF0ZU5hbWUpO1xuICAgIGlmICh0ZW1wbGF0ZSA9PT0gbnVsbCkge1xuICAgICAgcmV0dXJuIFByb21pc2UucmVqZWN0KHtcbiAgICAgICAgbWVzc2FnZTogYFRlbXBsYXRlICR7dGVtcGxhdGVOYW1lfSBub3QgZm91bmRgLFxuICAgICAgICBzdGF0dXNDb2RlOiA0MDQsXG4gICAgICB9KTtcbiAgICB9XG4gICAgdHJ5IHtcbiAgICAgIGFkZFJlcXVlc3RUb01ldHJpYygnaW50ZWdyYXRpb25zJywgJ2NyZWF0ZScsICdjb3VudCcpO1xuICAgICAgY29uc3QgcmVzdWx0ID0gYXdhaXQgdGhpcy5pbnN0YW5jZUJ1aWxkZXIuYnVpbGQodGVtcGxhdGUsIHtcbiAgICAgICAgbmFtZSxcbiAgICAgICAgaW5kZXhQYXR0ZXJuLFxuICAgICAgICBkYXRhU291cmNlTURTSWQsXG4gICAgICAgIGRhdGFTb3VyY2VNRFNMYWJlbCxcbiAgICAgICAgd29ya2Zsb3dzLFxuICAgICAgICBkYXRhU291cmNlLFxuICAgICAgICB0YWJsZU5hbWUsXG4gICAgICB9KTtcbiAgICAgIGNvbnN0IHRlc3QgPSBhd2FpdCB0aGlzLmNsaWVudC5jcmVhdGUoJ2ludGVncmF0aW9uLWluc3RhbmNlJywgcmVzdWx0KTtcbiAgICAgIHJldHVybiBQcm9taXNlLnJlc29sdmUoeyAuLi5yZXN1bHQsIGlkOiB0ZXN0LmlkIH0pO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgYWRkUmVxdWVzdFRvTWV0cmljKCdpbnRlZ3JhdGlvbnMnLCAnY3JlYXRlJywgZXJyKTtcbiAgICAgIHJldHVybiBQcm9taXNlLnJlamVjdCh7XG4gICAgICAgIG1lc3NhZ2U6IGVyci5tZXNzYWdlLFxuICAgICAgICBzdGF0dXNDb2RlOiA1MDAsXG4gICAgICB9KTtcbiAgICB9XG4gIH07XG5cbiAgZ2V0U3RhdGljID0gYXN5bmMgKHRlbXBsYXRlTmFtZTogc3RyaW5nLCBzdGF0aWNQYXRoOiBzdHJpbmcpOiBQcm9taXNlPEJ1ZmZlcj4gPT4ge1xuICAgIGNvbnN0IGludGVncmF0aW9uID0gYXdhaXQgdGhpcy5yZXBvc2l0b3J5LmdldEludGVncmF0aW9uKHRlbXBsYXRlTmFtZSk7XG4gICAgaWYgKGludGVncmF0aW9uID09PSBudWxsKSB7XG4gICAgICByZXR1cm4gUHJvbWlzZS5yZWplY3Qoe1xuICAgICAgICBtZXNzYWdlOiBgVGVtcGxhdGUgJHt0ZW1wbGF0ZU5hbWV9IG5vdCBmb3VuZGAsXG4gICAgICAgIHN0YXR1c0NvZGU6IDQwNCxcbiAgICAgIH0pO1xuICAgIH1cbiAgICBjb25zdCBkYXRhID0gYXdhaXQgaW50ZWdyYXRpb24uZ2V0U3RhdGljKHN0YXRpY1BhdGgpO1xuICAgIGlmIChkYXRhLm9rKSB7XG4gICAgICByZXR1cm4gZGF0YS52YWx1ZTtcbiAgICB9XG4gICAgY29uc3QgaXM0MDQgPSAoZGF0YS5lcnJvciBhcyB7IGNvZGU/OiBzdHJpbmcgfSkuY29kZSA9PT0gJ0VOT0VOVCc7XG4gICAgcmV0dXJuIFByb21pc2UucmVqZWN0KHtcbiAgICAgIG1lc3NhZ2U6IGRhdGEuZXJyb3IubWVzc2FnZSxcbiAgICAgIHN0YXR1c0NvZGU6IGlzNDA0ID8gNDA0IDogNTAwLFxuICAgIH0pO1xuICB9O1xuXG4gIGdldFNjaGVtYXMgPSBhc3luYyAodGVtcGxhdGVOYW1lOiBzdHJpbmcpOiBQcm9taXNlPHsgbWFwcGluZ3M6IHsgW2tleTogc3RyaW5nXTogdW5rbm93biB9IH0+ID0+IHtcbiAgICBjb25zdCBpbnRlZ3JhdGlvbiA9IGF3YWl0IHRoaXMucmVwb3NpdG9yeS5nZXRJbnRlZ3JhdGlvbih0ZW1wbGF0ZU5hbWUpO1xuICAgIGlmIChpbnRlZ3JhdGlvbiA9PT0gbnVsbCkge1xuICAgICAgcmV0dXJuIFByb21pc2UucmVqZWN0KHtcbiAgICAgICAgbWVzc2FnZTogYFRlbXBsYXRlICR7dGVtcGxhdGVOYW1lfSBub3QgZm91bmRgLFxuICAgICAgICBzdGF0dXNDb2RlOiA0MDQsXG4gICAgICB9KTtcbiAgICB9XG4gICAgY29uc3QgcmVzdWx0ID0gYXdhaXQgaW50ZWdyYXRpb24uZ2V0U2NoZW1hcygpO1xuICAgIGlmIChyZXN1bHQub2spIHtcbiAgICAgIHJldHVybiByZXN1bHQudmFsdWU7XG4gICAgfVxuICAgIGNvbnN0IGlzNDA0ID0gKHJlc3VsdC5lcnJvciBhcyB7IGNvZGU/OiBzdHJpbmcgfSkuY29kZSA9PT0gJ0VOT0VOVCc7XG4gICAgcmV0dXJuIFByb21pc2UucmVqZWN0KHtcbiAgICAgIG1lc3NhZ2U6IHJlc3VsdC5lcnJvci5tZXNzYWdlLFxuICAgICAgc3RhdHVzQ29kZTogaXM0MDQgPyA0MDQgOiA1MDAsXG4gICAgfSk7XG4gIH07XG5cbiAgZ2V0QXNzZXRzID0gYXN5bmMgKHRlbXBsYXRlTmFtZTogc3RyaW5nKTogUHJvbWlzZTxQYXJzZWRJbnRlZ3JhdGlvbkFzc2V0W10+ID0+IHtcbiAgICBjb25zdCBpbnRlZ3JhdGlvbiA9IGF3YWl0IHRoaXMucmVwb3NpdG9yeS5nZXRJbnRlZ3JhdGlvbih0ZW1wbGF0ZU5hbWUpO1xuICAgIGlmIChpbnRlZ3JhdGlvbiA9PT0gbnVsbCkge1xuICAgICAgcmV0dXJuIFByb21pc2UucmVqZWN0KHtcbiAgICAgICAgbWVzc2FnZTogYFRlbXBsYXRlICR7dGVtcGxhdGVOYW1lfSBub3QgZm91bmRgLFxuICAgICAgICBzdGF0dXNDb2RlOiA0MDQsXG4gICAgICB9KTtcbiAgICB9XG4gICAgY29uc3QgYXNzZXRzID0gYXdhaXQgaW50ZWdyYXRpb24uZ2V0QXNzZXRzKCk7XG4gICAgaWYgKGFzc2V0cy5vaykge1xuICAgICAgcmV0dXJuIGFzc2V0cy52YWx1ZTtcbiAgICB9XG4gICAgY29uc3QgaXM0MDQgPSAoYXNzZXRzLmVycm9yIGFzIHsgY29kZT86IHN0cmluZyB9KS5jb2RlID09PSAnRU5PRU5UJztcbiAgICByZXR1cm4gUHJvbWlzZS5yZWplY3Qoe1xuICAgICAgbWVzc2FnZTogYXNzZXRzLmVycm9yLm1lc3NhZ2UsXG4gICAgICBzdGF0dXNDb2RlOiBpczQwNCA/IDQwNCA6IDUwMCxcbiAgICB9KTtcbiAgfTtcblxuICBnZXRTYW1wbGVEYXRhID0gYXN5bmMgKHRlbXBsYXRlTmFtZTogc3RyaW5nKTogUHJvbWlzZTx7IHNhbXBsZURhdGE6IG9iamVjdFtdIHwgbnVsbCB9PiA9PiB7XG4gICAgY29uc3QgaW50ZWdyYXRpb24gPSBhd2FpdCB0aGlzLnJlcG9zaXRvcnkuZ2V0SW50ZWdyYXRpb24odGVtcGxhdGVOYW1lKTtcbiAgICBpZiAoaW50ZWdyYXRpb24gPT09IG51bGwpIHtcbiAgICAgIHJldHVybiBQcm9taXNlLnJlamVjdCh7XG4gICAgICAgIG1lc3NhZ2U6IGBUZW1wbGF0ZSAke3RlbXBsYXRlTmFtZX0gbm90IGZvdW5kYCxcbiAgICAgICAgc3RhdHVzQ29kZTogNDA0LFxuICAgICAgfSk7XG4gICAgfVxuICAgIGNvbnN0IHNhbXBsZURhdGEgPSBhd2FpdCBpbnRlZ3JhdGlvbi5nZXRTYW1wbGVEYXRhKCk7XG4gICAgaWYgKHNhbXBsZURhdGEub2spIHtcbiAgICAgIHJldHVybiBzYW1wbGVEYXRhLnZhbHVlO1xuICAgIH1cbiAgICBjb25zdCBpczQwNCA9IChzYW1wbGVEYXRhLmVycm9yIGFzIHsgY29kZT86IHN0cmluZyB9KS5jb2RlID09PSAnRU5PRU5UJztcbiAgICByZXR1cm4gUHJvbWlzZS5yZWplY3Qoe1xuICAgICAgbWVzc2FnZTogc2FtcGxlRGF0YS5lcnJvci5tZXNzYWdlLFxuICAgICAgc3RhdHVzQ29kZTogaXM0MDQgPyA0MDQgOiA1MDAsXG4gICAgfSk7XG4gIH07XG59XG4iXSwibWFwcGluZ3MiOiI7Ozs7OztBQUtBLElBQUFBLEtBQUEsR0FBQUMsc0JBQUEsQ0FBQUMsT0FBQTtBQUVBLElBQUFDLGVBQUEsR0FBQUQsT0FBQTtBQUNBLElBQUFFLHFCQUFBLEdBQUFGLE9BQUE7QUFDQSxJQUFBRyxnQkFBQSxHQUFBSCxPQUFBO0FBQ0EsSUFBQUksbUJBQUEsR0FBQUosT0FBQTtBQUNBLElBQUFLLFdBQUEsR0FBQUwsT0FBQTtBQUEwRCxTQUFBRCx1QkFBQU8sQ0FBQSxXQUFBQSxDQUFBLElBQUFBLENBQUEsQ0FBQUMsVUFBQSxHQUFBRCxDQUFBLEtBQUFFLE9BQUEsRUFBQUYsQ0FBQTtBQUFBLFNBQUFHLGdCQUFBSCxDQUFBLEVBQUFJLENBQUEsRUFBQUMsQ0FBQSxZQUFBRCxDQUFBLEdBQUFFLGNBQUEsQ0FBQUYsQ0FBQSxNQUFBSixDQUFBLEdBQUFPLE1BQUEsQ0FBQUMsY0FBQSxDQUFBUixDQUFBLEVBQUFJLENBQUEsSUFBQUssS0FBQSxFQUFBSixDQUFBLEVBQUFLLFVBQUEsTUFBQUMsWUFBQSxNQUFBQyxRQUFBLFVBQUFaLENBQUEsQ0FBQUksQ0FBQSxJQUFBQyxDQUFBLEVBQUFMLENBQUE7QUFBQSxTQUFBTSxlQUFBRCxDQUFBLFFBQUFRLENBQUEsR0FBQUMsWUFBQSxDQUFBVCxDQUFBLHVDQUFBUSxDQUFBLEdBQUFBLENBQUEsR0FBQUEsQ0FBQTtBQUFBLFNBQUFDLGFBQUFULENBQUEsRUFBQUQsQ0FBQSwyQkFBQUMsQ0FBQSxLQUFBQSxDQUFBLFNBQUFBLENBQUEsTUFBQUwsQ0FBQSxHQUFBSyxDQUFBLENBQUFVLE1BQUEsQ0FBQUMsV0FBQSxrQkFBQWhCLENBQUEsUUFBQWEsQ0FBQSxHQUFBYixDQUFBLENBQUFpQixJQUFBLENBQUFaLENBQUEsRUFBQUQsQ0FBQSx1Q0FBQVMsQ0FBQSxTQUFBQSxDQUFBLFlBQUFLLFNBQUEseUVBQUFkLENBQUEsR0FBQWUsTUFBQSxHQUFBQyxNQUFBLEVBQUFmLENBQUEsS0FYMUQ7QUFDQTtBQUNBO0FBQ0E7QUFVTyxNQUFNZ0IsbUJBQW1CLENBQUM7RUFLL0JDLFdBQVdBLENBQUNDLE1BQWtDLEVBQUVDLFVBQTRCLEVBQUU7SUFBQXJCLGVBQUE7SUFBQUEsZUFBQTtJQUFBQSxlQUFBO0lBQUFBLGVBQUEsb0NBV2xELE1BQU9zQixFQUFVLElBQXdCO01BQ25FLElBQUlDLFFBQTBDO01BQzlDLElBQUk7UUFDRkEsUUFBUSxHQUFHLE1BQU0sSUFBSSxDQUFDSCxNQUFNLENBQUNJLEdBQUcsQ0FBQyxzQkFBc0IsRUFBRUYsRUFBRSxDQUFDO01BQzlELENBQUMsQ0FBQyxPQUFPRyxHQUFHLEVBQUU7UUFBQSxJQUFBQyxXQUFBO1FBQ1osT0FBTyxFQUFBQSxXQUFBLEdBQUFELEdBQUcsQ0FBQ0UsTUFBTSxjQUFBRCxXQUFBLHVCQUFWQSxXQUFBLENBQVlFLFVBQVUsTUFBSyxHQUFHLEdBQUdDLE9BQU8sQ0FBQ0MsT0FBTyxDQUFDLENBQUNSLEVBQUUsQ0FBQyxDQUFDLEdBQUdPLE9BQU8sQ0FBQ0UsTUFBTSxDQUFDTixHQUFHLENBQUM7TUFDckY7TUFFQSxNQUFNTyxRQUFRLEdBQUdULFFBQVEsQ0FBQ1UsVUFBVSxDQUFDQyxNQUFNLENBQ3hDQyxNQUFNLENBQUV6QixDQUFpQixJQUFLQSxDQUFDLENBQUMwQixPQUFPLENBQUMsQ0FDeENDLEdBQUcsQ0FBRTNCLENBQWlCLElBQUs7UUFDMUIsT0FBTztVQUFFWSxFQUFFLEVBQUVaLENBQUMsQ0FBQzBCLE9BQU87VUFBRUUsSUFBSSxFQUFFNUIsQ0FBQyxDQUFDNkI7UUFBVSxDQUFDO01BQzdDLENBQUMsQ0FBQztNQUNKUCxRQUFRLENBQUNRLElBQUksQ0FBQztRQUFFbEIsRUFBRTtRQUFFZ0IsSUFBSSxFQUFFO01BQXVCLENBQUMsQ0FBQztNQUVuRCxNQUFNRyxNQUFNLEdBQUdaLE9BQU8sQ0FBQ2EsR0FBRyxDQUN4QlYsUUFBUSxDQUFDSyxHQUFHLENBQ1YsTUFBT00sS0FBbUMsSUFBc0I7UUFDOUQsSUFBSTtVQUNGLE1BQU0sSUFBSSxDQUFDdkIsTUFBTSxDQUFDd0IsTUFBTSxDQUFDRCxLQUFLLENBQUNMLElBQUksRUFBRUssS0FBSyxDQUFDckIsRUFBRSxDQUFDO1VBQzlDLE9BQU9PLE9BQU8sQ0FBQ0MsT0FBTyxDQUFDYSxLQUFLLENBQUNyQixFQUFFLENBQUM7UUFDbEMsQ0FBQyxDQUFDLE9BQU9HLEdBQUcsRUFBRTtVQUFBLElBQUFvQixZQUFBO1VBQ1osSUFBQUMsa0NBQWtCLEVBQUMsY0FBYyxFQUFFLFFBQVEsRUFBRXJCLEdBQUcsQ0FBQztVQUNqRCxPQUFPLEVBQUFvQixZQUFBLEdBQUFwQixHQUFHLENBQUNFLE1BQU0sY0FBQWtCLFlBQUEsdUJBQVZBLFlBQUEsQ0FBWWpCLFVBQVUsTUFBSyxHQUFHLEdBQUdDLE9BQU8sQ0FBQ0MsT0FBTyxDQUFDYSxLQUFLLENBQUNyQixFQUFFLENBQUMsR0FBR08sT0FBTyxDQUFDRSxNQUFNLENBQUNOLEdBQUcsQ0FBQztRQUN6RjtNQUNGLENBQ0YsQ0FDRixDQUFDO01BQ0QsSUFBQXFCLGtDQUFrQixFQUFDLGNBQWMsRUFBRSxRQUFRLEVBQUUsT0FBTyxDQUFDO01BQ3JELE9BQU9MLE1BQU07SUFDZixDQUFDO0lBRUQ7SUFBQXpDLGVBQUEsc0NBQzhCLFlBQXNEO01BQ2xGLE1BQU0rQyxlQUFlLEdBQUcsTUFBTSxJQUFJLENBQUMxQixVQUFVLENBQUMyQixrQkFBa0IsQ0FBQyxDQUFDO01BQ2xFLE1BQU1DLGFBQWEsR0FBRyxNQUFNcEIsT0FBTyxDQUFDYSxHQUFHLENBQUNLLGVBQWUsQ0FBQ1YsR0FBRyxDQUFFYSxDQUFDLElBQUtBLENBQUMsQ0FBQ0MsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDO01BQ2xGLE1BQU1DLE9BQU8sR0FBR0gsYUFBYSxDQUFDZCxNQUFNLENBQUVrQixHQUFHLElBQUtBLEdBQUcsQ0FBQ0MsRUFBRSxDQUF3QztNQUM1RixPQUFPekIsT0FBTyxDQUFDQyxPQUFPLENBQUM7UUFBRXlCLElBQUksRUFBRUgsT0FBTyxDQUFDZixHQUFHLENBQUVnQixHQUFHLElBQUtBLEdBQUcsQ0FBQy9DLEtBQUs7TUFBRSxDQUFDLENBQUM7SUFDbkUsQ0FBQztJQUVEO0lBQUFOLGVBQUEseUNBQ2lDLE1BQy9Cd0QsSUFBWSxJQUNpQztNQUM3QyxNQUFNQyxXQUFXLEdBQUcsTUFBTSxJQUFJLENBQUNwQyxVQUFVLENBQUNxQyxjQUFjLENBQUNGLElBQUksQ0FBQztNQUM5RCxNQUFNRyxNQUFNLEdBQUcsT0FBTUYsV0FBVyxhQUFYQSxXQUFXLHVCQUFYQSxXQUFXLENBQUVOLFNBQVMsQ0FBQyxDQUFDO01BQzdDLElBQUksQ0FBQ1EsTUFBTSxJQUFJLENBQUNBLE1BQU0sQ0FBQ0wsRUFBRSxFQUFFO1FBQ3pCLE9BQU96QixPQUFPLENBQUNDLE9BQU8sQ0FBQztVQUFFeUIsSUFBSSxFQUFFO1FBQUcsQ0FBQyxDQUFDO01BQ3RDO01BQ0EsT0FBTzFCLE9BQU8sQ0FBQ0MsT0FBTyxDQUFDO1FBQUV5QixJQUFJLEVBQUUsQ0FBQ0ksTUFBTSxDQUFDckQsS0FBSztNQUFFLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBQUFOLGVBQUEsa0NBRXlCLE1BQ3hCNEQsS0FBZ0MsSUFDYTtNQUM3QyxJQUFJQSxLQUFLLGFBQUxBLEtBQUssZUFBTEEsS0FBSyxDQUFFSixJQUFJLEVBQUU7UUFDZixPQUFPLElBQUksQ0FBQ0ssOEJBQThCLENBQUNELEtBQUssQ0FBQ0osSUFBSSxDQUFDO01BQ3hEO01BQ0EsT0FBTyxJQUFJLENBQUNNLDJCQUEyQixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUFBOUQsZUFBQSxrQ0FFeUIsTUFDeEIrRCxNQUFpQyxJQUNhO01BQUEsSUFBQUMscUJBQUE7TUFDOUMsSUFBQWxCLGtDQUFrQixFQUFDLGNBQWMsRUFBRSxLQUFLLEVBQUUsT0FBTyxDQUFDO01BQ2xELE1BQU1MLE1BQU0sR0FBRyxNQUFNLElBQUksQ0FBQ3JCLE1BQU0sQ0FBQzZDLElBQUksQ0FBQztRQUFFM0IsSUFBSSxFQUFFLHNCQUFzQjtRQUFFNEIsT0FBTyxFQUFFO01BQUssQ0FBQyxDQUFDO01BQ3RGLE9BQU9yQyxPQUFPLENBQUNDLE9BQU8sQ0FBQztRQUNyQnFDLEtBQUssRUFBRTFCLE1BQU0sQ0FBQzBCLEtBQUs7UUFDbkJaLElBQUksR0FBQVMscUJBQUEsR0FBRXZCLE1BQU0sQ0FBQzJCLGFBQWEsY0FBQUoscUJBQUEsdUJBQXBCQSxxQkFBQSxDQUFzQjNCLEdBQUcsQ0FBRWEsQ0FBQyxLQUFNO1VBQ3RDLEdBQUdBLENBQUMsQ0FBQ2pCLFVBQVc7VUFDaEJYLEVBQUUsRUFBRTRCLENBQUMsQ0FBQzVCO1FBQ1IsQ0FBQyxDQUFDO01BQ0osQ0FBQyxDQUFDO0lBQ0osQ0FBQztJQUFBdEIsZUFBQSxpQ0FFd0IsTUFDdkI0RCxLQUFnQyxJQUNPO01BQ3ZDLElBQUFkLGtDQUFrQixFQUFDLGNBQWMsRUFBRSxLQUFLLEVBQUUsT0FBTyxDQUFDO01BQ2xELE1BQU1MLE1BQU0sR0FBSSxNQUFNLElBQUksQ0FBQ3JCLE1BQU0sQ0FBQ0ksR0FBRyxDQUFDLHNCQUFzQixFQUFHLEdBQUVvQyxLQUFLLENBQUV0QyxFQUFHLEVBQUMsQ0FFM0U7TUFDRCxPQUFPTyxPQUFPLENBQUNDLE9BQU8sQ0FBQyxJQUFJLENBQUN1QyxxQkFBcUIsQ0FBQzVCLE1BQU0sQ0FBQyxDQUFDO0lBQzVELENBQUM7SUFBQXpDLGVBQUEsZ0NBRXVCLE1BQ3RCc0UsUUFBMEMsSUFDSDtNQUN2QyxNQUFNcEMsTUFBb0MsR0FBR29DLFFBQVEsQ0FBQ3JDLFVBQVUsQ0FBQ0MsTUFBTTtNQUN2RSxNQUFNcUMsTUFBYyxHQUFHckMsTUFBTSxHQUFHLE1BQU0sSUFBSSxDQUFDc0MsY0FBYyxDQUFDdEMsTUFBTSxDQUFDLEdBQUcsV0FBVztNQUUvRSxPQUFPO1FBQ0xaLEVBQUUsRUFBRWdELFFBQVEsQ0FBQ2hELEVBQUU7UUFDZmlELE1BQU07UUFDTixHQUFHRCxRQUFRLENBQUNyQztNQUNkLENBQUM7SUFDSCxDQUFDO0lBQUFqQyxlQUFBLHlCQUVnQixNQUFPa0MsTUFBd0IsSUFBc0I7TUFDcEUsTUFBTXVDLFFBQStDLEdBQUcsTUFBTTVDLE9BQU8sQ0FBQ2EsR0FBRyxDQUN2RVIsTUFBTSxDQUFDRyxHQUFHLENBQUMsTUFBT00sS0FBSyxJQUFLO1FBQzFCLElBQUk7VUFDRixNQUFNLElBQUksQ0FBQ3ZCLE1BQU0sQ0FBQ0ksR0FBRyxDQUFDbUIsS0FBSyxDQUFDSixTQUFTLEVBQUVJLEtBQUssQ0FBQ1AsT0FBTyxDQUFDO1VBQ3JELE9BQU87WUFBRWQsRUFBRSxFQUFFcUIsS0FBSyxDQUFDUCxPQUFPO1lBQUVtQyxNQUFNLEVBQUU7VUFBWSxDQUFDO1FBQ25ELENBQUMsQ0FBQyxPQUFPOUMsR0FBRyxFQUFFO1VBQUEsSUFBQWlELFlBQUE7VUFDWixNQUFNOUMsVUFBVSxJQUFBOEMsWUFBQSxHQUFHakQsR0FBRyxDQUFDRSxNQUFNLGNBQUErQyxZQUFBLHVCQUFWQSxZQUFBLENBQVk5QyxVQUFVO1VBQ3pDLElBQUlBLFVBQVUsSUFBSSxHQUFHLElBQUlBLFVBQVUsSUFBSUEsVUFBVSxHQUFHLEdBQUcsRUFBRTtZQUN2RCxPQUFPO2NBQUVOLEVBQUUsRUFBRXFCLEtBQUssQ0FBQ1AsT0FBTztjQUFFbUMsTUFBTSxFQUFFO1lBQWMsQ0FBQztVQUNyRDtVQUNBSSxPQUFPLENBQUNDLEtBQUssQ0FBQyw0QkFBNEIsRUFBRW5ELEdBQUcsQ0FBQztVQUNoRCxPQUFPO1lBQUVILEVBQUUsRUFBRXFCLEtBQUssQ0FBQ1AsT0FBTztZQUFFbUMsTUFBTSxFQUFFO1VBQVUsQ0FBQztRQUNqRDtNQUNGLENBQUMsQ0FDSCxDQUFDO01BRUQsTUFBTSxDQUFDTSxTQUFTLEVBQUVDLFdBQVcsRUFBRUMsT0FBTyxDQUFDLEdBQUcsQ0FDeENOLFFBQVEsQ0FBQ3RDLE1BQU0sQ0FBRWUsQ0FBQyxJQUFLQSxDQUFDLENBQUNxQixNQUFNLEtBQUssV0FBVyxDQUFDLENBQUNTLE1BQU0sRUFDdkRQLFFBQVEsQ0FBQ3RDLE1BQU0sQ0FBRWUsQ0FBQyxJQUFLQSxDQUFDLENBQUNxQixNQUFNLEtBQUssYUFBYSxDQUFDLENBQUNTLE1BQU0sRUFDekRQLFFBQVEsQ0FBQ3RDLE1BQU0sQ0FBRWUsQ0FBQyxJQUFLQSxDQUFDLENBQUNxQixNQUFNLEtBQUssU0FBUyxDQUFDLENBQUNTLE1BQU0sQ0FDdEQ7TUFDRCxJQUFJRCxPQUFPLEdBQUcsQ0FBQyxFQUFFLE9BQU8sU0FBUztNQUNqQyxJQUFJRCxXQUFXLEdBQUcsQ0FBQyxJQUFJRCxTQUFTLEdBQUcsQ0FBQyxFQUFFLE9BQU8scUJBQXFCO01BQ2xFLElBQUlDLFdBQVcsR0FBRyxDQUFDLEVBQUUsT0FBTyxhQUFhO01BQ3pDLE9BQU8sV0FBVztJQUNwQixDQUFDO0lBQUE5RSxlQUFBLGtDQUV5QixPQUN4QmlGLFlBQW9CLEVBQ3BCekIsSUFBWSxFQUNaMEIsWUFBb0IsRUFDcEJDLGVBQXdCLEVBQ3hCQyxrQkFBMkIsRUFDM0JDLFNBQW9CLEVBQ3BCQyxVQUFtQixFQUNuQkMsU0FBa0IsS0FDZTtNQUNqQyxNQUFNQyxRQUFRLEdBQUcsTUFBTSxJQUFJLENBQUNuRSxVQUFVLENBQUNxQyxjQUFjLENBQUN1QixZQUFZLENBQUM7TUFDbkUsSUFBSU8sUUFBUSxLQUFLLElBQUksRUFBRTtRQUNyQixPQUFPM0QsT0FBTyxDQUFDRSxNQUFNLENBQUM7VUFDcEIwRCxPQUFPLEVBQUcsWUFBV1IsWUFBYSxZQUFXO1VBQzdDckQsVUFBVSxFQUFFO1FBQ2QsQ0FBQyxDQUFDO01BQ0o7TUFDQSxJQUFJO1FBQ0YsSUFBQWtCLGtDQUFrQixFQUFDLGNBQWMsRUFBRSxRQUFRLEVBQUUsT0FBTyxDQUFDO1FBQ3JELE1BQU1MLE1BQU0sR0FBRyxNQUFNLElBQUksQ0FBQ2lELGVBQWUsQ0FBQ0MsS0FBSyxDQUFDSCxRQUFRLEVBQUU7VUFDeERoQyxJQUFJO1VBQ0owQixZQUFZO1VBQ1pDLGVBQWU7VUFDZkMsa0JBQWtCO1VBQ2xCQyxTQUFTO1VBQ1RDLFVBQVU7VUFDVkM7UUFDRixDQUFDLENBQUM7UUFDRixNQUFNSyxJQUFJLEdBQUcsTUFBTSxJQUFJLENBQUN4RSxNQUFNLENBQUN5RSxNQUFNLENBQUMsc0JBQXNCLEVBQUVwRCxNQUFNLENBQUM7UUFDckUsT0FBT1osT0FBTyxDQUFDQyxPQUFPLENBQUM7VUFBRSxHQUFHVyxNQUFNO1VBQUVuQixFQUFFLEVBQUVzRSxJQUFJLENBQUN0RTtRQUFHLENBQUMsQ0FBQztNQUNwRCxDQUFDLENBQUMsT0FBT0csR0FBRyxFQUFFO1FBQ1osSUFBQXFCLGtDQUFrQixFQUFDLGNBQWMsRUFBRSxRQUFRLEVBQUVyQixHQUFHLENBQUM7UUFDakQsT0FBT0ksT0FBTyxDQUFDRSxNQUFNLENBQUM7VUFDcEIwRCxPQUFPLEVBQUVoRSxHQUFHLENBQUNnRSxPQUFPO1VBQ3BCN0QsVUFBVSxFQUFFO1FBQ2QsQ0FBQyxDQUFDO01BQ0o7SUFDRixDQUFDO0lBQUE1QixlQUFBLG9CQUVXLE9BQU9pRixZQUFvQixFQUFFYSxVQUFrQixLQUFzQjtNQUMvRSxNQUFNckMsV0FBVyxHQUFHLE1BQU0sSUFBSSxDQUFDcEMsVUFBVSxDQUFDcUMsY0FBYyxDQUFDdUIsWUFBWSxDQUFDO01BQ3RFLElBQUl4QixXQUFXLEtBQUssSUFBSSxFQUFFO1FBQ3hCLE9BQU81QixPQUFPLENBQUNFLE1BQU0sQ0FBQztVQUNwQjBELE9BQU8sRUFBRyxZQUFXUixZQUFhLFlBQVc7VUFDN0NyRCxVQUFVLEVBQUU7UUFDZCxDQUFDLENBQUM7TUFDSjtNQUNBLE1BQU1tRSxJQUFJLEdBQUcsTUFBTXRDLFdBQVcsQ0FBQ3VDLFNBQVMsQ0FBQ0YsVUFBVSxDQUFDO01BQ3BELElBQUlDLElBQUksQ0FBQ3pDLEVBQUUsRUFBRTtRQUNYLE9BQU95QyxJQUFJLENBQUN6RixLQUFLO01BQ25CO01BQ0EsTUFBTTJGLEtBQUssR0FBSUYsSUFBSSxDQUFDbkIsS0FBSyxDQUF1QnNCLElBQUksS0FBSyxRQUFRO01BQ2pFLE9BQU9yRSxPQUFPLENBQUNFLE1BQU0sQ0FBQztRQUNwQjBELE9BQU8sRUFBRU0sSUFBSSxDQUFDbkIsS0FBSyxDQUFDYSxPQUFPO1FBQzNCN0QsVUFBVSxFQUFFcUUsS0FBSyxHQUFHLEdBQUcsR0FBRztNQUM1QixDQUFDLENBQUM7SUFDSixDQUFDO0lBQUFqRyxlQUFBLHFCQUVZLE1BQU9pRixZQUFvQixJQUF3RDtNQUM5RixNQUFNeEIsV0FBVyxHQUFHLE1BQU0sSUFBSSxDQUFDcEMsVUFBVSxDQUFDcUMsY0FBYyxDQUFDdUIsWUFBWSxDQUFDO01BQ3RFLElBQUl4QixXQUFXLEtBQUssSUFBSSxFQUFFO1FBQ3hCLE9BQU81QixPQUFPLENBQUNFLE1BQU0sQ0FBQztVQUNwQjBELE9BQU8sRUFBRyxZQUFXUixZQUFhLFlBQVc7VUFDN0NyRCxVQUFVLEVBQUU7UUFDZCxDQUFDLENBQUM7TUFDSjtNQUNBLE1BQU1hLE1BQU0sR0FBRyxNQUFNZ0IsV0FBVyxDQUFDMEMsVUFBVSxDQUFDLENBQUM7TUFDN0MsSUFBSTFELE1BQU0sQ0FBQ2EsRUFBRSxFQUFFO1FBQ2IsT0FBT2IsTUFBTSxDQUFDbkMsS0FBSztNQUNyQjtNQUNBLE1BQU0yRixLQUFLLEdBQUl4RCxNQUFNLENBQUNtQyxLQUFLLENBQXVCc0IsSUFBSSxLQUFLLFFBQVE7TUFDbkUsT0FBT3JFLE9BQU8sQ0FBQ0UsTUFBTSxDQUFDO1FBQ3BCMEQsT0FBTyxFQUFFaEQsTUFBTSxDQUFDbUMsS0FBSyxDQUFDYSxPQUFPO1FBQzdCN0QsVUFBVSxFQUFFcUUsS0FBSyxHQUFHLEdBQUcsR0FBRztNQUM1QixDQUFDLENBQUM7SUFDSixDQUFDO0lBQUFqRyxlQUFBLG9CQUVXLE1BQU9pRixZQUFvQixJQUF3QztNQUM3RSxNQUFNeEIsV0FBVyxHQUFHLE1BQU0sSUFBSSxDQUFDcEMsVUFBVSxDQUFDcUMsY0FBYyxDQUFDdUIsWUFBWSxDQUFDO01BQ3RFLElBQUl4QixXQUFXLEtBQUssSUFBSSxFQUFFO1FBQ3hCLE9BQU81QixPQUFPLENBQUNFLE1BQU0sQ0FBQztVQUNwQjBELE9BQU8sRUFBRyxZQUFXUixZQUFhLFlBQVc7VUFDN0NyRCxVQUFVLEVBQUU7UUFDZCxDQUFDLENBQUM7TUFDSjtNQUNBLE1BQU1NLE1BQU0sR0FBRyxNQUFNdUIsV0FBVyxDQUFDMkMsU0FBUyxDQUFDLENBQUM7TUFDNUMsSUFBSWxFLE1BQU0sQ0FBQ29CLEVBQUUsRUFBRTtRQUNiLE9BQU9wQixNQUFNLENBQUM1QixLQUFLO01BQ3JCO01BQ0EsTUFBTTJGLEtBQUssR0FBSS9ELE1BQU0sQ0FBQzBDLEtBQUssQ0FBdUJzQixJQUFJLEtBQUssUUFBUTtNQUNuRSxPQUFPckUsT0FBTyxDQUFDRSxNQUFNLENBQUM7UUFDcEIwRCxPQUFPLEVBQUV2RCxNQUFNLENBQUMwQyxLQUFLLENBQUNhLE9BQU87UUFDN0I3RCxVQUFVLEVBQUVxRSxLQUFLLEdBQUcsR0FBRyxHQUFHO01BQzVCLENBQUMsQ0FBQztJQUNKLENBQUM7SUFBQWpHLGVBQUEsd0JBRWUsTUFBT2lGLFlBQW9CLElBQStDO01BQ3hGLE1BQU14QixXQUFXLEdBQUcsTUFBTSxJQUFJLENBQUNwQyxVQUFVLENBQUNxQyxjQUFjLENBQUN1QixZQUFZLENBQUM7TUFDdEUsSUFBSXhCLFdBQVcsS0FBSyxJQUFJLEVBQUU7UUFDeEIsT0FBTzVCLE9BQU8sQ0FBQ0UsTUFBTSxDQUFDO1VBQ3BCMEQsT0FBTyxFQUFHLFlBQVdSLFlBQWEsWUFBVztVQUM3Q3JELFVBQVUsRUFBRTtRQUNkLENBQUMsQ0FBQztNQUNKO01BQ0EsTUFBTXlFLFVBQVUsR0FBRyxNQUFNNUMsV0FBVyxDQUFDNkMsYUFBYSxDQUFDLENBQUM7TUFDcEQsSUFBSUQsVUFBVSxDQUFDL0MsRUFBRSxFQUFFO1FBQ2pCLE9BQU8rQyxVQUFVLENBQUMvRixLQUFLO01BQ3pCO01BQ0EsTUFBTTJGLEtBQUssR0FBSUksVUFBVSxDQUFDekIsS0FBSyxDQUF1QnNCLElBQUksS0FBSyxRQUFRO01BQ3ZFLE9BQU9yRSxPQUFPLENBQUNFLE1BQU0sQ0FBQztRQUNwQjBELE9BQU8sRUFBRVksVUFBVSxDQUFDekIsS0FBSyxDQUFDYSxPQUFPO1FBQ2pDN0QsVUFBVSxFQUFFcUUsS0FBSyxHQUFHLEdBQUcsR0FBRztNQUM1QixDQUFDLENBQUM7SUFDSixDQUFDO0lBelBDLElBQUksQ0FBQzdFLE1BQU0sR0FBR0EsTUFBTTtJQUNwQixJQUFJLENBQUNDLFVBQVUsR0FDYkEsVUFBVSxhQUFWQSxVQUFVLGNBQVZBLFVBQVUsR0FDVixJQUFJa0YsMkJBQWUsQ0FBQyxDQUNsQixJQUFJQyxvQ0FBZ0IsQ0FBQyxJQUFJLENBQUNwRixNQUFNLENBQUMsRUFDakMsSUFBSXFGLHNDQUFxQixDQUFDQyxhQUFJLENBQUNDLElBQUksQ0FBQ0MsU0FBUyxFQUFFLHFCQUFxQixDQUFDLENBQUMsQ0FDdkUsQ0FBQztJQUNKLElBQUksQ0FBQ2xCLGVBQWUsR0FBRyxJQUFJbUIsZ0RBQTBCLENBQUMsSUFBSSxDQUFDekYsTUFBTSxDQUFDO0VBQ3BFO0FBa1BGO0FBQUMwRixPQUFBLENBQUE1RixtQkFBQSxHQUFBQSxtQkFBQSJ9