// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System;
using System.IO;
using System.Linq;
using System.Collections;
using System.Collections.Generic;
using System.Xml;
using System.Xml.Linq;
using XmlCoreTest.Common;

namespace CoreXml.Test.XLinq
{
    // used in DOMXmlFiles
    public enum DocumentStatus
    {
        valid,
        invalid,
        notWellFormed,
        all
    }

    //////////////////////////////////////////////////////////////////////////////////////////////////////
    /// <summary>
    /// Class for input generation used by all tests
    /// </summary>
    //////////////////////////////////////////////////////////////////////////////////////////////////////
    public static class InputSpace
    {
        // loads either the file for DMLCreate
        public static void Load(object loadObject, ref XDocument xDoc, ref XDocument xmlDoc)
        {
            Load(loadObject as string, ref xDoc, ref xmlDoc);
        }

        // Creates XDocument and XmlDocument from a input file
        public static void Load(string fileName, ref XDocument xDoc, ref XDocument xmlDoc)
        {
            XmlReaderSettings rs = new XmlReaderSettings();
            rs.IgnoreWhitespace = true;
            rs.DtdProcessing = DtdProcessing.Ignore;
            string filePath = Path.Combine("TestData", "XLinq", fileName);
            using (XmlReader r1 = XmlReader.Create(FilePathUtil.getStream(filePath), rs))
                xDoc = XDocument.Load(r1);
            using (XmlReader r2 = XmlReader.Create(FilePathUtil.getStream(filePath), rs))
            {
                xmlDoc = XDocument.Load(r2);
            }
        }

        // List of input xml files used in DOM Control files
        public static IEnumerable<string> DOMXmlFiles(DocumentStatus docStatus)
        {
            string status, file, filePath;
            XNamespace ns = "WebData/TestSpec";
            // look in each one of the DOM test control files for xml files used
            foreach (string controlFile in DOMControlFiles())
            {
                XDocument xDoc = null;
                XmlReaderSettings rs = new XmlReaderSettings();
                rs.IgnoreWhitespace = true;
                using (XmlReader reader = XmlReader.Create(FilePathUtil.getStream(controlFile), rs))
                {
                    xDoc = XDocument.Load(reader);
                }

                // get the filePath from the control file
                filePath = (string)xDoc.Descendants(ns + "global").Attributes("filePath").FirstOrDefault();
                if (filePath.StartsWith(@"$(DataPath)\StandardTests\"))
                {
                    filePath = Path.Combine(XLinqTestCase.StandardPath, filePath.Substring(@"$(DataPath)\StandardTests\".Length));
                }
                // get xml file used in each variation
                foreach (XElement e in xDoc.Descendants(ns + "variation").Where(x => x.Attribute("implemented").Value == "true"))
                {
                    file = e.Descendants(ns + "xml").FirstOrDefault().Value.Trim();
                    file = Path.Combine(filePath, file);
                    // check if the file exists, if not then continue
                    try
                    {
                        if (FilePathUtil.getStream(file) == null)
                            continue;
                    }
                    catch (Exception)
                    {
                        continue;
                    }
                    // if there are more than one "result" nodes then find the one that is used for System.Xml
                    IEnumerable<XElement> resultNodes = e.Descendants(ns + "result");
                    IEnumerable<XElement> systemXml = e.Descendants(ns + "System.Xml");
                    if (resultNodes.Count() > 1 && systemXml.Count() > 0)
                    {
                        string resultID = (string)systemXml.Select(x => x.Element(ns + "expectedResult")).Attributes("id").FirstOrDefault();
                        status = (string)resultNodes.Where(x => x.Attribute("id").Value == resultID).Attributes("expected").FirstOrDefault();
                    }
                    else
                        status = (string)resultNodes.Attributes("expected").FirstOrDefault();

                    switch (docStatus)
                    {
                        case DocumentStatus.valid:
                            if ("valid".Equals(status)) yield return file;
                            break;
                        case DocumentStatus.invalid:
                            if ("invalid".Equals(status)) yield return file;
                            break;
                        case DocumentStatus.notWellFormed:
                            if ("not-wf".Equals(status)) yield return file;
                            break;
                        default:
                            yield return file;
                            break;
                    }
                }
            }
        }

        // List of control files used in DOM tests
        public static IEnumerable<string> DOMControlFiles()
        {
            XDocument xDoc = null;
            XmlReaderSettings rs = new XmlReaderSettings();
            rs.IgnoreWhitespace = true;
            string masterControlFile = @"xlinq_master_xml.xml";
            using (XmlReader reader = XmlReader.Create(FilePathUtil.getStream(masterControlFile), rs))
            {
                xDoc = XDocument.Load(reader);
            }

            // Get test specific control files
            foreach (XElement x in xDoc.Descendants("testCase"))
            {
                yield return ((string)x.Attribute("uri"));
            }
        }

        public static XElement GetElement(int elementCount, int childElementCount)
        {
            XElement root = new XElement("root");

            for (int l = 0; l < elementCount; l++)
            {
                XElement nl = new XElement("e" + Guid.NewGuid().ToString(), Guid.NewGuid().ToString());
                root.Add(nl);
                for (int e = 0; e < childElementCount; e++)
                {
                    XElement child = new XElement("e" + Guid.NewGuid().ToString(), Guid.NewGuid().ToString());
                    nl.Add(child);
                }
            }
            return root;
        }

        public static XElement GetAttributeElement(int elementCount, int attributeCount)
        {
            XElement root = new XElement("root");

            for (int l = 0; l < elementCount; l++)
            {
                XElement nl = new XElement("e" + Guid.NewGuid().ToString(), Guid.NewGuid().ToString());
                for (int a = 0; a < attributeCount; a++)
                {
                    XAttribute attribute = new XAttribute("a" + Guid.NewGuid().ToString(), Guid.NewGuid().ToString());
                    nl.Add(attribute);
                }
                root.Add(nl);
            }
            return root;
        }

        public static XElement GetNamespaceElement(int namespaceCount, int elementCount, int childElementCount)
        {
            XElement root = new XElement("root");
            Random random = new Random(unchecked((int)DateTime.Now.Ticks));

            Guid[] namespaces = new Guid[namespaceCount];
            for (int n = 0; n < namespaceCount; n++)
                namespaces[n] = Guid.NewGuid();

            for (int l = 0; l < elementCount; l++)
            {
                XElement nl = new XElement("{n" + namespaces[random.Next(namespaceCount)] + "}p" + Guid.NewGuid().ToString(), Guid.NewGuid().ToString());
                root.Add(nl);
                for (int e = 0; e < childElementCount; e++)
                {
                    XElement child = new XElement("{n" + namespaces[random.Next(namespaceCount)] + "}c" + Guid.NewGuid().ToString(), Guid.NewGuid().ToString());
                    nl.Add(child);
                }
            }
            return root;
        }

        public static void RootElement(ref XDocument xDoc, ref XDocument xmlDoc)
        {
            xDoc = XDocument.Parse("<root/>");
            xmlDoc = XDocument.Parse(xDoc.ToString(SaveOptions.DisableFormatting));
        }

        public static void TextNodes(ref XDocument xDoc, ref XDocument xmlDoc)
        {
            xDoc = XDocument.Parse("<root>some text</root>");
            xmlDoc = XDocument.Parse(xDoc.ToString(SaveOptions.DisableFormatting));
        }

        public static void Contacts(ref XDocument xDoc, ref XDocument xmlDoc)
        {
            xDoc = new XDocument(
                new XDeclaration("1.0", "utf-8", "yes"),
                new XProcessingInstruction("AppName", "Processing Instruction Data"),
                new XComment("Personal Contacts"),
                new XElement("contacts",
                    new XAttribute("category", "friends"),
                    new XAttribute("gender", "male"),
                     new XElement("contact",
                        new XAttribute("netWorth", "100"),
                        new XElement("name", "John Hopkins"),
                        new XElement("phone",
                            new XAttribute("type", "home"),
                            "214-459-8907"),
                        new XElement("phone",
                            new XAttribute("type", "work"),
                            "817-283-9670")),
                    new XElement("contact",
                        new XAttribute("netWorth", "10"),
                        new XElement("name", "Patrick Hines"),
                        new XElement("phone",
                            new XAttribute("type", "home"),
                            "206-555-0144"),
                        new XElement("phone",
                            new XAttribute("type", "work"),
                            "425-555-0145"))));

            xmlDoc = XDocument.Parse(xDoc.ToString(SaveOptions.DisableFormatting));
        }

        public static void Teams(ref XDocument xDoc, ref XDocument xmlDoc)
        {
            XNamespace sp = "http://www.sportsleagues.com/";
            XNamespace nba = "http://www.nba.com/";
            xDoc = new XDocument(
                new XDeclaration("1.0", "utf-8", "yes"),
                new XProcessingInstruction("AppName", "Processing Instruction Data"),
                new XComment("Sport Leaques"),
                new XElement(sp + "SportLeagues",
                    new XAttribute(XNamespace.Xmlns + "sp", sp),
                    new XElement("League",
                        new XAttribute("type", "Basketball"),
                        new XAttribute("name", "National Basketball Association"),
                        new XAttribute("id", "NBA"),
                        new XAttribute(XNamespace.Xmlns + "nba", nba),
                        new XElement("commissioner", "David Stern"),
                        new XElement(nba + "Conference",
                            new XAttribute(nba + "name", "Eastern"),
                            new XElement(nba + "Division",
                                new XAttribute(nba + "name", "Central")
                            ),
                            new XElement(nba + "Division",
                                new XAttribute(nba + "name", "SouthEast")
                            ),
                            new XElement(nba + "Division",
                                new XAttribute(nba + "name", "Atlantic"),
                                new XElement(nba + "Teams",
                                    new XElement(nba + "Team",
                                        new XAttribute(nba + "name", "New Jersy Nets"),
                                        new XElement(nba + "Players",
                                            new XElement(nba + "Player",
                                                new XComment(" The best player on the team "),
                                                "\n" +
                                                "                Or Is he the best player on the team?\n" +
                                                "                ",
                                                new XElement(nba + "Name", "Richard Jefferson"),
                                                "<>,;:.@#$%^&*()?/`~+-",
                                                new XProcessingInstruction("Player", "processing instruction")
                                            ),
                                            new XElement(nba + "Player")
                                        )
                                    ),
                                    new XElement(nba + "Team")
                                )
                            )
                        ),
                        new XElement(nba + "Conference",
                            new XAttribute(nba + "name", "Western"),
                            new XElement(nba + "Division",
                                new XAttribute("name", "NorthWest")
                            ),
                            new XElement(nba + "Division",
                                new XAttribute(nba + "name", "SouthWest")
                            ),
                            new XElement(nba + "Division",
                                new XAttribute(nba + "name", "Pacific")
                            )
                        )
                    ),
                    new XElement("League",
                        new XAttribute("type", "Football"),
                        new XAttribute("name", "National Football League"),
                        new XAttribute("id", "NFL"),
                        new XAttribute(XNamespace.Xmlns + "nfl", "http://www.nfl.com"),
                        new XElement("commissioner", "Paul Taglibu"),
                        new XElement("Conference",
                            new XAttribute("name", "NFC"),
                            new XElement("Division",
                                new XAttribute("name", "East"),
                                new XElement("Teams",
                                    new XElement("Team",
                                        new XAttribute("name", "Dallas Cowboys"),
                                        new XComment(" The best team in the division "),
                                        "\n" +
                                        "            Or Is it the best team in the division?\n" +
                                        "            ",
                                        new XElement("Coach", "Bill Parcells"),
                                        "<>,;:.@#$%^&*()?/`~+-",
                                        new XProcessingInstruction("Cowboys", "processing instruction"),
                                        new XElement("Players")
                                    ),
                                    new XElement("Team",
                                        new XAttribute("name", "New Your Giants"),
                                        new XComment(" The second best team in the division "),
                                        "\n" +
                                        "            Or Is it the second best team in the division?\n" +
                                        "            ",
                                        new XElement("Coach", "Tom Coughlin"),
                                        "<>,;:.@#$%^&*()?/`~+-",
                                        new XProcessingInstruction("Giants", "processing instruction"),
                                        new XElement("Players")
                                    ),
                                    new XElement("Team"),
                                    new XElement("Team")
                                )
                            ),
                            new XElement("Division",
                                new XAttribute("name", "West"),
                                new XElement("Teams",
                                    new XElement("Team",
                                        new XAttribute("name", "Seattle Seahawks"),
                                        new XComment(" The best team in the division "),
                                        "\n" +
                                        "            Or Is it the best team in the division?\n" +
                                        "            ",
                                        new XElement("Coach", "Mike Holmgren"),
                                        "This should be a text node",
                                        new XProcessingInstruction("Seakhwaks", "processing instruction"),
                                        new XElement("Players")
                                    ),
                                    new XElement("Team",
                                        new XAttribute("name", "Arizona Cardinals"),
                                        new XComment(" The second best team in the division "),
                                        "\n" +
                                        "            Or Is it the second best team in the division?\n" +
                                        "            ",
                                        new XElement("Coach", "Dennis Green"),
                                        "<>,;:.@#$%^&*()?/`~+-",
                                        new XProcessingInstruction("Cardinals", "processing instruction"),
                                        new XElement("Players")
                                    ),
                                    new XElement("Team"),
                                    new XElement("Team")
                                )
                            ),
                            new XElement("Division",
                                new XAttribute("name", "North"),
                                new XElement("Teams")
                            ),
                            new XElement("Division",
                                new XAttribute("name", "South"),
                                new XElement("Teams")
                            )
                        ),
                        new XElement("Conference",
                            new XAttribute("name", "AFC"),
                            new XElement("Division",
                                new XAttribute("name", "East"),
                                new XElement("Teams")
                            ),
                            new XElement("Division",
                                new XAttribute("name", "West"),
                                new XElement("Teams")
                            ),
                            new XElement("Division",
                                new XAttribute("name", "North"),
                                new XElement("Teams")
                            ),
                            new XElement("Division",
                                new XAttribute("name", "South"),
                                new XElement("Teams")
                            )
                        )
                    ),
                    new XElement("League",
                        new XAttribute("type", "Baseball"),
                        new XAttribute("name", "Major League Baseball"),
                        new XAttribute("id", "MLB"),
                        new XAttribute(XNamespace.Xmlns + "mlb", "http://www.mlb.com")
                    ),
                    new XElement("League",
                        new XAttribute("type", "Hockey"),
                        new XAttribute("name", "National Hockey League"),
                        new XAttribute("id", "NHL"),
                        new XAttribute(XNamespace.Xmlns + "nhl", "http://www.nhl.com")
                    )
                )
            );

            xmlDoc = XDocument.Parse(xDoc.ToString(SaveOptions.DisableFormatting));
        }

        public static void Hamlet(ref XDocument xDoc, ref XDocument xmlDoc)
        {
            xDoc = new XDocument(
                new XElement("PLAY",
                    new XElement("TITLE", "The Tragedy of Hamlet, Prince of Denmark"),
                    new XElement("fm",
                        new XElement("p", "Text placed in the public domain by Moby Lexical Tools, 1992."),
                        new XElement("p", "SGML markup by Jon Bosak, 1992-1994."),
                        new XElement("p", "XML version by Jon Bosak, 1996-1997."),
                        new XElement("p", "This work may be freely copied and distributed worldwide.")
                    ),
                    new XElement("PERSONAE",
                        new XElement("TITLE", "Dramatis Personae"),
                        new XElement("PERSONA", "CLAUDIUS, king of Denmark. "),
                        new XElement("PERSONA", "HAMLET, son to the late, and nephew to the present king."),
                        new XElement("PERSONA", "POLONIUS, lord chamberlain. "),
                        new XElement("PERSONA", "HORATIO, friend to Hamlet."),
                        new XElement("PERSONA", "LAERTES, son to Polonius."),
                        new XElement("PERSONA", "LUCIANUS, nephew to the king."),
                        new XElement("PGROUP",
                            new XElement("PERSONA", "VOLTIMAND"),
                            new XElement("PERSONA", "CORNELIUS"),
                            new XElement("PERSONA", "ROSENCRANTZ"),
                            new XElement("PERSONA", "GUILDENSTERN"),
                            new XElement("PERSONA", "OSRIC"),
                            new XElement("GRPDESCR", "courtiers.")
                        ),
                        new XElement("PERSONA", "A Gentleman"),
                        new XElement("PERSONA", "A Priest. "),
                        new XElement("PGROUP",
                            new XElement("PERSONA", "MARCELLUS"),
                            new XElement("PERSONA", "BERNARDO"),
                            new XElement("GRPDESCR", "officers.")
                        ),
                        new XElement("PERSONA", "FRANCISCO, a soldier."),
                        new XElement("PERSONA", "REYNALDO, servant to Polonius."),
                        new XElement("PERSONA", "Players."),
                        new XElement("PERSONA", "Two Clowns, grave-diggers."),
                        new XElement("PERSONA", "FORTINBRAS, prince of Norway. "),
                        new XElement("PERSONA", "A Captain."),
                        new XElement("PERSONA", "English Ambassadors. "),
                        new XElement("PERSONA", "GERTRUDE, queen of Denmark, and mother to Hamlet. "),
                        new XElement("PERSONA", "OPHELIA, daughter to Polonius."),
                        new XElement("PERSONA", "Lords, Ladies, Officers, Soldiers, Sailors, Messengers, and other Attendants."),
                        new XElement("PERSONA", "Ghost of Hamlet's Father. ")
                    ),
                    new XElement("SCNDESCR", "SCENE  Denmark."),
                    new XElement("PLAYSUBT", "HAMLET"),
                    new XElement("ACT",
                        new XElement("TITLE", "ACT I"),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE I.  Elsinore. A platform before the castle."),
                            new XElement("STAGEDIR", "FRANCISCO at his post. Enter to him BERNARDO"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "Who's there?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "FRANCISCO"),
                                new XElement("LINE", "Nay, answer me: stand, and unfold yourself.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "Long live the king!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "FRANCISCO"),
                                new XElement("LINE", "Bernardo?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "He.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "FRANCISCO"),
                                new XElement("LINE", "You come most carefully upon your hour.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "'Tis now struck twelve; get thee to bed, Francisco.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "FRANCISCO"),
                                new XElement("LINE", "For this relief much thanks: 'tis bitter cold,"),
                                new XElement("LINE", "And I am sick at heart.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "Have you had quiet guard?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "FRANCISCO"),
                                new XElement("LINE", "Not a mouse stirring.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "Well, good night."),
                                new XElement("LINE", "If you do meet Horatio and Marcellus,"),
                                new XElement("LINE", "The rivals of my watch, bid them make haste.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "FRANCISCO"),
                                new XElement("LINE", "I think I hear them. Stand, ho! Who's there?")
                            ),
                            new XElement("STAGEDIR", "Enter HORATIO and MARCELLUS"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Friends to this ground.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "And liegemen to the Dane.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "FRANCISCO"),
                                new XElement("LINE", "Give you good night.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "O, farewell, honest soldier:"),
                                new XElement("LINE", "Who hath relieved you?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "FRANCISCO"),
                                new XElement("LINE", "Bernardo has my place."),
                                new XElement("LINE", "Give you good night.")
                            ),
                            new XElement("STAGEDIR", "Exit"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "Holla! Bernardo!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "Say,"),
                                new XElement("LINE", "What, is Horatio there?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "A piece of him.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "Welcome, Horatio: welcome, good Marcellus.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "What, has this thing appear'd again to-night?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "I have seen nothing.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "Horatio says 'tis but our fantasy,"),
                                new XElement("LINE", "And will not let belief take hold of him"),
                                new XElement("LINE", "Touching this dreaded sight, twice seen of us:"),
                                new XElement("LINE", "Therefore I have entreated him along"),
                                new XElement("LINE", "With us to watch the minutes of this night;"),
                                new XElement("LINE", "That if again this apparition come,"),
                                new XElement("LINE", "He may approve our eyes and speak to it.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Tush, tush, 'twill not appear.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "Sit down awhile;"),
                                new XElement("LINE", "And let us once again assail your ears,"),
                                new XElement("LINE", "That are so fortified against our story"),
                                new XElement("LINE", "What we have two nights seen.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Well, sit we down,"),
                                new XElement("LINE", "And let us hear Bernardo speak of this.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "Last night of all,"),
                                new XElement("LINE", "When yond same star that's westward from the pole"),
                                new XElement("LINE", "Had made his course to illume that part of heaven"),
                                new XElement("LINE", "Where now it burns, Marcellus and myself,"),
                                new XElement("LINE", "The bell then beating one,--")
                            ),
                            new XElement("STAGEDIR", "Enter Ghost"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "Peace, break thee off; look, where it comes again!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "In the same figure, like the king that's dead.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "Thou art a scholar; speak to it, Horatio.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "Looks it not like the king?  mark it, Horatio.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Most like: it harrows me with fear and wonder.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "It would be spoke to.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "Question it, Horatio.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "What art thou that usurp'st this time of night,"),
                                new XElement("LINE", "Together with that fair and warlike form"),
                                new XElement("LINE", "In which the majesty of buried Denmark"),
                                new XElement("LINE", "Did sometimes march? by heaven I charge thee, speak!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "It is offended.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "See, it stalks away!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Stay! speak, speak! I charge thee, speak!")
                            ),
                            new XElement("STAGEDIR", "Exit Ghost"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "'Tis gone, and will not answer.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "How now, Horatio! you tremble and look pale:"),
                                new XElement("LINE", "Is not this something more than fantasy?"),
                                new XElement("LINE", "What think you on't?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Before my God, I might not this believe"),
                                new XElement("LINE", "Without the sensible and true avouch"),
                                new XElement("LINE", "Of mine own eyes.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "Is it not like the king?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "As thou art to thyself:"),
                                new XElement("LINE", "Such was the very armour he had on"),
                                new XElement("LINE", "When he the ambitious Norway combated;"),
                                new XElement("LINE", "So frown'd he once, when, in an angry parle,"),
                                new XElement("LINE", "He smote the sledded Polacks on the ice."),
                                new XElement("LINE", "'Tis strange.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "Thus twice before, and jump at this dead hour,"),
                                new XElement("LINE", "With martial stalk hath he gone by our watch.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "In what particular thought to work I know not;"),
                                new XElement("LINE", "But in the gross and scope of my opinion,"),
                                new XElement("LINE", "This bodes some strange eruption to our state.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "Good now, sit down, and tell me, he that knows,"),
                                new XElement("LINE", "Why this same strict and most observant watch"),
                                new XElement("LINE", "So nightly toils the subject of the land,"),
                                new XElement("LINE", "And why such daily cast of brazen cannon,"),
                                new XElement("LINE", "And foreign mart for implements of war;"),
                                new XElement("LINE", "Why such impress of shipwrights, whose sore task"),
                                new XElement("LINE", "Does not divide the Sunday from the week;"),
                                new XElement("LINE", "What might be toward, that this sweaty haste"),
                                new XElement("LINE", "Doth make the night joint-labourer with the day:"),
                                new XElement("LINE", "Who is't that can inform me?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "That can I;"),
                                new XElement("LINE", "At least, the whisper goes so. Our last king,"),
                                new XElement("LINE", "Whose image even but now appear'd to us,"),
                                new XElement("LINE", "Was, as you know, by Fortinbras of Norway,"),
                                new XElement("LINE", "Thereto prick'd on by a most emulate pride,"),
                                new XElement("LINE", "Dared to the combat; in which our valiant Hamlet--"),
                                new XElement("LINE", "For so this side of our known world esteem'd him--"),
                                new XElement("LINE", "Did slay this Fortinbras; who by a seal'd compact,"),
                                new XElement("LINE", "Well ratified by law and heraldry,"),
                                new XElement("LINE", "Did forfeit, with his life, all those his lands"),
                                new XElement("LINE", "Which he stood seized of, to the conqueror:"),
                                new XElement("LINE", "Against the which, a moiety competent"),
                                new XElement("LINE", "Was gaged by our king; which had return'd"),
                                new XElement("LINE", "To the inheritance of Fortinbras,"),
                                new XElement("LINE", "Had he been vanquisher; as, by the same covenant,"),
                                new XElement("LINE", "And carriage of the article design'd,"),
                                new XElement("LINE", "His fell to Hamlet. Now, sir, young Fortinbras,"),
                                new XElement("LINE", "Of unimproved mettle hot and full,"),
                                new XElement("LINE", "Hath in the skirts of Norway here and there"),
                                new XElement("LINE", "Shark'd up a list of lawless resolutes,"),
                                new XElement("LINE", "For food and diet, to some enterprise"),
                                new XElement("LINE", "That hath a stomach in't; which is no other--"),
                                new XElement("LINE", "As it doth well appear unto our state--"),
                                new XElement("LINE", "But to recover of us, by strong hand"),
                                new XElement("LINE", "And terms compulsatory, those foresaid lands"),
                                new XElement("LINE", "So by his father lost: and this, I take it,"),
                                new XElement("LINE", "Is the main motive of our preparations,"),
                                new XElement("LINE", "The source of this our watch and the chief head"),
                                new XElement("LINE", "Of this post-haste and romage in the land.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "I think it be no other but e'en so:"),
                                new XElement("LINE", "Well may it sort that this portentous figure"),
                                new XElement("LINE", "Comes armed through our watch; so like the king"),
                                new XElement("LINE", "That was and is the question of these wars.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "A mote it is to trouble the mind's eye."),
                                new XElement("LINE", "In the most high and palmy state of Rome,"),
                                new XElement("LINE", "A little ere the mightiest Julius fell,"),
                                new XElement("LINE", "The graves stood tenantless and the sheeted dead"),
                                new XElement("LINE", "Did squeak and gibber in the Roman streets:"),
                                new XElement("LINE", "As stars with trains of fire and dews of blood,"),
                                new XElement("LINE", "Disasters in the sun; and the moist star"),
                                new XElement("LINE", "Upon whose influence Neptune's empire stands"),
                                new XElement("LINE", "Was sick almost to doomsday with eclipse:"),
                                new XElement("LINE", "And even the like precurse of fierce events,"),
                                new XElement("LINE", "As harbingers preceding still the fates"),
                                new XElement("LINE", "And prologue to the omen coming on,"),
                                new XElement("LINE", "Have heaven and earth together demonstrated"),
                                new XElement("LINE", "Unto our climatures and countrymen.--"),
                                new XElement("LINE", "But soft, behold! lo, where it comes again!"),
                                new XElement("STAGEDIR", "Re-enter Ghost"),
                                new XElement("LINE", "I'll cross it, though it blast me. Stay, illusion!"),
                                new XElement("LINE", "If thou hast any sound, or use of voice,"),
                                new XElement("LINE", "Speak to me:"),
                                new XElement("LINE", "If there be any good thing to be done,"),
                                new XElement("LINE", "That may to thee do ease and grace to me,"),
                                new XElement("LINE", "Speak to me:"),
                                new XElement("STAGEDIR", "Cock crows"),
                                new XElement("LINE", "If thou art privy to thy country's fate,"),
                                new XElement("LINE", "Which, happily, foreknowing may avoid, O, speak!"),
                                new XElement("LINE", "Or if thou hast uphoarded in thy life"),
                                new XElement("LINE", "Extorted treasure in the womb of earth,"),
                                new XElement("LINE", "For which, they say, you spirits oft walk in death,"),
                                new XElement("LINE", "Speak of it: stay, and speak! Stop it, Marcellus.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "Shall I strike at it with my partisan?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Do, if it will not stand.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "'Tis here!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "'Tis here!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "'Tis gone!"),
                                new XElement("STAGEDIR", "Exit Ghost"),
                                new XElement("LINE", "We do it wrong, being so majestical,"),
                                new XElement("LINE", "To offer it the show of violence;"),
                                new XElement("LINE", "For it is, as the air, invulnerable,"),
                                new XElement("LINE", "And our vain blows malicious mockery.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "It was about to speak, when the cock crew.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "And then it started like a guilty thing"),
                                new XElement("LINE", "Upon a fearful summons. I have heard,"),
                                new XElement("LINE", "The cock, that is the trumpet to the morn,"),
                                new XElement("LINE", "Doth with his lofty and shrill-sounding throat"),
                                new XElement("LINE", "Awake the god of day; and, at his warning,"),
                                new XElement("LINE", "Whether in sea or fire, in earth or air,"),
                                new XElement("LINE", "The extravagant and erring spirit hies"),
                                new XElement("LINE", "To his confine: and of the truth herein"),
                                new XElement("LINE", "This present object made probation.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "It faded on the crowing of the cock."),
                                new XElement("LINE", "Some say that ever 'gainst that season comes"),
                                new XElement("LINE", "Wherein our Saviour's birth is celebrated,"),
                                new XElement("LINE", "The bird of dawning singeth all night long:"),
                                new XElement("LINE", "And then, they say, no spirit dares stir abroad;"),
                                new XElement("LINE", "The nights are wholesome; then no planets strike,"),
                                new XElement("LINE", "No fairy takes, nor witch hath power to charm,"),
                                new XElement("LINE", "So hallow'd and so gracious is the time.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "So have I heard and do in part believe it."),
                                new XElement("LINE", "But, look, the morn, in russet mantle clad,"),
                                new XElement("LINE", "Walks o'er the dew of yon high eastward hill:"),
                                new XElement("LINE", "Break we our watch up; and by my advice,"),
                                new XElement("LINE", "Let us impart what we have seen to-night"),
                                new XElement("LINE", "Unto young Hamlet; for, upon my life,"),
                                new XElement("LINE", "This spirit, dumb to us, will speak to him."),
                                new XElement("LINE", "Do you consent we shall acquaint him with it,"),
                                new XElement("LINE", "As needful in our loves, fitting our duty?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "Let's do't, I pray; and I this morning know"),
                                new XElement("LINE", "Where we shall find him most conveniently.")
                            ),
                            new XElement("STAGEDIR", "Exeunt")
                        ),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE II.  A room of state in the castle."),
                            new XElement("STAGEDIR",
                                "Enter KING CLAUDIUS, QUEEN GERTRUDE, HAMLET,\n" +
                                "POLONIUS, LAERTES, VOLTIMAND, CORNELIUS, Lords,\n" +
                                "and Attendants"
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Though yet of Hamlet our dear brother's death"),
                                new XElement("LINE", "The memory be green, and that it us befitted"),
                                new XElement("LINE", "To bear our hearts in grief and our whole kingdom"),
                                new XElement("LINE", "To be contracted in one brow of woe,"),
                                new XElement("LINE", "Yet so far hath discretion fought with nature"),
                                new XElement("LINE", "That we with wisest sorrow think on him,"),
                                new XElement("LINE", "Together with remembrance of ourselves."),
                                new XElement("LINE", "Therefore our sometime sister, now our queen,"),
                                new XElement("LINE", "The imperial jointress to this warlike state,"),
                                new XElement("LINE", "Have we, as 'twere with a defeated joy,--"),
                                new XElement("LINE", "With an auspicious and a dropping eye,"),
                                new XElement("LINE", "With mirth in funeral and with dirge in marriage,"),
                                new XElement("LINE", "In equal scale weighing delight and dole,--"),
                                new XElement("LINE", "Taken to wife: nor have we herein barr'd"),
                                new XElement("LINE", "Your better wisdoms, which have freely gone"),
                                new XElement("LINE", "With this affair along. For all, our thanks."),
                                new XElement("LINE", "Now follows, that you know, young Fortinbras,"),
                                new XElement("LINE", "Holding a weak supposal of our worth,"),
                                new XElement("LINE", "Or thinking by our late dear brother's death"),
                                new XElement("LINE", "Our state to be disjoint and out of frame,"),
                                new XElement("LINE", "Colleagued with the dream of his advantage,"),
                                new XElement("LINE", "He hath not fail'd to pester us with message,"),
                                new XElement("LINE", "Importing the surrender of those lands"),
                                new XElement("LINE", "Lost by his father, with all bonds of law,"),
                                new XElement("LINE", "To our most valiant brother. So much for him."),
                                new XElement("LINE", "Now for ourself and for this time of meeting:"),
                                new XElement("LINE", "Thus much the business is: we have here writ"),
                                new XElement("LINE", "To Norway, uncle of young Fortinbras,--"),
                                new XElement("LINE", "Who, impotent and bed-rid, scarcely hears"),
                                new XElement("LINE", "Of this his nephew's purpose,--to suppress"),
                                new XElement("LINE", "His further gait herein; in that the levies,"),
                                new XElement("LINE", "The lists and full proportions, are all made"),
                                new XElement("LINE", "Out of his subject: and we here dispatch"),
                                new XElement("LINE", "You, good Cornelius, and you, Voltimand,"),
                                new XElement("LINE", "For bearers of this greeting to old Norway;"),
                                new XElement("LINE", "Giving to you no further personal power"),
                                new XElement("LINE", "To business with the king, more than the scope"),
                                new XElement("LINE", "Of these delated articles allow."),
                                new XElement("LINE", "Farewell, and let your haste commend your duty.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "CORNELIUS"),
                                new XElement("SPEAKER", "VOLTIMAND"),
                                new XElement("LINE", "In that and all things will we show our duty.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "We doubt it nothing: heartily farewell."),
                                new XElement("STAGEDIR", "Exeunt VOLTIMAND and CORNELIUS"),
                                new XElement("LINE", "And now, Laertes, what's the news with you?"),
                                new XElement("LINE", "You told us of some suit; what is't, Laertes?"),
                                new XElement("LINE", "You cannot speak of reason to the Dane,"),
                                new XElement("LINE", "And loose your voice: what wouldst thou beg, Laertes,"),
                                new XElement("LINE", "That shall not be my offer, not thy asking?"),
                                new XElement("LINE", "The head is not more native to the heart,"),
                                new XElement("LINE", "The hand more instrumental to the mouth,"),
                                new XElement("LINE", "Than is the throne of Denmark to thy father."),
                                new XElement("LINE", "What wouldst thou have, Laertes?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "My dread lord,"),
                                new XElement("LINE", "Your leave and favour to return to France;"),
                                new XElement("LINE", "From whence though willingly I came to Denmark,"),
                                new XElement("LINE", "To show my duty in your coronation,"),
                                new XElement("LINE", "Yet now, I must confess, that duty done,"),
                                new XElement("LINE", "My thoughts and wishes bend again toward France"),
                                new XElement("LINE", "And bow them to your gracious leave and pardon.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Have you your father's leave? What says Polonius?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "He hath, my lord, wrung from me my slow leave"),
                                new XElement("LINE", "By laboursome petition, and at last"),
                                new XElement("LINE", "Upon his will I seal'd my hard consent:"),
                                new XElement("LINE", "I do beseech you, give him leave to go.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Take thy fair hour, Laertes; time be thine,"),
                                new XElement("LINE", "And thy best graces spend it at thy will!"),
                                new XElement("LINE", "But now, my cousin Hamlet, and my son,--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Aside"),
                                    "  A little more than kin, and less than kind."
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "How is it that the clouds still hang on you?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Not so, my lord; I am too much i' the sun.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Good Hamlet, cast thy nighted colour off,"),
                                new XElement("LINE", "And let thine eye look like a friend on Denmark."),
                                new XElement("LINE", "Do not for ever with thy vailed lids"),
                                new XElement("LINE", "Seek for thy noble father in the dust:"),
                                new XElement("LINE", "Thou know'st 'tis common; all that lives must die,"),
                                new XElement("LINE", "Passing through nature to eternity.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Ay, madam, it is common.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "If it be,"),
                                new XElement("LINE", "Why seems it so particular with thee?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Seems, madam! nay it is; I know not 'seems.'"),
                                new XElement("LINE", "'Tis not alone my inky cloak, good mother,"),
                                new XElement("LINE", "Nor customary suits of solemn black,"),
                                new XElement("LINE", "Nor windy suspiration of forced breath,"),
                                new XElement("LINE", "No, nor the fruitful river in the eye,"),
                                new XElement("LINE", "Nor the dejected 'havior of the visage,"),
                                new XElement("LINE", "Together with all forms, moods, shapes of grief,"),
                                new XElement("LINE", "That can denote me truly: these indeed seem,"),
                                new XElement("LINE", "For they are actions that a man might play:"),
                                new XElement("LINE", "But I have that within which passeth show;"),
                                new XElement("LINE", "These but the trappings and the suits of woe.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "'Tis sweet and commendable in your nature, Hamlet,"),
                                new XElement("LINE", "To give these mourning duties to your father:"),
                                new XElement("LINE", "But, you must know, your father lost a father;"),
                                new XElement("LINE", "That father lost, lost his, and the survivor bound"),
                                new XElement("LINE", "In filial obligation for some term"),
                                new XElement("LINE", "To do obsequious sorrow: but to persever"),
                                new XElement("LINE", "In obstinate condolement is a course"),
                                new XElement("LINE", "Of impious stubbornness; 'tis unmanly grief;"),
                                new XElement("LINE", "It shows a will most incorrect to heaven,"),
                                new XElement("LINE", "A heart unfortified, a mind impatient,"),
                                new XElement("LINE", "An understanding simple and unschool'd:"),
                                new XElement("LINE", "For what we know must be and is as common"),
                                new XElement("LINE", "As any the most vulgar thing to sense,"),
                                new XElement("LINE", "Why should we in our peevish opposition"),
                                new XElement("LINE", "Take it to heart? Fie! 'tis a fault to heaven,"),
                                new XElement("LINE", "A fault against the dead, a fault to nature,"),
                                new XElement("LINE", "To reason most absurd: whose common theme"),
                                new XElement("LINE", "Is death of fathers, and who still hath cried,"),
                                new XElement("LINE", "From the first corse till he that died to-day,"),
                                new XElement("LINE", "'This must be so.' We pray you, throw to earth"),
                                new XElement("LINE", "This unprevailing woe, and think of us"),
                                new XElement("LINE", "As of a father: for let the world take note,"),
                                new XElement("LINE", "You are the most immediate to our throne;"),
                                new XElement("LINE", "And with no less nobility of love"),
                                new XElement("LINE", "Than that which dearest father bears his son,"),
                                new XElement("LINE", "Do I impart toward you. For your intent"),
                                new XElement("LINE", "In going back to school in Wittenberg,"),
                                new XElement("LINE", "It is most retrograde to our desire:"),
                                new XElement("LINE", "And we beseech you, bend you to remain"),
                                new XElement("LINE", "Here, in the cheer and comfort of our eye,"),
                                new XElement("LINE", "Our chiefest courtier, cousin, and our son.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Let not thy mother lose her prayers, Hamlet:"),
                                new XElement("LINE", "I pray thee, stay with us; go not to Wittenberg.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I shall in all my best obey you, madam.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Why, 'tis a loving and a fair reply:"),
                                new XElement("LINE", "Be as ourself in Denmark. Madam, come;"),
                                new XElement("LINE", "This gentle and unforced accord of Hamlet"),
                                new XElement("LINE", "Sits smiling to my heart: in grace whereof,"),
                                new XElement("LINE", "No jocund health that Denmark drinks to-day,"),
                                new XElement("LINE", "But the great cannon to the clouds shall tell,"),
                                new XElement("LINE", "And the king's rouse the heavens all bruit again,"),
                                new XElement("LINE", "Re-speaking earthly thunder. Come away.")
                            ),
                            new XElement("STAGEDIR", "Exeunt all but HAMLET"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "O, that this too too solid flesh would melt"),
                                new XElement("LINE", "Thaw and resolve itself into a dew!"),
                                new XElement("LINE", "Or that the Everlasting had not fix'd"),
                                new XElement("LINE", "His canon 'gainst self-slaughter! O God! God!"),
                                new XElement("LINE", "How weary, stale, flat and unprofitable,"),
                                new XElement("LINE", "Seem to me all the uses of this world!"),
                                new XElement("LINE", "Fie on't! ah fie! 'tis an unweeded garden,"),
                                new XElement("LINE", "That grows to seed; things rank and gross in nature"),
                                new XElement("LINE", "Possess it merely. That it should come to this!"),
                                new XElement("LINE", "But two months dead: nay, not so much, not two:"),
                                new XElement("LINE", "So excellent a king; that was, to this,"),
                                new XElement("LINE", "Hyperion to a satyr; so loving to my mother"),
                                new XElement("LINE", "That he might not beteem the winds of heaven"),
                                new XElement("LINE", "Visit her face too roughly. Heaven and earth!"),
                                new XElement("LINE", "Must I remember? why, she would hang on him,"),
                                new XElement("LINE", "As if increase of appetite had grown"),
                                new XElement("LINE", "By what it fed on: and yet, within a month--"),
                                new XElement("LINE", "Let me not think on't--Frailty, thy name is woman!--"),
                                new XElement("LINE", "A little month, or ere those shoes were old"),
                                new XElement("LINE", "With which she follow'd my poor father's body,"),
                                new XElement("LINE", "Like Niobe, all tears:--why she, even she--"),
                                new XElement("LINE", "O, God! a beast, that wants discourse of reason,"),
                                new XElement("LINE", "Would have mourn'd longer--married with my uncle,"),
                                new XElement("LINE", "My father's brother, but no more like my father"),
                                new XElement("LINE", "Than I to Hercules: within a month:"),
                                new XElement("LINE", "Ere yet the salt of most unrighteous tears"),
                                new XElement("LINE", "Had left the flushing in her galled eyes,"),
                                new XElement("LINE", "She married. O, most wicked speed, to post"),
                                new XElement("LINE", "With such dexterity to incestuous sheets!"),
                                new XElement("LINE", "It is not nor it cannot come to good:"),
                                new XElement("LINE", "But break, my heart; for I must hold my tongue.")
                            ),
                            new XElement("STAGEDIR", "Enter HORATIO, MARCELLUS, and BERNARDO"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Hail to your lordship!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I am glad to see you well:"),
                                new XElement("LINE", "Horatio,--or I do forget myself.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "The same, my lord, and your poor servant ever.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Sir, my good friend; I'll change that name with you:"),
                                new XElement("LINE", "And what make you from Wittenberg, Horatio? Marcellus?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "My good lord--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I am very glad to see you. Good even, sir."),
                                new XElement("LINE", "But what, in faith, make you from Wittenberg?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "A truant disposition, good my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I would not hear your enemy say so,"),
                                new XElement("LINE", "Nor shall you do mine ear that violence,"),
                                new XElement("LINE", "To make it truster of your own report"),
                                new XElement("LINE", "Against yourself: I know you are no truant."),
                                new XElement("LINE", "But what is your affair in Elsinore?"),
                                new XElement("LINE", "We'll teach you to drink deep ere you depart.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "My lord, I came to see your father's funeral.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I pray thee, do not mock me, fellow-student;"),
                                new XElement("LINE", "I think it was to see my mother's wedding.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Indeed, my lord, it follow'd hard upon.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Thrift, thrift, Horatio! the funeral baked meats"),
                                new XElement("LINE", "Did coldly furnish forth the marriage tables."),
                                new XElement("LINE", "Would I had met my dearest foe in heaven"),
                                new XElement("LINE", "Or ever I had seen that day, Horatio!"),
                                new XElement("LINE", "My father!--methinks I see my father.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Where, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "In my mind's eye, Horatio.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "I saw him once; he was a goodly king.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "He was a man, take him for all in all,"),
                                new XElement("LINE", "I shall not look upon his like again.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "My lord, I think I saw him yesternight.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Saw? who?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "My lord, the king your father.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "The king my father!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Season your admiration for awhile"),
                                new XElement("LINE", "With an attent ear, till I may deliver,"),
                                new XElement("LINE", "Upon the witness of these gentlemen,"),
                                new XElement("LINE", "This marvel to you.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "For God's love, let me hear.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Two nights together had these gentlemen,"),
                                new XElement("LINE", "Marcellus and Bernardo, on their watch,"),
                                new XElement("LINE", "In the dead vast and middle of the night,"),
                                new XElement("LINE", "Been thus encounter'd. A figure like your father,"),
                                new XElement("LINE", "Armed at point exactly, cap-a-pe,"),
                                new XElement("LINE", "Appears before them, and with solemn march"),
                                new XElement("LINE", "Goes slow and stately by them: thrice he walk'd"),
                                new XElement("LINE", "By their oppress'd and fear-surprised eyes,"),
                                new XElement("LINE", "Within his truncheon's length; whilst they, distilled"),
                                new XElement("LINE", "Almost to jelly with the act of fear,"),
                                new XElement("LINE", "Stand dumb and speak not to him. This to me"),
                                new XElement("LINE", "In dreadful secrecy impart they did;"),
                                new XElement("LINE", "And I with them the third night kept the watch;"),
                                new XElement("LINE", "Where, as they had deliver'd, both in time,"),
                                new XElement("LINE", "Form of the thing, each word made true and good,"),
                                new XElement("LINE", "The apparition comes: I knew your father;"),
                                new XElement("LINE", "These hands are not more like.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "But where was this?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "My lord, upon the platform where we watch'd.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Did you not speak to it?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "My lord, I did;"),
                                new XElement("LINE", "But answer made it none: yet once methought"),
                                new XElement("LINE", "It lifted up its head and did address"),
                                new XElement("LINE", "Itself to motion, like as it would speak;"),
                                new XElement("LINE", "But even then the morning cock crew loud,"),
                                new XElement("LINE", "And at the sound it shrunk in haste away,"),
                                new XElement("LINE", "And vanish'd from our sight.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "'Tis very strange.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "As I do live, my honour'd lord, 'tis true;"),
                                new XElement("LINE", "And we did think it writ down in our duty"),
                                new XElement("LINE", "To let you know of it.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Indeed, indeed, sirs, but this troubles me."),
                                new XElement("LINE", "Hold you the watch to-night?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "We do, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Arm'd, say you?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "Arm'd, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "From top to toe?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "My lord, from head to foot.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Then saw you not his face?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "O, yes, my lord; he wore his beaver up.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "What, look'd he frowningly?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "A countenance more in sorrow than in anger.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Pale or red?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Nay, very pale.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "And fix'd his eyes upon you?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Most constantly.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I would I had been there.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "It would have much amazed you.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Very like, very like. Stay'd it long?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "While one with moderate haste might tell a hundred.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("SPEAKER", "BERNARDO"),
                                new XElement("LINE", "Longer, longer.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Not when I saw't.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "His beard was grizzled--no?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "It was, as I have seen it in his life,"),
                                new XElement("LINE", "A sable silver'd.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I will watch to-night;"),
                                new XElement("LINE", "Perchance 'twill walk again.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "I warrant it will.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "If it assume my noble father's person,"),
                                new XElement("LINE", "I'll speak to it, though hell itself should gape"),
                                new XElement("LINE", "And bid me hold my peace. I pray you all,"),
                                new XElement("LINE", "If you have hitherto conceal'd this sight,"),
                                new XElement("LINE", "Let it be tenable in your silence still;"),
                                new XElement("LINE", "And whatsoever else shall hap to-night,"),
                                new XElement("LINE", "Give it an understanding, but no tongue:"),
                                new XElement("LINE", "I will requite your loves. So, fare you well:"),
                                new XElement("LINE", "Upon the platform, 'twixt eleven and twelve,"),
                                new XElement("LINE", "I'll visit you.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "All"),
                                new XElement("LINE", "Our duty to your honour.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Your loves, as mine to you: farewell."),
                                new XElement("STAGEDIR", "Exeunt all but HAMLET"),
                                new XElement("LINE", "My father's spirit in arms! all is not well;"),
                                new XElement("LINE", "I doubt some foul play: would the night were come!"),
                                new XElement("LINE", "Till then sit still, my soul: foul deeds will rise,"),
                                new XElement("LINE", "Though all the earth o'erwhelm them, to men's eyes.")
                            ),
                            new XElement("STAGEDIR", "Exit")
                        ),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE III.  A room in Polonius' house."),
                            new XElement("STAGEDIR", "Enter LAERTES and OPHELIA"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "My necessaries are embark'd: farewell:"),
                                new XElement("LINE", "And, sister, as the winds give benefit"),
                                new XElement("LINE", "And convoy is assistant, do not sleep,"),
                                new XElement("LINE", "But let me hear from you.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "Do you doubt that?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "For Hamlet and the trifling of his favour,"),
                                new XElement("LINE", "Hold it a fashion and a toy in blood,"),
                                new XElement("LINE", "A violet in the youth of primy nature,"),
                                new XElement("LINE", "Forward, not permanent, sweet, not lasting,"),
                                new XElement("LINE", "The perfume and suppliance of a minute; No more.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "No more but so?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Think it no more;"),
                                new XElement("LINE", "For nature, crescent, does not grow alone"),
                                new XElement("LINE", "In thews and bulk, but, as this temple waxes,"),
                                new XElement("LINE", "The inward service of the mind and soul"),
                                new XElement("LINE", "Grows wide withal. Perhaps he loves you now,"),
                                new XElement("LINE", "And now no soil nor cautel doth besmirch"),
                                new XElement("LINE", "The virtue of his will: but you must fear,"),
                                new XElement("LINE", "His greatness weigh'd, his will is not his own;"),
                                new XElement("LINE", "For he himself is subject to his birth:"),
                                new XElement("LINE", "He may not, as unvalued persons do,"),
                                new XElement("LINE", "Carve for himself; for on his choice depends"),
                                new XElement("LINE", "The safety and health of this whole state;"),
                                new XElement("LINE", "And therefore must his choice be circumscribed"),
                                new XElement("LINE", "Unto the voice and yielding of that body"),
                                new XElement("LINE", "Whereof he is the head. Then if he says he loves you,"),
                                new XElement("LINE", "It fits your wisdom so far to believe it"),
                                new XElement("LINE", "As he in his particular act and place"),
                                new XElement("LINE", "May give his saying deed; which is no further"),
                                new XElement("LINE", "Than the main voice of Denmark goes withal."),
                                new XElement("LINE", "Then weigh what loss your honour may sustain,"),
                                new XElement("LINE", "If with too credent ear you list his songs,"),
                                new XElement("LINE", "Or lose your heart, or your chaste treasure open"),
                                new XElement("LINE", "To his unmaster'd importunity."),
                                new XElement("LINE", "Fear it, Ophelia, fear it, my dear sister,"),
                                new XElement("LINE", "And keep you in the rear of your affection,"),
                                new XElement("LINE", "Out of the shot and danger of desire."),
                                new XElement("LINE", "The chariest maid is prodigal enough,"),
                                new XElement("LINE", "If she unmask her beauty to the moon:"),
                                new XElement("LINE", "Virtue itself 'scapes not calumnious strokes:"),
                                new XElement("LINE", "The canker galls the infants of the spring,"),
                                new XElement("LINE", "Too oft before their buttons be disclosed,"),
                                new XElement("LINE", "And in the morn and liquid dew of youth"),
                                new XElement("LINE", "Contagious blastments are most imminent."),
                                new XElement("LINE", "Be wary then; best safety lies in fear:"),
                                new XElement("LINE", "Youth to itself rebels, though none else near.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "I shall the effect of this good lesson keep,"),
                                new XElement("LINE", "As watchman to my heart. But, good my brother,"),
                                new XElement("LINE", "Do not, as some ungracious pastors do,"),
                                new XElement("LINE", "Show me the steep and thorny way to heaven;"),
                                new XElement("LINE", "Whiles, like a puff'd and reckless libertine,"),
                                new XElement("LINE", "Himself the primrose path of dalliance treads,"),
                                new XElement("LINE", "And recks not his own rede.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "O, fear me not."),
                                new XElement("LINE", "I stay too long: but here my father comes."),
                                new XElement("STAGEDIR", "Enter POLONIUS"),
                                new XElement("LINE", "A double blessing is a double grace,"),
                                new XElement("LINE", "Occasion smiles upon a second leave.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Yet here, Laertes! aboard, aboard, for shame!"),
                                new XElement("LINE", "The wind sits in the shoulder of your sail,"),
                                new XElement("LINE", "And you are stay'd for. There; my blessing with thee!"),
                                new XElement("LINE", "And these few precepts in thy memory"),
                                new XElement("LINE", "See thou character. Give thy thoughts no tongue,"),
                                new XElement("LINE", "Nor any unproportioned thought his act."),
                                new XElement("LINE", "Be thou familiar, but by no means vulgar."),
                                new XElement("LINE", "Those friends thou hast, and their adoption tried,"),
                                new XElement("LINE", "Grapple them to thy soul with hoops of steel;"),
                                new XElement("LINE", "But do not dull thy palm with entertainment"),
                                new XElement("LINE", "Of each new-hatch'd, unfledged comrade. Beware"),
                                new XElement("LINE", "Of entrance to a quarrel, but being in,"),
                                new XElement("LINE", "Bear't that the opposed may beware of thee."),
                                new XElement("LINE", "Give every man thy ear, but few thy voice;"),
                                new XElement("LINE", "Take each man's censure, but reserve thy judgment."),
                                new XElement("LINE", "Costly thy habit as thy purse can buy,"),
                                new XElement("LINE", "But not express'd in fancy; rich, not gaudy;"),
                                new XElement("LINE", "For the apparel oft proclaims the man,"),
                                new XElement("LINE", "And they in France of the best rank and station"),
                                new XElement("LINE", "Are of a most select and generous chief in that."),
                                new XElement("LINE", "Neither a borrower nor a lender be;"),
                                new XElement("LINE", "For loan oft loses both itself and friend,"),
                                new XElement("LINE", "And borrowing dulls the edge of husbandry."),
                                new XElement("LINE", "This above all: to thine ownself be true,"),
                                new XElement("LINE", "And it must follow, as the night the day,"),
                                new XElement("LINE", "Thou canst not then be false to any man."),
                                new XElement("LINE", "Farewell: my blessing season this in thee!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Most humbly do I take my leave, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "The time invites you; go; your servants tend.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Farewell, Ophelia; and remember well"),
                                new XElement("LINE", "What I have said to you.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "'Tis in my memory lock'd,"),
                                new XElement("LINE", "And you yourself shall keep the key of it.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Farewell.")
                            ),
                            new XElement("STAGEDIR", "Exit"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "What is't, Ophelia, be hath said to you?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "So please you, something touching the Lord Hamlet.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Marry, well bethought:"),
                                new XElement("LINE", "'Tis told me, he hath very oft of late"),
                                new XElement("LINE", "Given private time to you; and you yourself"),
                                new XElement("LINE", "Have of your audience been most free and bounteous:"),
                                new XElement("LINE", "If it be so, as so 'tis put on me,"),
                                new XElement("LINE", "And that in way of caution, I must tell you,"),
                                new XElement("LINE", "You do not understand yourself so clearly"),
                                new XElement("LINE", "As it behoves my daughter and your honour."),
                                new XElement("LINE", "What is between you? give me up the truth.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "He hath, my lord, of late made many tenders"),
                                new XElement("LINE", "Of his affection to me.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Affection! pooh! you speak like a green girl,"),
                                new XElement("LINE", "Unsifted in such perilous circumstance."),
                                new XElement("LINE", "Do you believe his tenders, as you call them?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "I do not know, my lord, what I should think.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Marry, I'll teach you: think yourself a baby;"),
                                new XElement("LINE", "That you have ta'en these tenders for true pay,"),
                                new XElement("LINE", "Which are not sterling. Tender yourself more dearly;"),
                                new XElement("LINE", "Or--not to crack the wind of the poor phrase,"),
                                new XElement("LINE", "Running it thus--you'll tender me a fool.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "My lord, he hath importuned me with love"),
                                new XElement("LINE", "In honourable fashion.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Ay, fashion you may call it; go to, go to.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "And hath given countenance to his speech, my lord,"),
                                new XElement("LINE", "With almost all the holy vows of heaven.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Ay, springes to catch woodcocks. I do know,"),
                                new XElement("LINE", "When the blood burns, how prodigal the soul"),
                                new XElement("LINE", "Lends the tongue vows: these blazes, daughter,"),
                                new XElement("LINE", "Giving more light than heat, extinct in both,"),
                                new XElement("LINE", "Even in their promise, as it is a-making,"),
                                new XElement("LINE", "You must not take for fire. From this time"),
                                new XElement("LINE", "Be somewhat scanter of your maiden presence;"),
                                new XElement("LINE", "Set your entreatments at a higher rate"),
                                new XElement("LINE", "Than a command to parley. For Lord Hamlet,"),
                                new XElement("LINE", "Believe so much in him, that he is young"),
                                new XElement("LINE", "And with a larger tether may he walk"),
                                new XElement("LINE", "Than may be given you: in few, Ophelia,"),
                                new XElement("LINE", "Do not believe his vows; for they are brokers,"),
                                new XElement("LINE", "Not of that dye which their investments show,"),
                                new XElement("LINE", "But mere implorators of unholy suits,"),
                                new XElement("LINE", "Breathing like sanctified and pious bawds,"),
                                new XElement("LINE", "The better to beguile. This is for all:"),
                                new XElement("LINE", "I would not, in plain terms, from this time forth,"),
                                new XElement("LINE", "Have you so slander any moment leisure,"),
                                new XElement("LINE", "As to give words or talk with the Lord Hamlet."),
                                new XElement("LINE", "Look to't, I charge you: come your ways.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "I shall obey, my lord.")
                            ),
                            new XElement("STAGEDIR", "Exeunt")
                        ),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE IV.  The platform."),
                            new XElement("STAGEDIR", "Enter HAMLET, HORATIO, and MARCELLUS"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "The air bites shrewdly; it is very cold.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "It is a nipping and an eager air.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "What hour now?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "I think it lacks of twelve.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "No, it is struck.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Indeed? I heard it not: then it draws near the season"),
                                new XElement("LINE", "Wherein the spirit held his wont to walk."),
                                new XElement("STAGEDIR", "A flourish of trumpets, and ordnance shot off, within"),
                                new XElement("LINE", "What does this mean, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "The king doth wake to-night and takes his rouse,"),
                                new XElement("LINE", "Keeps wassail, and the swaggering up-spring reels;"),
                                new XElement("LINE", "And, as he drains his draughts of Rhenish down,"),
                                new XElement("LINE", "The kettle-drum and trumpet thus bray out"),
                                new XElement("LINE", "The triumph of his pledge.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Is it a custom?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Ay, marry, is't:"),
                                new XElement("LINE", "But to my mind, though I am native here"),
                                new XElement("LINE", "And to the manner born, it is a custom"),
                                new XElement("LINE", "More honour'd in the breach than the observance."),
                                new XElement("LINE", "This heavy-headed revel east and west"),
                                new XElement("LINE", "Makes us traduced and tax'd of other nations:"),
                                new XElement("LINE", "They clepe us drunkards, and with swinish phrase"),
                                new XElement("LINE", "Soil our addition; and indeed it takes"),
                                new XElement("LINE", "From our achievements, though perform'd at height,"),
                                new XElement("LINE", "The pith and marrow of our attribute."),
                                new XElement("LINE", "So, oft it chances in particular men,"),
                                new XElement("LINE", "That for some vicious mole of nature in them,"),
                                new XElement("LINE", "As, in their birth--wherein they are not guilty,"),
                                new XElement("LINE", "Since nature cannot choose his origin--"),
                                new XElement("LINE", "By the o'ergrowth of some complexion,"),
                                new XElement("LINE", "Oft breaking down the pales and forts of reason,"),
                                new XElement("LINE", "Or by some habit that too much o'er-leavens"),
                                new XElement("LINE", "The form of plausive manners, that these men,"),
                                new XElement("LINE", "Carrying, I say, the stamp of one defect,"),
                                new XElement("LINE", "Being nature's livery, or fortune's star,--"),
                                new XElement("LINE", "Their virtues else--be they as pure as grace,"),
                                new XElement("LINE", "As infinite as man may undergo--"),
                                new XElement("LINE", "Shall in the general censure take corruption"),
                                new XElement("LINE", "From that particular fault: the dram of eale"),
                                new XElement("LINE", "Doth all the noble substance of a doubt"),
                                new XElement("LINE", "To his own scandal.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Look, my lord, it comes!")
                            ),
                            new XElement("STAGEDIR", "Enter Ghost"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Angels and ministers of grace defend us!"),
                                new XElement("LINE", "Be thou a spirit of health or goblin damn'd,"),
                                new XElement("LINE", "Bring with thee airs from heaven or blasts from hell,"),
                                new XElement("LINE", "Be thy intents wicked or charitable,"),
                                new XElement("LINE", "Thou comest in such a questionable shape"),
                                new XElement("LINE", "That I will speak to thee: I'll call thee Hamlet,"),
                                new XElement("LINE", "King, father, royal Dane: O, answer me!"),
                                new XElement("LINE", "Let me not burst in ignorance; but tell"),
                                new XElement("LINE", "Why thy canonized bones, hearsed in death,"),
                                new XElement("LINE", "Have burst their cerements; why the sepulchre,"),
                                new XElement("LINE", "Wherein we saw thee quietly inurn'd,"),
                                new XElement("LINE", "Hath oped his ponderous and marble jaws,"),
                                new XElement("LINE", "To cast thee up again. What may this mean,"),
                                new XElement("LINE", "That thou, dead corse, again in complete steel"),
                                new XElement("LINE", "Revisit'st thus the glimpses of the moon,"),
                                new XElement("LINE", "Making night hideous; and we fools of nature"),
                                new XElement("LINE", "So horridly to shake our disposition"),
                                new XElement("LINE", "With thoughts beyond the reaches of our souls?"),
                                new XElement("LINE", "Say, why is this? wherefore? what should we do?")
                            ),
                            new XElement("STAGEDIR", "Ghost beckons HAMLET"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "It beckons you to go away with it,"),
                                new XElement("LINE", "As if it some impartment did desire"),
                                new XElement("LINE", "To you alone.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "Look, with what courteous action"),
                                new XElement("LINE", "It waves you to a more removed ground:"),
                                new XElement("LINE", "But do not go with it.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "No, by no means.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "It will not speak; then I will follow it.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Do not, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Why, what should be the fear?"),
                                new XElement("LINE", "I do not set my life in a pin's fee;"),
                                new XElement("LINE", "And for my soul, what can it do to that,"),
                                new XElement("LINE", "Being a thing immortal as itself?"),
                                new XElement("LINE", "It waves me forth again: I'll follow it.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "What if it tempt you toward the flood, my lord,"),
                                new XElement("LINE", "Or to the dreadful summit of the cliff"),
                                new XElement("LINE", "That beetles o'er his base into the sea,"),
                                new XElement("LINE", "And there assume some other horrible form,"),
                                new XElement("LINE", "Which might deprive your sovereignty of reason"),
                                new XElement("LINE", "And draw you into madness? think of it:"),
                                new XElement("LINE", "The very place puts toys of desperation,"),
                                new XElement("LINE", "Without more motive, into every brain"),
                                new XElement("LINE", "That looks so many fathoms to the sea"),
                                new XElement("LINE", "And hears it roar beneath.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "It waves me still."),
                                new XElement("LINE", "Go on; I'll follow thee.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "You shall not go, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Hold off your hands.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Be ruled; you shall not go.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "My fate cries out,"),
                                new XElement("LINE", "And makes each petty artery in this body"),
                                new XElement("LINE", "As hardy as the Nemean lion's nerve."),
                                new XElement("LINE", "Still am I call'd. Unhand me, gentlemen."),
                                new XElement("LINE", "By heaven, I'll make a ghost of him that lets me!"),
                                new XElement("LINE", "I say, away! Go on; I'll follow thee.")
                            ),
                            new XElement("STAGEDIR", "Exeunt Ghost and HAMLET"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "He waxes desperate with imagination.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "Let's follow; 'tis not fit thus to obey him.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Have after. To what issue will this come?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "Something is rotten in the state of Denmark.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Heaven will direct it.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "Nay, let's follow him.")
                            ),
                            new XElement("STAGEDIR", "Exeunt")
                        ),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE V.  Another part of the platform."),
                            new XElement("STAGEDIR", "Enter GHOST and HAMLET"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Where wilt thou lead me? speak; I'll go no further.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Ghost"),
                                new XElement("LINE", "Mark me.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I will.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Ghost"),
                                new XElement("LINE", "My hour is almost come,"),
                                new XElement("LINE", "When I to sulphurous and tormenting flames"),
                                new XElement("LINE", "Must render up myself.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Alas, poor ghost!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Ghost"),
                                new XElement("LINE", "Pity me not, but lend thy serious hearing"),
                                new XElement("LINE", "To what I shall unfold.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Speak; I am bound to hear.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Ghost"),
                                new XElement("LINE", "So art thou to revenge, when thou shalt hear.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "What?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Ghost"),
                                new XElement("LINE", "I am thy father's spirit,"),
                                new XElement("LINE", "Doom'd for a certain term to walk the night,"),
                                new XElement("LINE", "And for the day confined to fast in fires,"),
                                new XElement("LINE", "Till the foul crimes done in my days of nature"),
                                new XElement("LINE", "Are burnt and purged away. But that I am forbid"),
                                new XElement("LINE", "To tell the secrets of my prison-house,"),
                                new XElement("LINE", "I could a tale unfold whose lightest word"),
                                new XElement("LINE", "Would harrow up thy soul, freeze thy young blood,"),
                                new XElement("LINE", "Make thy two eyes, like stars, start from their spheres,"),
                                new XElement("LINE", "Thy knotted and combined locks to part"),
                                new XElement("LINE", "And each particular hair to stand on end,"),
                                new XElement("LINE", "Like quills upon the fretful porpentine:"),
                                new XElement("LINE", "But this eternal blazon must not be"),
                                new XElement("LINE", "To ears of flesh and blood. List, list, O, list!"),
                                new XElement("LINE", "If thou didst ever thy dear father love--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "O God!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Ghost"),
                                new XElement("LINE", "Revenge his foul and most unnatural murder.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Murder!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Ghost"),
                                new XElement("LINE", "Murder most foul, as in the best it is;"),
                                new XElement("LINE", "But this most foul, strange and unnatural.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Haste me to know't, that I, with wings as swift"),
                                new XElement("LINE", "As meditation or the thoughts of love,"),
                                new XElement("LINE", "May sweep to my revenge.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Ghost"),
                                new XElement("LINE", "I find thee apt;"),
                                new XElement("LINE", "And duller shouldst thou be than the fat weed"),
                                new XElement("LINE", "That roots itself in ease on Lethe wharf,"),
                                new XElement("LINE", "Wouldst thou not stir in this. Now, Hamlet, hear:"),
                                new XElement("LINE", "'Tis given out that, sleeping in my orchard,"),
                                new XElement("LINE", "A serpent stung me; so the whole ear of Denmark"),
                                new XElement("LINE", "Is by a forged process of my death"),
                                new XElement("LINE", "Rankly abused: but know, thou noble youth,"),
                                new XElement("LINE", "The serpent that did sting thy father's life"),
                                new XElement("LINE", "Now wears his crown.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "O my prophetic soul! My uncle!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Ghost"),
                                new XElement("LINE", "Ay, that incestuous, that adulterate beast,"),
                                new XElement("LINE", "With witchcraft of his wit, with traitorous gifts,--"),
                                new XElement("LINE", "O wicked wit and gifts, that have the power"),
                                new XElement("LINE", "So to seduce!--won to his shameful lust"),
                                new XElement("LINE", "The will of my most seeming-virtuous queen:"),
                                new XElement("LINE", "O Hamlet, what a falling-off was there!"),
                                new XElement("LINE", "From me, whose love was of that dignity"),
                                new XElement("LINE", "That it went hand in hand even with the vow"),
                                new XElement("LINE", "I made to her in marriage, and to decline"),
                                new XElement("LINE", "Upon a wretch whose natural gifts were poor"),
                                new XElement("LINE", "To those of mine!"),
                                new XElement("LINE", "But virtue, as it never will be moved,"),
                                new XElement("LINE", "Though lewdness court it in a shape of heaven,"),
                                new XElement("LINE", "So lust, though to a radiant angel link'd,"),
                                new XElement("LINE", "Will sate itself in a celestial bed,"),
                                new XElement("LINE", "And prey on garbage."),
                                new XElement("LINE", "But, soft! methinks I scent the morning air;"),
                                new XElement("LINE", "Brief let me be. Sleeping within my orchard,"),
                                new XElement("LINE", "My custom always of the afternoon,"),
                                new XElement("LINE", "Upon my secure hour thy uncle stole,"),
                                new XElement("LINE", "With juice of cursed hebenon in a vial,"),
                                new XElement("LINE", "And in the porches of my ears did pour"),
                                new XElement("LINE", "The leperous distilment; whose effect"),
                                new XElement("LINE", "Holds such an enmity with blood of man"),
                                new XElement("LINE", "That swift as quicksilver it courses through"),
                                new XElement("LINE", "The natural gates and alleys of the body,"),
                                new XElement("LINE", "And with a sudden vigour doth posset"),
                                new XElement("LINE", "And curd, like eager droppings into milk,"),
                                new XElement("LINE", "The thin and wholesome blood: so did it mine;"),
                                new XElement("LINE", "And a most instant tetter bark'd about,"),
                                new XElement("LINE", "Most lazar-like, with vile and loathsome crust,"),
                                new XElement("LINE", "All my smooth body."),
                                new XElement("LINE", "Thus was I, sleeping, by a brother's hand"),
                                new XElement("LINE", "Of life, of crown, of queen, at once dispatch'd:"),
                                new XElement("LINE", "Cut off even in the blossoms of my sin,"),
                                new XElement("LINE", "Unhousel'd, disappointed, unanel'd,"),
                                new XElement("LINE", "No reckoning made, but sent to my account"),
                                new XElement("LINE", "With all my imperfections on my head:"),
                                new XElement("LINE", "O, horrible! O, horrible! most horrible!"),
                                new XElement("LINE", "If thou hast nature in thee, bear it not;"),
                                new XElement("LINE", "Let not the royal bed of Denmark be"),
                                new XElement("LINE", "A couch for luxury and damned incest."),
                                new XElement("LINE", "But, howsoever thou pursuest this act,"),
                                new XElement("LINE", "Taint not thy mind, nor let thy soul contrive"),
                                new XElement("LINE", "Against thy mother aught: leave her to heaven"),
                                new XElement("LINE", "And to those thorns that in her bosom lodge,"),
                                new XElement("LINE", "To prick and sting her. Fare thee well at once!"),
                                new XElement("LINE", "The glow-worm shows the matin to be near,"),
                                new XElement("LINE", "And 'gins to pale his uneffectual fire:"),
                                new XElement("LINE", "Adieu, adieu! Hamlet, remember me.")
                            ),
                            new XElement("STAGEDIR", "Exit"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "O all you host of heaven! O earth! what else?"),
                                new XElement("LINE", "And shall I couple hell? O, fie! Hold, hold, my heart;"),
                                new XElement("LINE", "And you, my sinews, grow not instant old,"),
                                new XElement("LINE", "But bear me stiffly up. Remember thee!"),
                                new XElement("LINE", "Ay, thou poor ghost, while memory holds a seat"),
                                new XElement("LINE", "In this distracted globe. Remember thee!"),
                                new XElement("LINE", "Yea, from the table of my memory"),
                                new XElement("LINE", "I'll wipe away all trivial fond records,"),
                                new XElement("LINE", "All saws of books, all forms, all pressures past,"),
                                new XElement("LINE", "That youth and observation copied there;"),
                                new XElement("LINE", "And thy commandment all alone shall live"),
                                new XElement("LINE", "Within the book and volume of my brain,"),
                                new XElement("LINE", "Unmix'd with baser matter: yes, by heaven!"),
                                new XElement("LINE", "O most pernicious woman!"),
                                new XElement("LINE", "O villain, villain, smiling, damned villain!"),
                                new XElement("LINE", "My tables,--meet it is I set it down,"),
                                new XElement("LINE", "That one may smile, and smile, and be a villain;"),
                                new XElement("LINE", "At least I'm sure it may be so in Denmark:"),
                                new XElement("STAGEDIR", "Writing"),
                                new XElement("LINE", "So, uncle, there you are. Now to my word;"),
                                new XElement("LINE", "It is 'Adieu, adieu! remember me.'"),
                                new XElement("LINE", "I have sworn 't.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Within"),
                                    "  My lord, my lord,--"
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Within"),
                                    " Lord Hamlet,--"
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Within"),
                                    " Heaven secure him!"
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "So be it!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Within"),
                                    "  Hillo, ho, ho, my lord!"
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Hillo, ho, ho, boy! come, bird, come.")
                            ),
                            new XElement("STAGEDIR", "Enter HORATIO and MARCELLUS"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "How is't, my noble lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "What news, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "O, wonderful!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Good my lord, tell it.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "No; you'll reveal it.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Not I, my lord, by heaven.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "Nor I, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "How say you, then; would heart of man once think it?"),
                                new XElement("LINE", "But you'll be secret?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "Ay, by heaven, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "There's ne'er a villain dwelling in all Denmark"),
                                new XElement("LINE", "But he's an arrant knave.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "There needs no ghost, my lord, come from the grave"),
                                new XElement("LINE", "To tell us this.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Why, right; you are i' the right;"),
                                new XElement("LINE", "And so, without more circumstance at all,"),
                                new XElement("LINE", "I hold it fit that we shake hands and part:"),
                                new XElement("LINE", "You, as your business and desire shall point you;"),
                                new XElement("LINE", "For every man has business and desire,"),
                                new XElement("LINE", "Such as it is; and for mine own poor part,"),
                                new XElement("LINE", "Look you, I'll go pray.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "These are but wild and whirling words, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I'm sorry they offend you, heartily;"),
                                new XElement("LINE", "Yes, 'faith heartily.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "There's no offence, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Yes, by Saint Patrick, but there is, Horatio,"),
                                new XElement("LINE", "And much offence too. Touching this vision here,"),
                                new XElement("LINE", "It is an honest ghost, that let me tell you:"),
                                new XElement("LINE", "For your desire to know what is between us,"),
                                new XElement("LINE", "O'ermaster 't as you may. And now, good friends,"),
                                new XElement("LINE", "As you are friends, scholars and soldiers,"),
                                new XElement("LINE", "Give me one poor request.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "What is't, my lord? we will.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Never make known what you have seen to-night.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "My lord, we will not.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Nay, but swear't.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "In faith,"),
                                new XElement("LINE", "My lord, not I.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "Nor I, my lord, in faith.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Upon my sword.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "MARCELLUS"),
                                new XElement("LINE", "We have sworn, my lord, already.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Indeed, upon my sword, indeed.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Ghost"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Beneath"),
                                    "  Swear."
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Ah, ha, boy! say'st thou so? art thou there,"),
                                new XElement("LINE", "truepenny?"),
                                new XElement("LINE", "Come on--you hear this fellow in the cellarage--"),
                                new XElement("LINE", "Consent to swear.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Propose the oath, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Never to speak of this that you have seen,"),
                                new XElement("LINE", "Swear by my sword.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Ghost"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Beneath"),
                                    "  Swear."
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Hic et ubique? then we'll shift our ground."),
                                new XElement("LINE", "Come hither, gentlemen,"),
                                new XElement("LINE", "And lay your hands again upon my sword:"),
                                new XElement("LINE", "Never to speak of this that you have heard,"),
                                new XElement("LINE", "Swear by my sword.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Ghost"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Beneath"),
                                    "  Swear."
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Well said, old mole! canst work i' the earth so fast?"),
                                new XElement("LINE", "A worthy pioner! Once more remove, good friends.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "O day and night, but this is wondrous strange!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "And therefore as a stranger give it welcome."),
                                new XElement("LINE", "There are more things in heaven and earth, Horatio,"),
                                new XElement("LINE", "Than are dreamt of in your philosophy. But come;"),
                                new XElement("LINE", "Here, as before, never, so help you mercy,"),
                                new XElement("LINE", "How strange or odd soe'er I bear myself,"),
                                new XElement("LINE", "As I perchance hereafter shall think meet"),
                                new XElement("LINE", "To put an antic disposition on,"),
                                new XElement("LINE", "That you, at such times seeing me, never shall,"),
                                new XElement("LINE", "With arms encumber'd thus, or this headshake,"),
                                new XElement("LINE", "Or by pronouncing of some doubtful phrase,"),
                                new XElement("LINE", "As 'Well, well, we know,' or 'We could, an if we would,'"),
                                new XElement("LINE", "Or 'If we list to speak,' or 'There be, an if they might,'"),
                                new XElement("LINE", "Or such ambiguous giving out, to note"),
                                new XElement("LINE", "That you know aught of me: this not to do,"),
                                new XElement("LINE", "So grace and mercy at your most need help you, Swear.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Ghost"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Beneath"),
                                    "  Swear."
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Rest, rest, perturbed spirit!"),
                                new XElement("STAGEDIR", "They swear"),
                                new XElement("LINE", "So, gentlemen,"),
                                new XElement("LINE", "With all my love I do commend me to you:"),
                                new XElement("LINE", "And what so poor a man as Hamlet is"),
                                new XElement("LINE", "May do, to express his love and friending to you,"),
                                new XElement("LINE", "God willing, shall not lack. Let us go in together;"),
                                new XElement("LINE", "And still your fingers on your lips, I pray."),
                                new XElement("LINE", "The time is out of joint: O cursed spite,"),
                                new XElement("LINE", "That ever I was born to set it right!"),
                                new XElement("LINE", "Nay, come, let's go together.")
                            ),
                            new XElement("STAGEDIR", "Exeunt")
                        )
                    ),
                    new XElement("ACT",
                        new XElement("TITLE", "ACT II"),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE I.  A room in POLONIUS' house."),
                            new XElement("STAGEDIR", "Enter POLONIUS and REYNALDO"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Give him this money and these notes, Reynaldo.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "REYNALDO"),
                                new XElement("LINE", "I will, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "You shall do marvellous wisely, good Reynaldo,"),
                                new XElement("LINE", "Before you visit him, to make inquire"),
                                new XElement("LINE", "Of his behavior.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "REYNALDO"),
                                new XElement("LINE", "My lord, I did intend it.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Marry, well said; very well said. Look you, sir,"),
                                new XElement("LINE", "Inquire me first what Danskers are in Paris;"),
                                new XElement("LINE", "And how, and who, what means, and where they keep,"),
                                new XElement("LINE", "What company, at what expense; and finding"),
                                new XElement("LINE", "By this encompassment and drift of question"),
                                new XElement("LINE", "That they do know my son, come you more nearer"),
                                new XElement("LINE", "Than your particular demands will touch it:"),
                                new XElement("LINE", "Take you, as 'twere, some distant knowledge of him;"),
                                new XElement("LINE", "As thus, 'I know his father and his friends,"),
                                new XElement("LINE", "And in part him: ' do you mark this, Reynaldo?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "REYNALDO"),
                                new XElement("LINE", "Ay, very well, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "'And in part him; but' you may say 'not well:"),
                                new XElement("LINE", "But, if't be he I mean, he's very wild;"),
                                new XElement("LINE", "Addicted so and so:' and there put on him"),
                                new XElement("LINE", "What forgeries you please; marry, none so rank"),
                                new XElement("LINE", "As may dishonour him; take heed of that;"),
                                new XElement("LINE", "But, sir, such wanton, wild and usual slips"),
                                new XElement("LINE", "As are companions noted and most known"),
                                new XElement("LINE", "To youth and liberty.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "REYNALDO"),
                                new XElement("LINE", "As gaming, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Ay, or drinking, fencing, swearing, quarrelling,"),
                                new XElement("LINE", "Drabbing: you may go so far.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "REYNALDO"),
                                new XElement("LINE", "My lord, that would dishonour him.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "'Faith, no; as you may season it in the charge"),
                                new XElement("LINE", "You must not put another scandal on him,"),
                                new XElement("LINE", "That he is open to incontinency;"),
                                new XElement("LINE", "That's not my meaning: but breathe his faults so quaintly"),
                                new XElement("LINE", "That they may seem the taints of liberty,"),
                                new XElement("LINE", "The flash and outbreak of a fiery mind,"),
                                new XElement("LINE", "A savageness in unreclaimed blood,"),
                                new XElement("LINE", "Of general assault.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "REYNALDO"),
                                new XElement("LINE", "But, my good lord,--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Wherefore should you do this?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "REYNALDO"),
                                new XElement("LINE", "Ay, my lord,"),
                                new XElement("LINE", "I would know that.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Marry, sir, here's my drift;"),
                                new XElement("LINE", "And I believe, it is a fetch of wit:"),
                                new XElement("LINE", "You laying these slight sullies on my son,"),
                                new XElement("LINE", "As 'twere a thing a little soil'd i' the working, Mark you,"),
                                new XElement("LINE", "Your party in converse, him you would sound,"),
                                new XElement("LINE", "Having ever seen in the prenominate crimes"),
                                new XElement("LINE", "The youth you breathe of guilty, be assured"),
                                new XElement("LINE", "He closes with you in this consequence;"),
                                new XElement("LINE", "'Good sir,' or so, or 'friend,' or 'gentleman,'"),
                                new XElement("LINE", "According to the phrase or the addition"),
                                new XElement("LINE", "Of man and country.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "REYNALDO"),
                                new XElement("LINE", "Very good, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "And then, sir, does he this--he does--what was I"),
                                new XElement("LINE", "about to say? By the mass, I was about to say"),
                                new XElement("LINE", "something: where did I leave?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "REYNALDO"),
                                new XElement("LINE", "At 'closes in the consequence,' at 'friend or so,'"),
                                new XElement("LINE", "and 'gentleman.'")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "At 'closes in the consequence,' ay, marry;"),
                                new XElement("LINE", "He closes thus: 'I know the gentleman;"),
                                new XElement("LINE", "I saw him yesterday, or t' other day,"),
                                new XElement("LINE", "Or then, or then; with such, or such; and, as you say,"),
                                new XElement("LINE", "There was a' gaming; there o'ertook in's rouse;"),
                                new XElement("LINE", "There falling out at tennis:' or perchance,"),
                                new XElement("LINE", "'I saw him enter such a house of sale,'"),
                                new XElement("LINE", "Videlicet, a brothel, or so forth."),
                                new XElement("LINE", "See you now;"),
                                new XElement("LINE", "Your bait of falsehood takes this carp of truth:"),
                                new XElement("LINE", "And thus do we of wisdom and of reach,"),
                                new XElement("LINE", "With windlasses and with assays of bias,"),
                                new XElement("LINE", "By indirections find directions out:"),
                                new XElement("LINE", "So by my former lecture and advice,"),
                                new XElement("LINE", "Shall you my son. You have me, have you not?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "REYNALDO"),
                                new XElement("LINE", "My lord, I have.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "God be wi' you; fare you well.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "REYNALDO"),
                                new XElement("LINE", "Good my lord!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Observe his inclination in yourself.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "REYNALDO"),
                                new XElement("LINE", "I shall, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "And let him ply his music.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "REYNALDO"),
                                new XElement("LINE", "Well, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Farewell!"),
                                new XElement("STAGEDIR", "Exit REYNALDO"),
                                new XElement("STAGEDIR", "Enter OPHELIA"),
                                new XElement("LINE", "How now, Ophelia! what's the matter?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "O, my lord, my lord, I have been so affrighted!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "With what, i' the name of God?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "My lord, as I was sewing in my closet,"),
                                new XElement("LINE", "Lord Hamlet, with his doublet all unbraced;"),
                                new XElement("LINE", "No hat upon his head; his stockings foul'd,"),
                                new XElement("LINE", "Ungarter'd, and down-gyved to his ancle;"),
                                new XElement("LINE", "Pale as his shirt; his knees knocking each other;"),
                                new XElement("LINE", "And with a look so piteous in purport"),
                                new XElement("LINE", "As if he had been loosed out of hell"),
                                new XElement("LINE", "To speak of horrors,--he comes before me.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Mad for thy love?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "My lord, I do not know;"),
                                new XElement("LINE", "But truly, I do fear it.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "What said he?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "He took me by the wrist and held me hard;"),
                                new XElement("LINE", "Then goes he to the length of all his arm;"),
                                new XElement("LINE", "And, with his other hand thus o'er his brow,"),
                                new XElement("LINE", "He falls to such perusal of my face"),
                                new XElement("LINE", "As he would draw it. Long stay'd he so;"),
                                new XElement("LINE", "At last, a little shaking of mine arm"),
                                new XElement("LINE", "And thrice his head thus waving up and down,"),
                                new XElement("LINE", "He raised a sigh so piteous and profound"),
                                new XElement("LINE", "As it did seem to shatter all his bulk"),
                                new XElement("LINE", "And end his being: that done, he lets me go:"),
                                new XElement("LINE", "And, with his head over his shoulder turn'd,"),
                                new XElement("LINE", "He seem'd to find his way without his eyes;"),
                                new XElement("LINE", "For out o' doors he went without their helps,"),
                                new XElement("LINE", "And, to the last, bended their light on me.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Come, go with me: I will go seek the king."),
                                new XElement("LINE", "This is the very ecstasy of love,"),
                                new XElement("LINE", "Whose violent property fordoes itself"),
                                new XElement("LINE", "And leads the will to desperate undertakings"),
                                new XElement("LINE", "As oft as any passion under heaven"),
                                new XElement("LINE", "That does afflict our natures. I am sorry."),
                                new XElement("LINE", "What, have you given him any hard words of late?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "No, my good lord, but, as you did command,"),
                                new XElement("LINE", "I did repel his fetters and denied"),
                                new XElement("LINE", "His access to me.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "That hath made him mad."),
                                new XElement("LINE", "I am sorry that with better heed and judgment"),
                                new XElement("LINE", "I had not quoted him: I fear'd he did but trifle,"),
                                new XElement("LINE", "And meant to wreck thee; but, beshrew my jealousy!"),
                                new XElement("LINE", "By heaven, it is as proper to our age"),
                                new XElement("LINE", "To cast beyond ourselves in our opinions"),
                                new XElement("LINE", "As it is common for the younger sort"),
                                new XElement("LINE", "To lack discretion. Come, go we to the king:"),
                                new XElement("LINE", "This must be known; which, being kept close, might"),
                                new XElement("LINE", "move"),
                                new XElement("LINE", "More grief to hide than hate to utter love.")
                            ),
                            new XElement("STAGEDIR", "Exeunt")
                        ),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE II.  A room in the castle."),
                            new XElement("STAGEDIR",
                                "Enter KING CLAUDIUS, QUEEN GERTRUDE, ROSENCRANTZ,\n" +
                                "GUILDENSTERN, and Attendants"
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Welcome, dear Rosencrantz and Guildenstern!"),
                                new XElement("LINE", "Moreover that we much did long to see you,"),
                                new XElement("LINE", "The need we have to use you did provoke"),
                                new XElement("LINE", "Our hasty sending. Something have you heard"),
                                new XElement("LINE", "Of Hamlet's transformation; so call it,"),
                                new XElement("LINE", "Sith nor the exterior nor the inward man"),
                                new XElement("LINE", "Resembles that it was. What it should be,"),
                                new XElement("LINE", "More than his father's death, that thus hath put him"),
                                new XElement("LINE", "So much from the understanding of himself,"),
                                new XElement("LINE", "I cannot dream of: I entreat you both,"),
                                new XElement("LINE", "That, being of so young days brought up with him,"),
                                new XElement("LINE", "And sith so neighbour'd to his youth and havior,"),
                                new XElement("LINE", "That you vouchsafe your rest here in our court"),
                                new XElement("LINE", "Some little time: so by your companies"),
                                new XElement("LINE", "To draw him on to pleasures, and to gather,"),
                                new XElement("LINE", "So much as from occasion you may glean,"),
                                new XElement("LINE", "Whether aught, to us unknown, afflicts him thus,"),
                                new XElement("LINE", "That, open'd, lies within our remedy.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Good gentlemen, he hath much talk'd of you;"),
                                new XElement("LINE", "And sure I am two men there are not living"),
                                new XElement("LINE", "To whom he more adheres. If it will please you"),
                                new XElement("LINE", "To show us so much gentry and good will"),
                                new XElement("LINE", "As to expend your time with us awhile,"),
                                new XElement("LINE", "For the supply and profit of our hope,"),
                                new XElement("LINE", "Your visitation shall receive such thanks"),
                                new XElement("LINE", "As fits a king's remembrance.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Both your majesties"),
                                new XElement("LINE", "Might, by the sovereign power you have of us,"),
                                new XElement("LINE", "Put your dread pleasures more into command"),
                                new XElement("LINE", "Than to entreaty.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "But we both obey,"),
                                new XElement("LINE", "And here give up ourselves, in the full bent"),
                                new XElement("LINE", "To lay our service freely at your feet,"),
                                new XElement("LINE", "To be commanded.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Thanks, Rosencrantz and gentle Guildenstern.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Thanks, Guildenstern and gentle Rosencrantz:"),
                                new XElement("LINE", "And I beseech you instantly to visit"),
                                new XElement("LINE", "My too much changed son. Go, some of you,"),
                                new XElement("LINE", "And bring these gentlemen where Hamlet is.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "Heavens make our presence and our practises"),
                                new XElement("LINE", "Pleasant and helpful to him!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Ay, amen!")
                            ),
                            new XElement("STAGEDIR",
                                "Exeunt ROSENCRANTZ, GUILDENSTERN, and some\n" +
                                "Attendants"
                            ),
                            new XElement("STAGEDIR", "Enter POLONIUS"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "The ambassadors from Norway, my good lord,"),
                                new XElement("LINE", "Are joyfully return'd.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Thou still hast been the father of good news.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Have I, my lord? I assure my good liege,"),
                                new XElement("LINE", "I hold my duty, as I hold my soul,"),
                                new XElement("LINE", "Both to my God and to my gracious king:"),
                                new XElement("LINE", "And I do think, or else this brain of mine"),
                                new XElement("LINE", "Hunts not the trail of policy so sure"),
                                new XElement("LINE", "As it hath used to do, that I have found"),
                                new XElement("LINE", "The very cause of Hamlet's lunacy.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "O, speak of that; that do I long to hear.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Give first admittance to the ambassadors;"),
                                new XElement("LINE", "My news shall be the fruit to that great feast.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Thyself do grace to them, and bring them in."),
                                new XElement("STAGEDIR", "Exit POLONIUS"),
                                new XElement("LINE", "He tells me, my dear Gertrude, he hath found"),
                                new XElement("LINE", "The head and source of all your son's distemper.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "I doubt it is no other but the main;"),
                                new XElement("LINE", "His father's death, and our o'erhasty marriage.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Well, we shall sift him."),
                                new XElement("STAGEDIR", "Re-enter POLONIUS, with VOLTIMAND and CORNELIUS"),
                                new XElement("LINE", "Welcome, my good friends!"),
                                new XElement("LINE", "Say, Voltimand, what from our brother Norway?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "VOLTIMAND"),
                                new XElement("LINE", "Most fair return of greetings and desires."),
                                new XElement("LINE", "Upon our first, he sent out to suppress"),
                                new XElement("LINE", "His nephew's levies; which to him appear'd"),
                                new XElement("LINE", "To be a preparation 'gainst the Polack;"),
                                new XElement("LINE", "But, better look'd into, he truly found"),
                                new XElement("LINE", "It was against your highness: whereat grieved,"),
                                new XElement("LINE", "That so his sickness, age and impotence"),
                                new XElement("LINE", "Was falsely borne in hand, sends out arrests"),
                                new XElement("LINE", "On Fortinbras; which he, in brief, obeys;"),
                                new XElement("LINE", "Receives rebuke from Norway, and in fine"),
                                new XElement("LINE", "Makes vow before his uncle never more"),
                                new XElement("LINE", "To give the assay of arms against your majesty."),
                                new XElement("LINE", "Whereon old Norway, overcome with joy,"),
                                new XElement("LINE", "Gives him three thousand crowns in annual fee,"),
                                new XElement("LINE", "And his commission to employ those soldiers,"),
                                new XElement("LINE", "So levied as before, against the Polack:"),
                                new XElement("LINE", "With an entreaty, herein further shown,"),
                                new XElement("STAGEDIR", "Giving a paper"),
                                new XElement("LINE", "That it might please you to give quiet pass"),
                                new XElement("LINE", "Through your dominions for this enterprise,"),
                                new XElement("LINE", "On such regards of safety and allowance"),
                                new XElement("LINE", "As therein are set down.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "It likes us well;"),
                                new XElement("LINE", "And at our more consider'd time well read,"),
                                new XElement("LINE", "Answer, and think upon this business."),
                                new XElement("LINE", "Meantime we thank you for your well-took labour:"),
                                new XElement("LINE", "Go to your rest; at night we'll feast together:"),
                                new XElement("LINE", "Most welcome home!")
                            ),
                            new XElement("STAGEDIR", "Exeunt VOLTIMAND and CORNELIUS"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "This business is well ended."),
                                new XElement("LINE", "My liege, and madam, to expostulate"),
                                new XElement("LINE", "What majesty should be, what duty is,"),
                                new XElement("LINE", "Why day is day, night night, and time is time,"),
                                new XElement("LINE", "Were nothing but to waste night, day and time."),
                                new XElement("LINE", "Therefore, since brevity is the soul of wit,"),
                                new XElement("LINE", "And tediousness the limbs and outward flourishes,"),
                                new XElement("LINE", "I will be brief: your noble son is mad:"),
                                new XElement("LINE", "Mad call I it; for, to define true madness,"),
                                new XElement("LINE", "What is't but to be nothing else but mad?"),
                                new XElement("LINE", "But let that go.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "More matter, with less art.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Madam, I swear I use no art at all."),
                                new XElement("LINE", "That he is mad, 'tis true: 'tis true 'tis pity;"),
                                new XElement("LINE", "And pity 'tis 'tis true: a foolish figure;"),
                                new XElement("LINE", "But farewell it, for I will use no art."),
                                new XElement("LINE", "Mad let us grant him, then: and now remains"),
                                new XElement("LINE", "That we find out the cause of this effect,"),
                                new XElement("LINE", "Or rather say, the cause of this defect,"),
                                new XElement("LINE", "For this effect defective comes by cause:"),
                                new XElement("LINE", "Thus it remains, and the remainder thus. Perpend."),
                                new XElement("LINE", "I have a daughter--have while she is mine--"),
                                new XElement("LINE", "Who, in her duty and obedience, mark,"),
                                new XElement("LINE", "Hath given me this: now gather, and surmise."),
                                new XElement("STAGEDIR", "Reads"),
                                new XElement("LINE", "'To the celestial and my soul's idol, the most"),
                                new XElement("LINE", "beautified Ophelia,'--"),
                                new XElement("LINE", "That's an ill phrase, a vile phrase; 'beautified' is"),
                                new XElement("LINE", "a vile phrase: but you shall hear. Thus:"),
                                new XElement("STAGEDIR", "Reads"),
                                new XElement("LINE", "'In her excellent white bosom, these, &c.'")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Came this from Hamlet to her?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Good madam, stay awhile; I will be faithful."),
                                new XElement("STAGEDIR", "Reads"),
                                new XElement("LINE", "'Doubt thou the stars are fire;"),
                                new XElement("LINE", "Doubt that the sun doth move;"),
                                new XElement("LINE", "Doubt truth to be a liar;"),
                                new XElement("LINE", "But never doubt I love."),
                                new XElement("LINE", "'O dear Ophelia, I am ill at these numbers;"),
                                new XElement("LINE", "I have not art to reckon my groans: but that"),
                                new XElement("LINE", "I love thee best, O most best, believe it. Adieu."),
                                new XElement("LINE", "'Thine evermore most dear lady, whilst"),
                                new XElement("LINE", "this machine is to him, HAMLET.'"),
                                new XElement("LINE", "This, in obedience, hath my daughter shown me,"),
                                new XElement("LINE", "And more above, hath his solicitings,"),
                                new XElement("LINE", "As they fell out by time, by means and place,"),
                                new XElement("LINE", "All given to mine ear.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "But how hath she"),
                                new XElement("LINE", "Received his love?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "What do you think of me?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "As of a man faithful and honourable.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "I would fain prove so. But what might you think,"),
                                new XElement("LINE", "When I had seen this hot love on the wing--"),
                                new XElement("LINE", "As I perceived it, I must tell you that,"),
                                new XElement("LINE", "Before my daughter told me--what might you,"),
                                new XElement("LINE", "Or my dear majesty your queen here, think,"),
                                new XElement("LINE", "If I had play'd the desk or table-book,"),
                                new XElement("LINE", "Or given my heart a winking, mute and dumb,"),
                                new XElement("LINE", "Or look'd upon this love with idle sight;"),
                                new XElement("LINE", "What might you think? No, I went round to work,"),
                                new XElement("LINE", "And my young mistress thus I did bespeak:"),
                                new XElement("LINE", "'Lord Hamlet is a prince, out of thy star;"),
                                new XElement("LINE", "This must not be:' and then I precepts gave her,"),
                                new XElement("LINE", "That she should lock herself from his resort,"),
                                new XElement("LINE", "Admit no messengers, receive no tokens."),
                                new XElement("LINE", "Which done, she took the fruits of my advice;"),
                                new XElement("LINE", "And he, repulsed--a short tale to make--"),
                                new XElement("LINE", "Fell into a sadness, then into a fast,"),
                                new XElement("LINE", "Thence to a watch, thence into a weakness,"),
                                new XElement("LINE", "Thence to a lightness, and, by this declension,"),
                                new XElement("LINE", "Into the madness wherein now he raves,"),
                                new XElement("LINE", "And all we mourn for.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Do you think 'tis this?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "It may be, very likely.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Hath there been such a time--I'd fain know that--"),
                                new XElement("LINE", "That I have positively said 'Tis so,'"),
                                new XElement("LINE", "When it proved otherwise?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Not that I know.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Pointing to his head and shoulder")
                                ),
                                new XElement("LINE", "Take this from this, if this be otherwise:"),
                                new XElement("LINE", "If circumstances lead me, I will find"),
                                new XElement("LINE", "Where truth is hid, though it were hid indeed"),
                                new XElement("LINE", "Within the centre.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "How may we try it further?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "You know, sometimes he walks four hours together"),
                                new XElement("LINE", "Here in the lobby.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "So he does indeed.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "At such a time I'll loose my daughter to him:"),
                                new XElement("LINE", "Be you and I behind an arras then;"),
                                new XElement("LINE", "Mark the encounter: if he love her not"),
                                new XElement("LINE", "And be not from his reason fall'n thereon,"),
                                new XElement("LINE", "Let me be no assistant for a state,"),
                                new XElement("LINE", "But keep a farm and carters.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "We will try it.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "But, look, where sadly the poor wretch comes reading.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Away, I do beseech you, both away:"),
                                new XElement("LINE", "I'll board him presently."),
                                new XElement("STAGEDIR",
                                    "Exeunt KING CLAUDIUS, QUEEN GERTRUDE, and\n" +
                                    "Attendants"
                                ),
                                new XElement("STAGEDIR", "Enter HAMLET, reading"),
                                new XElement("LINE", "O, give me leave:"),
                                new XElement("LINE", "How does my good Lord Hamlet?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Well, God-a-mercy.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Do you know me, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Excellent well; you are a fishmonger.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Not I, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Then I would you were so honest a man.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Honest, my lord!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Ay, sir; to be honest, as this world goes, is to be"),
                                new XElement("LINE", "one man picked out of ten thousand.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "That's very true, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "For if the sun breed maggots in a dead dog, being a"),
                                new XElement("LINE", "god kissing carrion,--Have you a daughter?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "I have, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Let her not walk i' the sun: conception is a"),
                                new XElement("LINE", "blessing: but not as your daughter may conceive."),
                                new XElement("LINE", "Friend, look to 't.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Aside"),
                                    "  How say you by that? Still harping on my"
                                ),
                                new XElement("LINE", "daughter: yet he knew me not at first; he said I"),
                                new XElement("LINE", "was a fishmonger: he is far gone, far gone: and"),
                                new XElement("LINE", "truly in my youth I suffered much extremity for"),
                                new XElement("LINE", "love; very near this. I'll speak to him again."),
                                new XElement("LINE", "What do you read, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Words, words, words.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "What is the matter, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Between who?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "I mean, the matter that you read, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Slanders, sir: for the satirical rogue says here"),
                                new XElement("LINE", "that old men have grey beards, that their faces are"),
                                new XElement("LINE", "wrinkled, their eyes purging thick amber and"),
                                new XElement("LINE", "plum-tree gum and that they have a plentiful lack of"),
                                new XElement("LINE", "wit, together with most weak hams: all which, sir,"),
                                new XElement("LINE", "though I most powerfully and potently believe, yet"),
                                new XElement("LINE", "I hold it not honesty to have it thus set down, for"),
                                new XElement("LINE", "yourself, sir, should be old as I am, if like a crab"),
                                new XElement("LINE", "you could go backward.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Aside"),
                                    "  Though this be madness, yet there is method"
                                ),
                                new XElement("LINE", "in 't. Will you walk out of the air, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Into my grave.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Indeed, that is out o' the air."),
                                new XElement("STAGEDIR", "Aside"),
                                new XElement("LINE", "How pregnant sometimes his replies are! a happiness"),
                                new XElement("LINE", "that often madness hits on, which reason and sanity"),
                                new XElement("LINE", "could not so prosperously be delivered of. I will"),
                                new XElement("LINE", "leave him, and suddenly contrive the means of"),
                                new XElement("LINE", "meeting between him and my daughter.--My honourable"),
                                new XElement("LINE", "lord, I will most humbly take my leave of you.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "You cannot, sir, take from me any thing that I will"),
                                new XElement("LINE", "more willingly part withal: except my life, except"),
                                new XElement("LINE", "my life, except my life.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Fare you well, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "These tedious old fools!")
                            ),
                            new XElement("STAGEDIR", "Enter ROSENCRANTZ and GUILDENSTERN"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "You go to seek the Lord Hamlet; there he is.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "To POLONIUS"),
                                    "  God save you, sir!"
                                )
                            ),
                            new XElement("STAGEDIR", "Exit POLONIUS"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "My honoured lord!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "My most dear lord!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "My excellent good friends! How dost thou,"),
                                new XElement("LINE", "Guildenstern? Ah, Rosencrantz! Good lads, how do ye both?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "As the indifferent children of the earth.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "Happy, in that we are not over-happy;"),
                                new XElement("LINE", "On fortune's cap we are not the very button.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Nor the soles of her shoe?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Neither, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Then you live about her waist, or in the middle of"),
                                new XElement("LINE", "her favours?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "'Faith, her privates we.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "In the secret parts of fortune? O, most true; she"),
                                new XElement("LINE", "is a strumpet. What's the news?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "None, my lord, but that the world's grown honest.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Then is doomsday near: but your news is not true."),
                                new XElement("LINE", "Let me question more in particular: what have you,"),
                                new XElement("LINE", "my good friends, deserved at the hands of fortune,"),
                                new XElement("LINE", "that she sends you to prison hither?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "Prison, my lord!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Denmark's a prison.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Then is the world one.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "A goodly one; in which there are many confines,"),
                                new XElement("LINE", "wards and dungeons, Denmark being one o' the worst.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "We think not so, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Why, then, 'tis none to you; for there is nothing"),
                                new XElement("LINE", "either good or bad, but thinking makes it so: to me"),
                                new XElement("LINE", "it is a prison.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Why then, your ambition makes it one; 'tis too"),
                                new XElement("LINE", "narrow for your mind.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "O God, I could be bounded in a nut shell and count"),
                                new XElement("LINE", "myself a king of infinite space, were it not that I"),
                                new XElement("LINE", "have bad dreams.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "Which dreams indeed are ambition, for the very"),
                                new XElement("LINE", "substance of the ambitious is merely the shadow of a dream.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "A dream itself is but a shadow.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Truly, and I hold ambition of so airy and light a"),
                                new XElement("LINE", "quality that it is but a shadow's shadow.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Then are our beggars bodies, and our monarchs and"),
                                new XElement("LINE", "outstretched heroes the beggars' shadows. Shall we"),
                                new XElement("LINE", "to the court? for, by my fay, I cannot reason.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "We'll wait upon you.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "No such matter: I will not sort you with the rest"),
                                new XElement("LINE", "of my servants, for, to speak to you like an honest"),
                                new XElement("LINE", "man, I am most dreadfully attended. But, in the"),
                                new XElement("LINE", "beaten way of friendship, what make you at Elsinore?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "To visit you, my lord; no other occasion.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Beggar that I am, I am even poor in thanks; but I"),
                                new XElement("LINE", "thank you: and sure, dear friends, my thanks are"),
                                new XElement("LINE", "too dear a halfpenny. Were you not sent for? Is it"),
                                new XElement("LINE", "your own inclining? Is it a free visitation? Come,"),
                                new XElement("LINE", "deal justly with me: come, come; nay, speak.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "What should we say, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Why, any thing, but to the purpose. You were sent"),
                                new XElement("LINE", "for; and there is a kind of confession in your looks"),
                                new XElement("LINE", "which your modesties have not craft enough to colour:"),
                                new XElement("LINE", "I know the good king and queen have sent for you.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "To what end, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "That you must teach me. But let me conjure you, by"),
                                new XElement("LINE", "the rights of our fellowship, by the consonancy of"),
                                new XElement("LINE", "our youth, by the obligation of our ever-preserved"),
                                new XElement("LINE", "love, and by what more dear a better proposer could"),
                                new XElement("LINE", "charge you withal, be even and direct with me,"),
                                new XElement("LINE", "whether you were sent for, or no?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Aside to GUILDENSTERN"),
                                    "  What say you?"
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Aside"),
                                    "  Nay, then, I have an eye of you.--If you"
                                ),
                                new XElement("LINE", "love me, hold not off.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "My lord, we were sent for.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I will tell you why; so shall my anticipation"),
                                new XElement("LINE", "prevent your discovery, and your secrecy to the king"),
                                new XElement("LINE", "and queen moult no feather. I have of late--but"),
                                new XElement("LINE", "wherefore I know not--lost all my mirth, forgone all"),
                                new XElement("LINE", "custom of exercises; and indeed it goes so heavily"),
                                new XElement("LINE", "with my disposition that this goodly frame, the"),
                                new XElement("LINE", "earth, seems to me a sterile promontory, this most"),
                                new XElement("LINE", "excellent canopy, the air, look you, this brave"),
                                new XElement("LINE", "o'erhanging firmament, this majestical roof fretted"),
                                new XElement("LINE", "with golden fire, why, it appears no other thing to"),
                                new XElement("LINE", "me than a foul and pestilent congregation of vapours."),
                                new XElement("LINE", "What a piece of work is a man! how noble in reason!"),
                                new XElement("LINE", "how infinite in faculty! in form and moving how"),
                                new XElement("LINE", "express and admirable! in action how like an angel!"),
                                new XElement("LINE", "in apprehension how like a god! the beauty of the"),
                                new XElement("LINE", "world! the paragon of animals! And yet, to me,"),
                                new XElement("LINE", "what is this quintessence of dust? man delights not"),
                                new XElement("LINE", "me: no, nor woman neither, though by your smiling"),
                                new XElement("LINE", "you seem to say so.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "My lord, there was no such stuff in my thoughts.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Why did you laugh then, when I said 'man delights not me'?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "To think, my lord, if you delight not in man, what"),
                                new XElement("LINE", "lenten entertainment the players shall receive from"),
                                new XElement("LINE", "you: we coted them on the way; and hither are they"),
                                new XElement("LINE", "coming, to offer you service.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "He that plays the king shall be welcome; his majesty"),
                                new XElement("LINE", "shall have tribute of me; the adventurous knight"),
                                new XElement("LINE", "shall use his foil and target; the lover shall not"),
                                new XElement("LINE", "sigh gratis; the humourous man shall end his part"),
                                new XElement("LINE", "in peace; the clown shall make those laugh whose"),
                                new XElement("LINE", "lungs are tickled o' the sere; and the lady shall"),
                                new XElement("LINE", "say her mind freely, or the blank verse shall halt"),
                                new XElement("LINE", "for't. What players are they?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Even those you were wont to take delight in, the"),
                                new XElement("LINE", "tragedians of the city.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "How chances it they travel? their residence, both"),
                                new XElement("LINE", "in reputation and profit, was better both ways.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "I think their inhibition comes by the means of the"),
                                new XElement("LINE", "late innovation.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Do they hold the same estimation they did when I was"),
                                new XElement("LINE", "in the city? are they so followed?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "No, indeed, are they not.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "How comes it? do they grow rusty?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Nay, their endeavour keeps in the wonted pace: but"),
                                new XElement("LINE", "there is, sir, an aery of children, little eyases,"),
                                new XElement("LINE", "that cry out on the top of question, and are most"),
                                new XElement("LINE", "tyrannically clapped for't: these are now the"),
                                new XElement("LINE", "fashion, and so berattle the common stages--so they"),
                                new XElement("LINE", "call them--that many wearing rapiers are afraid of"),
                                new XElement("LINE", "goose-quills and dare scarce come thither.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "What, are they children? who maintains 'em? how are"),
                                new XElement("LINE", "they escoted? Will they pursue the quality no"),
                                new XElement("LINE", "longer than they can sing? will they not say"),
                                new XElement("LINE", "afterwards, if they should grow themselves to common"),
                                new XElement("LINE", "players--as it is most like, if their means are no"),
                                new XElement("LINE", "better--their writers do them wrong, to make them"),
                                new XElement("LINE", "exclaim against their own succession?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "'Faith, there has been much to do on both sides; and"),
                                new XElement("LINE", "the nation holds it no sin to tarre them to"),
                                new XElement("LINE", "controversy: there was, for a while, no money bid"),
                                new XElement("LINE", "for argument, unless the poet and the player went to"),
                                new XElement("LINE", "cuffs in the question.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Is't possible?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "O, there has been much throwing about of brains.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Do the boys carry it away?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Ay, that they do, my lord; Hercules and his load too.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "It is not very strange; for mine uncle is king of"),
                                new XElement("LINE", "Denmark, and those that would make mows at him while"),
                                new XElement("LINE", "my father lived, give twenty, forty, fifty, an"),
                                new XElement("LINE", "hundred ducats a-piece for his picture in little."),
                                new XElement("LINE", "'Sblood, there is something in this more than"),
                                new XElement("LINE", "natural, if philosophy could find it out.")
                            ),
                            new XElement("STAGEDIR", "Flourish of trumpets within"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "There are the players.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Gentlemen, you are welcome to Elsinore. Your hands,"),
                                new XElement("LINE", "come then: the appurtenance of welcome is fashion"),
                                new XElement("LINE", "and ceremony: let me comply with you in this garb,"),
                                new XElement("LINE", "lest my extent to the players, which, I tell you,"),
                                new XElement("LINE", "must show fairly outward, should more appear like"),
                                new XElement("LINE", "entertainment than yours. You are welcome: but my"),
                                new XElement("LINE", "uncle-father and aunt-mother are deceived.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "In what, my dear lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I am but mad north-north-west: when the wind is"),
                                new XElement("LINE", "southerly I know a hawk from a handsaw.")
                            ),
                            new XElement("STAGEDIR", "Enter POLONIUS"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Well be with you, gentlemen!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Hark you, Guildenstern; and you too: at each ear a"),
                                new XElement("LINE", "hearer: that great baby you see there is not yet"),
                                new XElement("LINE", "out of his swaddling-clouts.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Happily he's the second time come to them; for they"),
                                new XElement("LINE", "say an old man is twice a child.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I will prophesy he comes to tell me of the players;"),
                                new XElement("LINE", "mark it. You say right, sir: o' Monday morning;"),
                                new XElement("LINE", "'twas so indeed.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "My lord, I have news to tell you.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "My lord, I have news to tell you."),
                                new XElement("LINE", "When Roscius was an actor in Rome,--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "The actors are come hither, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Buz, buz!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Upon mine honour,--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Then came each actor on his ass,--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "The best actors in the world, either for tragedy,"),
                                new XElement("LINE", "comedy, history, pastoral, pastoral-comical,"),
                                new XElement("LINE", "historical-pastoral, tragical-historical, tragical-"),
                                new XElement("LINE", "comical-historical-pastoral, scene individable, or"),
                                new XElement("LINE", "poem unlimited: Seneca cannot be too heavy, nor"),
                                new XElement("LINE", "Plautus too light. For the law of writ and the"),
                                new XElement("LINE", "liberty, these are the only men.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "O Jephthah, judge of Israel, what a treasure hadst thou!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "What a treasure had he, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Why,"),
                                new XElement("LINE", "'One fair daughter and no more,"),
                                new XElement("LINE", "The which he loved passing well.'")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Aside"),
                                    "  Still on my daughter."
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Am I not i' the right, old Jephthah?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "If you call me Jephthah, my lord, I have a daughter"),
                                new XElement("LINE", "that I love passing well.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Nay, that follows not.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "What follows, then, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Why,"),
                                new XElement("LINE", "'As by lot, God wot,'"),
                                new XElement("LINE", "and then, you know,"),
                                new XElement("LINE", "'It came to pass, as most like it was,'--"),
                                new XElement("LINE", "the first row of the pious chanson will show you"),
                                new XElement("LINE", "more; for look, where my abridgement comes."),
                                new XElement("STAGEDIR", "Enter four or five Players"),
                                new XElement("LINE", "You are welcome, masters; welcome, all. I am glad"),
                                new XElement("LINE", "to see thee well. Welcome, good friends. O, my old"),
                                new XElement("LINE", "friend! thy face is valenced since I saw thee last:"),
                                new XElement("LINE", "comest thou to beard me in Denmark? What, my young"),
                                new XElement("LINE", "lady and mistress! By'r lady, your ladyship is"),
                                new XElement("LINE", "nearer to heaven than when I saw you last, by the"),
                                new XElement("LINE", "altitude of a chopine. Pray God, your voice, like"),
                                new XElement("LINE", "apiece of uncurrent gold, be not cracked within the"),
                                new XElement("LINE", "ring. Masters, you are all welcome. We'll e'en"),
                                new XElement("LINE", "to't like French falconers, fly at any thing we see:"),
                                new XElement("LINE", "we'll have a speech straight: come, give us a taste"),
                                new XElement("LINE", "of your quality; come, a passionate speech.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Player"),
                                new XElement("LINE", "What speech, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I heard thee speak me a speech once, but it was"),
                                new XElement("LINE", "never acted; or, if it was, not above once; for the"),
                                new XElement("LINE", "play, I remember, pleased not the million; 'twas"),
                                new XElement("LINE", "caviare to the general: but it was--as I received"),
                                new XElement("LINE", "it, and others, whose judgments in such matters"),
                                new XElement("LINE", "cried in the top of mine--an excellent play, well"),
                                new XElement("LINE", "digested in the scenes, set down with as much"),
                                new XElement("LINE", "modesty as cunning. I remember, one said there"),
                                new XElement("LINE", "were no sallets in the lines to make the matter"),
                                new XElement("LINE", "savoury, nor no matter in the phrase that might"),
                                new XElement("LINE", "indict the author of affectation; but called it an"),
                                new XElement("LINE", "honest method, as wholesome as sweet, and by very"),
                                new XElement("LINE", "much more handsome than fine. One speech in it I"),
                                new XElement("LINE", "chiefly loved: 'twas Aeneas' tale to Dido; and"),
                                new XElement("LINE", "thereabout of it especially, where he speaks of"),
                                new XElement("LINE", "Priam's slaughter: if it live in your memory, begin"),
                                new XElement("LINE", "at this line: let me see, let me see--"),
                                new XElement("LINE", "'The rugged Pyrrhus, like the Hyrcanian beast,'--"),
                                new XElement("LINE", "it is not so:--it begins with Pyrrhus:--"),
                                new XElement("LINE", "'The rugged Pyrrhus, he whose sable arms,"),
                                new XElement("LINE", "Black as his purpose, did the night resemble"),
                                new XElement("LINE", "When he lay couched in the ominous horse,"),
                                new XElement("LINE", "Hath now this dread and black complexion smear'd"),
                                new XElement("LINE", "With heraldry more dismal; head to foot"),
                                new XElement("LINE", "Now is he total gules; horridly trick'd"),
                                new XElement("LINE", "With blood of fathers, mothers, daughters, sons,"),
                                new XElement("LINE", "Baked and impasted with the parching streets,"),
                                new XElement("LINE", "That lend a tyrannous and damned light"),
                                new XElement("LINE", "To their lord's murder: roasted in wrath and fire,"),
                                new XElement("LINE", "And thus o'er-sized with coagulate gore,"),
                                new XElement("LINE", "With eyes like carbuncles, the hellish Pyrrhus"),
                                new XElement("LINE", "Old grandsire Priam seeks.'"),
                                new XElement("LINE", "So, proceed you.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "'Fore God, my lord, well spoken, with good accent and"),
                                new XElement("LINE", "good discretion.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Player"),
                                new XElement("LINE", "'Anon he finds him"),
                                new XElement("LINE", "Striking too short at Greeks; his antique sword,"),
                                new XElement("LINE", "Rebellious to his arm, lies where it falls,"),
                                new XElement("LINE", "Repugnant to command: unequal match'd,"),
                                new XElement("LINE", "Pyrrhus at Priam drives; in rage strikes wide;"),
                                new XElement("LINE", "But with the whiff and wind of his fell sword"),
                                new XElement("LINE", "The unnerved father falls. Then senseless Ilium,"),
                                new XElement("LINE", "Seeming to feel this blow, with flaming top"),
                                new XElement("LINE", "Stoops to his base, and with a hideous crash"),
                                new XElement("LINE", "Takes prisoner Pyrrhus' ear: for, lo! his sword,"),
                                new XElement("LINE", "Which was declining on the milky head"),
                                new XElement("LINE", "Of reverend Priam, seem'd i' the air to stick:"),
                                new XElement("LINE", "So, as a painted tyrant, Pyrrhus stood,"),
                                new XElement("LINE", "And like a neutral to his will and matter,"),
                                new XElement("LINE", "Did nothing."),
                                new XElement("LINE", "But, as we often see, against some storm,"),
                                new XElement("LINE", "A silence in the heavens, the rack stand still,"),
                                new XElement("LINE", "The bold winds speechless and the orb below"),
                                new XElement("LINE", "As hush as death, anon the dreadful thunder"),
                                new XElement("LINE", "Doth rend the region, so, after Pyrrhus' pause,"),
                                new XElement("LINE", "Aroused vengeance sets him new a-work;"),
                                new XElement("LINE", "And never did the Cyclops' hammers fall"),
                                new XElement("LINE", "On Mars's armour forged for proof eterne"),
                                new XElement("LINE", "With less remorse than Pyrrhus' bleeding sword"),
                                new XElement("LINE", "Now falls on Priam."),
                                new XElement("LINE", "Out, out, thou strumpet, Fortune! All you gods,"),
                                new XElement("LINE", "In general synod 'take away her power;"),
                                new XElement("LINE", "Break all the spokes and fellies from her wheel,"),
                                new XElement("LINE", "And bowl the round nave down the hill of heaven,"),
                                new XElement("LINE", "As low as to the fiends!'")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "This is too long.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "It shall to the barber's, with your beard. Prithee,"),
                                new XElement("LINE", "say on: he's for a jig or a tale of bawdry, or he"),
                                new XElement("LINE", "sleeps: say on: come to Hecuba.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Player"),
                                new XElement("LINE", "'But who, O, who had seen the mobled queen--'")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "'The mobled queen?'")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "That's good; 'mobled queen' is good.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Player"),
                                new XElement("LINE", "'Run barefoot up and down, threatening the flames"),
                                new XElement("LINE", "With bisson rheum; a clout upon that head"),
                                new XElement("LINE", "Where late the diadem stood, and for a robe,"),
                                new XElement("LINE", "About her lank and all o'er-teemed loins,"),
                                new XElement("LINE", "A blanket, in the alarm of fear caught up;"),
                                new XElement("LINE", "Who this had seen, with tongue in venom steep'd,"),
                                new XElement("LINE", "'Gainst Fortune's state would treason have"),
                                new XElement("LINE", "pronounced:"),
                                new XElement("LINE", "But if the gods themselves did see her then"),
                                new XElement("LINE", "When she saw Pyrrhus make malicious sport"),
                                new XElement("LINE", "In mincing with his sword her husband's limbs,"),
                                new XElement("LINE", "The instant burst of clamour that she made,"),
                                new XElement("LINE", "Unless things mortal move them not at all,"),
                                new XElement("LINE", "Would have made milch the burning eyes of heaven,"),
                                new XElement("LINE", "And passion in the gods.'")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Look, whether he has not turned his colour and has"),
                                new XElement("LINE", "tears in's eyes. Pray you, no more.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "'Tis well: I'll have thee speak out the rest soon."),
                                new XElement("LINE", "Good my lord, will you see the players well"),
                                new XElement("LINE", "bestowed? Do you hear, let them be well used; for"),
                                new XElement("LINE", "they are the abstract and brief chronicles of the"),
                                new XElement("LINE", "time: after your death you were better have a bad"),
                                new XElement("LINE", "epitaph than their ill report while you live.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "My lord, I will use them according to their desert.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "God's bodykins, man, much better: use every man"),
                                new XElement("LINE", "after his desert, and who should 'scape whipping?"),
                                new XElement("LINE", "Use them after your own honour and dignity: the less"),
                                new XElement("LINE", "they deserve, the more merit is in your bounty."),
                                new XElement("LINE", "Take them in.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Come, sirs.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Follow him, friends: we'll hear a play to-morrow."),
                                new XElement("STAGEDIR", "Exit POLONIUS with all the Players but the First"),
                                new XElement("LINE", "Dost thou hear me, old friend; can you play the"),
                                new XElement("LINE", "Murder of Gonzago?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Player"),
                                new XElement("LINE", "Ay, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "We'll ha't to-morrow night. You could, for a need,"),
                                new XElement("LINE", "study a speech of some dozen or sixteen lines, which"),
                                new XElement("LINE", "I would set down and insert in't, could you not?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Player"),
                                new XElement("LINE", "Ay, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Very well. Follow that lord; and look you mock him"),
                                new XElement("LINE", "not."),
                                new XElement("STAGEDIR", "Exit First Player"),
                                new XElement("LINE", "My good friends, I'll leave you till night: you are"),
                                new XElement("LINE", "welcome to Elsinore.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Good my lord!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Ay, so, God be wi' ye;"),
                                new XElement("STAGEDIR", "Exeunt ROSENCRANTZ and GUILDENSTERN"),
                                new XElement("LINE", "Now I am alone."),
                                new XElement("LINE", "O, what a rogue and peasant slave am I!"),
                                new XElement("LINE", "Is it not monstrous that this player here,"),
                                new XElement("LINE", "But in a fiction, in a dream of passion,"),
                                new XElement("LINE", "Could force his soul so to his own conceit"),
                                new XElement("LINE", "That from her working all his visage wann'd,"),
                                new XElement("LINE", "Tears in his eyes, distraction in's aspect,"),
                                new XElement("LINE", "A broken voice, and his whole function suiting"),
                                new XElement("LINE", "With forms to his conceit? and all for nothing!"),
                                new XElement("LINE", "For Hecuba!"),
                                new XElement("LINE", "What's Hecuba to him, or he to Hecuba,"),
                                new XElement("LINE", "That he should weep for her? What would he do,"),
                                new XElement("LINE", "Had he the motive and the cue for passion"),
                                new XElement("LINE", "That I have? He would drown the stage with tears"),
                                new XElement("LINE", "And cleave the general ear with horrid speech,"),
                                new XElement("LINE", "Make mad the guilty and appal the free,"),
                                new XElement("LINE", "Confound the ignorant, and amaze indeed"),
                                new XElement("LINE", "The very faculties of eyes and ears. Yet I,"),
                                new XElement("LINE", "A dull and muddy-mettled rascal, peak,"),
                                new XElement("LINE", "Like John-a-dreams, unpregnant of my cause,"),
                                new XElement("LINE", "And can say nothing; no, not for a king,"),
                                new XElement("LINE", "Upon whose property and most dear life"),
                                new XElement("LINE", "A damn'd defeat was made. Am I a coward?"),
                                new XElement("LINE", "Who calls me villain? breaks my pate across?"),
                                new XElement("LINE", "Plucks off my beard, and blows it in my face?"),
                                new XElement("LINE", "Tweaks me by the nose? gives me the lie i' the throat,"),
                                new XElement("LINE", "As deep as to the lungs? who does me this?"),
                                new XElement("LINE", "Ha!"),
                                new XElement("LINE", "'Swounds, I should take it: for it cannot be"),
                                new XElement("LINE", "But I am pigeon-liver'd and lack gall"),
                                new XElement("LINE", "To make oppression bitter, or ere this"),
                                new XElement("LINE", "I should have fatted all the region kites"),
                                new XElement("LINE", "With this slave's offal: bloody, bawdy villain!"),
                                new XElement("LINE", "Remorseless, treacherous, lecherous, kindless villain!"),
                                new XElement("LINE", "O, vengeance!"),
                                new XElement("LINE", "Why, what an ass am I! This is most brave,"),
                                new XElement("LINE", "That I, the son of a dear father murder'd,"),
                                new XElement("LINE", "Prompted to my revenge by heaven and hell,"),
                                new XElement("LINE", "Must, like a whore, unpack my heart with words,"),
                                new XElement("LINE", "And fall a-cursing, like a very drab,"),
                                new XElement("LINE", "A scullion!"),
                                new XElement("LINE", "Fie upon't! foh! About, my brain! I have heard"),
                                new XElement("LINE", "That guilty creatures sitting at a play"),
                                new XElement("LINE", "Have by the very cunning of the scene"),
                                new XElement("LINE", "Been struck so to the soul that presently"),
                                new XElement("LINE", "They have proclaim'd their malefactions;"),
                                new XElement("LINE", "For murder, though it have no tongue, will speak"),
                                new XElement("LINE", "With most miraculous organ. I'll have these players"),
                                new XElement("LINE", "Play something like the murder of my father"),
                                new XElement("LINE", "Before mine uncle: I'll observe his looks;"),
                                new XElement("LINE", "I'll tent him to the quick: if he but blench,"),
                                new XElement("LINE", "I know my course. The spirit that I have seen"),
                                new XElement("LINE", "May be the devil: and the devil hath power"),
                                new XElement("LINE", "To assume a pleasing shape; yea, and perhaps"),
                                new XElement("LINE", "Out of my weakness and my melancholy,"),
                                new XElement("LINE", "As he is very potent with such spirits,"),
                                new XElement("LINE", "Abuses me to damn me: I'll have grounds"),
                                new XElement("LINE", "More relative than this: the play 's the thing"),
                                new XElement("LINE", "Wherein I'll catch the conscience of the king.")
                            ),
                            new XElement("STAGEDIR", "Exit")
                        )
                    ),
                    new XElement("ACT",
                        new XElement("TITLE", "ACT III"),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE I.  A room in the castle."),
                            new XElement("STAGEDIR",
                                "Enter KING CLAUDIUS, QUEEN GERTRUDE, POLONIUS,\n" +
                                "OPHELIA, ROSENCRANTZ, and GUILDENSTERN"
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "And can you, by no drift of circumstance,"),
                                new XElement("LINE", "Get from him why he puts on this confusion,"),
                                new XElement("LINE", "Grating so harshly all his days of quiet"),
                                new XElement("LINE", "With turbulent and dangerous lunacy?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "He does confess he feels himself distracted;"),
                                new XElement("LINE", "But from what cause he will by no means speak.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "Nor do we find him forward to be sounded,"),
                                new XElement("LINE", "But, with a crafty madness, keeps aloof,"),
                                new XElement("LINE", "When we would bring him on to some confession"),
                                new XElement("LINE", "Of his true state.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Did he receive you well?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Most like a gentleman.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "But with much forcing of his disposition.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Niggard of question; but, of our demands,"),
                                new XElement("LINE", "Most free in his reply.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Did you assay him?"),
                                new XElement("LINE", "To any pastime?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Madam, it so fell out, that certain players"),
                                new XElement("LINE", "We o'er-raught on the way: of these we told him;"),
                                new XElement("LINE", "And there did seem in him a kind of joy"),
                                new XElement("LINE", "To hear of it: they are about the court,"),
                                new XElement("LINE", "And, as I think, they have already order"),
                                new XElement("LINE", "This night to play before him.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "'Tis most true:"),
                                new XElement("LINE", "And he beseech'd me to entreat your majesties"),
                                new XElement("LINE", "To hear and see the matter.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "With all my heart; and it doth much content me"),
                                new XElement("LINE", "To hear him so inclined."),
                                new XElement("LINE", "Good gentlemen, give him a further edge,"),
                                new XElement("LINE", "And drive his purpose on to these delights.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "We shall, my lord.")
                            ),
                            new XElement("STAGEDIR", "Exeunt ROSENCRANTZ and GUILDENSTERN"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Sweet Gertrude, leave us too;"),
                                new XElement("LINE", "For we have closely sent for Hamlet hither,"),
                                new XElement("LINE", "That he, as 'twere by accident, may here"),
                                new XElement("LINE", "Affront Ophelia:"),
                                new XElement("LINE", "Her father and myself, lawful espials,"),
                                new XElement("LINE", "Will so bestow ourselves that, seeing, unseen,"),
                                new XElement("LINE", "We may of their encounter frankly judge,"),
                                new XElement("LINE", "And gather by him, as he is behaved,"),
                                new XElement("LINE", "If 't be the affliction of his love or no"),
                                new XElement("LINE", "That thus he suffers for.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "I shall obey you."),
                                new XElement("LINE", "And for your part, Ophelia, I do wish"),
                                new XElement("LINE", "That your good beauties be the happy cause"),
                                new XElement("LINE", "Of Hamlet's wildness: so shall I hope your virtues"),
                                new XElement("LINE", "Will bring him to his wonted way again,"),
                                new XElement("LINE", "To both your honours.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "Madam, I wish it may.")
                            ),
                            new XElement("STAGEDIR", "Exit QUEEN GERTRUDE"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Ophelia, walk you here. Gracious, so please you,"),
                                new XElement("LINE", "We will bestow ourselves."),
                                new XElement("STAGEDIR", "To OPHELIA"),
                                new XElement("LINE", "Read on this book;"),
                                new XElement("LINE", "That show of such an exercise may colour"),
                                new XElement("LINE", "Your loneliness. We are oft to blame in this,--"),
                                new XElement("LINE", "'Tis too much proved--that with devotion's visage"),
                                new XElement("LINE", "And pious action we do sugar o'er"),
                                new XElement("LINE", "The devil himself.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Aside"),
                                    "          O, 'tis too true!"
                                ),
                                new XElement("LINE", "How smart a lash that speech doth give my conscience!"),
                                new XElement("LINE", "The harlot's cheek, beautied with plastering art,"),
                                new XElement("LINE", "Is not more ugly to the thing that helps it"),
                                new XElement("LINE", "Than is my deed to my most painted word:"),
                                new XElement("LINE", "O heavy burthen!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "I hear him coming: let's withdraw, my lord.")
                            ),
                            new XElement("STAGEDIR", "Exeunt KING CLAUDIUS and POLONIUS"),
                            new XElement("STAGEDIR", "Enter HAMLET"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "To be, or not to be: that is the question:"),
                                new XElement("LINE", "Whether 'tis nobler in the mind to suffer"),
                                new XElement("LINE", "The slings and arrows of outrageous fortune,"),
                                new XElement("LINE", "Or to take arms against a sea of troubles,"),
                                new XElement("LINE", "And by opposing end them? To die: to sleep;"),
                                new XElement("LINE", "No more; and by a sleep to say we end"),
                                new XElement("LINE", "The heart-ache and the thousand natural shocks"),
                                new XElement("LINE", "That flesh is heir to, 'tis a consummation"),
                                new XElement("LINE", "Devoutly to be wish'd. To die, to sleep;"),
                                new XElement("LINE", "To sleep: perchance to dream: ay, there's the rub;"),
                                new XElement("LINE", "For in that sleep of death what dreams may come"),
                                new XElement("LINE", "When we have shuffled off this mortal coil,"),
                                new XElement("LINE", "Must give us pause: there's the respect"),
                                new XElement("LINE", "That makes calamity of so long life;"),
                                new XElement("LINE", "For who would bear the whips and scorns of time,"),
                                new XElement("LINE", "The oppressor's wrong, the proud man's contumely,"),
                                new XElement("LINE", "The pangs of despised love, the law's delay,"),
                                new XElement("LINE", "The insolence of office and the spurns"),
                                new XElement("LINE", "That patient merit of the unworthy takes,"),
                                new XElement("LINE", "When he himself might his quietus make"),
                                new XElement("LINE", "With a bare bodkin? who would fardels bear,"),
                                new XElement("LINE", "To grunt and sweat under a weary life,"),
                                new XElement("LINE", "But that the dread of something after death,"),
                                new XElement("LINE", "The undiscover'd country from whose bourn"),
                                new XElement("LINE", "No traveller returns, puzzles the will"),
                                new XElement("LINE", "And makes us rather bear those ills we have"),
                                new XElement("LINE", "Than fly to others that we know not of?"),
                                new XElement("LINE", "Thus conscience does make cowards of us all;"),
                                new XElement("LINE", "And thus the native hue of resolution"),
                                new XElement("LINE", "Is sicklied o'er with the pale cast of thought,"),
                                new XElement("LINE", "And enterprises of great pith and moment"),
                                new XElement("LINE", "With this regard their currents turn awry,"),
                                new XElement("LINE", "And lose the name of action.--Soft you now!"),
                                new XElement("LINE", "The fair Ophelia! Nymph, in thy orisons"),
                                new XElement("LINE", "Be all my sins remember'd.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "Good my lord,"),
                                new XElement("LINE", "How does your honour for this many a day?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I humbly thank you; well, well, well.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "My lord, I have remembrances of yours,"),
                                new XElement("LINE", "That I have longed long to re-deliver;"),
                                new XElement("LINE", "I pray you, now receive them.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "No, not I;"),
                                new XElement("LINE", "I never gave you aught.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "My honour'd lord, you know right well you did;"),
                                new XElement("LINE", "And, with them, words of so sweet breath composed"),
                                new XElement("LINE", "As made the things more rich: their perfume lost,"),
                                new XElement("LINE", "Take these again; for to the noble mind"),
                                new XElement("LINE", "Rich gifts wax poor when givers prove unkind."),
                                new XElement("LINE", "There, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Ha, ha! are you honest?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "My lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Are you fair?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "What means your lordship?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "That if you be honest and fair, your honesty should"),
                                new XElement("LINE", "admit no discourse to your beauty.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "Could beauty, my lord, have better commerce than"),
                                new XElement("LINE", "with honesty?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Ay, truly; for the power of beauty will sooner"),
                                new XElement("LINE", "transform honesty from what it is to a bawd than the"),
                                new XElement("LINE", "force of honesty can translate beauty into his"),
                                new XElement("LINE", "likeness: this was sometime a paradox, but now the"),
                                new XElement("LINE", "time gives it proof. I did love you once.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "Indeed, my lord, you made me believe so.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "You should not have believed me; for virtue cannot"),
                                new XElement("LINE", "so inoculate our old stock but we shall relish of"),
                                new XElement("LINE", "it: I loved you not.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "I was the more deceived.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Get thee to a nunnery: why wouldst thou be a"),
                                new XElement("LINE", "breeder of sinners? I am myself indifferent honest;"),
                                new XElement("LINE", "but yet I could accuse me of such things that it"),
                                new XElement("LINE", "were better my mother had not borne me: I am very"),
                                new XElement("LINE", "proud, revengeful, ambitious, with more offences at"),
                                new XElement("LINE", "my beck than I have thoughts to put them in,"),
                                new XElement("LINE", "imagination to give them shape, or time to act them"),
                                new XElement("LINE", "in. What should such fellows as I do crawling"),
                                new XElement("LINE", "between earth and heaven? We are arrant knaves,"),
                                new XElement("LINE", "all; believe none of us. Go thy ways to a nunnery."),
                                new XElement("LINE", "Where's your father?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "At home, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Let the doors be shut upon him, that he may play the"),
                                new XElement("LINE", "fool no where but in's own house. Farewell.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "O, help him, you sweet heavens!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "If thou dost marry, I'll give thee this plague for"),
                                new XElement("LINE", "thy dowry: be thou as chaste as ice, as pure as"),
                                new XElement("LINE", "snow, thou shalt not escape calumny. Get thee to a"),
                                new XElement("LINE", "nunnery, go: farewell. Or, if thou wilt needs"),
                                new XElement("LINE", "marry, marry a fool; for wise men know well enough"),
                                new XElement("LINE", "what monsters you make of them. To a nunnery, go,"),
                                new XElement("LINE", "and quickly too. Farewell.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "O heavenly powers, restore him!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I have heard of your paintings too, well enough; God"),
                                new XElement("LINE", "has given you one face, and you make yourselves"),
                                new XElement("LINE", "another: you jig, you amble, and you lisp, and"),
                                new XElement("LINE", "nick-name God's creatures, and make your wantonness"),
                                new XElement("LINE", "your ignorance. Go to, I'll no more on't; it hath"),
                                new XElement("LINE", "made me mad. I say, we will have no more marriages:"),
                                new XElement("LINE", "those that are married already, all but one, shall"),
                                new XElement("LINE", "live; the rest shall keep as they are. To a"),
                                new XElement("LINE", "nunnery, go.")
                            ),
                            new XElement("STAGEDIR", "Exit"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "O, what a noble mind is here o'erthrown!"),
                                new XElement("LINE", "The courtier's, soldier's, scholar's, eye, tongue, sword;"),
                                new XElement("LINE", "The expectancy and rose of the fair state,"),
                                new XElement("LINE", "The glass of fashion and the mould of form,"),
                                new XElement("LINE", "The observed of all observers, quite, quite down!"),
                                new XElement("LINE", "And I, of ladies most deject and wretched,"),
                                new XElement("LINE", "That suck'd the honey of his music vows,"),
                                new XElement("LINE", "Now see that noble and most sovereign reason,"),
                                new XElement("LINE", "Like sweet bells jangled, out of tune and harsh;"),
                                new XElement("LINE", "That unmatch'd form and feature of blown youth"),
                                new XElement("LINE", "Blasted with ecstasy: O, woe is me,"),
                                new XElement("LINE", "To have seen what I have seen, see what I see!")
                            ),
                            new XElement("STAGEDIR", "Re-enter KING CLAUDIUS and POLONIUS"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Love! his affections do not that way tend;"),
                                new XElement("LINE", "Nor what he spake, though it lack'd form a little,"),
                                new XElement("LINE", "Was not like madness. There's something in his soul,"),
                                new XElement("LINE", "O'er which his melancholy sits on brood;"),
                                new XElement("LINE", "And I do doubt the hatch and the disclose"),
                                new XElement("LINE", "Will be some danger: which for to prevent,"),
                                new XElement("LINE", "I have in quick determination"),
                                new XElement("LINE", "Thus set it down: he shall with speed to England,"),
                                new XElement("LINE", "For the demand of our neglected tribute"),
                                new XElement("LINE", "Haply the seas and countries different"),
                                new XElement("LINE", "With variable objects shall expel"),
                                new XElement("LINE", "This something-settled matter in his heart,"),
                                new XElement("LINE", "Whereon his brains still beating puts him thus"),
                                new XElement("LINE", "From fashion of himself. What think you on't?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "It shall do well: but yet do I believe"),
                                new XElement("LINE", "The origin and commencement of his grief"),
                                new XElement("LINE", "Sprung from neglected love. How now, Ophelia!"),
                                new XElement("LINE", "You need not tell us what Lord Hamlet said;"),
                                new XElement("LINE", "We heard it all. My lord, do as you please;"),
                                new XElement("LINE", "But, if you hold it fit, after the play"),
                                new XElement("LINE", "Let his queen mother all alone entreat him"),
                                new XElement("LINE", "To show his grief: let her be round with him;"),
                                new XElement("LINE", "And I'll be placed, so please you, in the ear"),
                                new XElement("LINE", "Of all their conference. If she find him not,"),
                                new XElement("LINE", "To England send him, or confine him where"),
                                new XElement("LINE", "Your wisdom best shall think.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "It shall be so:"),
                                new XElement("LINE", "Madness in great ones must not unwatch'd go.")
                            ),
                            new XElement("STAGEDIR", "Exeunt")
                        ),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE II.  A hall in the castle."),
                            new XElement("STAGEDIR", "Enter HAMLET and Players"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Speak the speech, I pray you, as I pronounced it to"),
                                new XElement("LINE", "you, trippingly on the tongue: but if you mouth it,"),
                                new XElement("LINE", "as many of your players do, I had as lief the"),
                                new XElement("LINE", "town-crier spoke my lines. Nor do not saw the air"),
                                new XElement("LINE", "too much with your hand, thus, but use all gently;"),
                                new XElement("LINE", "for in the very torrent, tempest, and, as I may say,"),
                                new XElement("LINE", "the whirlwind of passion, you must acquire and beget"),
                                new XElement("LINE", "a temperance that may give it smoothness. O, it"),
                                new XElement("LINE", "offends me to the soul to hear a robustious"),
                                new XElement("LINE", "periwig-pated fellow tear a passion to tatters, to"),
                                new XElement("LINE", "very rags, to split the ears of the groundlings, who"),
                                new XElement("LINE", "for the most part are capable of nothing but"),
                                new XElement("LINE", "inexplicable dumbshows and noise: I would have such"),
                                new XElement("LINE", "a fellow whipped for o'erdoing Termagant; it"),
                                new XElement("LINE", "out-herods Herod: pray you, avoid it.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Player"),
                                new XElement("LINE", "I warrant your honour.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Be not too tame neither, but let your own discretion"),
                                new XElement("LINE", "be your tutor: suit the action to the word, the"),
                                new XElement("LINE", "word to the action; with this special o'erstep not"),
                                new XElement("LINE", "the modesty of nature: for any thing so overdone is"),
                                new XElement("LINE", "from the purpose of playing, whose end, both at the"),
                                new XElement("LINE", "first and now, was and is, to hold, as 'twere, the"),
                                new XElement("LINE", "mirror up to nature; to show virtue her own feature,"),
                                new XElement("LINE", "scorn her own image, and the very age and body of"),
                                new XElement("LINE", "the time his form and pressure. Now this overdone,"),
                                new XElement("LINE", "or come tardy off, though it make the unskilful"),
                                new XElement("LINE", "laugh, cannot but make the judicious grieve; the"),
                                new XElement("LINE", "censure of the which one must in your allowance"),
                                new XElement("LINE", "o'erweigh a whole theatre of others. O, there be"),
                                new XElement("LINE", "players that I have seen play, and heard others"),
                                new XElement("LINE", "praise, and that highly, not to speak it profanely,"),
                                new XElement("LINE", "that, neither having the accent of Christians nor"),
                                new XElement("LINE", "the gait of Christian, pagan, nor man, have so"),
                                new XElement("LINE", "strutted and bellowed that I have thought some of"),
                                new XElement("LINE", "nature's journeymen had made men and not made them"),
                                new XElement("LINE", "well, they imitated humanity so abominably.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Player"),
                                new XElement("LINE", "I hope we have reformed that indifferently with us,"),
                                new XElement("LINE", "sir.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "O, reform it altogether. And let those that play"),
                                new XElement("LINE", "your clowns speak no more than is set down for them;"),
                                new XElement("LINE", "for there be of them that will themselves laugh, to"),
                                new XElement("LINE", "set on some quantity of barren spectators to laugh"),
                                new XElement("LINE", "too; though, in the mean time, some necessary"),
                                new XElement("LINE", "question of the play be then to be considered:"),
                                new XElement("LINE", "that's villanous, and shows a most pitiful ambition"),
                                new XElement("LINE", "in the fool that uses it. Go, make you ready."),
                                new XElement("STAGEDIR", "Exeunt Players"),
                                new XElement("STAGEDIR", "Enter POLONIUS, ROSENCRANTZ, and GUILDENSTERN"),
                                new XElement("LINE", "How now, my lord! I will the king hear this piece of work?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "And the queen too, and that presently.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Bid the players make haste."),
                                new XElement("STAGEDIR", "Exit POLONIUS"),
                                new XElement("LINE", "Will you two help to hasten them?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "We will, my lord.")
                            ),
                            new XElement("STAGEDIR", "Exeunt ROSENCRANTZ and GUILDENSTERN"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "What ho! Horatio!")
                            ),
                            new XElement("STAGEDIR", "Enter HORATIO"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Here, sweet lord, at your service.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Horatio, thou art e'en as just a man"),
                                new XElement("LINE", "As e'er my conversation coped withal.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "O, my dear lord,--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Nay, do not think I flatter;"),
                                new XElement("LINE", "For what advancement may I hope from thee"),
                                new XElement("LINE", "That no revenue hast but thy good spirits,"),
                                new XElement("LINE", "To feed and clothe thee? Why should the poor be flatter'd?"),
                                new XElement("LINE", "No, let the candied tongue lick absurd pomp,"),
                                new XElement("LINE", "And crook the pregnant hinges of the knee"),
                                new XElement("LINE", "Where thrift may follow fawning. Dost thou hear?"),
                                new XElement("LINE", "Since my dear soul was mistress of her choice"),
                                new XElement("LINE", "And could of men distinguish, her election"),
                                new XElement("LINE", "Hath seal'd thee for herself; for thou hast been"),
                                new XElement("LINE", "As one, in suffering all, that suffers nothing,"),
                                new XElement("LINE", "A man that fortune's buffets and rewards"),
                                new XElement("LINE", "Hast ta'en with equal thanks: and blest are those"),
                                new XElement("LINE", "Whose blood and judgment are so well commingled,"),
                                new XElement("LINE", "That they are not a pipe for fortune's finger"),
                                new XElement("LINE", "To sound what stop she please. Give me that man"),
                                new XElement("LINE", "That is not passion's slave, and I will wear him"),
                                new XElement("LINE", "In my heart's core, ay, in my heart of heart,"),
                                new XElement("LINE", "As I do thee.--Something too much of this.--"),
                                new XElement("LINE", "There is a play to-night before the king;"),
                                new XElement("LINE", "One scene of it comes near the circumstance"),
                                new XElement("LINE", "Which I have told thee of my father's death:"),
                                new XElement("LINE", "I prithee, when thou seest that act afoot,"),
                                new XElement("LINE", "Even with the very comment of thy soul"),
                                new XElement("LINE", "Observe mine uncle: if his occulted guilt"),
                                new XElement("LINE", "Do not itself unkennel in one speech,"),
                                new XElement("LINE", "It is a damned ghost that we have seen,"),
                                new XElement("LINE", "And my imaginations are as foul"),
                                new XElement("LINE", "As Vulcan's stithy. Give him heedful note;"),
                                new XElement("LINE", "For I mine eyes will rivet to his face,"),
                                new XElement("LINE", "And after we will both our judgments join"),
                                new XElement("LINE", "In censure of his seeming.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Well, my lord:"),
                                new XElement("LINE", "If he steal aught the whilst this play is playing,"),
                                new XElement("LINE", "And 'scape detecting, I will pay the theft.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "They are coming to the play; I must be idle:"),
                                new XElement("LINE", "Get you a place.")
                            ),
                            new XElement("STAGEDIR",
                                "Danish march. A flourish. Enter KING CLAUDIUS,\n" +
                                "QUEEN GERTRUDE, POLONIUS, OPHELIA, ROSENCRANTZ,\n" +
                                "GUILDENSTERN, and others"
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "How fares our cousin Hamlet?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Excellent, i' faith; of the chameleon's dish: I eat"),
                                new XElement("LINE", "the air, promise-crammed: you cannot feed capons so.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "I have nothing with this answer, Hamlet; these words"),
                                new XElement("LINE", "are not mine.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "No, nor mine now."),
                                new XElement("STAGEDIR", "To POLONIUS"),
                                new XElement("LINE", "My lord, you played once i' the university, you say?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "That did I, my lord; and was accounted a good actor.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "What did you enact?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "I did enact Julius Caesar: I was killed i' the"),
                                new XElement("LINE", "Capitol; Brutus killed me.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "It was a brute part of him to kill so capital a calf"),
                                new XElement("LINE", "there. Be the players ready?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Ay, my lord; they stay upon your patience.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Come hither, my dear Hamlet, sit by me.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "No, good mother, here's metal more attractive.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "To KING CLAUDIUS"),
                                    "  O, ho! do you mark that?"
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Lady, shall I lie in your lap?")
                            ),
                            new XElement("STAGEDIR", "Lying down at OPHELIA's feet"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "No, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I mean, my head upon your lap?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "Ay, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Do you think I meant country matters?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "I think nothing, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "That's a fair thought to lie between maids' legs.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "What is, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Nothing.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "You are merry, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Who, I?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "Ay, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "O God, your only jig-maker. What should a man do"),
                                new XElement("LINE", "but be merry? for, look you, how cheerfully my"),
                                new XElement("LINE", "mother looks, and my father died within these two hours.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "Nay, 'tis twice two months, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "So long? Nay then, let the devil wear black, for"),
                                new XElement("LINE", "I'll have a suit of sables. O heavens! die two"),
                                new XElement("LINE", "months ago, and not forgotten yet? Then there's"),
                                new XElement("LINE", "hope a great man's memory may outlive his life half"),
                                new XElement("LINE", "a year: but, by'r lady, he must build churches,"),
                                new XElement("LINE", "then; or else shall he suffer not thinking on, with"),
                                new XElement("LINE", "the hobby-horse, whose epitaph is 'For, O, for, O,"),
                                new XElement("LINE", "the hobby-horse is forgot.'"),
                                new XElement("STAGEDIR", "Hautboys play. The dumb-show enters")
                            ),
                            new XElement("STAGEDIR",
                                "Enter a King and a Queen very lovingly; the Queen\n" +
                                "embracing him, and he her. She kneels, and makes\n" +
                                "show of protestation unto him. He takes her up,\n" +
                                "and declines his head upon her neck: lays him down\n" +
                                "upon a bank of flowers: she, seeing him asleep,\n" +
                                "leaves him. Anon comes in a fellow, takes off his\n" +
                                "crown, kisses it, and pours poison in the King's\n" +
                                "ears, and exit. The Queen returns; finds the King\n" +
                                "dead, and makes passionate action. The Poisoner,\n" +
                                "with some two or three Mutes, comes in again,\n" +
                                "seeming to lament with her. The dead body is\n" +
                                "carried away. The Poisoner wooes the Queen with\n" +
                                "gifts: she seems loath and unwilling awhile, but\n" +
                                "in the end accepts his love"
                            ),
                            new XElement("STAGEDIR", "Exeunt"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "What means this, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Marry, this is miching mallecho; it means mischief.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "Belike this show imports the argument of the play.")
                            ),
                            new XElement("STAGEDIR", "Enter Prologue"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "We shall know by this fellow: the players cannot"),
                                new XElement("LINE", "keep counsel; they'll tell all.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "Will he tell us what this show meant?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Ay, or any show that you'll show him: be not you"),
                                new XElement("LINE", "ashamed to show, he'll not shame to tell you what it means.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "You are naught, you are naught: I'll mark the play.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Prologue"),
                                new XElement("LINE", "For us, and for our tragedy,"),
                                new XElement("LINE", "Here stooping to your clemency,"),
                                new XElement("LINE", "We beg your hearing patiently.")
                            ),
                            new XElement("STAGEDIR", "Exit"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Is this a prologue, or the posy of a ring?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "'Tis brief, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "As woman's love.")
                            ),
                            new XElement("STAGEDIR", "Enter two Players, King and Queen"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Player King"),
                                new XElement("LINE", "Full thirty times hath Phoebus' cart gone round"),
                                new XElement("LINE", "Neptune's salt wash and Tellus' orbed ground,"),
                                new XElement("LINE", "And thirty dozen moons with borrow'd sheen"),
                                new XElement("LINE", "About the world have times twelve thirties been,"),
                                new XElement("LINE", "Since love our hearts and Hymen did our hands"),
                                new XElement("LINE", "Unite commutual in most sacred bands.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Player Queen"),
                                new XElement("LINE", "So many journeys may the sun and moon"),
                                new XElement("LINE", "Make us again count o'er ere love be done!"),
                                new XElement("LINE", "But, woe is me, you are so sick of late,"),
                                new XElement("LINE", "So far from cheer and from your former state,"),
                                new XElement("LINE", "That I distrust you. Yet, though I distrust,"),
                                new XElement("LINE", "Discomfort you, my lord, it nothing must:"),
                                new XElement("LINE", "For women's fear and love holds quantity;"),
                                new XElement("LINE", "In neither aught, or in extremity."),
                                new XElement("LINE", "Now, what my love is, proof hath made you know;"),
                                new XElement("LINE", "And as my love is sized, my fear is so:"),
                                new XElement("LINE", "Where love is great, the littlest doubts are fear;"),
                                new XElement("LINE", "Where little fears grow great, great love grows there.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Player King"),
                                new XElement("LINE", "'Faith, I must leave thee, love, and shortly too;"),
                                new XElement("LINE", "My operant powers their functions leave to do:"),
                                new XElement("LINE", "And thou shalt live in this fair world behind,"),
                                new XElement("LINE", "Honour'd, beloved; and haply one as kind"),
                                new XElement("LINE", "For husband shalt thou--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Player Queen"),
                                new XElement("LINE", "O, confound the rest!"),
                                new XElement("LINE", "Such love must needs be treason in my breast:"),
                                new XElement("LINE", "In second husband let me be accurst!"),
                                new XElement("LINE", "None wed the second but who kill'd the first.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Aside"),
                                    "  Wormwood, wormwood."
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Player Queen"),
                                new XElement("LINE", "The instances that second marriage move"),
                                new XElement("LINE", "Are base respects of thrift, but none of love:"),
                                new XElement("LINE", "A second time I kill my husband dead,"),
                                new XElement("LINE", "When second husband kisses me in bed.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Player King"),
                                new XElement("LINE", "I do believe you think what now you speak;"),
                                new XElement("LINE", "But what we do determine oft we break."),
                                new XElement("LINE", "Purpose is but the slave to memory,"),
                                new XElement("LINE", "Of violent birth, but poor validity;"),
                                new XElement("LINE", "Which now, like fruit unripe, sticks on the tree;"),
                                new XElement("LINE", "But fall, unshaken, when they mellow be."),
                                new XElement("LINE", "Most necessary 'tis that we forget"),
                                new XElement("LINE", "To pay ourselves what to ourselves is debt:"),
                                new XElement("LINE", "What to ourselves in passion we propose,"),
                                new XElement("LINE", "The passion ending, doth the purpose lose."),
                                new XElement("LINE", "The violence of either grief or joy"),
                                new XElement("LINE", "Their own enactures with themselves destroy:"),
                                new XElement("LINE", "Where joy most revels, grief doth most lament;"),
                                new XElement("LINE", "Grief joys, joy grieves, on slender accident."),
                                new XElement("LINE", "This world is not for aye, nor 'tis not strange"),
                                new XElement("LINE", "That even our loves should with our fortunes change;"),
                                new XElement("LINE", "For 'tis a question left us yet to prove,"),
                                new XElement("LINE", "Whether love lead fortune, or else fortune love."),
                                new XElement("LINE", "The great man down, you mark his favourite flies;"),
                                new XElement("LINE", "The poor advanced makes friends of enemies."),
                                new XElement("LINE", "And hitherto doth love on fortune tend;"),
                                new XElement("LINE", "For who not needs shall never lack a friend,"),
                                new XElement("LINE", "And who in want a hollow friend doth try,"),
                                new XElement("LINE", "Directly seasons him his enemy."),
                                new XElement("LINE", "But, orderly to end where I begun,"),
                                new XElement("LINE", "Our wills and fates do so contrary run"),
                                new XElement("LINE", "That our devices still are overthrown;"),
                                new XElement("LINE", "Our thoughts are ours, their ends none of our own:"),
                                new XElement("LINE", "So think thou wilt no second husband wed;"),
                                new XElement("LINE", "But die thy thoughts when thy first lord is dead.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Player Queen"),
                                new XElement("LINE", "Nor earth to me give food, nor heaven light!"),
                                new XElement("LINE", "Sport and repose lock from me day and night!"),
                                new XElement("LINE", "To desperation turn my trust and hope!"),
                                new XElement("LINE", "An anchor's cheer in prison be my scope!"),
                                new XElement("LINE", "Each opposite that blanks the face of joy"),
                                new XElement("LINE", "Meet what I would have well and it destroy!"),
                                new XElement("LINE", "Both here and hence pursue me lasting strife,"),
                                new XElement("LINE", "If, once a widow, ever I be wife!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "If she should break it now!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Player King"),
                                new XElement("LINE", "'Tis deeply sworn. Sweet, leave me here awhile;"),
                                new XElement("LINE", "My spirits grow dull, and fain I would beguile"),
                                new XElement("LINE", "The tedious day with sleep.")
                            ),
                            new XElement("STAGEDIR", "Sleeps"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Player Queen"),
                                new XElement("LINE", "Sleep rock thy brain,"),
                                new XElement("LINE", "And never come mischance between us twain!")
                            ),
                            new XElement("STAGEDIR", "Exit"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Madam, how like you this play?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "The lady protests too much, methinks.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "O, but she'll keep her word.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Have you heard the argument? Is there no offence in 't?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "No, no, they do but jest, poison in jest; no offence"),
                                new XElement("LINE", "i' the world.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "What do you call the play?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "The Mouse-trap. Marry, how? Tropically. This play"),
                                new XElement("LINE", "is the image of a murder done in Vienna: Gonzago is"),
                                new XElement("LINE", "the duke's name; his wife, Baptista: you shall see"),
                                new XElement("LINE", "anon; 'tis a knavish piece of work: but what o'"),
                                new XElement("LINE", "that? your majesty and we that have free souls, it"),
                                new XElement("LINE", "touches us not: let the galled jade wince, our"),
                                new XElement("LINE", "withers are unwrung."),
                                new XElement("STAGEDIR", "Enter LUCIANUS"),
                                new XElement("LINE", "This is one Lucianus, nephew to the king.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "You are as good as a chorus, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I could interpret between you and your love, if I"),
                                new XElement("LINE", "could see the puppets dallying.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "You are keen, my lord, you are keen.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "It would cost you a groaning to take off my edge.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "Still better, and worse.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "So you must take your husbands. Begin, murderer;"),
                                new XElement("LINE", "pox, leave thy damnable faces, and begin. Come:"),
                                new XElement("LINE", "'the croaking raven doth bellow for revenge.'")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LUCIANUS"),
                                new XElement("LINE", "Thoughts black, hands apt, drugs fit, and time agreeing;"),
                                new XElement("LINE", "Confederate season, else no creature seeing;"),
                                new XElement("LINE", "Thou mixture rank, of midnight weeds collected,"),
                                new XElement("LINE", "With Hecate's ban thrice blasted, thrice infected,"),
                                new XElement("LINE", "Thy natural magic and dire property,"),
                                new XElement("LINE", "On wholesome life usurp immediately.")
                            ),
                            new XElement("STAGEDIR", "Pours the poison into the sleeper's ears"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "He poisons him i' the garden for's estate. His"),
                                new XElement("LINE", "name's Gonzago: the story is extant, and writ in"),
                                new XElement("LINE", "choice Italian: you shall see anon how the murderer"),
                                new XElement("LINE", "gets the love of Gonzago's wife.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "The king rises.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "What, frighted with false fire!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "How fares my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Give o'er the play.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Give me some light: away!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "All"),
                                new XElement("LINE", "Lights, lights, lights!")
                            ),
                            new XElement("STAGEDIR", "Exeunt all but HAMLET and HORATIO"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Why, let the stricken deer go weep,"),
                                new XElement("LINE", "The hart ungalled play;"),
                                new XElement("LINE", "For some must watch, while some must sleep:"),
                                new XElement("LINE", "So runs the world away."),
                                new XElement("LINE", "Would not this, sir, and a forest of feathers-- if"),
                                new XElement("LINE", "the rest of my fortunes turn Turk with me--with two"),
                                new XElement("LINE", "Provincial roses on my razed shoes, get me a"),
                                new XElement("LINE", "fellowship in a cry of players, sir?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Half a share.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "A whole one, I."),
                                new XElement("LINE", "For thou dost know, O Damon dear,"),
                                new XElement("LINE", "This realm dismantled was"),
                                new XElement("LINE", "Of Jove himself; and now reigns here"),
                                new XElement("LINE", "A very, very--pajock.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "You might have rhymed.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "O good Horatio, I'll take the ghost's word for a"),
                                new XElement("LINE", "thousand pound. Didst perceive?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Very well, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Upon the talk of the poisoning?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "I did very well note him.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Ah, ha! Come, some music! come, the recorders!"),
                                new XElement("LINE", "For if the king like not the comedy,"),
                                new XElement("LINE", "Why then, belike, he likes it not, perdy."),
                                new XElement("LINE", "Come, some music!")
                            ),
                            new XElement("STAGEDIR", "Re-enter ROSENCRANTZ and GUILDENSTERN"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "Good my lord, vouchsafe me a word with you.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Sir, a whole history.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "The king, sir,--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Ay, sir, what of him?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "Is in his retirement marvellous distempered.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "With drink, sir?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "No, my lord, rather with choler.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Your wisdom should show itself more richer to"),
                                new XElement("LINE", "signify this to his doctor; for, for me to put him"),
                                new XElement("LINE", "to his purgation would perhaps plunge him into far"),
                                new XElement("LINE", "more choler.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "Good my lord, put your discourse into some frame and"),
                                new XElement("LINE", "start not so wildly from my affair.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I am tame, sir: pronounce.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "The queen, your mother, in most great affliction of"),
                                new XElement("LINE", "spirit, hath sent me to you.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "You are welcome.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "Nay, good my lord, this courtesy is not of the right"),
                                new XElement("LINE", "breed. If it shall please you to make me a"),
                                new XElement("LINE", "wholesome answer, I will do your mother's"),
                                new XElement("LINE", "commandment: if not, your pardon and my return"),
                                new XElement("LINE", "shall be the end of my business.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Sir, I cannot.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "What, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Make you a wholesome answer; my wit's diseased: but,"),
                                new XElement("LINE", "sir, such answer as I can make, you shall command;"),
                                new XElement("LINE", "or, rather, as you say, my mother: therefore no"),
                                new XElement("LINE", "more, but to the matter: my mother, you say,--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Then thus she says; your behavior hath struck her"),
                                new XElement("LINE", "into amazement and admiration.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "O wonderful son, that can so astonish a mother! But"),
                                new XElement("LINE", "is there no sequel at the heels of this mother's"),
                                new XElement("LINE", "admiration? Impart.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "She desires to speak with you in her closet, ere you"),
                                new XElement("LINE", "go to bed.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "We shall obey, were she ten times our mother. Have"),
                                new XElement("LINE", "you any further trade with us?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "My lord, you once did love me.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "So I do still, by these pickers and stealers.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Good my lord, what is your cause of distemper? you"),
                                new XElement("LINE", "do, surely, bar the door upon your own liberty, if"),
                                new XElement("LINE", "you deny your griefs to your friend.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Sir, I lack advancement.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "How can that be, when you have the voice of the king"),
                                new XElement("LINE", "himself for your succession in Denmark?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Ay, but sir, 'While the grass grows,'--the proverb"),
                                new XElement("LINE", "is something musty."),
                                new XElement("STAGEDIR", "Re-enter Players with recorders"),
                                new XElement("LINE", "O, the recorders! let me see one. To withdraw with"),
                                new XElement("LINE", "you:--why do you go about to recover the wind of me,"),
                                new XElement("LINE", "as if you would drive me into a toil?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "O, my lord, if my duty be too bold, my love is too"),
                                new XElement("LINE", "unmannerly.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I do not well understand that. Will you play upon"),
                                new XElement("LINE", "this pipe?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "My lord, I cannot.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I pray you.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "Believe me, I cannot.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I do beseech you.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "I know no touch of it, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "'Tis as easy as lying: govern these ventages with"),
                                new XElement("LINE", "your lingers and thumb, give it breath with your"),
                                new XElement("LINE", "mouth, and it will discourse most eloquent music."),
                                new XElement("LINE", "Look you, these are the stops.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "But these cannot I command to any utterance of"),
                                new XElement("LINE", "harmony; I have not the skill.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Why, look you now, how unworthy a thing you make of"),
                                new XElement("LINE", "me! You would play upon me; you would seem to know"),
                                new XElement("LINE", "my stops; you would pluck out the heart of my"),
                                new XElement("LINE", "mystery; you would sound me from my lowest note to"),
                                new XElement("LINE", "the top of my compass: and there is much music,"),
                                new XElement("LINE", "excellent voice, in this little organ; yet cannot"),
                                new XElement("LINE", "you make it speak. 'Sblood, do you think I am"),
                                new XElement("LINE", "easier to be played on than a pipe? Call me what"),
                                new XElement("LINE", "instrument you will, though you can fret me, yet you"),
                                new XElement("LINE", "cannot play upon me."),
                                new XElement("STAGEDIR", "Enter POLONIUS"),
                                new XElement("LINE", "God bless you, sir!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "My lord, the queen would speak with you, and"),
                                new XElement("LINE", "presently.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Do you see yonder cloud that's almost in shape of a camel?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "By the mass, and 'tis like a camel, indeed.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Methinks it is like a weasel.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "It is backed like a weasel.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Or like a whale?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "Very like a whale.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Then I will come to my mother by and by. They fool"),
                                new XElement("LINE", "me to the top of my bent. I will come by and by.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "I will say so.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "By and by is easily said."),
                                new XElement("STAGEDIR", "Exit POLONIUS"),
                                new XElement("LINE", "Leave me, friends."),
                                new XElement("STAGEDIR", "Exeunt all but HAMLET"),
                                new XElement("LINE", "Tis now the very witching time of night,"),
                                new XElement("LINE", "When churchyards yawn and hell itself breathes out"),
                                new XElement("LINE", "Contagion to this world: now could I drink hot blood,"),
                                new XElement("LINE", "And do such bitter business as the day"),
                                new XElement("LINE", "Would quake to look on. Soft! now to my mother."),
                                new XElement("LINE", "O heart, lose not thy nature; let not ever"),
                                new XElement("LINE", "The soul of Nero enter this firm bosom:"),
                                new XElement("LINE", "Let me be cruel, not unnatural:"),
                                new XElement("LINE", "I will speak daggers to her, but use none;"),
                                new XElement("LINE", "My tongue and soul in this be hypocrites;"),
                                new XElement("LINE", "How in my words soever she be shent,"),
                                new XElement("LINE", "To give them seals never, my soul, consent!")
                            ),
                            new XElement("STAGEDIR", "Exit")
                        ),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE III.  A room in the castle."),
                            new XElement("STAGEDIR", "Enter KING CLAUDIUS, ROSENCRANTZ, and GUILDENSTERN"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "I like him not, nor stands it safe with us"),
                                new XElement("LINE", "To let his madness range. Therefore prepare you;"),
                                new XElement("LINE", "I your commission will forthwith dispatch,"),
                                new XElement("LINE", "And he to England shall along with you:"),
                                new XElement("LINE", "The terms of our estate may not endure"),
                                new XElement("LINE", "Hazard so dangerous as doth hourly grow"),
                                new XElement("LINE", "Out of his lunacies.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "We will ourselves provide:"),
                                new XElement("LINE", "Most holy and religious fear it is"),
                                new XElement("LINE", "To keep those many many bodies safe"),
                                new XElement("LINE", "That live and feed upon your majesty.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "The single and peculiar life is bound,"),
                                new XElement("LINE", "With all the strength and armour of the mind,"),
                                new XElement("LINE", "To keep itself from noyance; but much more"),
                                new XElement("LINE", "That spirit upon whose weal depend and rest"),
                                new XElement("LINE", "The lives of many. The cease of majesty"),
                                new XElement("LINE", "Dies not alone; but, like a gulf, doth draw"),
                                new XElement("LINE", "What's near it with it: it is a massy wheel,"),
                                new XElement("LINE", "Fix'd on the summit of the highest mount,"),
                                new XElement("LINE", "To whose huge spokes ten thousand lesser things"),
                                new XElement("LINE", "Are mortised and adjoin'd; which, when it falls,"),
                                new XElement("LINE", "Each small annexment, petty consequence,"),
                                new XElement("LINE", "Attends the boisterous ruin. Never alone"),
                                new XElement("LINE", "Did the king sigh, but with a general groan.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Arm you, I pray you, to this speedy voyage;"),
                                new XElement("LINE", "For we will fetters put upon this fear,"),
                                new XElement("LINE", "Which now goes too free-footed.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "We will haste us.")
                            ),
                            new XElement("STAGEDIR", "Exeunt ROSENCRANTZ and GUILDENSTERN"),
                            new XElement("STAGEDIR", "Enter POLONIUS"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "My lord, he's going to his mother's closet:"),
                                new XElement("LINE", "Behind the arras I'll convey myself,"),
                                new XElement("LINE", "To hear the process; and warrant she'll tax him home:"),
                                new XElement("LINE", "And, as you said, and wisely was it said,"),
                                new XElement("LINE", "'Tis meet that some more audience than a mother,"),
                                new XElement("LINE", "Since nature makes them partial, should o'erhear"),
                                new XElement("LINE", "The speech, of vantage. Fare you well, my liege:"),
                                new XElement("LINE", "I'll call upon you ere you go to bed,"),
                                new XElement("LINE", "And tell you what I know.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Thanks, dear my lord."),
                                new XElement("STAGEDIR", "Exit POLONIUS"),
                                new XElement("LINE", "O, my offence is rank it smells to heaven;"),
                                new XElement("LINE", "It hath the primal eldest curse upon't,"),
                                new XElement("LINE", "A brother's murder. Pray can I not,"),
                                new XElement("LINE", "Though inclination be as sharp as will:"),
                                new XElement("LINE", "My stronger guilt defeats my strong intent;"),
                                new XElement("LINE", "And, like a man to double business bound,"),
                                new XElement("LINE", "I stand in pause where I shall first begin,"),
                                new XElement("LINE", "And both neglect. What if this cursed hand"),
                                new XElement("LINE", "Were thicker than itself with brother's blood,"),
                                new XElement("LINE", "Is there not rain enough in the sweet heavens"),
                                new XElement("LINE", "To wash it white as snow? Whereto serves mercy"),
                                new XElement("LINE", "But to confront the visage of offence?"),
                                new XElement("LINE", "And what's in prayer but this two-fold force,"),
                                new XElement("LINE", "To be forestalled ere we come to fall,"),
                                new XElement("LINE", "Or pardon'd being down? Then I'll look up;"),
                                new XElement("LINE", "My fault is past. But, O, what form of prayer"),
                                new XElement("LINE", "Can serve my turn? 'Forgive me my foul murder'?"),
                                new XElement("LINE", "That cannot be; since I am still possess'd"),
                                new XElement("LINE", "Of those effects for which I did the murder,"),
                                new XElement("LINE", "My crown, mine own ambition and my queen."),
                                new XElement("LINE", "May one be pardon'd and retain the offence?"),
                                new XElement("LINE", "In the corrupted currents of this world"),
                                new XElement("LINE", "Offence's gilded hand may shove by justice,"),
                                new XElement("LINE", "And oft 'tis seen the wicked prize itself"),
                                new XElement("LINE", "Buys out the law: but 'tis not so above;"),
                                new XElement("LINE", "There is no shuffling, there the action lies"),
                                new XElement("LINE", "In his true nature; and we ourselves compell'd,"),
                                new XElement("LINE", "Even to the teeth and forehead of our faults,"),
                                new XElement("LINE", "To give in evidence. What then? what rests?"),
                                new XElement("LINE", "Try what repentance can: what can it not?"),
                                new XElement("LINE", "Yet what can it when one can not repent?"),
                                new XElement("LINE", "O wretched state! O bosom black as death!"),
                                new XElement("LINE", "O limed soul, that, struggling to be free,"),
                                new XElement("LINE", "Art more engaged! Help, angels! Make assay!"),
                                new XElement("LINE", "Bow, stubborn knees; and, heart with strings of steel,"),
                                new XElement("LINE", "Be soft as sinews of the newborn babe!"),
                                new XElement("LINE", "All may be well.")
                            ),
                            new XElement("STAGEDIR", "Retires and kneels"),
                            new XElement("STAGEDIR", "Enter HAMLET"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Now might I do it pat, now he is praying;"),
                                new XElement("LINE", "And now I'll do't. And so he goes to heaven;"),
                                new XElement("LINE", "And so am I revenged. That would be scann'd:"),
                                new XElement("LINE", "A villain kills my father; and for that,"),
                                new XElement("LINE", "I, his sole son, do this same villain send"),
                                new XElement("LINE", "To heaven."),
                                new XElement("LINE", "O, this is hire and salary, not revenge."),
                                new XElement("LINE", "He took my father grossly, full of bread;"),
                                new XElement("LINE", "With all his crimes broad blown, as flush as May;"),
                                new XElement("LINE", "And how his audit stands who knows save heaven?"),
                                new XElement("LINE", "But in our circumstance and course of thought,"),
                                new XElement("LINE", "'Tis heavy with him: and am I then revenged,"),
                                new XElement("LINE", "To take him in the purging of his soul,"),
                                new XElement("LINE", "When he is fit and season'd for his passage?"),
                                new XElement("LINE", "No!"),
                                new XElement("LINE", "Up, sword; and know thou a more horrid hent:"),
                                new XElement("LINE", "When he is drunk asleep, or in his rage,"),
                                new XElement("LINE", "Or in the incestuous pleasure of his bed;"),
                                new XElement("LINE", "At gaming, swearing, or about some act"),
                                new XElement("LINE", "That has no relish of salvation in't;"),
                                new XElement("LINE", "Then trip him, that his heels may kick at heaven,"),
                                new XElement("LINE", "And that his soul may be as damn'd and black"),
                                new XElement("LINE", "As hell, whereto it goes. My mother stays:"),
                                new XElement("LINE", "This physic but prolongs thy sickly days.")
                            ),
                            new XElement("STAGEDIR", "Exit"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Rising"),
                                    "  My words fly up, my thoughts remain below:"
                                ),
                                new XElement("LINE", "Words without thoughts never to heaven go.")
                            ),
                            new XElement("STAGEDIR", "Exit")
                        ),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE IV.  The Queen's closet."),
                            new XElement("STAGEDIR", "Enter QUEEN MARGARET and POLONIUS"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE", "He will come straight. Look you lay home to him:"),
                                new XElement("LINE", "Tell him his pranks have been too broad to bear with,"),
                                new XElement("LINE", "And that your grace hath screen'd and stood between"),
                                new XElement("LINE", "Much heat and him. I'll sconce me even here."),
                                new XElement("LINE", "Pray you, be round with him.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Within"),
                                    "  Mother, mother, mother!"
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "I'll warrant you,"),
                                new XElement("LINE", "Fear me not: withdraw, I hear him coming.")
                            ),
                            new XElement("STAGEDIR", "POLONIUS hides behind the arras"),
                            new XElement("STAGEDIR", "Enter HAMLET"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Now, mother, what's the matter?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Hamlet, thou hast thy father much offended.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Mother, you have my father much offended.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Come, come, you answer with an idle tongue.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Go, go, you question with a wicked tongue.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Why, how now, Hamlet!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "What's the matter now?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Have you forgot me?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "No, by the rood, not so:"),
                                new XElement("LINE", "You are the queen, your husband's brother's wife;"),
                                new XElement("LINE", "And--would it were not so!--you are my mother.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Nay, then, I'll set those to you that can speak.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Come, come, and sit you down; you shall not budge;"),
                                new XElement("LINE", "You go not till I set you up a glass"),
                                new XElement("LINE", "Where you may see the inmost part of you.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "What wilt thou do? thou wilt not murder me?"),
                                new XElement("LINE", "Help, help, ho!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Behind"),
                                    "  What, ho! help, help, help!"
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Drawing"),
                                    "  How now! a rat? Dead, for a ducat, dead!"
                                )
                            ),
                            new XElement("STAGEDIR", "Makes a pass through the arras"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LORD POLONIUS"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Behind"),
                                    "  O, I am slain!"
                                )
                            ),
                            new XElement("STAGEDIR", "Falls and dies"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "O me, what hast thou done?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Nay, I know not:"),
                                new XElement("LINE", "Is it the king?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "O, what a rash and bloody deed is this!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "A bloody deed! almost as bad, good mother,"),
                                new XElement("LINE", "As kill a king, and marry with his brother.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "As kill a king!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Ay, lady, 'twas my word."),
                                new XElement("STAGEDIR", "Lifts up the array and discovers POLONIUS"),
                                new XElement("LINE", "Thou wretched, rash, intruding fool, farewell!"),
                                new XElement("LINE", "I took thee for thy better: take thy fortune;"),
                                new XElement("LINE", "Thou find'st to be too busy is some danger."),
                                new XElement("LINE", "Leave wringing of your hands: peace! sit you down,"),
                                new XElement("LINE", "And let me wring your heart; for so I shall,"),
                                new XElement("LINE", "If it be made of penetrable stuff,"),
                                new XElement("LINE", "If damned custom have not brass'd it so"),
                                new XElement("LINE", "That it is proof and bulwark against sense.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "What have I done, that thou darest wag thy tongue"),
                                new XElement("LINE", "In noise so rude against me?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Such an act"),
                                new XElement("LINE", "That blurs the grace and blush of modesty,"),
                                new XElement("LINE", "Calls virtue hypocrite, takes off the rose"),
                                new XElement("LINE", "From the fair forehead of an innocent love"),
                                new XElement("LINE", "And sets a blister there, makes marriage-vows"),
                                new XElement("LINE", "As false as dicers' oaths: O, such a deed"),
                                new XElement("LINE", "As from the body of contraction plucks"),
                                new XElement("LINE", "The very soul, and sweet religion makes"),
                                new XElement("LINE", "A rhapsody of words: heaven's face doth glow:"),
                                new XElement("LINE", "Yea, this solidity and compound mass,"),
                                new XElement("LINE", "With tristful visage, as against the doom,"),
                                new XElement("LINE", "Is thought-sick at the act.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Ay me, what act,"),
                                new XElement("LINE", "That roars so loud, and thunders in the index?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Look here, upon this picture, and on this,"),
                                new XElement("LINE", "The counterfeit presentment of two brothers."),
                                new XElement("LINE", "See, what a grace was seated on this brow;"),
                                new XElement("LINE", "Hyperion's curls; the front of Jove himself;"),
                                new XElement("LINE", "An eye like Mars, to threaten and command;"),
                                new XElement("LINE", "A station like the herald Mercury"),
                                new XElement("LINE", "New-lighted on a heaven-kissing hill;"),
                                new XElement("LINE", "A combination and a form indeed,"),
                                new XElement("LINE", "Where every god did seem to set his seal,"),
                                new XElement("LINE", "To give the world assurance of a man:"),
                                new XElement("LINE", "This was your husband. Look you now, what follows:"),
                                new XElement("LINE", "Here is your husband; like a mildew'd ear,"),
                                new XElement("LINE", "Blasting his wholesome brother. Have you eyes?"),
                                new XElement("LINE", "Could you on this fair mountain leave to feed,"),
                                new XElement("LINE", "And batten on this moor? Ha! have you eyes?"),
                                new XElement("LINE", "You cannot call it love; for at your age"),
                                new XElement("LINE", "The hey-day in the blood is tame, it's humble,"),
                                new XElement("LINE", "And waits upon the judgment: and what judgment"),
                                new XElement("LINE", "Would step from this to this? Sense, sure, you have,"),
                                new XElement("LINE", "Else could you not have motion; but sure, that sense"),
                                new XElement("LINE", "Is apoplex'd; for madness would not err,"),
                                new XElement("LINE", "Nor sense to ecstasy was ne'er so thrall'd"),
                                new XElement("LINE", "But it reserved some quantity of choice,"),
                                new XElement("LINE", "To serve in such a difference. What devil was't"),
                                new XElement("LINE", "That thus hath cozen'd you at hoodman-blind?"),
                                new XElement("LINE", "Eyes without feeling, feeling without sight,"),
                                new XElement("LINE", "Ears without hands or eyes, smelling sans all,"),
                                new XElement("LINE", "Or but a sickly part of one true sense"),
                                new XElement("LINE", "Could not so mope."),
                                new XElement("LINE", "O shame! where is thy blush? Rebellious hell,"),
                                new XElement("LINE", "If thou canst mutine in a matron's bones,"),
                                new XElement("LINE", "To flaming youth let virtue be as wax,"),
                                new XElement("LINE", "And melt in her own fire: proclaim no shame"),
                                new XElement("LINE", "When the compulsive ardour gives the charge,"),
                                new XElement("LINE", "Since frost itself as actively doth burn"),
                                new XElement("LINE", "And reason panders will.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "O Hamlet, speak no more:"),
                                new XElement("LINE", "Thou turn'st mine eyes into my very soul;"),
                                new XElement("LINE", "And there I see such black and grained spots"),
                                new XElement("LINE", "As will not leave their tinct.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Nay, but to live"),
                                new XElement("LINE", "In the rank sweat of an enseamed bed,"),
                                new XElement("LINE", "Stew'd in corruption, honeying and making love"),
                                new XElement("LINE", "Over the nasty sty,--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "O, speak to me no more;"),
                                new XElement("LINE", "These words, like daggers, enter in mine ears;"),
                                new XElement("LINE", "No more, sweet Hamlet!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "A murderer and a villain;"),
                                new XElement("LINE", "A slave that is not twentieth part the tithe"),
                                new XElement("LINE", "Of your precedent lord; a vice of kings;"),
                                new XElement("LINE", "A cutpurse of the empire and the rule,"),
                                new XElement("LINE", "That from a shelf the precious diadem stole,"),
                                new XElement("LINE", "And put it in his pocket!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "No more!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "A king of shreds and patches,--"),
                                new XElement("STAGEDIR", "Enter Ghost"),
                                new XElement("LINE", "Save me, and hover o'er me with your wings,"),
                                new XElement("LINE", "You heavenly guards! What would your gracious figure?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Alas, he's mad!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Do you not come your tardy son to chide,"),
                                new XElement("LINE", "That, lapsed in time and passion, lets go by"),
                                new XElement("LINE", "The important acting of your dread command? O, say!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Ghost"),
                                new XElement("LINE", "Do not forget: this visitation"),
                                new XElement("LINE", "Is but to whet thy almost blunted purpose."),
                                new XElement("LINE", "But, look, amazement on thy mother sits:"),
                                new XElement("LINE", "O, step between her and her fighting soul:"),
                                new XElement("LINE", "Conceit in weakest bodies strongest works:"),
                                new XElement("LINE", "Speak to her, Hamlet.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "How is it with you, lady?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Alas, how is't with you,"),
                                new XElement("LINE", "That you do bend your eye on vacancy"),
                                new XElement("LINE", "And with the incorporal air do hold discourse?"),
                                new XElement("LINE", "Forth at your eyes your spirits wildly peep;"),
                                new XElement("LINE", "And, as the sleeping soldiers in the alarm,"),
                                new XElement("LINE", "Your bedded hair, like life in excrements,"),
                                new XElement("LINE", "Starts up, and stands on end. O gentle son,"),
                                new XElement("LINE", "Upon the heat and flame of thy distemper"),
                                new XElement("LINE", "Sprinkle cool patience. Whereon do you look?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "On him, on him! Look you, how pale he glares!"),
                                new XElement("LINE", "His form and cause conjoin'd, preaching to stones,"),
                                new XElement("LINE", "Would make them capable. Do not look upon me;"),
                                new XElement("LINE", "Lest with this piteous action you convert"),
                                new XElement("LINE", "My stern effects: then what I have to do"),
                                new XElement("LINE", "Will want true colour; tears perchance for blood.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "To whom do you speak this?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Do you see nothing there?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Nothing at all; yet all that is I see.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Nor did you nothing hear?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "No, nothing but ourselves.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Why, look you there! look, how it steals away!"),
                                new XElement("LINE", "My father, in his habit as he lived!"),
                                new XElement("LINE", "Look, where he goes, even now, out at the portal!")
                            ),
                            new XElement("STAGEDIR", "Exit Ghost"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "This the very coinage of your brain:"),
                                new XElement("LINE", "This bodiless creation ecstasy"),
                                new XElement("LINE", "Is very cunning in.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Ecstasy!"),
                                new XElement("LINE", "My pulse, as yours, doth temperately keep time,"),
                                new XElement("LINE", "And makes as healthful music: it is not madness"),
                                new XElement("LINE", "That I have utter'd: bring me to the test,"),
                                new XElement("LINE", "And I the matter will re-word; which madness"),
                                new XElement("LINE", "Would gambol from. Mother, for love of grace,"),
                                new XElement("LINE", "Lay not that mattering unction to your soul,"),
                                new XElement("LINE", "That not your trespass, but my madness speaks:"),
                                new XElement("LINE", "It will but skin and film the ulcerous place,"),
                                new XElement("LINE", "Whilst rank corruption, mining all within,"),
                                new XElement("LINE", "Infects unseen. Confess yourself to heaven;"),
                                new XElement("LINE", "Repent what's past; avoid what is to come;"),
                                new XElement("LINE", "And do not spread the compost on the weeds,"),
                                new XElement("LINE", "To make them ranker. Forgive me this my virtue;"),
                                new XElement("LINE", "For in the fatness of these pursy times"),
                                new XElement("LINE", "Virtue itself of vice must pardon beg,"),
                                new XElement("LINE", "Yea, curb and woo for leave to do him good.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "O Hamlet, thou hast cleft my heart in twain.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "O, throw away the worser part of it,"),
                                new XElement("LINE", "And live the purer with the other half."),
                                new XElement("LINE", "Good night: but go not to mine uncle's bed;"),
                                new XElement("LINE", "Assume a virtue, if you have it not."),
                                new XElement("LINE", "That monster, custom, who all sense doth eat,"),
                                new XElement("LINE", "Of habits devil, is angel yet in this,"),
                                new XElement("LINE", "That to the use of actions fair and good"),
                                new XElement("LINE", "He likewise gives a frock or livery,"),
                                new XElement("LINE", "That aptly is put on. Refrain to-night,"),
                                new XElement("LINE", "And that shall lend a kind of easiness"),
                                new XElement("LINE", "To the next abstinence: the next more easy;"),
                                new XElement("LINE", "For use almost can change the stamp of nature,"),
                                new XElement("LINE", "And either ... the devil, or throw him out"),
                                new XElement("LINE", "With wondrous potency. Once more, good night:"),
                                new XElement("LINE", "And when you are desirous to be bless'd,"),
                                new XElement("LINE", "I'll blessing beg of you. For this same lord,"),
                                new XElement("STAGEDIR", "Pointing to POLONIUS"),
                                new XElement("LINE", "I do repent: but heaven hath pleased it so,"),
                                new XElement("LINE", "To punish me with this and this with me,"),
                                new XElement("LINE", "That I must be their scourge and minister."),
                                new XElement("LINE", "I will bestow him, and will answer well"),
                                new XElement("LINE", "The death I gave him. So, again, good night."),
                                new XElement("LINE", "I must be cruel, only to be kind:"),
                                new XElement("LINE", "Thus bad begins and worse remains behind."),
                                new XElement("LINE", "One word more, good lady.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "What shall I do?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Not this, by no means, that I bid you do:"),
                                new XElement("LINE", "Let the bloat king tempt you again to bed;"),
                                new XElement("LINE", "Pinch wanton on your cheek; call you his mouse;"),
                                new XElement("LINE", "And let him, for a pair of reechy kisses,"),
                                new XElement("LINE", "Or paddling in your neck with his damn'd fingers,"),
                                new XElement("LINE", "Make you to ravel all this matter out,"),
                                new XElement("LINE", "That I essentially am not in madness,"),
                                new XElement("LINE", "But mad in craft. 'Twere good you let him know;"),
                                new XElement("LINE", "For who, that's but a queen, fair, sober, wise,"),
                                new XElement("LINE", "Would from a paddock, from a bat, a gib,"),
                                new XElement("LINE", "Such dear concernings hide? who would do so?"),
                                new XElement("LINE", "No, in despite of sense and secrecy,"),
                                new XElement("LINE", "Unpeg the basket on the house's top."),
                                new XElement("LINE", "Let the birds fly, and, like the famous ape,"),
                                new XElement("LINE", "To try conclusions, in the basket creep,"),
                                new XElement("LINE", "And break your own neck down.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Be thou assured, if words be made of breath,"),
                                new XElement("LINE", "And breath of life, I have no life to breathe"),
                                new XElement("LINE", "What thou hast said to me.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I must to England; you know that?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Alack,"),
                                new XElement("LINE", "I had forgot: 'tis so concluded on.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "There's letters seal'd: and my two schoolfellows,"),
                                new XElement("LINE", "Whom I will trust as I will adders fang'd,"),
                                new XElement("LINE", "They bear the mandate; they must sweep my way,"),
                                new XElement("LINE", "And marshal me to knavery. Let it work;"),
                                new XElement("LINE", "For 'tis the sport to have the engineer"),
                                new XElement("LINE", "Hoist with his own petard: and 't shall go hard"),
                                new XElement("LINE", "But I will delve one yard below their mines,"),
                                new XElement("LINE", "And blow them at the moon: O, 'tis most sweet,"),
                                new XElement("LINE", "When in one line two crafts directly meet."),
                                new XElement("LINE", "This man shall set me packing:"),
                                new XElement("LINE", "I'll lug the guts into the neighbour room."),
                                new XElement("LINE", "Mother, good night. Indeed this counsellor"),
                                new XElement("LINE", "Is now most still, most secret and most grave,"),
                                new XElement("LINE", "Who was in life a foolish prating knave."),
                                new XElement("LINE", "Come, sir, to draw toward an end with you."),
                                new XElement("LINE", "Good night, mother.")
                            ),
                            new XElement("STAGEDIR", "Exeunt severally; HAMLET dragging in POLONIUS")
                        )
                    ),
                    new XElement("ACT",
                        new XElement("TITLE", "ACT IV"),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE I.  A room in the castle."),
                            new XElement("STAGEDIR",
                                "Enter KING CLAUDIUS, QUEEN GERTRUDE, ROSENCRANTZ,\n" +
                                "and GUILDENSTERN"
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "There's matter in these sighs, these profound heaves:"),
                                new XElement("LINE", "You must translate: 'tis fit we understand them."),
                                new XElement("LINE", "Where is your son?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Bestow this place on us a little while."),
                                new XElement("STAGEDIR", "Exeunt ROSENCRANTZ and GUILDENSTERN"),
                                new XElement("LINE", "Ah, my good lord, what have I seen to-night!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "What, Gertrude? How does Hamlet?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Mad as the sea and wind, when both contend"),
                                new XElement("LINE", "Which is the mightier: in his lawless fit,"),
                                new XElement("LINE", "Behind the arras hearing something stir,"),
                                new XElement("LINE", "Whips out his rapier, cries, 'A rat, a rat!'"),
                                new XElement("LINE", "And, in this brainish apprehension, kills"),
                                new XElement("LINE", "The unseen good old man.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "O heavy deed!"),
                                new XElement("LINE", "It had been so with us, had we been there:"),
                                new XElement("LINE", "His liberty is full of threats to all;"),
                                new XElement("LINE", "To you yourself, to us, to every one."),
                                new XElement("LINE", "Alas, how shall this bloody deed be answer'd?"),
                                new XElement("LINE", "It will be laid to us, whose providence"),
                                new XElement("LINE", "Should have kept short, restrain'd and out of haunt,"),
                                new XElement("LINE", "This mad young man: but so much was our love,"),
                                new XElement("LINE", "We would not understand what was most fit;"),
                                new XElement("LINE", "But, like the owner of a foul disease,"),
                                new XElement("LINE", "To keep it from divulging, let it feed"),
                                new XElement("LINE", "Even on the pith of Life. Where is he gone?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "To draw apart the body he hath kill'd:"),
                                new XElement("LINE", "O'er whom his very madness, like some ore"),
                                new XElement("LINE", "Among a mineral of metals base,"),
                                new XElement("LINE", "Shows itself pure; he weeps for what is done.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "O Gertrude, come away!"),
                                new XElement("LINE", "The sun no sooner shall the mountains touch,"),
                                new XElement("LINE", "But we will ship him hence: and this vile deed"),
                                new XElement("LINE", "We must, with all our majesty and skill,"),
                                new XElement("LINE", "Both countenance and excuse. Ho, Guildenstern!"),
                                new XElement("STAGEDIR", "Re-enter ROSENCRANTZ and GUILDENSTERN"),
                                new XElement("LINE", "Friends both, go join you with some further aid:"),
                                new XElement("LINE", "Hamlet in madness hath Polonius slain,"),
                                new XElement("LINE", "And from his mother's closet hath he dragg'd him:"),
                                new XElement("LINE", "Go seek him out; speak fair, and bring the body"),
                                new XElement("LINE", "Into the chapel. I pray you, haste in this."),
                                new XElement("STAGEDIR", "Exeunt ROSENCRANTZ and GUILDENSTERN"),
                                new XElement("LINE", "Come, Gertrude, we'll call up our wisest friends;"),
                                new XElement("LINE", "And let them know, both what we mean to do,"),
                                new XElement("LINE", "And what's untimely done..."),
                                new XElement("LINE", "Whose whisper o'er the world's diameter,"),
                                new XElement("LINE", "As level as the cannon to his blank,"),
                                new XElement("LINE", "Transports his poison'd shot, may miss our name,"),
                                new XElement("LINE", "And hit the woundless air. O, come away!"),
                                new XElement("LINE", "My soul is full of discord and dismay.")
                            ),
                            new XElement("STAGEDIR", "Exeunt")
                        ),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE II.  Another room in the castle."),
                            new XElement("STAGEDIR", "Enter HAMLET"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Safely stowed.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Within"),
                                    "  Hamlet! Lord Hamlet!"
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "What noise? who calls on Hamlet?"),
                                new XElement("LINE", "O, here they come.")
                            ),
                            new XElement("STAGEDIR", "Enter ROSENCRANTZ and GUILDENSTERN"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "What have you done, my lord, with the dead body?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Compounded it with dust, whereto 'tis kin.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Tell us where 'tis, that we may take it thence"),
                                new XElement("LINE", "And bear it to the chapel.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Do not believe it.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Believe what?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "That I can keep your counsel and not mine own."),
                                new XElement("LINE", "Besides, to be demanded of a sponge! what"),
                                new XElement("LINE", "replication should be made by the son of a king?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Take you me for a sponge, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Ay, sir, that soaks up the king's countenance, his"),
                                new XElement("LINE", "rewards, his authorities. But such officers do the"),
                                new XElement("LINE", "king best service in the end: he keeps them, like"),
                                new XElement("LINE", "an ape, in the corner of his jaw; first mouthed, to"),
                                new XElement("LINE", "be last swallowed: when he needs what you have"),
                                new XElement("LINE", "gleaned, it is but squeezing you, and, sponge, you"),
                                new XElement("LINE", "shall be dry again.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "I understand you not, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I am glad of it: a knavish speech sleeps in a"),
                                new XElement("LINE", "foolish ear.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "My lord, you must tell us where the body is, and go"),
                                new XElement("LINE", "with us to the king.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "The body is with the king, but the king is not with"),
                                new XElement("LINE", "the body. The king is a thing--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "GUILDENSTERN"),
                                new XElement("LINE", "A thing, my lord!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Of nothing: bring me to him. Hide fox, and all after.")
                            ),
                            new XElement("STAGEDIR", "Exeunt")
                        ),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE III.  Another room in the castle."),
                            new XElement("STAGEDIR", "Enter KING CLAUDIUS, attended"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "I have sent to seek him, and to find the body."),
                                new XElement("LINE", "How dangerous is it that this man goes loose!"),
                                new XElement("LINE", "Yet must not we put the strong law on him:"),
                                new XElement("LINE", "He's loved of the distracted multitude,"),
                                new XElement("LINE", "Who like not in their judgment, but their eyes;"),
                                new XElement("LINE", "And where tis so, the offender's scourge is weigh'd,"),
                                new XElement("LINE", "But never the offence. To bear all smooth and even,"),
                                new XElement("LINE", "This sudden sending him away must seem"),
                                new XElement("LINE", "Deliberate pause: diseases desperate grown"),
                                new XElement("LINE", "By desperate appliance are relieved,"),
                                new XElement("LINE", "Or not at all."),
                                new XElement("STAGEDIR", "Enter ROSENCRANTZ"),
                                new XElement("LINE", "How now! what hath befall'n?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Where the dead body is bestow'd, my lord,"),
                                new XElement("LINE", "We cannot get from him.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "But where is he?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Without, my lord; guarded, to know your pleasure.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Bring him before us.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Ho, Guildenstern! bring in my lord.")
                            ),
                            new XElement("STAGEDIR", "Enter HAMLET and GUILDENSTERN"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Now, Hamlet, where's Polonius?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "At supper.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "At supper! where?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Not where he eats, but where he is eaten: a certain"),
                                new XElement("LINE", "convocation of politic worms are e'en at him. Your"),
                                new XElement("LINE", "worm is your only emperor for diet: we fat all"),
                                new XElement("LINE", "creatures else to fat us, and we fat ourselves for"),
                                new XElement("LINE", "maggots: your fat king and your lean beggar is but"),
                                new XElement("LINE", "variable service, two dishes, but to one table:"),
                                new XElement("LINE", "that's the end.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Alas, alas!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "A man may fish with the worm that hath eat of a"),
                                new XElement("LINE", "king, and cat of the fish that hath fed of that worm.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "What dost you mean by this?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Nothing but to show you how a king may go a"),
                                new XElement("LINE", "progress through the guts of a beggar.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Where is Polonius?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "In heaven; send hither to see: if your messenger"),
                                new XElement("LINE", "find him not there, seek him i' the other place"),
                                new XElement("LINE", "yourself. But indeed, if you find him not within"),
                                new XElement("LINE", "this month, you shall nose him as you go up the"),
                                new XElement("LINE", "stairs into the lobby.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Go seek him there.")
                            ),
                            new XElement("STAGEDIR", "To some Attendants"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "He will stay till ye come.")
                            ),
                            new XElement("STAGEDIR", "Exeunt Attendants"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Hamlet, this deed, for thine especial safety,--"),
                                new XElement("LINE", "Which we do tender, as we dearly grieve"),
                                new XElement("LINE", "For that which thou hast done,--must send thee hence"),
                                new XElement("LINE", "With fiery quickness: therefore prepare thyself;"),
                                new XElement("LINE", "The bark is ready, and the wind at help,"),
                                new XElement("LINE", "The associates tend, and every thing is bent"),
                                new XElement("LINE", "For England.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "For England!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Ay, Hamlet.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Good.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "So is it, if thou knew'st our purposes.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I see a cherub that sees them. But, come; for"),
                                new XElement("LINE", "England! Farewell, dear mother.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Thy loving father, Hamlet.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "My mother: father and mother is man and wife; man"),
                                new XElement("LINE", "and wife is one flesh; and so, my mother. Come, for England!")
                            ),
                            new XElement("STAGEDIR", "Exit"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Follow him at foot; tempt him with speed aboard;"),
                                new XElement("LINE", "Delay it not; I'll have him hence to-night:"),
                                new XElement("LINE", "Away! for every thing is seal'd and done"),
                                new XElement("LINE", "That else leans on the affair: pray you, make haste."),
                                new XElement("STAGEDIR", "Exeunt ROSENCRANTZ and GUILDENSTERN"),
                                new XElement("LINE", "And, England, if my love thou hold'st at aught--"),
                                new XElement("LINE", "As my great power thereof may give thee sense,"),
                                new XElement("LINE", "Since yet thy cicatrice looks raw and red"),
                                new XElement("LINE", "After the Danish sword, and thy free awe"),
                                new XElement("LINE", "Pays homage to us--thou mayst not coldly set"),
                                new XElement("LINE", "Our sovereign process; which imports at full,"),
                                new XElement("LINE", "By letters congruing to that effect,"),
                                new XElement("LINE", "The present death of Hamlet. Do it, England;"),
                                new XElement("LINE", "For like the hectic in my blood he rages,"),
                                new XElement("LINE", "And thou must cure me: till I know 'tis done,"),
                                new XElement("LINE", "Howe'er my haps, my joys were ne'er begun.")
                            ),
                            new XElement("STAGEDIR", "Exit")
                        ),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE IV.  A plain in Denmark."),
                            new XElement("STAGEDIR", "Enter FORTINBRAS, a Captain, and Soldiers, marching"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "PRINCE FORTINBRAS"),
                                new XElement("LINE", "Go, captain, from me greet the Danish king;"),
                                new XElement("LINE", "Tell him that, by his licence, Fortinbras"),
                                new XElement("LINE", "Craves the conveyance of a promised march"),
                                new XElement("LINE", "Over his kingdom. You know the rendezvous."),
                                new XElement("LINE", "If that his majesty would aught with us,"),
                                new XElement("LINE", "We shall express our duty in his eye;"),
                                new XElement("LINE", "And let him know so.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Captain"),
                                new XElement("LINE", "I will do't, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "PRINCE FORTINBRAS"),
                                new XElement("LINE", "Go softly on.")
                            ),
                            new XElement("STAGEDIR", "Exeunt FORTINBRAS and Soldiers"),
                            new XElement("STAGEDIR", "Enter HAMLET, ROSENCRANTZ, GUILDENSTERN, and others"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Good sir, whose powers are these?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Captain"),
                                new XElement("LINE", "They are of Norway, sir.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "How purposed, sir, I pray you?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Captain"),
                                new XElement("LINE", "Against some part of Poland.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Who commands them, sir?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Captain"),
                                new XElement("LINE", "The nephews to old Norway, Fortinbras.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Goes it against the main of Poland, sir,"),
                                new XElement("LINE", "Or for some frontier?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Captain"),
                                new XElement("LINE", "Truly to speak, and with no addition,"),
                                new XElement("LINE", "We go to gain a little patch of ground"),
                                new XElement("LINE", "That hath in it no profit but the name."),
                                new XElement("LINE", "To pay five ducats, five, I would not farm it;"),
                                new XElement("LINE", "Nor will it yield to Norway or the Pole"),
                                new XElement("LINE", "A ranker rate, should it be sold in fee.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Why, then the Polack never will defend it.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Captain"),
                                new XElement("LINE", "Yes, it is already garrison'd.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Two thousand souls and twenty thousand ducats"),
                                new XElement("LINE", "Will not debate the question of this straw:"),
                                new XElement("LINE", "This is the imposthume of much wealth and peace,"),
                                new XElement("LINE", "That inward breaks, and shows no cause without"),
                                new XElement("LINE", "Why the man dies. I humbly thank you, sir.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Captain"),
                                new XElement("LINE", "God be wi' you, sir.")
                            ),
                            new XElement("STAGEDIR", "Exit"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "ROSENCRANTZ"),
                                new XElement("LINE", "Wilt please you go, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I'll be with you straight go a little before."),
                                new XElement("STAGEDIR", "Exeunt all except HAMLET"),
                                new XElement("LINE", "How all occasions do inform against me,"),
                                new XElement("LINE", "And spur my dull revenge! What is a man,"),
                                new XElement("LINE", "If his chief good and market of his time"),
                                new XElement("LINE", "Be but to sleep and feed? a beast, no more."),
                                new XElement("LINE", "Sure, he that made us with such large discourse,"),
                                new XElement("LINE", "Looking before and after, gave us not"),
                                new XElement("LINE", "That capability and god-like reason"),
                                new XElement("LINE", "To fust in us unused. Now, whether it be"),
                                new XElement("LINE", "Bestial oblivion, or some craven scruple"),
                                new XElement("LINE", "Of thinking too precisely on the event,"),
                                new XElement("LINE", "A thought which, quarter'd, hath but one part wisdom"),
                                new XElement("LINE", "And ever three parts coward, I do not know"),
                                new XElement("LINE", "Why yet I live to say 'This thing's to do;'"),
                                new XElement("LINE", "Sith I have cause and will and strength and means"),
                                new XElement("LINE", "To do't. Examples gross as earth exhort me:"),
                                new XElement("LINE", "Witness this army of such mass and charge"),
                                new XElement("LINE", "Led by a delicate and tender prince,"),
                                new XElement("LINE", "Whose spirit with divine ambition puff'd"),
                                new XElement("LINE", "Makes mouths at the invisible event,"),
                                new XElement("LINE", "Exposing what is mortal and unsure"),
                                new XElement("LINE", "To all that fortune, death and danger dare,"),
                                new XElement("LINE", "Even for an egg-shell. Rightly to be great"),
                                new XElement("LINE", "Is not to stir without great argument,"),
                                new XElement("LINE", "But greatly to find quarrel in a straw"),
                                new XElement("LINE", "When honour's at the stake. How stand I then,"),
                                new XElement("LINE", "That have a father kill'd, a mother stain'd,"),
                                new XElement("LINE", "Excitements of my reason and my blood,"),
                                new XElement("LINE", "And let all sleep? while, to my shame, I see"),
                                new XElement("LINE", "The imminent death of twenty thousand men,"),
                                new XElement("LINE", "That, for a fantasy and trick of fame,"),
                                new XElement("LINE", "Go to their graves like beds, fight for a plot"),
                                new XElement("LINE", "Whereon the numbers cannot try the cause,"),
                                new XElement("LINE", "Which is not tomb enough and continent"),
                                new XElement("LINE", "To hide the slain? O, from this time forth,"),
                                new XElement("LINE", "My thoughts be bloody, or be nothing worth!")
                            ),
                            new XElement("STAGEDIR", "Exit")
                        ),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE V.  Elsinore. A room in the castle."),
                            new XElement("STAGEDIR", "Enter QUEEN GERTRUDE, HORATIO, and a Gentleman"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "I will not speak with her.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Gentleman"),
                                new XElement("LINE", "She is importunate, indeed distract:"),
                                new XElement("LINE", "Her mood will needs be pitied.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "What would she have?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Gentleman"),
                                new XElement("LINE", "She speaks much of her father; says she hears"),
                                new XElement("LINE", "There's tricks i' the world; and hems, and beats her heart;"),
                                new XElement("LINE", "Spurns enviously at straws; speaks things in doubt,"),
                                new XElement("LINE", "That carry but half sense: her speech is nothing,"),
                                new XElement("LINE", "Yet the unshaped use of it doth move"),
                                new XElement("LINE", "The hearers to collection; they aim at it,"),
                                new XElement("LINE", "And botch the words up fit to their own thoughts;"),
                                new XElement("LINE", "Which, as her winks, and nods, and gestures"),
                                new XElement("LINE", "yield them,"),
                                new XElement("LINE", "Indeed would make one think there might be thought,"),
                                new XElement("LINE", "Though nothing sure, yet much unhappily.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "'Twere good she were spoken with; for she may strew"),
                                new XElement("LINE", "Dangerous conjectures in ill-breeding minds.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Let her come in."),
                                new XElement("STAGEDIR", "Exit HORATIO"),
                                new XElement("LINE", "To my sick soul, as sin's true nature is,"),
                                new XElement("LINE", "Each toy seems prologue to some great amiss:"),
                                new XElement("LINE", "So full of artless jealousy is guilt,"),
                                new XElement("LINE", "It spills itself in fearing to be spilt.")
                            ),
                            new XElement("STAGEDIR", "Re-enter HORATIO, with OPHELIA"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "Where is the beauteous majesty of Denmark?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "How now, Ophelia!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Sings")
                                ),
                                new XElement("LINE", "How should I your true love know"),
                                new XElement("LINE", "From another one?"),
                                new XElement("LINE", "By his cockle hat and staff,"),
                                new XElement("LINE", "And his sandal shoon.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Alas, sweet lady, what imports this song?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "Say you? nay, pray you, mark."),
                                new XElement("STAGEDIR", "Sings"),
                                new XElement("LINE", "He is dead and gone, lady,"),
                                new XElement("LINE", "He is dead and gone;"),
                                new XElement("LINE", "At his head a grass-green turf,"),
                                new XElement("LINE", "At his heels a stone.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Nay, but, Ophelia,--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "Pray you, mark."),
                                new XElement("STAGEDIR", "Sings"),
                                new XElement("LINE", "White his shroud as the mountain snow,--")
                            ),
                            new XElement("STAGEDIR", "Enter KING CLAUDIUS"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Alas, look here, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Sings")
                                ),
                                new XElement("LINE", "Larded with sweet flowers"),
                                new XElement("LINE", "Which bewept to the grave did go"),
                                new XElement("LINE", "With true-love showers.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "How do you, pretty lady?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "Well, God 'ild you! They say the owl was a baker's"),
                                new XElement("LINE", "daughter. Lord, we know what we are, but know not"),
                                new XElement("LINE", "what we may be. God be at your table!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Conceit upon her father.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "Pray you, let's have no words of this; but when they"),
                                new XElement("LINE", "ask you what it means, say you this:"),
                                new XElement("STAGEDIR", "Sings"),
                                new XElement("LINE", "To-morrow is Saint Valentine's day,"),
                                new XElement("LINE", "All in the morning betime,"),
                                new XElement("LINE", "And I a maid at your window,"),
                                new XElement("LINE", "To be your Valentine."),
                                new XElement("LINE", "Then up he rose, and donn'd his clothes,"),
                                new XElement("LINE", "And dupp'd the chamber-door;"),
                                new XElement("LINE", "Let in the maid, that out a maid"),
                                new XElement("LINE", "Never departed more.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Pretty Ophelia!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "Indeed, la, without an oath, I'll make an end on't:"),
                                new XElement("STAGEDIR", "Sings"),
                                new XElement("LINE", "By Gis and by Saint Charity,"),
                                new XElement("LINE", "Alack, and fie for shame!"),
                                new XElement("LINE", "Young men will do't, if they come to't;"),
                                new XElement("LINE", "By cock, they are to blame."),
                                new XElement("LINE", "Quoth she, before you tumbled me,"),
                                new XElement("LINE", "You promised me to wed."),
                                new XElement("LINE", "So would I ha' done, by yonder sun,"),
                                new XElement("LINE", "An thou hadst not come to my bed.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "How long hath she been thus?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "I hope all will be well. We must be patient: but I"),
                                new XElement("LINE", "cannot choose but weep, to think they should lay him"),
                                new XElement("LINE", "i' the cold ground. My brother shall know of it:"),
                                new XElement("LINE", "and so I thank you for your good counsel. Come, my"),
                                new XElement("LINE", "coach! Good night, ladies; good night, sweet ladies;"),
                                new XElement("LINE", "good night, good night.")
                            ),
                            new XElement("STAGEDIR", "Exit"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Follow her close; give her good watch,"),
                                new XElement("LINE", "I pray you."),
                                new XElement("STAGEDIR", "Exit HORATIO"),
                                new XElement("LINE", "O, this is the poison of deep grief; it springs"),
                                new XElement("LINE", "All from her father's death. O Gertrude, Gertrude,"),
                                new XElement("LINE", "When sorrows come, they come not single spies"),
                                new XElement("LINE", "But in battalions. First, her father slain:"),
                                new XElement("LINE", "Next, your son gone; and he most violent author"),
                                new XElement("LINE", "Of his own just remove: the people muddied,"),
                                new XElement("LINE", "Thick and unwholesome in their thoughts and whispers,"),
                                new XElement("LINE", "For good Polonius' death; and we have done but greenly,"),
                                new XElement("LINE", "In hugger-mugger to inter him: poor Ophelia"),
                                new XElement("LINE", "Divided from herself and her fair judgment,"),
                                new XElement("LINE", "Without the which we are pictures, or mere beasts:"),
                                new XElement("LINE", "Last, and as much containing as all these,"),
                                new XElement("LINE", "Her brother is in secret come from France;"),
                                new XElement("LINE", "Feeds on his wonder, keeps himself in clouds,"),
                                new XElement("LINE", "And wants not buzzers to infect his ear"),
                                new XElement("LINE", "With pestilent speeches of his father's death;"),
                                new XElement("LINE", "Wherein necessity, of matter beggar'd,"),
                                new XElement("LINE", "Will nothing stick our person to arraign"),
                                new XElement("LINE", "In ear and ear. O my dear Gertrude, this,"),
                                new XElement("LINE", "Like to a murdering-piece, in many places"),
                                new XElement("LINE", "Gives me superfluous death.")
                            ),
                            new XElement("STAGEDIR", "A noise within"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Alack, what noise is this?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Where are my Switzers? Let them guard the door."),
                                new XElement("STAGEDIR", "Enter another Gentleman"),
                                new XElement("LINE", "What is the matter?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Gentleman"),
                                new XElement("LINE", "Save yourself, my lord:"),
                                new XElement("LINE", "The ocean, overpeering of his list,"),
                                new XElement("LINE", "Eats not the flats with more impetuous haste"),
                                new XElement("LINE", "Than young Laertes, in a riotous head,"),
                                new XElement("LINE", "O'erbears your officers. The rabble call him lord;"),
                                new XElement("LINE", "And, as the world were now but to begin,"),
                                new XElement("LINE", "Antiquity forgot, custom not known,"),
                                new XElement("LINE", "The ratifiers and props of every word,"),
                                new XElement("LINE", "They cry 'Choose we: Laertes shall be king:'"),
                                new XElement("LINE", "Caps, hands, and tongues, applaud it to the clouds:"),
                                new XElement("LINE", "'Laertes shall be king, Laertes king!'")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "How cheerfully on the false trail they cry!"),
                                new XElement("LINE", "O, this is counter, you false Danish dogs!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "The doors are broke.")
                            ),
                            new XElement("STAGEDIR", "Noise within"),
                            new XElement("STAGEDIR", "Enter LAERTES, armed; Danes following"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Where is this king? Sirs, stand you all without.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Danes"),
                                new XElement("LINE", "No, let's come in.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "I pray you, give me leave.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Danes"),
                                new XElement("LINE", "We will, we will.")
                            ),
                            new XElement("STAGEDIR", "They retire without the door"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "I thank you: keep the door. O thou vile king,"),
                                new XElement("LINE", "Give me my father!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Calmly, good Laertes.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "That drop of blood that's calm proclaims me bastard,"),
                                new XElement("LINE", "Cries cuckold to my father, brands the harlot"),
                                new XElement("LINE", "Even here, between the chaste unsmirched brow"),
                                new XElement("LINE", "Of my true mother.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "What is the cause, Laertes,"),
                                new XElement("LINE", "That thy rebellion looks so giant-like?"),
                                new XElement("LINE", "Let him go, Gertrude; do not fear our person:"),
                                new XElement("LINE", "There's such divinity doth hedge a king,"),
                                new XElement("LINE", "That treason can but peep to what it would,"),
                                new XElement("LINE", "Acts little of his will. Tell me, Laertes,"),
                                new XElement("LINE", "Why thou art thus incensed. Let him go, Gertrude."),
                                new XElement("LINE", "Speak, man.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Where is my father?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Dead.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "But not by him.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Let him demand his fill.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "How came he dead? I'll not be juggled with:"),
                                new XElement("LINE", "To hell, allegiance! vows, to the blackest devil!"),
                                new XElement("LINE", "Conscience and grace, to the profoundest pit!"),
                                new XElement("LINE", "I dare damnation. To this point I stand,"),
                                new XElement("LINE", "That both the worlds I give to negligence,"),
                                new XElement("LINE", "Let come what comes; only I'll be revenged"),
                                new XElement("LINE", "Most thoroughly for my father.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Who shall stay you?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "My will, not all the world:"),
                                new XElement("LINE", "And for my means, I'll husband them so well,"),
                                new XElement("LINE", "They shall go far with little.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Good Laertes,"),
                                new XElement("LINE", "If you desire to know the certainty"),
                                new XElement("LINE", "Of your dear father's death, is't writ in your revenge,"),
                                new XElement("LINE", "That, swoopstake, you will draw both friend and foe,"),
                                new XElement("LINE", "Winner and loser?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "None but his enemies.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Will you know them then?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "To his good friends thus wide I'll ope my arms;"),
                                new XElement("LINE", "And like the kind life-rendering pelican,"),
                                new XElement("LINE", "Repast them with my blood.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Why, now you speak"),
                                new XElement("LINE", "Like a good child and a true gentleman."),
                                new XElement("LINE", "That I am guiltless of your father's death,"),
                                new XElement("LINE", "And am most sensible in grief for it,"),
                                new XElement("LINE", "It shall as level to your judgment pierce"),
                                new XElement("LINE", "As day does to your eye.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Danes"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Within"),
                                    "                Let her come in."
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "How now! what noise is that?"),
                                new XElement("STAGEDIR", "Re-enter OPHELIA"),
                                new XElement("LINE", "O heat, dry up my brains! tears seven times salt,"),
                                new XElement("LINE", "Burn out the sense and virtue of mine eye!"),
                                new XElement("LINE", "By heaven, thy madness shall be paid by weight,"),
                                new XElement("LINE", "Till our scale turn the beam. O rose of May!"),
                                new XElement("LINE", "Dear maid, kind sister, sweet Ophelia!"),
                                new XElement("LINE", "O heavens! is't possible, a young maid's wits"),
                                new XElement("LINE", "Should be as moral as an old man's life?"),
                                new XElement("LINE", "Nature is fine in love, and where 'tis fine,"),
                                new XElement("LINE", "It sends some precious instance of itself"),
                                new XElement("LINE", "After the thing it loves.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Sings")
                                ),
                                new XElement("LINE", "They bore him barefaced on the bier;"),
                                new XElement("LINE", "Hey non nonny, nonny, hey nonny;"),
                                new XElement("LINE", "And in his grave rain'd many a tear:--"),
                                new XElement("LINE", "Fare you well, my dove!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Hadst thou thy wits, and didst persuade revenge,"),
                                new XElement("LINE", "It could not move thus.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Sings")
                                ),
                                new XElement("LINE", "You must sing a-down a-down,"),
                                new XElement("LINE", "An you call him a-down-a."),
                                new XElement("LINE", "O, how the wheel becomes it! It is the false"),
                                new XElement("LINE", "steward, that stole his master's daughter.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "This nothing's more than matter.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "There's rosemary, that's for remembrance; pray,"),
                                new XElement("LINE", "love, remember: and there is pansies. that's for thoughts.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "A document in madness, thoughts and remembrance fitted.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE", "There's fennel for you, and columbines: there's rue"),
                                new XElement("LINE", "for you; and here's some for me: we may call it"),
                                new XElement("LINE", "herb-grace o' Sundays: O you must wear your rue with"),
                                new XElement("LINE", "a difference. There's a daisy: I would give you"),
                                new XElement("LINE", "some violets, but they withered all when my father"),
                                new XElement("LINE", "died: they say he made a good end,--"),
                                new XElement("STAGEDIR", "Sings"),
                                new XElement("LINE", "For bonny sweet Robin is all my joy.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Thought and affliction, passion, hell itself,"),
                                new XElement("LINE", "She turns to favour and to prettiness.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OPHELIA"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Sings")
                                ),
                                new XElement("LINE", "And will he not come again?"),
                                new XElement("LINE", "And will he not come again?"),
                                new XElement("LINE", "No, no, he is dead:"),
                                new XElement("LINE", "Go to thy death-bed:"),
                                new XElement("LINE", "He never will come again."),
                                new XElement("LINE", "His beard was as white as snow,"),
                                new XElement("LINE", "All flaxen was his poll:"),
                                new XElement("LINE", "He is gone, he is gone,"),
                                new XElement("LINE", "And we cast away moan:"),
                                new XElement("LINE", "God ha' mercy on his soul!"),
                                new XElement("LINE", "And of all Christian souls, I pray God. God be wi' ye.")
                            ),
                            new XElement("STAGEDIR", "Exit"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Do you see this, O God?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Laertes, I must commune with your grief,"),
                                new XElement("LINE", "Or you deny me right. Go but apart,"),
                                new XElement("LINE", "Make choice of whom your wisest friends you will."),
                                new XElement("LINE", "And they shall hear and judge 'twixt you and me:"),
                                new XElement("LINE", "If by direct or by collateral hand"),
                                new XElement("LINE", "They find us touch'd, we will our kingdom give,"),
                                new XElement("LINE", "Our crown, our life, and all that we can ours,"),
                                new XElement("LINE", "To you in satisfaction; but if not,"),
                                new XElement("LINE", "Be you content to lend your patience to us,"),
                                new XElement("LINE", "And we shall jointly labour with your soul"),
                                new XElement("LINE", "To give it due content.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Let this be so;"),
                                new XElement("LINE", "His means of death, his obscure funeral--"),
                                new XElement("LINE", "No trophy, sword, nor hatchment o'er his bones,"),
                                new XElement("LINE", "No noble rite nor formal ostentation--"),
                                new XElement("LINE", "Cry to be heard, as 'twere from heaven to earth,"),
                                new XElement("LINE", "That I must call't in question.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "So you shall;"),
                                new XElement("LINE", "And where the offence is let the great axe fall."),
                                new XElement("LINE", "I pray you, go with me.")
                            ),
                            new XElement("STAGEDIR", "Exeunt")
                        ),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE VI.  Another room in the castle."),
                            new XElement("STAGEDIR", "Enter HORATIO and a Servant"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "What are they that would speak with me?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Servant"),
                                new XElement("LINE", "Sailors, sir: they say they have letters for you.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Let them come in."),
                                new XElement("STAGEDIR", "Exit Servant"),
                                new XElement("LINE", "I do not know from what part of the world"),
                                new XElement("LINE", "I should be greeted, if not from Lord Hamlet.")
                            ),
                            new XElement("STAGEDIR", "Enter Sailors"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Sailor"),
                                new XElement("LINE", "God bless you, sir.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Let him bless thee too.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Sailor"),
                                new XElement("LINE", "He shall, sir, an't please him. There's a letter for"),
                                new XElement("LINE", "you, sir; it comes from the ambassador that was"),
                                new XElement("LINE", "bound for England; if your name be Horatio, as I am"),
                                new XElement("LINE", "let to know it is.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Reads"),
                                    "  'Horatio, when thou shalt have overlooked"
                                ),
                                new XElement("LINE", "this, give these fellows some means to the king:"),
                                new XElement("LINE", "they have letters for him. Ere we were two days old"),
                                new XElement("LINE", "at sea, a pirate of very warlike appointment gave us"),
                                new XElement("LINE", "chase. Finding ourselves too slow of sail, we put on"),
                                new XElement("LINE", "a compelled valour, and in the grapple I boarded"),
                                new XElement("LINE", "them: on the instant they got clear of our ship; so"),
                                new XElement("LINE", "I alone became their prisoner. They have dealt with"),
                                new XElement("LINE", "me like thieves of mercy: but they knew what they"),
                                new XElement("LINE", "did; I am to do a good turn for them. Let the king"),
                                new XElement("LINE", "have the letters I have sent; and repair thou to me"),
                                new XElement("LINE", "with as much speed as thou wouldst fly death. I"),
                                new XElement("LINE", "have words to speak in thine ear will make thee"),
                                new XElement("LINE", "dumb; yet are they much too light for the bore of"),
                                new XElement("LINE", "the matter. These good fellows will bring thee"),
                                new XElement("LINE", "where I am. Rosencrantz and Guildenstern hold their"),
                                new XElement("LINE", "course for England: of them I have much to tell"),
                                new XElement("LINE", "thee. Farewell."),
                                new XElement("LINE", "'He that thou knowest thine, HAMLET.'"),
                                new XElement("LINE", "Come, I will make you way for these your letters;"),
                                new XElement("LINE", "And do't the speedier, that you may direct me"),
                                new XElement("LINE", "To him from whom you brought them.")
                            ),
                            new XElement("STAGEDIR", "Exeunt")
                        ),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE VII.  Another room in the castle."),
                            new XElement("STAGEDIR", "Enter KING CLAUDIUS and LAERTES"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Now must your conscience my acquaintance seal,"),
                                new XElement("LINE", "And you must put me in your heart for friend,"),
                                new XElement("LINE", "Sith you have heard, and with a knowing ear,"),
                                new XElement("LINE", "That he which hath your noble father slain"),
                                new XElement("LINE", "Pursued my life.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "It well appears: but tell me"),
                                new XElement("LINE", "Why you proceeded not against these feats,"),
                                new XElement("LINE", "So crimeful and so capital in nature,"),
                                new XElement("LINE", "As by your safety, wisdom, all things else,"),
                                new XElement("LINE", "You mainly were stirr'd up.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "O, for two special reasons;"),
                                new XElement("LINE", "Which may to you, perhaps, seem much unsinew'd,"),
                                new XElement("LINE", "But yet to me they are strong. The queen his mother"),
                                new XElement("LINE", "Lives almost by his looks; and for myself--"),
                                new XElement("LINE", "My virtue or my plague, be it either which--"),
                                new XElement("LINE", "She's so conjunctive to my life and soul,"),
                                new XElement("LINE", "That, as the star moves not but in his sphere,"),
                                new XElement("LINE", "I could not but by her. The other motive,"),
                                new XElement("LINE", "Why to a public count I might not go,"),
                                new XElement("LINE", "Is the great love the general gender bear him;"),
                                new XElement("LINE", "Who, dipping all his faults in their affection,"),
                                new XElement("LINE", "Would, like the spring that turneth wood to stone,"),
                                new XElement("LINE", "Convert his gyves to graces; so that my arrows,"),
                                new XElement("LINE", "Too slightly timber'd for so loud a wind,"),
                                new XElement("LINE", "Would have reverted to my bow again,"),
                                new XElement("LINE", "And not where I had aim'd them.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "And so have I a noble father lost;"),
                                new XElement("LINE", "A sister driven into desperate terms,"),
                                new XElement("LINE", "Whose worth, if praises may go back again,"),
                                new XElement("LINE", "Stood challenger on mount of all the age"),
                                new XElement("LINE", "For her perfections: but my revenge will come.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Break not your sleeps for that: you must not think"),
                                new XElement("LINE", "That we are made of stuff so flat and dull"),
                                new XElement("LINE", "That we can let our beard be shook with danger"),
                                new XElement("LINE", "And think it pastime. You shortly shall hear more:"),
                                new XElement("LINE", "I loved your father, and we love ourself;"),
                                new XElement("LINE", "And that, I hope, will teach you to imagine--"),
                                new XElement("STAGEDIR", "Enter a Messenger"),
                                new XElement("LINE", "How now! what news?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Messenger"),
                                new XElement("LINE", "Letters, my lord, from Hamlet:"),
                                new XElement("LINE", "This to your majesty; this to the queen.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "From Hamlet! who brought them?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Messenger"),
                                new XElement("LINE", "Sailors, my lord, they say; I saw them not:"),
                                new XElement("LINE", "They were given me by Claudio; he received them"),
                                new XElement("LINE", "Of him that brought them.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Laertes, you shall hear them. Leave us."),
                                new XElement("STAGEDIR", "Exit Messenger"),
                                new XElement("STAGEDIR", "Reads"),
                                new XElement("LINE", "'High and mighty, You shall know I am set naked on"),
                                new XElement("LINE", "your kingdom. To-morrow shall I beg leave to see"),
                                new XElement("LINE", "your kingly eyes: when I shall, first asking your"),
                                new XElement("LINE", "pardon thereunto, recount the occasion of my sudden"),
                                new XElement("LINE", "and more strange return.                  'HAMLET.'"),
                                new XElement("LINE", "What should this mean? Are all the rest come back?"),
                                new XElement("LINE", "Or is it some abuse, and no such thing?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Know you the hand?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "'Tis Hamlets character. 'Naked!"),
                                new XElement("LINE", "And in a postscript here, he says 'alone.'"),
                                new XElement("LINE", "Can you advise me?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "I'm lost in it, my lord. But let him come;"),
                                new XElement("LINE", "It warms the very sickness in my heart,"),
                                new XElement("LINE", "That I shall live and tell him to his teeth,"),
                                new XElement("LINE", "'Thus didest thou.'")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "If it be so, Laertes--"),
                                new XElement("LINE", "As how should it be so? how otherwise?--"),
                                new XElement("LINE", "Will you be ruled by me?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Ay, my lord;"),
                                new XElement("LINE", "So you will not o'errule me to a peace.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "To thine own peace. If he be now return'd,"),
                                new XElement("LINE", "As checking at his voyage, and that he means"),
                                new XElement("LINE", "No more to undertake it, I will work him"),
                                new XElement("LINE", "To an exploit, now ripe in my device,"),
                                new XElement("LINE", "Under the which he shall not choose but fall:"),
                                new XElement("LINE", "And for his death no wind of blame shall breathe,"),
                                new XElement("LINE", "But even his mother shall uncharge the practise"),
                                new XElement("LINE", "And call it accident.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "My lord, I will be ruled;"),
                                new XElement("LINE", "The rather, if you could devise it so"),
                                new XElement("LINE", "That I might be the organ.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "It falls right."),
                                new XElement("LINE", "You have been talk'd of since your travel much,"),
                                new XElement("LINE", "And that in Hamlet's hearing, for a quality"),
                                new XElement("LINE", "Wherein, they say, you shine: your sum of parts"),
                                new XElement("LINE", "Did not together pluck such envy from him"),
                                new XElement("LINE", "As did that one, and that, in my regard,"),
                                new XElement("LINE", "Of the unworthiest siege.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "What part is that, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "A very riband in the cap of youth,"),
                                new XElement("LINE", "Yet needful too; for youth no less becomes"),
                                new XElement("LINE", "The light and careless livery that it wears"),
                                new XElement("LINE", "Than settled age his sables and his weeds,"),
                                new XElement("LINE", "Importing health and graveness. Two months since,"),
                                new XElement("LINE", "Here was a gentleman of Normandy:--"),
                                new XElement("LINE", "I've seen myself, and served against, the French,"),
                                new XElement("LINE", "And they can well on horseback: but this gallant"),
                                new XElement("LINE", "Had witchcraft in't; he grew unto his seat;"),
                                new XElement("LINE", "And to such wondrous doing brought his horse,"),
                                new XElement("LINE", "As he had been incorpsed and demi-natured"),
                                new XElement("LINE", "With the brave beast: so far he topp'd my thought,"),
                                new XElement("LINE", "That I, in forgery of shapes and tricks,"),
                                new XElement("LINE", "Come short of what he did.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "A Norman was't?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "A Norman.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Upon my life, Lamond.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "The very same.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "I know him well: he is the brooch indeed"),
                                new XElement("LINE", "And gem of all the nation.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "He made confession of you,"),
                                new XElement("LINE", "And gave you such a masterly report"),
                                new XElement("LINE", "For art and exercise in your defence"),
                                new XElement("LINE", "And for your rapier most especially,"),
                                new XElement("LINE", "That he cried out, 'twould be a sight indeed,"),
                                new XElement("LINE", "If one could match you: the scrimers of their nation,"),
                                new XElement("LINE", "He swore, had had neither motion, guard, nor eye,"),
                                new XElement("LINE", "If you opposed them. Sir, this report of his"),
                                new XElement("LINE", "Did Hamlet so envenom with his envy"),
                                new XElement("LINE", "That he could nothing do but wish and beg"),
                                new XElement("LINE", "Your sudden coming o'er, to play with him."),
                                new XElement("LINE", "Now, out of this,--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "What out of this, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Laertes, was your father dear to you?"),
                                new XElement("LINE", "Or are you like the painting of a sorrow,"),
                                new XElement("LINE", "A face without a heart?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Why ask you this?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Not that I think you did not love your father;"),
                                new XElement("LINE", "But that I know love is begun by time;"),
                                new XElement("LINE", "And that I see, in passages of proof,"),
                                new XElement("LINE", "Time qualifies the spark and fire of it."),
                                new XElement("LINE", "There lives within the very flame of love"),
                                new XElement("LINE", "A kind of wick or snuff that will abate it;"),
                                new XElement("LINE", "And nothing is at a like goodness still;"),
                                new XElement("LINE", "For goodness, growing to a plurisy,"),
                                new XElement("LINE", "Dies in his own too much: that we would do"),
                                new XElement("LINE", "We should do when we would; for this 'would' changes"),
                                new XElement("LINE", "And hath abatements and delays as many"),
                                new XElement("LINE", "As there are tongues, are hands, are accidents;"),
                                new XElement("LINE", "And then this 'should' is like a spendthrift sigh,"),
                                new XElement("LINE", "That hurts by easing. But, to the quick o' the ulcer:--"),
                                new XElement("LINE", "Hamlet comes back: what would you undertake,"),
                                new XElement("LINE", "To show yourself your father's son in deed"),
                                new XElement("LINE", "More than in words?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "To cut his throat i' the church.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "No place, indeed, should murder sanctuarize;"),
                                new XElement("LINE", "Revenge should have no bounds. But, good Laertes,"),
                                new XElement("LINE", "Will you do this, keep close within your chamber."),
                                new XElement("LINE", "Hamlet return'd shall know you are come home:"),
                                new XElement("LINE", "We'll put on those shall praise your excellence"),
                                new XElement("LINE", "And set a double varnish on the fame"),
                                new XElement("LINE", "The Frenchman gave you, bring you in fine together"),
                                new XElement("LINE", "And wager on your heads: he, being remiss,"),
                                new XElement("LINE", "Most generous and free from all contriving,"),
                                new XElement("LINE", "Will not peruse the foils; so that, with ease,"),
                                new XElement("LINE", "Or with a little shuffling, you may choose"),
                                new XElement("LINE", "A sword unbated, and in a pass of practise"),
                                new XElement("LINE", "Requite him for your father.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "I will do't:"),
                                new XElement("LINE", "And, for that purpose, I'll anoint my sword."),
                                new XElement("LINE", "I bought an unction of a mountebank,"),
                                new XElement("LINE", "So mortal that, but dip a knife in it,"),
                                new XElement("LINE", "Where it draws blood no cataplasm so rare,"),
                                new XElement("LINE", "Collected from all simples that have virtue"),
                                new XElement("LINE", "Under the moon, can save the thing from death"),
                                new XElement("LINE", "That is but scratch'd withal: I'll touch my point"),
                                new XElement("LINE", "With this contagion, that, if I gall him slightly,"),
                                new XElement("LINE", "It may be death.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Let's further think of this;"),
                                new XElement("LINE", "Weigh what convenience both of time and means"),
                                new XElement("LINE", "May fit us to our shape: if this should fail,"),
                                new XElement("LINE", "And that our drift look through our bad performance,"),
                                new XElement("LINE", "'Twere better not assay'd: therefore this project"),
                                new XElement("LINE", "Should have a back or second, that might hold,"),
                                new XElement("LINE", "If this should blast in proof. Soft! let me see:"),
                                new XElement("LINE", "We'll make a solemn wager on your cunnings: I ha't."),
                                new XElement("LINE", "When in your motion you are hot and dry--"),
                                new XElement("LINE", "As make your bouts more violent to that end--"),
                                new XElement("LINE", "And that he calls for drink, I'll have prepared him"),
                                new XElement("LINE", "A chalice for the nonce, whereon but sipping,"),
                                new XElement("LINE", "If he by chance escape your venom'd stuck,"),
                                new XElement("LINE", "Our purpose may hold there."),
                                new XElement("STAGEDIR", "Enter QUEEN GERTRUDE"),
                                new XElement("LINE", "How now, sweet queen!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "One woe doth tread upon another's heel,"),
                                new XElement("LINE", "So fast they follow; your sister's drown'd, Laertes.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Drown'd! O, where?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "There is a willow grows aslant a brook,"),
                                new XElement("LINE", "That shows his hoar leaves in the glassy stream;"),
                                new XElement("LINE", "There with fantastic garlands did she come"),
                                new XElement("LINE", "Of crow-flowers, nettles, daisies, and long purples"),
                                new XElement("LINE", "That liberal shepherds give a grosser name,"),
                                new XElement("LINE", "But our cold maids do dead men's fingers call them:"),
                                new XElement("LINE", "There, on the pendent boughs her coronet weeds"),
                                new XElement("LINE", "Clambering to hang, an envious sliver broke;"),
                                new XElement("LINE", "When down her weedy trophies and herself"),
                                new XElement("LINE", "Fell in the weeping brook. Her clothes spread wide;"),
                                new XElement("LINE", "And, mermaid-like, awhile they bore her up:"),
                                new XElement("LINE", "Which time she chanted snatches of old tunes;"),
                                new XElement("LINE", "As one incapable of her own distress,"),
                                new XElement("LINE", "Or like a creature native and indued"),
                                new XElement("LINE", "Unto that element: but long it could not be"),
                                new XElement("LINE", "Till that her garments, heavy with their drink,"),
                                new XElement("LINE", "Pull'd the poor wretch from her melodious lay"),
                                new XElement("LINE", "To muddy death.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Alas, then, she is drown'd?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Drown'd, drown'd.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Too much of water hast thou, poor Ophelia,"),
                                new XElement("LINE", "And therefore I forbid my tears: but yet"),
                                new XElement("LINE", "It is our trick; nature her custom holds,"),
                                new XElement("LINE", "Let shame say what it will: when these are gone,"),
                                new XElement("LINE", "The woman will be out. Adieu, my lord:"),
                                new XElement("LINE", "I have a speech of fire, that fain would blaze,"),
                                new XElement("LINE", "But that this folly douts it.")
                            ),
                            new XElement("STAGEDIR", "Exit"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Let's follow, Gertrude:"),
                                new XElement("LINE", "How much I had to do to calm his rage!"),
                                new XElement("LINE", "Now fear I this will give it start again;"),
                                new XElement("LINE", "Therefore let's follow.")
                            ),
                            new XElement("STAGEDIR", "Exeunt")
                        )
                    ),
                    new XElement("ACT",
                        new XElement("TITLE", "ACT V"),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE I.  A churchyard."),
                            new XElement("STAGEDIR", "Enter two Clowns, with spades, &c"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "Is she to be buried in Christian burial that"),
                                new XElement("LINE", "wilfully seeks her own salvation?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Second Clown"),
                                new XElement("LINE", "I tell thee she is: and therefore make her grave"),
                                new XElement("LINE", "straight: the crowner hath sat on her, and finds it"),
                                new XElement("LINE", "Christian burial.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "How can that be, unless she drowned herself in her"),
                                new XElement("LINE", "own defence?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Second Clown"),
                                new XElement("LINE", "Why, 'tis found so.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "It must be 'se offendendo;' it cannot be else. For"),
                                new XElement("LINE", "here lies the point:  if I drown myself wittingly,"),
                                new XElement("LINE", "it argues an act: and an act hath three branches: it"),
                                new XElement("LINE", "is, to act, to do, to perform: argal, she drowned"),
                                new XElement("LINE", "herself wittingly.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Second Clown"),
                                new XElement("LINE", "Nay, but hear you, goodman delver,--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "Give me leave. Here lies the water; good: here"),
                                new XElement("LINE", "stands the man; good; if the man go to this water,"),
                                new XElement("LINE", "and drown himself, it is, will he, nill he, he"),
                                new XElement("LINE", "goes,--mark you that; but if the water come to him"),
                                new XElement("LINE", "and drown him, he drowns not himself: argal, he"),
                                new XElement("LINE", "that is not guilty of his own death shortens not his own life.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Second Clown"),
                                new XElement("LINE", "But is this law?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "Ay, marry, is't; crowner's quest law.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Second Clown"),
                                new XElement("LINE", "Will you ha' the truth on't? If this had not been"),
                                new XElement("LINE", "a gentlewoman, she should have been buried out o'"),
                                new XElement("LINE", "Christian burial.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "Why, there thou say'st: and the more pity that"),
                                new XElement("LINE", "great folk should have countenance in this world to"),
                                new XElement("LINE", "drown or hang themselves, more than their even"),
                                new XElement("LINE", "Christian. Come, my spade. There is no ancient"),
                                new XElement("LINE", "gentleman but gardeners, ditchers, and grave-makers:"),
                                new XElement("LINE", "they hold up Adam's profession.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Second Clown"),
                                new XElement("LINE", "Was he a gentleman?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "He was the first that ever bore arms.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Second Clown"),
                                new XElement("LINE", "Why, he had none.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "What, art a heathen? How dost thou understand the"),
                                new XElement("LINE", "Scripture? The Scripture says 'Adam digged:'"),
                                new XElement("LINE", "could he dig without arms? I'll put another"),
                                new XElement("LINE", "question to thee: if thou answerest me not to the"),
                                new XElement("LINE", "purpose, confess thyself--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Second Clown"),
                                new XElement("LINE", "Go to.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "What is he that builds stronger than either the"),
                                new XElement("LINE", "mason, the shipwright, or the carpenter?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Second Clown"),
                                new XElement("LINE", "The gallows-maker; for that frame outlives a"),
                                new XElement("LINE", "thousand tenants.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "I like thy wit well, in good faith: the gallows"),
                                new XElement("LINE", "does well; but how does it well? it does well to"),
                                new XElement("LINE", "those that do in: now thou dost ill to say the"),
                                new XElement("LINE", "gallows is built stronger than the church: argal,"),
                                new XElement("LINE", "the gallows may do well to thee. To't again, come.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Second Clown"),
                                new XElement("LINE", "'Who builds stronger than a mason, a shipwright, or"),
                                new XElement("LINE", "a carpenter?'")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "Ay, tell me that, and unyoke.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Second Clown"),
                                new XElement("LINE", "Marry, now I can tell.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "To't.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Second Clown"),
                                new XElement("LINE", "Mass, I cannot tell.")
                            ),
                            new XElement("STAGEDIR", "Enter HAMLET and HORATIO, at a distance"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "Cudgel thy brains no more about it, for your dull"),
                                new XElement("LINE", "ass will not mend his pace with beating; and, when"),
                                new XElement("LINE", "you are asked this question next, say 'a"),
                                new XElement("LINE", "grave-maker: 'the houses that he makes last till"),
                                new XElement("LINE", "doomsday. Go, get thee to Yaughan: fetch me a"),
                                new XElement("LINE", "stoup of liquor."),
                                new XElement("STAGEDIR", "Exit Second Clown"),
                                new XElement("STAGEDIR", "He digs and sings"),
                                new XElement("LINE", "In youth, when I did love, did love,"),
                                new XElement("LINE", "Methought it was very sweet,"),
                                new XElement("LINE", "To contract, O, the time, for, ah, my behove,"),
                                new XElement("LINE", "O, methought, there was nothing meet.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Has this fellow no feeling of his business, that he"),
                                new XElement("LINE", "sings at grave-making?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Custom hath made it in him a property of easiness.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "'Tis e'en so: the hand of little employment hath"),
                                new XElement("LINE", "the daintier sense.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Sings")
                                ),
                                new XElement("LINE", "But age, with his stealing steps,"),
                                new XElement("LINE", "Hath claw'd me in his clutch,"),
                                new XElement("LINE", "And hath shipped me intil the land,"),
                                new XElement("LINE", "As if I had never been such.")
                            ),
                            new XElement("STAGEDIR", "Throws up a skull"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "That skull had a tongue in it, and could sing once:"),
                                new XElement("LINE", "how the knave jowls it to the ground, as if it were"),
                                new XElement("LINE", "Cain's jaw-bone, that did the first murder! It"),
                                new XElement("LINE", "might be the pate of a politician, which this ass"),
                                new XElement("LINE", "now o'er-reaches; one that would circumvent God,"),
                                new XElement("LINE", "might it not?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "It might, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Or of a courtier; which could say 'Good morrow,"),
                                new XElement("LINE", "sweet lord! How dost thou, good lord?' This might"),
                                new XElement("LINE", "be my lord such-a-one, that praised my lord"),
                                new XElement("LINE", "such-a-one's horse, when he meant to beg it; might it not?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Ay, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Why, e'en so: and now my Lady Worm's; chapless, and"),
                                new XElement("LINE", "knocked about the mazzard with a sexton's spade:"),
                                new XElement("LINE", "here's fine revolution, an we had the trick to"),
                                new XElement("LINE", "see't. Did these bones cost no more the breeding,"),
                                new XElement("LINE", "but to play at loggats with 'em? mine ache to think on't.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("STAGEDIR", "Sings"),
                                new XElement("LINE", "A pick-axe, and a spade, a spade,"),
                                new XElement("LINE", "For and a shrouding sheet:"),
                                new XElement("LINE", "O, a pit of clay for to be made"),
                                new XElement("LINE", "For such a guest is meet.")
                            ),
                            new XElement("STAGEDIR", "Throws up another skull"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "There's another: why may not that be the skull of a"),
                                new XElement("LINE", "lawyer? Where be his quiddities now, his quillets,"),
                                new XElement("LINE", "his cases, his tenures, and his tricks? why does he"),
                                new XElement("LINE", "suffer this rude knave now to knock him about the"),
                                new XElement("LINE", "sconce with a dirty shovel, and will not tell him of"),
                                new XElement("LINE", "his action of battery? Hum! This fellow might be"),
                                new XElement("LINE", "in's time a great buyer of land, with his statutes,"),
                                new XElement("LINE", "his recognizances, his fines, his double vouchers,"),
                                new XElement("LINE", "his recoveries: is this the fine of his fines, and"),
                                new XElement("LINE", "the recovery of his recoveries, to have his fine"),
                                new XElement("LINE", "pate full of fine dirt? will his vouchers vouch him"),
                                new XElement("LINE", "no more of his purchases, and double ones too, than"),
                                new XElement("LINE", "the length and breadth of a pair of indentures? The"),
                                new XElement("LINE", "very conveyances of his lands will hardly lie in"),
                                new XElement("LINE", "this box; and must the inheritor himself have no more, ha?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Not a jot more, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Is not parchment made of sheepskins?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Ay, my lord, and of calf-skins too.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "They are sheep and calves which seek out assurance"),
                                new XElement("LINE", "in that. I will speak to this fellow. Whose"),
                                new XElement("LINE", "grave's this, sirrah?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "Mine, sir."),
                                new XElement("STAGEDIR", "Sings"),
                                new XElement("LINE", "O, a pit of clay for to be made"),
                                new XElement("LINE", "For such a guest is meet.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I think it be thine, indeed; for thou liest in't.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "You lie out on't, sir, and therefore it is not"),
                                new XElement("LINE", "yours: for my part, I do not lie in't, and yet it is mine.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "'Thou dost lie in't, to be in't and say it is thine:"),
                                new XElement("LINE", "'tis for the dead, not for the quick; therefore thou liest.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "'Tis a quick lie, sir; 'twill away gain, from me to"),
                                new XElement("LINE", "you.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "What man dost thou dig it for?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "For no man, sir.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "What woman, then?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "For none, neither.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Who is to be buried in't?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "One that was a woman, sir; but, rest her soul, she's dead.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "How absolute the knave is! we must speak by the"),
                                new XElement("LINE", "card, or equivocation will undo us. By the Lord,"),
                                new XElement("LINE", "Horatio, these three years I have taken a note of"),
                                new XElement("LINE", "it; the age is grown so picked that the toe of the"),
                                new XElement("LINE", "peasant comes so near the heel of the courtier, he"),
                                new XElement("LINE", "gaffs his kibe. How long hast thou been a"),
                                new XElement("LINE", "grave-maker?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "Of all the days i' the year, I came to't that day"),
                                new XElement("LINE", "that our last king Hamlet overcame Fortinbras.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "How long is that since?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "Cannot you tell that? every fool can tell that: it"),
                                new XElement("LINE", "was the very day that young Hamlet was born; he that"),
                                new XElement("LINE", "is mad, and sent into England.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Ay, marry, why was he sent into England?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "Why, because he was mad: he shall recover his wits"),
                                new XElement("LINE", "there; or, if he do not, it's no great matter there.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Why?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "'Twill, a not be seen in him there; there the men"),
                                new XElement("LINE", "are as mad as he.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "How came he mad?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "Very strangely, they say.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "How strangely?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "Faith, e'en with losing his wits.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Upon what ground?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "Why, here in Denmark: I have been sexton here, man"),
                                new XElement("LINE", "and boy, thirty years.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "How long will a man lie i' the earth ere he rot?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "I' faith, if he be not rotten before he die--as we"),
                                new XElement("LINE", "have many pocky corses now-a-days, that will scarce"),
                                new XElement("LINE", "hold the laying in--he will last you some eight year"),
                                new XElement("LINE", "or nine year: a tanner will last you nine year.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Why he more than another?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "Why, sir, his hide is so tanned with his trade, that"),
                                new XElement("LINE", "he will keep out water a great while; and your water"),
                                new XElement("LINE", "is a sore decayer of your whoreson dead body."),
                                new XElement("LINE", "Here's a skull now; this skull has lain in the earth"),
                                new XElement("LINE", "three and twenty years.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Whose was it?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "A whoreson mad fellow's it was: whose do you think it was?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Nay, I know not.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "A pestilence on him for a mad rogue! a' poured a"),
                                new XElement("LINE", "flagon of Rhenish on my head once. This same skull,"),
                                new XElement("LINE", "sir, was Yorick's skull, the king's jester.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "This?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Clown"),
                                new XElement("LINE", "E'en that.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Let me see."),
                                new XElement("STAGEDIR", "Takes the skull"),
                                new XElement("LINE", "Alas, poor Yorick! I knew him, Horatio: a fellow"),
                                new XElement("LINE", "of infinite jest, of most excellent fancy: he hath"),
                                new XElement("LINE", "borne me on his back a thousand times; and now, how"),
                                new XElement("LINE", "abhorred in my imagination it is! my gorge rims at"),
                                new XElement("LINE", "it. Here hung those lips that I have kissed I know"),
                                new XElement("LINE", "not how oft. Where be your gibes now? your"),
                                new XElement("LINE", "gambols? your songs? your flashes of merriment,"),
                                new XElement("LINE", "that were wont to set the table on a roar? Not one"),
                                new XElement("LINE", "now, to mock your own grinning? quite chap-fallen?"),
                                new XElement("LINE", "Now get you to my lady's chamber, and tell her, let"),
                                new XElement("LINE", "her paint an inch thick, to this favour she must"),
                                new XElement("LINE", "come; make her laugh at that. Prithee, Horatio, tell"),
                                new XElement("LINE", "me one thing.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "What's that, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Dost thou think Alexander looked o' this fashion i'"),
                                new XElement("LINE", "the earth?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "E'en so.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "And smelt so? pah!")
                            ),
                            new XElement("STAGEDIR", "Puts down the skull"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "E'en so, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "To what base uses we may return, Horatio! Why may"),
                                new XElement("LINE", "not imagination trace the noble dust of Alexander,"),
                                new XElement("LINE", "till he find it stopping a bung-hole?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "'Twere to consider too curiously, to consider so.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "No, faith, not a jot; but to follow him thither with"),
                                new XElement("LINE", "modesty enough, and likelihood to lead it: as"),
                                new XElement("LINE", "thus: Alexander died, Alexander was buried,"),
                                new XElement("LINE", "Alexander returneth into dust; the dust is earth; of"),
                                new XElement("LINE", "earth we make loam; and why of that loam, whereto he"),
                                new XElement("LINE", "was converted, might they not stop a beer-barrel?"),
                                new XElement("LINE", "Imperious Caesar, dead and turn'd to clay,"),
                                new XElement("LINE", "Might stop a hole to keep the wind away:"),
                                new XElement("LINE", "O, that that earth, which kept the world in awe,"),
                                new XElement("LINE", "Should patch a wall to expel the winter flaw!"),
                                new XElement("LINE", "But soft! but soft! aside: here comes the king."),
                                new XElement("STAGEDIR",
                                    "Enter Priest, &c. in procession; the Corpse of\n" +
                                    "OPHELIA, LAERTES and Mourners following; KING\n" +
                                    "CLAUDIUS, QUEEN GERTRUDE, their trains, &c"
                                ),
                                new XElement("LINE", "The queen, the courtiers: who is this they follow?"),
                                new XElement("LINE", "And with such maimed rites? This doth betoken"),
                                new XElement("LINE", "The corse they follow did with desperate hand"),
                                new XElement("LINE", "Fordo its own life: 'twas of some estate."),
                                new XElement("LINE", "Couch we awhile, and mark.")
                            ),
                            new XElement("STAGEDIR", "Retiring with HORATIO"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "What ceremony else?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "That is Laertes,"),
                                new XElement("LINE", "A very noble youth: mark.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "What ceremony else?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Priest"),
                                new XElement("LINE", "Her obsequies have been as far enlarged"),
                                new XElement("LINE", "As we have warrantise: her death was doubtful;"),
                                new XElement("LINE", "And, but that great command o'ersways the order,"),
                                new XElement("LINE", "She should in ground unsanctified have lodged"),
                                new XElement("LINE", "Till the last trumpet: for charitable prayers,"),
                                new XElement("LINE", "Shards, flints and pebbles should be thrown on her;"),
                                new XElement("LINE", "Yet here she is allow'd her virgin crants,"),
                                new XElement("LINE", "Her maiden strewments and the bringing home"),
                                new XElement("LINE", "Of bell and burial.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Must there no more be done?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Priest"),
                                new XElement("LINE", "No more be done:"),
                                new XElement("LINE", "We should profane the service of the dead"),
                                new XElement("LINE", "To sing a requiem and such rest to her"),
                                new XElement("LINE", "As to peace-parted souls.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Lay her i' the earth:"),
                                new XElement("LINE", "And from her fair and unpolluted flesh"),
                                new XElement("LINE", "May violets spring! I tell thee, churlish priest,"),
                                new XElement("LINE", "A ministering angel shall my sister be,"),
                                new XElement("LINE", "When thou liest howling.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "What, the fair Ophelia!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Sweets to the sweet: farewell!"),
                                new XElement("STAGEDIR", "Scattering flowers"),
                                new XElement("LINE", "I hoped thou shouldst have been my Hamlet's wife;"),
                                new XElement("LINE", "I thought thy bride-bed to have deck'd, sweet maid,"),
                                new XElement("LINE", "And not have strew'd thy grave.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "O, treble woe"),
                                new XElement("LINE", "Fall ten times treble on that cursed head,"),
                                new XElement("LINE", "Whose wicked deed thy most ingenious sense"),
                                new XElement("LINE", "Deprived thee of! Hold off the earth awhile,"),
                                new XElement("LINE", "Till I have caught her once more in mine arms:"),
                                new XElement("STAGEDIR", "Leaps into the grave"),
                                new XElement("LINE", "Now pile your dust upon the quick and dead,"),
                                new XElement("LINE", "Till of this flat a mountain you have made,"),
                                new XElement("LINE", "To o'ertop old Pelion, or the skyish head"),
                                new XElement("LINE", "Of blue Olympus.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Advancing"),
                                    "     What is he whose grief"
                                ),
                                new XElement("LINE", "Bears such an emphasis? whose phrase of sorrow"),
                                new XElement("LINE", "Conjures the wandering stars, and makes them stand"),
                                new XElement("LINE", "Like wonder-wounded hearers? This is I,"),
                                new XElement("LINE", "Hamlet the Dane.")
                            ),
                            new XElement("STAGEDIR", "Leaps into the grave"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "The devil take thy soul!")
                            ),
                            new XElement("STAGEDIR", "Grappling with him"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Thou pray'st not well."),
                                new XElement("LINE", "I prithee, take thy fingers from my throat;"),
                                new XElement("LINE", "For, though I am not splenitive and rash,"),
                                new XElement("LINE", "Yet have I something in me dangerous,"),
                                new XElement("LINE", "Which let thy wiseness fear: hold off thy hand.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Pluck them asunder.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Hamlet, Hamlet!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "All"),
                                new XElement("LINE", "Gentlemen,--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Good my lord, be quiet.")
                            ),
                            new XElement("STAGEDIR", "The Attendants part them, and they come out of the grave"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Why I will fight with him upon this theme"),
                                new XElement("LINE", "Until my eyelids will no longer wag.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "O my son, what theme?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I loved Ophelia: forty thousand brothers"),
                                new XElement("LINE", "Could not, with all their quantity of love,"),
                                new XElement("LINE", "Make up my sum. What wilt thou do for her?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "O, he is mad, Laertes.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "For love of God, forbear him.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "'Swounds, show me what thou'lt do:"),
                                new XElement("LINE", "Woo't weep? woo't fight? woo't fast? woo't tear thyself?"),
                                new XElement("LINE", "Woo't drink up eisel? eat a crocodile?"),
                                new XElement("LINE", "I'll do't. Dost thou come here to whine?"),
                                new XElement("LINE", "To outface me with leaping in her grave?"),
                                new XElement("LINE", "Be buried quick with her, and so will I:"),
                                new XElement("LINE", "And, if thou prate of mountains, let them throw"),
                                new XElement("LINE", "Millions of acres on us, till our ground,"),
                                new XElement("LINE", "Singeing his pate against the burning zone,"),
                                new XElement("LINE", "Make Ossa like a wart! Nay, an thou'lt mouth,"),
                                new XElement("LINE", "I'll rant as well as thou.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "This is mere madness:"),
                                new XElement("LINE", "And thus awhile the fit will work on him;"),
                                new XElement("LINE", "Anon, as patient as the female dove,"),
                                new XElement("LINE", "When that her golden couplets are disclosed,"),
                                new XElement("LINE", "His silence will sit drooping.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Hear you, sir;"),
                                new XElement("LINE", "What is the reason that you use me thus?"),
                                new XElement("LINE", "I loved you ever: but it is no matter;"),
                                new XElement("LINE", "Let Hercules himself do what he may,"),
                                new XElement("LINE", "The cat will mew and dog will have his day.")
                            ),
                            new XElement("STAGEDIR", "Exit"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "I pray you, good Horatio, wait upon him."),
                                new XElement("STAGEDIR", "Exit HORATIO"),
                                new XElement("STAGEDIR", "To LAERTES"),
                                new XElement("LINE", "Strengthen your patience in our last night's speech;"),
                                new XElement("LINE", "We'll put the matter to the present push."),
                                new XElement("LINE", "Good Gertrude, set some watch over your son."),
                                new XElement("LINE", "This grave shall have a living monument:"),
                                new XElement("LINE", "An hour of quiet shortly shall we see;"),
                                new XElement("LINE", "Till then, in patience our proceeding be.")
                            ),
                            new XElement("STAGEDIR", "Exeunt")
                        ),
                        new XElement("SCENE",
                            new XElement("TITLE", "SCENE II.  A hall in the castle."),
                            new XElement("STAGEDIR", "Enter HAMLET and HORATIO"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "So much for this, sir: now shall you see the other;"),
                                new XElement("LINE", "You do remember all the circumstance?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Remember it, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Sir, in my heart there was a kind of fighting,"),
                                new XElement("LINE", "That would not let me sleep: methought I lay"),
                                new XElement("LINE", "Worse than the mutines in the bilboes. Rashly,"),
                                new XElement("LINE", "And praised be rashness for it, let us know,"),
                                new XElement("LINE", "Our indiscretion sometimes serves us well,"),
                                new XElement("LINE", "When our deep plots do pall: and that should teach us"),
                                new XElement("LINE", "There's a divinity that shapes our ends,"),
                                new XElement("LINE", "Rough-hew them how we will,--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "That is most certain.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Up from my cabin,"),
                                new XElement("LINE", "My sea-gown scarf'd about me, in the dark"),
                                new XElement("LINE", "Groped I to find out them; had my desire."),
                                new XElement("LINE", "Finger'd their packet, and in fine withdrew"),
                                new XElement("LINE", "To mine own room again; making so bold,"),
                                new XElement("LINE", "My fears forgetting manners, to unseal"),
                                new XElement("LINE", "Their grand commission; where I found, Horatio,--"),
                                new XElement("LINE", "O royal knavery!--an exact command,"),
                                new XElement("LINE", "Larded with many several sorts of reasons"),
                                new XElement("LINE", "Importing Denmark's health and England's too,"),
                                new XElement("LINE", "With, ho! such bugs and goblins in my life,"),
                                new XElement("LINE", "That, on the supervise, no leisure bated,"),
                                new XElement("LINE", "No, not to stay the grinding of the axe,"),
                                new XElement("LINE", "My head should be struck off.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Is't possible?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Here's the commission: read it at more leisure."),
                                new XElement("LINE", "But wilt thou hear me how I did proceed?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "I beseech you.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Being thus be-netted round with villanies,--"),
                                new XElement("LINE", "Ere I could make a prologue to my brains,"),
                                new XElement("LINE", "They had begun the play--I sat me down,"),
                                new XElement("LINE", "Devised a new commission, wrote it fair:"),
                                new XElement("LINE", "I once did hold it, as our statists do,"),
                                new XElement("LINE", "A baseness to write fair and labour'd much"),
                                new XElement("LINE", "How to forget that learning, but, sir, now"),
                                new XElement("LINE", "It did me yeoman's service: wilt thou know"),
                                new XElement("LINE", "The effect of what I wrote?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Ay, good my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "An earnest conjuration from the king,"),
                                new XElement("LINE", "As England was his faithful tributary,"),
                                new XElement("LINE", "As love between them like the palm might flourish,"),
                                new XElement("LINE", "As peace should stiff her wheaten garland wear"),
                                new XElement("LINE", "And stand a comma 'tween their amities,"),
                                new XElement("LINE", "And many such-like 'As'es of great charge,"),
                                new XElement("LINE", "That, on the view and knowing of these contents,"),
                                new XElement("LINE", "Without debatement further, more or less,"),
                                new XElement("LINE", "He should the bearers put to sudden death,"),
                                new XElement("LINE", "Not shriving-time allow'd.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "How was this seal'd?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Why, even in that was heaven ordinant."),
                                new XElement("LINE", "I had my father's signet in my purse,"),
                                new XElement("LINE", "Which was the model of that Danish seal;"),
                                new XElement("LINE", "Folded the writ up in form of the other,"),
                                new XElement("LINE", "Subscribed it, gave't the impression, placed it safely,"),
                                new XElement("LINE", "The changeling never known. Now, the next day"),
                                new XElement("LINE", "Was our sea-fight; and what to this was sequent"),
                                new XElement("LINE", "Thou know'st already.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "So Guildenstern and Rosencrantz go to't.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Why, man, they did make love to this employment;"),
                                new XElement("LINE", "They are not near my conscience; their defeat"),
                                new XElement("LINE", "Does by their own insinuation grow:"),
                                new XElement("LINE", "'Tis dangerous when the baser nature comes"),
                                new XElement("LINE", "Between the pass and fell incensed points"),
                                new XElement("LINE", "Of mighty opposites.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Why, what a king is this!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Does it not, think'st thee, stand me now upon--"),
                                new XElement("LINE", "He that hath kill'd my king and whored my mother,"),
                                new XElement("LINE", "Popp'd in between the election and my hopes,"),
                                new XElement("LINE", "Thrown out his angle for my proper life,"),
                                new XElement("LINE", "And with such cozenage--is't not perfect conscience,"),
                                new XElement("LINE", "To quit him with this arm? and is't not to be damn'd,"),
                                new XElement("LINE", "To let this canker of our nature come"),
                                new XElement("LINE", "In further evil?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "It must be shortly known to him from England"),
                                new XElement("LINE", "What is the issue of the business there.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "It will be short: the interim is mine;"),
                                new XElement("LINE", "And a man's life's no more than to say 'One.'"),
                                new XElement("LINE", "But I am very sorry, good Horatio,"),
                                new XElement("LINE", "That to Laertes I forgot myself;"),
                                new XElement("LINE", "For, by the image of my cause, I see"),
                                new XElement("LINE", "The portraiture of his: I'll court his favours."),
                                new XElement("LINE", "But, sure, the bravery of his grief did put me"),
                                new XElement("LINE", "Into a towering passion.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Peace! who comes here?")
                            ),
                            new XElement("STAGEDIR", "Enter OSRIC"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "Your lordship is right welcome back to Denmark.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I humbly thank you, sir. Dost know this water-fly?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "No, my good lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Thy state is the more gracious; for 'tis a vice to"),
                                new XElement("LINE", "know him. He hath much land, and fertile: let a"),
                                new XElement("LINE", "beast be lord of beasts, and his crib shall stand at"),
                                new XElement("LINE", "the king's mess: 'tis a chough; but, as I say,"),
                                new XElement("LINE", "spacious in the possession of dirt.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "Sweet lord, if your lordship were at leisure, I"),
                                new XElement("LINE", "should impart a thing to you from his majesty.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I will receive it, sir, with all diligence of"),
                                new XElement("LINE", "spirit. Put your bonnet to his right use; 'tis for the head.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "I thank your lordship, it is very hot.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "No, believe me, 'tis very cold; the wind is"),
                                new XElement("LINE", "northerly.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "It is indifferent cold, my lord, indeed.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "But yet methinks it is very sultry and hot for my"),
                                new XElement("LINE", "complexion.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "Exceedingly, my lord; it is very sultry,--as"),
                                new XElement("LINE", "'twere,--I cannot tell how. But, my lord, his"),
                                new XElement("LINE", "majesty bade me signify to you that he has laid a"),
                                new XElement("LINE", "great wager on your head: sir, this is the matter,--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I beseech you, remember--")
                            ),
                            new XElement("STAGEDIR", "HAMLET moves him to put on his hat"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "Nay, good my lord; for mine ease, in good faith."),
                                new XElement("LINE", "Sir, here is newly come to court Laertes; believe"),
                                new XElement("LINE", "me, an absolute gentleman, full of most excellent"),
                                new XElement("LINE", "differences, of very soft society and great showing:"),
                                new XElement("LINE", "indeed, to speak feelingly of him, he is the card or"),
                                new XElement("LINE", "calendar of gentry, for you shall find in him the"),
                                new XElement("LINE", "continent of what part a gentleman would see.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Sir, his definement suffers no perdition in you;"),
                                new XElement("LINE", "though, I know, to divide him inventorially would"),
                                new XElement("LINE", "dizzy the arithmetic of memory, and yet but yaw"),
                                new XElement("LINE", "neither, in respect of his quick sail. But, in the"),
                                new XElement("LINE", "verity of extolment, I take him to be a soul of"),
                                new XElement("LINE", "great article; and his infusion of such dearth and"),
                                new XElement("LINE", "rareness, as, to make true diction of him, his"),
                                new XElement("LINE", "semblable is his mirror; and who else would trace"),
                                new XElement("LINE", "him, his umbrage, nothing more.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "Your lordship speaks most infallibly of him.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "The concernancy, sir? why do we wrap the gentleman"),
                                new XElement("LINE", "in our more rawer breath?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "Sir?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Is't not possible to understand in another tongue?"),
                                new XElement("LINE", "You will do't, sir, really.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "What imports the nomination of this gentleman?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "Of Laertes?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "His purse is empty already; all's golden words are spent.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Of him, sir.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "I know you are not ignorant--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I would you did, sir; yet, in faith, if you did,"),
                                new XElement("LINE", "it would not much approve me. Well, sir?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "You are not ignorant of what excellence Laertes is--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I dare not confess that, lest I should compare with"),
                                new XElement("LINE", "him in excellence; but, to know a man well, were to"),
                                new XElement("LINE", "know himself.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "I mean, sir, for his weapon; but in the imputation"),
                                new XElement("LINE", "laid on him by them, in his meed he's unfellowed.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "What's his weapon?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "Rapier and dagger.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "That's two of his weapons: but, well.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "The king, sir, hath wagered with him six Barbary"),
                                new XElement("LINE", "horses: against the which he has imponed, as I take"),
                                new XElement("LINE", "it, six French rapiers and poniards, with their"),
                                new XElement("LINE", "assigns, as girdle, hangers, and so: three of the"),
                                new XElement("LINE", "carriages, in faith, are very dear to fancy, very"),
                                new XElement("LINE", "responsive to the hilts, most delicate carriages,"),
                                new XElement("LINE", "and of very liberal conceit.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "What call you the carriages?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "I knew you must be edified by the margent ere you had done.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "The carriages, sir, are the hangers.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "The phrase would be more german to the matter, if we"),
                                new XElement("LINE", "could carry cannon by our sides: I would it might"),
                                new XElement("LINE", "be hangers till then. But, on: six Barbary horses"),
                                new XElement("LINE", "against six French swords, their assigns, and three"),
                                new XElement("LINE", "liberal-conceited carriages; that's the French bet"),
                                new XElement("LINE", "against the Danish. Why is this 'imponed,' as you call it?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "The king, sir, hath laid, that in a dozen passes"),
                                new XElement("LINE", "between yourself and him, he shall not exceed you"),
                                new XElement("LINE", "three hits: he hath laid on twelve for nine; and it"),
                                new XElement("LINE", "would come to immediate trial, if your lordship"),
                                new XElement("LINE", "would vouchsafe the answer.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "How if I answer 'no'?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "I mean, my lord, the opposition of your person in trial.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Sir, I will walk here in the hall: if it please his"),
                                new XElement("LINE", "majesty, 'tis the breathing time of day with me; let"),
                                new XElement("LINE", "the foils be brought, the gentleman willing, and the"),
                                new XElement("LINE", "king hold his purpose, I will win for him an I can;"),
                                new XElement("LINE", "if not, I will gain nothing but my shame and the odd hits.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "Shall I re-deliver you e'en so?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "To this effect, sir; after what flourish your nature will.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "I commend my duty to your lordship.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Yours, yours."),
                                new XElement("STAGEDIR", "Exit OSRIC"),
                                new XElement("LINE", "He does well to commend it himself; there are no"),
                                new XElement("LINE", "tongues else for's turn.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "This lapwing runs away with the shell on his head.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "He did comply with his dug, before he sucked it."),
                                new XElement("LINE", "Thus has he--and many more of the same bevy that I"),
                                new XElement("LINE", "know the dressy age dotes on--only got the tune of"),
                                new XElement("LINE", "the time and outward habit of encounter; a kind of"),
                                new XElement("LINE", "yesty collection, which carries them through and"),
                                new XElement("LINE", "through the most fond and winnowed opinions; and do"),
                                new XElement("LINE", "but blow them to their trial, the bubbles are out.")
                            ),
                            new XElement("STAGEDIR", "Enter a Lord"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Lord"),
                                new XElement("LINE", "My lord, his majesty commended him to you by young"),
                                new XElement("LINE", "Osric, who brings back to him that you attend him in"),
                                new XElement("LINE", "the hall: he sends to know if your pleasure hold to"),
                                new XElement("LINE", "play with Laertes, or that you will take longer time.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I am constant to my purpose; they follow the king's"),
                                new XElement("LINE", "pleasure: if his fitness speaks, mine is ready; now"),
                                new XElement("LINE", "or whensoever, provided I be so able as now.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Lord"),
                                new XElement("LINE", "The king and queen and all are coming down.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "In happy time.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "Lord"),
                                new XElement("LINE", "The queen desires you to use some gentle"),
                                new XElement("LINE", "entertainment to Laertes before you fall to play.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "She well instructs me.")
                            ),
                            new XElement("STAGEDIR", "Exit Lord"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "You will lose this wager, my lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I do not think so: since he went into France, I"),
                                new XElement("LINE", "have been in continual practise: I shall win at the"),
                                new XElement("LINE", "odds. But thou wouldst not think how ill all's here"),
                                new XElement("LINE", "about my heart: but it is no matter.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Nay, good my lord,--")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "It is but foolery; but it is such a kind of"),
                                new XElement("LINE", "gain-giving, as would perhaps trouble a woman.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "If your mind dislike any thing, obey it: I will"),
                                new XElement("LINE", "forestall their repair hither, and say you are not"),
                                new XElement("LINE", "fit.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Not a whit, we defy augury: there's a special"),
                                new XElement("LINE", "providence in the fall of a sparrow. If it be now,"),
                                new XElement("LINE", "'tis not to come; if it be not to come, it will be"),
                                new XElement("LINE", "now; if it be not now, yet it will come: the"),
                                new XElement("LINE", "readiness is all: since no man has aught of what he"),
                                new XElement("LINE", "leaves, what is't to leave betimes?")
                            ),
                            new XElement("STAGEDIR",
                                "Enter KING CLAUDIUS, QUEEN GERTRUDE, LAERTES,\n" +
                                "Lords, OSRIC, and Attendants with foils, &c"
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Come, Hamlet, come, and take this hand from me.")
                            ),
                            new XElement("STAGEDIR", "KING CLAUDIUS puts LAERTES' hand into HAMLET's"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Give me your pardon, sir: I've done you wrong;"),
                                new XElement("LINE", "But pardon't, as you are a gentleman."),
                                new XElement("LINE", "This presence knows,"),
                                new XElement("LINE", "And you must needs have heard, how I am punish'd"),
                                new XElement("LINE", "With sore distraction. What I have done,"),
                                new XElement("LINE", "That might your nature, honour and exception"),
                                new XElement("LINE", "Roughly awake, I here proclaim was madness."),
                                new XElement("LINE", "Was't Hamlet wrong'd Laertes? Never Hamlet:"),
                                new XElement("LINE", "If Hamlet from himself be ta'en away,"),
                                new XElement("LINE", "And when he's not himself does wrong Laertes,"),
                                new XElement("LINE", "Then Hamlet does it not, Hamlet denies it."),
                                new XElement("LINE", "Who does it, then? His madness: if't be so,"),
                                new XElement("LINE", "Hamlet is of the faction that is wrong'd;"),
                                new XElement("LINE", "His madness is poor Hamlet's enemy."),
                                new XElement("LINE", "Sir, in this audience,"),
                                new XElement("LINE", "Let my disclaiming from a purposed evil"),
                                new XElement("LINE", "Free me so far in your most generous thoughts,"),
                                new XElement("LINE", "That I have shot mine arrow o'er the house,"),
                                new XElement("LINE", "And hurt my brother.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "I am satisfied in nature,"),
                                new XElement("LINE", "Whose motive, in this case, should stir me most"),
                                new XElement("LINE", "To my revenge: but in my terms of honour"),
                                new XElement("LINE", "I stand aloof; and will no reconcilement,"),
                                new XElement("LINE", "Till by some elder masters, of known honour,"),
                                new XElement("LINE", "I have a voice and precedent of peace,"),
                                new XElement("LINE", "To keep my name ungored. But till that time,"),
                                new XElement("LINE", "I do receive your offer'd love like love,"),
                                new XElement("LINE", "And will not wrong it.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I embrace it freely;"),
                                new XElement("LINE", "And will this brother's wager frankly play."),
                                new XElement("LINE", "Give us the foils. Come on.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Come, one for me.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I'll be your foil, Laertes: in mine ignorance"),
                                new XElement("LINE", "Your skill shall, like a star i' the darkest night,"),
                                new XElement("LINE", "Stick fiery off indeed.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "You mock me, sir.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "No, by this hand.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Give them the foils, young Osric. Cousin Hamlet,"),
                                new XElement("LINE", "You know the wager?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Very well, my lord"),
                                new XElement("LINE", "Your grace hath laid the odds o' the weaker side.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "I do not fear it; I have seen you both:"),
                                new XElement("LINE", "But since he is better'd, we have therefore odds.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "This is too heavy, let me see another.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "This likes me well. These foils have all a length?")
                            ),
                            new XElement("STAGEDIR", "They prepare to play"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "Ay, my good lord.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Set me the stoops of wine upon that table."),
                                new XElement("LINE", "If Hamlet give the first or second hit,"),
                                new XElement("LINE", "Or quit in answer of the third exchange,"),
                                new XElement("LINE", "Let all the battlements their ordnance fire:"),
                                new XElement("LINE", "The king shall drink to Hamlet's better breath;"),
                                new XElement("LINE", "And in the cup an union shall he throw,"),
                                new XElement("LINE", "Richer than that which four successive kings"),
                                new XElement("LINE", "In Denmark's crown have worn. Give me the cups;"),
                                new XElement("LINE", "And let the kettle to the trumpet speak,"),
                                new XElement("LINE", "The trumpet to the cannoneer without,"),
                                new XElement("LINE", "The cannons to the heavens, the heavens to earth,"),
                                new XElement("LINE", "'Now the king dunks to Hamlet.' Come, begin:"),
                                new XElement("LINE", "And you, the judges, bear a wary eye.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Come on, sir.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Come, my lord.")
                            ),
                            new XElement("STAGEDIR", "They play"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "One.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "No.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Judgment.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "A hit, a very palpable hit.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Well; again.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Stay; give me drink. Hamlet, this pearl is thine;"),
                                new XElement("LINE", "Here's to thy health."),
                                new XElement("STAGEDIR", "Trumpets sound, and cannon shot off within"),
                                new XElement("LINE", "Give him the cup.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I'll play this bout first; set it by awhile. Come."),
                                new XElement("STAGEDIR", "They play"),
                                new XElement("LINE", "Another hit; what say you?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "A touch, a touch, I do confess.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Our son shall win.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "He's fat, and scant of breath."),
                                new XElement("LINE", "Here, Hamlet, take my napkin, rub thy brows;"),
                                new XElement("LINE", "The queen carouses to thy fortune, Hamlet.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Good madam!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Gertrude, do not drink.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "I will, my lord; I pray you, pardon me.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Aside"),
                                    "  It is the poison'd cup: it is too late."
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "I dare not drink yet, madam; by and by.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "Come, let me wipe thy face.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "My lord, I'll hit him now.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "I do not think't.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE",
                                    new XElement("STAGEDIR", "Aside"),
                                    "  And yet 'tis almost 'gainst my conscience."
                                )
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Come, for the third, Laertes: you but dally;"),
                                new XElement("LINE", "I pray you, pass with your best violence;"),
                                new XElement("LINE", "I am afeard you make a wanton of me.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Say you so? come on.")
                            ),
                            new XElement("STAGEDIR", "They play"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "Nothing, neither way.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Have at you now!")
                            ),
                            new XElement("STAGEDIR",
                                "LAERTES wounds HAMLET; then in scuffling, they\n" +
                                "change rapiers, and HAMLET wounds LAERTES"
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "Part them; they are incensed.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Nay, come, again.")
                            ),
                            new XElement("STAGEDIR", "QUEEN GERTRUDE falls"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "Look to the queen there, ho!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "They bleed on both sides. How is it, my lord?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "How is't, Laertes?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "Why, as a woodcock to mine own springe, Osric;"),
                                new XElement("LINE", "I am justly kill'd with mine own treachery.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "How does the queen?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "She swounds to see them bleed.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "QUEEN GERTRUDE"),
                                new XElement("LINE", "No, no, the drink, the drink,--O my dear Hamlet,--"),
                                new XElement("LINE", "The drink, the drink! I am poison'd.")
                            ),
                            new XElement("STAGEDIR", "Dies"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "O villany! Ho! let the door be lock'd:"),
                                new XElement("LINE", "Treachery! Seek it out.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "It is here, Hamlet: Hamlet, thou art slain;"),
                                new XElement("LINE", "No medicine in the world can do thee good;"),
                                new XElement("LINE", "In thee there is not half an hour of life;"),
                                new XElement("LINE", "The treacherous instrument is in thy hand,"),
                                new XElement("LINE", "Unbated and envenom'd: the foul practise"),
                                new XElement("LINE", "Hath turn'd itself on me lo, here I lie,"),
                                new XElement("LINE", "Never to rise again: thy mother's poison'd:"),
                                new XElement("LINE", "I can no more: the king, the king's to blame.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "The point!--envenom'd too!"),
                                new XElement("LINE", "Then, venom, to thy work.")
                            ),
                            new XElement("STAGEDIR", "Stabs KING CLAUDIUS"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "All"),
                                new XElement("LINE", "Treason! treason!")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "KING CLAUDIUS"),
                                new XElement("LINE", "O, yet defend me, friends; I am but hurt.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Here, thou incestuous, murderous, damned Dane,"),
                                new XElement("LINE", "Drink off this potion. Is thy union here?"),
                                new XElement("LINE", "Follow my mother.")
                            ),
                            new XElement("STAGEDIR", "KING CLAUDIUS dies"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "LAERTES"),
                                new XElement("LINE", "He is justly served;"),
                                new XElement("LINE", "It is a poison temper'd by himself."),
                                new XElement("LINE", "Exchange forgiveness with me, noble Hamlet:"),
                                new XElement("LINE", "Mine and my father's death come not upon thee,"),
                                new XElement("LINE", "Nor thine on me.")
                            ),
                            new XElement("STAGEDIR", "Dies"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "Heaven make thee free of it! I follow thee."),
                                new XElement("LINE", "I am dead, Horatio. Wretched queen, adieu!"),
                                new XElement("LINE", "You that look pale and tremble at this chance,"),
                                new XElement("LINE", "That are but mutes or audience to this act,"),
                                new XElement("LINE", "Had I but time--as this fell sergeant, death,"),
                                new XElement("LINE", "Is strict in his arrest--O, I could tell you--"),
                                new XElement("LINE", "But let it be. Horatio, I am dead;"),
                                new XElement("LINE", "Thou livest; report me and my cause aright"),
                                new XElement("LINE", "To the unsatisfied.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Never believe it:"),
                                new XElement("LINE", "I am more an antique Roman than a Dane:"),
                                new XElement("LINE", "Here's yet some liquor left.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "As thou'rt a man,"),
                                new XElement("LINE", "Give me the cup: let go; by heaven, I'll have't."),
                                new XElement("LINE", "O good Horatio, what a wounded name,"),
                                new XElement("LINE", "Things standing thus unknown, shall live behind me!"),
                                new XElement("LINE", "If thou didst ever hold me in thy heart"),
                                new XElement("LINE", "Absent thee from felicity awhile,"),
                                new XElement("LINE", "And in this harsh world draw thy breath in pain,"),
                                new XElement("LINE", "To tell my story."),
                                new XElement("STAGEDIR", "March afar off, and shot within"),
                                new XElement("LINE", "What warlike noise is this?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "OSRIC"),
                                new XElement("LINE", "Young Fortinbras, with conquest come from Poland,"),
                                new XElement("LINE", "To the ambassadors of England gives"),
                                new XElement("LINE", "This warlike volley.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HAMLET"),
                                new XElement("LINE", "O, I die, Horatio;"),
                                new XElement("LINE", "The potent poison quite o'er-crows my spirit:"),
                                new XElement("LINE", "I cannot live to hear the news from England;"),
                                new XElement("LINE", "But I do prophesy the election lights"),
                                new XElement("LINE", "On Fortinbras: he has my dying voice;"),
                                new XElement("LINE", "So tell him, with the occurrents, more and less,"),
                                new XElement("LINE", "Which have solicited. The rest is silence.")
                            ),
                            new XElement("STAGEDIR", "Dies"),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Now cracks a noble heart. Good night sweet prince:"),
                                new XElement("LINE", "And flights of angels sing thee to thy rest!"),
                                new XElement("LINE", "Why does the drum come hither?")
                            ),
                            new XElement("STAGEDIR", "March within"),
                            new XElement("STAGEDIR",
                                "Enter FORTINBRAS, the English Ambassadors,\n" +
                                "and others"
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "PRINCE FORTINBRAS"),
                                new XElement("LINE", "Where is this sight?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "What is it ye would see?"),
                                new XElement("LINE", "If aught of woe or wonder, cease your search.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "PRINCE FORTINBRAS"),
                                new XElement("LINE", "This quarry cries on havoc. O proud death,"),
                                new XElement("LINE", "What feast is toward in thine eternal cell,"),
                                new XElement("LINE", "That thou so many princes at a shot"),
                                new XElement("LINE", "So bloodily hast struck?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "First Ambassador"),
                                new XElement("LINE", "The sight is dismal;"),
                                new XElement("LINE", "And our affairs from England come too late:"),
                                new XElement("LINE", "The ears are senseless that should give us hearing,"),
                                new XElement("LINE", "To tell him his commandment is fulfill'd,"),
                                new XElement("LINE", "That Rosencrantz and Guildenstern are dead:"),
                                new XElement("LINE", "Where should we have our thanks?")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Not from his mouth,"),
                                new XElement("LINE", "Had it the ability of life to thank you:"),
                                new XElement("LINE", "He never gave commandment for their death."),
                                new XElement("LINE", "But since, so jump upon this bloody question,"),
                                new XElement("LINE", "You from the Polack wars, and you from England,"),
                                new XElement("LINE", "Are here arrived give order that these bodies"),
                                new XElement("LINE", "High on a stage be placed to the view;"),
                                new XElement("LINE", "And let me speak to the yet unknowing world"),
                                new XElement("LINE", "How these things came about: so shall you hear"),
                                new XElement("LINE", "Of carnal, bloody, and unnatural acts,"),
                                new XElement("LINE", "Of accidental judgments, casual slaughters,"),
                                new XElement("LINE", "Of deaths put on by cunning and forced cause,"),
                                new XElement("LINE", "And, in this upshot, purposes mistook"),
                                new XElement("LINE", "Fall'n on the inventors' reads: all this can I"),
                                new XElement("LINE", "Truly deliver.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "PRINCE FORTINBRAS"),
                                new XElement("LINE", "Let us haste to hear it,"),
                                new XElement("LINE", "And call the noblest to the audience."),
                                new XElement("LINE", "For me, with sorrow I embrace my fortune:"),
                                new XElement("LINE", "I have some rights of memory in this kingdom,"),
                                new XElement("LINE", "Which now to claim my vantage doth invite me.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "HORATIO"),
                                new XElement("LINE", "Of that I shall have also cause to speak,"),
                                new XElement("LINE", "And from his mouth whose voice will draw on more;"),
                                new XElement("LINE", "But let this same be presently perform'd,"),
                                new XElement("LINE", "Even while men's minds are wild; lest more mischance"),
                                new XElement("LINE", "On plots and errors, happen.")
                            ),
                            new XElement("SPEECH",
                                new XElement("SPEAKER", "PRINCE FORTINBRAS"),
                                new XElement("LINE", "Let four captains"),
                                new XElement("LINE", "Bear Hamlet, like a soldier, to the stage;"),
                                new XElement("LINE", "For he was likely, had he been put on,"),
                                new XElement("LINE", "To have proved most royally: and, for his passage,"),
                                new XElement("LINE", "The soldiers' music and the rites of war"),
                                new XElement("LINE", "Speak loudly for him."),
                                new XElement("LINE", "Take up the bodies: such a sight as this"),
                                new XElement("LINE", "Becomes the field, but here shows much amiss."),
                                new XElement("LINE", "Go, bid the soldiers shoot.")
                            ),
                            new XElement("STAGEDIR",
                                "A dead march. Exeunt, bearing off the dead\n" +
                                "bodies; after which a peal of ordnance is shot off")
                            )
                      )
                 )
            );

            xmlDoc = XDocument.Parse(xDoc.ToString(SaveOptions.DisableFormatting));
        }
    }
}
