// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef NET_QUIC_CRYPTO_QUIC_SERVER_INFO_H_
#define NET_QUIC_CRYPTO_QUIC_SERVER_INFO_H_

#include <string>
#include <vector>

#include "base/memory/ref_counted.h"
#include "base/memory/weak_ptr.h"
#include "base/time/time.h"
#include "net/base/completion_callback.h"
#include "net/base/net_export.h"
#include "net/quic/quic_session_key.h"

namespace net {

class X509Certificate;

// QuicServerInfo is an interface for fetching information about a QUIC server.
// This information may be stored on disk so does not include keys or other
// sensitive information. Primarily it's intended for caching the QUIC server's
// crypto config.
class NET_EXPORT_PRIVATE QuicServerInfo {
 public:
  QuicServerInfo(const QuicSessionKey& server_key);
  virtual ~QuicServerInfo();

  // Start will commence the lookup. This must be called before any other
  // methods. By opportunistically calling this early, it may be possible to
  // overlap this object's lookup and reduce latency.
  virtual void Start() = 0;

  // WaitForDataReady returns OK if the fetch of the requested data has
  // completed. Otherwise it returns ERR_IO_PENDING and will call |callback| on
  // the current thread when ready.
  //
  // Only a single callback can be outstanding at a given time and, in the
  // event that WaitForDataReady returns OK, it's the caller's responsibility
  // to delete |callback|.
  //
  // |callback| may be NULL, in which case ERR_IO_PENDING may still be returned
  // but, obviously, a callback will never be made.
  virtual int WaitForDataReady(const CompletionCallback& callback) = 0;

  // Returns true if data is loaded from disk cache and ready (WaitForDataReady
  // doesn't have a pending callback).
  virtual bool IsDataReady() = 0;

  // Returns true if the object is ready to persist data, in other words, if
  // data is loaded from disk cache and ready and there are no pending writes.
  virtual bool IsReadyToPersist() = 0;

  // Persist allows for the server information to be updated for future users.
  // This is a fire and forget operation: the caller may drop its reference
  // from this object and the store operation will still complete. This can
  // only be called once WaitForDataReady has returned OK or called its
  // callback.
  virtual void Persist() = 0;

  struct State {
    State();
    ~State();

    void Clear();

    // This class matches QuicClientCryptoConfig::CachedState.
    std::string server_config;         // A serialized handshake message.
    std::string source_address_token;  // An opaque proof of IP ownership.
    std::vector<std::string> certs;    // A list of certificates in leaf-first
                                       // order.
    std::string server_config_sig;     // A signature of |server_config_|.

   private:
    DISALLOW_COPY_AND_ASSIGN(State);
  };

  // Once the data is ready, it can be read using the following members. These
  // members can then be updated before calling |Persist|.
  const State& state() const;
  State* mutable_state();

 protected:
  // Parse parses pickled data and fills out the public member fields of this
  // object. It returns true iff the parse was successful. The public member
  // fields will be set to something sane in any case.
  bool Parse(const std::string& data);
  std::string Serialize();
  State state_;

 private:
  // ParseInner is a helper function for Parse.
  bool ParseInner(const std::string& data);

  // SerializeInner is a helper function for Serialize.
  std::string SerializeInner() const;

  // This is the QUIC server (hostname, port, is_https) tuple for which we
  // restore the crypto_config.
  const QuicSessionKey server_key_;
};

class QuicServerInfoFactory {
 public:
  virtual ~QuicServerInfoFactory();

  // GetForServer returns a fresh, allocated QuicServerInfo for the given
  // |server_key| or NULL on failure.
  virtual QuicServerInfo* GetForServer(const QuicSessionKey& server_key) = 0;
};

}  // namespace net

#endif  // NET_QUIC_CRYPTO_QUIC_SERVER_INFO_H_
