//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Device/InstrumentModel.cpp
//! @brief     Implement class InstrumentModel
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Device/InstrumentModel.h"
#include "GUI/Model/Device/BackgroundItems.h"
#include "GUI/Model/Device/InstrumentItems.h"
#include "GUI/Support/Util/String.h"
#include <QUuid>

namespace {
namespace Tag {

const QString Instrument("Instrument");
const QString SelectedIndex("SelectedIndex");

} // namespace Tag
} // namespace

InstrumentModel::~InstrumentModel() = default;

void InstrumentModel::clear()
{
    m_instruments.clear();
}

void InstrumentModel::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // instruments
    for (const auto& sel : m_instruments) {
        w->writeStartElement(Tag::Instrument);
        sel.writeTo(w);
        w->writeEndElement();
    }

    // selected index
    w->writeStartElement(Tag::SelectedIndex);
    XML::writeAttribute(w, XML::Attrib::value, m_selectedIndex);
    w->writeEndElement();
}

void InstrumentModel::readFrom(QXmlStreamReader* r)
{
    m_instruments.clear();

    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // instrument
        if (tag == Tag::Instrument) {
            addEmptyInstrumentSelection().readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // selected index
        } else if (tag == Tag::SelectedIndex) {
            XML::readAttribute(r, XML::Attrib::value, &m_selectedIndex);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}

void InstrumentModel::emplace_back(InstrumentItem* item)
{
    addEmptyInstrumentSelection().setCurrentItem(item);
}

InstrumentItem* InstrumentModel::insertItemCopy(const InstrumentItem& source)
{
    auto* copy = source.createItemCopy();
    copy->setId(QUuid::createUuid().toString());
    emplace_back(copy);
    return copy;
}

QVector<InstrumentItem*> InstrumentModel::instrumentItems() const
{
    QVector<InstrumentItem*> output;
    for (const auto& sel : m_instruments)
        output.append(sel.currentItem());
    return output;
}

QStringList InstrumentModel::instrumentNames() const
{
    QStringList existingNames;
    for (const auto* item : instrumentItems())
        existingNames << item->instrumentName();
    return existingNames;
}

SelectionProperty<InstrumentItemCatalog>& InstrumentModel::addEmptyInstrumentSelection()
{
    m_instruments.push_back(nullptr);
    return m_instruments.back();
}

QString InstrumentModel::suggestInstrumentName(const QString& baseName) const
{
    return GUI::Util::String::suggestName(instrumentNames(), baseName);
}

QVector<InstrumentItem*>
InstrumentModel::instrumentItems(const std::function<bool(const InstrumentItem*)>& accept) const
{
    QVector<InstrumentItem*> result;
    for (auto* p : instrumentItems())
        if (accept(p))
            result << p;

    return result;
}

QVector<InstrumentItem*> InstrumentModel::instrument2DItems() const
{
    QVector<InstrumentItem*> result;
    for (auto* p : instrumentItems())
        if (auto* p2D = dynamic_cast<InstrumentItem*>(p))
            result << p2D;

    return result;
}

InstrumentItem* InstrumentModel::findInstrumentItemById(const QString& instrumentId) const
{
    for (auto* instrument : instrumentItems())
        if (instrument->id() == instrumentId)
            return instrument;

    return nullptr;
}

bool InstrumentModel::instrumentExists(const QString& instrumentId) const
{
    return findInstrumentItemById(instrumentId) != nullptr;
}

void InstrumentModel::removeInstrument(InstrumentItem* instrument)
{
    m_instruments.delete_element(instrument);
}
