/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file lastcall.c
 * \brief Contains all function needed for last call list
 */

#include <ffgtk.h>

/** Last call - incoming list */
static GList *psLastcallsIn = NULL;
/** Last call - outgoing list */
static GList *psLastcallsOut = NULL;
/** Last call - missed list */
static GList *psLastcallsMissed = NULL;

/**
 * \brief Get last call list
 * \param nType type of list
 * \return list pointer
 */
GList *getLastcalls( gint nType ) {
	switch ( nType ) {
		case 0:
			return psLastcallsIn;
		case 1:
			return psLastcallsOut;
		case 2:
			return psLastcallsMissed;
	}

	return NULL;
}

/**
 * \brief Add last call from xmlnode
 * \param psNode xml node
 */
static void addLastcall( xmlnode *psNode ) {
	xmlnode *psChild = NULL;
	xmlnode *psChild2 = NULL;
	gchar *pnNumber = NULL;

	psChild = xmlnode_get_child( psNode, "in" );
	if ( psChild != NULL ) {
		for ( psChild2 = xmlnode_get_child( psChild, "number" ); psChild2 != NULL; psChild2 = xmlnode_get_next_twin( psChild2 ) ) {
			pnNumber = xmlnode_get_data( psChild2 );
			psLastcallsIn = g_list_append( psLastcallsIn, g_strdup( pnNumber ) );
			g_free( pnNumber );
		}
	}

	psChild = xmlnode_get_child( psNode, "out" );
	if ( psChild != NULL ) {
		for ( psChild2 = xmlnode_get_child( psChild, "number" ); psChild2 != NULL; psChild2 = xmlnode_get_next_twin( psChild2 ) ) {
			pnNumber = xmlnode_get_data( psChild2 );
			psLastcallsOut = g_list_append( psLastcallsOut, g_strdup( pnNumber ) );
			g_free( pnNumber );
		}
	}

	psChild = xmlnode_get_child( psNode, "missed" );
	if ( psChild != NULL ) {
		for ( psChild2 = xmlnode_get_child( psChild, "number" ); psChild2 != NULL; psChild2 = xmlnode_get_next_twin( psChild2 ) ) {
			pnNumber = xmlnode_get_data( psChild2 );
			psLastcallsMissed = g_list_append( psLastcallsMissed, g_strdup( pnNumber ) );
			g_free( pnNumber );
		}
	}
}

/**
 * \brief Load last calls
 */
void LoadLastcalls( void ) {
	xmlnode *psNode = NULL;
	DIR *psDir = NULL;
	gchar *pnFile = NULL;
	struct sProfile *psProfile = getActiveProfile();

	psDir = opendir( getUserDir() );
	if ( psDir == NULL ) {
		g_mkdir( getUserDir(), 0755 );
	} else {
		closedir( psDir );
	}

	if ( psProfile == NULL || psProfile -> pnName == NULL ) {
		return;
	}

	pnFile = g_build_filename( getProfilesDir(), psProfile -> pnName, "lastcall.xml", NULL );
	if ( pnFile == NULL ) {
		return;
	}

	psNode = readXmlFromFile( pnFile, _( "lastcall" ) );
	if ( psNode == NULL ) {
		Debug( KERN_DEBUG, "Could not read lastcall.xml\n" );
		g_free( pnFile );
		return;
	}

	addLastcall( psNode );

	xmlnode_free( psNode );

	g_free( pnFile );
}

/**
 * \brief Convert last call entry to xml node
 * \param psList last call list
 * \param nType list type
 * \return created xml node
 */
static xmlnode *lastcallToXmlnode( GList *psList, gint nType ) {
	xmlnode *node, *child;

	switch ( nType ) {
		case 0:
			node = xmlnode_new( "in" );
			break;
		case 1:
			node = xmlnode_new( "out" );
			break;
		case 2:
			node = xmlnode_new( "missed" );
			break;
		default:
			Debug( KERN_WARNING, "Wrong type (%d), abort!\n", nType );
			return NULL;
	}

	while ( psList != NULL ) {
		child = xmlnode_new_child( node, "number" );
		xmlnode_insert_data(child, psList -> data, -1 );
		psList = psList -> next;
	}

	return node;
}

/**
 * \brief Create xmlnode from all last call lists
 * \return created xmlnode
 */
static xmlnode *lastcallsToXmlnode(void){
	xmlnode *node, *child;

	node = xmlnode_new( "lastcall" );
	xmlnode_set_attrib( node, "version", "1.0" );

	child = lastcallToXmlnode( psLastcallsIn, 0 );
	xmlnode_insert_child( node, child );
	child = lastcallToXmlnode( psLastcallsOut, 1 );
	xmlnode_insert_child( node, child );
	child = lastcallToXmlnode( psLastcallsMissed, 2 );
	xmlnode_insert_child( node, child );

	return node;
}

/**
 * \brief Save last calls
 */
void SaveLastCalls( void ) {
	xmlnode *node;
	char *pnData;

	node = lastcallsToXmlnode();
	pnData = xmlnode_to_formatted_str( node, NULL );
	if ( pnData != NULL ) {
		int nFile, nResult;
		char *pnFile = g_build_filename( getProfilesDir(), getActiveProfile() -> pnName, "lastcall.xml", NULL );

		nFile = open( pnFile, O_RDWR | O_CREAT | O_TRUNC, 0600 );
		if ( nFile > 0 ) {
			nResult = write( nFile, pnData, strlen( pnData ) );
			if ( nResult != strlen( pnData ) ) {
				Debug( KERN_WARNING, "Could not save file %s\n", pnFile );
			}
			close( nFile );
		}
	}
	g_free( pnData );
	xmlnode_free( node );
}

/**
 * \brief Add new number to list type
 * \param nType list type
 * \param pnNumber number to add
 */
void lastCallAddNumber( gint nType, gchar *pnNumber ) {
	GList *psList = NULL;
	GList *psTmp = NULL;

	if ( pnNumber == NULL || strlen( pnNumber ) <= 0 ) {
		Debug( KERN_DEBUG, "Not adding unknown call\n" );
		return;
	}

	switch ( nType ) {
		case 0:
			psList = psLastcallsIn;
			break;
		case 1:
			psList = psLastcallsOut;
 			break;
		case 2:
			psList = psLastcallsMissed;
			break;
		default:
			Debug( KERN_DEBUG, "Unknown type (%d), abort\n", nType );
			return;
	}

	if ( psList != NULL && g_list_length( psList ) > 3 ) {
		while ( ( psTmp = g_list_nth( psList, 4 ) ) != NULL ) {
			psList = g_list_remove_link( psList, psTmp );
			g_free( psTmp -> data );
			g_list_free( psTmp );
		}
	}

	switch ( nType ) {
		case 0:
			psLastcallsIn = g_list_prepend( psLastcallsIn, g_strdup( pnNumber ) );
			break;
		case 1:
			psLastcallsOut = g_list_prepend( psLastcallsOut, g_strdup( pnNumber ) );
			break;
		case 2:
			psLastcallsMissed = g_list_prepend( psList, g_strdup( pnNumber ) );
			break;
	}

	SaveLastCalls();
}
