/**
 * Author......: See docs/credits.txt
 * License.....: MIT
 */

#include "common.h"
#include "types.h"
#include "modules.h"
#include "bitops.h"
#include "convert.h"
#include "shared.h"
#include "memory.h"

static const u32   ATTACK_EXEC    = ATTACK_EXEC_OUTSIDE_KERNEL;
static const u32   DGST_POS0      = 0;
static const u32   DGST_POS1      = 1;
static const u32   DGST_POS2      = 2;
static const u32   DGST_POS3      = 3;
static const u32   DGST_SIZE      = DGST_SIZE_4_16;
static const u32   HASH_CATEGORY  = HASH_CATEGORY_FDE;
static const char *HASH_NAME      = "LUKS v1 RIPEMD-160 + Serpent";
static const u64   KERN_TYPE      = 14642; // old kernel used here
static const u32   OPTI_TYPE      = OPTI_TYPE_ZERO_BYTE
                                  | OPTI_TYPE_SLOW_HASH_SIMD_LOOP;
static const u64   OPTS_TYPE      = OPTS_TYPE_STOCK_MODULE
                                  | OPTS_TYPE_PT_GENERATE_LE;
static const u32   SALT_TYPE      = SALT_TYPE_EMBEDDED;
static const char *ST_PASS        = "hashcat";
static const char *ST_HASH        = "$luks$1$ripemd160$serpent$xts-plain64$256$107691$de4943f59812487b6fdde7cc4a2126286070f2b658cb3cc3b78f3b6c174f94eb$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$2ab5d1395feac8ca9307d7101c27e4d563574958eb8745a5b3dc1bb26884ea886d313475755843c28babbb96d86dd2079f697647af8ff47a27e564ca68aeb6ce0b5b2e0d5cd36b860fd314714058b45454245e7fa8e7ed269a324be1b04959a883738b27eab0388c5825e6a8e0e1797be9a8467104372b6fe34161e6141ceff562f5d4402629440a71c821ffbf5bcd6b141028a79f4913c2648b664574bcdf5c58e0974e8108868b4d5b1f6f05b13068c7d8b045034cf14ea3e0bb5ca8abc373a4688f2f07eb075b6c965652ac8a3a5ee822c98b1d2c579f58ec8ec9fc23d1a06abc76334b29762175f346b0397b58211184a3bda7d859fe9075da3aa843c2c32a3177f7191e5d001ae9af20dbab1c6bd5154ce9a7f76f917f5e101180878303121552c4537d247b542576c082c039524934692afd45c1b87923c96177dd00019fd89e4d796d1bdcf2b0860fb81815e874943ad6dc722844b6a4f61ee7b33613dd1d1eb6c6b9891d834b79300c377ccee475d179eba065edecd92fd13d5cc1723d32f3de2bb3a896f4e55a13b471b4dc30fdc2ce46a4722c3a9bb8af72af39faa79810421803aeab1745f23b775f1a0fccc268147569082ca26d62db1fa9c8db33fa81b75d455f2d60c4e8aaf621e02a6954654b2bc1dc8333defc9582bb4c990abec0010c9c5f2e4601f7f4e4d83aff42f763763b909c18b206b9ca6997c8b8";

u32         module_attack_exec    (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ATTACK_EXEC;     }
u32         module_dgst_pos0      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS0;       }
u32         module_dgst_pos1      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS1;       }
u32         module_dgst_pos2      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS2;       }
u32         module_dgst_pos3      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS3;       }
u32         module_dgst_size      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_SIZE;       }
u32         module_hash_category  (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_CATEGORY;   }
const char *module_hash_name      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_NAME;       }
u64         module_kern_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return KERN_TYPE;       }
u32         module_opti_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTI_TYPE;       }
u64         module_opts_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTS_TYPE;       }
u32         module_salt_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return SALT_TYPE;       }
const char *module_st_hash        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_HASH;         }
const char *module_st_pass        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_PASS;         }

#define LUKS_STRIPES        (                                   4000)
#define LUKS_SALT_LEN       (                                     32)
#define LUKS_SALT_HEX_LEN   (                      LUKS_SALT_LEN * 2)
#define LUKS_CT_LEN         (                                    512)
#define LUKS_CT_HEX_LEN     (                        LUKS_CT_LEN * 2)
#define LUKS_AF_MIN_LEN     (HC_LUKS_KEY_SIZE_128 / 8 * LUKS_STRIPES)
#define LUKS_AF_MIN_HEX_LEN (                    LUKS_AF_MIN_LEN * 2)
#define LUKS_AF_MAX_LEN     (HC_LUKS_KEY_SIZE_512 / 8 * LUKS_STRIPES)
#define LUKS_AF_MAX_HEX_LEN (                    LUKS_AF_MAX_LEN * 2)

typedef enum hc_luks_hash_type
{
  HC_LUKS_HASH_TYPE_SHA1      = 1,
  HC_LUKS_HASH_TYPE_SHA256    = 2,
  HC_LUKS_HASH_TYPE_SHA512    = 3,
  HC_LUKS_HASH_TYPE_RIPEMD160 = 4,
  HC_LUKS_HASH_TYPE_WHIRLPOOL = 5,

} hc_luks_hash_type_t;

typedef enum hc_luks_key_size
{
  HC_LUKS_KEY_SIZE_128 = 128,
  HC_LUKS_KEY_SIZE_256 = 256,
  HC_LUKS_KEY_SIZE_512 = 512,

} hc_luks_key_size_t;

typedef enum hc_luks_cipher_type
{
  HC_LUKS_CIPHER_TYPE_AES     = 1,
  HC_LUKS_CIPHER_TYPE_SERPENT = 2,
  HC_LUKS_CIPHER_TYPE_TWOFISH = 3,

} hc_luks_cipher_type_t;

typedef enum hc_luks_cipher_mode
{
  HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256 = 1,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN        = 2,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN64      = 3,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN        = 4,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN64      = 5,

} hc_luks_cipher_mode_t;

typedef struct luks
{
  int hash_type;   // hc_luks_hash_type_t
  int key_size;    // hc_luks_key_size_t
  int cipher_type; // hc_luks_cipher_type_t
  int cipher_mode; // hc_luks_cipher_mode_t

  u32 ct_buf[LUKS_CT_LEN / 4];

  u32 af_buf[LUKS_AF_MAX_LEN / 4];
  u32 af_len;

} luks_t;

typedef struct luks_tmp
{
  u32 ipad32[8];
  u64 ipad64[8];

  u32 opad32[8];
  u64 opad64[8];

  u32 dgst32[32];
  u64 dgst64[16];

  u32 out32[32];
  u64 out64[16];

} luks_tmp_t;

static const char *SIGNATURE_LUKS = "$luks$1$ripemd160$serpent$";

bool module_unstable_warning (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra, MAYBE_UNUSED const hc_device_param_t *device_param)
{
  if ((device_param->opencl_platform_vendor_id == VENDOR_ID_APPLE) && (device_param->opencl_device_type & CL_DEVICE_TYPE_GPU))
  {
    if (device_param->is_metal == true)
    {
      if (strncmp (device_param->device_name, "Intel", 5) == 0)
      {
        // Intel Iris Graphics, Metal Version 244.303: failed to create 'm14642_comp' pipeline, timeout reached
        return true;
      }
    }
  }

  return false;
}

u64 module_esalt_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 esalt_size = (const u64) sizeof (luks_t);

  return esalt_size;
}

u64 module_tmp_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 tmp_size = (const u64) sizeof (luks_tmp_t);

  return tmp_size;
}

int module_hash_decode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED void *digest_buf, MAYBE_UNUSED salt_t *salt, MAYBE_UNUSED void *esalt_buf, MAYBE_UNUSED void *hook_salt_buf, MAYBE_UNUSED hashinfo_t *hash_info, const char *line_buf, MAYBE_UNUSED const int line_len)
{
  luks_t *luks = (luks_t *) esalt_buf;

  hc_token_t token;

  memset (&token, 0, sizeof (hc_token_t));

  token.token_cnt  = 7;

  token.signatures_cnt    = 1;
  token.signatures_buf[0] = SIGNATURE_LUKS;

  // signature with hash and cipher type
  token.len[0]     = 26;
  token.attr[0]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_SIGNATURE;

  // cipher mode
  token.sep[1]     = '$';
  token.len_min[1] = 9;
  token.len_max[1] = 16;
  token.attr[1]    = TOKEN_ATTR_VERIFY_LENGTH;

  // key size
  token.sep[2]     = '$';
  token.len[2]     = 3;
  token.attr[2]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // iter
  token.sep[3]     = '$';
  token.len_min[3] = 1;
  token.len_max[3] = 10;
  token.attr[3]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // salt
  token.sep[4]     = '$';
  token.len[4]     = LUKS_SALT_HEX_LEN;
  token.attr[4]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // af
  token.sep[5]     = '$';
  token.len_min[5] = LUKS_AF_MIN_HEX_LEN;
  token.len_max[5] = LUKS_AF_MAX_HEX_LEN;
  token.attr[5]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // ct
  token.sep[6]     = '$';
  token.len[6]     = LUKS_CT_HEX_LEN;
  token.attr[6]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  const int rc_tokenizer = input_tokenizer ((const u8 *) line_buf, line_len, &token);

  if (rc_tokenizer != PARSER_OK) return (rc_tokenizer);

  // hash type

  luks->hash_type = HC_LUKS_HASH_TYPE_RIPEMD160;

  // cipher type

  luks->cipher_type = HC_LUKS_CIPHER_TYPE_SERPENT;

  // cipher mode

  const u8 *cipher_mode_pos = token.buf[1];
  const u32 cipher_mode_len = token.len[1];

  if ((strncmp ((const char *) cipher_mode_pos, "cbc-essiv:sha256", 16) == 0) && (cipher_mode_len == 16))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "cbc-plain",    9) == 0) && (cipher_mode_len ==  9))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "cbc-plain64", 11) == 0) && (cipher_mode_len == 11))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN64;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "xts-plain",    9) == 0) && (cipher_mode_len ==  9))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "xts-plain64", 11) == 0) && (cipher_mode_len == 11))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN64;
  }
  else
  {
    return (PARSER_LUKS_CIPHER_MODE);
  }

  // key size

  const u8 *key_size_pos = token.buf[2];

  const u32 key_size = hc_strtoul ((const char *) key_size_pos, NULL, 10);

  if (key_size == 128)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_128;
  }
  else if (key_size == 256)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_256;
  }
  else if (key_size == 512)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_512;
  }
  else
  {
    return (PARSER_LUKS_KEY_SIZE);
  }

  // salt

  const u8 *salt_pos = token.buf[4];

  salt->salt_len = hex_decode (salt_pos, LUKS_SALT_HEX_LEN, (u8 *) salt->salt_buf);

  // iter

  const u8 *iter_pos = token.buf[3];

  const u32 iter = hc_strtoul ((const char *) iter_pos, NULL, 10);

  salt->salt_iter = iter - 1;

  // af

  u32 af_len = token.len[5] / 2;

  if (af_len != (key_size / 8) * LUKS_STRIPES) return (PARSER_LUKS_FILE_SIZE);

  const u8 *af_pos = token.buf[5];

  luks->af_len = hex_decode (af_pos, token.len[5], (u8 *) luks->af_buf);

  // ct

  u32 ct_len = token.len[6] / 2;

  if (ct_len != LUKS_CT_LEN) return (PARSER_LUKS_FILE_SIZE);

  const u8 *ct_pos = token.buf[6];

  hex_decode (ct_pos, LUKS_CT_HEX_LEN, (u8 *) luks->ct_buf);

  return (PARSER_OK);
}

int module_hash_encode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const void *digest_buf, MAYBE_UNUSED const salt_t *salt, MAYBE_UNUSED const void *esalt_buf, MAYBE_UNUSED const void *hook_salt_buf, MAYBE_UNUSED const hashinfo_t *hash_info, char *line_buf, MAYBE_UNUSED const int line_size)
{
  const luks_t *luks = (const luks_t *) esalt_buf;

  // cipher mode

  const char *cipher_mode = "";

  switch (luks->cipher_mode)
  {
    case HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256: cipher_mode = "cbc-essiv:sha256"; break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN:        cipher_mode = "cbc-plain";        break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN64:      cipher_mode = "cbc-plain64";      break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN:        cipher_mode = "xts-plain";        break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN64:      cipher_mode = "xts-plain64";      break;
    default:                                                                     return 0;
  }

  // key size

  u32 key_size = 0;

  switch (luks->key_size)
  {
    case HC_LUKS_KEY_SIZE_128: key_size = 128; break;
    case HC_LUKS_KEY_SIZE_256: key_size = 256; break;
    case HC_LUKS_KEY_SIZE_512: key_size = 512; break;
    default:                                   return 0;
  }

  // iterations

  u32 iterations = salt->salt_iter + 1;

  // salt

  char salt_buf[LUKS_SALT_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) salt->salt_buf, LUKS_SALT_LEN, (u8 *) salt_buf);

  // af

  char af_buf[LUKS_AF_MAX_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) luks->af_buf, luks->af_len, (u8 *) af_buf);

  // ct

  char ct_buf[LUKS_CT_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) luks->ct_buf, LUKS_CT_LEN, (u8 *) ct_buf);

  // output

  int line_len = snprintf (line_buf, line_size, "%s%s$%u$%u$%s$%s$%s",
    SIGNATURE_LUKS,
    cipher_mode,
    key_size,
    iterations,
    salt_buf,
    af_buf,
    ct_buf);

  return line_len;
}

void module_init (module_ctx_t *module_ctx)
{
  module_ctx->module_context_size             = MODULE_CONTEXT_SIZE_CURRENT;
  module_ctx->module_interface_version        = MODULE_INTERFACE_VERSION_CURRENT;

  module_ctx->module_attack_exec              = module_attack_exec;
  module_ctx->module_benchmark_esalt          = MODULE_DEFAULT;
  module_ctx->module_benchmark_hook_salt      = MODULE_DEFAULT;
  module_ctx->module_benchmark_mask           = MODULE_DEFAULT;
  module_ctx->module_benchmark_charset        = MODULE_DEFAULT;
  module_ctx->module_benchmark_salt           = MODULE_DEFAULT;
  module_ctx->module_bridge_name              = MODULE_DEFAULT;
  module_ctx->module_bridge_type              = MODULE_DEFAULT;
  module_ctx->module_build_plain_postprocess  = MODULE_DEFAULT;
  module_ctx->module_deep_comp_kernel         = MODULE_DEFAULT;
  module_ctx->module_deprecated_notice        = MODULE_DEFAULT;
  module_ctx->module_dgst_pos0                = module_dgst_pos0;
  module_ctx->module_dgst_pos1                = module_dgst_pos1;
  module_ctx->module_dgst_pos2                = module_dgst_pos2;
  module_ctx->module_dgst_pos3                = module_dgst_pos3;
  module_ctx->module_dgst_size                = module_dgst_size;
  module_ctx->module_dictstat_disable         = MODULE_DEFAULT;
  module_ctx->module_esalt_size               = module_esalt_size;
  module_ctx->module_extra_buffer_size        = MODULE_DEFAULT;
  module_ctx->module_extra_tmp_size           = MODULE_DEFAULT;
  module_ctx->module_extra_tuningdb_block     = MODULE_DEFAULT;
  module_ctx->module_forced_outfile_format    = MODULE_DEFAULT;
  module_ctx->module_hash_binary_count        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_parse        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_save         = MODULE_DEFAULT;
  module_ctx->module_hash_decode_postprocess  = MODULE_DEFAULT;
  module_ctx->module_hash_decode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_decode_zero_hash    = MODULE_DEFAULT;
  module_ctx->module_hash_decode              = module_hash_decode;
  module_ctx->module_hash_encode_status       = MODULE_DEFAULT;
  module_ctx->module_hash_encode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_encode              = module_hash_encode;
  module_ctx->module_hash_init_selftest       = MODULE_DEFAULT;
  module_ctx->module_hash_mode                = MODULE_DEFAULT;
  module_ctx->module_hash_category            = module_hash_category;
  module_ctx->module_hash_name                = module_hash_name;
  module_ctx->module_hashes_count_min         = MODULE_DEFAULT;
  module_ctx->module_hashes_count_max         = MODULE_DEFAULT;
  module_ctx->module_hlfmt_disable            = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_size    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_init    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_term    = MODULE_DEFAULT;
  module_ctx->module_hook12                   = MODULE_DEFAULT;
  module_ctx->module_hook23                   = MODULE_DEFAULT;
  module_ctx->module_hook_salt_size           = MODULE_DEFAULT;
  module_ctx->module_hook_size                = MODULE_DEFAULT;
  module_ctx->module_jit_build_options        = MODULE_DEFAULT;
  module_ctx->module_jit_cache_disable        = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_max       = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_min       = MODULE_DEFAULT;
  module_ctx->module_kern_type                = module_kern_type;
  module_ctx->module_kern_type_dynamic        = MODULE_DEFAULT;
  module_ctx->module_opti_type                = module_opti_type;
  module_ctx->module_opts_type                = module_opts_type;
  module_ctx->module_outfile_check_disable    = MODULE_DEFAULT;
  module_ctx->module_outfile_check_nocomp     = MODULE_DEFAULT;
  module_ctx->module_potfile_custom_check     = MODULE_DEFAULT;
  module_ctx->module_potfile_disable          = MODULE_DEFAULT;
  module_ctx->module_potfile_keep_all_hashes  = MODULE_DEFAULT;
  module_ctx->module_pwdump_column            = MODULE_DEFAULT;
  module_ctx->module_pw_max                   = MODULE_DEFAULT;
  module_ctx->module_pw_min                   = MODULE_DEFAULT;
  module_ctx->module_salt_max                 = MODULE_DEFAULT;
  module_ctx->module_salt_min                 = MODULE_DEFAULT;
  module_ctx->module_salt_type                = module_salt_type;
  module_ctx->module_separator                = MODULE_DEFAULT;
  module_ctx->module_st_hash                  = module_st_hash;
  module_ctx->module_st_pass                  = module_st_pass;
  module_ctx->module_tmp_size                 = module_tmp_size;
  module_ctx->module_unstable_warning         = module_unstable_warning;
  module_ctx->module_warmup_disable           = MODULE_DEFAULT;
}
