/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

//
// .NAME:
//  SuperPage
//
// .AUTHOR:
//  Gilberto Camara and Fernando Ii
//
// .SUMMARY:
//  Defines a concrete SuperPage class for the page hierarchy in PlotMod.
//  The SuperPage class is an example a Component class
//  of the Composite "pattern".
//
//  Its methods allow for the insertion and removal of nodes and
//  on a tree structure, and also for the drawing method.
//
//
// .DESCRIPTION:
//  Refer to the "Design Patterns" book, page 163 for more details
//
//
// .DESCENDENT:
//
//
// .RELATED:
//  PlotMod, DataObject, SubPage, Text, Legend
//  Builder, Visitor, Device, DataUnit
//
// .ASCENDENT:
//  Presentable
//
#ifndef SuperPage_H
#define SuperPage_H

#include "Device.h"
#include "Preferences.h"
#include "Presentable.h"

#ifndef NOMETVIEW_QT
class uPlotBase;
#endif

class SuperPage : public Presentable
{
public:
    // Constructors
    SuperPage(const MvRequest& inRequest);
    SuperPage(const SuperPage&);

    virtual Presentable* Clone() const { return new SuperPage(*this); }

    // Destructor
    ~SuperPage();

    // Methods
    // Overriden from Presentable class

    // Create a reply request for creation
    virtual void CreateReply(MvRequest& replyRequest);

    // Respond to a drop action
    virtual void Drop(PmContext& context);

    // Returns the preferences
    virtual Preferences& GetPreferences()
    {
        return preferences_;
    }

    // Set print option: All subpages or visible ones only
    virtual void PrintAll(bool);
    virtual bool PrintAll();

    // Return my Name
    virtual string Name();

    //	virtual Cached SuperPageName()
    //		{ return this->Name(); }

    virtual string MacroPlotIndexName(bool useIndex = false);

    // Returns a pointer to the icon data base
    virtual MvIconDataBase& IconDataBase() const
    {
        return *(iconDataBase_.get());
    }

    //	virtual void    Close();
    // Close all objects associated with the SuperPage

    virtual void RemoveAllData()
    {
        RemoveAllDataChildren();
    }

    virtual void Draw();
    // Draws the superpage and all its children

    virtual void DrawProlog();
    virtual void DrawTrailer(MvRequest&);

    virtual bool IsVisible()
    {
        return true;
    }

    // Implements the visitor pattern
    virtual void Visit(Visitor& v);

    // Returns the device associated to the superpage
    virtual Device& GetDevice() const
    {
        return *myDevice_;
    }

    // Creates the device for the superpage
    virtual void SetDeviceInfo(const MvRequest&);
    virtual void SetDeviceInfo();

    // Indicates the size of the superpage
    virtual PaperSize GetMySize();

    // Retrieve all requests for the whole tree.
    // The requests have already been built and saved internally, but
    // they are not updated automatically if the layers
    // visibility/order/transparency are modified by the uPlot Sidebar.
    // Boolean checkIconStatus = false means use the saved requests;
    // otherwise, rebuild the requests by analysing the whole tree.
    void GetAllRequests(MvRequest&, bool checkIconStatus = false);

    // Rebuild the request by analisying the icon features: order,
    // visibility and transparency.
    void SortRequest(MvRequest&, MvRequest&, bool save_layer = true);

    // Generate a description of the superpage's device
    virtual void DescribeDevice(ObjectInfo& myDescription);

    // Generate a description of the superpage
    virtual void DescribeYourself(ObjectInfo& myDescription);

    // Insert Visdef in the database
    void InsertVisDef(MvRequest&, bool);

    // Call EraseDraw for all children
    //   virtual void EraseDraw ( int visdefId = 0 );

    //	virtual void UpdateWidget ( MvRequest& inRequest, const int treeNodeId, int& newTreeNodeId );
    // Update SuperPage Widget

    // Retrieve Text lists: title and annotation
    // If there is no Text Title here, search up the tree (i.e the Root).
    // If there is no Text Annotation, do not need to search up the tree,
    // because this type of icon is not mandatory to have.
    virtual bool RetrieveTextTitle(MvIconList&);
    void RetrieveTextAnnotation(MvIconList&);

    // Retrieve the Legend
    // If failed, retrieves the root's Legend (default)
    virtual bool RetrieveLegend(MvIcon&);

    // Zoom methods
    //virtual void ZoomRequest ( int id, const Location& zoomCoord );
    virtual void ZoomRequest(int id, const string&);
    virtual void ZoomRequest(int id, int izoom);

    // Layer methods
    virtual bool UpdateLayerTransparency(int, int);
    virtual bool UpdateLayerVisibility(int, bool);
    virtual bool UpdateLayerStackingOrder(int, int);

    // --  Internal methods

    //	void  SetMacroType ( MacroType type )
    //		{ macroType_ = type; }
    // Indicates what is the type of macro

    //	void  SetZoomPageId ( int id )
    //		{ zoomPageId_ = id; }

    // Generate a new macro file based on the plot displayed on the screen
    void GenerateMacro();

    // Provide a list of the names of the pages
    string ListMyPages();

    //	void PrintingFinished ( MvRequest& );
    // Complete the printing job - tell the user

    //	void SaveInFile();
    // Save the contents of the superpage in a file

    virtual void PendingDrawingsAdd()
    {
        pendingDrawings_++;
    }

    virtual void PendingDrawingsRemove()
    {
        pendingDrawings_--;
    }

    //	virtual bool DrawingIsFinished()
    //		{ return  (  pendingDrawings_ == 0 ) &&
    //			  (  hasNewpage_  == true  ); }

    // Returns the number of SuperPages for this SuperPage
    virtual int PageIndex()
    {
        return superPageIndex_;
    }

    // Returns the number of paper pages
    virtual int PaperPageIndex()
    {
        return paperPageIndex_;
    }

    // Sets the number of paper pages
    virtual void PaperPageIndex(int paperPageIndex)
    {
        paperPageIndex_ = paperPageIndex;
    }

    // Increments and returns the number of pictures printed
    virtual int IncrementPrintIndex();

    //        virtual void NeedsRedrawing ( bool yesno );

    //        virtual bool NeedsRedrawing ();

    //        virtual void RedrawIfWindow(int visdefId = 0);
    // Sets redrawing flags and does AddTask if needed.

    virtual void RemoveIcon(MvIcon&);

    //	virtual void SetInputPresentableId ( int );

    //	virtual void SetBackgroundDraw ( bool );

    //	virtual bool GetBackgroundDraw ()
    //		{ return backDrawEnabled_; }

    //	virtual void PrintInfo ();

    // Dealing with Export plotting
    // Save a copy of the plotting. Set sync to true to make sure
    // that the file will be created before returning to the caller.
    virtual bool ExportPlot(MvRequest*, bool sync = false);

    // Get the appropriate requests from the Display Window to
    // be used by ecCharts
    virtual MvRequest ExportToEcCharts();

    // Dealing with Print plotting
    // Recreate the plotting and send it to a given printer
    virtual void PrintFile(MvRequest&);

    // Insert one Page
    virtual Page* InsertOnePage(MvRequest&);
    void InsertOnePage(Page*);

    // Dealing with Contents tools
    virtual void contentsRequest(MvRequest&);

#ifndef NOMETVIEW_QT
    static void setPlotApplication(uPlotBase* pa)
    {
        plotApplication_ = pa;
    }

#ifdef METVIEW_WEATHER_ROOM
    void plotWeatherRoom(bool checkSync = false);
#endif

#endif

private:
    // Copy Constructor and Operator=
    SuperPage& operator=(const SuperPage&);

    // Members
    PaperSize size_;                         // size of superpage in paper coords (cms)
    string macroFileName_;                   // name of the macro file which is produced
    std::unique_ptr<MvIconDataBase> iconDataBase_;  // icon data base (one per superpage)
    std::unique_ptr<Device> myDevice_;              // device (plot media)
    Preferences preferences_;                // preferences associated to the superpage

    int pageIndex_;  // number of pages used in Macro generation

    int superPageIndex_;  // number of Super Pages for this SuperPage,
                          // incremented when receives New Page in a macro
    int paperPageIndex_;  // number of paper pages used for this SuperPage

    //	bool			  backDrawEnabled_; // true if background drawing is enabled

    static int nrSuperPages_;  // Total number of superpages currently active;
    int printIndex_;           // Number of pictures printed

#ifndef NOMETVIEW_QT
    static uPlotBase* plotApplication_;
#endif
};

#endif
