/* nexp_tclvars.c

   Initialize important Tcl global variables, their C counterparts,
   and the variables subsystem.

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"

#define VERBOSE_VARNAME "verbose"
#define TSFORMAT_VARNAME "tsformat"

int vflag; /* Verbose flag */
int tflag = 1; /* Timestamp format flag. See tcpdump(8) for documentation of
		  possible values. */

/*
 * This function gets called whenever a Tcl variable registered with
 * Tcl_TraceVar() is written to. The function just synchronizes the registered
 * Tcl variable with its C counterpart. This way users can modify the Tcl
 * variable and C code can have immediate access to the new value.
 */
static char *
update_int(ClientData clientData, Tcl_Interp *interp, const char *name1,
	   const char *name2 _U_, int flags _U_)
{
    int *intptr;
    Tcl_Obj *obj;

    intptr = clientData;

    obj = Tcl_GetVar2Ex(interp, name1, NULL, 0);
    Tcl_GetIntFromObj(interp, obj, intptr);

    return NULL;
}

/*
 * This function gets called whenever a Tcl variable registered with
 * Tcl_TraceVar() is written to. The function just synchronizes the registered
 * Tcl variable with its C counterpart. This way users can modify the Tcl
 * variable and C code can have immediate access to the new value.
 */
static char *
update_double(ClientData clientData, Tcl_Interp *interp, const char *name1,
	      const char *name2 _U_, int flags _U_)
{
    double *doubleptr;
    Tcl_Obj *obj;

    doubleptr = clientData;

    obj = Tcl_GetVar2Ex(interp, name1, NULL, 0);
    Tcl_GetDoubleFromObj(interp, obj, doubleptr);

    return NULL;
}

int
setup_tcl_vartrace(Tcl_Interp *interp, enum tclvar_types type,
		   const char *name, void *varptr)
{
    Tcl_Obj *obj;
    Tcl_VarTraceProc *trace_proc;

    switch (type) {
    case TCLVAR_INT:
	obj = Tcl_NewIntObj(*(int *) varptr);
	trace_proc = &update_int;
	break;
    case TCLVAR_DOUBLE:
	obj = Tcl_NewDoubleObj(*(double *) varptr);
	trace_proc = &update_double;
	break;
    }

    Tcl_SetVar2Ex(interp, name, NULL, obj, 0);

    return Tcl_TraceVar(interp, name, TCL_TRACE_WRITES, trace_proc, varptr);
}

void
nexp_init_tclvars(Tcl_Interp *interp)
{
    /*
     * The global variable vflag and the Tcl variable "verbose" will
     * always be in sync. We use Tcl_TraceVar() to accomplish this.
     *
     * vflag defaults to 0 but can be modified when the application starts via
     * the -v CLI switch.
     */
    setup_tcl_vartrace(interp, TCLVAR_INT, VERBOSE_VARNAME, &vflag);

    /*
     * The global variable tflag and the Tcl variable "tsformat" will
     * always be in sync. We use Tcl_TraceVar() to accomplish this.
     *
     * vflag defaults to 0 but can be modified when the application starts via
     * the -t CLI switch.
     */
    setup_tcl_vartrace(interp, TCLVAR_INT, TSFORMAT_VARNAME, &tflag);
}
