/* numbers.c
  
   Functions to handle numeric specifications.

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "util.h"
#include "missing.h"

/*
 * This keeps track of the order in which numspecs are created. It must
 * be set to zero before numspecs begin to be created.
 */
int numspec_seq;

/*********************************************************************
 *                               Numbers                             *
 *********************************************************************/

/*
 * Returns the next 32-bit number based on a numeric specification.
 */
uint32_t
num_next(numspec_t *spec)
{
    uint32_t retval;

    if (!spec) return 0;

    switch (spec->type) {
    case NUM_FIXED:
	retval = spec->num_u32.value;
	break;
    case NUM_RANGE:
	retval = spec->num_u32range.current;

	/*
	 * We test for spec->change_freq == 0 because it is possible to
	 * initialize a numspec without registering it. In this case
	 * spec->change_freq will be zero.
	 */
	if (spec->change_freq == 0
	    || ++spec->change_counter % spec->change_freq == 0) {
	    if (spec->num_u32range.start < spec->num_u32range.end) {
		spec->num_u32range.current++;
		if (spec->num_u32range.current > spec->num_u32range.end)
		    spec->num_u32range.current = spec->num_u32range.start;
	    } else {
		spec->num_u32range.current--;
		if (spec->num_u32range.current < spec->num_u32range.end)
		    spec->num_u32range.current = spec->num_u32range.start;
	    }
	}
	break;
    case NUM_RANDOM:
	if (spec->num_u32rand.value) {
	    /*
	     * Pseudo-random. Algorithm stolen from ISIC.
	     */
	    if (rand() <= RAND_MAX*spec->num_u32rand.probability)
		retval = rand();
	    else
		retval = spec->num_u32rand.value;
	} else {
	    if (spec->num_u32rand.end)
		/*
		 * The number specification calls for returning a random
		 * number inside a range.
		 */
		retval = _num_random_int(spec->num_u32rand.start,
					 spec->num_u32rand.end);
	    else
		retval = rand();
	}
	break;
    case NUM_LIST:
	retval = spec->num_u32list.values[spec->num_u32list.curr_index];

	/*
	 * We test for spec->change_freq == 0 because it is possible to
	 * initialize a numspec without registering it. In this case
	 * spec->change_freq will be zero.
	 */
	if (spec->change_freq == 0
	    || ++spec->change_counter % spec->change_freq == 0) {
	    spec->num_u32list.curr_index++;
	    if ( (int) spec->num_u32list.curr_index >= spec->nvalues)
		spec->num_u32list.curr_index = 0;
	}
	break;
    case NUM_ADD:
	retval = spec->num_u32op.current;
	spec->num_u32op.current += spec->num_u32op.operand;
	break;
    case NUM_SUB:
	retval = spec->num_u32op.current;
	spec->num_u32op.current -= spec->num_u32op.operand;
	break;
    case NUM_MUL:
	retval = spec->num_u32op.current;
	spec->num_u32op.current *= spec->num_u32op.operand;
	break;
    case NUM_DIV:
	retval = spec->num_u32op.current;
	spec->num_u32op.current /= spec->num_u32op.operand;
	break;
    case NUM_MOD:
	retval = spec->num_u32op.current;
	spec->num_u32op.current %= spec->num_u32op.operand;
	break;
    case NUM_SHL:
	retval = spec->num_u32op.current;
	spec->num_u32op.current <<= spec->num_u32op.operand;
	break;
    case NUM_SHR:
	retval = spec->num_u32op.current;
	spec->num_u32op.current >>= spec->num_u32op.operand;
	break;
    case IP_FIXED:
    case IP_LIST:
    case IP_RANGE:
    case IP_RANDOM:
    case IP_ADD:
	/* Nothing to with IP addresses in num_next() */
	break;
    } 

    return retval;
}

const char *
num_info(numspec_t *spec)
{
    static char info[128];

    if (spec == NULL) \
	strlcpy(info, "uninitialized number specification", sizeof(info) );
    else
	switch(spec->type) {
	case NUM_FIXED:
	    snprintf(info, sizeof(info), "fixed number, value: %d (0x%x)",
		    spec->num_u32.value, spec->num_u32.value);
	    break;
	case NUM_RANGE:
	    snprintf(info, sizeof(info), "range, from %d to %d (%d values)",
		    spec->num_u32range.start, spec->num_u32range.end,
		    spec->nvalues);
	    break;
	case NUM_RANDOM:
	    if (spec->num_u32rand.value) {
		/* Pseudo-random */
		snprintf(info, sizeof(info),
			 "%d (0x%x), with %.0f%% probability of being random",
			 spec->num_u32rand.value, spec->num_u32rand.value,
			 100.0*spec->num_u32rand.probability);
	    } else {
		if (spec->num_u32rand.end)
		    snprintf(info, sizeof(info),
			     "random number between %d and %d",
			     spec->num_u32rand.start, spec->num_u32rand.end);
		else
		    strlcpy(info, "random number", sizeof(info) );
	    }
	    break;
	case NUM_LIST:
	    snprintf(info, sizeof(info), "list with %d values", spec->nvalues);
	    break;
	case NUM_ADD:
	    snprintf(info, sizeof(info),
		     "number starting from %d, incremented by %d",
		     spec->num_u32op.current, spec->num_u32op.operand);
	    break;
	case NUM_SUB:
	    snprintf(info, sizeof(info),
		     "number starting from %d, decremented by %d",
		     spec->num_u32op.current, spec->num_u32op.operand);
	    break;
	case NUM_MUL:
	    snprintf(info, sizeof(info),
		     "number starting from %d, multiplied by %d",
		     spec->num_u32op.current, spec->num_u32op.operand);
	    break;
	case NUM_DIV:
	    snprintf(info, sizeof(info),
		     "number starting from %d, divided by %d",
		     spec->num_u32op.current, spec->num_u32op.operand);
	    break;
	case NUM_MOD:
	    snprintf(info, sizeof(info),
		     "number starting from %d, modulus %d",
		     spec->num_u32op.current, spec->num_u32op.operand);
	    break;
	case NUM_SHL:
	    snprintf(info, sizeof(info),
		     "number starting from %d, shifted left by %d",
		     spec->num_u32op.current, spec->num_u32op.operand);
	    break;
	case NUM_SHR:
	    snprintf(info, sizeof(info),
		     "number starting from %d, shifted right by %d",
		     spec->num_u32op.current, spec->num_u32op.operand);
	    break;
	case IP_FIXED:
	    strlcpy(info, _num_ipaddr2str(spec->num_ip.value), sizeof(info) );
	    break;
	case IP_RANGE:
	    strlcpy(info, "range, from ", sizeof(info) );
	    strlcat(info, _num_ipaddr2str(spec->num_iprange.start), sizeof(info) );
	    strlcat(info, " to ", sizeof(info) );
	    strlcat(info, _num_ipaddr2str(spec->num_iprange.end), sizeof(info) );
	    break;
	case IP_RANDOM:
	    if (spec->num_iprand.end) {
		strlcpy(info, "random IP address between ", sizeof(info) );
		strlcat(info, _num_ipaddr2str(spec->num_iprand.start),
			sizeof(info) );
		strlcat(info, " and ", sizeof(info) );
		strlcat(info, _num_ipaddr2str(spec->num_iprand.end),
			sizeof(info) );
	    } else
		strlcpy(info, "random IP address", sizeof(info) );
	    break;
	case IP_LIST:
	    snprintf(info, sizeof(info), "list with %d values", spec->nvalues);
	    break;
	case IP_ADD:
	    snprintf(info, sizeof(info),
		     "IP address incremented by 1");
	    break;
	default:
	    strlcpy(info, "invalid number specification", sizeof(info) );
	}

    return info;
}

void
num_destroy(numspec_t *spec)
{
    if (!spec)
	return;

    if (spec->type == NUM_LIST || spec->type == IP_LIST)
	free(spec->type == NUM_LIST ? spec->num_u32list.values
				    : spec->num_iplist.values);

    /* spec->spec was allocated via strdup() so we need to free() it */
    free(spec->spec);
    free(spec);
}

const char *
num_spec(numspec_t *spec)
{
    return spec->spec;
}

/*********************************************************************
 *                          IPv4 addresses                           *
 *********************************************************************/

/*
 * Returns the next IP address based on the IP address specification.
 */
ip_addr_t
ip_next(numspec_t *spec)
{
    ip_addr_t retval;

    if (!spec) return 0;

    switch (spec->type) {
    case IP_FIXED:
	retval = spec->num_ip.value;
	break;
    case IP_ADD:
	retval = spec->num_ipop.current;
	spec->num_ipop.current = htonl(ntohl(spec->num_ipop.current)
				       + spec->num_ipop.operand);
	break;
    case IP_RANGE:
	retval = spec->num_iprange.current;

	/*
	 * We test for spec->change_freq == 0 because it is possible to
	 * initialize a numspec without registering it. In this case
	 * spec->change_freq will be zero.
	 */
	if (spec->change_freq == 0
	    || ++spec->change_counter % spec->change_freq == 0) {
	    if (ntohl(spec->num_iprange.start)
		< ntohl(spec->num_iprange.end) ) {
		spec->num_iprange.current = htonl(ntohl(spec->num_iprange.current) + 1);
		if (ntohl(spec->num_iprange.current)
		    > ntohl(spec->num_iprange.end) )
		    spec->num_iprange.current = spec->num_iprange.start;
	    } else {
		spec->num_iprange.current = htonl(ntohl(spec->num_iprange.current) - 1);
		if (ntohl(spec->num_iprange.current)
		    < ntohl(spec->num_iprange.end) )
		    spec->num_iprange.current = spec->num_iprange.start;
	    }
	}
	break;
    case IP_RANDOM:
	if (spec->num_iprand.end)
	    /*
	     * The IP address specification calls for returning a random
	     * IP address inside an address range.
	     */
	    retval = htonl(ntohl(spec->num_iprand.start)
			   + (rand()
			      % (ntohl(spec->num_iprand.end)
				 - ntohl(spec->num_iprand.start) + 1) ) );
	else
	    retval = rand();
	break;
    case IP_LIST:
	retval = spec->num_iplist.values[spec->num_iplist.curr_index];

	/*
	 * We test for spec->change_freq == 0 because it is possible to
	 * initialize a numspec without registering it. In this case
	 * spec->change_freq will be zero.
	 */
	if (spec->change_freq == 0
	    || ++spec->change_counter % spec->change_freq == 0) {
	    spec->num_iplist.curr_index++;
	    if ( (int) spec->num_iplist.curr_index >= spec->nvalues)
		spec->num_iplist.curr_index = 0;
	}
	break;
    case NUM_FIXED:
    case NUM_LIST:
    case NUM_RANGE:
    case NUM_RANDOM:
    case NUM_ADD:
    case NUM_SUB:
    case NUM_MUL:
    case NUM_DIV:
    case NUM_MOD:
    case NUM_SHL:
    case NUM_SHR:
	/* Nothing to do with numbers in ip_next() */
	break;
    } 

    return retval;
}

int
num_getfreq(numspec_t *spec)
{
    return spec->change_freq;
}

void
num_setfreq(numspec_t *spec, int change_freq)
{
    spec->change_freq = change_freq;
}

int
num_nvalues(numspec_t *spec)
{
    return spec->nvalues;
}

int
num_seq(numspec_t *spec)
{
    return spec->sequence;
}
