"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ensureNodeFactory = void 0;
function ensureNodeFactory(factoryLike) {
    if ("factory" in factoryLike && factoryLike.factory != null) {
        return normalizeNodeFactory(factoryLike.factory);
    }
    else if (!("updateSourceFileNode" in factoryLike)) {
        return normalizeNodeFactory(factoryLike);
    }
    return createNodeFactory(factoryLike);
}
exports.ensureNodeFactory = ensureNodeFactory;
function normalizeNodeFactory(factory) {
    const badCreateImportEqualsDeclaration = factory.createImportEqualsDeclaration.length === 4;
    const badCreateImportSpecifier = factory.createImportSpecifier.length === 2;
    const badCreateExportSpecifier = factory.createExportSpecifier.length === 2;
    const badCreateMappedTypeNodeA = factory.createMappedTypeNode.length === 4;
    const badCreateMappedTypeNodeB = factory.createMappedTypeNode.length === 5;
    const missingCreateClassStaticBlockDeclaration = factory.createClassStaticBlockDeclaration == null;
    const missingCreateAssertClause = factory.createAssertClause == null;
    const missingCreateAssertEntry = factory.createAssertEntry == null;
    const missingCreateJSDocMemberName = factory.createJSDocMemberName == null;
    const missingCreateJSDocLinkCode = factory.createJSDocLinkCode == null;
    const missingCreateJSDocLinkPlain = factory.createJSDocLinkPlain == null;
    const needsModifications = badCreateImportEqualsDeclaration ||
        badCreateImportSpecifier ||
        badCreateExportSpecifier ||
        badCreateMappedTypeNodeA ||
        badCreateMappedTypeNodeB ||
        missingCreateClassStaticBlockDeclaration ||
        missingCreateAssertClause ||
        missingCreateAssertEntry;
    if (needsModifications) {
        /**
         * The following helpers are internal TypeScript helpers that have been inlined for reuse inside factory helpers when the full TypeScript namespace is not available
         */
        const withOriginal = "original" in factory.updateBlock(factory.createBlock([]), []);
        const setOriginalNode = (node, original) => {
            node.original = original;
            return node;
        };
        const setTextRangeEnd = (range, end) => {
            range.end = end;
            return range;
        };
        const setTextRangePos = (range, pos) => {
            range.pos = pos;
            return range;
        };
        const setTextRangePosEnd = (range, pos, end) => setTextRangeEnd(setTextRangePos(range, pos), end);
        const setTextRange = (range, loc) => (loc != null ? setTextRangePosEnd(range, loc.pos, loc.end) : range);
        const updateWithoutOriginal = (updated, original) => {
            if (updated !== original) {
                setTextRange(updated, original);
            }
            return updated;
        };
        const updateWithOriginal = (updated, original) => {
            if (updated !== original) {
                setOriginalNode(updated, original);
                setTextRange(updated, original);
            }
            return updated;
        };
        const update = withOriginal ? updateWithOriginal : updateWithoutOriginal;
        return {
            ...factory,
            ...(badCreateImportEqualsDeclaration
                ? {
                    createImportEqualsDeclaration(decorators, modifiers, isTypeOnly, name, moduleReference) {
                        return factory.createImportEqualsDeclaration(decorators, modifiers, name, moduleReference);
                    },
                    updateImportEqualsDeclaration(node, decorators, modifiers, isTypeOnly, name, moduleReference) {
                        return factory.updateImportEqualsDeclaration(node, decorators, modifiers, name, moduleReference);
                    }
                }
                : {}),
            ...(badCreateImportSpecifier
                ? {
                    createImportSpecifier(isTypeOnly, propertyName, name) {
                        return factory.createImportSpecifier(propertyName, name);
                    },
                    updateImportSpecifier(node, isTypeOnly, propertyName, name) {
                        return factory.updateImportSpecifier(node, propertyName, name);
                    }
                }
                : {}),
            ...(badCreateExportSpecifier
                ? {
                    createExportSpecifier(isTypeOnly, propertyName, name) {
                        return factory.createExportSpecifier(propertyName, name);
                    },
                    updateExportSpecifier(node, isTypeOnly, propertyName, name) {
                        return factory.updateExportSpecifier(node, propertyName, name);
                    }
                }
                : {}),
            ...(badCreateMappedTypeNodeA
                ? {
                    createMappedTypeNode(readonlyToken, typeParameter, nameType, questionToken, type) {
                        return factory.createMappedTypeNode(readonlyToken, typeParameter, questionToken, type);
                    },
                    updateMappedTypeNode(node, readonlyToken, typeParameter, nameType, questionToken, type) {
                        return factory.updateMappedTypeNode(node, readonlyToken, typeParameter, questionToken, type);
                    }
                }
                : {}),
            ...(badCreateMappedTypeNodeB
                ? {
                    createMappedTypeNode(readonlyToken, typeParameter, nameType, questionToken, type, members) {
                        return factory.createMappedTypeNode(readonlyToken, typeParameter, nameType, questionToken, type);
                    },
                    updateMappedTypeNode(node, readonlyToken, typeParameter, nameType, questionToken, type, members) {
                        return factory.updateMappedTypeNode(node, readonlyToken, typeParameter, nameType, questionToken, type);
                    }
                }
                : {}),
            ...(missingCreateClassStaticBlockDeclaration
                ? (() => {
                    function createClassStaticBlockDeclaration(decorators, modifiers, body) {
                        const node = factory.createEmptyStatement();
                        node.body = body;
                        node.decorators = decorators == null ? undefined : factory.createNodeArray(decorators);
                        node.modifiers = modifiers == null ? undefined : factory.createNodeArray(modifiers);
                        node.body = body;
                        node.transformFlags = 8388608 /* ContainsClassFields */;
                        return node;
                    }
                    function updateClassStaticBlockDeclaration(node, decorators, modifiers, body) {
                        return decorators === node.decorators && modifiers === node.modifiers && body === node.body
                            ? node
                            : update(createClassStaticBlockDeclaration(decorators, modifiers, body), node);
                    }
                    return {
                        createClassStaticBlockDeclaration,
                        updateClassStaticBlockDeclaration
                    };
                })()
                : {}),
            ...(missingCreateAssertClause
                ? (() => {
                    function createAssertClause(elements, multiLine) {
                        const node = factory.createEmptyStatement();
                        node.elements = elements;
                        node.multiLine = multiLine;
                        node.transformFlags |= 4 /* ContainsESNext */;
                        return node;
                    }
                    function updateAssertClause(node, elements, multiLine) {
                        return node.elements !== elements || node.multiLine !== multiLine ? update(createAssertClause(elements, multiLine), node) : node;
                    }
                    return {
                        createAssertClause,
                        updateAssertClause
                    };
                })()
                : {}),
            ...(missingCreateAssertEntry
                ? (() => {
                    function createAssertEntry(name, value) {
                        const node = factory.createEmptyStatement();
                        node.name = name;
                        node.value = value;
                        node.transformFlags |= 4 /* ContainsESNext */;
                        return node;
                    }
                    function updateAssertEntry(node, name, value) {
                        return node.name !== name || node.value !== value ? update(createAssertEntry(name, value), node) : node;
                    }
                    return {
                        createAssertEntry,
                        updateAssertEntry
                    };
                })()
                : {}),
            ...(missingCreateJSDocMemberName
                ? (() => {
                    function createJSDocMemberName(left, right) {
                        const base = factory.createJSDocComment(undefined, undefined);
                        delete base.comment;
                        delete base.tags;
                        const node = base;
                        node.left = left;
                        node.right = right;
                        return node;
                    }
                    function updateJSDocMemberName(node, left, right) {
                        return left === node.left && right === node.right ? node : update(createJSDocMemberName(left, right), node);
                    }
                    return {
                        createJSDocMemberName,
                        updateJSDocMemberName
                    };
                })()
                : {}),
            ...(missingCreateJSDocLinkCode
                ? (() => {
                    function createJSDocLinkCode(name, text) {
                        const base = factory.createJSDocComment(undefined, undefined);
                        delete base.comment;
                        delete base.tags;
                        const node = base;
                        node.name = name;
                        node.text = text;
                        return node;
                    }
                    function updateJSDocLinkCode(node, name, text) {
                        return name === node.name && text === node.text ? node : update(createJSDocLinkCode(name, text), node);
                    }
                    return {
                        createJSDocLinkCode,
                        updateJSDocLinkCode
                    };
                })()
                : {}),
            ...(missingCreateJSDocLinkPlain
                ? (() => {
                    function createJSDocLinkPlain(name, text) {
                        const base = factory.createJSDocComment(undefined, undefined);
                        delete base.comment;
                        delete base.tags;
                        const node = base;
                        node.name = name;
                        node.text = text;
                        return node;
                    }
                    function updateJSDocLinkPlain(node, name, text) {
                        return name === node.name && text === node.text ? node : update(createJSDocLinkPlain(name, text), node);
                    }
                    return {
                        createJSDocLinkPlain,
                        updateJSDocLinkPlain
                    };
                })()
                : {})
        };
    }
    return factory;
}
function createNodeFactory(typescript) {
    function createToken(token) {
        return typescript.createToken(token);
    }
    function createConstructorTypeNode(modifiersOrTypeParameters, typeParametersOrParameters, parametersOrType, typeOrUndefined) {
        if (arguments.length >= 4) {
            return typescript.createConstructorTypeNode(typeParametersOrParameters, parametersOrType, typeOrUndefined);
        }
        return typescript.createConstructorTypeNode(modifiersOrTypeParameters, typeParametersOrParameters, parametersOrType);
    }
    function updateConstructorTypeNode(node, modifiersOrTypeParameters, typeParametersOrParameters, parametersOrType, typeOrUndefined) {
        if (arguments.length >= 5) {
            return typescript.updateConstructorTypeNode(node, typeParametersOrParameters, parametersOrType, typeOrUndefined);
        }
        return typescript.updateConstructorTypeNode(node, modifiersOrTypeParameters, typeParametersOrParameters, parametersOrType);
    }
    function createNamedTupleMember(dotDotDotToken, name, questionToken, type) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.NamedTupleMember) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.TupleType);
        node.dotDotDotToken = dotDotDotToken;
        node.name = name;
        node.questionToken = questionToken;
        node.type = type;
        node.transformFlags = 1 /* ContainsTypeScript */;
        return node;
    }
    function createJSDocComment(comment, tags) {
        if ("createJSDocComment" in typescript) {
            return typescript.createJSDocComment(comment, tags);
        }
        const node = typescript.createNode(typescript.SyntaxKind.JSDocComment);
        node.comment = comment;
        node.tags = typescript.createNodeArray(tags);
        return node;
    }
    function createJSDocParameterTag(tagName, name, isBracketed, typeExpression, isNameFirst, comment) {
        var _a;
        if ("createJSDocParameterTag" in typescript) {
            return typescript.createJSDocParameterTag(tagName, name, isBracketed, typeExpression, isNameFirst, comment);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocParameterTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        if (tagName != null)
            node.tagName = tagName;
        node.name = name;
        node.isBracketed = isBracketed;
        node.typeExpression = typeExpression;
        if (isNameFirst != null)
            node.isNameFirst = isNameFirst;
        node.comment = comment;
        return node;
    }
    function createJSDocPrivateTag(tagName, comment) {
        var _a;
        if ("createJSDocPrivateTag" in typescript) {
            return typescript.createJSDocPrivateTag(tagName, comment);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocPrivateTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        if (tagName != null)
            node.tagName = tagName;
        node.comment = comment;
        return node;
    }
    function createJSDocAugmentsTag(tagName, className, comment) {
        var _a;
        if ("createJSDocAugmentsTag" in typescript) {
            return typescript.createJSDocAugmentsTag(tagName, className, comment);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocAugmentsTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        if (tagName != null)
            node.tagName = tagName;
        node.class = className;
        node.comment = comment;
        return node;
    }
    function createJSDocDeprecatedTag(tagName, comment) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocDeprecatedTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        node.tagName = tagName;
        node.comment = comment;
        return node;
    }
    function createJSDocFunctionType(parameters, type) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocFunctionType) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        node.parameters = typescript.createNodeArray(parameters);
        node.type = type;
        return node;
    }
    function createJSDocLink(name, text) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocLink) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        node.name = name;
        node.text = text;
        return node;
    }
    function createJSDocNameReference(name) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocNameReference) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        node.name = name;
        return node;
    }
    function createJSDocNamepathType(type) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocNamepathType) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        node.type = type;
        return node;
    }
    function createJSDocNonNullableType(type) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocNonNullableType) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        node.type = type;
        return node;
    }
    function createJSDocNullableType(type) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocNullableType) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        node.type = type;
        return node;
    }
    function createJSDocOptionalType(type) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocOptionalType) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        node.type = type;
        return node;
    }
    function createJSDocOverrideTag(tagName, comment) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocOverrideTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        node.tagName = tagName;
        node.comment = comment;
        return node;
    }
    function createJSDocSeeTag(tagName, nameExpression, comment) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocSeeTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        if (tagName != null) {
            node.tagName = tagName;
        }
        node.name = nameExpression;
        node.comment = comment;
        return node;
    }
    function createJSDocText(text) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocText) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        node.text = text;
        return node;
    }
    function createJSDocUnknownTag(tagName, comment) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        node.tagName = tagName;
        node.comment = comment;
        return node;
    }
    function createJSDocUnknownType() {
        var _a;
        return typescript.createNode((_a = typescript.SyntaxKind.JSDocUnknownType) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
    }
    function createJSDocVariadicType(type) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocVariadicType) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        node.type = type;
        return node;
    }
    function createJSDocAllType() {
        var _a;
        return typescript.createNode((_a = typescript.SyntaxKind.JSDocAllType) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
    }
    function createTemplateLiteralType(head, templateSpans) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.TemplateLiteralType) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.StringLiteral);
        node.head = head;
        node.templateSpans = typescript.createNodeArray(templateSpans);
        node.transformFlags = 1 /* ContainsTypeScript */;
        return node;
    }
    function createTemplateLiteralTypeSpan(type, literal) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.TemplateLiteralTypeSpan) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.StringLiteral);
        node.type = type;
        node.literal = literal;
        node.transformFlags = 1 /* ContainsTypeScript */;
        return node;
    }
    function createJSDocAuthorTag(tagName, comment) {
        var _a;
        if ("createJSDocAuthorTag" in typescript) {
            return typescript.createJSDocAuthorTag(tagName, comment);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocAuthorTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        if (tagName != null)
            node.tagName = tagName;
        node.comment = comment;
        return node;
    }
    function createJSDocCallbackTag(tagName, typeExpression, fullName, comment) {
        var _a;
        if ("createJSDocCallbackTag" in typescript) {
            return typescript.createJSDocCallbackTag(tagName, typeExpression, fullName, comment);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocCallbackTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        if (tagName != null)
            node.tagName = tagName;
        node.typeExpression = typeExpression;
        node.fullName = fullName;
        node.comment = comment;
        return node;
    }
    function createJSDocClassTag(tagName, comment) {
        var _a;
        if ("createJSDocClassTag" in typescript) {
            return typescript.createJSDocClassTag(tagName, comment);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocClassTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        if (tagName != null)
            node.tagName = tagName;
        node.comment = comment;
        return node;
    }
    function createJSDocEnumTag(tagName, typeExpression, comment) {
        var _a;
        if ("createJSDocEnumTag" in typescript) {
            return typescript.createJSDocEnumTag(tagName, typeExpression, comment);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocEnumTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        if (tagName != null)
            node.tagName = tagName;
        node.typeExpression = typeExpression;
        node.comment = comment;
        return node;
    }
    function createJSDocImplementsTag(tagName, className, comment) {
        var _a;
        if ("createJSDocImplementsTag" in typescript) {
            return typescript.createJSDocImplementsTag(tagName, className, comment);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocImplementsTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        if (tagName != null)
            node.tagName = tagName;
        node.class = className;
        node.comment = comment;
        return node;
    }
    function createJSDocPropertyTag(tagName, name, isBracketed, typeExpression, isNameFirst, comment) {
        var _a;
        if ("createJSDocPropertyTag" in typescript) {
            return typescript.createJSDocPropertyTag(tagName, name, isBracketed, typeExpression, isNameFirst, comment);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocPropertyTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        if (tagName != null)
            node.tagName = tagName;
        node.name = name;
        node.isBracketed = isBracketed;
        node.typeExpression = typeExpression;
        if (isNameFirst != null)
            node.isNameFirst = isNameFirst;
        node.comment = comment;
        return node;
    }
    function createJSDocProtectedTag(tagName, comment) {
        var _a;
        if ("createJSDocProtectedTag" in typescript) {
            return typescript.createJSDocProtectedTag(tagName, comment);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocProtectedTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        if (tagName != null)
            node.tagName = tagName;
        node.comment = comment;
        return node;
    }
    function createJSDocPublicTag(tagName, comment) {
        var _a;
        if ("createJSDocPublicTag" in typescript) {
            return typescript.createJSDocPublicTag(tagName, comment);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocPublicTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        if (tagName != null)
            node.tagName = tagName;
        node.comment = comment;
        return node;
    }
    function createJSDocReadonlyTag(tagName, comment) {
        var _a;
        if ("createJSDocReadonlyTag" in typescript) {
            return typescript.createJSDocReadonlyTag(tagName, comment);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocReadonlyTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        if (tagName != null)
            node.tagName = tagName;
        node.comment = comment;
        return node;
    }
    function createJSDocReturnTag(tagName, typeExpression, comment) {
        var _a;
        if ("createJSDocReturnTag" in typescript) {
            return typescript.createJSDocReturnTag(tagName, typeExpression, comment);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocReturnTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        if (tagName != null)
            node.tagName = tagName;
        node.typeExpression = typeExpression;
        node.comment = comment;
        return node;
    }
    function createJSDocSignature(typeParameters, parameters, type) {
        var _a;
        if ("createJSDocSignature" in typescript) {
            return typescript.createJSDocSignature(typeParameters, parameters, type);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocSignature) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        node.typeParameters = typeParameters;
        node.parameters = parameters;
        node.type = type;
        return node;
    }
    function createJSDocTemplateTag(tagName, constraint, typeParameters, comment) {
        var _a;
        if ("createJSDocTemplateTag" in typescript) {
            return typescript.createJSDocTemplateTag(tagName, constraint, typeParameters, comment);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocTemplateTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        if (tagName != null)
            node.tagName = tagName;
        node.constraint = constraint;
        node.typeParameters = typescript.createNodeArray(typeParameters);
        node.comment = comment;
        return node;
    }
    function createJSDocThisTag(tagName, typeExpression, comment) {
        var _a;
        if ("createJSDocThisTag" in typescript) {
            return typescript.createJSDocThisTag(tagName, typeExpression, comment);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocThisTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        if (tagName != null)
            node.tagName = tagName;
        node.typeExpression = typeExpression;
        node.comment = comment;
        return node;
    }
    function createJSDocTypeExpression(type) {
        var _a;
        if ("createJSDocTypeExpression" in typescript) {
            return typescript.createJSDocTypeExpression(type);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocTypeExpression) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        node.type = type;
        return node;
    }
    function createJSDocTypeLiteral(jsDocPropertyTags, isArrayType) {
        var _a;
        if ("createJSDocTypeLiteral" in typescript) {
            return typescript.createJSDocTypeLiteral(jsDocPropertyTags, isArrayType);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocTypeLiteral) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        node.jsDocPropertyTags = jsDocPropertyTags;
        if (isArrayType != null)
            node.isArrayType = isArrayType;
        return node;
    }
    function createJSDocTypeTag(tagName, typeExpression, comment) {
        var _a;
        if ("createJSDocTypeTag" in typescript) {
            return typescript.createJSDocTypeTag(tagName, typeExpression, comment);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocTypeTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        if (tagName != null)
            node.tagName = tagName;
        node.typeExpression = typeExpression;
        node.comment = comment;
        return node;
    }
    function createJSDocTypedefTag(tagName, typeExpression, fullName, comment) {
        var _a;
        if ("createJSDocTypedefTag" in typescript) {
            return typescript.createJSDocTypedefTag(tagName, typeExpression, fullName, comment);
        }
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocTypedefTag) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        if (tagName != null)
            node.tagName = tagName;
        node.typeExpression = typeExpression;
        node.fullName = fullName;
        node.comment = comment;
        return node;
    }
    function createJSDocMemberName(left, right) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocMemberName) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        node.left = left;
        node.right = right;
        return node;
    }
    function createJSDocLinkCode(name, text) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocLinkCode) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        node.name = name;
        node.text = text;
        return node;
    }
    function createJSDocLinkPlain(name, text) {
        var _a;
        const node = typescript.createNode((_a = typescript.SyntaxKind.JSDocLinkPlain) !== null && _a !== void 0 ? _a : typescript.SyntaxKind.JSDocComment);
        node.name = name;
        node.text = text;
        return node;
    }
    function createClassStaticBlockDeclaration(decorators, modifiers, body) {
        const node = typescript.createEmptyStatement();
        node.body = body;
        node.decorators = decorators == null ? undefined : typescript.createNodeArray(decorators);
        node.modifiers = modifiers == null ? undefined : typescript.createNodeArray(modifiers);
        node.transformFlags = 8388608 /* ContainsClassFields */;
        return node;
    }
    function createAssertClause(elements, multiLine) {
        const node = typescript.createEmptyStatement();
        node.elements = elements;
        node.multiLine = multiLine;
        node.transformFlags |= 4 /* ContainsESNext */;
        return node;
    }
    function createAssertEntry(name, value) {
        const node = typescript.createEmptyStatement();
        node.name = name;
        node.value = value;
        node.transformFlags |= 4 /* ContainsESNext */;
        return node;
    }
    const { updateSourceFileNode, ...common } = typescript;
    return {
        ...common,
        createToken,
        createConstructorTypeNode,
        updateConstructorTypeNode,
        createJSDocComment,
        createJSDocParameterTag,
        createJSDocPrivateTag,
        createJSDocAugmentsTag,
        createJSDocAuthorTag,
        createJSDocCallbackTag,
        createJSDocClassTag,
        createJSDocEnumTag,
        createJSDocImplementsTag,
        createJSDocPropertyTag,
        createJSDocProtectedTag,
        createJSDocPublicTag,
        createJSDocReadonlyTag,
        createJSDocReturnTag,
        createJSDocSignature,
        createJSDocTemplateTag,
        createJSDocThisTag,
        createJSDocTypeExpression,
        createJSDocTypeLiteral,
        createJSDocTypeTag,
        createJSDocTypedefTag,
        createJSDocAllType,
        createJSDocDeprecatedTag,
        createJSDocFunctionType,
        createJSDocLink,
        createJSDocNameReference,
        createJSDocNamepathType,
        createJSDocNonNullableType,
        createJSDocNullableType,
        createJSDocOptionalType,
        createJSDocOverrideTag,
        createJSDocSeeTag,
        createJSDocText,
        createJSDocUnknownTag,
        createJSDocUnknownType,
        createJSDocVariadicType,
        createJSDocMemberName,
        createJSDocLinkCode,
        createJSDocLinkPlain,
        createTemplateLiteralType,
        createTemplateLiteralTypeSpan,
        createClassStaticBlockDeclaration,
        createAssertClause,
        createAssertEntry,
        createComma(left, right) {
            return typescript.createComma(left, right);
        },
        createAssignment(left, right) {
            return typescript.createAssignment(left, right);
        },
        createLessThan(left, right) {
            return typescript.createLessThan(left, right);
        },
        createSourceFile(statements, endOfFileToken, flags) {
            const sourceFile = typescript.createSourceFile("", "", 0, undefined, 0);
            sourceFile.endOfFileToken = endOfFileToken;
            sourceFile.flags |= flags;
            sourceFile.statements = typescript.createNodeArray(statements);
            return sourceFile;
        },
        createClassExpression(decorators, modifiers, name, typeParameters, heritageClauses, members) {
            return typescript.createClassExpression(modifiers, name, typeParameters, heritageClauses, members);
        },
        createExpressionWithTypeArguments(expression, typeArguments) {
            return typescript.createExpressionWithTypeArguments(typeArguments, expression);
        },
        updateExpressionWithTypeArguments(node, expression, typeArguments) {
            return typescript.updateExpressionWithTypeArguments(node, typeArguments, expression);
        },
        updateImportClause(node, isTypeOnly, name, namedBindings) {
            return typescript.updateImportClause(node, name, namedBindings, isTypeOnly);
        },
        updateExportDeclaration(node, decorators, modifiers, isTypeOnly, exportClause, moduleSpecifier) {
            return typescript.updateExportDeclaration(node, decorators, modifiers, exportClause, moduleSpecifier, isTypeOnly);
        },
        createTypePredicateNode(assertsModifier, parameterName, type) {
            return typescript.createTypePredicateNode(parameterName, type);
        },
        updateTypePredicateNode(node, assertsModifier, parameterName, type) {
            return typescript.updateTypePredicateNode(node, parameterName, type);
        },
        createMethodSignature(modifiers, name, questionToken, typeParameters, parameters, type) {
            const methodSignature = typescript.createMethodSignature(typeParameters, parameters, type, name, questionToken);
            // Also set the modifiers
            // Workaround for: https://github.com/microsoft/TypeScript/issues/35959
            if (modifiers != null) {
                methodSignature.modifiers = typescript.createNodeArray(modifiers);
            }
            return methodSignature;
        },
        updateMethodSignature(node, modifiers, name, questionToken, typeParameters, parameters, type) {
            const methodSignature = typescript.updateMethodSignature(node, typeParameters, parameters, type, name, questionToken);
            // Also set the modifiers
            // Workaround for: https://github.com/microsoft/TypeScript/issues/35959
            if (modifiers !== methodSignature.modifiers) {
                methodSignature.modifiers = modifiers == null ? modifiers : typescript.createNodeArray(modifiers);
            }
            return methodSignature;
        },
        updatePropertySignature(node, modifiers, name, questionToken, type) {
            return typescript.updatePropertySignature(node, modifiers, name, questionToken, type, undefined);
        },
        createAwaitExpression(expression) {
            return typescript.createAwait(expression);
        },
        createBinaryExpression(left, operator, right) {
            return typescript.createBinary(left, operator, right);
        },
        createBitwiseAnd(left, right) {
            return typescript.createBinary(left, typescript.SyntaxKind.AmpersandToken, right);
        },
        createBitwiseNot(operand) {
            return typescript.createPrefix(typescript.SyntaxKind.TildeToken, operand);
        },
        createBitwiseOr(left, right) {
            return typescript.createBinary(left, typescript.SyntaxKind.BarToken, right);
        },
        createBitwiseXor(left, right) {
            return typescript.createBinary(left, typescript.SyntaxKind.CaretToken, right);
        },
        createBreakStatement(label) {
            return typescript.createBreak(label);
        },
        createCommaListExpression(elements) {
            return typescript.createCommaList(elements);
        },
        createConditionalExpression(condition, questionToken, whenTrue, colonToken, whenFalse) {
            if (questionToken == null || colonToken == null) {
                return typescript.createConditional(condition, whenTrue, whenFalse);
            }
            return typescript.createConditional(condition, questionToken, whenTrue, colonToken, whenFalse);
        },
        createConstructorDeclaration(decorators, modifiers, parameters, body) {
            return typescript.createConstructor(decorators, modifiers, parameters, body);
        },
        createContinueStatement(label) {
            return typescript.createContinue(label);
        },
        createDeleteExpression(expression) {
            return typescript.createDelete(expression);
        },
        createDivide(left, right) {
            return typescript.createBinary(left, typescript.SyntaxKind.SlashToken, right);
        },
        createDoStatement(statement, expression) {
            return typescript.createDo(statement, expression);
        },
        createElementAccessExpression(expression, index) {
            return typescript.createElementAccess(expression, index);
        },
        createEquality(left, right) {
            return typescript.createBinary(left, typescript.SyntaxKind.EqualsEqualsToken, right);
        },
        createExponent(left, right) {
            return typescript.createBinary(left, typescript.SyntaxKind.AsteriskAsteriskToken, right);
        },
        createForInStatement(initializer, expression, statement) {
            return typescript.createForIn(initializer, expression, statement);
        },
        createForOfStatement(awaitModifier, initializer, expression, statement) {
            return typescript.createForOf(awaitModifier, initializer, expression, statement);
        },
        createForStatement(initializer, condition, incrementor, statement) {
            return typescript.createFor(initializer, condition, incrementor, statement);
        },
        createGreaterThan(left, right) {
            return typescript.createBinary(left, typescript.SyntaxKind.GreaterThanToken, right);
        },
        createGreaterThanEquals(left, right) {
            return typescript.createBinary(left, typescript.SyntaxKind.GreaterThanEqualsToken, right);
        },
        createIfStatement(expression, thenStatement, elseStatement) {
            return typescript.createIf(expression, thenStatement, elseStatement);
        },
        createInequality(left, right) {
            return typescript.createBinary(left, typescript.SyntaxKind.ExclamationEqualsToken, right);
        },
        createLabeledStatement(label, statement) {
            return typescript.createLabel(label, statement);
        },
        createLeftShift(left, right) {
            return typescript.createBinary(left, typescript.SyntaxKind.LessThanLessThanToken, right);
        },
        createLessThanEquals(left, right) {
            return typescript.createBinary(left, typescript.SyntaxKind.LessThanEqualsToken, right);
        },
        createMethodDeclaration(decorators, modifiers, asteriskToken, name, questionToken, typeParameters, parameters, type, body) {
            return typescript.createMethod(decorators, modifiers, asteriskToken, name, questionToken, typeParameters, parameters, type, body);
        },
        createModulo(left, right) {
            return typescript.createBinary(left, typescript.SyntaxKind.PercentToken, right);
        },
        createMultiply(left, right) {
            return typescript.createBinary(left, typescript.SyntaxKind.AsteriskToken, right);
        },
        createNamedTupleMember,
        createNewExpression(expression, typeArguments, argumentsArray) {
            return typescript.createNew(expression, typeArguments, argumentsArray);
        },
        createParameterDeclaration(decorators, modifiers, dotDotDotToken, name, questionToken, type, initializer) {
            return typescript.createParameter(decorators, modifiers, dotDotDotToken, name, questionToken, type, initializer);
        },
        createParenthesizedExpression(expression) {
            return typescript.createParen(expression);
        },
        createPostfixDecrement(operand) {
            return typescript.createPostfix(operand, typescript.SyntaxKind.MinusMinusToken);
        },
        createPostfixUnaryExpression(operand, operator) {
            return typescript.createPostfix(operand, operator);
        },
        createPrefixDecrement(operand) {
            return typescript.createPrefix(typescript.SyntaxKind.MinusMinusToken, operand);
        },
        createPrefixIncrement(operand) {
            return typescript.createPrefix(typescript.SyntaxKind.PlusPlusToken, operand);
        },
        createPrefixMinus(operand) {
            return typescript.createPrefix(typescript.SyntaxKind.MinusToken, operand);
        },
        createPrefixPlus(operand) {
            return typescript.createPrefix(typescript.SyntaxKind.PlusToken, operand);
        },
        createPrefixUnaryExpression(operator, operand) {
            return typescript.createPrefix(operator, operand);
        },
        createPropertyDeclaration(decorators, modifiers, name, questionOrExclamationToken, type, initializer) {
            return typescript.createProperty(decorators, modifiers, name, questionOrExclamationToken, type, initializer);
        },
        createRightShift(left, right) {
            return typescript.createBinary(left, typescript.SyntaxKind.GreaterThanGreaterThanToken, right);
        },
        createSetAccessorDeclaration(decorators, modifiers, name, parameters, body) {
            return typescript.createSetAccessor(decorators, modifiers, name, parameters, body);
        },
        createSpreadElement(expression) {
            return typescript.createSpread(expression);
        },
        createSwitchStatement(expression, caseBlock) {
            return typescript.createSwitch(expression, caseBlock);
        },
        createTaggedTemplateExpression(tag, typeArguments, template) {
            return typescript.createTaggedTemplate(tag, typeArguments, template);
        },
        createThrowStatement(expression) {
            return typescript.createThrow(expression);
        },
        createTryStatement(tryBlock, catchClause, finallyBlock) {
            return typescript.createTry(tryBlock, catchClause, finallyBlock);
        },
        createTypeOfExpression(expression) {
            return typescript.createTypeOf(expression);
        },
        createUnsignedRightShift(left, right) {
            return typescript.createBinary(left, typescript.SyntaxKind.GreaterThanGreaterThanGreaterThanToken, right);
        },
        createVoidExpression(expression) {
            return typescript.createVoid(expression);
        },
        createWhileStatement(expression, statement) {
            return typescript.createWhile(expression, statement);
        },
        createWithStatement(expression, statement) {
            return typescript.createWith(expression, statement);
        },
        createYieldExpression(asteriskToken, expression) {
            return typescript.createYield(asteriskToken, expression);
        },
        restoreOuterExpressions(outerExpression, innerExpression, kinds) {
            return innerExpression;
        },
        updateAwaitExpression(node, expression) {
            return typescript.updateAwait(node, expression);
        },
        updateBinaryExpression(node, left, operator, right) {
            return typescript.updateBinary(node, left, right, operator);
        },
        updateBreakStatement(node, label) {
            return typescript.updateBreak(node, label);
        },
        updateCommaListExpression(node, elements) {
            return typescript.updateCommaList(node, elements);
        },
        updateConditionalExpression(node, condition, questionToken, whenTrue, colonToken, whenFalse) {
            return typescript.updateConditional(node, condition, questionToken, whenTrue, colonToken, whenFalse);
        },
        updateContinueStatement(node, label) {
            return typescript.updateContinue(node, label);
        },
        updateDeleteExpression(node, expression) {
            return typescript.updateDelete(node, expression);
        },
        updateDoStatement(node, statement, expression) {
            return typescript.updateDo(node, statement, expression);
        },
        updateElementAccessExpression(node, expression, argumentExpression) {
            return typescript.updateElementAccess(node, expression, argumentExpression);
        },
        updateForInStatement(node, initializer, expression, statement) {
            return typescript.updateForIn(node, initializer, expression, statement);
        },
        updateForOfStatement(node, awaitModifier, initializer, expression, statement) {
            return typescript.updateForOf(node, awaitModifier, initializer, expression, statement);
        },
        updateForStatement(node, initializer, condition, incrementor, statement) {
            return typescript.updateFor(node, initializer, condition, incrementor, statement);
        },
        updateIfStatement(node, expression, thenStatement, elseStatement) {
            return typescript.updateIf(node, expression, thenStatement, elseStatement);
        },
        updateJSDocAugmentsTag(node, tagName, className, comment) {
            return tagName === node.tagName && className === node.class && comment === node.comment
                ? node
                : typescript.setTextRange(createJSDocAugmentsTag(tagName, className, comment), node);
        },
        updateJSDocAuthorTag(node, tagName, comment) {
            return tagName === node.tagName && comment === node.comment ? node : typescript.setTextRange(createJSDocAuthorTag(tagName, comment), node);
        },
        updateJSDocCallbackTag(node, tagName, typeExpression, fullName, comment) {
            return tagName === node.tagName && typeExpression === node.typeExpression && fullName === node.fullName && comment === node.comment
                ? node
                : typescript.setTextRange(createJSDocCallbackTag(tagName, typeExpression, fullName, comment), node);
        },
        updateJSDocClassTag(node, tagName, comment) {
            return tagName === node.tagName && comment === node.comment ? node : typescript.setTextRange(createJSDocClassTag(tagName, comment), node);
        },
        updateJSDocComment(node, comment, tags) {
            return comment === node.comment && tags === node.tags ? node : typescript.setTextRange(createJSDocComment(comment, tags), node);
        },
        updateJSDocDeprecatedTag(node, tagName, comment) {
            return tagName === node.tagName && comment === node.comment ? node : typescript.setTextRange(createJSDocDeprecatedTag(tagName, comment), node);
        },
        updateJSDocEnumTag(node, tagName, typeExpression, comment) {
            return tagName === node.tagName && typeExpression === node.typeExpression ? node : typescript.setTextRange(createJSDocEnumTag(tagName, typeExpression, comment), node);
        },
        updateJSDocFunctionType(node, parameters, type) {
            return parameters === node.parameters && type === node.type ? node : typescript.setTextRange(createJSDocFunctionType(parameters, type), node);
        },
        updateJSDocImplementsTag(node, tagName, className, comment) {
            return tagName === node.tagName && className === node.class && comment === node.comment
                ? node
                : typescript.setTextRange(createJSDocImplementsTag(tagName, className, comment), node);
        },
        updateJSDocLink(node, name, text) {
            return name === node.name && text === node.text ? node : typescript.setTextRange(createJSDocLink(name, text), node);
        },
        updateJSDocNameReference(node, name) {
            return name === node.name ? node : typescript.setTextRange(createJSDocNameReference(name), node);
        },
        updateJSDocNamepathType(node, type) {
            return type === node.type ? node : typescript.setTextRange(createJSDocNamepathType(type), node);
        },
        updateJSDocNonNullableType(node, type) {
            return type === node.type ? node : typescript.setTextRange(createJSDocNonNullableType(type), node);
        },
        updateJSDocNullableType(node, type) {
            return type === node.type ? node : typescript.setTextRange(createJSDocNullableType(type), node);
        },
        updateJSDocOptionalType(node, type) {
            return type === node.type ? node : typescript.setTextRange(createJSDocOptionalType(type), node);
        },
        updateJSDocOverrideTag(node, tagName, comment) {
            return tagName === node.tagName && comment === node.comment ? node : typescript.setTextRange(createJSDocOverrideTag(tagName, comment), node);
        },
        updateJSDocParameterTag(node, tagName, name, isBracketed, typeExpression, isNameFirst, comment) {
            return tagName === node.tagName &&
                name === node.name &&
                isBracketed === node.isBracketed &&
                typeExpression === node.typeExpression &&
                isNameFirst === node.isNameFirst &&
                comment === node.comment
                ? node
                : typescript.setTextRange(createJSDocParameterTag(tagName, name, isBracketed, typeExpression, isNameFirst, comment), node);
        },
        updateJSDocPrivateTag(node, tagName, comment) {
            return tagName === node.tagName && comment === node.comment ? node : typescript.setTextRange(createJSDocPrivateTag(tagName, comment), node);
        },
        updateJSDocPropertyTag(node, tagName, name, isBracketed, typeExpression, isNameFirst, comment) {
            return tagName === node.tagName &&
                name === node.name &&
                isBracketed === node.isBracketed &&
                typeExpression === node.typeExpression &&
                isNameFirst === node.isNameFirst &&
                comment === node.comment
                ? node
                : typescript.setTextRange(createJSDocPropertyTag(tagName, name, isBracketed, typeExpression, isNameFirst, comment), node);
        },
        updateJSDocProtectedTag(node, tagName, comment) {
            return tagName === node.tagName && comment === node.comment ? node : typescript.setTextRange(createJSDocProtectedTag(tagName, comment), node);
        },
        updateJSDocPublicTag(node, tagName, comment) {
            return tagName === node.tagName && comment === node.comment ? node : typescript.setTextRange(createJSDocPublicTag(tagName, comment), node);
        },
        updateJSDocReadonlyTag(node, tagName, comment) {
            return tagName === node.tagName && comment === node.comment ? node : typescript.setTextRange(createJSDocReadonlyTag(tagName, comment), node);
        },
        updateJSDocReturnTag(node, tagName, typeExpression, comment) {
            return tagName === node.tagName && comment === node.comment && typeExpression === node.typeExpression
                ? node
                : typescript.setTextRange(createJSDocReturnTag(tagName, typeExpression, comment), node);
        },
        updateJSDocSeeTag(node, tagName, nameExpression, comment) {
            return tagName === node.tagName && nameExpression === node.name && comment === node.comment
                ? node
                : typescript.setTextRange(createJSDocSeeTag(tagName, nameExpression, comment), node);
        },
        updateJSDocSignature(node, typeParameters, parameters, type) {
            return typeParameters === node.typeParameters && parameters === node.parameters && type === node.type
                ? node
                : typescript.setTextRange(createJSDocSignature(typeParameters, parameters, type), node);
        },
        updateJSDocTemplateTag(node, tagName, constraint, typeParameters, comment) {
            return tagName === node.tagName && constraint === node.constraint && typeParameters === node.typeParameters && comment === node.comment
                ? node
                : typescript.setTextRange(createJSDocTemplateTag(tagName, constraint, typeParameters, comment), node);
        },
        updateJSDocText(node, text) {
            return text === node.text ? node : typescript.setTextRange(createJSDocText(text), node);
        },
        updateJSDocThisTag(node, tagName, typeExpression, comment) {
            return tagName === node.tagName && typeExpression === node.typeExpression && comment === node.comment
                ? node
                : typescript.setTextRange(createJSDocThisTag(tagName, typeExpression, comment), node);
        },
        updateJSDocTypeExpression(node, type) {
            return type === node.type ? node : typescript.setTextRange(createJSDocTypeExpression(type), node);
        },
        updateJSDocTypeLiteral(node, jsDocPropertyTags, isArrayType) {
            return jsDocPropertyTags === node.jsDocPropertyTags && isArrayType === node.isArrayType
                ? node
                : typescript.setTextRange(createJSDocTypeLiteral(jsDocPropertyTags, isArrayType), node);
        },
        updateJSDocTypeTag(node, tagName, typeExpression, comment) {
            return tagName === node.tagName && typeExpression === node.typeExpression && comment === node.comment
                ? node
                : typescript.setTextRange(createJSDocTypeTag(tagName, typeExpression, comment), node);
        },
        updateJSDocTypedefTag(node, tagName, typeExpression, fullName, comment) {
            return tagName === node.tagName && typeExpression === node.typeExpression && fullName === node.fullName && comment === node.comment
                ? node
                : typescript.setTextRange(createJSDocTypedefTag(tagName, typeExpression, fullName, comment), node);
        },
        updateJSDocUnknownTag(node, tagName, comment) {
            return tagName === node.tagName && comment === node.comment ? node : typescript.setTextRange(createJSDocUnknownTag(tagName, comment), node);
        },
        updateJSDocVariadicType(node, type) {
            return type === node.type ? node : typescript.setTextRange(createJSDocVariadicType(type), node);
        },
        updateJSDocMemberName(node, left, right) {
            return left === node.left && right === node.right ? node : typescript.setTextRange(createJSDocMemberName(left, right), node);
        },
        updateJSDocLinkCode(node, name, text) {
            return name === node.name && text === node.text ? node : typescript.setTextRange(createJSDocLinkCode(name, text), node);
        },
        updateJSDocLinkPlain(node, name, text) {
            return name === node.name && text === node.text ? node : typescript.setTextRange(createJSDocLinkPlain(name, text), node);
        },
        updateLabeledStatement(node, label, statement) {
            return typescript.updateLabel(node, label, statement);
        },
        updateMethodDeclaration(node, decorators, modifiers, asteriskToken, name, questionToken, typeParameters, parameters, type, body) {
            return typescript.updateMethod(node, decorators, modifiers, asteriskToken, name, questionToken, typeParameters, parameters, type, body);
        },
        updateNamedTupleMember(node, dotDotDotToken, name, questionToken, type) {
            return dotDotDotToken === node.dotDotDotToken && name === node.name && questionToken === node.questionToken && type === node.type
                ? node
                : typescript.setTextRange(createNamedTupleMember(dotDotDotToken, name, questionToken, type), node);
        },
        updateNewExpression(node, expression, typeArguments, argumentsArray) {
            return typescript.updateNew(node, expression, typeArguments, argumentsArray);
        },
        updateObjectLiteralExpression(node, properties) {
            return typescript.updateObjectLiteral(node, properties);
        },
        updateParameterDeclaration(node, decorators, modifiers, dotDotDotToken, name, questionToken, type, initializer) {
            return typescript.updateParameter(node, decorators, modifiers, dotDotDotToken, name, questionToken, type, initializer);
        },
        updateParenthesizedExpression(node, expression) {
            return typescript.updateParen(node, expression);
        },
        updatePostfixUnaryExpression(node, operand) {
            return typescript.updatePostfix(node, operand);
        },
        updatePrefixUnaryExpression(node, operand) {
            return typescript.updatePrefix(node, operand);
        },
        updatePropertyAccessExpression(node, expression, name) {
            return typescript.updatePropertyAccess(node, expression, name);
        },
        updatePropertyDeclaration(node, decorators, modifiers, name, questionOrExclamationToken, type, initializer) {
            return typescript.updateProperty(node, decorators, modifiers, name, questionOrExclamationToken, type, initializer);
        },
        updateReturnStatement(node, expression) {
            return typescript.updateReturn(node, expression);
        },
        updateSetAccessorDeclaration(node, decorators, modifiers, name, parameters, body) {
            return typescript.updateSetAccessor(node, decorators, modifiers, name, parameters, body);
        },
        updateSpreadElement(node, expression) {
            return typescript.updateSpread(node, expression);
        },
        updateSwitchStatement(node, expression, caseBlock) {
            return typescript.updateSwitch(node, expression, caseBlock);
        },
        updateTaggedTemplateExpression(node, tag, typeArguments, template) {
            return typescript.updateTaggedTemplate(node, tag, typeArguments, template);
        },
        updateTemplateLiteralType(node, head, templateSpans) {
            return head === node.head && templateSpans === node.templateSpans ? node : typescript.setTextRange(createTemplateLiteralType(head, templateSpans), node);
        },
        updateTemplateLiteralTypeSpan(node, type, literal) {
            return type === node.type && literal === node.literal ? node : typescript.setTextRange(createTemplateLiteralTypeSpan(type, literal), node);
        },
        updateClassStaticBlockDeclaration(node, decorators, modifiers, body) {
            return decorators === node.decorators && modifiers === node.modifiers && body === node.body
                ? node
                : typescript.setTextRange(createClassStaticBlockDeclaration(decorators, modifiers, body), node);
        },
        updateAssertClause(node, elements, multiLine) {
            return node.elements !== elements || node.multiLine !== multiLine ? typescript.setTextRange(createAssertClause(elements, multiLine), node) : node;
        },
        updateAssertEntry(node, name, value) {
            return node.name !== name || node.value !== value ? typescript.setTextRange(createAssertEntry(name, value), node) : node;
        },
        updateThrowStatement(node, expression) {
            return typescript.updateThrow(node, expression);
        },
        updateTryStatement(node, tryBlock, catchClause, finallyBlock) {
            return typescript.updateTry(node, tryBlock, catchClause, finallyBlock);
        },
        updateTypeOfExpression(node, expression) {
            return typescript.updateTypeOf(node, expression);
        },
        updateVoidExpression(node, expression) {
            return typescript.updateVoid(node, expression);
        },
        updateWhileStatement(node, expression, statement) {
            return typescript.updateWhile(node, expression, statement);
        },
        updateWithStatement(node, expression, statement) {
            return typescript.updateWith(node, expression, statement);
        },
        updateYieldExpression(node, asteriskToken, expression) {
            return typescript.updateYield(node, asteriskToken, expression);
        },
        createImportClause(isTypeOnly, name, namedBindings) {
            return typescript.createImportClause(name, namedBindings, isTypeOnly);
        },
        createCallExpression(expression, typeArguments, argumentsArray) {
            return typescript.createCall(expression, typeArguments, argumentsArray);
        },
        updateCallExpression(node, expression, typeArguments, argumentsArray) {
            return typescript.updateCall(node, expression, typeArguments, argumentsArray);
        },
        createArrayLiteralExpression(elements, multiLine) {
            return typescript.createArrayLiteral(elements, multiLine);
        },
        updateArrayLiteralExpression(node, elements) {
            return typescript.updateArrayLiteral(node, elements);
        },
        updateSourceFile(node, statements, isDeclarationFile, referencedFiles, typeReferences, hasNoDefaultLib, libReferences) {
            return typescript.updateSourceFileNode(node, statements, isDeclarationFile, referencedFiles, typeReferences, hasNoDefaultLib, libReferences);
        },
        updateClassExpression(node, decorators, modifiers, name, typeParameters, heritageClauses, members) {
            return typescript.updateClassExpression(node, modifiers, name, typeParameters, heritageClauses, members);
        },
        createPropertyAccessExpression(expression, name) {
            return typescript.createPropertyAccess(expression, name);
        },
        createGetAccessorDeclaration(decorators, modifiers, name, parameters, type, body) {
            return typescript.createGetAccessor(decorators, modifiers, name, parameters, type, body);
        },
        updateGetAccessorDeclaration(node, decorators, modifiers, name, parameters, type, body) {
            return typescript.updateGetAccessor(node, decorators, modifiers, name, parameters, type, body);
        },
        createReturnStatement(expression) {
            return typescript.createReturn(expression);
        },
        createObjectLiteralExpression(properties, multiLine) {
            return typescript.createObjectLiteral(properties, multiLine);
        },
        createVariableDeclaration(name, exclamationToken, type, initializer) {
            if (typescript.createVariableDeclaration.length === 4) {
                return typescript.createVariableDeclaration(name, exclamationToken, type, initializer);
            }
            return typescript.createVariableDeclaration(name, type, initializer);
        },
        updateVariableDeclaration(node, name, exclamationToken, type, initializer) {
            if (typescript.updateVariableDeclaration.length === 4) {
                return typescript.updateVariableDeclaration(node, name, type, initializer);
            }
            return typescript.updateVariableDeclaration(node, name, exclamationToken, type, initializer);
        },
        createPropertyAccessChain(expression, questionDotToken, name) {
            if ("createPropertyAccessChain" in typescript) {
                return typescript.createPropertyAccessChain(expression, questionDotToken, name);
            }
            const node = typescript.createPropertyAccess(expression, name);
            node.questionDotToken = questionDotToken;
            return node;
        },
        updatePropertyAccessChain(node, expression, questionDotToken, name) {
            if ("updatePropertyAccessChain" in typescript) {
                return typescript.updatePropertyAccessChain(node, expression, questionDotToken, name);
            }
            const newNode = typescript.updatePropertyAccess(node, expression, name);
            newNode.questionDotToken = questionDotToken;
            return newNode;
        },
        createImportEqualsDeclaration(decorators, modifiers, isTypeOnly, name, moduleReference) {
            return typescript.createImportEqualsDeclaration(decorators, modifiers, name, moduleReference);
        },
        updateImportEqualsDeclaration(node, decorators, modifiers, isTypeOnly, name, moduleReference) {
            const normalizedName = typeof name === "string" ? typescript.createIdentifier(name) : name;
            return typescript.updateImportEqualsDeclaration(node, decorators, modifiers, normalizedName, moduleReference);
        },
        createMappedTypeNode(readonlyToken, typeParameter, nameType, questionToken, type, members) {
            return typescript.createMappedTypeNode(readonlyToken, typeParameter, questionToken, type);
        },
        updateMappedTypeNode(node, readonlyToken, typeParameter, nameType, questionToken, type, members) {
            return typescript.updateMappedTypeNode(node, readonlyToken, typeParameter, questionToken, type);
        },
        createImportSpecifier(isTypeOnly, propertyName, name) {
            return typescript.createImportSpecifier(propertyName, name);
        },
        updateImportSpecifier(node, isTypeOnly, propertyName, name) {
            return typescript.updateImportSpecifier(node, propertyName, name);
        },
        createExportSpecifier(isTypeOnly, propertyName, name) {
            return typescript.createExportSpecifier(propertyName, name);
        },
        updateExportSpecifier(node, isTypeOnly, propertyName, name) {
            return typescript.updateExportSpecifier(node, propertyName, name);
        },
        createExportDeclaration(decorators, modifiers, isTypeOnly, exportClause, moduleSpecifier) {
            return typescript.createExportDeclaration(decorators, modifiers, exportClause, moduleSpecifier, isTypeOnly);
        },
        updateConstructorDeclaration(node, decorators, modifiers, parameters, body) {
            return typescript.updateConstructor(node, decorators, modifiers, parameters, body);
        },
        /**
         * Some TypeScript versions require that the value is a string argument
         */
        createNumericLiteral(value, numericLiteralFlags) {
            return typescript.createNumericLiteral(String(value), numericLiteralFlags);
        }
    };
}
//# sourceMappingURL=index.js.map