###############################################################################
# 
#  Copyright (2008) Alexander Stukowski
#
#  This file is part of OVITO (Open Visualization Tool).
#
#  OVITO is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  OVITO is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
###############################################################################

#
# This Python package is part of the highlevel scripting interface
# for the AtomViz plugin.
#

import math
import sys

from Scripting import *
from AtomViz import *

# The default mapping from columns to data channels that
# is used when importing LAMMP dump files.
lammpsDefaultColumnMapping = ColumnChannelMapping()
lammpsDefaultColumnMapping.DefineStandardColumn(0, DataChannelIdentifier.AtomIndexChannel)
lammpsDefaultColumnMapping.DefineStandardColumn(1, DataChannelIdentifier.AtomTypeChannel)
lammpsDefaultColumnMapping.DefineStandardColumn(2, DataChannelIdentifier.PositionChannel, 0)
lammpsDefaultColumnMapping.DefineStandardColumn(3, DataChannelIdentifier.PositionChannel, 1)
lammpsDefaultColumnMapping.DefineStandardColumn(4, DataChannelIdentifier.PositionChannel, 2)

# The default mapping from columns to data channels that
# is used when importing XYZ files.
xyzDefaultColumnMapping = ColumnChannelMapping()
xyzDefaultColumnMapping.DefineStandardColumn(0, DataChannelIdentifier.AtomTypeChannel)
xyzDefaultColumnMapping.DefineStandardColumn(1, DataChannelIdentifier.PositionChannel)
xyzDefaultColumnMapping.DefineStandardColumn(2, DataChannelIdentifier.PositionChannel, 1)
xyzDefaultColumnMapping.DefineStandardColumn(3, DataChannelIdentifier.PositionChannel, 2)
xyzDefaultColumnMapping.DefineStandardColumn(4, DataChannelIdentifier.AtomIndexChannel, 0)

def ImportLAMMPSDumpFile(filepath, binary = False, columns = lammpsDefaultColumnMapping, pbc = None, multiTimesteps = False, dataset = None):
	""" Imports a LAMMPS dump file into the current scene. 
		
		Parameters:
		
		filepath  -  The filename of the LAMMPS dump file to be imported.
					 If this filename contains at least one * or % character
					 then it will be treated as a wildcard string and a sequence of 
					 dump files be read that match the wildcard pattern string. 
					 
		binary	-  A boolean flag that indicates whether the dump file is in binary
					 or text format.
					 
		columns   -  A ColumnChannelMapping object that maps data column in the input dump file
					 to data channels in the destination AtomsObject. 
					 
		pbc	   -  An array of three boolean values that control whether periodic boundary
					 conditions should be enabled for the imported simulation box. This information
					 has to be specified here because it is not given in the LAMMPS dump file.
					 
		multiTimesteps - A boolean flag that indicates whether the input file contains multiple 
						 timesteps. If set to True the input file is first scanned to determine
						 the number of timesteps contained in the dump file.
	
		dataset   -  The scene data set where the atoms should be stored. If not specified
					 then the current data set is used.
					 
		Return value: The object node in the scene that contains imported atoms. It
						 can be used to apply modifiers to it.
	"""
	
	if binary == True:
		parser = LAMMPSBinaryDumpParser()
	else:
		parser = LAMMPSTextDumpParser()
	
	# Setup parser
	if "*" in filepath or "%" in filepath:
		sys.stderr.write("Using wildcard pattern to load multiple atoms files.\n")
		parser.UseWildcardFilename = True
	else:
		parser.UseWildcardFilename = False
		parser.MovieFileEnabled = multiTimesteps
	# Define the column to data channel mapping
	parser.ColumnMapping = columns
		
	# Get the current data set
	if dataset == None: dataset = DataSetManager.Instance.CurrentSet
	  
	# Let the parser import the atoms file.
	parser.ImportFile(filepath, dataset, True)

	# Find the scene node that contains the imported atoms.
	for node in dataset.SceneRoot.Children:
		if node.IsObjectNode:
			if isinstance(node.SceneObject, AtomsImportObject):
				
				# Set the periodic boundary conditions.
				if pbc != None:
					simCell = node.SceneObject.Atoms.SimulationCell
					simCell.SetPeriodicity(pbc[0], pbc[1], pbc[2])
				
				return node
	
	# No scene found that evaluates to a AtomsObject
	raise RuntimeError, "Something went wrong. Scene does not contains an AtomsObject after LAMMPS dump file import."

def ImportLAMMPSDataFile(filepath, pbc = None, dataset = None):
	""" Imports a LAMMPS data file into the current scene. 
		
		Parameters:
		
		filepath  -  The filename of the LAMMPS data file to be imported.
					 
		pbc	   -  An array of three boolean values that control whether periodic boundary
					 conditions should be enabled for the imported simulation box. This information
					 has to be specified here because it is not given in the LAMMPS data file.
					 
		dataset   -  The scene data set where the atoms should be stored. If not specified
					 then the current data set is used.
					 
		Return value: The object node in the scene that contains imported atoms. It
						 can be used to apply modifiers to it.
	"""
	
	parser = LAMMPSDataParser()
	
	# Get the current data set
	if dataset == None: dataset = DataSetManager.Instance.CurrentSet
	  
	# Let the parser import the atoms file.
	parser.ImportFile(filepath, dataset, True)

	# Find the scene node that contains the imported atoms.
	for node in DataSetManager.Instance.CurrentSet.SceneRoot.Children:
		if node.IsObjectNode:
			if isinstance(node.SceneObject, AtomsImportObject):
				
				# Set the periodic boundary conditions.
				if pbc != None:
					simCell = node.SceneObject.Atoms.SimulationCell
					simCell.SetPeriodicity(pbc[0], pbc[1], pbc[2])
				
				return node
	
	# No scene found that evaluates to a AtomsObject
	raise RuntimeError, "Something went wrong. Scene does not contains an AtomsObject after LAMMPS data file import."

def ImportXYZFile(filepath, columns = xyzDefaultColumnMapping, pbc = None, multiTimesteps = False, dataset = None):
	""" Imports an XYZ file into the current scene. 
		
		Parameters:
		
		filepath  -  The filename of the XYZ file to be imported.
					 If this filename contains at least one * or % character
					 then it will be treated as a wildcard string and a sequence of 
					 files be read that match the wildcard pattern string. 
					 
		columns   -  A ColumnChannelMapping object that maps data column in the input file
					 to data channels in the destination AtomsObject. 
					 
		pbc	   -  An array of three boolean values that control whether periodic boundary
					 conditions should be enabled for the imported simulation box.
					 
		multiTimesteps - A boolean flag that indicates whether the input file contains multiple 
						 timesteps. If set to True the input file is first scanned to determine
						 the number of timesteps contained in the file.
	
		dataset   -  The scene data set where the atoms should be stored. If not specified
					 then the current data set is used.
					 
		Return value: The object node in the scene that contains imported atoms. It
						 can be used to apply modifiers to it.
	"""

	parser = XYZParser()
	
	# Setup parser
	if "*" in filepath or "%" in filepath:
		sys.stderr.write("Using wildcard pattern to load multiple atoms files.\n")
		parser.UseWildcardFilename = True
	else:
		parser.UseWildcardFilename = False
		parser.MovieFileEnabled = multiTimesteps
	# Define the column to data channel mapping
	parser.ColumnMapping = columns
		
	# Get the current data set
	if dataset == None: dataset = DataSetManager.Instance.CurrentSet
	  
	# Let the parser import the atoms file.
	parser.ImportFile(filepath, dataset, True)

	# Find the scene node that contains the imported atoms.
	for node in DataSetManager.Instance.CurrentSet.SceneRoot.Children:
		if node.IsObjectNode:
			if isinstance(node.SceneObject, AtomsImportObject):
				
				# Set the periodic boundary conditions.
				if pbc != None:
					simCell = node.SceneObject.Atoms.SimulationCell
					simCell.SetPeriodicity(pbc[0], pbc[1], pbc[2])
				
				return node
	
	# No scene found that evaluates to a AtomsObject
	raise RuntimeError, "Something went wrong. Scene does not contains an AtomsObject after XYZ file import."

def ImportIMDAtomFile(filepath, pbc = None, dataset = None):
	""" Imports an IMD atoms file into the current scene. 
		
		Parameters:
		
		filepath  -  The filename of theIMD file to be imported.
					 
		pbc	   -  An array of three boolean values that control whether periodic boundary
					 conditions should be enabled for the imported simulation box. This information
					 has to be specified here because it is not given in the IMD file.
					 
		dataset   -  The scene data set where the atoms should be stored. If not specified
					 then the current data set is used.
					 
		Return value: The object node in the scene that contains imported atoms. It
						 can be used to apply modifiers to it.
	"""
	
	parser = IMDAtomFileParser()
	
	# Get the current data set
	if dataset == None: dataset = DataSetManager.Instance.CurrentSet
	  
	# Let the parser import the atoms file.
	parser.ImportFile(filepath, dataset, True)

	# Find the scene node that contains the imported atoms.
	for node in DataSetManager.Instance.CurrentSet.SceneRoot.Children:
		if node.IsObjectNode:
			if isinstance(node.SceneObject, AtomsImportObject):
				
				# Set the periodic boundary conditions.
				if pbc != None:
					simCell = node.SceneObject.Atoms.SimulationCell
					simCell.SetPeriodicity(pbc[0], pbc[1], pbc[2])
				
				return node
	
	# No scene found that evaluates to a AtomsObject
	raise RuntimeError, "Something went wrong. Scene does not contains an AtomsObject after IMD file import."

def GetAtomsObjectNode():
	""" 
		Looks up the scene node in the current scene that
		contains an AtomsObject.
		
		If there is no scene object with atoms in the scene or
		if there is more than one AtomsObject in the scene than
		this function throws an exception.
		
		This function returns the ObjectNode that contains the atoms.
	"""
	if DataSetManager.Instance.CurrentSet == None:
		raise RuntimeError, "There is no current scene."
	
	atomsNode = None
	
	for node in DataSetManager.Instance.CurrentSet.SceneRoot.Children:
		if not node.IsObjectNode: continue
		
		result = node.EvalPipeline(AnimManager.Instance.Time)
		if not isinstance(result.Result, AtomsObject): continue
		
		if atomsNode != None:
			raise RuntimeError, "There are more object nodes in the scene that contain an AtomsObject."
		atomsNode = node

	if atomsNode == None:
		raise RuntimeError, "There is no object node in the scene that contain an AtomsObject."
		
	return atomsNode

def GetNumberOfMovieFrames(objNode):
	""" Returns the number of movie frames that have been imported. 
	
		Parameter:
		
		objNode	   - An object node whos modification stack contains an atoms import object.
	"""
	obj = objNode.SceneObject

	while obj != None and not isinstance(obj, AtomsImportObject) and isinstance(obj, ModifiedObject):
		obj = obj.InputObject
		
	if not isinstance(obj, AtomsImportObject):
		raise RuntimeError, "The object node does not contain an AtomsImportObject."
	
	return obj.Parser.NumberOfMovieFrames

	