///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file StandardKeyedControllers.h
 * \brief Contains the definition of the Core::StandardKeyedController class and
 *        its derived classes.
 */

#ifndef __OVITO_STD_KEYED_CONTROLLERS_H
#define __OVITO_STD_KEYED_CONTROLLERS_H

#include <core/Core.h>
#include "Controller.h"
#include <core/undo/UndoManager.h>
#include <core/scene/animation/AnimManager.h>
#include <core/data/ObjectLoadStream.h>
#include <core/data/ObjectSaveStream.h>

namespace Core {

template<class BaseControllerClass,
		 typename _ValueType,
		 typename _KeyType,
		 typename _NullValue,
		 class KeyInterpolator>
class StandardKeyedController : public BaseControllerClass
{
protected:
	typedef _NullValue NullValue;
	typedef _ValueType ValueType;
	typedef _KeyType KeyType;
	typedef map<TimeTicks, KeyType> KeyArray;
	typedef typename KeyArray::iterator Key;
	typedef typename KeyArray::const_iterator ConstKey;
	typedef StandardKeyedController<BaseControllerClass, ValueType, KeyType, NullValue, KeyInterpolator> ThisClassType;

	// This class stores the previous key values before the have been changed.
	class KeyChangeOperation : public UndoableOperation {
	public:
		KeyChangeOperation(StandardKeyedController* ctrl) : controller(ctrl), storedKeys(ctrl->keys) {}
		virtual void undo() {
			storedKeys.swap(controller->keys);
			controller->notifyDependents(REFTARGET_CHANGED);
		}
		virtual void redo() { undo(); }
		/// Provides a localized, human readable description of this operation.
		virtual QString displayName() const { return "Keyed controller change"; }
	private:
		intrusive_ptr<StandardKeyedController> controller;
		KeyArray storedKeys;
	};

public:
    // Default constructor.
	StandardKeyedController(bool isLoading) : BaseControllerClass(isLoading) {}

	/// Queries the controller for its absolute value at a certain time.
	virtual void getValue(TimeTicks time, ValueType& result, TimeInterval& validityInterval) {
		if(keys.empty()) {
			result = NullValue();
			return;
		}

		// Handle out of range cases.
		if(time <= keys.begin()->first) {
			result = keys.begin()->second;
			if(keys.size() != 1)
				validityInterval.intersect(TimeInterval(TimeNegativeInfinity, keys.begin()->first));
		}
		else if(time >= keys.rbegin()->first) {
			result = keys.rbegin()->second;
			if(keys.size() != 1)
				validityInterval.intersect(TimeInterval(keys.rbegin()->first, TimePositiveInfinity));
		}
		else {
			// Intersect validity interval.
			validityInterval.intersect(TimeInterval(time));

			ConstKey lastKey = keys.begin();
			ConstKey key = lastKey;
			while((++key) != keys.end()) {
				if(key->first == time) {
					// No interpolation neccessary.
					result = key->second;
					return;
				}
				else if(key->first > time) {
					// Interpolate between two keys.
					KeyInterpolator keyInterpolator;
					result = keyInterpolator(time, *lastKey, *key);
					return;
				}
				lastKey = key;
			}

			// Should not happen.
			OVITO_ASSERT_MSG(false, "StandardKeyedController::getValue", "Invalid controller keys.");
			result = NullValue();
		}
	}

	/// Sets the controller's value at the specified time.
	virtual void setValue(TimeTicks time, const ValueType& newValue, bool isAbsoluteValue) {
		if(keys.empty()) {
			// Handle undo.
			if(UNDO_MANAGER.isRecording())
				UNDO_MANAGER.addOperation(new KeyChangeOperation(this));
			// Automatically create a second key at time 0 if the controller still has its default value (i.e. no keys).
			if(time != 0 && ANIM_MANAGER.isAnimating() && newValue != NullValue())
				keys[0] = NullValue();
			// Set initial value.
			keys[time] = newValue;
			this->updateKeys();
			this->notifyDependents(REFTARGET_CHANGED);
			return;
		}

		ValueType deltaValue(newValue);
		ValueType oldValue;
		// Get delta from new absolute value.
		if(isAbsoluteValue) {
			TimeInterval iv;
			getValue(time, oldValue, iv);
			if(newValue == oldValue) return;
		}
		else if(deltaValue == NullValue()) return;	// Nothing to do.

		// Handle undo.
		if(UNDO_MANAGER.isRecording())
			UNDO_MANAGER.addOperation(new KeyChangeOperation(this));

		if(!ANIM_MANAGER.isAnimating()) {
			if(keys.size() == 1 && isAbsoluteValue) {
				keys.begin()->second = newValue;
			}
			else {

				if(isAbsoluteValue) deltaValue -= oldValue;
				// Apply delta value to all keys.
				for(Key key = keys.begin(); key != keys.end(); ++key)
					key->second += deltaValue;
			}
		}
		else {
			if(isAbsoluteValue) deltaValue -= oldValue;
			Key key = insertKey(time);
			key->second += deltaValue;
		}
		this->updateKeys();

		// Send change message.
		this->notifyDependents(REFTARGET_CHANGED);
	}

	/// This will rescale the time values of all keys from the old animation interval to the new interval.
	virtual void rescaleTime(const TimeInterval& oldAnimationInterval, const TimeInterval& newAnimationInterval) {
		OVITO_ASSERT(!oldAnimationInterval.isInfinite());
		OVITO_ASSERT(!newAnimationInterval.isInfinite());
		if(oldAnimationInterval.duration() == 0 && oldAnimationInterval.start() == newAnimationInterval.start()) return;
		// Handle undo.
		if(UNDO_MANAGER.isRecording())
			UNDO_MANAGER.addOperation(new KeyChangeOperation(this));
		KeyArray newKeys;
		TimeTicks newTime;
		for(Key key = keys.begin(); key != keys.end(); ++key) {
			if(oldAnimationInterval.duration() == 0)
				newTime = key->first - oldAnimationInterval.start() + newAnimationInterval.start();
			else
				newTime = (key->first - oldAnimationInterval.start()) * newAnimationInterval.duration()
							/ oldAnimationInterval.duration() + newAnimationInterval.start();
			newKeys.insert(make_pair(newTime, key->second));
		}
		keys = newKeys;
		updateKeys();
		// Send change message.
		this->notifyDependents(REFTARGET_CHANGED);
	}

	/// \brief Creates a new key at the given time with the specified value.
	/// \param time The animation where a key should be created.
	/// \param value The absolute value of the new animation key.
	/// Any existing key at that time is replaced with the new key.
	virtual void createKey(TimeTicks time, const ValueType& value) {
		// Get existing key if there is one.
		Key key = keys.find(time);
		if(key != keys.end()) {
			// Check if existing key already has the same value.
			if(value == (ValueType)key->second) return;	// nothing to do.
		}

		// Handle undo.
		if(UNDO_MANAGER.isRecording())
			UNDO_MANAGER.addOperation(new KeyChangeOperation(this));

		if(key == keys.end())	// Create a new key.
			key = keys.insert(make_pair(time, value)).first;
		else
			key->second = value;
		updateKeys();

		// Send change message.
		this->notifyDependents(REFTARGET_CHANGED);
	}

protected:

	/// Saves the class' contents to the given stream.
	virtual void saveToStream(ObjectSaveStream& stream) {
		BaseControllerClass::saveToStream(stream);
		stream.beginChunk(0x001);
		stream << (quint32)keys.size();
		for(Key key = keys.begin(); key != keys.end(); ++key) {
			stream << key->first;
			stream << key->second;
		}
		stream.endChunk();
	}

	/// Loads the class' contents from the given stream.
	virtual void loadFromStream(ObjectLoadStream& stream) {
		BaseControllerClass::loadFromStream(stream);
		stream.expectChunk(0x001);
		quint32 nkeys;
		stream >> nkeys;
		TimeTicks time;
		for(; nkeys != 0; nkeys--) {
			stream >> time;
			stream >> keys[time];
		}
		stream.closeChunk();
	}

	/// Creates a copy of this object.
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper) {
		// Let the base class create an instance of this class.
		intrusive_ptr<ThisClassType> clone = static_object_cast<ThisClassType>(BaseControllerClass::clone(deepCopy, cloneHelper));
		clone->keys = this->keys;
		return clone;
	}

	/// The list of controller keys.
    KeyArray keys;

	/// Inserts a new key at the specified time.
	virtual Key insertKey(const TimeTicks& time) {
		// Get existing key if there is one.
		Key key = keys.find(time);
		if(key != keys.end()) return key;

		// Get initial value of new key.
		TimeInterval iv;
		ValueType v;
		getValue(time, v, iv);

		// Create new key.
		return keys.insert(make_pair(time, v)).first;
	}

	/// This is called each time the value or time positions of one or more keys has been changed.
	virtual void updateKeys() {}
};

/**
 * \brief Default implementation of the value interpolator concept that does linear interpolation.
 *
 * This template class interpolates linearly between two values of abitrary types.
 * The value 0.0 of the interpolation parameter \c t is mapped to the first value.
 * The value 1.0 of the interpolation parameter \c t is mapped to the second value.
 *
 * \author Alexander Stukowski
 */
template<typename ValueType>
struct LinearValueInterpolator {
	ValueType operator()(FloatType t, const ValueType& value1, const ValueType& value2) const {
		return (ValueType)(value1 + (t * (value2 - value1)));
	}
};

/**
 * \brief Implementation of the value interpolator concept for rotations.
 *
 * This class is required because the Base::Rotation class does not support the standard
 * addition, scalar multiplication and subtraction operators.
 */
template<>
struct LinearValueInterpolator<Rotation> {
	Rotation operator()(FloatType t, const Rotation& value1, const Rotation& value2) const {
		return Rotation::interpolate(value1, value2, t);
	}
};

/**
 * \brief Implementation of the value interpolator concept for scaling values.
 *
 * This class is required because the Base::Scaling class does not support the standard
 * addition, scalar multiplication and subtraction operators.
 */
template<>
struct LinearValueInterpolator<Scaling> {
	Scaling operator()(FloatType t, const Scaling& value1, const Scaling& value2) const {
		return Scaling::interpolate(value1, value2, t);
	}
};

/**
 * \brief Default implementation of the value interpolator concept that does smooth interpolation.
 *
 * This template class interpolates using a cubic spline between two values of abitrary types.
 * The value 0.0 of the interpolation parameter \c t is mapped to the first value.
 * The value 1.0 of the interpolation parameter \c t is mapped to the second value.
 *
 * \author Alexander Stukowski
 */
template<typename ValueType>
struct SplineValueInterpolator {
	ValueType operator()(FloatType t, const ValueType& value1, const ValueType& value2, const ValueType& outPoint1, const ValueType& inPoint2) const {
		FloatType Ti = 1.0 - t;
		FloatType U2 = square(t), T2 = square(Ti);
		FloatType U3 = U2 * t, T3 = T2 * Ti;
		return value1 * T3 + outPoint1 * (3.0 * t * T2) + inPoint2 * (3.0 * U2 * Ti) + value2 * U3;
	}
};

/**
 * \brief Implementation of the smooth value interpolator concept for rotations.
 *
 * This class is required because the Base::Rotation class does not support the standard
 * addition, scalar multiplication and subtraction operators.
 */
template<>
struct SplineValueInterpolator<Rotation> {
	Rotation operator()(FloatType t, const Rotation& value1, const Rotation& value2, const Rotation& outPoint1, const Rotation& inPoint2) const {
		return Rotation::interpolateQuad(value1, value2, outPoint1, inPoint2, t);
	}
};

/**
 * \brief Implementation of the smooth value interpolator concept for scaling values.
 *
 * This class is required because the Base::Scaling class does not support the standard
 * addition, scalar multiplication and subtraction operators.
 */
template<>
struct SplineValueInterpolator<Scaling> {
	Scaling operator()(FloatType t, const Scaling& value1, const Scaling& value2, const Scaling& outPoint1, const Scaling& inPoint2) const {
		return Scaling::interpolateQuad(value1, value2, outPoint1, inPoint2, t);
	}
};

/**
 * \brief Base template class for keyed float controllers.
 * \author Alexander Stukowski
 */
template<typename KeyType, class KeyInterpolator>
class KeyedFloatController : public StandardKeyedController<FloatController, FloatType, KeyType, FloatType, KeyInterpolator>
{
public:
	KeyedFloatController(bool isLoading = false) : StandardKeyedController<FloatController, FloatType, KeyType, FloatType, KeyInterpolator>(isLoading) {}
};

/**
 * \brief Base template class for keyed vector controllers.
 * \author Alexander Stukowski
 */
template<typename KeyType, class KeyInterpolator>
class KeyedVectorController : public StandardKeyedController<VectorController, Vector3, KeyType, NullVector, KeyInterpolator>
{
public:
	KeyedVectorController(bool isLoading = false) : StandardKeyedController<VectorController, Vector3, KeyType, NullVector, KeyInterpolator>(isLoading) {}
};

/**
 * \brief Base template class for keyed position controllers.
 * \author Alexander Stukowski
 */
template<typename KeyType, class KeyInterpolator>
class KeyedPositionController : public StandardKeyedController<PositionController, Vector3, KeyType, NullVector, KeyInterpolator>
{
public:
	KeyedPositionController(bool isLoading = false) : StandardKeyedController<PositionController, Vector3, KeyType, NullVector, KeyInterpolator>(isLoading) {}

	virtual void changeParent(TimeTicks time, const AffineTransformation& oldParentTM, const AffineTransformation& newParentTM, SceneNode* contextNode) {
		if(this->keys.empty()) return;
		AffineTransformation rel = oldParentTM * newParentTM.inverse();

		// Handle undo.
		typedef StandardKeyedController<PositionController, Vector3, KeyType, NullVector, KeyInterpolator> BaseClassType;
		if(UNDO_MANAGER.isRecording())
			UNDO_MANAGER.addOperation(new typename BaseClassType::KeyChangeOperation(this));

		if(!ANIM_MANAGER.isAnimating()) {
			// Apply delta value to all keys.
			for(typename BaseClassType::KeyArray::iterator key = this->keys.begin(); key != this->keys.end(); ++key)
				key->second = (rel * (ORIGIN + (Vector3)key->second)) - ORIGIN;
		}
		else {
			typename BaseClassType::KeyArray::iterator key = this->insertKey(time);
			key->second = (rel * (ORIGIN + (Vector3)key->second)) - ORIGIN;
		}

		// Send change message.
		this->notifyDependents(REFTARGET_CHANGED);
	}
};

/**
 * \brief Base template class for keyed rotation controllers.
 * \author Alexander Stukowski
 */
template<typename KeyType, class KeyInterpolator>
class KeyedRotationController : public StandardKeyedController<RotationController, Rotation, KeyType, NullRotation, KeyInterpolator>
{
public:
	KeyedRotationController(bool isLoading = false) : StandardKeyedController<RotationController, Rotation, KeyType, NullRotation, KeyInterpolator>(isLoading) {}

	virtual void changeParent(TimeTicks time, const AffineTransformation& oldParentTM, const AffineTransformation& newParentTM, SceneNode* contextNode) {
		// to be implemented.
	}
};

/**
 * \brief Base template class for keyed scaling controllers.
 * \author Alexander Stukowski
 */
template<typename KeyType, class KeyInterpolator>
class KeyedScalingController : public StandardKeyedController<ScalingController, Scaling, KeyType, IdentityScaling, KeyInterpolator>
{
public:
	KeyedScalingController(bool isLoading = false) : StandardKeyedController<ScalingController, Scaling, KeyType, IdentityScaling, KeyInterpolator>(isLoading) {}

	virtual void changeParent(TimeTicks time, const AffineTransformation& oldParentTM, const AffineTransformation& newParentTM, SceneNode* contextNode) {
		// to be implemented.
	}
};

};	// End of namespace

#endif // __OVITO_STD_KEYED_CONTROLLERS_H
