///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file ViewportGrid.h
 * \brief Contains the definition of the Core::ViewportGrid class.
 */

#ifndef __OVITO_VIEWPORT_GRID_H
#define __OVITO_VIEWPORT_GRID_H

#include <core/Core.h>

namespace Core {

class Viewport;		// defined in Viewport.h

/**
 * \brief The construction grid that is displayed in a viewport.
 *
 * Every Viewport has a current grid that can be accessed through the Viewport::grid() method.
 *
 * \author Alexander Stukowski
 * \sa Viewport
 */
class CORE_DLLEXPORT ViewportGrid : public QObject
{
	Q_OBJECT
public:

	/// \brief Returns whether the grid is currently enabled.
	/// \return \c true if the grid is displayed in the viewport; \c false otherwise.
	/// \sa setVisible()
	bool isVisible() const;

	/// \brief Turns the grid display on or off.
	/// \param visible Controls the display of the construction grid.
	/// \sa isVisible()
	void setVisible(bool visible);

	/// \brief Returns the grid matrix that defines the orientation of the grid.
	/// \return A transformation matrix that defines the coordinate system of the grid.
	///         The matrix transforms from grid coordinates to world coordinates.
	/// \sa setGridMatrix(), inverseGridMatrix()
	const AffineTransformation& gridMatrix() const { return _gridMatrix; }

	/// \brief Returns the inverse of the grid matrix that defines the orientation of the grid.
	/// \return The inverse of the transformation matrix that defines the coordinate system of the grid.
	///         This matrix transforms from world coordinates to grid coordinates.
	/// \sa gridMatrix()
	const AffineTransformation& inverseGridMatrix() const { return _inverseGridMatrix; }

	/// \brief Sets the orientation of the grid plane.
	/// \param gridmat The transformation matrix that defines the grid orientation.
	///                It transforms from grid coordinates to world coordinates.
	/// \sa gridMatrix()
	void setGridMatrix(const AffineTransformation& gridmat);

	/// \brief Returns the current spacing of the grid.
	/// \return The current spacing between the minor grid lines in world units.
	FloatType gridSpacing() const { return _gridSpacing; }

	/// \brief Returns the visible part of the grid.
	/// \return The bounds in the grid plane of the visible grid part.
	const Box2& visibleGridRect() const { return _visibleGridRect; }

	/// \brief Computes the intersection point of a ray going through a point in the
	///        viewport plane with the grid plane.
	/// \param[in] viewportPosition A 2d point in viewport coordinates, in the range [-1,+1].
	/// \param[out] intersectionPoint The coordinates of the intersection point in grid plane coordinates.
	///                               The point can be transformed to world coordinates using the gridMatrix() transformation.
	/// \param[in] epsilon This threshold value is used to test whether the ray is parallel to the grid plane.
	/// \return \c true if an intersection has been found; \c false if not.
	/// \sa screenComputePlaneIntersection()
	bool viewportComputePlaneIntersection(const Point2& viewportPosition, Point3& intersectionPoint, FloatType epsilon = FLOATTYPE_EPSILON);

	/// \brief Computes the intersection of a ray going through a pixel in the
	///        3d window with the grid plane.
	/// \param[in] screenPosition A 2d point in the 3d window. This is relative to the upper left corner of the Window3D.
	/// \param[out] intersectionPoint The coordinates of the intersection point in grid plane coordinates.
	///                               The point can be transformed to world coordinates using the gridMatrix() transformation.
	/// \param[in] epsilon This threshold value is used to test whether the ray is parallel to the grid plane.
	/// \return \c true if an intersection has been found; \c false if not.
	/// \sa viewportComputePlaneIntersection()
	bool screenComputePlaneIntersection(const Point2I& screenPosition, Point3& intersectionPoint, FloatType epsilon = FLOATTYPE_EPSILON);

	/// \brief Computes a length in 3d space given by two viewport points.
	/// \param ray Specifies the base point (in grid coordinates) from which the length vector originates.
	/// \param p1 The base point in screen coordinates.
	/// \param p2 The second screen point that is used to determine the length of the vector whose direction is given by \a ray.
	/// \return The length of the computed vector.
	FloatType computeConstructionLength(const Ray3& ray, const Point2I& p1, const Point2I& p2);

protected:

	/// Constructor.
	ViewportGrid();

	/// The viewport the grid belongs to.
	Viewport* viewport;

	/// Transformation matrix from grid to world space.
	AffineTransformation _gridMatrix;

	/// The inverse of the above matrix.
	AffineTransformation _inverseGridMatrix;

	// The size of the displayed grid cells.
	FloatType _gridSpacing;

	/// Stores the visible part of the grid.
	Box2 _visibleGridRect;

	/// Renders the grid into the viewport.
	void renderGrid();

	/// Calculates the visible grid area and the grid spacing.
	void updateGrid();

	friend class Viewport;
};

};

#endif // __OVITO_VIEWPORT_GRID_H
