# This class installs and configures Plumgrid Neutron Plugin.
#
# === Parameters
#
# [*director_server*]
#   IP address of the PLUMgrid Director Server
#   Defaults to 127.0.0.1
#
# [*director_server_port*]
#   Port of the PLUMgrid Director Server.
#   Defaults to 443
#
# [*username*]
#   PLUMgrid platform username
#
# [*password*]
#   PLUMgrid platform password
#
# [*servertimeout*]
#   Request timeout duration (seconds) to PLUMgrid paltform
#   Defaults to 99
#
# [*connection*]
#   Database connection
#   Defaults to http://127.0.0.1:35357/v2.0
#
# [*admin_password*]
#   Keystone admin password
#
# [*controller_priv_host*]
#   Controller private host IP
#   Defaults to 127.0.0.1
#
# [*auth_protocol*]
#   Authorization protocol
#   Defaults to http
#
# [*nova_metadata_ip*]
#   Nova metadata IP
#   Defaults to 127.0.0.1
#
# [*nova_metadata_port*]
#   Nova metadata port
#   Defaults to 8775
#
# [*metadata_proxy_shared_secret*]
#   Neutron metadata shared secret key
#
# [*package_ensure*]
#   (optional) Ensure state for package.
#   Defaults to 'present'.
#
class neutron::plugins::plumgrid (
  $director_server              = '127.0.0.1',
  $director_server_port         = '443',
  $username                     = undef,
  $password                     = undef,
  $servertimeout                = '99',
  $connection                   = 'http://127.0.0.1:35357/v2.0',
  $admin_password               = undef,
  $controller_priv_host         = '127.0.0.1',
  $auth_protocol                = 'http',
  $nova_metadata_ip             = '127.0.0.1',
  $nova_metadata_port           = '8775',
  $metadata_proxy_shared_secret = undef,
  $package_ensure               = 'present'
) {

  include ::neutron::params

  Neutron_plugin_plumgrid<||> ~> Service['neutron-server']
  Neutron_plumlib_plumgrid<||> ~> Service['neutron-server']

  ensure_resource('file', '/etc/neutron/plugins/plumgrid', {
    ensure => directory,
    owner  => 'root',
    group  => 'neutron',
    mode   => '0640'}
  )

  # Ensure the neutron package is installed before config is set
  # under both RHEL and Ubuntu
  if ($::neutron::params::server_package) {
    Package['neutron-server'] -> Neutron_plugin_plumgrid<||>
    Package['neutron-server'] -> Neutron_plumlib_plumgrid<||>
  } else {
    Package['neutron'] -> Neutron_plugin_plumgrid<||>
    Package['neutron'] -> Neutron_plumlib_plumgrid<||>
  }

  package { 'neutron-plugin-plumgrid':
    ensure => $package_ensure,
    name   => $::neutron::params::plumgrid_plugin_package
  }

  package { 'neutron-plumlib-plumgrid':
    ensure => $package_ensure,
    name   => $::neutron::params::plumgrid_pythonlib_package
  }
  warning('neutron-plumlib-plumgrid package management is deprecated, it will be dropped in a future release.')

  if $::osfamily == 'Debian' {
    file_line { '/etc/default/neutron-server:NEUTRON_PLUGIN_CONFIG':
      path    => '/etc/default/neutron-server',
      match   => '^NEUTRON_PLUGIN_CONFIG=(.*)$',
      line    => "NEUTRON_PLUGIN_CONFIG=${::neutron::params::plumgrid_config_file}",
      require => [ Package['neutron-server'], Package['neutron-plugin-plumgrid'] ],
      notify  => Service['neutron-server'],
    }
  }

  if $::osfamily == 'Redhat' {
    file { '/etc/neutron/plugin.ini':
      ensure  => link,
      target  => $::neutron::params::plumgrid_config_file,
      require => Package['neutron-plugin-plumgrid'],
    }
  }

  neutron_plugin_plumgrid {
    'PLUMgridDirector/director_server':      value => $director_server;
    'PLUMgridDirector/director_server_port': value => $director_server_port;
    'PLUMgridDirector/username':             value => $username;
    'PLUMgridDirector/password':             value => $password, secret =>true;
    'PLUMgridDirector/servertimeout':        value => $servertimeout;
    'database/connection':                   value => $connection;
  }

  neutron_plumlib_plumgrid {
    'keystone_authtoken/admin_user' :                value => 'admin';
    'keystone_authtoken/admin_password':             value => $admin_password, secret =>true;
    'keystone_authtoken/auth_uri':                   value => "${auth_protocol}://${controller_priv_host}:35357/v2.0";
    'keystone_authtoken/admin_tenant_name':          value => 'admin';
    'PLUMgridMetadata/enable_pg_metadata' :          value => 'True';
    'PLUMgridMetadata/metadata_mode':                value => 'local';
    'PLUMgridMetadata/nova_metadata_ip':             value => $nova_metadata_ip;
    'PLUMgridMetadata/nova_metadata_port':           value => $nova_metadata_port;
    'PLUMgridMetadata/metadata_proxy_shared_secret': value => $metadata_proxy_shared_secret;
  }
}
