/* 
 *  Copyright (c) 2012 Daisuke Okanohara
 * 
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 * 
 *   1. Redistributions of source code must retain the above Copyright
 *      notice, this list of conditions and the following disclaimer.
 *
 *   2. Redistributions in binary form must reproduce the above Copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *
 *   3. Neither the name of the authors nor the names of its contributors
 *      may be used to endorse or promote products derived from this
 *      software without specific prior written permission.
 */

#include <cassert>
#include "Const.hpp"
#include "EnumCoder.hpp"

namespace rsdic{

uint64_t EnumCoder::Encode(uint64_t val, uint64_t rank_sb){
  uint64_t code = 0;
  for (uint64_t i = 0; i < kSmallBlockSize; ++i){
    if ((val >> i) & 1LLU){
      code += kCombinationTable64_[kSmallBlockSize - i - 1][rank_sb];
      --rank_sb;
    }
  }
  return code;
}

uint64_t EnumCoder::Decode(uint64_t code, uint64_t rank_sb){
  uint64_t ret = 0;
  for (uint64_t i = 0; i < kSmallBlockSize; ++i){
    uint64_t zero_case_num = 
      kCombinationTable64_[kSmallBlockSize - i - 1][rank_sb];
    if (code >= zero_case_num){
      ret |= (1LLU << i);
      code -= zero_case_num;
      --rank_sb;
    }
  }
  return ret;
}


bool EnumCoder::GetBit(uint64_t code, uint64_t rank_sb, uint64_t pos){
  if (Len(rank_sb) == kSmallBlockSize){
    return (code >> pos) & 1LLU;
  }
  for (uint64_t i = 0; i < pos; ++i){
    uint64_t zero_case_num = 
      kCombinationTable64_[kSmallBlockSize - i - 1][rank_sb];
    if (code >= zero_case_num){
      code -= zero_case_num;
      --rank_sb;
    }
  }
  return (code >= kCombinationTable64_[kSmallBlockSize - pos - 1][rank_sb]);
}

uint64_t EnumCoder::PopCount(uint64_t code){
  uint64_t r = code;
  r = (r & 0x5555555555555555ULL) +
    ((r >> 1) & 0x5555555555555555ULL);
  r = (r & 0x3333333333333333ULL) +
    ((r >> 2) & 0x3333333333333333ULL);
  r = (r + (r >> 4)) & 0x0f0f0f0f0f0f0f0fULL;
  r = r + (r >>  8);
  r = r + (r >> 16);
  r = r + (r >> 32);
  return (uint64_t)(r & 0x7f);

}

uint64_t EnumCoder::Rank(uint64_t code, uint64_t rank_sb, uint64_t pos){
  if (Len(rank_sb) == kSmallBlockSize){
    return PopCount(code & ((1LLU << pos) - 1));
  }

  uint64_t cur_rank = rank_sb;
  for (uint64_t i = 0; i < pos; ++i){
    uint64_t zero_case_num = 
      kCombinationTable64_[kSmallBlockSize - i - 1][cur_rank];
    if (code >= zero_case_num){
      code -= zero_case_num;
      --cur_rank;
    }
  }
  return rank_sb - cur_rank;
}

uint64_t EnumCoder::SelectRaw(uint64_t code, uint64_t num){
  uint64_t offset = 0;
  for (; offset < kSmallBlockSize; offset += 8){
    uint8_t r = kPopCount_[(code >> offset) & 0xff];
    if (num > r){
      num -= r;
    } else {
      break;
    }
  }
  
  for (; offset < kSmallBlockSize; ++offset){
    if ((code >> offset) & 1LLU){
      --num;
      if (num == 0) return offset;
      }
  }
  assert(false);
}

uint64_t EnumCoder::Select0(uint64_t code, uint64_t rank_sb, uint64_t num){
  if (Len(rank_sb) == kSmallBlockSize){
    return SelectRaw(~code, num);
  }
  for (uint64_t offset = 0; offset < kSmallBlockSize; ++ offset){
    uint64_t zero_case_num = kCombinationTable64_[kSmallBlockSize - offset - 1][rank_sb];
    if (code >= zero_case_num){
      code -= zero_case_num;
      --rank_sb;
    } else {
      --num;
      if (num == 0) return offset;
    }
  }
  assert(false);
}

uint64_t EnumCoder::Select1(uint64_t code, uint64_t rank_sb, uint64_t num){
  assert(num <= rank_sb);
  if (Len(rank_sb) == kSmallBlockSize){
    return SelectRaw(code, num);
  }

  for (uint64_t offset = 0; offset < kSmallBlockSize; ++ offset){
    uint64_t zero_case_num = kCombinationTable64_[kSmallBlockSize - offset - 1][rank_sb];
    if (code >= zero_case_num){
      --num;
      if (num == 0) return offset;
      code -= zero_case_num;
      --rank_sb;
    } 
  }
  assert(false);
}

uint64_t EnumCoder::Select(uint64_t code, uint64_t rank_sb, uint64_t num, bool bit){
  if (num == 0) return 0;
  if (bit) return Select1(code, rank_sb, num);
  else return Select0(code, rank_sb, num);
}

const uint8_t EnumCoder::kEnumCodeLength_[65] = {
  0,  6,  11, 16, 20, 23, 27, 30, 33, 35, 38, 40, 42, 44, 46, 64,
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
  64, 64, 46, 44, 42, 40, 38, 35, 33, 30, 27, 23, 20, 16, 11, 6, 
  0};


/*
const uint64_t EnumCoder::kEnumCodeLength_[65] = {
  0,  6,  11, 16, 20, 23, 27, 30, 33, 35, 38, 40, 42, 44, 46, 48, 
  49, 51, 52, 53, 55, 56, 57, 58, 58, 59, 60, 60, 60, 61, 61, 61,
  61, 61, 61, 61, 60, 60, 60, 59, 58, 58, 57, 56, 55, 53, 52, 51, 
  49, 48, 46, 44, 42, 40, 38, 35, 33, 30, 27, 23, 20, 16, 11, 6, 
  0};
*/

const uint8_t EnumCoder::kPopCount_[256] = {
0,1,1,2,1,2,2,3,1,2,2,3,2,3,3,4,
  1,2,2,3,2,3,3,4,2,3,3,4,3,4,4,5,
  1,2,2,3,2,3,3,4,2,3,3,4,3,4,4,5,
  2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,
  1,2,2,3,2,3,3,4,2,3,3,4,3,4,4,5,
  2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,
  2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,
  3,4,4,5,4,5,5,6,4,5,5,6,5,6,6,7,
  1,2,2,3,2,3,3,4,2,3,3,4,3,4,4,5,
  2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,
  2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,
  3,4,4,5,4,5,5,6,4,5,5,6,5,6,6,7,
  2,3,3,4,3,4,4,5,3,4,4,5,4,5,5,6,
  3,4,4,5,4,5,5,6,4,5,5,6,5,6,6,7,
  3,4,4,5,4,5,5,6,4,5,5,6,5,6,6,7,
  4,5,5,6,5,6,6,7,5,6,6,7,6,7,7,8
};


const uint64_t EnumCoder::kCombinationTable64_[65][65] = 
{ 
{ 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 2LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 3LLU, 3LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 4LLU, 6LLU, 4LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 5LLU, 10LLU, 10LLU, 5LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 6LLU, 15LLU, 20LLU, 15LLU, 6LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 7LLU, 21LLU, 35LLU, 35LLU, 21LLU, 7LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 8LLU, 28LLU, 56LLU, 70LLU, 56LLU, 28LLU, 8LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 9LLU, 36LLU, 84LLU, 126LLU, 126LLU, 84LLU, 36LLU, 9LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 10LLU, 45LLU, 120LLU, 210LLU, 252LLU, 210LLU, 120LLU, 45LLU, 10LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 11LLU, 55LLU, 165LLU, 330LLU, 462LLU, 462LLU, 330LLU, 165LLU, 55LLU, 11LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 12LLU, 66LLU, 220LLU, 495LLU, 792LLU, 924LLU, 792LLU, 495LLU, 220LLU, 66LLU, 12LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 13LLU, 78LLU, 286LLU, 715LLU, 1287LLU, 1716LLU, 1716LLU, 1287LLU, 715LLU, 286LLU, 78LLU, 13LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 14LLU, 91LLU, 364LLU, 1001LLU, 2002LLU, 3003LLU, 3432LLU, 3003LLU, 2002LLU, 1001LLU, 364LLU, 91LLU, 14LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 15LLU, 105LLU, 455LLU, 1365LLU, 3003LLU, 5005LLU, 6435LLU, 6435LLU, 5005LLU, 3003LLU, 1365LLU, 455LLU, 105LLU, 15LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 16LLU, 120LLU, 560LLU, 1820LLU, 4368LLU, 8008LLU, 11440LLU, 12870LLU, 11440LLU, 8008LLU, 4368LLU, 1820LLU, 560LLU, 120LLU, 16LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 17LLU, 136LLU, 680LLU, 2380LLU, 6188LLU, 12376LLU, 19448LLU, 24310LLU, 24310LLU, 19448LLU, 12376LLU, 6188LLU, 2380LLU, 680LLU, 136LLU, 17LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 18LLU, 153LLU, 816LLU, 3060LLU, 8568LLU, 18564LLU, 31824LLU, 43758LLU, 48620LLU, 43758LLU, 31824LLU, 18564LLU, 8568LLU, 3060LLU, 816LLU, 153LLU, 18LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 19LLU, 171LLU, 969LLU, 3876LLU, 11628LLU, 27132LLU, 50388LLU, 75582LLU, 92378LLU, 92378LLU, 75582LLU, 50388LLU, 27132LLU, 11628LLU, 3876LLU, 969LLU, 171LLU, 19LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 20LLU, 190LLU, 1140LLU, 4845LLU, 15504LLU, 38760LLU, 77520LLU, 125970LLU, 167960LLU, 184756LLU, 167960LLU, 125970LLU, 77520LLU, 38760LLU, 15504LLU, 4845LLU, 1140LLU, 190LLU, 20LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 21LLU, 210LLU, 1330LLU, 5985LLU, 20349LLU, 54264LLU, 116280LLU, 203490LLU, 293930LLU, 352716LLU, 352716LLU, 293930LLU, 203490LLU, 116280LLU, 54264LLU, 20349LLU, 5985LLU, 1330LLU, 210LLU, 21LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 22LLU, 231LLU, 1540LLU, 7315LLU, 26334LLU, 74613LLU, 170544LLU, 319770LLU, 497420LLU, 646646LLU, 705432LLU, 646646LLU, 497420LLU, 319770LLU, 170544LLU, 74613LLU, 26334LLU, 7315LLU, 1540LLU, 231LLU, 22LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 23LLU, 253LLU, 1771LLU, 8855LLU, 33649LLU, 100947LLU, 245157LLU, 490314LLU, 817190LLU, 1144066LLU, 1352078LLU, 1352078LLU, 1144066LLU, 817190LLU, 490314LLU, 245157LLU, 100947LLU, 33649LLU, 8855LLU, 1771LLU, 253LLU, 23LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 24LLU, 276LLU, 2024LLU, 10626LLU, 42504LLU, 134596LLU, 346104LLU, 735471LLU, 1307504LLU, 1961256LLU, 2496144LLU, 2704156LLU, 2496144LLU, 1961256LLU, 1307504LLU, 735471LLU, 346104LLU, 134596LLU, 42504LLU, 10626LLU, 2024LLU, 276LLU, 24LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 25LLU, 300LLU, 2300LLU, 12650LLU, 53130LLU, 177100LLU, 480700LLU, 1081575LLU, 2042975LLU, 3268760LLU, 4457400LLU, 5200300LLU, 5200300LLU, 4457400LLU, 3268760LLU, 2042975LLU, 1081575LLU, 480700LLU, 177100LLU, 53130LLU, 12650LLU, 2300LLU, 300LLU, 25LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 26LLU, 325LLU, 2600LLU, 14950LLU, 65780LLU, 230230LLU, 657800LLU, 1562275LLU, 3124550LLU, 5311735LLU, 7726160LLU, 9657700LLU, 10400600LLU, 9657700LLU, 7726160LLU, 5311735LLU, 3124550LLU, 1562275LLU, 657800LLU, 230230LLU, 65780LLU, 14950LLU, 2600LLU, 325LLU, 26LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 27LLU, 351LLU, 2925LLU, 17550LLU, 80730LLU, 296010LLU, 888030LLU, 2220075LLU, 4686825LLU, 8436285LLU, 13037895LLU, 17383860LLU, 20058300LLU, 20058300LLU, 17383860LLU, 13037895LLU, 8436285LLU, 4686825LLU, 2220075LLU, 888030LLU, 296010LLU, 80730LLU, 17550LLU, 2925LLU, 351LLU, 27LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 28LLU, 378LLU, 3276LLU, 20475LLU, 98280LLU, 376740LLU, 1184040LLU, 3108105LLU, 6906900LLU, 13123110LLU, 21474180LLU, 30421755LLU, 37442160LLU, 40116600LLU, 37442160LLU, 30421755LLU, 21474180LLU, 13123110LLU, 6906900LLU, 3108105LLU, 1184040LLU, 376740LLU, 98280LLU, 20475LLU, 3276LLU, 378LLU, 28LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 29LLU, 406LLU, 3654LLU, 23751LLU, 118755LLU, 475020LLU, 1560780LLU, 4292145LLU, 10015005LLU, 20030010LLU, 34597290LLU, 51895935LLU, 67863915LLU, 77558760LLU, 77558760LLU, 67863915LLU, 51895935LLU, 34597290LLU, 20030010LLU, 10015005LLU, 4292145LLU, 1560780LLU, 475020LLU, 118755LLU, 23751LLU, 3654LLU, 406LLU, 29LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 30LLU, 435LLU, 4060LLU, 27405LLU, 142506LLU, 593775LLU, 2035800LLU, 5852925LLU, 14307150LLU, 30045015LLU, 54627300LLU, 86493225LLU, 119759850LLU, 145422675LLU, 155117520LLU, 145422675LLU, 119759850LLU, 86493225LLU, 54627300LLU, 30045015LLU, 14307150LLU, 5852925LLU, 2035800LLU, 593775LLU, 142506LLU, 27405LLU, 4060LLU, 435LLU, 30LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 31LLU, 465LLU, 4495LLU, 31465LLU, 169911LLU, 736281LLU, 2629575LLU, 7888725LLU, 20160075LLU, 44352165LLU, 84672315LLU, 141120525LLU, 206253075LLU, 265182525LLU, 300540195LLU, 300540195LLU, 265182525LLU, 206253075LLU, 141120525LLU, 84672315LLU, 44352165LLU, 20160075LLU, 7888725LLU, 2629575LLU, 736281LLU, 169911LLU, 31465LLU, 4495LLU, 465LLU, 31LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 32LLU, 496LLU, 4960LLU, 35960LLU, 201376LLU, 906192LLU, 3365856LLU, 10518300LLU, 28048800LLU, 64512240LLU, 129024480LLU, 225792840LLU, 347373600LLU, 471435600LLU, 565722720LLU, 601080390LLU, 565722720LLU, 471435600LLU, 347373600LLU, 225792840LLU, 129024480LLU, 64512240LLU, 28048800LLU, 10518300LLU, 3365856LLU, 906192LLU, 201376LLU, 35960LLU, 4960LLU, 496LLU, 32LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 33LLU, 528LLU, 5456LLU, 40920LLU, 237336LLU, 1107568LLU, 4272048LLU, 13884156LLU, 38567100LLU, 92561040LLU, 193536720LLU, 354817320LLU, 573166440LLU, 818809200LLU, 1037158320LLU, 1166803110LLU, 1166803110LLU, 1037158320LLU, 818809200LLU, 573166440LLU, 354817320LLU, 193536720LLU, 92561040LLU, 38567100LLU, 13884156LLU, 4272048LLU, 1107568LLU, 237336LLU, 40920LLU, 5456LLU, 528LLU, 33LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 34LLU, 561LLU, 5984LLU, 46376LLU, 278256LLU, 1344904LLU, 5379616LLU, 18156204LLU, 52451256LLU, 131128140LLU, 286097760LLU, 548354040LLU, 927983760LLU, 1391975640LLU, 1855967520LLU, 2203961430LLU, 2333606220LLU, 2203961430LLU, 1855967520LLU, 1391975640LLU, 927983760LLU, 548354040LLU, 286097760LLU, 131128140LLU, 52451256LLU, 18156204LLU, 5379616LLU, 1344904LLU, 278256LLU, 46376LLU, 5984LLU, 561LLU, 34LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 35LLU, 595LLU, 6545LLU, 52360LLU, 324632LLU, 1623160LLU, 6724520LLU, 23535820LLU, 70607460LLU, 183579396LLU, 417225900LLU, 834451800LLU, 1476337800LLU, 2319959400LLU, 3247943160LLU, 4059928950LLU, 4537567650LLU, 4537567650LLU, 4059928950LLU, 3247943160LLU, 2319959400LLU, 1476337800LLU, 834451800LLU, 417225900LLU, 183579396LLU, 70607460LLU, 23535820LLU, 6724520LLU, 1623160LLU, 324632LLU, 52360LLU, 6545LLU, 595LLU, 35LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 36LLU, 630LLU, 7140LLU, 58905LLU, 376992LLU, 1947792LLU, 8347680LLU, 30260340LLU, 94143280LLU, 254186856LLU, 600805296LLU, 1251677700LLU, 2310789600LLU, 3796297200LLU, 5567902560LLU, 7307872110LLU, 8597496600LLU, 9075135300LLU, 8597496600LLU, 7307872110LLU, 5567902560LLU, 3796297200LLU, 2310789600LLU, 1251677700LLU, 600805296LLU, 254186856LLU, 94143280LLU, 30260340LLU, 8347680LLU, 1947792LLU, 376992LLU, 58905LLU, 7140LLU, 630LLU, 36LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 37LLU, 666LLU, 7770LLU, 66045LLU, 435897LLU, 2324784LLU, 10295472LLU, 38608020LLU, 124403620LLU, 348330136LLU, 854992152LLU, 1852482996LLU, 3562467300LLU, 6107086800LLU, 9364199760LLU, 12875774670LLU, 15905368710LLU, 17672631900LLU, 17672631900LLU, 15905368710LLU, 12875774670LLU, 9364199760LLU, 6107086800LLU, 3562467300LLU, 1852482996LLU, 854992152LLU, 348330136LLU, 124403620LLU, 38608020LLU, 10295472LLU, 2324784LLU, 435897LLU, 66045LLU, 7770LLU, 666LLU, 37LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 38LLU, 703LLU, 8436LLU, 73815LLU, 501942LLU, 2760681LLU, 12620256LLU, 48903492LLU, 163011640LLU, 472733756LLU, 1203322288LLU, 2707475148LLU, 5414950296LLU, 9669554100LLU, 15471286560LLU, 22239974430LLU, 28781143380LLU, 33578000610LLU, 35345263800LLU, 33578000610LLU, 28781143380LLU, 22239974430LLU, 15471286560LLU, 9669554100LLU, 5414950296LLU, 2707475148LLU, 1203322288LLU, 472733756LLU, 163011640LLU, 48903492LLU, 12620256LLU, 2760681LLU, 501942LLU, 73815LLU, 8436LLU, 703LLU, 38LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 39LLU, 741LLU, 9139LLU, 82251LLU, 575757LLU, 3262623LLU, 15380937LLU, 61523748LLU, 211915132LLU, 635745396LLU, 1676056044LLU, 3910797436LLU, 8122425444LLU, 15084504396LLU, 25140840660LLU, 37711260990LLU, 51021117810LLU, 62359143990LLU, 68923264410LLU, 68923264410LLU, 62359143990LLU, 51021117810LLU, 37711260990LLU, 25140840660LLU, 15084504396LLU, 8122425444LLU, 3910797436LLU, 1676056044LLU, 635745396LLU, 211915132LLU, 61523748LLU, 15380937LLU, 3262623LLU, 575757LLU, 82251LLU, 9139LLU, 741LLU, 39LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 40LLU, 780LLU, 9880LLU, 91390LLU, 658008LLU, 3838380LLU, 18643560LLU, 76904685LLU, 273438880LLU, 847660528LLU, 2311801440LLU, 5586853480LLU, 12033222880LLU, 23206929840LLU, 40225345056LLU, 62852101650LLU, 88732378800LLU, 113380261800LLU, 131282408400LLU, 137846528820LLU, 131282408400LLU, 113380261800LLU, 88732378800LLU, 62852101650LLU, 40225345056LLU, 23206929840LLU, 12033222880LLU, 5586853480LLU, 2311801440LLU, 847660528LLU, 273438880LLU, 76904685LLU, 18643560LLU, 3838380LLU, 658008LLU, 91390LLU, 9880LLU, 780LLU, 40LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 41LLU, 820LLU, 10660LLU, 101270LLU, 749398LLU, 4496388LLU, 22481940LLU, 95548245LLU, 350343565LLU, 1121099408LLU, 3159461968LLU, 7898654920LLU, 17620076360LLU, 35240152720LLU, 63432274896LLU, 103077446706LLU, 151584480450LLU, 202112640600LLU, 244662670200LLU, 269128937220LLU, 269128937220LLU, 244662670200LLU, 202112640600LLU, 151584480450LLU, 103077446706LLU, 63432274896LLU, 35240152720LLU, 17620076360LLU, 7898654920LLU, 3159461968LLU, 1121099408LLU, 350343565LLU, 95548245LLU, 22481940LLU, 4496388LLU, 749398LLU, 101270LLU, 10660LLU, 820LLU, 41LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 42LLU, 861LLU, 11480LLU, 111930LLU, 850668LLU, 5245786LLU, 26978328LLU, 118030185LLU, 445891810LLU, 1471442973LLU, 4280561376LLU, 11058116888LLU, 25518731280LLU, 52860229080LLU, 98672427616LLU, 166509721602LLU, 254661927156LLU, 353697121050LLU, 446775310800LLU, 513791607420LLU, 538257874440LLU, 513791607420LLU, 446775310800LLU, 353697121050LLU, 254661927156LLU, 166509721602LLU, 98672427616LLU, 52860229080LLU, 25518731280LLU, 11058116888LLU, 4280561376LLU, 1471442973LLU, 445891810LLU, 118030185LLU, 26978328LLU, 5245786LLU, 850668LLU, 111930LLU, 11480LLU, 861LLU, 42LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 43LLU, 903LLU, 12341LLU, 123410LLU, 962598LLU, 6096454LLU, 32224114LLU, 145008513LLU, 563921995LLU, 1917334783LLU, 5752004349LLU, 15338678264LLU, 36576848168LLU, 78378960360LLU, 151532656696LLU, 265182149218LLU, 421171648758LLU, 608359048206LLU, 800472431850LLU, 960566918220LLU, 1052049481860LLU, 1052049481860LLU, 960566918220LLU, 800472431850LLU, 608359048206LLU, 421171648758LLU, 265182149218LLU, 151532656696LLU, 78378960360LLU, 36576848168LLU, 15338678264LLU, 5752004349LLU, 1917334783LLU, 563921995LLU, 145008513LLU, 32224114LLU, 6096454LLU, 962598LLU, 123410LLU, 12341LLU, 903LLU, 43LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 44LLU, 946LLU, 13244LLU, 135751LLU, 1086008LLU, 7059052LLU, 38320568LLU, 177232627LLU, 708930508LLU, 2481256778LLU, 7669339132LLU, 21090682613LLU, 51915526432LLU, 114955808528LLU, 229911617056LLU, 416714805914LLU, 686353797976LLU, 1029530696964LLU, 1408831480056LLU, 1761039350070LLU, 2012616400080LLU, 2104098963720LLU, 2012616400080LLU, 1761039350070LLU, 1408831480056LLU, 1029530696964LLU, 686353797976LLU, 416714805914LLU, 229911617056LLU, 114955808528LLU, 51915526432LLU, 21090682613LLU, 7669339132LLU, 2481256778LLU, 708930508LLU, 177232627LLU, 38320568LLU, 7059052LLU, 1086008LLU, 135751LLU, 13244LLU, 946LLU, 44LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 45LLU, 990LLU, 14190LLU, 148995LLU, 1221759LLU, 8145060LLU, 45379620LLU, 215553195LLU, 886163135LLU, 3190187286LLU, 10150595910LLU, 28760021745LLU, 73006209045LLU, 166871334960LLU, 344867425584LLU, 646626422970LLU, 1103068603890LLU, 1715884494940LLU, 2438362177020LLU, 3169870830126LLU, 3773655750150LLU, 4116715363800LLU, 4116715363800LLU, 3773655750150LLU, 3169870830126LLU, 2438362177020LLU, 1715884494940LLU, 1103068603890LLU, 646626422970LLU, 344867425584LLU, 166871334960LLU, 73006209045LLU, 28760021745LLU, 10150595910LLU, 3190187286LLU, 886163135LLU, 215553195LLU, 45379620LLU, 8145060LLU, 1221759LLU, 148995LLU, 14190LLU, 990LLU, 45LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 46LLU, 1035LLU, 15180LLU, 163185LLU, 1370754LLU, 9366819LLU, 53524680LLU, 260932815LLU, 1101716330LLU, 4076350421LLU, 13340783196LLU, 38910617655LLU, 101766230790LLU, 239877544005LLU, 511738760544LLU, 991493848554LLU, 1749695026860LLU, 2818953098830LLU, 4154246671960LLU, 5608233007146LLU, 6943526580276LLU, 7890371113950LLU, 8233430727600LLU, 7890371113950LLU, 6943526580276LLU, 5608233007146LLU, 4154246671960LLU, 2818953098830LLU, 1749695026860LLU, 991493848554LLU, 511738760544LLU, 239877544005LLU, 101766230790LLU, 38910617655LLU, 13340783196LLU, 4076350421LLU, 1101716330LLU, 260932815LLU, 53524680LLU, 9366819LLU, 1370754LLU, 163185LLU, 15180LLU, 1035LLU, 46LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 47LLU, 1081LLU, 16215LLU, 178365LLU, 1533939LLU, 10737573LLU, 62891499LLU, 314457495LLU, 1362649145LLU, 5178066751LLU, 17417133617LLU, 52251400851LLU, 140676848445LLU, 341643774795LLU, 751616304549LLU, 1503232609098LLU, 2741188875414LLU, 4568648125690LLU, 6973199770790LLU, 9762479679106LLU, 12551759587422LLU, 14833897694226LLU, 16123801841550LLU, 16123801841550LLU, 14833897694226LLU, 12551759587422LLU, 9762479679106LLU, 6973199770790LLU, 4568648125690LLU, 2741188875414LLU, 1503232609098LLU, 751616304549LLU, 341643774795LLU, 140676848445LLU, 52251400851LLU, 17417133617LLU, 5178066751LLU, 1362649145LLU, 314457495LLU, 62891499LLU, 10737573LLU, 1533939LLU, 178365LLU, 16215LLU, 1081LLU, 47LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 48LLU, 1128LLU, 17296LLU, 194580LLU, 1712304LLU, 12271512LLU, 73629072LLU, 377348994LLU, 1677106640LLU, 6540715896LLU, 22595200368LLU, 69668534468LLU, 192928249296LLU, 482320623240LLU, 1093260079344LLU, 2254848913647LLU, 4244421484512LLU, 7309837001104LLU, 11541847896480LLU, 16735679449896LLU, 22314239266528LLU, 27385657281648LLU, 30957699535776LLU, 32247603683100LLU, 30957699535776LLU, 27385657281648LLU, 22314239266528LLU, 16735679449896LLU, 11541847896480LLU, 7309837001104LLU, 4244421484512LLU, 2254848913647LLU, 1093260079344LLU, 482320623240LLU, 192928249296LLU, 69668534468LLU, 22595200368LLU, 6540715896LLU, 1677106640LLU, 377348994LLU, 73629072LLU, 12271512LLU, 1712304LLU, 194580LLU, 17296LLU, 1128LLU, 48LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 49LLU, 1176LLU, 18424LLU, 211876LLU, 1906884LLU, 13983816LLU, 85900584LLU, 450978066LLU, 2054455634LLU, 8217822536LLU, 29135916264LLU, 92263734836LLU, 262596783764LLU, 675248872536LLU, 1575580702584LLU, 3348108992991LLU, 6499270398159LLU, 11554258485616LLU, 18851684897584LLU, 28277527346376LLU, 39049918716424LLU, 49699896548176LLU, 58343356817424LLU, 63205303218876LLU, 63205303218876LLU, 58343356817424LLU, 49699896548176LLU, 39049918716424LLU, 28277527346376LLU, 18851684897584LLU, 11554258485616LLU, 6499270398159LLU, 3348108992991LLU, 1575580702584LLU, 675248872536LLU, 262596783764LLU, 92263734836LLU, 29135916264LLU, 8217822536LLU, 2054455634LLU, 450978066LLU, 85900584LLU, 13983816LLU, 1906884LLU, 211876LLU, 18424LLU, 1176LLU, 49LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 50LLU, 1225LLU, 19600LLU, 230300LLU, 2118760LLU, 15890700LLU, 99884400LLU, 536878650LLU, 2505433700LLU, 10272278170LLU, 37353738800LLU, 121399651100LLU, 354860518600LLU, 937845656300LLU, 2250829575120LLU, 4923689695575LLU, 9847379391150LLU, 18053528883775LLU, 30405943383200LLU, 47129212243960LLU, 67327446062800LLU, 88749815264600LLU, 108043253365600LLU, 121548660036300LLU, 126410606437752LLU, 121548660036300LLU, 108043253365600LLU, 88749815264600LLU, 67327446062800LLU, 47129212243960LLU, 30405943383200LLU, 18053528883775LLU, 9847379391150LLU, 4923689695575LLU, 2250829575120LLU, 937845656300LLU, 354860518600LLU, 121399651100LLU, 37353738800LLU, 10272278170LLU, 2505433700LLU, 536878650LLU, 99884400LLU, 15890700LLU, 2118760LLU, 230300LLU, 19600LLU, 1225LLU, 50LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 51LLU, 1275LLU, 20825LLU, 249900LLU, 2349060LLU, 18009460LLU, 115775100LLU, 636763050LLU, 3042312350LLU, 12777711870LLU, 47626016970LLU, 158753389900LLU, 476260169700LLU, 1292706174900LLU, 3188675231420LLU, 7174519270695LLU, 14771069086725LLU, 27900908274925LLU, 48459472266975LLU, 77535155627160LLU, 114456658306760LLU, 156077261327400LLU, 196793068630200LLU, 229591913401900LLU, 247959266474052LLU, 247959266474052LLU, 229591913401900LLU, 196793068630200LLU, 156077261327400LLU, 114456658306760LLU, 77535155627160LLU, 48459472266975LLU, 27900908274925LLU, 14771069086725LLU, 7174519270695LLU, 3188675231420LLU, 1292706174900LLU, 476260169700LLU, 158753389900LLU, 47626016970LLU, 12777711870LLU, 3042312350LLU, 636763050LLU, 115775100LLU, 18009460LLU, 2349060LLU, 249900LLU, 20825LLU, 1275LLU, 51LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 52LLU, 1326LLU, 22100LLU, 270725LLU, 2598960LLU, 20358520LLU, 133784560LLU, 752538150LLU, 3679075400LLU, 15820024220LLU, 60403728840LLU, 206379406870LLU, 635013559600LLU, 1768966344600LLU, 4481381406320LLU, 10363194502115LLU, 21945588357420LLU, 42671977361650LLU, 76360380541900LLU, 125994627894135LLU, 191991813933920LLU, 270533919634160LLU, 352870329957600LLU, 426384982032100LLU, 477551179875952LLU, 495918532948104LLU, 477551179875952LLU, 426384982032100LLU, 352870329957600LLU, 270533919634160LLU, 191991813933920LLU, 125994627894135LLU, 76360380541900LLU, 42671977361650LLU, 21945588357420LLU, 10363194502115LLU, 4481381406320LLU, 1768966344600LLU, 635013559600LLU, 206379406870LLU, 60403728840LLU, 15820024220LLU, 3679075400LLU, 752538150LLU, 133784560LLU, 20358520LLU, 2598960LLU, 270725LLU, 22100LLU, 1326LLU, 52LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 53LLU, 1378LLU, 23426LLU, 292825LLU, 2869685LLU, 22957480LLU, 154143080LLU, 886322710LLU, 4431613550LLU, 19499099620LLU, 76223753060LLU, 266783135710LLU, 841392966470LLU, 2403979904200LLU, 6250347750920LLU, 14844575908435LLU, 32308782859535LLU, 64617565719070LLU, 119032357903550LLU, 202355008436035LLU, 317986441828055LLU, 462525733568080LLU, 623404249591760LLU, 779255311989700LLU, 903936161908052LLU, 973469712824056LLU, 973469712824056LLU, 903936161908052LLU, 779255311989700LLU, 623404249591760LLU, 462525733568080LLU, 317986441828055LLU, 202355008436035LLU, 119032357903550LLU, 64617565719070LLU, 32308782859535LLU, 14844575908435LLU, 6250347750920LLU, 2403979904200LLU, 841392966470LLU, 266783135710LLU, 76223753060LLU, 19499099620LLU, 4431613550LLU, 886322710LLU, 154143080LLU, 22957480LLU, 2869685LLU, 292825LLU, 23426LLU, 1378LLU, 53LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 54LLU, 1431LLU, 24804LLU, 316251LLU, 3162510LLU, 25827165LLU, 177100560LLU, 1040465790LLU, 5317936260LLU, 23930713170LLU, 95722852680LLU, 343006888770LLU, 1108176102180LLU, 3245372870670LLU, 8654327655120LLU, 21094923659355LLU, 47153358767970LLU, 96926348578605LLU, 183649923622620LLU, 321387366339585LLU, 520341450264090LLU, 780512175396135LLU, 1085929983159840LLU, 1402659561581460LLU, 1683191473897752LLU, 1877405874732108LLU, 1946939425648112LLU, 1877405874732108LLU, 1683191473897752LLU, 1402659561581460LLU, 1085929983159840LLU, 780512175396135LLU, 520341450264090LLU, 321387366339585LLU, 183649923622620LLU, 96926348578605LLU, 47153358767970LLU, 21094923659355LLU, 8654327655120LLU, 3245372870670LLU, 1108176102180LLU, 343006888770LLU, 95722852680LLU, 23930713170LLU, 5317936260LLU, 1040465790LLU, 177100560LLU, 25827165LLU, 3162510LLU, 316251LLU, 24804LLU, 1431LLU, 54LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 55LLU, 1485LLU, 26235LLU, 341055LLU, 3478761LLU, 28989675LLU, 202927725LLU, 1217566350LLU, 6358402050LLU, 29248649430LLU, 119653565850LLU, 438729741450LLU, 1451182990950LLU, 4353548972850LLU, 11899700525790LLU, 29749251314475LLU, 68248282427325LLU, 144079707346575LLU, 280576272201225LLU, 505037289962205LLU, 841728816603675LLU, 1300853625660225LLU, 1866442158555975LLU, 2488589544741300LLU, 3085851035479212LLU, 3560597348629860LLU, 3824345300380220LLU, 3824345300380220LLU, 3560597348629860LLU, 3085851035479212LLU, 2488589544741300LLU, 1866442158555975LLU, 1300853625660225LLU, 841728816603675LLU, 505037289962205LLU, 280576272201225LLU, 144079707346575LLU, 68248282427325LLU, 29749251314475LLU, 11899700525790LLU, 4353548972850LLU, 1451182990950LLU, 438729741450LLU, 119653565850LLU, 29248649430LLU, 6358402050LLU, 1217566350LLU, 202927725LLU, 28989675LLU, 3478761LLU, 341055LLU, 26235LLU, 1485LLU, 55LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 56LLU, 1540LLU, 27720LLU, 367290LLU, 3819816LLU, 32468436LLU, 231917400LLU, 1420494075LLU, 7575968400LLU, 35607051480LLU, 148902215280LLU, 558383307300LLU, 1889912732400LLU, 5804731963800LLU, 16253249498640LLU, 41648951840265LLU, 97997533741800LLU, 212327989773900LLU, 424655979547800LLU, 785613562163430LLU, 1346766106565880LLU, 2142582442263900LLU, 3167295784216200LLU, 4355031703297275LLU, 5574440580220512LLU, 6646448384109072LLU, 7384942649010080LLU, 7648690600760440LLU, 7384942649010080LLU, 6646448384109072LLU, 5574440580220512LLU, 4355031703297275LLU, 3167295784216200LLU, 2142582442263900LLU, 1346766106565880LLU, 785613562163430LLU, 424655979547800LLU, 212327989773900LLU, 97997533741800LLU, 41648951840265LLU, 16253249498640LLU, 5804731963800LLU, 1889912732400LLU, 558383307300LLU, 148902215280LLU, 35607051480LLU, 7575968400LLU, 1420494075LLU, 231917400LLU, 32468436LLU, 3819816LLU, 367290LLU, 27720LLU, 1540LLU, 56LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 57LLU, 1596LLU, 29260LLU, 395010LLU, 4187106LLU, 36288252LLU, 264385836LLU, 1652411475LLU, 8996462475LLU, 43183019880LLU, 184509266760LLU, 707285522580LLU, 2448296039700LLU, 7694644696200LLU, 22057981462440LLU, 57902201338905LLU, 139646485582065LLU, 310325523515700LLU, 636983969321700LLU, 1210269541711230LLU, 2132379668729310LLU, 3489348548829780LLU, 5309878226480100LLU, 7522327487513475LLU, 9929472283517787LLU, 12220888964329584LLU, 14031391033119152LLU, 15033633249770520LLU, 15033633249770520LLU, 14031391033119152LLU, 12220888964329584LLU, 9929472283517787LLU, 7522327487513475LLU, 5309878226480100LLU, 3489348548829780LLU, 2132379668729310LLU, 1210269541711230LLU, 636983969321700LLU, 310325523515700LLU, 139646485582065LLU, 57902201338905LLU, 22057981462440LLU, 7694644696200LLU, 2448296039700LLU, 707285522580LLU, 184509266760LLU, 43183019880LLU, 8996462475LLU, 1652411475LLU, 264385836LLU, 36288252LLU, 4187106LLU, 395010LLU, 29260LLU, 1596LLU, 57LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 58LLU, 1653LLU, 30856LLU, 424270LLU, 4582116LLU, 40475358LLU, 300674088LLU, 1916797311LLU, 10648873950LLU, 52179482355LLU, 227692286640LLU, 891794789340LLU, 3155581562280LLU, 10142940735900LLU, 29752626158640LLU, 79960182801345LLU, 197548686920970LLU, 449972009097765LLU, 947309492837400LLU, 1847253511032930LLU, 3342649210440540LLU, 5621728217559090LLU, 8799226775309880LLU, 12832205713993575LLU, 17451799771031262LLU, 22150361247847371LLU, 26252279997448736LLU, 29065024282889672LLU, 30067266499541040LLU, 29065024282889672LLU, 26252279997448736LLU, 22150361247847371LLU, 17451799771031262LLU, 12832205713993575LLU, 8799226775309880LLU, 5621728217559090LLU, 3342649210440540LLU, 1847253511032930LLU, 947309492837400LLU, 449972009097765LLU, 197548686920970LLU, 79960182801345LLU, 29752626158640LLU, 10142940735900LLU, 3155581562280LLU, 891794789340LLU, 227692286640LLU, 52179482355LLU, 10648873950LLU, 1916797311LLU, 300674088LLU, 40475358LLU, 4582116LLU, 424270LLU, 30856LLU, 1653LLU, 58LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 59LLU, 1711LLU, 32509LLU, 455126LLU, 5006386LLU, 45057474LLU, 341149446LLU, 2217471399LLU, 12565671261LLU, 62828356305LLU, 279871768995LLU, 1119487075980LLU, 4047376351620LLU, 13298522298180LLU, 39895566894540LLU, 109712808959985LLU, 277508869722315LLU, 647520696018735LLU, 1397281501935165LLU, 2794563003870330LLU, 5189902721473470LLU, 8964377427999630LLU, 14420954992868970LLU, 21631432489303455LLU, 30284005485024837LLU, 39602161018878633LLU, 48402641245296107LLU, 55317304280338408LLU, 59132290782430712LLU, 59132290782430712LLU, 55317304280338408LLU, 48402641245296107LLU, 39602161018878633LLU, 30284005485024837LLU, 21631432489303455LLU, 14420954992868970LLU, 8964377427999630LLU, 5189902721473470LLU, 2794563003870330LLU, 1397281501935165LLU, 647520696018735LLU, 277508869722315LLU, 109712808959985LLU, 39895566894540LLU, 13298522298180LLU, 4047376351620LLU, 1119487075980LLU, 279871768995LLU, 62828356305LLU, 12565671261LLU, 2217471399LLU, 341149446LLU, 45057474LLU, 5006386LLU, 455126LLU, 32509LLU, 1711LLU, 59LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 60LLU, 1770LLU, 34220LLU, 487635LLU, 5461512LLU, 50063860LLU, 386206920LLU, 2558620845LLU, 14783142660LLU, 75394027566LLU, 342700125300LLU, 1399358844975LLU, 5166863427600LLU, 17345898649800LLU, 53194089192720LLU, 149608375854525LLU, 387221678682300LLU, 925029565741050LLU, 2044802197953900LLU, 4191844505805495LLU, 7984465725343800LLU, 14154280149473100LLU, 23385332420868600LLU, 36052387482172425LLU, 51915437974328292LLU, 69886166503903470LLU, 88004802264174740LLU, 103719945525634515LLU, 114449595062769120LLU, 118264581564861424LLU, 114449595062769120LLU, 103719945525634515LLU, 88004802264174740LLU, 69886166503903470LLU, 51915437974328292LLU, 36052387482172425LLU, 23385332420868600LLU, 14154280149473100LLU, 7984465725343800LLU, 4191844505805495LLU, 2044802197953900LLU, 925029565741050LLU, 387221678682300LLU, 149608375854525LLU, 53194089192720LLU, 17345898649800LLU, 5166863427600LLU, 1399358844975LLU, 342700125300LLU, 75394027566LLU, 14783142660LLU, 2558620845LLU, 386206920LLU, 50063860LLU, 5461512LLU, 487635LLU, 34220LLU, 1770LLU, 60LLU, 1LLU, 0LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 61LLU, 1830LLU, 35990LLU, 521855LLU, 5949147LLU, 55525372LLU, 436270780LLU, 2944827765LLU, 17341763505LLU, 90177170226LLU, 418094152866LLU, 1742058970275LLU, 6566222272575LLU, 22512762077400LLU, 70539987842520LLU, 202802465047245LLU, 536830054536825LLU, 1312251244423350LLU, 2969831763694950LLU, 6236646703759395LLU, 12176310231149295LLU, 22138745874816900LLU, 37539612570341700LLU, 59437719903041025LLU, 87967825456500717LLU, 121801604478231762LLU, 157890968768078210LLU, 191724747789809255LLU, 218169540588403635LLU, 232714176627630544LLU, 232714176627630544LLU, 218169540588403635LLU, 191724747789809255LLU, 157890968768078210LLU, 121801604478231762LLU, 87967825456500717LLU, 59437719903041025LLU, 37539612570341700LLU, 22138745874816900LLU, 12176310231149295LLU, 6236646703759395LLU, 2969831763694950LLU, 1312251244423350LLU, 536830054536825LLU, 202802465047245LLU, 70539987842520LLU, 22512762077400LLU, 6566222272575LLU, 1742058970275LLU, 418094152866LLU, 90177170226LLU, 17341763505LLU, 2944827765LLU, 436270780LLU, 55525372LLU, 5949147LLU, 521855LLU, 35990LLU, 1830LLU, 61LLU, 1LLU, 0LLU, 0LLU, 0LLU},
{ 1LLU, 62LLU, 1891LLU, 37820LLU, 557845LLU, 6471002LLU, 61474519LLU, 491796152LLU, 3381098545LLU, 20286591270LLU, 107518933731LLU, 508271323092LLU, 2160153123141LLU, 8308281242850LLU, 29078984349975LLU, 93052749919920LLU, 273342452889765LLU, 739632519584070LLU, 1849081298960175LLU, 4282083008118300LLU, 9206478467454345LLU, 18412956934908690LLU, 34315056105966195LLU, 59678358445158600LLU, 96977332473382725LLU, 147405545359541742LLU, 209769429934732479LLU, 279692573246309972LLU, 349615716557887465LLU, 409894288378212890LLU, 450883717216034179LLU, 465428353255261088LLU, 450883717216034179LLU, 409894288378212890LLU, 349615716557887465LLU, 279692573246309972LLU, 209769429934732479LLU, 147405545359541742LLU, 96977332473382725LLU, 59678358445158600LLU, 34315056105966195LLU, 18412956934908690LLU, 9206478467454345LLU, 4282083008118300LLU, 1849081298960175LLU, 739632519584070LLU, 273342452889765LLU, 93052749919920LLU, 29078984349975LLU, 8308281242850LLU, 2160153123141LLU, 508271323092LLU, 107518933731LLU, 20286591270LLU, 3381098545LLU, 491796152LLU, 61474519LLU, 6471002LLU, 557845LLU, 37820LLU, 1891LLU, 62LLU, 1LLU, 0LLU, 0LLU},
{ 1LLU, 63LLU, 1953LLU, 39711LLU, 595665LLU, 7028847LLU, 67945521LLU, 553270671LLU, 3872894697LLU, 23667689815LLU, 127805525001LLU, 615790256823LLU, 2668424446233LLU, 10468434365991LLU, 37387265592825LLU, 122131734269895LLU, 366395202809685LLU, 1012974972473835LLU, 2588713818544245LLU, 6131164307078475LLU, 13488561475572645LLU, 27619435402363035LLU, 52728013040874885LLU, 93993414551124795LLU, 156655690918541325LLU, 244382877832924467LLU, 357174975294274221LLU, 489462003181042451LLU, 629308289804197437LLU, 759510004936100355LLU, 860778005594247069LLU, 916312070471295267LLU, 916312070471295267LLU, 860778005594247069LLU, 759510004936100355LLU, 629308289804197437LLU, 489462003181042451LLU, 357174975294274221LLU, 244382877832924467LLU, 156655690918541325LLU, 93993414551124795LLU, 52728013040874885LLU, 27619435402363035LLU, 13488561475572645LLU, 6131164307078475LLU, 2588713818544245LLU, 1012974972473835LLU, 366395202809685LLU, 122131734269895LLU, 37387265592825LLU, 10468434365991LLU, 2668424446233LLU, 615790256823LLU, 127805525001LLU, 23667689815LLU, 3872894697LLU, 553270671LLU, 67945521LLU, 7028847LLU, 595665LLU, 39711LLU, 1953LLU, 63LLU, 1LLU, 0LLU},
{ 1LLU, 64LLU, 2016LLU, 41664LLU, 635376LLU, 7624512LLU, 74974368LLU, 621216192LLU, 4426165368LLU, 27540584512LLU, 151473214816LLU, 743595781824LLU, 3284214703056LLU, 13136858812224LLU, 47855699958816LLU, 159518999862720LLU, 488526937079580LLU, 1379370175283520LLU, 3601688791018080LLU, 8719878125622720LLU, 19619725782651120LLU, 41107996877935680LLU, 80347448443237920LLU, 146721427591999680LLU, 250649105469666120LLU, 401038568751465792LLU, 601557853127198688LLU, 846636978475316672LLU, 1118770292985239888LLU, 1388818294740297792LLU, 1620288010530347424LLU, 1777090076065542336LLU, 1832624140942590534LLU, 1777090076065542336LLU, 1620288010530347424LLU, 1388818294740297792LLU, 1118770292985239888LLU, 846636978475316672LLU, 601557853127198688LLU, 401038568751465792LLU, 250649105469666120LLU, 146721427591999680LLU, 80347448443237920LLU, 41107996877935680LLU, 19619725782651120LLU, 8719878125622720LLU, 3601688791018080LLU, 1379370175283520LLU, 488526937079580LLU, 159518999862720LLU, 47855699958816LLU, 13136858812224LLU, 3284214703056LLU, 743595781824LLU, 151473214816LLU, 27540584512LLU, 4426165368LLU, 621216192LLU, 74974368LLU, 7624512LLU, 635376LLU, 41664LLU, 2016LLU, 64LLU, 1LLU}};

}
