/**
 * <p>
 * (C) Siemens Aktiengesellschaft, Germany, 2015
 * </p>
 * 
 * @since Sol7
 * @author  zelgerpe
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program (see the file LICENSE.txt for more
 * details); if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
 */

package org.acplt.oncrpc.apps.jrpcgen;

import java.io.IOException;

/**
 * base class providing storage of a javadoc comment. for use at jrpcgen data model.
 * @author zelgerpe
 */
public class JrpcgenDocumentable 
{
  protected String documentation; /* javadoc comment out of .x file */
  static String eol=System.getProperty("line.separator");

  public static String emptyDocumentation() {
	  return "/** TODO */";
  }
  
  public JrpcgenDocumentable()
  {
	  this(emptyDocumentation());
  }

  public JrpcgenDocumentable(String jdoc)
  {
	  this.documentation = getNormalizedDocumentation(jdoc);
  }

  public boolean documentationIsEmpty() {
	  return emptyDocumentation().equals(documentation);
  }
  
  public boolean documentationIsProvided() {
	  return (emptyDocumentation().compareTo(documentation) != 0);
  }
  
  public String getDocumentation() {
	  return documentation;
  }
  
  void setDocumentation(String jdoc)
  {
    documentation = getNormalizedDocumentation(jdoc);
  }

  /**
   * format javadoc comment. 
   * In case it contains a deprecated javadoc tag a {@link Deprecated} java annotation is generated.
   * @param prefix is prefixed each comment line
   * @return prefixed pretty formatted javadoc comment.
   */
  public String getIdentedDocu(final String prefix)
  {
    return getIdentedDocu(prefix, true);
  }
  
  public String getIdentedDocu(final String prefix, final boolean isWithDeprecatedAnnotation)
  {
    StringBuilder sb = new StringBuilder();
    String[] sa = documentation.split("\n|\r\n");
    for (int i = 0; i < sa.length; i++)
    {
      if (i == 0)
        sb.append(prefix).append(sa[0]);
      else
        {
        int istar=sa[i].indexOf('*');
        if(istar>=0)
          sb.append(prefix).append(' ').append(sa[i].substring(istar));
        else
          sb.append(prefix).append(sa[i]);
        }
      if (i < sa.length - 1) sb.append(eol);
    }
    // generate @Deprecated java annotation if @deprecated javadoc tag is found see USF10633 
    if(isWithDeprecatedAnnotation)
      appendDeprecatedOrNothing(sb, prefix);
    return sb.toString();    
  }
  
  public void writeDocumentation(JrpcgenJavaFile javaFile) {
	  writeDocumentation(javaFile, true);
  }
  
  public void writeDocumentation(JrpcgenJavaFile javaFile, final boolean isWithDeprecatedAnnotation) {
	  String[] sa = documentation.split("\n|\r\n");
	  
	  for (int i = 0; i < sa.length; i++)
	  {
		  if (i == 0)
			  javaFile.beginLine().append(sa[0]);
		  else
		  {
			  int istar=sa[i].indexOf('*');
			  if(istar>=0)
				  javaFile.beginNewLine().append(' ').append(sa[i].substring(istar));
			  else
				  javaFile.beginNewLine().append(sa[i]);
		  }
	  }
	  // generate @Deprecated java annotation if @deprecated javadoc tag is found see USF10633 
	  if(isWithDeprecatedAnnotation && isDeprecated()) {
		  javaFile.beginNewLine().append("@Deprecated");
	  }
  }
  
  protected void appendDeprecatedOrNothing(JrpcgenJavaFile javaFile) {
	  if (isDeprecated()) {
		  javaFile.beginNewLine().append("@Deprecated");
	  }
  }
  
  /** generate \n+prefix+@Deprecated java annotation if <i>deprecated</i> javadoc tag is found, otherwise do nothing.
   *  see USF10633 
   *  @return p */
  protected Appendable appendDeprecatedOrNothing(Appendable p, String prefix)
  {
    if(isDeprecated()) try
    {
      p.append(eol).append(prefix).append("@Deprecated");
    }
    catch (IOException e)
    {
      e.printStackTrace(System.err);
    }
    return p;
  }
  
  protected boolean isDeprecated()
  {
    return documentation.contains("@deprecated");
  }
  
  private static String getNormalizedDocumentation(String jdoc) {
	  if (emptyDocumentation().equals(jdoc) || (jdoc == null) || jdoc.isEmpty())
		  return emptyDocumentation();
	  
	  return jdoc;
  }
  
}
