#!/usr/bin/python3


# convert-data.py - converts the cached data to Python code
#
#   Copyright 2023-2025 Paul Wise <pabs@debian.org>
#
# This program is freely distributable per the following license:
#
#  Permission to use, copy, modify, and distribute this software and its
#  documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appears in all copies and that
#  both that copyright notice and this permission notice appear in
#  supporting documentation.
#
#  I DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
#  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL I
#  BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
#  DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
#  WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
#  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
#  SOFTWARE.


import ast
import pprint


warning = 'THIS FILE IS AUTOGENERATED - DO NOT EDIT - see cache README for more info'


def parse_arch_usertags_table(arch_usertags_table):
    arch_usertags = {}
    for line in arch_usertags_table.split('\n'):
        fields = line.split('||')
        fields = [field.strip() for field in fields]
        fields = [field.removeprefix('[[') for field in fields]
        fields = [field.removesuffix(']]') for field in fields]
        docs_name = fields[1].split('|')
        if len(docs_name) > 1:
            name = docs_name[1]
            docs = docs_name[0]
        else:
            name = docs = docs_name[0]
        if docs and not docs.startswith(('http://', 'https://')):
            docs = 'https://wiki.debian.org/' + docs
        user = fields[2].removeprefix('DebianBTSUser:')
        if not user:
            user = None
        usertags = fields[3].split()
        if not usertags:
            usertags = None
        xcc = fields[4].split('|')[1] if fields[4] else None
        arch_usertags[name] = (docs, user, usertags, xcc)
    return arch_usertags


def parse_arch_names_wml(arch_names_wml):
    arch_names = {}
    for line in arch_names_wml.split('\n'):
        fields = line.split('=>')
        arch, name = [field.strip(",' \t") for field in fields]
        name = name.removeprefix('<gettext domain="distrib">')
        name = name.removesuffix('</gettext>')
        arch_names[arch] = name
    return arch_names


def create_arch_descriptions(arch_names, arch_usertags, arch_status):
    arches = sorted(arch_names.keys() | arch_usertags.keys() | arch_status.keys())
    descriptions = {}
    for arch in arches:
        name = arch_names.get(arch, '')
        name = name.removesuffix(' (%s)' % arch)
        name = name.removesuffix(' (%s)' % arch.split('-')[-1])
        usertags = arch_usertags.get(arch)
        url = usertags[0] if usertags else None
        descriptions[arch] = ' '.join(filter(None, [name, url]))
    descriptions['all'] = 'Architecture independent binary packages'
    return descriptions


def load(filename):
    with open('cache/' + filename) as f:
        next(f)  # Skip warning header
        next(f)  # Skip blank line
        return f.read().strip()


def format(data):
    return pprint.pformat(data, indent=4, sort_dicts=False)


def convert_arch_info():
    arch_usertags_table = load('wiki-Teams-Debbugs-ArchitectureTags-usertags-table.moin')
    arch_names_wml = load('salsa-webmaster-team-webwml-releases-arches-only.data')
    wannabuild_arches = load('udd-wannabuild-architecture-vancouvered.py')

    arch_usertags = parse_arch_usertags_table(arch_usertags_table)
    arch_names = parse_arch_names_wml(arch_names_wml)
    wannabuild_arches = ast.literal_eval(wannabuild_arches)

    arch_status = {arch: not unofficial for arch, unofficial in wannabuild_arches}
    order_arches = sorted(wannabuild_arches, key=lambda item: (item[1], item[0]))
    order_arches = [arch for arch, unofficial in order_arches]
    arch_descriptions = create_arch_descriptions(arch_names, arch_usertags, arch_status)

    with open('reportbug/data.py', 'w') as f:
        f.write(
            f'''# {warning}

arch_descriptions = {format(arch_descriptions)}

arch_status = {format(arch_status)}

arch_usertags = {format(arch_usertags)}

order_arches = {format(order_arches)}
'''
            .replace(' = {   ', ' = {\n    ')
            .replace('}', ',\n}')
            .replace(' = [   ', ' = [\n    ')
            .replace(']\n\n', ',\n]\n\n')
            .replace(': (   ', ': (\n        ')
            .replace('),', ',\n    ),')
            .replace('                ', '        ')
            .replace('              ', '        ')
            .replace('            ', '        ')
            .replace('          ', '        ')
            .replace('         ', '        ')
            .replace("'\n        '", "")
            .replace("'", '"')
        )


convert_arch_info()
