// @HEADER
// ************************************************************************
//
//               Rapid Optimization Library (ROL) Package
//                 Copyright (2014) Sandia Corporation
//
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact lead developers:
//              Drew Kouri   (dpkouri@sandia.gov) and
//              Denis Ridzal (dridzal@sandia.gov)
//
// ************************************************************************
// @HEADER

/** \file
    \brief  Contains definitions for the equality constrained NLP:

            minimize  x^2 + y^2
          subject to  (x-2)^2 + y^2 = 1

    \author Created by D. Ridzal and D. Kouri.
 */

#ifndef ROL_PARABOLOIDCIRCLE_HPP
#define ROL_PARABOLOIDCIRCLE_HPP

#include "ROL_Objective.hpp"
#include "ROL_StdVector.hpp"
#include "ROL_EqualityConstraint.hpp"
#include "Teuchos_SerialDenseVector.hpp"
#include "Teuchos_SerialDenseSolver.hpp"

namespace ROL {
namespace ZOO {

  /** \brief Objective function:
             f(x,y) = x^2 + y^2
   */
  template< class Real, class XPrim=StdVector<Real>, class XDual=StdVector<Real> >
  class Objective_ParaboloidCircle : public Objective<Real> {

  typedef std::vector<Real> vector;
  typedef Vector<Real>      V;

  typedef typename vector::size_type uint;
   

  private:

    template<class VectorType>
    Teuchos::RCP<const vector> getVector( const V& x ) {
      using Teuchos::dyn_cast;
      return dyn_cast<const VectorType>(x).getVector();
    }

    template<class VectorType>
    Teuchos::RCP<vector> getVector( V& x ) {
      using Teuchos::dyn_cast;
      return dyn_cast<VectorType>(x).getVector();
    }

  public:
    Objective_ParaboloidCircle() {}

    Real value( const Vector<Real> &x, Real &tol ) {
 
      using Teuchos::RCP;
      RCP<const vector> xp = getVector<XPrim>(x); 

      uint n = xp->size();
      TEUCHOS_TEST_FOR_EXCEPTION( (n != 2), std::invalid_argument, ">>> ERROR (ROL_ParaboloidCircle, objective value): "
                                                                   "Primal vector x must be of length 2.");

      Real x1 = (*xp)[0];
      Real x2 = (*xp)[1];

      Real val = x1*x1 + x2*x2;

      return val;
    }

    void gradient( Vector<Real> &g, const Vector<Real> &x, Real &tol ) {

      using Teuchos::RCP;
      RCP<const vector> xp = getVector<XPrim>(x);
      RCP<vector> gp = getVector<XDual>(g); 

      uint n = xp->size();
      TEUCHOS_TEST_FOR_EXCEPTION( (n != 2), std::invalid_argument, ">>> ERROR (ROL_ParaboloidCircle, objective gradient): "
                                                                   " Primal vector x must be of length 2."); 

      n = gp->size();
      TEUCHOS_TEST_FOR_EXCEPTION( (n != 2), std::invalid_argument, ">>> ERROR (ROL_ParaboloidCircle, objective gradient): "
                                                                   "Gradient vector g must be of length 2."); 

      Real x1 = (*xp)[0];
      Real x2 = (*xp)[1];

      Real two(2);

      (*gp)[0] = two*x1;
      (*gp)[1] = two*x2;
    }

    void hessVec( Vector<Real> &hv, const Vector<Real> &v, const Vector<Real> &x, Real &tol ) {

      using Teuchos::RCP;
      RCP<const vector> xp = getVector<XPrim>(x);
      RCP<const vector> vp = getVector<XPrim>(v);
      RCP<vector> hvp = getVector<XDual>(hv);

      uint n = xp->size();
      TEUCHOS_TEST_FOR_EXCEPTION( (n != 2), std::invalid_argument, ">>> ERROR (ROL_ParaboloidCircle, objective hessVec): "
                                                                   "Primal vector x must be of length 2."); 

      n = vp->size();
      TEUCHOS_TEST_FOR_EXCEPTION( (n != 2), std::invalid_argument, ">>> ERROR (ROL_ParaboloidCircle, objective hessVec): "
                                                                   "Input vector v must be of length 2."); 

      n = hvp->size();
      TEUCHOS_TEST_FOR_EXCEPTION( (n != 2), std::invalid_argument, ">>> ERROR (ROL_ParaboloidCircle, objective hessVec): "
                                                                   "Output vector hv must be of length 2."); 

      Real v1 = (*vp)[0];
      Real v2 = (*vp)[1];

      Real two(2);

      (*hvp)[0] = two*v1;
      (*hvp)[1] = two*v2;
    }

  };


  /** \brief Equality constraint c(x,y) = (x-2)^2 + y^2 - 1.
   */
  template<class Real, class XPrim=StdVector<Real>, class XDual=StdVector<Real>, class CPrim=StdVector<Real>, class CDual=StdVector<Real> >
  class EqualityConstraint_ParaboloidCircle : public EqualityConstraint<Real> {

    typedef std::vector<Real> vector;
    typedef Vector<Real>      V;

    typedef typename vector::size_type uint;

  private:
    template<class VectorType>
    Teuchos::RCP<const vector> getVector( const V& x ) {
      using Teuchos::dyn_cast;
      return dyn_cast<const VectorType>(x).getVector();
    }

    template<class VectorType> 
    Teuchos::RCP<vector> getVector( V& x ) {
      using Teuchos::dyn_cast;
      return dyn_cast<VectorType>(x).getVector(); 
    }

  public:
    EqualityConstraint_ParaboloidCircle() {}

    void value( Vector<Real> &c, const Vector<Real> &x, Real &tol ) {

      using Teuchos::RCP;
      RCP<const vector> xp = getVector<XPrim>(x);
      RCP<vector> cp = getVector<CPrim>(c);

      uint n = xp->size();
      TEUCHOS_TEST_FOR_EXCEPTION( (n != 2), std::invalid_argument, ">>> ERROR (ROL_ParaboloidCircle, constraint value): "
                                                                   "Primal vector x must be of length 2.");

      uint m = cp->size();
      TEUCHOS_TEST_FOR_EXCEPTION( (m != 1), std::invalid_argument, ">>> ERROR (ROL_ParaboloidCircle, constraint value): "
                                                                   "Constraint vector c must be of length 1.");

      Real x1 = (*xp)[0];
      Real x2 = (*xp)[1];
 
      Real one(1), two(2);

      (*cp)[0] = (x1-two)*(x1-two) + x2*x2 - one;
    }
  
    void applyJacobian( Vector<Real> &jv, const Vector<Real> &v, const Vector<Real> &x, Real &tol ) {

      using Teuchos::RCP;
      RCP<const vector> xp = getVector<XPrim>(x);
      RCP<const vector> vp = getVector<XPrim>(v);
      RCP<vector> jvp = getVector<CPrim>(jv);

      uint n = xp->size();
      TEUCHOS_TEST_FOR_EXCEPTION( (n != 2), std::invalid_argument, ">>> ERROR (ROL_ParaboloidCircle, constraint applyJacobian): "
                                                                   "Primal vector x must be of length 2.");

      uint d = vp->size();
      TEUCHOS_TEST_FOR_EXCEPTION( (d != 2), std::invalid_argument, ">>> ERROR (ROL_ParaboloidCircle, constraint applyJacobian): "
                                                                   "Input vector v must be of length 2.");
      d = jvp->size();
      TEUCHOS_TEST_FOR_EXCEPTION( (d != 1), std::invalid_argument, ">>> ERROR (ROL_ParaboloidCircle, constraint applyJacobian): "
                                                                   "Output vector jv must be of length 1.");
      
      Real x1 = (*xp)[0];
      Real x2 = (*xp)[1];

      Real v1 = (*vp)[0];
      Real v2 = (*vp)[1];

      Real two(2);

      (*jvp)[0] = two*(x1-two)*v1 + two*x2*v2;
    } //applyJacobian

    void applyAdjointJacobian( Vector<Real> &ajv, const Vector<Real> &v, const Vector<Real> &x, Real &tol ) {

      using Teuchos::RCP;
      RCP<const vector> xp = getVector<XPrim>(x);
      RCP<const vector> vp = getVector<CDual>(v);
      RCP<vector> ajvp = getVector<XDual>(ajv);

      uint n = xp->size();
      TEUCHOS_TEST_FOR_EXCEPTION( (n != 2), std::invalid_argument, ">>> ERROR (ROL_ParaboloidCircle, constraint applyAdjointJacobian): "
                                                                   "Primal vector x must be of length 2.");

      uint d = vp->size();
      TEUCHOS_TEST_FOR_EXCEPTION( (d != 1), std::invalid_argument, ">>> ERROR (ROL_ParaboloidCircle, constraint applyAdjointJacobian): "
                                                                   "Input vector v must be of length 1.");

      d = ajvp->size();
      TEUCHOS_TEST_FOR_EXCEPTION( (d != 2), std::invalid_argument, ">>> ERROR (ROL_ParaboloidCircle, constraint applyAdjointJacobian): "
                                                                   "Output vector ajv must be of length 2.");
      
      Real x1 = (*xp)[0];
      Real x2 = (*xp)[1];

      Real v1 = (*vp)[0];

      Real two(2);

      (*ajvp)[0] = two*(x1-two)*v1;
      (*ajvp)[1] = two*x2*v1;

    } //applyAdjointJacobian

    void applyAdjointHessian( Vector<Real> &ahuv, const Vector<Real> &u, const Vector<Real> &v, const Vector<Real> &x, Real &tol ) {
      
      bool useFD = true;

      if (useFD) {
        EqualityConstraint<Real>::applyAdjointHessian( ahuv, u, v, x, tol );
      }
      else {
        using Teuchos::RCP;
        RCP<const vector> xp = getVector<XPrim>(x);
        RCP<const vector> up = getVector<CDual>(u);
        RCP<const vector> vp = getVector<XPrim>(v);
        RCP<vector> ahuvp = getVector<XDual>(ahuv);

        uint n = xp->size();
        TEUCHOS_TEST_FOR_EXCEPTION( (n != 2), std::invalid_argument, ">>> ERROR (ROL_ParaboloidCircle, constraint applyAdjointHessian): "
                                                                     "Primal vector x must be of length 2.");

        n = vp->size();
        TEUCHOS_TEST_FOR_EXCEPTION( (n != 2), std::invalid_argument, ">>> ERROR (ROL_ParaboloidCircle, constraint applyAdjointHessian): "
                                                                     "Direction vector v must be of length 2.");

        n = ahuvp->size();
        TEUCHOS_TEST_FOR_EXCEPTION( (n != 2), std::invalid_argument, ">>> ERROR (ROL_ParaboloidCircle, constraint applyAdjointHessian): "
                                                                     "Output vector ahuv must be of length 2.");
        uint d = up->size();
        TEUCHOS_TEST_FOR_EXCEPTION( (d != 1), std::invalid_argument, ">>> ERROR (ROL_ParaboloidCircle, constraint applyAdjointHessian): "
                                                                     "Dual constraint vector u must be of length 1.");
        
        Real v1 = (*vp)[0];
        Real v2 = (*vp)[1];

        Real u1 = (*up)[0];

        Real two(2);

        (*ahuvp)[0] = two*u1*v1;
        (*ahuvp)[1] = two*u1*v2;
      }
    } //applyAdjointHessian

  };


  template<class Real, class XPrim, class XDual, class CPrim, class CDual>
  void getParaboloidCircle( Teuchos::RCP<Objective<Real> > &obj,
                            Teuchos::RCP<EqualityConstraint<Real> > &constr,
                            Vector<Real> &x0,
                            Vector<Real> &sol ) {

    typedef std::vector<Real> vector;
    
    typedef typename vector::size_type uint;

    using Teuchos::RCP;       using Teuchos::rcp;
    using Teuchos::dyn_cast; 

    // Cast initial guess and solution vectors.
    RCP<vector> x0p  = dyn_cast<XPrim>(x0).getVector(); 
    RCP<vector> solp = dyn_cast<XPrim>(sol).getVector();

    uint n = 2;

    // Resize vectors.
    x0p->resize(n);
    solp->resize(n);
    // Instantiate objective function.
    obj = Teuchos::rcp( new Objective_ParaboloidCircle<Real, XPrim, XDual> );
    // Instantiate constraints.
    constr = Teuchos::rcp( new EqualityConstraint_ParaboloidCircle<Real, XPrim, XDual, CPrim, CDual> );
    // later we will bundle equality constraints into constraints ...
    //std::vector<Teuchos::RCP<EqualityConstraint<Real> > > eqc( 1, Teuchos::rcp( new EqualityConstraint_ParaboloidCircle<Real> ) );
    //constr = Teuchos::rcp( new Constraints<Real>(eqc) );

    // Get initial guess.
    Real zero(0), one(1);
    (*x0p)[0] = static_cast<Real>(rand())/static_cast<Real>(RAND_MAX);
    (*x0p)[1] = static_cast<Real>(rand())/static_cast<Real>(RAND_MAX);
    // Get solution.
    (*solp)[0] = one;
    (*solp)[1] = zero;
  }

} // End ZOO Namespace
} // End ROL Namespace

#endif
