/*
 *  This code is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This code is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this code; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* Copyright (C) 2020-2022 Max-Planck-Society
   Author: Martin Reinecke */

#include "ducc0/math/gridding_kernel.h"

namespace ducc0 {

namespace detail_gridding_kernel {

using namespace std;

vector<double> getCoeffs(size_t W, size_t D, const function<double(double)> &func)
  {
  vector<double> coeff(W*(D+1));
  vector<double> chebroot(D+1);
  for (size_t i=0; i<=D; ++i)
    chebroot[i] = cos((2*i+1.)*pi/(2*D+2));
  vector<double> y(D+1), lcf(D+1), C((D+1)*(D+1)), lcf2(D+1);
  for (size_t i=0; i<W; ++i)
    {
    double l = -1+2.*i/double(W);
    double r = -1+2.*(i+1)/double(W);
    // function values at Chebyshev nodes
    double avg = 0;
    for (size_t j=0; j<=D; ++j)
      {
      y[j] = func(chebroot[j]*(r-l)*0.5 + (r+l)*0.5);
      avg += y[j];
      }
    avg/=(D+1);
    for (size_t j=0; j<=D; ++j)
      y[j] -= avg;
    // Chebyshev coefficients
    for (size_t j=0; j<=D; ++j)
      {
      lcf[j] = 0;
      for (size_t k=0; k<=D; ++k)
        lcf[j] += 2./(D+1)*y[k]*cos(j*(2*k+1)*pi/(2*D+2));
      }
    lcf[0] *= 0.5;
    // Polynomial coefficients
    fill(C.begin(), C.end(), 0.);
    C[0] = 1.;
    C[1*(D+1) + 1] = 1.;
    for (size_t j=2; j<=D; ++j)
      {
      C[j*(D+1) + 0] = -C[(j-2)*(D+1) + 0];
      for (size_t k=1; k<=j; ++k)
        C[j*(D+1) + k] = 2*C[(j-1)*(D+1) + k-1] - C[(j-2)*(D+1) + k];
      }
    for (size_t j=0; j<=D; ++j) lcf2[j] = 0;
    for (size_t j=0; j<=D; ++j)
      for (size_t k=0; k<=D; ++k)
        lcf2[k] += C[j*(D+1) + k]*lcf[j];
    lcf2[0] += avg;
    for (size_t j=0; j<=D; ++j)
      coeff[j*W + i] = lcf2[D-j];
    }
  return coeff;
  }

const vector<KernelParams> KernelDB {
{ 4, 1.20,   0.013816362, 1.30709338717990464, 0.588199361165364421, 1,  true},
{ 4, 1.25,  0.0085879935, 1.33188061184353357, 0.594018046061197902, 1,  true},
{ 4, 1.30,  0.0057357209, 1.36280729505750875, 0.596257019042968683, 1,  true},
{ 4, 1.35,   0.004250634, 1.38681869506835920, 0.598430379231771004, 1,  true},
{ 4, 1.40,  0.0033433175, 1.42907129923502607, 0.595376586914062478, 1,  true},
{ 4, 1.45,  0.0028117363, 1.48220782809787344, 0.588401285807291785, 1,  true},
{ 4, 1.50,  0.0023847392, 1.55200220743815076, 0.577678934733072857, 1,  true},
{ 4, 1.55,  0.0020357655, 1.61562355889214393, 0.568545023600260491, 1,  true},
{ 4, 1.60,  0.0017140276, 1.66557473076714380, 0.562941996256510202, 1,  true},
{ 4, 1.65,  0.0014739182, 1.70882788764105853, 0.558200836181640403, 1,  true},
{ 4, 1.70,   0.001256288, 1.75829043918185701, 0.552664438883463616, 1,  true},
{ 4, 1.75,  0.0010615833, 1.78472069634331620, 0.551698048909505245, 1,  true},
{ 4, 1.80, 0.00090960415, 1.81566416422526022, 0.549645741780599106, 1,  true},
{ 4, 1.85, 0.00077764143, 1.82287784152560750, 0.551604970296223796, 1,  true},
{ 4, 1.90,   0.000674396, 1.84992726643880223, 0.550009918212890803, 1,  true},
{ 4, 1.95, 0.00058999304, 1.88333621554904518, 0.547602589925130112, 1,  true},
{ 4, 2.00, 0.00051953173, 1.90694291856553799, 0.546800994873046786, 1,  true},
{ 4, 2.05, 0.00047169452, 1.92870330810546897, 0.545942433675130068, 1,  true},
{ 4, 2.10, 0.00043035853, 1.94683447943793375, 0.545642852783203014, 1,  true},
{ 4, 2.15, 0.00039993456, 1.95983327229817705, 0.545700836181640558, 1,  true},
{ 4, 2.20, 0.00036768497, 1.97344377305772567, 0.546037038167317745, 1,  true},
{ 4, 2.25, 0.00034392926, 1.98338402642144063, 0.546487172444661384, 1,  true},
{ 4, 2.30, 0.00032273579, 1.99225379096137201, 0.547050221761067768, 1,  true},
{ 4, 2.35, 0.00030386782, 2.00018734402126697, 0.547697194417317679, 1,  true},
{ 4, 2.40, 0.00029032885, 2.00592770046658009, 0.548258209228515314, 1,  true},
{ 4, 2.45, 0.00027519289, 2.01244015163845535, 0.548992156982421986, 1,  true},
{ 4, 2.50, 0.00026442815, 2.01718461778428892, 0.549589284261067679, 1,  true},
{ 5, 1.20,  0.0045447572, 1.47271143595377563, 0.547661844889323124, 1,  true},
{ 5, 1.25,   0.002565032, 1.51477381388346322, 0.547609456380208193, 1,  true},
{ 5, 1.30,  0.0014950085, 1.56042251586914049, 0.546399434407552009, 1,  true},
{ 5, 1.35, 0.00093190915, 1.59968931410047732, 0.545594278971354263, 1,  true},
{ 5, 1.40, 0.00058294697, 1.62376225789388040, 0.547126770019531361, 1,  true},
{ 5, 1.45,  0.0004191174, 1.67027257283528652, 0.544658406575520782, 1,  true},
{ 5, 1.50, 0.00032211632, 1.71065928141276036, 0.543058522542317568, 1,  true},
{ 5, 1.55, 0.00025889559, 1.74115142822265612, 0.542447662353515714, 1,  true},
{ 5, 1.60, 0.00021176206, 1.76781667073567705, 0.542211151123046964, 1,  true},
{ 5, 1.65, 0.00018155438, 1.80731658935546857, 0.539609527587890825, 1,  true},
{ 5, 1.70, 0.00015202986, 1.83974456787109331, 0.537987009684244954, 1,  true},
{ 5, 1.75, 0.00012394032, 1.87821875678168393, 0.535735321044921831, 1,  true},
{ 5, 1.80, 0.00010185554, 1.92916208902994835, 0.531913503011067768, 1,  true},
{ 5, 1.85, 8.6084819e-05, 1.97668134901258674, 0.528661346435546742, 1,  true},
{ 5, 1.90, 7.5510239e-05, 2.01164194742838554, 0.526721445719401160, 1,  true},
{ 5, 1.95, 6.6235913e-05, 2.04016469319661509, 0.525634002685546831, 1,  true},
{ 5, 2.00, 5.8045964e-05, 2.06404741075303821, 0.525147755940755356, 1,  true},
{ 5, 2.05, 5.1833314e-05, 2.08155195448133679, 0.525099436442057366, 1,  true},
{ 5, 2.10, 4.6296138e-05, 2.09675140380859348, 0.525319671630859353, 1,  true},
{ 5, 2.15, 4.2186584e-05, 2.10787777370876706, 0.525670623779296853, 1,  true},
{ 5, 2.20, 3.7799771e-05, 2.11950395372178768, 0.526264699300130090, 1,  true},
{ 5, 2.25, 3.4593093e-05, 2.12779930962456554, 0.526887766520182321, 1,  true},
{ 5, 2.30, 3.1764919e-05, 2.13487870958116410, 0.527610015869140359, 1,  true},
{ 5, 2.35, 2.9305975e-05, 2.14067128499348991, 0.528427378336588482, 1,  true},
{ 5, 2.40, 2.7603323e-05, 2.14426320393880232, 0.529153188069661606, 1,  true},
{ 5, 2.45,   2.58328e-05, 2.14864959716796866, 0.530002593994140625, 1,  true},
{ 5, 2.50,  2.461415e-05, 2.15946265326605946, 0.529833730061848529, 1,  true},
{ 6, 1.20, 0.00087716416, 1.48702875773111987, 0.543441772460937278, 1,  true},
{ 6, 1.25, 0.00052617233, 1.55846243964301179, 0.538631693522135246, 1,  true},
{ 6, 1.30, 0.00030990674, 1.62978795369466201, 0.533901977539062544, 1,  true},
{ 6, 1.35, 0.00018685304, 1.67900339762369777, 0.531911468505859464, 1,  true},
{ 6, 1.40, 0.00011048179, 1.72446373833550348, 0.530658213297526271, 1,  true},
{ 6, 1.45, 7.5272528e-05, 1.77476925320095469, 0.528379058837890714, 1,  true},
{ 6, 1.50, 5.4519899e-05, 1.81697709825303821, 0.527150217692057388, 1,  true},
{ 6, 1.55, 4.1327954e-05, 1.84778255886501697, 0.526968638102213394, 1,  true},
{ 6, 1.60, 3.1940841e-05, 1.87210167778862813, 0.527448781331380179, 1,  true},
{ 6, 1.65, 2.6524553e-05, 1.89124738905164880, 0.527779897054036606, 1,  true},
{ 6, 1.70, 2.1912792e-05, 1.92978871663411455, 0.525992584228515692, 1,  true},
{ 6, 1.75,  1.752611e-05, 1.96848670111762192, 0.524261728922526204, 1,  true},
{ 6, 1.80, 1.3983041e-05, 1.99691992865668366, 0.523411814371744888, 1,  true},
{ 6, 1.85, 1.0909295e-05, 2.02382388644748223, 0.522790781656901182, 1,  true},
{ 6, 1.90, 9.1159743e-06, 2.06700524224175330, 0.520410919189453147, 1,  true},
{ 6, 1.95, 7.9473035e-06, 2.08590850830078134, 0.520696258544921631, 1,  true},
{ 6, 2.00, 6.9082129e-06, 2.10745629204644125, 0.520621490478515692, 1,  true},
{ 6, 2.05, 6.0489563e-06, 2.13673621283637205, 0.519232940673827970, 1,  true},
{ 6, 2.10, 5.2599096e-06, 2.16241302490234455, 0.518203481038411518, 1,  true},
{ 6, 2.15, 4.7022734e-06, 2.17432640923394116, 0.518426767985025960, 1,  true},
{ 6, 2.20, 4.1154589e-06, 2.18673960367838571, 0.518857574462890714, 1,  true},
{ 6, 2.25, 3.6872152e-06, 2.19571753607855902, 0.519317881266276093, 1,  true},
{ 6, 2.30, 3.3135316e-06, 2.20337880452473955, 0.519864654541015603, 1,  true},
{ 6, 2.35, 2.9949297e-06, 2.20968983968098920, 0.520482126871744888, 1,  true},
{ 6, 2.40,  2.778763e-06, 2.21353166368272536, 0.521029408772786695, 1,  true},
{ 6, 2.45, 2.5517287e-06, 2.23052520751953054, 0.520584869384765669, 1,  true},
{ 6, 2.50, 2.3518687e-06, 2.25309499104817679, 0.519467417399088638, 1,  true},
{ 7, 1.20, 0.00027656574, 1.57363950941297781, 0.528833516438802165, 1,  true},
{ 7, 1.25, 0.00012917467, 1.62363544040256103, 0.528018188476562345, 1,  true},
{ 7, 1.30, 7.2382792e-05, 1.68306528727213545, 0.525804901123047119, 1,  true},
{ 7, 1.35,  4.356952e-05, 1.73462778727213518, 0.523948923746745066, 1,  true},
{ 7, 1.40, 2.4981206e-05, 1.78285200330946147, 0.522594960530598951, 1,  true},
{ 7, 1.45, 1.4834317e-05, 1.81916283501519116, 0.522077687581380068, 1,  true},
{ 7, 1.50, 9.8593424e-06, 1.87820756700303804, 0.519385019938151005, 1,  true},
{ 7, 1.55, 7.1596313e-06, 1.91922556559244817, 0.518272145589192657, 1,  true},
{ 7, 1.60, 5.2459533e-06, 1.95363786485460023, 0.517866770426432477, 1,  true},
{ 7, 1.65, 4.0705255e-06, 1.97871042887369786, 0.517832692464192723, 1,  true},
{ 7, 1.70, 3.1711049e-06, 2.00276845296223982, 0.517857615152994666, 1,  true},
{ 7, 1.75, 2.4800325e-06, 2.02892269558376803, 0.517637888590494311, 1,  true},
{ 7, 1.80, 1.9970539e-06, 2.06023915608723929, 0.516692860921223907, 1,  true},
{ 7, 1.85, 1.6496563e-06, 2.08682607014974009, 0.516387685139973907, 1,  true},
{ 7, 1.90, 1.4148184e-06, 2.09541609022352526, 0.517656707763671786, 1,  true},
{ 7, 1.95,  1.226067e-06, 2.10332353379991410, 0.518966929117838260, 1,  true},
{ 7, 2.00, 1.0736112e-06, 2.13156653510199590, 0.518265024820963704, 1,  true},
{ 7, 2.05,  9.562875e-07, 2.17690378824869812, 0.515382639567056988, 1,  true},
{ 7, 2.10, 8.4064676e-07, 2.22028656005859348, 0.512581634521484086, 1,  true},
{ 7, 2.15, 7.5439059e-07, 2.23297949896918446, 0.512734731038411051, 1,  true},
{ 7, 2.20,  6.667494e-07, 2.24713083902994715, 0.512998199462890270, 1,  true},
{ 7, 2.25, 6.0166015e-07, 2.25832807752821108, 0.513249460856119444, 1,  true},
{ 7, 2.30, 5.4448389e-07, 2.26957007514105769, 0.513472747802734553, 1,  true},
{ 7, 2.35, 4.9472313e-07, 2.28155059814453098, 0.513612620035807321, 1,  true},
{ 7, 2.40, 4.5845311e-07, 2.29392649332682330, 0.513477325439453236, 1,  true},
{ 7, 2.45, 4.2218381e-07, 2.30428076850043340, 0.513808949788411296, 1,  true},
{ 7, 2.50, 3.9604759e-07, 2.31114756266275956, 0.514217376708984375, 1,  true},
{ 8, 1.20, 8.6642191e-05, 1.62311698065863697, 0.521696472167968661, 1,  true},
{ 8, 1.25, 3.2467478e-05, 1.68515777587890647, 0.519925181070963416, 1,  true},
{ 8, 1.30, 1.6610246e-05, 1.74423607720269080, 0.518215688069661451, 1,  true},
{ 8, 1.35,  9.778419e-06, 1.78767479790581585, 0.517632293701171697, 1,  true},
{ 8, 1.40, 5.9162971e-06, 1.82943505181206589, 0.517185719807942479, 1,  true},
{ 8, 1.45, 3.8721512e-06, 1.87169511583116299, 0.516173044840494644, 1,  true},
{ 8, 1.50, 2.6605377e-06, 1.92122107611762094, 0.514544423421223862, 1,  true},
{ 8, 1.55, 1.9558123e-06, 1.96090749104817696, 0.513719431559244599, 1,  true},
{ 8, 1.60, 1.4846871e-06, 1.99985165066189197, 0.512932586669921808, 1,  true},
{ 8, 1.65, 1.1859573e-06, 2.02751278347439223, 0.512713877360025716, 1,  true},
{ 8, 1.70, 9.6331998e-07, 2.05249209933810839, 0.512742869059244488, 1,  true},
{ 8, 1.75, 7.9697968e-07, 2.07719167073567679, 0.512768809000650916, 1,  true},
{ 8, 1.80, 6.7710041e-07, 2.10146230061848982, 0.512584177652994466, 1,  true},
{ 8, 1.85, 5.8369677e-07, 2.12952253553602455, 0.512125905354817501, 1,  true},
{ 8, 1.90, 5.1917955e-07, 2.14036543104383759, 0.513058217366536229, 1,  true},
{ 8, 1.95, 4.6639866e-07, 2.14727325439453187, 0.514326731363932144, 1,  true},
{ 8, 2.00, 4.2301959e-07, 2.15810869004991313, 0.515234629313151005, 1,  true},
{ 8, 2.05, 3.9184676e-07, 2.20948469373914858, 0.512186431884765647, 1,  true},
{ 8, 2.10, 3.6266925e-07, 2.23797387017144178, 0.511113230387369444, 1,  true},
{ 8, 2.15, 3.3814129e-07, 2.26647796630859455, 0.509743499755859086, 1,  true},
{ 8, 2.20, 3.1640043e-07, 2.28009219699435706, 0.510066986083984042, 1,  true},
{ 8, 2.25, 2.9859388e-07, 2.29611223008897580, 0.509888966878255245, 1,  true},
{ 8, 2.30, 2.8275659e-07, 2.30705210367838509, 0.510179901123046564, 1,  true},
{ 8, 2.35,  2.685671e-07, 2.32135637071397483, 0.510184478759765581, 1,  true},
{ 8, 2.40, 2.5685046e-07, 2.33250885009765652, 0.510236358642577992, 1,  true},
{ 8, 2.45, 2.4679219e-07, 2.34000973171658000, 0.510860951741536340, 1,  true},
{ 8, 2.50, 2.3834227e-07, 2.33702579074435679, 0.512065887451171986, 1,  true},
{ 4, 1.20,   0.027638471, 1.30709338717990464, 0.588199361165364421, 2,  true},
{ 4, 1.25,   0.017179163, 1.33188061184353357, 0.594018046061197902, 2,  true},
{ 4, 1.30,   0.011473319, 1.36280729505750875, 0.596257019042968683, 2,  true},
{ 4, 1.35,  0.0085024842, 1.38681869506835920, 0.598430379231771004, 2,  true},
{ 4, 1.40,  0.0066874095, 1.42907129923502607, 0.595376586914062478, 2,  true},
{ 4, 1.45,  0.0056239728, 1.48220782809787344, 0.588401285807291785, 2,  true},
{ 4, 1.50,  0.0047698072, 1.55200220743815076, 0.577678934733072857, 2,  true},
{ 4, 1.55,  0.0040717361, 1.61562355889214393, 0.568545023600260491, 2,  true},
{ 4, 1.60,  0.0034281853, 1.66557473076714380, 0.562941996256510202, 2,  true},
{ 4, 1.65,  0.0029479114, 1.70882788764105853, 0.558200836181640403, 2,  true},
{ 4, 1.70,   0.002512611, 1.75829043918185701, 0.552664438883463616, 2,  true},
{ 4, 1.75,  0.0021231746, 1.78472069634331620, 0.551698048909505245, 2,  true},
{ 4, 1.80,   0.001819194, 1.81566416422526022, 0.549645741780599106, 2,  true},
{ 4, 1.85,  0.0015552516, 1.82287784152560750, 0.551604970296223796, 2,  true},
{ 4, 1.90,  0.0013487471, 1.84992726643880223, 0.550009918212890803, 2,  true},
{ 4, 1.95,  0.0011799293, 1.88333621554904518, 0.547602589925130112, 2,  true},
{ 4, 2.00,   0.001038997, 1.90694291856553799, 0.546800994873046786, 2,  true},
{ 4, 2.05, 0.00094331549, 1.92870330810546897, 0.545942433675130068, 2,  true},
{ 4, 2.10, 0.00086063741, 1.94683447943793375, 0.545642852783203014, 2,  true},
{ 4, 2.15, 0.00079978431, 1.95983327229817705, 0.545700836181640558, 2,  true},
{ 4, 2.20,  0.0007352812, 1.97344377305772567, 0.546037038167317745, 2,  true},
{ 4, 2.25, 0.00068776638, 1.98338402642144063, 0.546487172444661384, 2,  true},
{ 4, 2.30, 0.00064537642, 1.99225379096137201, 0.547050221761067768, 2,  true},
{ 4, 2.35, 0.00060763777, 2.00018734402126697, 0.547697194417317679, 2,  true},
{ 4, 2.40, 0.00058055746, 2.00592770046658009, 0.548258209228515314, 2,  true},
{ 4, 2.45, 0.00055028377, 2.01244015163845535, 0.548992156982421986, 2,  true},
{ 4, 2.50,  0.0005287527, 2.01718461778428892, 0.549589284261067679, 2,  true},
{ 5, 1.20,  0.0091081213, 1.47271143595377563, 0.547661844889323124, 2,  true},
{ 5, 1.25,  0.0051393036, 1.51477381388346322, 0.547609456380208193, 2,  true},
{ 5, 1.30,  0.0029950485, 1.56042251586914049, 0.546399434407552009, 2,  true},
{ 5, 1.35,  0.0018668854, 1.59968931410047732, 0.545594278971354263, 2,  true},
{ 5, 1.40,  0.0011678541, 1.62376225789388040, 0.547126770019531361, 2,  true},
{ 5, 1.45, 0.00083949653, 1.67027257283528652, 0.544658406575520782, 2,  true},
{ 5, 1.50, 0.00064510114, 1.71065928141276036, 0.543058522542317568, 2,  true},
{ 5, 1.55, 0.00051838121, 1.74115142822265612, 0.542447662353515714, 2,  true},
{ 5, 1.60, 0.00042394497, 1.76781667073567705, 0.542211151123046964, 2,  true},
{ 5, 1.65, 0.00036339906, 1.80731658935546857, 0.539609527587890825, 2,  true},
{ 5, 1.70, 0.00030426196, 1.83974456787109331, 0.537987009684244954, 2,  true},
{ 5, 1.75, 0.00024801871, 1.87821875678168393, 0.535735321044921831, 2,  true},
{ 5, 1.80, 0.00020379638, 1.92916208902994835, 0.531913503011067768, 2,  true},
{ 5, 1.85, 0.00017221407, 1.97668134901258674, 0.528661346435546742, 2,  true},
{ 5, 1.90, 0.00015103689, 2.01164194742838554, 0.526721445719401160, 2,  true},
{ 5, 1.95, 0.00013246567, 2.04016469319661509, 0.525634002685546831, 2,  true},
{ 5, 2.00,  0.0001160673, 2.06404741075303821, 0.525147755940755356, 2,  true},
{ 5, 2.05, 0.00010362875, 2.08155195448133679, 0.525099436442057366, 2,  true},
{ 5, 2.10, 9.2543096e-05, 2.09675140380859348, 0.525319671630859353, 2,  true},
{ 5, 2.15,  8.431446e-05, 2.10787777370876706, 0.525670623779296853, 2,  true},
{ 5, 2.20, 7.5533655e-05, 2.11950395372178768, 0.526264699300130090, 2,  true},
{ 5, 2.25, 6.9114106e-05, 2.12779930962456554, 0.526887766520182321, 2,  true},
{ 5, 2.30, 6.3452285e-05, 2.13487870958116410, 0.527610015869140359, 2,  true},
{ 5, 2.35,  5.852956e-05, 2.14067128499348991, 0.528427378336588482, 2,  true},
{ 5, 2.40, 5.5120029e-05, 2.14426320393880232, 0.529153188069661606, 2,  true},
{ 5, 2.45, 5.1575852e-05, 2.14864959716796866, 0.530002593994140625, 2,  true},
{ 5, 2.50, 4.9135627e-05, 2.15946265326605946, 0.529833730061848529, 2,  true},
{ 6, 1.20,  0.0018267494, 1.48702875773111987, 0.543441772460937278, 2,  true},
{ 6, 1.25,  0.0010812568, 1.55846243964301179, 0.538631693522135246, 2,  true},
{ 6, 1.30, 0.00063328686, 1.62978795369466178, 0.533901977539062544, 2,  true},
{ 6, 1.35, 0.00038126795, 1.67900339762369777, 0.531911468505859464, 2,  true},
{ 6, 1.40, 0.00022540405, 1.72446373833550348, 0.530658213297526271, 2,  true},
{ 6, 1.45, 0.00015329233, 1.77476925320095469, 0.528379058837890714, 2,  true},
{ 6, 1.50, 0.00011089008, 1.81697709825303821, 0.527150217692057388, 2,  true},
{ 6, 1.55, 8.3901712e-05, 1.84778255886501697, 0.526968638102213394, 2,  true},
{ 6, 1.60, 6.4776566e-05, 1.87210167778862835, 0.527448781331380290, 2,  true},
{ 6, 1.65, 5.3697783e-05, 1.89141523573133630, 0.527762095133463505, 2,  true},
{ 6, 1.70, 4.4299041e-05, 1.93276146782769054, 0.525681813557942856, 2,  true},
{ 6, 1.75, 3.5391822e-05, 1.96848670111762192, 0.524261728922526204, 2,  true},
{ 6, 1.80, 2.8212779e-05, 1.99691992865668366, 0.523411814371744888, 2,  true},
{ 6, 1.85, 2.1992397e-05, 2.02382388644748223, 0.522790781656901182, 2,  true},
{ 6, 1.90, 1.8350135e-05, 2.06700524224175330, 0.520410919189453147, 2,  true},
{ 6, 1.95, 1.5972068e-05, 2.08590850830078134, 0.520696258544921631, 2,  true},
{ 6, 2.00, 1.3859999e-05, 2.10745629204644125, 0.520621490478515692, 2,  true},
{ 6, 2.05, 1.2116191e-05, 2.13673621283637205, 0.519232940673827970, 2,  true},
{ 6, 2.10, 1.0516966e-05, 2.16247270372178901, 0.518196360270182232, 2,  true},
{ 6, 2.15, 9.3851633e-06, 2.17432640923394116, 0.518426767985025960, 2,  true},
{ 6, 2.20, 8.1990803e-06, 2.18673960367838571, 0.518857574462890714, 2,  true},
{ 6, 2.25, 7.3319484e-06, 2.19571753607855902, 0.519317881266276093, 2,  true},
{ 6, 2.30, 6.5752637e-06, 2.20337880452473955, 0.519864654541015603, 2,  true},
{ 6, 2.35, 5.9298948e-06, 2.20968983968098920, 0.520482126871744888, 2,  true},
{ 6, 2.40, 5.4904875e-06, 2.21353166368272536, 0.521029408772786695, 2,  true},
{ 6, 2.45, 5.0308227e-06, 2.23052520751953054, 0.520584869384765669, 2,  true},
{ 6, 2.50,  4.625964e-06, 2.25309499104817679, 0.519467417399088638, 2,  true},
{ 7, 1.20,  0.0007571742, 1.57580659654405375, 0.528578694661458459, 2,  true},
{ 7, 1.25,  0.0003367945, 1.63487184312608536, 0.526731363932291696, 2,  true},
{ 7, 1.30, 0.00017840443, 1.68659379747178795, 0.525407663981120043, 2,  true},
{ 7, 1.35, 0.00010456205, 1.73841366238064232, 0.523535919189453192, 2,  true},
{ 7, 1.40, 5.9581163e-05, 1.78485124376085036, 0.522390492757161540, 2,  true},
{ 7, 1.45, 3.5499217e-05, 1.81957685682508630, 0.522037506103515625, 2,  true},
{ 7, 1.50,  2.341159e-05, 1.87820756700303804, 0.519385019938150894, 2,  true},
{ 7, 1.55, 1.6709902e-05, 1.91922556559244817, 0.518272145589192657, 2,  true},
{ 7, 1.60, 1.2162659e-05, 1.95363786485460023, 0.517866770426432477, 2,  true},
{ 7, 1.65, 9.3333251e-06, 1.97871042887369786, 0.517832692464192723, 2,  true},
{ 7, 1.70, 7.2180318e-06, 2.00276845296223982, 0.517857615152994666, 2,  true},
{ 7, 1.75, 5.6180048e-06, 2.02892269558376759, 0.517637888590494311, 2,  true},
{ 7, 1.80, 4.4823027e-06, 2.06023915608723929, 0.516692860921223907, 2,  true},
{ 7, 1.85, 3.6615236e-06, 2.09614715576171839, 0.515417734781901116, 2,  true},
{ 7, 1.90, 3.1044253e-06, 2.09791141086154465, 0.517407989501953436, 2,  true},
{ 7, 1.95, 2.6579785e-06, 2.11293928358289929, 0.518028004964192657, 2,  true},
{ 7, 2.00,  2.292283e-06, 2.13445349799262107, 0.517974090576171942, 2,  true},
{ 7, 2.05, 2.0048563e-06, 2.20160708957248241, 0.512737274169921653, 2,  true},
{ 7, 2.10, 1.7431904e-06, 2.22028656005859348, 0.512581634521484086, 2,  true},
{ 7, 2.15, 1.5434754e-06, 2.23297949896918402, 0.512734731038411162, 2,  true},
{ 7, 2.20, 1.3478827e-06, 2.24713083902994715, 0.512998199462890270, 2,  true},
{ 7, 2.25, 1.2005263e-06, 2.25832807752821108, 0.513249460856119444, 2,  true},
{ 7, 2.30, 1.0711768e-06, 2.26957007514105769, 0.513472747802734553, 2,  true},
{ 7, 2.35,  9.585121e-07, 2.28155059814453098, 0.513612620035807321, 2,  true},
{ 7, 2.40, 8.7454818e-07, 2.29392649332682330, 0.513477325439453236, 2,  true},
{ 7, 2.45, 7.9380344e-07, 2.30712670220269001, 0.513562774658203103, 2,  true},
{ 7, 2.50, 7.3427447e-07, 2.31122216118706580, 0.514211273193359131, 2,  true},
{ 8, 1.20, 0.00073262844, 1.66550945705837661, 0.516498819986978996, 2,  true},
{ 8, 1.25, 0.00026113174, 1.71241234673394072, 0.516673533121744799, 2,  true},
{ 8, 1.30,  0.0001097567, 1.74426591661241281, 0.518212127685547030, 2,  true},
{ 8, 1.35, 5.7327797e-05, 1.78767852783203129, 0.517631785074869954, 2,  true},
{ 8, 1.40, 3.1679034e-05, 1.83155364990234371, 0.516968027750650871, 2,  true},
{ 8, 1.45, 1.9085793e-05, 1.87205691867404500, 0.516137440999349106, 2,  true},
{ 8, 1.50, 1.2356395e-05, 1.92130686442057286, 0.514534759521484197, 2,  true},
{ 8, 1.55, 8.3078466e-06, 1.96372358534071156, 0.513400522867838327, 2,  true},
{ 8, 1.60, 5.9643915e-06, 2.00066104465060768, 0.512841542561849106, 2,  true},
{ 8, 1.65, 4.4710511e-06, 2.02756873236762081, 0.512707773844401027, 2,  true},
{ 8, 1.70, 3.4523033e-06, 2.05249209933810794, 0.512742869059244599, 2,  true},
{ 8, 1.75, 2.7373352e-06, 2.07740800645616330, 0.512746938069661429, 2,  true},
{ 8, 1.80, 2.2109254e-06, 2.10268198649088545, 0.512466684977213371, 2,  true},
{ 8, 1.85, 1.8013756e-06, 2.13499806722005214, 0.511604054768880157, 2,  true},
{ 8, 1.90, 1.5252763e-06, 2.16502397325303830, 0.510731252034505090, 2,  true},
{ 8, 1.95, 1.3043269e-06, 2.19154747856987875, 0.510195668538411362, 2,  true},
{ 8, 2.00, 1.1225258e-06, 2.21735483805338607, 0.509709421793619666, 2,  true},
{ 8, 2.05,  9.913048e-07, 2.23568742540147580, 0.509645334879556988, 2,  true},
{ 8, 2.10, 8.7847996e-07, 2.25366193983290009, 0.509593963623046542, 2,  true},
{ 8, 2.15,  7.863896e-07, 2.26725006103515625, 0.509669240315754779, 2,  true},
{ 8, 2.20, 7.1103723e-07, 2.28289710150824599, 0.509803517659505046, 2,  true},
{ 8, 2.25,  6.487745e-07, 2.29611223008897580, 0.509888966878255023, 2,  true},
{ 8, 2.30,  5.937697e-07, 2.31027103000217027, 0.509893544514973596, 2,  true},
{ 8, 2.35, 5.4509224e-07, 2.32824554443359366, 0.509586842854817479, 2,  true},
{ 8, 2.40, 5.0481753e-07, 2.34016638861762161, 0.509583791097005245, 2,  true},
{ 8, 2.45, 4.7251724e-07, 2.35166948106553786, 0.509892018636067701, 2,  true},
{ 8, 2.50, 4.4514883e-07, 2.35677575005425410, 0.510489654541015692, 2,  true},
{ 4, 1.20,    0.04150764, 1.30709338717990464, 0.588199361165364421, 3,  true},
{ 4, 1.25,   0.025790962, 1.33188061184353357, 0.594018046061197902, 3,  true},
{ 4, 1.30,   0.017221092, 1.36280729505750875, 0.596257019042968683, 3,  true},
{ 4, 1.35,   0.012760142, 1.38681869506835920, 0.598430379231771004, 3,  true},
{ 4, 1.40,   0.010034843, 1.42907129923502607, 0.595376586914062478, 3,  true},
{ 4, 1.45,  0.0084382415, 1.48220782809787344, 0.588401285807291785, 3,  true},
{ 4, 1.50,  0.0071561917, 1.55200220743815076, 0.577678934733072857, 3,  true},
{ 4, 1.55,  0.0061085655, 1.61562355889214393, 0.568545023600260491, 3,  true},
{ 4, 1.60,  0.0051429525, 1.66557473076714380, 0.562941996256510202, 3,  true},
{ 4, 1.65,  0.0044223467, 1.70882788764105853, 0.558200836181640403, 3,  true},
{ 4, 1.70,  0.0037692633, 1.75829043918185701, 0.552664438883463616, 3,  true},
{ 4, 1.75,  0.0031850242, 1.78472069634331620, 0.551698048909505245, 3,  true},
{ 4, 1.80,  0.0027289867, 1.81566416422526022, 0.549645741780599106, 3,  true},
{ 4, 1.85,  0.0023330248, 1.82287784152560750, 0.551604970296223796, 3,  true},
{ 4, 1.90,  0.0020232305, 1.84992726643880223, 0.550009918212890803, 3,  true},
{ 4, 1.95,  0.0017699727, 1.88333621554904518, 0.547602589925130112, 3,  true},
{ 4, 2.00,  0.0015585499, 1.90694291856553799, 0.546800994873046786, 3,  true},
{ 4, 2.05,    0.00141501, 1.92870330810546897, 0.545942433675130068, 3,  true},
{ 4, 2.10,  0.0012909783, 1.94683447943793375, 0.545642852783203014, 3,  true},
{ 4, 2.15,  0.0011996866, 1.95983327229817705, 0.545700836181640558, 3,  true},
{ 4, 2.20,   0.001102923, 1.97344377305772567, 0.546037038167317745, 3,  true},
{ 4, 2.25,  0.0010316431, 1.98338402642144063, 0.546487172444661384, 3,  true},
{ 4, 2.30, 0.00096805153, 1.99225379096137201, 0.547050221761067768, 3,  true},
{ 4, 2.35, 0.00091143774, 2.00018734402126697, 0.547697194417317679, 3,  true},
{ 4, 2.40, 0.00087081226, 2.00592770046658009, 0.548258209228515314, 3,  true},
{ 4, 2.45, 0.00082539805, 2.01244015163845535, 0.548992156982421986, 3,  true},
{ 4, 2.50, 0.00079309819, 2.01718461778428892, 0.549589284261067679, 3,  true},
{ 5, 1.20,   0.013925114, 1.47271143595377563, 0.547661844889323124, 3,  true},
{ 5, 1.25,  0.0078059267, 1.51477381388346322, 0.547609456380208193, 3,  true},
{ 5, 1.30,  0.0045340779, 1.56053441365559875, 0.546380106608072791, 3,  true},
{ 5, 1.35,   0.002821534, 1.59968931410047732, 0.545594278971354263, 3,  true},
{ 5, 1.40,  0.0017635314, 1.62376225789388018, 0.547126770019531250, 3,  true},
{ 5, 1.45,  0.0012659595, 1.67027257283528652, 0.544658406575520782, 3,  true},
{ 5, 1.50, 0.00097191815, 1.71065928141276036, 0.543058522542317568, 3,  true},
{ 5, 1.55, 0.00078030659, 1.74115142822265612, 0.542447662353515714, 3,  true},
{ 5, 1.60, 0.00063781753, 1.76781667073567705, 0.542211151123046964, 3,  true},
{ 5, 1.65, 0.00054641212, 1.80731658935546857, 0.539609527587890825, 3,  true},
{ 5, 1.70,  0.0004573431, 1.83974456787109331, 0.537987009684244954, 3,  true},
{ 5, 1.75, 0.00037273199, 1.87821875678168393, 0.535735321044921831, 3,  true},
{ 5, 1.80,  0.0003062092, 1.92916208902994835, 0.531913503011067768, 3,  true},
{ 5, 1.85, 0.00025869826, 1.97668134901258674, 0.528661346435546742, 3,  true},
{ 5, 1.90, 0.00022684339, 2.01164194742838554, 0.526721445719401160, 3,  true},
{ 5, 1.95, 0.00019891815, 2.04016469319661509, 0.525634002685546831, 3,  true},
{ 5, 2.00, 0.00017426706, 2.06404741075303821, 0.525147755940755356, 3,  true},
{ 5, 2.05, 0.00015557231, 2.08155195448133679, 0.525099436442057366, 3,  true},
{ 5, 2.10, 0.00013891336, 2.09675140380859348, 0.525319671630859353, 3,  true},
{ 5, 2.15, 0.00012654554, 2.10787777370876706, 0.525670623779296853, 3,  true},
{ 5, 2.20, 0.00011335614, 2.11950395372178768, 0.526264699300130090, 3,  true},
{ 5, 2.25,  0.0001037115, 2.12779930962456554, 0.526887766520182321, 3,  true},
{ 5, 2.30, 9.5205561e-05, 2.13487870958116410, 0.527610015869140359, 3,  true},
{ 5, 2.35, 8.7810064e-05, 2.14067128499348991, 0.528427378336588482, 3,  true},
{ 5, 2.40, 8.2686012e-05, 2.14426320393880232, 0.529153188069661606, 3,  true},
{ 5, 2.45, 7.7362661e-05, 2.14864959716796866, 0.530002593994140625, 3,  true},
{ 5, 2.50, 7.3695816e-05, 2.15946265326605946, 0.529833730061848529, 3,  true},
{ 6, 1.20,  0.0045594551, 1.50582012600368942, 0.540683492024739643, 3,  true},
{ 6, 1.25,  0.0021050185, 1.56927736070421009, 0.537108357747395737, 3,  true},
{ 6, 1.30,  0.0011154959, 1.62996512518988745, 0.533878580729166607, 3,  true},
{ 6, 1.35, 0.00064507169, 1.67900339762369777, 0.531911468505859464, 3,  true},
{ 6, 1.40, 0.00037315221, 1.72458309597439219, 0.530643971761067812, 3,  true},
{ 6, 1.45, 0.00024824541, 1.77476925320095469, 0.528379058837890714, 3,  true},
{ 6, 1.50, 0.00017724061, 1.81697709825303821, 0.527150217692057388, 3,  true},
{ 6, 1.55,  0.0001324623, 1.84778255886501697, 0.526968638102213394, 3,  true},
{ 6, 1.60, 0.00010158561, 1.87210167778862835, 0.527448781331380290, 3,  true},
{ 6, 1.65, 8.3583491e-05, 1.89237382676866250, 0.527660369873046875, 3,  true},
{ 6, 1.70, 6.8551138e-05, 1.93276146782769054, 0.525681813557942856, 3,  true},
{ 6, 1.75, 5.4616423e-05, 1.96848670111762192, 0.524261728922526204, 3,  true},
{ 6, 1.80, 4.3449372e-05, 1.99702436659071147, 0.523401641845703258, 3,  true},
{ 6, 1.85, 3.3827605e-05, 2.02382388644748223, 0.522790781656901182, 3,  true},
{ 6, 1.90, 2.8156339e-05, 2.06700524224175330, 0.520410919189453147, 3,  true},
{ 6, 1.95, 2.4445725e-05, 2.08590850830078134, 0.520696258544921631, 3,  true},
{ 6, 2.00, 2.1164377e-05, 2.10745629204644125, 0.520621490478515692, 3,  true},
{ 6, 2.05, 1.8468148e-05, 2.13673621283637205, 0.519232940673827970, 3,  true},
{ 6, 2.10, 1.6005415e-05, 2.16247270372178857, 0.518196360270182121, 3,  true},
{ 6, 2.15, 1.4258813e-05, 2.17432640923394116, 0.518426767985025960, 3,  true},
{ 6, 2.20, 1.2444461e-05, 2.18673960367838571, 0.518857574462890714, 3,  true},
{ 6, 2.25, 1.1114558e-05, 2.19571753607855946, 0.519317881266276093, 3,  true},
{ 6, 2.30, 9.9546958e-06, 2.20337880452473955, 0.519864654541015603, 3,  true},
{ 6, 2.35,  8.965476e-06, 2.20968983968098920, 0.520482126871744888, 3,  true},
{ 6, 2.40, 8.2885097e-06, 2.21353166368272536, 0.521029408772786695, 3,  true},
{ 6, 2.45, 7.5852234e-06, 2.23052520751953054, 0.520584869384765669, 3,  true},
{ 6, 2.50, 6.9655517e-06, 2.25309499104817679, 0.519467417399088638, 3,  true},
//{ 7, 1.20,  0.0062397048, 1.95579749213324638, 0.468906402587890514, 3,  true},
{ 7, 1.25,  0.0021701769, 1.80653703477647554, 0.501760101318359353, 3,  true},
{ 7, 1.30,  0.0008254969, 1.74045766194661389, 0.518593088785807343, 3,  true},
{ 7, 1.35, 0.00038442438, 1.76697370741102411, 0.520175933837890514, 3,  true},
{ 7, 1.40, 0.00018937161, 1.80296749538845447, 0.520410919189453147, 3,  true},
{ 7, 1.45, 0.00010361285, 1.82994232177734317, 0.521003468831379934, 3,  true},
{ 7, 1.50,  6.235019e-05, 1.87820756700303826, 0.519385019938151005, 3,  true},
{ 7, 1.55, 4.0294879e-05, 1.91923302544487862, 0.518271636962890581, 3,  true},
{ 7, 1.60, 2.7808874e-05, 1.95363786485460067, 0.517866770426432477, 3,  true},
{ 7, 1.65,  2.026018e-05, 1.97871042887369786, 0.517832692464192612, 3,  true},
{ 7, 1.70, 1.5136149e-05, 2.00276845296224026, 0.517857615152994666, 3,  true},
{ 7, 1.75,  1.151624e-05, 2.02899729410807383, 0.517629750569661207, 3,  true},
{ 7, 1.80, 8.9463191e-06, 2.06023915608723929, 0.516692860921223907, 3,  true},
{ 7, 1.85, 7.0887638e-06, 2.09950408935546839, 0.515063222249348862, 3,  true},
{ 7, 1.90, 5.8925966e-06, 2.11824323866102393, 0.515318044026692790, 3,  true},
{ 7, 1.95, 4.9355286e-06, 2.13582611083984419, 0.515685272216796831, 3,  true},
{ 7, 2.00, 4.1417575e-06, 2.18169674343532982, 0.512996164957682188, 3,  true},
{ 7, 2.05, 3.5555298e-06, 2.20350562201605849, 0.512531789143879868, 3,  true},
{ 7, 2.10, 3.0496661e-06, 2.22028656005859348, 0.512581634521484086, 3,  true},
{ 7, 2.15, 2.6606782e-06, 2.23297949896918446, 0.512734731038411162, 3,  true},
{ 7, 2.20,  2.303615e-06, 2.24713083902994715, 0.512998199462890270, 3,  true},
{ 7, 2.25, 2.0304322e-06, 2.25832807752821108, 0.513249460856119444, 3,  true},
{ 7, 2.30, 1.7924162e-06, 2.26957007514105769, 0.513472747802734553, 3,  true},
{ 7, 2.35, 1.5860954e-06, 2.28171844482421804, 0.513596852620442856, 3,  true},
{ 7, 2.40, 1.4287925e-06, 2.29392649332682330, 0.513477325439453236, 3,  true},
{ 7, 2.45, 1.2852443e-06, 2.30712670220269001, 0.513562774658203103, 3,  true},
{ 7, 2.50, 1.1779738e-06, 2.31585845947265678, 0.513826243082682321, 3,  true},
//{ 8, 1.20,  0.0080380716, 2.40528344048394072, 0.411779530843098873, 3,  true},
//{ 8, 1.25,  0.0029139281, 2.28206159803602393, 0.437241363525390669, 3,  true},
//{ 8, 1.30,  0.0011816208, 2.18023088243272545, 0.459330495198567779, 3,  true},
//{ 8, 1.35, 0.00055073447, 2.09329376220703045, 0.477956390380859408, 3,  true},
//{ 8, 1.40, 0.00026332174, 2.02678171793619777, 0.493080902099609419, 3,  true},
//{ 8, 1.45, 0.00014081099, 2.10365549723307277, 0.488327280680338593, 3,  true},
//{ 8, 1.50, 7.9734568e-05, 1.98848656548394076, 0.507173411051432321, 3,  true},
//{ 8, 1.55, 4.5950156e-05, 2.02879960801866321, 0.506347401936849018, 3,  true},
//{ 8, 1.60,  2.919248e-05, 2.03395809597439259, 0.509394582112630001, 3,  true},
{ 8, 1.65, 1.9452728e-05, 2.05627424452040009, 0.509817759195963283, 3,  true},
{ 8, 1.70, 1.3629565e-05, 2.07524464925130303, 0.510539499918619577, 3,  true},
{ 8, 1.75, 9.9866112e-06, 2.09811655680338527, 0.510826365152994732, 3,  true},
{ 8, 1.80, 7.4765723e-06, 2.12325252956814214, 0.510612742106119621, 3,  true},
{ 8, 1.85, 5.6599612e-06, 2.14303978814019125, 0.510897064208984397, 3,  true},
{ 8, 1.90, 4.4795853e-06, 2.16549021402994812, 0.510690053304036273, 3,  true},
{ 8, 1.95, 3.5909836e-06, 2.19277462429470438, 0.510074615478515847, 3,  true},
{ 8, 2.00, 2.9123621e-06, 2.21736975775824741, 0.509707895914713438, 3,  true},
{ 8, 2.05, 2.4525224e-06, 2.23633270263671724, 0.509581756591796942, 3,  true},
{ 8, 2.10, 2.0782745e-06, 2.25366566975911509, 0.509593454996744688, 3,  true},
{ 8, 2.15, 1.7816567e-06, 2.26725006103515581, 0.509669240315755001, 3,  true},
{ 8, 2.20, 1.5571862e-06, 2.28289710150824643, 0.509803517659505379, 3,  true},
{ 8, 2.25, 1.3738399e-06, 2.29649641248914893, 0.509857432047526093, 3,  true},
{ 8, 2.30, 1.2158171e-06, 2.31117740207248312, 0.509815724690755201, 3,  true},
{ 8, 2.35, 1.0793857e-06, 2.32826419406467000, 0.509585316975911473, 3,  true},
{ 8, 2.40, 9.6672011e-07, 2.34473554823133679, 0.509188588460286473, 3,  true},
{ 8, 2.45, 8.8091041e-07, 2.36683909098307277, 0.508585357666015647, 3,  true},
{ 8, 2.50, 8.0898294e-07, 2.38400794135199634, 0.508149464925130023, 3,  true},
{ 4, 1.20,   0.013815407, 1.30709338717990464, 0.588199361165364421, 1, false},
{ 5, 1.20,  0.0045431455, 1.47271143595377563, 0.547661844889323124, 1, false},
{ 6, 1.20, 0.00087410709, 1.48702875773111987, 0.543441772460937278, 1, false},
{ 7, 1.20, 0.00027149305, 1.57363950941297781, 0.528833516438802165, 1, false},
{ 8, 1.20, 7.8031902e-05, 1.62251086764865415, 0.521761576334635269, 1, false},
{ 9, 1.20, 2.2359735e-05, 1.65120146009657076, 0.517889404296875133, 1, false},
{10, 1.20, 5.9440654e-06, 1.69237518310546853, 0.513090769449869866, 1, false},
{11, 1.20, 1.7714079e-06, 1.70876074896918384, 0.511258697509765381, 1, false},
{12, 1.20, 5.2940891e-07, 1.72945437961154469, 0.508924102783202992, 1, false},
{13, 1.20,  1.616469e-07, 1.74369896782769063, 0.507493845621744821, 1, false},
{14, 1.20, 4.7466199e-08, 1.74225175645616304, 0.507634226481119666, 1, false},
{15, 1.20, 1.3705269e-08, 1.75197194417317714, 0.506649017333984264, 1, false},
{16, 1.20, 4.1584532e-09, 1.75695885552300313, 0.506128184000650960, 1, false},
{ 4, 1.25,  0.0085872483, 1.33188061184353357, 0.594018046061197902, 1, false},
{ 5, 1.25,  0.0025638577, 1.51477381388346322, 0.547609456380208193, 1, false},
{ 6, 1.25, 0.00052419465, 1.55846243964301179, 0.538631693522135246, 1, false},
{ 7, 1.25, 0.00012600907, 1.62363544040256103, 0.528018188476562345, 1, false},
{ 8, 1.25, 2.7541576e-05, 1.68515777587890647, 0.519925181070963416, 1, false},
{ 9, 1.25,   8.03598e-06, 1.71001400417751714, 0.516446685791015603, 1, false},
{10, 1.25,  2.032079e-06, 1.74668290879991273, 0.512401072184244577, 1, false},
{11, 1.25, 5.3950624e-07, 1.76964806450737866, 0.509976450602213416, 1, false},
{12, 1.25, 1.3863609e-07, 1.78036787245008643, 0.508847300211588749, 1, false},
{13, 1.25, 3.7582521e-08, 1.78752933078341969, 0.508087921142578502, 1, false},
{14, 1.25, 9.7976609e-09, 1.79297129313150982, 0.507515716552734419, 1, false},
{15, 1.25, 2.4353272e-09, 1.79002092149522563, 0.507702382405598951, 1, false},
{16, 1.25, 6.1772516e-10, 1.80235951741536460, 0.506502024332682210, 1, false},
{ 4, 1.30,  0.0057351145, 1.36280729505750875, 0.596257019042968683, 1, false},
{ 5, 1.30,  0.0014941066, 1.56042251586914049, 0.546399434407552009, 1, false},
{ 6, 1.30, 0.00030851593, 1.62978795369466201, 0.533901977539062544, 1, false},
{ 7, 1.30, 7.0253983e-05, 1.68304663764105911, 0.525806935628255534, 1, false},
{ 8, 1.30, 1.3471765e-05, 1.74423607720269080, 0.518215688069661340, 1, false},
{ 9, 1.30, 3.2516874e-06, 1.77520938449435794, 0.514374542236327836, 1, false},
{10, 1.30, 7.6463475e-07, 1.80346357557508630, 0.511223093668619510, 1, false},
{11, 1.30, 1.8880224e-07, 1.82270626491970456, 0.509227244059244799, 1, false},
{12, 1.30, 4.4308557e-08, 1.82085249159071205, 0.509485626220703036, 1, false},
{13, 1.30, 1.0231249e-08, 1.82708892822265589, 0.508758290608723796, 1, false},
{14, 1.30, 2.3178081e-09, 1.83420189751519080, 0.507956695556640692, 1, false},
{15, 1.30, 5.6881187e-10, 1.85050913492838509, 0.506482696533203325, 1, false},
{16, 1.30, 1.1262137e-10, 1.78474680582682299, 0.512872060139973995, 1, false},
{ 4, 1.35,  0.0042501164, 1.38681869506835920, 0.598430379231771004, 1, false},
{ 5, 1.35,  0.0009311744, 1.59968931410047732, 0.545594278971354263, 1, false},
{ 6, 1.35,   0.000185778, 1.67900339762369777, 0.531911468505859464, 1, false},
{ 7, 1.35, 4.2003532e-05, 1.73462778727213518, 0.523948923746745066, 1, false},
{ 8, 1.35, 7.5361412e-06, 1.78767479790581585, 0.517632293701171697, 1, false},
{ 9, 1.35,   1.64935e-06, 1.82514563666449625, 0.513285064697265758, 1, false},
{10, 1.35, 3.5204803e-07, 1.84514923095703098, 0.511175282796223707, 1, false},
{11, 1.35, 7.7359073e-08, 1.85591006808810710, 0.509974924723307299, 1, false},
{12, 1.35, 1.7120337e-08, 1.86972944471571201, 0.508644358317057343, 1, false},
{13, 1.35, 3.6481697e-09, 1.86772647433810768, 0.508736928304036495, 1, false},
{14, 1.35, 6.9429066e-10, 1.86868506537543366, 0.508473968505859464, 1, false},
{15, 1.35, 1.5483281e-10, 1.88645443386501666, 0.506858062744140359, 1, false},
{16, 1.35, 3.2736447e-11, 1.89248572455512143, 0.506277720133463616, 1, false},
{ 4, 1.40,  0.0033428725, 1.42907129923502607, 0.595376586914062478, 1, false},
{ 5, 1.40, 0.00058233056, 1.62376225789388040, 0.547126770019531361, 1, false},
{ 6, 1.40, 0.00010962619, 1.72446373833550348, 0.530658213297526271, 1, false},
{ 7, 1.40, 2.3781042e-05, 1.78181135389539902, 0.522699228922526182, 1, false},
{ 8, 1.40, 4.2558576e-06, 1.82943505181206612, 0.517185719807942368, 1, false},
{ 9, 1.40, 8.5886687e-07, 1.87064327663845464, 0.512637074788411362, 1, false},
{10, 1.40, 1.7240192e-07, 1.89010603162977397, 0.510741933186848796, 1, false},
{11, 1.40, 3.2659198e-08, 1.88815528021918388, 0.510795338948567546, 1, false},
{12, 1.40, 6.5804064e-09, 1.90691307915581620, 0.508948008219400894, 1, false},
{13, 1.40, 1.3268381e-09, 1.91943817138671879, 0.507719167073567790, 1, false},
{14, 1.40, 2.5054332e-10, 1.91767764621310755, 0.507796478271484331, 1, false},
{15, 1.40, 4.5470594e-11, 1.92086300320095482, 0.507323964436848907, 1, false},
{16, 1.40, 8.5081942e-12, 1.93046756320529456, 0.506463368733724106, 1, false},
{ 4, 1.45,  0.0028113458, 1.48220782809787344, 0.588401285807291785, 1, false},
{ 5, 1.45, 0.00041859306, 1.67027257283528652, 0.544658406575520782, 1, false},
{ 6, 1.45, 7.4569503e-05, 1.77476925320095469, 0.528379058837890714, 1, false},
{ 7, 1.45, 1.3873599e-05, 1.81916283501519116, 0.522077687581380068, 1, false},
{ 8, 1.45, 2.5854811e-06, 1.87167646620008665, 0.516175079345702947, 1, false},
{ 9, 1.45, 4.9172616e-07, 1.90793134901258621, 0.512265777587890381, 1, false},
{10, 1.45, 9.4654697e-08, 1.93040042453342009, 0.510135142008463660, 1, false},
{11, 1.45, 1.6664682e-08, 1.91693539089626719, 0.511350250244140492, 1, false},
{12, 1.45, 2.9986209e-09, 1.93405948215060786, 0.509625498453776027, 1, false},
{13, 1.45, 5.4304454e-10, 1.95315670437283018, 0.507810211181640714, 1, false},
{14, 1.45, 1.0425362e-10, 1.96645389133029469, 0.506643422444661429, 1, false},
{15, 1.45, 1.8444399e-11, 1.90480194091796884, 0.512058766682942701, 1, false},
{16, 1.45, 3.1326664e-12, 1.96340281168619835, 0.506584930419921697, 1, false},
{ 4, 1.50,  0.0023843893, 1.55200220743815076, 0.577678934733072857, 1, false},
{ 5, 1.50, 0.00032165452, 1.71065928141276036, 0.543058522542317568, 1, false},
{ 6, 1.50, 5.3916807e-05, 1.81697709825303821, 0.527150217692057388, 1, false},
{ 7, 1.50, 9.0668317e-06, 1.87820756700303804, 0.519385019938151005, 1, false},
{ 8, 1.50,  1.623544e-06, 1.91592831081814241, 0.515136464436848796, 1, false},
{ 9, 1.50, 2.8332531e-07, 1.94864722357855880, 0.511623891194661340, 1, false},
{10, 1.50, 5.1587061e-08, 1.96809505886501723, 0.509795888264973684, 1, false},
{11, 1.50, 8.9143687e-09, 1.95902014838324612, 0.510626475016276227, 1, false},
{12, 1.50,  1.503315e-09, 1.97183990478515603, 0.509373728434244777, 1, false},
{13, 1.50, 2.2394352e-10, 1.97394731309678861, 0.508898671468098862, 1, false},
{14, 1.50, 4.0377966e-11, 1.99085879855685755, 0.507420094807942590, 1, false},
{15, 1.50, 6.8224504e-12, 2.00304073757595535, 0.506375376383463305, 1, false},
{16, 1.50, 1.2343819e-12, 2.00665503607855999, 0.505981699625650982, 1, false},
{ 4, 1.55,  0.0020354502, 1.61562355889214393, 0.568545023600260491, 1, false},
{ 5, 1.55, 0.00025848611, 1.74115142822265612, 0.542447662353515714, 1, false},
{ 6, 1.55,  4.080595e-05, 1.84778255886501697, 0.526968638102213394, 1, false},
{ 7, 1.55, 6.4941452e-06, 1.91922556559244795, 0.518272145589192546, 1, false},
{ 8, 1.55, 1.1021222e-06, 1.96090749104817696, 0.513719431559244599, 1, false},
{ 9, 1.55, 1.7487073e-07, 1.96196679009331620, 0.513517506917317901, 1, false},
{10, 1.55, 2.9467914e-08, 1.99232838948567714, 0.510506439208984419, 1, false},
{11, 1.55, 4.8539272e-09, 2.00348086886935883, 0.509428660074869644, 1, false},
{12, 1.55, 8.1230807e-10, 2.01106753879123179, 0.508689626057942546, 1, false},
{13, 1.55, 1.1559056e-10, 1.99755028618706554, 0.509466298421224040, 1, false},
{14, 1.55,  1.837536e-11, 2.01643490261501634, 0.507848358154296964, 1, false},
{15, 1.55, 2.6822035e-12, 2.02784101698133723, 0.506849416097005290, 1, false},
{16, 1.55, 4.7836785e-13, 2.03843214246961812, 0.505949401855468772, 1, false},
{ 4, 1.60,  0.0017137364, 1.66557473076714380, 0.562941996256510202, 1, false},
{ 5, 1.60,  0.0002113896, 1.76781667073567705, 0.542211151123046964, 1, false},
{ 6, 1.60, 3.1474491e-05, 1.87210167778862813, 0.527448781331380179, 1, false},
{ 7, 1.60, 4.6654552e-06, 1.95363786485460023, 0.517866770426432477, 1, false},
{ 8, 1.60, 7.5861993e-07, 1.99141828748914929, 0.513874053955077903, 1, false},
{ 9, 1.60, 1.1134349e-07, 1.98512217203776031, 0.514410654703776005, 1, false},
{10, 1.60, 1.7042622e-08, 2.01318240695529616, 0.511495717366536540, 1, false},
{11, 1.60, 2.5879344e-09, 2.03020952012803813, 0.509832000732421853, 1, false},
{12, 1.60, 4.1693971e-10, 2.04782223171657973, 0.508222198486328103, 1, false},
{13, 1.60, 6.1658363e-11, 2.05101131863064312, 0.507862599690754979, 1, false},
{14, 1.60, 9.3611423e-12, 2.05649431016710116, 0.507271575927734197, 1, false},
{15, 1.60, 1.3004065e-12, 2.01804436577690938, 0.510756937662760313, 1, false},
{16, 1.60, 1.8593896e-13, 2.02783542209201340, 0.509182739257812433, 1, false},
{ 4, 1.65,  0.0014736472, 1.70882788764105853, 0.558200836181640403, 1, false},
{ 5, 1.65, 0.00018121468, 1.80731658935546857, 0.539609527587890825, 1, false},
{ 6, 1.65, 2.6103499e-05, 1.89124738905164880, 0.527779897054036606, 1, false},
{ 7, 1.65, 3.5565044e-06, 1.97871042887369786, 0.517832692464192723, 1, false},
{ 8, 1.65, 5.5312216e-07, 2.02737477620442652, 0.512729136149088327, 1, false},
{ 9, 1.65, 7.9987151e-08, 2.01524505615234384, 0.514055633544921897, 1, false},
{10, 1.65, 1.1242769e-08, 2.03220876057942634, 0.512120819091796853, 1, false},
{11, 1.65, 1.5765572e-09, 2.05390947129991286, 0.510055287679036407, 1, false},
{12, 1.65, 2.3613366e-10, 2.07090674506293437, 0.508435312906900916, 1, false},
{13, 1.65, 3.4102018e-11, 2.08033972846137116, 0.507582346598307255, 1, false},
{14, 1.65, 5.1066387e-12, 2.09127587212456634, 0.506627146402994555, 1, false},
{15, 1.65,  7.240791e-13, 2.09200320773654447, 0.506441497802734397, 1, false},
{16, 1.65, 9.6551207e-14, 2.04190470377604161, 0.511186726888020804, 1, false},
{ 4, 1.70,  0.0012560336, 1.75829043918185701, 0.552664438883463616, 1, false},
{ 5, 1.70, 0.00015171534, 1.83974456787109331, 0.537987009684244954, 1, false},
{ 6, 1.70, 2.1530094e-05, 1.92978871663411455, 0.525992584228515692, 1, false},
{ 7, 1.70, 2.7099306e-06, 2.00241783989800304, 0.517897796630859220, 1, false},
{ 8, 1.70, 4.0192318e-07, 2.05249209933810794, 0.512742869059244488, 1, false},
{ 9, 1.70, 5.7819133e-08, 2.04652421739366375, 0.513499704996744466, 1, false},
{10, 1.70, 7.7595465e-09, 2.05339101155599080, 0.512527211507161184, 1, false},
{11, 1.70, 9.9443921e-10, 2.06418914794921893, 0.511281077067057388, 1, false},
{12, 1.70, 1.3537401e-10, 2.08515879313151054, 0.509268951416015692, 1, false},
{13, 1.70, 1.8605538e-11, 2.10078718397352437, 0.507962799072265603, 1, false},
{14, 1.70, 2.6160996e-12, 2.11506161159939232, 0.506755828857421986, 1, false},
{15, 1.70, 3.7266125e-13, 2.12537112765841929, 0.505952707926431988, 1, false},
{16, 1.70, 5.6814687e-14, 2.13444417317708313, 0.505234781901041763, 1, false},
{ 4, 1.75,  0.0010613413, 1.78472069634331620, 0.551698048909505245, 1, false},
{ 5, 1.75, 0.00012364642, 1.87821875678168393, 0.535735321044921831, 1, false},
{ 6, 1.75,   1.71735e-05, 1.96848670111762192, 0.524261728922526204, 1, false},
{ 7, 1.75,  2.056967e-06, 2.02892269558376803, 0.517637888590494311, 1, false},
{ 8, 1.75, 2.9031371e-07, 2.07719167073567679, 0.512768809000650916, 1, false},
{ 9, 1.75, 4.0373911e-08, 2.10331607394748143, 0.510445404052734530, 1, false},
{10, 1.75, 5.4481297e-09, 2.10092146131727464, 0.510645294189453258, 1, false},
{11, 1.75, 6.9725682e-10, 2.10065290662977411, 0.510514068603515558, 1, false},
{12, 1.75, 8.3421663e-11, 2.10685204399956616, 0.509598541259765669, 1, false},
{13, 1.75, 1.0843658e-11, 2.12432301839192705, 0.508163706461588482, 1, false},
{14, 1.75,  1.391908e-12, 2.14296145968967000, 0.506688181559244777, 1, false},
{15, 1.75, 1.9028993e-13, 2.15020870632595518, 0.506039174397786340, 1, false},
{16, 1.75, 3.2250116e-14, 2.16093410915798634, 0.505217488606770959, 1, false},
{ 4, 1.80, 0.00090937354, 1.81566416422526022, 0.549645741780599106, 1, false},
{ 5, 1.80, 0.00010158062, 1.92916208902994835, 0.531913503011067768, 1, false},
{ 6, 1.80, 1.3655431e-05, 1.99691992865668366, 0.523411814371744888, 1, false},
{ 7, 1.80, 1.6092487e-06, 2.06023915608723929, 0.516692860921223907, 1, false},
{ 8, 1.80, 2.1708426e-07, 2.10110795762803804, 0.512618255615234109, 1, false},
{ 9, 1.80, 2.8953329e-08, 2.12062293158637161, 0.510943349202474084, 1, false},
{10, 1.80, 3.8249681e-09, 2.13734046088324670, 0.509422047932942657, 1, false},
{11, 1.80, 4.9665488e-10, 2.13370751274956616, 0.509656016031900805, 1, false},
{12, 1.80, 5.9088731e-11, 2.12671017116970473, 0.509776560465494577, 1, false},
{13, 1.80, 7.3962456e-12, 2.14630347357855955, 0.508229319254557166, 1, false},
{14, 1.80, 8.4779051e-13, 2.10544586181640581, 0.512052663167317790, 1, false},
{15, 1.80, 1.1759538e-13, 2.13324873182508590, 0.508927663167317745, 1, false},
{16, 1.80,  1.789478e-14, 2.14086337619357581, 0.508614603678385402, 1, false},
{ 4, 1.85, 0.00077742026, 1.82287784152560750, 0.551604970296223796, 1, false},
{ 5, 1.85, 8.5826364e-05, 1.97668134901258674, 0.528661346435546742, 1, false},
{ 6, 1.85,  1.060363e-05, 2.02382388644748223, 0.522790781656901182, 1, false},
{ 7, 1.85, 1.2915421e-06, 2.08198835584852526, 0.516882069905598818, 1, false},
{ 8, 1.85, 1.6360767e-07, 2.11999257405599018, 0.513011932373046986, 1, false},
{ 9, 1.85, 2.0780526e-08, 2.13356204562717044, 0.511795806884765447, 1, false},
{10, 1.85, 2.6100516e-09, 2.15372229682074634, 0.509919484456380134, 1, false},
{11, 1.85, 3.3400666e-10, 2.16671735975477509, 0.508801015218098618, 1, false},
{12, 1.85,  3.959553e-11, 2.17137230767144151, 0.508251190185546764, 1, false},
{13, 1.85, 4.9773284e-12, 2.18463965521918491, 0.507171376546223907, 1, false},
{14, 1.85, 5.6480952e-13, 2.18464338514539991, 0.506948089599609020, 1, false},
{15, 1.85, 7.3413075e-14, 2.14894239637586892, 0.509914652506510468, 1, false},
{16, 1.85, 1.4193522e-14, 2.16014522976345535, 0.509142303466796897, 1, false},
{ 4, 1.90, 0.00067418314, 1.84992726643880223, 0.550009918212890803, 1, false},
{ 5, 1.90,   7.52643e-05, 2.01164194742838554, 0.526721445719401160, 1, false},
{ 6, 1.90, 8.8289709e-06, 2.06700524224175330, 0.520410919189453147, 1, false},
{ 7, 1.90, 1.0791604e-06, 2.09541609022352526, 0.517656707763671786, 1, false},
{ 8, 1.90,  1.285904e-07, 2.12510257297092009, 0.514389292399088371, 1, false},
{ 9, 1.90, 1.5539072e-08, 2.15220421685112884, 0.511920928955078125, 1, false},
{10, 1.90, 1.8542669e-09, 2.17061513264973893, 0.510152435302734353, 1, false},
{11, 1.90, 2.3094423e-10, 2.18844790988498250, 0.508651479085286296, 1, false},
{12, 1.90, 2.6414976e-11, 2.19517669677734384, 0.507954661051432166, 1, false},
{13, 1.90,  3.274423e-12, 2.20817175971137170, 0.506905364990234530, 1, false},
{14, 1.90, 3.6867183e-13, 2.21245371500651000, 0.506451670328776249, 1, false},
{15, 1.90, 4.6995716e-14, 2.22232496473524277, 0.505666605631510446, 1, false},
{16, 1.90, 1.0620113e-14, 2.23213840060763902, 0.504876708984374867, 1, false},
{ 4, 1.95, 0.00058978801, 1.88333621554904518, 0.547602589925130112, 1, false},
{ 5, 1.95, 6.6001033e-05, 2.04016469319661509, 0.525634002685546831, 1, false},
{ 6, 1.95, 7.6754019e-06, 2.08590850830078134, 0.520696258544921631, 1, false},
{ 7, 1.95, 9.1020834e-07, 2.10214114718967027, 0.519080352783203303, 1, false},
{ 8, 1.95, 1.0189464e-07, 2.13631846110026080, 0.515235646565755157, 1, false},
{ 9, 1.95, 1.1779895e-08, 2.16356930202907982, 0.512669626871744888, 1, false},
{10, 1.95,  1.317752e-09, 2.18178999159071196, 0.510828908284505112, 1, false},
{11, 1.95, 1.5922142e-10, 2.20117441813151027, 0.509173838297526049, 1, false},
{12, 1.95, 1.7243425e-11, 2.21349809434678813, 0.508074696858723862, 1, false},
{13, 1.95, 2.0920773e-12, 2.22576209174262019, 0.507048288981119866, 1, false},
{14, 1.95, 2.3011745e-13, 2.23883175320095473, 0.506095631917317768, 1, false},
{15, 1.95, 2.8602883e-14, 2.24485744900173589, 0.505522664388020848, 1, false},
{16, 1.95, 8.2964783e-15, 2.25228000217013857, 0.504876708984374867, 1, false},
{ 4, 2.00, 0.00051933362, 1.90694291856553799, 0.546800994873046786, 1, false},
{ 5, 2.00, 5.7820989e-05, 2.06404741075303821, 0.525147755940755356, 1, false},
{ 6, 2.00, 6.6501224e-06, 2.10745629204644125, 0.520621490478515692, 1, false},
{ 7, 2.00, 7.7714857e-07, 2.13151058620876643, 0.518270619710286762, 1, false},
{ 8, 2.00,  8.198995e-08, 2.14249894883897563, 0.516482289632161362, 1, false},
{ 9, 2.00, 9.1181362e-09, 2.17356550428602446, 0.513518524169921942, 1, false},
{10, 2.00, 9.6725041e-10, 2.20186072455512116, 0.510932668050130157, 1, false},
{11, 2.00, 1.1203138e-10, 2.21397179497612839, 0.509688568115234331, 1, false},
{12, 2.00, 1.2269086e-11, 2.24148000081380294, 0.507639821370442501, 1, false},
{13, 2.00, 1.3735521e-12, 2.24892120361328107, 0.506914011637369821, 1, false},
{14, 2.00, 1.5905584e-13, 2.25044860839843697, 0.506738789876302165, 1, false},
{15, 2.00, 1.9623665e-14, 2.24747585720486143, 0.506626383463541607, 1, false},
{16, 2.00, 7.5125438e-15, 2.24828338623046831, 0.506981150309244621, 1, false},
{ 4, 2.05, 0.00047150197, 1.92870330810546897, 0.545942433675130068, 1, false},
{ 5, 2.05, 5.1616059e-05, 2.08155195448133679, 0.525099436442057366, 1, false},
{ 6, 2.05, 5.8021435e-06, 2.13673621283637205, 0.519232940673827970, 1, false},
{ 7, 2.05, 6.7617867e-07, 2.17350582546657956, 0.515741221110026071, 1, false},
{ 8, 2.05, 6.9553745e-08, 2.15772077772352455, 0.516779327392578147, 1, false},
{ 9, 2.05, 7.5109528e-09, 2.17791086832682312, 0.514518483479817657, 1, false},
{10, 2.05, 7.7082801e-10, 2.21550852457682268, 0.511269378662109308, 1, false},
{11, 2.05,  8.612464e-11, 2.23040584988064206, 0.509801991780599040, 1, false},
{12, 2.05, 9.2274332e-12, 2.25799238416883696, 0.507725779215494555, 1, false},
{13, 2.05, 1.0097669e-12, 2.26661597357855804, 0.506922658284505112, 1, false},
{14, 2.05, 1.1393051e-13, 2.22743123372395901, 0.509872436523437389, 1, false},
{15, 2.05, 1.3585144e-14, 2.24851837158203072, 0.508230336507161429, 1, false},
{16, 2.05, 5.9281344e-15, 2.24473063151041652, 0.508546956380208526, 1, false},
{ 4, 2.10, 0.00043017109, 1.94683447943793375, 0.545642852783203014, 1, false},
{ 5, 2.10, 4.6085986e-05, 2.09675140380859348, 0.525319671630859353, 1, false},
{ 6, 2.10, 5.0232581e-06, 2.16241302490234455, 0.518203481038411518, 1, false},
{ 7, 2.10, 5.7504767e-07, 2.21955549452039946, 0.512659962972005001, 1, false},
{ 8, 2.10, 5.9966208e-08, 2.17143944634331598, 0.517146046956380223, 1, false},
{ 9, 2.10, 6.4315103e-09, 2.21104380289713598, 0.513404083251952970, 1, false},
{10, 2.10, 6.2627526e-10, 2.22412465413411553, 0.511929066975911451, 1, false},
{11, 2.10, 6.8756031e-11, 2.23974558512369804, 0.510313161214192568, 1, false},
{12, 2.10, 7.0195203e-12, 2.26767900254991339, 0.508244578043619777, 1, false},
{13, 2.10, 7.5377441e-13, 2.22048051622178777, 0.511962127685547053, 1, false},
{14, 2.10, 8.6143203e-14, 2.24638112386067634, 0.509975433349609375, 1, false},
{15, 2.10, 1.0125126e-14, 2.25160861545138902, 0.509761555989583393, 1, false},
{16, 2.10, 5.7853129e-15, 2.20574917263454884, 0.514025370279948102, 1, false},
{ 4, 2.15, 0.00039975177, 1.95983327229817705, 0.545700836181640558, 1, false},
{ 5, 2.15,  4.198288e-05, 2.10787777370876706, 0.525670623779296853, 1, false},
{ 6, 2.15, 4.4743957e-06, 2.17432640923394116, 0.518426767985025960, 1, false},
{ 7, 2.15, 5.0018357e-07, 2.23297949896918446, 0.512734731038411051, 1, false},
{ 8, 2.15, 5.3254189e-08, 2.19627329508463598, 0.516251881917317523, 1, false},
{ 9, 2.15, 5.6017754e-09, 2.22688106960720544, 0.513256072998046653, 1, false},
{10, 2.15, 5.3877436e-10, 2.22816043429904509, 0.512646738688151027, 1, false},
{11, 2.15,  5.848809e-11, 2.25718299018012170, 0.510106658935546631, 1, false},
{12, 2.15, 5.7663117e-12, 2.27178565131293375, 0.508858489990234419, 1, false},
{13, 2.15,  6.189742e-13, 2.23387468126085098, 0.512366485595703081, 1, false},
{14, 2.15, 6.4181541e-14, 2.30832027859157973, 0.506116994222005290, 1, false},
{15, 2.15, 8.7127311e-15, 2.31010504828559027, 0.505754089355468617, 1, false},
{16, 2.15, 5.0738943e-15, 2.25038519965277750, 0.510668945312499956, 1, false},
{ 4, 2.20, 0.00036750595, 1.97344377305772567, 0.546037038167317745, 1, false},
{ 5, 2.20, 3.7601268e-05, 2.11950395372178768, 0.526264699300130090, 1, false},
{ 6, 2.20, 3.8946142e-06, 2.18673960367838571, 0.518857574462890714, 1, false},
{ 7, 2.20, 4.2176257e-07, 2.24713083902994715, 0.512998199462890270, 1, false},
{ 8, 2.20, 4.3757382e-08, 2.26494496663411438, 0.511482493082681899, 1, false},
{ 9, 2.20, 4.5740499e-09, 2.27114783393012099, 0.510934702555338682, 1, false},
{10, 2.20, 4.5191604e-10, 2.27525821261935723, 0.510403188069661562, 1, false},
{11, 2.20, 4.5773935e-11, 2.29960717095269063, 0.508342234293619577, 1, false},
{12, 2.20, 4.4851095e-12, 2.30357954237196205, 0.507871754964192901, 1, false},
{13, 2.20, 4.9025636e-13, 2.25579172770182224, 0.511590830485025849, 1, false},
{14, 2.20, 4.5520105e-14, 2.33169013129340330, 0.505726114908854329, 1, false},
{15, 2.20, 7.2469695e-15, 2.30224236382378411, 0.507592264811198013, 1, false},
{16, 2.20, 4.5519717e-15, 2.25403679741753482, 0.511529032389322902, 1, false},
{ 4, 2.25, 0.00034375373, 1.98338402642144063, 0.546487172444661384, 1, false},
{ 5, 2.25,  3.439937e-05, 2.12779930962456554, 0.526887766520182321, 1, false},
{ 6, 2.25, 3.4727917e-06, 2.19571753607855902, 0.519317881266276093, 1, false},
{ 7, 2.25, 3.6506545e-07, 2.25832807752821108, 0.513249460856119444, 1, false},
{ 8, 2.25, 3.6870286e-08, 2.28025631374782956, 0.511336008707682144, 1, false},
{ 9, 2.25, 3.7758473e-09, 2.29882388644748259, 0.509762827555338527, 1, false},
{10, 2.25, 3.6647238e-10, 2.30602637396918420, 0.509100596110026005, 1, false},
{11, 2.25, 3.5905262e-11, 2.31697370741102437, 0.508083852132161451, 1, false},
{12, 2.25, 3.4272367e-12, 2.32586585150824599, 0.507328033447265736, 1, false},
{13, 2.25, 3.2095066e-13, 2.33827904595269054, 0.506330617268880179, 1, false},
{14, 2.25, 3.2572104e-14, 2.34635993109808982, 0.505700174967447791, 1, false},
{15, 2.25, 5.6249746e-15, 2.34157443576388857, 0.505541992187499845, 1, false},
{16, 2.25, 4.1714486e-15, 2.28971354166666652, 0.509049479166666541, 1, false},
{ 4, 2.30, 0.00032256354, 1.99225379096137201, 0.547050221761067768, 1, false},
{ 5, 2.30, 3.1575688e-05, 2.13487870958116410, 0.527610015869140359, 1, false},
{ 6, 2.30, 3.1051064e-06, 2.20337880452473955, 0.519864654541015603, 1, false},
{ 7, 2.30, 3.1576811e-07, 2.26955888536241268, 0.513473765055338482, 1, false},
{ 8, 2.30, 3.1023915e-08, 2.28904028998480946, 0.511767323811848751, 1, false},
{ 9, 2.30, 3.0923132e-09, 2.31326989067925348, 0.509712473551432010, 1, false},
{10, 2.30,   2.92813e-10, 2.32252756754557277, 0.508909352620442790, 1, false},
{11, 2.30, 2.7700105e-11, 2.33217688666449563, 0.507999928792317590, 1, false},
{12, 2.30, 2.5787506e-12, 2.34049835205078072, 0.507298533121744888, 1, false},
{13, 2.30, 2.4911138e-13, 2.34607832166883634, 0.506789906819661407, 1, false},
{14, 2.30, 2.5326772e-14, 2.35365193684895768, 0.506177775065104196, 1, false},
{15, 2.30, 5.0189717e-15, 2.33074832492404438, 0.507601674397786695, 1, false},
{16, 2.30, 3.7559908e-15, 2.28815443250868045, 0.511583455403645804, 1, false},
{ 4, 2.35, 0.00030369868, 2.00018734402126697, 0.547697194417317679, 1, false},
{ 5, 2.35, 2.9120969e-05, 2.14067128499348991, 0.528427378336588482, 1, false},
{ 6, 2.35, 2.7921133e-06, 2.20968983968098920, 0.520482126871744888, 1, false},
{ 7, 2.35, 2.7335207e-07, 2.28155059814453187, 0.513612620035807210, 1, false},
{ 8, 2.35, 2.6018776e-08, 2.30517968071831536, 0.511560312906900849, 1, false},
{ 9, 2.35,  2.525367e-09, 2.32373233371310839, 0.509988657633463127, 1, false},
{10, 2.35, 2.3207815e-10, 2.33070356580946259, 0.509332529703776182, 1, false},
{11, 2.35, 2.1484147e-11, 2.35038265652126777, 0.507772572835286540, 1, false},
{12, 2.35, 2.0964478e-12, 2.35220659044053804, 0.507554880777994710, 1, false},
{13, 2.35, 1.9788865e-13, 2.34172736273871562, 0.508011372884114687, 1, false},
{14, 2.35, 1.9421937e-14, 2.35412190755208384, 0.507126871744791652, 1, false},
{15, 2.35, 4.3178387e-15, 2.33332010904947884, 0.508687845865885224, 1, false},
{16, 2.35, 3.5764489e-15, 2.36537509494357634, 0.504006449381510513, 1, false},
{ 4, 2.40, 0.00029016261, 2.00592770046658009, 0.548258209228515314, 1, false},
{ 5, 2.40, 2.7422246e-05, 2.14426320393880232, 0.529153188069661606, 1, false},
{ 6, 2.40, 2.5811282e-06, 2.21353166368272536, 0.521029408772786695, 1, false},
{ 7, 2.40, 2.4395236e-07, 2.29392649332682330, 0.513477325439453236, 1, false},
{ 8, 2.40, 2.2635538e-08, 2.30989430745442670, 0.512092336018880268, 1, false},
{ 9, 2.40, 2.1487134e-09, 2.33141225179036482, 0.510267893473307144, 1, false},
{10, 2.40, 1.9525547e-10, 2.34538455539279589, 0.509139251708984220, 1, false},
{11, 2.40, 1.8364802e-11, 2.36717105441623232, 0.507470448811848884, 1, false},
{12, 2.40, 1.7536355e-12, 2.37904713948567714, 0.506586964925130334, 1, false},
{13, 2.40, 1.5813936e-13, 2.33895043267144098, 0.509288279215494799, 1, false},
{14, 2.40, 1.5335344e-14, 2.35107828776041705, 0.508315022786458126, 1, false},
{15, 2.40, 4.1536105e-15, 2.34058973524305536, 0.508540852864583282, 1, false},
{16, 2.40, 3.3491859e-15, 2.33942599826388875, 0.511319986979166585, 1, false},
{ 4, 2.45, 0.00027502894, 2.01244015163845535, 0.548992156982421986, 1, false},
{ 5, 2.45, 2.5654799e-05, 2.14864959716796866, 0.530002593994140625, 1, false},
{ 6, 2.45, 2.3584489e-06, 2.23052520751953054, 0.520584869384765669, 1, false},
{ 7, 2.45, 2.1299005e-07, 2.30423973931206483, 0.513812510172525938, 1, false},
{ 8, 2.45, 1.9065345e-08, 2.31863352457682304, 0.512532806396484242, 1, false},
{ 9, 2.45,  1.760873e-09, 2.33694373236762143, 0.510904693603515647, 1, false},
{10, 2.45, 1.6221195e-10, 2.36320241292317723, 0.508957163492838260, 1, false},
{11, 2.45, 1.4935124e-11, 2.38042721218533027, 0.507595062255859042, 1, false},
{12, 2.45,  1.393832e-12, 2.38868526882595456, 0.506927235921224018, 1, false},
{13, 2.45, 1.2693878e-13, 2.33679080539279571, 0.511076100667317568, 1, false},
{14, 2.45, 1.1917414e-14, 2.35007866753472294, 0.509834798177083215, 1, false},
{15, 2.45, 3.7478109e-15, 2.34827524820963562, 0.509776051839192723, 1, false},
{16, 2.45, 3.0994407e-15, 2.34861653645833401, 0.511319986979166585, 1, false},
{ 4, 2.50, 0.00026426635, 2.01718461778428892, 0.549589284261067679, 1, false},
{ 5, 2.50, 2.4439176e-05, 2.15946265326605946, 0.529833730061848529, 1, false},
{ 6, 2.50, 2.1628261e-06, 2.25309499104817679, 0.519467417399088638, 1, false},
{ 7, 2.50, 1.9164717e-07, 2.30846947564019045, 0.514437103271484064, 1, false},
{ 8, 2.50, 1.6787971e-08, 2.33213212754991250, 0.512434641520182366, 1, false},
{ 9, 2.50, 1.5268243e-09, 2.35149417453342036, 0.510761769612630090, 1, false},
{10, 2.50, 1.4005755e-10, 2.37159847683376723, 0.509220123291015292, 1, false},
{11, 2.50, 1.2625363e-11, 2.38795047336154509, 0.507892100016275494, 1, false},
{12, 2.50, 1.1723073e-12, 2.39123280843099018, 0.507511138916015181, 1, false},
{13, 2.50, 1.0751716e-13, 2.39824133978949661, 0.506931304931640403, 1, false},
{14, 2.50,  1.077695e-14, 2.37836642795138831, 0.508345540364583459, 1, false},
{15, 2.50, 3.6241528e-15, 2.35148111979166652, 0.509773763020833326, 1, false},
{16, 2.50, 2.9646412e-15, 2.33321567111545125, 0.510513814290364354, 1, false},
{ 4, 1.20,   0.027630814, 1.30709338717990464, 0.588199361165364421, 2, false},
{ 5, 1.20,  0.0090862909, 1.47271143595377563, 0.547661844889323124, 2, false},
{ 6, 1.20,  0.0017482142, 1.48702875773111987, 0.543441772460937278, 2, false},
{ 7, 1.20,  0.0005429861, 1.57363950941297781, 0.528833516438802165, 2, false},
{ 8, 1.20, 0.00015606381, 1.62251086764865415, 0.521761576334635269, 2, false},
{ 9, 1.20, 4.4719474e-05, 1.65120146009657076, 0.517889404296875133, 2, false},
{10, 1.20,  1.188814e-05, 1.69237518310546853, 0.513090769449869866, 2, false},
{11, 1.20, 3.5428437e-06, 1.70876074896918384, 0.511258697509765381, 2, false},
{12, 1.20, 1.0588976e-06, 1.72945437961154469, 0.508924102783202992, 2, false},
{13, 1.20, 3.2352583e-07, 1.74369896782769063, 0.507493845621744821, 2, false},
{14, 1.20, 9.5676077e-08, 1.74225175645616304, 0.507634226481119666, 2, false},
{15, 1.20, 2.9601364e-08, 1.75197194417317714, 0.506649017333984153, 2, false},
{16, 1.20, 1.4565997e-08, 1.76991288926866353, 0.504910532633463305, 2, false},
{ 4, 1.25,   0.017174497, 1.33188061184353357, 0.594018046061197902, 2, false},
{ 5, 1.25,  0.0051277154, 1.51477381388346322, 0.547609456380208193, 2, false},
{ 6, 1.25,  0.0010483893, 1.55846243964301179, 0.538631693522135246, 2, false},
{ 7, 1.25, 0.00025201815, 1.62363544040256103, 0.528018188476562345, 2, false},
{ 8, 1.25, 5.5083153e-05, 1.68515777587890647, 0.519925181070963416, 2, false},
{ 9, 1.25, 1.6071961e-05, 1.71001400417751714, 0.516446685791015603, 2, false},
{10, 1.25, 4.0641604e-06, 1.74668290879991273, 0.512401072184244577, 2, false},
{11, 1.25, 1.0790185e-06, 1.76964806450737866, 0.509976450602213416, 2, false},
{12, 1.25, 2.7728809e-07, 1.78036787245008643, 0.508847300211588749, 2, false},
{13, 1.25, 7.5207271e-08, 1.78752933078341969, 0.508087921142578502, 2, false},
{14, 1.25, 1.9707719e-08, 1.79297129313150982, 0.507515716552734419, 2, false},
{15, 1.25, 5.1856713e-09, 1.79311303032769098, 0.507442982991536562, 2, false},
{16, 1.25, 2.0137551e-09, 1.80572391086154527, 0.506224822998046720, 2, false},
{ 4, 1.30,   0.011470229, 1.36280729505750875, 0.596257019042968683, 2, false},
{ 5, 1.30,  0.0029882132, 1.56042251586914049, 0.546399434407552009, 2, false},
{ 6, 1.30, 0.00061703186, 1.62978795369466201, 0.533901977539062544, 2, false},
{ 7, 1.30, 0.00014050797, 1.68304663764105911, 0.525806935628255534, 2, false},
{ 8, 1.30,  2.694353e-05, 1.74423607720269080, 0.518215688069661340, 2, false},
{ 9, 1.30, 6.5033751e-06, 1.77520938449435794, 0.514374542236327836, 2, false},
{10, 1.30, 1.5292703e-06, 1.80346357557508630, 0.511223093668619510, 2, false},
{11, 1.30, 3.7760627e-07, 1.82270626491970456, 0.509227244059244799, 2, false},
{12, 1.30, 8.8621514e-08, 1.82085249159071205, 0.509485626220703036, 2, false},
{13, 1.30, 2.0473021e-08, 1.82708892822265589, 0.508758290608723796, 2, false},
{14, 1.30, 4.6605257e-09, 1.83420189751519080, 0.507956695556640692, 2, false},
{15, 1.30, 1.1935629e-09, 1.85243004692925317, 0.506316375732421831, 2, false},
{16, 1.30, 3.7888952e-10, 1.85057254367404500, 0.506397755940754979, 2, false},
{ 4, 1.35,  0.0085002328, 1.38681869506835920, 0.598430379231771004, 2, false},
{ 5, 1.35,  0.0018623488, 1.59968931410047732, 0.545594278971354263, 2, false},
{ 6, 1.35,   0.000371556, 1.67900339762369777, 0.531911468505859464, 2, false},
{ 7, 1.35, 8.4007064e-05, 1.73462778727213518, 0.523948923746745066, 2, false},
{ 8, 1.35, 1.5072283e-05, 1.78767479790581585, 0.517632293701171697, 2, false},
{ 9, 1.35, 3.2987001e-06, 1.82514563666449625, 0.513285064697265758, 2, false},
{10, 1.35, 7.0409638e-07, 1.84514923095703098, 0.511175282796223707, 2, false},
{11, 1.35, 1.5471886e-07, 1.85591006808810710, 0.509974924723307299, 2, false},
{12, 1.35, 3.4242213e-08, 1.86972944471571201, 0.508644358317057343, 2, false},
{13, 1.35, 7.2998086e-09, 1.86772647433810768, 0.508736928304036495, 2, false},
{14, 1.35, 1.3963203e-09, 1.86868506537543366, 0.508473968505859464, 2, false},
{15, 1.35, 3.2543842e-10, 1.88662601047091982, 0.506844838460286384, 2, false},
{16, 1.35, 9.9868928e-11, 1.90032229953342013, 0.505679066975911473, 2, false},
{ 4, 1.40,  0.0066857449, 1.42907129923502607, 0.595376586914062478, 2, false},
{ 5, 1.40,  0.0011646611, 1.62376225789388040, 0.547126770019531361, 2, false},
{ 6, 1.40, 0.00021925238, 1.72446373833550348, 0.530658213297526271, 2, false},
{ 7, 1.40, 4.7562085e-05, 1.78181135389539902, 0.522699228922526182, 2, false},
{ 8, 1.40, 8.5117152e-06, 1.82943505181206612, 0.517185719807942368, 2, false},
{ 9, 1.40, 1.7177338e-06, 1.87064327663845464, 0.512637074788411362, 2, false},
{10, 1.40, 3.4480399e-07, 1.89010603162977397, 0.510741933186848796, 2, false},
{11, 1.40,  6.531871e-08, 1.88815528021918388, 0.510795338948567546, 2, false},
{12, 1.40, 1.3161431e-08, 1.90691307915581620, 0.508948008219400894, 2, false},
{13, 1.40, 2.6549019e-09, 1.91943817138671879, 0.507719167073567790, 2, false},
{14, 1.40, 5.0362062e-10, 1.91767764621310755, 0.507796478271484331, 2, false},
{15, 1.40, 9.6086651e-11, 1.92086300320095482, 0.507323964436848907, 2, false},
{16, 1.40, 2.7093353e-11, 1.93481665717230888, 0.506169382731119777, 2, false},
{ 4, 1.45,  0.0056226916, 1.48220782809787344, 0.588401285807291785, 2, false},
{ 5, 1.45, 0.00083718611, 1.67027257283528652, 0.544658406575520782, 2, false},
{ 6, 1.45, 0.00014913901, 1.77476925320095469, 0.528379058837890714, 2, false},
{ 7, 1.45, 2.7747197e-05, 1.81916283501519116, 0.522077687581380068, 2, false},
{ 8, 1.45, 5.1709622e-06, 1.87167646620008665, 0.516175079345702947, 2, false},
{ 9, 1.45, 9.8345236e-07, 1.90793134901258621, 0.512265777587890381, 2, false},
{10, 1.45, 1.8930947e-07, 1.93040042453342009, 0.510135142008463660, 2, false},
{11, 1.45,  3.332952e-08, 1.91693539089626719, 0.511350250244140492, 2, false},
{12, 1.45, 5.9975333e-09, 1.93405948215060786, 0.509625498453776027, 2, false},
{13, 1.45, 1.0866241e-09, 1.95315670437283018, 0.507810211181640714, 2, false},
{14, 1.45,  2.094927e-10, 1.96645389133029469, 0.506643422444661429, 2, false},
{15, 1.45, 3.9230381e-11, 1.90479821099175362, 0.512058766682942590, 2, false},
{16, 1.45, 9.8637235e-12, 1.96671498616536478, 0.506378936767577725, 2, false},
{ 4, 1.50,  0.0047687786, 1.55200220743815076, 0.577678934733072857, 2, false},
{ 5, 1.50, 0.00064330903, 1.71065928141276036, 0.543058522542317568, 2, false},
{ 6, 1.50, 0.00010783361, 1.81697709825303821, 0.527150217692057388, 2, false},
{ 7, 1.50, 1.8133663e-05, 1.87820756700303804, 0.519385019938151005, 2, false},
{ 8, 1.50, 3.2470879e-06, 1.91592831081814241, 0.515136464436848796, 2, false},
{ 9, 1.50, 5.6665064e-07, 1.94864722357855880, 0.511623891194661340, 2, false},
{10, 1.50, 1.0317417e-07, 1.96809505886501723, 0.509795888264973684, 2, false},
{11, 1.50, 1.7828823e-08, 1.95902014838324612, 0.510626475016276227, 2, false},
{12, 1.50, 3.0067838e-09, 1.97183990478515603, 0.509373728434244777, 2, false},
{13, 1.50, 4.4817001e-10, 1.97394731309678861, 0.508898671468098862, 2, false},
{14, 1.50, 8.1249349e-11, 1.99085879855685755, 0.507420094807942590, 2, false},
{15, 1.50, 1.4508907e-11, 2.00304073757595535, 0.506375376383463305, 2, false},
{16, 1.50, 3.9889501e-12, 2.01239166259765634, 0.505605824788411429, 2, false},
{ 4, 1.55,  0.0040709003, 1.61562355889214393, 0.568545023600260491, 2, false},
{ 5, 1.55, 0.00051697223, 1.74115142822265612, 0.542447662353515714, 2, false},
{ 6, 1.55,   8.16119e-05, 1.84778255886501697, 0.526968638102213394, 2, false},
{ 7, 1.55,  1.298829e-05, 1.91922556559244795, 0.518272145589192546, 2, false},
{ 8, 1.55, 2.2042444e-06, 1.96090749104817696, 0.513719431559244599, 2, false},
{ 9, 1.55, 3.4974148e-07, 1.96196679009331620, 0.513517506917317901, 2, false},
{10, 1.55, 5.8935855e-08, 1.99232838948567714, 0.510506439208984419, 2, false},
{11, 1.55, 9.7079011e-09, 2.00348086886935883, 0.509428660074869644, 2, false},
{12, 1.55, 1.6246975e-09, 2.01106753879123179, 0.508689626057942546, 2, false},
{13, 1.55, 2.3132914e-10, 1.99755028618706554, 0.509466298421224040, 2, false},
{14, 1.55,  3.699667e-11, 2.01643490261501634, 0.507848358154296964, 2, false},
{15, 1.55, 5.7787035e-12, 2.02789323594835080, 0.506847381591796986, 2, false},
{16, 1.55, 1.6329489e-12, 2.04246605767144107, 0.505713144938151005, 2, false},
{ 4, 1.60,  0.0034274729, 1.66557473076714380, 0.562941996256510202, 2, false},
{ 5, 1.60, 0.00042277921, 1.76781667073567705, 0.542211151123046964, 2, false},
{ 6, 1.60, 6.2948983e-05, 1.87210167778862813, 0.527448781331380179, 2, false},
{ 7, 1.60, 9.3309104e-06, 1.95363786485460023, 0.517866770426432477, 2, false},
{ 8, 1.60, 1.5172399e-06, 1.99141828748914929, 0.513874053955077903, 2, false},
{ 9, 1.60, 2.2268699e-07, 1.98512217203776031, 0.514410654703776005, 2, false},
{10, 1.60, 3.4085262e-08, 2.01318240695529616, 0.511495717366536540, 2, false},
{11, 1.60, 5.1758982e-09, 2.03020952012803813, 0.509832000732421853, 2, false},
{12, 1.60, 8.3392806e-10, 2.04782223171657973, 0.508222198486328103, 2, false},
{13, 1.60, 1.2339824e-10, 2.05101131863064312, 0.507862599690754979, 2, false},
{14, 1.60, 1.8857143e-11, 2.05649431016710116, 0.507271575927734197, 2, false},
{15, 1.60, 2.8427489e-12, 2.01824018690321161, 0.510732269287109242, 2, false},
{16, 1.60, 7.7510717e-13, 2.02804056803385357, 0.509260559082031272, 2, false},
{ 4, 1.65,  0.0029472944, 1.70882788764105853, 0.558200836181640403, 2, false},
{ 5, 1.65, 0.00036242937, 1.80731658935546857, 0.539609527587890825, 2, false},
{ 6, 1.65, 5.2206998e-05, 1.89124738905164880, 0.527779897054036606, 2, false},
{ 7, 1.65, 7.1130088e-06, 1.97871042887369786, 0.517832692464192723, 2, false},
{ 8, 1.65, 1.1062443e-06, 2.02737477620442652, 0.512729136149088327, 2, false},
{ 9, 1.65, 1.5997431e-07, 2.01524505615234384, 0.514055633544921897, 2, false},
{10, 1.65, 2.2485549e-08, 2.03220876057942634, 0.512120819091796853, 2, false},
{11, 1.65, 3.1531336e-09, 2.05390947129991286, 0.510055287679036407, 2, false},
{12, 1.65, 4.7229812e-10, 2.07090674506293437, 0.508435312906900916, 2, false},
{13, 1.65, 6.8253405e-11, 2.08033972846137116, 0.507582346598307255, 2, false},
{14, 1.65, 1.0291321e-11, 2.09127587212456634, 0.506627146402994555, 2, false},
{15, 1.65, 1.5733668e-12, 2.09200320773654447, 0.506441497802734397, 2, false},
{16, 1.65, 4.1632968e-13, 2.05187852647569402, 0.510007731119791474, 2, false},
{ 4, 1.70,  0.0025120672, 1.75829043918185701, 0.552664438883463616, 2, false},
{ 5, 1.70, 0.00030343068, 1.83974456787109331, 0.537987009684244954, 2, false},
{ 6, 1.70, 4.3060189e-05, 1.92978871663411455, 0.525992584228515692, 2, false},
{ 7, 1.70, 5.4198612e-06, 2.00241783989800304, 0.517897796630859220, 2, false},
{ 8, 1.70, 8.0384636e-07, 2.05249209933810794, 0.512742869059244488, 2, false},
{ 9, 1.70, 1.1563827e-07, 2.04652421739366375, 0.513499704996744466, 2, false},
{10, 1.70, 1.5519101e-08, 2.05339101155599080, 0.512527211507161184, 2, false},
{11, 1.70, 1.9888918e-09, 2.06418914794921893, 0.511281077067057388, 2, false},
{12, 1.70, 2.7076896e-10, 2.08515879313151054, 0.509268951416015692, 2, false},
{13, 1.70, 3.7243447e-11, 2.10078718397352437, 0.507962799072265603, 2, false},
{14, 1.70, 5.2817391e-12, 2.11506161159939232, 0.506755828857421986, 2, false},
{15, 1.70, 8.2091438e-13, 2.12537112765841929, 0.505952707926431988, 2, false},
{16, 1.70, 2.2836638e-13, 2.13444417317708313, 0.505234781901041763, 2, false},
{ 4, 1.75,  0.0021226826, 1.78472069634331620, 0.551698048909505245, 2, false},
{ 5, 1.75, 0.00024729283, 1.87821875678168393, 0.535735321044921831, 2, false},
{ 6, 1.75,    3.4347e-05, 1.96848670111762192, 0.524261728922526204, 2, false},
{ 7, 1.75, 4.1139339e-06, 2.02892269558376803, 0.517637888590494311, 2, false},
{ 8, 1.75, 5.8062743e-07, 2.07719167073567679, 0.512768809000650916, 2, false},
{ 9, 1.75, 8.0747826e-08, 2.10331607394748143, 0.510445404052734530, 2, false},
{10, 1.75, 1.0896265e-08, 2.10092146131727464, 0.510645294189453258, 2, false},
{11, 1.75,  1.394523e-09, 2.10065290662977411, 0.510514068603515558, 2, false},
{12, 1.75, 1.6685807e-10, 2.10685204399956616, 0.509598541259765669, 2, false},
{13, 1.75, 2.1709563e-11, 2.12432301839192705, 0.508163706461588482, 2, false},
{14, 1.75, 2.8187972e-12, 2.14311438666449705, 0.506678517659505223, 2, false},
{15, 1.75, 4.3021724e-13, 2.15020870632595518, 0.506039174397786340, 2, false},
{16, 1.75, 1.3786536e-13, 2.16093410915798634, 0.505217488606770959, 2, false},
{ 4, 1.80,  0.0018187471, 1.81566416422526022, 0.549645741780599106, 2, false},
{ 5, 1.80, 0.00020316124, 1.92916208902994835, 0.531913503011067768, 2, false},
{ 6, 1.80, 2.7310863e-05, 1.99691992865668366, 0.523411814371744888, 2, false},
{ 7, 1.80, 3.2184975e-06, 2.06023915608723929, 0.516692860921223907, 2, false},
{ 8, 1.80, 4.3416853e-07, 2.10110795762803804, 0.512618255615234109, 2, false},
{ 9, 1.80,  5.790666e-08, 2.12062293158637161, 0.510943349202474084, 2, false},
{10, 1.80, 7.6499403e-09, 2.13734046088324670, 0.509422047932942657, 2, false},
{11, 1.80, 9.9331646e-10, 2.13370751274956616, 0.509656016031900805, 2, false},
{12, 1.80,   1.18188e-10, 2.12671017116970473, 0.509776560465494577, 2, false},
{13, 1.80, 1.4808048e-11, 2.14630347357855955, 0.508229319254557166, 2, false},
{14, 1.80, 1.7205951e-12, 2.10544586181640581, 0.512052663167317790, 2, false},
{15, 1.80, 2.7112982e-13, 2.13324873182508590, 0.508927663167317745, 2, false},
{16, 1.80, 9.2432339e-14, 2.17575683593749947, 0.505806477864583326, 2, false},
{ 4, 1.85,  0.0015548405, 1.82287784152560750, 0.551604970296223796, 2, false},
{ 5, 1.85, 0.00017165273, 1.97668134901258674, 0.528661346435546742, 2, false},
{ 6, 1.85, 2.1207259e-05, 2.02382388644748223, 0.522790781656901182, 2, false},
{ 7, 1.85, 2.5830842e-06, 2.08198835584852526, 0.516882069905598818, 2, false},
{ 8, 1.85, 3.2721534e-07, 2.11999257405599018, 0.513011932373046986, 2, false},
{ 9, 1.85, 4.1561054e-08, 2.13356204562717044, 0.511795806884765447, 2, false},
{10, 1.85, 5.2201064e-09, 2.15372229682074634, 0.509919484456380134, 2, false},
{11, 1.85, 6.6801813e-10, 2.16671735975477509, 0.508801015218098618, 2, false},
{12, 1.85, 7.9198356e-11, 2.17137230767144151, 0.508251190185546764, 2, false},
{13, 1.85, 9.9653524e-12, 2.18463965521918491, 0.507171376546223907, 2, false},
{14, 1.85, 1.1454323e-12, 2.18464338514539991, 0.506948089599609020, 2, false},
{15, 1.85, 1.6940164e-13, 2.19788648817274268, 0.505950419108072702, 2, false},
{16, 1.85, 6.0166976e-14, 2.21154920789930598, 0.504844156901041563, 2, false},
{ 4, 1.90,  0.0013483663, 1.84992726643880223, 0.550009918212890803, 2, false},
{ 5, 1.90,  0.0001505286, 2.01164194742838554, 0.526721445719401160, 2, false},
{ 6, 1.90, 1.7657942e-05, 2.06700524224175330, 0.520410919189453147, 2, false},
{ 7, 1.90, 2.1583209e-06, 2.09541609022352526, 0.517656707763671786, 2, false},
{ 8, 1.90,  2.571808e-07, 2.12510257297092009, 0.514389292399088371, 2, false},
{ 9, 1.90, 3.1078146e-08, 2.15220421685112884, 0.511920928955078125, 2, false},
{10, 1.90, 3.7085363e-09, 2.17061513264973893, 0.510152435302734353, 2, false},
{11, 1.90, 4.6189212e-10, 2.18844790988498250, 0.508651479085286296, 2, false},
{12, 1.90,  5.283541e-11, 2.19517669677734384, 0.507954661051432166, 2, false},
{13, 1.90, 6.5567471e-12, 2.20817175971137170, 0.506905364990234530, 2, false},
{14, 1.90, 7.4876468e-13, 2.21245371500651000, 0.506451670328776249, 2, false},
{15, 1.90, 1.0987385e-13, 2.22502916124131955, 0.505518595377604019, 2, false},
{16, 1.90, 4.3904399e-14, 2.23213840060763902, 0.504876708984374867, 2, false},
{ 4, 1.95,   0.001179576, 1.88333621554904518, 0.547602589925130112, 2, false},
{ 5, 1.95, 0.00013200207, 2.04016469319661509, 0.525634002685546831, 2, false},
{ 6, 1.95, 1.5350804e-05, 2.08590850830078134, 0.520696258544921631, 2, false},
{ 7, 1.95, 1.8204167e-06, 2.10214114718967027, 0.519080352783203303, 2, false},
{ 8, 1.95, 2.0378929e-07, 2.13631846110026080, 0.515235646565755157, 2, false},
{ 9, 1.95, 2.3559792e-08, 2.16356930202907982, 0.512669626871744888, 2, false},
{10, 1.95, 2.6355058e-09, 2.18178999159071196, 0.510828908284505112, 2, false},
{11, 1.95, 3.1844565e-10, 2.20117441813151027, 0.509173838297526049, 2, false},
{12, 1.95, 3.4490979e-11, 2.21349809434678813, 0.508074696858723862, 2, false},
{13, 1.95, 4.1900745e-12, 2.22576209174262019, 0.507048288981119866, 2, false},
{14, 1.95,  4.685642e-13, 2.23883175320095473, 0.506095631917317768, 2, false},
{15, 1.95, 6.9809174e-14, 2.24861907958984375, 0.505327606201171675, 2, false},
{16, 1.95, 3.2796943e-14, 2.25592041015625044, 0.504429117838541607, 2, false},
{ 4, 2.00,  0.0010386672, 1.90694291856553799, 0.546800994873046786, 2, false},
{ 5, 2.00, 0.00011564198, 2.06404741075303821, 0.525147755940755356, 2, false},
{ 6, 2.00, 1.3300245e-05, 2.10745629204644125, 0.520621490478515692, 2, false},
{ 7, 2.00, 1.5542971e-06, 2.13151058620876643, 0.518270619710286762, 2, false},
{ 8, 2.00,  1.639799e-07, 2.14249894883897563, 0.516482289632161362, 2, false},
{ 9, 2.00, 1.8236273e-08, 2.17356550428602446, 0.513518524169921942, 2, false},
{10, 2.00, 1.9345023e-09, 2.20186072455512116, 0.510932668050130157, 2, false},
{11, 2.00, 2.2406492e-10, 2.21397179497612839, 0.509688568115234331, 2, false},
{12, 2.00, 2.4541257e-11, 2.24148000081380294, 0.507639821370442501, 2, false},
{13, 2.00, 2.7515159e-12, 2.24892120361328107, 0.506914011637369821, 2, false},
{14, 2.00, 3.2435552e-13, 2.25044860839843697, 0.506738789876302165, 2, false},
{15, 2.00, 4.8048055e-14, 2.24747585720486143, 0.506626383463541607, 2, false},
{16, 2.00, 2.5131751e-14, 2.27874009874131955, 0.504424031575520848, 2, false},
{ 4, 2.05, 0.00094300395, 1.92870330810546897, 0.545942433675130068, 2, false},
{ 5, 2.05, 0.00010323212, 2.08155195448133679, 0.525099436442057366, 2, false},
{ 6, 2.05, 1.1604287e-05, 2.13673621283637205, 0.519232940673827970, 2, false},
{ 7, 2.05, 1.3523573e-06, 2.17350582546657956, 0.515741221110026071, 2, false},
{ 8, 2.05, 1.3910749e-07, 2.15772077772352455, 0.516779327392578147, 2, false},
{ 9, 2.05, 1.5021906e-08, 2.17791086832682312, 0.514518483479817657, 2, false},
{10, 2.05, 1.5416572e-09, 2.21550852457682268, 0.511269378662109308, 2, false},
{11, 2.05, 1.7225099e-10, 2.23040584988064206, 0.509801991780599040, 2, false},
{12, 2.05, 1.8457297e-11, 2.25799238416883696, 0.507725779215494555, 2, false},
{13, 2.05, 2.0229806e-12, 2.26661597357855804, 0.506922658284505112, 2, false},
{14, 2.05,   2.32967e-13, 2.22743123372395901, 0.509872436523437389, 2, false},
{15, 2.05, 3.4057151e-14, 2.24851837158203072, 0.508230336507161429, 2, false},
{16, 2.05, 2.1135293e-14, 2.28975830078124964, 0.504876708984374867, 2, false},
{ 4, 2.10, 0.00086034219, 1.94683447943793375, 0.545642852783203014, 2, false},
{ 5, 2.10, 9.2171972e-05, 2.09675140380859348, 0.525319671630859353, 2, false},
{ 6, 2.10, 1.0046516e-05, 2.16241302490234455, 0.518203481038411518, 2, false},
{ 7, 2.10, 1.1500953e-06, 2.21955549452039946, 0.512659962972005001, 2, false},
{ 8, 2.10, 1.1993242e-07, 2.17143944634331598, 0.517146046956380223, 2, false},
{ 9, 2.10, 1.2863021e-08, 2.21104380289713598, 0.513404083251952970, 2, false},
{10, 2.10, 1.2525514e-09, 2.22412465413411553, 0.511929066975911451, 2, false},
{11, 2.10, 1.3751342e-10, 2.23974558512369804, 0.510313161214192568, 2, false},
{12, 2.10,  1.404097e-11, 2.26767900254991339, 0.508244578043619777, 2, false},
{13, 2.10, 1.5104659e-12, 2.22048051622178777, 0.511962127685547053, 2, false},
{14, 2.10, 1.7622925e-13, 2.24638112386067634, 0.509975433349609375, 2, false},
{15, 2.10, 2.5917971e-14, 2.26238810221354081, 0.508665974934895848, 2, false},
{16, 2.10,  1.760438e-14, 2.34405483669704839, 0.502705383300781405, 2, false},
{ 4, 2.15, 0.00079950354, 1.95983327229817705, 0.545700836181640558, 2, false},
{ 5, 2.15,  8.396576e-05, 2.10787777370876706, 0.525670623779296853, 2, false},
{ 6, 2.15, 8.9487914e-06, 2.17432640923394116, 0.518426767985025960, 2, false},
{ 7, 2.15, 1.0003671e-06, 2.23297949896918446, 0.512734731038411051, 2, false},
{ 8, 2.15, 1.0650838e-07, 2.19627329508463598, 0.516251881917317523, 2, false},
{ 9, 2.15, 1.1203551e-08, 2.22688106960720544, 0.513256072998046653, 2, false},
{10, 2.15, 1.0775494e-09, 2.22816043429904509, 0.512646738688151027, 2, false},
{11, 2.15, 1.1697725e-10, 2.25718299018012170, 0.510106658935546631, 2, false},
{12, 2.15, 1.1534164e-11, 2.27178565131293375, 0.508858489990234419, 2, false},
{13, 2.15,  1.240233e-12, 2.23387468126085098, 0.512366485595703081, 2, false},
{14, 2.15, 1.3126925e-13, 2.30832027859157973, 0.506116994222005290, 2, false},
{15, 2.15, 2.1176376e-14, 2.31364474826388866, 0.505541992187499734, 2, false},
{16, 2.15, 1.5296998e-14, 2.37511206732855928, 0.501588694254557321, 2, false},
{ 4, 2.20,  0.0007350119, 1.97344377305772567, 0.546037038167317745, 2, false},
{ 5, 2.20, 7.5202537e-05, 2.11950395372178768, 0.526264699300130090, 2, false},
{ 6, 2.20, 7.7892284e-06, 2.18673960367838571, 0.518857574462890714, 2, false},
{ 7, 2.20, 8.4352514e-07, 2.24713083902994715, 0.512998199462890270, 2, false},
{ 8, 2.20, 8.7514764e-08, 2.26494496663411438, 0.511482493082681899, 2, false},
{ 9, 2.20, 9.1481001e-09, 2.27114783393012099, 0.510934702555338682, 2, false},
{10, 2.20, 9.0383263e-10, 2.27525821261935723, 0.510403188069661562, 2, false},
{11, 2.20, 9.1548702e-11, 2.29960717095269063, 0.508342234293619577, 2, false},
{12, 2.20, 8.9714426e-12, 2.30357954237196205, 0.507871754964192901, 2, false},
{13, 2.20, 9.8235294e-13, 2.25579172770182224, 0.511590830485025849, 2, false},
{14, 2.20,  9.336502e-14, 2.33169013129340330, 0.505726114908854329, 2, false},
{15, 2.20, 1.7775364e-14, 2.30224236382378411, 0.507592264811198013, 2, false},
{16, 2.20, 1.3178514e-14, 2.39444240993923652, 0.501569620768229085, 2, false},
{ 4, 2.25, 0.00068750745, 1.98338402642144063, 0.546487172444661384, 2, false},
{ 5, 2.25, 6.8798741e-05, 2.12779930962456554, 0.526887766520182321, 2, false},
{ 6, 2.25, 6.9455833e-06, 2.19571753607855902, 0.519317881266276093, 2, false},
{ 7, 2.25, 7.3013089e-07, 2.25832807752821108, 0.513249460856119444, 2, false},
{ 8, 2.25, 7.3740572e-08, 2.28025631374782956, 0.511336008707682144, 2, false},
{ 9, 2.25, 7.5516948e-09, 2.29882388644748259, 0.509762827555338527, 2, false},
{10, 2.25, 7.3294518e-10, 2.30602637396918420, 0.509100596110026005, 2, false},
{11, 2.25, 7.1811192e-11, 2.31697370741102437, 0.508083852132161451, 2, false},
{12, 2.25, 6.8554541e-12, 2.32586585150824599, 0.507328033447265736, 2, false},
{13, 2.25, 6.4327647e-13, 2.33827904595269054, 0.506330617268880179, 2, false},
{14, 2.25, 6.7026174e-14, 2.34635993109808982, 0.505700174967447791, 2, false},
{15, 2.25, 1.3819742e-14, 2.34157443576388857, 0.505541992187499845, 2, false},
{16, 2.25, 1.1356449e-14, 2.38293185763888937, 0.502848307291666519, 2, false},
{ 4, 2.30, 0.00064512709, 1.99225379096137201, 0.547050221761067768, 2, false},
{ 5, 2.30, 6.3151377e-05, 2.13487870958116410, 0.527610015869140359, 2, false},
{ 6, 2.30, 6.2102127e-06, 2.20337880452473955, 0.519864654541015603, 2, false},
{ 7, 2.30, 6.3153623e-07, 2.26955888536241268, 0.513473765055338482, 2, false},
{ 8, 2.30, 6.2047829e-08, 2.28904028998480946, 0.511767323811848751, 2, false},
{ 9, 2.30, 6.1846265e-09, 2.31326989067925348, 0.509712473551432010, 2, false},
{10, 2.30, 5.8562634e-10, 2.32252756754557277, 0.508909352620442790, 2, false},
{11, 2.30,  5.540074e-11, 2.33217688666449563, 0.507999928792317590, 2, false},
{12, 2.30, 5.1582894e-12, 2.34049835205078072, 0.507298533121744888, 2, false},
{13, 2.30, 4.9934178e-13, 2.34607832166883634, 0.506789906819661407, 2, false},
{14, 2.30, 5.2186506e-14, 2.35365193684895768, 0.506177775065104196, 2, false},
{15, 2.30, 1.2169554e-14, 2.33074832492404438, 0.507601674397786695, 2, false},
{16, 2.30, 1.0068399e-14, 2.41578504774305580, 0.500638834635416696, 2, false},
{ 4, 2.35, 0.00060739736, 2.00018734402126697, 0.547697194417317679, 2, false},
{ 5, 2.35, 5.8241938e-05, 2.14067128499348991, 0.528427378336588482, 2, false},
{ 6, 2.35, 5.5842265e-06, 2.20968983968098920, 0.520482126871744888, 2, false},
{ 7, 2.35, 5.4670414e-07, 2.28155059814453187, 0.513612620035807210, 2, false},
{ 8, 2.35, 5.2037551e-08, 2.30517968071831536, 0.511560312906900849, 2, false},
{ 9, 2.35, 5.0507342e-09, 2.32373233371310839, 0.509988657633463127, 2, false},
{10, 2.35, 4.6415656e-10, 2.33070356580946259, 0.509332529703776182, 2, false},
{11, 2.35, 4.2968707e-11, 2.35038265652126777, 0.507772572835286540, 2, false},
{12, 2.35, 4.1935249e-12, 2.35220659044053804, 0.507554880777994710, 2, false},
{13, 2.35,  3.966966e-13, 2.34172736273871562, 0.508011372884114687, 2, false},
{14, 2.35, 4.0098276e-14, 2.35412190755208384, 0.507126871744791652, 2, false},
{15, 2.35, 1.0354054e-14, 2.34626668294270768, 0.507597859700520737, 2, false},
{16, 2.35, 9.1074408e-15, 2.42102186414930598, 0.502773030598958393, 2, false},
{ 4, 2.40, 0.00058032523, 2.00592770046658009, 0.548258209228515314, 2, false},
{ 5, 2.40, 5.4844492e-05, 2.14426320393880232, 0.529153188069661606, 2, false},
{ 6, 2.40, 5.1622563e-06, 2.21353166368272536, 0.521029408772786695, 2, false},
{ 7, 2.40, 4.8790471e-07, 2.29392649332682330, 0.513477325439453236, 2, false},
{ 8, 2.40, 4.5271076e-08, 2.30989430745442670, 0.512092336018880268, 2, false},
{ 9, 2.40, 4.2974268e-09, 2.33141225179036482, 0.510267893473307144, 2, false},
{10, 2.40, 3.9051113e-10, 2.34538455539279589, 0.509139251708984220, 2, false},
{11, 2.40, 3.6729918e-11, 2.36717105441623232, 0.507470448811848884, 2, false},
{12, 2.40, 3.5077579e-12, 2.37904713948567714, 0.506586964925130334, 2, false},
{13, 2.40, 3.1702584e-13, 2.33895043267144098, 0.509288279215494799, 2, false},
{14, 2.40, 3.1695681e-14, 2.35107828776041705, 0.508315022786458126, 2, false},
{15, 2.40, 9.7154883e-15, 2.34058973524305536, 0.508540852864583282, 2, false},
{16, 2.40, 8.1910843e-15, 2.43121202256944402, 0.503002929687500155, 2, false},
{ 4, 2.45, 0.00055005788, 2.01244015163845535, 0.548992156982421986, 2, false},
{ 5, 2.45, 5.1309599e-05, 2.14864959716796866, 0.530002593994140625, 2, false},
{ 6, 2.45, 4.7168978e-06, 2.23052520751953054, 0.520584869384765669, 2, false},
{ 7, 2.45,  4.259801e-07, 2.30423973931206483, 0.513812510172525938, 2, false},
{ 8, 2.45, 3.8130691e-08, 2.31863352457682304, 0.512532806396484242, 2, false},
{ 9, 2.45,  3.521746e-09, 2.33694373236762143, 0.510904693603515647, 2, false},
{10, 2.45, 3.2442404e-10, 2.36320241292317723, 0.508957163492838260, 2, false},
{11, 2.45, 2.9870495e-11, 2.38042721218533027, 0.507595062255859042, 2, false},
{12, 2.45, 2.7880605e-12, 2.38868526882595456, 0.506927235921224018, 2, false},
{13, 2.45, 2.5450194e-13, 2.33679080539279571, 0.511076100667317568, 2, false},
{14, 2.45, 2.4696607e-14, 2.35007866753472294, 0.509834798177083215, 2, false},
{15, 2.45, 8.6673822e-15, 2.34827524820963562, 0.509776051839192723, 2, false},
{16, 2.45, 7.9794978e-15, 2.35947808159722250, 0.511319986979166585, 2, false},
{ 4, 2.50,  0.0005285327, 2.01718461778428892, 0.549589284261067679, 2, false},
{ 5, 2.50, 4.8878353e-05, 2.15946265326605946, 0.529833730061848529, 2, false},
{ 6, 2.50, 4.3256521e-06, 2.25309499104817679, 0.519467417399088638, 2, false},
{ 7, 2.50, 3.8329434e-07, 2.30846947564019045, 0.514437103271484064, 2, false},
{ 8, 2.50, 3.3575942e-08, 2.33213212754991250, 0.512434641520182366, 2, false},
{ 9, 2.50, 3.0536485e-09, 2.35149417453342036, 0.510761769612630090, 2, false},
{10, 2.50, 2.8011519e-10, 2.37159847683376723, 0.509220123291015292, 2, false},
{11, 2.50, 2.5250917e-11, 2.38795047336154509, 0.507892100016275494, 2, false},
{12, 2.50, 2.3449365e-12, 2.39123280843099018, 0.507511138916015181, 2, false},
{13, 2.50,  2.155162e-13, 2.39824133978949661, 0.506931304931640403, 2, false},
{14, 2.50, 2.2253421e-14, 2.37836642795138831, 0.508345540364583459, 2, false},
{15, 2.50, 8.2349925e-15, 2.35148111979166652, 0.509773763020833326, 2, false},
{16, 2.50, 7.1866641e-15, 2.35853814019097285, 0.512795003255208326, 2, false},
{ 4, 1.20,   0.041446222, 1.30709338717990464, 0.588199361165364421, 3, false},
{ 5, 1.20,   0.013629436, 1.47271143595377563, 0.547661844889323124, 3, false},
{ 6, 1.20,  0.0026223213, 1.48702875773111987, 0.543441772460937278, 3, false},
{ 7, 1.20, 0.00081447916, 1.57363950941297781, 0.528833516438802165, 3, false},
{ 8, 1.20, 0.00023409579, 1.62251086764865415, 0.521761576334635269, 3, false},
{ 9, 1.20, 6.7079641e-05, 1.65120146009657076, 0.517889404296875133, 3, false},
{10, 1.20, 1.7834137e-05, 1.69237518310546853, 0.513090769449869866, 3, false},
{11, 1.20, 5.3241984e-06, 1.70876074896918384, 0.511258697509765381, 3, false},
{12, 1.20, 1.6365303e-06, 1.72945437961154491, 0.508924102783202881, 3, false},
{13, 1.20, 7.1906035e-07, 1.74623904758029513, 0.507221730550130090, 3, false},
//{14, 1.20, 1.2303757e-06, 1.78480275472005223, 0.503347524007161540, 3, false},
//{15, 1.20,  2.363096e-06, 1.93205651177300375, 0.486727142333984508, 3, false},
//{16, 1.20, 3.6129712e-06, 2.08135799831814206, 0.469492848714192723, 3, false},
{ 4, 1.25,   0.025761745, 1.33188061184353357, 0.594018046061197902, 3, false},
{ 5, 1.25,  0.0076915731, 1.51477381388346322, 0.547609456380208193, 3, false},
{ 6, 1.25,  0.0015725839, 1.55846243964301179, 0.538631693522135246, 3, false},
{ 7, 1.25, 0.00037802722, 1.62363544040256103, 0.528018188476562345, 3, false},
{ 8, 1.25, 8.2624744e-05, 1.68515777587890647, 0.519925181070963416, 3, false},
{ 9, 1.25, 2.4108007e-05, 1.71001400417751714, 0.516446685791015603, 3, false},
{10, 1.25, 6.0964926e-06, 1.74668290879991273, 0.512401072184244577, 3, false},
{11, 1.25, 1.6195385e-06, 1.76964806450737866, 0.509976450602213416, 3, false},
{12, 1.25, 4.2023445e-07, 1.78036787245008643, 0.508847300211588749, 3, false},
{13, 1.25,  1.306458e-07, 1.79408654106987808, 0.507438913981120066, 3, false},
{14, 1.25, 9.9033504e-08, 1.81174401177300304, 0.505699920654296808, 3, false},
//{15, 1.25, 2.1372065e-07, 1.89445139567057286, 0.497390492757161518, 3, false},
//{16, 1.25, 3.5916982e-07, 2.04013112386067741, 0.481364695231119799, 3, false},
{ 4, 1.30,   0.017205343, 1.36280729505750875, 0.596257019042968683, 3, false},
{ 5, 1.30,  0.0044823198, 1.56042251586914049, 0.546399434407552009, 3, false},
{ 6, 1.30,  0.0009255478, 1.62978795369466201, 0.533901977539062544, 3, false},
{ 7, 1.30, 0.00021076195, 1.68304663764105911, 0.525806935628255534, 3, false},
{ 8, 1.30, 4.0415299e-05, 1.74423607720269080, 0.518215688069661340, 3, false},
{ 9, 1.30, 9.7550762e-06, 1.77520938449435794, 0.514374542236327836, 3, false},
{10, 1.30, 2.2939521e-06, 1.80346357557508630, 0.511223093668619510, 3, false},
{11, 1.30, 5.6657229e-07, 1.82270626491970456, 0.509227244059244799, 3, false},
{12, 1.30, 1.3356139e-07, 1.82085249159071205, 0.509485626220703036, 3, false},
{13, 1.30, 3.3045491e-08, 1.83200497097439241, 0.508312733968098729, 3, false},
{14, 1.30, 1.4550131e-08, 1.85990854899088531, 0.505644480387369533, 3, false},
//{15, 1.30, 2.4402664e-08, 1.89877065022786451, 0.502065785725911540, 3, false},
//{16, 1.30, 4.4693519e-08, 2.02004547119140643, 0.489640045166015558, 3, false},
{ 4, 1.35,   0.012750349, 1.38681869506835920, 0.598430379231771004, 3, false},
{ 5, 1.35,  0.0027935232, 1.59968931410047732, 0.545594278971354263, 3, false},
{ 6, 1.35,   0.000557334, 1.67900339762369777, 0.531911468505859464, 3, false},
{ 7, 1.35,  0.0001260106, 1.73462778727213518, 0.523948923746745066, 3, false},
{ 8, 1.35, 2.2608425e-05, 1.78767479790581585, 0.517632293701171697, 3, false},
{ 9, 1.35, 4.9480543e-06, 1.82514563666449625, 0.513285064697265758, 3, false},
{10, 1.35, 1.0561576e-06, 1.84514923095703098, 0.511175282796223707, 3, false},
{11, 1.35, 2.3212036e-07, 1.85591006808810710, 0.509974924723307299, 3, false},
{12, 1.35, 5.1494208e-08, 1.86972944471571201, 0.508644358317057343, 3, false},
{13, 1.35,  1.138375e-08, 1.87267608642578098, 0.508304595947265625, 3, false},
{14, 1.35, 3.5079381e-09, 1.88038957383897576, 0.507539621988932210, 3, false},
//{15, 1.35, 4.0548476e-09, 1.91844601101345558, 0.504120127360025871, 3, false},
//{16, 1.35, 7.8390451e-09, 2.01599850124783053, 0.494870249430338471, 3, false},
{ 4, 1.40,   0.010028617, 1.42907129923502607, 0.595376586914062478, 3, false},
{ 5, 1.40,  0.0017469917, 1.62376225789388040, 0.547126770019531361, 3, false},
{ 6, 1.40, 0.00032887857, 1.72446373833550348, 0.530658213297526271, 3, false},
{ 7, 1.40, 7.1343128e-05, 1.78181135389539902, 0.522699228922526182, 3, false},
{ 8, 1.40, 1.2767573e-05, 1.82943505181206612, 0.517185719807942368, 3, false},
{ 9, 1.40, 2.5766022e-06, 1.87064327663845464, 0.512637074788411362, 3, false},
{10, 1.40, 5.1721007e-07, 1.89010603162977397, 0.510741933186848796, 3, false},
{11, 1.40, 9.7990407e-08, 1.88815528021918388, 0.510795338948567546, 3, false},
{12, 1.40, 1.9775866e-08, 1.90691307915581620, 0.508948008219400894, 3, false},
{13, 1.40, 4.0756893e-09, 1.91943817138671879, 0.507719167073567790, 3, false},
{14, 1.40, 1.0288549e-09, 1.92299652099609353, 0.507370249430338593, 3, false},
{15, 1.40, 8.0851857e-10, 1.96251508924696161, 0.503927357991536318, 3, false},
//{16, 1.40, 1.5114393e-09, 2.02583058675130134, 0.498289235432942790, 3, false},
{ 4, 1.45,  0.0084340373, 1.48220782809787344, 0.588401285807291785, 3, false},
{ 5, 1.45,  0.0012557792, 1.67027257283528652, 0.544658406575520782, 3, false},
{ 6, 1.45, 0.00022370851, 1.77476925320095469, 0.528379058837890714, 3, false},
{ 7, 1.45, 4.1620796e-05, 1.81916283501519116, 0.522077687581380068, 3, false},
{ 8, 1.45, 7.7564435e-06, 1.87167646620008665, 0.516175079345702947, 3, false},
{ 9, 1.45, 1.4751791e-06, 1.90793134901258621, 0.512265777587890381, 3, false},
{10, 1.45, 2.8396575e-07, 1.93040042453342009, 0.510135142008463660, 3, false},
{11, 1.45, 4.9998671e-08, 1.91693539089626719, 0.511350250244140492, 3, false},
{12, 1.45, 9.0073681e-09, 1.93405948215060786, 0.509625498453776027, 3, false},
{13, 1.45, 1.6571393e-09, 1.95315670437283018, 0.507810211181640714, 3, false},
{14, 1.45, 3.8188569e-10, 1.96678585476345469, 0.506616465250650960, 3, false},
{15, 1.45, 2.2273757e-10, 1.99709150526258683, 0.504047902425130090, 3, false},
//{16, 1.45, 3.7020618e-10, 2.03698679606119848, 0.500698598225911340, 3, false},
{ 4, 1.50,   0.007153168, 1.55200220743815076, 0.577678934733072857, 3, false},
{ 5, 1.50, 0.00096496355, 1.71065928141276036, 0.543058522542317568, 3, false},
{ 6, 1.50, 0.00016175042, 1.81697709825303821, 0.527150217692057388, 3, false},
{ 7, 1.50, 2.7200495e-05, 1.87820756700303804, 0.519385019938151005, 3, false},
{ 8, 1.50,  4.870632e-06, 1.91592831081814241, 0.515136464436848796, 3, false},
{ 9, 1.50, 8.4997625e-07, 1.94864722357855880, 0.511623891194661340, 3, false},
{10, 1.50, 1.5476196e-07, 1.96809505886501723, 0.509795888264973684, 3, false},
{11, 1.50,  2.674505e-08, 1.95902014838324612, 0.510626475016276227, 3, false},
{12, 1.50, 4.5144856e-09, 1.97183990478515603, 0.509373728434244777, 3, false},
{13, 1.50, 6.8283968e-10, 1.97394731309678861, 0.508898671468098862, 3, false},
{14, 1.50, 1.4599734e-10, 1.99087371826171844, 0.507419077555338438, 3, false},
{15, 1.50, 7.5296568e-11, 2.02536807590060741, 0.504684193929036407, 3, false},
//{16, 1.50, 1.1397531e-10, 2.06267479790581598, 0.501650238037109619, 3, false},
{ 4, 1.55,  0.0061063505, 1.61562355889214393, 0.568545023600260491, 3, false},
{ 5, 1.55, 0.00077545834, 1.74115142822265612, 0.542447662353515714, 3, false},
{ 6, 1.55, 0.00012241785, 1.84778255886501697, 0.526968638102213394, 3, false},
{ 7, 1.55, 1.9482436e-05, 1.91922556559244795, 0.518272145589192546, 3, false},
{ 8, 1.55, 3.3063667e-06, 1.96090749104817696, 0.513719431559244599, 3, false},
{ 9, 1.55, 5.2461237e-07, 1.96196679009331620, 0.513517506917317901, 3, false},
{10, 1.55, 8.8404119e-08, 1.99232838948567714, 0.510506439208984419, 3, false},
{11, 1.55, 1.4562607e-08, 2.00348086886935883, 0.509428660074869644, 3, false},
{12, 1.55, 2.4387385e-09, 2.01106753879123179, 0.508689626057942546, 3, false},
{13, 1.55, 3.5106362e-10, 1.99755028618706554, 0.509466298421224040, 3, false},
{14, 1.55, 6.4089206e-11, 2.01755388047960027, 0.507781219482421942, 3, false},
{15, 1.55, 2.6815991e-11, 2.04735599093966991, 0.505527496337890736, 3, false},
//{16, 1.55, 3.6018465e-11, 2.08681488037109375, 0.502400461832681944, 3, false},
{ 4, 1.60,  0.0051412093, 1.66557473076714380, 0.562941996256510202, 3, false},
{ 5, 1.60, 0.00063416881, 1.76781667073567705, 0.542211151123046964, 3, false},
{ 6, 1.60, 9.4423474e-05, 1.87210167778862813, 0.527448781331380179, 3, false},
{ 7, 1.60, 1.3996366e-05, 1.95363786485460023, 0.517866770426432477, 3, false},
{ 8, 1.60, 2.2758598e-06, 1.99141828748914929, 0.513874053955077903, 3, false},
{ 9, 1.60, 3.3403057e-07, 1.98512217203776031, 0.514410654703776005, 3, false},
{10, 1.60,  5.112808e-08, 2.01318240695529616, 0.511495717366536540, 3, false},
{11, 1.60, 7.7642371e-09, 2.03020952012803813, 0.509832000732421853, 3, false},
{12, 1.60, 1.2516932e-09, 2.04782223171657973, 0.508222198486328103, 3, false},
{13, 1.60, 1.8679518e-10, 2.05101131863064312, 0.507862599690754979, 3, false},
{14, 1.60,  3.183807e-11, 2.05666961669921911, 0.507260386149088527, 3, false},
{15, 1.60, 1.1402462e-11, 2.07673661973741330, 0.505748240152994688, 3, false},
//{16, 1.60, 1.4072305e-11, 2.11808285183376777, 0.502554066975911651, 3, false},
{ 4, 1.65,  0.0044209415, 1.70882788764105853, 0.558200836181640403, 3, false},
{ 5, 1.65, 0.00054364405, 1.80731658935546857, 0.539609527587890825, 3, false},
{ 6, 1.65, 7.8310497e-05, 1.89124738905164880, 0.527779897054036606, 3, false},
{ 7, 1.65, 1.0669513e-05, 1.97871042887369786, 0.517832692464192723, 3, false},
{ 8, 1.65, 1.6593665e-06, 2.02737477620442652, 0.512729136149088327, 3, false},
{ 9, 1.65, 2.3996152e-07, 2.01524505615234384, 0.514055633544921897, 3, false},
{10, 1.65, 3.3728431e-08, 2.03220876057942634, 0.512120819091796853, 3, false},
{11, 1.65, 4.7299116e-09, 2.05390947129991286, 0.510055287679036407, 3, false},
{12, 1.65, 7.0886193e-10, 2.07090674506293437, 0.508435312906900916, 3, false},
{13, 1.65, 1.0319888e-10, 2.08033972846137116, 0.507582346598307255, 3, false},
{14, 1.65, 1.7030402e-11, 2.09127587212456634, 0.506627146402994555, 3, false},
{15, 1.65, 5.4556804e-12, 2.10138024224175401, 0.505838775634765425, 3, false},
//{16, 1.65, 6.0499602e-12, 2.14197302924262134, 0.502792104085286407, 3, false},
{ 4, 1.70,  0.0037681007, 1.75829043918185701, 0.552664438883463616, 3, false},
{ 5, 1.70, 0.00045514602, 1.83974456787109331, 0.537987009684244954, 3, false},
{ 6, 1.70, 6.4590283e-05, 1.92978871663411455, 0.525992584228515692, 3, false},
{ 7, 1.70, 8.1297918e-06, 2.00241783989800304, 0.517897796630859220, 3, false},
{ 8, 1.70, 1.2057696e-06, 2.05249209933810794, 0.512742869059244488, 3, false},
{ 9, 1.70, 1.7345744e-07, 2.04652421739366375, 0.513499704996744466, 3, false},
{10, 1.70, 2.3278718e-08, 2.05339101155599080, 0.512527211507161184, 3, false},
{11, 1.70, 2.9834657e-09, 2.06418914794921893, 0.511281077067057388, 3, false},
{12, 1.70, 4.0639272e-10, 2.08515879313151054, 0.509268951416015692, 3, false},
{13, 1.70, 5.6310644e-11, 2.10078718397352437, 0.507962799072265603, 3, false},
{14, 1.70, 8.7517377e-12, 2.11526675754123383, 0.506744130452474129, 3, false},
{15, 1.70, 2.7541776e-12, 2.12576276991102464, 0.505927785237630157, 3, false},
//{16, 1.70, 2.9108591e-12, 2.17013583713107661, 0.502714538574218772, 3, false},
{ 4, 1.75,  0.0031840239, 1.78472069634331620, 0.551698048909505245, 3, false},
{ 5, 1.75, 0.00037093925, 1.87821875678168393, 0.535735321044921831, 3, false},
{ 6, 1.75,   5.15205e-05, 1.96848670111762192, 0.524261728922526204, 3, false},
{ 7, 1.75, 6.1709009e-06, 2.02892269558376803, 0.517637888590494311, 3, false},
{ 8, 1.75, 8.7094116e-07, 2.07719167073567679, 0.512768809000650916, 3, false},
{ 9, 1.75, 1.2112176e-07, 2.10331607394748143, 0.510445404052734530, 3, false},
{10, 1.75, 1.6344439e-08, 2.10092146131727464, 0.510645294189453258, 3, false},
{11, 1.75, 2.0918625e-09, 2.10065290662977411, 0.510514068603515558, 3, false},
{12, 1.75, 2.5043281e-10, 2.10685204399956616, 0.509598541259765669, 3, false},
{13, 1.75, 3.2824997e-11, 2.12432301839192705, 0.508163706461588482, 3, false},
{14, 1.75, 4.6873557e-12, 2.14312557644314250, 0.506677500406901293, 3, false},
{15, 1.75, 1.4671295e-12, 2.15334370930989483, 0.505855814615885357, 3, false},
//{16, 1.75, 1.5452808e-12, 2.19921620686848929, 0.502564748128255356, 3, false},
{ 4, 1.80,  0.0027281206, 1.81566416422526022, 0.549645741780599106, 3, false},
{ 5, 1.80, 0.00030474186, 1.92916208902994835, 0.531913503011067768, 3, false},
{ 6, 1.80, 4.0966294e-05, 1.99691992865668366, 0.523411814371744888, 3, false},
{ 7, 1.80, 4.8277462e-06, 2.06023915608723929, 0.516692860921223907, 3, false},
{ 8, 1.80,  6.512528e-07, 2.10110795762803804, 0.512618255615234109, 3, false},
{ 9, 1.80, 8.6860004e-08, 2.12062293158637161, 0.510943349202474084, 3, false},
{10, 1.80, 1.1474937e-08, 2.13734046088324670, 0.509422047932942657, 3, false},
{11, 1.80, 1.4900238e-09, 2.13370751274956616, 0.509656016031900805, 3, false},
{12, 1.80, 1.7737321e-10, 2.12671017116970473, 0.509776560465494577, 3, false},
{13, 1.80, 2.2386587e-11, 2.14743737114800304, 0.508159637451171986, 3, false},
{14, 1.80, 2.8892214e-12, 2.10544586181640536, 0.512052663167317790, 3, false},
{15, 1.80, 8.5791121e-13, 2.17534840901692661, 0.505950164794921720, 3, false},
//{16, 1.80, 8.6835326e-13, 2.22360992431640536, 0.502550506591797119, 3, false},
{ 4, 1.85,  0.0023322608, 1.82287784152560750, 0.551604970296223796, 3, false},
{ 5, 1.85, 0.00025747909, 1.97668134901258674, 0.528661346435546742, 3, false},
{ 6, 1.85, 3.1810889e-05, 2.02382388644748223, 0.522790781656901182, 3, false},
{ 7, 1.85, 3.8746264e-06, 2.08198835584852526, 0.516882069905598818, 3, false},
{ 8, 1.85, 4.9082302e-07, 2.11999257405599018, 0.513011932373046986, 3, false},
{ 9, 1.85, 6.2341591e-08, 2.13356204562717044, 0.511795806884765447, 3, false},
{10, 1.85, 7.8301776e-09, 2.15372229682074634, 0.509919484456380134, 3, false},
{11, 1.85, 1.0020588e-09, 2.16671735975477509, 0.508801015218098618, 3, false},
{12, 1.85, 1.1885262e-10, 2.17137230767144151, 0.508251190185546764, 3, false},
{13, 1.85, 1.5041179e-11, 2.18463965521918491, 0.507171376546223907, 3, false},
{14, 1.85, 1.8789625e-12, 2.18464338514539991, 0.506948089599609020, 3, false},
{15, 1.85, 5.1887707e-13, 2.20492858886718768, 0.505540466308593617, 3, false},
{16, 1.85, 4.9637721e-13, 2.24612935384114509, 0.502656555175781339, 3, false},
{ 4, 1.90,  0.0020225494, 1.84992726643880223, 0.550009918212890803, 3, false},
{ 5, 1.90,  0.0002257929, 2.01164194742838554, 0.526721445719401160, 3, false},
{ 6, 1.90, 2.6486913e-05, 2.06700524224175330, 0.520410919189453147, 3, false},
{ 7, 1.90, 3.2374813e-06, 2.09541609022352526, 0.517656707763671786, 3, false},
{ 8, 1.90, 3.8577121e-07, 2.12510257297092009, 0.514389292399088371, 3, false},
{ 9, 1.90, 4.6617227e-08, 2.15220421685112884, 0.511920928955078125, 3, false},
{10, 1.90, 5.5628174e-09, 2.17061513264973893, 0.510152435302734353, 3, false},
{11, 1.90, 6.9286019e-10, 2.18844790988498250, 0.508651479085286296, 3, false},
{12, 1.90, 7.9290365e-11, 2.19517669677734384, 0.507954661051432166, 3, false},
{13, 1.90, 9.8965337e-12, 2.20817175971137170, 0.506905364990234530, 3, false},
{14, 1.90, 1.2251775e-12, 2.21245371500651000, 0.506451670328776249, 3, false},
{15, 1.90, 3.2863528e-13, 2.22906494140624956, 0.505279541015624778, 3, false},
{16, 1.90, 3.1562028e-13, 2.26889309353298607, 0.502519734700520915, 3, false},
{ 4, 1.95,   0.001769364, 1.88333621554904518, 0.547602589925130112, 3, false},
{ 5, 1.95,  0.0001980031, 2.04016469319661509, 0.525634002685546831, 3, false},
{ 6, 1.95, 2.3026206e-05, 2.08590850830078134, 0.520696258544921631, 3, false},
{ 7, 1.95,  2.730625e-06, 2.10214114718967027, 0.519080352783203303, 3, false},
{ 8, 1.95, 3.0568394e-07, 2.13631846110026080, 0.515235646565755157, 3, false},
{ 9, 1.95, 3.5339694e-08, 2.16356930202907982, 0.512669626871744888, 3, false},
{10, 1.95, 3.9532683e-09, 2.18178999159071196, 0.510828908284505112, 3, false},
{11, 1.95,  4.776842e-10, 2.20117441813151027, 0.509173838297526049, 3, false},
{12, 1.95, 5.1762217e-11, 2.21349809434678813, 0.508074696858723862, 3, false},
{13, 1.95, 6.3266466e-12, 2.22576209174262019, 0.507048288981119866, 3, false},
{14, 1.95, 7.6885514e-13, 2.23883175320095473, 0.506095631917317768, 3, false},
{15, 1.95, 2.0931511e-13, 2.24868621826171866, 0.505333709716796697, 3, false},
{16, 1.95,  2.050985e-13, 2.28954569498697902, 0.502578226725260535, 3, false},
{ 4, 2.00,  0.0015580009, 1.90694291856553799, 0.546800994873046786, 3, false},
{ 5, 2.00, 0.00017346297, 2.06404741075303821, 0.525147755940755356, 3, false},
{ 6, 2.00, 1.9950367e-05, 2.10745629204644125, 0.520621490478515692, 3, false},
{ 7, 2.00, 2.3314457e-06, 2.13151058620876643, 0.518270619710286762, 3, false},
{ 8, 2.00, 2.4596985e-07, 2.14249894883897563, 0.516482289632161362, 3, false},
{ 9, 2.00, 2.7354414e-08, 2.17356550428602446, 0.513518524169921942, 3, false},
{10, 2.00, 2.9017603e-09, 2.20186072455512116, 0.510932668050130157, 3, false},
{11, 2.00,  3.361087e-10, 2.21397179497612839, 0.509688568115234331, 3, false},
{12, 2.00, 3.6829545e-11, 2.24148000081380294, 0.507639821370442501, 3, false},
{13, 2.00, 4.1553645e-12, 2.24892120361328107, 0.506914011637369821, 3, false},
{14, 2.00, 5.2930439e-13, 2.25466528998480920, 0.506470998128255023, 3, false},
{15, 2.00, 1.4040047e-13, 2.25586632622612848, 0.506162770589192568, 3, false},
{16, 2.00, 1.3802599e-13, 2.30979546440972205, 0.502591959635416807, 3, false},
{ 4, 2.05,  0.0014145059, 1.92870330810546897, 0.545942433675130068, 3, false},
{ 5, 2.05, 0.00015484818, 2.08155195448133679, 0.525099436442057366, 3, false},
{ 6, 2.05, 1.7406431e-05, 2.13673621283637205, 0.519232940673827970, 3, false},
{ 7, 2.05,  2.028536e-06, 2.17350582546657956, 0.515741221110026071, 3, false},
{ 8, 2.05, 2.0866124e-07, 2.15772077772352455, 0.516779327392578147, 3, false},
{ 9, 2.05, 2.2532863e-08, 2.17791086832682312, 0.514518483479817657, 3, false},
{10, 2.05,  2.312491e-09, 2.21550852457682268, 0.511269378662109308, 3, false},
{11, 2.05,   2.58385e-10, 2.23040584988064206, 0.509801991780599040, 3, false},
{12, 2.05, 2.7699008e-11, 2.25799238416883696, 0.507725779215494555, 3, false},
{13, 2.05, 3.0548308e-12, 2.26661597357855804, 0.506922658284505112, 3, false},
{14, 2.05, 3.8241342e-13, 2.22907613118489589, 0.509736633300781228, 3, false},
{15, 2.05, 1.0054434e-13, 2.24838222927517339, 0.508189392089843595, 3, false},
{16, 2.05, 9.8986316e-14, 2.32577260335286473, 0.502707163492838727, 3, false},
{ 4, 2.10,  0.0012905133, 1.94683447943793375, 0.545642852783203014, 3, false},
{ 5, 2.10, 0.00013825796, 2.09675140380859348, 0.525319671630859353, 3, false},
{ 6, 2.10, 1.5069774e-05, 2.16241302490234455, 0.518203481038411518, 3, false},
{ 7, 2.10,  1.725143e-06, 2.21955549452039946, 0.512659962972005001, 3, false},
{ 8, 2.10, 1.7989863e-07, 2.17143944634331598, 0.517146046956380223, 3, false},
{ 9, 2.10, 1.9294534e-08, 2.21104380289713598, 0.513404083251952970, 3, false},
{10, 2.10, 1.8788313e-09, 2.22412465413411553, 0.511929066975911451, 3, false},
{11, 2.10, 2.0627664e-10, 2.23974558512369804, 0.510313161214192568, 3, false},
{12, 2.10, 2.1071284e-11, 2.26767900254991339, 0.508244578043619777, 3, false},
{13, 2.10, 2.2821409e-12, 2.22048051622178777, 0.511962127685547053, 3, false},
{14, 2.10, 2.8772368e-13, 2.25465596516927080, 0.509201558430989576, 3, false},
{15, 2.10, 7.4977376e-14, 2.26222398546006831, 0.508659871419270826, 3, false},
{16, 2.10, 7.2698728e-14, 2.35554487440321170, 0.501833343505859286, 3, false},
{ 4, 2.15,  0.0011992553, 1.95983327229817705, 0.545700836181640558, 3, false},
{ 5, 2.15, 0.00012594864, 2.10787777370876706, 0.525670623779296853, 3, false},
{ 6, 2.15, 1.3423187e-05, 2.17432640923394116, 0.518426767985025960, 3, false},
{ 7, 2.15, 1.5005507e-06, 2.23297949896918446, 0.512734731038411051, 3, false},
{ 8, 2.15, 1.5976257e-07, 2.19627329508463598, 0.516251881917317523, 3, false},
{ 9, 2.15, 1.6805329e-08, 2.22688106960720544, 0.513256072998046653, 3, false},
{10, 2.15, 1.6163274e-09, 2.22816043429904509, 0.512646738688151027, 3, false},
{11, 2.15, 1.7547081e-10, 2.25718299018012170, 0.510106658935546631, 3, false},
{12, 2.15, 1.7308758e-11, 2.27178565131293375, 0.508858489990234419, 3, false},
{13, 2.15, 1.8724472e-12, 2.23387468126085098, 0.512366485595703081, 3, false},
{14, 2.15, 2.1326748e-13, 2.30832027859157973, 0.506116994222005290, 3, false},
{15, 2.15, 5.6503445e-14, 2.31436089409722179, 0.505541992187499734, 3, false},
{16, 2.15, 5.4697116e-14, 2.37556152343749982, 0.501582845052083282, 3, false},
{ 4, 2.20,  0.0011025179, 1.97344377305772567, 0.546037038167317745, 3, false},
{ 5, 2.20, 0.00011280381, 2.11950395372178768, 0.526264699300130090, 3, false},
{ 6, 2.20, 1.1683843e-05, 2.18673960367838571, 0.518857574462890714, 3, false},
{ 7, 2.20, 1.2652877e-06, 2.24713083902994715, 0.512998199462890270, 3, false},
{ 8, 2.20, 1.3127215e-07, 2.26494496663411438, 0.511482493082681899, 3, false},
{ 9, 2.20, 1.3722152e-08, 2.27114783393012099, 0.510934702555338682, 3, false},
{10, 2.20, 1.3557514e-09, 2.27525821261935723, 0.510403188069661562, 3, false},
{11, 2.20, 1.3732683e-10, 2.29960717095269063, 0.508342234293619577, 3, false},
{12, 2.20, 1.3462918e-11, 2.30357954237196205, 0.507871754964192901, 3, false},
{13, 2.20, 1.4828143e-12, 2.25579172770182224, 0.511590830485025849, 3, false},
{14, 2.20, 1.5240984e-13, 2.33169013129340330, 0.505726114908854329, 3, false},
{15, 2.20, 4.2666582e-14, 2.34941474066840250, 0.504525756835937589, 3, false},
{16, 2.20,  4.350582e-14, 2.39430813259548581, 0.501571655273437500, 3, false},
{ 4, 2.25,  0.0010312612, 1.98338402642144063, 0.546487172444661384, 3, false},
{ 5, 2.25, 0.00010319811, 2.12779930962456554, 0.526887766520182321, 3, false},
{ 6, 2.25, 1.0418375e-05, 2.19571753607855902, 0.519317881266276093, 3, false},
{ 7, 2.25, 1.0951963e-06, 2.25832807752821108, 0.513249460856119444, 3, false},
{ 8, 2.25, 1.1061086e-07, 2.28025631374782956, 0.511336008707682144, 3, false},
{ 9, 2.25, 1.1327543e-08, 2.29882388644748259, 0.509762827555338527, 3, false},
{10, 2.25, 1.0994197e-09, 2.30602637396918420, 0.509100596110026005, 3, false},
{11, 2.25, 1.0771979e-10, 2.31697370741102437, 0.508083852132161451, 3, false},
{12, 2.25, 1.0287677e-11, 2.32586585150824599, 0.507328033447265736, 3, false},
{13, 2.25, 9.7149266e-13, 2.33827904595269054, 0.506330617268880179, 3, false},
{14, 2.25, 1.1007942e-13, 2.34637485080295116, 0.505700174967447791, 3, false},
{15, 2.25, 3.4634308e-14, 2.34067925347222205, 0.505843098958333126, 3, false},
{16, 2.25, 3.7331853e-14, 2.42448696560329946, 0.500596618652343728, 3, false},
{ 4, 2.30, 0.00096769063, 1.99225379096137201, 0.547050221761067768, 3, false},
{ 5, 2.30, 9.4727065e-05, 2.13487870958116410, 0.527610015869140359, 3, false},
{ 6, 2.30, 9.3153191e-06, 2.20337880452473955, 0.519864654541015603, 3, false},
{ 7, 2.30, 9.4730434e-07, 2.26955888536241268, 0.513473765055338482, 3, false},
{ 8, 2.30, 9.3071744e-08, 2.28904028998480946, 0.511767323811848751, 3, false},
{ 9, 2.30, 9.2769408e-09, 2.31326989067925348, 0.509712473551432010, 3, false},
{10, 2.30, 8.7844109e-10, 2.32252756754557277, 0.508909352620442790, 3, false},
{11, 2.30, 8.3103511e-11, 2.33217688666449563, 0.507999928792317590, 3, false},
{12, 2.30, 7.7409942e-12, 2.34049835205078072, 0.507298533121744888, 3, false},
{13, 2.30, 7.5421465e-13, 2.34607832166883634, 0.506789906819661407, 3, false},
{14, 2.30, 8.5861563e-14, 2.35363701714409723, 0.506177775065104196, 3, false},
{15, 2.30, 2.7909572e-14, 2.38203667534722197, 0.504256184895833348, 3, false},
{16, 2.30, 2.9037173e-14, 2.43938056098090339, 0.500613403320312345, 3, false},
{ 4, 2.35, 0.00091109603, 2.00018734402126697, 0.547697194417317679, 3, false},
{ 5, 2.35, 8.7362907e-05, 2.14067128499348991, 0.528427378336588482, 3, false},
{ 6, 2.35, 8.3763398e-06, 2.20968983968098920, 0.520482126871744888, 3, false},
{ 7, 2.35, 8.2005622e-07, 2.28155059814453187, 0.513612620035807210, 3, false},
{ 8, 2.35, 7.8056328e-08, 2.30517968071831536, 0.511560312906900849, 3, false},
{ 9, 2.35, 7.5761021e-09, 2.32373233371310839, 0.509988657633463127, 3, false},
{10, 2.35, 6.9623612e-10, 2.33070356580946259, 0.509332529703776182, 3, false},
{11, 2.35, 6.4454973e-11, 2.35038265652126777, 0.507772572835286540, 3, false},
{12, 2.35, 6.2931201e-12, 2.35220659044053804, 0.507554880777994710, 3, false},
{13, 2.35, 5.9923646e-13, 2.34172736273871562, 0.508011372884114687, 3, false},
{14, 2.35, 6.6065931e-14, 2.35412190755208384, 0.507126871744791652, 3, false},
{15, 2.35, 2.4012349e-14, 2.35584513346354107, 0.507138061523437322, 3, false},
{16, 2.35, 2.4399231e-14, 2.42357499864366321, 0.502361297607421875, 3, false},
{ 4, 2.40, 0.00087048784, 2.00592770046658009, 0.548258209228515314, 3, false},
{ 5, 2.40, 8.2266738e-05, 2.14426320393880232, 0.529153188069661606, 3, false},
{ 6, 2.40, 7.7433845e-06, 2.21353166368272536, 0.521029408772786695, 3, false},
{ 7, 2.40, 7.3185707e-07, 2.29392649332682330, 0.513477325439453236, 3, false},
{ 8, 2.40, 6.7906614e-08, 2.30989430745442670, 0.512092336018880268, 3, false},
{ 9, 2.40, 6.4461408e-09, 2.33141225179036482, 0.510267893473307144, 3, false},
{10, 2.40, 5.8576773e-10, 2.34538455539279589, 0.509139251708984220, 3, false},
{11, 2.40, 5.5096404e-11, 2.36717105441623232, 0.507470448811848884, 3, false},
{12, 2.40, 5.2638546e-12, 2.37904713948567714, 0.506586964925130334, 3, false},
{13, 2.40, 4.7890714e-13, 2.33895043267144098, 0.509288279215494799, 3, false},
{14, 2.40, 5.2262697e-14, 2.35107828776041705, 0.508315022786458126, 3, false},
{15, 2.40, 2.0930731e-14, 2.37529296874999973, 0.505843098958333126, 3, false},
{16, 2.40, 2.0700099e-14, 2.42124565972222161, 0.504003906250000133, 3, false},
{ 4, 2.45, 0.00082508682, 2.01244015163845535, 0.548992156982421986, 3, false},
{ 5, 2.45, 7.6964398e-05, 2.14864959716796866, 0.530002593994140625, 3, false},
{ 6, 2.45, 7.0753467e-06, 2.23052520751953054, 0.520584869384765669, 3, false},
{ 7, 2.45, 6.3897015e-07, 2.30423973931206483, 0.513812510172525938, 3, false},
{ 8, 2.45, 5.7196036e-08, 2.31863352457682304, 0.512532806396484242, 3, false},
{ 9, 2.45, 5.2826195e-09, 2.33694373236762143, 0.510904693603515647, 3, false},
{10, 2.45, 4.8663691e-10, 2.36320241292317723, 0.508957163492838260, 3, false},
{11, 2.45, 4.4807014e-11, 2.38042721218533027, 0.507595062255859042, 3, false},
{12, 2.45, 4.1839375e-12, 2.38868526882595456, 0.506927235921224018, 3, false},
{13, 2.45,  3.845637e-13, 2.33679080539279571, 0.511076100667317568, 3, false},
{14, 2.45, 4.0956394e-14, 2.35007866753472294, 0.509834798177083215, 3, false},
{15, 2.45, 1.8479256e-14, 2.34827524820963562, 0.509776051839192723, 3, false},
{16, 2.45, 1.7927619e-14, 2.45225253634982643, 0.503007507324218950, 3, false},
{ 4, 2.50, 0.00079279905, 2.01718461778428892, 0.549589284261067679, 3, false},
{ 5, 2.50, 7.3317529e-05, 2.15946265326605946, 0.529833730061848529, 3, false},
{ 6, 2.50, 6.4884782e-06, 2.25309499104817679, 0.519467417399088638, 3, false},
{ 7, 2.50,  5.749415e-07, 2.30846947564019045, 0.514437103271484064, 3, false},
{ 8, 2.50, 5.0363914e-08, 2.33213212754991250, 0.512434641520182366, 3, false},
{ 9, 2.50, 4.5804732e-09, 2.35149417453342036, 0.510761769612630090, 3, false},
{10, 2.50,  4.201735e-10, 2.37159847683376723, 0.509220123291015292, 3, false},
{11, 2.50, 3.7877442e-11, 2.38795047336154509, 0.507892100016275494, 3, false},
{12, 2.50, 3.5189578e-12, 2.39123280843099018, 0.507511138916015181, 3, false},
{13, 2.50, 3.2547081e-13, 2.39824133978949661, 0.506931304931640403, 3, false},
{14, 2.50, 3.6529298e-14, 2.37836642795138831, 0.508345540364583459, 3, false},
{15, 2.50, 1.6499646e-14, 2.43243543836805465, 0.504073079427083348, 3, false},
{16, 2.50, 1.6277483e-14, 2.46149902343749982, 0.503055826822916718, 3, false}};

template<typename T> T esknew (T v, T beta, T e0)
  {
  auto tmp = (1-v)*(1+v);
  auto tmp2 = tmp>=0;
  return tmp2*exp(beta*(pow(tmp*tmp2, e0)-1));
  }

const KernelParams &getKernel(size_t idx)
  {
  MR_assert(idx<KernelDB.size(), "no appropriate kernel found");
  return KernelDB[idx];
  }

shared_ptr<PolynomialKernel> selectKernel(size_t idx)
  {
  MR_assert(idx<KernelDB.size(), "no appropriate kernel found");
  auto supp = KernelDB[idx].W;
  auto beta = KernelDB[idx].beta*supp;
  auto e0 = KernelDB[idx].e0;
  auto lam = [beta,e0](double v){return esknew(v, beta, e0);};
  return make_shared<PolynomialKernel>(supp, supp+3, lam, GLFullCorrection(supp, lam));
  }

double bestEpsilon(size_t ndim, bool singleprec,
  double ofactor_min, double ofactor_max)
  {
  MR_assert((ndim>=1) && (ndim<=3), "bad dimensionality");
  double res = 1000.;
  for (const auto &krn:KernelDB)
    if ((krn.ndim==ndim) && (krn.singleprec==singleprec)
      && (krn.epsilon<=res)
      && (krn.ofactor<=ofactor_max) && (krn.ofactor>=ofactor_min))
      res = krn.epsilon;
  MR_assert(res<1000., "no appropriate kernel found");
  return res;
  }

}}
